/*
Copyright 2011-14 Newcastle University
   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at
       http://www.apache.org/licenses/LICENSE-2.0
   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/
(function() {
/** @file Contains code to load in the other script files, and initialise the exam.
 * Creates the global {@link Numbas} object, inside which everything else is stored, so as not to conflict with anything else that might be running in the page.
 */
    const _globalThis = (typeof globalThis !== 'undefined') ? globalThis : (typeof global !== 'undefined') ? global : window;
    if(typeof window == 'undefined') {
        window = _globalThis.window = _globalThis;
        _globalThis.alert = function(m) {
            console.error(m);
        }
    }
    if(!_globalThis.Numbas) {
        _globalThis.Numbas = {}
    }

// Polyfill for Promise.withResolvers. Remove once it's reached 95% compatibility.
if(!Promise.withResolvers) {
    Promise.withResolvers = function() {
        const out = {};
        out.promise = new this((resolve, reject) => {
            out.resolve = resolve;
            out.reject = reject;
        });
        return out;
    }
}

/** @namespace Numbas */
/** Extensions should add objects to this so they can be accessed */
Numbas.extensions = {};
/** A function for displaying debug info in the console. It will try to give a reference back to the line that called it, if it can.
 *
 * @param {string} msg - Text to display.
 * @param {boolean} [noStack=false] - Don't show the stack trace.
 * @param {Error} error
 */
Numbas.debug = function(msg, noStack, error) {
    if(window.console) {
        var e = new Error(msg);
        if(e.stack && !noStack) {
            var words = e.stack.split('\n')[2];
            if(error) {
                console.error(msg, error);
            } else {
                console.error(msg, " " + words);
            }
        } else {
            console.log(msg);
        }
    }
};
/** Display an error in a nice alert box. Also sends the error to the console via {@link Numbas.debug}.
 *
 * @param {Error} e
 */
Numbas.showError = function(e) {
    var message = (e || e.message) + '';
    message += ' <br> ' + e.stack.replace(/\n/g, '<br>\n');
    Numbas.debug(message, false, e);
    Numbas.display?.showAlert && Numbas.display.showAlert(message);
    throw(e);
};
/** Generic error class. Extends JavaScript's `Error`.
 *
 * @class
 * @param {string} message - A description of the error. Localised by R.js.
 * @param {object} args - Arguments for the error message.
 * @param {Error} originalError - If this is a re-thrown error, the original error object.
 */
Numbas.Error = function(message, args, originalError) {
    var e = new Error();
    e.name = "Numbas Error";
    e.message = _globalThis.R && R.apply(e, [message, args]);
    e.originalMessage = message;
    e.originalMessages = [message];
    if(originalError !== undefined) {
        e.originalError = originalError;
        if(originalError.originalMessages) {
            e.originalMessages = e.originalMessages.concat(originalError.originalMessages.filter(function(m) {
                return m != message
            }));
        }
    }
    return e;
}

var scriptreqs = Numbas.scriptreqs = {};
/** Keep track of loading status of a script and its dependencies.
 *
 * @param {string} file - Name of the script.
 * @param {Array.<string>} fdeps - Scripts which this one depends on.
 * @param {Function} callback
 * @global
 * @class
 * @property {string} file - Name of the script.
 * @property {boolean} loaded - Has the script been loaded yet?
 * @property {boolean} executed - Has the script been run?
 * @property {Array.<string>} backdeps - Scripts which depend on this one (need this one to run first)
 * @property {Array.<string>} fdeps - Scripts which this one depends on (it must run after them)
 * @property {Function} callback - The function to run when all this script's dependencies have run (this is the script itself)
 */
var RequireScript = Numbas.RequireScript = function(file, fdeps, callback) {
    this.file = file;
    scriptreqs[file] = this;
    this.backdeps = [];
    this.fdeps = fdeps || [];
    this.callback = callback;

    const {promise, resolve} = Promise.withResolvers();
    this.promise = promise;
    this.resolve = resolve;

    this.promise.then(() => {
        this.callback;
    });
}
RequireScript.prototype = {
    loaded: false,
    executed: false,
    backdeps: [],
    fdeps: [],
    callback: null,


    /** Try to run this script. It will run if all of its dependencies have run.
     * Once it has run, every script which depends on it will try to run.
     */
    script_loaded: function() {
        Promise.all(this.fdeps.map((r) => scriptreqs[r].promise)).then(() => {
            this.executed = true;

            if(this.callback) {
                var module = { exports: {} };
                this.callback.apply(window, [module]);
                for(var x in module.exports) {
                    window[x] = module.exports[x];
                    if(typeof global !== 'undefined') {
                        global[x] = module.exports[x];
                    }
                }
            }

            this.resolve();
        });
    }
};
/** Ask to load a javascript file. Unless `noreq` is set, the file's code must be wrapped in a call to Numbas.queueScript with its filename as the first parameter.
 *
 * @memberof Numbas
 * @param {string} file
 * @param {boolean} noreq - Don't create a {@link Numbas.RequireScript} object.
 * @returns {Numbas.RequireScript}
 */
var loadScript = Numbas.loadScript = function(file, noreq) {
    if(!noreq) {
        if(scriptreqs[file] !== undefined) {
            return scriptreqs[file];
        }
        var req = new RequireScript(file);
        return req;
    }
    return scriptreqs[file];
}

/**
 * Queue up a file's code to be executed.
 * Each script should be wrapped in this function.
 *
 * @param {string} file - Name of the script.
 * @param {Array.<string>} deps - A list of other scripts which need to be run before this one can be run.
 * @param {Function} callback - A function wrapping up this file's code.
 *
 * @returns {Promise} - Resolves when the file has been executed.
 */
Numbas.queueScript = function(file, deps, callback) {
    if(typeof(deps) == 'string') {
        deps = [deps];
    }

    for(var i = 0;i < deps.length;i++) {
        var dep = deps[i];
        deps[i] = dep;
        loadScript(dep);
        scriptreqs[dep].backdeps.push(file);
    }

    var req = scriptreqs[file];
    if(req) {
        req.fdeps = deps;
        req.callback = callback;
    } else {
        req = new RequireScript(file, deps, callback);
    }
    req.script_loaded();

    return req.promise;
}

/** Empty; kept for backwards compatibility. */
Numbas.tryInit = function() {
}


Numbas.awaitScripts = function(deps) {
    return Promise.all(deps.map((file) => loadScript(file).promise));
}

var extension_callbacks = {};
/** A wrapper round {@link Numbas.queueScript} to register extensions easily.
 * The extension is not run immediately - call {@link Numbas.activateExtension} to run the extension.
 *
 * @param {string} name - Unique name of the extension.
 * @param {Array.<string>} deps - A list of other scripts which need to be run before this one can be run.
 * @param {Function} callback - Code to set up the extension. It's given the object `Numbas.extensions.<name>` as a parameter, which contains a {@link Numbas.jme.Scope} object.
 *
 * @returns {Promise} - Resolves when the extension has been activated.
 */
Numbas.addExtension = function(name, deps, callback) {
    deps.push('jme');
    return Numbas.queueScript('extensions/' + name + '/' + name + '.js', deps, function() {
        var extension = Numbas.extensions[name] = {
            scope: new Numbas.jme.Scope()
        };
        extension_callbacks[name] = {
            callback: callback,
            extension: extension,
            activated: false
        }
    });
}

Numbas.extension_url_root = {};

/**
 * Get the URL of a standalone file from an extension.
 *
 * @param {string} extension - The name of the extension.
 * @param {string} path - The path to the script, relative to the extension's `standalone_scripts` folder.
 * @returns {string}
 */
Numbas.getStandaloneFileURL = function(extension, path) {
    const root = Numbas.extension_url_root[extension] || `extensions/${extension}`;
    return root + '/standalone_scripts/' + path;
}

/**
 * Load a standalone script from an extension.
 * Inserts a <script> tag into the page's head.
 *
 * @param {string} extension - The name of the extension.
 * @param {string} path - The path to the script, relative to the extension's `standalone_scripts` folder.
 * @param {string} [type] - The type of the script, such as `"module"`.
 */
Numbas.loadStandaloneScript = function(extension, path, type) {
    var script = document.createElement('script');
    if(type) { 
        script.setAttribute('type',type);
    }
    script.setAttribute('src', Numbas.getStandaloneFileURL(extension, path));
    document.head.appendChild(script);
}

/** Run the extension with the given name. The extension must have already been registered with {@link Numbas.addExtension}.
 *
 * @param {string} name
 */
Numbas.activateExtension = function(name) {
    var cb = extension_callbacks[name];
    if(!cb) {
        throw(new Numbas.Error("extension.not found", {name: name}));
    }
    if(!cb.activated) {
        cb.callback(cb.extension);
        cb.activated = true;
    }
}

/** Check all required scripts have executed - the theme should call this once the document has loaded.
 *
 * @returns {Array.<object>} A list of files which have not loaded.
 */
Numbas.checkAllScriptsLoaded = function() {
    var fails = [];
    for(var file in scriptreqs) {
        var req = scriptreqs[file];
        if(req.executed) {
            continue;
        }
        if(req.fdeps.every(function(f) {
            return scriptreqs[f].executed
        })) {
            var err = new Numbas.Error('die.script not loaded', {file:req.file});
            Numbas.display && Numbas.display.die(err);
        }
        fails.push({file: req.file, req: req, fdeps: req.fdeps.filter(function(f) {
            return !scriptreqs[f].executed
        })});
    };
    return fails;
}
})();

/** Resources to do with localisation: `preferred_locale` is the code of the locale to use, and `resources` is a dictionary of localisations.
 *
 * @name locale
 * @memberof Numbas
 * @type {object}
 */

/** Definitions of marking scripts for the built-in part types.
 *
 * @name raw_marking_scripts
 * @memberof Numbas
 * @type {Object<string>}
 */

/** Marking scripts for the built-in part types.
 *
 * @name marking_scripts
 * @memberof Numbas
 * @type {Object<Numbas.marking.MarkingScript>}
 */
;
Numbas.queueScript('diagnostic_scripts', ['diagnostic', 'marking'], function() {
            Numbas.raw_diagnostic_scripts = {"mastery": "// Mastery diagnostic script\n// The student must answer every question correctly.\n// They start with a topic that has no dependencies.\n// After answering a question, if they get it correct, it's done forever.\n// If it's incorrect, the question is put on the end of that topic's \"queue\", \n// so they'll be asked it again later.\n// Once all the questions in the topic are answered correctly, the next topic\n// with no unmet dependencies is picked.\n\n//////////////\n// Functions\n//////////////\n\nupdate_where (Update items in a list which satisfy the given predicate, applying the given function to them):\n    ((predicate, action, list) -> (if(predicate(x), action(x), x) for: x of: list))\n\n\nquestion_queue_for_topic (When starting a topic, this function makes a queue of questions which must be answered):\n    (topic) -> (\n        [\"question\": q, \"status\": \"unknown\"]\n        for: q\n        of: topic[\"topic\"][\"questions\"]\n    )\n\n\nstart_topic (A function to update the state, setting the current topic and filling the question queue from that topic):\n    (state,topic) -> \n        merge(\n            state,\n            [\n                \"current_topic\": topic,\n                \"question_queue\": question_queue_for_topic(topic)\n            ]\n        )\n\n\nget_next_question (A function to get the next question from the queue):\n    (state) -> \n        let(\n            queue, state[\"question_queue\"],\n\n            if(len(queue)>0,\n                queue[0][\"question\"], \n                nothing\n            )\n        )\n\n\nnext_topic (The next topic to assess):\n    (state) ->\n        let(\n            topics, state[\"topics\"], // List of the state object for each topic\n\n            topicdict, dict([t[\"topic\"][\"name\"],t] for: t of: topics), // A mapping from topic names to topic state objects\n\n            available_topics,   // Topics that we can move to next: either no dependencies, or all their dependencies have been passed.\n                filter(\n                    t -> let(\n                        all_deps_passed, all(topicdict[topicname][\"status\"] <> \"unknown\" for: topicname of: t[\"topic\"][\"depends_on\"]),\n                        all_deps_passed and t[\"status\"]=\"unknown\"\n                    )\n                    , topics\n                ),\n\n           if(len(available_topics)>0,available_topics[0],nothing)\n        )\n\n\n/////////////////////\n// Initial variables\n/////////////////////\n\nfirst_topic (The first topic to assess):\n    // Picks the first topic which doesn't depend on anything.\n    let(\n        topics, pre_state[\"topics\"],\n        filter(t -> len(t[\"topic\"][\"depends_on\"])=0, topics)[0]\n    )\n\n\nfirst_question (The first question to show the student):\n    get_next_question(state)\n\n\npre_state (A template for the `state` variable, which will be filled in with the chosen start topic):\n    [\n        \"topics\":   // For each topic, both the given info about that topic and a status, either \"passed\" or \"unknown\".\n            [\n                \"topic\": topic,\n                \"status\": if(len(topic[\"questions\"])=0,\"passed\",\"unknown\")  // A topic is \"passed\" if there are no questions left unasked.\n            ]\n            for: topic\n            of: values(topics)\n        ,\n        \"finished\": false   // Is the exam over?\n    ]\n\n\nstate (The initial state variable):\n    start_topic(pre_state, first_topic)\n\n\n/////////////////////////////\n// Notes used when moving on\n/////////////////////////////\n\ncorrect (Did the student get the current question right?):\n    current_question[\"credit\"]=1\n\n\nafter_answering (The state after the student answers a question):\n    let(\n        queue, state[\"question_queue\"],\n        empty_queue, len(queue) = 0,\n\n        nquestion, \n            // Set the status of this question in the queue.\n            if(not empty_queue, \n              merge(\n                queue[0],\n                [\"status\": if(correct,\"passed\",\"failed\")]\n              ), \n              nothing\n            ),\n\n        nqueue, \n            // Change the queue: either remove the current question if correct, or add it to the end.\n            queue[1..len(queue)] + if(correct or empty_queue, [], [nquestion]), \n\n        ntopics,\n            // Update the list of topics, setting the current topic to \"passed\" if the queue is now empty.\n            if(len(nqueue)=0,\n                update_where(t -> t=state[\"current_topic\"], t -> t+[\"status\": \"passed\"], state[\"topics\"]),\n                state[\"topics\"]\n            ),\n\n        merge(\n            // Return a new state with the new list of topics and question queue\n            state,\n            [\"topics\": ntopics, \"question_queue\": nqueue]\n        )\n    )\n\n\n///////////\n// Actions\n///////////\n\naction_next_question_same_topic (Move to the next question in the queue):\n    [\n        \"label\": translate(\"diagnostic.move to next question in topic\"),\n        \"state\": after_answering,\n        \"next_question\": get_next_question(after_answering)\n    ]\n\naction_next_topic (Move to the next topic):\n    let(\n        state, after_answering, // Start with the state we get from answering the question.\n\n        topic, next_topic(state), // Pick a new topic.\n\n        nstate, \n            if(topic <> nothing, \n                start_topic(state, topic)   // Update the state with the new topic.\n            , \n                state   // Otherwise, there's no next topic, so this action won't be used.\n            ),\n\n        [\n            \"label\": translate(\"diagnostic.move to next topic\"),\n            \"state\": nstate,\n            \"next_question\": get_next_question(nstate)\n        ]\n    )\n\nnext_actions (The list of possible actions after answering a question):\n    let(\n        state, after_answering,\n        queue_empty, len(state[\"question_queue\"])=0,\n        actions, \n            switch(\n                not queue_empty,\n                    [action_next_question_same_topic]   // Move to the next question in the queue\n            ,   next_topic(state) <> nothing,\n                    [action_next_topic] // Move to the next topic\n            ,\n                []  // End the exam\n            ),\n\n        [\n            \"feedback\": \"\",\n            \"actions\": actions\n        ]\n    )\n\nafter_exam_ended (The state after the exam has finished):\n    merge(\n        after_answering,\n        [\"finished\": true]\n    )\n\n\n//////////////////\n// Feedback notes\n//////////////////\n\nprogress (Summarise the student's progress through the exam):\n    let(\n        passed_topics, filter(t -> t[\"status\"] = \"passed\", state[\"topics\"])\n    ,   num_passed_topics, len(passed_topics)\n    ,   num_topics, len(state[\"topics\"])\n    ,   exam_progress, num_passed_topics/num_topics\n    ,   topic_credit, 1-len(state[\"question_queue\"])/len(state[\"current_topic\"][\"topic\"][\"questions\"])\n    ,   current_topic, state[\"current_topic\"][\"topic\"][\"name\"]\n    ,   lo_progress,\n            let(\n                ltopics, filter(t -> lo[\"name\"] in t[\"topic\"][\"learning_objectives\"], state[\"topics\"]),\n                passed, filter(t -> t[\"status\"]=\"passed\", ltopics),\n                p, len(passed)/len(ltopics),\n                [\"name\": lo[\"name\"], \"progress\": p, \"credit\": p]\n            )\n            for: lo\n            of: learning_objectives\n    ,   topic_progress, [[\"name\": \"Current topic: {current_topic}\", \"progress\": topic_credit, \"credit\": topic_credit]]\n\n    ,   if(state[\"finished\"], [], topic_progress)\n          + lo_progress\n          + [\n                [\"name\": translate(\"control.total\"), \"progress\": exam_progress, \"credit\": exam_progress]\n            ]\n    )\n\nfeedback (A text description of the current state): \n    if(state[\"finished\"],\n        translate(\"diagnostic.complete\")\n    ,\n        translate(\"diagnostic.studying topic\", [\"topic\": state[\"current_topic\"][\"topic\"][\"name\"]])\n    )\n\n", "diagnosys": "state (Produces the initial value of the state object): // should be renamed \"initial_state\"\n    [\n        \"topics\": map(\n            [\n                \"topic\": topic,\n                \"status\": \"unknown\" // \"unknown\", \"passed\", or \"failed\"\n            ],\n            topic,\n            values(topics)\n        ),\n        \"retries\": 3,\n        \"finished\": false,\n    ]\n\ntopics_by_objective (A dictionary mapping a learning objective name to a list of indices of topics):\n    dict(map(\n        let(\n            ltopics, values(topics),\n            indices, filter(lo[\"name\"] in ltopics[j][\"learning_objectives\"], j, 0..len(ltopics)-1),\n            [lo[\"name\"],indices]\n        ),\n        lo,\n        learning_objectives\n    ))\n\nunknown_topics (Which topics are still unknown?): \n    map(x[\"topic\"],x,filter(x[\"status\"]=\"unknown\",x,state[\"topics\"]))\n\nfirst_topic (The first topic to pick a question on):\n    unknown_topics[floor(len(unknown_topics)/2)][\"name\"]\n\nfirst_question (The first question to show the student):\n    random(topics[first_topic][\"questions\"])\n\nget_dependents (An expression which gets the topics to update after answering a question):\n    expression(\"\"\"\n        [target] + flatten(map(eval(get_dependents,[\"target\":t,\"correct\":correct]),t,topics[target][if(correct,\"depends_on\",\"leads_to\")]))\n    \"\"\")\n\ncorrect (Did the student get the current question right?):\n    current_question[\"credit\"]=1\n\nafter_answering (Update the state after the student answers a question):\n    let(\n        ntopics, eval(get_dependents,[\"target\":current_topic,\"correct\":correct])\n    ,   nstate, state + ['topics': map(\n                    if(tstate[\"topic\"][\"name\"] in ntopics, tstate + [\"status\":if(correct,\"passed\",\"failed\")], tstate),\n                    tstate,\n                    state[\"topics\"]\n                )]\n    , nstate\n    )\n\naction_retry (Use up one retry and visit the same topic again):\n    [\n        \"label\": translate(\"diagnostic.use retry\"),\n        \"state\": state + [\"retries\": state[\"retries\"]-1],\n        \"next_question\": random(topics[current_topic][\"questions\"])\n    ]\n\naction_stop (Stop the exam):\n    [\n        \"label\": translate(\"diagnostic.end test\"),\n        \"state\": state,\n        \"next_question\": nothing\n    ]\n\naction_move_on (Move to the next topic, or end the exam if there are no more):\n    let(\n        state, after_answering,\n        immediate_next_topics, topics[current_topic][if(correct, \"leads_to\", \"depends_on\")],\n        unknown_topics, map(x[\"topic\"],x,filter(x[\"status\"]=\"unknown\",x,state[\"topics\"])),\n        unknown_immediate_topics, filter(x[\"name\"] in immediate_next_topics,x,unknown_topics),\n        next_topics, if(len(unknown_immediate_topics), unknown_immediate_topics, unknown_topics),\n        finished, len(next_topics)=0 or state[\"finished\"],\n        topic,\n            if(not finished,\n                next_topics[floor(len(next_topics)/2)][\"name\"]\n            ,\n                nothing\n            ),\n        [\n            \"label\": translate(\"diagnostic.move to next topic\"),\n            \"state\": after_answering,\n            \"next_question\": if(not finished, random(topics[topic][\"questions\"]), nothing)\n        ]\n    )\n\ncan_move_on:\n    action_move_on[\"next_question\"]<>nothing\n\nnext_actions (Actions to offer to the student when they ask to move on):\n    let(\n        feedback, retries_feedback+\"\\n\\n\"+translate(\"diagnostic.next step question\")\n    ,   [\n            \"feedback\": feedback,\n            \"actions\": if(not correct and state[\"retries\"]>0, [action_retry], []) + if(can_move_on,[action_move_on],[action_stop])\n        ]\n    )\n\nafter_exam_ended (Update the state after the exam ends):\n    let(\n        state, after_answering,\n        ntopics, map(t+[\"status\": if(t[\"status\"]=\"unknown\",\"failed\",t[\"status\"])],t,state[\"topics\"]),\n        state+[\"finished\": true]\n    )\n\nfinished (Is the test finished? True if there are no unknown topics):\n    len(unknown_topics)=0 or state[\"finished\"]\n\ntotal_progress:\n    let(\n        num_topics, len(state[\"topics\"]),\n        known, filter(tstate[\"status\"]<>\"unknown\",tstate,state[\"topics\"]),\n        passed, filter(tstate[\"status\"]=\"passed\",tstate,known),\n        num_known, len(known),\n        num_passed, len(passed),\n        [\n            \"name\": translate(\"control.total\"),\n            \"progress\": if(num_topics>0,num_known/num_topics,0), \n            \"credit\": if(num_known>0,num_passed/num_topics,0)\n        ]\n    )\n\nlearning_objective_progress:\n    map(\n        let(\n            tstates, map(state[\"topics\"][j],j,topics_by_objective[lo[\"name\"]]),\n            known, filter(tstate[\"status\"]<>\"unknown\",tstate,tstates),\n            passed, filter(tstate[\"status\"]=\"passed\",tstate,known),\n            num_topics, len(tstates),\n            num_known, len(known),\n            num_passed, len(passed),\n            [\"name\": lo[\"name\"], \"progress\": if(num_topics>0,num_known/num_topics,0), \"credit\": if(finished,num_passed/num_topics,if(num_known>0,num_passed/num_known,0))]\n        ),\n        lo,\n        learning_objectives\n    )\n\nprogress (Progress on each of the learning objectives, plus total progress):\n    learning_objective_progress+\n    total_progress\n\nretries_feedback:\n    translate(\"diagnostic.now assessing topic\", [\"current_topic\": current_topic]) + \" \" +\n    let(\n        retries, state[\"retries\"], \n        pluralise(retries, translate(\"diagnostic.one retry left\"), translate(\"diagnostic.retries left\", [\"retries\": retries ]))\n    )\n    + \" \" +\n    let(\n        p,total_progress[\"progress\"],\n        percentage, dpformat(100p, 0),\n        translate(\"diagnostic.percentage completed\", [\"percentage\": percentage])\n    )\n\nweak_objective_threshold (The amount of credit below which a learning objective is considered weak):\n    0.6\n\nfinished_feedback:\n    let(\n        weak_objectives, filter(p[\"credit\"]<weak_objective_threshold, p, learning_objective_progress),\n    //\n        translate(\"diagnostic.test is over\") + \" \"\n        +\n        if(len(weak_objectives)=0,\n            translate(\"diagnostic.passed all lo\")\n        ,\n            translate(\"diagnostic.more work on lo\", [\"los\": \n                join(map(lo[\"name\"], lo, weak_objectives),\", \")])\n        )\n    )\n\nfeedback:\n    if(finished,\n        finished_feedback,\n        retries_feedback\n    )\n\n"};
        });
        ;

        Numbas.queueScript('localisation-resources', ['i18next'], function() {
        Numbas.locale = {
            preferred_locale: "en-GB",
            resources: {"ja-jp": {"translation": {"page.loading": "\u8aad\u307f\u8fbc\u307f\u4e2d...", "page.saving": "\u4fdd\u5b58\u4e2d\u3002\u6570\u79d2\u9593\u304b\u304b\u308b\u3053\u3068\u304c\u3042\u308a\u307e\u3059\u3002", "mathjax.math processing error": "\"{{-message}}\" when texifying <code>{{expression}}</code>", "die.numbas failed": "Numbas has failed", "die.sorry": "\u7533\u3057\u8a33\u3042\u308a\u307e\u305b\u3093\u3002Numbas\u306f\u30a8\u30e9\u30fc\u3092\u767a\u751f\u3057\u3053\u308c\u4ee5\u4e0a\u7d9a\u884c\u3067\u304d\u307e\u305b\u3093\u3002\u30a8\u30e9\u30fc\u306e\u8a73\u7d30\u306f\u4ee5\u4e0b\u3092\u53c2\u7167\u3057\u3066\u304f\u3060\u3055\u3044\u3002", "die.error": "\u30a8\u30e9\u30fc", "modal.ok": "OK", "modal.cancel": "\u30ad\u30e3\u30f3\u30bb\u30eb", "exam.exam name": "\u8a66\u9a13\u540d:", "exam.random seed": "\u30bb\u30c3\u30b7\u30e7\u30f3ID:", "exam.student name": "\u5b66\u751f\u6c0f\u540d:", "exam.number of questions": "\u554f\u984c\u6570:", "exam.marks available": "\u63a1\u70b9\u53ef\u80fd:", "exam.pass percentage": "\u5408\u683c\u7387:", "exam.time allowed": "\u5236\u9650\u6642\u9593:", "exam.passed": "\u5408\u683c", "exam.failed": "\u4e0d\u5408\u683c", "exam.review header": "\u8b1b\u8a55:\u2423", "frontpage.start": "\u958b\u59cb", "suspend.paused header": "\u4e00\u6642\u4f11\u6b62\u4e2d", "suspend.exam suspended": "\u8a66\u9a13\u306f\u4e2d\u65ad\u3055\u308c\u307e\u3057\u305f\u3002<em>\u518d\u958b</em>\u3092\u30af\u30ea\u30c3\u30af\u3057\u3066\u7d9a\u3051\u3066\u304f\u3060\u3055\u3044\u3002", "suspend.you can resume": "\u6b21\u56de\u3053\u306e\u30a2\u30af\u30c6\u30a3\u30d3\u30c6\u30a3\u3092\u59cb\u3081\u308b\u3068\u304d\uff0c\u30bb\u30c3\u30b7\u30e7\u30f3\u3092\u518d\u958b\u3059\u308b\u3053\u3068\u304c\u3067\u304d\u307e\u3059\u3002", "suspend.resume": "\u518d\u958b", "result.exit": "\u8a66\u9a13\u7d42\u4e86", "result.print": "Print this results summary", "result.exam summary": "\u8a66\u9a13\u6982\u8981", "result.performance summary": "\u6210\u7e3e\u306e\u6982\u8981", "result.exam start": "\u8a66\u9a13\u958b\u59cb:", "result.exam stop": "\u8a66\u9a13\u7d42\u4e86:", "result.time spent": "\u7d4c\u904e\u6642\u9593:", "result.questions attempted": "Questions Attempted:", "result.score": "\u5f97\u70b9:", "result.result": "\u7d50\u679c:", "result.question number": "\u554f\u984c\u756a\u53f7", "result.question score": "\u70b9\u6570", "result.question review title": "Review this question", "result.click a question to review": "\u554f\u984c\u756a\u53f7\u3092\u30af\u30ea\u30c3\u30af\u3059\u308b\u3068\uff0c\u3042\u306a\u305f\u306e\u89e3\u7b54\u304c\u3069\u306e\u3088\u3046\u306b\u63a1\u70b9\u3055\u308c\u305f\u306e\u304b\uff0c\u305d\u3057\u3066\uff0c\u53ef\u80fd\u306a\u3089\u6a21\u7bc4\u89e3\u7b54\u304c\u8868\u793a\u3055\u308c\u307e\u3059\u3002", "end.exam has finished": "\u8a66\u9a13\u306f\u7d42\u4e86\u3057\u307e\u3057\u305f\u3002\u3053\u306e\u30a6\u30a3\u30f3\u30c9\u30a6\u3092\u9589\u3058\u3066\u3082\u304b\u307e\u3044\u307e\u305b\u3093\u3002", "control.confirm leave": "\u307e\u3060\u8a66\u9a13\u306f\u7d42\u4e86\u3057\u3066\u3044\u307e\u305b\u3093\u3002", "control.not all questions answered": "\u3053\u306e\u8a66\u9a13\u306e\u554f\u984c\u306f\u5168\u3066\u306f\u5b8c\u4e86\u3057\u3066\u3044\u307e\u305b\u3093\u3002", "control.not all questions submitted": "\uff11\u3064\u4ee5\u4e0a\u306e\u89e3\u7b54\u304c\u4fee\u6b63\u3055\u308c\u307e\u3057\u305f\u304c\u3001\u307e\u3060\u4fee\u6b63\u3055\u308c\u305f\u89e3\u7b54\u306f\u63d0\u51fa\u3055\u308c\u3066\u3044\u307e\u305b\u3093\u3002\u5404\u554f\u984c\u304c\u63d0\u51fa\u3055\u308c\u3066\u3044\u308b\u304b\u78ba\u8a8d\u3057\u3066\u304f\u3060\u3055\u3044\u3002", "control.confirm end": "\u672c\u5f53\u306b\u8a66\u9a13\u3092\u7d42\u4e86\u3057\u305f\u3044\u3067\u3059\u304b\uff1f\u8a66\u9a13\u3092\u7d42\u4e86\u3059\u308b\u3068\u3001\u89e3\u7b54\u3092\u5909\u66f4\u3059\u308b\u3053\u3068\u306f\u3067\u304d\u306a\u304f\u306a\u308a\u307e\u3059\u3002", "control.confirm regen": "\u3053\u306e\u554f\u984c\u3092\u518d\u30e9\u30f3\u30c0\u30e0\u5316\u3057\u307e\u3059\u304b\uff1fOK\u3092\u30af\u30ea\u30c3\u30af\u3059\u308b\u3068\u3001\u73fe\u5728\u306e\u554f\u984c\u306b\u5bfe\u3059\u308b\u3042\u306a\u305f\u306e\u89e3\u7b54\u3068\u70b9\u6570\u304c\u3059\u3079\u3066\u5931\u308f\u308c\u307e\u3059\u3002", "control.confirm reveal": "\u3053\u306e\u554f\u984c\u306e\u7b54\u3048\u3092\u8868\u793a\u3057\u307e\u3059\u304b\uff1f\u73fe\u6642\u70b9\u3067\u306e\u70b9\u6570\u3067\u78ba\u5b9a\u3068\u306a\u308a\u3001\u5f8c\u3067\u3053\u306e\u554f\u984c\u306b\u7b54\u3048\u308b\u3053\u3068\u306f\u3067\u304d\u306a\u304f\u306a\u308a\u307e\u3059\u3002", "control.proceed anyway": "\u3068\u306b\u304b\u304f\u5148\u306b\u9032\u307f\u307e\u3059\u304b\uff1f", "control.regen": "\u540c\u985e\u306e\u4ed6\u306e\u554f\u984c\u3092\u3084\u308a\u307e\u3057\u3087\u3046", "control.submit answer": "\u89e3\u7b54\u306e\u63d0\u51fa", "control.submit all parts": "\u5168\u3066\u306e\u30d1\u30fc\u30c8\u3092\u63d0\u51fa", "control.submit again": "\u518d\u63d0\u51fa", "control.submit": "\u63d0\u51fa", "control.previous": "\u524d", "control.next": "\u6b21", "control.advice": "\u30a2\u30c9\u30d0\u30a4\u30b9", "control.reveal": "Reveal answers", "control.total": "\u5408\u8a08", "control.pause": "\u4e00\u6642\u4f11\u6b62\u3059\u308b", "control.end exam": "\u8a66\u9a13\u3092\u7d42\u4e86\u3059\u308b", "control.back to results": "\u7d50\u679c\u306b\u623b\u308b", "display.part.jme.error making maths": "\u6570\u5f0f\u8868\u793a\u306b\u30a8\u30e9\u30fc\u304c\u767a\u751f\u3057\u307e\u3057\u305f", "exam.xml.bad root": "Root element of exam XML should be 'exam'", "exam.changeQuestion.no questions": "\u3053\u306e\u8a66\u9a13\u306b\u306f\u554f\u984c\u304c\u3042\u308a\u307e\u305b\u3093\uff01.exam\u30d5\u30a1\u30a4\u30eb\u3092\u78ba\u8a8d\u3057\u3066\u304f\u3060\u3055\u3044\u3002", "feedback.you were awarded": "You were awarded <strong>{{count,niceNumber}}</strong> $t(mark).", "feedback.taken away": "<strong>{{count,niceNumber}}</strong> $t(mark) $t(was) taken away.", "jme.tokenise.invalid": "\u7121\u52b9\u306a\u5f0f\u3067\u3059: <code>{{expression}}</code>", "jme.shunt.not enough arguments": "\u6f14\u7b97 {{op}}\u306e\u305f\u3081\u306e\u5909\u6570\u304c\u8db3\u308a\u307e\u305b\u3093\u3002", "jme.shunt.no left bracket in function": "\u95a2\u6570\u3084\u7d44\u306b\u5bfe\u5fdc\u3059\u308b\u5de6\u62ec\u5f27\u304c\u3042\u308a\u307e\u305b\u3093\u3002", "jme.shunt.no left square bracket": "\u5bfe\u5fdc\u3059\u308b\u5de6\u62ec\u5f27\u304c\u3042\u308a\u307e\u305b\u3093\u3002", "jme.shunt.no left bracket": "\u5bfe\u5fdc\u3059\u308b\u5de6\u62ec\u5f27\u304c\u3042\u308a\u307e\u305b\u3093\u3002", "jme.shunt.no right bracket": "\u5bfe\u5fdc\u3059\u308b\u53f3\u62ec\u5f27\u304c\u3042\u308a\u307e\u305b\u3093\u3002", "jme.shunt.no right square bracket": "\u30ea\u30b9\u30c8\u306e\u7d42\u308f\u308a\u306b\u5bfe\u5fdc\u3059\u308b\u53f3\u306e\u89d2\u62ec\u5f27\u304c\u3042\u308a\u307e\u305b\u3093\u3002", "jme.shunt.missing operator": "\u5f0f\u304c\u8a55\u4fa1\u3067\u304d\u307e\u305b\u3093\u3002-- \u6f14\u7b97\u5b50\u304c\u3042\u308a\u307e\u305b\u3093\u3002", "jme.typecheck.function maybe implicit multiplication": "\u6f14\u7b97 {{name}} \u306f\u5b9a\u7fa9\u3055\u308c\u3066\u3044\u307e\u305b\u3093\u3002\u3042\u306a\u305f\u304c\u8868\u3057\u305f\u304b\u3063\u305f\u306e\u306f <br/><code>{{first}}*{{possibleOp}}(...)</code>\u3067\u3059\u304b\uff1f", "jme.typecheck.function not defined": "\u6f14\u7b97 <code>{{op}}</code> \u306f\u5b9a\u7fa9\u3055\u308c\u3066\u3044\u307e\u305b\u3093\u3002 <code>{{op}}</code> \u306f\u5909\u6570\u3067\u3059\u304b\u3001\u3042\u306a\u305f\u304c\u8868\u3057\u305f\u304b\u3063\u305f\u306e\u306f <code>{{suggestion}}*(...)</code> \u3067\u3059\u304b\uff1f", "jme.typecheck.op not defined": "\u6f14\u7b97 '{{op}}' \u306f\u5b9a\u7fa9\u3055\u308c\u3066\u3044\u307e\u305b\u3093\u3002", "jme.typecheck.no right type definition": "No definition of '{{op}}' of correct type found.", "jme.typecheck.no right type unbound name": "\u5909\u6570 <code>{{name}}</code> \u306f\u5b9a\u7fa9\u3055\u308c\u3066\u3044\u307e\u305b\u3093\u3002", "jme.typecheck.map not on enumerable": "\u6f14\u7b97<code>map</code>\u306f\u3001{{type}}\u3067\u306f\u306a\u304f\u3001\u30ea\u30b9\u30c8\u3084\u7bc4\u56f2\u306b\u5bfe\u3057\u3066\u52d5\u4f5c\u3057\u307e\u3059\u3002", "jme.evaluate.undefined variable": "\u5909\u6570 {{name}} \u306f\u672a\u5b9a\u7fa9\u3067\u3059\u3002", "jme.thtml.not html": "Passed a non-HTML value into the THTML constructor.", "jme.func.switch.no default case": "No default case for Switch statement", "jme.func.listval.invalid index": "Invalid list index {{index}} on list of size {{size}}", "jme.func.listval.not a list": "Object is not subscriptable", "jme.func.matrix.invalid row type": "Can't construct a matrix from rows of type {{type}}", "jme.func.except.continuous range": "Can't use the 'except' operator on continuous ranges.", "jme.matrix.reports bad size": "Matrix reports its size incorrectly - must be an error in constructor function", "jme.texsubvars.no right bracket": "No matching <code>]</code> in {{op}} arguments.", "jme.texsubvars.missing parameter": "Missing parameter in {{op}}: {{parameter}}", "jme.texsubvars.no right brace": "{{op}}\u306b\u5bfe\u5fdc\u3059\u308b<code>}</code>\u304c\u3042\u308a\u307e\u305b\u3093\u3002", "jme.user javascript.error": "Error in user-defined javascript function <code>{{name}}</code>: {{-message}}", "jme.variables.error making function": "Error making function <code>{{name}}</code>: {{-message}}", "jme.variables.syntax error in function definition": "\u95a2\u6570\u5b9a\u7fa9\u306e\u6587\u6cd5\u30a8\u30e9\u30fc", "jme.variables.variable not defined": "\u5909\u6570 <code>{{name}}</code> \u306f\u5b9a\u7fa9\u3055\u308c\u3066\u3044\u307e\u305b\u3093\u3002", "jme.variables.empty definition": "\u5909\u6570 <code>{{name}}</code> \u306e\u5b9a\u7fa9\u304c\u7a7a\u3067\u3059\u3002", "jme.variables.circular reference": "<code>{{name}}</code> \u306e\u5b9a\u7fa9\u306e\u4e2d\u3067\u5909\u6570\u306e\u53c2\u7167\u304c\u5faa\u74b0\u3057\u3066\u3044\u307e\u3059\u3002", "jme.variables.error computing dependency": "Error computing referenced variable <code>{{name}}</code>", "jme.variables.error evaluating variable": "Error evaluating variable {{name}}: {{-message}}", "jme.variables.question took too many runs to generate variables": "A valid set of question variables was not generated in time.", "jme.display.unknown token type": "Can't texify token type {{type}}", "jme.display.collectRuleset.no sets": "No sets given to collectRuleset!", "jme.display.collectRuleset.set not defined": "Ruleset {{name}} has not been defined", "jme.display.simplifyTree.no scope given": "Numbas.jme.display.simplifyTree must be given a Scope", "math.precround.complex": "Can't round to a complex number of decimal places", "math.siground.complex": "Can't round to a complex number of sig figs", "math.combinations.complex": "\u8907\u7d20\u6570\u306e\u7d44\u307f\u5408\u308f\u305b\u306e\u6570\u306f\u8a08\u7b97\u3067\u304d\u307e\u305b\u3093\u3002", "math.permutations.complex": "\u8907\u7d20\u6570\u306e\u9806\u5217\u306f\u8a08\u7b97\u3067\u304d\u307e\u305b\u3093\u3002", "math.gcf.complex": "\u8907\u7d20\u6570\u306e\u6700\u5927\u516c\u7d04\u6570(GCD)\u306f\u8a08\u7b97\u3067\u304d\u307e\u305b\u3093\u3002", "math.lcm.complex": "\u8907\u7d20\u6570\u306e\u6700\u5c0f\u516c\u500d\u6570(LCM)\u306f\u8a08\u7b97\u3067\u304d\u307e\u305b\u3093\u3002", "math.lt.order complex numbers": "\u8907\u7d20\u6570\u3092\u9806\u5e8f\u4ed8\u3051\u3059\u308b\u3053\u3068\u306f\u3067\u304d\u307e\u305b\u3093\u3002", "math.choose.empty selection": "Empty selection given to random function", "matrixmath.abs.non-square": "\u6b63\u65b9\u884c\u5217\u3067\u306a\u3044\u884c\u5217\u306e\u884c\u5217\u5f0f\u306f\u8a08\u7b97\u3067\u304d\u307e\u305b\u3093\u3002", "matrixmath.abs.too big": "\u3059\u307f\u307e\u305b\u3093\u30013\u6b21\u4ee5\u4e0a\u306e\u6b63\u65b9\u884c\u5217\u306e\u884c\u5217\u5f0f\u306f\u8a08\u7b97\u3067\u304d\u307e\u305b\u3093\u3002", "matrixmath.mul.different sizes": "\u30b5\u30a4\u30ba\u306e\u7570\u306a\u308b\u884c\u5217\u306e\u639b\u3051\u7b97\u306f\u3067\u304d\u307e\u305b\u3093\u3002", "vectormath.cross.not 3d": "3\u6b21\u5143\u30d9\u30af\u30c8\u30eb\u306e\u307f\u5916\u7a4d\u3092\u8a08\u7b97\u3059\u308b\u3053\u3068\u304c\u3067\u304d\u307e\u3059\u3002", "vectormath.dot.matrix too big": "$1 \\times N$ or $N \\times 1$ \u3067\u306f\u306a\u3044\u884c\u5217\u306e\u30c9\u30c3\u30c8\u7a4d\u306f\u8a08\u7b97\u3067\u304d\u307e\u305b\u3093\u3002", "vectormath.cross.matrix too big": "Can't calculate cross product of a matrix which isn't $1 \\times N$ or $N \\times 1$.", "part.with steps answer prompt": "\u89e3\u7b54: ", "part.script.error": "Error in part {{path}} custom script {{script}}: {{-message}}", "part.marking.steps no matter": "Because you received full marks for the part, your answers to the steps aren't counted.", "part.marking.revealed steps no penalty": "You revealed the steps.", "part.marking.used variable replacements": "This part was marked using your answers to previous parts.", "part.marking.variable replacement part not answered": "\u6700\u521d\u306b {{part}} \u3092\u89e3\u7b54\u3057\u306a\u3051\u308c\u3070\u306a\u308a\u307e\u305b\u3093\u3002", "part.marking.resubmit because of variable replacement": "\u3053\u306e\u90e8\u5206\u306e\u63a1\u70b9\u306f\u3001\u3042\u306a\u305f\u304c\u5909\u66f4\u3057\u305f\u4ed6\u306e\u90e8\u5206\u306e\u89e3\u7b54\u306b\u4f9d\u5b58\u3057\u307e\u3059\u3002\u70b9\u6570\u3092\u66f4\u65b0\u3059\u308b\u305f\u3081\u306b\u3053\u306e\u90e8\u5206\u3092\u518d\u63d0\u51fa\u3057\u3066\u304f\u3060\u3055\u3044\u3002", "part.marking.not submitted": "\u89e3\u7b54\u304c\u63d0\u51fa\u3055\u308c\u3066\u3044\u307e\u305b\u3093\u3002", "part.marking.did not answer": "\u3053\u306e\u554f\u984c\u306b\u89e3\u7b54\u3057\u3066\u3044\u307e\u305b\u3093", "part.marking.nothing entered": "\u89e3\u7b54\u3092\u5165\u529b\u3057\u3066\u3044\u307e\u305b\u3093", "part.marking.incorrect": "\u9593\u9055\u3044\u3067\u3059", "part.marking.correct": "\u6b63\u89e3\u3067\u3059", "part.marking.uncaught error": "{{part}}: {{-message}} \u63a1\u70b9\u6642\u306e\u30a8\u30e9\u30fc", "part.marking.no result": "\u3053\u306e\u90e8\u5206\u306f\u63a1\u70b9\u3067\u304d\u307e\u305b\u3093\u3067\u3057\u305f\u3002", "part.correct answer": "\u671f\u5f85\u3055\u308c\u308b\u89e3\u7b54:", "part.missing type attribute": "{{part}}: Missing part type attribute", "part.unknown type": "{{part}}: Unrecognised part type {{type}}", "part.setting not present": "Property '{{property}}' not set", "part.jme.answer missing": "\u6b63\u89e3\u306f\u3042\u308a\u307e\u305b\u3093\u3002", "part.jme.answer too long": "\u89e3\u7b54\u304c\u9577\u3059\u304e\u307e\u3059\u3002", "part.jme.answer too short": "\u89e3\u7b54\u304c\u77ed\u3059\u304e\u307e\u3059\u3002", "part.jme.answer invalid": "\u89e3\u7b54\u304c\u6570\u5f0f\u3068\u3057\u3066\u6b63\u3057\u304f\u3042\u308a\u307e\u305b\u3093\u3002", "part.jme.marking.correct": "\u3042\u306a\u305f\u306e\u89e3\u7b54\u306f\u6570\u5024\u7684\u306b\u306f\u6b63\u3057\u3044\u3067\u3059\u3002", "part.jme.must-have bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.must-have one": "\u89e3\u7b54\u306f {{strings}} \u3092\u542b\u307e\u306a\u304f\u3066\u306f\u3044\u3051\u307e\u305b\u3093\u3002", "part.jme.must-have several": "\u89e3\u7b54\u306f {{strings}} \u306e\u3059\u3079\u3066\u3092\u542b\u307e\u306a\u304f\u3066\u306f\u3044\u3051\u307e\u305b\u3093\u3002", "part.jme.not-allowed bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.not-allowed one": "\u89e3\u7b54\u306f {{strings}} \u3092\u542b\u3093\u3067\u306f\u3044\u3051\u307e\u305b\u3093\u3002", "part.jme.not-allowed several": "\u89e3\u7b54\u306f {{strings}} \u306e\u3046\u3061\u306e\u3069\u308c\u3082\u542b\u3093\u3067\u306f\u3044\u3051\u307e\u305b\u3093\u3002", "part.jme.unexpected variable name": "Your answer was interpreted to use the unexpected variable name <code>{{name}}</code>.", "part.jme.unexpected variable name suggestion": "Your answer was interpreted to use the unexpected variable name <code>{{name}}</code>. Did you mean <code>{{suggestion}}</code>?", "part.patternmatch.display answer missing": "Display answer is missing", "part.patternmatch.correct except case": "Your answer is correct, except for the case.", "part.numberentry.correct except decimal": "\u3042\u306a\u305f\u306e\u89e3\u7b54\u306f\u6709\u52b9\u306a\u7bc4\u56f2\u5185\u306b\u3042\u308a\u307e\u3059\u304c\u3001\u5c0f\u6570\u3067\u7b54\u3048\u3066\u306f\u3044\u3051\u307e\u305b\u3093\u3002", "part.numberentry.correct except fraction": "\u3042\u306a\u305f\u306e\u89e3\u7b54\u306f\u6709\u52b9\u306a\u7bc4\u56f2\u5185\u306b\u3042\u308a\u307e\u3059\u304c\u3001\u5206\u6570\u3067\u7b54\u3048\u3066\u306f\u3044\u3051\u307e\u305b\u3093\u3002", "part.numberentry.answer invalid": "\u6709\u52b9\u306a\u6570\u304c\u5165\u529b\u3055\u308c\u307e\u305b\u3093\u3067\u3057\u305f\u3002", "part.numberentry.answer not integer": "\u7121\u52b9\u306a\u89e3\u7b54\u3067\u3059\u3002\u5c0f\u6570\u3067\u306f\u306a\u304f\u30010\u4ee5\u4e0a\u306e\u6574\u6570\u3092\u5165\u529b\u3059\u308b\u5fc5\u8981\u304c\u3042\u308a\u307e\u3059\u3002", "part.numberentry.answer not integer or decimal": "\u7121\u52b9\u306a\u89e3\u7b54\u3067\u3059\u3002\u6574\u6570\u3082\u3057\u304f\u306f\u5c0f\u6570\u3092\u5165\u529b\u3059\u308b\u5fc5\u8981\u304c\u3042\u308a\u307e\u3059\u3002", "part.numberentry.zero sig fig": "This part is set up to round the student's answer to zero significant figures, which has no meaning.", "part.mcq.options def not a list": "{{properties}} \u3067\u5b9a\u7fa9\u3055\u308c\u308b\u5f0f\u306f\u30ea\u30b9\u30c8\u3067\u306f\u3042\u308a\u307e\u305b\u3093\u3002", "part.mcq.marking matrix string empty": "The custom marking matrix expression is empty.", "part.mcq.choices missing": "\u9078\u629e\u80a2\u306e\u5b9a\u7fa9\u304c\u3042\u308a\u307e\u305b\u3093\u3002", "part.mcq.matrix not a number": "Part {{part}} marking matrix cell ({{row}},{{column}}) does not evaluate to a number", "part.mcq.wrong number of choices": "\u9078\u629e\u80a2\u306e\u9078\u629e\u6570\u304c\u9593\u9055\u3063\u3066\u3044\u307e\u3059", "part.mcq.no choices selected": "\u9078\u629e\u80a2\u304c\u9078\u629e\u3055\u308c\u3066\u3044\u307e\u305b\u3093", "part.mcq.matrix not a list": "Marking matrix, defined by JME expression, is not a list but it should be.", "part.mcq.matrix wrong type": "Element of invalid type '{{type}}' used in marking matrix.", "part.mcq.matrix mix of numbers and lists": "Mix of numbers and lists used in marking matrix.", "part.mcq.matrix wrong size": "Marking matrix is the wrong size.", "part.mcq.correct choice": "\u6b63\u89e3\u3092\u9078\u3073\u307e\u3057\u305f\u3002", "part.matrix.answer invalid": "\u6709\u52b9\u306a\u89e3\u7b54\u3067\u306f\u3042\u308a\u307e\u305b\u3093\u3002", "part.matrix.invalid cell": "\u89e3\u7b54\u306e\u3046\u3061\u3001\u3044\u304f\u3064\u304b\u306e\u6b04\u304c\u7a7a\u3082\u3057\u304f\u306f\u7121\u52b9\u3067\u3059\u3002", "part.matrix.some incorrect": "\u89e3\u7b54\u306e\u3046\u3061\u3001\u3044\u304f\u3064\u304b\u306e\u6b04\u306f\u9593\u9055\u3063\u3066\u3044\u307e\u3059\u304c\u3001\u6b8b\u308a\u306e\u6b04\u306f\u63a1\u70b9\u3055\u308c\u3066\u3044\u307e\u3059\u3002", "part.matrix.empty": "\u307e\u3060\u89e3\u7b54\u304c\u5165\u529b\u3055\u308c\u3066\u3044\u307e\u305b\u3093", "part.matrix.empty cell": "\u89e3\u7b54\u306e\u3046\u3061\u3001\u3044\u304f\u3064\u304b\u306e\u6b04\u304c\u7a7a\u3067\u3059\u3002", "part.matrix.size mismatch": "The question author hasn't allowed the student to decide the dimensions of their answer, but the correct answer is {{correct_dimensions}} while the answer input is {{input_dimensions}}", "part.gapfill.feedback header": "<strong>{{name}}</strong>", "part.extension.not implemented": "Part hasn't implemented the <code>{{name}}</code> method.", "question.loaded name mismatch": "Can't resume this attempt - the package has changed since the last session.", "question.error": "\u554f {{number}}: {{-message}}", "question.preamble.error": "Error in preamble: {{-message}}", "question.preamble.syntax error": "Syntax error in preamble", "question.unsupported part type": "Unsupported part type", "question.header": "\u554f {{number}}", "question.submit part": "Save answer", "question.show steps": "Show steps", "question.show steps penalty": "You will lose <strong>{{count,niceNumber}}</strong> $t(mark).", "question.show steps no penalty": "Your score will not be affected.", "question.show steps already penalised": "You have already shown steps. You can show them again with no further penalty.", "question.hide steps": "Hide steps", "question.hide steps no penalty": "Your score will not be affected.", "question.advice": "Advice", "question.no such part": "Can't find part {{path}}.", "question.can not submit": "\u89e3\u7b54\u3092\u63d0\u51fa\u3067\u304d\u307e\u305b\u3093\u3002\u30a8\u30e9\u30fc\u3092\u30c1\u30a7\u30c3\u30af\u3057\u3066\u304f\u3060\u3055\u3044\u3002", "question.answer submitted": "\u63d0\u51fa\u3055\u308c\u305f\u89e3\u7b54", "question.score feedback.show": "\u30d5\u30a3\u30fc\u30c9\u30d0\u30c3\u30af\u3092\u8868\u793a\u3059\u308b", "question.score feedback.hide": "\u30d5\u30a3\u30fc\u30c9\u30d0\u30c3\u30af\u3092\u96a0\u3059", "question.score feedback.answered total actual": "\u70b9\u6570: {{score,niceNumber}}/{{marks,niceNumber}}", "question.score feedback.answered total": "{{marksString}}. Answered.", "question.score feedback.answered actual": "\u70b9\u6570: {{scoreString}}", "question.score feedback.answered": "\u89e3\u7b54\u6e08\u307f\u3002", "question.score feedback.unanswered": "\u672a\u89e3\u7b54\u3002", "question.score feedback.unanswered total": "{{marksString}}.", "question.score feedback.correct": "\u6b63\u89e3\u3067\u3059", "question.score feedback.partial": "\u90e8\u5206\u7684\u306b\u6b63\u89e3\u3067\u3059", "question.score feedback.wrong": "\u9593\u9055\u3044\u3067\u3059", "question.selector.unsubmitted changes": "Unsubmitted changes.", "timing.no accumulator": "no timing accumulator {{name}}", "timing.time remaining": "\u6b8b\u308a\u6642\u9593:", "xml.could not load": "XML\u30c9\u30ad\u30e5\u30e1\u30f3\u30c8\u304c\u30ed\u30fc\u30c9\u3067\u304d\u307e\u305b\u3093\u3067\u3057\u305f\u3002: {{-message}}", "xml.property not number": "Property {{name}} should be a number, but isn't ({{value}}), in node {{element}}", "xml.property not boolean": "Property {{name}} should be a boolean, but isn't ({{value}}), in node {{element}}", "xml.error in variable definition": "\u5909\u6570 <code>{{name}}</code> \u306e\u5b9a\u7fa9\u306b\u304a\u3051\u308b\u30a8\u30e9\u30fc", "scorm.error initialising": "Error initialising SCORM protocol: {{-message}}", "scorm.failed save": "<p>The request to save data to the server failed. Press <b>OK</b> to try again.</p>\n<p>If you get this message repeatedly, check your internet connection or use a different computer. Your previously submitted answers have been successfully saved and will be restored if you resume this session on a different computer.</p>\n<p>If this message appears persistently and you can't save <em>any</em> answers, please contact your lecturer or teacher.</p>", "scorm.no exam suspend data": "Failed to resume: no exam suspend data.", "scorm.error loading suspend data": "Error loading suspend data: {{-message}}", "scorm.error loading question": "Error loading question {{number}}: {{-message}}", "scorm.no question suspend data": "No question suspend data", "scorm.error loading part": "Error loading part {{part}}: {{-message}}", "scorm.no part suspend data": "No part suspend data", "util.product.non list": "Passed a non-list to <code>Numbas.util.product</code>", "mark": "\u63a1\u70b9", "was": "was", "part": "\u90e8\u5206", "gap": "gap", "step": "step", "jme.substituteTree.undefined variable": "\u672a\u5b9a\u7fa9\u306e\u5909\u6570: <code>{{name}}</code>", "jme.user javascript.returned undefined": "User-defined javascript function <code>{{name}}</code> returned <code>undefined</code>.", "part.marking.steps change": "You were awarded <strong>{{count,niceNumber}}</strong> $t(mark) for your answers to the steps.", "part.marking.revealed steps with penalty": "You revealed the steps. The maximum you can score for this part is <strong>{{count,niceNumber}}</strong> $t(mark). Your scores will be scaled down accordingly.", "part.marking.total score": "\u3053\u306e\u90e8\u5206\u306b\u5bfe\u3059\u308b\u3042\u306a\u305f\u306e\u5f97\u70b9\u306f\u3001<strong>{{count,niceNumber}}</strong> $t(mark) \u3067\u3057\u305f\u3002", "part.numberentry.precision type.dp": "\u5c11\u6570\u4f4d", "part.numberentry.precision type.dp_plural": "\u5c11\u6570\u4f4d", "part.numberentry.precision type.sigfig": "\u6709\u52b9\u6570\u5b57", "part.numberentry.precision type.sigfig_plural": "\u6709\u52b9\u6570\u5b57", "part.numberentry.give your answer to precision": "{{count,niceNumber}} {{precisionType}}\u306b\u4e38\u3081\u3066\u7b54\u3048\u3066\u304f\u3060\u3055\u3044\u3002", "question.unsubmitted changes": "\u89e3\u7b54\u304c\u4fee\u6b63\u3055\u308c\u307e\u3057\u305f\u304c\u3001\u63d0\u51fa\u306f\u3055\u308c\u3066\u3044\u307e\u305b\u3093\u3002\u89e3\u7b54\u3092\u78ba\u8a8d\u3057\u3066<strong>\u89e3\u7b54\u3092\u63d0\u51fa\u3059\u308b</strong>\u30dc\u30bf\u30f3\u3092\u62bc\u3057\u3066\u304f\u3060\u3055\u3044\u3002", "question.unsubmitted changes_plural": "\u89e3\u7b54\u304c\u4fee\u6b63\u3055\u308c\u307e\u3057\u305f\u304c\u3001\u63d0\u51fa\u306f\u3055\u308c\u3066\u3044\u307e\u305b\u3093\u3002\u5404\u90e8\u5206\u306e\u89e3\u7b54\u3092\u78ba\u8a8d\u3057\u3066<strong>\u3059\u3079\u3066\u306e\u89e3\u7b54\u3092\u63d0\u51fa\u3059\u308b</strong>\u30dc\u30bf\u30f3\u3092\u62bc\u3057\u3066\u304f\u3060\u3055\u3044\u3002", "util.equality not defined for type": "Equality not defined for type {{type}}", "mark_plural": "\u63a1\u70b9", "was_plural": "were", "die.script not loaded": "\u30d5\u30a1\u30a4\u30eb<code>{{file}}</code>\u304c\u30ed\u30fc\u30c9\u3055\u308c\u306a\u304b\u3063\u305f\u305f\u3081\u3001Numbas\u306f\u30b9\u30bf\u30fc\u30c8\u3067\u304d\u307e\u305b\u3093\u3067\u3057\u305f\u3002\u30d5\u30a1\u30a4\u30eb\u304c<code>scripts.js</code>\u306b\u542b\u307e\u308c\u3066\u3044\u308b\u304b\u78ba\u304b\u3081\u3066\u304f\u3060\u3055\u3044\u3002", "math.combinations.n less than zero": "\u7d44\u307f\u5408\u308f\u305b\u306e\u6570\u304c\u8a08\u7b97\u3067\u304d\u307e\u305b\u3093: n\u304c0\u672a\u6e80\u3067\u3059\u3002", "math.combinations.k less than zero": "\u7d44\u307f\u5408\u308f\u305b\u306e\u6570\u304c\u8a08\u7b97\u3067\u304d\u307e\u305b\u3093: k\u304c0\u672a\u6e80\u3067\u3059\u3002", "math.combinations.n less than k": "\u7d44\u307f\u5408\u308f\u305b\u306e\u6570\u304c\u8a08\u7b97\u3067\u304d\u307e\u305b\u3093: n\u304ck\u672a\u6e80\u3067\u3059\u3002", "math.permutations.n less than zero": "\u9806\u5217\u304c\u8a08\u7b97\u3067\u304d\u307e\u305b\u3093: n\u304c0\u672a\u6e80\u3067\u3059\u3002", "math.permutations.k less than zero": "\u9806\u5217\u304c\u8a08\u7b97\u3067\u304d\u307e\u305b\u3093: k\u304c0\u672a\u6e80\u3067\u3059\u3002", "math.permutations.n less than k": "\u9806\u5217\u304c\u8a08\u7b97\u3067\u304d\u307e\u305b\u3093: n\u304ck\u672a\u6e80\u3067\u3059\u3002", "part.numberentry.give your answer to precision_0": "\u89e3\u7b54\u3092\u6700\u3082\u8fd1\u3044\u6574\u6570\u306b\u4e38\u3081\u3066\u304f\u3060\u3055\u3044\u3002", "mathjax.error": "MathJax\u30d7\u30ed\u30bb\u30b9\u30a8\u30e9\u30fc: {{-message}}", "mathjax.error with context": "{{context}}\u3067\u306eMathJax\u51e6\u7406\u306e\u30a8\u30e9\u30fc: {{-message}}", "exam.introduction": "\u8a66\u9a13\u306e\u8aac\u660e", "exam.feedback": "\u8a66\u9a13\u306e\u30d5\u30a3\u30fc\u30c9\u30d0\u30c3\u30af\u30e1\u30c3\u30bb\u30fc\u30b8", "jme.tokenise.keypair key not a string": "Dictionary key should be a string, not {{type}}.", "jme.shunt.list mixed argument types": "Can't parse {{mode}}: mix of dictionary and list elements", "jme.func.listval.key not in dict": "Dictionary does not contain the key <code>{{key}}</code>", "part.prompt": "prompt", "part.feedback": "\u30d5\u30a3\u30fc\u30c9\u30d0\u30c3\u30af", "part.numberentry.answer not reduced": "\u65e2\u7d04\u5206\u6570\u306b\u306a\u3063\u3066\u3044\u307e\u305b\u3093\u3002", "part.numberentry.give your answer as a reduced fraction": "\u65e2\u7d04\u5206\u6570\u306b\u3057\u3066\u304f\u3060\u3055\u3044\u3002", "part.numberentry.negative decimal places": "This part is set up to round the student's answer to a negative number of decimal places, which has no meaning.", "part.mcq.choices": "\u9078\u629e\u80a2", "part.mcq.answers": "\u89e3\u7b54", "part.mcq.matrix cell empty": "Part {{part}} marking matrix cell ({{row}},{{column}}) is empty", "part.mcq.matrix jme error": "Part {{part}} marking matrix cell ({{row}},{{column}}) gives a JME error: {{-error}}", "question.statement": "Statement", "ruleset.circular reference": "\u30eb\u30fc\u30eb\u30bb\u30c3\u30c8<code>{{name}}</code>\u306e\u5b9a\u7fa9\u3067\u53c2\u7167\u304c\u5faa\u74b0\u3057\u3066\u3044\u307e\u3059\u3002", "ruleset.set not defined": "\u30eb\u30fc\u30eb\u30bb\u30c3\u30c8{{name}}\u306f\u5b9a\u7fa9\u3055\u308c\u3066\u3044\u307e\u305b\u3093\u3002", "jme.evaluate.no scope given": "Numbas.jme.evaluate must be given a Scope", "answer.number.not a number": "Your answer is not a valid number.", "answer.number.fractions not allowed": "You may not enter a fraction.", "answer.jme.invalid expression": "{{-message}}", "answer.matrix.fractions not allowed": "You may not enter fractions.", "answer.matrix.some cell not a number": "One or more of the cells in your answer is not a valid number.", "exam.enter password": "Password:", "exam.password.correct": "This password is correct. You can start the exam.", "exam.password.incorrect": "This password is incorrect.", "frontpage.scorm.lms not connected": "This exam is running in standalone mode. Your answers and marks will not be saved!", "result.question review": "Review", "control.confirm regen no marks": "Would you like to re-randomise this question?", "control.confirm reveal no marks": "Would you like to reveal the answer to this question?", "jme.tokenise.invalid near": "Invalid expression: <code>{{expression}}</code> at position {{position}} near <code>{{nearby}}</code>", "jme.tokenise.number.object not complex": "Invalid object passed into number constructor.", "jme.subvars.null substitution": "Empty variable substitution: <code>$t(left brace){{str}}$t(right brace)", "jme.type.type already registered": "The data type {{type}} has already been registered so can't be registered again.", "jme.type.no cast method": "Can't automatically convert from {{from}} to {{to}}.", "jme.display.simplifyTree.empty expression": "Expression is empty", "jme.display.simplifyTree.stuck in a loop": "Simplifier is stuck in a loop: <code>{{expr}}</code>", "math.niceNumber.undefined": "Was expecting a number, but got <code>undefined</code>", "math.rangeToList.zero step size": "Can't convert a range with step size zero to a list.", "part.error": "{{path}}: {{-message}}", "part.marking.revealed steps": "You revealed the steps.", "part.marking.maximum scaled down": "The maximum you can score for this part is <strong>{{count,niceNumber}}</strong> $t(mark). Your scores will be scaled down accordingly.", "part.marking.minimum score applied": "The minimum score for this part is <strong>{{score,niceNumber}}</strong>.", "part.marking.maximum score applied": "The maximum score for this part is <strong>{{score,niceNumber}}</strong>.", "part.marking.error in marking script": "There was an error in this part's marking algorithm. Please report this. {{-message}}", "part.marking.no result after replacement": "This part could not be marked using your answers to previous parts.", "part.marking.missing required note": "The marking algorithm does not define the note <code>{{note}}</code>", "marking.apply.not a list": "The first argument to <code>apply</code> must be a list, and isn't", "marking.apply marking script.script not found": "Marking script <code>{{name}}</code> not found", "marking.note.compilation error": "Error compiling note <code>{{name}}</code>: {{-message}}", "marking.note.error evaluating note": "Error evaluating note <code>{{name}}</code> - {{-message}}", "marking.note.invalid definition": "Invalid note definition: <code>{{source}}</code>. {{-hint}}", "marking.note.invalid definition.missing colon": "You might be missing a colon after the name and description", "marking.note.invalid definition.description missing closing bracket": "You might be missing a closing bracket", "marking.note.empty expression": "The note <code>{{name}}</code> is empty.", "marking.script.error parsing notes": "Error parsing marking script: {{- message}}", "part.feedback out of date": "This feedback is based on your last submitted answer. Save your changed answer to get updated feedback.", "part.jme.invalid value generator expression": "Invalid value generator expression for variable <code>{{name}}</code>: {{-message}}", "part.mcq.incorrect choice": "You chose an incorrect answer.", "part.matrix.not all cells same precision": "You have not given every cell in your answer to the same precision.", "part.gapfill.error marking gap": "Error marking {{name}}: {{-message}}", "part.custom.empty setting": "No value given.", "part.custom.unrecognised input type": "Unrecognised setting type <code>{{input_type}}</code>", "part.custom.error evaluating input option": "Error evaluating input option <code>{{option}}</code>: {{-error}}", "part.custom.input option missing": "Definition of input option <code>{{option}}</code> is missing.", "part.custom.error evaluating setting": "Error evaluating setting <code>{{setting}}</code>: {{-error}}", "question.error creating question": "Error while creating question {{number}}: {{-message}}", "question.score feedback.not marked": "Not marked", "question.score feedback.partially answered": "Partially answered", "question.score feedback.score total": "{{marksString}}", "question.score feedback.score actual": "Score: {{scoreString}}", "question.score feedback.score total actual": "Score: {{score,niceNumber}}/{{marks,niceNumber}}", "variable.error in variable definition": "Error in definition of variable <code>{{name}}</code>", "left brace": "{", "right brace": "}", "extension.not found": "Couldn't load the extension <code>{{name}}</code>.", "control.toggle navigation menu": "Toggle the navigation menu", "part.input title": "Answer for part {{name}}", "part.correct answer title": "Expected answer for part {{name}}", "part.jme.must-match.failed": "Your answer is not in the expected form.", "question.score feedback.none": "", "control.submit part.confirm remove next parts": "<p>One or more subsequent parts depend on your answer to this part. Submitting this part again will invalidate those parts, and remove them from the question. This cannot be undone.</p>\n<p>Would you like to submit this part again?</p>", "control.back to menu": "Go back to the menu", "display.error making html": "Error making HTML in {{contextDescription}}: {{-message}}", "jme.subvars.error compiling": "{{-message}} in <code>{{expression}}</code>", "jme.variables.empty name": "A question variable has not been given a name.", "jme.calculus.unknown derivative": "Don't know how to differentiate <code>{{tree}}</code>", "math.order complex numbers": "Can't order complex numbers", "menu.choose a question": "Choose a question.", "part.choose next part.answered": "What do you want to do next?", "part.choose next part.unanswered": "Or, you could:", "part.choose next part.will be locked": "(This part will be locked)", "part.reached dead end": "There's nothing more to do from here.", "part.next part.penalty amount": "(lose {{count}} $t(mark))", "part.marking.counts towards objective": "This part counts towards the objective <strong>\u201c{{objective}}\u201d</strong>.", "part.numberentry.answer not integer or decimal or fraction": "Your answer is invalid. You must enter an integer, a decimal or a fraction.", "question": "Question", "question.progress": "Question progress", "question.score feedback.unattempted": "Not attempted", "question.score feedback.attempted": "Attempted", "question.score feedback.score actual.plain": "{{scoreString}}", "question.score feedback.score total actual.plain": "{{score,niceNumber}}/{{marks,niceNumber}}", "question.objectives": "Objectives", "question.penalties": "Penalties", "question.back to previous part": "Go back to the previous part", "end.print": "Print your exam transcript", "math.shuffle_together.lists not all the same length": "Not all lists are the same length.", "jme.parse signature.invalid signature string": "Invalid function signature string: {{str}}", "part.custom.expected answer has wrong type": "The expected answer for this part has the wrong type. It should be <code>{{shouldbe}}</code>.", "part.custom.input option has wrong type": "The answer input setting <code>{{option}}</code> has the wrong type. It should be <code>{{shouldbe}}</code>.", "matrix input.size control legend": "Size", "matrix input.rows": "Rows", "matrix input.columns": "Columns", "part.jme.error checking numerically": "There was an error numerically checking your answer: {{-message}}", "part.gapfill.cyclic adaptive marking": "There is a cycle in the adaptive marking for this part: <strong>{{name1}}</strong> relies on <strong>{{name2}}</strong>, which eventually relies on <strong>{{name1}}</strong>.", "modal.style.background colour": "Background colour", "modal.style.text colour": "Text colour", "modal.style.text size": "Text size", "modal.style.explanation": "Use these controls to change the appearance of the exam.", "modal.style.reset to defaults": "Reset to defaults", "modal.style.text size preview": "Most text will be this big.", "control.style options": "Display options", "part.marking.partially correct": "Your answer is partially correct.", "part.marking.error in adaptive marking": "There was an error in the adaptive marking for this part. Please report this. {{-message}}", "page.skip to content": "Skip to content", "result.learning objective": "Learning objective", "jme.interpreted as": "interpreted as", "jme.script.note.compilation error": "Error compiling note <code>{{name}}</code>: {{-message}}", "jme.script.note.error evaluating note": "Error evaluating note <code>{{name}}</code> - {{-message}}", "jme.script.note.invalid definition": "Invalid note definition: <code>{{source}}</code>. {{-hint}}", "jme.script.note.invalid definition.missing colon": "You might be missing a colon after the name and description", "jme.script.note.invalid definition.description missing closing bracket": "You might be missing a closing bracket", "jme.script.note.empty expression": "The note <code>{{name}}</code> is empty.", "jme.script.error parsing notes": "Error parsing marking script: {{- message}}", "matrix input.cell label": "Row {{row}}, column {{column}}", "control.move to next question": "Move to the next question", "diagnostic.use retry": "Use one retry and try this topic again.", "diagnostic.move to next topic": "Move on to the next topic.", "diagnostic.next step question": "What would you like to do next?", "diagnostic.now assessing topic": "Now assessing {{current_topic}}", "diagnostic.one retry left": "You have 1 retry left", "diagnostic.retries left": "You have {{retries}} retries left.", "diagnostic.percentage completed": "You've completed <strong>{{percentage}}%</strong> of the test.", "diagnostic.test is over": "The test is over.", "diagnostic.passed all lo": "You have passed all learning objectives.", "diagnostic.more work on lo": "You need to do some more work on the following learning objectives: {{los}}.", "diagnostic.move to next question in topic": "Move on to the next question in topic.", "diagnostic.complete": "Complete!", "diagnostic.studying topic": "Studying {{topic}}", "display.answer widget.unknown widget type": "The answer widget type <code>{{name}}</code> is not recognised.", "jme.shunt.expected argument before comma": "Expected to see something between the opening bracket and the comma", "part.waiting for pre submit": "Your answer is being marked. Please wait.", "diagnostic.end test": "End the test.", "page.no stylesheet loaded": "The page's stylesheet file has not loaded.", "modal.confirm": "Confirm", "modal.alert": "Alert", "suspend.resumed header": "Attempt resumed", "jme.vector.value not an array of numbers": "Tried to construct a vector using a value that is not an array of numbers.", "jme.matrix.value not the right type": "Tried to construct a vector using a value of the wrong type.", "jme.subvars.html inserted twice": "An HTML value has been embedded twice. Consider defining a function to generate a new value each time it is used.", "jme.variables.invalid function language": "The language <code>{{language}}</code> is not valid.", "jme.variables.duplicate definition": "There is more than one definition of the variable <code>{{name}}</code>.", "math.random_integer_partition.invalid k": "The size of the partition must be between 1 and {{n}}.", "part.marking.parameter already in scope": "There is a variable named <code>{{name}}</code>, which is also the name of a marking parameter. Please rename the variable.", "part.marking.adaptive variable replacement refers to self": "This part refers to itself in a variable replacement for adaptive marking.", "part.marking.adaptive variable replacement refers to nothing": "This part contains an invalid variable replacement for adaptive marking.", "part.numberentry.display answer wrong type": "The display answer for this part is a value of type <code>{{got_type}}</code>, but should be a <code>{{want_type}}</code>.", "part.matrix.invalid type in prefilled": "There is an invalid value of type <code>{{n}}</code> in the array of pre-filled cells.", "diagnostic.make a choice": "Make a choice", "matrixmath.not square": "This operation only works on a square matrix.", "matrixmath.not invertible": "This operation only works on an invertible matrix.", "matrixmath.gauss-jordan elimination.not enough columns": "There must be at least as many columns as rows.", "question.required extension not available": "This question requires the extension <code>{{-extension}}</code> but it is not available.", "util.formatNumberNotation.unrecognised syntax": "The number formatting syntax <code>{{syntax}}</code> is not recognised.", "worksheet.number of exams to generate": "Number of sheets to generate", "worksheet.starting with id": "Starting with ID", "worksheet.show exam id": "Show sheet ID?", "worksheet.page break between questions": "Page breaks between questions?", "worksheet.page margins": "Page margins (mm)", "worksheet.text size": "Text size (pt)", "worksheet.generate": "Generate", "worksheet.generating exams": "Generating sheets", "worksheet.sheet id": "Sheet ID:", "worksheet.print single": "Print this sheet", "worksheet.print several": "Print these sheets", "worksheet.answer sheets": "Answer sheets", "worksheet.question sheets": "Question sheets", "worksheet.reconfigure": "Generate different sheets", "worksheet.show sheet": "Preview the sheet with ID:", "accessibility statement": "Accessibility statement and guide to adapting Numbas to your needs.", "exam.enter your name": "Your name:", "exam.attempt download security warning": "This exam is configured to allow you to download your data, but it is not running in a secure browser context. You will not be able to download your exam data. Contact your lecturer or teacher for help.", "result.download exam object": "Download your exam data", "control.return to question": "Return to the question", "control.show introduction": "Introduction", "analysis.header": "Analyse attempt data", "analysis.help.upload files": "Upload attempt data files that your students have given you.", "analysis.help.file input label": "Choose student attempt data files, or drag files onto this window.", "analysis.table.total": "Exam totals", "analysis.table.question": "Exam and question totals", "analysis.table.all": "All details", "analysis.student name.anonymous": "No name given", "analysis.expected": "Expected results", "analysis.start time": "Start time", "analysis.maximum": "Maximum Marks", "analysis.file": "File", "analysis.download": "Download", "analysis.delete": "Delete", "analysis.view results": "View results", "analysis.upload files": "Upload files", "analysis.upload more": "Upload more files", "analysis.attempt data": "Attempt data", "analysis.select format": "Select data to show", "analysis.download this table": "Download this table", "analysis.student": "Student Results", "analysis.question key": "Question Key", "analysis.question name": "Question Name", "analysis.group": "Group", "analysis.question": "Question", "analysis.part": "Part", "analysis.gap": "Gap", "analysis.record type": "Record Type", "analysis.score": "Score", "analysis.marks available": "Marks Available", "analysis.percentage": "Percentage", "analysis.answer": "Answer", "analysis.student name": "Student Name", "analysis.summary.no files": "You have not uploaded any files yet.", "analysis.summary.no decrypted files": "You have not uploaded any valid files yet.", "analysis.summary.one file": "One attempt.", "analysis.summary.several files": "{{num_files,niceNumber}} attempts.", "analysis.not secure context": "This page must be opened in a secure browser context. A secure context is either a page served over HTTPS, or a file loaded from your device.", "jme.shunt.pipe right hand takes no arguments": "The expression on the right-hand side of the pipe operator must be a function application.", "question.explore.no parts defined": "There are no parts defined in this question.", "answer": "answer", "worksheet.answersheet show question content": "Show question content in answer sheets?", "modal.confirm end exam": "Write <code>{{endConfirmation}}</code> in the box to confirm:", "modal.end exam button": "End exam", "lightbox.zoom in on image": "Zoom in on this image", "exam.progress": "Progress", "exam.questions answered": "{{numAnsweredQuestions}} of {{numQuestions}} questions answered.", "result.question marks available": "Marks Available", "result.question answered": "Answered?", "control.confirm end.correct": "You may now end the exam.", "control.confirm end.incorrect": "This is not the expected text.", "control.confirm end.password": "end", "jme.typecheck.for in name wrong type": "The name in a <code>for</code> statement must be a name or list of names, not {{type}}.", "jme.makeFast.no fast definition of function": "The function <code>{{name}}</code> here isn't defined in a way that can be made fast.", "part.show feedback": "Show feedback", "part.hide feedback": "Hide feedback", "part.feedback title": "Feedback for {{name}}.", "part.jme.must-match.warning": "Your answer is not in the expected form: {{-message}}", "part.numberentry.write your answer as a fraction": "Write your answer as a fraction.", "question.nav.label": "Question controls", "question.answer saved": "Answer saved", "question.all answers saved": "All answers saved", "analysis.back to results": "Back to results", "analysis.review": "Review", "analysis.review this": "Review this attempt", "analysis.reviewing attempt": "Reviewing attempt by {{student_name}}.", "part.there is new feedback": "The feedback has changed.", "modal.style.colour scheme": "Colour scheme", "modal.style.automatic colour scheme": "Automatic", "modal.style.light colour scheme": "Light", "modal.style.dark colour scheme": "Dark", "modal.style.custom colour scheme": "Custom", "modal.style.main font": "Main font", "modal.style.font.sans serif": "Sans serif", "modal.style.font.serif": "Serif", "modal.style.font.monospace": "Monospace", "modal.style.spacing": "Spacing", "modal.style.font weight": "Font weight", "modal.style.forced colours warning": "Your browser has overridden the colours used in this page because of a setting such as high contrast mode. Changes to the colour scheme settings here might not have any effect.", "modal.style.text preview": "Most text will look like this.", "modal.style.more options": "More options", "modal.style.main colour": "Main (brand) colour", "modal.style.primary colour": "Primary button colour", "modal.style.link colour": "Link colour", "modal.style.success colour": "Success/correct colour", "modal.style.info colour": "Info colour", "modal.style.warning colour": "Warning colour", "modal.style.danger colour": "Danger/incorrect colour", "modal.style.muted colour": "Muted text colour", "modal.style.highlight colour": "Highlight colour", "exam.error loading exam definition": "There was an error while loading the exam definition: {{text}}", "exam.no exam definition": "No exam definition was given.", "jme.typecheck.wrong arguments for anonymous function": "Wrong number of arguments for this anonymous function.", "worksheet.top": "Top", "worksheet.left": "Left", "worksheet.bottom": "Bottom", "worksheet.right": "Right"}}, "zh-cn": {"translation": {"page.loading": "\u7f51\u9875\u8f7d\u5165\u4e2d...", "page.saving": "<p>\u4fdd\u5b58</p>\n<p>\u8fd9\u53ef\u80fd\u9700\u8981\u51e0\u79d2\u949f</p>", "mathjax.math processing error": "\n\"{{-message}}\" when texifying <code>{{expression}}</code>", "die.numbas failed": "\u56e0\u67d0\u4e9b\u539f\u56e0\uff0cNumbas\u65e0\u6cd5\u542f\u52a8\u3002", "die.sorry": "\u5f88\u62b1\u6b49\uff0c\u7531\u4e8e\u67d0\u4e9b\u9519\u8bef\u5bfc\u81f4Numbas\u65e0\u6cd5\u7ee7\u7eed\u8fd0\u4f5c\u3002\u8bf7\u770b\u4e0b\u9762\u9519\u8bef\u8bf4\u660e\u3002", "die.error": "\u9519\u8bef", "modal.ok": "\u597d\u7684", "modal.cancel": "\u53d6\u6d88", "exam.exam name": "\u8003\u8bd5\u540d\u79f0:", "exam.random seed": "\u4f1a\u8bddID:", "exam.student name": "\u5b66\u751f\u540d\u5b57:", "exam.number of questions": "\u95ee\u9898\u6570\u91cf:", "exam.marks available": "\u53ef\u5f97\u5230\u5206\u6570:", "exam.pass percentage": "\u53ca\u683c\u7387:", "exam.time allowed": "\u5141\u8bb8\u65f6\u95f4:", "exam.passed": "\u53ca\u683c", "exam.failed": "\u4e0d\u53ca\u683c", "exam.review header": "\u56de\u987e:", "frontpage.start": "\u5f00\u59cb", "suspend.paused header": "\u6682\u505c", "suspend.exam suspended": "\u8003\u8bd5\u5df2\u88ab\u6682\u505c\u3002\u8bf7\u6309<em>\u6062\u590d<em>\u6062\u590d\u8003\u8bd5\u3002", "suspend.you can resume": "\u4e0b\u6b21\u5f00\u59cb\u6b64\u6d3b\u52a8\u65f6\uff0c\u60a8\u5c06\u53ef\u4ee5\u7ee7\u7eed\u6b64\u4f1a\u8bdd\u3002", "suspend.resume": "\u6062\u590d", "result.exit": "\u9000\u51fa\u8003\u8bd5", "result.print": "\u6253\u5370\u6b64\u7ed3\u679c\u603b\u7ed3", "result.exam summary": "\u8003\u8bd5\u603b\u7ed3", "result.performance summary": "\u8868\u73b0\u603b\u7ed3", "result.exam start": "\u8003\u8bd5\u5f00\u59cb:", "result.exam stop": "\u8003\u8bd5\u7ed3\u675f:", "result.time spent": "\u6240\u82b1\u8d39\u65f6\u95f4:", "result.questions attempted": "\u5df2\u5b8c\u6210\u7684\u9898\u76ee:", "result.score": "\u5f97\u5206:", "result.result": "\u6210\u7ee9:", "result.question number": "\u95ee\u9898", "result.question score": "\u5f97\u5206", "result.question review title": "\u56de\u987e\u8be5\u95ee\u9898", "result.click a question to review": "\u60f3\u67e5\u770b\u6539\u7b54\u6848\u7684\u65b9\u5f0f\u4ee5\u53ca\u5b8c\u6574\u7b54\u6848\uff0c\u8bf7\u6309\u95ee\u9898\u53f7\u7801\u3002", "end.exam has finished": "\u8003\u8bd5\u7ed3\u675f\u3002 \u60a8\u73b0\u5728\u53ef\u4ee5\u5173\u95ed\u6b64\u7a97\u53e3\u3002", "control.confirm leave": "\u60a8\u5c1a\u672a\u5b8c\u6210\u8003\u8bd5\u3002", "control.not all questions answered": "\u60a8\u5c1a\u672a\u5b8c\u6210\u8003\u8bd5\u4e2d\u7684\u6240\u6709\u95ee\u9898\u3002", "control.not all questions submitted": "\u60a8\u5df2\u5bf9\u4e00\u4e2a\u6216\u591a\u4e2a\u7b54\u6848\u8fdb\u884c\u4e86\u66f4\u6539\uff0c\u4f46\u672a\u63d0\u4ea4\u3002 \u8bf7\u68c0\u67e5\u6bcf\u4e2a\u95ee\u9898\u662f\u5426\u5df2\u63d0\u4ea4\u3002", "control.confirm end": "\u60a8\u786e\u5b9a\u8981\u7ed3\u675f\u8003\u8bd5\u5417\uff1f \u8003\u8bd5\u7ed3\u675f\u540e\uff0c\u60a8\u5c06\u65e0\u6cd5\u66f4\u6539\u4efb\u4f55\u7b54\u6848\u3002", "control.confirm regen": "\u60a8\u60f3\u91cd\u65b0\u968f\u673a\u5316\u8fd9\u4e2a\u95ee\u9898\u5417\uff1f \u5982\u679c\u6309\u201c\u786e\u5b9a\u201d\uff0c\u5219\u5f53\u524d\u95ee\u9898\u7684\u6240\u6709\u7b54\u6848\u548c\u5206\u6570\u90fd\u5c06\u4e22\u5931\u3002", "control.confirm reveal": "\u60a8\u60f3\u663e\u793a\u8fd9\u4e2a\u95ee\u9898\u7684\u7b54\u6848\u5417\uff1f \u5230\u76ee\u524d\u4e3a\u6b62\uff0c\u60a8\u6536\u5230\u7684\u6240\u6709\u5206\u6570\u90fd\u5c06\u88ab\u9501\u5b9a\uff0c\u4ee5\u540e\u60a8\u5c06\u65e0\u6cd5\u56de\u7b54\u8be5\u95ee\u9898\u3002", "control.proceed anyway": "\u4ecd\u8981\u7ee7\u7eed\uff1f", "control.regen": "\u5c1d\u8bd5\u7c7b\u4f3c\u95ee\u9898", "control.submit answer": "\u63d0\u4ea4\u7b54\u6848", "control.submit all parts": "\u63d0\u4ea4\u6240\u6709\u90e8\u5206", "control.submit again": "\u518d\u6b21\u63d0\u4ea4", "control.submit": "\u63d0\u4ea4", "control.previous": "\u4e0a\u4e00\u4e2a", "control.next": "\u4e0b\u4e00\u4e2a", "control.advice": "\u5fe0\u544a", "control.reveal": "\u663e\u793a\u7b54\u6848", "control.total": "\u603b\u5206", "control.pause": "\u6682\u505c", "control.end exam": "\u7ed3\u675f\u8003\u8bd5", "control.back to results": "\u56de\u5230\u6210\u7ee9", "display.part.jme.error making maths": "\u8fdb\u884c\u6570\u5b66\u663e\u793a\u65f6\u51fa\u9519", "exam.xml.bad root": "Root element of exam XML should be 'exam'", "exam.changeQuestion.no questions": "\u8fd9\u9879\u8003\u8bd5\u6ca1\u6709\u95ee\u9898\uff01\u68c0\u67e5.exam\u6587\u4ef6\u662f\u5426\u6709\u9519\u8bef\u3002", "feedback.you were awarded": "\u60a8\u88ab\u6388\u4e88<strong>{{count,niceNumber}}</strong> $t(mark).", "feedback.taken away": "<strong>{{count,niceNumber}}</strong> $t(mark) $t(was) taken away.", "jme.tokenise.invalid": "\u65e0\u6548\u8868\u8fbe:  <code>{{expression}}</code>", "jme.shunt.not enough arguments": "\u64cd\u4f5c\u53c2\u6570\u4e0d\u8db3 <code>{{op}}</code>", "jme.shunt.no left bracket in function": "\u51fd\u6570\u5e94\u7528\u7a0b\u5e8f\u6216\u5143\u7ec4\u4e2d\u6ca1\u6709\u5339\u914d\u7684\u5de6\u62ec\u53f7", "jme.shunt.no left square bracket": "\u6ca1\u6709\u5339\u914d\u7684\u5de6\u65b9\u62ec\u53f7", "jme.shunt.no left bracket": "\u6ca1\u6709\u5339\u914d\u7684\u5de6\u62ec\u53f7", "jme.shunt.no right bracket": "\u6ca1\u6709\u5339\u914d\u7684\u53f3\u62ec\u53f7", "jme.shunt.no right square bracket": "\u6ca1\u6709\u5339\u914d\u7684\u53f3\u65b9\u62ec\u53f7\u5728\u7ed3\u675f\u5217\u8868", "jme.shunt.missing operator": "\u65e0\u6cd5\u8bc4\u4f30\u8868\u8fbe\u5f0f-\u7f3a\u5c11\u8fd0\u7b97\u7b26\u3002", "jme.typecheck.function maybe implicit multiplication": "\u672a\u5b9a\u4e49\u51fd\u6570 <code>{{name}}</code>\u3002 \u60a8\u662f\u8bf4<code> {{first}}*{{possibleOp}}(...)</code>\u5417\uff1f", "jme.typecheck.function not defined": "\u672a\u5b9a\u4e49\u51fd\u6570<code>{{op}}</code>\u3002<code>{{op}}</code> \u662f\u53d8\u91cf\u5417\uff1f\u60a8\u7684\u610f\u601d\u662f<code>{{suggestion}}*(...)</code>\u5417\uff1f", "jme.typecheck.op not defined": "\u672a\u5b9a\u4e49\u8fd0\u7b97 '{{op}}'\u3002", "jme.typecheck.no right type definition": "\u627e\u4e0d\u5230\u6b63\u786e\u7c7b\u578b\u7684'{{op}}'\u7684\u5b9a\u4e49\u3002", "jme.typecheck.no right type unbound name": "\u672a\u5b9a\u4e49\u53d8\u91cf<code>{{name}}</code>\u3002", "jme.typecheck.map not on enumerable": "<code>map</code>\u8fd0\u7b97\u5fc5\u987b\u5728\u5217\u8868\u6216\u8303\u56f4\u5185\u5de5\u4f5c\uff0c\u800c\u4e0d\u662f{{type}}", "jme.evaluate.undefined variable": "\u672a\u5b9a\u4e49\u53d8\u91cf{{name}}", "jme.thtml.not html": "\u5c06\u975eHTML\u503c\u4f20\u9012\u7ed9THTML\u6784\u9020\u51fd\u6570\u3002", "jme.func.switch.no default case": "Switch\u8bed\u53e5\u6ca1\u6709\u9ed8\u8ba4\u60c5\u51b5", "jme.func.listval.invalid index": "\u5927\u5c0f\u4e3a{{size}}\u7684\u5217\u8868\u4e0a\u7684\u5217\u8868\u7d22\u5f15{{index}}\u65e0\u6548", "jme.func.listval.not a list": "\u5bf9\u8c61\u4e0d\u53ef\u4e0b\u6807", "jme.func.matrix.invalid row type": "\u65e0\u6cd5\u4ece{{type}}\u7c7b\u578b\u7684\u884c\u6784\u9020\u77e9\u9635", "jme.func.except.continuous range": "\u4e0d\u80fd\u5728\u8fde\u7eed\u8303\u56f4\u5185\u4f7f\u7528'except'\u8fd0\u7b97\u7b26\u3002", "jme.matrix.reports bad size": "\u77e9\u9635\u9519\u8bef\u5730\u62a5\u544a\u5176\u5927\u5c0f - \u80af\u5b9a\u662f\u6784\u9020\u51fd\u6570\u4e2d\u51fa\u73b0\u9519\u8bef", "jme.texsubvars.no right bracket": "{{op}}\u53c2\u6570\u4e2d\u6ca1\u6709\u5339\u914d\u7684<code>]</code>\u3002", "jme.texsubvars.missing parameter": "{{op}}\u4e2d\u7f3a\u5c11\u53c2\u6570: {{parameter}}", "jme.texsubvars.no right brace": "{{op}}\u4e2d\u6ca1\u6709\u5339\u914d\u7684 <code>}</code>", "jme.user javascript.error": "\u7528\u6237\u5b9a\u4e49\u7684javascript\u51fd\u6570<code>{{name}}</code>\u4e2d\u51fa\u73b0\u9519\u8bef: {{-message}}", "jme.variables.error making function": "Error making function <code>{{name}}</code>: {{-message}}", "jme.variables.syntax error in function definition": "\u51fd\u6570<code> {{name}} </ code>\u51fa\u9519: {{-message}}", "jme.variables.variable not defined": "\u672a\u5b9a\u4e49\u53d8\u91cf<code>{{name}}</code>\u3002", "jme.variables.empty definition": "\u53d8\u91cf<code>{{name}}</code>\u7684\u5b9a\u4e49\u4e3a\u7a7a\u3002", "jme.variables.circular reference": "<code>{{name}}</code>\u5b9a\u4e49\u4e2d\u7684\u5faa\u73af\u53d8\u91cf\u5f15\u7528", "jme.variables.error computing dependency": "\u8ba1\u7b97\u5f15\u7528\u7684\u53d8\u91cf<<code>{{name}}</code>\u65f6\u51fa\u9519", "jme.variables.error evaluating variable": "\u8bc4\u4f30\u53d8\u91cf{{name}}\u65f6\u51fa\u9519: {{-message}}", "jme.variables.question took too many runs to generate variables": "\u6ca1\u6709\u53ca\u65f6\u751f\u6210\u6709\u6548\u7684\u95ee\u9898\u53d8\u91cf\u96c6\u3002", "jme.display.unknown token type": "Can't texify token type {{type}}", "jme.display.collectRuleset.no sets": "\u6ca1\u6709\u8bbe\u7f6e\u7ed9collectRuleset\u7684\u96c6\u5408\uff01", "jme.display.collectRuleset.set not defined": "Ruleset {{name}}\u5c1a\u672a\u5b9a\u4e49", "jme.display.simplifyTree.no scope given": "Numbas.jme.display.simplifyTree must be given a Scope", "math.precround.complex": "\u65e0\u6cd5\u820d\u5165\u5230\u590d\u6570\u7684\u5c0f\u6570\u4f4d\u6570", "math.siground.complex": "\u65e0\u6cd5\u820d\u5165\u5230\u590d\u6570\u7684\u6709\u6548\u6570\u5b57", "math.combinations.complex": "\u65e0\u6cd5\u8ba1\u7b97\u590d\u6570\u7684\u7ec4\u5408", "math.permutations.complex": "\u65e0\u6cd5\u8ba1\u7b97\u590d\u6570\u7684\u6392\u5217", "math.gcf.complex": "\u65e0\u6cd5\u8ba1\u7b97\u590d\u6570\u7684\u6700\u5927\u516c\u7ea6\u6570", "math.lcm.complex": "\u65e0\u6cd5\u8ba1\u7b97\u590d\u6570\u7684\u6700\u5c0f\u516c\u500d\u6570", "math.lt.order complex numbers": "\u65e0\u6cd5\u6392\u5e8f\u590d\u6570", "math.choose.empty selection": "\u7a7a\u9009\u62e9\u8d4b\u4e88\u968f\u673a\u51fd\u6570", "matrixmath.abs.non-square": "\u65e0\u6cd5\u8ba1\u7b97\u975e\u6b63\u65b9\u5f62\u77e9\u9635\u7684\u884c\u5217\u5f0f\u3002", "matrixmath.abs.too big": "\u62b1\u6b49\uff0c\u8fd8\u4e0d\u80fd\u8ba1\u7b97\u5927\u4e8e3x3\u7684\u77e9\u9635\u7684\u884c\u5217\u5f0f\u3002", "matrixmath.mul.different sizes": "\u4e0d\u80fd\u5c06\u4e0d\u540c\u5927\u5c0f\u7684\u77e9\u9635\u76f8\u4e58\u3002", "vectormath.cross.not 3d": "\u53ea\u80fd\u53d63\u7ef4\u5411\u91cf\u7684\u53c9\u79ef\u3002", "vectormath.dot.matrix too big": "\u65e0\u6cd5\u8ba1\u7b97\u4e0d\u662f$1 \\times N$\u6216$N \\times 1$\u7684\u77e9\u9635\u7684\u70b9\u79ef\u3002", "vectormath.cross.matrix too big": "\u65e0\u6cd5\u8ba1\u7b97\u4e0d\u662f$1 \\times N$\u6216$N \\times 1$\u7684\u77e9\u9635\u7684\u53c9\u79ef\u3002", "part.with steps answer prompt": "\u56de\u7b54:", "part.script.error": "\u90e8\u5206{{path}}\u81ea\u5b9a\u4e49\u811a\u672c{{script}}\u4e2d\u7684\u9519\u8bef: {{-message}}", "part.marking.steps no matter": "\u56e0\u4e3a\u60a8\u5728\u8fd9\u90e8\u5206\u95ee\u9898\u4e0a\u83b7\u5f97\u4e86\u6ee1\u5206\uff0c\u6240\u4ee5\u4e0d\u8ba1\u7b97\u6b65\u9aa4\u7684\u7b54\u6848\u3002", "part.marking.revealed steps no penalty": "\u60a8\u663e\u793a\u4e86\u6b65\u9aa4\u3002", "part.marking.used variable replacements": "\u8be5\u90e8\u5206\u5df2\u4f7f\u7528\u60a8\u5bf9\u5148\u524d\u90e8\u5206\u7684\u7b54\u6848\u8fdb\u884c\u4e86\u8bc4\u5206\u3002", "part.marking.variable replacement part not answered": "\u60a8\u5fc5\u987b\u5148\u56de\u7b54{{part}}\u3002", "part.marking.resubmit because of variable replacement": "\u8be5\u90e8\u5206\u7684\u8bc4\u5206\u53d6\u51b3\u4e8e\u60a8\u5bf9\u5176\u4ed6\u90e8\u5206\u6240\u505a\u7684\u7b54\u6848\u66f4\u6539\u3002 \u518d\u6b21\u63d0\u4ea4\u6b64\u90e8\u5206\u4ee5\u66f4\u65b0\u60a8\u7684\u5206\u6570\u3002", "part.marking.not submitted": "\u6ca1\u6709\u7b54\u6848\u88ab\u63d0\u4ea4\u3002", "part.marking.did not answer": "\u60a8\u6ca1\u6709\u56de\u7b54\u8fd9\u4e2a\u95ee\u9898\u3002", "part.marking.nothing entered": "\u60a8\u6ca1\u6709\u8f93\u5165\u7b54\u6848\u3002", "part.marking.incorrect": "\u60a8\u7684\u7b54\u6848\u4e0d\u6b63\u786e\u3002", "part.marking.correct": "\u60a8\u7684\u56de\u7b54\u662f\u6b63\u786e\u7684\u3002", "part.marking.uncaught error": "\u8bc4\u5206\u65f6\u51fa\u9519: {{-message}}", "part.marking.no result": "\u65e0\u6cd5\u5bf9\u6b64\u90e8\u5206\u8bc4\u5206\u3002", "part.correct answer": "\u9884\u671f\u7b54\u6848:", "part.missing type attribute": "{{part}}: \u7f3a\u5c11\u96f6\u4ef6\u7c7b\u578b\u5c5e\u6027", "part.unknown type": "{{part}}: \u65e0\u6cd5\u8bc6\u522b\u7684\u96f6\u4ef6\u7c7b\u578b {{type}}", "part.setting not present": "\u672a\u8bbe\u7f6e\u5c5e\u6027 '{{property}}'", "part.jme.answer missing": "\u7f3a\u5c11\u6b63\u786e\u7b54\u6848", "part.jme.answer too long": "\u60a8\u7684\u7b54\u6848\u592a\u957f\u3002", "part.jme.answer too short": "\u60a8\u7684\u7b54\u6848\u592a\u77ed\u4e86\u3002", "part.jme.answer invalid": "\u60a8\u7684\u7b54\u6848\u4e0d\u662f\u6709\u6548\u7684\u6570\u5b66\u8868\u8fbe\u5f0f\u3002<br/>{{-message}}.\u3002", "part.jme.marking.correct": "\u60a8\u7684\u7b54\u6848\u5728\u6570\u5b57\u4e0a\u662f\u6b63\u786e\u7684\u3002", "part.jme.must-have bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.must-have one": "\u60a8\u7684\u7b54\u6848\u5fc5\u987b\u5305\u542b: {{strings}}", "part.jme.must-have several": "\u60a8\u7684\u7b54\u6848\u5fc5\u987b\u5305\u542b\u4ee5\u4e0b\u6240\u6709\u5185\u5bb9: {{strings}}", "part.jme.not-allowed bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.not-allowed one": "\u60a8\u7684\u7b54\u6848\u4e0d\u5f97\u5305\u542b: {{strings}}", "part.jme.not-allowed several": "\u60a8\u7684\u7b54\u6848\u4e0d\u5f97\u5305\u542b\u4ee5\u4e0b\u4efb\u4f55\u5185\u5bb9: {{strings}}", "part.jme.unexpected variable name": "\u60a8\u7684\u7b54\u6848\u88ab\u89e3\u91ca\u4e3a\u4f7f\u7528\u4e86\u610f\u5916\u7684\u53d8\u91cf\u540d\u79f0<code>{{name}}</code>\u3002", "part.jme.unexpected variable name suggestion": "\u60a8\u7684\u7b54\u6848\u88ab\u89e3\u91ca\u4e3a\u4f7f\u7528\u4e86\u610f\u5916\u7684\u53d8\u91cf\u540d\u79f0 <code>{{name}}</code>\u3002 \u60a8\u662f\u8bf4<code>{{suggestion}}</code>\u5417\uff1f", "part.patternmatch.display answer missing": "\u663e\u793a\u7b54\u6848\u4e22\u5931", "part.patternmatch.correct except case": "\u9664\u60c5\u51b5\u5916\uff0c\u60a8\u7684\u7b54\u6848\u662f\u6b63\u786e\u7684\u3002", "part.numberentry.correct except decimal": "\u60a8\u7684\u7b54\u6848\u5728\u5141\u8bb8\u7684\u8303\u56f4\u5185\uff0c\u4f46\u4e0d\u5141\u8bb8\u4f7f\u7528\u5341\u8fdb\u5236\u6570\u5b57\u3002", "part.numberentry.correct except fraction": "\u60a8\u7684\u7b54\u6848\u5728\u5141\u8bb8\u7684\u8303\u56f4\u5185\uff0c\u4f46\u5206\u6570\u662f\u4e0d\u88ab\u5141\u8bb8\u7684\u3002", "part.numberentry.answer invalid": "\u60a8\u8f93\u5165\u7684\u53f7\u7801\u65e0\u6548\u3002", "part.numberentry.answer not integer": "\u60a8\u7684\u7b54\u6848\u65e0\u6548\u3002 \u60a8\u5fc5\u987b\u8f93\u5165\u6574\u6570\uff0c\u800c\u4e0d\u662f\u5c0f\u6570", "part.numberentry.answer not integer or decimal": "\u60a8\u7684\u7b54\u6848\u65e0\u6548\u3002 \u60a8\u5fc5\u987b\u8f93\u5165\u6574\u6570\u6216\u5c0f\u6570\u3002", "part.numberentry.zero sig fig": "\u8bbe\u7f6e\u8be5\u90e8\u5206\u7684\u76ee\u7684\u662f\u5c06\u5b66\u751f\u7684\u7b54\u6848\u56db\u820d\u4e94\u5165\u5230\u96f6\u6709\u6548\u6570\u5b57\uff0c\u8fd9\u6ca1\u6709\u4efb\u4f55\u610f\u4e49\u3002", "part.mcq.options def not a list": "\u5b9a\u4e49{{properties}}\u7684\u8868\u8fbe\u5f0f\u4e0d\u662f\u5217\u8868\u3002", "part.mcq.marking matrix string empty": "\u81ea\u5b9a\u4e49\u6807\u8bb0\u77e9\u9635\u8868\u8fbe\u5f0f\u4e3a\u7a7a\u3002", "part.mcq.choices missing": "\u7f3a\u5931\u9009\u62e9\u7684\u5b9a\u4e49", "part.mcq.matrix not a number": "\u90e8\u5206{{part}}\u6807\u8bb0\u77e9\u9635\u5355\u5143\u683c({{row}},{{column}})\u7684\u8ba1\u7b97\u7ed3\u679c\u4e0d\u662f\u6570\u5b57", "part.mcq.wrong number of choices": "\u60a8\u6240\u9009\u62e9\u7684\u9009\u9879\u6570\u76ee\u662f\u9519\u8bef\u7684", "part.mcq.no choices selected": "\u672a\u9009\u62e9\u4efb\u4f55\u9009\u9879\u3002", "part.mcq.matrix not a list": "\u7531JME\u8868\u8fbe\u5f0f\u5b9a\u4e49\u7684\u6807\u8bb0\u77e9\u9635\u4e0d\u662f\u5217\u8868\uff0c\u4f46\u5e94\u8be5\u662f\u5217\u8868\u3002", "part.mcq.matrix wrong type": "\u6807\u8bb0\u77e9\u9635\u4e2d\u4f7f\u7528\u7684\u65e0\u6548\u7c7b\u578b'{{type}}'\u7684\u5143\u7d20\u3002", "part.mcq.matrix mix of numbers and lists": "\u77e9\u9635\u4e2d\u4f7f\u7528\u7684\u6570\u5b57\u548c\u5217\u8868\u7684\u6df7\u5408\u3002", "part.mcq.matrix wrong size": "\u77e9\u9635\u6709\u5927\u5c0f\u9519\u8bef\u3002", "part.mcq.correct choice": "\u60a8\u9009\u62e9\u4e86\u6b63\u786e\u7684\u7b54\u6848\u3002", "part.matrix.answer invalid": "\u60a8\u7684\u7b54\u6848\u65e0\u6548\u3002", "part.matrix.invalid cell": "\u60a8\u7684\u7b54\u6848\u4e2d\u7684\u4e00\u4e2a\u6216\u591a\u4e2a\u5355\u5143\u683c\u4e3a\u7a7a\u6216\u65e0\u6548\u3002", "part.matrix.some incorrect": "\u60a8\u7684\u7b54\u6848\u4e2d\u7684\u4e00\u4e2a\u6216\u591a\u4e2a\u5355\u5143\u683c\u4e0d\u6b63\u786e\uff0c\u4f46\u5176\u4f59\u90e8\u5206\u5219\u88ab\u6388\u4e88\u5206\u6570\u3002", "part.matrix.empty": "\u60a8\u5c1a\u672a\u8f93\u5165\u7b54\u6848\u3002", "part.matrix.empty cell": "\u60a8\u7684\u7b54\u6848\u4e2d\u7684\u4e00\u4e2a\u6216\u591a\u4e2a\u5355\u5143\u683c\u4e3a\u7a7a\u3002", "part.matrix.size mismatch": "\u95ee\u9898\u4f5c\u8005\u4e0d\u5141\u8bb8\u5b66\u751f\u51b3\u5b9a\u7b54\u6848\u7684\u7ef4\u5ea6\uff0c\u4f46\u662f\u6b63\u786e\u7684\u7b54\u6848\u662f{{correct_dimensions}}\uff0c\u800c\u7b54\u6848\u8f93\u5165\u662f{{input_dimensions}}", "part.gapfill.feedback header": "<strong>{{name}}</strong>", "part.extension.not implemented": "\u90e8\u4ef6\u5c1a\u672a\u5b9e\u73b0<code>{{name}}</code>\u65b9\u6cd5\u3002", "question.loaded name mismatch": "\u65e0\u6cd5\u6062\u590d\u6b64\u5c1d\u8bd5-\u81ea\u4e0a\u6b21\u4f1a\u8bdd\u4ee5\u6765\uff0c\u7a0b\u5e8f\u5305\u5df2\u66f4\u6539\u3002", "question.error": "\u95ee\u9898{{number}}: {{-message}}", "question.preamble.error": "\u524d\u8a00\u4e2d\u7684\u9519\u8bef: {{-message}}", "question.preamble.syntax error": "\u524d\u8a00\u4e2d\u7684\u8bed\u6cd5\u9519\u8bef", "question.unsupported part type": "\u4e0d\u652f\u6301\u7684\u96f6\u4ef6\u7c7b\u578b", "question.header": "\u95ee\u9898{{number}}", "question.submit part": "\u63d0\u4ea4\u90e8\u5206", "question.show steps": "\u663e\u793a\u6b65\u9aa4", "question.show steps penalty": "\u60a8\u5c06\u4e22\u5931<strong>{{count,niceNumber}}</strong> $t(mark)\u3002", "question.show steps no penalty": "\u60a8\u7684\u5206\u6570\u4e0d\u4f1a\u53d7\u5230\u5f71\u54cd\u3002", "question.show steps already penalised": "\u60a8\u5df2\u7ecf\u663e\u793a\u4e86\u6b65\u9aa4\u3002 \u60a8\u53ef\u4ee5\u518d\u6b21\u65e0\u53d7\u60e9\u7f5a\u7684\u663e\u793a\u5b83\u4eec\u3002", "question.hide steps": "\u9690\u85cf\u6b65\u9aa4", "question.hide steps no penalty": "\u60a8\u7684\u5206\u6570\u4e0d\u4f1a\u53d7\u5230\u5f71\u54cd\u3002", "question.advice": "\u5efa\u8bae", "question.no such part": "\u627e\u4e0d\u5230\u90e8\u5206{{path}}", "question.can not submit": "\u65e0\u6cd5\u63d0\u4ea4\u7b54\u6848-\u68c0\u67e5\u9519\u8bef\u3002", "question.answer submitted": "\u7b54\u6848\u5df2\u63d0\u4ea4", "question.score feedback.show": "\u663e\u793a\u53cd\u9988", "question.score feedback.hide": "\u9690\u85cf\u53cd\u9988", "question.score feedback.answered total actual": "\u5f97\u5206: {{score,niceNumber}}/{{marks,niceNumber}}", "question.score feedback.answered total": "{{marksString}}\u3002\u5df2\u56de\u7b54\u3002", "question.score feedback.answered actual": "\u5f97\u5206:  {{scoreString}}", "question.score feedback.answered": "\u5df2\u56de\u7b54", "question.score feedback.unanswered": "\u672a\u56de\u7b54", "question.score feedback.unanswered total": "{{marksString}}\u3002", "question.score feedback.correct": "\u4f60\u7684\u7b54\u6848\u662f\u6b63\u786e\u7684", "question.score feedback.partial": "\u60a8\u7684\u7b54\u6848\u90e8\u5206\u6b63\u786e", "question.score feedback.wrong": "\u4f60\u7684\u7b54\u6848\u4e0d\u6b63\u786e", "question.selector.unsubmitted changes": "\u672a\u63d0\u4ea4\u7684\u66f4\u6539\u3002", "timing.no accumulator": "\u6ca1\u6709\u8ba1\u65f6\u7d2f\u52a0\u5668{{name}}", "timing.time remaining": "\u5269\u4f59\u65f6\u95f4:", "xml.could not load": "\u65e0\u6cd5\u52a0\u8f7dXML\u6587\u6863: {{-message}}", "xml.property not number": "\u8282\u70b9{{element}}\u4e2d\u7684\u5c5e\u6027{{name}}\u5e94\u8be5\u662f\u6570\u5b57\uff0c\u4f46\u4e0d\u80fd\u662f ({{value}})\u3002", "xml.property not boolean": "\u8282\u70b9{{element}}\u4e2d\u7684\u5c5e\u6027{{name}}\u5e94\u8be5\u662f\u5e03\u5c14\u503c\uff0c\u800c\u4e0d\u662f ({{value}})", "xml.error in variable definition": "\u53d8\u91cf<code>{{name}}</code>\u7684\u5b9a\u4e49\u6709\u9519\u8bef", "scorm.error initialising": "\u521d\u59cb\u5316SCORM\u534f\u8bae\u65f6\u51fa\u9519\uff1a{{-message}}", "scorm.failed save": "<p>\u5c06\u6570\u636e\u4fdd\u5b58\u5230\u670d\u52a1\u5668\u7684\u8bf7\u6c42\u5931\u8d25\u3002 \u6309 <b>OK</b> \u5728\u7ee7\u7eed\u5c1d\u8bd5\u3002</p>\n<p>\u5982\u679c\u60a8\u53cd\u590d\u6536\u5230\u6b64\u6d88\u606f\uff0c\u8bf7\u68c0\u67e5\u60a8\u7684\u4e92\u8054\u7f51\u8fde\u63a5\u6216\u4f7f\u7528\u5176\u4ed6\u8ba1\u7b97\u673a\u3002 \u60a8\u5148\u524d\u63d0\u4ea4\u7684\u7b54\u6848\u5df2\u6210\u529f\u4fdd\u5b58\uff0c\u5982\u679c\u60a8\u5728\u53e6\u4e00\u53f0\u8ba1\u7b97\u673a\u4e0a\u7ee7\u7eed\u8fdb\u884c\u6b64\u4f1a\u8bdd\uff0c\u5219\u5c06\u8fd8\u539f\u8be5\u7b54\u6848\u3002</p>\n<p>\u5982\u679c\u6b64\u6d88\u606f\u6301\u7eed\u51fa\u73b0\uff0c\u5e76\u4e14\u60a8\u65e0\u6cd5\u4fdd\u5b58<em>\u4efb\u4f55</em>\u7b54\u6848\uff0c\u8bf7\u4e0e\u60a8\u7684\u8bb2\u5e08\u6216\u8001\u5e08\u8054\u7cfb\u3002</p>", "scorm.no exam suspend data": "\u6062\u590d\u5931\u8d25: \u6ca1\u6709\u8003\u8bd5\u6682\u505c\u6570\u636e\u3002", "scorm.error loading suspend data": "\u52a0\u8f7d\u6682\u505c\u6570\u636e\u65f6\u51fa\u9519: {{-message}}", "scorm.error loading question": "\u52a0\u8f7d\u95ee\u9898{{number}}\u65f6\u51fa\u9519: {{-message}}", "scorm.no question suspend data": "\u6ca1\u6709\u95ee\u9898\u6682\u505c\u6570\u636e", "scorm.error loading part": "\u52a0\u8f7d\u90e8\u5206{{part}}\u65f6\u51fa\u9519: {{-message}}", "scorm.no part suspend data": "\u6ca1\u6709\u90e8\u5206\u6682\u505c\u6570\u636e", "util.product.non list": "\u5c06\u975e\u5217\u8868\u4f20\u9012\u7ed9<code>Numbas.util.product</code>", "mark": "\u5206\u6570", "was": "\u662f", "part": "\u90e8\u5206", "gap": "\u95f4\u9699", "step": "\u6b65\u9aa4", "jme.substituteTree.undefined variable": "\u672a\u5b9a\u4e49\u7684\u53d8\u91cf: <code>{{name}}</code>", "jme.user javascript.returned undefined": "\u7528\u6237\u5b9a\u4e49\u7684javascript\u51fd\u6570<code>{{name}}</code>\u8fd4\u56de\u4e86<code>undefined</code>\u3002", "part.marking.steps change": "\u60a8\u5bf9\u6b65\u9aa4\u7684\u56de\u7b54\u83b7\u5f97\u4e86<strong>{{count,niceNumber}}</strong> $t(mark)\u3002", "part.marking.revealed steps with penalty": "\u60a8\u900f\u9732\u4e86\u6b65\u9aa4\u3002 \u60a8\u53ef\u4ee5\u4e3a\u6b64\u90e8\u5206\u8bc4\u5206\u7684\u6700\u9ad8\u5206\u662f<strong>{{count,niceNumber}}</strong> $t(mark)\u3002 \u60a8\u7684\u5206\u6570\u5c06\u76f8\u5e94\u964d\u4f4e\u3002", "part.marking.total score": "\u60a8\u4e3a\u6b64\u90e8\u5206\u8bc4\u5206\u4e86<strong>{{count,niceNumber}}</strong> $t(mark)\u3002", "part.numberentry.precision type.dp": "\u5c0f\u6570\u4f4d", "part.numberentry.precision type.dp_plural": "\u5c0f\u6570\u4f4d", "part.numberentry.precision type.sigfig": "\u6709\u6548\u6570\u5b57", "part.numberentry.precision type.sigfig_plural": "\u6709\u6548\u6570\u5b57", "part.numberentry.give your answer to precision": "\u5c06\u7b54\u6848\u56db\u820d\u4e94\u5165\u5230{{count,niceNumber}} {{precisionType}}\u3002", "question.unsubmitted changes": "\u60a8\u5df2\u66f4\u6539\u7b54\u6848\uff0c\u4f46\u672a\u63d0\u4ea4\u3002 \u8bf7\u68c0\u67e5\u60a8\u7684\u7b54\u6848\uff0c\u7136\u540e\u6309 <strong>\u63d0\u4ea4\u7b54\u6848</strong>\u6309\u94ae\u3002", "question.unsubmitted changes_plural": "\u60a8\u5df2\u5bf9\u7b54\u6848\u8fdb\u884c\u4e86\u66f4\u6539\uff0c\u4f46\u672a\u63d0\u4ea4\u3002 \u8bf7\u68c0\u67e5\u6bcf\u4e2a\u90e8\u5206\u7684\u7b54\u6848\uff0c\u7136\u540e\u6309<strong>\u63d0\u4ea4\u6240\u6709\u90e8\u5206</strong>\u6309\u94ae\u3002", "util.equality not defined for type": "\u672a\u4e3a\u7c7b\u578b{{type}}\u5b9a\u4e49\u7684\u76f8\u7b49\u6027", "mark_plural": "\u5206\u6570", "was_plural": "\u662f", "die.script not loaded": "Numbas\u65e0\u6cd5\u542f\u52a8\uff0c\u56e0\u4e3a\u672a\u52a0\u8f7d\u6587\u4ef6<code>{{file}}</code>\u3002\u68c0\u67e5\u5b83\u662f\u5426\u5305\u542b\u5728<code>scripts.js</code>\u4e2d\u3002", "math.combinations.n less than zero": "\u65e0\u6cd5\u8ba1\u7b97\u7ec4\u5408: n\u5c0f\u4e8e\u96f6", "math.combinations.k less than zero": "\u65e0\u6cd5\u8ba1\u7b97\u7ec4\u5408: k\u5c0f\u4e8e\u96f6", "math.combinations.n less than k": "\u65e0\u6cd5\u8ba1\u7b97\u7ec4\u5408: n\u5c0f\u4e8ek", "math.permutations.n less than zero": "\u65e0\u6cd5\u8ba1\u7b97\u6392\u5217: n\u5c0f\u4e8e\u96f6", "math.permutations.k less than zero": "\u65e0\u6cd5\u8ba1\u7b97\u6392\u5217: k\u5c0f\u4e8e\u96f6", "math.permutations.n less than k": "\u65e0\u6cd5\u8ba1\u7b97\u6392\u5217: n\u5c0f\u4e8ek", "part.numberentry.give your answer to precision_0": "\u5c06\u7b54\u6848\u56db\u820d\u4e94\u5165\u5230\u6700\u63a5\u8fd1\u7684\u6574\u6570\u3002", "mathjax.error": "MathJax\u5904\u7406\u9519\u8bef\uff1a{{-message}}", "mathjax.error with context": "{{-context}}\u4e2d\u7684MathJax\u5904\u7406\u9519\u8bef\uff1a{{-message}}", "exam.introduction": "\u8003\u8bd5\u7b80\u4ecb", "exam.feedback": "\u8003\u8bd5\u53cd\u9988\u4fe1\u606f", "jme.tokenise.keypair key not a string": "\u5b57\u5178\u952e\u5e94\u4e3a\u5b57\u7b26\u4e32\uff0c\u800c\u4e0d\u662f{{type}}\u3002", "jme.shunt.list mixed argument types": "\u65e0\u6cd5\u89e3\u6790{{mode}}: \u5b57\u5178\u548c\u5217\u8868\u5143\u7d20\u7684\u6df7\u5408", "jme.func.listval.key not in dict": "\u5b57\u5178\u4e0d\u5305\u542bkey<code>{{key}}</code>", "part.prompt": "\u63d0\u793a", "part.feedback": "\u53cd\u9988", "part.numberentry.answer not reduced": "\u60a8\u7684\u56de\u7b54\u8fd8\u6ca1\u51cf\u5c11\u5230\u6700\u4f4e\u9650\u5ea6\u3002", "part.numberentry.give your answer as a reduced fraction": "\u5c06\u7b54\u6848\u964d\u4f4e\u5230\u6700\u4f4e\u9650\u5ea6\u3002", "part.numberentry.negative decimal places": "\u8bbe\u7f6e\u6b64\u90e8\u5206\u7684\u76ee\u7684\u662f\u5c06\u5b66\u751f\u7684\u7b54\u6848\u56db\u820d\u4e94\u5165\u4e3a\u8d1f\u7684\u5c0f\u6570\u4f4d\u6570\uff0c\u8fd9\u6ca1\u6709\u4efb\u4f55\u610f\u4e49\u3002", "part.mcq.choices": "\u9009\u62e9", "part.mcq.answers": "\u7b54\u6848", "part.mcq.matrix cell empty": "\u90e8\u5206{{part}}\u77e9\u9635\u5355\u5143\u683c ({{row}},{{column}})\u4e3a\u7a7a", "part.mcq.matrix jme error": "\u90e8\u5206{{part}}\u77e9\u9635\u5355\u5143\u683c ({{row}},{{column}})\u7ed9\u51fa\u4e86JME\u9519\u8bef\uff1a{{-error}}", "question.statement": "\u58f0\u660e", "ruleset.circular reference": "\u89c4\u5219\u96c6code>{{name}}</code>\u5b9a\u4e49\u4e2d\u7684\u5faa\u73af\u5f15\u7528", "ruleset.set not defined": "\u89c4\u5219\u96c6{{name}}\u5c1a\u672a\u5b9a\u4e49", "jme.evaluate.no scope given": "\u5fc5\u987b\u7ed9Numbas.jme.evaluate\u4e00\u4e2aScope", "answer.number.not a number": "\u60a8\u7684\u7b54\u6848\u65e0\u6548\u3002", "answer.number.fractions not allowed": "\u60a8\u4e0d\u80fd\u8f93\u5165\u5206\u6570\u3002", "answer.jme.invalid expression": "{{-message}}", "answer.matrix.fractions not allowed": "\u60a8\u4e0d\u80fd\u8f93\u5165\u5206\u6570\u3002", "answer.matrix.some cell not a number": "\u60a8\u7684\u7b54\u6848\u4e2d\u7684\u4e00\u4e2a\u6216\u591a\u4e2a\u5355\u5143\u683c\u4e0d\u662f\u6709\u6548\u6570\u5b57\u3002", "exam.enter password": "\u5bc6\u7801:", "exam.password.correct": "\u8be5\u5bc6\u7801\u6b63\u786e\u3002 \u60a8\u53ef\u4ee5\u5f00\u59cb\u8003\u8bd5\u3002", "exam.password.incorrect": "\u8be5\u5bc6\u7801\u9519\u8bef\u3002", "frontpage.scorm.lms not connected": "\u8be5\u8003\u8bd5\u4ee5\u72ec\u7acb\u6a21\u5f0f\u8fd0\u884c\u3002 \u60a8\u7684\u7b54\u6848\u548c\u6807\u8bb0\u5c06\u4e0d\u4f1a\u4fdd\u5b58\uff01", "result.question review": "\u56de\u987e", "control.confirm regen no marks": "\u60a8\u60f3\u91cd\u65b0\u968f\u673a\u5316\u8fd9\u4e2a\u95ee\u9898\u5417\uff1f", "control.confirm reveal no marks": "\u60a8\u60f3\u663e\u793a\u8fd9\u4e2a\u95ee\u9898\u7684\u7b54\u6848\u5417\uff1f", "jme.tokenise.invalid near": "\u65e0\u6548\u7684\u8868\u8fbe\u5f0f: <code>{{expression}}</code>\u9644\u8fd1\u7684\u4f4d\u7f6e{{position}}\u4e0a\u7684<code>{{nearby}}</code>", "jme.tokenise.number.object not complex": "\u65e0\u6548\u7684\u5bf9\u8c61\u4f20\u9012\u7ed9\u6570\u5b57\u6784\u9020\u51fd\u6570\u3002", "jme.subvars.null substitution": "\u7a7a\u53d8\u91cf\u66ff\u6362: <code>$t(left brace){{str}}$t(right brace)", "jme.type.type already registered": "\u6570\u636e\u7c7b\u578b{{type}}\u5df2\u88ab\u6ce8\u518c\uff0c\u56e0\u6b64\u65e0\u6cd5\u518d\u6b21\u6ce8\u518c\u3002", "jme.type.no cast method": "\u65e0\u6cd5\u81ea\u52a8\u4ece{{from}}\u8f6c\u6362\u4e3a{{to}}\u3002", "jme.display.simplifyTree.empty expression": "\u8868\u8fbe\u5f0f\u4e3a\u7a7a", "jme.display.simplifyTree.stuck in a loop": "\u7b80\u5316\u7a0b\u5e8f\u9677\u5165\u5faa\u73af: <code>{{expr}}</code>", "math.niceNumber.undefined": "\u539f\u672c\u671f\u671b\u6709\u4e00\u4e2a\u6570\u5b57\uff0c\u4f46 <code>undefined</code>", "math.rangeToList.zero step size": "\u65e0\u6cd5\u5c06\u6b65\u957f\u4e3a\u96f6\u7684\u8303\u56f4\u8f6c\u6362\u4e3a\u5217\u8868\u3002", "part.error": "{{path}}: {{-message}}", "part.marking.revealed steps": "\u60a8\u663e\u793a\u4e86\u6b65\u9aa4\u3002", "part.marking.maximum scaled down": "\u60a8\u53ef\u4ee5\u4e3a\u6b64\u90e8\u5206\u8bc4\u5206\u7684\u6700\u9ad8\u5206\u662f<strong>{{count,niceNumber}}</strong> $t(mark)\u3002 \u60a8\u7684\u5206\u6570\u5c06\u76f8\u5e94\u964d\u4f4e\u3002", "part.marking.minimum score applied": "\u8fd9\u90e8\u5206\u7684\u6700\u4f4e\u5206\u6570\u662f<strong>{{score,niceNumber}}</strong>\u3002", "part.marking.maximum score applied": "\u8fd9\u90e8\u5206\u7684\u6700\u9ad8\u5206\u6570\u662f<strong>{{score,niceNumber}}</strong>\u3002", "part.marking.error in marking script": "\u8be5\u90e8\u5206\u7684\u8bc4\u5206\u7b97\u6cd5\u51fa\u9519\u3002 \u8bf7\u62a5\u544a\u6b64\u9519\u8bef\u3002 {{-message}}", "part.marking.no result after replacement": "\u65e0\u6cd5\u4f7f\u7528\u60a8\u5bf9\u5148\u524d\u90e8\u5206\u7684\u56de\u7b54\u6765\u8bc4\u5206\u8fd9\u90e8\u5206\u3002", "part.marking.missing required note": "\u8bc4\u5206\u7b97\u6cd5\u672a\u5b9a\u4e49\u6ce8\u91ca<code>{{note}}</code>", "marking.apply.not a list": " <code>apply</code>\u7684\u7b2c\u4e00\u4e2a\u53c2\u6570\u5fc5\u987b\u662f\u4e00\u4e2a\u5217\u8868\uff0c\u5e76\u4e14\u4e0d\u662f", "marking.apply marking script.script not found": "\u627e\u4e0d\u5230\u6807\u8bb0\u811a\u672c<code>{{name}}</code", "marking.note.compilation error": "\u7f16\u8bd1\u6ce8\u91ca<code>{{name}}</code>\u65f6\u51fa\u9519: {{-message}}\n", "marking.note.error evaluating note": "\u8bc4\u4f30\u7b14\u8bb0 <code>{{name}}</code> - {{-message}}\u65f6\u51fa\u9519", "marking.note.invalid definition": "\u65e0\u6548\u7684\u6ce8\u91ca\u5b9a\u4e49: <code>{{source}}</code>\u3002{{-hint}}", "marking.note.invalid definition.missing colon": "\u60a8\u53ef\u80fd\u4f1a\u5728\u540d\u79f0\u548c\u63cf\u8ff0\u540e\u9057\u6f0f\u4e00\u4e2a\u5192\u53f7", "marking.note.invalid definition.description missing closing bracket": "\u60a8\u53ef\u80fd\u7f3a\u5c11\u53f3\u62ec\u53f7", "marking.note.empty expression": "\u6ce8\u91ca<code>{{name}}</code>>\u4e3a\u7a7a\u3002", "marking.script.error parsing notes": "\u89e3\u6790\u6807\u8bb0\u811a\u672c\u65f6\u51fa\u9519: {{- message}}", "part.feedback out of date": "\u8be5\u53cd\u9988\u57fa\u4e8e\u60a8\u6700\u540e\u63d0\u4ea4\u7684\u7b54\u6848\u3002 \u63d0\u4ea4\u66f4\u6539\u7684\u7b54\u6848\u4ee5\u83b7\u53d6\u66f4\u65b0\u7684\u53cd\u9988\u3002", "part.jme.invalid value generator expression": "\u53d8\u91cf<code>{{name}}</code>\u7684\u65e0\u6548\u503c\u751f\u6210\u5668\u8868\u8fbe\u5f0f: {{-message}}", "part.mcq.incorrect choice": "\u60a8\u9009\u62e9\u4e86\u9519\u8bef\u7684\u7b54\u6848\u3002", "part.matrix.not all cells same precision": "\u60a8\u6ca1\u6709\u4ee5\u76f8\u540c\u7684\u7cbe\u5ea6\u7ed9\u51fa\u7b54\u6848\u4e2d\u7684\u6bcf\u4e2a\u5355\u5143\u683c\u3002", "part.gapfill.error marking gap": "\u8bc4\u5206{{name}}\u65f6\u51fa\u9519: {{-message}}", "part.custom.empty setting": "\u6ca1\u6709\u7ed9\u51fa\u503c\u3002", "part.custom.unrecognised input type": "\u65e0\u6cd5\u8bc6\u522b\u7684\u8bbe\u7f6e\u7c7b\u578b<code>{{input_type}}</code>", "part.custom.error evaluating input option": "\u8bc4\u4f30\u8f93\u5165\u9009\u9879<code>{{option}}</code>\u65f6\u51fa\u9519: {{-error}}", "part.custom.input option missing": "\u8f93\u5165\u9009\u9879 <code>{{option}}</code>\u7684\u5b9a\u4e49\u4e22\u5931\u3002", "part.custom.error evaluating setting": "\u8bc4\u4f30\u8bbe\u7f6e<code>{{setting}}</code>\u65f6\u51fa\u9519: {{-error}}", "question.error creating question": "\u521b\u5efa\u95ee\u9898{{number}}\u65f6\u51fa\u9519: {{-message}}", "question.score feedback.not marked": "\u672a\u8bc4\u5206", "question.score feedback.partially answered": "\u90e8\u5206\u56de\u7b54", "question.score feedback.score total": "{{marksString}}", "question.score feedback.score actual": "\u5f97\u5206: {{scoreString}}", "question.score feedback.score total actual": "\u5f97\u5206: {{score,niceNumber}}/{{marks,niceNumber}}", "variable.error in variable definition": "\u53d8\u91cf<code>{{name}}</code>\u7684\u5b9a\u4e49\u9519\u8bef", "left brace": "{", "right brace": "}", "extension.not found": "\u65e0\u6cd5\u52a0\u8f7d\u6269\u5c55\uff1a<code>{{name}}</code>\u3002", "control.toggle navigation menu": "\u5207\u6362\u5bfc\u822a\u83dc\u5355", "part.input title": "\u7b2c{{name}}\u6b65\u7b54\u6848", "part.correct answer title": "\u7b2c{{name}}\u6b65\u6b63\u786e\u7b54\u6848", "part.jme.must-match.failed": "\u7b54\u6848\u683c\u5f0f\u5b58\u5728\u9519\u8bef\u3002", "question.score feedback.none": "", "control.submit part.confirm remove next parts": "<p>One or more subsequent parts depend on your answer to this part. Submitting this part again will invalidate those parts, and remove them from the question. This cannot be undone.</p>\n<p>Would you like to submit this part again?</p>", "control.back to menu": "\u8fd4\u56de\u83dc\u5355", "display.error making html": "Error making HTML in {{contextDescription}}: {{-message}}", "jme.subvars.error compiling": "{{-message}} in <code>{{expression}}</code>", "jme.variables.empty name": "\u95ee\u9898\u4e2d\u7684\u4e00\u4e2a\u53d8\u91cf\u8fd8\u6ca1\u88ab\u547d\u540d", "jme.calculus.unknown derivative": "Don't know how to differentiate <code>{{tree}}</code>", "math.order complex numbers": "Can't order complex numbers", "menu.choose a question": "\u9009\u62e9\u4e00\u4e2a\u95ee\u9898", "part.choose next part.answered": "\u60a8\u63a5\u4e0b\u6765\u60f3\u505a\u4ec0\u4e48 \uff1f", "part.choose next part.unanswered": "\u4e0d\u7136\uff0c \u60a8\u53ef\u4ee5\uff1a", "part.choose next part.will be locked": "\uff08\u8be5\u90e8\u5206\u5c06\u4f1a\u88ab\u9501\u5b9a\uff09", "part.reached dead end": "\u81ea\u6b64\u4ee5\u4e0b\uff0c\u672a\u6709\u53ef\u505a\u4e4b\u4e8b\u4e86", "part.next part.penalty amount": "(lose {{count}} $t(mark))", "part.marking.counts towards objective": "This part counts towards the objective <strong>\u201c{{objective}}\u201d</strong>.", "part.numberentry.answer not integer or decimal or fraction": "\u4f60\u7684\u7b54\u6848\u662f\u65e0\u6548\u7684\u3002\u4f60\u5fc5\u987b\u8f93\u5165\u4e00\u4e2a\u6574\u6570\u3001\u4e00\u4e2a\u5c0f\u6570\u6216\u4e00\u4e2a\u5206\u6570\u3002", "question": "\u9898\u76ee", "question.progress": "\u9898\u76ee\u8fdb\u5ea6 \uff1a", "question.score feedback.unattempted": "\u672a\u5c1d\u8bd5", "question.score feedback.attempted": "\u5df2\u5c1d\u8bd5", "question.score feedback.score actual.plain": "{{scoreString}}", "question.score feedback.score total actual.plain": "{{score,niceNumber}}/{{marks,niceNumber}}", "question.objectives": "\u76ee\u6807", "question.penalties": "\u60e9\u7f5a", "question.back to previous part": "\u8fd4\u56de\u4e0a\u4e00\u90e8\u5206", "end.print": "\u6253\u5370\u60a8\u7684\u6210\u7ee9\u5355", "math.shuffle_together.lists not all the same length": "list\u957f\u5ea6\u672a\u5168\u90e8\u76f8\u540c\u3002", "jme.parse signature.invalid signature string": "Invalid function signature string: {{str}}", "part.custom.expected answer has wrong type": "The expected answer for this part has the wrong type. It should be <code>{{shouldbe}}</code>.", "part.custom.input option has wrong type": "The answer input setting <code>{{option}}</code> has the wrong type. It should be <code>{{shouldbe}}</code>.", "matrix input.size control legend": "Size", "matrix input.rows": "\u884c", "matrix input.columns": "\u5217", "part.jme.error checking numerically": "There was an error numerically checking your answer: {{-message}}", "part.gapfill.cyclic adaptive marking": "There is a cycle in the adaptive marking for this part: <strong>{{name1}}</strong> relies on <strong>{{name2}}</strong>, which eventually relies on <strong>{{name1}}</strong>.", "modal.style.background colour": "\u80cc\u666f\u8272", "modal.style.text colour": "\u5b57\u7b26\u989c\u8272", "modal.style.text size": "\u5b57\u7b26\u5927\u5c0f", "modal.style.explanation": "\u7528\u8fd9\u4e9b\u63a7\u5236\u6765\u6539\u53d8\u8003\u8bd5\u7684\u5916\u89c2\u3002", "modal.style.reset to defaults": "\u91cd\u7f6e", "modal.style.text size preview": "\u5927\u591a\u6570\u5b57\u7b26\u90fd\u5c06\u5982\u6b64\u5927\u5c0f", "control.style options": "\u663e\u793a\u9009\u9879", "part.marking.partially correct": "\u60a8\u7684\u7b54\u6848\u90e8\u5206\u6b63\u786e\u3002", "part.marking.error in adaptive marking": "There was an error in the adaptive marking for this part. Please report this. {{-message}}", "page.skip to content": "Skip to content", "result.learning objective": "\u5b66\u4e60\u76ee\u6807", "jme.interpreted as": "interpreted as", "jme.script.note.compilation error": "Error compiling note <code>{{name}}</code>: {{-message}}", "jme.script.note.error evaluating note": "Error evaluating note <code>{{name}}</code> - {{-message}}", "jme.script.note.invalid definition": "Invalid note definition: <code>{{source}}</code>. {{-hint}}", "jme.script.note.invalid definition.missing colon": "\u60a8\u53ef\u80fd\u5728\u540d\u79f0\u548c\u63cf\u8ff0\u540e\u9762\u7f3a\u5c11\u4e00\u4e2a\u5192\u53f7", "jme.script.note.invalid definition.description missing closing bracket": "\u60a8\u53ef\u80fd\u6ca1\u6253\u62ec\u53f7\u7684\u540e\u4e00\u534a\u3002", "jme.script.note.empty expression": "The note <code>{{name}}</code> is empty.", "jme.script.error parsing notes": "Error parsing marking script: {{- message}}", "matrix input.cell label": "Row {{row}}, column {{column}}", "control.move to next question": "\u8f6c\u5230\u4e0b\u4e00\u4e2a\u95ee\u9898", "diagnostic.use retry": "\u4f7f\u7528\u6389\u4e00\u6b21\u91cd\u8bd5\uff0c\u5e76\u518d\u5c1d\u8bd5\u4e00\u4e0b\u8fd9\u4e2a\u4e3b\u9898\u3002", "diagnostic.move to next topic": "\u8f6c\u5230\u4e0b\u4e00\u4e2a\u8bdd\u9898\u3002", "diagnostic.next step question": "\u60a8\u63a5\u4e0b\u6765\u60f3\u505a\u4ec0\u4e48\uff1f", "diagnostic.now assessing topic": "Now assessing {{current_topic}}", "diagnostic.one retry left": "\u60a8\u8fd8\u67091\u6b21\u91cd\u8bd5\u673a\u4f1a", "diagnostic.retries left": "You have {{retries}} retries left.", "diagnostic.percentage completed": "You've completed <strong>{{percentage}}%</strong> of the test.", "diagnostic.test is over": "\u6d4b\u8bd5\u7ed3\u675f\u3002", "diagnostic.passed all lo": "\u60a8\u5df2\u7ecf\u901a\u8fc7\u4e86\u6240\u6709\u7684\u5b66\u4e60\u76ee\u6807\u3002", "diagnostic.more work on lo": "You need to do some more work on the following learning objectives: {{los}}.", "diagnostic.move to next question in topic": "\u8f6c\u8fdb\u5230\u4e3b\u9898\u4e2d\u7684\u4e0b\u4e00\u4e2a\u95ee\u9898\u3002", "diagnostic.complete": "\u5b8c\u6210\u4e86\uff01", "diagnostic.studying topic": "Studying {{topic}}", "display.answer widget.unknown widget type": "The answer widget type <code>{{name}}</code> is not recognised.", "jme.shunt.expected argument before comma": "Expected to see something between the opening bracket and the comma", "part.waiting for pre submit": "\u60a8\u7684\u56de\u7b54\u6b63\u5728\u8bc4\u5206\u79cd\u3002\u8bf7\u7b49\u5f85\u3002", "diagnostic.end test": "\u7ed3\u675f\u6d4b\u8bd5", "page.no stylesheet loaded": "The page's stylesheet file has not loaded.", "modal.confirm": "\u786e\u5b9a", "modal.alert": "\u8b66\u544a", "suspend.resumed header": "Attempt resumed", "jme.vector.value not an array of numbers": "Tried to construct a vector using a value that is not an array of numbers.", "jme.matrix.value not the right type": "\u6784\u5efa\u5411\u91cf\u65f6\u5019\u7528\u4e86\u4e0d\u6b63\u786e\u7684\u6570\u503c\u7c7b\u578b\u3002", "jme.subvars.html inserted twice": "An HTML value has been embedded twice. Consider defining a function to generate a new value each time it is used.", "jme.variables.invalid function language": "The language <code>{{language}}</code> is not valid.", "jme.variables.duplicate definition": "There is more than one definition of the variable <code>{{name}}</code>.", "math.random_integer_partition.invalid k": "The size of the partition must be between 1 and {{n}}.", "part.marking.parameter already in scope": "There is a variable named <code>{{name}}</code>, which is also the name of a marking parameter. Please rename the variable.", "part.marking.adaptive variable replacement refers to self": "This part refers to itself in a variable replacement for adaptive marking.", "part.marking.adaptive variable replacement refers to nothing": "This part contains an invalid variable replacement for adaptive marking.", "part.numberentry.display answer wrong type": "The display answer for this part is a value of type <code>{{got_type}}</code>, but should be a <code>{{want_type}}</code>.", "part.matrix.invalid type in prefilled": "There is an invalid value of type <code>{{n}}</code> in the array of pre-filled cells.", "diagnostic.make a choice": "\u505a\u51fa\u9009\u62e9", "matrixmath.not square": "This operation only works on a square matrix.", "matrixmath.not invertible": "This operation only works on an invertible matrix.", "matrixmath.gauss-jordan elimination.not enough columns": "\u77e9\u9635\u7684\u884c\u5217\u6570\u76ee\u5fc5\u987b\u81f3\u5c11\u4e00\u81f4\u3002", "question.required extension not available": "This question requires the extension <code>{{-extension}}</code> but it is not available.", "util.formatNumberNotation.unrecognised syntax": "The number formatting syntax <code>{{syntax}}</code> is not recognised.", "worksheet.number of exams to generate": "\u5f85\u8f93\u51fa\u5f20\u6570", "worksheet.starting with id": "Starting with ID", "worksheet.show exam id": "\u663e\u793a\u5377ID \uff1f", "worksheet.page break between questions": "\u60f3\u5728\u4e24\u9898\u4e4b\u95f4\u6362\u9875 \uff1f", "worksheet.page margins": "\u9875\u8fb9\u8ddd\uff08\u6beb\u7c73\uff09", "worksheet.text size": "\u5b57\u7b26\u5927\u5c0f\uff08\u78c5\uff09", "worksheet.generate": "\u8f93\u51fa", "worksheet.generating exams": "Generating sheets", "worksheet.sheet id": "\u5377\u7f16\u53f7", "worksheet.print single": "\u6253\u5370\u8be5\u5377", "worksheet.print several": "\u6253\u5370\u8fd9\u4e9b\u5377", "worksheet.answer sheets": "\u7b54\u5377", "worksheet.question sheets": "\u8bd5\u5377", "worksheet.reconfigure": "\u8f93\u51fa\u4e0d\u540c\u5377", "worksheet.show sheet": "\u4e0e\u7f16\u53f7\u4e00\u8d77\u9884\u89c8\u8be5\u5377", "accessibility statement": "Accessibility statement and guide to adapting Numbas to your needs.", "exam.enter your name": "Your name:", "exam.attempt download security warning": "This exam is configured to allow you to download your data, but it is not running in a secure browser context. You will not be able to download your exam data. Contact your lecturer or teacher for help.", "result.download exam object": "Download your exam data", "control.return to question": "Return to the question", "control.show introduction": "Introduction", "analysis.header": "Analyse attempt data", "analysis.help.upload files": "Upload attempt data files that your students have given you.", "analysis.help.file input label": "Choose student attempt data files, or drag files onto this window.", "analysis.table.total": "Exam totals", "analysis.table.question": "Exam and question totals", "analysis.table.all": "All details", "analysis.student name.anonymous": "No name given", "analysis.expected": "Expected results", "analysis.start time": "Start time", "analysis.maximum": "Maximum Marks", "analysis.file": "File", "analysis.download": "Download", "analysis.delete": "Delete", "analysis.view results": "View results", "analysis.upload files": "Upload files", "analysis.upload more": "Upload more files", "analysis.attempt data": "Attempt data", "analysis.select format": "Select data to show", "analysis.download this table": "Download this table", "analysis.student": "Student Results", "analysis.question key": "Question Key", "analysis.question name": "Question Name", "analysis.group": "Group", "analysis.question": "Question", "analysis.part": "Part", "analysis.gap": "Gap", "analysis.record type": "Record Type", "analysis.score": "Score", "analysis.marks available": "Marks Available", "analysis.percentage": "Percentage", "analysis.answer": "Answer", "analysis.student name": "Student Name", "analysis.summary.no files": "You have not uploaded any files yet.", "analysis.summary.no decrypted files": "You have not uploaded any valid files yet.", "analysis.summary.one file": "One attempt.", "analysis.summary.several files": "{{num_files,niceNumber}} attempts.", "analysis.not secure context": "This page must be opened in a secure browser context. A secure context is either a page served over HTTPS, or a file loaded from your device.", "jme.shunt.pipe right hand takes no arguments": "The expression on the right-hand side of the pipe operator must be a function application.", "question.explore.no parts defined": "There are no parts defined in this question.", "answer": "answer", "worksheet.answersheet show question content": "Show question content in answer sheets?", "modal.confirm end exam": "Write <code>{{endConfirmation}}</code> in the box to confirm:", "modal.end exam button": "End exam", "lightbox.zoom in on image": "Zoom in on this image", "exam.progress": "Progress", "exam.questions answered": "{{numAnsweredQuestions}} of {{numQuestions}} questions answered.", "result.question marks available": "Marks Available", "result.question answered": "Answered?", "control.confirm end.correct": "You may now end the exam.", "control.confirm end.incorrect": "This is not the expected text.", "control.confirm end.password": "end", "jme.typecheck.for in name wrong type": "The name in a <code>for</code> statement must be a name or list of names, not {{type}}.", "jme.makeFast.no fast definition of function": "The function <code>{{name}}</code> here isn't defined in a way that can be made fast.", "part.show feedback": "Show feedback", "part.hide feedback": "Hide feedback", "part.feedback title": "Feedback for {{name}}.", "part.jme.must-match.warning": "Your answer is not in the expected form: {{-message}}", "part.numberentry.write your answer as a fraction": "Write your answer as a fraction.", "question.nav.label": "Question controls", "question.answer saved": "Answer saved", "question.all answers saved": "All answers saved", "analysis.back to results": "Back to results", "analysis.review": "Review", "analysis.review this": "Review this attempt", "analysis.reviewing attempt": "Reviewing attempt by {{student_name}}.", "part.there is new feedback": "The feedback has changed.", "modal.style.colour scheme": "Colour scheme", "modal.style.automatic colour scheme": "Automatic", "modal.style.light colour scheme": "Light", "modal.style.dark colour scheme": "Dark", "modal.style.custom colour scheme": "Custom", "modal.style.main font": "Main font", "modal.style.font.sans serif": "Sans serif", "modal.style.font.serif": "Serif", "modal.style.font.monospace": "Monospace", "modal.style.spacing": "Spacing", "modal.style.font weight": "Font weight", "modal.style.forced colours warning": "Your browser has overridden the colours used in this page because of a setting such as high contrast mode. Changes to the colour scheme settings here might not have any effect.", "modal.style.text preview": "Most text will look like this.", "modal.style.more options": "More options", "modal.style.main colour": "Main (brand) colour", "modal.style.primary colour": "Primary button colour", "modal.style.link colour": "Link colour", "modal.style.success colour": "Success/correct colour", "modal.style.info colour": "Info colour", "modal.style.warning colour": "Warning colour", "modal.style.danger colour": "Danger/incorrect colour", "modal.style.muted colour": "Muted text colour", "modal.style.highlight colour": "Highlight colour", "exam.error loading exam definition": "There was an error while loading the exam definition: {{text}}", "exam.no exam definition": "No exam definition was given.", "jme.typecheck.wrong arguments for anonymous function": "Wrong number of arguments for this anonymous function.", "worksheet.top": "Top", "worksheet.left": "Left", "worksheet.bottom": "Bottom", "worksheet.right": "Right"}}, "ar-sa": {"translation": {"page.loading": "\u062a\u062d\u0645\u064a\u0644...\n\u062a\u062d\u0645\u064a\u0644 \u0627\u0644\u0635\u0641\u062d\u0629", "page.saving": "<p>\u062d\u0641\u0638</p>\n<p>\u0642\u062f \u064a\u0633\u062a\u063a\u0631\u0642 \u0647\u0630\u0627 \u0628\u0636\u0639 \u062b\u0648\u0627\u0646</p>", "mathjax.math processing error": "\"{{-message}}\" \u0639\u0646\u062f \u062a\u062d\u062f\u064a\u062f <code> {{\u062a\u0639\u0628\u064a\u0631}} </ code>", "die.numbas failed": "\u0641\u0634\u0644 Numbas ", "die.sorry": "\u0639\u0630\u0631\u0627\u064b\u060c \u062d\u0635\u0644 \u062e\u0637\u0623 \u0641\u064a Numbas \u0648\u0644\u0627 \u064a\u0645\u0643\u0646 \u0627\u0644\u0645\u062a\u0627\u0628\u0639\u0629.\n\u0641\u064a\u0645\u0627 \u064a\u0644\u064a \u0648\u0635\u0641 \u0627\u0644\u062e\u0637\u0623.", "die.error": "\u062e\u0637\u0623", "modal.ok": "\u0645\u0648\u0627\u0641\u0642", "modal.cancel": "\u0627\u0644\u063a\u0627\u0621", "exam.exam name": "\u0627\u0633\u0645 \u0627\u0644\u0627\u0645\u062a\u062d\u0627\u0646", "exam.random seed": "\u0631\u0642\u0645 \u0627\u0644\u062c\u0644\u0633\u0629", "exam.student name": "\u0627\u0633\u0645 \u0627\u0644\u0637\u0627\u0644\u0628", "exam.number of questions": "\u0639\u062f\u062f \u0627\u0644\u0623\u0633\u0626\u0644\u0629", "exam.marks available": "\u0627\u0644\u062f\u0631\u062c\u0627\u062a \u0627\u0644\u0645\u062a\u0627\u062d\u0629", "exam.pass percentage": "\u0646\u0633\u0628\u0629 \u0627\u0644\u0646\u062c\u0627\u062d", "exam.time allowed": "\u0627\u0644\u0648\u0642\u062a \u0627\u0644\u0645\u062a\u0627\u062d", "exam.passed": "\u0646\u062c\u0627\u062d", "exam.failed": "\u0641\u0634\u0644", "exam.review header": "\u0645\u0631\u0627\u062c\u0639\u0629", "frontpage.start": "\u0627\u0644\u0628\u062f\u0621", "suspend.paused header": "\u062a\u0648\u0642\u0641 \u0645\u0624\u0642\u062a", "suspend.exam suspended": "\u0627\u0644\u0627\u0645\u062a\u062d\u0627\u0646 \u0645\u062a\u0648\u0642\u0641 \u0628\u0634\u0643\u0644 \u0645\u0624\u0642\u062a. \u0627\u0636\u063a\u0637 <em>\u0645\u062a\u0627\u0628\u0639\u0629</em> \u0644\u0644\u0627\u0633\u062a\u0645\u0631\u0627\u0631.", "suspend.you can resume": "\u064a\u0645\u0643\u0646\u0643 \u0625\u0643\u0645\u0627\u0644 \u0627\u0644\u0625\u062e\u062a\u0628\u0627\u0631 \u0641\u064a \u0627\u0644\u0645\u0631\u0629 \u0627\u0644\u0642\u0627\u062f\u0645\u0629", "suspend.resume": "\u0645\u062a\u0627\u0628\u0639\u0629", "result.exit": "\u0645\u063a\u0627\u062f\u0631\u0629 \u0627\u0644\u0625\u062e\u062a\u0628\u0627\u0631", "result.print": "\u0637\u0628\u0627\u0639\u0629 \u0645\u0644\u062e\u0635 \u0627\u0644\u0646\u062a\u0627\u0626\u062c", "result.exam summary": "\u0645\u0644\u062e\u0635 \u0627\u0644\u0625\u062e\u062a\u0628\u0627\u0631", "result.performance summary": "\u0645\u0644\u062e\u0635 \u0627\u0644\u0623\u062f\u0627\u0621", "result.exam start": "\u0628\u062f\u0621 \u0627\u0644\u0627\u0645\u062a\u062d\u0627\u0646:", "result.exam stop": "\u0627\u0646\u062a\u0647\u0627\u0621 \u0627\u0644\u0627\u0645\u062a\u062d\u0627\u0646:", "result.time spent": "\u0627\u0644\u0648\u0642\u062a \u0627\u0644\u0645\u0646\u0642\u0636\u064a:", "result.questions attempted": "\u0639\u062f\u062f \u0627\u0644\u0623\u0633\u0626\u0644\u0629 \u0627\u0644\u062a\u064a \u062a\u0645\u062a \u0645\u062d\u0627\u0648\u0644\u062a\u0647\u0627", "result.score": "\u0627\u0644\u062f\u0631\u062c\u0629", "result.result": "\u0627\u0644\u0646\u062a\u064a\u062c\u0629:", "result.question number": "\u0631\u0642\u0645 \u0627\u0644\u0633\u0624\u0627\u0644", "result.question score": "\u0627\u0644\u062f\u0631\u062c\u0629", "result.question review title": "\u0645\u0631\u0627\u062c\u0639\u0629 \u0647\u0630\u0627 \u0627\u0644\u0633\u0624\u0627\u0644", "result.click a question to review": "\u0627\u0636\u063a\u0637 \u0639\u0644\u0649 \u0631\u0642\u0645 \u0633\u0624\u0627\u0644 \u0644\u0631\u0624\u064a\u0629 \u0643\u064a\u0641\u064a\u0629 \u062a\u0642\u064a\u064a\u0645 \u0627\u062c\u0627\u0628\u0627\u062a\u0643\u060c \u0648\u0627\u0644\u062d\u0644\u0648\u0644 \u0627\u0644\u0643\u0627\u0645\u0644\u0629 \u0627\u0644\u0645\u062a\u0648\u0641\u0631\u0629.", "end.exam has finished": "\u0647\u0630\u0627 \u0627\u0644\u0627\u0645\u062a\u062d\u0627\u0646 \u0642\u062f \u0627\u0646\u062a\u0647\u0649. \u0628\u0625\u0645\u0643\u0627\u0646\u0643 \u0627\u063a\u0644\u0627\u0642 \u0647\u0630\u0647 \u0627\u0644\u0646\u0627\u0641\u0630\u0629.", "control.confirm leave": "\u0644\u0645 \u062a\u0643\u0645\u0644 \u0627\u0644\u0627\u0645\u062a\u062d\u0627\u0646.", "control.not all questions answered": "\u0644\u0645 \u062a\u0643\u0645\u0644 \u0643\u0644 \u0623\u0633\u0626\u0644\u0629 \u0627\u0644\u0625\u0645\u062a\u062d\u0627\u0646.", "control.not all questions submitted": "\u0644\u0642\u062f \u0642\u0645\u062a \u0628\u062a\u0639\u062f\u064a\u0644 \u0625\u062c\u0627\u0628\u0629 \u0623\u0648 \u0623\u0643\u062b\u0631 \u0648\u0644\u0645 \u062a\u0642\u0645 \u0628\u062a\u0623\u0643\u064a\u062f \u0627\u0644\u062a\u0639\u062f\u064a\u0644. \u064a\u0631\u062c\u0649 \u062a\u0623\u0643\u064a\u062f \u062a\u0639\u062f\u064a\u0644\u0627\u062a\u0643 \u0644\u0643\u0644 \u0627\u0644\u0623\u0633\u0626\u0644\u0629.", "control.confirm end": "\u0647\u0644 \u0623\u0646\u062a \u0645\u062a\u0623\u0643\u062f \u0623\u0646\u0643 \u062a\u0631\u064a\u062f \u0625\u0646\u0647\u0627\u0621 \u0627\u0644\u0627\u062e\u062a\u0628\u0627\u0631\u061f \u0628\u0639\u062f \u0625\u0646\u0647\u0627\u0621 \u0627\u0644\u0627\u062e\u062a\u0628\u0627\u0631 \u060c \u0644\u0646 \u062a\u062a\u0645\u0643\u0646 \u0645\u0646 \u062a\u063a\u064a\u064a\u0631 \u0623\u064a \u0645\u0646 \u0625\u062c\u0627\u0628\u0627\u062a\u0643.", "control.confirm regen": "\u0647\u0644 \u062a\u0631\u063a\u0628 \u0628\u0627\u0633\u062a\u0628\u062f\u0627\u0644 \u0647\u0630\u0627 \u0627\u0644\u0633\u0624\u0627\u0644 \u0639\u0634\u0648\u0627\u0626\u064a\u0627\u064b\u061f \u0625\u0630\u0627 \u0646\u0642\u0631\u062a \"\u0645\u0648\u0627\u0641\u0642\" \u0641\u0633\u0648\u0641 \u062a\u0641\u0642\u062f \u0643\u0627\u0641\u0629 \u0625\u062c\u0627\u0628\u0627\u062a\u0643 \u0648\u0639\u0644\u0627\u0645\u0627\u062a\u0643 \u0639\u0646 \u0627\u0644\u0633\u0624\u0627\u0644 \u0627\u0644\u062d\u0627\u0644\u064a", "control.confirm reveal": "\u0647\u0644 \u062a\u0631\u063a\u0628 \u0628\u0627\u0644\u0643\u0634\u0641 \u0639\u0646 \u0625\u062c\u0627\u0628\u0629 \u0647\u0630\u0627 \u0627\u0644\u0633\u0624\u0627\u0644\u061f \u0633\u064a\u062a\u0645 \u0625\u0642\u0641\u0627\u0644 \u0623\u064a \u0639\u0644\u0627\u0645\u0627\u062a \u062a\u0644\u0642\u064a\u062a\u0647\u0627 \u062d\u062a\u0649 \u0627\u0644\u0622\u0646 \u0648\u0644\u0646 \u062a\u062a\u0645\u0643\u0646 \u0645\u0646 \u0627\u0644\u0625\u062c\u0627\u0628\u0629 \u0639\u0644\u0649 \u0647\u0630\u0627 \u0627\u0644\u0633\u0624\u0627\u0644 \u0644\u0627\u062d\u0642\u064b\u0627.", "control.proceed anyway": "\u0627\u0644\u0645\u062a\u0627\u0628\u0639\u0629 \u0639\u0644\u0649 \u0623\u064a \u062d\u0627\u0644", "control.regen": "\u0645\u062d\u0627\u0648\u0644\u0629 \u0633\u0624\u0627\u0644 \u0622\u062e\u0631 \u0645\u062b\u0644 \u0647\u0630\u0627", "control.submit answer": "\u0623\u0631\u0633\u0644 \u0627\u0644\u0625\u062c\u0627\u0628\u0629", "control.submit all parts": "\u0623\u0631\u0633\u0644 \u0643\u0644 \u0627\u0644\u0623\u062c\u0632\u0627\u0621", "control.submit again": "\u0623\u0631\u0633\u0644 \u0645\u0631\u0629 \u062b\u0627\u0646\u064a\u0629", "control.submit": "\u0623\u0631\u0633\u0644", "control.previous": "\u0627\u0644\u0633\u0627\u0628\u0642", "control.next": "\u0627\u0644\u062a\u0627\u0644\u064a", "control.advice": "\u0645\u0633\u0627\u0639\u062f\u0629", "control.reveal": "\u0643\u0634\u0641 \u0627\u0644\u0625\u062c\u0627\u0628\u0627\u062a", "control.total": "\u0627\u0644\u0645\u062c\u0645\u0648\u0639", "control.pause": "\u0625\u064a\u0642\u0627\u0641 \u0645\u0624\u0642\u062a", "control.end exam": "\u0625\u0646\u0647\u0627\u0621 \u0627\u0644\u0627\u0645\u062a\u062d\u0627\u0646", "control.back to results": "\u0627\u0644\u0639\u0648\u062f\u0629 \u0625\u0644\u0649 \u0627\u0644\u0646\u062a\u0627\u0626\u062c", "display.part.jme.error making maths": "\u064a\u0648\u062c\u062f \u062e\u0637\u0623 \u0641\u064a \u0639\u0631\u0636 \u0627\u0644\u0635\u064a\u063a\u0629 \u0627\u0644\u0631\u064a\u0627\u0636\u064a\u0629", "exam.xml.bad root": "\u0627\u0644\u0639\u0646\u0635\u0631 \u0627\u0644\u0623\u0633\u0627\u0633\u064a \u0641\u064a \u0645\u0644\u0641 XML \u0627\u0644\u062e\u0627\u0635 \u0628\u0627\u0644\u0625\u0645\u062a\u062d\u0627\u0646\u060c \u064a\u062c\u0628 \u0623\u0646 \u064a\u0643\u0648\u0646 'exam'", "exam.changeQuestion.no questions": "\u0647\u0630\u0627 \u0627\u0644\u0627\u0645\u062a\u062d\u0627\u0646 \u0644\u0627 \u064a\u062d\u0648\u064a \u0623\u064a \u0623\u0633\u0626\u0644\u0629! \u062a\u0623\u0643\u062f \u0645\u0646 \u062e\u0644\u0648 \u0645\u0644\u0641 exam. \u0645\u0646 \u0627\u0644\u0623\u062e\u0637\u0627\u0621.", "feedback.you were awarded": "\n\u062a\u0645 \u0645\u0646\u062d\u0643 <strong>{{count,niceNumber}}</strong> $t(mark).", "feedback.taken away": "\n\u062a\u0645 \u0633\u062d\u0628 <strong> {{count\u060c niceNumber}} </strong> $ t (\u0639\u0644\u0627\u0645\u0629) $ t (\u0639\u0644\u0627\u0645\u0629).", "jme.tokenise.invalid": "Invalid expression: <code>{{expression}}</code>", "jme.shunt.not enough arguments": "\u0644\u0627 \u062a\u0648\u062c\u062f \u0648\u0633\u064a\u0637\u0627\u062a \u0643\u0627\u0641\u064a\u0629 \u0644\u0644\u0639\u0645\u0644\u064a\u0629 <code> {{op}} </code>", "jme.shunt.no left bracket in function": "\u0644\u0627 \u064a\u0648\u062c\u062f \u0642\u0648\u0633 \u0623\u064a\u0633\u0631 \u0645\u0637\u0627\u0628\u0642 \u0641\u064a \u062a\u0637\u0628\u064a\u0642 \u0627\u0644\u0648\u0638\u064a\u0641\u0629 \u0623\u0648 \u0627\u0644\u0645\u062c\u0645\u0648\u0639\u0629", "jme.shunt.no left square bracket": "\u0644\u0645 \u064a\u062a\u0645 \u0625\u063a\u0644\u0627\u0642 \u0627\u0644\u0642\u0648\u0633 \u0627\u0644\u0623\u064a\u0633\u0631", "jme.shunt.no left bracket": "\u0644\u0645 \u064a\u062a\u0645 \u0625\u063a\u0644\u0627\u0642 \u0627\u0644\u0642\u0648\u0633 \u0627\u0644\u0623\u064a\u0633\u0631", "jme.shunt.no right bracket": "\u0644\u0645 \u064a\u062a\u0645 \u0625\u063a\u0644\u0627\u0642 \u0627\u0644\u0642\u0648\u0633 \u0627\u0644\u0623\u064a\u0645\u0646", "jme.shunt.no right square bracket": "\u0644\u0645 \u064a\u062a\u0645 \u0625\u063a\u0644\u0627\u0642 \u0627\u0644\u0642\u0648\u0633 \u0627\u0644\u0623\u064a\u0645\u0646 \u0644\u0625\u0646\u0647\u0627\u0621 \u0627\u0644\u0642\u0627\u0626\u0645\u0629", "jme.shunt.missing operator": "\u0644\u0627\u064a\u0645\u0643\u0646 \u062d\u0633\u0627\u0628 \u0642\u064a\u0645\u0629 \u0627\u0644\u062a\u0639\u0628\u064a\u0631 \u0627\u0644\u0631\u064a\u0627\u0636\u064a\u060c \u064a\u0648\u062c\u062f \u0645\u0624\u062b\u0631 \u0645\u0641\u0642\u0648\u062f", "jme.typecheck.function maybe implicit multiplication": "\n\u0627\u0644\u0648\u0638\u064a\u0641\u0629 <code> {{name}} </code> \u063a\u064a\u0631 \u0645\u062d\u062f\u062f\u0629. \u0647\u0644 \u062a\u0642\u0635\u062f <code> {{first}} * {{\u0627\u0644\u0645\u062d\u062a\u0645\u0644\u0629Op}} (...) </code>\u061f", "jme.typecheck.function not defined": "\u0627\u0644\u0648\u0638\u064a\u0641\u0629 <code> {{op}} </code> \u063a\u064a\u0631 \u0645\u062d\u062f\u062f\u0629. \u0647\u0644 <code> {{op}} </code> \u0645\u062a\u063a\u064a\u0631 \u060c \u0648\u0647\u0644 \u062a\u0642\u0635\u062f <code> {{Suggestion}} * (...) </code>\u061f", "jme.typecheck.op not defined": "\u0627\u0644\u0639\u0645\u0644\u064a\u0629 \"{{op}}\" \u063a\u064a\u0631 \u0645\u062d\u062f\u062f\u0629.", "jme.typecheck.no right type definition": "\u0644\u0645 \u064a\u062a\u0645 \u0627\u0644\u0639\u062b\u0648\u0631 \u0639\u0644\u0649 \u062a\u0639\u0631\u064a\u0641 '{{op}}' \u0645\u0646 \u0627\u0644\u0646\u0648\u0639 \u0627\u0644\u0635\u062d\u064a\u062d.", "jme.typecheck.no right type unbound name": "\u0627\u0644\u0645\u062a\u063a\u064a\u0631 <code> {{name}} </code> \u063a\u064a\u0631 \u0645\u0639\u0631\u0641.", "jme.typecheck.map not on enumerable": "\u064a\u062c\u0628 \u0623\u0646 \u062a\u0639\u0645\u0644 \u0639\u0645\u0644\u064a\u0629 <code> map </code> \u0639\u0628\u0631 \u0642\u0627\u0626\u0645\u0629 \u0623\u0648 \u0646\u0637\u0627\u0642 \u060c \u0648\u0644\u064a\u0633 {{type}}", "jme.evaluate.undefined variable": "\u0627\u0644\u0645\u062a\u063a\u064a\u0631 {{\u0627\u0644\u0625\u0633\u0645}} \u063a\u064a\u0631 \u0645\u0648\u062c\u0648\u062f", "jme.thtml.not html": "\u062a\u0645\u0631\u064a\u0631 \u0642\u064a\u0645\u0629 \u063a\u064a\u0631 HTML \u0625\u0644\u0649 \u0645\u064f\u0646\u0634\u0626 THTML.", "jme.func.switch.no default case": "\u0644\u0645 \u064a\u062a\u0645 \u0625\u062e\u062a\u064a\u0627\u0631 \u0642\u064a\u0645\u0629 \u0625\u0641\u062a\u0631\u0627\u0636\u064a\u0629 \u0644\u062c\u0645\u0644\u0629 \u0627\u0644\u062a\u062d\u0648\u064a\u0644 (Switch)", "jme.func.listval.invalid index": "\u0641\u0647\u0631\u0633 \u0642\u0627\u0626\u0645\u0629 \u063a\u064a\u0631 \u0635\u0627\u0644\u062d {{index}} \u0641\u064a \u0642\u0627\u0626\u0645\u0629 \u0627\u0644\u062d\u062c\u0645 {{size}}", "jme.func.listval.not a list": "\u063a\u0631\u0636 \u063a\u064a\u0631 \u0642\u0627\u0628\u0644 \u0644\u0644\u0643\u062a\u0627\u0628\u0629", "jme.func.matrix.invalid row type": "\u0644\u0627 \u064a\u0645\u0643\u0646 \u0625\u0646\u0634\u0627\u0621 \u0645\u0635\u0641\u0648\u0641\u0629 \u0645\u0646 \u0623\u0633\u0637\u0631 \u0645\u0646 \u0627\u0644\u0646\u0648\u0639 {{type}}.", "jme.func.except.continuous range": "\u0644\u0627 \u064a\u0645\u0643\u0646\u0643 \u0627\u0633\u062a\u0639\u0645\u0627\u0644 \u0645\u0634\u063a\u0651\u0644 \u0637\u0645\u0627 \u0639\u062f\u0627\u0637 \u0641\u064a \u0627\u0644\u0646\u0637\u0627\u0642\u0627\u062a \u0627\u0644\u0645\u0633\u062a\u0645\u0631\u0629", "jme.matrix.reports bad size": "\u062a\u0634\u064a\u0631 \u0627\u0644\u0645\u0635\u0641\u0648\u0641\u0629 \u0625\u0644\u0649 \u062d\u062c\u0645\u0647\u0627 \u0628\u0634\u0643\u0644 \u063a\u064a\u0631 \u0635\u062d\u064a\u062d - \u0644\u0627 \u0628\u062f \u0627\u0646\u0647 \u062e\u0637\u0623 \u0641\u064a \u062f\u0627\u0644\u0651\u0629 \u0627\u0644\u0645\u064f\u0646\u0634\u0626 (Constructor)", "jme.texsubvars.no right bracket": "\u0644\u0627 \u064a\u0648\u062c\u062f <code>] </code> \u0645\u0637\u0627\u0628\u0642 \u0641\u064a \u0648\u0633\u064a\u0637\u0627\u062a {{op}}.", "jme.texsubvars.missing parameter": "\u0627\u0644\u0645\u0639\u0644\u0645\u0629 \u0645\u0641\u0642\u0648\u062f\u0629 \u0641\u064a {{op}}: {{\u0645\u0639\u0644\u0645\u0629}}", "jme.texsubvars.no right brace": "\u0644\u0627 \u064a\u0648\u062c\u062f <code>} </code> \u0645\u0637\u0627\u0628\u0642 \u0641\u064a {{op}}", "jme.user javascript.error": "\u062e\u0637\u0623 \u0641\u064a \u0648\u0638\u064a\u0641\u0629 \u062c\u0627\u0641\u0627 \u0633\u0643\u0631\u064a\u0628\u062a \u0627\u0644\u0645\u0639\u0631\u0641\u0629 \u0645\u0646 \u0642\u0628\u0644 \u0627\u0644\u0645\u0633\u062a\u062e\u062f\u0645 <code> {{name}} </code>: {{-message}", "jme.variables.error making function": "\u062e\u0637\u0623 \u0641\u064a \u0639\u0645\u0644 \u0627\u0644\u0648\u0638\u064a\u0641\u0629 <code> {{name}} </code>: {{-message}}", "jme.variables.syntax error in function definition": "\u062e\u0637\u0623 \u0641\u064a \u0643\u062a\u0627\u0628\u0629 \u0625\u0633\u0645 \u0627\u0644\u062f\u0627\u0644\u0629", "jme.variables.variable not defined": "\u0627\u0644\u0645\u062a\u063a\u064a\u0631 <code> {{name}} </code> \u063a\u064a\u0631 \u0645\u0639\u0631\u0641.", "jme.variables.empty definition": "\u062a\u0639\u0631\u064a\u0641 \u0627\u0644\u0645\u062a\u063a\u064a\u0631 <code> {{name}} </code> \u0641\u0627\u0631\u063a.", "jme.variables.circular reference": "\u0645\u0631\u062c\u0639 \u0645\u062a\u063a\u064a\u0631 \u062f\u0627\u0626\u0631\u064a \u0641\u064a \u062a\u0639\u0631\u064a\u0641 <code> {{name}} </code>", "jme.variables.error computing dependency": "\u062d\u062f\u062b \u062e\u0637\u0623 \u0641\u064a \u0627\u0644\u062d\u0648\u0633\u0628\u0629 \u064a\u0634\u064a\u0631 \u0625\u0644\u0649 \u0627\u0644\u0645\u062a\u063a\u064a\u0631 <code> {{name}} </code>", "jme.variables.error evaluating variable": "\u062e\u0637\u0623 \u0641\u064a \u062a\u0642\u064a\u064a\u0645 \u0627\u0644\u0645\u062a\u063a\u064a\u0631 {{name}}: {{-message}}", "jme.variables.question took too many runs to generate variables": "\u0644\u0645 \u064a\u062a\u0645 \u0625\u0646\u0634\u0627\u0621 \u0645\u062c\u0645\u0648\u0639\u0629 \u0635\u0627\u0644\u062d\u0629 \u0645\u0646 \u0645\u062a\u063a\u064a\u0631\u0627\u062a \u0627\u0644\u0633\u0624\u0627\u0644 \u0641\u064a \u0627\u0644\u0648\u0642\u062a \u0627\u0644\u0645\u0646\u0627\u0633\u0628.", "jme.display.unknown token type": "\u0644\u0627 \u064a\u0645\u0643\u0646 \u062a\u062d\u062f\u064a\u062f \u0646\u0648\u0639 \u0627\u0644\u0631\u0645\u0632 \u0627\u0644\u0645\u0645\u064a\u0632 {{type}}", "jme.display.collectRuleset.no sets": "\u0644\u0645 \u064a\u062a\u0645 \u062a\u0633\u0644\u064a\u0645 \u0645\u062c\u0645\u0648\u0639\u0627\u062a \u0644 collectRuleset!", "jme.display.collectRuleset.set not defined": "\u0644\u0645 \u064a\u062a\u0645 \u062a\u062d\u062f\u064a\u062f \u0645\u062c\u0645\u0648\u0639\u0629 \u0627\u0644\u0642\u0648\u0627\u0639\u062f {{name}}", "jme.display.simplifyTree.no scope given": "\u064a\u062c\u0628 \u0625\u0639\u0637\u0627\u0621 \u0645\u062c\u0627\u0644 \u0644 Numbas.jme.display.simplify \u0627\u0644\u0634\u062c\u0631\u0629", "math.precround.complex": "\u0644\u0627 \u064a\u0645\u0643\u0646 \u0627\u0644\u062a\u0642\u0631\u064a\u0628 \u0625\u0644\u0649 \u0639\u062f\u062f \u0645\u0639\u0642\u062f \u0645\u0646 \u0627\u0644\u0645\u0646\u0627\u0632\u0644 \u0627\u0644\u0639\u0634\u0631\u064a\u0629.", "math.siground.complex": "\u0644\u0627 \u064a\u0645\u0643\u0646 \u0627\u0644\u062a\u0642\u0631\u064a\u0628 \u0625\u0644\u0649 \u0639\u062f\u062f \u0645\u0639\u0642\u062f \u0645\u0646 \u0627\u0644\u0623\u0631\u0642\u0627\u0645 \u0627\u0644\u0645\u0647\u0645\u0629.", "math.combinations.complex": "\u0644\u0627 \u064a\u0645\u0643\u0646 \u062d\u0633\u0627\u0628 \u062a\u0648\u0644\u064a\u0641\u0627\u062a \u0644\u0623\u0639\u062f\u0627\u062f \u0627\u0644\u0645\u0631\u0643\u0628\u0629.", "math.permutations.complex": "\u0644\u0627 \u064a\u0645\u0643\u0646 \u062d\u0633\u0627\u0628 \u062a\u0628\u0627\u062f\u064a\u0644 \u0644\u0623\u0639\u062f\u0627\u062f \u0627\u0644\u0645\u0631\u0643\u0628\u0629.", "math.gcf.complex": "\u0644\u0627 \u064a\u0645\u0643\u0646 \u062d\u0633\u0627\u0628 \"\u0627\u0644\u0639\u0627\u0645\u0644 \u0627\u0644\u0645\u0634\u062a\u0631\u0643 \u0627\u0644\u0623\u0643\u0628\u0631\" \u0644\u0623\u0639\u062f\u0627\u062f \u0627\u0644\u0645\u0631\u0643\u0628\u0629.", "math.lcm.complex": "\u0644\u0627 \u064a\u0645\u0643\u0646 \u062d\u0633\u0627\u0628 \"\u0627\u0644\u0645\u0636\u0627\u0639\u0641 \u0627\u0644\u0645\u0634\u062a\u0631\u0643 \u0627\u0644\u0623\u0635\u063a\u0631\" \u0644\u0623\u0639\u062f\u0627\u062f \u0645\u0631\u0643\u0628\u0629.", "math.lt.order complex numbers": "\u0644\u0627\u064a\u0645\u0643\u0646 \u062a\u0631\u062a\u064a\u0628 \u0627\u0644\u0623\u0639\u062f\u0627\u062f \u0627\u0644\u0645\u0631\u0643\u0628\u0629", "math.choose.empty selection": "\u062a\u0645 \u0625\u0639\u0637\u0627\u0621 \u0627\u0644\u062f\u0627\u0644\u0629 \u0627\u0644\u0639\u0634\u0648\u0627\u0626\u064a\u0629 \u0627\u062e\u062a\u064a\u0627\u0631 \u0641\u0627\u0631\u063a.", "matrixmath.abs.non-square": "\u0644\u0627 \u064a\u0645\u0643\u0646 \u062d\u0633\u0627\u0628 \u0645\u062d\u062f\u062f\u0629 \u0645\u0635\u0641\u0648\u0641\u0629 \u0644\u064a\u0633\u062a \u0645\u0631\u0628\u0639\u0629.", "matrixmath.abs.too big": "\u0639\u0630\u0631\u064b\u0627 \u060c \u0644\u0627 \u064a\u0645\u0643\u0646 \u0641\u064a \u0627\u0644\u0648\u0642\u062a \u0627\u0644\u062d\u0627\u0644\u064a \u062d\u0633\u0627\u0628 \u0645\u062d\u062f\u062f\u0629 \u0645\u0635\u0641\u0648\u0641\u0629 \u0645\u0642\u0627\u064a\u064a\u0633\u0647\u0627 \u0623\u0643\u0628\u0631 \u0645\u0646 3x3.", "matrixmath.mul.different sizes": "\u0644\u0627 \u064a\u0645\u0643\u0646 \u062c\u062f\u0627\u0621 \u0645\u0635\u0641\u0648\u0641\u0627\u062a \u0645\u062e\u062a\u0644\u0641\u0629 \u0627\u0644\u0623\u062d\u062c\u0627\u0645.", "vectormath.cross.not 3d": "\u064a\u0645\u0643\u0646 \u0623\u062e\u0630 \u0627\u0644\u0645\u0646\u062a\u062c \u0627\u0644\u0645\u062a\u0642\u0627\u0637\u0639 \u0644\u0646\u0627\u0642\u0644\u0627\u062a \u062b\u0644\u0627\u062b\u064a\u0629 \u0627\u0644\u0623\u0628\u0639\u0627\u062f \u0641\u0642\u0637.", "vectormath.dot.matrix too big": "\u0644\u0627 \u064a\u0645\u0643\u0646 \u062d\u0633\u0627\u0628 \u0627\u0644\u0645\u0646\u062a\u062c \u0627\u0644\u0646\u0642\u0637\u064a \u0644\u0645\u0635\u0641\u0648\u0641\u0629 \u0644\u064a\u0633\u062a $ 1 \\ times N $ \u0623\u0648 $ N \\ \u0645\u0636\u0631\u0648\u0628\u0629 $ 1.", "vectormath.cross.matrix too big": "\u0644\u0627 \u064a\u0645\u0643\u0646 \u062d\u0633\u0627\u0628 \u0627\u0644\u0645\u0646\u062a\u062c \u0627\u0644\u0645\u062a\u0642\u0627\u0637\u0639 \u0644\u0645\u0635\u0641\u0648\u0641\u0629 \u0644\u064a\u0633\u062a $ 1 \\ times N $ \u0623\u0648 $ N \\ \u0645\u0636\u0631\u0648\u0628\u0629 $ 1.", "part.with steps answer prompt": "\u0627\u0644\u0625\u062c\u0627\u0628\u0629:", "part.script.error": "\u062e\u0637\u0623 \u0641\u064a \u0627\u0644\u062c\u0632\u0621 {{path}} \u0627\u0644\u0628\u0631\u0646\u0627\u0645\u062c \u0627\u0644\u0646\u0635\u064a \u0627\u0644\u0645\u062e\u0635\u0635 {{script}}: {{-message}}", "part.marking.steps no matter": "\u0628\u0633\u0628\u0628 \u062d\u0635\u0648\u0644\u0643 \u0639\u0644\u0649 \u0639\u0644\u0627\u0645\u0627\u062a \u0643\u0627\u0645\u0644\u0629 \u0644\u0644\u062c\u0632\u0621\u060c \u0625\u062c\u0627\u0628\u0627\u062a\u0643 \u0639\u0644\u0649 \u0627\u0644\u062e\u0637\u0648\u0627\u062a \u0644\u0646 \u062a\u062d\u062a\u0633\u0628.", "part.marking.revealed steps no penalty": "\u0642\u0645\u062a \u0628\u0627\u0644\u0643\u0634\u0641 \u0639\u0646 \u0647\u0630\u0647 \u0627\u0644\u062e\u0637\u0648\u0629", "part.marking.used variable replacements": "\u062a\u0645 \u062a\u0635\u062d\u064a\u062d \u0647\u0630\u0627 \u0627\u0644\u0633\u0624\u0627\u0644 \u0628\u0646\u0627\u0621 \u0639\u0644\u0649 \u0625\u062c\u0627\u0628\u062a\u0643 \u0641\u064a \u062c\u0632\u0621 \u0633\u0627\u0628\u0642", "part.marking.variable replacement part not answered": "\u064a\u062c\u0628 \u0639\u0644\u064a\u0643 \u0623\u0648\u0644\u0627 \u0627\u0644\u0625\u062c\u0627\u0628\u0629 \u0639\u0646 {{\u0642\u0633\u0645}}.", "part.marking.resubmit because of variable replacement": "\u0639\u0644\u0627\u0645\u0627\u062a \u0647\u0630\u0627 \u0627\u0644\u062c\u0632\u0621 \u062a\u0639\u062a\u0645\u062f \u0639\u0644\u0649 \u0625\u062c\u0627\u0628\u0627\u062a\u0643 \u0639\u0644\u0649 \u0623\u062c\u0632\u0627\u0621 \u0623\u062e\u0631\u0649 \u0642\u0645\u062a \u0628\u062a\u063a\u064a\u064a\u0631\u0647\u0627. \u0623\u0631\u0633\u0644 \u0647\u0630\u0627 \u0627\u0644\u062c\u0632\u0621 \u0645\u0631\u0629 \u0623\u062e\u0631\u0649 \u0645\u0646 \u0623\u062c\u0644 \u062a\u062d\u062f\u064a\u062b \u062f\u0631\u062c\u0627\u062a\u0643.", "part.marking.not submitted": "\u0644\u0645 \u064a\u062a\u0645 \u0625\u0631\u0633\u0627\u0644 \u0627\u0644\u0625\u062c\u0627\u0628\u0629", "part.marking.did not answer": "\u0644\u0645 \u062a\u0642\u0645 \u0628\u0627\u0644\u0625\u062c\u0627\u0628\u0629 \u0639\u0644\u0649 \u0647\u0630\u0627 \u0627\u0644\u0633\u0624\u0627\u0644", "part.marking.nothing entered": "\u0644\u0645 \u062a\u0642\u0645 \u0628\u0625\u062f\u062e\u0627\u0644 \u0627\u0644\u0625\u062c\u0627\u0628\u0629", "part.marking.incorrect": "\u0625\u062c\u0627\u0628\u062a\u0643 \u063a\u064a\u0631 \u0635\u062d\u064a\u062d\u0629", "part.marking.correct": "\u0625\u062c\u0627\u0628\u0629 \u0635\u062d\u064a\u062d\u0629", "part.marking.uncaught error": "\u062e\u0637\u0623 \u0641\u064a \u0627\u0644\u0639\u0644\u0627\u0645\u0627\u062a {{-\u0631\u0633\u0627\u0644\u0629}}", "part.marking.no result": "This part could not be marked.", "part.correct answer": "\u0627\u0644\u0625\u062c\u0627\u0628\u0629 \u0627\u0644\u0645\u062a\u0648\u0642\u0639\u0629: ", "part.missing type attribute": "{{part}}: \u0633\u0645\u0629 \u0646\u0648\u0639 \u0627\u0644\u062c\u0632\u0621 \u0645\u0641\u0642\u0648\u062f\u0629", "part.unknown type": "{{part}}: \u0646\u0648\u0639 \u062c\u0632\u0621 \u063a\u064a\u0631 \u0645\u0639\u0631\u0648\u0641 {{type}}", "part.setting not present": "\u0644\u0645 \u064a\u062a\u0645 \u062a\u0639\u064a\u064a\u0646 \u0627\u0644\u062e\u0627\u0635\u064a\u0629 \"{{property}}\"", "part.jme.answer missing": "\u0627\u0644\u0625\u062c\u0627\u0628\u0629 \u0627\u0644\u0635\u062d\u064a\u062d\u0629 \u0645\u0641\u0642\u0648\u062f\u0629", "part.jme.answer too long": "\u0627\u0644\u0625\u062c\u0627\u0628\u0629 \u0623\u0637\u0648\u0644 \u0645\u0646 \u0627\u0644\u0645\u062a\u0648\u0642\u0639 ", "part.jme.answer too short": "\u0627\u0644\u0625\u062c\u0627\u0628\u0629 \u0623\u0642\u0635\u0631\u0645\u0646 \u0627\u0644\u0645\u062a\u0648\u0642\u0639 ", "part.jme.answer invalid": "\u0625\u062c\u0627\u0628\u062a\u0643 \u0644\u064a\u0633\u062a \u062a\u0639\u0628\u064a\u0631\u064b\u0627 \u0631\u064a\u0627\u0636\u064a\u064b\u0627 \u0635\u0627\u0644\u062d\u064b\u0627. <br/> {{-message}}.", "part.jme.marking.correct": "\u0627\u0644\u0625\u062c\u0627\u0628\u0629 \u063a\u064a\u0631 \u0635\u062d\u064a\u062d\u0629 \u0639\u062f\u062f\u064a\u0627", "part.jme.must-have bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.must-have one": "\u064a\u062c\u0628 \u0623\u0646 \u062a\u062d\u062a\u0648\u064a \u0625\u062c\u0627\u0628\u062a\u0643 \u0639\u0644\u0649: {{strings}}", "part.jme.must-have several": "\u0625\u062c\u0627\u0628\u062a\u0643 \u064a\u062c\u0628 \u0627\u0646 \u062a\u062d\u0648\u0644 \u0643\u0644 \u0627\u0644 {{strings}}", "part.jme.not-allowed bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.not-allowed one": "\u0625\u062c\u0627\u0628\u062a\u0643 \u0644\u0627 \u062a\u062d\u062a\u0648\u064a \u0639\u0644\u0649 {{strings}}", "part.jme.not-allowed several": "\u064a\u062c\u0628 \u0623\u0644\u0627 \u062a\u062d\u062a\u0648\u064a \u0625\u062c\u0627\u0628\u062a\u0643 \u0639\u0644\u0649 \u0623\u064a \u0645\u0646: {{strings}}", "part.jme.unexpected variable name": "\u062a\u0645 \u062a\u0641\u0633\u064a\u0631 \u0625\u062c\u0627\u0628\u062a\u0643 \u0644\u0627\u0633\u062a\u062e\u062f\u0627\u0645 \u0627\u0633\u0645 \u0627\u0644\u0645\u062a\u063a\u064a\u0631 \u063a\u064a\u0631 \u0627\u0644\u0645\u062a\u0648\u0642\u0639 <code> {{name}} </code>.", "part.jme.unexpected variable name suggestion": "\u062a\u0645 \u062a\u0641\u0633\u064a\u0631 \u0625\u062c\u0627\u0628\u062a\u0643 \u0644\u0627\u0633\u062a\u062e\u062f\u0627\u0645 \u0627\u0633\u0645 \u0627\u0644\u0645\u062a\u063a\u064a\u0631 \u063a\u064a\u0631 \u0627\u0644\u0645\u062a\u0648\u0642\u0639 <code> {{name}} </code>. \u0647\u0644 \u062a\u0642\u0635\u062f <code> {{Suggestion}} </code>\u061f", "part.patternmatch.display answer missing": "\u0644\u0645 \u064a\u062a\u0645 \u0639\u0631\u0636 \u0627\u0644\u0646\u062a\u0627\u0626\u062c", "part.patternmatch.correct except case": "\u0625\u062c\u0627\u0628\u062a\u0643 \u0635\u062d\u064a\u062d\u0629 \u060c \u0628\u0627\u0633\u062a\u062b\u0646\u0627\u0621 \u0627\u0644\u062d\u0627\u0644\u0629.", "part.numberentry.correct except decimal": "\u0625\u062c\u0627\u0628\u062a\u0643 \u062a\u0642\u0639 \u0636\u0645\u0646 \u0627\u0644\u0646\u0637\u0627\u0642 \u0627\u0644\u0645\u0633\u0645\u0648\u062d \u0628\u0647 \u060c \u0648\u0644\u0643\u0646 \u0627\u0644\u0623\u0631\u0642\u0627\u0645 \u0627\u0644\u0639\u0634\u0631\u064a\u0629 \u063a\u064a\u0631 \u0645\u0633\u0645\u0648\u062d \u0628\u0647\u0627.", "part.numberentry.correct except fraction": "\u0625\u062c\u0627\u0628\u062a\u0643 \u062a\u0642\u0639 \u0636\u0645\u0646 \u0627\u0644\u0646\u0637\u0627\u0642 \u0627\u0644\u0645\u0633\u0645\u0648\u062d \u0628\u0647 \u060c \u0648\u0644\u0643\u0646 \u0627\u0644\u0643\u0633\u0648\u0631 \u063a\u064a\u0631 \u0645\u0633\u0645\u0648\u062d \u0628\u0647\u0627.", "part.numberentry.answer invalid": "\u0644\u0645 \u062a\u0642\u0645 \u0628\u0625\u062f\u062e\u0627\u0644 \u0631\u0642\u0645 \u0635\u0627\u0644\u062d", "part.numberentry.answer not integer": "\u0625\u062c\u0627\u0628\u062a\u0643 \u063a\u064a\u0631 \u0635\u0627\u0644\u062d\u0629\u060c \u0642\u0645\u062a \u0628\u0625\u062f\u062e\u0627\u0644 \u0639\u062f\u062f \u0635\u062d\u064a\u062d \u0648 \u0644\u064a\u0633 \u0639\u062f\u062f \u0639\u0634\u0631\u064a", "part.numberentry.answer not integer or decimal": "\u0625\u062c\u0627\u0628\u062a\u0643 \u063a\u064a\u0631 \u0635\u0627\u0644\u062d\u0629. \u064a\u062c\u0628 \u0625\u062f\u062e\u0627\u0644 \u0639\u062f\u062f \u0635\u062d\u064a\u062d \u0623\u0648 \u0639\u062f\u062f \u0639\u0634\u0631\u064a.", "part.numberentry.zero sig fig": "\u062a\u0645 \u0625\u0639\u062f\u0627\u062f \u0647\u0630\u0627 \u0627\u0644\u062c\u0632\u0621 \u0644\u062a\u0642\u0631\u064a\u0628 \u0625\u062c\u0627\u0628\u0629 \u0627\u0644\u0637\u0627\u0644\u0628 \u0625\u0644\u0649 \u0635\u0641\u0631 \u0645\u0646 \u0627\u0644\u0623\u0631\u0642\u0627\u0645 \u0627\u0644\u0645\u0647\u0645\u0629 \u060c \u0648\u0627\u0644\u062a\u064a \u0644\u0627 \u0645\u0639\u0646\u0649 \u0644\u0647\u0627.", "part.mcq.options def not a list": "\u0627\u0644\u062a\u0639\u0628\u064a\u0631 \u0627\u0644\u0630\u064a \u064a\u0639\u0631\u0651\u0641 {{properties}} \u0644\u064a\u0633 \u0642\u0627\u0626\u0645\u0629.", "part.mcq.marking matrix string empty": "\u062a\u0639\u0628\u064a\u0631 \u0645\u0635\u0641\u0648\u0641\u0629 \u0627\u0644\u0639\u0644\u0627\u0645\u0627\u062a \u0627\u0644\u0645\u062e\u0635\u0635\u0629 \u0641\u0627\u0631\u063a .", "part.mcq.choices missing": "\u062a\u0639\u0631\u064a\u0641 \u0627\u0644\u0627\u062e\u062a\u064a\u0627\u0631\u0627\u062a \u0645\u0641\u0642\u0648\u062f", "part.mcq.matrix not a number": "\u0627\u0644\u062c\u0632\u0621 {{part}} \u0648\u0636\u0639 \u0639\u0644\u0627\u0645\u0629 \u0639\u0644\u0649 \u062e\u0644\u064a\u0629 \u0627\u0644\u0645\u0635\u0641\u0648\u0641\u0629 ({{row}} \u060c {{\u0627\u0644\u0639\u0645\u0648\u062f}}) \u0644\u0627 \u064a\u062a\u0645 \u062a\u0642\u064a\u064a\u0645\u0647 \u0625\u0644\u0649 \u0631\u0642\u0645", "part.mcq.wrong number of choices": "\u0627\u062e\u062a\u0631\u062a \u0639\u062f\u062f\u064b\u0627 \u062e\u0627\u0637\u0626\u064b\u0627 \u0645\u0646 \u0627\u0644\u062e\u064a\u0627\u0631\u0627\u062a.", "part.mcq.no choices selected": "\u0644\u0645 \u064a\u062a\u0645 \u0627\u0644\u0625\u062e\u062a\u064a\u0627\u0631 ", "part.mcq.matrix not a list": "\u0645\u0635\u0641\u0648\u0641\u0629 \u0627\u0644\u0648\u0633\u0645 \u060c \u0627\u0644\u0645\u0639\u0631\u0641\u0629 \u0628\u0648\u0627\u0633\u0637\u0629 \u062a\u0639\u0628\u064a\u0631 JME \u060c \u0644\u064a\u0633\u062a \u0642\u0627\u0626\u0645\u0629 \u0648\u0644\u0643\u0646 \u064a\u062c\u0628 \u0623\u0646 \u062a\u0643\u0648\u0646 \u0643\u0630\u0644\u0643.", "part.mcq.matrix wrong type": "\u0639\u0646\u0635\u0631 \u0645\u0646 \u0627\u0644\u0646\u0648\u0639 \u063a\u064a\u0631 \u0627\u0644\u0635\u0627\u0644\u062d '{{type}}' \u064a\u0633\u062a\u062e\u062f\u0645 \u0641\u064a \u0645\u0635\u0641\u0648\u0641\u0629 \u0627\u0644\u0648\u0633\u0645.", "part.mcq.matrix mix of numbers and lists": "\u0627\u0633\u062a\u062e\u062f\u0627\u0645 \u0645\u0632\u064a\u062c \u0645\u0646 \u0627\u0644\u0623\u0631\u0642\u0627\u0645 \u0648\u0627\u0644\u0642\u0648\u0627\u0626\u0645 \u0641\u064a \u0645\u0635\u0641\u0648\u0641\u0629 \u0627\u0644\u0639\u0644\u0627\u0645\u0627\u062a.", "part.mcq.matrix wrong size": "\u062e\u0637\u0623 \u0641\u064a \u062d\u062c\u0645 \u0645\u0635\u0641\u0648\u0641\u0629 \u0627\u0644\u0639\u0644\u0627\u0645\u0627\u062a.", "part.mcq.correct choice": "\u0642\u0645\u062a \u0628\u0625\u062e\u062a\u064a\u0627\u0631 \u0627\u0644\u0625\u062c\u0627\u0628\u0629 \u0627\u0644\u0635\u062d\u064a\u062d\u0629", "part.matrix.answer invalid": "\u0625\u062c\u0627\u0628\u062a\u0643 \u063a\u064a\u0631 \u0645\u0642\u0628\u0648\u0644\u0629", "part.matrix.invalid cell": "\u0648\u0627\u062d\u062f\u0629 \u0623\u0648 \u0623\u0643\u062b\u0631 \u0645\u0646 \u0627\u0644\u062e\u0644\u0627\u064a\u0627 \u0641\u064a \u0625\u062c\u0627\u0628\u062a\u0643 \u0641\u0627\u0631\u063a\u0629 \u0623\u0648 \u063a\u064a\u0631 \u0635\u0627\u0644\u062d\u0629.", "part.matrix.some incorrect": "\u062e\u0644\u064a\u0629 \u0648\u0627\u062d\u062f\u0629 \u0623\u0648 \u0623\u0643\u062b\u0631 \u0641\u064a \u0625\u062c\u0627\u0628\u062a\u0643 \u063a\u064a\u0631 \u0635\u062d\u064a\u062d\u0629 \u060c \u0648\u0644\u0643\u0646 \u062a\u0645 \u0645\u0646\u062d\u0643 \u0639\u0644\u0627\u0645\u0627\u062a \u0644\u0644\u0628\u0642\u064a\u0629.", "part.matrix.empty": "\u0644\u0645 \u062a\u0642\u0645 \u0628\u0625\u062e\u062a\u064a\u0627\u0631 \u0623\u064a \u0625\u062c\u0627\u0628\u0629", "part.matrix.empty cell": "\u062e\u0644\u064a\u0629 \u0623\u0648 \u0623\u0643\u062b\u0631 \u0641\u064a \u0625\u062c\u0627\u0628\u062a\u0643 \u063a\u064a\u0631 \u0645\u0643\u062a\u0645\u0644\u0629", "part.matrix.size mismatch": "\u0644\u0645 \u064a\u0633\u0645\u062d \u0645\u0624\u0644\u0641 \u0627\u0644\u0633\u0624\u0627\u0644 \u0644\u0644\u0637\u0627\u0644\u0628 \u0628\u062a\u062d\u062f\u064a\u062f \u0623\u0628\u0639\u0627\u062f \u0625\u062c\u0627\u0628\u062a\u0647 \u060c \u0644\u0643\u0646 \u0627\u0644\u0625\u062c\u0627\u0628\u0629 \u0627\u0644\u0635\u062d\u064a\u062d\u0629 \u0647\u064a {{correct_dimensions}} \u0628\u064a\u0646\u0645\u0627 \u0625\u062f\u062e\u0627\u0644 \u0627\u0644\u0625\u062c\u0627\u0628\u0629 \u0647\u0648 {{input_dimensions}}", "part.gapfill.feedback header": "<strong>{{name}}</strong>", "part.extension.not implemented": "\u0644\u0645 \u064a\u0646\u0641\u0630 \u0627\u0644\u062c\u0632\u0621 \u0637\u0631\u064a\u0642\u0629 <code> {{name}} </code>.", "question.loaded name mismatch": "\u0644\u0627 \u064a\u0645\u0643\u0646 \u0627\u0633\u062a\u0626\u0646\u0627\u0641 \u0647\u0630\u0647 \u0627\u0644\u0645\u062d\u0627\u0648\u0644\u0629 - \u0644\u0642\u062f \u062a\u063a\u064a\u0631\u062a \u0627\u0644\u062d\u0632\u0645\u0629 \u0645\u0646\u0630 \u0627\u0644\u062c\u0644\u0633\u0629 \u0627\u0644\u0623\u062e\u064a\u0631\u0629.", "question.error": "\u0627\u0644\u0633\u0624\u0627\u0644 {{number}}: {{-message}}", "question.preamble.error": "\n\u062e\u0637\u0623 \u0641\u064a \u0627\u0644\u062f\u064a\u0628\u0627\u062c\u0629: {{-message}}", "question.preamble.syntax error": "\u062e\u0637\u0623 \u0641\u064a \u0628\u0646\u0627\u0621 \u0627\u0644\u062c\u0645\u0644\u0629 \u0641\u064a \u0627\u0644\u062f\u064a\u0628\u0627\u062c\u0629", "question.unsupported part type": "\u0646\u0648\u0639 \u062c\u0632\u0626\u064a \u063a\u064a\u0631 \u0645\u062f\u0639\u0648\u0645", "question.header": "\u0633\u0624\u0627\u0644 {{number}}", "question.submit part": "\u0625\u0631\u0633\u0644 \u0627\u0644\u0625\u062c\u0627\u0628\u0629", "question.show steps": "\u0623\u0638\u0647\u0631 \u0627\u0644\u062e\u0637\u0648\u0627\u062a", "question.show steps penalty": "\n\u0633\u062a\u0641\u0642\u062f <strong> {{count\u060c niceNumber}} </strong> $ t (\u0639\u0644\u0627\u0645\u0629).", "question.show steps no penalty": "\u0644\u0646 \u062a\u062a\u0623\u062b\u0631 \u0646\u062a\u064a\u062c\u062a\u0643 \u0628\u0630\u0644\u0643", "question.show steps already penalised": "\u0644\u0642\u062f \u0642\u0645\u062a \u0645\u0646 \u0642\u0628\u0644 \u0628\u0639\u0631\u0636 \u0627\u0644\u062e\u0637\u0648\u0627\u062a. \u064a\u0645\u0643\u0646\u0643 \u0625\u0639\u0627\u062f\u0629 \u0625\u0638\u0647\u0627\u0631\u0647\u0645 \u0628\u062f\u0648\u0646 \u0639\u0642\u0648\u0628\u0629 \u0625\u0636\u0627\u0641\u064a\u0629.", "question.hide steps": "\u0623\u062e\u0641\u064a \u0627\u0644\u062e\u0637\u0648\u0627\u062a", "question.hide steps no penalty": "\u0644\u0646 \u062a\u062a\u0623\u062b\u0631 \u0646\u062a\u064a\u062c\u062a\u0643", "question.advice": "\u0646\u0635\u064a\u062d\u0629", "question.no such part": "\u0644\u0627 \u064a\u0645\u0643\u0646 \u0627\u0644\u0639\u062b\u0648\u0631 \u0639\u0644\u0649 \u0627\u0644\u062c\u0632\u0621 {{path}}", "question.can not submit": "\u0644\u0627 \u064a\u0645\u0643\u0646 \u0625\u0631\u0633\u0627\u0644 \u0625\u062c\u0627\u0628\u0629 - \u062a\u062d\u0642\u0642 \u0645\u0646 \u0627\u0644\u0623\u062e\u0637\u0627\u0621.", "question.answer submitted": "\u062a\u0645 \u0625\u0631\u0633\u0627\u0644 \u0627\u0644\u0646\u062a\u0627\u0626\u062c", "question.score feedback.show": "\u0627\u0639\u0631\u0636 \u0627\u0644\u062a\u063a\u0630\u064a\u0629 \u0627\u0644\u0631\u0627\u062c\u0639\u0629", "question.score feedback.hide": "\u0623\u062e\u0641 \u0627\u0644\u062a\u063a\u0630\u064a\u0629 \u0627\u0644\u0631\u0627\u062c\u0639\u0629", "question.score feedback.ascore total actual": "Score: {{score,niceNumber}}/{{marks,niceNumber}}", "question.score feedback.score total": "{{marksString}}", "question.score feedback.score actual": "\u0627\u0644\u0646\u062a\u064a\u062c\u0629: {{ScoreString}}", "question.score feedback.answered": "\u062a\u0645\u062a \u0627\u0644\u0625\u062c\u0627\u0628\u0629 \u0639\u0644\u064a\u0647", "question.score feedback.unanswered": "\u0644\u0645 \u064a\u062a\u0645 \u0627\u0644\u0625\u062c\u0627\u0628\u0629 \u0639\u0644\u064a\u0647", "question.score feedback.correct": "\u0625\u062c\u0627\u0628\u062a\u0643 \u0635\u062d\u064a\u062d\u0629", "question.score feedback.partial": "\u0625\u062c\u0627\u0628\u0629 \u0635\u062d\u064a\u062d\u0629 \u062c\u0632\u0626\u064a\u0627", "question.score feedback.wrong": "\u0625\u062c\u0627\u0628\u062a\u0643 \u063a\u064a\u0631 \u0635\u062d\u064a\u062d\u0629", "question.selector.unsubmitted changes": "\u0644\u0645 \u064a\u062a\u0645 \u0625\u0631\u0633\u0627\u0644 \u0627\u0644\u0646\u062a\u0627\u0626\u062c", "timing.no accumulator": "\n\u0644\u0627 \u064a\u0648\u062c\u062f \u062a\u0631\u0627\u0643\u0645 \u062a\u0648\u0642\u064a\u062a {{name}}", "timing.time remaining": "\u0627\u0644\u0648\u0642\u062a \u0627\u0644\u0645\u062a\u0628\u0642\u064a", "xml.could not load": "\u062a\u0639\u0630\u0631 \u062a\u062d\u0645\u064a\u0644 \u0645\u0633\u062a\u0646\u062f XML: {{-message}}", "xml.property not number": "\u064a\u062c\u0628 \u0623\u0646 \u062a\u0643\u0648\u0646 \u0627\u0644\u062e\u0627\u0635\u064a\u0629 {{name}} \u0631\u0642\u0645\u064b\u0627 \u060c \u0648\u0644\u0643\u0646\u0647\u0627 \u0644\u064a\u0633\u062a ({{value}}) \u060c \u0641\u064a \u0627\u0644\u0639\u0642\u062f\u0629 {{element}}", "xml.property not boolean": "\u064a\u062c\u0628 \u0623\u0646 \u062a\u0643\u0648\u0646 \u0627\u0644\u062e\u0627\u0635\u064a\u0629 {{name}} \u0645\u0646\u0637\u0642\u064a\u0629 \u060c \u0648\u0644\u0643\u0646\u0647\u0627 \u0644\u064a\u0633\u062a ({{value}}) \u060c \u0641\u064a \u0627\u0644\u0639\u0642\u062f\u0629 {{element}}", "xml.error in variable definition": "Error in definition of variable <code>{{name}}</code>", "scorm.error initialising": "\u062e\u0637\u0623 \u0641\u064a \u062a\u0647\u064a\u0626\u0629 \u0628\u0631\u0648\u062a\u0648\u0643\u0648\u0644 SCORM: {{-message}}", "scorm.failed save": "<p> \u0641\u0634\u0644 \u0637\u0644\u0628 \u062d\u0641\u0638 \u0627\u0644\u0628\u064a\u0627\u0646\u0627\u062a \u0639\u0644\u0649 \u0627\u0644\u062e\u0627\u062f\u0645. \u0627\u0636\u063a\u0637 \u0639\u0644\u0649 <b> OK </b> \u0644\u0644\u0645\u062d\u0627\u0648\u0644\u0629 \u0645\u0631\u0629 \u0623\u062e\u0631\u0649. </ p>\n<p> \u0625\u0630\u0627 \u062a\u0644\u0642\u064a\u062a \u0647\u0630\u0647 \u0627\u0644\u0631\u0633\u0627\u0644\u0629 \u0628\u0634\u0643\u0644 \u0645\u062a\u0643\u0631\u0631 \u060c \u0641\u062a\u062d\u0642\u0642 \u0645\u0646 \u0627\u062a\u0635\u0627\u0644 \u0627\u0644\u0625\u0646\u062a\u0631\u0646\u062a \u0623\u0648 \u0627\u0633\u062a\u062e\u062f\u0645 \u062c\u0647\u0627\u0632 \u0643\u0645\u0628\u064a\u0648\u062a\u0631 \u0645\u062e\u062a\u0644\u0641\u064b\u0627. \u062a\u0645 \u062d\u0641\u0638 \u0625\u062c\u0627\u0628\u0627\u062a\u0643 \u0627\u0644\u0645\u0631\u0633\u0644\u0629 \u0645\u0633\u0628\u0642\u064b\u0627 \u0628\u0646\u062c\u0627\u062d \u0648\u0633\u062a\u062a\u0645 \u0627\u0633\u062a\u0639\u0627\u062f\u062a\u0647\u0627 \u0625\u0630\u0627 \u0627\u0633\u062a\u0623\u0646\u0641\u062a \u0647\u0630\u0647 \u0627\u0644\u062c\u0644\u0633\u0629 \u0639\u0644\u0649 \u062c\u0647\u0627\u0632 \u0643\u0645\u0628\u064a\u0648\u062a\u0631 \u0645\u062e\u062a\u0644\u0641. </ p>\n<p> \u0625\u0630\u0627 \u0638\u0647\u0631\u062a \u0647\u0630\u0647 \u0627\u0644\u0631\u0633\u0627\u0644\u0629 \u0628\u0627\u0633\u062a\u0645\u0631\u0627\u0631 \u0648\u0644\u0627 \u064a\u0645\u0643\u0646\u0643 \u062d\u0641\u0638 <em> \u0623\u064a \u0625\u062c\u0627\u0628\u0627\u062a </ em> \u060c \u0641\u064a\u0631\u062c\u0649 \u0627\u0644\u0627\u062a\u0635\u0627\u0644 \u0628\u0627\u0644\u0645\u062d\u0627\u0636\u0631 \u0623\u0648 \u200b\u200b\u0627\u0644\u0645\u0639\u0644\u0645. </ p>", "scorm.no exam suspend data": "\u0641\u0634\u0644\u062a \u0625\u0639\u0627\u062f\u0629 \u062a\u062d\u0645\u064a\u0644 \u0627\u0644\u0625\u062e\u062a\u0628\u0627\u0631: \u0644\u0627\u064a\u0648\u062c\u062f \u0628\u064a\u0627\u0646\u0627\u062a \u0645\u0639\u0644\u0642\u0629 \u0644\u0644\u0625\u062e\u062a\u0628\u0627\u0631", "scorm.error loading suspend data": "\u062e\u0637\u0623 \u0641\u064a \u062a\u062d\u0645\u064a\u0644 \u0628\u064a\u0627\u0646\u0627\u062a \u0627\u0644\u062a\u0639\u0644\u064a\u0642: {{-message}}", "scorm.error loading question": "\n\u062e\u0637\u0623 \u0641\u064a \u062a\u062d\u0645\u064a\u0644 \u0627\u0644\u0633\u0624\u0627\u0644 {{number}}: {{-message}}", "scorm.no question suspend data": "\u0644\u0627\u064a\u0648\u062c\u062f \u0628\u064a\u0627\u0646\u0627\u062a \u0645\u0639\u0644\u0642\u0629 \u0644\u0644\u0633\u0624\u0627\u0644", "scorm.error loading part": "\u062e\u0637\u0623 \u0641\u064a \u062a\u062d\u0645\u064a\u0644 \u0627\u0644\u062c\u0632\u0621 {{part}}: {{-message}}", "scorm.no part suspend data": "\u0644\u0627\u064a\u0648\u062c\u062f \u0628\u064a\u0627\u0646\u0627\u062a \u0645\u0639\u0644\u0642\u0629 \u0644\u0647\u0630\u0627 \u0627\u0644\u062c\u0632\u0621", "util.product.non list": "\u0645\u0627 \u062a\u0645 \u062a\u0645\u0631\u064a\u0631\u0647 \u0644\u064a\u0633 \u0642\u0627\u0626\u0645\u0629 \u0625\u0644\u0649 <code> Numbas.util.product </code>", "mark": "\u062f\u0631\u062c\u0629", "was": "\u0643\u0627\u0646", "part": "\u062c\u0632\u0621", "gap": "\u0641\u0631\u0627\u063a", "step": "\u062e\u0637\u0648\u0629", "jme.substituteTree.undefined variable": "\u0645\u062a\u063a\u064a\u0631 \u063a\u064a\u0631 \u0645\u062d\u062f\u062f: <code> {{name}} </code>", "jme.user javascript.returned undefined": "\u062a\u0645 \u0625\u0631\u062c\u0627\u0639 \u0648\u0638\u064a\u0641\u0629 \u062c\u0627\u0641\u0627 \u0633\u0643\u0631\u064a\u0628\u062a \u0627\u0644\u0645\u0639\u0631\u0641\u0629 \u0628\u0648\u0627\u0633\u0637\u0629 \u0627\u0644\u0645\u0633\u062a\u062e\u062f\u0645 <code> {{name}} </code> <code> undefined </code>.", "part.marking.steps change": "\u062a\u0645 \u0645\u0646\u062d\u0643 <strong> {{count\u060c niceNumber}} </strong> $ t (\u0639\u0644\u0627\u0645\u0629) \u0645\u0642\u0627\u0628\u0644 \u0625\u062c\u0627\u0628\u0627\u062a\u0643 \u0639\u0644\u0649 \u0627\u0644\u062e\u0637\u0648\u0627\u062a.", "part.marking.revealed steps with penalty": "You revealed the steps. The maximum you can score for this part is <strong>{{count,niceNumber}}</strong> $t(mark). Your scores will be scaled down accordingly.", "part.marking.total score": "\u0644\u0642\u062f \u0623\u062d\u0631\u0632\u062a <strong> {{count\u060c niceNumber}} </strong> $ t (\u0639\u0644\u0627\u0645\u0629) \u0641\u064a \u0647\u0630\u0627 \u0627\u0644\u062c\u0632\u0621.", "part.numberentry.precision type.dp": "\u062e\u0627\u0646\u0629 \u0639\u0634\u0631\u064a\u0629", "part.numberentry.precision type.dp_plural": "\u062e\u0627\u0646\u0627\u062a \u0639\u0634\u0631\u064a\u0629", "part.numberentry.precision type.sigfig": "\u0631\u0642\u0645 \u0630\u0648 \u0627\u0647\u0645\u064a\u0629", "part.numberentry.precision type.sigfig_plural": "\u0623\u0631\u0642\u0627\u0645 \u0645\u0647\u0645\u0629", "part.numberentry.give your answer to precision": "\u062a\u0642\u0631\u064a\u0628 \u0625\u062c\u0627\u0628\u062a\u0643 \u0625\u0644\u0649 {{count\u060c niceNumber}} {{\u0627\u0644\u062f\u0642\u0629Type}}.", "question.unsubmitted changes": "\u0644\u0642\u062f \u0642\u0645\u062a \u0628\u062a\u063a\u064a\u064a\u0631 \u0625\u062c\u0627\u0628\u062a\u0643 \u0648\u0644\u0643\u0646 \u0644\u0645 \u062a\u0631\u0633\u0644\u0647\u0627. \u064a\u0631\u062c\u0649 \u0627\u0644\u062a\u062d\u0642\u0642 \u0645\u0646 \u0625\u062c\u0627\u0628\u062a\u0643 \u062b\u0645 \u0627\u0644\u0636\u063a\u0637 \u0639\u0644\u0649 \u0632\u0631 <strong> \u0625\u0631\u0633\u0627\u0644 \u0625\u062c\u0627\u0628\u0629 </ strong>.", "question.unsubmitted changes_plural": "\u0644\u0642\u062f \u0623\u062c\u0631\u064a\u062a \u062a\u063a\u064a\u064a\u0631\u0627\u062a \u0639\u0644\u0649 \u0625\u062c\u0627\u0628\u0627\u062a\u0643 \u0648\u0644\u0643\u0646 \u0644\u0645 \u062a\u0631\u0633\u0644\u0647\u0627. \u064a\u0631\u062c\u0649 \u0627\u0644\u062a\u062d\u0642\u0642 \u0645\u0646 \u0625\u062c\u0627\u0628\u0627\u062a\u0643 \u0644\u0643\u0644 \u062c\u0632\u0621 \u062b\u0645 \u0627\u0644\u0636\u063a\u0637 \u0639\u0644\u0649 \u0627\u0644\u0632\u0631 <strong> \u0625\u0631\u0633\u0627\u0644 \u062c\u0645\u064a\u0639 \u0627\u0644\u0623\u062c\u0632\u0627\u0621 </ strong>.", "util.equality not defined for type": "\u0644\u0645 \u064a\u062a\u0645 \u062a\u062d\u062f\u064a\u062f \u0627\u0644\u0645\u0633\u0627\u0648\u0627\u0629 \u0644\u0644\u0646\u0648\u0639 {{type}}", "mark_plural": "\u062f\u0631\u062c\u0627\u062a", "was_plural": "\u0643\u0627\u0646\u062a", "die.script not loaded": "\u062a\u0639\u0630\u0631 \u0628\u062f\u0621 Numbas \u0644\u0623\u0646\u0647 \u0644\u0645 \u064a\u062a\u0645 \u062a\u062d\u0645\u064a\u0644 \u0627\u0644\u0645\u0644\u0641 <code> {{file}} </code>. \u062a\u062d\u0642\u0642 \u0645\u0646 \u0623\u0646\u0647 \u0645\u062f\u0631\u062c \u0641\u064a <code> scripts.js </code>.", "math.combinations.n less than zero": "\u0644\u0627\u064a\u0645\u0643\u0646 \u062d\u0633\u0627\u0628 \u0627\u0644\u062a\u0648\u0627\u0641\u064a\u0642 \u0644\u0623\u0646 \u0642\u064a\u0645\u0629 \u0646 \u0623\u0635\u063a\u0631 \u0645\u0646 \u0635\u0641\u0631", "math.combinations.k less than zero": "\u0644\u0627\u064a\u0645\u0643\u0646 \u062d\u0633\u0627\u0628 \u0627\u0644\u062a\u0648\u0627\u0641\u064a\u0642 \u0644\u0623\u0646 \u0642\u064a\u0645\u0629 \u0643 \u0623\u0635\u063a\u0631 \u0645\u0646 \u0635\u0641\u0631", "math.combinations.n less than k": "\u0644\u0627\u064a\u0645\u0643\u0646 \u062d\u0633\u0627\u0628 \u0627\u0644\u062a\u0648\u0627\u0641\u064a\u0642 \u0644\u0623\u0646 \u0642\u064a\u0645\u0629 \u0646 \u0623\u0635\u063a\u0631 \u0645\u0646 \u0643", "math.permutations.n less than zero": "\u0644\u0627\u064a\u0645\u0643\u0646 \u062d\u0633\u0627\u0628 \u0627\u0644\u062a\u0628\u0627\u062f\u064a\u0644 \u0644\u0623\u0646 \u0642\u064a\u0645\u0629 \u0646 \u0623\u0635\u063a\u0631 \u0645\u0646 \u0635\u0641\u0631", "math.permutations.k less than zero": "\u0644\u0627\u064a\u0645\u0643\u0646 \u062d\u0633\u0627\u0628 \u0627\u0644\u062a\u0628\u0627\u062f\u064a\u0644 \u0644\u0623\u0646 \u0642\u064a\u0645\u0629 \u0643 \u0623\u0635\u063a\u0631 \u0645\u0646 \u0635\u0641\u0631", "math.permutations.n less than k": "\u0644\u0627\u064a\u0645\u0643\u0646 \u062d\u0633\u0627\u0628 \u0627\u0644\u062a\u0628\u0627\u062f\u064a\u0644 \u0644\u0623\u0646 \u0642\u064a\u0645\u0629 \u0646 \u0623\u0635\u063a\u0631 \u0645\u0646 \u0643", "part.numberentry.give your answer to precision_0": "\u0642\u0631\u0628 \u0625\u062c\u0627\u0628\u062a\u0643 \u0625\u0644\u0649 \u0623\u0642\u0631\u0628 \u0639\u062f\u062f \u0635\u062d\u064a\u062d", "mathjax.error": "\u062e\u0637\u0623 \u0641\u064a \u0645\u0639\u0627\u0644\u062c\u0629 MathJax: {{-message}}", "mathjax.error with context": "\u062e\u0637\u0623 \u0641\u064a \u0645\u0639\u0627\u0644\u062c\u0629 MathJax \u0641\u064a {{-context}}: {{-message}}", "exam.introduction": "\u0645\u0642\u062f\u0645\u0629 \u0627\u0644\u0625\u062e\u062a\u0628\u0627\u0631", "exam.feedback": "\u0631\u0633\u0627\u0644\u0629 \u0627\u0644\u0645\u0644\u0627\u062d\u0638\u0627\u062a \u0639\u0646 \u0627\u0644\u0627\u0645\u062a\u062d\u0627\u0646", "jme.tokenise.keypair key not a string": "\u064a\u062c\u0628 \u0623\u0646 \u064a\u0643\u0648\u0646 \u0645\u0641\u062a\u0627\u062d \u0627\u0644\u0642\u0627\u0645\u0648\u0633 \u0633\u0644\u0633\u0644\u0629 \u0648\u0644\u064a\u0633 {{type}}.", "jme.shunt.list mixed argument types": "\u0644\u0627 \u064a\u0645\u0643\u0646 \u062a\u062d\u0644\u064a\u0644 {{mode}}: \u0645\u0632\u064a\u062c \u0645\u0646 \u0627\u0644\u0642\u0627\u0645\u0648\u0633 \u0648\u0639\u0646\u0627\u0635\u0631 \u0627\u0644\u0642\u0627\u0626\u0645\u0629", "jme.func.listval.key not in dict": "\u0644\u0627 \u064a\u062d\u062a\u0648\u064a \u0627\u0644\u0642\u0627\u0645\u0648\u0633 \u0639\u0644\u0649 \u0627\u0644\u0645\u0641\u062a\u0627\u062d <code> {{key}} </code>", "part.prompt": "\u0623\u0645\u0631", "part.feedback": "\u062a\u063a\u0630\u064a\u0629 \u0631\u0627\u062c\u0639\u0629", "part.numberentry.answer not reduced": "\u0625\u062c\u0627\u0628\u062a\u0643 \u0644\u064a\u0633\u062a \u0641\u064a \u0627\u0644\u0635\u064a\u063a\u0629 \u0627\u0644\u0645\u062e\u062a\u0635\u0631\u0629", "part.numberentry.give your answer as a reduced fraction": "\u0623\u0643\u062a\u0628 \u0627\u0644\u0625\u062c\u0627\u0628\u0629 \u0641\u064a \u0623\u0628\u0633\u0637 \u0635\u0648\u0631\u0629 \u0645\u0645\u0643\u0646\u0629", "part.numberentry.negative decimal places": "\u062a\u0645 \u0625\u0639\u062f\u0627\u062f \u0647\u0630\u0627 \u0627\u0644\u062c\u0632\u0621 \u0644\u064a\u0642\u0648\u0645 \u0628\u062a\u0642\u0631\u064a\u0628 \u0625\u062c\u0627\u0628\u0629 \u0627\u0644\u0637\u0627\u0644\u0628 \u0625\u0644\u0649 \u0639\u062f\u062f \u0633\u0627\u0644\u0628 \u0645\u0646 \u0627\u0644\u0645\u0646\u0627\u0632\u0644 \u0627\u0644\u0639\u0634\u0631\u064a\u0629\u060c \u0648\u0627\u0644\u062a\u064a \u0644\u0627 \u0645\u0639\u0646\u0649 \u0644\u0647\u0627.", "part.mcq.choices": "\u0625\u062e\u062a\u064a\u0627\u0631\u0627\u062a", "part.mcq.answers": "\u0625\u062c\u0627\u0628\u0627\u062a", "part.mcq.matrix cell empty": "\u0627\u0644\u062c\u0632\u0621 {{part}} \u0648\u0636\u0639 \u0639\u0644\u0627\u0645\u0629 \u0639\u0644\u0649 \u062e\u0644\u064a\u0629 \u0627\u0644\u0645\u0635\u0641\u0648\u0641\u0629 ({{row}} \u060c {{\u0639\u0645\u0648\u062f}}) \u0641\u0627\u0631\u063a", "part.mcq.matrix jme error": "\u064a\u0642\u062f\u0645 \u0627\u0644\u062c\u0632\u0621 {{part}} \u062e\u0644\u064a\u0629 \u0645\u0635\u0641\u0648\u0641\u0629 \u0648\u0633\u0645 ({{row}} \u060c {{\u0639\u0645\u0648\u062f}}) \u062e\u0637\u0623 JME: {{-error}}", "question.statement": "\u062c\u0645\u0644\u0629", "ruleset.circular reference": "\u0645\u0631\u062c\u0639 \u062f\u0627\u0626\u0631\u064a \u0641\u064a \u062a\u0639\u0631\u064a\u0641 \u0645\u062c\u0645\u0648\u0639\u0629 \u0627\u0644\u0642\u0648\u0627\u0639\u062f <code> {{name}} </code>", "ruleset.set not defined": "\u0644\u0645 \u064a\u062a\u0645 \u062a\u062d\u062f\u064a\u062f \u0645\u062c\u0645\u0648\u0639\u0629 \u0627\u0644\u0642\u0648\u0627\u0639\u062f {{name}}", "jme.evaluate.no scope given": "\u064a\u062c\u0628 \u0625\u0639\u0637\u0627\u0621 Numbas.jme.evaluate \u0646\u0637\u0627\u0642\u064b\u0627\n", "question.score feedback.answered total actual": "Score: {{score,niceNumber}}/{{marks,niceNumber}}", "question.score feedback.answered total": "{{marksString}}. Answered.", "question.score feedback.answered actual": "Score: {{scoreString}}", "question.score feedback.unanswered total": "{{marksString}}.", "answer.number.not a number": "\u0625\u062c\u0627\u0628\u062a\u0643 \u0644\u064a\u0633\u062a \u0631\u0642\u0645\u064b\u0627 \u0635\u0627\u0644\u062d\u064b\u0627.", "answer.number.fractions not allowed": "\u0644\u0627 \u064a\u0645\u0643\u0646\u0643 \u0625\u062f\u062e\u0627\u0644 \u0643\u0633\u0631.", "answer.jme.invalid expression": "{{-message}}\n", "answer.matrix.fractions not allowed": "\u0644\u0627 \u064a\u0645\u0643\u0646\u0643 \u0625\u062f\u062e\u0627\u0644 \u0643\u0633\u0648\u0631.", "answer.matrix.some cell not a number": "\u0648\u0627\u062d\u062f\u0629 \u0623\u0648 \u0623\u0643\u062b\u0631 \u0645\u0646 \u0627\u0644\u062e\u0644\u0627\u064a\u0627 \u0641\u064a \u0625\u062c\u0627\u0628\u062a\u0643 \u0644\u0627 \u062a\u062d\u0645\u0644 \u0631\u0642\u0645\u064b\u0627 \u0635\u0627\u0644\u062d\u064b\u0627.", "exam.enter password": "\u0643\u0644\u0645\u0629 \u0627\u0644\u0645\u0631\u0648\u0631:", "exam.password.correct": "\u0643\u0644\u0645\u0629 \u0627\u0644\u0645\u0631\u0648\u0631 \u0635\u062d\u064a\u062d\u0629\u060c \u064a\u0645\u0643\u0646\u0643 \u0628\u062f\u0621 \u0627\u0644\u0625\u062e\u062a\u0628\u0627\u0631.", "exam.password.incorrect": "\u0643\u0644\u0645\u0629 \u0627\u0644\u0645\u0631\u0648\u0631 \u0647\u0630\u0647 \u063a\u064a\u0631 \u0635\u062d\u064a\u062d\u0629.", "frontpage.scorm.lms not connected": "\u0647\u0630\u0627 \u0627\u0644\u0627\u062e\u062a\u0628\u0627\u0631 \u064a\u0639\u0645\u0644 \u0627\u0644\u0622\u0646 \u0641\u064a \"\u0627\u0644\u0648\u0636\u0639\u064a\u0629 \u0627\u0644\u0645\u0633\u062a\u0642\u0644\u0629\". \u0644\u0646 \u064a\u062a\u0645 \u062d\u0641\u0638 \u0625\u062c\u0627\u0628\u0627\u062a\u0643 \u0648\u0639\u0644\u0627\u0645\u0627\u062a\u0643!", "result.question review": "\u0645\u0631\u0627\u062c\u0639\u0629", "control.confirm regen no marks": "\u0647\u0644 \u062a\u0631\u063a\u0628 \u0628\u0625\u0639\u0627\u062f\u0629 \u062e\u0644\u0637 \u0627\u0644\u0633\u0624\u0627\u0644 \u0639\u0634\u0648\u0627\u0626\u064a\u0627\u061f", "control.confirm reveal no marks": "\u0647\u0644 \u062a\u0631\u063a\u0628 \u0641\u064a \u0643\u0634\u0641 \u0625\u062c\u0627\u0628\u0629 \u0647\u0630\u0627 \u0627\u0644\u0633\u0624\u0627\u0644\u061f", "jme.tokenise.invalid near": "\u062a\u0639\u0628\u064a\u0631 \u063a\u064a\u0631 \u0635\u0627\u0644\u062d: <code> {{\u062a\u0639\u0628\u064a\u0631}} </ code> \u0641\u064a \u0627\u0644\u0645\u0648\u0636\u0639 {{position}} \u0628\u0627\u0644\u0642\u0631\u0628 \u0645\u0646 <code> {{nearby}} </code>", "jme.tokenise.number.object not complex": "\u062a\u0645 \u062a\u0645\u0631\u064a\u0631 \u063a\u0631\u0636 \u063a\u064a\u0631 \u0635\u0627\u0644\u062d \u0625\u0644\u0649 \u0645\u064f\u0646\u0634\u0626 \u0627\u0644\u0631\u0642\u0645.", "jme.subvars.null substitution": "\u0627\u0633\u062a\u0628\u062f\u0627\u0644 \u0627\u0644\u0645\u062a\u063a\u064a\u0631 \u0627\u0644\u0641\u0627\u0631\u063a: <code> $ t (\u0642\u0648\u0633 \u0623\u064a\u0633\u0631) {{str}} $ t (\u0642\u0648\u0633 \u0623\u064a\u0645\u0646)", "jme.type.type already registered": "\u062a\u0645 \u062a\u0633\u062c\u064a\u0644 \u0646\u0648\u0639 \u0627\u0644\u0628\u064a\u0627\u0646\u0627\u062a {{type}} \u0628\u0627\u0644\u0641\u0639\u0644 \u060c \u0644\u0630\u0627 \u0644\u0627 \u064a\u0645\u0643\u0646 \u0627\u0644\u062a\u0633\u062c\u064a\u0644 \u0645\u0631\u0629 \u0623\u062e\u0631\u0649.\n", "jme.type.no cast method": "\u0644\u0627 \u064a\u0645\u0643\u0646 \u0627\u0644\u062a\u062d\u0648\u064a\u0644 \u062a\u0644\u0642\u0627\u0626\u064a\u064b\u0627 \u0645\u0646 {{from}} \u0625\u0644\u0649 {{to}}.", "jme.display.simplifyTree.empty expression": "\u0627\u0644\u062a\u0639\u0628\u064a\u0631 \u0641\u0627\u0631\u063a", "jme.display.simplifyTree.stuck in a loop": "\u0627\u0644\u0645\u0639\u0644\u0642 \u0639\u0627\u0644\u0642 \u0641\u064a \u062d\u0644\u0642\u0629: <code> {{expr}} </code>\n", "math.niceNumber.undefined": "\u0643\u0627\u0646 \u064a\u062a\u0648\u0642\u0639 \u0631\u0642\u0645\u064b\u0627 \u060c \u0648\u0644\u0643\u0646 \u062a\u0645 \u0627\u0644\u062d\u0635\u0648\u0644 \u0639\u0644\u0649 <code> \u063a\u064a\u0631 \u0645\u0639\u0631\u0641 </ code>", "math.rangeToList.zero step size": "\u0644\u0627 \u064a\u0645\u0643\u0646 \u062a\u062d\u0648\u064a\u0644 \u0646\u0637\u0627\u0642 \u062d\u062c\u0645 \u062e\u0637\u0648\u062a\u0647 \u0635\u0641\u0631\u060c \u0625\u0644\u0649 \u0642\u0627\u0626\u0645\u0629.", "part.error": "{{path}}: {{-message}}\n", "part.marking.revealed steps": "\u0642\u0645\u062a \u0628\u0627\u0644\u0643\u0634\u0641 \u0639\u0646 \u0627\u0644\u062e\u0637\u0648\u0627\u062a", "part.marking.maximum scaled down": "\u0627\u0644\u062d\u062f \u0627\u0644\u0623\u0642\u0635\u0649 \u0627\u0644\u0630\u064a \u064a\u0645\u0643\u0646\u0643 \u062a\u0633\u062c\u064a\u0644\u0647 \u0644\u0647\u0630\u0627 \u0627\u0644\u062c\u0632\u0621 \u0647\u0648 <strong> {{count\u060c niceNumber}} </strong> $ t (\u0639\u0644\u0627\u0645\u0629). \u0633\u064a\u062a\u0645 \u062a\u062e\u0641\u064a\u0636 \u062f\u0631\u062c\u0627\u062a\u0643 \u0648\u0641\u0642\u064b\u0627 \u0644\u0630\u0644\u0643.", "part.marking.minimum score applied": "\u0627\u0644\u062d\u062f \u0627\u0644\u0623\u062f\u0646\u0649 \u0645\u0646 \u0627\u0644\u0646\u0642\u0627\u0637 \u0641\u064a \u0647\u0630\u0627 \u0627\u0644\u062c\u0632\u0621 \u0647\u0648 <strong> {{Score\u060c niceNumber}} </strong>.", "part.marking.maximum score applied": "\u0627\u0644\u062f\u0631\u062c\u0629 \u0627\u0644\u0642\u0635\u0648\u0649 \u0644\u0647\u0630\u0627 \u0627\u0644\u062c\u0632\u0621 \u0647\u064a <strong> {{Score\u060c niceNumber}} </strong>.\n", "part.marking.error in marking script": "\u062d\u062f\u062b \u062e\u0637\u0623 \u0641\u064a \u062e\u0648\u0627\u0631\u0632\u0645\u064a\u0629 \u0648\u0636\u0639 \u0627\u0644\u0639\u0644\u0627\u0645\u0627\u062a \u0644\u0647\u0630\u0627 \u0627\u0644\u062c\u0632\u0621. \u064a\u0631\u062c\u0649 \u0627\u0644\u0625\u0628\u0644\u0627\u063a \u0639\u0646 \u0647\u0630\u0627. {{-\u0631\u0633\u0627\u0644\u0629}}\n", "part.marking.no result after replacement": "\u0644\u0627 \u064a\u0645\u0643\u0646 \u062a\u0645\u064a\u064a\u0632 \u0647\u0630\u0627 \u0627\u0644\u062c\u0632\u0621 \u0628\u0627\u0633\u062a\u062e\u062f\u0627\u0645 \u0625\u062c\u0627\u0628\u0627\u062a\u0643 \u0639\u0644\u0649 \u0627\u0644\u0623\u062c\u0632\u0627\u0621 \u0627\u0644\u0633\u0627\u0628\u0642\u0629.\n", "part.marking.missing required note": "\u0644\u0627 \u062a\u062d\u062f\u062f \u062e\u0648\u0627\u0631\u0632\u0645\u064a\u0629 \u0648\u0636\u0639 \u0627\u0644\u0639\u0644\u0627\u0645\u0627\u062a \u0627\u0644\u0645\u0644\u0627\u062d\u0638\u0629 <code> {{note}} </code>", "marking.apply.not a list": "\u064a\u062c\u0628 \u0623\u0646 \u062a\u0643\u0648\u0646 \u0627\u0644\u0648\u0633\u064a\u0637\u0629 \u0627\u0644\u0623\u0648\u0644\u0649 \u0644\u0640 <code> \u062a\u0637\u0628\u064a\u0642 </code> \u0642\u0627\u0626\u0645\u0629 \u0648\u0644\u064a\u0633\u062a \u0643\u0630\u0644\u0643", "marking.apply marking script.script not found": "\u0627\u0644\u0646\u0635 \u0627\u0644\u0628\u0631\u0645\u062c\u064a \u0644\u0644\u0639\u0644\u0627\u0645\u0629 <code> {{name}} </code> \u063a\u064a\u0631 \u0645\u0648\u062c\u0648\u062f", "marking.note.compilation error": "\u062e\u0637\u0623 \u0641\u064a \u062a\u062c\u0645\u064a\u0639 \u0627\u0644\u0645\u0644\u0627\u062d\u0638\u0629 <code> {{name}} </code>: {{-message}}", "marking.note.error evaluating note": "\u062e\u0637\u0623 \u0641\u064a \u062a\u0642\u064a\u064a\u0645 \u0627\u0644\u0645\u0644\u0627\u062d\u0638\u0629 <code> {{name}} </code> - {{-message}}", "marking.note.invalid definition": "\u062a\u0639\u0631\u064a\u0641 \u0645\u0644\u0627\u062d\u0638\u0629 \u063a\u064a\u0631 \u0635\u0627\u0644\u062d: <code> {{source}} </code>. {{-\u0645\u0644\u062d\u0648\u0638\u0629}}", "marking.note.invalid definition.missing colon": "\u0644\u0627 \u062a\u0648\u062c\u062f \u0646\u0642\u0637\u062a\u064a\u0646 \u0628\u0639\u062f \u0627\u0644\u0625\u0633\u0645 \u0648\u0627\u0644\u0635\u0641\u0629", "marking.note.invalid definition.description missing closing bracket": "\u0642\u062f \u064a\u0643\u0648\u0646 \u0627\u062d\u062f \u0627\u0644\u0623\u0642\u0648\u0627\u0633 \u063a\u064a\u0631 \u0645\u063a\u0644\u0642.", "marking.note.empty expression": "\n\u0627\u0644\u0645\u0644\u0627\u062d\u0638\u0629 <code> {{name}} </code> \u0641\u0627\u0631\u063a\u0629.", "marking.script.error parsing notes": "\u062e\u0637\u0623 \u0641\u064a \u062a\u062d\u0644\u064a\u0644 \u0627\u0644\u0646\u0635 \u0627\u0644\u0628\u0631\u0645\u062c\u064a \u0644\u0644\u062a\u0645\u064a\u064a\u0632: {{- message}}", "part.feedback out of date": "\u0647\u0630\u0627 \u0627\u0644\u0645\u0644\u0627\u062d\u0638\u0627\u062a \u062a\u0633\u062a\u0646\u062f \u0625\u0644\u0649 \u0625\u062c\u0627\u0628\u062a\u0643 \u0627\u0644\u0623\u062e\u064a\u0631\u0629 \u0627\u0644\u062a\u064a \u062a\u0645 \u0625\u0631\u0633\u0627\u0644\u0647\u0627. \u0623\u0631\u0633\u0644 \u0625\u062c\u0627\u0628\u062a\u0643 \u0627\u0644\u062a\u064a \u062a\u0645 \u062a\u063a\u064a\u064a\u0631\u0647\u0627 \u0644\u0644\u062d\u0635\u0648\u0644 \u0639\u0644\u0649 \u0645\u0644\u0627\u062d\u0638\u0627\u062a \u0645\u062d\u062f\u062b\u0629.", "part.jme.invalid value generator expression": "\u062a\u0639\u0628\u064a\u0631 \u0645\u0646\u0634\u0626 \u0642\u064a\u0645\u0629 \u063a\u064a\u0631 \u0635\u0627\u0644\u062d \u0644\u0644\u0645\u062a\u063a\u064a\u0631 <code> {{name}} </code>: {{-message}}", "part.mcq.incorrect choice": "\u0642\u0645\u062a \u0628\u0627\u062e\u062a\u064a\u0627\u0631 \u0625\u062c\u0627\u0628\u0629 \u062e\u0627\u0637\u0626\u0629", "part.matrix.not all cells same precision": "\u062f\u0642\u0629 \u0627\u0644\u0623\u0639\u062f\u0627\u062f \u0627\u0644\u0645\u0642\u062f\u0645\u0629 (\u0639\u062f\u062f \u0627\u0644\u0623\u0631\u0642\u0627\u0645 \u0628\u0639\u062f \u0627\u0644\u0641\u0627\u0635\u0644\u0629) \u062a\u062e\u062a\u0644\u0641 \u0628\u064a\u0646 \u0627\u0644\u062e\u0644\u0627\u064a\u0627.", "part.gapfill.error marking gap": "\u062e\u0637\u0623 \u0641\u064a \u0648\u0636\u0639 \u0639\u0644\u0627\u0645\u0629 \u0639\u0644\u0649 {{name}}: {{-message}}", "part.custom.empty setting": "\u0644\u0645 \u064a\u062a\u0645 \u0625\u0639\u0637\u0627\u0621 \u0642\u064a\u0645\u0629.", "part.custom.unrecognised input type": "\u0646\u0648\u0639 \u0625\u0639\u062f\u0627\u062f \u063a\u064a\u0631 \u0645\u0639\u0631\u0648\u0641 <code> {{input_type}} </code>", "part.custom.error evaluating input option": "\u062e\u0637\u0623 \u0641\u064a \u062a\u0642\u064a\u064a\u0645 \u062e\u064a\u0627\u0631 \u0627\u0644\u0625\u062f\u062e\u0627\u0644 <code> {{option}} </code>: {{-error}}", "part.custom.input option missing": "\u062a\u0639\u0631\u064a\u0641 \u062e\u064a\u0627\u0631 \u0627\u0644\u0625\u062f\u062e\u0627\u0644 <code> {{option}} </code> \u0645\u0641\u0642\u0648\u062f.", "part.custom.error evaluating setting": "\u062e\u0637\u0623 \u0641\u064a \u062a\u0642\u064a\u064a\u0645 \u0627\u0644\u0625\u0639\u062f\u0627\u062f <code> {{setting}} </code>: {{-error}}", "question.error creating question": "\u062e\u0637\u0623 \u0623\u062b\u0646\u0627\u0621 \u0625\u0646\u0634\u0627\u0621 \u0627\u0644\u0633\u0624\u0627\u0644 {{number}}: {{-message}}", "question.score feedback.not marked": "\u0644\u0645 \u064a\u062a\u0645 \u0648\u0636\u0639 \u0639\u0644\u0627\u0645\u0629", "question.score feedback.partially answered": "\u0625\u062c\u0627\u0628\u0629 \u062c\u0632\u0626\u064a\u0629", "question.score feedback.score total actual": "\u0627\u0644\u0646\u062a\u064a\u062c\u0629: {{Score\u060c niceNumber}} / {{\u0639\u0644\u0627\u0645\u0627\u062a \u060c niceNumber}}", "variable.error in variable definition": "\u062e\u0637\u0623 \u0641\u064a \u062a\u0639\u0631\u064a\u0641 \u0627\u0644\u0645\u062a\u063a\u064a\u0631 <code> {{name}} </code>", "left brace": "}", "right brace": "{", "extension.not found": "\u062a\u0639\u0630\u0631 \u062a\u062d\u0645\u064a\u0644 \u0627\u0644\u0645\u0644\u062d\u0642 <code> {{name}} </code>.", "control.toggle navigation menu": "\u0642\u0645 \u0628\u0627\u0644\u062a\u0628\u062f\u064a\u0644 \u0641\u064a \u0642\u0627\u0626\u0645\u0629 \u0627\u0644\u0625\u0646\u062a\u0642\u0627\u0644", "part.input title": "\u0627\u0644\u0625\u062c\u0627\u0628\u0629 \u0644\u0644\u062c\u0632\u0621 {{name}}", "part.correct answer title": "\u0627\u0644\u0625\u062c\u0627\u0628\u0629 \u0627\u0644\u0645\u062a\u0648\u0642\u0639\u0629 \u0644\u0644\u062c\u0632\u0621 {{name}}", "part.jme.must-match.failed": "\u0625\u062c\u0627\u0628\u062a\u0643 \u0644\u064a\u0633\u062a \u0628\u0627\u0644\u0635\u064a\u063a\u0629 \u0627\u0644\u0635\u062d\u064a\u062d\u0629.", "question.score feedback.none": "", "control.submit part.confirm remove next parts": "<p> \u064a\u0639\u062a\u0645\u062f \u062c\u0632\u0621 \u0648\u0627\u062d\u062f \u0623\u0648 \u0623\u0643\u062b\u0631 \u0645\u0646 \u0627\u0644\u0623\u062c\u0632\u0627\u0621 \u0627\u0644\u0644\u0627\u062d\u0642\u0629 \u0639\u0644\u0649 \u0625\u062c\u0627\u0628\u062a\u0643 \u0644\u0647\u0630\u0627 \u0627\u0644\u062c\u0632\u0621. \u0633\u064a\u0624\u062f\u064a \u062a\u0642\u062f\u064a\u0645 \u0647\u0630\u0627 \u0627\u0644\u062c\u0632\u0621 \u0645\u0631\u0629 \u0623\u062e\u0631\u0649 \u0625\u0644\u0649 \u0625\u0628\u0637\u0627\u0644 \u062a\u0644\u0643 \u0627\u0644\u0623\u062c\u0632\u0627\u0621 \u0648\u0625\u0632\u0627\u0644\u062a\u0647\u0627 \u0645\u0646 \u0627\u0644\u0633\u0624\u0627\u0644. \u0647\u0630\u0627 \u0644\u0627 \u064a\u0645\u0643\u0646 \u0627\u0644\u062a\u0631\u0627\u062c\u0639 \u0639\u0646\u0647 </ p>\n<p> \u0647\u0644 \u062a\u0631\u063a\u0628 \u0641\u064a \u0625\u0631\u0633\u0627\u0644 \u0647\u0630\u0627 \u0627\u0644\u062c\u0632\u0621 \u0645\u0631\u0629 \u0623\u062e\u0631\u0649\u061f </ p>", "control.back to menu": "\u0627\u0631\u062c\u0639 \u0625\u0644\u0649 \u0627\u0644\u0642\u0627\u0626\u0645\u0629\n", "display.error making html": "\u062e\u0637\u0623 \u0641\u064a \u0639\u0645\u0644 HTML \u0641\u064a {{contextDescription}}: {{-message}}", "jme.subvars.error compiling": "{{-\u0631\u0633\u0627\u0644\u0629}} \u0641\u064a <code> {{\u062a\u0639\u0628\u064a\u0631}} </ code>", "jme.variables.empty name": "\u0644\u0645 \u064a\u062a\u0645 \u062a\u062d\u062f\u064a\u062f \u0627\u0633\u0645 \u0644\u0645\u062a\u063a\u064a\u0631 \u0627\u0644\u0633\u0624\u0627\u0644.\n", "jme.calculus.unknown derivative": "\u0644\u0627 \u062a\u0639\u0631\u0641 \u0643\u064a\u0641\u064a\u0629 \u0627\u0644\u062a\u0641\u0631\u064a\u0642 \u0628\u064a\u0646 <code> {{tree}} </code>", "math.order complex numbers": "\u0644\u0627 \u064a\u0645\u0643\u0646 \u0637\u0644\u0628 \u0623\u0631\u0642\u0627\u0645 \u0645\u0639\u0642\u062f\u0629", "menu.choose a question": "\u0627\u062e\u062a\u0631 \u0633\u0624\u0627\u0644\u0627.", "part.choose next part.answered": "\u0645\u0627\u0630\u0627 \u062a\u0631\u064a\u062f \u0627\u0646 \u062a\u0641\u0639\u0644 \u0644\u0627\u062d\u0642\u0627\u061f", "part.choose next part.unanswered": "\u0623\u0648 \u064a\u0645\u0643\u0646\u0643:", "part.choose next part.will be locked": "(\u0633\u064a\u062a\u0645 \u0642\u0641\u0644 \u0647\u0630\u0627 \u0627\u0644\u062c\u0632\u0621)", "part.reached dead end": "\u0644\u0627 \u064a\u0648\u062c\u062f \u0634\u064a\u0621 \u0622\u062e\u0631 \u064a\u0645\u0643\u0646 \u0627\u0644\u0642\u064a\u0627\u0645 \u0628\u0647 \u0645\u0646 \u0647\u0646\u0627.", "part.next part.penalty amount": "(\u062e\u0633\u0627\u0631\u0629 {{count}} $ t (\u0639\u0644\u0627\u0645\u0629))", "part.marking.counts towards objective": "\u064a\u062a\u0645 \u0627\u062d\u062a\u0633\u0627\u0628 \u0647\u0630\u0627 \u0627\u0644\u062c\u0632\u0621 \u0645\u0646 \u0627\u0644\u0647\u062f\u0641 <strong> \"{{target}}\" </strong>.", "part.numberentry.answer not integer or decimal or fraction": "\u0625\u062c\u0627\u0628\u062a\u0643 \u063a\u064a\u0631 \u0635\u0627\u0644\u062d\u0629. \u064a\u062c\u0628 \u0625\u062f\u062e\u0627\u0644 \u0639\u062f\u062f \u0635\u062d\u064a\u062d \u0623\u0648 \u0639\u0634\u0631\u064a \u0623\u0648 \u0643\u0633\u0631.", "question": "\u0633\u0624\u0627\u0644", "question.progress": "\u062a\u0642\u062f\u0645 \u0627\u0644\u0633\u0624\u0627\u0644:\n", "question.score feedback.unattempted": "\u0644\u0645 \u062a\u062d\u0627\u0648\u0644", "question.score feedback.attempted": "\u062a\u0645\u062a \u0627\u0644\u0645\u062d\u0627\u0648\u0644\u0629", "question.score feedback.score actual.plain": "\n{{ScoreString}}", "question.score feedback.score total actual.plain": "{{score,niceNumber}}/{{marks,niceNumber}}", "question.objectives": "\u0627\u0644\u0623\u0647\u062f\u0627\u0641", "question.penalties": "\u0627\u0644\u0639\u0642\u0648\u0628\u0627\u062a", "question.back to previous part": "\u0639\u062f \u0625\u0644\u0649 \u0627\u0644\u062c\u0632\u0621 \u0627\u0644\u0633\u0627\u0628\u0642", "end.print": "Print your exam transcript", "math.shuffle_together.lists not all the same length": "Not all lists are the same length.", "jme.parse signature.invalid signature string": "Invalid function signature string: {{str}}", "part.custom.expected answer has wrong type": "The expected answer for this part has the wrong type. It should be <code>{{shouldbe}}</code>.", "part.custom.input option has wrong type": "The answer input setting <code>{{option}}</code> has the wrong type. It should be <code>{{shouldbe}}</code>.", "matrix input.size control legend": "Size", "matrix input.rows": "Rows", "matrix input.columns": "Columns", "part.jme.error checking numerically": "There was an error numerically checking your answer: {{-message}}", "part.gapfill.cyclic adaptive marking": "There is a cycle in the adaptive marking for this part: <strong>{{name1}}</strong> relies on <strong>{{name2}}</strong>, which eventually relies on <strong>{{name1}}</strong>.", "modal.style.background colour": "Background colour", "modal.style.text colour": "Text colour", "modal.style.text size": "Text size", "modal.style.explanation": "Use these controls to change the appearance of the exam.", "modal.style.reset to defaults": "Reset to defaults", "modal.style.text size preview": "Most text will be this big.", "control.style options": "Display options", "part.marking.partially correct": "Your answer is partially correct.", "part.marking.error in adaptive marking": "There was an error in the adaptive marking for this part. Please report this. {{-message}}", "page.skip to content": "Skip to content", "result.learning objective": "Learning objective", "jme.interpreted as": "interpreted as", "jme.script.note.compilation error": "Error compiling note <code>{{name}}</code>: {{-message}}", "jme.script.note.error evaluating note": "Error evaluating note <code>{{name}}</code> - {{-message}}", "jme.script.note.invalid definition": "Invalid note definition: <code>{{source}}</code>. {{-hint}}", "jme.script.note.invalid definition.missing colon": "You might be missing a colon after the name and description", "jme.script.note.invalid definition.description missing closing bracket": "You might be missing a closing bracket", "jme.script.note.empty expression": "The note <code>{{name}}</code> is empty.", "jme.script.error parsing notes": "Error parsing marking script: {{- message}}", "matrix input.cell label": "Row {{row}}, column {{column}}", "control.move to next question": "Move to the next question", "diagnostic.use retry": "Use one retry and try this topic again.", "diagnostic.move to next topic": "Move on to the next topic.", "diagnostic.next step question": "What would you like to do next?", "diagnostic.now assessing topic": "Now assessing {{current_topic}}", "diagnostic.one retry left": "You have 1 retry left", "diagnostic.retries left": "You have {{retries}} retries left.", "diagnostic.percentage completed": "You've completed <strong>{{percentage}}%</strong> of the test.", "diagnostic.test is over": "The test is over.", "diagnostic.passed all lo": "You have passed all learning objectives.", "diagnostic.more work on lo": "You need to do some more work on the following learning objectives: {{los}}.", "diagnostic.move to next question in topic": "Move on to the next question in topic.", "diagnostic.complete": "Complete!", "diagnostic.studying topic": "Studying {{topic}}", "display.answer widget.unknown widget type": "The answer widget type <code>{{name}}</code> is not recognised.", "jme.shunt.expected argument before comma": "Expected to see something between the opening bracket and the comma", "part.waiting for pre submit": "Your answer is being marked. Please wait.", "diagnostic.end test": "End the test.", "page.no stylesheet loaded": "The page's stylesheet file has not loaded.", "modal.confirm": "Confirm", "modal.alert": "Alert", "suspend.resumed header": "Attempt resumed", "jme.vector.value not an array of numbers": "Tried to construct a vector using a value that is not an array of numbers.", "jme.matrix.value not the right type": "Tried to construct a vector using a value of the wrong type.", "jme.subvars.html inserted twice": "An HTML value has been embedded twice. Consider defining a function to generate a new value each time it is used.", "jme.variables.invalid function language": "The language <code>{{language}}</code> is not valid.", "jme.variables.duplicate definition": "There is more than one definition of the variable <code>{{name}}</code>.", "math.random_integer_partition.invalid k": "The size of the partition must be between 1 and {{n}}.", "part.marking.parameter already in scope": "There is a variable named <code>{{name}}</code>, which is also the name of a marking parameter. Please rename the variable.", "part.marking.adaptive variable replacement refers to self": "This part refers to itself in a variable replacement for adaptive marking.", "part.marking.adaptive variable replacement refers to nothing": "This part contains an invalid variable replacement for adaptive marking.", "part.numberentry.display answer wrong type": "The display answer for this part is a value of type <code>{{got_type}}</code>, but should be a <code>{{want_type}}</code>.", "part.matrix.invalid type in prefilled": "There is an invalid value of type <code>{{n}}</code> in the array of pre-filled cells.", "diagnostic.make a choice": "Make a choice", "matrixmath.not square": "This operation only works on a square matrix.", "matrixmath.not invertible": "This operation only works on an invertible matrix.", "matrixmath.gauss-jordan elimination.not enough columns": "There must be at least as many columns as rows.", "question.required extension not available": "This question requires the extension <code>{{-extension}}</code> but it is not available.", "util.formatNumberNotation.unrecognised syntax": "The number formatting syntax <code>{{syntax}}</code> is not recognised.", "worksheet.number of exams to generate": "Number of sheets to generate", "worksheet.starting with id": "Starting with ID", "worksheet.show exam id": "Show sheet ID?", "worksheet.page break between questions": "Page breaks between questions?", "worksheet.page margins": "Page margins (mm)", "worksheet.text size": "Text size (pt)", "worksheet.generate": "Generate", "worksheet.generating exams": "Generating sheets", "worksheet.sheet id": "Sheet ID:", "worksheet.print single": "Print this sheet", "worksheet.print several": "Print these sheets", "worksheet.answer sheets": "Answer sheets", "worksheet.question sheets": "Question sheets", "worksheet.reconfigure": "Generate different sheets", "worksheet.show sheet": "Preview the sheet with ID:", "accessibility statement": "Accessibility statement and guide to adapting Numbas to your needs.", "exam.enter your name": "Your name:", "exam.attempt download security warning": "This exam is configured to allow you to download your data, but it is not running in a secure browser context. You will not be able to download your exam data. Contact your lecturer or teacher for help.", "result.download exam object": "Download your exam data", "control.return to question": "Return to the question", "control.show introduction": "Introduction", "analysis.header": "Analyse attempt data", "analysis.help.upload files": "Upload attempt data files that your students have given you.", "analysis.help.file input label": "Choose student attempt data files, or drag files onto this window.", "analysis.table.total": "Exam totals", "analysis.table.question": "Exam and question totals", "analysis.table.all": "All details", "analysis.student name.anonymous": "No name given", "analysis.expected": "Expected results", "analysis.start time": "Start time", "analysis.maximum": "Maximum Marks", "analysis.file": "File", "analysis.download": "Download", "analysis.delete": "Delete", "analysis.view results": "View results", "analysis.upload files": "Upload files", "analysis.upload more": "Upload more files", "analysis.attempt data": "Attempt data", "analysis.select format": "Select data to show", "analysis.download this table": "Download this table", "analysis.student": "Student Results", "analysis.question key": "Question Key", "analysis.question name": "Question Name", "analysis.group": "Group", "analysis.question": "Question", "analysis.part": "Part", "analysis.gap": "Gap", "analysis.record type": "Record Type", "analysis.score": "Score", "analysis.marks available": "Marks Available", "analysis.percentage": "Percentage", "analysis.answer": "Answer", "analysis.student name": "Student Name", "analysis.summary.no files": "You have not uploaded any files yet.", "analysis.summary.no decrypted files": "You have not uploaded any valid files yet.", "analysis.summary.one file": "One attempt.", "analysis.summary.several files": "{{num_files,niceNumber}} attempts.", "analysis.not secure context": "This page must be opened in a secure browser context. A secure context is either a page served over HTTPS, or a file loaded from your device.", "jme.shunt.pipe right hand takes no arguments": "The expression on the right-hand side of the pipe operator must be a function application.", "question.explore.no parts defined": "There are no parts defined in this question.", "answer": "answer", "worksheet.answersheet show question content": "Show question content in answer sheets?", "modal.confirm end exam": "Write <code>{{endConfirmation}}</code> in the box to confirm:", "modal.end exam button": "End exam", "lightbox.zoom in on image": "Zoom in on this image", "exam.progress": "Progress", "exam.questions answered": "{{numAnsweredQuestions}} of {{numQuestions}} questions answered.", "result.question marks available": "Marks Available", "result.question answered": "Answered?", "control.confirm end.correct": "You may now end the exam.", "control.confirm end.incorrect": "This is not the expected text.", "control.confirm end.password": "end", "jme.typecheck.for in name wrong type": "The name in a <code>for</code> statement must be a name or list of names, not {{type}}.", "jme.makeFast.no fast definition of function": "The function <code>{{name}}</code> here isn't defined in a way that can be made fast.", "part.show feedback": "Show feedback", "part.hide feedback": "Hide feedback", "part.feedback title": "Feedback for {{name}}.", "part.jme.must-match.warning": "Your answer is not in the expected form: {{-message}}", "part.numberentry.write your answer as a fraction": "Write your answer as a fraction.", "question.nav.label": "Question controls", "question.answer saved": "Answer saved", "question.all answers saved": "All answers saved", "analysis.back to results": "Back to results", "analysis.review": "Review", "analysis.review this": "Review this attempt", "analysis.reviewing attempt": "Reviewing attempt by {{student_name}}.", "part.there is new feedback": "The feedback has changed.", "modal.style.colour scheme": "Colour scheme", "modal.style.automatic colour scheme": "Automatic", "modal.style.light colour scheme": "Light", "modal.style.dark colour scheme": "Dark", "modal.style.custom colour scheme": "Custom", "modal.style.main font": "Main font", "modal.style.font.sans serif": "Sans serif", "modal.style.font.serif": "Serif", "modal.style.font.monospace": "Monospace", "modal.style.spacing": "Spacing", "modal.style.font weight": "Font weight", "modal.style.forced colours warning": "Your browser has overridden the colours used in this page because of a setting such as high contrast mode. Changes to the colour scheme settings here might not have any effect.", "modal.style.text preview": "Most text will look like this.", "modal.style.more options": "More options", "modal.style.main colour": "Main (brand) colour", "modal.style.primary colour": "Primary button colour", "modal.style.link colour": "Link colour", "modal.style.success colour": "Success/correct colour", "modal.style.info colour": "Info colour", "modal.style.warning colour": "Warning colour", "modal.style.danger colour": "Danger/incorrect colour", "modal.style.muted colour": "Muted text colour", "modal.style.highlight colour": "Highlight colour", "exam.error loading exam definition": "There was an error while loading the exam definition: {{text}}", "exam.no exam definition": "No exam definition was given.", "jme.typecheck.wrong arguments for anonymous function": "Wrong number of arguments for this anonymous function.", "worksheet.top": "Top", "worksheet.left": "Left", "worksheet.bottom": "Bottom", "worksheet.right": "Right"}}, "pl-pl": {"translation": {"page.loading": "\u0141adowanie...", "page.saving": "<p>Zapisywanie.</p>\n<p>To mo\u017ce potrwa\u0107 par\u0119 sekund.</p>", "mathjax.math processing error": "\"{{-message}}\" when texifying <code>{{expression}}</code>", "die.numbas failed": "Numbas has failed", "die.sorry": "Przepraszamy, Numbas napotka\u0142o b\u0142\u0105d, co znaczy, \u017ce nie mo\u017ce kontynuowa\u0107. Poni\u017cej jest opis b\u0142\u0119du.", "die.error": "B\u0142\u0105d", "modal.ok": "OK", "modal.cancel": "Anuluj", "exam.exam name": "Nazwa Egzaminu:", "exam.random seed": "Numer Sesji:", "exam.student name": "Imi\u0119 Ucznia:", "exam.number of questions": "Liczba Pyta\u0144:", "exam.marks available": "Dost\u0119pne Oceny:", "exam.pass percentage": "Pr\u00f3g procentowy:", "exam.time allowed": "Czas dozwolony:", "exam.passed": "Zdane", "exam.failed": "Niezdane", "exam.review header": "Review: ", "frontpage.start": "Rozpocznij", "suspend.paused header": "Wstrzymaj", "suspend.exam suspended": "Egzamin zosta\u0142 wstrzymany. Wci\u015bnij <em>Wzn\u00f3w</em> by kontynuowa\u0107 egzamin.", "suspend.you can resume": "B\u0119dziesz m\u00f3g\u0142/mog\u0142a wznowi\u0107 t\u0105 sesj\u0119, kiedy rozpoczniesz to zadanie nast\u0119pnym razem.", "suspend.resume": "Wzn\u00f3w", "result.exit": "Zako\u0144cz egzamin", "result.print": "Wydrukuj te podsumowanie wynik\u00f3w", "result.exam summary": "Podsumowanie Egzaminu", "result.performance summary": "Performance Summary", "result.exam start": "Rozpocznij Egzamin:", "result.exam stop": "Zako\u0144cz Egzamin:", "result.time spent": "Czas po\u015bwi\u0119cony:", "result.questions attempted": "Zadania podj\u0119te:", "result.score": "Liczba punkt\u00f3w:", "result.result": "Wynik:", "result.question number": "Numer pytania", "result.question score": "Liczba Punkt\u00f3w", "result.question review title": "Review this question", "result.click a question to review": "Kliknij na numer pytania, by zobaczy\u0107, jak Twoje odpowiedzi zosta\u0142y ocenione i, gdzie to mo\u017cliwe, modelowe odpowiedzi.", "end.exam has finished": "Egzamin zosta\u0142 zako\u0144czony. Mo\u017cesz zamkn\u0105\u0107 to okno.", "control.confirm leave": "Nie zako\u0144czy\u0142e\u015b/\u0142a\u015b jeszcze tego egzaminu.", "control.not all questions answered": "Nie odpowiedzia\u0142e\u015b/\u0142a\u015b na ka\u017cde pytanie w tym egzaminie.", "control.not all questions submitted": "Dokona\u0142e\u015b/\u0142a\u015b zmian w przynajmniej jednej ze swoich odpowiedzi, ale nie zatwierdzi\u0142e\u015b/\u0142a\u015b jej. Prosz\u0119 sprawd\u017a czy ka\u017cde zdanie zosta\u0142o zatwierdzone.", "control.confirm end": "Czy jeste\u015b pewny/na, \u017ce chcesz zako\u0144czy\u0107 egzamin? Po jego zako\u0144czeniu nie b\u0119dziesz m\u00f3g\u0142/mog\u0142a ju\u017c dokonywa\u0107 zmian na swoich odpowiedziach.", "control.confirm regen": "Czy chcia\u0142by\u015b/chcia\u0142aby\u015b wylosowa\u0107 nowe dane to tego zadania? Je\u015bli klikniesz OK, wszystkie Twoje odpowiedzi i punkty za to zadanie zostan\u0105 utracone.", "control.confirm reveal": "Czy chcesz zobaczy\u0107 odpowiedzi do tego zadania? Wszelkie punkty, kt\u00f3re uzyska\u0142e\u015b/\u0142a\u015b za to zadanie zostan\u0105 zablokowane i nie b\u0119dziesz ju\u017c p\u00f3\u017aniej m\u00f3g\u0142/mog\u0142a go wykona\u0107.", "control.proceed anyway": "Proceed anyway?", "control.regen": "Spr\u00f3buj podobne zadanie.", "control.submit answer": "Zatwierd\u017a odpowied\u017a", "control.submit all parts": "Zatwierd\u017a wszystkie cz\u0119\u015bci", "control.submit again": "Zatwierd\u017a ponownie", "control.submit": "Zatwierd\u017a", "control.previous": "Poprzednie", "control.next": "Nast\u0119pne", "control.advice": "Porada", "control.reveal": "Poka\u017c odpowiedzi", "control.total": "Suma", "control.pause": "Wstrzymaj", "control.end exam": "Zako\u0144cz Egzamin", "control.back to results": "Wr\u00f3\u0107 do wynik\u00f3w", "display.part.jme.error making maths": "Error making maths display", "exam.xml.bad root": "Root element of exam XML should be 'exam'", "exam.changeQuestion.no questions": "Ten egzamin nie zawiera \u017cadnych pyta\u0144! Sprawd\u017a plik .exam nie ma b\u0142\u0119d\u00f3w.", "feedback.you were awarded": "You were awarded <strong>{{count,niceNumber}}</strong> $t(mark).", "feedback.taken away": "<strong>{{count,niceNumber}}</strong> $t(mark) $t(was) taken away.", "jme.tokenise.invalid": "Invalid expression: <code>{{expression}}</code>", "jme.shunt.not enough arguments": "Not enough arguments for operation <code>{{op}}</code>", "jme.shunt.no left bracket in function": "No matching left bracket in function application or tuple", "jme.shunt.no left square bracket": "Brak pasuj\u0105cego lewego nawiasu", "jme.shunt.no left bracket": "Brak pasuj\u0105cego lewego nawiasu", "jme.shunt.no right bracket": "Brak pasuj\u0105cego prawego nawiasu", "jme.shunt.no right square bracket": "Brak pasuj\u0105cego prawego kwadratowego nawiasu na ko\u0144cu tej listy.", "jme.shunt.missing operator": "Wyra\u017cenie nie mo\u017ce by\u0107 obliczone -- brakuje operatora.", "jme.typecheck.function maybe implicit multiplication": "Operacja {{name}} nie jest zdefiniowana. Czy mia\u0142e\u015b/\u0142a\u015b na my\u015bli <br/><code>{{first}}*{{possibleOp}}(...)</code>?", "jme.typecheck.function not defined": "Operacja {{op}} nie jest zdefiniowana. Czy mia\u0142e\u015b/\u0142a\u015b na my\u015bli <br/><code>{{suggestion}}*(...)</code>?", "jme.typecheck.op not defined": "Operacja '{{op}}' nie jest zdefiniowana.", "jme.typecheck.no right type definition": "Nie zaleziono definicji '{{op}}' poprawnego typu.", "jme.typecheck.no right type unbound name": "Zmienna <code>{{name}}</code> nie jest zdefiniowana.", "jme.typecheck.map not on enumerable": "<code>map</code> operacja musi zosta\u0107 wykonana na li\u015bcie albo przedziale, a nie na {{type}}", "jme.evaluate.undefined variable": "Zmienna {{name}} nie jest zdefiniowana.", "jme.thtml.not html": "Passed a non-HTML value into the THTML constructor.", "jme.func.switch.no default case": "No default case for Switch statement", "jme.func.listval.invalid index": "Invalid list index {{index}} on list of size {{size}}", "jme.func.listval.not a list": "Obiekt nie mo\u017ce mie\u0107 indeksu dolnego.", "jme.func.matrix.invalid row type": "Nie mo\u017cna skonstruowa\u0107 macierzy z rz\u0119d\u00f3w typu {{type}}", "jme.func.except.continuous range": "Can't use the 'except' operator on continuous ranges.", "jme.matrix.reports bad size": "Matrix reports its size incorrectly - must be an error in constructor function", "jme.texsubvars.no right bracket": "No matching <code>]</code> in {{op}} arguments.", "jme.texsubvars.missing parameter": "Brakuje parametra w {{op}}: {{parameter}}", "jme.texsubvars.no right brace": "Brak pasuj\u0105cego <code>}</code> w {{op}}", "jme.user javascript.error": "Error in user-defined javascript function <code>{{name}}</code>: {{-message}}", "jme.variables.error making function": "B\u0142\u0105d tworzenia funkcji <code>{{name}}</code>: {{-message}}", "jme.variables.syntax error in function definition": "B\u0142\u0105d sk\u0142adni w definicji funkcji", "jme.variables.variable not defined": "Zmienna <code>{{name}}</code> nie zosta\u0142a zdefiniowana.", "jme.variables.empty definition": "Definicja zmiennej <code>{{name}}</code> jest pusta.", "jme.variables.circular reference": "Circular variable reference in definition of <code>{{name}}</code>", "jme.variables.error computing dependency": "Error computing referenced variable <code>{{name}}</code>", "jme.variables.error evaluating variable": "Error evaluating variable {{name}}: {{-message}}", "jme.variables.question took too many runs to generate variables": "A valid set of question variables was not generated in time.", "jme.display.unknown token type": "Can't texify token type {{type}}", "jme.display.collectRuleset.no sets": "No sets given to collectRuleset!", "jme.display.collectRuleset.set not defined": "Ruleset {{name}} has not been defined", "jme.display.simplifyTree.no scope given": "Numbas.jme.display.simplifyTree must be given a Scope", "math.precround.complex": "Can't round to a complex number of decimal places", "math.siground.complex": "Can't round to a complex number of sig figs", "math.combinations.complex": "Can't compute combinations of complex numbers", "math.permutations.complex": "Can't compute permutations of complex numbers", "math.gcf.complex": "Can't compute GCF of complex numbers", "math.lcm.complex": "Can't compute LCM of complex numbers", "math.lt.order complex numbers": "Can't order complex numbers", "math.choose.empty selection": "Empty selection given to random function", "matrixmath.abs.non-square": "Can't compute the determinant of a matrix which isn't square.", "matrixmath.abs.too big": "Sorry, can't compute the determinant of a matrix bigger than 3x3 yet.", "matrixmath.mul.different sizes": "Nie mo\u017cna mno\u017cy\u0107 macierzy kt\u00f3re maj\u0105 inne rozmiary.", "vectormath.cross.not 3d": "Can only take the cross product of 3-dimensional vectors.", "vectormath.dot.matrix too big": "Can't calculate dot product of a matrix which isn't $1 \\times N$ or $N \\times 1$.", "vectormath.cross.matrix too big": "Can't calculate cross product of a matrix which isn't $1 \\times N$ or $N \\times 1$.", "part.with steps answer prompt": "Odpowied\u017a: ", "part.script.error": "Error in part {{path}} custom script {{script}}: {{-message}}", "part.marking.steps no matter": "Because you received full marks for the part, your answers to the steps aren't counted.", "part.marking.revealed steps no penalty": "You revealed the steps.", "part.marking.used variable replacements": "Ta cz\u0119\u015b\u0107 by\u0142a oceniona u\u017cywaj\u0105c twoich odpowiedzi do poprzednich cz\u0119\u015bci.", "part.marking.variable replacement part not answered": "You must answer {{part}} first.", "part.marking.resubmit because of variable replacement": "Ocena tej cz\u0119\u015bci zale\u017cy na odpowiedzi do innych cz\u0119\u015bci, kt\u00f3re zosta\u0142y zmienione. Ponownie zatwierd\u017a t\u0105 cz\u0119\u015b\u0107 \u017ceby zaktualizowa\u0107 twoj\u0105 ocen\u0119.", "part.marking.not submitted": "No answer submitted.", "part.marking.did not answer": "You did not answer this question.", "part.marking.nothing entered": "You did not enter an answer.", "part.marking.incorrect": "Your answer is incorrect.", "part.marking.correct": "Your answer is correct.", "part.marking.uncaught error": "Error when marking: {{-message}}", "part.marking.no result": "This part could not be marked.", "part.correct answer": "Expected answer:", "part.missing type attribute": "{{part}}: Missing part type attribute", "part.unknown type": "{{part}}: Unrecognised part type {{type}}", "part.setting not present": "Property '{{property}}' not set", "part.jme.answer missing": "Correct answer is missing", "part.jme.answer too long": "Your answer is too long.", "part.jme.answer too short": "Your answer is too short.", "part.jme.answer invalid": "Your answer is not a valid mathematical expression.<br/>{{-message}}.", "part.jme.marking.correct": "Your answer is numerically correct.", "part.jme.must-have bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.must-have one": "Your answer must contain: {{strings}}", "part.jme.must-have several": "Your answer must contain all of: {{strings}}", "part.jme.not-allowed bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.not-allowed one": "Your answer must not contain: {{strings}}", "part.jme.not-allowed several": "Your answer must not contain any of: {{strings}}", "part.jme.unexpected variable name": "Your answer was interpreted to use the unexpected variable name <code>{{name}}</code>.", "part.jme.unexpected variable name suggestion": "Your answer was interpreted to use the unexpected variable name <code>{{name}}</code>. Did you mean <code>{{suggestion}}</code>?", "part.patternmatch.display answer missing": "Display answer is missing", "part.patternmatch.correct except case": "Your answer is correct, except for the case.", "part.numberentry.correct except decimal": "Your answer is within the allowed range, but decimal numbers are not allowed.", "part.numberentry.correct except fraction": "Your answer is within the allowed range, but fractions are not allowed.", "part.numberentry.answer invalid": "You did not enter a valid number.", "part.numberentry.answer not integer": "Your answer is invalid. You must enter a whole number, not a decimal.", "part.numberentry.answer not integer or decimal": "Your answer is invalid. You must enter an integer or a decimal.", "part.numberentry.zero sig fig": "This part is set up to round the student's answer to zero significant figures, which has no meaning.", "part.mcq.options def not a list": "The expression defining the {{properties}} is not a list.", "part.mcq.marking matrix string empty": "The custom marking matrix expression is empty.", "part.mcq.choices missing": "Definition of choices is missing", "part.mcq.matrix not a number": "Part {{part}} marking matrix cell ({{row}},{{column}}) does not evaluate to a number", "part.mcq.wrong number of choices": "You selected the wrong number of choices.", "part.mcq.no choices selected": "No choices selected.", "part.mcq.matrix not a list": "Marking matrix, defined by JME expression, is not a list but it should be.", "part.mcq.matrix wrong type": "Element of invalid type '{{type}}' used in marking matrix.", "part.mcq.matrix mix of numbers and lists": "Mix of numbers and lists used in marking matrix.", "part.mcq.matrix wrong size": "Marking matrix is the wrong size.", "part.mcq.correct choice": "You chose a correct answer.", "part.matrix.answer invalid": "Your answer is not valid.", "part.matrix.invalid cell": "One or more of the cells in your answer is empty or invalid.", "part.matrix.some incorrect": "One or more of the cells in your answer is incorrect, but you have been awarded marks for the rest.", "part.matrix.empty": "You have not entered an answer.", "part.matrix.empty cell": "One or more of the cells in your answer is empty.", "part.matrix.size mismatch": "The question author hasn't allowed the student to decide the dimensions of their answer, but the correct answer is {{correct_dimensions}} while the answer input is {{input_dimensions}}", "part.gapfill.feedback header": "<strong>{{name}}</strong>", "part.extension.not implemented": "Part hasn't implemented the <code>{{name}}</code> method.", "question.loaded name mismatch": "Can't resume this attempt - the package has changed since the last session.", "question.error": "Question {{number}}: {{-message}}", "question.preamble.error": "Error in preamble: {{-message}}", "question.preamble.syntax error": "Syntax error in preamble", "question.unsupported part type": "Unsupported part type", "question.header": "Question {{number}}", "question.submit part": "Save answer", "question.show steps": "Show steps", "question.show steps penalty": "You will lose <strong>{{count,niceNumber}}</strong> $t(mark).", "question.show steps no penalty": "Your score will not be affected.", "question.show steps already penalised": "You have already shown steps. You can show them again with no further penalty.", "question.hide steps": "Hide steps", "question.hide steps no penalty": "Your score will not be affected.", "question.advice": "Advice", "question.no such part": "Can't find part {{path}}.", "question.can not submit": "Can not submit answer - check for errors.", "question.answer submitted": "Answer submitted", "question.score feedback.show": "Show feedback", "question.score feedback.hide": "Hide feedback", "question.score feedback.score total actual": "Score: {{score,niceNumber}}/{{marks,niceNumber}}", "question.score feedback.score total": "{{marksString}}", "question.score feedback.score actual": "Score: {{scoreString}}", "question.score feedback.answered": "Answered", "question.score feedback.unanswered": "Unanswered", "question.score feedback.correct": "Your answer is correct", "question.score feedback.partial": "Your answer is partially correct", "question.score feedback.wrong": "Your answer is incorrect", "question.selector.unsubmitted changes": "Unsubmitted changes.", "timing.no accumulator": "no timing accumulator {{name}}", "timing.time remaining": "Time remaining:", "xml.could not load": "Couldn't load an XML document: {{-message}}", "xml.property not number": "Property {{name}} should be a number, but isn't ({{value}}), in node {{element}}", "xml.property not boolean": "Property {{name}} should be a boolean, but isn't ({{value}}), in node {{element}}", "xml.error in variable definition": "Error in definition of variable <code>{{name}}</code>", "scorm.error initialising": "Error initialising SCORM protocol: {{-message}}", "scorm.failed save": "<p>The request to save data to the server failed. Press <b>OK</b> to try again.</p>\n<p>If you get this message repeatedly, check your internet connection or use a different computer. Your previously submitted answers have been successfully saved and will be restored if you resume this session on a different computer.</p>\n<p>If this message appears persistently and you can't save <em>any</em> answers, please contact your lecturer or teacher.</p>", "scorm.no exam suspend data": "Failed to resume: no exam suspend data.", "scorm.error loading suspend data": "Error loading suspend data: {{-message}}", "scorm.error loading question": "Error loading question {{number}}: {{-message}}", "scorm.no question suspend data": "No question suspend data", "scorm.error loading part": "Error loading part {{part}}: {{-message}}", "scorm.no part suspend data": "No part suspend data", "util.product.non list": "Passed a non-list to <code>Numbas.util.product</code>", "mark": "mark", "was": "was", "part": "part", "gap": "gap", "step": "step", "jme.substituteTree.undefined variable": "Undefined variable: <code>{{name}}</code>", "jme.user javascript.returned undefined": "User-defined javascript function <code>{{name}}</code> returned <code>undefined</code>.", "part.marking.steps change": "You were awarded <strong>{{count,niceNumber}}</strong> $t(mark) for your answers to the steps.", "part.marking.revealed steps with penalty": "You revealed the steps. The maximum you can score for this part is <strong>{{count,niceNumber}}</strong> $t(mark). Your scores will be scaled down accordingly.", "part.marking.total score": "You scored <strong>{{count,niceNumber}}</strong> $t(mark) for this part.", "part.numberentry.precision type.dp": "decimal place", "part.numberentry.precision type.dp_plural": "decimal places", "part.numberentry.precision type.sigfig": "significant figure", "part.numberentry.precision type.sigfig_plural": "significant figures", "part.numberentry.give your answer to precision": "Round your answer to {{count,niceNumber}} {{precisionType}}.", "question.unsubmitted changes": "You have made a change to your answer but not submitted it. Please check your answer and then press the <strong>Save answer</strong> button.", "question.unsubmitted changes_plural": "You have made changes to your answers but not submitted them. Please check your answers to each part and then press the <strong>Save all answers</strong> button.", "util.equality not defined for type": "Equality not defined for type {{type}}", "mark_plural": "marks", "was_plural": "were", "die.script not loaded": "Numbas couldn't start because the file <code>{{file}}</code> was not loaded. Check that it's included in <code>scripts.js</code>.", "math.combinations.n less than zero": "Can't compute combinations: n is less than zero", "math.combinations.k less than zero": "Can't compute combinations: k is less than zero", "math.combinations.n less than k": "Can't compute combinations: n is less than k", "math.permutations.n less than zero": "Can't compute permutations: n is less than zero", "math.permutations.k less than zero": "Can't compute permutations: k is less than zero", "math.permutations.n less than k": "Can't compute permutations: n is less than k", "part.numberentry.give your answer to precision_0": "Round your answer to the nearest integer.", "mathjax.error": "MathJax processing error: {{-message}}", "mathjax.error with context": "MathJax processing error in {{-context}}: {{-message}}", "exam.introduction": "Exam introduction", "exam.feedback": "Exam feedback message", "jme.tokenise.keypair key not a string": "Dictionary key should be a string, not {{type}}.", "jme.shunt.list mixed argument types": "Can't parse {{mode}}: mix of dictionary and list elements", "jme.func.listval.key not in dict": "Dictionary does not contain the key <code>{{key}}</code>", "part.prompt": "prompt", "part.feedback": "Feedback", "part.numberentry.answer not reduced": "Your answer is not reduced to lowest terms.", "part.numberentry.give your answer as a reduced fraction": "Reduce your answer to lowest terms.", "part.numberentry.negative decimal places": "This part is set up to round the student's answer to a negative number of decimal places, which has no meaning.", "part.mcq.choices": "choices", "part.mcq.answers": "answers", "part.mcq.matrix cell empty": "Part {{part}} marking matrix cell ({{row}},{{column}}) is empty", "part.mcq.matrix jme error": "Part {{part}} marking matrix cell ({{row}},{{column}}) gives a JME error: {{-error}}", "question.statement": "Statement", "ruleset.circular reference": "Circular reference in definition of ruleset <code>{{name}}</code>", "ruleset.set not defined": "Ruleset {{name}} has not been defined", "jme.evaluate.no scope given": "Numbas.jme.evaluate must be given a Scope", "question.score feedback.answered total actual": "Score: {{score,niceNumber}}/{{marks,niceNumber}}", "question.score feedback.answered total": "{{marksString}}. Answered.", "question.score feedback.answered actual": "Score: {{scoreString}}", "question.score feedback.unanswered total": "{{marksString}}.", "answer.number.not a number": "Your answer is not a valid number.", "answer.number.fractions not allowed": "You may not enter a fraction.", "answer.jme.invalid expression": "{{-message}}", "answer.matrix.fractions not allowed": "You may not enter fractions.", "answer.matrix.some cell not a number": "One or more of the cells in your answer is not a valid number.", "exam.enter password": "Password:", "exam.password.correct": "This password is correct. You can start the exam.", "exam.password.incorrect": "This password is incorrect.", "frontpage.scorm.lms not connected": "This exam is running in standalone mode. Your answers and marks will not be saved!", "result.question review": "Review", "control.confirm regen no marks": "Would you like to re-randomise this question?", "control.confirm reveal no marks": "Would you like to reveal the answer to this question?", "jme.tokenise.invalid near": "Invalid expression: <code>{{expression}}</code> at position {{position}} near <code>{{nearby}}</code>", "jme.tokenise.number.object not complex": "Invalid object passed into number constructor.", "jme.subvars.null substitution": "Empty variable substitution: <code>$t(left brace){{str}}$t(right brace)", "jme.type.type already registered": "The data type {{type}} has already been registered so can't be registered again.", "jme.type.no cast method": "Can't automatically convert from {{from}} to {{to}}.", "jme.display.simplifyTree.empty expression": "Expression is empty", "jme.display.simplifyTree.stuck in a loop": "Simplifier is stuck in a loop: <code>{{expr}}</code>", "math.niceNumber.undefined": "Was expecting a number, but got <code>undefined</code>", "math.rangeToList.zero step size": "Can't convert a range with step size zero to a list.", "part.error": "{{path}}: {{-message}}", "part.marking.revealed steps": "You revealed the steps.", "part.marking.maximum scaled down": "The maximum you can score for this part is <strong>{{count,niceNumber}}</strong> $t(mark). Your scores will be scaled down accordingly.", "part.marking.minimum score applied": "The minimum score for this part is <strong>{{score,niceNumber}}</strong>.", "part.marking.maximum score applied": "The maximum score for this part is <strong>{{score,niceNumber}}</strong>.", "part.marking.error in marking script": "There was an error in this part's marking algorithm. Please report this. {{-message}}", "part.marking.no result after replacement": "This part could not be marked using your answers to previous parts.", "part.marking.missing required note": "The marking algorithm does not define the note <code>{{note}}</code>", "marking.apply.not a list": "The first argument to <code>apply</code> must be a list, and isn't", "marking.apply marking script.script not found": "Marking script <code>{{name}}</code> not found", "marking.note.compilation error": "Error compiling note <code>{{name}}</code>: {{-message}}", "marking.note.error evaluating note": "Error evaluating note <code>{{name}}</code> - {{-message}}", "marking.note.invalid definition": "Invalid note definition: <code>{{source}}</code>. {{-hint}}", "marking.note.invalid definition.missing colon": "You might be missing a colon after the name and description", "marking.note.invalid definition.description missing closing bracket": "You might be missing a closing bracket", "marking.note.empty expression": "The note <code>{{name}}</code> is empty.", "marking.script.error parsing notes": "Error parsing marking script: {{- message}}", "part.feedback out of date": "This feedback is based on your last submitted answer. Save your changed answer to get updated feedback.", "part.jme.invalid value generator expression": "Invalid value generator expression for variable <code>{{name}}</code>: {{-message}}", "part.mcq.incorrect choice": "You chose an incorrect answer.", "part.matrix.not all cells same precision": "You have not given every cell in your answer to the same precision.", "part.gapfill.error marking gap": "Error marking {{name}}: {{-message}}", "part.custom.empty setting": "No value given.", "part.custom.unrecognised input type": "Unrecognised setting type <code>{{input_type}}</code>", "part.custom.error evaluating input option": "Error evaluating input option <code>{{option}}</code>: {{-error}}", "part.custom.input option missing": "Definition of input option <code>{{option}}</code> is missing.", "part.custom.error evaluating setting": "Error evaluating setting <code>{{setting}}</code>: {{-error}}", "question.error creating question": "Error while creating question {{number}}: {{-message}}", "question.score feedback.not marked": "Not marked", "question.score feedback.partially answered": "Partially answered", "variable.error in variable definition": "Error in definition of variable <code>{{name}}</code>", "left brace": "{", "right brace": "}", "extension.not found": "Couldn't load the extension <code>{{name}}</code>.", "control.toggle navigation menu": "Toggle the navigation menu", "part.input title": "Answer for part {{name}}", "part.correct answer title": "Expected answer for part {{name}}", "part.jme.must-match.failed": "Your answer is not in the expected form.", "question.score feedback.none": "", "control.submit part.confirm remove next parts": "<p>One or more subsequent parts depend on your answer to this part. Submitting this part again will invalidate those parts, and remove them from the question. This cannot be undone.</p>\n<p>Would you like to submit this part again?</p>", "control.back to menu": "Go back to the menu", "display.error making html": "Error making HTML in {{contextDescription}}: {{-message}}", "jme.subvars.error compiling": "{{-message}} in <code>{{expression}}</code>", "jme.variables.empty name": "A question variable has not been given a name.", "jme.calculus.unknown derivative": "Don't know how to differentiate <code>{{tree}}</code>", "math.order complex numbers": "Can't order complex numbers", "menu.choose a question": "Choose a question.", "part.choose next part.answered": "What do you want to do next?", "part.choose next part.unanswered": "Or, you could:", "part.choose next part.will be locked": "(This part will be locked)", "part.reached dead end": "There's nothing more to do from here.", "part.next part.penalty amount": "(lose {{count}} $t(mark))", "part.marking.counts towards objective": "This part counts towards the objective <strong>\u201c{{objective}}\u201d</strong>.", "part.numberentry.answer not integer or decimal or fraction": "Your answer is invalid. You must enter an integer, a decimal or a fraction.", "question": "Question", "question.progress": "Question progress", "question.score feedback.unattempted": "Not attempted", "question.score feedback.attempted": "Attempted", "question.score feedback.score actual.plain": "{{scoreString}}", "question.score feedback.score total actual.plain": "{{score,niceNumber}}/{{marks,niceNumber}}", "question.objectives": "Objectives", "question.penalties": "Penalties", "question.back to previous part": "Go back to the previous part", "end.print": "Print your exam transcript", "math.shuffle_together.lists not all the same length": "Not all lists are the same length.", "jme.parse signature.invalid signature string": "Invalid function signature string: {{str}}", "part.custom.expected answer has wrong type": "The expected answer for this part has the wrong type. It should be <code>{{shouldbe}}</code>.", "part.custom.input option has wrong type": "The answer input setting <code>{{option}}</code> has the wrong type. It should be <code>{{shouldbe}}</code>.", "matrix input.size control legend": "Size", "matrix input.rows": "Rows", "matrix input.columns": "Columns", "part.jme.error checking numerically": "There was an error numerically checking your answer: {{-message}}", "part.gapfill.cyclic adaptive marking": "There is a cycle in the adaptive marking for this part: <strong>{{name1}}</strong> relies on <strong>{{name2}}</strong>, which eventually relies on <strong>{{name1}}</strong>.", "modal.style.background colour": "Background colour", "modal.style.text colour": "Text colour", "modal.style.text size": "Text size", "modal.style.explanation": "Use these controls to change the appearance of the exam.", "modal.style.reset to defaults": "Reset to defaults", "modal.style.text size preview": "Most text will be this big.", "control.style options": "Display options", "part.marking.partially correct": "Your answer is partially correct.", "part.marking.error in adaptive marking": "There was an error in the adaptive marking for this part. Please report this. {{-message}}", "page.skip to content": "Skip to content", "result.learning objective": "Learning objective", "jme.interpreted as": "interpreted as", "jme.script.note.compilation error": "Error compiling note <code>{{name}}</code>: {{-message}}", "jme.script.note.error evaluating note": "Error evaluating note <code>{{name}}</code> - {{-message}}", "jme.script.note.invalid definition": "Invalid note definition: <code>{{source}}</code>. {{-hint}}", "jme.script.note.invalid definition.missing colon": "You might be missing a colon after the name and description", "jme.script.note.invalid definition.description missing closing bracket": "You might be missing a closing bracket", "jme.script.note.empty expression": "The note <code>{{name}}</code> is empty.", "jme.script.error parsing notes": "Error parsing marking script: {{- message}}", "matrix input.cell label": "Row {{row}}, column {{column}}", "control.move to next question": "Move to the next question", "diagnostic.use retry": "Use one retry and try this topic again.", "diagnostic.move to next topic": "Move on to the next topic.", "diagnostic.next step question": "What would you like to do next?", "diagnostic.now assessing topic": "Now assessing {{current_topic}}", "diagnostic.one retry left": "You have 1 retry left", "diagnostic.retries left": "You have {{retries}} retries left.", "diagnostic.percentage completed": "You've completed <strong>{{percentage}}%</strong> of the test.", "diagnostic.test is over": "The test is over.", "diagnostic.passed all lo": "You have passed all learning objectives.", "diagnostic.more work on lo": "You need to do some more work on the following learning objectives: {{los}}.", "diagnostic.move to next question in topic": "Move on to the next question in topic.", "diagnostic.complete": "Complete!", "diagnostic.studying topic": "Studying {{topic}}", "display.answer widget.unknown widget type": "The answer widget type <code>{{name}}</code> is not recognised.", "jme.shunt.expected argument before comma": "Expected to see something between the opening bracket and the comma", "part.waiting for pre submit": "Your answer is being marked. Please wait.", "diagnostic.end test": "End the test.", "page.no stylesheet loaded": "The page's stylesheet file has not loaded.", "modal.confirm": "Confirm", "modal.alert": "Alert", "suspend.resumed header": "Attempt resumed", "jme.vector.value not an array of numbers": "Tried to construct a vector using a value that is not an array of numbers.", "jme.matrix.value not the right type": "Tried to construct a vector using a value of the wrong type.", "jme.subvars.html inserted twice": "An HTML value has been embedded twice. Consider defining a function to generate a new value each time it is used.", "jme.variables.invalid function language": "The language <code>{{language}}</code> is not valid.", "jme.variables.duplicate definition": "There is more than one definition of the variable <code>{{name}}</code>.", "math.random_integer_partition.invalid k": "The size of the partition must be between 1 and {{n}}.", "part.marking.parameter already in scope": "There is a variable named <code>{{name}}</code>, which is also the name of a marking parameter. Please rename the variable.", "part.marking.adaptive variable replacement refers to self": "This part refers to itself in a variable replacement for adaptive marking.", "part.marking.adaptive variable replacement refers to nothing": "This part contains an invalid variable replacement for adaptive marking.", "part.numberentry.display answer wrong type": "The display answer for this part is a value of type <code>{{got_type}}</code>, but should be a <code>{{want_type}}</code>.", "part.matrix.invalid type in prefilled": "There is an invalid value of type <code>{{n}}</code> in the array of pre-filled cells.", "diagnostic.make a choice": "Make a choice", "matrixmath.not square": "This operation only works on a square matrix.", "matrixmath.not invertible": "This operation only works on an invertible matrix.", "matrixmath.gauss-jordan elimination.not enough columns": "There must be at least as many columns as rows.", "question.required extension not available": "This question requires the extension <code>{{-extension}}</code> but it is not available.", "util.formatNumberNotation.unrecognised syntax": "The number formatting syntax <code>{{syntax}}</code> is not recognised.", "worksheet.number of exams to generate": "Number of sheets to generate", "worksheet.starting with id": "Starting with ID", "worksheet.show exam id": "Show sheet ID?", "worksheet.page break between questions": "Page breaks between questions?", "worksheet.page margins": "Page margins (mm)", "worksheet.text size": "Text size (pt)", "worksheet.generate": "Generate", "worksheet.generating exams": "Generating sheets", "worksheet.sheet id": "Sheet ID:", "worksheet.print single": "Print this sheet", "worksheet.print several": "Print these sheets", "worksheet.answer sheets": "Answer sheets", "worksheet.question sheets": "Question sheets", "worksheet.reconfigure": "Generate different sheets", "worksheet.show sheet": "Preview the sheet with ID:", "accessibility statement": "Accessibility statement and guide to adapting Numbas to your needs.", "exam.enter your name": "Your name:", "exam.attempt download security warning": "This exam is configured to allow you to download your data, but it is not running in a secure browser context. You will not be able to download your exam data. Contact your lecturer or teacher for help.", "result.download exam object": "Download your exam data", "control.return to question": "Return to the question", "control.show introduction": "Introduction", "analysis.header": "Analyse attempt data", "analysis.help.upload files": "Upload attempt data files that your students have given you.", "analysis.help.file input label": "Choose student attempt data files, or drag files onto this window.", "analysis.table.total": "Exam totals", "analysis.table.question": "Exam and question totals", "analysis.table.all": "All details", "analysis.student name.anonymous": "No name given", "analysis.expected": "Expected results", "analysis.start time": "Start time", "analysis.maximum": "Maximum Marks", "analysis.file": "File", "analysis.download": "Download", "analysis.delete": "Delete", "analysis.view results": "View results", "analysis.upload files": "Upload files", "analysis.upload more": "Upload more files", "analysis.attempt data": "Attempt data", "analysis.select format": "Select data to show", "analysis.download this table": "Download this table", "analysis.student": "Student Results", "analysis.question key": "Question Key", "analysis.question name": "Question Name", "analysis.group": "Group", "analysis.question": "Question", "analysis.part": "Part", "analysis.gap": "Gap", "analysis.record type": "Record Type", "analysis.score": "Score", "analysis.marks available": "Marks Available", "analysis.percentage": "Percentage", "analysis.answer": "Answer", "analysis.student name": "Student Name", "analysis.summary.no files": "You have not uploaded any files yet.", "analysis.summary.no decrypted files": "You have not uploaded any valid files yet.", "analysis.summary.one file": "One attempt.", "analysis.summary.several files": "{{num_files,niceNumber}} attempts.", "analysis.not secure context": "This page must be opened in a secure browser context. A secure context is either a page served over HTTPS, or a file loaded from your device.", "jme.shunt.pipe right hand takes no arguments": "The expression on the right-hand side of the pipe operator must be a function application.", "question.explore.no parts defined": "There are no parts defined in this question.", "answer": "answer", "worksheet.answersheet show question content": "Show question content in answer sheets?", "modal.confirm end exam": "Write <code>{{endConfirmation}}</code> in the box to confirm:", "modal.end exam button": "End exam", "lightbox.zoom in on image": "Zoom in on this image", "exam.progress": "Progress", "exam.questions answered": "{{numAnsweredQuestions}} of {{numQuestions}} questions answered.", "result.question marks available": "Marks Available", "result.question answered": "Answered?", "control.confirm end.correct": "You may now end the exam.", "control.confirm end.incorrect": "This is not the expected text.", "control.confirm end.password": "end", "jme.typecheck.for in name wrong type": "The name in a <code>for</code> statement must be a name or list of names, not {{type}}.", "jme.makeFast.no fast definition of function": "The function <code>{{name}}</code> here isn't defined in a way that can be made fast.", "part.show feedback": "Show feedback", "part.hide feedback": "Hide feedback", "part.feedback title": "Feedback for {{name}}.", "part.jme.must-match.warning": "Your answer is not in the expected form: {{-message}}", "part.numberentry.write your answer as a fraction": "Write your answer as a fraction.", "question.nav.label": "Question controls", "question.answer saved": "Answer saved", "question.all answers saved": "All answers saved", "analysis.back to results": "Back to results", "analysis.review": "Review", "analysis.review this": "Review this attempt", "analysis.reviewing attempt": "Reviewing attempt by {{student_name}}.", "part.there is new feedback": "The feedback has changed.", "modal.style.colour scheme": "Colour scheme", "modal.style.automatic colour scheme": "Automatic", "modal.style.light colour scheme": "Light", "modal.style.dark colour scheme": "Dark", "modal.style.custom colour scheme": "Custom", "modal.style.main font": "Main font", "modal.style.font.sans serif": "Sans serif", "modal.style.font.serif": "Serif", "modal.style.font.monospace": "Monospace", "modal.style.spacing": "Spacing", "modal.style.font weight": "Font weight", "modal.style.forced colours warning": "Your browser has overridden the colours used in this page because of a setting such as high contrast mode. Changes to the colour scheme settings here might not have any effect.", "modal.style.text preview": "Most text will look like this.", "modal.style.more options": "More options", "modal.style.main colour": "Main (brand) colour", "modal.style.primary colour": "Primary button colour", "modal.style.link colour": "Link colour", "modal.style.success colour": "Success/correct colour", "modal.style.info colour": "Info colour", "modal.style.warning colour": "Warning colour", "modal.style.danger colour": "Danger/incorrect colour", "modal.style.muted colour": "Muted text colour", "modal.style.highlight colour": "Highlight colour", "exam.error loading exam definition": "There was an error while loading the exam definition: {{text}}", "exam.no exam definition": "No exam definition was given.", "jme.typecheck.wrong arguments for anonymous function": "Wrong number of arguments for this anonymous function.", "worksheet.top": "Top", "worksheet.left": "Left", "worksheet.bottom": "Bottom", "worksheet.right": "Right"}}, "it-it": {"translation": {"page.loading": "Sto caricando...", "page.saving": "<p>Salvataggio in corso.</p>\n<p>Potrebbe richiedere qualche secondo.</p>", "mathjax.math processing error": "\"{{-message}}\" mentre compila <code>{{expression}}</code>", "die.numbas failed": "Numbas ha fallito", "die.sorry": "Mi spiace. Nunbas ha incontrato un errore e non pu\u00f2 continuare. In basso troverai una descrizione.", "die.error": "Errore", "modal.ok": "OK", "modal.cancel": "Annulla", "exam.exam name": "Nome esame:", "exam.random seed": "ID sessione:", "exam.student name": "Nome studente:", "exam.number of questions": "Numero di domande:", "exam.marks available": "Punti disponibili:", "exam.pass percentage": "Percentuale superamento:", "exam.time allowed": "Tempo disponibile:", "exam.passed": "Superato", "exam.failed": "Non superato", "exam.review header": "Controllo: ", "frontpage.start": "Inizio", "suspend.paused header": "In pausa", "suspend.exam suspended": "Il test \u00e8 sospeso. Premi <em>Riprendi</em> per continuare", "suspend.you can resume": "Puoi riprendere la sessione la prossima volta che apri quest'attivit\u00e0.", "suspend.resume": "Riprendi", "result.exit": "Esci dall'esame", "result.print": "Stampa i risultati", "result.exam summary": "Argomenti del test", "result.performance summary": "Riepilogo delle prestazioni", "result.exam start": "Inizio test:", "result.exam stop": "Fine test:", "result.time spent": "Tempo impiegato:", "result.questions attempted": "Domande tentate:", "result.score": "Puteggio:", "result.result": "Risultato:", "result.question number": "Domanda numero", "result.question score": "Punteggio", "result.question review title": "Rivedi questa domanda", "result.click a question to review": "Clicca sul numero di una domanda per visualizzare che voti hanno ricevuto le tue risposte e, dove possibile, le soluzioni complete.", "end.exam has finished": "L'esame \u00e8 terminato. Adesso puoi chiudere questa finestra.", "control.confirm leave": "Non hai ancora terminato l'esame.", "control.not all questions answered": "Non hai completato tutte le domande di questo esame.", "control.not all questions submitted": "Hai fatto cambiamenti a una o pi\u00f9 risposte ma non le hai inviate. Per favore controlla che ogni risposta sia stata inviata.", "control.confirm end": "Sei sicuro di voler finire l'esame? Dopo che avrai finito l'esame, non sarai in grado di cambiare nessuna delle tue risposte.", "control.confirm regen": "Vorresti rigenerare questa domanda? Se clicchi OK, tutte le tue risposte e i punteggi per questa domanda saranno persi.", "control.confirm reveal": "Vuoi mostrare la risposta a questa domanda? I punti che hai ricevuto fino a ora saranno bloccati e non potrai rispondere a questa domanda pi\u00f9 tardi.", "control.proceed anyway": "Procedi in ogni caso?", "control.regen": "Prova un'altra domanda come questa.", "control.submit answer": "Invia la risposta", "control.submit all parts": "Invia tutte le parti", "control.submit again": "Reinvia", "control.submit": "Invia", "control.previous": "Precedente", "control.next": "Successivo", "control.advice": "Suggerimento", "control.reveal": "Mostra le risposte", "control.total": "Totale", "control.pause": "Pausa", "control.end exam": "Finisci l'esame", "control.back to results": "Torna ai risultati", "display.part.jme.error making maths": "Errore nel mostrare la matematica", "exam.xml.bad root": "L'elemento root dell'XML dell'esame dovrebbe essere 'exam'", "exam.changeQuestion.no questions": "Questo esame non contiene domande! Controlla gli errori nel file .exam.", "feedback.you were awarded": "Ti sono stati assegnati <strong>{{count,niceNumber}}</strong> $t(mark).", "feedback.taken away": "<strong>{{count,niceNumber}}</strong> $t(mark) $t(was) tolto.", "jme.tokenise.invalid": "Invalid expression: <code>{{expression}}</code>", "jme.shunt.not enough arguments": "Non ci sono abbastanza argomenti per l'operazione <code>{{op}}</code>", "jme.shunt.no left bracket in function": "Manca la parentesi sinistra nell'applicare una funzione o in un vettore", "jme.shunt.no left square bracket": "Manca una parentesi quadra sinistra", "jme.shunt.no left bracket": "Manca una parentesi sinistra", "jme.shunt.no right bracket": "Manca una parentesi destra", "jme.shunt.no right square bracket": "Manca una parentesi quadra a destra per chiudere la lista", "jme.shunt.missing operator": "L'espressione non pu\u00f2 essere computata -- manca un operatore", "jme.typecheck.function maybe implicit multiplication": "La funzione <code>{{name}}</code> non \u00e8 definita. Volevi dire <code>{{first}}*{{possibleOp}}(...)</code>?", "jme.typecheck.function not defined": "La funzione <code>{{op}}</code> non \u00e8 definita. <code>{{op}}</code> \u00e8 una variabile, e intendevi <code>{{suggestion}}*(...)</code>?", "jme.typecheck.op not defined": "L'operazione '{{op}}' non \u00e8 definita.", "jme.typecheck.no right type definition": "Non ho trovato una definizione di '{{op}}' del tipo giusto.", "jme.typecheck.no right type unbound name": "La variabile <code>{{name}}</code> non \u00e8 definita.", "jme.typecheck.map not on enumerable": "L'operazione <code>map</code> deve funzionare con una lista o un intervallo, non {{type}}", "jme.evaluate.undefined variable": "La variabile {{name}} non \u00e8 definita", "jme.thtml.not html": "\u00c8 stato passato un valore non HTML nel costruttore THTML.", "jme.func.switch.no default case": "No default case for Switch statement", "jme.func.listval.invalid index": "Indice delle liste {{index}} non valido in una lista di dimensione {{size}}", "jme.func.listval.not a list": "Object is not subscriptable", "jme.func.matrix.invalid row type": "Non posso costruire una riga a partire da righe del tipo {{type}}", "jme.func.except.continuous range": "Non posso usare l'operatore 'eccetto' su intervalli continui.", "jme.matrix.reports bad size": "La matrice riporta la sua misura in modo incorretto - ci dev'essere un errore nella funzione di costruzione", "jme.texsubvars.no right bracket": "No matching <code>]</code> in {{op}} arguments.", "jme.texsubvars.missing parameter": "Parametro mancante in {{op}}: {{parameter}}", "jme.texsubvars.no right brace": "No matching <code>}</code> in {{op}}", "jme.user javascript.error": "Error in user-defined javascript function <code>{{name}}</code>: {{-message}}", "jme.variables.error making function": "Error making function <code>{{name}}</code>: {{-message}}", "jme.variables.syntax error in function definition": "Errore di sintassi nella definizione della funzione", "jme.variables.variable not defined": "La variabile <code>{{name}}</code> non \u00e8 definita.", "jme.variables.empty definition": "La definizione della variabile <code>{{name}}</code> \u00e8 vuota.", "jme.variables.circular reference": "Riferimento circolare alle variabili nella definizione di <code>{{name}}</code>", "jme.variables.error computing dependency": "Error computing referenced variable <code>{{name}}</code>", "jme.variables.error evaluating variable": "Error evaluating variable {{name}}: {{-message}}", "jme.variables.question took too many runs to generate variables": "Non \u00e8 stato generato in tempo un insieme valido di variabili per la domanda.", "jme.display.unknown token type": "Can't texify token type {{type}}", "jme.display.collectRuleset.no sets": "No sets given to collectRuleset!", "jme.display.collectRuleset.set not defined": "Ruleset {{name}} has not been defined", "jme.display.simplifyTree.no scope given": "Numbas.jme.display.simplifyTree must be given a Scope", "math.precround.complex": "Non posso arrotondare a un numero complesso di cifre decimali", "math.siground.complex": "Non posso arrotondare a un numero complesso di cifre significative", "math.combinations.complex": "Can't compute combinations of complex numbers", "math.permutations.complex": "Can't compute permutations of complex numbers", "math.gcf.complex": "Non posso calcolare l'MCD di numeri complessi", "math.lcm.complex": "Non posso calcolare l'mcm di numeri complessi", "math.lt.order complex numbers": "Non posso ordinare numeri complessi", "math.choose.empty selection": "Empty selection given to random function", "matrixmath.abs.non-square": "Non posso calcolare il determinante di una matrice non quadrata.", "matrixmath.abs.too big": "Mi dispiace, non so ancora calcolare il determinante di una matrice pi\u00f9 grande di 3x3.", "matrixmath.mul.different sizes": "Non posso moltiplicare matrici di dimensioni diverse.", "vectormath.cross.not 3d": "Can only take the cross product of 3-dimensional vectors.", "vectormath.dot.matrix too big": "Can't calculate dot product of a matrix which isn't $1 \\times N$ or $N \\times 1$.", "vectormath.cross.matrix too big": "Can't calculate cross product of a matrix which isn't $1 \\times N$ or $N \\times 1$.", "part.with steps answer prompt": "Risposta: ", "part.script.error": "Error in part {{path}} custom script {{script}}: {{-message}}", "part.marking.steps no matter": "Poich\u00e9 hai avuto il punteggio pieno per la parte, le tue risposte ai passaggi non vengono contate.", "part.marking.revealed steps no penalty": "You revealed the steps.", "part.marking.used variable replacements": "Questa parte \u00e8 stata valutata usando le tue risposte alle parti precedenti.", "part.marking.variable replacement part not answered": "Devi rispondere prima a {{part}}.", "part.marking.resubmit because of variable replacement": "Il voto di questa parte dipende dalle tue risposte alle altre parti, che hai cambiato. Invia di nuovo questa parte per aggiornare il tuo punteggio.", "part.marking.not submitted": "Non hai inviato una risposta.", "part.marking.did not answer": "Non hai risposto a questa domanda.", "part.marking.nothing entered": "Non hai inserito una risposta.", "part.marking.incorrect": "La tua risposta non \u00e8 corretta.", "part.marking.correct": "La tua risposta \u00e8 corretta.", "part.marking.uncaught error": "Errore nel valutare: {{-message}}", "part.marking.no result": "This part could not be marked.", "part.correct answer": "Risposta attesa:", "part.missing type attribute": "{{part}}: Manca l'attributo del tipo di parte", "part.unknown type": "{{part}}: Tipo di parte {{type}} non riconosciuto", "part.setting not present": "Propriet\u00e0 '{{property}}' non impostata", "part.jme.answer missing": "Manca la risposta corretta", "part.jme.answer too long": "La tua risposta \u00e8 troppo lunga.", "part.jme.answer too short": "La tua risposta \u00e8 troppo corta.", "part.jme.answer invalid": "La tua risposta non \u00e8 un'espressione matematica valida.<br/>{{-message}}.", "part.jme.marking.correct": "La tua risposta \u00e8 numericamente corretta.", "part.jme.must-have bits": "<span class=\"monospace\">{{string}}</span>\n", "part.jme.must-have one": "La tua risposta deve contenere: {{strings}}", "part.jme.must-have several": "La tua risposta deve contenere tutti i seguenti elementi: {{strings}}", "part.jme.not-allowed bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.not-allowed one": "La tua risposta non deve contenere:  {{strings}}", "part.jme.not-allowed several": "La tua risposta non deve contenere nessuno dei seguenti elementi: {{strings}}", "part.jme.unexpected variable name": "La tua risposta \u00e8 stata interpretata come utilizzante il nome della variabile inatteso <code>{{name}}</code>.", "part.jme.unexpected variable name suggestion": "Your answer was interpreted to use the unexpected variable name <code>{{name}}</code>. Did you mean <code>{{suggestion}}</code>?", "part.patternmatch.display answer missing": "Display answer is missing", "part.patternmatch.correct except case": "La tua risposta \u00e8 corretta, eccettuato il caso.", "part.numberentry.correct except decimal": "La tua risposta \u00e8 nell'intervallo permesso, ma non sono permessi numeri decimali.", "part.numberentry.correct except fraction": "La tua risposta \u00e8 nell'intervallo richiesto, ma le frazioni non sono permesse.", "part.numberentry.answer invalid": "Non hai inserito un numero valido.", "part.numberentry.answer not integer": "La tua risposta non \u00e8 valida. Devi inserire un numero intero, non un decimale.", "part.numberentry.answer not integer or decimal": "La tua risposta non \u00e8 valida. Devi inserire un intero o un decimale.", "part.numberentry.zero sig fig": "Questa parte \u00e8 impostata per arrotondare la risposta dello studente a zero cifre significative, il che non ha senso.", "part.mcq.options def not a list": "L'espressione che definisce le {{properties}} non \u00e8 una lista.", "part.mcq.marking matrix string empty": "The custom marking matrix expression is empty.", "part.mcq.choices missing": "Manca la definizione delle scelte", "part.mcq.matrix not a number": "Part {{part}} marking matrix cell ({{row}},{{column}}) does not evaluate to a number", "part.mcq.wrong number of choices": "Hai scelto il numero errato di scelte.", "part.mcq.no choices selected": "Nessuna scelta selezionata.", "part.mcq.matrix not a list": "Marking matrix, defined by JME expression, is not a list but it should be.", "part.mcq.matrix wrong type": "Element of invalid type '{{type}}' used in marking matrix.", "part.mcq.matrix mix of numbers and lists": "Mix of numbers and lists used in marking matrix.", "part.mcq.matrix wrong size": "Marking matrix is the wrong size.", "part.mcq.correct choice": "Hai scelto una risposta corretta.", "part.matrix.answer invalid": "La tua risposta non \u00e8 valida.", "part.matrix.invalid cell": "One or more of the cells in your answer is empty or invalid.", "part.matrix.some incorrect": "One or more of the cells in your answer is incorrect, but you have been awarded marks for the rest.", "part.matrix.empty": "Non hai inserito una risposta.", "part.matrix.empty cell": "Uno o pi\u00f9 campi nella tua risposta sono vuoti.", "part.matrix.size mismatch": "L'autore della domanda non ha permesso allo studente di decidere le dimensioni della loro risposta, ma la risposta corretta \u00e8 {{correct_dimensions}} mentre l'input della risposta \u00e8 {{input_dimensions}}", "part.gapfill.feedback header": "<strong>{{name}}</strong>", "part.extension.not implemented": "Part hasn't implemented the <code>{{name}}</code> method.", "question.loaded name mismatch": "Can't resume this attempt - the package has changed since the last session.", "question.error": "Domanda {{number}}: {{-message}}", "question.preamble.error": "Errore nel preambolo: {{-message}}", "question.preamble.syntax error": "Errore di sintassi nel preambolo", "question.unsupported part type": "Tipo di parte non supportato", "question.header": "Domanda {{number}}", "question.submit part": "Invia parte", "question.show steps": "Mostra i passaggi", "question.show steps penalty": "You will lose <strong>{{count,niceNumber}}</strong> $t(mark).", "question.show steps no penalty": "Il tuo punteggio non sar\u00e0 influenzato.", "question.show steps already penalised": "Hai gi\u00e0 mostrato i passaggi. Puoi mostrarli di nuovo senza ulteriori penalit\u00e0.", "question.hide steps": "Nascondi i passaggi", "question.hide steps no penalty": "Il tuo punteggio non sar\u00e0 influenzato.", "question.advice": "Soluzione", "question.no such part": "Non trovo la parte {{path}}", "question.can not submit": "Non posso inviare la risposta - controlla se hai fatto errori.", "question.answer submitted": "Risposta inviata", "question.score feedback.show": "Mostra il feedback", "question.score feedback.hide": "Nascondi il feedback", "question.score feedback.score total actual": "Punteggio: {{score,niceNumber}}/{{marks,niceNumber}}", "question.score feedback.score total": "{{marksString}}", "question.score feedback.score actual": "Punteggio: {{scoreString}}", "question.score feedback.answered": "Answered", "question.score feedback.unanswered": "Unanswered", "question.score feedback.correct": "La tua risposta \u00e8 corretta", "question.score feedback.partial": "La tua risposta \u00e8 parzialmente corretta", "question.score feedback.wrong": "La tua risposta \u00e8 errata", "question.selector.unsubmitted changes": "Cambiamenti non inviati.", "timing.no accumulator": "no timing accumulator {{name}}", "timing.time remaining": "Tempo rimanente:", "xml.could not load": "Non ho potuto caricare un documento XML: {{-message}}", "xml.property not number": "La propriet\u00e0 {{name}} dovrebbe essere un numero, ma non \u00e8 ({{value}}), nel nodo {{element}}", "xml.property not boolean": "La propriet\u00e0 {{name}} dovrebbe essere una variabile booleana, ma non \u00e8 ({{value}}), nel nodo {{element}}", "xml.error in variable definition": "Error in definition of variable <code>{{name}}</code>", "scorm.error initialising": "Errore nell'inizializzazione del protocollo SCORM: {{-message}}", "scorm.failed save": "<p>The request to save data to the server failed. Press <b>OK</b> to try again.</p>\n<p>If you get this message repeatedly, check your internet connection or use a different computer. Your previously submitted answers have been successfully saved and will be restored if you resume this session on a different computer.</p>\n<p>If this message appears persistently and you can't save <em>any</em> answers, please contact your lecturer or teacher.</p>", "scorm.no exam suspend data": "Failed to resume: no exam suspend data.", "scorm.error loading suspend data": "Error loading suspend data: {{-message}}", "scorm.error loading question": "Error loading question {{number}}: {{-message}}", "scorm.no question suspend data": "No question suspend data", "scorm.error loading part": "Error loading part {{part}}: {{-message}}", "scorm.no part suspend data": "No part suspend data", "util.product.non list": "Passed a non-list to <code>Numbas.util.product</code>", "mark": "mark", "was": "was", "part": "parte", "gap": "gap", "step": "passaggio", "jme.substituteTree.undefined variable": "Variabile non definita: <code>{{name}}</code>", "jme.user javascript.returned undefined": "User-defined javascript function <code>{{name}}</code> returned <code>undefined</code>.", "part.marking.steps change": "You were awarded <strong>{{count,niceNumber}}</strong> $t(mark) for your answers to the steps.", "part.marking.revealed steps with penalty": "You revealed the steps. The maximum you can score for this part is <strong>{{count,niceNumber}}</strong> $t(mark). Your scores will be scaled down accordingly.", "part.marking.total score": "You scored <strong>{{count,niceNumber}}</strong> $t(mark) for this part.", "part.numberentry.precision type.dp": "posizione decimale", "part.numberentry.precision type.dp_plural": "posizioni decimali", "part.numberentry.precision type.sigfig": "cifra significativa", "part.numberentry.precision type.sigfig_plural": "cifre significative", "part.numberentry.give your answer to precision": "Arrotonda la tua risposta a {{count,niceNumber}} {{precisionType}}.", "question.unsubmitted changes": "Hai fatto un cambiamento alla tua risposta ma non l'hai inviato. Per favore controlla la risposta e poi premi il bottone <strong>Invia risposta</strong>", "question.unsubmitted changes_plural": "Hai fatto cambiamenti alle tue risposte ma non li hai inviati. Per favore controlla le tue risposte a ciascuna parte e poi premi il bottone <strong>Invia tutte le parti</strong>", "util.equality not defined for type": "Equality not defined for type {{type}}", "mark_plural": "marks", "was_plural": "were", "die.script not loaded": "Numbas couldn't start because the file <code>{{file}}</code> was not loaded. Check that it's included in <code>scripts.js</code>.", "math.combinations.n less than zero": "Non posso calcolare le combinazioni: n \u00e8 pi\u00f9 piccolo di zero", "math.combinations.k less than zero": "Non posso calcolare le combinazioni: k \u00e8 pi\u00f9 piccolo di zero", "math.combinations.n less than k": "Non posso calcolare le combinazioni: n \u00e8 pi\u00f9 piccolo di k", "math.permutations.n less than zero": "Non posso calcolare le permutazioni: n \u00e8 pi\u00f9 piccolo di zero", "math.permutations.k less than zero": "Non posso calcolare le permutazioni: k \u00e8 pi\u00f9 piccolo di zero", "math.permutations.n less than k": "Non posso calcolare le permutazioni: n \u00e8 pi\u00f9 piccolo di k", "part.numberentry.give your answer to precision_0": "Arrotonda la tua risposta all'intero pi\u00f9 vicino.", "mathjax.error": "MathJax processing error: {{-message}}", "mathjax.error with context": "MathJax processing error in {{-context}}: {{-message}}", "exam.introduction": "Introduzione all'esame", "exam.feedback": "Exam feedback message", "jme.tokenise.keypair key not a string": "Dictionary key should be a string, not {{type}}.", "jme.shunt.list mixed argument types": "Can't parse {{mode}}: mix of dictionary and list elements", "jme.func.listval.key not in dict": "Dictionary does not contain the key <code>{{key}}</code>", "part.prompt": "prompt", "part.feedback": "Feedback", "part.numberentry.answer not reduced": "La tua risposta non \u00e8 ridotta ai minimi termini.", "part.numberentry.give your answer as a reduced fraction": "Riduci la tua risposta ai minimi termini.", "part.numberentry.negative decimal places": "Questa parte \u00e8 impostata per arrotondare la risposta dello studente a un numero negativo di cifre decimali, che non ha senso.", "part.mcq.choices": "opzioni", "part.mcq.answers": "risposta", "part.mcq.matrix cell empty": "Part {{part}} marking matrix cell ({{row}},{{column}}) is empty", "part.mcq.matrix jme error": "Part {{part}} marking matrix cell ({{row}},{{column}}) gives a JME error: {{-error}}", "question.statement": "Enunciato", "ruleset.circular reference": "Circular reference in definition of ruleset <code>{{name}}</code>", "ruleset.set not defined": "L'insieme di regole {{name}} non \u00e8 stato definito", "jme.evaluate.no scope given": "Numbas.jme.evaluate must be given a Scope", "question.score feedback.answered total actual": "Score: {{score,niceNumber}}/{{marks,niceNumber}}", "question.score feedback.answered total": "{{marksString}}. Answered.", "question.score feedback.answered actual": "Score: {{scoreString}}", "question.score feedback.unanswered total": "{{marksString}}.", "answer.number.not a number": "La risposta non \u00e8 valida", "answer.number.fractions not allowed": "Non puoi inserire una frazione.", "answer.jme.invalid expression": "{{-message}}", "answer.matrix.fractions not allowed": "Non puoi inserire frazioni.", "answer.matrix.some cell not a number": "One or more of the cells in your answer is not a valid number.", "exam.enter password": "Password:", "exam.password.correct": "Questa password \u00e8 corretta. Puoi cominciare l'esame.", "exam.password.incorrect": "Questa password \u00e8 errata.", "frontpage.scorm.lms not connected": "This exam is running in standalone mode. Your answers and marks will not be saved!", "result.question review": "Revisione", "control.confirm regen no marks": "Vorresti ri-randomizzare questa domanda?", "control.confirm reveal no marks": "Vorresti rivelare la risposta a questa domanda?", "jme.tokenise.invalid near": "Invalid expression: <code>{{expression}}</code> at position {{position}} near <code>{{nearby}}</code>", "jme.tokenise.number.object not complex": "Invalid object passed into number constructor.", "jme.subvars.null substitution": "Empty variable substitution: <code>$t(left brace){{str}}$t(right brace)", "jme.type.type already registered": "The data type {{type}} has already been registered so can't be registered again.", "jme.type.no cast method": "Non posso convertire automaticamente da {{from}} a {{to}}.", "jme.display.simplifyTree.empty expression": "L'espressione \u00e8 vuota", "jme.display.simplifyTree.stuck in a loop": "Simplifier is stuck in a loop: <code>{{expr}}</code>", "math.niceNumber.undefined": "Was expecting a number, but got <code>undefined</code>", "math.rangeToList.zero step size": "Can't convert a range with step size zero to a list.", "part.error": "{{path}}: {{-message}}", "part.marking.revealed steps": "Hai mostrato i passaggi.", "part.marking.maximum scaled down": "The maximum you can score for this part is <strong>{{count,niceNumber}}</strong> $t(mark). Your scores will be scaled down accordingly.", "part.marking.minimum score applied": "Il punteggio minimo per questa parte \u00e8 <strong>{{score,niceNumber}}</strong>.", "part.marking.maximum score applied": "The maximum score for this part is <strong>{{score,niceNumber}}</strong>.", "part.marking.error in marking script": "There was an error in this part's marking algorithm. Please report this. {{-message}}", "part.marking.no result after replacement": "Non si \u00e8 potuto attribuire punteggio a questa parte utilizzando le tue risposte alle parti precedenti.", "part.marking.missing required note": "The marking algorithm does not define the note <code>{{note}}</code>", "marking.apply.not a list": "The first argument to <code>apply</code> must be a list, and isn't", "marking.apply marking script.script not found": "Marking script <code>{{name}}</code> not found", "marking.note.compilation error": "Error compiling note <code>{{name}}</code>: {{-message}}", "marking.note.error evaluating note": "Error evaluating note <code>{{name}}</code> - {{-message}}", "marking.note.invalid definition": "Invalid note definition: <code>{{source}}</code>. {{-hint}}", "marking.note.invalid definition.missing colon": "Potrebbe mancare una virgola dopo il nome e la descrizione", "marking.note.invalid definition.description missing closing bracket": "Potrebbe mancare una parentesi di chiusura", "marking.note.empty expression": "Il nodo <code>{{name}}</code> \u00e8 vuoto.", "marking.script.error parsing notes": "Error parsing marking script: {{- message}}", "part.feedback out of date": "This feedback is based on your last submitted answer. Save your changed answer to get updated feedback.", "part.jme.invalid value generator expression": "Invalid value generator expression for variable <code>{{name}}</code>: {{-message}}", "part.mcq.incorrect choice": "Hai scelto una risposta errata.", "part.matrix.not all cells same precision": "Non hai dato ogni cella nella tua risposta con la stessa precisione.", "part.gapfill.error marking gap": "Error marking {{name}}: {{-message}}", "part.custom.empty setting": "Nessun valore dato.", "part.custom.unrecognised input type": "Unrecognised setting type <code>{{input_type}}</code>", "part.custom.error evaluating input option": "Error evaluating input option <code>{{option}}</code>: {{-error}}", "part.custom.input option missing": "Definition of input option <code>{{option}}</code> is missing.", "part.custom.error evaluating setting": "Error evaluating setting <code>{{setting}}</code>: {{-error}}", "question.error creating question": "Errore nella creazione della domanda {{number}}: {{-message}}", "question.score feedback.not marked": "Non valutato", "question.score feedback.partially answered": "Parzialmente risposto", "variable.error in variable definition": "Errore nella definizione della variabile <code>{{name}}</code>", "left brace": "{", "right brace": "}", "extension.not found": "Non sono riuscito a caricare l'estensione <code>{{name}}</code>.", "control.toggle navigation menu": "Toggle the navigation menu", "part.input title": "Risposta alla parte {{name}}", "part.correct answer title": "Risposta attesa alla parte {{name}}", "part.jme.must-match.failed": "La tua risposta non \u00e8 scritta nella forma corretta.", "question.score feedback.none": "", "control.submit part.confirm remove next parts": "<p>One or more subsequent parts depend on your answer to this part. Submitting this part again will invalidate those parts, and remove them from the question. This cannot be undone.</p>\n<p>Would you like to submit this part again?</p>", "control.back to menu": "Go back to the menu", "display.error making html": "Error making HTML in {{contextDescription}}: {{-message}}", "jme.subvars.error compiling": "{{-message}} in <code>{{expression}}</code>", "jme.variables.empty name": "A question variable has not been given a name.", "jme.calculus.unknown derivative": "Don't know how to differentiate <code>{{tree}}</code>", "math.order complex numbers": "Can't order complex numbers", "menu.choose a question": "Choose a question.", "part.choose next part.answered": "What do you want to do next?", "part.choose next part.unanswered": "Or, you could:", "part.choose next part.will be locked": "(This part will be locked)", "part.reached dead end": "There's nothing more to do from here.", "part.next part.penalty amount": "(lose {{count}} $t(mark))", "part.marking.counts towards objective": "This part counts towards the objective <strong>\u201c{{objective}}\u201d</strong>.", "part.numberentry.answer not integer or decimal or fraction": "Your answer is invalid. You must enter an integer, a decimal or a fraction.", "question": "Question", "question.progress": "Question progress", "question.score feedback.unattempted": "Not attempted", "question.score feedback.attempted": "Attempted", "question.score feedback.score actual.plain": "{{scoreString}}", "question.score feedback.score total actual.plain": "{{score,niceNumber}}/{{marks,niceNumber}}", "question.objectives": "Objectives", "question.penalties": "Penalties", "question.back to previous part": "Go back to the previous part", "end.print": "Print your exam transcript", "math.shuffle_together.lists not all the same length": "Not all lists are the same length.", "jme.parse signature.invalid signature string": "Invalid function signature string: {{str}}", "part.custom.expected answer has wrong type": "The expected answer for this part has the wrong type. It should be <code>{{shouldbe}}</code>.", "part.custom.input option has wrong type": "The answer input setting <code>{{option}}</code> has the wrong type. It should be <code>{{shouldbe}}</code>.", "matrix input.size control legend": "Size", "matrix input.rows": "Rows", "matrix input.columns": "Columns", "part.jme.error checking numerically": "There was an error numerically checking your answer: {{-message}}", "part.gapfill.cyclic adaptive marking": "There is a cycle in the adaptive marking for this part: <strong>{{name1}}</strong> relies on <strong>{{name2}}</strong>, which eventually relies on <strong>{{name1}}</strong>.", "modal.style.background colour": "Background colour", "modal.style.text colour": "Text colour", "modal.style.text size": "Text size", "modal.style.explanation": "Use these controls to change the appearance of the exam.", "modal.style.reset to defaults": "Reset to defaults", "modal.style.text size preview": "Most text will be this big.", "control.style options": "Display options", "part.marking.partially correct": "Your answer is partially correct.", "part.marking.error in adaptive marking": "There was an error in the adaptive marking for this part. Please report this. {{-message}}", "page.skip to content": "Skip to content", "result.learning objective": "Learning objective", "jme.interpreted as": "interpreted as", "jme.script.note.compilation error": "Error compiling note <code>{{name}}</code>: {{-message}}", "jme.script.note.error evaluating note": "Error evaluating note <code>{{name}}</code> - {{-message}}", "jme.script.note.invalid definition": "Invalid note definition: <code>{{source}}</code>. {{-hint}}", "jme.script.note.invalid definition.missing colon": "You might be missing a colon after the name and description", "jme.script.note.invalid definition.description missing closing bracket": "You might be missing a closing bracket", "jme.script.note.empty expression": "The note <code>{{name}}</code> is empty.", "jme.script.error parsing notes": "Error parsing marking script: {{- message}}", "matrix input.cell label": "Row {{row}}, column {{column}}", "control.move to next question": "Move to the next question", "diagnostic.use retry": "Use one retry and try this topic again.", "diagnostic.move to next topic": "Move on to the next topic.", "diagnostic.next step question": "What would you like to do next?", "diagnostic.now assessing topic": "Now assessing {{current_topic}}", "diagnostic.one retry left": "You have 1 retry left", "diagnostic.retries left": "You have {{retries}} retries left.", "diagnostic.percentage completed": "You've completed <strong>{{percentage}}%</strong> of the test.", "diagnostic.test is over": "The test is over.", "diagnostic.passed all lo": "You have passed all learning objectives.", "diagnostic.more work on lo": "You need to do some more work on the following learning objectives: {{los}}.", "diagnostic.move to next question in topic": "Move on to the next question in topic.", "diagnostic.complete": "Complete!", "diagnostic.studying topic": "Studying {{topic}}", "display.answer widget.unknown widget type": "The answer widget type <code>{{name}}</code> is not recognised.", "jme.shunt.expected argument before comma": "Expected to see something between the opening bracket and the comma", "part.waiting for pre submit": "Your answer is being marked. Please wait.", "diagnostic.end test": "End the test.", "page.no stylesheet loaded": "The page's stylesheet file has not loaded.", "modal.confirm": "Confirm", "modal.alert": "Alert", "suspend.resumed header": "Attempt resumed", "jme.vector.value not an array of numbers": "Tried to construct a vector using a value that is not an array of numbers.", "jme.matrix.value not the right type": "Tried to construct a vector using a value of the wrong type.", "jme.subvars.html inserted twice": "An HTML value has been embedded twice. Consider defining a function to generate a new value each time it is used.", "jme.variables.invalid function language": "The language <code>{{language}}</code> is not valid.", "jme.variables.duplicate definition": "There is more than one definition of the variable <code>{{name}}</code>.", "math.random_integer_partition.invalid k": "The size of the partition must be between 1 and {{n}}.", "part.marking.parameter already in scope": "There is a variable named <code>{{name}}</code>, which is also the name of a marking parameter. Please rename the variable.", "part.marking.adaptive variable replacement refers to self": "This part refers to itself in a variable replacement for adaptive marking.", "part.marking.adaptive variable replacement refers to nothing": "This part contains an invalid variable replacement for adaptive marking.", "part.numberentry.display answer wrong type": "The display answer for this part is a value of type <code>{{got_type}}</code>, but should be a <code>{{want_type}}</code>.", "part.matrix.invalid type in prefilled": "There is an invalid value of type <code>{{n}}</code> in the array of pre-filled cells.", "diagnostic.make a choice": "Make a choice", "matrixmath.not square": "This operation only works on a square matrix.", "matrixmath.not invertible": "This operation only works on an invertible matrix.", "matrixmath.gauss-jordan elimination.not enough columns": "There must be at least as many columns as rows.", "question.required extension not available": "This question requires the extension <code>{{-extension}}</code> but it is not available.", "util.formatNumberNotation.unrecognised syntax": "The number formatting syntax <code>{{syntax}}</code> is not recognised.", "worksheet.number of exams to generate": "Number of sheets to generate", "worksheet.starting with id": "Starting with ID", "worksheet.show exam id": "Show sheet ID?", "worksheet.page break between questions": "Page breaks between questions?", "worksheet.page margins": "Page margins (mm)", "worksheet.text size": "Text size (pt)", "worksheet.generate": "Generate", "worksheet.generating exams": "Generating sheets", "worksheet.sheet id": "Sheet ID:", "worksheet.print single": "Print this sheet", "worksheet.print several": "Print these sheets", "worksheet.answer sheets": "Answer sheets", "worksheet.question sheets": "Question sheets", "worksheet.reconfigure": "Generate different sheets", "worksheet.show sheet": "Preview the sheet with ID:", "accessibility statement": "Accessibility statement and guide to adapting Numbas to your needs.", "exam.enter your name": "Your name:", "exam.attempt download security warning": "This exam is configured to allow you to download your data, but it is not running in a secure browser context. You will not be able to download your exam data. Contact your lecturer or teacher for help.", "result.download exam object": "Download your exam data", "control.return to question": "Return to the question", "control.show introduction": "Introduction", "analysis.header": "Analyse attempt data", "analysis.help.upload files": "Upload attempt data files that your students have given you.", "analysis.help.file input label": "Choose student attempt data files, or drag files onto this window.", "analysis.table.total": "Exam totals", "analysis.table.question": "Exam and question totals", "analysis.table.all": "All details", "analysis.student name.anonymous": "No name given", "analysis.expected": "Expected results", "analysis.start time": "Start time", "analysis.maximum": "Maximum Marks", "analysis.file": "File", "analysis.download": "Download", "analysis.delete": "Delete", "analysis.view results": "View results", "analysis.upload files": "Upload files", "analysis.upload more": "Upload more files", "analysis.attempt data": "Attempt data", "analysis.select format": "Select data to show", "analysis.download this table": "Download this table", "analysis.student": "Student Results", "analysis.question key": "Question Key", "analysis.question name": "Question Name", "analysis.group": "Group", "analysis.question": "Question", "analysis.part": "Part", "analysis.gap": "Gap", "analysis.record type": "Record Type", "analysis.score": "Score", "analysis.marks available": "Marks Available", "analysis.percentage": "Percentage", "analysis.answer": "Answer", "analysis.student name": "Student Name", "analysis.summary.no files": "You have not uploaded any files yet.", "analysis.summary.no decrypted files": "You have not uploaded any valid files yet.", "analysis.summary.one file": "One attempt.", "analysis.summary.several files": "{{num_files,niceNumber}} attempts.", "analysis.not secure context": "This page must be opened in a secure browser context. A secure context is either a page served over HTTPS, or a file loaded from your device.", "jme.shunt.pipe right hand takes no arguments": "The expression on the right-hand side of the pipe operator must be a function application.", "question.explore.no parts defined": "There are no parts defined in this question.", "answer": "answer", "worksheet.answersheet show question content": "Show question content in answer sheets?", "modal.confirm end exam": "Write <code>{{endConfirmation}}</code> in the box to confirm:", "modal.end exam button": "End exam", "lightbox.zoom in on image": "Zoom in on this image", "exam.progress": "Progress", "exam.questions answered": "{{numAnsweredQuestions}} of {{numQuestions}} questions answered.", "result.question marks available": "Marks Available", "result.question answered": "Answered?", "control.confirm end.correct": "You may now end the exam.", "control.confirm end.incorrect": "This is not the expected text.", "control.confirm end.password": "end", "jme.typecheck.for in name wrong type": "The name in a <code>for</code> statement must be a name or list of names, not {{type}}.", "jme.makeFast.no fast definition of function": "The function <code>{{name}}</code> here isn't defined in a way that can be made fast.", "part.show feedback": "Show feedback", "part.hide feedback": "Hide feedback", "part.feedback title": "Feedback for {{name}}.", "part.jme.must-match.warning": "Your answer is not in the expected form: {{-message}}", "part.numberentry.write your answer as a fraction": "Write your answer as a fraction.", "question.nav.label": "Question controls", "question.answer saved": "Answer saved", "question.all answers saved": "All answers saved", "analysis.back to results": "Back to results", "analysis.review": "Review", "analysis.review this": "Review this attempt", "analysis.reviewing attempt": "Reviewing attempt by {{student_name}}.", "part.there is new feedback": "The feedback has changed.", "modal.style.colour scheme": "Colour scheme", "modal.style.automatic colour scheme": "Automatic", "modal.style.light colour scheme": "Light", "modal.style.dark colour scheme": "Dark", "modal.style.custom colour scheme": "Custom", "modal.style.main font": "Main font", "modal.style.font.sans serif": "Sans serif", "modal.style.font.serif": "Serif", "modal.style.font.monospace": "Monospace", "modal.style.spacing": "Spacing", "modal.style.font weight": "Font weight", "modal.style.forced colours warning": "Your browser has overridden the colours used in this page because of a setting such as high contrast mode. Changes to the colour scheme settings here might not have any effect.", "modal.style.text preview": "Most text will look like this.", "modal.style.more options": "More options", "modal.style.main colour": "Main (brand) colour", "modal.style.primary colour": "Primary button colour", "modal.style.link colour": "Link colour", "modal.style.success colour": "Success/correct colour", "modal.style.info colour": "Info colour", "modal.style.warning colour": "Warning colour", "modal.style.danger colour": "Danger/incorrect colour", "modal.style.muted colour": "Muted text colour", "modal.style.highlight colour": "Highlight colour", "exam.error loading exam definition": "There was an error while loading the exam definition: {{text}}", "exam.no exam definition": "No exam definition was given.", "jme.typecheck.wrong arguments for anonymous function": "Wrong number of arguments for this anonymous function.", "worksheet.top": "Top", "worksheet.left": "Left", "worksheet.bottom": "Bottom", "worksheet.right": "Right"}}, "nb-no": {"translation": {"page.loading": "Laster...", "page.saving": "<p>Lagrer.</p> <p>Dette kan ta noen sekunder.</p>", "mathjax.math processing error": "\"{{-message}}\" when texifying <code>{{expression}}</code>", "die.numbas failed": "Numbas har feilet", "die.sorry": "Beklager, det har oppst\u00e5tt en feil, og Numbas kan ikke fortsette. Nedenfor er en beskrivelse av feilen.", "die.error": "Feil", "modal.ok": "OK", "modal.cancel": "Avbryt", "exam.exam name": "Eksamen navn:", "exam.random seed": "Sesjon ID:", "exam.student name": "Studentens navn:", "exam.number of questions": "Antall sp\u00f8rsm\u00e5l:", "exam.marks available": "Mulige poeng:", "exam.pass percentage": "Grense for best\u00e5tt:", "exam.time allowed": "Tillatt tid:", "exam.passed": "Best\u00e5tt", "exam.failed": "Ikke best\u00e5tt", "exam.review header": "Gjennomgang: ", "frontpage.start": "Start", "suspend.paused header": "Pause", "suspend.exam suspended": "Eksamen er avbrutt. Klikk Fortsett for \u00e5 g\u00e5 videre.", "suspend.you can resume": "Du kan fortsette eksamen neste gang du starter denne aktiviteten.", "suspend.resume": "Fortsett", "result.exit": "Avslutt eksamen", "result.print": "Skriv ut detaljert resultat", "result.exam summary": "Eksamen oversikt", "result.performance summary": "Resultatsammendrag", "result.exam start": "Eksamen start:", "result.exam stop": "Eksamen slutt:", "result.time spent": "Tidsbruk:", "result.questions attempted": "Antall besvarte sp\u00f8rsm\u00e5l:", "result.score": "Poengsum:", "result.result": "Resultat:", "result.question number": "Sp\u00f8rsm\u00e5l nummer", "result.question score": "Poengsum", "result.question review title": "G\u00e5 igjennom dette sp\u00f8rsm\u00e5let", "result.click a question to review": "Klikk p\u00e5 et sp\u00f8rsm\u00e5lnummer for \u00e5 se karaktersetting, og om mulig, fullstendig l\u00f8sning.", "end.exam has finished": "Eksamen er avsluttet. Du kan n\u00e5 lukke vinduet.", "control.confirm leave": "Du har ikke levert besvarelse.", "control.not all questions answered": "Du har ikke svart p\u00e5 alle sp\u00f8rsm\u00e5lene i denne eksamen.", "control.not all questions submitted": "Du har endret ett eller flere svar men ikke lagret dem. Vennligst se om svarene er lagret.", "control.confirm end": "Er du sikker p\u00e5 at du vil avslutte? Etter dette vil du ikke kunne endre p\u00e5 svarene dine..", "control.confirm regen": "Vil du lage nye tilfeldige tall i denne oppgaven? Hvis du klikker OK vil svarene og oppn\u00e5dde poeng bli annullert.", "control.confirm reveal": "Vil du se svaret p\u00e5 dette sp\u00f8rsm\u00e5let? Alle poeng du har f\u00e5tt hittil vil bli l\u00e5st - og du kan ikke besvare dette sp\u00f8rsm\u00e5let senere.", "control.proceed anyway": "Fortsett likevel?", "control.regen": "Pr\u00f8v et lignende sp\u00f8rsm\u00e5l", "control.submit answer": "Send inn svar", "control.submit all parts": "Send inn alle delsvar", "control.submit again": "Send inn p\u00e5 nytt", "control.submit": "Send inn", "control.previous": "Forrige", "control.next": "Neste", "control.advice": "Svarforslag", "control.reveal": "Vis svar", "control.total": "Totalt", "control.pause": "Pause", "control.end exam": "Avslutt eksamen", "control.back to results": "Go back to results", "display.part.jme.error making maths": "Feil i visning av matematisk uttrykk", "exam.xml.bad root": "Elementet p\u00e5 \u00f8verste niv\u00e5 i XML m\u00e5 v\u00e6re 'exam'", "exam.changeQuestion.no questions": "Eksamen inneholder ingen sp\u00f8rsm\u00e5l! Sjekk .exam-fila for feil.", "feedback.you were awarded": "Du oppn\u00e5dde {{count}} $t(mark).", "feedback.taken away": "<strong>{{count}}</strong> $t(mark) er trukket fra.", "jme.tokenise.invalid": "Ugyldig uttrykk: <code>{{expression}}</code>", "jme.shunt.not enough arguments": "Det mangler argumenter for \u00e5 utf\u00f8re {{op}}", "jme.shunt.no left bracket in function": "Venstre parentes mangler i funksjon eller tuppel", "jme.shunt.no left square bracket": "Venstre parentes mangler", "jme.shunt.no left bracket": "Venstre parentes mangler", "jme.shunt.no right bracket": "H\u00f8yre parentes mangler", "jme.shunt.no right square bracket": "H\u00f8yre parentes mangler i slutten av liste", "jme.shunt.missing operator": "Uttrykket kan ikke evalueres -- operator mangler.", "jme.typecheck.function maybe implicit multiplication": "Operasjon {{name}} er ikke definert. Mente du <br/><code>{{first}}*{{possibleOp}}(...)</code>?", "jme.typecheck.function not defined": "Operasjon {{op}} er ikke definert. Mente du <br/><code>{{suggestion}}*(...)</code>?", "jme.typecheck.op not defined": "Operasjon '{{op}}' er ikke definert.", "jme.typecheck.no right type definition": "Finner ikke definisjon av '{{op}}' med korrekt type.", "jme.typecheck.no right type unbound name": "Variabel <code>{{name}}</code> er ikke definert.", "jme.typecheck.map not on enumerable": "<code>map</code> operasjonen m\u00e5 gjelde en liste eller range, ikke {{type}}", "jme.evaluate.undefined variable": "Variabel {{name}} er udefinert", "jme.thtml.not html": "Passed a non-HTML value into the THTML constructor.", "jme.func.switch.no default case": "Switch-setning mangler standardverdi", "jme.func.listval.invalid index": "Ugyldig listeindeks {{index}} for en liste med st\u00f8rrelse {{size}}", "jme.func.listval.not a list": "Objektet kan ikke indekseres", "jme.func.matrix.invalid row type": "Kan ikke danne matrise ut fra rader av type {{type}}", "jme.func.except.continuous range": "Kan ikke bruke operator 'except' p\u00e5 et kontinuerlig omr\u00e5de.", "jme.matrix.reports bad size": "Matrix reports its size incorrectly - must be an error in constructor function", "jme.texsubvars.no right bracket": "Ingen samsvarende <code>]</code> i {{op}} argumenter.", "jme.texsubvars.missing parameter": "Manglende parameter in {{op}}: {{parameter}}", "jme.texsubvars.no right brace": "Ingen samsvarende <code>}</code> i {{op}}.", "jme.user javascript.error": "Feil i brukerdefinert javascript funksjon <code>{{name}}</code><br/>{{-message}}", "jme.variables.error making function": "Feil med funksjonskode <code>{{name}}</code>: {{-message}}", "jme.variables.syntax error in function definition": "Syntax feil i definisjonen av funksjon", "jme.variables.variable not defined": "Variabel <code>{{name}}</code> er ikke definert.", "jme.variables.empty definition": "Definisjonen av variabel {{name}} er tom.", "jme.variables.circular reference": "Sirkul\u00e6r referanse til variabel <code>{{name}}</code>", "jme.variables.error computing dependency": "Feil ved beregning av referert variabel <code>{{name}}</code>", "jme.variables.error evaluating variable": "Feil ved evaluering av variabel {{name}}: {{-message}}", "jme.variables.question took too many runs to generate variables": "Et gyldig sett med variabler i sp\u00f8rsm\u00e5l ble ikke generert p\u00e5 normal tid", "jme.display.unknown token type": "Kan ikke lage tekstbilde av token type {{type}}", "jme.display.collectRuleset.no sets": "Ingen sett ble oppgitt til collectRuleset!", "jme.display.collectRuleset.set not defined": "Regelsett {{name}} er ikke definert", "jme.display.simplifyTree.no scope given": "Numbas.jme.display.simplifyTree m\u00e5 f\u00e5 angitt et Scope", "math.precround.complex": "Kan ikke avrunde til antall desimaler gitt som komplekst tall", "math.siground.complex": "Kan ikke avrunde til antall signifikante siffer gitt som komplekst tall", "math.combinations.complex": "Kan ikke beregne kombinasjoner for komplekse tall", "math.permutations.complex": "Kan ikke beregne permutasjoner for komplekse tall", "math.gcf.complex": "Kan ikke beregne GCF for komplekse tall", "math.lcm.complex": "Kan ikke beregne LCM for komplekse tall", "math.lt.order complex numbers": "Kan ikke sortere komplekse tall", "math.choose.empty selection": "Slumpfunksjon har tomt tallomr\u00e5de", "matrixmath.abs.non-square": "Kan ikke beregne determinanten til en matrise som ikke er kvadratisk.", "matrixmath.abs.too big": "Kan ikke beregne determinanten til en matrise st\u00f8rre enn 3x3.", "matrixmath.mul.different sizes": "Kan ikke multiplisere matriser med ulike dimensjoner.", "vectormath.cross.not 3d": "Kan bare beregne kryssprodukt til 3-dimensjonale vektorer.", "vectormath.dot.matrix too big": "Kan ikke beregne prikkproduktet til en matrise som ikke er $1 \\times N$ eller $N \\times 1$.", "vectormath.cross.matrix too big": "Kan ikke beregne kryssproduktet til en matrise som ikke er $1 \\times N$ eller $N \\times 1$.", "part.with steps answer prompt": "Svar: ", "part.script.error": "Feil i del {{path}} brukerdefinert skript {{script}}: {{-message}}", "part.marking.steps no matter": "Ettersom du fikk alt riktig i oppgaven blir ikke delsvarene telt opp.", "part.marking.revealed steps no penalty": "Du valgte \u00e5 se svarforslag.", "part.marking.used variable replacements": "This part was marked using your answers to previous parts.", "part.marking.variable replacement part not answered": "Du m\u00e5 svare p\u00e5 {{part}} f\u00f8rst", "part.marking.resubmit because of variable replacement": "This part's marking depends on your answers to other parts, which you have changed. Save your answer to this part again to update your score.", "part.marking.not submitted": "Du svarte ikke", "part.marking.did not answer": "Du svarte ikke p\u00e5 dette sp\u00f8rsm\u00e5let.", "part.marking.nothing entered": "Du svarte ikke.", "part.marking.incorrect": "Svaret er feil.", "part.marking.correct": "Svaret er riktig.", "part.marking.uncaught error": "Error when marking: {{-message}}", "part.marking.no result": "Denne delen kunne ikke merkes.", "part.correct answer": "Riktig svar:", "part.missing type attribute": "Sp\u00f8rsm\u00e5lsdel har feil atributt", "part.unknown type": "Ukjent sp\u00f8rsm\u00e5lsdel {{type}}", "part.setting not present": "Egenskap '{{property}}' er ikke angitt", "part.jme.answer missing": "Korrekt svar for et JME felt mangler (%s)", "part.jme.answer too long": "Svaret er for langt.", "part.jme.answer too short": "Svaret er for kort.", "part.jme.answer invalid": "Svaret er ikke et gyldig matematisk uttrykk.<br/>{{-message}}", "part.jme.marking.correct": "Svaret er numerisk korrekt.", "part.jme.must-have bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.must-have one": "Svaret m\u00e5 inneholde: {{strings}}", "part.jme.must-have several": "Svaret m\u00e5 inneholde alle: {{strings}}", "part.jme.not-allowed bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.not-allowed one": "Svaret m\u00e5 ikke inneholde: {{strings}}", "part.jme.not-allowed several": "Svaret m\u00e5 ikke inneholde disse: {{strings}}", "part.jme.unexpected variable name": "Svaret ditt er tolket til \u00e5 bruke det uventede variabelnavnet <code>{{name}}</code>.", "part.jme.unexpected variable name suggestion": "Svaret ditt er tolket til \u00e5 bruke det uventede variabelnavnet <code>{{name}}</code>. Mente du <code>{{suggestion}}</code>?", "part.patternmatch.display answer missing": "Display answer is missing from a Pattern Match part (%s)", "part.patternmatch.correct except case": "Svaret er riktig, unntatt i dette tilfellet.", "part.numberentry.correct except decimal": "Svaret er i riktig intervall, men desimaltall er ikke tillatt.", "part.numberentry.correct except fraction": "Ditt svar er innenfor riktig omr\u00e5de, men br\u00f8k er ikke tillatt som svar.", "part.numberentry.answer invalid": "Du svarte ikke med et gyldig tall.", "part.numberentry.answer not integer": "Ditt svar er ikke gyldig. Tast inn et heltall, ikke desimaltall.", "part.numberentry.answer not integer or decimal": "Ditt svar er ikke gyldig. Tast inn et heltall eller et desimaltall.", "part.numberentry.zero sig fig": "This part is set up to round the student's answer to zero significant figures, which has no meaning.", "part.mcq.options def not a list": "The expression defining the {{properties}} is not a list.", "part.mcq.marking matrix string empty": "The custom marking matrix expression is empty.", "part.mcq.choices missing": "Svarmuligheter mangler i flervalgstesten (%s)", "part.mcq.matrix not a number": "Del %s evaluering av matrisecelle {{row}},{{column}} gir ikke et tall", "part.mcq.wrong number of choices": "Du merket av feil antall valg.", "part.mcq.no choices selected": "Ingen av valgene er merket.", "part.mcq.matrix not a list": "Score matrise for flervalg oppgave definert som JME uttrykk er ikke en liste slik den skal v\u00e6re.", "part.mcq.matrix wrong type": "Element av ugyldig type '%s' er brukt i score matrise.", "part.mcq.matrix mix of numbers and lists": "En blanding av tall og lister er brukt i score matrise.", "part.mcq.matrix wrong size": "Score matrise er av feil dimensjon.", "part.mcq.correct choice": "Du valgte riktig svar.", "part.matrix.answer invalid": "Svaret ditt er ugyldig.", "part.matrix.invalid cell": "En eller flere av cellene i ditt svar er tomme eller ugyldige", "part.matrix.some incorrect": "En eller flere av cellene i ditt svar er ikke riktig besvart, men du har f\u00e5tt poeng for resten", "part.matrix.empty": "Du har ikke gitt noe svar.", "part.matrix.empty cell": "En eller flere av cellene i ditt svar er tomme.", "part.matrix.size mismatch": "The question author hasn't allowed the student to decide the dimensions of their answer, but the correct answer is {{correct_dimensions}} while the answer input is {{input_dimensions}}", "part.gapfill.feedback header": "<strong>Boks {{index}}</strong>", "part.extension.not implemented": "Part hasn't implemented the <code>{{name}}</code> method.", "question.loaded name mismatch": "Kan ikke fortsette dette fors\u00f8ket - pakken er endret siden siste sesjon.", "question.error": "Sp\u00f8rsm\u00e5l {{number}}: {{-message}}", "question.preamble.error": "Feil i startkoden: {{-message}}", "question.preamble.syntax error": "Syntaks feil i startkoden", "question.unsupported part type": "Ikke gyldig svartype", "question.header": "Sp\u00f8rsm\u00e5l {{number}}", "question.submit part": "Send inn svar", "question.show steps": "Vis tips", "question.show steps penalty": "Du vil miste <strong>{{count}}</strong> $t(mark).", "question.show steps no penalty": "Din score vil ikke bli p\u00e5virket.", "question.show steps already penalised": "Du har allerede vist tips. Du kan se tips p\u00e5 nytt uten \u00e5 tape poeng.", "question.hide steps": "Skjul tips", "question.hide steps no penalty": "Din score vil ikke bli p\u00e5virket.", "question.advice": "Svarforslag", "question.no such part": "Finner ikke sp\u00f8rsm\u00e5l {{path}}", "question.can not submit": "Kan ikke sende inn svar - sjekk mulige feil.", "question.answer submitted": "Svaret er sendt inn", "question.score feedback.show": "Vis vurdering", "question.score feedback.hide": "Skjul vurdering", "question.score feedback.answered total actual": "Poengsum: {{score}}/{{marks}}", "question.score feedback.answered total": "{{marksString}}. Besvart.", "question.score feedback.answered actual": "Poengsum: {{scoreString}}", "question.score feedback.answered": "Besvart.", "question.score feedback.unanswered": "Ubesvart.", "question.score feedback.unanswered total": "{{marksString}}.", "question.score feedback.correct": "Ditt svar er riktig", "question.score feedback.partial": "Ditt svar er delvis riktig", "question.score feedback.wrong": "Ditt svar er feil", "question.selector.unsubmitted changes": "Endringer som ikke er sendt inn.", "timing.no accumulator": "no timing accumulator {{name}}", "timing.time remaining": "Tid igjen: %s", "xml.could not load": "Kan ikke laste et XML dokument: {{-message}}", "xml.property not number": "Egenskap {{name}} m\u00e5 v\u00e6re et tall, men er ikke ({{value}}), i node {{element}}", "xml.property not boolean": "Egenskap {{name}} m\u00e5 v\u00e6re en boolsk verdi, men er ikke ({{value}}), i node {{element}}", "xml.error in variable definition": "Feil ved definisjon av variabel <code>{{name}}</code>", "scorm.error initialising": "Feil ved initiering av SCORM protokoll: {{-message}}", "scorm.failed save": "<p>Skriving av data til serveren feilet. Klikk <b>OK</b> og pr\u00f8v en gang til.</p>\n<p>Hvis denne feilen gjentar seg ofte b\u00f8r du sjekke forbindelsen til internet eller pr\u00f8ve en annen datamaskin. Dine tidligere innsendte svar er lagret og blir gjentatt hvis du fortsette \u00f8kten p\u00e5 en annen datamaskin.</p>\n<p> Hvis denne gjentar seg ofte og du kan ikke lagre <em>noen</em> svar b\u00f8r du ta kontakt med din l\u00e6rer.</p>", "scorm.no exam suspend data": "Kan ikke fortsette: finner ikke sesjonsdata.", "scorm.error loading suspend data": "Feil ved lasting av sesjonsdata: {{-message}}", "scorm.error loading question": "Feil ved lasting av sp\u00f8rsm\u00e5l {{number}}: {{-message}}", "scorm.no question suspend data": "Ingen sesjonsdata for sp\u00f8rsm\u00e5l", "scorm.error loading part": "Feil ved lasting av del {{part}}: {{-message}}", "scorm.no part suspend data": "Ingen sesjonsdata for delen", "util.product.non list": "Passed a non-list to <code>Numbas.util.product</code>", "mark": "poeng", "was": "var", "part": "del", "gap": "mellomrom", "step": "steg", "jme.substituteTree.undefined variable": "Udefinert variabel: <code>{{name}}</code>", "jme.user javascript.returned undefined": "Brukerdefinert javascript funksjon <code>{{name}}</code> returnerte <code>undefined</code>.", "part.marking.steps change": "Du oppn\u00e5dde <strong>{{count}}</strong> $t(mark) for delsvarene", "part.marking.revealed steps with penalty": "Du valgte \u00e5 se svarforslag. Maksimal poengsum for denne oppgaven er <strong>{{count}}</strong> $t(mark). Din poengsum blir dermed redusert.", "part.marking.total score": "Du fikk <strong>{{count}}</strong> $t(mark) for denne oppgaven.", "part.numberentry.precision type.dp": "decimal place", "part.numberentry.precision type.dp_plural": "decimal places", "part.numberentry.precision type.sigfig": "signifikant siffer", "part.numberentry.precision type.sigfig_plural": "signifikante sifre", "part.numberentry.give your answer to precision": "Round your answer to {{count,niceNumber}} {{precisionType}}.", "question.unsubmitted changes": "Du har gjort endring i svaret, men ikke sendt det inn. Vennligst sjekk svaret og klikk <strong>Send inn svar</strong>.", "question.unsubmitted changes_plural": "Du har gjort endringer i dine svar, men ikke sendt de inn. Vennligst se over svarene og klikk <strong>Send inn alle delsvar</strong>.", "util.equality not defined for type": "Equality not defined for type {{type}}", "mark_plural": "poeng", "was_plural": "var", "die.script not loaded": "Numbas kunne ikke starte fordi filen <code>{{file}}</code> ikke ble lastet inn. Kontroller at den er inkludert i <code>scripts.js</code>", "math.combinations.n less than zero": "Kan ikke utregne kombinasjoner: n er mindre enn null", "math.combinations.k less than zero": "Kan ikke utregne kombinasjoner: k er mindre enn null", "math.combinations.n less than k": "Kan ikke utregne kombinasjoner: n er mindre enn k", "math.permutations.n less than zero": "Kan ikke utregne permutasjonene: n er mindre enn null", "math.permutations.k less than zero": "Kan ikke utregne permutasjonene: k er mindre enn null", "math.permutations.n less than k": "Kan ikke utregne permutasjonene: n er mindre enn k", "part.numberentry.give your answer to precision_0": "Avrund svaret ditt til n\u00e6rmeste heltall.", "mathjax.error": "MathJax processing error: {{-message}}", "mathjax.error with context": "MathJax processing error in {{-context}}: {{-message}}", "exam.introduction": "Exam introduction", "exam.feedback": "Exam feedback message", "jme.tokenise.keypair key not a string": "Dictionary key should be a string, not {{type}}.", "jme.shunt.list mixed argument types": "Can't parse {{mode}}: mix of dictionary and list elements", "jme.func.listval.key not in dict": "Dictionary does not contain the key <code>{{key}}</code>", "part.prompt": "prompt", "part.feedback": "tilbakemelding", "part.numberentry.answer not reduced": "Svaret ditt er ikke forkortet mest mulig.", "part.numberentry.give your answer as a reduced fraction": "Forkort svaret mest mulig.", "part.numberentry.negative decimal places": "This part is set up to round the student's answer to a negative number of decimal places, which has no meaning.", "part.mcq.choices": "valg", "part.mcq.answers": "svar", "part.mcq.matrix cell empty": "Part {{part}} marking matrix cell ({{row}},{{column}}) is empty", "part.mcq.matrix jme error": "Part {{part}} marking matrix cell ({{row}},{{column}}) gives a JME error: {{-error}}", "question.statement": "Uttrykk", "ruleset.circular reference": "Circular reference in definition of ruleset <code>{{name}}</code>", "ruleset.set not defined": "Ruleset {{name}} has not been defined", "jme.evaluate.no scope given": "Numbas.jme.evaluate must be given a Scope", "answer.number.not a number": "Svaret ditt er ikke et gyldig tall.", "answer.number.fractions not allowed": "Du kan ikke skrive en br\u00f8k.", "answer.jme.invalid expression": "{{-message}}", "answer.matrix.fractions not allowed": "Du kan ikke skrive br\u00f8ker.", "answer.matrix.some cell not a number": "One or more of the cells in your answer is not a valid number.", "exam.enter password": "Passord:", "exam.password.correct": "Passordet er korrekt. Du kan starte eksamen.", "exam.password.incorrect": "Feil passord.", "frontpage.scorm.lms not connected": "This exam is running in standalone mode. Your answers and marks will not be saved!", "result.question review": "Review", "control.confirm regen no marks": "Would you like to re-randomise this question?", "control.confirm reveal no marks": "Would you like to reveal the answer to this question?", "jme.tokenise.invalid near": "Invalid expression: <code>{{expression}}</code> at position {{position}} near <code>{{nearby}}</code>", "jme.tokenise.number.object not complex": "Invalid object passed into number constructor.", "jme.subvars.null substitution": "Empty variable substitution: <code>$t(left brace){{str}}$t(right brace)", "jme.type.type already registered": "The data type {{type}} has already been registered so can't be registered again.", "jme.type.no cast method": "Can't automatically convert from {{from}} to {{to}}.", "jme.display.simplifyTree.empty expression": "Uttrykket er tomt", "jme.display.simplifyTree.stuck in a loop": "Simplifier is stuck in a loop: <code>{{expr}}</code>", "math.niceNumber.undefined": "Was expecting a number, but got <code>undefined</code>", "math.rangeToList.zero step size": "Can't convert a range with step size zero to a list.", "part.error": "{{path}}: {{-message}}", "part.marking.revealed steps": "Du avsl\u00f8rte stegene.", "part.marking.maximum scaled down": "The maximum you can score for this part is <strong>{{count,niceNumber}}</strong> $t(mark). Your scores will be scaled down accordingly.", "part.marking.minimum score applied": "The minimum score for this part is <strong>{{score,niceNumber}}</strong>.", "part.marking.maximum score applied": "The maximum score for this part is <strong>{{score,niceNumber}}</strong>.", "part.marking.error in marking script": "There was an error in this part's marking algorithm. Please report this. {{-message}}", "part.marking.no result after replacement": "This part could not be marked using your answers to previous parts.", "part.marking.missing required note": "The marking algorithm does not define the note <code>{{note}}</code>", "marking.apply.not a list": "The first argument to <code>apply</code> must be a list, and isn't", "marking.apply marking script.script not found": "Marking script <code>{{name}}</code> not found", "marking.note.compilation error": "Error compiling note <code>{{name}}</code>: {{-message}}", "marking.note.error evaluating note": "Error evaluating note <code>{{name}}</code> - {{-message}}", "marking.note.invalid definition": "Invalid note definition: <code>{{source}}</code>. {{-hint}}", "marking.note.invalid definition.missing colon": "Mulig du mangler kolon etter navn og beskrivelse", "marking.note.invalid definition.description missing closing bracket": "You might be missing a closing bracket", "marking.note.empty expression": "The note <code>{{name}}</code> is empty.", "marking.script.error parsing notes": "Error parsing marking script: {{- message}}", "part.feedback out of date": "This feedback is based on your last submitted answer. Save your changed answer to get updated feedback.", "part.jme.invalid value generator expression": "Invalid value generator expression for variable <code>{{name}}</code>: {{-message}}", "part.mcq.incorrect choice": "Du valgte et feil svar", "part.matrix.not all cells same precision": "You have not given every cell in your answer to the same precision.", "part.gapfill.error marking gap": "Error marking {{name}}: {{-message}}", "part.custom.empty setting": "No value given.", "part.custom.unrecognised input type": "Unrecognised setting type <code>{{input_type}}</code>", "part.custom.error evaluating input option": "Error evaluating input option <code>{{option}}</code>: {{-error}}", "part.custom.input option missing": "Definition of input option <code>{{option}}</code> is missing.", "part.custom.error evaluating setting": "Error evaluating setting <code>{{setting}}</code>: {{-error}}", "question.error creating question": "Error while creating question {{number}}: {{-message}}", "question.score feedback.not marked": "Not marked", "question.score feedback.partially answered": "Partially answered", "question.score feedback.score total": "{{marksString}}", "question.score feedback.score actual": "Poengsum: {{scoreString}}", "question.score feedback.score total actual": "Poengsum: {{score,niceNumber}}/{{marks,niceNumber}}", "variable.error in variable definition": "Error in definition of variable <code>{{name}}</code>", "left brace": "{", "right brace": "}", "extension.not found": "Couldn't load the extension <code>{{name}}</code>.", "control.toggle navigation menu": "Toggle the navigation menu", "part.input title": "Answer for part {{name}}", "part.correct answer title": "Expected answer for part {{name}}", "part.jme.must-match.failed": "Your answer is not in the expected form.", "question.score feedback.none": "", "control.submit part.confirm remove next parts": "<p>One or more subsequent parts depend on your answer to this part. Submitting this part again will invalidate those parts, and remove them from the question. This cannot be undone.</p>\n<p>Would you like to submit this part again?</p>", "control.back to menu": "Go back to the menu", "display.error making html": "Error making HTML in {{contextDescription}}: {{-message}}", "jme.subvars.error compiling": "{{-message}} in <code>{{expression}}</code>", "jme.variables.empty name": "A question variable has not been given a name.", "jme.calculus.unknown derivative": "Don't know how to differentiate <code>{{tree}}</code>", "math.order complex numbers": "Can't order complex numbers", "menu.choose a question": "Choose a question.", "part.choose next part.answered": "What do you want to do next?", "part.choose next part.unanswered": "Or, you could:", "part.choose next part.will be locked": "(This part will be locked)", "part.reached dead end": "There's nothing more to do from here.", "part.next part.penalty amount": "(lose {{count}} $t(mark))", "part.marking.counts towards objective": "This part counts towards the objective <strong>\u201c{{objective}}\u201d</strong>.", "part.numberentry.answer not integer or decimal or fraction": "Your answer is invalid. You must enter an integer, a decimal or a fraction.", "question": "Question", "question.progress": "Question progress", "question.score feedback.unattempted": "Not attempted", "question.score feedback.attempted": "Attempted", "question.score feedback.score actual.plain": "{{scoreString}}", "question.score feedback.score total actual.plain": "{{score,niceNumber}}/{{marks,niceNumber}}", "question.objectives": "Objectives", "question.penalties": "Penalties", "question.back to previous part": "Go back to the previous part", "end.print": "Print your exam transcript", "math.shuffle_together.lists not all the same length": "Not all lists are the same length.", "jme.parse signature.invalid signature string": "Invalid function signature string: {{str}}", "part.custom.expected answer has wrong type": "The expected answer for this part has the wrong type. It should be <code>{{shouldbe}}</code>.", "part.custom.input option has wrong type": "The answer input setting <code>{{option}}</code> has the wrong type. It should be <code>{{shouldbe}}</code>.", "matrix input.size control legend": "Size", "matrix input.rows": "Rows", "matrix input.columns": "Columns", "part.jme.error checking numerically": "There was an error numerically checking your answer: {{-message}}", "part.gapfill.cyclic adaptive marking": "There is a cycle in the adaptive marking for this part: <strong>{{name1}}</strong> relies on <strong>{{name2}}</strong>, which eventually relies on <strong>{{name1}}</strong>.", "modal.style.background colour": "Background colour", "modal.style.text colour": "Text colour", "modal.style.text size": "Text size", "modal.style.explanation": "Use these controls to change the appearance of the exam.", "modal.style.reset to defaults": "Reset to defaults", "modal.style.text size preview": "Most text will be this big.", "control.style options": "Display options", "part.marking.partially correct": "Your answer is partially correct.", "part.marking.error in adaptive marking": "There was an error in the adaptive marking for this part. Please report this. {{-message}}", "page.skip to content": "Skip to content", "result.learning objective": "Learning objective", "jme.interpreted as": "interpreted as", "jme.script.note.compilation error": "Error compiling note <code>{{name}}</code>: {{-message}}", "jme.script.note.error evaluating note": "Error evaluating note <code>{{name}}</code> - {{-message}}", "jme.script.note.invalid definition": "Invalid note definition: <code>{{source}}</code>. {{-hint}}", "jme.script.note.invalid definition.missing colon": "You might be missing a colon after the name and description", "jme.script.note.invalid definition.description missing closing bracket": "You might be missing a closing bracket", "jme.script.note.empty expression": "The note <code>{{name}}</code> is empty.", "jme.script.error parsing notes": "Error parsing marking script: {{- message}}", "matrix input.cell label": "Row {{row}}, column {{column}}", "control.move to next question": "Move to the next question", "diagnostic.use retry": "Use one retry and try this topic again.", "diagnostic.move to next topic": "Move on to the next topic.", "diagnostic.next step question": "What would you like to do next?", "diagnostic.now assessing topic": "Now assessing {{current_topic}}", "diagnostic.one retry left": "You have 1 retry left", "diagnostic.retries left": "You have {{retries}} retries left.", "diagnostic.percentage completed": "You've completed <strong>{{percentage}}%</strong> of the test.", "diagnostic.test is over": "The test is over.", "diagnostic.passed all lo": "You have passed all learning objectives.", "diagnostic.more work on lo": "You need to do some more work on the following learning objectives: {{los}}.", "diagnostic.move to next question in topic": "Move on to the next question in topic.", "diagnostic.complete": "Complete!", "diagnostic.studying topic": "Studying {{topic}}", "display.answer widget.unknown widget type": "The answer widget type <code>{{name}}</code> is not recognised.", "jme.shunt.expected argument before comma": "Expected to see something between the opening bracket and the comma", "part.waiting for pre submit": "Your answer is being marked. Please wait.", "diagnostic.end test": "End the test.", "page.no stylesheet loaded": "The page's stylesheet file has not loaded.", "modal.confirm": "Confirm", "modal.alert": "Alert", "suspend.resumed header": "Attempt resumed", "jme.vector.value not an array of numbers": "Tried to construct a vector using a value that is not an array of numbers.", "jme.matrix.value not the right type": "Tried to construct a vector using a value of the wrong type.", "jme.subvars.html inserted twice": "An HTML value has been embedded twice. Consider defining a function to generate a new value each time it is used.", "jme.variables.invalid function language": "The language <code>{{language}}</code> is not valid.", "jme.variables.duplicate definition": "There is more than one definition of the variable <code>{{name}}</code>.", "math.random_integer_partition.invalid k": "The size of the partition must be between 1 and {{n}}.", "part.marking.parameter already in scope": "There is a variable named <code>{{name}}</code>, which is also the name of a marking parameter. Please rename the variable.", "part.marking.adaptive variable replacement refers to self": "This part refers to itself in a variable replacement for adaptive marking.", "part.marking.adaptive variable replacement refers to nothing": "This part contains an invalid variable replacement for adaptive marking.", "part.numberentry.display answer wrong type": "The display answer for this part is a value of type <code>{{got_type}}</code>, but should be a <code>{{want_type}}</code>.", "part.matrix.invalid type in prefilled": "There is an invalid value of type <code>{{n}}</code> in the array of pre-filled cells.", "diagnostic.make a choice": "Make a choice", "matrixmath.not square": "This operation only works on a square matrix.", "matrixmath.not invertible": "This operation only works on an invertible matrix.", "matrixmath.gauss-jordan elimination.not enough columns": "There must be at least as many columns as rows.", "question.required extension not available": "This question requires the extension <code>{{-extension}}</code> but it is not available.", "util.formatNumberNotation.unrecognised syntax": "The number formatting syntax <code>{{syntax}}</code> is not recognised.", "worksheet.number of exams to generate": "Number of sheets to generate", "worksheet.starting with id": "Starting with ID", "worksheet.show exam id": "Show sheet ID?", "worksheet.page break between questions": "Page breaks between questions?", "worksheet.page margins": "Page margins (mm)", "worksheet.text size": "Text size (pt)", "worksheet.generate": "Generate", "worksheet.generating exams": "Generating sheets", "worksheet.sheet id": "Sheet ID:", "worksheet.print single": "Print this sheet", "worksheet.print several": "Print these sheets", "worksheet.answer sheets": "Answer sheets", "worksheet.question sheets": "Question sheets", "worksheet.reconfigure": "Generate different sheets", "worksheet.show sheet": "Preview the sheet with ID:", "accessibility statement": "Accessibility statement and guide to adapting Numbas to your needs.", "exam.enter your name": "Your name:", "exam.attempt download security warning": "This exam is configured to allow you to download your data, but it is not running in a secure browser context. You will not be able to download your exam data. Contact your lecturer or teacher for help.", "result.download exam object": "Download your exam data", "control.return to question": "Return to the question", "control.show introduction": "Introduction", "analysis.header": "Analyse attempt data", "analysis.help.upload files": "Upload attempt data files that your students have given you.", "analysis.help.file input label": "Choose student attempt data files, or drag files onto this window.", "analysis.table.total": "Exam totals", "analysis.table.question": "Exam and question totals", "analysis.table.all": "All details", "analysis.student name.anonymous": "No name given", "analysis.expected": "Expected results", "analysis.start time": "Start time", "analysis.maximum": "Maximum Marks", "analysis.file": "File", "analysis.download": "Download", "analysis.delete": "Delete", "analysis.view results": "View results", "analysis.upload files": "Upload files", "analysis.upload more": "Upload more files", "analysis.attempt data": "Attempt data", "analysis.select format": "Select data to show", "analysis.download this table": "Download this table", "analysis.student": "Student Results", "analysis.question key": "Question Key", "analysis.question name": "Question Name", "analysis.group": "Group", "analysis.question": "Question", "analysis.part": "Part", "analysis.gap": "Gap", "analysis.record type": "Record Type", "analysis.score": "Score", "analysis.marks available": "Marks Available", "analysis.percentage": "Percentage", "analysis.answer": "Answer", "analysis.student name": "Student Name", "analysis.summary.no files": "You have not uploaded any files yet.", "analysis.summary.no decrypted files": "You have not uploaded any valid files yet.", "analysis.summary.one file": "One attempt.", "analysis.summary.several files": "{{num_files,niceNumber}} attempts.", "analysis.not secure context": "This page must be opened in a secure browser context. A secure context is either a page served over HTTPS, or a file loaded from your device.", "jme.shunt.pipe right hand takes no arguments": "The expression on the right-hand side of the pipe operator must be a function application.", "question.explore.no parts defined": "There are no parts defined in this question.", "answer": "answer", "worksheet.answersheet show question content": "Show question content in answer sheets?", "modal.confirm end exam": "Write <code>{{endConfirmation}}</code> in the box to confirm:", "modal.end exam button": "End exam", "lightbox.zoom in on image": "Zoom in on this image", "exam.progress": "Progress", "exam.questions answered": "{{numAnsweredQuestions}} of {{numQuestions}} questions answered.", "result.question marks available": "Marks Available", "result.question answered": "Answered?", "control.confirm end.correct": "You may now end the exam.", "control.confirm end.incorrect": "This is not the expected text.", "control.confirm end.password": "end", "jme.typecheck.for in name wrong type": "The name in a <code>for</code> statement must be a name or list of names, not {{type}}.", "jme.makeFast.no fast definition of function": "The function <code>{{name}}</code> here isn't defined in a way that can be made fast.", "part.show feedback": "Show feedback", "part.hide feedback": "Hide feedback", "part.feedback title": "Feedback for {{name}}.", "part.jme.must-match.warning": "Your answer is not in the expected form: {{-message}}", "part.numberentry.write your answer as a fraction": "Write your answer as a fraction.", "question.nav.label": "Question controls", "question.answer saved": "Answer saved", "question.all answers saved": "All answers saved", "analysis.back to results": "Back to results", "analysis.review": "Review", "analysis.review this": "Review this attempt", "analysis.reviewing attempt": "Reviewing attempt by {{student_name}}.", "part.there is new feedback": "The feedback has changed.", "modal.style.colour scheme": "Colour scheme", "modal.style.automatic colour scheme": "Automatic", "modal.style.light colour scheme": "Light", "modal.style.dark colour scheme": "Dark", "modal.style.custom colour scheme": "Custom", "modal.style.main font": "Main font", "modal.style.font.sans serif": "Sans serif", "modal.style.font.serif": "Serif", "modal.style.font.monospace": "Monospace", "modal.style.spacing": "Spacing", "modal.style.font weight": "Font weight", "modal.style.forced colours warning": "Your browser has overridden the colours used in this page because of a setting such as high contrast mode. Changes to the colour scheme settings here might not have any effect.", "modal.style.text preview": "Most text will look like this.", "modal.style.more options": "More options", "modal.style.main colour": "Main (brand) colour", "modal.style.primary colour": "Primary button colour", "modal.style.link colour": "Link colour", "modal.style.success colour": "Success/correct colour", "modal.style.info colour": "Info colour", "modal.style.warning colour": "Warning colour", "modal.style.danger colour": "Danger/incorrect colour", "modal.style.muted colour": "Muted text colour", "modal.style.highlight colour": "Highlight colour", "exam.error loading exam definition": "There was an error while loading the exam definition: {{text}}", "exam.no exam definition": "No exam definition was given.", "jme.typecheck.wrong arguments for anonymous function": "Wrong number of arguments for this anonymous function.", "worksheet.top": "Top", "worksheet.left": "Left", "worksheet.bottom": "Bottom", "worksheet.right": "Right"}}, "tr-tr": {"translation": {"page.loading": "Y\u00fckleniyor...", "page.saving": "<p>Kaydediliyor.</p>\n<p>Bu biraz vakit alabilir.</p>", "mathjax.math processing error": "\"{{-message}}\" metinle\u015ftirilirken <code>{{expression}}</code>", "die.numbas failed": "Numbas hata verdi", "die.sorry": "\u00dczg\u00fcn\u00fcz, Numbas devam edemiyecek seviyede bir hatayla kar\u015f\u0131la\u015ft\u0131. A\u015fa\u011f\u0131dan hatan\u0131n detaylar\u0131na ula\u015fabilirsiniz.", "die.error": "Hata", "modal.ok": "Tamam", "modal.cancel": "\u0130ptal", "exam.exam name": "S\u0131nav Ad\u0131:", "exam.random seed": "Oturum ID'si:", "exam.student name": "\u00d6\u011frenci Ad\u0131:", "exam.number of questions": "Soru adedi:", "exam.marks available": "\u0130\u015faretlenmi\u015f olanlar:", "exam.pass percentage": "Ge\u00e7me Oran\u0131:", "exam.time allowed": "Ge\u00e7en Zaman:", "exam.passed": "Ge\u00e7ti", "exam.failed": "Ba\u015far\u0131s\u0131z", "exam.review header": "\u0130nceleme: ", "frontpage.start": "Ba\u015fla", "suspend.paused header": "Durduruldu", "suspend.exam suspended": "S\u0131nav ask\u0131ya al\u0131nd\u0131. Devam etmek i\u00e7in <em>Devam</em>  tu\u015funa bas\u0131n\u0131z", "suspend.you can resume": "Daha sonra ba\u015flad\u0131\u011f\u0131n\u0131zda bu oturumdan devam edebilirsiniz", "suspend.resume": "Devam", "result.exit": "S\u0131navdan \u00c7\u0131k", "result.print": "Sonucun \u00f6zetini \u00e7\u0131kt\u0131 al", "result.exam summary": "S\u0131nav \u00d6zeti", "result.performance summary": "Performans \u00d6zeti", "result.exam start": "S\u0131nav Ba\u015flad\u0131:", "result.exam stop": "S\u0131nav Bitti:", "result.time spent": "Harcanan Zaman:", "result.questions attempted": "Yap\u0131lan Sorular:", "result.score": "Puan:", "result.result": "Sonu\u00e7:", "result.question number": "Soru Adedi", "result.question score": "Puan", "result.question review title": "Bu soruyu incele", "result.click a question to review": "Cevaplar\u0131n\u0131n nas\u0131l oldu\u011funu g\u00f6rmek i\u00e7in soru adedine t\u0131kla ve e\u011fer mevcutsa \u00e7\u00f6z\u00fcmlerini g\u00f6r", "end.exam has finished": "S\u0131nav bitti. Art\u0131k bu pencereyi kapatabilirsin.", "control.confirm leave": "S\u0131nav\u0131n\u0131 bitirmedin.", "control.not all questions answered": "Bu s\u0131navdaki t\u00fcm sorular\u0131 \u00e7\u00f6zmedin.", "control.not all questions submitted": "Bir veya daha fazla soruda de\u011fi\u015fiklik yapt\u0131n fakat g\u00f6ndermedin. L\u00fctfen her soruyu g\u00f6nderdi\u011finden emin ol.", "control.confirm end": "S\u0131nav\u0131 bitirmek istedi\u011fine emin misin? S\u0131nav\u0131 bitirdikten sonra cevaplar\u0131n\u0131 de\u011fi\u015ftiremezsin.", "control.confirm regen": "Would you like to re-randomise this question? If you click OK, all your answers and marks for the current question will be lost.", "control.confirm reveal": "Would you like to reveal the answer to this question? Any marks you have received so far will be locked and you will not be able to answer this question later.", "control.proceed anyway": "Proceed anyway?", "control.regen": "Bunun gibi ba\u015fka bir soru dene", "control.submit answer": "Cevab\u0131 g\u00f6nder", "control.submit all parts": "T\u00fcm par\u00e7alar\u0131 yolla", "control.submit again": "Tekrar G\u00f6nder", "control.submit": "G\u00f6nder", "control.previous": "\u00d6nceki", "control.next": "Sonraki", "control.advice": "Tavsiye", "control.reveal": "Cevaplar\u0131 \u0130ncele", "control.total": "Total", "control.pause": "Durdur", "control.end exam": "S\u0131nav\u0131 Bitir", "control.back to results": "Sonu\u00e7lara geri d\u00f6n", "display.part.jme.error making maths": "Matematik \u00c7\u0131kt\u0131lar Olu\u015fturulamad\u0131", "exam.xml.bad root": "Root element of exam XML should be 'exam'", "exam.changeQuestion.no questions": "This exam contains no questions! Check the .exam file for errors.", "feedback.you were awarded": "You were awarded <strong>{{count,niceNumber}}</strong> $t(mark).", "feedback.taken away": "<strong>{{count,niceNumber}}</strong> $t(mark) $t(was) taken away.", "jme.tokenise.invalid": "Invalid expression: <code>{{expression}}</code>", "jme.shunt.not enough arguments": "Not enough arguments for operation <code>{{op}}</code>", "jme.shunt.no left bracket in function": "No matching left bracket in function application or tuple", "jme.shunt.no left square bracket": "No matching left bracket", "jme.shunt.no left bracket": "No matching left bracket", "jme.shunt.no right bracket": "No matching right bracket", "jme.shunt.no right square bracket": "No matching right square bracket to end list", "jme.shunt.missing operator": "Expression can't be evaluated -- missing an operator.", "jme.typecheck.function maybe implicit multiplication": "Function <code>{{name}}</code> is not defined. Did you mean <code>{{first}}*{{possibleOp}}(...)</code>?", "jme.typecheck.function not defined": "Function <code>{{op}}</code> is not defined. Is <code>{{op}}</code> a variable, and did you mean <code>{{suggestion}}*(...)</code>?", "jme.typecheck.op not defined": "Operation '{{op}}' is not defined.", "jme.typecheck.no right type definition": "No definition of '{{op}}' of correct type found.", "jme.typecheck.no right type unbound name": "Variable <code>{{name}}</code> is not defined.", "jme.typecheck.map not on enumerable": "<code>map</code> operation must work over a list or a range, not {{type}}", "jme.evaluate.undefined variable": "Variable {{name}} is undefined", "jme.thtml.not html": "Passed a non-HTML value into the THTML constructor.", "jme.func.switch.no default case": "No default case for Switch statement", "jme.func.listval.invalid index": "Invalid list index {{index}} on list of size {{size}}", "jme.func.listval.not a list": "Object is not subscriptable", "jme.func.matrix.invalid row type": "Can't construct a matrix from rows of type {{type}}", "jme.func.except.continuous range": "Can't use the 'except' operator on continuous ranges.", "jme.matrix.reports bad size": "Matrix reports its size incorrectly - must be an error in constructor function", "jme.texsubvars.no right bracket": "No matching <code>]</code> in {{op}} arguments.", "jme.texsubvars.missing parameter": "Missing parameter in {{op}}: {{parameter}}", "jme.texsubvars.no right brace": "No matching <code>}</code> in {{op}}", "jme.user javascript.error": "Error in user-defined javascript function <code>{{name}}</code>: {{-message}}", "jme.variables.error making function": "Error making function <code>{{name}}</code>: {{-message}}", "jme.variables.syntax error in function definition": "Syntax error in function definition", "jme.variables.variable not defined": "Variable <code>{{name}}</code> is not defined.", "jme.variables.empty definition": "Definition of variable <code>{{name}}</code> is empty.", "jme.variables.circular reference": "Circular variable reference in definition of <code>{{name}}</code>", "jme.variables.error computing dependency": "Error computing referenced variable <code>{{name}}</code>", "jme.variables.error evaluating variable": "Error evaluating variable {{name}}: {{-message}}", "jme.variables.question took too many runs to generate variables": "A valid set of question variables was not generated in time.", "jme.display.unknown token type": "Can't texify token type {{type}}", "jme.display.collectRuleset.no sets": "No sets given to collectRuleset!", "jme.display.collectRuleset.set not defined": "Ruleset {{name}} has not been defined", "jme.display.simplifyTree.no scope given": "Numbas.jme.display.simplifyTree must be given a Scope", "math.precround.complex": "Can't round to a complex number of decimal places", "math.siground.complex": "Can't round to a complex number of sig figs", "math.combinations.complex": "Can't compute combinations of complex numbers", "math.permutations.complex": "Can't compute permutations of complex numbers", "math.gcf.complex": "Can't compute GCF of complex numbers", "math.lcm.complex": "Can't compute LCM of complex numbers", "math.lt.order complex numbers": "Can't order complex numbers", "math.choose.empty selection": "Empty selection given to random function", "matrixmath.abs.non-square": "Can't compute the determinant of a matrix which isn't square.", "matrixmath.abs.too big": "Sorry, can't compute the determinant of a matrix bigger than 3x3 yet.", "matrixmath.mul.different sizes": "Can't multiply matrices of different sizes.", "vectormath.cross.not 3d": "Can only take the cross product of 3-dimensional vectors.", "vectormath.dot.matrix too big": "Can't calculate dot product of a matrix which isn't $1 \\times N$ or $N \\times 1$.", "vectormath.cross.matrix too big": "Can't calculate cross product of a matrix which isn't $1 \\times N$ or $N \\times 1$.", "part.with steps answer prompt": "Answer: ", "part.script.error": "Error in part {{path}} custom script {{script}}: {{-message}}", "part.marking.steps no matter": "Because you received full marks for the part, your answers to the steps aren't counted.", "part.marking.revealed steps no penalty": "You revealed the steps.", "part.marking.used variable replacements": "This part was marked using your answers to previous parts.", "part.marking.variable replacement part not answered": "You must answer {{part}} first.", "part.marking.resubmit because of variable replacement": "This part's marking depends on your answers to other parts, which you have changed. Save your answer to this part again to update your score.", "part.marking.not submitted": "No answer submitted.", "part.marking.did not answer": "You did not answer this question.", "part.marking.nothing entered": "You did not enter an answer.", "part.marking.incorrect": "Your answer is incorrect.", "part.marking.correct": "Your answer is correct.", "part.marking.uncaught error": "Error when marking: {{-message}}", "part.marking.no result": "This part could not be marked.", "part.correct answer": "Expected answer:", "part.missing type attribute": "{{part}}: Missing part type attribute", "part.unknown type": "{{part}}: Unrecognised part type {{type}}", "part.setting not present": "Property '{{property}}' not set", "part.jme.answer missing": "Correct answer is missing", "part.jme.answer too long": "Your answer is too long.", "part.jme.answer too short": "Your answer is too short.", "part.jme.answer invalid": "Your answer is not a valid mathematical expression.<br/>{{-message}}.", "part.jme.marking.correct": "Your answer is numerically correct.", "part.jme.must-have bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.must-have one": "Your answer must contain: {{strings}}", "part.jme.must-have several": "Your answer must contain all of: {{strings}}", "part.jme.not-allowed bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.not-allowed one": "Your answer must not contain: {{strings}}", "part.jme.not-allowed several": "Your answer must not contain any of: {{strings}}", "part.jme.unexpected variable name": "Your answer was interpreted to use the unexpected variable name <code>{{name}}</code>.", "part.jme.unexpected variable name suggestion": "Your answer was interpreted to use the unexpected variable name <code>{{name}}</code>. Did you mean <code>{{suggestion}}</code>?", "part.patternmatch.display answer missing": "Display answer is missing", "part.patternmatch.correct except case": "Your answer is correct, except for the case.", "part.numberentry.correct except decimal": "Your answer is within the allowed range, but decimal numbers are not allowed.", "part.numberentry.correct except fraction": "Your answer is within the allowed range, but fractions are not allowed.", "part.numberentry.answer invalid": "You did not enter a valid number.", "part.numberentry.answer not integer": "Your answer is invalid. You must enter a whole number, not a decimal.", "part.numberentry.answer not integer or decimal": "Your answer is invalid. You must enter an integer or a decimal.", "part.numberentry.zero sig fig": "This part is set up to round the student's answer to zero significant figures, which has no meaning.", "part.mcq.options def not a list": "The expression defining the {{properties}} is not a list.", "part.mcq.marking matrix string empty": "The custom marking matrix expression is empty.", "part.mcq.choices missing": "Definition of choices is missing", "part.mcq.matrix not a number": "Part {{part}} marking matrix cell ({{row}},{{column}}) does not evaluate to a number", "part.mcq.wrong number of choices": "You selected the wrong number of choices.", "part.mcq.no choices selected": "No choices selected.", "part.mcq.matrix not a list": "Marking matrix, defined by JME expression, is not a list but it should be.", "part.mcq.matrix wrong type": "Element of invalid type '{{type}}' used in marking matrix.", "part.mcq.matrix mix of numbers and lists": "Mix of numbers and lists used in marking matrix.", "part.mcq.matrix wrong size": "Marking matrix is the wrong size.", "part.mcq.correct choice": "You chose a correct answer.", "part.matrix.answer invalid": "Your answer is not valid.", "part.matrix.invalid cell": "One or more of the cells in your answer is empty or invalid.", "part.matrix.some incorrect": "One or more of the cells in your answer is incorrect, but you have been awarded marks for the rest.", "part.matrix.empty": "You have not entered an answer.", "part.matrix.empty cell": "One or more of the cells in your answer is empty.", "part.matrix.size mismatch": "The question author hasn't allowed the student to decide the dimensions of their answer, but the correct answer is {{correct_dimensions}} while the answer input is {{input_dimensions}}", "part.gapfill.feedback header": "<strong>{{name}}</strong>", "part.extension.not implemented": "Part hasn't implemented the <code>{{name}}</code> method.", "question.loaded name mismatch": "Can't resume this attempt - the package has changed since the last session.", "question.error": "Question {{number}}: {{-message}}", "question.preamble.error": "Error in preamble: {{-message}}", "question.preamble.syntax error": "Syntax error in preamble", "question.unsupported part type": "Unsupported part type", "question.header": "Question {{number}}", "question.submit part": "Save answer", "question.show steps": "Show steps", "question.show steps penalty": "You will lose <strong>{{count,niceNumber}}</strong> $t(mark).", "question.show steps no penalty": "Your score will not be affected.", "question.show steps already penalised": "You have already shown steps. You can show them again with no further penalty.", "question.hide steps": "Hide steps", "question.hide steps no penalty": "Your score will not be affected.", "question.advice": "Advice", "question.no such part": "Can't find part {{path}}.", "question.can not submit": "Can not submit answer - check for errors.", "question.answer submitted": "Answer submitted", "question.score feedback.show": "Show feedback", "question.score feedback.hide": "Hide feedback", "question.score feedback.answered total actual": "Score: {{score,niceNumber}}/{{marks,niceNumber}}", "question.score feedback.answered total": "{{marksString}}. Answered.", "question.score feedback.answered actual": "Score: {{scoreString}}", "question.score feedback.answered": "Answered", "question.score feedback.unanswered": "Unanswered", "question.score feedback.unanswered total": "{{marksString}}.", "question.score feedback.correct": "Your answer is correct", "question.score feedback.partial": "Your answer is partially correct", "question.score feedback.wrong": "Your answer is incorrect", "question.selector.unsubmitted changes": "Unsubmitted changes.", "timing.no accumulator": "no timing accumulator {{name}}", "timing.time remaining": "Time remaining:", "xml.could not load": "Couldn't load an XML document: {{-message}}", "xml.property not number": "Property {{name}} should be a number, but isn't ({{value}}), in node {{element}}", "xml.property not boolean": "Property {{name}} should be a boolean, but isn't ({{value}}), in node {{element}}", "xml.error in variable definition": "Error in definition of variable <code>{{name}}</code>", "scorm.error initialising": "Error initialising SCORM protocol: {{-message}}", "scorm.failed save": "<p>The request to save data to the server failed. Press <b>OK</b> to try again.</p>\n<p>If you get this message repeatedly, check your internet connection or use a different computer. Your previously submitted answers have been successfully saved and will be restored if you resume this session on a different computer.</p>\n<p>If this message appears persistently and you can't save <em>any</em> answers, please contact your lecturer or teacher.</p>", "scorm.no exam suspend data": "Failed to resume: no exam suspend data.", "scorm.error loading suspend data": "Error loading suspend data: {{-message}}", "scorm.error loading question": "Error loading question {{number}}: {{-message}}", "scorm.no question suspend data": "No question suspend data", "scorm.error loading part": "Error loading part {{part}}: {{-message}}", "scorm.no part suspend data": "No part suspend data", "util.product.non list": "Passed a non-list to <code>Numbas.util.product</code>", "mark": "mark", "was": "was", "part": "part", "gap": "gap", "step": "step", "jme.substituteTree.undefined variable": "Undefined variable: <code>{{name}}</code>", "jme.user javascript.returned undefined": "User-defined javascript function <code>{{name}}</code> returned <code>undefined</code>.", "part.marking.steps change": "You were awarded <strong>{{count,niceNumber}}</strong> $t(mark) for your answers to the steps.", "part.marking.revealed steps with penalty": "You revealed the steps. The maximum you can score for this part is <strong>{{count,niceNumber}}</strong> $t(mark). Your scores will be scaled down accordingly.", "part.marking.total score": "You scored <strong>{{count,niceNumber}}</strong> $t(mark) for this part.", "part.numberentry.precision type.dp": "decimal place", "part.numberentry.precision type.dp_plural": "decimal places", "part.numberentry.precision type.sigfig": "significant figure", "part.numberentry.precision type.sigfig_plural": "significant figures", "part.numberentry.give your answer to precision": "Round your answer to {{count,niceNumber}} {{precisionType}}.", "question.unsubmitted changes": "You have made a change to your answer but not submitted it. Please check your answer and then press the <strong>Save answer</strong> button.", "question.unsubmitted changes_plural": "You have made changes to your answers but not submitted them. Please check your answers to each part and then press the <strong>Save all answers</strong> button.", "util.equality not defined for type": "Equality not defined for type {{type}}", "mark_plural": "marks", "was_plural": "were", "die.script not loaded": "Numbas couldn't start because the file <code>{{file}}</code> was not loaded. Check that it's included in <code>scripts.js</code>.", "math.combinations.n less than zero": "Can't compute combinations: n is less than zero", "math.combinations.k less than zero": "Can't compute combinations: k is less than zero", "math.combinations.n less than k": "Can't compute combinations: n is less than k", "math.permutations.n less than zero": "Can't compute permutations: n is less than zero", "math.permutations.k less than zero": "Can't compute permutations: k is less than zero", "math.permutations.n less than k": "Can't compute permutations: n is less than k", "part.numberentry.give your answer to precision_0": "Round your answer to the nearest integer.", "mathjax.error": "MathJax processing error: {{-message}}", "mathjax.error with context": "MathJax processing error in {{-context}}: {{-message}}", "exam.introduction": "Exam introduction", "exam.feedback": "Exam feedback message", "jme.tokenise.keypair key not a string": "Dictionary key should be a string, not {{type}}.", "jme.shunt.list mixed argument types": "Can't parse {{mode}}: mix of dictionary and list elements", "jme.func.listval.key not in dict": "Dictionary does not contain the key <code>{{key}}</code>", "part.prompt": "prompt", "part.feedback": "Feedback", "part.numberentry.answer not reduced": "Your answer is not reduced to lowest terms.", "part.numberentry.give your answer as a reduced fraction": "Reduce your answer to lowest terms.", "part.numberentry.negative decimal places": "This part is set up to round the student's answer to a negative number of decimal places, which has no meaning.", "part.mcq.choices": "choices", "part.mcq.answers": "answers", "part.mcq.matrix cell empty": "Part {{part}} marking matrix cell ({{row}},{{column}}) is empty", "part.mcq.matrix jme error": "Part {{part}} marking matrix cell ({{row}},{{column}}) gives a JME error: {{-error}}", "question.statement": "Statement", "ruleset.circular reference": "Circular reference in definition of ruleset <code>{{name}}</code>", "ruleset.set not defined": "Ruleset {{name}} has not been defined", "jme.evaluate.no scope given": "Numbas.jme.evaluate must be given a Scope", "answer.number.not a number": "Your answer is not a valid number.", "answer.number.fractions not allowed": "You may not enter a fraction.", "answer.jme.invalid expression": "{{-message}}", "answer.matrix.fractions not allowed": "You may not enter fractions.", "answer.matrix.some cell not a number": "One or more of the cells in your answer is not a valid number.", "exam.enter password": "Password:", "exam.password.correct": "This password is correct. You can start the exam.", "exam.password.incorrect": "This password is incorrect.", "frontpage.scorm.lms not connected": "This exam is running in standalone mode. Your answers and marks will not be saved!", "result.question review": "Review", "control.confirm regen no marks": "Would you like to re-randomise this question?", "control.confirm reveal no marks": "Would you like to reveal the answer to this question?", "jme.tokenise.invalid near": "Invalid expression: <code>{{expression}}</code> at position {{position}} near <code>{{nearby}}</code>", "jme.tokenise.number.object not complex": "Invalid object passed into number constructor.", "jme.subvars.null substitution": "Empty variable substitution: <code>$t(left brace){{str}}$t(right brace)", "jme.type.type already registered": "The data type {{type}} has already been registered so can't be registered again.", "jme.type.no cast method": "Can't automatically convert from {{from}} to {{to}}.", "jme.display.simplifyTree.empty expression": "Expression is empty", "jme.display.simplifyTree.stuck in a loop": "Simplifier is stuck in a loop: <code>{{expr}}</code>", "math.niceNumber.undefined": "Was expecting a number, but got <code>undefined</code>", "math.rangeToList.zero step size": "Can't convert a range with step size zero to a list.", "part.error": "{{path}}: {{-message}}", "part.marking.revealed steps": "You revealed the steps.", "part.marking.maximum scaled down": "The maximum you can score for this part is <strong>{{count,niceNumber}}</strong> $t(mark). Your scores will be scaled down accordingly.", "part.marking.minimum score applied": "The minimum score for this part is <strong>{{score,niceNumber}}</strong>.", "part.marking.maximum score applied": "The maximum score for this part is <strong>{{score,niceNumber}}</strong>.", "part.marking.error in marking script": "There was an error in this part's marking algorithm. Please report this. {{-message}}", "part.marking.no result after replacement": "This part could not be marked using your answers to previous parts.", "part.marking.missing required note": "The marking algorithm does not define the note <code>{{note}}</code>", "marking.apply.not a list": "The first argument to <code>apply</code> must be a list, and isn't", "marking.apply marking script.script not found": "Marking script <code>{{name}}</code> not found", "marking.note.compilation error": "Error compiling note <code>{{name}}</code>: {{-message}}", "marking.note.error evaluating note": "Error evaluating note <code>{{name}}</code> - {{-message}}", "marking.note.invalid definition": "Invalid note definition: <code>{{source}}</code>. {{-hint}}", "marking.note.invalid definition.missing colon": "You might be missing a colon after the name and description", "marking.note.invalid definition.description missing closing bracket": "You might be missing a closing bracket", "marking.note.empty expression": "The note <code>{{name}}</code> is empty.", "marking.script.error parsing notes": "Error parsing marking script: {{- message}}", "part.feedback out of date": "This feedback is based on your last submitted answer. Save your changed answer to get updated feedback.", "part.jme.invalid value generator expression": "Invalid value generator expression for variable <code>{{name}}</code>: {{-message}}", "part.mcq.incorrect choice": "You chose an incorrect answer.", "part.matrix.not all cells same precision": "You have not given every cell in your answer to the same precision.", "part.gapfill.error marking gap": "Error marking {{name}}: {{-message}}", "part.custom.empty setting": "No value given.", "part.custom.unrecognised input type": "Unrecognised setting type <code>{{input_type}}</code>", "part.custom.error evaluating input option": "Error evaluating input option <code>{{option}}</code>: {{-error}}", "part.custom.input option missing": "Definition of input option <code>{{option}}</code> is missing.", "part.custom.error evaluating setting": "Error evaluating setting <code>{{setting}}</code>: {{-error}}", "question.error creating question": "Error while creating question {{number}}: {{-message}}", "question.score feedback.not marked": "Not marked", "question.score feedback.partially answered": "Partially answered", "question.score feedback.score total": "{{marksString}}", "question.score feedback.score actual": "Score: {{scoreString}}", "question.score feedback.score total actual": "Score: {{score,niceNumber}}/{{marks,niceNumber}}", "variable.error in variable definition": "Error in definition of variable <code>{{name}}</code>", "left brace": "{", "right brace": "}", "extension.not found": "Couldn't load the extension <code>{{name}}</code>.", "control.toggle navigation menu": "Toggle the navigation menu", "part.input title": "Answer for part {{name}}", "part.correct answer title": "Expected answer for part {{name}}", "part.jme.must-match.failed": "Your answer is not in the expected form.", "question.score feedback.none": "", "control.submit part.confirm remove next parts": "<p>One or more subsequent parts depend on your answer to this part. Submitting this part again will invalidate those parts, and remove them from the question. This cannot be undone.</p>\n<p>Would you like to submit this part again?</p>", "control.back to menu": "Go back to the menu", "display.error making html": "Error making HTML in {{contextDescription}}: {{-message}}", "jme.subvars.error compiling": "{{-message}} in <code>{{expression}}</code>", "jme.variables.empty name": "A question variable has not been given a name.", "jme.calculus.unknown derivative": "Don't know how to differentiate <code>{{tree}}</code>", "math.order complex numbers": "Can't order complex numbers", "menu.choose a question": "Choose a question.", "part.choose next part.answered": "What do you want to do next?", "part.choose next part.unanswered": "Or, you could:", "part.choose next part.will be locked": "(This part will be locked)", "part.reached dead end": "There's nothing more to do from here.", "part.next part.penalty amount": "(lose {{count}} $t(mark))", "part.marking.counts towards objective": "This part counts towards the objective <strong>\u201c{{objective}}\u201d</strong>.", "part.numberentry.answer not integer or decimal or fraction": "Your answer is invalid. You must enter an integer, a decimal or a fraction.", "question": "Question", "question.progress": "Question progress", "question.score feedback.unattempted": "Not attempted", "question.score feedback.attempted": "Attempted", "question.score feedback.score actual.plain": "{{scoreString}}", "question.score feedback.score total actual.plain": "{{score,niceNumber}}/{{marks,niceNumber}}", "question.objectives": "Objectives", "question.penalties": "Penalties", "question.back to previous part": "Go back to the previous part", "end.print": "Print your exam transcript", "math.shuffle_together.lists not all the same length": "Not all lists are the same length.", "jme.parse signature.invalid signature string": "Invalid function signature string: {{str}}", "part.custom.expected answer has wrong type": "The expected answer for this part has the wrong type. It should be <code>{{shouldbe}}</code>.", "part.custom.input option has wrong type": "The answer input setting <code>{{option}}</code> has the wrong type. It should be <code>{{shouldbe}}</code>.", "matrix input.size control legend": "Size", "matrix input.rows": "Rows", "matrix input.columns": "Columns", "part.jme.error checking numerically": "There was an error numerically checking your answer: {{-message}}", "part.gapfill.cyclic adaptive marking": "There is a cycle in the adaptive marking for this part: <strong>{{name1}}</strong> relies on <strong>{{name2}}</strong>, which eventually relies on <strong>{{name1}}</strong>.", "modal.style.background colour": "Background colour", "modal.style.text colour": "Text colour", "modal.style.text size": "Text size", "modal.style.explanation": "Use these controls to change the appearance of the exam.", "modal.style.reset to defaults": "Reset to defaults", "modal.style.text size preview": "Most text will be this big.", "control.style options": "Display options", "part.marking.partially correct": "Your answer is partially correct.", "part.marking.error in adaptive marking": "There was an error in the adaptive marking for this part. Please report this. {{-message}}", "page.skip to content": "Skip to content", "result.learning objective": "Learning objective", "jme.interpreted as": "interpreted as", "jme.script.note.compilation error": "Error compiling note <code>{{name}}</code>: {{-message}}", "jme.script.note.error evaluating note": "Error evaluating note <code>{{name}}</code> - {{-message}}", "jme.script.note.invalid definition": "Invalid note definition: <code>{{source}}</code>. {{-hint}}", "jme.script.note.invalid definition.missing colon": "You might be missing a colon after the name and description", "jme.script.note.invalid definition.description missing closing bracket": "You might be missing a closing bracket", "jme.script.note.empty expression": "The note <code>{{name}}</code> is empty.", "jme.script.error parsing notes": "Error parsing marking script: {{- message}}", "matrix input.cell label": "Row {{row}}, column {{column}}", "control.move to next question": "Move to the next question", "diagnostic.use retry": "Use one retry and try this topic again.", "diagnostic.move to next topic": "Move on to the next topic.", "diagnostic.next step question": "What would you like to do next?", "diagnostic.now assessing topic": "Now assessing {{current_topic}}", "diagnostic.one retry left": "You have 1 retry left", "diagnostic.retries left": "You have {{retries}} retries left.", "diagnostic.percentage completed": "You've completed <strong>{{percentage}}%</strong> of the test.", "diagnostic.test is over": "The test is over.", "diagnostic.passed all lo": "You have passed all learning objectives.", "diagnostic.more work on lo": "You need to do some more work on the following learning objectives: {{los}}.", "diagnostic.move to next question in topic": "Move on to the next question in topic.", "diagnostic.complete": "Complete!", "diagnostic.studying topic": "Studying {{topic}}", "display.answer widget.unknown widget type": "The answer widget type <code>{{name}}</code> is not recognised.", "jme.shunt.expected argument before comma": "Expected to see something between the opening bracket and the comma", "part.waiting for pre submit": "Your answer is being marked. Please wait.", "diagnostic.end test": "End the test.", "page.no stylesheet loaded": "The page's stylesheet file has not loaded.", "modal.confirm": "Confirm", "modal.alert": "Alert", "suspend.resumed header": "Attempt resumed", "jme.vector.value not an array of numbers": "Tried to construct a vector using a value that is not an array of numbers.", "jme.matrix.value not the right type": "Tried to construct a vector using a value of the wrong type.", "jme.subvars.html inserted twice": "An HTML value has been embedded twice. Consider defining a function to generate a new value each time it is used.", "jme.variables.invalid function language": "The language <code>{{language}}</code> is not valid.", "jme.variables.duplicate definition": "There is more than one definition of the variable <code>{{name}}</code>.", "math.random_integer_partition.invalid k": "The size of the partition must be between 1 and {{n}}.", "part.marking.parameter already in scope": "There is a variable named <code>{{name}}</code>, which is also the name of a marking parameter. Please rename the variable.", "part.marking.adaptive variable replacement refers to self": "This part refers to itself in a variable replacement for adaptive marking.", "part.marking.adaptive variable replacement refers to nothing": "This part contains an invalid variable replacement for adaptive marking.", "part.numberentry.display answer wrong type": "The display answer for this part is a value of type <code>{{got_type}}</code>, but should be a <code>{{want_type}}</code>.", "part.matrix.invalid type in prefilled": "There is an invalid value of type <code>{{n}}</code> in the array of pre-filled cells.", "diagnostic.make a choice": "Make a choice", "matrixmath.not square": "This operation only works on a square matrix.", "matrixmath.not invertible": "This operation only works on an invertible matrix.", "matrixmath.gauss-jordan elimination.not enough columns": "There must be at least as many columns as rows.", "question.required extension not available": "This question requires the extension <code>{{-extension}}</code> but it is not available.", "util.formatNumberNotation.unrecognised syntax": "The number formatting syntax <code>{{syntax}}</code> is not recognised.", "worksheet.number of exams to generate": "Number of sheets to generate", "worksheet.starting with id": "Starting with ID", "worksheet.show exam id": "Show sheet ID?", "worksheet.page break between questions": "Page breaks between questions?", "worksheet.page margins": "Page margins (mm)", "worksheet.text size": "Text size (pt)", "worksheet.generate": "Generate", "worksheet.generating exams": "Generating sheets", "worksheet.sheet id": "Sheet ID:", "worksheet.print single": "Print this sheet", "worksheet.print several": "Print these sheets", "worksheet.answer sheets": "Answer sheets", "worksheet.question sheets": "Question sheets", "worksheet.reconfigure": "Generate different sheets", "worksheet.show sheet": "Preview the sheet with ID:", "accessibility statement": "Accessibility statement and guide to adapting Numbas to your needs.", "exam.enter your name": "Your name:", "exam.attempt download security warning": "This exam is configured to allow you to download your data, but it is not running in a secure browser context. You will not be able to download your exam data. Contact your lecturer or teacher for help.", "result.download exam object": "Download your exam data", "control.return to question": "Return to the question", "control.show introduction": "Introduction", "analysis.header": "Analyse attempt data", "analysis.help.upload files": "Upload attempt data files that your students have given you.", "analysis.help.file input label": "Choose student attempt data files, or drag files onto this window.", "analysis.table.total": "Exam totals", "analysis.table.question": "Exam and question totals", "analysis.table.all": "All details", "analysis.student name.anonymous": "No name given", "analysis.expected": "Expected results", "analysis.start time": "Start time", "analysis.maximum": "Maximum Marks", "analysis.file": "File", "analysis.download": "Download", "analysis.delete": "Delete", "analysis.view results": "View results", "analysis.upload files": "Upload files", "analysis.upload more": "Upload more files", "analysis.attempt data": "Attempt data", "analysis.select format": "Select data to show", "analysis.download this table": "Download this table", "analysis.student": "Student Results", "analysis.question key": "Question Key", "analysis.question name": "Question Name", "analysis.group": "Group", "analysis.question": "Question", "analysis.part": "Part", "analysis.gap": "Gap", "analysis.record type": "Record Type", "analysis.score": "Score", "analysis.marks available": "Marks Available", "analysis.percentage": "Percentage", "analysis.answer": "Answer", "analysis.student name": "Student Name", "analysis.summary.no files": "You have not uploaded any files yet.", "analysis.summary.no decrypted files": "You have not uploaded any valid files yet.", "analysis.summary.one file": "One attempt.", "analysis.summary.several files": "{{num_files,niceNumber}} attempts.", "analysis.not secure context": "This page must be opened in a secure browser context. A secure context is either a page served over HTTPS, or a file loaded from your device.", "jme.shunt.pipe right hand takes no arguments": "The expression on the right-hand side of the pipe operator must be a function application.", "question.explore.no parts defined": "There are no parts defined in this question.", "answer": "answer", "worksheet.answersheet show question content": "Show question content in answer sheets?", "modal.confirm end exam": "Write <code>{{endConfirmation}}</code> in the box to confirm:", "modal.end exam button": "End exam", "lightbox.zoom in on image": "Zoom in on this image", "exam.progress": "Progress", "exam.questions answered": "{{numAnsweredQuestions}} of {{numQuestions}} questions answered.", "result.question marks available": "Marks Available", "result.question answered": "Answered?", "control.confirm end.correct": "You may now end the exam.", "control.confirm end.incorrect": "This is not the expected text.", "control.confirm end.password": "end", "jme.typecheck.for in name wrong type": "The name in a <code>for</code> statement must be a name or list of names, not {{type}}.", "jme.makeFast.no fast definition of function": "The function <code>{{name}}</code> here isn't defined in a way that can be made fast.", "part.show feedback": "Show feedback", "part.hide feedback": "Hide feedback", "part.feedback title": "Feedback for {{name}}.", "part.jme.must-match.warning": "Your answer is not in the expected form: {{-message}}", "part.numberentry.write your answer as a fraction": "Write your answer as a fraction.", "question.nav.label": "Question controls", "question.answer saved": "Answer saved", "question.all answers saved": "All answers saved", "analysis.back to results": "Back to results", "analysis.review": "Review", "analysis.review this": "Review this attempt", "analysis.reviewing attempt": "Reviewing attempt by {{student_name}}.", "part.there is new feedback": "The feedback has changed.", "modal.style.colour scheme": "Colour scheme", "modal.style.automatic colour scheme": "Automatic", "modal.style.light colour scheme": "Light", "modal.style.dark colour scheme": "Dark", "modal.style.custom colour scheme": "Custom", "modal.style.main font": "Main font", "modal.style.font.sans serif": "Sans serif", "modal.style.font.serif": "Serif", "modal.style.font.monospace": "Monospace", "modal.style.spacing": "Spacing", "modal.style.font weight": "Font weight", "modal.style.forced colours warning": "Your browser has overridden the colours used in this page because of a setting such as high contrast mode. Changes to the colour scheme settings here might not have any effect.", "modal.style.text preview": "Most text will look like this.", "modal.style.more options": "More options", "modal.style.main colour": "Main (brand) colour", "modal.style.primary colour": "Primary button colour", "modal.style.link colour": "Link colour", "modal.style.success colour": "Success/correct colour", "modal.style.info colour": "Info colour", "modal.style.warning colour": "Warning colour", "modal.style.danger colour": "Danger/incorrect colour", "modal.style.muted colour": "Muted text colour", "modal.style.highlight colour": "Highlight colour", "exam.error loading exam definition": "There was an error while loading the exam definition: {{text}}", "exam.no exam definition": "No exam definition was given.", "jme.typecheck.wrong arguments for anonymous function": "Wrong number of arguments for this anonymous function.", "worksheet.top": "Top", "worksheet.left": "Left", "worksheet.bottom": "Bottom", "worksheet.right": "Right"}}, "en-school": {"translation": {"page.loading": "Loading...", "page.saving": "<p>Saving.</p>\n<p>This might take a few seconds.</p>", "mathjax.math processing error": "MathJax processing error: \"{{message}}\" when texifying <code>{{expression}}</code>", "die.numbas failed": "Numbas has failed", "die.sorry": "Sorry, Numbas has encountered an error which means it can't continue. Below is a description of the error.", "die.script not loaded": "Numbas couldn't start because the file <code>{{file}}</code> was not loaded. Check that it's included in <code>scripts.js</code>.", "die.error": "Error", "modal.ok": "OK", "modal.cancel": "Cancel", "exam.exam name": "Test Name:", "exam.random seed": "Session ID:", "exam.student name": "Student's Name:", "exam.number of questions": "Number of Questions:", "exam.marks available": "Marks Available:", "exam.pass percentage": "Pass Percentage:", "exam.time allowed": "Time Allowed:", "exam.passed": "Passed", "exam.failed": "Failed", "exam.review header": "Review: ", "frontpage.start": "Start the test", "suspend.paused header": "Paused", "suspend.exam suspended": "The test is paused. Press <em>Resume</em> to continue.", "suspend.you can resume": "You will be able to resume this session the next time you start this activity.", "suspend.resume": "Resume", "result.exit": "Exit Test", "result.print": "Print this results summary", "result.review": "Review", "result.exam summary": "Test Summary", "result.performance summary": "Performance Summary", "result.exam start": "Test Start:", "result.exam stop": "Test Stop:", "result.time spent": "Time Spent:", "result.questions attempted": "Questions Attempted:", "result.score": "Score:", "result.result": "Result:", "result.detailed question breakdown": "Detailed Question Breakdown and Feedback", "result.question number": "Question Number", "result.question score": "Score", "result.question review title": "Look at this question again", "result.click a question to review": "Click on a question number to see how your answers were marked.", "end.exam has finished": "The test has finished. You can close this window now.", "control.confirm leave": "You haven't finished the test.", "control.not all questions answered": "You have not completed every question in this test.", "control.not all questions submitted": "You have changed one of your answers but not submitted it. Please check each question has been submitted.", "control.confirm end": "Are you sure you want to end the test?<br>After you end the test, you will not be able to change any of your answers.", "control.confirm regen": "Would you like to re-randomise this question?<br> If you click OK, all your answers and marks for the current question will be lost.", "control.confirm reveal": "Would you like to reveal the answer to this question?<br>Any marks you have received so far will be locked and you will not be able to answer this question later.", "control.proceed anyway": "Proceed anyway?", "control.regen": "Try another question like this one", "control.submit answer": "Check my answer", "control.submit all parts": "Check every part", "control.submit again": "Check again", "control.submit": "Check", "control.previous": "Previous", "control.next": "Next", "control.advice": "Worked solution", "control.reveal": "Reveal answers", "control.total": "Total score", "control.pause": "Pause", "control.end exam": "End Test", "control.back to results": "Go back to results", "display.part.jme.error making maths": "Error making maths display", "exam.xml.bad root": "Root element of exam XML should be 'exam'", "exam.changeQuestion.no questions": "This test contains no questions! Check the .exam file for errors.", "feedback.you were awarded": "You were awarded <strong>{{count,niceNumber}}</strong> $t(mark).", "feedback.taken away": "<strong>{{count,niceNumber}}</strong> $t(mark) $t(was) taken away.", "jme.tokenise.invalid": "Invalid expression: <code>{{expression}}</code>", "jme.shunt.not enough arguments": "Not enough arguments for operation {{op}}", "jme.shunt.no left bracket in function": "No matching left bracket in function application or tuple", "jme.shunt.no left square bracket": "No matching left bracket", "jme.shunt.no left bracket": "No matching left bracket", "jme.shunt.no right bracket": "No matching right bracket", "jme.shunt.no right square bracket": "No matching right square bracket to end list", "jme.shunt.missing operator": "Expression can't be evaluated -- missing an operator.", "jme.typecheck.function maybe implicit multiplication": "Operation {{name}} is not defined. Did you mean <br/><code>{{first}}*{{possibleOp}}(...)</code>?", "jme.typecheck.function not defined": "Operation {{op}} is not defined. Did you mean <br/><code>{{suggestion}}*(...)</code>?", "jme.typecheck.op not defined": "Operation '{{op}}' is not defined.", "jme.typecheck.no right type definition": "No definition of '{{op}}' of correct type found.", "jme.typecheck.no right type unbound name": "Variable <code>{{name}}</code> is not defined.", "jme.typecheck.map not on enumerable": "<code>map</code> operation must work over a list or a range, not {{type}}", "jme.evaluate.undefined variable": "Variable {{name}} is undefined", "jme.substituteTree.undefined variable": "Undefined variable: <code>{{name}}</code>", "jme.thtml.not html": "Passed a non-HTML value into the THTML constructor.", "jme.func.switch.no default case": "No default case for Switch statement", "jme.func.listval.invalid index": "Invalid list index {{index}} on list of size {{size}}", "jme.func.listval.not a list": "Object is not subscriptable", "jme.func.matrix.invalid row type": "Can't construct a matrix from rows of type {{type}}", "jme.func.except.continuous range": "Can't use the 'except' operator on continuous ranges.", "jme.matrix.reports bad size": "Matrix reports its size incorrectly - must be an error in constructor function", "jme.texsubvars.no right bracket": "No matching <code>]</code> in {{op}} arguments.", "jme.texsubvars.missing parameter": "Missing parameter in {{op}}: {{parameter}}", "jme.texsubvars.no right brace": "No matching <code>}</code> in {{op}}", "jme.user javascript.error": "Error in user-defined javascript function <code>{{name}}</code>: {{message}}", "jme.user javascript.returned undefined": "User-defined javascript function <code>{{name}}</code> returned <code>undefined</code>.", "jme.variables.error making function": "Error making function <code>{{name}}</code>: {{message}}", "jme.variables.syntax error in function definition": "Syntax error in function definition", "jme.variables.variable not defined": "Variable <code>{{name}}</code> is not defined.", "jme.variables.empty definition": "Definition of variable <code>{{name}}</code> is empty.", "jme.variables.circular reference": "Circular variable reference in definition of <code>{{name}}</code>", "jme.variables.error computing dependency": "Error computing referenced variable <code>{{name}}</code>", "jme.variables.error evaluating variable": "Error evaluating variable {{name}}: {{message}}", "jme.variables.question took too many runs to generate variables": "A valid set of question variables was not generated in time.", "jme.display.unknown token type": "Can't texify token type {{type}}", "jme.display.collectRuleset.no sets": "No sets given to collectRuleset!", "jme.display.collectRuleset.set not defined": "Ruleset {{name}} has not been defined", "jme.display.simplifyTree.no scope given": "Numbas.jme.display.simplifyTree must be given a Scope", "math.precround.complex": "Can't round to a complex number of decimal places", "math.siground.complex": "Can't round to a complex number of sig figs", "math.combinations.complex": "Can't compute combinations of complex numbers", "math.combinations.n less than zero": "Can't compute combinations: n is less than zero", "math.combinations.k less than zero": "Can't compute combinations: k is less than zero", "math.combinations.n less than k": "Can't compute combinations: n is less than k", "math.permutations.n less than zero": "Can't compute permutations: n is less than zero", "math.permutations.k less than zero": "Can't compute permutations: k is less than zero", "math.permutations.n less than k": "Can't compute permutations: n is less than k", "math.permutations.complex": "Can't compute permutations of complex numbers", "math.gcf.complex": "Can't compute GCF of complex numbers", "math.lcm.complex": "Can't compute LCM of complex numbers", "math.lt.order complex numbers": "Can't order complex numbers", "math.choose.empty selection": "Empty selection given to random function", "matrixmath.abs.non-square": "Can't compute the determinant of a matrix which isn't square.", "matrixmath.abs.too big": "Sorry, can't compute the determinant of a matrix bigger than 3x3 yet.", "matrixmath.mul.different sizes": "Can't multiply matrices of different sizes.", "vectormath.cross.not 3d": "Can only take the cross product of 3-dimensional vectors.", "vectormath.dot.matrix too big": "Can't calculate dot product of a matrix which isn't $1 \\times N$ or $N \\times 1$.", "vectormath.cross.matrix too big": "Can't calculate cross product of a matrix which isn't $1 \\times N$ or $N \\times 1$.", "part.with steps answer prompt": "Answer: ", "part.script.error": "Error in part {{path}} custom script {{script}}: {{message}}", "part.marking.steps no matter": "Because you received full marks for the part, your answers to the steps aren't counted.", "part.marking.steps change": "You were awarded <strong>{{count,niceNumber}}</strong> $t(mark) for your answers to the steps", "part.marking.revealed steps with penalty": "You revealed the steps. The maximum you can score for this part is <strong>{{count,niceNumber}}</strong> $t(mark). Your scores will be scaled down accordingly.", "part.marking.revealed steps no penalty": "You revealed the steps.", "part.marking.used variable replacements": "This part was marked using your answers to previous parts.", "part.marking.variable replacement part not answered": "You must answer {{part}} first", "part.marking.resubmit because of variable replacement": "This part's marking depends on your answers to other parts, which you have changed. Submit this part again to update your score.", "part.marking.not submitted": "No answer submitted", "part.marking.did not answer": "You did not answer this question.", "part.marking.total score": "You scored <strong>{{count,niceNumber}}</strong> $t(mark) for this part.", "part.marking.nothing entered": "You did not enter an answer.", "part.marking.incorrect": "Your answer is incorrect.", "part.marking.correct": "Your answer is correct.", "part.marking.uncaught error": "Error when marking {{part}}: {{message}}", "part.marking.no result": "This part could not be marked.", "part.correct answer": "Expected answer:", "part.missing type attribute": "{{part}}: Missing part type attribute", "part.unknown type": "{{part}}: Unrecognised part type {{type}}", "part.setting not present": "Property '{{property}}' not set", "part.jme.answer missing": "Correct answer is missing", "part.jme.answer too long": "Your answer is too long.", "part.jme.answer too short": "Your answer is too short.", "part.jme.answer invalid": "Your answer is not a valid mathematical expression.<br/>{{message}}.", "part.jme.marking.correct": "Your answer is numerically correct.", "part.jme.must-have bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.must-have one": "Your answer must contain: {{strings}}", "part.jme.must-have several": "Your answer must contain all of: {{strings}}", "part.jme.not-allowed bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.not-allowed one": "Your answer must not contain: {{strings}}", "part.jme.not-allowed several": "Your answer must not contain any of: {{strings}}", "part.jme.unexpected variable name": "Your answer was interpreted to use the unexpected variable name <code>{{name}}</code>.", "part.jme.unexpected variable name suggestion": "Your answer was interpreted to use the unexpected variable name <code>{{name}}</code>. Did you mean <code>{{suggestion}}</code>?", "part.patternmatch.display answer missing": "Display answer is missing", "part.patternmatch.correct except case": "Your answer is correct, except for the case.", "part.numberentry.correct except decimal": "Your answer is within the allowed range, but decimal numbers are not allowed.", "part.numberentry.correct except fraction": "Your answer is within the allowed range, but fractions are not allowed.", "part.numberentry.answer invalid": "You did not enter a valid number.", "part.numberentry.answer not integer": "Your answer is invalid. You must enter a whole number, not a decimal.", "part.numberentry.answer not integer or decimal": "Your answer is invalid. You must enter a whole number or a decimal.", "part.numberentry.zero sig fig": "This part is set up to round the student's answer to zero significant figures, which has no meaning.", "part.numberentry.precision type.dp": "decimal place", "part.numberentry.precision type.dp_plural": "decimal places", "part.numberentry.precision type.sigfig": "significant figure", "part.numberentry.precision type.sigfig_plural": "significant figures", "part.numberentry.give your answer to precision": "Round your answer to {{count,niceNumber}} {{precisionType}}.", "part.numberentry.give your answer to precision_0": "Round your answer to the nearest integer.", "part.mcq.options def not a list": "The expression defining the {{properties}} is not a list.", "part.mcq.marking matrix string empty": "The custom marking matrix expression is empty.", "part.mcq.choices missing": "Definition of choices is missing", "part.mcq.matrix cell empty": "Part {{part}} marking matrix cell ({{row}},{{column}}) is empty", "part.mcq.matrix jme error": "Part {{part}} marking matrix cell ({{row}},{{column}}) gives a JME error, {{error}}", "part.mcq.matrix not a number": "Part {{part}} marking matrix cell ({{row}},{{column}}) does not evaluate to a number", "part.mcq.wrong number of choices": "You selected the wrong number of choices.", "part.mcq.no choices selected": "No choices selected.", "part.mcq.matrix not a list": "Marking matrix, defined by JME expression, is not a list but it should be.", "part.mcq.matrix wrong type": "Element of invalid type '{{type}}' used in marking matrix.", "part.mcq.matrix mix of numbers and lists": "Mix of numbers and lists used in marking matrix.", "part.mcq.matrix wrong size": "Marking matrix is the wrong size.", "part.mcq.correct choice": "You chose the correct answer.", "part.matrix.answer invalid": "Your answer is not valid.", "part.matrix.invalid cell": "One or more of the cells in your answer is empty or invalid.", "part.matrix.some incorrect": "One or more of the cells in your answer is incorrect, but you have been awarded marks for the rest.", "part.matrix.empty": "You have not entered an answer.", "part.matrix.empty cell": "One or more of the cells in your answer is empty.", "part.matrix.size mismatch": "The question author hasn't allowed the student to decide the dimensions of their answer, but the correct answer is {{correct_dimensions}} while the answer input is {{input_dimensions}}", "part.gapfill.feedback header": "<strong>Answer {{index}}</strong>", "part.extension.not implemented": "Part hasn't implemented the <code>{{name}}</code> method.", "question.loaded name mismatch": "Can't resume this attempt - the package has changed since the last session.", "question.error": "Question {{number}}: {{message}}", "question.preamble.error": "Error in preamble: {{message}}", "question.preamble.syntax error": "Syntax error in preamble", "question.unsupported part type": "Unsupported part type", "question.header": "Question {{number}}", "question.submit part": "Check answer", "question.show steps": "Show steps", "question.show steps penalty": "You will lose <strong>{{count,niceNumber}}</strong> $t(mark).", "question.show steps no penalty": "Your score will not be affected.", "question.show steps already penalised": "You have already shown steps. You can show them again with no further penalty.", "question.hide steps": "Hide steps", "question.hide steps no penalty": "Your score will not be affected.", "question.advice": "Advice", "question.no such part": "Can't find part {{path}}", "question.can not submit": "Can not submit answer - check for errors.", "question.answer submitted": "Answer submitted", "question.unsubmitted changes": "You have made a change to your answer but not submitted it. Please check your answer and then press the <strong>Submit answer</strong> button.", "question.unsubmitted changes_plural": "You have made changes to your answers but not submitted them. Please check your answers to each part and then press the <strong>Submit all parts</strong> button.", "question.score feedback.show": "Show feedback", "question.score feedback.hide": "Hide feedback", "question.score feedback.score total actual": "Score: {{score,niceNumber}}/{{marks,niceNumber}}", "question.score feedback.score total": "{{marksString}}. Answered.", "question.score feedback.score actual": "Score: {{scoreString}}", "question.score feedback.answered": "Answered.", "question.score feedback.unanswered": "Unanswered.", "question.score feedback.correct": "Your answer is correct", "question.score feedback.partial": "Your answer is partially correct", "question.score feedback.wrong": "Your answer is incorrect", "question.selector.unsubmitted changes": "Answer changed.", "timing.no accumulator": "no timing accumulator {{name}}", "timing.time remaining": "Time remaining:", "xml.could not load": "Couldn't load an XML document: {{message}}", "xml.property not number": "Property {{name}} should be a number, but isn't ({{value}}), in node {{element}}", "xml.property not boolean": "Property {{name}} should be a boolean, but isn't ({{value}}), in node {{element}}", "xml.error in variable definition": "Error in definition of variable <code>{{name}}</code>", "scorm.error initialising": "Error initialising SCORM protocol: {{message}}", "scorm.failed save": "<p>The request to save data to the server failed. Press <b>OK</b> to try again.</p>\n<p>If you get this message repeatedly, check your internet connection or use a different computer. Your previously submitted answers have been successfully saved and will be restored if you resume this session on a different computer.</p>\n<p>If this message appears persistently and you can't save <em>any</em> answers, please contact your lecturer or teacher.</p>", "scorm.no exam suspend data": "Failed to resume: no exam suspend data.", "scorm.error loading suspend data": "Error loading suspend data: {{message}}", "scorm.error loading question": "Error loading question {{number}}: {{message}}", "scorm.no question suspend data": "No question suspend data", "scorm.error loading part": "Error loading part {{part}}: {{message}}", "scorm.no part suspend data": "No part suspend data", "util.product.non list": "Passed a non-list to <code>Numbas.util.product</code>", "util.equality not defined for type": "Equality not defined for type {{type}}", "mark": "mark", "mark_plural": "marks", "was": "was", "was_plural": "were", "part": "part", "gap": "gap", "step": "step"}}, "af-za": {"translation": {"page.loading": "Word gelaai...", "page.saving": "<p>Stoor tans.</p>\n<p>Dit kan 'n paar sekondes neem.</p>", "mathjax.math processing error": "\"{{-message}}\" wanneer <code>{{expression}}</code> na tex verwerk word", "die.numbas failed": "Numbas het gefaal", "die.sorry": "Jammer, Numbas het 'n fout te\u00ebgekom wat beteken dit kan nie voortgaan nie. Hieronder is 'n beskrywing van die fout.", "die.error": "Fout", "modal.ok": "OK", "modal.cancel": "Kanselleer", "exam.exam name": "Eksamen Naam:", "exam.random seed": "Sessie ID:", "exam.student name": "Student se naam:", "exam.number of questions": "Aantal vrae:", "exam.marks available": "Punte beskikbaar:", "exam.pass percentage": "Slaagpersentasie:", "exam.time allowed": "Tyd toegelaat:", "exam.passed": "Geslaag", "exam.failed": "Gedruip", "exam.review header": "Hersien: ", "frontpage.start": "Begin", "suspend.paused header": "Onderbreek", "suspend.exam suspended": "Die eksamen is opgeskort. Druk <em>Hervat</em> om voort te gaan.", "suspend.you can resume": "Jy sal hierdie sessie kan hervat wanneer jy hierdie aktiwiteit die volgende keer begin.", "suspend.resume": "Hervat", "result.exit": "Verlaat eksamen", "result.print": "Druk hierdie resultate-opsomming", "result.exam summary": "Eksamenopsomming", "result.performance summary": "Prestasie-opsomming", "result.exam start": "Eksamen begin:", "result.exam stop": "Eksamen eindig:", "result.time spent": "Tyd spandeer:", "result.questions attempted": "Vrae aangepak:", "result.score": "Telling:", "result.result": "Resultaat:", "result.question number": "Vraagnommer", "result.question score": "Telling", "result.question review title": "Hersien hierdie vraag", "result.click a question to review": "Klik op 'n vraagnommer om te sien hoe jou antwoorde gemerk is en, waar beskikbaar, volledige oplossings.", "end.exam has finished": "Die eksamen is klaar. Jy kan hierdie venster nou toemaak.", "control.confirm leave": "Jy het nie die eksamen voltooi nie.", "control.not all questions answered": "Jy het nie elke vraag in hierdie eksamen voltooi nie.", "control.not all questions submitted": "Jy het veranderinge aan een of meer antwoorde aangebring, maar dit nie ingedien nie. Kontroleer asseblief dat elke vraag ingedien is.", "control.confirm end": "Is jy seker jy wil die eksamen be\u00ebindig? Nadat jy die eksamen voltooi het, sal jy nie enige van jou antwoorde kan verander nie.", "control.confirm regen": "Wil jy waardes in hierdie vraag weer skommel? As jy OK klik, sal al jou antwoorde en punte vir die huidige vraag verlore gaan.", "control.confirm reveal": "Wil jy die antwoord op hierdie vraag ontbloot? Enige punte wat jy tot dusver ontvang het, sal vasgemaak word en jy sal nie later hierdie vraag kan beantwoord nie.", "control.proceed anyway": "Gaan in elk geval voort?", "control.regen": "Probeer 'n soortgelyke vraag", "control.submit answer": "Dien antwoord in", "control.submit all parts": "Dien al die afdelings in", "control.submit again": "Dien weer in", "control.submit": "Dien in", "control.previous": "Vorige", "control.next": "Volgende", "control.advice": "Raad", "control.reveal": "Wys antwoorde", "control.total": "Totaal", "control.pause": "Skort op", "control.end exam": "Eindig Eksamen", "control.back to results": "Gaan terug na resultate", "display.part.jme.error making maths": "Kon nie wiskunde vertoon nie", "exam.xml.bad root": "Wortelelement van eksamen-XML moet 'eksamen' wees", "exam.changeQuestion.no questions": "Hierdie eksamen bevat geen vrae nie! Gaan die .exam-l\u00eaer na vir foute.", "feedback.you were awarded": "<strong>{{count,niceNumber}}</strong> $t(mark) is aan jou toegeken.", "feedback.taken away": "<strong>{{count,niceNumber}}</strong> $t(mark) $t(was) weggeneem.", "jme.shunt.not enough arguments": "Nie genoeg argumente vir bewerking <code>{{op}}</code> nie", "jme.shunt.no left bracket in function": "Geen bypassende linkerhakie in funksietoepassing of tupel nie", "jme.shunt.no left square bracket": "Geen bypassende linkerhakie nie", "jme.shunt.no left bracket": "Geen bypassende linkerhakie nie", "jme.shunt.no right bracket": "Geen bypassende regterhakie nie", "jme.shunt.no right square bracket": "Geen bypassende regter, vierkantige hakie om die lys te sluit nie", "jme.shunt.missing operator": "Uitdrukking kan nie ge\u00ebvalueer word nie -- 'n operator ontbreek.", "jme.typecheck.function maybe implicit multiplication": "Funksie <code>{{name}}</code> is nie gedefinieer nie. Bedoel jy <code>{{first}}*{{possibleOp}}(...)</code>?", "jme.typecheck.function not defined": "Funksie <code>{{op}}</code> is nie gedefinieer nie. Is <code>{{op}}</code> 'n veranderlike, en het jy bedoel <code>{{suggestion}}*(...)</code> bedoel?", "jme.typecheck.op not defined": "Operasie '{{op}}' is nie gedefinieer nie.", "jme.typecheck.no right type definition": "Geen definisie van '{{op}}' met korrekte tipe gevind nie.", "jme.typecheck.no right type unbound name": "Veranderlike <code>{{name}}</code> is nie gedefinieer nie.", "jme.typecheck.map not on enumerable": "<code>afbeeldings</code>-bewerking moet oor 'n lys of 'n reeks werk, nie {{type}} nie", "jme.evaluate.undefined variable": "Veranderlike {{name}} is ongedefinieerd", "jme.thtml.not html": "Het 'n nie-HTML-waarde na die THTML-konstruktor deurgegee.", "jme.func.switch.no default case": "Geen versteksaak vir Switch-stelling nie", "jme.func.listval.invalid index": "Ongeldige lysindeks {{index}} op lys van grootte {{size}}", "jme.func.listval.not a list": "Voorwerp is nie onderskryfbaar nie", "jme.func.matrix.invalid row type": "Kan nie 'n matriks konstrueer uit rye van tipe {{type}} nie", "jme.func.except.continuous range": "Kan nie die 'except'-operator op kontinue reekse gebruik nie.", "jme.matrix.reports bad size": "Matriks rapporteer sy grootte verkeerd - daar moet 'n fout in konstruktor funksie wees", "jme.texsubvars.no right bracket": "Geen ooreenstemmende <code>]</code> in {{op}} argumente.", "jme.texsubvars.missing parameter": "Ontbrekende parameter in {{op}}: {{parameter}}", "jme.texsubvars.no right brace": "Geen ooreenstemmende <code>}</code> in {{op}}", "jme.user javascript.error": "Fout in gebruikergedefinieerde javascript-funksie <code>{{name}}</code>: {{-message}}", "jme.variables.error making function": "Kon nie funksie <code>{{name}}</code> skep nie: {{-message}}", "jme.variables.syntax error in function definition": "Sintaksisfout in funksiedefinisie", "jme.variables.variable not defined": "Veranderlike <code>{{name}}</code> is nie gedefinieer nie.", "jme.variables.empty definition": "Definisie van veranderlike <code>{{name}}</code> is leeg.", "jme.variables.circular reference": "Veranderlike se definisie verwys na self in <code>{{name}}</code>", "jme.variables.error computing dependency": "Fout met die berekening van verwysde veranderlike <code>{{name}}</code>", "jme.variables.error evaluating variable": "Kon nie veranderlike {{name}} evalueer nie: {{-message}}", "jme.variables.question took too many runs to generate variables": "'n Geldige stel vraagveranderlikes is nie betyds gegenereer nie.", "jme.display.unknown token type": "Kan nie teken tipe {{type}} in tex omskakel nie", "jme.display.collectRuleset.no sets": "Geen versamelings is aan collectRuleset gegee nie!", "jme.display.collectRuleset.set not defined": "Ruleset {{name}} is nie gedefinieer nie", "jme.display.simplifyTree.no scope given": "Numbas.jme.display.simplifyTree moet 'n afbakening h\u00ea", "math.precround.complex": "Kan nie tot 'n komplekse getal desimale plekke afrond nie", "math.siground.complex": "Kan nie tot 'n komplekse getal beduidende syfers afrond nie", "math.combinations.complex": "Kan nie kombinasies van komplekse getalle bereken nie", "math.permutations.complex": "Kan nie permutasies van komplekse getalle bereken nie", "math.gcf.complex": "Kan nie GGD van komplekse getalle bereken nie", "math.lcm.complex": "Kan nie KGV van komplekse getalle bereken nie", "math.lt.order complex numbers": "Kan nie komplekse getalle orden nie", "math.choose.empty selection": "Le\u00eb seleksie gegee aan lukrake funksie", "matrixmath.abs.non-square": "Kan nie die determinant van 'n nie-vierkantige matriks bereken nie.", "matrixmath.abs.too big": "Jammer, kan nog nie die determinant van 'n matriks groter as 3x3 bereken nie.", "matrixmath.mul.different sizes": "Kan nie matrikse van verskillende groottes vermenigvuldig nie.", "vectormath.cross.not 3d": "Kan slegs die kruisproduk van 3-dimensionele vektore bereken.", "vectormath.dot.matrix too big": "Kan nie skalaarproduk van 'n matriks bereken wat nie $1 \\times N$ of $N \\times 1$ is nie.", "vectormath.cross.matrix too big": "Kan nie kruisproduk van 'n matriks bereken wat nie $1 \\times N$ of $N \\times 1$ is nie.", "part.with steps answer prompt": "Antwoord: ", "part.script.error": "Fout in deel {{path}} gepasmaakte skrif {{script}}: {{-message}}", "part.marking.steps no matter": "Omdat jy volpunte vir die afdeling gekry het, word jou antwoorde in die stappe afdeling nie getel nie.", "part.marking.used variable replacements": "Hierdie deel is gemerk deur jou antwoorde op vorige afdelings te gebruik.", "part.marking.variable replacement part not answered": "Jy moet eers {{part}} antwoord.", "part.marking.resubmit because of variable replacement": "Hierdie deel se nasien hang van jou antwoorde in ander afdelings af, wat jy verander het. Dien hierdie deel weer in om jou punt op te dateer.", "part.marking.not submitted": "Geen antwoord ingedien nie.", "part.marking.did not answer": "Jy het nie hierdie vraag beantwoord nie.", "part.marking.nothing entered": "Jy het nie 'n antwoord ingevoer nie.", "part.marking.incorrect": "Jou antwoord is verkeerd.", "part.marking.correct": "Jou antwoord is korrek.", "part.marking.uncaught error": "Nasienfout: {{-message}}", "part.correct answer": "Antwoord wat verwag is:", "part.missing type attribute": "{{part}}: Ontbrekende afdelingtipe-kenmerk", "part.unknown type": "{{part}}: Onbekende afdelingtipe {{type}}", "part.setting not present": "Eienskap '{{property}}' nie gestel nie", "part.jme.answer missing": "Korrekte antwoord ontbreek", "part.jme.answer too long": "Jou antwoord is te lank.", "part.jme.answer too short": "Jou antwoord is te kort.", "part.jme.answer invalid": "Jou antwoord is nie 'n geldige wiskundige uitdrukking nie.<br/>{{-message}}.", "part.jme.marking.correct": "Jou antwoord is numeries korrek.", "part.jme.must-have bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.must-have one": "Jou antwoord moet die volgende bevat: {{strings}}", "part.jme.must-have several": "Jou antwoord moet al die volgende bevat: {{strings}}", "part.jme.not-allowed bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.not-allowed one": "Jou antwoord moet die volgende bevat nie: {{strings}}", "part.jme.not-allowed several": "Jou antwoord moet nie enige van die volgende bevat nie: {{strings}}", "part.jme.unexpected variable name": "Jou antwoord is ge\u00efnterpreteer om die onverwagte veranderlike naam <code>{{name}}</code> te gebruik.", "part.jme.unexpected variable name suggestion": "Jou antwoord is ge\u00efnterpreteer om die onverwagte veranderlike naam <code>{{name}}</code> te gebruik. Bedoel jy <code>{{suggestion}}</code>?", "part.patternmatch.display answer missing": "Vertoonantwoord ontbreek", "part.patternmatch.correct except case": "Jou antwoord is korrek, behalwe vir die kas.", "part.numberentry.correct except decimal": "Jou antwoord is binne die toegelate reeks, maar desimale getalle word nie toegelaat nie.", "part.numberentry.correct except fraction": "Jou antwoord is binne die toegelate reeks, maar breuke word nie toegelaat nie.", "part.numberentry.answer invalid": "Jy het nie 'n geldige getal ingevoer nie.", "part.numberentry.answer not integer": "Jou antwoord is ongeldig. Jy moet 'n heelgetal invoer, nie 'n desimale getal nie.", "part.numberentry.answer not integer or decimal": "Jou antwoord is ongeldig. Jy moet 'n heelgetal of 'n desimale getal invoer.", "part.numberentry.zero sig fig": "Hierdie deel is opgestel om die student se antwoord af te rond tot nul beduidende syfers, wat geen betekenis het nie.", "part.mcq.options def not a list": "Die uitdrukking wat die {{properties}} definieer, is nie 'n lys nie.", "part.mcq.marking matrix string empty": "Die gepasmaakte merkmatriks-uitdrukking is leeg.", "part.mcq.choices missing": "Definisie van keuses ontbreek", "part.mcq.matrix not a number": "Afdeling {{part}} se merkmatrikssel ({{row}},{{column}}) evalueer nie na 'n getal nie", "part.mcq.wrong number of choices": "Jy het die verkeerde aantal keuses gekies.", "part.mcq.no choices selected": "Geen keuses gemaak nie.", "part.mcq.matrix not a list": "Merkmatriks, gedefinieer deur JME-uitdrukking, is nie 'n lys nie, maar dit behoort te wees.", "part.mcq.matrix wrong type": "Element van ongeldige tipe '{{type}}' word in merkmatriks gebruik.", "part.mcq.matrix mix of numbers and lists": "Mengsel van getalle en lyste word in die merkmatriks gebruik.", "part.mcq.matrix wrong size": "Merkmatriks is die verkeerde grootte.", "part.mcq.correct choice": "Jy het 'n korrekte antwoord gekies.", "part.matrix.answer invalid": "Jou antwoord is nie geldig nie.", "part.matrix.invalid cell": "Een of meer van die selle in jou antwoord is leeg of ongeldig.", "part.matrix.some incorrect": "Een of meer van die selle in jou antwoord is verkeerd, maar jy het punte vir die res gekry.", "part.matrix.empty": "Jy het nie 'n antwoord ingevoer nie.", "part.matrix.empty cell": "Een of meer van die selle in jou antwoord is leeg.", "part.matrix.size mismatch": "Die vraagskrywer het nie die student toegelaat om die afmetings van hul antwoord te bepaal nie, maar die korrekte antwoord is {{correct_dimensions}} terwyl die antwoordinvoer {{input_dimensions}} is", "part.gapfill.feedback header": "<strong>{{name}}</strong>", "part.extension.not implemented": "Afdeling het nie die <code>{{name}}</code>-metode ge\u00efmplementeer nie.", "question.loaded name mismatch": "Kan nie hierdie poging hervat nie - die pakket het sedert die laaste sessie verander.", "question.error": "Antwoord {{number}}: {{-message}}", "question.preamble.error": "Fout in aanhef: {{-message}}", "question.preamble.syntax error": "Sintaksisfout in aanhef", "question.unsupported part type": "Ongesteunde afdelingtipe", "question.header": "Vraag {{number}}", "question.submit part": "Dien afdeling in", "question.show steps": "Wys stappe", "question.show steps penalty": "Jy sal <strong>{{count,niceNumber}}</strong> $t(mark) verloor.", "question.show steps no penalty": "Jou telling sal nie be\u00efnvloed word nie.", "question.show steps already penalised": "Jy het reeds stappe gewys. Jy kan hulle weer wys sonder enige verdere strafpunte.", "question.hide steps": "Verberg stappe", "question.hide steps no penalty": "Jou telling sal nie be\u00efnvloed word nie.", "question.advice": "Raad", "question.no such part": "Kan nie deel {{path}} vind nie", "question.can not submit": "Kan nie antwoord indien nie - kyk vir foute.", "question.answer submitted": "Antwoorde ingedien", "question.score feedback.show": "Wys terugvoer", "question.score feedback.hide": "Verberg terugvoer", "question.score feedback.answered": "Beantwoord", "question.score feedback.unanswered": "Ongeantwoord", "question.score feedback.correct": "Jou antwoord is reg", "question.score feedback.partial": "Jou antwoord is gedeeltelik reg", "question.score feedback.wrong": "Jou antwoord is verkeerd", "question.selector.unsubmitted changes": "Oningediende veranderinge.", "timing.no accumulator": "geen tydsberekening akkumulator {{name}}", "timing.time remaining": "Oorblywende tyd:", "xml.could not load": "Kon nie 'n XML-dokument laai nie: {{-message}}", "xml.property not number": "Eienskap {{name}} moet 'n getal wees, maar is nie ({{value}}), in nodus {{element}}", "xml.property not boolean": "Eienskap {{name}} moet 'n boolean wees, maar is nie ({{value}}), in nodus {{element}}", "scorm.error initialising": "Fout met inisiasie van SCORM-protokol: {{-message}}", "scorm.failed save": "<p>Die versoek om data op die bediener te stoor het misluk. Druk <b>OK</b> om weer te probeer.</p>\n<p>As jy hierdie boodskap herhaaldelik kry, toets jou internetverbinding of gebruik 'n ander rekenaar. Jou voorheen ingediende antwoorde is suksesvol gestoor en sal herstel word as jy hierdie sessie op 'n ander rekenaar hervat.</p>\n<p>As hierdie boodskap aanhoudend verskyn en jy nie <em>enige</em> antwoorde kan stoor nie, kontak asseblief jou dosent of onderwyser.</p>", "scorm.no exam suspend data": "Kon nie hervat nie: geen eksamenopskortingsdata nie.", "scorm.error loading suspend data": "Kon nie opskortdata laai nie: {{-message}}", "scorm.error loading question": "Kon nie vraag {{number}} laai nie: {{-message}}", "scorm.no question suspend data": "Geen vraagopskort data nie", "scorm.error loading part": "Kon nie deel {{part}} laai nie: {{-message}}", "scorm.no part suspend data": "Geen afdeling-opskortdata nie", "util.product.non list": "Het 'n nie-lys na <code>Numbas.util.product</code> deurgegee", "mark": "punt", "was": "was", "part": "afdeling", "gap": "gaping", "step": "stap", "jme.substituteTree.undefined variable": "Ongedefinieerde veranderlike: <code>{{name}}</code>", "jme.user javascript.returned undefined": "Gebruikergedefinieerde javascript-funksie <code>{{name}}</code> het <code>undefined</code> teruggestuur.", "part.marking.steps change": "<strong>{{count,niceNumber}}</strong> $t(mark) is aan jou toegeken vir jou antwoorde op die stappe.", "part.marking.total score": "Jy het <strong>{{count,niceNumber}}</strong> $t(mark) vir hierdie afdeling gekry.", "part.numberentry.precision type.dp": "desimale plek", "part.numberentry.precision type.dp_plural": "desimale plekke", "part.numberentry.precision type.sigfig": "beduidende syfer", "part.numberentry.precision type.sigfig_plural": "beduidende syfers", "part.numberentry.give your answer to precision": "Rond jou antwoord af tot {{count,niceNumber}} {{precisionType}}.", "question.unsubmitted changes": "Jy het jou antwoord verander, maar dit nie ingedien nie. Kontroleer asseblief jou antwoord en druk dan die <strong>Dien antwoord in</strong>-knoppie.", "question.unsubmitted changes_plural": "Jy het jou antwoorde verander, maar dit nie ingedien nie. Kontroleer asseblief jou antwoorde vir elke afdeling en druk dan die <strong>Dien alle dele in</strong>-knoppie.", "util.equality not defined for type": "Gelykgeid nie gedefinieer vir tipe {{type}} nie", "mark_plural": "punte", "was_plural": "was", "die.script not loaded": "Numbas kon nie begin nie omdat die l\u00eaer <code>{{file}}</code> nie gelaai is nie. Maak seker dat dit by <code>scripts.js</code> ingesluit is.", "math.combinations.n less than zero": "Kan nie kombinasies bereken nie: n is minder as nul", "math.combinations.k less than zero": "Kan nie kombinasies bereken nie: k is minder as nul", "math.combinations.n less than k": "Kan nie kombinasies bereken nie: n is minder as k", "math.permutations.n less than zero": "Kan nie permutasies bereken nie: n is minder as nul", "math.permutations.k less than zero": "Kan nie permutasies bereken nie: k is minder as nul", "math.permutations.n less than k": "Kan nie permutasies bereken nie: n is minder as k", "part.numberentry.give your answer to precision_0": "Rond jou antwoord af tot die naaste heelgetal.", "mathjax.error": "MathJax verwerkingsfout: {{-message}}", "mathjax.error with context": "MathJax-verwerkingsfout in {{-context}}: {{-message}}", "exam.introduction": "Eksamen inleiding", "exam.feedback": "Eksamen terugvoerboodskap", "jme.tokenise.keypair key not a string": "Woordeboeksleutel moet 'n string wees, nie {{type}} nie.", "jme.shunt.list mixed argument types": "Kan nie {{mode}} ontleed nie: mengsel van woordeboek- en lyselemente", "jme.func.listval.key not in dict": "Woordeboek bevat nie die sleutel <code>{{key}}</code> nie", "part.prompt": "vraagteks", "part.feedback": "terugvoer", "part.numberentry.answer not reduced": "Jou antwoord is nie tot die kleinste terme vereenvoudig nie.", "part.numberentry.give your answer as a reduced fraction": "Vereenvoudig jou antowwrd tot die kleinste terme.", "part.numberentry.negative decimal places": "Hierdie deel is opgestel om die student se antwoord af te rond tot 'n negatiewe aantal desimale plekke, wat geen betekenis het nie.", "part.mcq.choices": "keuses", "part.mcq.answers": "antwoorde", "part.mcq.matrix cell empty": "Deel {{part}} merkmatrikssel ({{row}},{{column}}) is leeg", "part.mcq.matrix jme error": "Deel {{part}} merkmatrikssel ({{row}},{{column}}) gee 'n JME-fout: {{-error}}", "question.statement": "Stelling", "ruleset.circular reference": "Selfverwysing in definisie van re\u00eblstel <code>{{name}}</code>", "ruleset.set not defined": "Re\u00eblstel {{name}} is nie gedefinieer nie", "jme.evaluate.no scope given": "Numbas.jme.evaluate moet 'n Afbakening gegee word", "answer.number.not a number": "Jou antwoord is nie 'n geldige nommer nie.", "answer.number.fractions not allowed": "Jy mag nie 'n breuk invoer nie.", "answer.jme.invalid expression": "{{-message}}", "answer.matrix.fractions not allowed": "Jy mag nie breuke invoer nie.", "answer.matrix.some cell not a number": "Een of meer van die selle in jou antwoord is nie 'n geldige getal nie.", "exam.enter password": "Wagwoord:", "exam.password.correct": "Hierdie wagwoord is korrek. Jy mag die eksamen begin.", "exam.password.incorrect": "Hierdie wagwoord is verkeerd.", "frontpage.scorm.lms not connected": "Hierdie eksamen verloop op sy eie. Jou antwoorde en punte sal nie gestoor word nie!", "result.question review": "Hersien", "control.confirm regen no marks": "Wil jy hierdie vraag se getalle weer kies?", "control.confirm reveal no marks": "Wil jy die antwoord op hierdie vraag blootstel?", "jme.tokenise.invalid near": "Ongeldige uitdrukking: <code>{{expression}}</code> by posisie {{position}} naby <code>{{nearby}}</code>", "jme.tokenise.number.object not complex": "Ongeldige voorwerp deurgegee in getalkonstruktor.", "jme.subvars.null substitution": "Le\u00eb veranderlike vervanging: <code>$t(left brace){{str}}$t(right brace)", "jme.type.type already registered": "Die datatipe {{type}} is reeds geregistreer en kan dus nie weer geregistreer word nie.", "jme.type.no cast method": "Kan nie outomaties van {{from}} na {{to}} omskakel nie.", "jme.display.simplifyTree.empty expression": "Uitdrukking is leeg", "jme.display.simplifyTree.stuck in a loop": "Vereenvoudiger is in 'n lus vasgevang: <code>{{expr}}</code>", "math.niceNumber.undefined": "Het 'n getal verwag, maar het <code>undefined</code> gekry", "math.rangeToList.zero step size": "Kan nie 'n reeks met stapgrootte nul na 'n lys omskakel nie.", "part.error": "{{path}}: {{-message}}", "part.marking.revealed steps": "Jy het die stappe ontbloot.", "part.marking.maximum scaled down": "Die maksimum punt wat jy vir hierdie afdeling kan kry is <strong>{{count,niceNumber}}</strong> $t(punt). Jou punte sal dienooreenkomstig afgeskaal word.", "part.marking.minimum score applied": "Die minimum telling vir hierdie deel is <strong>{{score,niceNumber}}</strong>.", "part.marking.maximum score applied": "Die aksimum telling vir hierdie deel is <strong>{{score,niceNumber}}</strong>.", "part.marking.error in marking script": "Daar was 'n fout in hierdie deel se merkalgoritme. Rapporteer dit asseblief. {{-message}}", "part.marking.no result after replacement": "Hierdie deel kon nie nagesien word deur jou antwoorde by vorige afdelings te gebruik nie.", "part.marking.missing required note": "Die merkalgoritme definieer nie die nota <code>{{note}}</code> nie", "marking.apply.not a list": "Die eerste argument om <code>toe te pas</code> moet 'n lys wees, en is nie", "marking.apply marking script.script not found": "Merkskrif <code>{{name}}</code> nie gevind nie", "marking.note.compilation error": "Kon nie nota <code>{{name}}</code> saamstel nie: {{-message}}", "marking.note.error evaluating note": "Kon nie nota <code>{{name}}</code> evalueer nie - {{-message}}", "marking.note.invalid definition": "Ongeldige notadefinisie: <code>{{source}}</code>. {{-hint}}", "marking.note.invalid definition.missing colon": "Jy kort dalk 'n dubbelpunt na die naam en beskrywing", "marking.note.invalid definition.description missing closing bracket": "Jy kort dalk 'n sluitingshakie", "marking.note.empty expression": "Die nota <code>{{name}}</code> is leeg.", "marking.script.error parsing notes": "Kon nie merkskrif ontleed nie: {{- message}}", "part.feedback out of date": "Hierdie terugvoer is gebaseer op jou laaste ingediende antwoord. Dien jou veranderde antwoord in om opgedateerde terugvoer te kry.", "part.jme.invalid value generator expression": "Ongeldige waardegenerator-uitdrukking vir veranderlike <code>{{name}}</code>: {{-message}}", "part.mcq.incorrect choice": "Jy het 'n verkeerde antwoord gekies.", "part.matrix.not all cells same precision": "Jy het nie elke sel in jou antwoord met dieselfde beduidende syfers gegee nie.", "part.gapfill.error marking gap": "Kon nie {{name}} merk nie: {{-message}}", "part.custom.empty setting": "Geen waarde gegee nie.", "part.custom.unrecognised input type": "Onherkende instellingtipe <code>{{input_type}}</code>", "part.custom.error evaluating input option": "Fout met die evaluering van invoeropsie <code>{{option}}</code>: {{-error}}", "part.custom.input option missing": "Definisie van invoeropsie <code>{{option}}</code> ontbreek.", "part.custom.error evaluating setting": "Kon nie instelling <code>{{setting}}</code> evalueer: {{-error}}", "question.error creating question": "Kon nie vraag {{number}} skep nie: {{-message}}", "question.score feedback.not marked": "Nie gemerk nie", "question.score feedback.partially answered": "Deels beantwoord", "question.score feedback.score total": "{{marksString}}", "question.score feedback.score actual": "Punt: {{scoreString}}", "question.score feedback.score total actual": "Punt: {{score,niceNumber}}/{{marks,niceNumber}}", "variable.error in variable definition": "Fout in definisie van veranderlike <code>{{name}}</code>", "left brace": "{", "right brace": "}", "extension.not found": "Kon nie die uitbreiding <code>{{name}}</code> laai nie.", "control.toggle navigation menu": "Skakel die navigasiekieslys om", "part.input title": "Antwoord vir afdeling {{name}}", "part.correct answer title": "Verwagte antwoord vir afdeling {{name}}", "part.jme.must-match.failed": "Jou antwoord is nie in die regte formaat nie.", "control.submit part.confirm remove next parts": "<p>Een of meer opeenvolgende afdelings hang af van jou antwoord in hierdie deel. Deur hierdie deel weer in te dien, sal daardie afdelings ongeldig word en hulle word uit die vraag verwyder. Dit kan nie terug verander word nie.</p>\n<p>Wil jy hierdie deel weer indien?</p>", "control.back to menu": "Gaan terug na die kieslys", "display.error making html": "Kon nie HTML skep in {{contextDescription}}: {{-message}}", "jme.subvars.error compiling": "{{-message}} in <code>{{expression}}</code>", "jme.variables.empty name": "'n Vraagveranderlike het nie 'n naam nie.", "jme.calculus.unknown derivative": "Weet nie hoe om <code>{{tree}}</code> te differensieer nie", "math.order complex numbers": "Kan nie komplekse getalle orden nie", "menu.choose a question": "Kies 'n vraag.", "part.choose next part.answered": "Wat vil jy volgende doen?", "part.choose next part.unanswered": "Of jy kan:", "part.choose next part.will be locked": "(Hierdie afdeling sal gesluit wees)", "part.reached dead end": "Daar is niks meer om verder te doen nie.", "part.next part.penalty amount": "(verloor {{count}} $t(mark))", "part.marking.counts towards objective": "Hierdie afdeling tel vir die doelwit <strong>\u201c{{objective}}\u201d</strong>.", "part.numberentry.answer not integer or decimal or fraction": "Jou antwoord is ongeldig. Jy moet 'n heelgetal, 'n desimale getal of 'n breuk invoer.", "question": "Vraag", "question.progress": "Vordering met vraag:", "question.score feedback.unattempted": "Nie gepoog nie", "question.score feedback.attempted": "Gepoog", "question.score feedback.score actual.plain": "{{scoreString}}", "question.score feedback.score total actual.plain": "{{score,niceNumber}}/{{marks,niceNumber}}", "question.objectives": "Doelwitte", "question.penalties": "Strafpunte", "question.back to previous part": "Gaan terug na die vorige deel", "end.print": "Druk jou eksamenafskrif", "math.shuffle_together.lists not all the same length": "Nie alle lyste is ewe lank nie.", "jme.parse signature.invalid signature string": "Ongeldige funksie handtekeningstring: {{str}}", "part.custom.expected answer has wrong type": "Die verwagte antwoord vir hierdie deel het die verkeerde tipe. Dit moet <code>{{shouldbe}}</code> wees.", "part.custom.input option has wrong type": "Die antwoordinvoerstelling <code>{{option}}</code> het die verkeerde tipe. Dit moet <code>{{shouldbe}}</code> wees.", "matrix input.size control legend": "Grootte", "matrix input.rows": "Rye", "matrix input.columns": "Kolomme", "part.jme.error checking numerically": "Kon nie jou antwoord numeries nagaan nie: {{-message}}", "part.gapfill.cyclic adaptive marking": "Daar is 'n siklus in die aanpasbare merkswyse vir hierdie deel: <strong>{{name1}}</strong> maak staat op <strong>{{name2}}</strong>, wat uiteindelik staatmaak op <strong>{{name1} }</strong>.", "modal.style.background colour": "Agtergrond kleur", "modal.style.text colour": "Teks kleur", "modal.style.text size": "Teks grootte", "modal.style.explanation": "Gebruik hierdie kontroles om die voorkoms van die eksamen te verander.", "modal.style.reset to defaults": "Stel terug na verstekwaardes", "modal.style.text size preview": "Die meeste teks sal so groot wees.", "control.style options": "Vertoon opsies", "part.marking.partially correct": "Jou antwoord is gedeeltelik korrek.", "part.marking.error in adaptive marking": "Daar was 'n fout in die aanpasbare merkswyse vir hierdie afdeling. Rapporteer dit asseblief. {{-message}}", "page.skip to content": "Slaan oor na inhoud", "result.learning objective": "Leerdoelwit", "jme.interpreted as": "ge\u00efnterpreteer as", "jme.script.note.compilation error": "Kon nie nota <code>{{name}}</code> saamstel nie: {{-message}}", "jme.script.note.error evaluating note": "Kon nie nota <code>{{name}}</code> evalueer nie - {{-message}}", "jme.script.note.invalid definition": "Ongeldige notadefinisie: <code>{{source}}</code>. {{-hint}}", "jme.script.note.invalid definition.missing colon": "Dalk ontbreek 'n dubbelpunt na die naam en beskrywing", "jme.script.note.invalid definition.description missing closing bracket": "Jy kort dalk 'n sluitingshakie", "jme.script.note.empty expression": "Die nota <code>{{name}}</code> is leeg.", "jme.script.error parsing notes": "Kon nie merkskrif ontleed nie: {{- message}}", "matrix input.cell label": "Ry {{row}}, kolom {{column}}", "control.move to next question": "Gaan na die volgende vraag", "diagnostic.use retry": "Gebruik een herprobeerslag en doen hierdie onderwerp weer.", "diagnostic.move to next topic": "Gaan aan na die volgende onderwerp.", "diagnostic.next step question": "Wat wil jy volgende doen?", "diagnostic.now assessing topic": "{{current_topic}} word nou assesseer", "diagnostic.one retry left": "Jy het nog 1 herprobeerslag oor", "diagnostic.retries left": "Jy het nog {{retries}} herprobeerslae oor", "diagnostic.percentage completed": "Jy het <strong>{{percentage}}%</strong> van die toets voltooi.", "diagnostic.test is over": "Die toets is verby.", "diagnostic.passed all lo": "Jy het alle leerdoelwitte geslaag.", "diagnostic.more work on lo": "Jy moet nog aan die volgende leerdoelwitte werk: {{los}}.", "diagnostic.move to next question in topic": "Beweeg na die volgende vraag in die onderwerp.", "diagnostic.complete": "Voltooi!", "diagnostic.studying topic": "Bestudeer {{topic}}", "display.answer widget.unknown widget type": "Die antwoord legstuktipe <code>{{name}}</code> word nie herken nie.", "jme.shunt.expected argument before comma": "Het verwag om iets tussen die openinghakie en die komma te sien", "part.waiting for pre submit": "Jou antwoord word gemerk. Wag asseblief.", "diagnostic.end test": "Be\u00ebindig die toets.", "page.no stylesheet loaded": "Die bladsy se stylbladl\u00eaer is nie gelaai nie.", "modal.confirm": "Bevestig", "modal.alert": "Pasop", "suspend.resumed header": "Poging hervat", "jme.vector.value not an array of numbers": "Het probeer om 'n vektor te konstrueer deur 'n waarde te gebruik wat nie 'n reeks van getalle is nie.", "jme.matrix.value not the right type": "Het probeer om 'n vektor te konstrueer deur 'n waarde van die verkeerde tipe te gebruik.", "jme.subvars.html inserted twice": "'n HTML-waarde is twee keer ingebed. Oorweeg om 'n funksie te definieer om 'n nuwe waarde te genereer elke keer as dit gebruik word.", "jme.variables.invalid function language": "Die taal <code>{{language}}</code> is nie geldig nie.", "jme.variables.duplicate definition": "Daar is meer as een definisie van die veranderlike <code>{{name}}</code>.", "math.random_integer_partition.invalid k": "Die grootte van die partisie moet tussen 1 en {{n}} wees.", "part.marking.parameter already in scope": "Daar is 'n veranderlike genaamd <code>{{name}}</code>, wat ook die naam van 'n merkparameter is. Hernoem asseblief die veranderlike.", "part.marking.adaptive variable replacement refers to self": "Hierdie deel verwys na self in 'n veranderlike vervanging vir aanpasbare merk.", "part.marking.adaptive variable replacement refers to nothing": "Hierdie deel bevat 'n ongeldige veranderlike vervanging vir aanpasbare merk.", "part.numberentry.display answer wrong type": "Die vertoonantwoord vir hierdie deel is 'n waarde van tipe <code>{{got_type}}</code>, maar moet 'n <code>{{want_type}}</code> wees.", "part.matrix.invalid type in prefilled": "Daar is 'n ongeldige waarde van tipe <code>{{n}}</code> in die reeks voorafgevulde selle.", "diagnostic.make a choice": "Maak 'n keuse", "matrixmath.not square": "Hierdie bewerking werk slegs op 'n vierkantige matriks.", "matrixmath.not invertible": "Hierdie bewerking werk slegs op 'n omkeerbare matriks.", "matrixmath.gauss-jordan elimination.not enough columns": "Daar moet ten minste soveel kolomme as rye wees.", "question.required extension not available": "Hierdie vraag vereis die uitbreiding <code>{{-extension}}</code> maar dit is nie beskikbaar nie.", "util.formatNumberNotation.unrecognised syntax": "Die getalformatering sintaksis <code>{{syntax}}</code> word nie herken nie.", "worksheet.number of exams to generate": "Aantal bladsye om te genereer", "worksheet.starting with id": "Begin met ID", "worksheet.show exam id": "Wys blad-ID?", "worksheet.page break between questions": "Bladsybreuke tussen vrae?", "worksheet.page margins": "Bladsykantlyne (mm)", "worksheet.text size": "Teksgrootte (pt)", "worksheet.generate": "Skep", "worksheet.generating exams": "Skep bladsye", "worksheet.sheet id": "Bladsy ID:", "worksheet.print single": "Druk hierdie bladsy", "worksheet.print several": "Deuk hierdie bladsye", "worksheet.answer sheets": "Antwoordbladsye", "worksheet.question sheets": "Vraebladsye", "worksheet.reconfigure": "Generate different sheets", "worksheet.show sheet": "Preview the sheet with ID:", "accessibility statement": "Accessibility statement and guide to adapting Numbas to your needs.", "exam.enter your name": "Your name:", "exam.attempt download security warning": "This exam is configured to allow you to download your data, but it is not running in a secure browser context. You will not be able to download your exam data. Contact your lecturer or teacher for help.", "result.download exam object": "Download your exam data", "control.return to question": "Keer terug na die vraag", "control.show introduction": "Inleiding", "analysis.header": "Ontleed pogingdata", "analysis.help.upload files": "Laai pogingdata l\u00eaers op wat jou studente vir jou gegee het.", "analysis.help.file input label": "Kies studente pogingdata l\u00eaers, of sleep l\u00eaers na hierdie venster.", "analysis.table.total": "Eksamentotale", "analysis.table.question": "Eksamen- en vraagtotale", "analysis.table.all": "Alle besonderhede", "analysis.student name.anonymous": "Geen naam gegee nie", "analysis.expected": "Verwagte resultate", "analysis.start time": "Begin tyd", "analysis.maximum": "Maksimum punte", "analysis.file": "L\u00eaer", "analysis.download": "Laai af", "analysis.delete": "Vee uit", "analysis.view results": "Bekyk resultate", "analysis.upload files": "Laai l\u00eaers op", "analysis.upload more": "Laai meer l\u00eaers op", "analysis.attempt data": "Pogingdata", "analysis.select format": "Kies data om te wys", "analysis.download this table": "Download this table", "analysis.student": "Student Results", "analysis.question key": "Question Key", "analysis.question name": "Question Name", "analysis.group": "Group", "analysis.question": "Question", "analysis.part": "Part", "analysis.gap": "Gap", "analysis.record type": "Record Type", "analysis.score": "Score", "analysis.marks available": "Marks Available", "analysis.percentage": "Percentage", "analysis.answer": "Answer", "analysis.student name": "Student Name", "analysis.summary.no files": "You have not uploaded any files yet.", "analysis.summary.no decrypted files": "You have not uploaded any valid files yet.", "analysis.summary.one file": "One attempt.", "analysis.summary.several files": "{{num_files,niceNumber}} attempts.", "analysis.not secure context": "This page must be opened in a secure browser context. A secure context is either a page served over HTTPS, or a file loaded from your device.", "jme.shunt.pipe right hand takes no arguments": "The expression on the right-hand side of the pipe operator must be a function application.", "question.explore.no parts defined": "There are no parts defined in this question.", "answer": "answer", "worksheet.answersheet show question content": "Show question content in answer sheets?", "modal.confirm end exam": "Write <code>{{endConfirmation}}</code> in the box to confirm:", "modal.end exam button": "End exam", "lightbox.zoom in on image": "Zoom in on this image", "exam.progress": "Progress", "exam.questions answered": "{{numAnsweredQuestions}} of {{numQuestions}} questions answered.", "result.question marks available": "Marks Available", "result.question answered": "Answered?", "control.confirm end.correct": "You may now end the exam.", "control.confirm end.incorrect": "This is not the expected text.", "control.confirm end.password": "end", "jme.typecheck.for in name wrong type": "The name in a <code>for</code> statement must be a name or list of names, not {{type}}.", "jme.makeFast.no fast definition of function": "The function <code>{{name}}</code> here isn't defined in a way that can be made fast.", "part.show feedback": "Show feedback", "part.hide feedback": "Hide feedback", "part.feedback title": "Feedback for {{name}}.", "part.jme.must-match.warning": "Your answer is not in the expected form: {{-message}}", "part.numberentry.write your answer as a fraction": "Write your answer as a fraction.", "question.nav.label": "Question controls", "question.answer saved": "Answer saved", "question.all answers saved": "All answers saved", "analysis.back to results": "Back to results", "analysis.review": "Review", "analysis.review this": "Review this attempt", "analysis.reviewing attempt": "Reviewing attempt by {{student_name}}.", "part.there is new feedback": "The feedback has changed.", "modal.style.colour scheme": "Colour scheme", "modal.style.automatic colour scheme": "Automatic", "modal.style.light colour scheme": "Light", "modal.style.dark colour scheme": "Dark", "modal.style.custom colour scheme": "Custom", "modal.style.main font": "Main font", "modal.style.font.sans serif": "Sans serif", "modal.style.font.serif": "Serif", "modal.style.font.monospace": "Monospace", "modal.style.spacing": "Spacing", "modal.style.font weight": "Font weight", "modal.style.forced colours warning": "Your browser has overridden the colours used in this page because of a setting such as high contrast mode. Changes to the colour scheme settings here might not have any effect.", "modal.style.text preview": "Most text will look like this.", "modal.style.more options": "More options", "modal.style.main colour": "Main (brand) colour", "modal.style.primary colour": "Primary button colour", "modal.style.link colour": "Link colour", "modal.style.success colour": "Success/correct colour", "modal.style.info colour": "Info colour", "modal.style.warning colour": "Warning colour", "modal.style.danger colour": "Danger/incorrect colour", "modal.style.muted colour": "Muted text colour", "modal.style.highlight colour": "Highlight colour", "exam.error loading exam definition": "There was an error while loading the exam definition: {{text}}", "exam.no exam definition": "No exam definition was given.", "jme.typecheck.wrong arguments for anonymous function": "Wrong number of arguments for this anonymous function.", "worksheet.top": "Top", "worksheet.left": "Left", "worksheet.bottom": "Bottom", "worksheet.right": "Right"}}, "ko-kr": {"translation": {"page.loading": "\ubd88\ub7ec\uc624\ub294 \uc911...", "page.saving": "<p>\uc800\uc7a5 \uc911.</p>\n<p>\uc7a0\uc2dc\ub9cc \uae30\ub2e4\ub824 \uc8fc\uc2ed\uc2dc\uc624.</p>", "mathjax.math processing error": "\"{{message}}\" when texifying <code>{{expression}}</code>", "die.numbas failed": "Numbas has failed", "die.sorry": "Sorry, Numbas has encountered an error which means it can't continue. Below is a description of the error.", "die.error": "\uc624\ub958", "modal.ok": "\ud655\uc778", "modal.cancel": "\ucde8\uc18c", "exam.exam name": "\uc2dc\ud5d8 \uc774\ub984:", "exam.random seed": "Session ID:", "exam.student name": "\ud559\uc0dd \uc774\ub984:", "exam.number of questions": "\uc9c8\ubb38\uc758 \uc218:", "exam.marks available": "Marks Available:", "exam.pass percentage": "\ud569\uaca9 \ud37c\uc13c\ud2b8:", "exam.time allowed": "Time Allowed:", "exam.passed": "\ud569\uaca9", "exam.failed": "\ubd88\ud569\uaca9", "exam.review header": "Review: ", "frontpage.start": "\uc2dc\uc791", "suspend.paused header": "\uc77c\uc2dc\uc815\uc9c0", "suspend.exam suspended": "\uc2dc\ud5d8\uc774 \uc77c\uc2dc\uc815\uc9c0\ub418\uc5c8\uc2b5\ub2c8\ub2e4. <em>\uacc4\uc18d\ud558\uae30</em>\ub97c \ub20c\ub7ec \uacc4\uc18d\ud558\uc2ed\uc2dc\uc624.", "suspend.you can resume": "You will be able to resume this session the next time you start this activity.", "suspend.resume": "\uc7ac\uac1c", "result.exit": "\uc2dc\ud5d8 \uc885\ub8cc", "result.print": "Print this results summary", "result.exam summary": "Exam Summary", "result.performance summary": "Performance Summary", "result.exam start": "Exam Start:", "result.exam stop": "Exam Stop:", "result.time spent": "Time Spent:", "result.questions attempted": "Questions Attempted:", "result.score": "\uc810\uc218:", "result.result": "\uacb0\uacfc:", "result.question number": "\ubb38\uc81c \ubc88\ud638", "result.question score": "\uc810\uc218", "result.question review title": "Review this question", "result.click a question to review": "Click on a question number to see how your answers were marked and, where available, full solutions.", "end.exam has finished": "The exam has finished. You may now close this window.", "control.confirm leave": "You haven't finished the exam.", "control.not all questions answered": "You have not completed every question in this exam.", "control.not all questions submitted": "You have made changes to one or more answers but not submitted them. Please check each question has been submitted.", "control.confirm end": "Are you sure you want to end the exam? After you end the exam, you will not be able to change any of your answers.", "control.confirm regen": "Would you like to re-randomise this question? If you click OK, all your answers and marks for the current question will be lost.", "control.confirm reveal": "Would you like to reveal the answer to this question? Any marks you have received so far will be locked and you will not be able to answer this question later.", "control.proceed anyway": "Proceed anyway?", "control.regen": "Try another question like this one", "control.submit answer": "Submit answer", "control.submit all parts": "Submit all parts", "control.submit again": "Submit again", "control.submit": "Submit", "control.previous": "Previous", "control.next": "Next", "control.advice": "Advice", "control.reveal": "Reveal answers", "control.total": "Total", "control.pause": "Pause", "control.end exam": "End Exam", "control.back to results": "Go back to results", "display.part.jme.error making maths": "Error making maths display", "exam.xml.bad root": "Root element of exam XML should be 'exam'", "exam.changeQuestion.no questions": "This exam contains no questions! Check the .exam file for errors.", "feedback.you were awarded": "You were awarded <strong>{{count,niceNumber}}</strong> $t(mark).", "feedback.taken away": "<strong>{{count,niceNumber}}</strong> $t(mark) $t(was) taken away.", "jme.tokenise.invalid": "Invalid expression: <code>{{expression}}</code>", "jme.shunt.not enough arguments": "Not enough arguments for operation {{op}}", "jme.shunt.no left bracket in function": "No matching left bracket in function application or tuple", "jme.shunt.no left square bracket": "No matching left bracket", "jme.shunt.no left bracket": "No matching left bracket", "jme.shunt.no right bracket": "No matching right bracket", "jme.shunt.no right square bracket": "No matching right square bracket to end list", "jme.shunt.missing operator": "Expression can't be evaluated -- missing an operator.", "jme.typecheck.function maybe implicit multiplication": "Operation {{name}} is not defined. Did you mean <br/><code>{{first}}*{{possibleOp}}(...)</code>?", "jme.typecheck.function not defined": "Operation <code>{{op}}</code> is not defined. Is <code>{{op}}</code> a variable, and did you mean <code>{{suggestion}}*(...)</code>?", "jme.typecheck.op not defined": "Operation '{{op}}' is not defined.", "jme.typecheck.no right type definition": "No definition of '{{op}}' of correct type found.", "jme.typecheck.no right type unbound name": "Variable <code>{{name}}</code> is not defined.", "jme.typecheck.map not on enumerable": "<code>map</code> operation must work over a list or a range, not {{type}}", "jme.evaluate.undefined variable": "Variable {{name}} is undefined", "jme.thtml.not html": "Passed a non-HTML value into the THTML constructor.", "jme.func.switch.no default case": "No default case for Switch statement", "jme.func.listval.invalid index": "Invalid list index {{index}} on list of size {{size}}", "jme.func.listval.not a list": "Object is not subscriptable", "jme.func.matrix.invalid row type": "Can't construct a matrix from rows of type {{type}}", "jme.func.except.continuous range": "Can't use the 'except' operator on continuous ranges.", "jme.matrix.reports bad size": "Matrix reports its size incorrectly - must be an error in constructor function", "jme.texsubvars.no right bracket": "No matching <code>]</code> in {{op}} arguments.", "jme.texsubvars.missing parameter": "Missing parameter in {{op}}: {{parameter}}", "jme.texsubvars.no right brace": "No matching <code>}</code> in {{op}}", "jme.user javascript.error": "Error in user-defined javascript function <code>{{name}}</code>: {{message}}", "jme.variables.error making function": "Error making function <code>{{name}}</code>: {{message}}", "jme.variables.syntax error in function definition": "Syntax error in function definition", "jme.variables.variable not defined": "Variable <code>{{name}}</code> is not defined.", "jme.variables.empty definition": "Definition of variable <code>{{name}}</code> is empty.", "jme.variables.circular reference": "Circular variable reference in definition of <code>{{name}}</code>", "jme.variables.error computing dependency": "Error computing referenced variable <code>{{name}}</code>", "jme.variables.error evaluating variable": "Error evaluating variable {{name}}: {{message}}", "jme.variables.question took too many runs to generate variables": "A valid set of question variables was not generated in time.", "jme.display.unknown token type": "Can't texify token type {{type}}", "jme.display.collectRuleset.no sets": "No sets given to collectRuleset!", "jme.display.collectRuleset.set not defined": "Ruleset {{name}} has not been defined", "jme.display.simplifyTree.no scope given": "Numbas.jme.display.simplifyTree must be given a Scope", "math.precround.complex": "Can't round to a complex number of decimal places", "math.siground.complex": "Can't round to a complex number of sig figs", "math.combinations.complex": "Can't compute combinations of complex numbers", "math.permutations.complex": "Can't compute permutations of complex numbers", "math.gcf.complex": "Can't compute GCF of complex numbers", "math.lcm.complex": "Can't compute LCM of complex numbers", "math.lt.order complex numbers": "Can't order complex numbers", "math.choose.empty selection": "Empty selection given to random function", "matrixmath.abs.non-square": "Can't compute the determinant of a matrix which isn't square.", "matrixmath.abs.too big": "Sorry, can't compute the determinant of a matrix bigger than 3x3 yet.", "matrixmath.mul.different sizes": "Can't multiply matrices of different sizes.", "vectormath.cross.not 3d": "Can only take the cross product of 3-dimensional vectors.", "vectormath.dot.matrix too big": "Can't calculate dot product of a matrix which isn't $1 \\times N$ or $N \\times 1$.", "vectormath.cross.matrix too big": "Can't calculate cross product of a matrix which isn't $1 \\times N$ or $N \\times 1$.", "part.with steps answer prompt": "Answer: ", "part.script.error": "Error in part {{path}} custom script {{script}}: {{message}}", "part.marking.steps no matter": "Because you received full marks for the part, your answers to the steps aren't counted.", "part.marking.revealed steps no penalty": "You revealed the steps.", "part.marking.used variable replacements": "This part was marked using your answers to previous parts.", "part.marking.variable replacement part not answered": "You must answer {{part}} first", "part.marking.resubmit because of variable replacement": "This part's marking depends on your answers to other parts, which you have changed. Submit this part again to update your score.", "part.marking.not submitted": "No answer submitted", "part.marking.did not answer": "You did not answer this question.", "part.marking.nothing entered": "You did not enter an answer.", "part.marking.incorrect": "Your answer is incorrect.", "part.marking.correct": "Your answer is correct.", "part.marking.uncaught error": "Error when marking {{part}}: {{message}}", "part.marking.no result": "This part could not be marked.", "part.correct answer": "Expected answer:", "part.missing type attribute": "{{part}}: Missing part type attribute", "part.unknown type": "{{part}}: Unrecognised part type {{type}}", "part.setting not present": "Property '{{property}}' not set", "part.jme.answer missing": "Correct answer is missing", "part.jme.answer too long": "Your answer is too long.", "part.jme.answer too short": "Your answer is too short.", "part.jme.answer invalid": "Your answer is not a valid mathematical expression.<br/>{{message}}.", "part.jme.marking.correct": "Your answer is numerically correct.", "part.jme.must-have bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.must-have one": "Your answer must contain: {{strings}}", "part.jme.must-have several": "Your answer must contain all of: {{strings}}", "part.jme.not-allowed bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.not-allowed one": "Your answer must not contain: {{strings}}", "part.jme.not-allowed several": "Your answer must not contain any of: {{strings}}", "part.jme.unexpected variable name": "Your answer was interpreted to use the unexpected variable name <code>{{name}}</code>.", "part.jme.unexpected variable name suggestion": "Your answer was interpreted to use the unexpected variable name <code>{{name}}</code>. Did you mean <code>{{suggestion}}</code>?", "part.patternmatch.display answer missing": "Display answer is missing", "part.patternmatch.correct except case": "Your answer is correct, except for the case.", "part.numberentry.correct except decimal": "Your answer is within the allowed range, but decimal numbers are not allowed.", "part.numberentry.correct except fraction": "Your answer is within the allowed range, but fractions are not allowed.", "part.numberentry.answer invalid": "You did not enter a valid number.", "part.numberentry.answer not integer": "Your answer is invalid. You must enter a whole number, not a decimal.", "part.numberentry.answer not integer or decimal": "Your answer is invalid. You must enter an integer or a decimal.", "part.numberentry.zero sig fig": "This part is set up to round the student's answer to zero significant figures, which has no meaning.", "part.mcq.options def not a list": "The expression defining the {{properties}} is not a list.", "part.mcq.marking matrix string empty": "The custom marking matrix expression is empty.", "part.mcq.choices missing": "Definition of choices is missing", "part.mcq.matrix not a number": "Part {{part}} marking matrix cell ({{row}},{{column}}) does not evaluate to a number", "part.mcq.wrong number of choices": "You selected the wrong number of choices.", "part.mcq.no choices selected": "No choices selected.", "part.mcq.matrix not a list": "Marking matrix, defined by JME expression, is not a list but it should be.", "part.mcq.matrix wrong type": "Element of invalid type '{{type}}' used in marking matrix.", "part.mcq.matrix mix of numbers and lists": "Mix of numbers and lists used in marking matrix.", "part.mcq.matrix wrong size": "Marking matrix is the wrong size.", "part.mcq.correct choice": "You chose the correct answer.", "part.matrix.answer invalid": "Your answer is not valid.", "part.matrix.invalid cell": "One or more of the cells in your answer is empty or invalid.", "part.matrix.some incorrect": "One or more of the cells in your answer is incorrect, but you have been awarded marks for the rest.", "part.matrix.empty": "You have not entered an answer.", "part.matrix.empty cell": "One or more of the cells in your answer is empty.", "part.matrix.size mismatch": "The question author hasn't allowed the student to decide the dimensions of their answer, but the correct answer is {{correct_dimensions}} while the answer input is {{input_dimensions}}", "part.gapfill.feedback header": "<strong>Gap {{index}}</strong>", "part.extension.not implemented": "Part hasn't implemented the <code>{{name}}</code> method.", "question.loaded name mismatch": "Can't resume this attempt - the package has changed since the last session.", "question.error": "Question {{number}}: {{message}}", "question.preamble.error": "Error in preamble: {{message}}", "question.preamble.syntax error": "Syntax error in preamble", "question.unsupported part type": "Unsupported part type", "question.header": "Question {{number}}", "question.submit part": "Submit part", "question.show steps": "Show steps", "question.show steps penalty": "You will lose <strong>{{count,niceNumber}}</strong> $t(mark).", "question.show steps no penalty": "Your score will not be affected.", "question.show steps already penalised": "You have already shown steps. You can show them again with no further penalty.", "question.hide steps": "Hide steps", "question.hide steps no penalty": "Your score will not be affected.", "question.advice": "Advice", "question.no such part": "Can't find part {{path}}", "question.can not submit": "Can not submit answer - check for errors.", "question.answer submitted": "Answer submitted", "question.score feedback.show": "Show feedback", "question.score feedback.hide": "Hide feedback", "question.score feedback.ascore total actual": "Score: {{score,niceNumber}}/{{marks,niceNumber}}", "question.score feedback.score total": "{{marksString}}. Answered.", "question.score feedback.score actual": "Score: {{scoreString}}", "question.score feedback.answered": "Answered.", "question.score feedback.unanswered": "Unanswered.", "question.score feedback.correct": "Your answer is correct", "question.score feedback.partial": "Your answer is partially correct", "question.score feedback.wrong": "Your answer is incorrect", "question.selector.unsubmitted changes": "Unsubmitted changes.", "timing.no accumulator": "no timing accumulator {{name}}", "timing.time remaining": "Time remaining:", "xml.could not load": "Couldn't load an XML document: {{message}}", "xml.property not number": "Property {{name}} should be a number, but isn't ({{value}}), in node {{element}}", "xml.property not boolean": "Property {{name}} should be a boolean, but isn't ({{value}}), in node {{element}}", "xml.error in variable definition": "Error in definition of variable <code>{{name}}</code>", "scorm.error initialising": "Error initialising SCORM protocol: {{message}}", "scorm.failed save": "<p>The request to save data to the server failed. Press <b>OK</b> to try again.</p>\n<p>If you get this message repeatedly, check your internet connection or use a different computer. Your previously submitted answers have been successfully saved and will be restored if you resume this session on a different computer.</p>\n<p>If this message appears persistently and you can't save <em>any</em> answers, please contact your lecturer or teacher.</p>", "scorm.no exam suspend data": "Failed to resume: no exam suspend data.", "scorm.error loading suspend data": "Error loading suspend data: {{message}}", "scorm.error loading question": "Error loading question {{number}}: {{message}}", "scorm.no question suspend data": "No question suspend data", "scorm.error loading part": "Error loading part {{part}}: {{message}}", "scorm.no part suspend data": "No part suspend data", "util.product.non list": "Passed a non-list to <code>Numbas.util.product</code>", "mark": "mark", "was": "was", "part": "part", "gap": "gap", "step": "step", "jme.substituteTree.undefined variable": "Undefined variable: <code>{{name}}</code>", "jme.user javascript.returned undefined": "User-defined javascript function <code>{{name}}</code> returned <code>undefined</code>.", "part.marking.steps change": "You were awarded <strong>{{count,niceNumber}}</strong> $t(mark) for your answers to the steps.", "part.marking.revealed steps with penalty": "You revealed the steps. The maximum you can score for this part is <strong>{{count,niceNumber}}</strong> $t(mark). Your scores will be scaled down accordingly.", "part.marking.total score": "You scored <strong>{{count,niceNumber}}</strong> $t(mark) for this part.", "part.numberentry.precision type.dp": "decimal place", "part.numberentry.precision type.dp_plural": "decimal places", "part.numberentry.precision type.sigfig": "significant figure", "part.numberentry.precision type.sigfig_plural": "significant figures", "part.numberentry.give your answer to precision": "Round your answer to {{count,niceNumber}} {{precisionType}}.", "question.unsubmitted changes": "You have made a change to your answer but not submitted it. Please check your answer and then press the <strong>Submit answer</strong> button.", "question.unsubmitted changes_plural": "You have made changes to your answers but not submitted them. Please check your answers to each part and then press the <strong>Submit all parts</strong> button.", "util.equality not defined for type": "Equality not defined for type {{type}}", "mark_plural": "marks", "was_plural": "were", "die.script not loaded": "Numbas couldn't start because the file <code>{{file}}</code> was not loaded. Check that it's included in <code>scripts.js</code>.", "math.combinations.n less than zero": "Can't compute combinations: n is less than zero", "math.combinations.k less than zero": "Can't compute combinations: k is less than zero", "math.combinations.n less than k": "Can't compute combinations: n is less than k", "math.permutations.n less than zero": "Can't compute permutations: n is less than zero", "math.permutations.k less than zero": "Can't compute permutations: k is less than zero", "math.permutations.n less than k": "Can't compute permutations: n is less than k", "part.numberentry.give your answer to precision_0": "Round your answer to the nearest integer.", "mathjax.error": "MathJax processing error: {{message}}", "mathjax.error with context": "MathJax processing error in {{context}}: {{message}}", "exam.introduction": "Exam introduction", "exam.feedback": "Exam feedback message", "jme.tokenise.keypair key not a string": "Dictionary key should be a string, not {{type}}.", "jme.shunt.list mixed argument types": "Can't parse {{mode}}: mix of dictionary and list elements", "jme.func.listval.key not in dict": "Dictionary does not contain the key <code>{{key}}</code>", "part.prompt": "prompt", "part.feedback": "feedback", "part.numberentry.answer not reduced": "Your answer is not reduced to lowest terms.", "part.numberentry.give your answer as a reduced fraction": "Reduce your answer to lowest terms.", "part.numberentry.negative decimal places": "This part is set up to round the student's answer to a negative number of decimal places, which has no meaning.", "part.mcq.choices": "choices", "part.mcq.answers": "answers", "part.mcq.matrix cell empty": "Part {{part}} marking matrix cell ({{row}},{{column}}) is empty", "part.mcq.matrix jme error": "Part {{part}} marking matrix cell ({{row}},{{column}}) gives a JME error: {{error}}", "question.statement": "Statement", "ruleset.circular reference": "Circular reference in definition of ruleset <code>{{name}}</code>", "ruleset.set not defined": "Ruleset {{name}} has not been defined", "jme.evaluate.no scope given": "Numbas.jme.evaluate must be given a Scope"}}, "es-es": {"translation": {"page.loading": "Cargando...", "page.saving": "<p>Guardando.</p>\n<p>Esto podr\u00eda demorar unos segundos.</p>", "mathjax.math processing error": "Error de procesamiento en MathJax: \"{{-message}}\" al compilar <code>{{expression}}</code> ", "die.numbas failed": "Numbas ha fallado", "die.sorry": "Lo siento, Numbas ha encontrado un error, por lo tanto no puede continuar. A continuaci\u00f3n se muestra una descripci\u00f3n del error.", "die.error": "Error", "modal.ok": "OK", "modal.cancel": "Cancelar", "exam.exam name": "Nombre del examen:", "exam.random seed": "ID de la sesi\u00f3n:", "exam.student name": "Nombre del estudiante: ", "exam.number of questions": "N\u00famero de preguntas:", "exam.marks available": "Puntuaci\u00f3n m\u00e1xima:", "exam.pass percentage": "Porcentaje de aprobaci\u00f3n:", "exam.time allowed": "Tiempo permitido:", "exam.passed": "Aprobado... \u00a1Muy bien!", "exam.failed": "Reprobado... Int\u00e9ntelo otra vez", "exam.review header": "Revisi\u00f3n: ", "frontpage.start": "Comenzar examen", "suspend.paused header": "En pausa", "suspend.exam suspended": "Se ha detenido el examen. Presione <em>Reanudar</em> para continuar.", "suspend.you can resume": "Usted podr\u00e1 reanudar esta sesi\u00f3n la pr\u00f3xima vez que inicie esta actividad.", "suspend.resume": "Reanudar", "result.exit": "Salir del examen", "result.print": "Imprimir informe de resultados", "result.exam summary": "Informe del examen", "result.performance summary": "Informe de rendimiento", "result.exam start": "Inicio del examen:", "result.exam stop": "Termin\u00f3 el examen:", "result.time spent": "Tiempo invertido en resolver el examen:", "result.questions attempted": "Preguntas respondidas:", "result.score": "Puntuaci\u00f3n:", "result.result": "Resultado:", "result.question number": "N\u00famero de la pregunta", "result.question score": "Puntuaci\u00f3n", "result.question review title": "Revisar esta pregunta", "result.click a question to review": "Pulse sobre el n\u00famero de la pregunta para revisar sus respuestas y, si est\u00e1 disponible, la soluci\u00f3n al problema.", "end.exam has finished": "El examen ha finalizado. Ahora puede cerrar esta ventana.", "control.confirm leave": "Usted no ha terminado de responder su examen.", "control.not all questions answered": "Usted no ha contestado todas las preguntas de su examen.", "control.not all questions submitted": "Ha realizado cambios en una o m\u00e1s respuestas, pero no las ha enviado. Por favor, compruebe que se ha respondido cada pregunta.", "control.confirm end": "\u00bfEst\u00e1 seguro de que quiere terminar el examen? En cuanto lo finalice, ya no ser\u00e1 posible cambiar las respuestas.", "control.confirm regen": "\u00bfQuiere nuevos valores para la pregunta? Si usted pulsa sobre OK, todas sus respuestas y puntuaci\u00f3n para esta pregunta se perder\u00e1n.", "control.confirm reveal": "\u00bfQuiere mostrar las respuestas a esta pregunta? Perder\u00e1 la puntuaci\u00f3n recibida hasta ahora.", "control.proceed anyway": "\u00bfDesea continuar?", "control.regen": "Intentar una nueva versi\u00f3n de esta pregunta", "control.submit answer": "Enviar Respuesta", "control.submit all parts": "Enviar todas las partes de la pregunta", "control.submit again": "Enviar nuevamente", "control.submit": "Enviar", "control.previous": "Anterior", "control.next": "Siguiente", "control.advice": "Consejo", "control.reveal": "Mostrar las respuestas correctas", "control.total": "Total", "control.pause": "Pausa", "control.end exam": "Finalizar examen", "control.back to results": "Volver a los resultados", "display.part.jme.error making maths": "Error al desplegar escritura matem\u00e1tica", "exam.xml.bad root": "El elemento ra\u00edz de un examen XML deber\u00eda ser 'exam'", "exam.changeQuestion.no questions": "\u00a1Este examen no tiene preguntas! Revise el archivo con extensi\u00f3n .exam .", "feedback.you were awarded": "Ha logrado <strong>{{count}}</strong> $t(mark).", "feedback.taken away": "%s %s ha sido quitado.", "jme.tokenise.invalid": "Expresi\u00f3n inv\u00e1lida: <code>{{expression}}</code>", "jme.shunt.not enough arguments": "No hay suficientes argumentos para la operaci\u00f3n {{op}}", "jme.shunt.no left bracket in function": "No concuerda el par\u00e9ntesis de la izquierda al aplicar la funci\u00f3n o tupla", "jme.shunt.no left square bracket": "No concuerda el corchete de la izquierda", "jme.shunt.no left bracket": "No concuerda el par\u00e9ntesis de la izquierda", "jme.shunt.no right bracket": "No concuerda el par\u00e9ntesis de la derecha", "jme.shunt.no right square bracket": "No concuerda el corchete de la derecha al final de la lista", "jme.shunt.missing operator": "La expresi\u00f3n no puede ser evaluada -- falta un operador.", "jme.typecheck.function maybe implicit multiplication": "La operaci\u00f3n {{name}} no est\u00e1 definida. \u00bfTal vez quiso decir <br/><code>{{first}}*{{possibleOp}}(...)</code>?", "jme.typecheck.function not defined": "La operaci\u00f3n {{op}} no est\u00e1 definida. \u00bfTal vez quiso decir <code>{{suggestion}}*(...)</code>?", "jme.typecheck.op not defined": "La operaci\u00f3n '{{op}}' no est\u00e1 definida.", "jme.typecheck.no right type definition": "No se ha encontrado una definici\u00f3n para '{{op}}' del tipo correcto.\n", "jme.typecheck.no right type unbound name": "La variable <code>{{name}}</code> no est\u00e1 definida.", "jme.typecheck.map not on enumerable": "<code>map</code> la operaci\u00f3n funciona sobre una lista o un rango, no sobre{{type}}", "jme.evaluate.undefined variable": "La variable {{name}} no est\u00e1 definida", "jme.thtml.not html": "Se ha pasado un valor no HTML al constructor THTML.\n", "jme.func.switch.no default case": "No ha establecido una opci\u00f3n por defecto.", "jme.func.listval.invalid index": "\u00cdndice {{index}} de lista inv\u00e1lido en una lista de tama\u00f1o {{size}}", "jme.func.listval.not a list": "El objeto no se pude indizar", "jme.func.matrix.invalid row type": "No se puede construir una matriz a partir de filas de tipo {{type}}", "jme.func.except.continuous range": "No se puede usar el operador 'except' en intervalos continuos.", "jme.matrix.reports bad size": "No se puede construir una matriz a partir de filas de tipo {{type}}", "jme.texsubvars.no right bracket": "Falta <code>]</code> en los argumentos de {{op}}.", "jme.texsubvars.missing parameter": "Falta un par\u00e1metro en {{op}}: {{parameter}}", "jme.texsubvars.no right brace": "No corresponde <code>}</code> en {{op}}", "jme.user javascript.error": "Error en la funci\u00f3n javascript definida por el usuario <code> {{name}} </ code>: {{-message}}", "jme.variables.error making function": "Hubo un error en la construcci\u00f3n de la funci\u00f3n <code>{{name}}</code>: {{-message}}", "jme.variables.syntax error in function definition": "Error de sintaxis en la definici\u00f3n de la funci\u00f3n", "jme.variables.variable not defined": "La variable {{name}} no est\u00e1 definida.", "jme.variables.empty definition": "La variable definida <code>{{name}}</code> est\u00e1 vac\u00eda", "jme.variables.circular reference": "Hay una referencia circular en la definici\u00f3n de la variable <code>{{name}}</code>.", "jme.variables.error computing dependency": "Error al calcular la variable referenciada <code> {{name}} </ code>", "jme.variables.error evaluating variable": "Error al evaluar la variable {{name}}: {{-message}}", "jme.variables.question took too many runs to generate variables": "Un conjunto v\u00e1lido de variables de pregunta no se gener\u00f3 a tiempo.", "jme.display.unknown token type": "No puedo pasar a texto la expresi\u00f3n de tipo {{type}}", "jme.display.collectRuleset.no sets": "No se han proporcionado conjuntos a collectRuleset!", "jme.display.collectRuleset.set not defined": "El conjunto de reglas para {{name}}, no ha sido definido.", "jme.display.simplifyTree.no scope given": "Hay que proporcionar un \u00e1mbito de actuaci\u00f3n a Numbas.jme.display.simplifyTree", "math.precround.complex": "No se puede aproximar un n\u00famero complejo a un n\u00famero de cifras decimales.", "math.siground.complex": "No se puede aproximar un n\u00famero complejo a un n\u00famero de cifras significativas.", "math.combinations.complex": "No se puede calcular combinaciones de n\u00fameros complejos.", "math.permutations.complex": "No se puede calcular permutaciones de n\u00fameros complejos.", "math.gcf.complex": "No se puede calcular MCD de n\u00fameros complejos.", "math.lcm.complex": "No se puede calcular MCM de n\u00fameros complejos.", "math.lt.order complex numbers": "No se pueden ordenar n\u00fameros complejos.", "math.choose.empty selection": "Se le ha proporcionado una selecci\u00f3n vac\u00eda a una funci\u00f3n aleatoria", "matrixmath.abs.non-square": "No se puede calcular el determinante de una matriz que no es cuadrada.", "matrixmath.abs.too big": "Lo sentimos, no se puede calcular el determinante de una matriz mayor que 3x3.", "matrixmath.mul.different sizes": "No se pueden multiplicar matrices de diferentes tama\u00f1os (columnas-filas).", "vectormath.cross.not 3d": "Solo puedo hacer el producto cruzado de vectores tridimensionales.", "vectormath.dot.matrix too big": "Solo puedo calcular el producto escalar de matrices que o bien tengan una sola fila o una sola columna.", "vectormath.cross.matrix too big": "Solo puedo calcular el producto cruzado de matrices que o bien tengan una sola fila o una sola columna.", "part.with steps answer prompt": "Respuesta.", "part.script.error": "Error en la parte {{path}} del script personalizado {{script}}: {{-message}}", "part.marking.steps no matter": "Como has recibido la m\u00e1xima puntuaci\u00f3n para esta parte, las respuestas de los pasos intermedios no se tienen en consideraci\u00f3n.", "part.marking.revealed steps no penalty": "Has revelado los pasos.", "part.marking.used variable replacements": "Esta parte ha sido puntuada usando tus respuestas a las partes anteriores.", "part.marking.variable replacement part not answered": "Debes responder primero a {{part}}", "part.marking.resubmit because of variable replacement": "La puntuaci\u00f3n de esta parte depende de las respuestas que has dado en partes anteriores, las cuales has cambiado. Por favor, env\u00eda de nuevo la respuesta de esta parte para actualizar tu nota.", "part.marking.not submitted": "No ha enviado respuesta", "part.marking.did not answer": "Usted no ha respondido esta pregunta.", "part.marking.nothing entered": "Usted no ha introducido una respuesta.", "part.marking.incorrect": "Su respuesta es incorrecta.", "part.marking.correct": "Su respuesta es correcta.", "part.marking.uncaught error": "Error al puntuar {{part}}: {{-message}}", "part.marking.no result": "Esta parte no se pudo puntuar.", "part.correct answer": "Respuesta correcta:", "part.missing type attribute": "{{part}}: falta atributo de tipo en la parte", "part.unknown type": "{{part}}: tipo {{type}} de parte no reconocido", "part.setting not present": "Propiedad '{{property}}' no establecida", "part.jme.answer missing": "Falta la respuesta correcta.", "part.jme.answer too long": "Su respuesta es demasiado larga.", "part.jme.answer too short": "Su respuesta es demasiado corta.", "part.jme.answer invalid": "Su respuesta no es una expresi\u00f3n matem\u00e1tica v\u00e1lida. <br/> {{-message}}.", "part.jme.marking.correct": "Su respuesta en num\u00e9ricamente correcta.", "part.jme.must-have bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.must-have one": "Su respuesta debe contener: {{strings}}", "part.jme.must-have several": "Su respuesta debe contener todas las cadenas: {{strings}}", "part.jme.not-allowed bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.not-allowed one": "Su respuesta no debe contener: {{strings}}", "part.jme.not-allowed several": "Su respuesta no debe contener: {{strings}}\n", "part.jme.unexpected variable name": "Se ha interpretado que tu respuesta usa el nombre de variable <code>{{name}}</code>, el cual es inapropiado.", "part.jme.unexpected variable name suggestion": "Se ha interpretado que tu respuesta usa el nombre de variable <code>{{name}}</code>, el cual es inapropiado. \u00bfQuiz\u00e1s quer\u00edas decir  <code>{{suggestion}}</code>?", "part.patternmatch.display answer missing": "Falta respuesta para mostrar.", "part.patternmatch.correct except case": "Su respuesta es correcta, excepto para el caso.", "part.numberentry.correct except decimal": "Su respuesta est\u00e1 dentro del rango permitido, pero los n\u00fameros decimales no est\u00e1n permitidos.", "part.numberentry.correct except fraction": "Su respuesta est\u00e1 dentro del rango permitido, pero no se permiten fracciones.", "part.numberentry.answer invalid": "No ha introducido un n\u00famero v\u00e1lido.", "part.numberentry.answer not integer": "Su respuesta no es v\u00e1lida. Debe ingresar un n\u00famero entero, no un decimal.", "part.numberentry.answer not integer or decimal": "Su respuesta no es v\u00e1lida. Debe ingresar un n\u00famero entero o decimal.", "part.numberentry.zero sig fig": "Esta parte est\u00e1 configurada para redondear la respuesta del estudiante a cero cifras significativas, lo cual no tiene sentido.", "part.mcq.options def not a list": "La expresi\u00f3n que define las {{properties}} no es una lista.", "part.mcq.marking matrix string empty": "La matriz de puntuaciones personalizada est\u00e1 vac\u00eda.", "part.mcq.choices missing": "Falta la definici\u00f3n de opciones", "part.mcq.matrix not a number": "La celda {{row}},{{column}} de la matriz de puntuaciones de la parte {{part}} no se evalua como un n\u00famero.", "part.mcq.wrong number of choices": "Ha seleccionado un n\u00famero incorrecto de opciones.", "part.mcq.no choices selected": "No hay opciones seleccionadas.", "part.mcq.matrix not a list": "La matriz de puntuaciones, definida por la expresi\u00f3n JME, no es una lista, y deber\u00eda serlo.", "part.mcq.matrix wrong type": "Se ha usado en la matriz de puntuaciones un elemento de tipo '{{type}}', el cual no es v\u00e1lido.", "part.mcq.matrix mix of numbers and lists": "Se ha usado una mezcla de listas y n\u00fameros en la matriz de puntuaciones.", "part.mcq.matrix wrong size": "La matriz de puntuaciones tiene un tama\u00f1o err\u00f3neo.", "part.mcq.correct choice": "Usted eligi\u00f3 la respuesta correcta.", "part.matrix.answer invalid": "Su respuesta no es v\u00e1lida.", "part.matrix.invalid cell": "Una o m\u00e1s de las celdas de la respuesta est\u00e1 vac\u00eda o es inv\u00e1lida.", "part.matrix.some incorrect": "Una o m\u00e1s celdas de tu respuesta son incorrectas, pero se te ha puntuado el resto.", "part.matrix.empty": "No ha ingresado una respuesta.", "part.matrix.empty cell": "Una o m\u00e1s de las celdas de la respuesta est\u00e1 vac\u00eda.", "part.matrix.size mismatch": "El autor de la pregunta no ha permitido al estudiante decidir la dimensiones de su respuesta; la respuesta correcta tiene {{correct_dimensions}} mientras que la entrada es {{input_dimensions}}", "part.gapfill.feedback header": "<strong>{{name}}</strong>", "part.extension.not implemented": "Esta parte no tiene implementado el m\u00e9todo <code>{{name}}</code>.", "question.loaded name mismatch": "No se puede reanudar este intento - el paquete ha cambiado desde la \u00faltima sesi\u00f3n.", "question.error": "Pregunta {{number}}: {{-message}}", "question.preamble.error": "Error en el pre\u00e1mbulo: {{-message}}", "question.preamble.syntax error": "Error de sintaxis en el pre\u00e1mbulo", "question.unsupported part type": "Tipo de parte no soportado", "question.header": "Pregunta {{number}}", "question.submit part": "Enviar esta parte", "question.show steps": "Mostrar pasos", "question.show steps penalty": "Usted perder\u00e1 <strong>{{count}}</strong> $t(mark).", "question.show steps no penalty": "Su puntuaci\u00f3n no se ver\u00e1 afectada.", "question.show steps already penalised": "Ya se han mostrado los pasos, no tendr\u00e1 penalizaci\u00f3n adicional.", "question.hide steps": "Ocultar pasos", "question.hide steps no penalty": "Su puntuaci\u00f3n no se ver\u00e1 afectada.", "question.advice": "Soluci\u00f3n", "question.no such part": "No puedo encontrar la parte {{path}}", "question.can not submit": "No se puede enviar la respuesta - compruebe si hay errores.", "question.answer submitted": "Respuesta enviada", "question.score feedback.show": "Mostrar retroalimentaci\u00f3n", "question.score feedback.hide": "Ocultar retroalimentaci\u00f3n", "question.score feedback.answered total actual": "Puntuaci\u00f3n: {{score}}/{{marks}}", "question.score feedback.answered total": "{{marksString}}. Respondida.", "question.score feedback.answered actual": "Puntuaci\u00f3n: {{scoreString}}", "question.score feedback.answered": "Respondida.", "question.score feedback.unanswered": "No respondida.", "question.score feedback.unanswered total": "{{marksString}}.", "question.score feedback.correct": "Su respuesta es correcta", "question.score feedback.partial": "Su respuesta es parcialmente correcta", "question.score feedback.wrong": "Su respuesta es incorrecta", "question.selector.unsubmitted changes": "Cambios no enviados.", "timing.no accumulator": "no hay acumulador de tiempo {{name}}", "timing.time remaining": "Tiempo restante: %s", "xml.could not load": "No se pudo cargar un documento XML: {{-message}}.", "xml.property not number": "La propiedad {{name}} en el nodo {{element}} deber\u00eda ser un numero, pero no es ({{value}})", "xml.property not boolean": "La propiedad {{name}} en el nodo {{element}} deber\u00eda ser un valor booleano, pero no es ({{value}})", "xml.error in variable definition": "Error en la definici\u00f3n de la variable <code>{{name}}</code>", "scorm.error initialising": "Error al inicializar el protocolo SCORM: {{-message}}.", "scorm.failed save": "<p>Ha fallado la petici\u00f3n de guardar los datos en el servidor. Pulsa  <b>OK</b> para intentarlo de nuevo.</p>\n<p>Si este mensaje aparece a menudo, comprueba tu conexi\u00f3n a internet o utiliza otro ordenador. Las respuestas enviadas con anterioridad han sido guardadas y se restaurar\u00e1n cuando restablezcas esta sesi\u00f3n en otro ordenador.</p>\n<p>Si persiste este mensaje y no puedes guardar <em>ninguna</em> respuesta, por favor, contacta con tu profesor.</p> ", "scorm.no exam suspend data": "No se ha podido reanudar: no hay datos del examen suspendido.", "scorm.error loading suspend data": "Error al cargar datos suspendidos: {{-message}}", "scorm.error loading question": "Error al cargar la pregunta {{number}}: {{-message}}", "scorm.no question suspend data": "No hay datos de la pregunta suspendida", "scorm.error loading part": "Error ala cargar la parte {{part}}: {{-message}}", "scorm.no part suspend data": "No hay datos suspendidos de esta parte", "util.product.non list": "A <code>Numbas.util.product</code> se le a pasado un argumento que no es una lista", "mark": "punto", "was": "fue", "part": "Parte.", "gap": "casilla", "step": "Paso.", "jme.substituteTree.undefined variable": "Variable no definida: <code> {{name}} </ code>", "jme.user javascript.returned undefined": "Funci\u00f3n javascript <code>{{name}}</code>, definida por el usuario, devolvi\u00f3 <code>undefined</code>.", "part.marking.steps change": "Has ganado <strong>{{count,niceNumber}}</strong> $t(mark) por tus respuestas a los pasos.", "part.marking.revealed steps with penalty": "Has revelado los pasos. Lo m\u00e1ximo que puedes puntear para esta parte es <strong>{{count,niceNumber}}</strong> $t(mark). Su puntuaci\u00f3n ser\u00e1 reducida en consecuencia.", "part.marking.total score": "Has obtenido <strong>{{count,niceNumber}}</strong> $t(mark) en esta parte.", "part.numberentry.precision type.dp": "posici\u00f3n decimal", "part.numberentry.precision type.dp_plural": "N\u00famero de decimales.", "part.numberentry.precision type.sigfig": "cifra significativa", "part.numberentry.precision type.sigfig_plural": "cifras significativas", "part.numberentry.give your answer to precision": "Redondea tu respuesta a {{count,niceNumber}} {{precisionType}}.", "question.unsubmitted changes": "Has realizado un cambio en tu respuesta, pero no la has enviado a\u00fan. Por favor, revisa tu respuesta y luego presiona el bot\u00f3n <strong>Enviar respuesta</strong>.", "question.unsubmitted changes_plural": "Has realizado cambios a tus respuestas, pero no las has enviado. Por favor, revisa tus respuestas a cada parte y luego presiona el bot\u00f3n  <strong>Enviar todas las partes </strong>. ", "util.equality not defined for type": "Igualdad no definida para el tipo {{type}} ", "mark_plural": "Puntos", "was_plural": "fueron", "die.script not loaded": "Numbas no pudo iniciar porque el archivo <code>{{file}}</code> no carg\u00f3. Compruebe que est\u00e1 incluido en <code>scripts.js</code>. ", "math.combinations.n less than zero": "No se puede calcular combinaciones: n es menor que cero.", "math.combinations.k less than zero": "No se puede calcular combinaciones: k es menor que cero.", "math.combinations.n less than k": "No se puede calcular combinaciones: n es menor que k.", "math.permutations.n less than zero": "No se puede calcular permutaciones: n es menor que cero.", "math.permutations.k less than zero": "No se puede calcular permutacione: k es menor que cero.", "math.permutations.n less than k": "No se puede calcular permutaciones: n es menor que k.", "part.numberentry.give your answer to precision_0": "Redondea tu respuesta al entero m\u00e1s cercano.", "mathjax.error": "MathJax procesando error: {{-message}}", "mathjax.error with context": "MathJax procesando error en {{context}}: {{-message}}", "exam.introduction": "Introducci\u00f3n al examen", "exam.feedback": "Mensaje de retroalimentaci\u00f3n del examen", "jme.tokenise.keypair key not a string": "La clave del diccionario tiene que ser una cadena, no {{type}}", "jme.shunt.list mixed argument types": "No puede analizar gram\u00e1ticamente {{mode}}: mezcla de diccionario y lista de elementos", "jme.func.listval.key not in dict": "El diccionario no contiene la clave <code>{{key}}</code>", "part.prompt": "Indicaci\u00f3n", "part.feedback": "Retroalimentaci\u00f3n", "part.numberentry.answer not reduced": "Su respuesta no est\u00e1 reducida a su menor expresi\u00f3n.", "part.numberentry.give your answer as a reduced fraction": "Reduzca su respuesta a la menor expresi\u00f3n.", "part.numberentry.negative decimal places": "Esta parte est\u00e1 configurada para redondear la respuesta del estudiante a un n\u00famero negativo de posiciones decimales, lo cual no tiene sentido.", "part.mcq.choices": "opciones", "part.mcq.answers": "respuestas", "part.mcq.matrix cell empty": "La parte {{part}} que marca la celda de una matriz ({{row}},{{column}}) est\u00e1 vac\u00eda", "part.mcq.matrix jme error": "La parte {{part}} que marca la celda de una matriz ({{row}}.{{column}}) produce un error JME: {{error}}", "question.statement": "Declaraci\u00f3n", "ruleset.circular reference": "Referencia circular en la definici\u00f3n de ruleset <code>{{name}}</code>", "ruleset.set not defined": "El conjunto de reglas {{name}} no ha sido definido", "jme.evaluate.no scope given": "Hay que darle a Numbas.jme.evaluate un Scope\n", "answer.number.not a number": "La respuesta no es v\u00e1lida", "answer.number.fractions not allowed": "La entrada no tiene que ser una fracci\u00f3n", "answer.jme.invalid expression": "{{-message}}", "answer.matrix.fractions not allowed": "No puedes introducir fracciones.", "answer.matrix.some cell not a number": "Una o m\u00e1s celdas en tu respuesta no es un n\u00famero v\u00e1lido", "exam.enter password": "Contrase\u00f1a", "exam.password.correct": "La contrase\u00f1a es correcta. Ud. puede comenzar el examen.", "exam.password.incorrect": "La contrase\u00f1a es incorrecta.", "frontpage.scorm.lms not connected": "Este examen se est\u00e1 desarrollando en modo independiente. \u00a1Sus respuestas y calificaciones no se guardar\u00e1n!", "result.question review": "Revisar", "control.confirm regen no marks": "\u00bfDesea elegir otra pregunta aleatoria?", "control.confirm reveal no marks": "\u00bfDesea que se le revele la respuesta?", "jme.tokenise.invalid near": "Expresi\u00f3n inv\u00e1lida: <code>{{expression}}</code> en la posici\u00f3n {{position}} cerca de <code>{{nearby}}</code>", "jme.tokenise.number.object not complex": "Se ha pasado un objeto inv\u00e1lido al constructor num\u00e9rico.", "jme.subvars.null substitution": "Sustituci\u00f3n de una variable vac\u00eda: <code>$t(left brace){{str}}$t(right brace)", "jme.type.type already registered": "Ya se ha registrado el tipo de dato {{type}}, por lo que no se puede registrar de nuevo.", "jme.type.no cast method": "No se puede convertir autom\u00e1ticamente de {{from}} en {{}}.", "jme.display.simplifyTree.empty expression": "La expresi\u00f3n est\u00e1 vac\u00eda.", "jme.display.simplifyTree.stuck in a loop": "El simplificador se ha bloqueado en la iteraci\u00f3n: <code>{{expr}}</code>", "math.niceNumber.undefined": "Se esperaba un n\u00famero, pero se obtuvo <code>undefined</code>", "math.rangeToList.zero step size": "No es posible convertir un intervalo con longitud de paso nula a una lista.", "part.error": "{{path}}: {{-message}}", "part.marking.revealed steps": "Ud. ha revelado los pasos.", "part.marking.maximum scaled down": "El puntaje m\u00e1ximo para esta parte es <strong>{{count,niceNumber}}</strong> $t(mark). El puntaje se disminuir\u00e1 acordemente.", "part.marking.minimum score applied": "El puntaje m\u00ednimo para esta parte es <strong>{{score,niceNumber}}</strong>.", "part.marking.maximum score applied": "El puntaje m\u00e1ximo para esta parte es <strong>{{score,niceNumber}}</strong>.", "part.marking.error in marking script": "Se encontr\u00f3 un error en el algoritmo de puntuaci\u00f3n. Por favor rep\u00f3rtelo. {{-message}}", "part.marking.no result after replacement": "No se pudo puntuar esta parte usando las repuestas a las partes anteriores.", "part.marking.missing required note": "El algoritmo de puntuaci\u00f3n no define la nota <code>{{note}}</code>", "marking.apply.not a list": "El primer argumento para <code>apply</code> debe ser una lista, y no lo es", "marking.apply marking script.script not found": "No se encontr\u00f3 el algoritmo de puntuaci\u00f3n <code>{{name}}</code>.", "marking.note.compilation error": "Error al compilar la nota <code>{{name}}</code>: {{-message}}", "marking.note.error evaluating note": "Error al evaluar la nota <code>{{name}}</code> - {{-message}}", "marking.note.invalid definition": "Definici\u00f3n de nota inv\u00e1lida: <code>{{source}}</code>. {{-hint}}", "marking.note.invalid definition.missing colon": "Ud. parece haber olvidado los dos puntos despu\u00e9s del nombre y la descripci\u00f3n.", "marking.note.invalid definition.description missing closing bracket": "Ud. parece haber olvidado el corchete", "marking.note.empty expression": "La nota <code>{{name}}</code> est\u00e1 vac\u00eda.", "marking.script.error parsing notes": "Ud. parece haber olvidado el corchete del cierre.", "part.feedback out of date": "Esta retroalimentaci\u00f3n se basa en la \u00faltima respuesta enviada. Env\u00ede la respuesta cambiada para obtener una retroalimentaci\u00f3n actualizada.", "part.jme.invalid value generator expression": "Invalid value generator expression for variable <code>{{name}}</code>: {{-message}}", "part.mcq.incorrect choice": "La respuesta seleccionada es incorrecta.", "part.matrix.not all cells same precision": "Todas las celdas en su respuesta no tienen la misma precisi\u00f3n.", "part.gapfill.error marking gap": "Error al puntuar {{name}}: {{-message}}", "part.custom.empty setting": "Ning\u00fan valor enviado.", "part.custom.unrecognised input type": "Unrecognised setting type <code>{{input_type}}</code>", "part.custom.error evaluating input option": "Error al evaluar la opci\u00f3n de entrada <code>{{option}}</code>: {{-error}}", "part.custom.input option missing": "Definition of input option <code>{{option}}</code> is missing.", "part.custom.error evaluating setting": "Error al evaluar el ajuste <code>{{setting}}</code>: {{-error}}", "question.error creating question": "Error mientras se genera la pregunta {{number}}: {{-message}}", "question.score feedback.not marked": "No anotado.", "question.score feedback.partially answered": "Respuesta parcial.", "question.score feedback.score total": "{{marksString}}", "question.score feedback.score actual": "Puntuaci\u00f3n: {{scoreString}}", "question.score feedback.score total actual": "Puntuaci\u00f3n: {{score,niceNumber}}/{{marks,niceNumber}}", "variable.error in variable definition": "Error en la definici\u00f3n de la variable <code>{{name}}</code>", "left brace": "{", "right brace": "}", "extension.not found": "No se pudo cargar la extensi\u00f3n <code>{{name}}</code>.", "control.toggle navigation menu": "Toggle the navigation menu", "part.input title": "Respuesta para parte {{name}}", "part.correct answer title": "Respuesta esperada para parte {{name}}", "part.jme.must-match.failed": "Su respuesta no est\u00e1 escrita en la forma correcta", "question.score feedback.none": "", "control.submit part.confirm remove next parts": "<p>One or more subsequent parts depend on your answer to this part. Submitting this part again will invalidate those parts, and remove them from the question. This cannot be undone.</p>\n<p>Would you like to submit this part again?</p>", "control.back to menu": "Volver al men\u00fa", "display.error making html": "Error making HTML in {{contextDescription}}: {{-message}}", "jme.subvars.error compiling": "{{-message}} in <code>{{expression}}</code>", "jme.variables.empty name": "Una variable de la pregunta no ha sido nombrada.", "jme.calculus.unknown derivative": "No s\u00e9 diferenciar <code>{{tree}}</code>", "math.order complex numbers": "No puede ordenar n\u00fameros complejos", "menu.choose a question": "Escoja una pregunta", "part.choose next part.answered": "\u00bfQu\u00e9 quiere hacer ahora?", "part.choose next part.unanswered": "O podr\u00eda:", "part.choose next part.will be locked": "(Esta parte estar\u00e1 bloqueada)", "part.reached dead end": "No hay m\u00e1s por hacer.", "part.next part.penalty amount": "(lose {{count}} $t(mark))", "part.marking.counts towards objective": "This part counts towards the objective <strong>\u201c{{objective}}\u201d</strong>.", "part.numberentry.answer not integer or decimal or fraction": "Su pregunta es inv\u00e1lida. Debe ingresar un entero, un decimal o una fracci\u00f3n.", "question": "Pregunta", "question.progress": "Progreso de la pregunta:", "question.score feedback.unattempted": "No resuelta", "question.score feedback.attempted": "Resuelta", "question.score feedback.score actual.plain": "{{scoreString}}", "question.score feedback.score total actual.plain": "{{score,niceNumber}}/{{marks,niceNumber}}", "question.objectives": "Objetivos", "question.penalties": "Penalties", "question.back to previous part": "Retorne a la parte previa", "end.print": "Imprima la transcripci\u00f3n de su examen", "math.shuffle_together.lists not all the same length": "No todas las listas tienen la misma longitud.", "jme.parse signature.invalid signature string": "Invalid function signature string: {{str}}", "part.custom.expected answer has wrong type": "La respuesta esperada para esta parte es de un tipo incorrecto. Deber\u00eda ser <code>{{shouldbe}}</code>.", "part.custom.input option has wrong type": "The answer input setting <code>{{option}}</code> has the wrong type. It should be <code>{{shouldbe}}</code>.", "matrix input.size control legend": "Dimensiones", "matrix input.rows": "Filas", "matrix input.columns": "Columnas", "part.jme.error checking numerically": "There was an error numerically checking your answer: {{-message}}", "part.gapfill.cyclic adaptive marking": "There is a cycle in the adaptive marking for this part: <strong>{{name1}}</strong> relies on <strong>{{name2}}</strong>, which eventually relies on <strong>{{name1}}</strong>.", "modal.style.background colour": "Color de fondo", "modal.style.text colour": "Color de texto", "modal.style.text size": "Tama\u00f1o de texto", "modal.style.explanation": "Use estos controles para cambiar la apariencia del examen.", "modal.style.reset to defaults": "Volver a los valores por defecto", "modal.style.text size preview": "La mayor\u00eda del texto tendr\u00e1 este tama\u00f1o.", "control.style options": "Opciones de visualizaci\u00f3n", "part.marking.partially correct": "Tu respuesta es parcialmente correcta.", "part.marking.error in adaptive marking": "There was an error in the adaptive marking for this part. Please report this. {{-message}}", "page.skip to content": "Ir al contenido", "result.learning objective": "Objetivo del aprendizaje", "jme.interpreted as": "Interpretado como", "jme.script.note.compilation error": "Error compiling note <code>{{name}}</code>: {{-message}}", "jme.script.note.error evaluating note": "Error evaluating note <code>{{name}}</code> - {{-message}}", "jme.script.note.invalid definition": "Invalid note definition: <code>{{source}}</code>. {{-hint}}", "jme.script.note.invalid definition.missing colon": "You might be missing a colon after the name and description", "jme.script.note.invalid definition.description missing closing bracket": "You might be missing a closing bracket", "jme.script.note.empty expression": "The note <code>{{name}}</code> is empty.", "jme.script.error parsing notes": "Error parsing marking script: {{- message}}", "matrix input.cell label": "Row {{row}}, column {{column}}", "control.move to next question": "Move to the next question", "diagnostic.use retry": "Use one retry and try this topic again.", "diagnostic.move to next topic": "Ir al siguiente tema.", "diagnostic.next step question": "What would you like to do next?", "diagnostic.now assessing topic": "Now assessing {{current_topic}}", "diagnostic.one retry left": "Te queda 1 intento", "diagnostic.retries left": "You have {{retries}} retries left.", "diagnostic.percentage completed": "You've completed <strong>{{percentage}}%</strong> of the test.", "diagnostic.test is over": "La prueba ha terminado.", "diagnostic.passed all lo": "You have passed all learning objectives.", "diagnostic.more work on lo": "You need to do some more work on the following learning objectives: {{los}}.", "diagnostic.move to next question in topic": "Move on to the next question in topic.", "diagnostic.complete": "Complete!", "diagnostic.studying topic": "Studying {{topic}}", "display.answer widget.unknown widget type": "The answer widget type <code>{{name}}</code> is not recognised.", "jme.shunt.expected argument before comma": "Expected to see something between the opening bracket and the comma", "part.waiting for pre submit": "Your answer is being marked. Please wait.", "diagnostic.end test": "End the test.", "page.no stylesheet loaded": "The page's stylesheet file has not loaded.", "modal.confirm": "Confirm", "modal.alert": "Alert", "suspend.resumed header": "Attempt resumed", "jme.vector.value not an array of numbers": "Tried to construct a vector using a value that is not an array of numbers.", "jme.matrix.value not the right type": "Tried to construct a vector using a value of the wrong type.", "jme.subvars.html inserted twice": "An HTML value has been embedded twice. Consider defining a function to generate a new value each time it is used.", "jme.variables.invalid function language": "The language <code>{{language}}</code> is not valid.", "jme.variables.duplicate definition": "There is more than one definition of the variable <code>{{name}}</code>.", "math.random_integer_partition.invalid k": "The size of the partition must be between 1 and {{n}}.", "part.marking.parameter already in scope": "There is a variable named <code>{{name}}</code>, which is also the name of a marking parameter. Please rename the variable.", "part.marking.adaptive variable replacement refers to self": "This part refers to itself in a variable replacement for adaptive marking.", "part.marking.adaptive variable replacement refers to nothing": "This part contains an invalid variable replacement for adaptive marking.", "part.numberentry.display answer wrong type": "The display answer for this part is a value of type <code>{{got_type}}</code>, but should be a <code>{{want_type}}</code>.", "part.matrix.invalid type in prefilled": "There is an invalid value of type <code>{{n}}</code> in the array of pre-filled cells.", "diagnostic.make a choice": "Toma una decisi\u00f3n", "matrixmath.not square": "This operation only works on a square matrix.", "matrixmath.not invertible": "This operation only works on an invertible matrix.", "matrixmath.gauss-jordan elimination.not enough columns": "There must be at least as many columns as rows.", "question.required extension not available": "This question requires the extension <code>{{-extension}}</code> but it is not available.", "util.formatNumberNotation.unrecognised syntax": "The number formatting syntax <code>{{syntax}}</code> is not recognised.", "worksheet.number of exams to generate": "Number of sheets to generate", "worksheet.starting with id": "Starting with ID", "worksheet.show exam id": "Show sheet ID?", "worksheet.page break between questions": "Page breaks between questions?", "worksheet.page margins": "Page margins (mm)", "worksheet.text size": "Text size (pt)", "worksheet.generate": "Generate", "worksheet.generating exams": "Generating sheets", "worksheet.sheet id": "Sheet ID:", "worksheet.print single": "Print this sheet", "worksheet.print several": "Print these sheets", "worksheet.answer sheets": "Answer sheets", "worksheet.question sheets": "Question sheets", "worksheet.reconfigure": "Generate different sheets", "worksheet.show sheet": "Preview the sheet with ID:", "accessibility statement": "Accessibility statement and guide to adapting Numbas to your needs.", "exam.enter your name": "Your name:", "exam.attempt download security warning": "This exam is configured to allow you to download your data, but it is not running in a secure browser context. You will not be able to download your exam data. Contact your lecturer or teacher for help.", "result.download exam object": "Download your exam data", "control.return to question": "Return to the question", "control.show introduction": "Introduction", "analysis.header": "Analyse attempt data", "analysis.help.upload files": "Upload attempt data files that your students have given you.", "analysis.help.file input label": "Choose student attempt data files, or drag files onto this window.", "analysis.table.total": "Exam totals", "analysis.table.question": "Exam and question totals", "analysis.table.all": "All details", "analysis.student name.anonymous": "No name given", "analysis.expected": "Expected results", "analysis.start time": "Start time", "analysis.maximum": "Maximum Marks", "analysis.file": "File", "analysis.download": "Download", "analysis.delete": "Delete", "analysis.view results": "View results", "analysis.upload files": "Upload files", "analysis.upload more": "Upload more files", "analysis.attempt data": "Attempt data", "analysis.select format": "Select data to show", "analysis.download this table": "Download this table", "analysis.student": "Student Results", "analysis.question key": "Question Key", "analysis.question name": "Question Name", "analysis.group": "Group", "analysis.question": "Question", "analysis.part": "Part", "analysis.gap": "Gap", "analysis.record type": "Record Type", "analysis.score": "Score", "analysis.marks available": "Marks Available", "analysis.percentage": "Percentage", "analysis.answer": "Answer", "analysis.student name": "Student Name", "analysis.summary.no files": "You have not uploaded any files yet.", "analysis.summary.no decrypted files": "You have not uploaded any valid files yet.", "analysis.summary.one file": "One attempt.", "analysis.summary.several files": "{{num_files,niceNumber}} attempts.", "analysis.not secure context": "This page must be opened in a secure browser context. A secure context is either a page served over HTTPS, or a file loaded from your device.", "jme.shunt.pipe right hand takes no arguments": "The expression on the right-hand side of the pipe operator must be a function application.", "question.explore.no parts defined": "There are no parts defined in this question.", "answer": "answer", "worksheet.answersheet show question content": "Show question content in answer sheets?", "modal.confirm end exam": "Write <code>{{endConfirmation}}</code> in the box to confirm:", "modal.end exam button": "End exam", "lightbox.zoom in on image": "Zoom in on this image", "exam.progress": "Progress", "exam.questions answered": "{{numAnsweredQuestions}} of {{numQuestions}} questions answered.", "result.question marks available": "Marks Available", "result.question answered": "Answered?", "control.confirm end.correct": "You may now end the exam.", "control.confirm end.incorrect": "This is not the expected text.", "control.confirm end.password": "end", "jme.typecheck.for in name wrong type": "The name in a <code>for</code> statement must be a name or list of names, not {{type}}.", "jme.makeFast.no fast definition of function": "The function <code>{{name}}</code> here isn't defined in a way that can be made fast.", "part.show feedback": "Show feedback", "part.hide feedback": "Hide feedback", "part.feedback title": "Feedback for {{name}}.", "part.jme.must-match.warning": "Your answer is not in the expected form: {{-message}}", "part.numberentry.write your answer as a fraction": "Write your answer as a fraction.", "question.nav.label": "Question controls", "question.answer saved": "Answer saved", "question.all answers saved": "All answers saved", "analysis.back to results": "Back to results", "analysis.review": "Review", "analysis.review this": "Review this attempt", "analysis.reviewing attempt": "Reviewing attempt by {{student_name}}.", "part.there is new feedback": "The feedback has changed.", "modal.style.colour scheme": "Colour scheme", "modal.style.automatic colour scheme": "Automatic", "modal.style.light colour scheme": "Light", "modal.style.dark colour scheme": "Dark", "modal.style.custom colour scheme": "Custom", "modal.style.main font": "Main font", "modal.style.font.sans serif": "Sans serif", "modal.style.font.serif": "Serif", "modal.style.font.monospace": "Monospace", "modal.style.spacing": "Spacing", "modal.style.font weight": "Font weight", "modal.style.forced colours warning": "Your browser has overridden the colours used in this page because of a setting such as high contrast mode. Changes to the colour scheme settings here might not have any effect.", "modal.style.text preview": "Most text will look like this.", "modal.style.more options": "More options", "modal.style.main colour": "Main (brand) colour", "modal.style.primary colour": "Primary button colour", "modal.style.link colour": "Link colour", "modal.style.success colour": "Success/correct colour", "modal.style.info colour": "Info colour", "modal.style.warning colour": "Warning colour", "modal.style.danger colour": "Danger/incorrect colour", "modal.style.muted colour": "Muted text colour", "modal.style.highlight colour": "Highlight colour", "exam.error loading exam definition": "There was an error while loading the exam definition: {{text}}", "exam.no exam definition": "No exam definition was given.", "jme.typecheck.wrong arguments for anonymous function": "Wrong number of arguments for this anonymous function.", "worksheet.top": "Top", "worksheet.left": "Left", "worksheet.bottom": "Bottom", "worksheet.right": "Right"}}, "he-il": {"translation": {"page.loading": "\u05d4\u05d3\u05e3 \u05d1\u05d8\u05e2\u05d9\u05e0\u05d4", "page.saving": "<p>\u05d4\u05d3\u05e3 \u05d1\u05e9\u05de\u05d9\u05e8\u05d4</p>\n<p>\u05d4\u05e4\u05e2\u05d5\u05dc\u05d4 \u05e2\u05e9\u05d5\u05d9\u05d4 \u05dc\u05e7\u05d7\u05ea \u05de\u05e1\u05e4\u05e8 \u05e9\u05e0\u05d9\u05d5\u05ea</p>", "mathjax.math processing error": "\"{{-message}}\" \u05db\u05d0\u05e9\u05e8 \u05de\u05e2\u05d1\u05d3 \u05d0\u05ea <code>{{expression}}</code>", "die.numbas failed": "\u05d4\u05d0\u05e4\u05dc\u05d9\u05e7\u05e6\u05d9\u05d4 \u05e0\u05db\u05e9\u05dc\u05d4", "die.sorry": "\u05d4\u05d0\u05e4\u05dc\u05d9\u05e7\u05e6\u05d9\u05d4 \u05e0\u05ea\u05e7\u05dc\u05d4 \u05d1\u05e9\u05d2\u05d9\u05d0\u05d4 \u05d5\u05dc\u05d0 \u05d9\u05db\u05d5\u05dc\u05d4 \u05dc\u05d4\u05de\u05e9\u05d9\u05da. \u05ea\u05d0\u05d5\u05e8 \u05d4\u05e9\u05d2\u05d9\u05d0\u05d4 \u05de\u05d5\u05e4\u05d9\u05e2 \u05de\u05d8\u05d4.", "die.error": "\u05e9\u05d2\u05d9\u05d0\u05d4", "modal.ok": "OK", "modal.cancel": "Cancel", "exam.exam name": "\u05e9\u05dd \u05d4\u05e4\u05e2\u05d9\u05dc\u05d5\u05ea:", "exam.random seed": "\u05de\u05e1\u05e4\u05e8 \u05de\u05e4\u05d2\u05e9:", "exam.student name": "\u05e9\u05dd \u05d4\u05ea\u05dc\u05de\u05d9\u05d3:", "exam.number of questions": "\u05de\u05e1\u05e4\u05e8 \u05d4\u05e9\u05d0\u05dc\u05d5\u05ea:", "exam.marks available": "\u05e6\u05d9\u05d5\u05e0\u05d9\u05dd:", "exam.pass percentage": "\u05d0\u05d7\u05d5\u05d6 \u05d4\u05e6\u05dc\u05d7\u05d4:", "exam.time allowed": "\u05d6\u05de\u05df \u05de\u05d5\u05e7\u05e6\u05d4:", "exam.passed": "\u05e2\u05d1\u05e8", "exam.failed": "\u05e0\u05db\u05e9\u05dc", "exam.review header": "\u05e1\u05e7\u05d9\u05e8\u05d4: ", "frontpage.start": "\u05d4\u05ea\u05d7\u05dc", "suspend.paused header": "\u05d4\u05d5\u05e4\u05e1\u05e7", "suspend.exam suspended": "\n\u05d4\u05de\u05d1\u05d3\u05e7 \u05d4\u05d5\u05e4\u05e1\u05e7. \u05dc\u05d7\u05e5 \u05e2\u05dc  <em>\u05d4\u05de\u05e9\u05da</em> \u05e2\u05dc \u05de\u05e0\u05ea \u05dc\u05d4\u05de\u05e9\u05d9\u05da.", "suspend.you can resume": "\u05ea\u05d5\u05db\u05dc \u05dc\u05d4\u05de\u05e9\u05d9\u05da \u05d0\u05ea \u05d4\u05de\u05e4\u05d2\u05e9 \u05d1\u05e4\u05e2\u05dd \u05d4\u05d1\u05d0\u05d4 \u05e9\u05ea\u05ea\u05d7\u05d9\u05dc \u05d0\u05ea \u05d4\u05e4\u05e2\u05d9\u05dc\u05d5\u05ea \u05d4\u05d6\u05d5.", "suspend.resume": "\u05d4\u05de\u05e9\u05da", "result.exit": "\u05e1\u05d9\u05d9\u05dd \u05e4\u05e2\u05d9\u05dc\u05d5\u05ea", "result.print": "\u05d4\u05d3\u05e4\u05e1 \u05d0\u05ea \u05d3\u05e3 \u05d4\u05ea\u05d5\u05e6\u05d0\u05d5\u05ea", "result.exam summary": "\u05d3\u05e3 \u05ea\u05d5\u05e6\u05d0\u05d5\u05ea", "result.performance summary": "\u05e1\u05d9\u05db\u05d5\u05dd \u05d1\u05d9\u05e6\u05d5\u05e2 \u05d4\u05e4\u05e2\u05d9\u05dc\u05d5\u05ea", "result.exam start": "\u05d4\u05ea\u05d7\u05dc\u05ea \u05e4\u05e2\u05d9\u05dc\u05d5\u05ea", "result.exam stop": "\u05e1\u05d9\u05d5\u05dd \u05e4\u05e2\u05d9\u05dc\u05d5\u05ea:", "result.time spent": "\u05d6\u05de\u05df \u05d4\u05e4\u05e2\u05d9\u05dc\u05d5\u05ea:", "result.questions attempted": "\u05e9\u05d0\u05dc\u05d5\u05ea \u05e9\u05e0\u05d9\u05e1\u05d9\u05ea:", "result.score": "\u05e6\u05d9\u05d5\u05df:", "result.result": "\u05ea\u05d5\u05e6\u05d0\u05d5\u05ea:", "result.question number": "\u05de\u05e1\u05e4\u05e8 \u05e9\u05d0\u05dc\u05d4:", "result.question score": "\u05e6\u05d9\u05d5\u05df", "result.question review title": "\u05e6\u05e4\u05d4 \u05d1\u05e9\u05d0\u05dc\u05d4", "result.click a question to review": "\u05d4\u05e7\u05dc\u05e7 \u05e2\u05dc \u05de\u05e1\u05e4\u05e8 \u05d4\u05e9\u05d0\u05dc\u05d4 \u05db\u05d3\u05d9 \u05dc\u05e8\u05d0\u05d5\u05ea \u05d0\u05ea \u05d4\u05e6\u05d9\u05d5\u05df \u05dc\u05ea\u05e9\u05d5\u05d1\u05d5\u05ea \u05e9\u05e0\u05ea\u05ea \u05d5\u05d0\u05ea \u05d4\u05e4\u05ea\u05e8\u05d5\u05df \u05d4\u05de\u05dc\u05d0 (\u05d1\u05de\u05d9\u05d3\u05d4 \u05d5\u05e7\u05d9\u05d9\u05dd).", "end.exam has finished": "\u05d4\u05e4\u05e2\u05d9\u05dc\u05d5\u05ea \u05d4\u05e1\u05ea\u05d9\u05d9\u05de\u05d4. \u05d0\u05ea\u05d4 \u05d9\u05db\u05d5\u05dc \u05dc\u05e1\u05d2\u05d5\u05e8 \u05d0\u05ea \u05d4\u05d7\u05dc\u05d5\u05df.", "control.confirm leave": "\u05dc\u05d0 \u05e1\u05d9\u05d9\u05de\u05ea \u05d0\u05ea \u05d4\u05e4\u05e2\u05d9\u05dc\u05d5\u05ea.", "control.not all questions answered": "\u05dc\u05d0 \u05e1\u05d9\u05d9\u05de\u05ea \u05d0\u05ea \u05db\u05dc \u05d4\u05e9\u05d0\u05dc\u05d5\u05ea \u05d1\u05e4\u05e2\u05d9\u05dc\u05d5\u05ea.", "control.not all questions submitted": "\u05d1\u05d9\u05e6\u05e2\u05ea \u05e9\u05d9\u05e0\u05d5\u05d9\u05d9\u05dd \u05dc\u05ea\u05e9\u05d5\u05d1\u05d4 \u05d0\u05d7\u05ea \u05d0\u05d5 \u05d9\u05d5\u05ea\u05e8, \u05d0\u05d1\u05dc \u05dc\u05d0 \u05d4\u05d2\u05e9\u05ea \u05d0\u05d5\u05ea\u05dd. \u05d0\u05e0\u05d0 \u05d5\u05d5\u05d3\u05d0 \u05e9\u05db\u05dc \u05ea\u05e9\u05d5\u05d1\u05d5\u05ea\u05d9\u05da \u05d4\u05d5\u05d2\u05e9\u05d5.", "control.confirm end": "\u05d4\u05d0\u05dd \u05d0\u05ea\u05d4 \u05d1\u05d8\u05d5\u05d7 \u05e9\u05d1\u05e8\u05e6\u05d5\u05e0\u05da \u05dc\u05e1\u05d9\u05d9\u05dd \u05d0\u05ea \u05d4\u05e4\u05e2\u05d9\u05dc\u05d5\u05ea? \u05d0\u05d7\u05e8\u05d9 \u05e1\u05d9\u05d5\u05dd \u05d4\u05e4\u05e2\u05d9\u05dc\u05d5\u05ea, \u05dc\u05d0 \u05ea\u05d5\u05db\u05dc \u05dc\u05e9\u05e0\u05d5\u05ea \u05d0\u05ea \u05ea\u05e9\u05d5\u05d1\u05d5\u05ea\u05d9\u05da. ", "control.confirm regen": "\u05d4\u05d0\u05dd \u05d0\u05ea\u05d4 \u05de\u05e2\u05d5\u05e0\u05d9\u05d9\u05df \u05dc\u05d1\u05d7\u05d5\u05e8 \u05e2\u05e8\u05db\u05d9\u05dd \u05d7\u05d3\u05e9\u05d9\u05dd \u05dc\u05e9\u05d0\u05dc\u05d4? \u05d0\u05dd \u05db\u05df, \u05d4\u05ea\u05e9\u05d5\u05d1\u05d5\u05ea \u05d5\u05d4\u05e6\u05d9\u05d5\u05e0\u05d9\u05dd \u05e9\u05dc \u05e9\u05d0\u05dc\u05d4 \u05d6\u05d5 \u05d9\u05de\u05d7\u05e7\u05d5.", "control.confirm reveal": "\u05d4\u05d0\u05dd \u05d0\u05ea\u05d4 \u05de\u05e2\u05d5\u05e0\u05d9\u05d9\u05df \u05dc\u05e8\u05d0\u05d5\u05ea \u05d0\u05ea \u05d4\u05e4\u05ea\u05e8\u05d5\u05df \u05e9\u05dc \u05d4\u05e9\u05d0\u05dc\u05d4? \u05d0\u05dd \u05db\u05df, \u05d4\u05ea\u05e9\u05d5\u05d1\u05d5\u05ea \u05d5\u05d4\u05e6\u05d9\u05d5\u05e0\u05d9\u05dd \u05e9\u05dc \u05e9\u05d0\u05dc\u05d4 \u05d6\u05d5 \u05d9\u05e0\u05e2\u05dc\u05d5 \u05d5\u05dc\u05d0 \u05ea\u05d5\u05db\u05dc \u05dc\u05e2\u05e0\u05d5\u05ea \u05e2\u05dc\u05d9\u05d4 \u05e9\u05d5\u05d1.", "control.proceed anyway": "\u05dc\u05d4\u05de\u05e9\u05d9\u05da?", "control.regen": "\u05d1\u05d7\u05e8 \u05e2\u05e8\u05db\u05d9\u05dd \u05d7\u05d3\u05e9\u05d9\u05dd \u05dc\u05e9\u05d0\u05dc\u05d4 \u05d6\u05d5", "control.submit answer": "\u05d4\u05d2\u05e9 \u05d0\u05ea \u05d4\u05ea\u05e9\u05d5\u05d1\u05d4", "control.submit all parts": "\u05d4\u05d2\u05e9 \u05d0\u05ea \u05db\u05dc \u05d7\u05dc\u05e7\u05d9 \u05d4\u05ea\u05e9\u05d5\u05d1\u05d4", "control.submit again": "\u05d4\u05d2\u05e9 \u05de\u05d7\u05d3\u05e9", "control.submit": "\u05d4\u05d2\u05e9", "control.previous": "\u05d4\u05e7\u05d5\u05d3\u05dd", "control.next": "\u05d4\u05d1\u05d0", "control.advice": "\u05e8\u05de\u05d6", "control.reveal": "\u05d2\u05dc\u05d4 \u05d0\u05ea \u05d4\u05ea\u05e9\u05d5\u05d1\u05d5\u05ea", "control.total": "\u05e1\u05d9\u05db\u05d5\u05dd", "control.pause": "\u05d4\u05e4\u05e1\u05e7", "control.end exam": "\u05e1\u05d9\u05d9\u05dd \u05e4\u05e2\u05d9\u05dc\u05d5\u05ea", "control.back to results": "\u05d7\u05d6\u05d5\u05e8 \u05dc\u05ea\u05d5\u05e6\u05d0\u05d5\u05ea", "display.part.jme.error making maths": "\u05e9\u05d2\u05d9\u05d0\u05d4 \u05d1\u05e2\u05ea \u05d4\u05e6\u05d2\u05ea \u05e0\u05ea\u05d5\u05e0\u05d9\u05dd", "exam.xml.bad root": "Root element of exam XML should be 'exam'", "exam.changeQuestion.no questions": "\u05d1\u05e4\u05e2\u05d9\u05dc\u05d5\u05ea \u05d6\u05d5 \u05d0\u05d9\u05df \u05e9\u05d0\u05dc\u05d5\u05ea! (\u05d7\u05e4\u05e9 \u05e9\u05d2\u05d9\u05d0\u05d5\u05ea \u05d1\u05e7\u05d5\u05d1\u05e5 exam)", "feedback.you were awarded": "\n\u05e7\u05d9\u05d1\u05dc\u05ea <strong>{{count,niceNumber}}</strong> $t(mark).", "feedback.taken away": "\n<strong>{{count,niceNumber}}</strong> $t(mark) $t(was) \u05e0\u05dc\u05e7\u05d7\u05d5.", "jme.shunt.not enough arguments": "Not enough arguments for operation <code>{{op}}</code>", "jme.shunt.no left bracket in function": "No matching left bracket in function application or tuple", "jme.shunt.no left square bracket": "No matching left bracket", "jme.shunt.no left bracket": "No matching left bracket", "jme.shunt.no right bracket": "No matching right bracket", "jme.shunt.no right square bracket": "No matching right square bracket to end list", "jme.shunt.missing operator": "Expression can't be evaluated -- missing an operator.", "jme.typecheck.function maybe implicit multiplication": "Function <code>{{name}}</code> is not defined. Did you mean <code>{{first}}*{{possibleOp}}(...)</code>?", "jme.typecheck.function not defined": "Function <code>{{op}}</code> is not defined. Is <code>{{op}}</code> a variable, and did you mean <code>{{suggestion}}*(...)</code>?", "jme.typecheck.op not defined": "Operation '{{op}}' is not defined.", "jme.typecheck.no right type definition": "No definition of '{{op}}' of correct type found.", "jme.typecheck.no right type unbound name": "Variable <code>{{name}}</code> is not defined.", "jme.typecheck.map not on enumerable": "<code>map</code> operation must work over a list or a range, not {{type}}", "jme.evaluate.undefined variable": "Variable {{name}} is undefined", "jme.thtml.not html": "Passed a non-HTML value into the THTML constructor.", "jme.func.switch.no default case": "No default case for Switch statement", "jme.func.listval.invalid index": "Invalid list index {{index}} on list of size {{size}}", "jme.func.listval.not a list": "Object is not subscriptable", "jme.func.matrix.invalid row type": "Can't construct a matrix from rows of type {{type}}", "jme.func.except.continuous range": "Can't use the 'except' operator on continuous ranges.", "jme.matrix.reports bad size": "Matrix reports its size incorrectly - must be an error in constructor function", "jme.texsubvars.no right bracket": "No matching <code>]</code> in {{op}} arguments.", "jme.texsubvars.missing parameter": "Missing parameter in {{op}}: {{parameter}}", "jme.texsubvars.no right brace": "No matching <code>}</code> in {{op}}", "jme.user javascript.error": "Error in user-defined javascript function <code>{{name}}</code>: {{-message}}", "jme.variables.error making function": "Error making function <code>{{name}}</code>: {{-message}}", "jme.variables.syntax error in function definition": "Syntax error in function definition", "jme.variables.variable not defined": "Variable <code>{{name}}</code> is not defined.", "jme.variables.empty definition": "Definition of variable <code>{{name}}</code> is empty.", "jme.variables.circular reference": "Circular variable reference in definition of <code>{{name}}</code>", "jme.variables.error computing dependency": "Error computing referenced variable <code>{{name}}</code>", "jme.variables.error evaluating variable": "Error evaluating variable {{name}}: {{-message}}", "jme.variables.question took too many runs to generate variables": "A valid set of question variables was not generated in time.", "jme.display.unknown token type": "Can't texify token type {{type}}", "jme.display.collectRuleset.no sets": "No sets given to collectRuleset!", "jme.display.collectRuleset.set not defined": "Ruleset {{name}} has not been defined", "jme.display.simplifyTree.no scope given": "Numbas.jme.display.simplifyTree must be given a Scope", "math.precround.complex": "Can't round to a complex number of decimal places", "math.siground.complex": "Can't round to a complex number of sig figs", "math.combinations.complex": "Can't compute combinations of complex numbers", "math.permutations.complex": "Can't compute permutations of complex numbers", "math.gcf.complex": "Can't compute GCF of complex numbers", "math.lcm.complex": "Can't compute LCM of complex numbers", "math.lt.order complex numbers": "Can't order complex numbers", "math.choose.empty selection": "Empty selection given to random function", "matrixmath.abs.non-square": "Can't compute the determinant of a matrix which isn't square.", "matrixmath.abs.too big": "Sorry, can't compute the determinant of a matrix bigger than 3x3 yet.", "matrixmath.mul.different sizes": "Can't multiply matrices of different sizes.", "vectormath.cross.not 3d": "Can only take the cross product of 3-dimensional vectors.", "vectormath.dot.matrix too big": "Can't calculate dot product of a matrix which isn't $1 \\times N$ or $N \\times 1$.", "vectormath.cross.matrix too big": "Can't calculate cross product of a matrix which isn't $1 \\times N$ or $N \\times 1$.", "part.with steps answer prompt": "Answer: ", "part.script.error": "Error in part {{path}} custom script {{script}}: {{-message}}", "part.marking.steps no matter": "Because you received full marks for the part, your answers to the steps aren't counted.", "part.marking.used variable replacements": "This part was marked using your answers to previous parts.", "part.marking.variable replacement part not answered": "You must answer {{part}} first.", "part.marking.resubmit because of variable replacement": "This part's marking depends on your answers to other parts, which you have changed. Save your answer to this part again to update your score.", "part.marking.not submitted": "No answer submitted.", "part.marking.did not answer": "You did not answer this question.", "part.marking.nothing entered": "You did not enter an answer.", "part.marking.incorrect": "Your answer is incorrect.", "part.marking.correct": "Your answer is correct.", "part.marking.uncaught error": "Error when marking: {{-message}}", "part.correct answer": "Expected answer:", "part.missing type attribute": "{{part}}: Missing part type attribute", "part.unknown type": "{{part}}: Unrecognised part type {{type}}", "part.setting not present": "Property '{{property}}' not set", "part.jme.answer missing": "Correct answer is missing", "part.jme.answer too long": "Your answer is too long.", "part.jme.answer too short": "Your answer is too short.", "part.jme.answer invalid": "Your answer is not a valid mathematical expression.<br/>{{-message}}.", "part.jme.marking.correct": "Your answer is numerically correct.", "part.jme.must-have bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.must-have one": "Your answer must contain: {{strings}}", "part.jme.must-have several": "Your answer must contain all of: {{strings}}", "part.jme.not-allowed bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.not-allowed one": "Your answer must not contain: {{strings}}", "part.jme.not-allowed several": "Your answer must not contain any of: {{strings}}", "part.jme.unexpected variable name": "Your answer was interpreted to use the unexpected variable name <code>{{name}}</code>.", "part.jme.unexpected variable name suggestion": "Your answer was interpreted to use the unexpected variable name <code>{{name}}</code>. Did you mean <code>{{suggestion}}</code>?", "part.patternmatch.display answer missing": "Display answer is missing", "part.patternmatch.correct except case": "Your answer is correct, except for the case.", "part.numberentry.correct except decimal": "Your answer is within the allowed range, but decimal numbers are not allowed.", "part.numberentry.correct except fraction": "Your answer is within the allowed range, but fractions are not allowed.", "part.numberentry.answer invalid": "You did not enter a valid number.", "part.numberentry.answer not integer": "Your answer is invalid. You must enter a whole number, not a decimal.", "part.numberentry.answer not integer or decimal": "Your answer is invalid. You must enter an integer or a decimal.", "part.numberentry.zero sig fig": "This part is set up to round the student's answer to zero significant figures, which has no meaning.", "part.mcq.options def not a list": "The expression defining the {{properties}} is not a list.", "part.mcq.marking matrix string empty": "The custom marking matrix expression is empty.", "part.mcq.choices missing": "Definition of choices is missing", "part.mcq.matrix not a number": "Part {{part}} marking matrix cell ({{row}},{{column}}) does not evaluate to a number", "part.mcq.wrong number of choices": "You selected the wrong number of choices.", "part.mcq.no choices selected": "No choices selected.", "part.mcq.matrix not a list": "Marking matrix, defined by JME expression, is not a list but it should be.", "part.mcq.matrix wrong type": "Element of invalid type '{{type}}' used in marking matrix.", "part.mcq.matrix mix of numbers and lists": "Mix of numbers and lists used in marking matrix.", "part.mcq.matrix wrong size": "Marking matrix is the wrong size.", "part.mcq.correct choice": "You chose a correct answer.", "part.matrix.answer invalid": "Your answer is not valid.", "part.matrix.invalid cell": "One or more of the cells in your answer is empty or invalid.", "part.matrix.some incorrect": "One or more of the cells in your answer is incorrect, but you have been awarded marks for the rest.", "part.matrix.empty": "You have not entered an answer.", "part.matrix.empty cell": "One or more of the cells in your answer is empty.", "part.matrix.size mismatch": "The question author hasn't allowed the student to decide the dimensions of their answer, but the correct answer is {{correct_dimensions}} while the answer input is {{input_dimensions}}", "part.gapfill.feedback header": "<strong>{{name}}</strong>", "part.extension.not implemented": "Part hasn't implemented the <code>{{name}}</code> method.", "question.loaded name mismatch": "Can't resume this attempt - the package has changed since the last session.", "question.error": "Question {{number}}: {{-message}}", "question.preamble.error": "Error in preamble: {{-message}}", "question.preamble.syntax error": "Syntax error in preamble", "question.unsupported part type": "Unsupported part type", "question.header": "Question {{number}}", "question.submit part": "Save answer", "question.show steps": "Show steps", "question.show steps penalty": "You will lose <strong>{{count,niceNumber}}</strong> $t(mark).", "question.show steps no penalty": "Your score will not be affected.", "question.show steps already penalised": "You have already shown steps. You can show them again with no further penalty.", "question.hide steps": "Hide steps", "question.hide steps no penalty": "Your score will not be affected.", "question.advice": "Advice", "question.no such part": "Can't find part {{path}}.", "question.can not submit": "Can not submit answer - check for errors.", "question.answer submitted": "Answer submitted", "question.score feedback.show": "Show feedback", "question.score feedback.hide": "Hide feedback", "question.score feedback.answered": "Answered", "question.score feedback.unanswered": "Unanswered", "question.score feedback.correct": "Your answer is correct", "question.score feedback.partial": "Your answer is partially correct", "question.score feedback.wrong": "Your answer is incorrect", "question.selector.unsubmitted changes": "Unsubmitted changes.", "timing.no accumulator": "no timing accumulator {{name}}", "timing.time remaining": "\u05d4\u05d6\u05de\u05df \u05d4\u05e0\u05d5\u05ea\u05e8:", "xml.could not load": "Couldn't load an XML document: {{-message}}", "xml.property not number": "Property {{name}} should be a number, but isn't ({{value}}), in node {{element}}", "xml.property not boolean": "Property {{name}} should be a boolean, but isn't ({{value}}), in node {{element}}", "scorm.error initialising": "Error initialising SCORM protocol: {{-message}}", "scorm.failed save": "<p>The request to save data to the server failed. Press <b>OK</b> to try again.</p>\n<p>If you get this message repeatedly, check your internet connection or use a different computer. Your previously submitted answers have been successfully saved and will be restored if you resume this session on a different computer.</p>\n<p>If this message appears persistently and you can't save <em>any</em> answers, please contact your lecturer or teacher.</p>", "scorm.no exam suspend data": "Failed to resume: no exam suspend data.", "scorm.error loading suspend data": "Error loading suspend data: {{-message}}", "scorm.error loading question": "Error loading question {{number}}: {{-message}}", "scorm.no question suspend data": "No question suspend data", "scorm.error loading part": "Error loading part {{part}}: {{-message}}", "scorm.no part suspend data": "No part suspend data", "util.product.non list": "Passed a non-list to <code>Numbas.util.product</code>", "mark": "mark", "was": "\u05d4\u05d9\u05d4", "part": "part", "gap": "\u05de\u05e8\u05d5\u05d5\u05d7", "step": "\u05e6\u05e2\u05d3", "jme.substituteTree.undefined variable": "Undefined variable: <code>{{name}}</code>", "jme.user javascript.returned undefined": "User-defined javascript function <code>{{name}}</code> returned <code>undefined</code>.", "part.marking.steps change": "You were awarded <strong>{{count,niceNumber}}</strong> $t(mark) for your answers to the steps.", "part.marking.total score": "You scored <strong>{{count,niceNumber}}</strong> $t(mark) for this part.", "part.numberentry.precision type.dp": "decimal place", "part.numberentry.precision type.dp_plural": "decimal places", "part.numberentry.precision type.sigfig": "significant figure", "part.numberentry.precision type.sigfig_plural": "significant figures", "part.numberentry.give your answer to precision": "Round your answer to {{count,niceNumber}} {{precisionType}}.", "question.unsubmitted changes": "You have made a change to your answer but not submitted it. Please check your answer and then press the <strong>Save answer</strong> button.", "question.unsubmitted changes_plural": "You have made changes to your answers but not submitted them. Please check your answers to each part and then press the <strong>Save all answers</strong> button.", "util.equality not defined for type": "Equality not defined for type {{type}}", "mark_plural": "marks", "was_plural": "\u05d4\u05d9\u05d5", "die.script not loaded": "\u05d4\u05d0\u05e4\u05dc\u05d9\u05e7\u05e6\u05d9\u05d4 \u05dc\u05d0 \u05d9\u05db\u05d5\u05dc\u05d4 \u05dc\u05d4\u05ea\u05d7\u05d9\u05dc \u05db\u05d9\u05d5\u05d5\u05df \u05e9\u05d4\u05e7\u05d5\u05d1\u05e5 <code>{{file}}</code> \u05dc\u05d0 \u05e0\u05d8\u05e2\u05df. \u05d1\u05d3\u05d5\u05e7 \u05e9\u05d4\u05e7\u05d5\u05d1\u05e5 \u05e0\u05db\u05dc\u05dc \u05d1 <code>scripts.js</code>.", "math.combinations.n less than zero": "Can't compute combinations: n is less than zero", "math.combinations.k less than zero": "Can't compute combinations: k is less than zero", "math.combinations.n less than k": "Can't compute combinations: n is less than k", "math.permutations.n less than zero": "Can't compute permutations: n is less than zero", "math.permutations.k less than zero": "Can't compute permutations: k is less than zero", "math.permutations.n less than k": "Can't compute permutations: n is less than k", "part.numberentry.give your answer to precision_0": "Round your answer to the nearest integer.", "mathjax.error": "MathJax processing error: {{-message}}", "mathjax.error with context": "MathJax processing error in {{-context}}: {{-message}}", "exam.introduction": "\u05d4\u05e7\u05d3\u05de\u05d4 \u05dc\u05e4\u05e2\u05d9\u05dc\u05d5\u05ea", "exam.feedback": "\u05de\u05e9\u05d5\u05d1 \u05dc\u05e4\u05e2\u05d9\u05dc\u05d5\u05ea", "jme.tokenise.keypair key not a string": "Dictionary key should be a string, not {{type}}.", "jme.shunt.list mixed argument types": "Can't parse {{mode}}: mix of dictionary and list elements", "jme.func.listval.key not in dict": "Dictionary does not contain the key <code>{{key}}</code>", "part.prompt": "prompt", "part.feedback": "Feedback", "part.numberentry.answer not reduced": "Your answer is not reduced to lowest terms.", "part.numberentry.give your answer as a reduced fraction": "Reduce your answer to lowest terms.", "part.numberentry.negative decimal places": "This part is set up to round the student's answer to a negative number of decimal places, which has no meaning.", "part.mcq.choices": "choices", "part.mcq.answers": "answers", "part.mcq.matrix cell empty": "Part {{part}} marking matrix cell ({{row}},{{column}}) is empty", "part.mcq.matrix jme error": "Part {{part}} marking matrix cell ({{row}},{{column}}) gives a JME error: {{-error}}", "question.statement": "Statement", "ruleset.circular reference": "Circular reference in definition of ruleset <code>{{name}}</code>", "ruleset.set not defined": "Ruleset {{name}} has not been defined", "jme.evaluate.no scope given": "Numbas.jme.evaluate must be given a Scope", "answer.number.not a number": "\u05ea\u05e9\u05d5\u05d1\u05ea\u05da \u05d0\u05d9\u05e0\u05d4 \u05de\u05e1\u05e4\u05e8 \u05d7\u05d5\u05e7\u05d9.", "answer.number.fractions not allowed": "\u05d0\u05d9\u05e0\u05da \u05e8\u05e9\u05d0\u05d9 \u05dc\u05d4\u05e7\u05dc\u05d9\u05d3 \u05e9\u05d1\u05e8.", "answer.jme.invalid expression": "{{-message}}", "answer.matrix.fractions not allowed": "\u05d0\u05d9\u05e0\u05da \u05e8\u05e9\u05d0\u05d9 \u05dc\u05d4\u05e7\u05dc\u05d9\u05d3 \u05e9\u05d1\u05e8\u05d9\u05dd.", "answer.matrix.some cell not a number": "\u05d0\u05d7\u05d3 \u05d0\u05d5 \u05d9\u05d5\u05ea\u05e8 \u05de\u05d4\u05ea\u05d0\u05d9\u05dd \u05d1\u05ea\u05e9\u05d5\u05d1\u05ea\u05da \u05d0\u05d9\u05e0\u05d5 \u05de\u05e1\u05e4\u05e8 \u05d7\u05d5\u05e7\u05d9.", "exam.enter password": "\u05e1\u05d9\u05e1\u05de\u05d4:", "exam.password.correct": "\u05d4\u05e1\u05d9\u05e1\u05de\u05d4 \u05e0\u05db\u05d5\u05e0\u05d4. \u05d0\u05ea\u05d4 \u05e8\u05e9\u05d0\u05d9 \u05dc\u05d4\u05ea\u05d7\u05d9\u05dc \u05d1\u05e4\u05e2\u05d9\u05dc\u05d5\u05ea.", "exam.password.incorrect": "\u05d4\u05e1\u05d9\u05e1\u05de\u05d4 \u05e9\u05d2\u05d5\u05d9\u05d4.", "frontpage.scorm.lms not connected": "\u05e4\u05e2\u05d9\u05dc\u05d5\u05ea \u05d6\u05d5 \u05dc\u05dc\u05d0 \u05d7\u05d9\u05d1\u05d5\u05e8 \u05dc\u05e9\u05e8\u05ea. \u05d4\u05ea\u05e9\u05d5\u05d1\u05d5\u05ea \u05d5\u05d4\u05e6\u05d9\u05d5\u05e0\u05d9\u05dd \u05dc\u05d0 \u05d9\u05e9\u05de\u05e8\u05d5. ", "result.question review": "Review", "control.confirm regen no marks": "\u05d4\u05d0\u05dd \u05d0\u05ea\u05d4 \u05de\u05e2\u05d5\u05e0\u05d9\u05d9\u05df \u05dc\u05d1\u05d7\u05d5\u05e8 \u05e2\u05e8\u05db\u05d9\u05dd \u05d7\u05d3\u05e9\u05d9\u05dd \u05dc\u05e9\u05d0\u05dc\u05d4?", "control.confirm reveal no marks": "\u05d4\u05d0\u05dd \u05d0\u05ea\u05d4 \u05de\u05e2\u05d5\u05e0\u05d9\u05d9\u05df \u05dc\u05e8\u05d0\u05d5\u05ea \u05d0\u05ea \u05d4\u05e4\u05ea\u05e8\u05d5\u05df \u05e9\u05dc \u05d4\u05e9\u05d0\u05dc\u05d4?", "jme.tokenise.invalid near": "Invalid expression: <code>{{expression}}</code> at position {{position}} near <code>{{nearby}}</code>", "jme.tokenise.number.object not complex": "Invalid object passed into number constructor.", "jme.subvars.null substitution": "Empty variable substitution: <code>$t(left brace){{str}}$t(right brace)", "jme.type.type already registered": "The data type {{type}} has already been registered so can't be registered again.", "jme.type.no cast method": "Can't automatically convert from {{from}} to {{to}}.", "jme.display.simplifyTree.empty expression": "Expression is empty", "jme.display.simplifyTree.stuck in a loop": "Simplifier is stuck in a loop: <code>{{expr}}</code>", "math.niceNumber.undefined": "Was expecting a number, but got <code>undefined</code>", "math.rangeToList.zero step size": "Can't convert a range with step size zero to a list.", "part.error": "{{path}}: {{-message}}", "part.marking.revealed steps": "You revealed the steps.", "part.marking.maximum scaled down": "The maximum you can score for this part is <strong>{{count,niceNumber}}</strong> $t(mark). Your scores will be scaled down accordingly.", "part.marking.minimum score applied": "The minimum score for this part is <strong>{{score,niceNumber}}</strong>.", "part.marking.maximum score applied": "The maximum score for this part is <strong>{{score,niceNumber}}</strong>.", "part.marking.error in marking script": "There was an error in this part's marking algorithm. Please report this. {{-message}}", "part.marking.no result after replacement": "This part could not be marked using your answers to previous parts.", "part.marking.missing required note": "The marking algorithm does not define the note <code>{{note}}</code>", "marking.apply.not a list": "The first argument to <code>apply</code> must be a list, and isn't", "marking.apply marking script.script not found": "Marking script <code>{{name}}</code> not found", "marking.note.compilation error": "Error compiling note <code>{{name}}</code>: {{-message}}", "marking.note.error evaluating note": "Error evaluating note <code>{{name}}</code> - {{-message}}", "marking.note.invalid definition": "Invalid note definition: <code>{{source}}</code>. {{-hint}}", "marking.note.invalid definition.missing colon": "You might be missing a colon after the name and description", "marking.note.invalid definition.description missing closing bracket": "You might be missing a closing bracket", "marking.note.empty expression": "The note <code>{{name}}</code> is empty.", "marking.script.error parsing notes": "Error parsing marking script: {{- message}}", "part.feedback out of date": "This feedback is based on your last submitted answer. Save your changed answer to get updated feedback.", "part.jme.invalid value generator expression": "Invalid value generator expression for variable <code>{{name}}</code>: {{-message}}", "part.mcq.incorrect choice": "You chose an incorrect answer.", "part.matrix.not all cells same precision": "You have not given every cell in your answer to the same precision.", "part.gapfill.error marking gap": "Error marking {{name}}: {{-message}}", "part.custom.empty setting": "No value given.", "part.custom.unrecognised input type": "Unrecognised setting type <code>{{input_type}}</code>", "part.custom.error evaluating input option": "Error evaluating input option <code>{{option}}</code>: {{-error}}", "part.custom.input option missing": "Definition of input option <code>{{option}}</code> is missing.", "part.custom.error evaluating setting": "Error evaluating setting <code>{{setting}}</code>: {{-error}}", "question.error creating question": "Error while creating question {{number}}: {{-message}}", "question.score feedback.not marked": "Not marked", "question.score feedback.partially answered": "Partially answered", "question.score feedback.score total": "{{marksString}}", "question.score feedback.score actual": "Score: {{scoreString}}", "question.score feedback.score total actual": "Score: {{score,niceNumber}}/{{marks,niceNumber}}", "variable.error in variable definition": "Error in definition of variable <code>{{name}}</code>", "left brace": "{", "right brace": "}", "extension.not found": "Couldn't load the extension <code>{{name}}</code>.", "control.toggle navigation menu": "\u05d4\u05d7\u05dc\u05e3 \u05d0\u05ea \u05ea\u05e4\u05e8\u05d9\u05d8 \u05d4\u05e0\u05d9\u05d5\u05d5\u05d8", "part.input title": "Answer for part {{name}}", "part.correct answer title": "Expected answer for part {{name}}", "part.jme.must-match.failed": "Your answer is not in the expected form.", "question.score feedback.none": "", "control.submit part.confirm remove next parts": "<p>One or more subsequent parts depend on your answer to this part. Submitting this part again will invalidate those parts, and remove them from the question. This cannot be undone.</p>\n<p>Would you like to submit this part again?</p>", "control.back to menu": "Go back to the menu", "display.error making html": "Error making HTML in {{contextDescription}}: {{-message}}", "jme.subvars.error compiling": "{{-message}} in <code>{{expression}}</code>", "jme.variables.empty name": "A question variable has not been given a name.", "jme.calculus.unknown derivative": "Don't know how to differentiate <code>{{tree}}</code>", "math.order complex numbers": "Can't order complex numbers", "menu.choose a question": "Choose a question.", "part.choose next part.answered": "What do you want to do next?", "part.choose next part.unanswered": "Or, you could:", "part.choose next part.will be locked": "(This part will be locked)", "part.reached dead end": "There's nothing more to do from here.", "part.next part.penalty amount": "(lose {{count}} $t(mark))", "part.marking.counts towards objective": "This part counts towards the objective <strong>\u201c{{objective}}\u201d</strong>.", "part.numberentry.answer not integer or decimal or fraction": "Your answer is invalid. You must enter an integer, a decimal or a fraction.", "question": "Question", "question.progress": "Question progress", "question.score feedback.unattempted": "Not attempted", "question.score feedback.attempted": "Attempted", "question.score feedback.score actual.plain": "{{scoreString}}", "question.score feedback.score total actual.plain": "{{score,niceNumber}}/{{marks,niceNumber}}", "question.objectives": "Objectives", "question.penalties": "Penalties", "question.back to previous part": "Go back to the previous part", "end.print": "Print your exam transcript", "math.shuffle_together.lists not all the same length": "Not all lists are the same length.", "jme.parse signature.invalid signature string": "Invalid function signature string: {{str}}", "part.custom.expected answer has wrong type": "The expected answer for this part has the wrong type. It should be <code>{{shouldbe}}</code>.", "part.custom.input option has wrong type": "The answer input setting <code>{{option}}</code> has the wrong type. It should be <code>{{shouldbe}}</code>.", "matrix input.size control legend": "Size", "matrix input.rows": "Rows", "matrix input.columns": "Columns", "part.jme.error checking numerically": "There was an error numerically checking your answer: {{-message}}", "part.gapfill.cyclic adaptive marking": "There is a cycle in the adaptive marking for this part: <strong>{{name1}}</strong> relies on <strong>{{name2}}</strong>, which eventually relies on <strong>{{name1}}</strong>.", "modal.style.background colour": "Background colour", "modal.style.text colour": "Text colour", "modal.style.text size": "Text size", "modal.style.explanation": "Use these controls to change the appearance of the exam.", "modal.style.reset to defaults": "Reset to defaults", "modal.style.text size preview": "Most text will be this big.", "control.style options": "Display options", "part.marking.partially correct": "Your answer is partially correct.", "part.marking.error in adaptive marking": "There was an error in the adaptive marking for this part. Please report this. {{-message}}", "page.skip to content": "Skip to content", "result.learning objective": "Learning objective", "jme.interpreted as": "interpreted as", "jme.script.note.compilation error": "Error compiling note <code>{{name}}</code>: {{-message}}", "jme.script.note.error evaluating note": "Error evaluating note <code>{{name}}</code> - {{-message}}", "jme.script.note.invalid definition": "Invalid note definition: <code>{{source}}</code>. {{-hint}}", "jme.script.note.invalid definition.missing colon": "You might be missing a colon after the name and description", "jme.script.note.invalid definition.description missing closing bracket": "You might be missing a closing bracket", "jme.script.note.empty expression": "The note <code>{{name}}</code> is empty.", "jme.script.error parsing notes": "Error parsing marking script: {{- message}}", "matrix input.cell label": "Row {{row}}, column {{column}}", "control.move to next question": "Move to the next question", "diagnostic.use retry": "Use one retry and try this topic again.", "diagnostic.move to next topic": "Move on to the next topic.", "diagnostic.next step question": "What would you like to do next?", "diagnostic.now assessing topic": "Now assessing {{current_topic}}", "diagnostic.one retry left": "You have 1 retry left", "diagnostic.retries left": "You have {{retries}} retries left.", "diagnostic.percentage completed": "You've completed <strong>{{percentage}}%</strong> of the test.", "diagnostic.test is over": "The test is over.", "diagnostic.passed all lo": "You have passed all learning objectives.", "diagnostic.more work on lo": "You need to do some more work on the following learning objectives: {{los}}.", "diagnostic.move to next question in topic": "Move on to the next question in topic.", "diagnostic.complete": "Complete!", "diagnostic.studying topic": "Studying {{topic}}", "display.answer widget.unknown widget type": "The answer widget type <code>{{name}}</code> is not recognised.", "jme.shunt.expected argument before comma": "Expected to see something between the opening bracket and the comma", "part.waiting for pre submit": "Your answer is being marked. Please wait.", "diagnostic.end test": "End the test.", "page.no stylesheet loaded": "The page's stylesheet file has not loaded.", "modal.confirm": "Confirm", "modal.alert": "Alert", "suspend.resumed header": "Attempt resumed", "jme.vector.value not an array of numbers": "Tried to construct a vector using a value that is not an array of numbers.", "jme.matrix.value not the right type": "Tried to construct a vector using a value of the wrong type.", "jme.subvars.html inserted twice": "An HTML value has been embedded twice. Consider defining a function to generate a new value each time it is used.", "jme.variables.invalid function language": "The language <code>{{language}}</code> is not valid.", "jme.variables.duplicate definition": "There is more than one definition of the variable <code>{{name}}</code>.", "math.random_integer_partition.invalid k": "The size of the partition must be between 1 and {{n}}.", "part.marking.parameter already in scope": "There is a variable named <code>{{name}}</code>, which is also the name of a marking parameter. Please rename the variable.", "part.marking.adaptive variable replacement refers to self": "This part refers to itself in a variable replacement for adaptive marking.", "part.marking.adaptive variable replacement refers to nothing": "This part contains an invalid variable replacement for adaptive marking.", "part.numberentry.display answer wrong type": "The display answer for this part is a value of type <code>{{got_type}}</code>, but should be a <code>{{want_type}}</code>.", "part.matrix.invalid type in prefilled": "There is an invalid value of type <code>{{n}}</code> in the array of pre-filled cells.", "diagnostic.make a choice": "Make a choice", "matrixmath.not square": "This operation only works on a square matrix.", "matrixmath.not invertible": "This operation only works on an invertible matrix.", "matrixmath.gauss-jordan elimination.not enough columns": "There must be at least as many columns as rows.", "question.required extension not available": "This question requires the extension <code>{{-extension}}</code> but it is not available.", "util.formatNumberNotation.unrecognised syntax": "The number formatting syntax <code>{{syntax}}</code> is not recognised.", "worksheet.number of exams to generate": "Number of sheets to generate", "worksheet.starting with id": "Starting with ID", "worksheet.show exam id": "Show sheet ID?", "worksheet.page break between questions": "Page breaks between questions?", "worksheet.page margins": "Page margins (mm)", "worksheet.text size": "Text size (pt)", "worksheet.generate": "Generate", "worksheet.generating exams": "Generating sheets", "worksheet.sheet id": "Sheet ID:", "worksheet.print single": "Print this sheet", "worksheet.print several": "Print these sheets", "worksheet.answer sheets": "Answer sheets", "worksheet.question sheets": "Question sheets", "worksheet.reconfigure": "Generate different sheets", "worksheet.show sheet": "Preview the sheet with ID:", "accessibility statement": "Accessibility statement and guide to adapting Numbas to your needs.", "exam.enter your name": "Your name:", "exam.attempt download security warning": "This exam is configured to allow you to download your data, but it is not running in a secure browser context. You will not be able to download your exam data. Contact your lecturer or teacher for help.", "result.download exam object": "Download your exam data", "control.return to question": "Return to the question", "control.show introduction": "Introduction", "analysis.header": "Analyse attempt data", "analysis.help.upload files": "Upload attempt data files that your students have given you.", "analysis.help.file input label": "Choose student attempt data files, or drag files onto this window.", "analysis.table.total": "Exam totals", "analysis.table.question": "Exam and question totals", "analysis.table.all": "All details", "analysis.student name.anonymous": "No name given", "analysis.expected": "Expected results", "analysis.start time": "Start time", "analysis.maximum": "Maximum Marks", "analysis.file": "File", "analysis.download": "Download", "analysis.delete": "Delete", "analysis.view results": "View results", "analysis.upload files": "Upload files", "analysis.upload more": "Upload more files", "analysis.attempt data": "Attempt data", "analysis.select format": "Select data to show", "analysis.download this table": "Download this table", "analysis.student": "Student Results", "analysis.question key": "Question Key", "analysis.question name": "Question Name", "analysis.group": "Group", "analysis.question": "Question", "analysis.part": "Part", "analysis.gap": "Gap", "analysis.record type": "Record Type", "analysis.score": "Score", "analysis.marks available": "Marks Available", "analysis.percentage": "Percentage", "analysis.answer": "Answer", "analysis.student name": "Student Name", "analysis.summary.no files": "You have not uploaded any files yet.", "analysis.summary.no decrypted files": "You have not uploaded any valid files yet.", "analysis.summary.one file": "One attempt.", "analysis.summary.several files": "{{num_files,niceNumber}} attempts.", "analysis.not secure context": "This page must be opened in a secure browser context. A secure context is either a page served over HTTPS, or a file loaded from your device.", "jme.shunt.pipe right hand takes no arguments": "The expression on the right-hand side of the pipe operator must be a function application.", "question.explore.no parts defined": "There are no parts defined in this question.", "answer": "answer", "worksheet.answersheet show question content": "Show question content in answer sheets?", "modal.confirm end exam": "Write <code>{{endConfirmation}}</code> in the box to confirm:", "modal.end exam button": "End exam", "lightbox.zoom in on image": "Zoom in on this image", "exam.progress": "Progress", "exam.questions answered": "{{numAnsweredQuestions}} of {{numQuestions}} questions answered.", "result.question marks available": "Marks Available", "result.question answered": "Answered?", "control.confirm end.correct": "You may now end the exam.", "control.confirm end.incorrect": "This is not the expected text.", "control.confirm end.password": "end", "jme.typecheck.for in name wrong type": "The name in a <code>for</code> statement must be a name or list of names, not {{type}}.", "jme.makeFast.no fast definition of function": "The function <code>{{name}}</code> here isn't defined in a way that can be made fast.", "part.show feedback": "Show feedback", "part.hide feedback": "Hide feedback", "part.feedback title": "Feedback for {{name}}.", "part.jme.must-match.warning": "Your answer is not in the expected form: {{-message}}", "part.numberentry.write your answer as a fraction": "Write your answer as a fraction.", "question.nav.label": "Question controls", "question.answer saved": "Answer saved", "question.all answers saved": "All answers saved", "analysis.back to results": "Back to results", "analysis.review": "Review", "analysis.review this": "Review this attempt", "analysis.reviewing attempt": "Reviewing attempt by {{student_name}}.", "part.there is new feedback": "The feedback has changed.", "modal.style.colour scheme": "Colour scheme", "modal.style.automatic colour scheme": "Automatic", "modal.style.light colour scheme": "Light", "modal.style.dark colour scheme": "Dark", "modal.style.custom colour scheme": "Custom", "modal.style.main font": "Main font", "modal.style.font.sans serif": "Sans serif", "modal.style.font.serif": "Serif", "modal.style.font.monospace": "Monospace", "modal.style.spacing": "Spacing", "modal.style.font weight": "Font weight", "modal.style.forced colours warning": "Your browser has overridden the colours used in this page because of a setting such as high contrast mode. Changes to the colour scheme settings here might not have any effect.", "modal.style.text preview": "Most text will look like this.", "modal.style.more options": "More options", "modal.style.main colour": "Main (brand) colour", "modal.style.primary colour": "Primary button colour", "modal.style.link colour": "Link colour", "modal.style.success colour": "Success/correct colour", "modal.style.info colour": "Info colour", "modal.style.warning colour": "Warning colour", "modal.style.danger colour": "Danger/incorrect colour", "modal.style.muted colour": "Muted text colour", "modal.style.highlight colour": "Highlight colour", "exam.error loading exam definition": "There was an error while loading the exam definition: {{text}}", "exam.no exam definition": "No exam definition was given.", "jme.typecheck.wrong arguments for anonymous function": "Wrong number of arguments for this anonymous function.", "worksheet.top": "Top", "worksheet.left": "Left", "worksheet.bottom": "Bottom", "worksheet.right": "Right"}}, "en-gb": {"translation": {"accessibility statement": "Accessibility statement and guide to adapting Numbas to your needs.", "answer.number.not a number": "Your answer is not a valid number.", "answer.number.fractions not allowed": "You may not enter a fraction.", "answer.jme.invalid expression": "{{-message}}", "answer.matrix.fractions not allowed": "You may not enter fractions.", "answer.matrix.some cell not a number": "One or more of the cells in your answer is not a valid number.", "page.no stylesheet loaded": "The page's stylesheet file has not loaded.", "page.loading": "Loading...", "page.saving": "<p>Saving.</p>\n<p>This might take a few seconds.</p>", "page.skip to content": "Skip to content", "mathjax.math processing error": "\"{{-message}}\" when texifying <code>{{expression}}</code>", "mathjax.error": "MathJax processing error: {{-message}}", "mathjax.error with context": "MathJax processing error in {{-context}}: {{-message}}", "die.numbas failed": "Numbas has failed", "die.sorry": "Sorry, Numbas has encountered an error which means it can't continue. Below is a description of the error.", "die.script not loaded": "Numbas couldn't start because the file <code>{{file}}</code> was not loaded. Check that it's included in <code>scripts.js</code>.", "die.error": "Error", "extension.not found": "Couldn't load the extension <code>{{name}}</code>.", "modal.confirm": "Confirm", "modal.confirm end exam": "Write <code>{{endConfirmation}}</code> in the box to confirm:", "modal.end exam button": "End exam", "modal.alert": "Alert", "modal.ok": "OK", "modal.cancel": "Cancel", "modal.style.colour scheme": "Colour scheme", "modal.style.automatic colour scheme": "Automatic", "modal.style.light colour scheme": "Light", "modal.style.dark colour scheme": "Dark", "modal.style.custom colour scheme": "Custom", "modal.style.invert images in dark mode": "Invert images in dark mode", "modal.style.background colour": "Background colour", "modal.style.text colour": "Text colour", "modal.style.text size": "Text size", "modal.style.main font": "Main font", "modal.style.font.sans serif": "Sans serif", "modal.style.font.serif": "Serif", "modal.style.font.monospace": "Monospace", "modal.style.spacing": "Spacing", "modal.style.font weight": "Font weight", "modal.style.explanation": "Use these controls to change the appearance of the exam.", "modal.style.forced colours warning": "Your browser has overridden the colours used in this page because of a setting such as high contrast mode. Changes to the colour scheme settings here might not have any effect.", "modal.style.reset to defaults": "Reset to defaults", "modal.style.text preview": "Most text will look like this.", "modal.style.more options": "More options", "modal.style.main colour": "Main (brand) colour", "modal.style.primary colour": "Primary button colour", "modal.style.link colour": "Link colour", "modal.style.success colour": "Success/correct colour", "modal.style.info colour": "Info colour", "modal.style.warning colour": "Warning colour", "modal.style.danger colour": "Danger/incorrect colour", "modal.style.muted colour": "Muted text colour", "modal.style.highlight colour": "Highlight colour", "lightbox.zoom in on image": "Zoom in on this image", "exam.error loading exam definition": "There was an error while loading the exam definition: {{text}}", "exam.no exam definition": "No exam definition was given.", "exam.enter password": "Password:", "exam.password.correct": "This password is correct. You can start the exam.", "exam.password.incorrect": "This password is incorrect.", "exam.exam name": "Exam Name:", "exam.introduction": "Exam introduction", "exam.feedback": "Exam feedback message", "exam.random seed": "Session ID:", "exam.student name": "Student's Name:", "exam.number of questions": "Number of Questions:", "exam.marks available": "Marks Available:", "exam.pass percentage": "Pass Percentage:", "exam.time allowed": "Time Allowed:", "exam.passed": "Passed", "exam.failed": "Failed", "exam.review header": "Review: ", "exam.enter your name": "Your name:", "exam.attempt download security warning": "This exam is configured to allow you to download your data, but it is not running in a secure browser context. You will not be able to download your exam data. Contact your lecturer or teacher for help.", "exam.progress": "Progress", "exam.questions answered": "{{numAnsweredQuestions}} of {{numQuestions}} questions answered.", "frontpage.start": "Start", "frontpage.scorm.lms not connected": "This exam is running in standalone mode. Your answers and marks will not be saved!", "suspend.paused header": "Paused", "suspend.resumed header": "Attempt resumed", "suspend.exam suspended": "The Exam has been suspended. Press <em>Resume</em> to continue.", "suspend.you can resume": "You will be able to resume this session the next time you start this activity.", "suspend.resume": "Resume", "result.exit": "Exit Exam", "result.print": "Print this results summary", "result.exam summary": "Exam Summary", "result.performance summary": "Performance Summary", "result.exam start": "Exam Start:", "result.exam stop": "Exam Stop:", "result.time spent": "Time Spent:", "result.questions attempted": "Questions Attempted:", "result.score": "Score:", "result.result": "Result:", "result.question number": "Question Number", "result.learning objective": "Learning objective", "result.question score": "Score", "result.question marks available": "Marks Available", "result.question answered": "Answered?", "result.question review title": "Review this question", "result.question review": "Review", "result.click a question to review": "Click on a question number to see how your answers were marked and, where available, full solutions.", "result.download exam object": "Download your exam data", "end.exam has finished": "The exam has finished. You may now close this window.", "end.print": "Print your exam transcript", "control.confirm leave": "You haven't finished the exam.", "control.not all questions answered": "You have not completed every question in this exam.", "control.not all questions submitted": "You have made changes to one or more answers but not submitted them. Please check each question has been submitted.", "control.confirm end": "Are you sure you want to end the exam? After you end the exam, you will not be able to change any of your answers.", "control.confirm regen": "Would you like to re-randomise this question? If you click OK, all your answers and marks for the current question will be lost.", "control.confirm regen no marks": "Would you like to re-randomise this question?", "control.confirm reveal": "Would you like to reveal the answer to this question? Any marks you have received so far will be locked and you will not be able to answer this question later.", "control.confirm reveal no marks": "Would you like to reveal the answer to this question?", "control.submit part.confirm remove next parts": "<p>One or more subsequent parts depend on your answer to this part. Submitting this part again will invalidate those parts, and remove them from the question. This cannot be undone.</p>\n<p>Would you like to submit this part again?</p>", "control.proceed anyway": "Proceed anyway?", "control.regen": "Try another question like this one", "control.submit answer": "Save answer", "control.submit all parts": "Save all answers", "control.previous": "Previous", "control.next": "Next", "control.advice": "Advice", "control.reveal": "Reveal answers", "control.total": "Total", "control.pause": "Pause", "control.end exam": "End Exam", "control.back to results": "Go back to results", "control.back to menu": "Go back to the menu", "control.toggle navigation menu": "Toggle the navigation menu", "control.return to question": "Return to the question", "control.style options": "Display options", "control.move to next question": "Move to the next question", "control.show introduction": "Introduction", "control.confirm end.correct": "You may now end the exam.", "control.confirm end.incorrect": "This is not the expected text.", "control.confirm end.password": "end", "display.answer widget.unknown widget type": "The answer widget type <code>{{name}}</code> is not recognised.", "display.part.jme.error making maths": "Error making maths display", "display.error making html": "Error making HTML in {{contextDescription}}: {{-message}}", "exam.xml.bad root": "Root element of exam XML should be 'exam'", "exam.changeQuestion.no questions": "This exam contains no questions! Check the .exam file for errors.", "analysis.header": "Analyse attempt data", "analysis.help.upload files": "Upload attempt data files that your students have given you.", "analysis.help.file input label": "Choose student attempt data files, or drag files onto this window.", "analysis.table.total": "Exam totals", "analysis.table.question": "Exam and question totals", "analysis.table.all": "All details", "analysis.student name.anonymous": "No name given", "analysis.expected": "Expected results", "analysis.start time": "Start time", "analysis.maximum": "Maximum Marks", "analysis.file": "File", "analysis.download": "Download", "analysis.delete": "Delete", "analysis.view results": "View results", "analysis.upload files": "Upload files", "analysis.upload more": "Upload more files", "analysis.back to results": "Back to results", "analysis.review": "Review", "analysis.review this": "Review this attempt", "analysis.reviewing attempt": "Reviewing attempt by {{student_name}}.", "analysis.attempt data": "Attempt data", "analysis.select format": "Select data to show", "analysis.download this table": "Download this table", "analysis.student": "Student Results", "analysis.question key": "Question Key", "analysis.question name": "Question Name", "analysis.group": "Group", "analysis.question": "Question", "analysis.part": "Part", "analysis.gap": "Gap", "analysis.record type": "Record Type", "analysis.score": "Score", "analysis.marks available": "Marks Available", "analysis.percentage": "Percentage", "analysis.answer": "Answer", "analysis.student name": "Student Name", "analysis.summary.no files": "You have not uploaded any files yet.", "analysis.summary.no decrypted files": "You have not uploaded any valid files yet.", "analysis.summary.one file": "One attempt.", "analysis.summary.several files": "{{num_files,niceNumber}} attempts.", "analysis.not secure context": "This page must be opened in a secure browser context. A secure context is either a page served over HTTPS, or a file loaded from your device.", "feedback.you were awarded": "You were awarded <strong>{{count,niceNumber}}</strong> $t(mark).", "feedback.taken away": "<strong>{{count,niceNumber}}</strong> $t(mark) $t(was) taken away.", "jme.tokenise.invalid near": "Invalid expression: <code>{{expression}}</code> at position {{position}} near <code>{{nearby}}</code>", "jme.tokenise.keypair key not a string": "Dictionary key should be a string, not {{type}}.", "jme.tokenise.number.object not complex": "Invalid object passed into number constructor.", "jme.shunt.not enough arguments": "Not enough arguments for operation <code>{{op}}</code>", "jme.shunt.no left bracket in function": "No matching left bracket in function application or tuple", "jme.shunt.no left square bracket": "No matching left bracket", "jme.shunt.no left bracket": "No matching left bracket", "jme.shunt.no right bracket": "No matching right bracket", "jme.shunt.no right square bracket": "No matching right square bracket to end list", "jme.shunt.missing operator": "Expression can't be evaluated -- missing an operator.", "jme.shunt.list mixed argument types": "Can't parse {{mode}}: mix of dictionary and list elements", "jme.shunt.expected argument before comma": "Expected to see something between the opening bracket and the comma", "jme.shunt.pipe right hand takes no arguments": "The expression on the right-hand side of the pipe operator must be a function application.", "jme.typecheck.function maybe implicit multiplication": "Function <code>{{name}}</code> is not defined. Did you mean <code>{{first}}*{{possibleOp}}(...)</code>?", "jme.typecheck.function not defined": "Function <code>{{op}}</code> is not defined. Is <code>{{op}}</code> a variable, and did you mean <code>{{suggestion}}*(...)</code>?", "jme.typecheck.op not defined": "Operation '{{op}}' is not defined.", "jme.typecheck.no right type definition": "No definition of '{{op}}' of correct type found.", "jme.typecheck.no right type unbound name": "Variable <code>{{name}}</code> is not defined.", "jme.typecheck.map not on enumerable": "<code>map</code> operation must work over a list or a range, not {{type}}", "jme.typecheck.for in name wrong type": "The name in a <code>for</code> statement must be a name or list of names, not {{type}}.", "jme.typecheck.wrong arguments for anonymous function": "Wrong number of arguments for this anonymous function.", "jme.evaluate.undefined variable": "Variable {{name}} is undefined", "jme.evaluate.no scope given": "Numbas.jme.evaluate must be given a Scope", "jme.substituteTree.undefined variable": "Undefined variable: <code>{{name}}</code>", "jme.thtml.not html": "Passed a non-HTML value into the THTML constructor.", "jme.func.switch.no default case": "No default case for Switch statement", "jme.func.listval.invalid index": "Invalid list index {{index}} on list of size {{size}}", "jme.func.listval.not a list": "Object is not subscriptable", "jme.func.listval.key not in dict": "Dictionary does not contain the key <code>{{key}}</code>", "jme.func.matrix.invalid row type": "Can't construct a matrix from rows of type {{type}}", "jme.func.except.continuous range": "Can't use the 'except' operator on continuous ranges.", "jme.vector.value not an array of numbers": "Tried to construct a vector using a value that is not an array of numbers.", "jme.matrix.reports bad size": "Matrix reports its size incorrectly - must be an error in constructor function", "jme.matrix.value not the right type": "Tried to construct a vector using a value of the wrong type.", "jme.parse signature.invalid signature string": "Invalid function signature string: {{str}}", "jme.subvars.null substitution": "Empty variable substitution: <code>$t(left brace){{str}}$t(right brace)", "jme.subvars.error compiling": "{{-message}} in <code>{{expression}}</code>", "jme.subvars.html inserted twice": "An HTML value has been embedded twice. Consider defining a function to generate a new value each time it is used.", "jme.texsubvars.no right bracket": "No matching <code>]</code> in {{op}} arguments.", "jme.texsubvars.missing parameter": "Missing parameter in {{op}}: {{parameter}}", "jme.texsubvars.no right brace": "No matching <code>}</code> in {{op}}", "jme.type.type already registered": "The data type {{type}} has already been registered so can't be registered again.", "jme.type.no cast method": "Can't automatically convert from {{from}} to {{to}}.", "jme.user javascript.error": "Error in user-defined javascript function <code>{{name}}</code>: {{-message}}", "jme.user javascript.returned undefined": "User-defined javascript function <code>{{name}}</code> returned <code>undefined</code>.", "jme.variables.error making function": "Error making function <code>{{name}}</code>: {{-message}}", "jme.variables.invalid function language": "The language <code>{{language}}</code> is not valid.", "jme.variables.syntax error in function definition": "Syntax error in function definition", "jme.variables.variable not defined": "Variable <code>{{name}}</code> is not defined.", "jme.variables.empty definition": "Definition of variable <code>{{name}}</code> is empty.", "jme.variables.empty name": "A question variable has not been given a name.", "jme.variables.circular reference": "Circular variable reference in definition of <code>{{name}}</code>", "jme.variables.error computing dependency": "Error computing referenced variable <code>{{name}}</code>", "jme.variables.error evaluating variable": "Error evaluating variable {{name}}: {{-message}}", "jme.variables.question took too many runs to generate variables": "A valid set of question variables was not generated in time.", "jme.variables.duplicate definition": "There is more than one definition of the variable <code>{{name}}</code>.", "jme.display.unknown token type": "Can't texify token type {{type}}", "jme.display.collectRuleset.no sets": "No sets given to collectRuleset!", "jme.display.collectRuleset.set not defined": "Ruleset {{name}} has not been defined", "jme.display.simplifyTree.no scope given": "Numbas.jme.display.simplifyTree must be given a Scope", "jme.display.simplifyTree.empty expression": "Expression is empty", "jme.display.simplifyTree.stuck in a loop": "Simplifier is stuck in a loop: <code>{{expr}}</code>", "jme.calculus.unknown derivative": "Don't know how to differentiate <code>{{tree}}</code>", "jme.interpreted as": "interpreted as", "jme.makeFast.no fast definition of function": "The function <code>{{name}}</code> here isn't defined in a way that can be made fast.", "math.precround.complex": "Can't round to a complex number of decimal places", "math.siground.complex": "Can't round to a complex number of sig figs", "math.combinations.complex": "Can't compute combinations of complex numbers", "math.combinations.n less than zero": "Can't compute combinations: n is less than zero", "math.combinations.k less than zero": "Can't compute combinations: k is less than zero", "math.combinations.n less than k": "Can't compute combinations: n is less than k", "math.permutations.n less than zero": "Can't compute permutations: n is less than zero", "math.permutations.k less than zero": "Can't compute permutations: k is less than zero", "math.permutations.n less than k": "Can't compute permutations: n is less than k", "math.permutations.complex": "Can't compute permutations of complex numbers", "math.gcf.complex": "Can't compute GCF of complex numbers", "math.lcm.complex": "Can't compute LCM of complex numbers", "math.order complex numbers": "Can't order complex numbers", "math.niceNumber.undefined": "Was expecting a number, but got <code>undefined</code>", "math.choose.empty selection": "Empty selection given to random function", "math.rangeToList.zero step size": "Can't convert a range with step size zero to a list.", "math.shuffle_together.lists not all the same length": "Not all lists are the same length.", "math.random_integer_partition.invalid k": "The size of the partition must be between 1 and {{n}}.", "matrixmath.not square": "This operation only works on a square matrix.", "matrixmath.not invertible": "This operation only works on an invertible matrix.", "matrixmath.gauss-jordan elimination.not enough columns": "There must be at least as many columns as rows.", "matrixmath.abs.non-square": "Can't compute the determinant of a matrix which isn't square.", "matrixmath.abs.too big": "Sorry, can't compute the determinant of a matrix bigger than 3x3 yet.", "matrixmath.mul.different sizes": "Can't multiply matrices of different sizes.", "vectormath.cross.not 3d": "Can only take the cross product of 3-dimensional vectors.", "vectormath.dot.matrix too big": "Can't calculate dot product of a matrix which isn't $1 \\times N$ or $N \\times 1$.", "vectormath.cross.matrix too big": "Can't calculate cross product of a matrix which isn't $1 \\times N$ or $N \\times 1$.", "menu.choose a question": "Choose a question.", "part.error": "{{path}}: {{-message}}", "part.with steps answer prompt": "Answer: ", "part.prompt": "prompt", "part.feedback": "Feedback", "part.choose next part.answered": "What do you want to do next?", "part.choose next part.unanswered": "Or, you could:", "part.choose next part.will be locked": "(This part will be locked)", "part.reached dead end": "There's nothing more to do from here.", "part.next part.penalty amount": "(lose {{count}} $t(mark))", "part.next part.confirm penalty": "Would you like to take this option? You will lose {{count,niceNumber}} $t(mark).", "part.script.error": "Error in part {{path}} custom script {{script}}: {{-message}}", "part.marking.steps no matter": "Because you received full marks for the part, your answers to the steps aren't counted.", "part.marking.steps change": "You were awarded <strong>{{count,niceNumber}}</strong> $t(mark) for your answers to the steps.", "part.marking.revealed steps": "You revealed the steps.", "part.marking.maximum scaled down": "The maximum you can score for this part is <strong>{{count,niceNumber}}</strong> $t(mark). Your scores will be scaled down accordingly.", "part.marking.used variable replacements": "This part was marked using your answers to previous parts.", "part.marking.variable replacement part not answered": "You must answer {{part}} first.", "part.marking.resubmit because of variable replacement": "This part's marking depends on your answers to other parts, which you have changed. Save your answer to this part again to update your score.", "part.marking.not submitted": "No answer submitted.", "part.marking.did not answer": "You did not answer this question.", "part.marking.total score": "You scored <strong>{{count,niceNumber}}</strong> $t(mark) for this part.", "part.marking.counts towards objective": "This part counts towards the objective <strong>\u201c{{objective}}\u201d</strong>.", "part.marking.minimum score applied": "The minimum score for this part is <strong>{{score,niceNumber}}</strong>.", "part.marking.maximum score applied": "The maximum score for this part is <strong>{{score,niceNumber}}</strong>.", "part.marking.nothing entered": "You did not enter an answer.", "part.marking.incorrect": "Your answer is incorrect.", "part.marking.partially correct": "Your answer is partially correct.", "part.marking.correct": "Your answer is correct.", "part.marking.uncaught error": "Error when marking: {{-message}}", "part.marking.parameter already in scope": "There is a variable named <code>{{name}}</code>, which is also the name of a marking parameter. Please rename the variable.", "part.marking.error in marking script": "There was an error in this part's marking algorithm. Please report this. {{-message}}", "part.marking.error in adaptive marking": "There was an error in the adaptive marking for this part. Please report this. {{-message}}", "part.marking.adaptive variable replacement refers to self": "This part refers to itself in a variable replacement for adaptive marking.", "part.marking.adaptive variable replacement refers to nothing": "This part contains an invalid variable replacement for adaptive marking.", "part.marking.no result after replacement": "This part could not be marked using your answers to previous parts.", "part.marking.missing required note": "The marking algorithm does not define the note <code>{{note}}</code>", "marking.apply.not a list": "The first argument to <code>apply</code> must be a list, and isn't", "marking.apply marking script.script not found": "Marking script <code>{{name}}</code> not found", "jme.script.note.compilation error": "Error compiling note <code>{{name}}</code>: {{-message}}", "jme.script.note.error evaluating note": "Error evaluating note <code>{{name}}</code> - {{-message}}", "jme.script.note.invalid definition": "Invalid note definition: <code>{{source}}</code>. {{-hint}}", "jme.script.note.invalid definition.missing colon": "You might be missing a colon after the name and description", "jme.script.note.invalid definition.description missing closing bracket": "You might be missing a closing bracket", "jme.script.note.empty expression": "The note <code>{{name}}</code> is empty.", "jme.script.error parsing notes": "Error parsing marking script: {{- message}}", "part.correct answer": "Expected answer:", "part.missing type attribute": "{{part}}: Missing part type attribute", "part.unknown type": "{{part}}: Unrecognised part type {{type}}", "part.setting not present": "Property '{{property}}' not set", "part.feedback out of date": "This feedback is based on your last submitted answer. Save your changed answer to get updated feedback.", "part.there is new feedback": "The feedback has changed.", "part.input title": "Answer for part {{name}}", "part.feedback title": "Feedback for {{name}}.", "part.correct answer title": "Expected answer for part {{name}}", "part.waiting for pre submit": "Your answer is being marked. Please wait.", "part.jme.answer missing": "Correct answer is missing", "part.jme.answer too long": "Your answer is too long.", "part.jme.answer too short": "Your answer is too short.", "part.jme.answer invalid": "Your answer is not a valid mathematical expression.<br/>{{-message}}.", "part.jme.marking.correct": "Your answer is numerically correct.", "part.jme.must-have bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.must-have one": "Your answer must contain: {{strings}}", "part.jme.must-have several": "Your answer must contain all of: {{strings}}", "part.jme.not-allowed bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.not-allowed one": "Your answer must not contain: {{strings}}", "part.jme.not-allowed several": "Your answer must not contain any of: {{strings}}", "part.jme.must-match.failed": "Your answer is not in the expected form.", "part.jme.must-match.warning": "Your answer is not in the expected form: {{-message}}", "part.jme.unexpected variable name": "Your answer was interpreted to use the unexpected variable name <code>{{name}}</code>.", "part.jme.unexpected variable name suggestion": "Your answer was interpreted to use the unexpected variable name <code>{{name}}</code>. Did you mean <code>{{suggestion}}</code>?", "part.jme.invalid value generator expression": "Invalid value generator expression for variable <code>{{name}}</code>: {{-message}}", "part.jme.error checking numerically": "There was an error numerically checking your answer: {{-message}}", "part.patternmatch.display answer missing": "Display answer is missing", "part.patternmatch.correct except case": "Your answer is correct, except for the case.", "part.numberentry.correct except decimal": "Your answer is within the allowed range, but decimal numbers are not allowed.", "part.numberentry.correct except fraction": "Your answer is within the allowed range, but fractions are not allowed.", "part.numberentry.answer invalid": "You did not enter a valid number.", "part.numberentry.answer not reduced": "Your answer is not reduced to lowest terms.", "part.numberentry.give your answer as a reduced fraction": "Reduce your answer to lowest terms.", "part.numberentry.write your answer as a fraction": "Write your answer as a fraction.", "part.numberentry.answer not integer": "Your answer is invalid. You must enter a whole number, not a decimal.", "part.numberentry.answer not integer or decimal": "Your answer is invalid. You must enter an integer or a decimal.", "part.numberentry.answer not integer or decimal or fraction": "Your answer is invalid. You must enter an integer, a decimal or a fraction.", "part.numberentry.zero sig fig": "This part is set up to round the student's answer to zero significant figures, which has no meaning.", "part.numberentry.negative decimal places": "This part is set up to round the student's answer to a negative number of decimal places, which has no meaning.", "part.numberentry.precision type.dp": "decimal place", "part.numberentry.precision type.dp_plural": "decimal places", "part.numberentry.precision type.sigfig": "significant figure", "part.numberentry.precision type.sigfig_plural": "significant figures", "part.numberentry.give your answer to precision": "Round your answer to {{count,niceNumber}} {{precisionType}}.", "part.numberentry.give your answer to precision_0": "Round your answer to the nearest integer.", "part.numberentry.display answer wrong type": "The display answer for this part is a value of type <code>{{got_type}}</code>, but should be a <code>{{want_type}}</code>.", "part.mcq.choices": "choices", "part.mcq.answers": "answers", "part.mcq.options def not a list": "The expression defining the {{properties}} is not a list.", "part.mcq.marking matrix string empty": "The custom marking matrix expression is empty.", "part.mcq.choices missing": "Definition of choices is missing", "part.mcq.matrix cell empty": "Part {{part}} marking matrix cell ({{row}},{{column}}) is empty", "part.mcq.matrix jme error": "Part {{part}} marking matrix cell ({{row}},{{column}}) gives a JME error: {{-error}}", "part.mcq.matrix not a number": "Part {{part}} marking matrix cell ({{row}},{{column}}) does not evaluate to a number", "part.mcq.wrong number of choices": "You selected the wrong number of choices.", "part.mcq.no choices selected": "No choices selected.", "part.mcq.matrix not a list": "Marking matrix, defined by JME expression, is not a list but it should be.", "part.mcq.matrix wrong type": "Element of invalid type '{{type}}' used in marking matrix.", "part.mcq.matrix mix of numbers and lists": "Mix of numbers and lists used in marking matrix.", "part.mcq.matrix wrong size": "Marking matrix is the wrong size.", "part.mcq.correct choice": "You chose a correct answer.", "part.mcq.incorrect choice": "You chose an incorrect answer.", "part.matrix.answer invalid": "Your answer is not valid.", "part.matrix.invalid cell": "One or more of the cells in your answer is empty or invalid.", "part.matrix.some incorrect": "One or more of the cells in your answer is incorrect, but you have been awarded marks for the rest.", "part.matrix.not all cells same precision": "You have not given every cell in your answer to the same precision.", "part.matrix.empty": "You have not entered an answer.", "part.matrix.empty cell": "One or more of the cells in your answer is empty.", "part.matrix.size mismatch": "The question author hasn't allowed the student to decide the dimensions of their answer, but the correct answer is {{correct_dimensions}} while the answer input is {{input_dimensions}}", "part.matrix.invalid type in prefilled": "There is an invalid value of type <code>{{n}}</code> in the array of pre-filled cells.", "part.gapfill.feedback header": "<strong>{{name}}</strong>", "part.gapfill.error marking gap": "Error marking {{name}}: {{-message}}", "part.gapfill.cyclic adaptive marking": "There is a cycle in the adaptive marking for this part: <strong>{{name1}}</strong> relies on <strong>{{name2}}</strong>, which eventually relies on <strong>{{name1}}</strong>.", "part.extension.not implemented": "Part hasn't implemented the <code>{{name}}</code> method.", "part.custom.empty setting": "No value given.", "part.custom.unrecognised input type": "Unrecognised setting type <code>{{input_type}}</code>", "part.custom.error evaluating input option": "Error evaluating input option <code>{{option}}</code>: {{-error}}", "part.custom.input option missing": "Definition of input option <code>{{option}}</code> is missing.", "part.custom.error evaluating setting": "Error evaluating setting <code>{{setting}}</code>: {{-error}}", "part.custom.expected answer has wrong type": "The expected answer for this part has the wrong type. It should be <code>{{shouldbe}}</code>.", "part.custom.input option has wrong type": "The answer input setting <code>{{option}}</code> has the wrong type. It should be <code>{{shouldbe}}</code>.", "question": "Question", "question.nav.label": "Question controls", "question.loaded name mismatch": "Can't resume this attempt - the package has changed since the last session.", "question.error": "Question {{number}}: {{-message}}", "question.error creating question": "Error while creating question {{number}}: {{-message}}", "question.preamble.error": "Error in preamble: {{-message}}", "question.preamble.syntax error": "Syntax error in preamble", "question.required extension not available": "This question requires the extension <code>{{-extension}}</code> but it is not available.", "question.unsupported part type": "Unsupported part type", "question.header": "Question {{number}}", "question.submit part": "Save answer", "question.answer saved": "Answer saved", "question.all answers saved": "All answers saved", "question.show steps": "Show steps", "question.show steps penalty": "You will lose <strong>{{count,niceNumber}}</strong> $t(mark).", "question.show steps no penalty": "Your score will not be affected.", "question.show steps already penalised": "You have already shown steps. You can show them again with no further penalty.", "question.hide steps": "Hide steps", "question.hide steps no penalty": "Your score will not be affected.", "question.statement": "Statement", "question.advice": "Advice", "question.progress": "Question progress", "question.no such part": "Can't find part {{path}}.", "question.can not submit": "Can not submit answer - check for errors.", "question.answer submitted": "Answer submitted", "question.unsubmitted changes": "You have made a change to your answer but not submitted it. Please check your answer and then press the <strong>Save answer</strong> button.", "question.unsubmitted changes_plural": "You have made changes to your answers but not submitted them. Please check your answers to each part and then press the <strong>Save all answers</strong> button.", "question.score feedback.show": "Show feedback", "question.score feedback.hide": "Hide feedback", "question.score feedback.not marked": "Not marked", "question.score feedback.unanswered": "Unanswered", "question.score feedback.partially answered": "Partially answered", "question.score feedback.answered": "Answered", "question.score feedback.unattempted": "Not attempted", "question.score feedback.attempted": "Attempted", "question.score feedback.score total": "{{marksString}}", "question.score feedback.score actual": "Score: {{scoreString}}", "question.score feedback.score total actual": "Score: {{score,niceNumber}}/{{marks,niceNumber}}", "question.score feedback.score actual.plain": "{{scoreString}}", "question.score feedback.score total actual.plain": "{{score,niceNumber}}/{{marks,niceNumber}}", "question.score feedback.correct": "Your answer is correct", "question.score feedback.partial": "Your answer is partially correct", "question.score feedback.wrong": "Your answer is incorrect", "question.objectives": "Objectives", "question.penalties": "Penalties", "question.selector.unsubmitted changes": "Unsubmitted changes.", "question.back to previous part": "Go back to the previous part", "question.explore.no parts defined": "There are no parts defined in this question.", "ruleset.circular reference": "Circular reference in definition of ruleset <code>{{name}}</code>", "ruleset.set not defined": "Ruleset {{name}} has not been defined", "timing.no accumulator": "no timing accumulator {{name}}", "timing.time remaining": "Time remaining:", "variable.error in variable definition": "Error in definition of variable <code>{{name}}</code>", "xml.could not load": "Couldn't load an XML document: {{-message}}", "xml.property not number": "Property {{name}} should be a number, but isn't ({{value}}), in node {{element}}", "xml.property not boolean": "Property {{name}} should be a boolean, but isn't ({{value}}), in node {{element}}", "scorm.error initialising": "Error initialising SCORM protocol: {{-message}}", "scorm.failed save": "<p>The request to save data to the server failed. Press <b>OK</b> to try again.</p>\n<p>If you get this message repeatedly, check your internet connection or use a different computer. Your previously submitted answers have been successfully saved and will be restored if you resume this session on a different computer.</p>\n<p>If this message appears persistently and you can't save <em>any</em> answers, please contact your lecturer or teacher.</p>", "scorm.no exam suspend data": "Failed to resume: no exam suspend data.", "scorm.error loading suspend data": "Error loading suspend data: {{-message}}", "scorm.error loading question": "Error loading question {{number}}: {{-message}}", "scorm.no question suspend data": "No question suspend data", "scorm.error loading part": "Error loading part {{part}}: {{-message}}", "scorm.no part suspend data": "No part suspend data", "util.product.non list": "Passed a non-list to <code>Numbas.util.product</code>", "util.equality not defined for type": "Equality not defined for type {{type}}", "util.formatNumberNotation.unrecognised syntax": "The number formatting syntax <code>{{syntax}}</code> is not recognised.", "mark": "mark", "mark_plural": "marks", "answer": "answer", "was": "was", "was_plural": "were", "part": "part", "gap": "gap", "step": "step", "left brace": "{", "right brace": "}", "math.lt.order complex numbers": "Can't order complex numbers", "matrix input.size control legend": "Size", "matrix input.rows": "Rows", "matrix input.columns": "Columns", "matrix input.cell label": "Row {{row}}, column {{column}}", "marking.note.compilation error": "Error compiling note <code>{{name}}</code>: {{-message}}", "marking.note.error evaluating note": "Error evaluating note <code>{{name}}</code> - {{-message}}", "marking.note.invalid definition": "Invalid note definition: <code>{{source}}</code>. {{-hint}}", "marking.note.invalid definition.missing colon": "You might be missing a colon after the name and description", "marking.note.invalid definition.description missing closing bracket": "You might be missing a closing bracket", "marking.note.empty expression": "The note <code>{{name}}</code> is empty.", "marking.script.error parsing notes": "Error parsing marking script: {{- message}}", "diagnostic.use retry": "Use one retry and try this topic again.", "diagnostic.move to next topic": "Move on to the next topic.", "diagnostic.next step question": "What would you like to do next?", "diagnostic.now assessing topic": "Now assessing {{current_topic}}", "diagnostic.one retry left": "You have 1 retry left", "diagnostic.retries left": "You have {{retries}} retries left.", "diagnostic.percentage completed": "You've completed <strong>{{percentage}}%</strong> of the test.", "diagnostic.end test": "End the test.", "diagnostic.test is over": "The test is over.", "diagnostic.passed all lo": "You have passed all learning objectives.", "diagnostic.more work on lo": "You need to do some more work on the following learning objectives: {{los}}.", "diagnostic.move to next question in topic": "Move on to the next question in topic.", "diagnostic.make a choice": "Make a choice", "diagnostic.complete": "Complete!", "diagnostic.studying topic": "Studying {{topic}}", "worksheet.number of exams to generate": "Number of sheets to generate", "worksheet.starting with id": "Starting with ID", "worksheet.show exam id": "Show sheet ID?", "worksheet.page break between questions": "Page breaks between questions?", "worksheet.page margins": "Page margins (mm)", "worksheet.text size": "Text size (pt)", "worksheet.generate": "Generate", "worksheet.generating exams": "Generating sheets", "worksheet.sheet id": "Sheet ID:", "worksheet.print single": "Print this sheet", "worksheet.print several": "Print these sheets", "worksheet.answer sheets": "Answer sheets", "worksheet.question sheets": "Question sheets", "worksheet.reconfigure": "Generate different sheets", "worksheet.show sheet": "Preview the sheet with ID:", "worksheet.answersheet show question content": "Show question content in answer sheets?", "worksheet.top": "Top", "worksheet.left": "Left", "worksheet.bottom": "Bottom", "worksheet.right": "Right", "modal.style.text size preview": "Most text will be this big.", "part.show feedback": "Show feedback", "part.hide feedback": "Hide feedback"}}, "sq-al": {"translation": {"page.loading": "Duke u ngarkuar ...", "page.saving": "<p>Duke ruajtur.</p>\n<p>Kjo mund t\u00eb k\u00ebrkoj\u00eb disa seconda.</p>", "mathjax.math processing error": "\"{{-message}}\" gjat\u00eb tex-ifikimit t\u00eb <code>{{expression}}</code>", "die.numbas failed": "Numbas d\u00ebshtoi", "die.sorry": "Na vjen keq, Numbas ka hasur n\u00eb nj\u00eb gabim q\u00eb do t\u00eb thot\u00eb se nuk mund t\u00eb vazhdoj\u00eb. M\u00eb posht\u00eb \u00ebsht\u00eb nj\u00eb p\u00ebrshkrim i gabimit.", "die.error": "Gabim", "modal.ok": "OK", "modal.cancel": "Anulo", "exam.exam name": "Emri i Provimit:", "exam.random seed": "ID e Sesionit:", "exam.student name": "Emri i Studentit:", "exam.number of questions": "Numri i Pyetjeve:", "exam.marks available": "Vler\u00ebsime n\u00eb dispozicion:", "exam.pass percentage": "P\u00ebrqindja p\u00ebr t\u00eb kaluar:", "exam.time allowed": "Koha e lejuar:", "exam.passed": "Pranohet", "exam.failed": "Nuk pranohet", "exam.review header": "Koment: ", "frontpage.start": "Fillo", "suspend.paused header": "I nd\u00ebrprer\u00eb p\u00ebrkoh\u00ebsisht", "suspend.exam suspended": "Provimi \u00ebsht\u00eb nd\u00ebrprer\u00eb p\u00ebrkoh\u00ebsisht. Shtypni <em>Vazhdo</em> p\u00ebr t\u00eb vazhduar.", "suspend.you can resume": "Ju mund ta vazhdoni k\u00ebt\u00eb sesion aty ku e keni l\u00ebn\u00eb, kur t\u00eb filloni k\u00ebt\u00eb aktivitet her\u00ebn tjet\u00ebr.", "suspend.resume": "Vazhdo", "result.exit": "Dil nga Provimi", "result.print": "Printoni k\u00ebt\u00eb p\u00ebrmbledhje t\u00eb rezultateve", "result.exam summary": "P\u00ebrmbledhje e provimit", "result.performance summary": "P\u00ebrmbledhje e rezultateve", "result.exam start": "Fillimi i Provimit:", "result.exam stop": "Mbarimi i Provimit:", "result.time spent": "Koha e p\u00ebrdorur:", "result.questions attempted": "Pyetjte t\u00eb tratjuara:", "result.score": "Vler\u00ebsimi:", "result.result": "Rezultati:", "result.question number": "Pyetja Nr", "result.question score": "Vler\u00ebsimi", "result.question review title": "Komento k\u00ebt\u00eb pyetje", "result.click a question to review": "Kliko mbi nj\u00eb num\u00ebr pyetjeje p\u00ebr t\u00eb par\u00eb se si jan\u00eb vler\u00ebsuar p\u00ebrgjigjet tuaja dhe kur \u00ebsht\u00eb e mundur, zgjidhjet e plota.", "end.exam has finished": "Provimi ka mbaruar. Tani mund ta mbyllni k\u00ebt\u00eb dritare.", "control.confirm leave": "Akoma nuk e keni mbaruar provimin.", "control.not all questions answered": "Nuk i keni trajtuar t\u00eb gjitha pyetjet e provimit.", "control.not all questions submitted": "Ju keni b\u00ebr\u00eb ndryshime n\u00eb nj\u00eb ose m\u00eb shum\u00eb p\u00ebrgjigje, por nuk i keni dor\u00ebzuar ato. Ju lutemi kontrolloni q\u00eb \u00e7do pyetje t\u00eb jet\u00eb dor\u00ebzuar.", "control.confirm end": "Jeni t\u00eb sigurt q\u00eb doni t\u00eb p\u00ebrfundoni provimin? Pas k\u00ebsaj, nuk do t\u00eb jeni n\u00eb gjendje t\u00eb ndryshoni asnj\u00eb nga p\u00ebrgjigjet tuaja.", "control.confirm regen": "D\u00ebshironi t\u00eb ri-krijoni k\u00ebt\u00eb pyetje? N\u00ebse klikoni OK, t\u00eb gjitha p\u00ebrgjigjet dhe vler\u00ebsimet tuaja p\u00ebr pyetjen aktuale do t\u00eb humbasin.", "control.confirm reveal": "D\u00ebshironi t\u00eb zbuloni p\u00ebrgjigjen p\u00ebr k\u00ebt\u00eb pyetje? \u00c7do vler\u00ebsim q\u00eb keni marr\u00eb deri tani do t\u00eb bllokohet dhe nuk do t\u00eb jeni n\u00eb gjendje t'i p\u00ebrgjigjeni k\u00ebsaj pyetjeje m\u00eb von\u00eb.", "control.proceed anyway": "Vazhdo gjith\u00ebsesi?", "control.regen": "Provo nj\u00eb pyetje tjet\u00ebr t\u00eb ngjashme", "control.submit answer": "Dor\u00ebzo p\u00ebrgjigjen", "control.submit all parts": "Dor\u00ebzo t\u00eb gjitha", "control.submit again": "Dor\u00ebzo p\u00ebrs\u00ebri", "control.submit": "Dor\u00ebzo", "control.previous": "Paraardh\u00ebs", "control.next": "Pasardh\u00ebs", "control.advice": "Udh\u00ebzim", "control.reveal": "Zbulo p\u00ebrgjigjet", "control.total": "Total", "control.pause": "Nd\u00ebrprit p\u00ebrkoh\u00ebsisht", "control.end exam": "P\u00ebrfundo Provimin", "control.back to results": "Kthehu tek rezultatet", "display.part.jme.error making maths": "Gabim n\u00eb paraqitjen e shprehjeve matematikore", "exam.xml.bad root": "Elementi rr\u00ebnj\u00eb i provimit XML duhet t\u00eb jet\u00eb 'exam'", "exam.changeQuestion.no questions": "Provimi nuk p\u00ebrmban asnj\u00eb pyetje! Kontrolloni skedarin .exam p\u00ebr gabime.", "feedback.you were awarded": "Jeni vler\u00ebsuar me <strong>{{count,niceNumber}}</strong> $t(mark).", "feedback.taken away": "<strong>{{count,niceNumber}}</strong> $t(mark) $t(was) zbritur.", "jme.tokenise.invalid": "Shprehje e gabuar: <code>{{expression}}</code>", "jme.shunt.not enough arguments": "Nuk ka argumente t\u00eb mjaftuesh\u00ebm p\u00ebr operatorin <code>{{op}}</code>", "jme.shunt.no left bracket in function": "Nuk kombinon kllapa majtas n\u00eb aplikimin e funksionit ose tek sistemi i renditur", "jme.shunt.no left square bracket": "Nuk kombinon kllapa majtas", "jme.shunt.no left bracket": "Nuk kombinon kllapa majtas", "jme.shunt.no right bracket": "Nuk kombinon kllapa djathtas", "jme.shunt.no right square bracket": "Nuk kombinon kllapa djathtas p\u00ebr t\u00eb mbyllur list\u00ebn", "jme.shunt.missing operator": "Shprehja nuk mund t\u00eb vler\u00ebsohet - mungon nj\u00eb operator.", "jme.typecheck.function maybe implicit multiplication": "Funksioni <code>{{name}}</code> nuk \u00ebsht\u00eb i p\u00ebrcaktuar. Mos keni dashur:<br/><code>{{first}}*{{possibleOp}}(...)</code>?", "jme.typecheck.function not defined": "Funksioni <code>{{op}}</code> nuk \u00ebsht\u00eb i p\u00ebrcaktuar. A \u00ebsht\u00eb <code>{{op}}</code> nj\u00eb variab\u00ebl dhe mos keni dashur: <br/><code>{{suggestion}}*(...)</code>?", "jme.typecheck.op not defined": "Veprimi '{{op}}' nuk \u00ebsht\u00eb i p\u00ebrcaktuar.", "jme.typecheck.no right type definition": "Nuk ka nj\u00eb p\u00ebrcaktim t\u00eb '{{op}}' p\u00ebr tipin e gjetur.", "jme.typecheck.no right type unbound name": "Variabli <code>{{name}}</code> nuk \u00ebsht\u00eb i p\u00ebrcaktuar.", "jme.typecheck.map not on enumerable": "veprimi <code>map</code> duhet t\u00eb p\u00ebrdoret p\u00ebr nj\u00eb list\u00eb ose zon\u00eb, interval, jo p\u00ebr {{type}}", "jme.evaluate.undefined variable": "Variabli {{name}} nuk \u00ebsht\u00eb i p\u00ebrcaktuar", "jme.thtml.not html": "Konstruktorit THTML i \u00ebsht\u00eb kaluar nj\u00eb vler\u00eb jo-HTML.", "jme.func.switch.no default case": "Asnj\u00eb rast i parazgjedhur p\u00ebr deklarat\u00ebn Switch", "jme.func.listval.invalid index": "Indeks liste i gabuar {{index}} n\u00eb list\u00ebn me madh\u00ebsi {{size}}", "jme.func.listval.not a list": "Objekti nuk \u00ebsht\u00eb i p\u00ebrshkruesh\u00ebm", "jme.func.matrix.invalid row type": "Nuk mund t\u00eb krijohet nj\u00eb matric\u00eb me rreshta t\u00eb tipit {{type}}", "jme.func.except.continuous range": "Nuk mund t\u00eb p\u00ebrdoret operatori 'except' n\u00eb intervale t\u00eb vazhduar.", "jme.matrix.reports bad size": "Matrix reports its size incorrectly - must be an error in constructor function", "jme.texsubvars.no right bracket": "No matching <code>]</code> in {{op}} arguments.", "jme.texsubvars.missing parameter": "Mungon parametri n\u00eb {{op}}: {{parameter}}", "jme.texsubvars.no right brace": "No matching <code>}</code> in {{op}}", "jme.user javascript.error": "Error in user-defined javascript function <code>{{name}}</code>: {{-message}}", "jme.variables.error making function": "Error making function <code>{{name}}</code>: {{-message}}", "jme.variables.syntax error in function definition": "Gabim sintaksor\u00eb gjat\u00eb definimit t\u00eb funksionit", "jme.variables.variable not defined": "Variabli <code>{{name}}</code> nuk \u00ebsht\u00eb i p\u00ebrcaktuar.", "jme.variables.empty definition": "P\u00ebrcaktimi i variablit <code>{{name}}</code> \u00ebsht\u00eb bosh.", "jme.variables.circular reference": "Referenc\u00eb rrethore n\u00eb p\u00ebrcaktimin e variablit <code>{{name}}</code>", "jme.variables.error computing dependency": "Error computing referenced variable <code>{{name}}</code>", "jme.variables.error evaluating variable": "Error evaluating variable {{name}}: {{-message}}", "jme.variables.question took too many runs to generate variables": "A valid set of question variables was not generated in time.", "jme.display.unknown token type": "Can't texify token type {{type}}", "jme.display.collectRuleset.no sets": "No sets given to collectRuleset!", "jme.display.collectRuleset.set not defined": "Ruleset {{name}} has not been defined", "jme.display.simplifyTree.no scope given": "Numbas.jme.display.simplifyTree must be given a Scope", "math.precround.complex": "Can't round to a complex number of decimal places", "math.siground.complex": "Can't round to a complex number of sig figs", "math.combinations.complex": "Can't compute combinations of complex numbers", "math.permutations.complex": "Can't compute permutations of complex numbers", "math.gcf.complex": "Smund t\u00eb llogaritet GCF i numrave kompleks", "math.lcm.complex": "Can't compute LCM of complex numbers", "math.lt.order complex numbers": "Can't order complex numbers", "math.choose.empty selection": "Empty selection given to random function", "matrixmath.abs.non-square": "Nuk mund t\u00eb llogaritet p\u00ebrcaktori i nj\u00eb matrice jo katrore.", "matrixmath.abs.too big": "Na vjen keq, akoma nuk mund t\u00eb llogarisim p\u00ebrcaktorin e nj\u00eb matrice m\u00eb t\u00eb madhe se 3x3.", "matrixmath.mul.different sizes": "Nuk mund t\u00eb shum\u00ebzohen dy matrica me p\u00ebrmasa t\u00eb ndryshme.", "vectormath.cross.not 3d": "Can only take the cross product of 3-dimensional vectors.", "vectormath.dot.matrix too big": "Can't calculate dot product of a matrix which isn't $1 \\times N$ or $N \\times 1$.", "vectormath.cross.matrix too big": "Can't calculate cross product of a matrix which isn't $1 \\times N$ or $N \\times 1$.", "part.with steps answer prompt": "P\u00ebrgjigje: ", "part.script.error": "Error in part {{path}} custom script {{script}}: {{-message}}", "part.marking.steps no matter": "Meqen\u00ebse keni marr\u00eb t\u00eb gjtha pik\u00ebt p\u00ebr pjes\u00ebn, p\u00ebrgjigjet p\u00ebr hapat nuk do t\u00eb llogariten.", "part.marking.revealed steps no penalty": "Keni zbuluar hapat.", "part.marking.used variable replacements": "Kjo k\u00ebrkes\u00eb \u00ebsht\u00eb vler\u00ebsuar duke p\u00ebrdorur p\u00ebrgjigjet e dh\u00ebna n\u00eb k\u00ebrkesat paraardh\u00ebse.", "part.marking.variable replacement part not answered": "Duhet ti p\u00ebrgjigjeni {{part}} m\u00eb par\u00eb.", "part.marking.resubmit because of variable replacement": "Vler\u00ebsimi i k\u00ebsaj k\u00ebrkese varet nga p\u00ebrgjigjet e k\u00ebrkesave t\u00eb tjera, t\u00eb cilat ju i keni ndryshuar. Dor\u00ebzoni p\u00ebrs\u00ebri p\u00ebrgjigjen p\u00ebr k\u00ebt\u00eb k\u00ebrkes\u00eb p\u00ebr t\u00eb azhornuar rezultatin.", "part.marking.not submitted": "Nuk \u00ebsht\u00eb dor\u00ebzuar p\u00ebrgjigje.", "part.marking.did not answer": "Nuk i jeni p\u00ebrgjigjur k\u00ebsaj pyetjeje.", "part.marking.nothing entered": "Nuk keni dh\u00ebn\u00eb nj\u00eb p\u00ebrgjigje.", "part.marking.incorrect": "P\u00ebrgjigja juaj \u00ebsht\u00eb e pasakt\u00eb.", "part.marking.correct": "P\u00ebrgjigja juaj \u00ebsht\u00eb e sakt\u00eb.", "part.marking.uncaught error": "Gabim n\u00eb vler\u00ebsim: {{-message}}", "part.marking.no result": "Kjo k\u00ebrkes\u00eb nuk mund t\u00eb vler\u00ebsohet.", "part.correct answer": "P\u00ebrgjigja e pritshme:", "part.missing type attribute": "{{part}}: Missing part type attribute", "part.unknown type": "{{part}}: Unrecognised part type {{type}}", "part.setting not present": "Property '{{property}}' not set", "part.jme.answer missing": "Mungon p\u00ebrgjigja e sakt\u00eb", "part.jme.answer too long": "P\u00ebrgjigja juaj \u00ebsht\u00eb shum\u00eb e gjat\u00eb.", "part.jme.answer too short": "P\u00ebrgjigja juaj \u00ebsht\u00eb shum\u00eb e shkurt\u00eb.", "part.jme.answer invalid": "P\u00ebrgjigja juaj nuk \u00ebsht\u00eb nj\u00eb shprehje e vlefshme matematike.<br/>{{-message}}.", "part.jme.marking.correct": "P\u00ebrgjigja juaj \u00ebsht\u00eb numerikisht e sakt\u00eb.", "part.jme.must-have bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.must-have one": "P\u00ebrgjigja juaj duhet t\u00eb p\u00ebrmbaj\u00eb: {{strings}}", "part.jme.must-have several": "P\u00ebrgjigja juaj duhet t\u00eb p\u00ebrmbaj\u00eb t\u00eb gjitha fjal\u00ebt: {{strings}}", "part.jme.not-allowed bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.not-allowed one": "P\u00ebrgjigja juaj nuk duhet t\u00eb p\u00ebrmbaj\u00eb: {{strings}}", "part.jme.not-allowed several": "P\u00ebrgjigja juaj nuk duhet t\u00eb p\u00ebrmbaj\u00eb asnj\u00eb nga fjal\u00ebt: {{strings}}", "part.jme.unexpected variable name": "P\u00ebrgjigja juaj \u00ebsht\u00eb interpretuar sikur p\u00ebrdor variablin e pap\u00ebrcaktuar <code>{{name}}</code>.", "part.jme.unexpected variable name suggestion": "P\u00ebrgjigja juaj \u00ebsht\u00eb interpretuar sikur p\u00ebrdor variablin e pap\u00ebrcaktuar <code>{{name}}</code>. Ndoshta keni dashur <code>{{suggestion}}</code>?", "part.patternmatch.display answer missing": "Paraqitja e p\u00ebrgjigjes mungon", "part.patternmatch.correct except case": "P\u00ebrgjigja juaj \u00ebsht\u00eb e sakt\u00eb, p\u00ebrve\u00e7 k\u00ebtij rasti.", "part.numberentry.correct except decimal": "P\u00ebrgjigja juaj \u00ebsht\u00eb brenda intervalit t\u00eb lejuar, por nuk pranohen numrat dhjetor\u00eb.", "part.numberentry.correct except fraction": "P\u00ebrgjigja juaj \u00ebsht\u00eb brenda intervalit t\u00eb lejuar, por nuk pranohen thyesat.", "part.numberentry.answer invalid": "Nuk keni shkruar nj\u00eb num\u00ebr.", "part.numberentry.answer not integer": "P\u00ebrgjigja jua \u00ebsht\u00eb e gabuar. Duhet t\u00eb shkruani nj\u00eb num\u00ebr t\u00eb plot\u00eb dhe jo nj\u00eb num\u00ebr dhjetor.", "part.numberentry.answer not integer or decimal": "P\u00ebrgjigja jua \u00ebsht\u00eb e gabuar. Duhet t\u00eb shkruani nj\u00eb num\u00ebr t\u00eb plot\u00eb ose dhjetor.", "part.numberentry.zero sig fig": "Kjo k\u00ebrkes\u00eb \u00ebsht\u00eb nd\u00ebrtuar n\u00eb m\u00ebnyr\u00eb q\u00eb t\u00eb rrumbullakos\u00eb p\u00ebrgjigjen e studentit me zero shifra me vlere, gj\u00eb q\u00eb nuk ka kuptim.", "part.mcq.options def not a list": "The expression defining the {{properties}} is not a list.", "part.mcq.marking matrix string empty": "The custom marking matrix expression is empty.", "part.mcq.choices missing": "Definition of choices is missing", "part.mcq.matrix not a number": "Part {{part}} marking matrix cell ({{row}},{{column}}) does not evaluate to a number", "part.mcq.wrong number of choices": "Keni zgjedhur numrin e gabuar t\u00eb opsioneve.", "part.mcq.no choices selected": "Nuk \u00ebsht\u00eb b\u00ebr\u00eb asnj\u00eb zgjedhje.", "part.mcq.matrix not a list": "Marking matrix, defined by JME expression, is not a list but it should be.", "part.mcq.matrix wrong type": "Element of invalid type '{{type}}' used in marking matrix.", "part.mcq.matrix mix of numbers and lists": "Mix of numbers and lists used in marking matrix.", "part.mcq.matrix wrong size": "Marking matrix is the wrong size.", "part.mcq.correct choice": "Keni zgjedhur nj\u00eb p\u00ebrgjigje t\u00eb sakt\u00eb.", "part.matrix.answer invalid": "P\u00ebrgjigja juaj nuk \u00ebsht\u00eb e sakt\u00eb.", "part.matrix.invalid cell": "Nj\u00eb ose m\u00eb shum\u00eb qeliza n\u00eb p\u00ebrgjigjen tuaj jan\u00eb bosh ose t\u00eb pavlefshme.", "part.matrix.some incorrect": "Nj\u00eb ose m\u00eb shum\u00eb qeliza n\u00eb p\u00ebrgjigjen tuaj jan\u00eb jo t\u00eb sakta, por jeni vler\u00ebsuar mbi pjes\u00ebn tjet\u00ebr.", "part.matrix.empty": "Nuk keni shkruar nj\u00eb p\u00ebrgjigje.", "part.matrix.empty cell": "Nj\u00eb ose m\u00eb shum\u00eb qeliza n\u00eb p\u00ebrgjigjen tuaj jan\u00eb bosh.", "part.matrix.size mismatch": "Autori i pyetjes nuk i ka lejuar studentit t\u00eb vendos\u00eb dimensionin e p\u00ebrgjgigjes, por p\u00ebrgjigja e sakt\u00eb \u00ebsht\u00eb me {{correct_dimensions}} nd\u00ebrsa p\u00ebrgjigja e dh\u00ebn\u00eb \u00ebsht\u00eb me {{input_dimensions}}", "part.gapfill.feedback header": "<strong>Hap\u00ebsir\u00eb {{name}}</strong>", "part.extension.not implemented": "K\u00ebrkesa nuk ka implementuar metod\u00ebn <code>{{name}}</code>.", "question.loaded name mismatch": "Nuk mund t\u00eb vazhdohet ky sesion, sepse aplikacioni ka ndryshuar.", "question.error": "Pyetja {{number}}: {{-message}}", "question.preamble.error": "Error in preamble: {{-message}}", "question.preamble.syntax error": "Syntax error in preamble", "question.unsupported part type": "Unsupported part type", "question.header": "Pyetja {{number}}", "question.submit part": "Dor\u00ebzo k\u00ebt\u00eb p\u00ebrgjigje", "question.show steps": "Paraqit hapat", "question.show steps penalty": "Do ju zbriten <strong>{{count,niceNumber}}</strong> $t(mark).", "question.show steps no penalty": "Vler\u00ebsimi nuk do t\u00eb ndikohet.", "question.show steps already penalised": "Tashm\u00eb ju i kani paraqitur hapat. Paraqitja e tyre p\u00ebrs\u00ebri nuk ju penalizon.", "question.hide steps": "Fshih hapat", "question.hide steps no penalty": "Vler\u00ebsimi nuk do t\u00eb ndikohet.", "question.advice": "Udh\u00ebzim", "question.no such part": "K\u00ebrkesa {{path}} nuk mund t\u00eb gjendet", "question.can not submit": "P\u00ebrgjigja nuk mund t\u00eb dor\u00ebzohet - kontrolloni p\u00ebr gabime.", "question.answer submitted": "P\u00ebrgjigja u dor\u00ebzua", "question.score feedback.show": "Paraqit vler\u00ebsim", "question.score feedback.hide": "Fshih vler\u00ebsim", "question.score feedback.score total actual": "Score: {{score,niceNumber}}/{{marks,niceNumber}}", "question.score feedback.score total": "{{marksString}}", "question.score feedback.score actual": "Score: {{scoreString}}", "question.score feedback.answered": "Me p\u00ebrgjigje", "question.score feedback.unanswered": "Pa p\u00ebrgjigje", "question.score feedback.correct": "P\u00ebrgjigja juaj \u00ebsht\u00eb e sakt\u00eb", "question.score feedback.partial": "P\u00ebrgjigja juaj \u00ebsht\u00eb pjes\u00ebrisht e sakt\u00eb", "question.score feedback.wrong": "P\u00ebrgjigja juaj \u00ebsht\u00eb e gabuar", "question.selector.unsubmitted changes": "Ndryshime t\u00eb pa dor\u00ebzuara.", "timing.no accumulator": "nuk ka akumulator kohor {{name}}", "timing.time remaining": "Koha e mbetur:", "xml.could not load": "Nuk mund t\u00eb ngarkohet dokumenti XML: {{-message}}", "xml.property not number": "Property {{name}} should be a number, but isn't ({{value}}), in node {{element}}", "xml.property not boolean": "Property {{name}} should be a boolean, but isn't ({{value}}), in node {{element}}", "xml.error in variable definition": "Error in definition of variable <code>{{name}}</code>", "scorm.error initialising": "Error initialising SCORM protocol: {{-message}}", "scorm.failed save": "<p>K\u00ebrkesa p\u00ebr t\u00eb ruajtur t\u00eb dh\u00ebnat n\u00eb server d\u00ebshtoi. Klikoni <b>OK</b> p\u00ebr ta provuar p\u00ebrs\u00ebri.</p>\n<p>N\u00ebse ky mesazh p\u00ebrs\u00ebritet disa her\u00eb, kontrolloni lidhjen tuaj me internetin ose p\u00ebrdorni nj\u00eb kompjuter tjet\u00ebr. P\u00ebrgjigjet tuaja t\u00eb m\u00ebparshme jan\u00eb ruajtur me sukses dhe do t\u00eb ringarkohen n\u00ebse e vazhdoni k\u00ebt\u00eb sesion n\u00eb nj\u00eb kompjuter tjet\u00ebr.</p>\n<p>N\u00ebse ky mesazh shafet vazhdimisht dhe nuk mund t\u00eb ruani <em>asnj\u00eb</em> p\u00ebrgjigje, ju lutem kontaktoni pedagogun ose m\u00ebsuesin.</p>", "scorm.no exam suspend data": "Vazhdimi i sesionit d\u00ebshtoi: nuk ka t\u00eb dh\u00ebna mbi provimin.", "scorm.error loading suspend data": "Gabim n\u00eb ngarkimin e t\u00eb dh\u00ebnave t\u00eb sesionit: {{-message}}", "scorm.error loading question": "Gabim n\u00eb ngarkimin e pyetjes {{number}}: {{-message}}", "scorm.no question suspend data": "Nuk ka t\u00eb dh\u00ebna t\u00eb pezulluara p\u00ebr pyetjen", "scorm.error loading part": "Gabim n\u00eb ngarkimin e k\u00ebrkes\u00ebs {{part}}: {{-message}}", "scorm.no part suspend data": "Nuk ka t\u00eb dh\u00ebna t\u00eb pezulluara p\u00ebr k\u00ebrkes\u00ebn", "util.product.non list": "Passed a non-list to <code>Numbas.util.product</code>", "mark": "pik\u00eb", "was": "ishte", "part": "k\u00ebrkes\u00eb", "gap": "hap\u00ebsir\u00eb", "step": "hap", "jme.substituteTree.undefined variable": "Variab\u00ebl i pap\u00ebrcaktuar: <code>{{name}}</code>", "jme.user javascript.returned undefined": "User-defined javascript function <code>{{name}}</code> returned <code>undefined</code>.", "part.marking.steps change": "Jeni vler\u00ebsuar me <strong>{{count,niceNumber}}</strong> $t(mark) p\u00ebr p\u00ebrgjigjet e dh\u00ebna tek hapat.", "part.marking.revealed steps with penalty": "Keni paraqitur hapat. Maksimumi i vler\u00ebsimit p\u00ebr k\u00ebt\u00eb k\u00ebrkes\u00eb \u00ebsht\u00eb <strong>{{count,niceNumber}}</strong> $t(mark). Vler\u00ebsimi juaj do t\u00eb ulet bazuar mbi k\u00ebt\u00eb.", "part.marking.total score": "Jeni vler\u00ebsuar me <strong>{{count,niceNumber}}</strong> $t(mark) p\u00ebr k\u00ebt\u00eb k\u00ebrkes\u00eb.", "part.numberentry.precision type.dp": "shif\u00ebr dhjetore", "part.numberentry.precision type.dp_plural": "shifra dhjetore", "part.numberentry.precision type.sigfig": "shif\u00ebr me vler\u00eb", "part.numberentry.precision type.sigfig_plural": "shifra me vler\u00eb", "part.numberentry.give your answer to precision": "Rrumbullakoseni p\u00ebrgjigjen me {{count,niceNumber}} {{precisionType}}.", "question.unsubmitted changes": "Keni ndryshuar p\u00ebrgjigjen por nuk e keni dor\u00ebzuar. Ju lutemi kontrolloni p\u00ebrgjigjen dhe klikoni butonin <strong>Dor\u00ebzo p\u00ebrgjigje</strong>.", "question.unsubmitted changes_plural": "Keni ndryshuar p\u00ebrgjigjet por nuk i keni dor\u00ebzuar. Ju lutemi kontrolloni p\u00ebrgjigjet dhe klikoni butonin <strong>Dor\u00ebzo t\u00eb gjitha p\u00ebrgjigjet</strong>.", "util.equality not defined for type": "Barazim i pa definuar p\u00ebr tipin {{type}}", "mark_plural": "pik\u00eb", "was_plural": "jan\u00eb", "die.script not loaded": "Numbas nuk mund t\u00eb filloj\u00eb pasi skedari <code>{{file}}</code> nuk \u00ebsht\u00eb ngarkuar. Sigurohuni q\u00eb ai t\u00eb jet\u00eb i p\u00ebrfshir\u00eb n\u00eb <code>scripts.js</code>.", "math.combinations.n less than zero": "Can't compute combinations: n is less than zero", "math.combinations.k less than zero": "Can't compute combinations: k is less than zero", "math.combinations.n less than k": "Can't compute combinations: n is less than k", "math.permutations.n less than zero": "Can't compute permutations: n is less than zero", "math.permutations.k less than zero": "Can't compute permutations: k is less than zero", "math.permutations.n less than k": "Can't compute permutations: n is less than k", "part.numberentry.give your answer to precision_0": "Rrumbullakoseni p\u00ebrgjigjen n\u00eb numrin e plot\u00eb m\u00eb t\u00eb af\u00ebrt.", "mathjax.error": "MathJax processing error: {{-message}}", "mathjax.error with context": "MathJax processing error in {{-context}}: {{-message}}", "exam.introduction": "Prezantim i provimit", "exam.feedback": "Mesazhi i vler\u00ebsimit t\u00eb provimit", "jme.tokenise.keypair key not a string": "Dictionary key should be a string, not {{type}}.", "jme.shunt.list mixed argument types": "Can't parse {{mode}}: mix of dictionary and list elements", "jme.func.listval.key not in dict": "Dictionary does not contain the key <code>{{key}}</code>", "part.prompt": "prompt", "part.feedback": "vler\u00ebsim", "part.numberentry.answer not reduced": "P\u00ebrgjigja juaj nuk \u00ebsht\u00eb thjeshtuar n\u00eb termat m\u00eb t\u00eb vegj\u00ebl.", "part.numberentry.give your answer as a reduced fraction": "Thjeshtoni p\u00ebrgjigjen n\u00eb termat m\u00eb t\u00eb vegj\u00ebl.", "part.numberentry.negative decimal places": "This part is set up to round the student's answer to a negative number of decimal places, which has no meaning.", "part.mcq.choices": "opsione", "part.mcq.answers": "p\u00ebrgjigje", "part.mcq.matrix cell empty": "Part {{part}} marking matrix cell ({{row}},{{column}}) is empty", "part.mcq.matrix jme error": "Part {{part}} marking matrix cell ({{row}},{{column}}) gives a JME error: {{-error}}", "question.statement": "Formulimi", "ruleset.circular reference": "Circular reference in definition of ruleset <code>{{name}}</code>", "ruleset.set not defined": "Ruleset {{name}} has not been defined", "jme.evaluate.no scope given": "Numbas.jme.evaluate must be given a Scope", "question.score feedback.answered total actual": "Vler\u00ebsimi: {{score,niceNumber}}/{{marks,niceNumber}}", "question.score feedback.answered total": "{{marksString}}. Me p\u00ebrgjigje.", "question.score feedback.answered actual": "Vler\u00ebsimi: {{scoreString}}", "question.score feedback.unanswered total": "{{marksString}}.", "answer.number.not a number": "P\u00ebrgjigja juaj nuk \u00ebsht\u00eb num\u00ebr.", "answer.number.fractions not allowed": "Nuk mund t\u00eb shkruani thyes\u00eb.", "answer.jme.invalid expression": "{{-message}}", "answer.matrix.fractions not allowed": "Nuk mund t\u00eb shkruani thyesa.", "answer.matrix.some cell not a number": "Vlerate n\u00eb nj\u00eb ose disa qeliza t\u00eb p\u00ebrgjigjes nuk jan\u00eb numra.", "exam.enter password": "Fjal\u00ebkalimi:", "exam.password.correct": "Fjal\u00ebkalimi \u00ebsht\u00eb i sakt\u00eb. Ju mund t\u00eb filloni provimin.", "exam.password.incorrect": "Fjal\u00ebkalimi nuk \u00ebsht\u00eb i sakt\u00eb.", "frontpage.scorm.lms not connected": "This exam is running in standalone mode. Your answers and marks will not be saved!", "result.question review": "Review", "control.confirm regen no marks": "Doni ta ri-randomizoni k\u00ebt\u00eb pyetje?", "control.confirm reveal no marks": "Doni ti shfaqni p\u00ebrgjigjet e k\u00ebsaj pyetjeje?", "jme.tokenise.invalid near": "Shprehje e gabuar: <code>{{expression}}</code> n\u00eb pozicionin {{position}} pran\u00eb <code>{{nearby}}</code>", "jme.tokenise.number.object not complex": "Invalid object passed into number constructor.", "jme.subvars.null substitution": "Empty variable substitution: <code>$t(left brace){{str}}$t(right brace)", "jme.type.type already registered": "The data type {{type}} has already been registered so can't be registered again.", "jme.type.no cast method": "Nuk mund t\u00eb b\u00ebhet konvertim automatik nga {{from}} n\u00eb {{to}}.", "jme.display.simplifyTree.empty expression": "Shprehja \u00ebsht\u00eb bosh", "jme.display.simplifyTree.stuck in a loop": "Simplifier is stuck in a loop: <code>{{expr}}</code>", "math.niceNumber.undefined": "Was expecting a number, but got <code>undefined</code>", "math.rangeToList.zero step size": "Can't convert a range with step size zero to a list.", "part.error": "{{path}}: {{-message}}", "part.marking.revealed steps": "You revealed the steps.", "part.marking.maximum scaled down": "The maximum you can score for this part is <strong>{{count,niceNumber}}</strong> $t(mark). Your scores will be scaled down accordingly.", "part.marking.minimum score applied": "The minimum score for this part is <strong>{{score,niceNumber}}</strong>.", "part.marking.maximum score applied": "The maximum score for this part is <strong>{{score,niceNumber}}</strong>.", "part.marking.error in marking script": "There was an error in this part's marking algorithm. Please report this. {{-message}}", "part.marking.no result after replacement": "This part could not be marked using your answers to previous parts.", "part.marking.missing required note": "The marking algorithm does not define the note <code>{{note}}</code>", "marking.apply.not a list": "The first argument to <code>apply</code> must be a list, and isn't", "marking.apply marking script.script not found": "Marking script <code>{{name}}</code> not found", "marking.note.compilation error": "Error compiling note <code>{{name}}</code>: {{-message}}", "marking.note.error evaluating note": "Error evaluating note <code>{{name}}</code> - {{-message}}", "marking.note.invalid definition": "Invalid note definition: <code>{{source}}</code>. {{-hint}}", "marking.note.invalid definition.missing colon": "You might be missing a colon after the name and description", "marking.note.invalid definition.description missing closing bracket": "You might be missing a closing bracket", "marking.note.empty expression": "The note <code>{{name}}</code> is empty.", "marking.script.error parsing notes": "Error parsing marking script: {{- message}}", "part.feedback out of date": "Ky vler\u00ebsim bazohet n\u00eb p\u00ebrgjigjen e fundit t\u00eb dor\u00ebzuar nga ju. Dor\u00ebzoni p\u00ebrgjigjen e ndryshuar p\u00ebr t\u00eb marr\u00eb nj\u00eb vleresim t\u00eb p\u00ebrdit\u00ebsuar.", "part.jme.invalid value generator expression": "Invalid value generator expression for variable <code>{{name}}</code>: {{-message}}", "part.mcq.incorrect choice": "Keni zgjedhur nj\u00eb p\u00ebrgjigje jo t\u00eb sakt\u00eb.", "part.matrix.not all cells same precision": "You have not given every cell in your answer to the same precision.", "part.gapfill.error marking gap": "Error marking {{name}}: {{-message}}", "part.custom.empty setting": "No value given.", "part.custom.unrecognised input type": "Unrecognised setting type <code>{{input_type}}</code>", "part.custom.error evaluating input option": "Error evaluating input option <code>{{option}}</code>: {{-error}}", "part.custom.input option missing": "Definition of input option <code>{{option}}</code> is missing.", "part.custom.error evaluating setting": "Error evaluating setting <code>{{setting}}</code>: {{-error}}", "question.error creating question": "Error while creating question {{number}}: {{-message}}", "question.score feedback.not marked": "E pa vler\u00ebsuar", "question.score feedback.partially answered": "P\u00ebrgjigjur pjes\u00ebrisht", "variable.error in variable definition": "Error in definition of variable <code>{{name}}</code>", "left brace": "{", "right brace": "}", "extension.not found": "Couldn't load the extension <code>{{name}}</code>.", "control.toggle navigation menu": "Paraqit menun\u00eb e navigimit", "part.input title": "Answer for part {{name}}", "part.correct answer title": "P\u00ebrgjigja e pritshme e k\u00ebrkes\u00ebs {{name}}", "part.jme.must-match.failed": "P\u00ebrgjigja juaj nuk \u00ebsht\u00eb n\u00eb form\u00ebn e duhur.", "question.score feedback.none": "", "control.submit part.confirm remove next parts": "<p>Nj\u00eb ose disa nga k\u00ebrkesat vijuese varen nga p\u00ebrgjigjet e k\u00ebsaj k\u00ebrkese. Duke e dor\u00ebzuar edhe nj\u00eb her\u00eb k\u00ebt\u00eb k\u00ebrkes\u00eb do t\u00eb anulohen p\u00ebrgjigjet e tyre, dhe do t\u00eb hiqen nga pyetja. Ky verpim nuk mund t\u00eb kthehet.</p>\n<p>Doni ta dor\u00ebzoni k\u00ebt\u00eb k\u00ebrkes\u00eb p\u00ebrs\u00ebri?</p>", "control.back to menu": "Kthehu pas tek menuja", "display.error making html": "Gabim n\u00eb krijimin e HTML n\u00eb {{contextDescription}}: {{-message}}", "jme.subvars.error compiling": "{{-message}} n\u00eb <code>{{expression}}</code>", "jme.variables.empty name": "A question variable has not been given a name.", "jme.calculus.unknown derivative": "Don't know how to differentiate <code>{{tree}}</code>", "math.order complex numbers": "Can't order complex numbers", "menu.choose a question": "Zgjidhni nj\u00eb pyetje.", "part.choose next part.answered": "What do you want to do next?", "part.choose next part.unanswered": "Or, you could:", "part.choose next part.will be locked": "(This part will be locked)", "part.reached dead end": "There's nothing more to do from here.", "part.next part.penalty amount": "(lose {{count}} $t(mark))", "part.marking.counts towards objective": "This part counts towards the objective <strong>\u201c{{objective}}\u201d</strong>.", "part.numberentry.answer not integer or decimal or fraction": "P\u00ebrgjigja jua \u00ebsht\u00eb e gabuar. Duhet t\u00eb shkruani nj\u00eb num\u00ebr t\u00eb plot\u00eb, dhjetor ose thyes\u00eb.", "question": "Pyetje", "question.progress": "Question progress", "question.score feedback.unattempted": "Not attempted", "question.score feedback.attempted": "Attempted", "question.score feedback.score actual.plain": "{{scoreString}}", "question.score feedback.score total actual.plain": "{{score,niceNumber}}/{{marks,niceNumber}}", "question.objectives": "Objectives", "question.penalties": "Penalties", "question.back to previous part": "Kthehu tek k\u00ebrkesa paraardh\u00ebse", "end.print": "Printo transkriptin e provimit", "math.shuffle_together.lists not all the same length": "Not all lists are the same length.", "jme.parse signature.invalid signature string": "Invalid function signature string: {{str}}", "part.custom.expected answer has wrong type": "The expected answer for this part has the wrong type. It should be <code>{{shouldbe}}</code>.", "part.custom.input option has wrong type": "The answer input setting <code>{{option}}</code> has the wrong type. It should be <code>{{shouldbe}}</code>.", "matrix input.size control legend": "Madh\u00ebsia", "matrix input.rows": "Rreshta", "matrix input.columns": "Kolona", "part.jme.error checking numerically": "There was an error numerically checking your answer: {{-message}}", "part.gapfill.cyclic adaptive marking": "There is a cycle in the adaptive marking for this part: <strong>{{name1}}</strong> relies on <strong>{{name2}}</strong>, which eventually relies on <strong>{{name1}}</strong>.", "modal.style.background colour": "Background colour", "modal.style.text colour": "Text colour", "modal.style.text size": "Text size", "modal.style.explanation": "Use these controls to change the appearance of the exam.", "modal.style.reset to defaults": "Reset to defaults", "modal.style.text size preview": "Most text will be this big.", "control.style options": "Opsione t\u00eb paraqitjes", "part.marking.partially correct": "P\u00ebrgjigja \u00ebsht\u00eb pjes\u00ebrisht e sakt\u00eb.", "part.marking.error in adaptive marking": "There was an error in the adaptive marking for this part. Please report this. {{-message}}", "page.skip to content": "Skip to content", "result.learning objective": "Learning objective", "jme.interpreted as": "interpreted as", "jme.script.note.compilation error": "Error compiling note <code>{{name}}</code>: {{-message}}", "jme.script.note.error evaluating note": "Error evaluating note <code>{{name}}</code> - {{-message}}", "jme.script.note.invalid definition": "Invalid note definition: <code>{{source}}</code>. {{-hint}}", "jme.script.note.invalid definition.missing colon": "You might be missing a colon after the name and description", "jme.script.note.invalid definition.description missing closing bracket": "You might be missing a closing bracket", "jme.script.note.empty expression": "The note <code>{{name}}</code> is empty.", "jme.script.error parsing notes": "Error parsing marking script: {{- message}}", "matrix input.cell label": "Row {{row}}, column {{column}}", "control.move to next question": "Move to the next question", "diagnostic.use retry": "Use one retry and try this topic again.", "diagnostic.move to next topic": "Move on to the next topic.", "diagnostic.next step question": "What would you like to do next?", "diagnostic.now assessing topic": "Now assessing {{current_topic}}", "diagnostic.one retry left": "You have 1 retry left", "diagnostic.retries left": "You have {{retries}} retries left.", "diagnostic.percentage completed": "You've completed <strong>{{percentage}}%</strong> of the test.", "diagnostic.test is over": "The test is over.", "diagnostic.passed all lo": "You have passed all learning objectives.", "diagnostic.more work on lo": "You need to do some more work on the following learning objectives: {{los}}.", "diagnostic.move to next question in topic": "Move on to the next question in topic.", "diagnostic.complete": "Complete!", "diagnostic.studying topic": "Studying {{topic}}", "display.answer widget.unknown widget type": "The answer widget type <code>{{name}}</code> is not recognised.", "jme.shunt.expected argument before comma": "Expected to see something between the opening bracket and the comma", "part.waiting for pre submit": "Your answer is being marked. Please wait.", "diagnostic.end test": "End the test.", "page.no stylesheet loaded": "The page's stylesheet file has not loaded.", "modal.confirm": "Confirm", "modal.alert": "Alert", "suspend.resumed header": "Attempt resumed", "jme.vector.value not an array of numbers": "Tried to construct a vector using a value that is not an array of numbers.", "jme.matrix.value not the right type": "Tried to construct a vector using a value of the wrong type.", "jme.subvars.html inserted twice": "An HTML value has been embedded twice. Consider defining a function to generate a new value each time it is used.", "jme.variables.invalid function language": "The language <code>{{language}}</code> is not valid.", "jme.variables.duplicate definition": "There is more than one definition of the variable <code>{{name}}</code>.", "math.random_integer_partition.invalid k": "The size of the partition must be between 1 and {{n}}.", "part.marking.parameter already in scope": "There is a variable named <code>{{name}}</code>, which is also the name of a marking parameter. Please rename the variable.", "part.marking.adaptive variable replacement refers to self": "This part refers to itself in a variable replacement for adaptive marking.", "part.marking.adaptive variable replacement refers to nothing": "This part contains an invalid variable replacement for adaptive marking.", "part.numberentry.display answer wrong type": "The display answer for this part is a value of type <code>{{got_type}}</code>, but should be a <code>{{want_type}}</code>.", "part.matrix.invalid type in prefilled": "There is an invalid value of type <code>{{n}}</code> in the array of pre-filled cells.", "diagnostic.make a choice": "Make a choice", "matrixmath.not square": "This operation only works on a square matrix.", "matrixmath.not invertible": "This operation only works on an invertible matrix.", "matrixmath.gauss-jordan elimination.not enough columns": "There must be at least as many columns as rows.", "question.required extension not available": "This question requires the extension <code>{{-extension}}</code> but it is not available.", "util.formatNumberNotation.unrecognised syntax": "The number formatting syntax <code>{{syntax}}</code> is not recognised.", "worksheet.number of exams to generate": "Number of sheets to generate", "worksheet.starting with id": "Starting with ID", "worksheet.show exam id": "Show sheet ID?", "worksheet.page break between questions": "Page breaks between questions?", "worksheet.page margins": "Page margins (mm)", "worksheet.text size": "Text size (pt)", "worksheet.generate": "Generate", "worksheet.generating exams": "Generating sheets", "worksheet.sheet id": "Sheet ID:", "worksheet.print single": "Print this sheet", "worksheet.print several": "Print these sheets", "worksheet.answer sheets": "Answer sheets", "worksheet.question sheets": "Question sheets", "worksheet.reconfigure": "Generate different sheets", "worksheet.show sheet": "Preview the sheet with ID:", "accessibility statement": "Accessibility statement and guide to adapting Numbas to your needs.", "exam.enter your name": "Your name:", "exam.attempt download security warning": "This exam is configured to allow you to download your data, but it is not running in a secure browser context. You will not be able to download your exam data. Contact your lecturer or teacher for help.", "result.download exam object": "Download your exam data", "control.return to question": "Return to the question", "control.show introduction": "Introduction", "analysis.header": "Analyse attempt data", "analysis.help.upload files": "Upload attempt data files that your students have given you.", "analysis.help.file input label": "Choose student attempt data files, or drag files onto this window.", "analysis.table.total": "Exam totals", "analysis.table.question": "Exam and question totals", "analysis.table.all": "All details", "analysis.student name.anonymous": "No name given", "analysis.expected": "Expected results", "analysis.start time": "Start time", "analysis.maximum": "Maximum Marks", "analysis.file": "File", "analysis.download": "Download", "analysis.delete": "Delete", "analysis.view results": "View results", "analysis.upload files": "Upload files", "analysis.upload more": "Upload more files", "analysis.attempt data": "Attempt data", "analysis.select format": "Select data to show", "analysis.download this table": "Download this table", "analysis.student": "Student Results", "analysis.question key": "Question Key", "analysis.question name": "Question Name", "analysis.group": "Group", "analysis.question": "Question", "analysis.part": "Part", "analysis.gap": "Gap", "analysis.record type": "Record Type", "analysis.score": "Score", "analysis.marks available": "Marks Available", "analysis.percentage": "Percentage", "analysis.answer": "Answer", "analysis.student name": "Student Name", "analysis.summary.no files": "You have not uploaded any files yet.", "analysis.summary.no decrypted files": "You have not uploaded any valid files yet.", "analysis.summary.one file": "One attempt.", "analysis.summary.several files": "{{num_files,niceNumber}} attempts.", "analysis.not secure context": "This page must be opened in a secure browser context. A secure context is either a page served over HTTPS, or a file loaded from your device.", "jme.shunt.pipe right hand takes no arguments": "The expression on the right-hand side of the pipe operator must be a function application.", "question.explore.no parts defined": "There are no parts defined in this question.", "answer": "answer", "worksheet.answersheet show question content": "Show question content in answer sheets?", "modal.confirm end exam": "Write <code>{{endConfirmation}}</code> in the box to confirm:", "modal.end exam button": "End exam", "lightbox.zoom in on image": "Zoom in on this image", "exam.progress": "Progress", "exam.questions answered": "{{numAnsweredQuestions}} of {{numQuestions}} questions answered.", "result.question marks available": "Marks Available", "result.question answered": "Answered?", "control.confirm end.correct": "You may now end the exam.", "control.confirm end.incorrect": "This is not the expected text.", "control.confirm end.password": "end", "jme.typecheck.for in name wrong type": "The name in a <code>for</code> statement must be a name or list of names, not {{type}}.", "jme.makeFast.no fast definition of function": "The function <code>{{name}}</code> here isn't defined in a way that can be made fast.", "part.show feedback": "Show feedback", "part.hide feedback": "Hide feedback", "part.feedback title": "Feedback for {{name}}.", "part.jme.must-match.warning": "Your answer is not in the expected form: {{-message}}", "part.numberentry.write your answer as a fraction": "Write your answer as a fraction.", "question.nav.label": "Question controls", "question.answer saved": "Answer saved", "question.all answers saved": "All answers saved", "analysis.back to results": "Back to results", "analysis.review": "Review", "analysis.review this": "Review this attempt", "analysis.reviewing attempt": "Reviewing attempt by {{student_name}}.", "part.there is new feedback": "The feedback has changed.", "modal.style.colour scheme": "Colour scheme", "modal.style.automatic colour scheme": "Automatic", "modal.style.light colour scheme": "Light", "modal.style.dark colour scheme": "Dark", "modal.style.custom colour scheme": "Custom", "modal.style.main font": "Main font", "modal.style.font.sans serif": "Sans serif", "modal.style.font.serif": "Serif", "modal.style.font.monospace": "Monospace", "modal.style.spacing": "Spacing", "modal.style.font weight": "Font weight", "modal.style.forced colours warning": "Your browser has overridden the colours used in this page because of a setting such as high contrast mode. Changes to the colour scheme settings here might not have any effect.", "modal.style.text preview": "Most text will look like this.", "modal.style.more options": "More options", "modal.style.main colour": "Main (brand) colour", "modal.style.primary colour": "Primary button colour", "modal.style.link colour": "Link colour", "modal.style.success colour": "Success/correct colour", "modal.style.info colour": "Info colour", "modal.style.warning colour": "Warning colour", "modal.style.danger colour": "Danger/incorrect colour", "modal.style.muted colour": "Muted text colour", "modal.style.highlight colour": "Highlight colour", "exam.error loading exam definition": "There was an error while loading the exam definition: {{text}}", "exam.no exam definition": "No exam definition was given.", "jme.typecheck.wrong arguments for anonymous function": "Wrong number of arguments for this anonymous function.", "worksheet.top": "Top", "worksheet.left": "Left", "worksheet.bottom": "Bottom", "worksheet.right": "Right"}}, "de-de": {"translation": {"page.loading": "Wird geladen...", "page.saving": "<p>Wird gespeichert.</p>\n<p>Dies kann einen Moment dauern.</p>", "mathjax.math processing error": "\"{{-message}}\" beim TeXen von <code>{{expression}}</code>", "die.numbas failed": "Ausf\u00fchren von Numbas ist gescheitert", "die.sorry": "Entschuldigung, Numbas hat einen Fehler festgestellt, der eine weitere Bearbeitung unm\u00f6glich macht. Es folgt eine Beschreibung des Fehlers.", "die.error": "Fehler", "modal.ok": "OK", "modal.cancel": "Abbrechen", "exam.exam name": "Testname:", "exam.random seed": "Session ID:", "exam.student name": "Name der/s Teilnehmer/in:", "exam.number of questions": "Anzahl der Aufgaben:", "exam.marks available": "Erreichbare Punkte:", "exam.pass percentage": "Notwendige Punktzahl (in Prozent):", "exam.time allowed": "Erlaubte Zeit:", "exam.passed": "Bestanden", "exam.failed": "Nicht bestanden", "exam.review header": "\u00dcberblick: ", "frontpage.start": "Start", "suspend.paused header": "Angehalten", "suspend.exam suspended": "Der Test wurde unterbrochen. Klicken Sie <em>Fortfahren</em> um weiterzumachen.", "suspend.you can resume": "Sie k\u00f6nnen diese Sitzung beim n\u00e4chsten Start fortsetzen.", "suspend.resume": "Fortfahren", "result.exit": "Test beenden", "result.print": "Ergebnisse ausdrucken", "result.exam summary": "Zusammenfassung des Tests", "result.performance summary": "Zusammenfassung der Leistung", "result.exam start": "Test begonnen:", "result.exam stop": "Test beendet:", "result.time spent": "Vergangene Zeit:", "result.questions attempted": "Bearbeitete Aufgaben:", "result.score": "Punktzahl:", "result.result": "Ergebnis:", "result.question number": "Aufgabennummer", "result.question score": "Punkte", "result.question review title": "Aufgabe nochmals durchsehen", "result.click a question to review": "Klicken Sie auf eine Aufgabennummer, um die Bewertung Ihrer Antworten und, wenn vorhanden, die L\u00f6sungen zu sehen.", "end.exam has finished": "Der Test wurde beendet. Sie k\u00f6nnen das Fenster nun schlie\u00dfen.", "control.confirm leave": "Sie haben den Test noch nicht beendet.", "control.not all questions answered": "Sie haben nicht alle Aufgaben in diesem Test bearbeitet.", "control.not all questions submitted": "Sie haben eine oder mehrere Antworten ge\u00e4ndert, aber nicht eingereicht. \u00dcberpr\u00fcfen Sie bitte, dass alle Antworten eingereicht wurden.", "control.confirm end": "Sind Sie sich sicher den Test zu beenden? Nach Beendigung des Tests k\u00f6nnen Sie ihre Antworten nicht mehr \u00e4ndern.", "control.confirm regen": "Wollen Sie diese Aufgabe mit neuen zuf\u00e4lligen Werten bearbeiten? Wenn Sie OK klicken, gehen alle ihre Antworten und Punkte f\u00fcr die aktuelle Aufgabe verloren.", "control.confirm reveal": "Wollen Sie die L\u00f6sung zu dieser Aufgabe ansehen? Alle bisher erreichten Punkte bleiben erhalten und sie k\u00f6nnen diese Aufgabe sp\u00e4ter nicht nochmal bearbeiten.", "control.proceed anyway": "Dennoch fortfahren?", "control.regen": "Eine andere Aufgabe von dieser Art probieren", "control.submit answer": "Antwort einreichen", "control.submit all parts": "Alle Abschnitte einreichen", "control.submit again": "Erneut einreichen", "control.submit": "Einreichen", "control.previous": "Vorhergehend", "control.next": "N\u00e4chste", "control.advice": "Hinweis", "control.reveal": "Antworten aufdecken", "control.total": "Gesamtergebnis", "control.pause": "Pause", "control.end exam": "Test beenden", "control.back to results": "Zur\u00fcck zu den Ergebnissen", "display.part.jme.error making maths": "Fehler bei der mathematischen Darstellung", "exam.xml.bad root": "Grundelement des Test XML sollte 'exam' sein", "exam.changeQuestion.no questions": "Dieser Test enth\u00e4lt keine Aufgaben! \u00dcberpr\u00fcfen sie die .exam Datei auf Fehler.", "feedback.you were awarded": "Sie erhalten <strong>{{count,niceNumber}}</strong> $t(mark).", "feedback.taken away": "<strong>{{count,niceNumber}}</strong> $t(mark) $t(was) abgezogen.", "jme.tokenise.invalid": "Ung\u00fcltiger Ausdruck: <code>{{expression}}</code>", "jme.shunt.not enough arguments": "Nicht gen\u00fcgend Argumente f\u00fcr die Funktion <code>{{op}}</code>", "jme.shunt.no left bracket in function": "Fehlende linke Klammer in der Funktion oder dem Tupel", "jme.shunt.no left square bracket": "Keine passende linke eckige Klammer", "jme.shunt.no left bracket": "Keine passende linke Klammer", "jme.shunt.no right bracket": "Keine passende rechte Klammer", "jme.shunt.no right square bracket": "Keine passende rechte eckige Klammer", "jme.shunt.missing operator": "Ausdruck kann nicht ausgewertet werden - ein Operator fehlt.", "jme.typecheck.function maybe implicit multiplication": "Funktion <code>{{name}}</code> ist nicht definiert. Meinen Sie <code>{{first}}*{{possibleOp}}(...)</code>?", "jme.typecheck.function not defined": "Funktion <code>{{op}}</code> ist nicht definiert. Ist <code>{{op}}</code> eine Variable, und meinen Sie <code>{{suggestion}}*(...)</code>?", "jme.typecheck.op not defined": "Operation '{{op}}' ist nicht definiert.", "jme.typecheck.no right type definition": "Keine Definition von '{{op}}' mit korrektem Typ gefunden.", "jme.typecheck.no right type unbound name": "Variable <code>{{name}}</code> ist nicht definiert.", "jme.typecheck.map not on enumerable": "<code>map</code> operiert auf einer Liste oder Bereich nicht auf {{type}}", "jme.evaluate.undefined variable": "Variable {{name}} ist nicht definiert", "jme.thtml.not html": "Ein nicht-HTML Wert wurde an einen THTML Konstruktor \u00fcbergeben.", "jme.func.switch.no default case": "Kein Standardwert f\u00fcr Switch Anweisung", "jme.func.listval.invalid index": "Ung\u00fcltiger Listenindex {{index}} f\u00fcr eine Liste der Gr\u00f6\u00dfe {{size}}", "jme.func.listval.not a list": "Objekt ist nicht indizierbar", "jme.func.matrix.invalid row type": "Matrix kann nicht mit Zeilen vom Typ {{type}} erstellt werden", "jme.func.except.continuous range": "'Exklusiv' Operator kann nicht auf kontinuierliche Bereiche angewendet werden.", "jme.matrix.reports bad size": "Matrix meldet ihre Gr\u00f6\u00dfe falsch - es muss ein Fehler im Konstruktor vorliegen", "jme.texsubvars.no right bracket": "Keine passende <code>]</code> in {{op}} Argumenten.", "jme.texsubvars.missing parameter": "Fehlende Parameter in {{op}}: {{parameter}}", "jme.texsubvars.no right brace": "Keine passende <code>}</code> in {{op}}", "jme.user javascript.error": "Fehler in Benutzer-definierter Javascript Funktion <code>{{name}}</code>: {{-message}}", "jme.variables.error making function": "Fehler beim Erstellen der Funktion <code>{{name}}</code>: {{-message}}", "jme.variables.syntax error in function definition": "Syntaxfehler in der Definition der Funktion", "jme.variables.variable not defined": "Variable  <code>{{name}}</code> ist nicht definiert.", "jme.variables.empty definition": "Definition der Variable  <code>{{name}}</code> ist leer.", "jme.variables.circular reference": "Zirkul\u00e4re Definition der Variable <code>{{name}}</code>", "jme.variables.error computing dependency": "Fehler bei der Berechnung der referenzierten Variablen <code>{{name}}</code>", "jme.variables.error evaluating variable": "Fehler beim Auswerten der Variable {{name}}: {{-message}}", "jme.variables.question took too many runs to generate variables": "Es konnte kein g\u00fcltiger Variablensatz in angemessener Zeit gebildet werden.", "jme.display.unknown token type": "Token vom Typ {{type}} konnte nicht mit TeX dargestellt werden", "jme.display.collectRuleset.no sets": "Keine Daten gegeben zum Erstellen von Regels\u00e4tzen!", "jme.display.collectRuleset.set not defined": "Regelsatz {{name}} wurde nicht definiert", "jme.display.simplifyTree.no scope given": "Numbas.jme.display.simplifyTree muss ein Geltungsbereich \u00fcbergeben werden", "math.precround.complex": "Beim Runden darf die Anzahl der Dezimalstellen nicht komplex sein", "math.siground.complex": "Beim Runden darf die Anzahl der signifikanten Ziffern nicht komplex sein", "math.combinations.complex": "Kombinationen k\u00f6nnen nicht von komplexen Zahlen berechnet werden", "math.permutations.complex": "Permutationen k\u00f6nnen nicht von komplexen Zahlen angegeben werden", "math.gcf.complex": "ggT kann nicht von komplexen Zahlen berechnet werden", "math.lcm.complex": "kgV kann nicht von komplexen Zahlen berechnet werden", "math.lt.order complex numbers": "Komplexe Zahlen k\u00f6nnen nicht sortiert werden", "math.choose.empty selection": "Leere Auswahl wurde der Zufallsfunktion \u00fcbergeben", "matrixmath.abs.non-square": "Determinanten k\u00f6nnen nur von quadratischen Matrizen berechnet werden.", "matrixmath.abs.too big": "Determinanten k\u00f6nnen im Moment leider nicht f\u00fcr Matrizen gr\u00f6\u00dfer als 3x3 bestimmt werden.", "matrixmath.mul.different sizes": "Matrizen mit verschiedenen Gr\u00f6\u00dfen k\u00f6nnen nicht multipliziert werden.", "vectormath.cross.not 3d": "Nur Kreuzprodukte von 3-dimensionalen Vektoren m\u00f6glich.", "vectormath.dot.matrix too big": "Das Skalarprodukt von Matrizen ist nur f\u00fcr  $1 \\times N$ or $N \\times 1$ definiert.", "vectormath.cross.matrix too big": "Das Kreuzprodukt von Matrizen ist nur f\u00fcr  $1 \\times N$ or $N \\times 1$ definiert.", "part.with steps answer prompt": "Antwort: ", "part.script.error": "Fehler im Abschnitt {{path}} des Benutzerscripts {{script}}: {{-message}}", "part.marking.steps no matter": "Da Sie die volle Punktzahl f\u00fcr diesen Abschnitt erhalten haben, werden ihre Antworten zu den Teilschritten nicht gez\u00e4hlt.", "part.marking.revealed steps no penalty": "Sie haben die Tipps aufgedeckt.", "part.marking.used variable replacements": "Dieser Abschnitt wurde anhand Ihrer Antworten in vorhergehenden Abschnitten bewertet.", "part.marking.variable replacement part not answered": "Sie m\u00fcssen zuerst {{part}} beantworten.", "part.marking.resubmit because of variable replacement": "Die Bewertung dieses Abschnitts h\u00e4ngt von ihren Antworten aus anderen Abschnitten, die sie g\u00e4ndert haben, ab. Reichen sie diesen Abschnitt erneut ein, um ihren Punktestand zu aktualisieren.", "part.marking.not submitted": "Keine Antwort eingereicht.", "part.marking.did not answer": "Sie haben diese Aufgabe nicht beantwortet.", "part.marking.nothing entered": "Sie haben keine Antwort eingegeben.", "part.marking.incorrect": "Ihre Antwort ist nicht richtig.", "part.marking.correct": "Ihre Antwort ist richtig.", "part.marking.uncaught error": "Fehler bei der Bewertung von: {{-message}}", "part.marking.no result": "Dieser Abschnitt kann nicht bewertet werden.", "part.correct answer": "Erwartete Antwort:", "part.missing type attribute": "{{part}}: Fehlendes Attribut f\u00fcr den Typ des Abschnitts", "part.unknown type": "{{part}}: Unbekanntet Typ des Abschnitts {{type}}", "part.setting not present": "Einstellung '{{property}}' nicht gesetzt", "part.jme.answer missing": "Korrekte Antwort fehlt", "part.jme.answer too long": "Ihre Antwort ist zu lang.", "part.jme.answer too short": "Ihre Antwort ist zu kurz.", "part.jme.answer invalid": "Ihre Antwort ist kein g\u00fcltiger mathematischer Ausdruck.<br/>{{-message}}.", "part.jme.marking.correct": "Ihre Antwort ist numerisch korrekt.", "part.jme.must-have bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.must-have one": "Ihre Antwort muss enthalten: {{strings}}", "part.jme.must-have several": "Ihre Antwort muss alle folgenden Teile enthalten: {{strings}}", "part.jme.not-allowed bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.not-allowed one": "Ihre Antwort darf nicht enthalten: {{strings}}", "part.jme.not-allowed several": "Ihre Antwort darf keinen der folgenden Teile enthalten: {{strings}}", "part.jme.unexpected variable name": "In ihrer Antwort tritt der unerwartete Variablennamen <code>{{name}}</code> auf.", "part.jme.unexpected variable name suggestion": "In ihrer Antwort tritt der unerwartete Variablennamen <code>{{name}}</code> auf. Meinen sie <code>{{suggestion}}</code>?", "part.patternmatch.display answer missing": "Anzeigeantwort fehlt", "part.patternmatch.correct except case": "Ihre Antwort ist bis auf Gro\u00df-/Kleinschreibung richtig.", "part.numberentry.correct except decimal": "Ihre Antwort ist im erlaubten Intervall, aber Dezimalzahlen sind nicht erlaubt.", "part.numberentry.correct except fraction": "Ihre Antwort ist im erlaubten Intervall, aber Br\u00fcche sind nicht erlaubt.", "part.numberentry.answer invalid": "Sie haben keine g\u00fcltige Zahl eingegeben.", "part.numberentry.answer not integer": "Ihre Antwort ist ung\u00fcltig. Sie m\u00fcssen eine ganze Zahl, nicht eine Dezimalzahl angeben.", "part.numberentry.answer not integer or decimal": "Ihre Antwort ist ung\u00fcltig. Sie m\u00fcssen eine ganze oder eine Dezimalzahl angeben.", "part.numberentry.zero sig fig": "Dieser Abschnitt ist so eingestellt, dass die Antwort der Studenten auf null signifikante Stellen gerundet werden. Dies ergibt keinen Sinn.", "part.mcq.options def not a list": "Der Ausdruck, der {{properties}} definiert, ist keine Liste.", "part.mcq.marking matrix string empty": "Die angepasste Bewertungsmatrix ist leer.", "part.mcq.choices missing": "Definition der Auswahlm\u00f6glichkeiten fehlt", "part.mcq.matrix not a number": "Abschnitt {{part}} Zelle ({{row}},{{column}}) der Bewertungsmatrix wird nicht mit einer Zahl belegt", "part.mcq.wrong number of choices": "Sie haben die falsche Anzahl von Auswahlm\u00f6glichkeiten gew\u00e4hlt.", "part.mcq.no choices selected": "Keine Auswahl get\u00e4tigt.", "part.mcq.matrix not a list": "Die Bewertungsmatrix, definiert durch einen JME-Ausdruck, ist keine Liste, was sie sein sollte.", "part.mcq.matrix wrong type": "Element mit ung\u00fcltigem Typ '{{type}}' in Bewertungsmatrix verwendet.", "part.mcq.matrix mix of numbers and lists": "Mix von Zahlen und Listen in Bewertungsmatrix verwendet.", "part.mcq.matrix wrong size": "Bewertungsmatrix hat falsche Gr\u00f6\u00dfe.", "part.mcq.correct choice": "Sie haben eine richtige Antwort gew\u00e4hlt.", "part.matrix.answer invalid": "Ihre Antwort ist nicht g\u00fcltig.", "part.matrix.invalid cell": "Eine oder mehrere Zellen in ihren Antworten sind leer oder ung\u00fcltig.", "part.matrix.some incorrect": "Eine oder mehrere Zellen in ihrer Antwort sind nicht richtig, aber sie haben f\u00fcr den Rest Punkte erhalten.", "part.matrix.empty": "Sie haben keine Antwort angegeben.", "part.matrix.empty cell": "Eine oder mehrere Zellen in Ihrer Antwort sind leer.", "part.matrix.size mismatch": "Der Aufgabensteller erwartet eine bestimmte Dimension der L\u00f6sung. Die korrekte Dimension ist {{correct_dimensions}}, aber Ihre Antwort hat die Dimension {{input_dimensions}}", "part.gapfill.feedback header": "<strong>{{name}}</strong>", "part.extension.not implemented": "Abschnitt hat nicht die Methode <code>{{name}}</code> implementiert.", "question.loaded name mismatch": "Der L\u00f6sungsversuch kann nicht wieder aufgenommen werden - das Paket hat sich seit der letzten Sitzung ge\u00e4ndert.", "question.error": "Aufgabe {{number}}: {{-message}}", "question.preamble.error": "Fehler in der Pr\u00e4ambel : {{-message}}", "question.preamble.syntax error": "Syntaxfehler in der Pr\u00e4ambel", "question.unsupported part type": "Nicht unterst\u00fctzer Abschnittstyp", "question.header": "Aufgabe {{number}}", "question.submit part": "Abschnitt einreichen", "question.show steps": "Zeige Tipps", "question.show steps penalty": "Sie verlieren <strong>{{count,niceNumber}}</strong> $t(mark).", "question.show steps no penalty": "Ihr Punktestand wird nicht ver\u00e4ndert.", "question.show steps already penalised": "Sie haben die Tipps schon gesehen. Sie k\u00f6nnen sie sich noch einmal ohne weiteren Punktabzug ansehen.", "question.hide steps": "Verberge Tipps", "question.hide steps no penalty": "Ihr Punktestand wird nicht ver\u00e4ndert.", "question.advice": "Hinweise", "question.no such part": "Abschnitt {{path}} kann nicht gefunden werden", "question.can not submit": "Antwort kann nicht eingereicht werden - bitte auf Fehler \u00fcberpr\u00fcfen.", "question.answer submitted": "Antwort eingereicht", "question.score feedback.show": "Zeige Wertung", "question.score feedback.hide": "Blende Wertung aus", "question.score feedback.answered total actual": "Punktestand: {{score,niceNumber}}/{{marks,niceNumber}}", "question.score feedback.answered total": "{{marksString}}. Beantwortet.", "question.score feedback.answered actual": "Punktestand: {{scoreString}}", "question.score feedback.answered": "Beantwortet", "question.score feedback.unanswered": "Unbeantwortet", "question.score feedback.unanswered total": "{{marksString}}.", "question.score feedback.correct": "Ihre Antwort ist richtig", "question.score feedback.partial": "Ihre Antwort ist teilweise richtig", "question.score feedback.wrong": "Ihre Antwort ist nicht richtig", "question.selector.unsubmitted changes": "Nicht eingereichte \u00c4nderungen.", "timing.no accumulator": "kein Zeitregister {{name}}", "timing.time remaining": "Verbleibende Zeit:", "xml.could not load": "Ein XML Dokument konnte nicht geladen werden: {{-message}}", "xml.property not number": "Einstellung {{name}} sollte eine Zahl sein, aber ist es nicht ({{value}}), im Knoten {{element}}", "xml.property not boolean": "Einstellung {{name}} sollte ein boolescher Wert sein, aber ist es nicht ({{value}}), im Knoten {{element}}", "xml.error in variable definition": "Fehler in der Definition der Variablen <code>{{name}}</code> ", "scorm.error initialising": "Fehler bei der Initialisierung des SCORM-Protokolls: {{-message}}", "scorm.failed save": "<p>Das Speichern der Daten auf dem Server schlug fehl. Klicken Sie <b>OK</b>, um es nochmal zu versuchen.</p>\n<p>Wenn Sie diese Meldung wiederholt erhalten, pr\u00fcfen Sie die Internetverbindung oder benutzen Sie einen anderen Computer. Ihre vorher eingereichte Antwort wurde erfolgreich gespeichert und wird wierderhergestellt, wenn Sie die Sitzung an einem anderen Computer fortsetzen.</p>\n<p>Wenn diese Meldung dauerhaft erscheint und Sie <em>gar keine</em> Antworten speichern k\u00f6nnen, nehmen Sie bitte mit Ihrer Dozent*in oder Lehrer*in Kontakt auf.</p>", "scorm.no exam suspend data": "Fortsetzung fehlgeschlagen: keine Daten vom unterbrochenen Test.", "scorm.error loading suspend data": "Fehler beim Laden der eingefrorenen Daten: {{-message}}", "scorm.error loading question": "Fehler beim Laden der Aufgabe {{number}}: {{-message}}", "scorm.no question suspend data": "Keine Daten zur unterbrochenen Aufgabe", "scorm.error loading part": "Fehler beim Laden des Abschnitts {{part}}: {{-message}}", "scorm.no part suspend data": "Keine Daten von der Unterbrechung des Abschnitts", "util.product.non list": "Keine Liste \u00fcbergeben an <code>Numbas.util.product</code>", "mark": "Punkt", "was": "wurde", "part": "Abschnitt", "gap": "L\u00fccke", "step": "Schritt", "jme.substituteTree.undefined variable": "Nicht definierte Variable: <code>{{name}}</code>", "jme.user javascript.returned undefined": "Benutzerdefinierte Javascriptfunktion <code>{{name}}</code> liefert <code>undefined</code>.", "part.marking.steps change": "Sie erhielten <strong>{{count,niceNumber}}</strong> $t(mark) f\u00fcr ihre Antworten in den L\u00f6sungsschritten.", "part.marking.revealed steps with penalty": "Sie haben sich Tipps anzeigen lassen. F\u00fcr diesen Abschnitt k\u00f6nnten sie <strong>{{count,niceNumber}}</strong> $t(mark) erhalten. Ihr Punktestand wird entsprechend reduziert.", "part.marking.total score": "Sie erhalten <strong>{{count,niceNumber}}</strong> $t(mark) f\u00fcr diesen Abschnitt.", "part.numberentry.precision type.dp": "Dezimalstelle", "part.numberentry.precision type.dp_plural": "Dezimalstellen", "part.numberentry.precision type.sigfig": "signifikante Stelle", "part.numberentry.precision type.sigfig_plural": "signifikante Stellen", "part.numberentry.give your answer to precision": "Runden Sie ihre Antwort auf {{count,niceNumber}} {{precisionType}}.", "question.unsubmitted changes": "Sie haben ihre Antwort ge\u00e4ndert ohne sie einzureichen. Bitte pr\u00fcfen Sie Ihre Antwort und klicken Sie <strong>Antwort einreichen</strong>.", "question.unsubmitted changes_plural": "Sie haben \u00c4nderungen an ihren Antworten vorgenommen ohne sie einzureichen. Bitte pr\u00fcfen Sie ihre Antworten und klicken Sie <strong>Alle Abschnitte einreichen</strong>.", "util.equality not defined for type": "Der Gleichheitsoperator ist f\u00fcr den Typ {{type}} nicht definiert", "mark_plural": "Punkte", "was_plural": "wurden", "die.script not loaded": "Numbas konnte nicht starten, da die Datei <code>{{file}}</code> nicht geladen wurde. \u00dcberpr\u00fcfen sie, dass sie in <code>scripts.js</code> enthalten ist.", "math.combinations.n less than zero": "Kombinationen konnten nicht berechnet werden: n ist kleiner als null", "math.combinations.k less than zero": "Kombinationen konnten nicht berechnet werden: k ist kleiner als null", "math.combinations.n less than k": "Kombinationen konnten nicht berechnet werden: n ist kleiner als k", "math.permutations.n less than zero": "Permutationen konnten nicht berechnet werden: n ist kleiner als null", "math.permutations.k less than zero": "Permutationen konnten nicht berechnet werden: k ist kleiner als null", "math.permutations.n less than k": "Permutationen konnten nicht berechnet werden: n ist kleiner als k", "part.numberentry.give your answer to precision_0": "Runden sie ihre Antwort auf die n\u00e4chste ganze Zahl.", "mathjax.error": "MathJax Verarbeitungsfehler: {{-message}}", "mathjax.error with context": "MathJax Verarbeitungsfehler bei {{-context}}: {{-message}}", "exam.introduction": "Test Einf\u00fchrung", "exam.feedback": "Test Bewertungsnachricht", "jme.tokenise.keypair key not a string": "Dictionary key muss eine Zeichenkette sein, nicht {{type}}.", "jme.shunt.list mixed argument types": "Kann {{mode}} nicht parsen: Mischung von Dictionary- und Listenelementen", "jme.func.listval.key not in dict": "Dictionary enth\u00e4lt keinen key <code>{{key}}</code>", "part.prompt": "Aufforderung", "part.feedback": "Wertung", "part.numberentry.answer not reduced": "Ihre Antwort ist nicht vollst\u00e4ndig vereinfacht.", "part.numberentry.give your answer as a reduced fraction": "Vereinfachen Sie ihre Antwort vollst\u00e4ndig.", "part.numberentry.negative decimal places": "Dieser Abschnitt ist so eingestellt, dass die Antwort der Studenten auf eine negative Zahl von Dezimalstellen gerundet wird. Dies ergibt keinen Sinn.", "part.mcq.choices": "M\u00f6glichkeiten", "part.mcq.answers": "Antworten", "part.mcq.matrix cell empty": "Abschnitt {{part}} Bewertungsmatrix Zelle ({{row}},{{column}}) ist leer", "part.mcq.matrix jme error": "Abschnitt {{part}} Bewertungsmatrix Zelle ({{row}},{{column}}) f\u00fchrt zu einem JME Fehler: {{-error}}", "question.statement": "Beschreibung", "ruleset.circular reference": "Zirkul\u00e4re Referenz in der Definition des Regelsatzes <code>{{name}}</code>", "ruleset.set not defined": "Regelsatz {{name}} wurde nicht definiert", "jme.evaluate.no scope given": "Numbas.jme.evaluate muss ein Geltungsbereich \u00fcbergeben werden", "answer.number.not a number": "Sie haben keine g\u00fcltige Zahl eingegeben.", "answer.number.fractions not allowed": "Sie d\u00fcrfen keine Bruchzahl eingeben.", "answer.jme.invalid expression": "{{-message}}", "answer.matrix.fractions not allowed": "Sie d\u00fcrfen keine Bruchzahlen eingeben.", "answer.matrix.some cell not a number": "Eine oder mehrere der Zellen in ihrer Antwort enthalten keine g\u00fcltige(n) Zahle(n).", "exam.enter password": "Passwort:", "exam.password.correct": "Dieses Passwort ist korrekt. Sie k\u00f6nnen den Test beginnen.", "exam.password.incorrect": "Dieses Passwort ist falsch.", "frontpage.scorm.lms not connected": "Der Test l\u00e4uft im Standalone-Modus. Ihre Antworten und Bewertungen werden nicht gespeichert!", "result.question review": "\u00dcberblick", "control.confirm regen no marks": "Wollen Sie diese Aufgabe mit neuen zuf\u00e4lligen Werten bearbeiten?", "control.confirm reveal no marks": "Wollen Sie die L\u00f6sung zu dieser Aufgabe ansehen?", "jme.tokenise.invalid near": "Ung\u00fcltiger Ausdruck: <code>{{expression}}</code> an Position {{position}} nahe bei <code>{{nearby}}</code>", "jme.tokenise.number.object not complex": "Ung\u00fcltiges Objekt an die Zahlerzeugung \u00fcbergeben.", "jme.subvars.null substitution": "Leere Variablen-Substitution: <code>$t(left brace){{str}}$t(right brace)", "jme.type.type already registered": "Der Daten-Typ {{type}} wurde bereits vergeben und kann nicht erneut vergeben werden.", "jme.type.no cast method": "Kann nicht automatisch von {{from}} nach {{to}} konvertieren.", "jme.display.simplifyTree.empty expression": "Ausdruck ist leer", "jme.display.simplifyTree.stuck in a loop": "Vereinfachung h\u00e4ngt in Schleife: <code>{{expr}}</code>", "math.niceNumber.undefined": "Zahl erwartet, aber <code>undefined</code> erhalten", "math.rangeToList.zero step size": "Kann einen Bereich mit Schrittanzahl Null nicht in eine Liste konvertieren", "part.error": "{{path}}: {{-message}}", "part.marking.revealed steps": "Sie haben sich die Schritte anzeigen lassen.", "part.marking.maximum scaled down": "Die maximal erzielbare Punktzahl in diesem Teil betr\u00e4gt <strong>{{count,niceNumber}}</strong> $t(mark). Ihre Punktzahl wird entsprechend verringert.", "part.marking.minimum score applied": "Die Mindestpunktzahl f\u00fcr diesen Teil betr\u00e4gt <strong>{{score,niceNumber}}</strong>.", "part.marking.maximum score applied": "Die H\u00f6chstpunktzahl f\u00fcr diesen Teil betr\u00e4gt <strong>{{score,niceNumber}}</strong>.", "part.marking.error in marking script": "Ein Fehler im Bewertungs-Algorithmus f\u00fcr diesen Teil ist aufgetreten. Bitte melden Sie dies. {{-message}}", "part.marking.no result after replacement": "Dieser Abschnitt konnte nicht unter Verwendung ihrer Antworten zu den vorigen Abschnitten bewertet werden.", "part.marking.missing required note": "Der Bewertungs-Algorithmus definiert die marking note <code>{{note}}</code> nicht", "marking.apply.not a list": "Das erste Argument in <code>apply</code> muss vom Typ Liste sein, ist es aber nicht", "marking.apply marking script.script not found": "Beurteilungs-Skript <code>{{name}}</code> nicht gefunden", "marking.note.compilation error": "Kompilieren der marking note <code>{{name}}</code> fehlgeschlagen: {{-message}}", "marking.note.error evaluating note": "Auswertung der marking note <code>{{name}}</code> fehlgeschlagen - {{-message}}", "marking.note.invalid definition": "Ung\u00fcltig definierte marking note:<code>{{source}}</code>. {{-hint}}", "marking.note.invalid definition.missing colon": "M\u00f6glicherweise fehlt ein Doppelpunkt nach dem Namen und der Beschreibung", "marking.note.invalid definition.description missing closing bracket": "M\u00f6glicherweise fehlt eine schlie\u00dfende Klammer", "marking.note.empty expression": "Die marking note <code>{{name}}</code> ist leer.", "marking.script.error parsing notes": "Fehler beim Parsen des Bewertungsskripts: {{- message}}", "part.feedback out of date": "Das Feedback basiert auf ihrer zuletzt eingereichten Antwort. Reichen Sie ihre ver\u00e4nderte Antwort ein um ein aktualisiertes Feedback zu erhalten.", "part.jme.invalid value generator expression": "Ung\u00fcltiger Ausdruck zur Erzeugung von Werten f\u00fcr die Variable <code>{{name}}</code>: {{-message}}", "part.mcq.incorrect choice": "Sie haben eine falsche Antwort ausgew\u00e4hlt.", "part.matrix.not all cells same precision": "Sie haben nicht jede Zelle ihrer Antwort mit derselben Genauigkeit angegeben.", "part.gapfill.error marking gap": "Bewertungsfehler bei {{name}}: {{-message}}", "part.custom.empty setting": "Kein Wert angegeben.", "part.custom.unrecognised input type": "Nicht erkannter Einstellungs-Typ <code>{{input_type}}</code>", "part.custom.error evaluating input option": "Fehler beim Auswerten der Eingabe-Option <code>{{option}}</code>: {{-error}}", "part.custom.input option missing": "Definition der Eingabe-Option <code>{{option}}</code> fehlt.", "part.custom.error evaluating setting": "Fehler beim Auswerten der Einstellung <code>{{setting}}</code>: {{-error}}", "question.error creating question": "Fehler beim Erzeugen der Frage {{number}}: {{-message}}", "question.score feedback.not marked": "Nicht bewertet", "question.score feedback.partially answered": "Teilweise beantwortet", "question.score feedback.score total": "{{marksString}}", "question.score feedback.score actual": "Punktestand: {{scoreString}}", "question.score feedback.score total actual": "Punktestand: {{score,niceNumber}}/{{marks,niceNumber}}", "variable.error in variable definition": "Fehler in der Definition der Variable <code>{{name}}</code>", "left brace": "{", "right brace": "}", "extension.not found": "Erweiterung <code>{{name}}</code> konnte nicht geladen werden.", "control.toggle navigation menu": "Navigationsmen\u00fc ein-/ausschalten", "part.input title": "Antwort zum Teil {{name}}", "part.correct answer title": "Erwartete Antwort zum Teil {{name}}", "part.jme.must-match.failed": "Ihre Antwort ist nicht im richtigen Format.", "question.score feedback.none": "", "control.submit part.confirm remove next parts": "<p>Einer oder mehrere der folgenden Teile h\u00e4ngen von Ihrer Antwort in diesem Abschnitt ab. Wenn Sie diesen Abschnitt erneut einreichen, werden diese folgenden Teile ung\u00fcltig und entfernt. Dies kann nicht r\u00fcckg\u00e4ngig gemacht werden.</p>\n<p>M\u00f6chten Sie diesen Abschnitt erneut einreichen?</p>", "control.back to menu": "Zur\u00fcck zum Menu", "display.error making html": "Fehler beim Erzeugen von HTML in {{contextDescription}}: {{-message}}", "jme.subvars.error compiling": "{{-message}} in <code>{{expression}}</code>", "jme.variables.empty name": "Einer Fragevariable wurde kein Name zugewiesen.", "jme.calculus.unknown derivative": "Kann <code>{{tree}}</code> nicht ableiten.", "math.order complex numbers": "Kann komplexe Zahlen nicht anordnen", "menu.choose a question": "W\u00e4hlen Sie eine Frage aus.", "part.choose next part.answered": "Was m\u00f6chten Sie als n\u00e4chstes machen?", "part.choose next part.unanswered": "Sie k\u00f6nnten auch:", "part.choose next part.will be locked": "(Dieser Teil wird gesperrt werden)", "part.reached dead end": "An diese Stelle gibt es nichts mehr zu tun.", "part.next part.penalty amount": "(Abzug von {{count}} $t(mark))", "part.marking.counts towards objective": "Dieser Abschnitt z\u00e4hlt f\u00fcr das Ziel <strong>\u201c{{objective}}\u201d</strong>.", "part.numberentry.answer not integer or decimal or fraction": "Ihre Antwort ist ung\u00fcltig. Sie m\u00fcssen eine ganze Zahl, eine Dezimalzahl oder eine Bruchzahl eingeben.", "question": "Frage", "question.progress": "Fortschritt bei dieser Frage:", "question.score feedback.unattempted": "Kein Versuch unternommen", "question.score feedback.attempted": "Versuch unternommen", "question.score feedback.score actual.plain": "{{scoreString}}", "question.score feedback.score total actual.plain": "{{score,niceNumber}}/{{marks,niceNumber}}", "question.objectives": "Ziele", "question.penalties": "Abz\u00fcge", "question.back to previous part": "Zur\u00fcck zum vorherigen Teil", "end.print": "Drucke eine Abschrift des Tests", "math.shuffle_together.lists not all the same length": "Nicht alle Listen haben die gleiche L\u00e4nge.", "jme.parse signature.invalid signature string": "Ung\u00fcltiger Signaturstring der Funktion: {{str}}", "part.custom.expected answer has wrong type": "Die erwartete Antwort f\u00fcr diesen Abschnitt hat den falschen Typ. Es sollte <code>{{shouldbe}}</code> sein.", "part.custom.input option has wrong type": "Die Eingabe-Einstellung <code>{{option}}</code> hat den falschen Typ. Es sollte <code>{{shouldbe}}</code> sein.", "matrix input.size control legend": "Gr\u00f6\u00dfe", "matrix input.rows": "Zeilen", "matrix input.columns": "Spalten", "part.jme.error checking numerically": "Beim numerischen \u00dcberpr\u00fcfen der Antwort ist ein Fehler aufgetreten: {{-message}}", "part.gapfill.cyclic adaptive marking": "Es gibt eine zirkul\u00e4re Referenz in der adaptiven Bewertung dieses Abschnitts: <strong>{{name1}}</strong> ben\u00f6tigt <strong>{{name2}}</strong>, was sich letztlich auf <strong>{{name1}}</strong> bezieht.", "modal.style.background colour": "Hintergrundfarbe", "modal.style.text colour": "Schriftfarbe", "modal.style.text size": "Schriftgr\u00f6\u00dfe", "modal.style.explanation": "Benutzen Sie diese Einstellungen, um das Erscheinungsbild des Tests zu ver\u00e4ndern.", "modal.style.reset to defaults": "Auf die Vorgabewerte zur\u00fccksetzen", "modal.style.text size preview": "Der Text wird meist diese Gr\u00f6\u00dfe haben.", "control.style options": "Anzeigeeinstellungen", "part.marking.partially correct": "Ihre Antwort ist teilweise korrekt.", "part.marking.error in adaptive marking": "Fehler bei der adaptiven Bewertung f\u00fcr diesen Abschnitt. Bitte melden Sie dies. {{-message}}", "page.skip to content": "Weiter zum Inhalt", "result.learning objective": "Lernziel", "jme.interpreted as": "interpretiert als", "jme.script.note.compilation error": "Fehler beim Kompilieren, <code>{{name}}</code>: {{-message}}", "jme.script.note.error evaluating note": "Fehler beim Auswerten, <code>{{name}}</code>: {{-message}}", "jme.script.note.invalid definition": "Ung\u00fcltige Definition: <code>{{source}}</code>. {{-hint}}", "jme.script.note.invalid definition.missing colon": "Vielleicht fehlt ein Doppelpunkt nach Name und Beschreibung", "jme.script.note.invalid definition.description missing closing bracket": "Vielleicht fehlt eine schlie\u00dfende Klammer", "jme.script.note.empty expression": "Die marking note <code>{{name}}</code> ist leer.", "jme.script.error parsing notes": "Fehler beim Parsen des Bewertungsskripts: {{- message}}", "matrix input.cell label": "Zeile {{row}}, Spalte {{column}}", "control.move to next question": "Weiter zur n\u00e4chsten Frage", "diagnostic.use retry": "Nutze einen Versuch und probiere das Thema noch einmal.", "diagnostic.move to next topic": "Weiter zum n\u00e4chsten Thema.", "diagnostic.next step question": "Was m\u00f6chtest du als n\u00e4chstes machen?", "diagnostic.now assessing topic": "{{current_topic}} wird gepr\u00fcft", "diagnostic.one retry left": "Du hast noch einen Versuch \u00fcbrig", "diagnostic.retries left": "Du hast noch {{retries}} Versuche \u00fcbrig.", "diagnostic.percentage completed": "Du hast bereits <strong>{{percentage}}%</strong> des Tests geschafft.", "diagnostic.test is over": "Die Pr\u00fcfung ist beendet.", "diagnostic.passed all lo": "Du hast alle Lernziele erreicht.", "diagnostic.more work on lo": "Du solltest noch folgende Lernziele weiter \u00fcben: {{los}}.", "diagnostic.move to next question in topic": "Weiter zur n\u00e4chsten Frage im Thema.", "diagnostic.complete": "Geschafft!", "diagnostic.studying topic": "Aktuelles Thema: {{topic}}", "display.answer widget.unknown widget type": "Der Antwort-Widget-Typ <code>{{name}}</code> wurde nicht erkannt.", "jme.shunt.expected argument before comma": "Eintrag zwischen \u00f6ffnender Klammer und Komma erwartet", "part.waiting for pre submit": "Die Antwort wird jetzt bewertet. Bitte warten.", "diagnostic.end test": "Den Test beenden.", "page.no stylesheet loaded": "Die Stylesheet-Datei der Seite wurde nicht geladen.", "modal.confirm": "Best\u00e4tigen", "modal.alert": "Achtung", "suspend.resumed header": "Versuch wieder aufgenommen", "jme.vector.value not an array of numbers": "Ein Vektor kann nur aus einem Array von Zahlen konstruiert werden.", "jme.matrix.value not the right type": "Falscher Typ, um einen Vektor zu konstruieren.", "jme.subvars.html inserted twice": "Ein HTML-Wert wurde zweifach eingebettet. Definieren Sie gegebenenfalls eine Funktion, um jeweils einen neuen Wert zu erzeugen.", "jme.variables.invalid function language": "Die Sprache <code>{{language}}</code> ist ung\u00fcltig.", "jme.variables.duplicate definition": "Es gibt mehrere Definitionen der Variable <code>{{name}}</code>.", "math.random_integer_partition.invalid k": "Die Gr\u00f6\u00dfe der Partition muss zwischen 1 und {{n}} liegen.", "part.marking.parameter already in scope": "Es gibt eine Variable <code>{{name}}</code>, was auch die Bezeichnung eines Bewertungsparameters ist. Bitte benennen Sie die Variable um.", "part.marking.adaptive variable replacement refers to self": "Dieser Teil enth\u00e4lt eine selbstbez\u00fcgliche Variablenersetzung f\u00fcr adaptive Bewertung.", "part.marking.adaptive variable replacement refers to nothing": "Dieser Teil enth\u00e4lt eine ung\u00fcltige Variablenersetzung f\u00fcr adaptive Bewertung.", "part.numberentry.display answer wrong type": "Die angezeigte Antwort f\u00fcr diesen Teil ist vom Typ <code>{{got_type}}</code> statt vom Typ <code>{{want_type}}</code>.", "part.matrix.invalid type in prefilled": "Das Array der vorausgef\u00fcllten Zellen enth\u00e4lt einen ung\u00fcltigen Wert vom Typ <code>{{n}}</code>.", "diagnostic.make a choice": "Auswahl treffen", "matrixmath.not square": "Diese Operation kann nur auf eine quadratische Matrix angewendet werden.", "matrixmath.not invertible": "Diese Operation kann nur auf eine invertierbare Matrix angewendet werden.", "matrixmath.gauss-jordan elimination.not enough columns": "Es muss mindestens so viele Spalten wie Zeilen geben.", "question.required extension not available": "Diese Frage ben\u00f6tigt die Erweiterung <code>{{-extension}}</code>, aber diese ist nicht verf\u00fcgbar.", "util.formatNumberNotation.unrecognised syntax": "Die Formatierungssyntax <code>{{syntax}}</code> wurde nicht erkannt.", "worksheet.number of exams to generate": "Anzahl von zu erzeugenden Bl\u00e4ttern", "worksheet.starting with id": "Beginne mit ID", "worksheet.show exam id": "Zeige Blatt-ID?", "worksheet.page break between questions": "Seitenumbruch zwischen Fragen?", "worksheet.page margins": "Seitenr\u00e4nder (mm)", "worksheet.text size": "Schriftgr\u00f6\u00dfe (pt)", "worksheet.generate": "Erzeugen", "worksheet.generating exams": "Erzeuge Bl\u00e4tter", "worksheet.sheet id": "Blatt-ID:", "worksheet.print single": "Dieses Blatt drucken", "worksheet.print several": "Diese Bl\u00e4tter drucken", "worksheet.answer sheets": "Antwortbl\u00e4tter", "worksheet.question sheets": "Aufgabenbl\u00e4tter", "worksheet.reconfigure": "Erzeuge andere Bl\u00e4tter", "worksheet.show sheet": "Vorschau f\u00fcr Blatt ID:", "accessibility statement": "Barrierefreiheit und Anpassungsm\u00f6glichkeiten von Numbas.", "exam.enter your name": "Ihr Name:", "exam.attempt download security warning": "Dieser Test ist zwar so eingerichtet, dass Sie Ihre Daten herunterladen k\u00f6nnen, aber l\u00e4uft nicht in einem sicheren Browser-Kontext. Sie k\u00f6nnen daher Ihre Testdaten nicht herunterladen. Nehmen Sie gegebenenfalls mit Ihrer Dozent*in Kontakt auf.", "result.download exam object": "Testdaten herunterladen", "control.return to question": "Zur Frage zur\u00fcckkehren", "control.show introduction": "Einf\u00fchrung", "analysis.header": "Versuchsdaten analysieren", "analysis.help.upload files": "Versuchsdaten-Dateien Ihrer Studierenden hochladen.", "analysis.help.file input label": "W\u00e4hlen Sie Versuchsdaten-Dateien aus, oder ziehen Sie Dateien hierher.", "analysis.table.total": "Punktzahl des Tests", "analysis.table.question": "Punktzahlen des Tests und der Fragen", "analysis.table.all": "Alle Details", "analysis.student name.anonymous": "Kein Name angegeben", "analysis.expected": "Erwartete Ergebnisse", "analysis.start time": "Startzeitpunkt", "analysis.maximum": "Maximale Punktzahl", "analysis.file": "Datei", "analysis.download": "Download", "analysis.delete": "L\u00f6schen", "analysis.view results": "Ergebnisse anzeigen", "analysis.upload files": "Dateien hochladen", "analysis.upload more": "Weitere Dateien hochladen", "analysis.attempt data": "Daten des Versuchs", "analysis.select format": "Daten zur Anzeige ausw\u00e4hlen", "analysis.download this table": "Diese Tabelle herunterladen", "analysis.student": "Ergebnisse der Studierenden", "analysis.question key": "Frage-Schl\u00fcssel", "analysis.question name": "Name der Frage", "analysis.group": "Gruppe", "analysis.question": "Frage", "analysis.part": "Teil", "analysis.gap": "L\u00fccke", "analysis.record type": "Eintragstyp", "analysis.score": "Punktzahl", "analysis.marks available": "Bewertung verf\u00fcgbar", "analysis.percentage": "Prozentsatz", "analysis.answer": "Antwort", "analysis.student name": "Name des Studierenden", "analysis.summary.no files": "Sie haben noch keine Dateien hochgeladen.", "analysis.summary.no decrypted files": "Sie haben noch keine g\u00fcltigen Dateien hochgeladen.", "analysis.summary.one file": "Ein Versuch.", "analysis.summary.several files": "{{num_files,niceNumber}} Versuche.", "analysis.not secure context": "Diese Seite muss \u00fcber eine HTTPS-Verbindung oder als Datei auf Ihrem eigenen Ger\u00e4t geladen werden.", "jme.shunt.pipe right hand takes no arguments": "Der Ausdruck auf der rechten Seite des Pipe-Operators muss eine Funktion sein.", "question.explore.no parts defined": "In dieser Frage wurden keine Teile angelegt.", "answer": "Antwort", "worksheet.answersheet show question content": "Inhalt der Frage in Antwortbl\u00e4ttern zeigen?", "modal.confirm end exam": "Schreiben Sie zur Best\u00e4tigung <code>{{endConfirmation}}</code> in die Box:", "modal.end exam button": "Test beenden", "lightbox.zoom in on image": "Hereinzoomen", "exam.progress": "Fortschritt", "exam.questions answered": "{{numAnsweredQuestions}} von {{numQuestions}} Fragen beantwortet.", "result.question marks available": "Bewertungen verf\u00fcgbar", "result.question answered": "Beantwortet?", "control.confirm end.correct": "Sie k\u00f6nnen den Test nun beenden.", "control.confirm end.incorrect": "Dies ist nicht der erwartete Text.", "control.confirm end.password": "Beenden", "jme.typecheck.for in name wrong type": "Der Name in einem <code>for</code>-Statement muss ein Name oder eine Liste von Namen sein, nicht {{type}}.", "jme.makeFast.no fast definition of function": "Die Funktion <code>{{name}}</code> hier ist nicht so definiert, dass sie schnell ausgef\u00fchrt werden kann.", "part.show feedback": "Feedback anzeigen", "part.hide feedback": "Feedback verbergen", "part.feedback title": "Feedback f\u00fcr {{name}}.", "part.jme.must-match.warning": "Ihre Antwort hat nicht die erwartete Form: {{-message}}", "part.numberentry.write your answer as a fraction": "Schreiben Sie Ihre Antwort als Bruchzahl.", "question.nav.label": "Fragen-Navigation", "question.answer saved": "Antwort gespeichert", "question.all answers saved": "Alle Antworten gespeichert", "analysis.back to results": "Zur\u00fcck zu den Ergebnissen", "analysis.review": "\u00dcberpr\u00fcfung", "analysis.review this": "Versuch \u00fcberpr\u00fcfen", "analysis.reviewing attempt": "\u00dcberpr\u00fcfung: Versuch von {{student_name}}.", "part.there is new feedback": "Die R\u00fcckmeldung wurde ge\u00e4ndert.", "modal.style.colour scheme": "Colour scheme", "modal.style.automatic colour scheme": "Automatic", "modal.style.light colour scheme": "Light", "modal.style.dark colour scheme": "Dark", "modal.style.custom colour scheme": "Custom", "modal.style.main font": "Main font", "modal.style.font.sans serif": "Sans serif", "modal.style.font.serif": "Serif", "modal.style.font.monospace": "Monospace", "modal.style.spacing": "Spacing", "modal.style.font weight": "Font weight", "modal.style.forced colours warning": "Your browser has overridden the colours used in this page because of a setting such as high contrast mode. Changes to the colour scheme settings here might not have any effect.", "modal.style.text preview": "Most text will look like this.", "modal.style.more options": "More options", "modal.style.main colour": "Main (brand) colour", "modal.style.primary colour": "Primary button colour", "modal.style.link colour": "Link colour", "modal.style.success colour": "Success/correct colour", "modal.style.info colour": "Info colour", "modal.style.warning colour": "Warning colour", "modal.style.danger colour": "Danger/incorrect colour", "modal.style.muted colour": "Muted text colour", "modal.style.highlight colour": "Highlight colour", "exam.error loading exam definition": "There was an error while loading the exam definition: {{text}}", "exam.no exam definition": "No exam definition was given.", "jme.typecheck.wrong arguments for anonymous function": "Wrong number of arguments for this anonymous function.", "worksheet.top": "Top", "worksheet.left": "Left", "worksheet.bottom": "Bottom", "worksheet.right": "Right"}}, "in-id": {"translation": {"page.loading": "Memuat...", "page.saving": "<p>Menyimpan.</p>\n<p>Hal ini akan memerlukan waktu beberapa detik.</p>", "mathjax.math processing error": "Terjadi kesalahan \"{{-message}}\" ketika membuat teks dari sintaks <code>{{expression}}</code>", "die.numbas failed": "Numbas telah gagal", "die.sorry": "Maaf, Numbas mengalami masalah sehingga proses tidak dapat dilanjutkan. Berikut ini adalah deskripsi dari masalah tersebut.", "die.error": "Kesalahan", "modal.ok": "Ok", "modal.cancel": "Batalkan", "exam.exam name": "Nama Ujian:", "exam.random seed": "ID Sesi:", "exam.student name": "Nama Siswa:", "exam.number of questions": "Jumlah Soal:", "exam.marks available": "Total Skor:", "exam.pass percentage": "Persentase Kelulusan:", "exam.time allowed": "Waktu yang Diizinkan:", "exam.passed": "Lulus", "exam.failed": "Gagal", "exam.review header": "Review: ", "frontpage.start": "Mulai", "suspend.paused header": "Terhenti Sementara", "suspend.exam suspended": "Ujian ditangguhkan. Tekan <em>Lanjutkan</em> untuk melanjutkan.", "suspend.you can resume": "Anda akan bisa melanjutkan sesi ini setelah Anda memulai aktivitas ini.", "suspend.resume": "Lanjutkan", "result.exit": "Keluar Ujian", "result.print": "Cetak ringkasan hasil ujian ini", "result.exam summary": "Ringkasan Ujian", "result.performance summary": "Ulasan Kinerja", "result.exam start": "Ujian Mulai:", "result.exam stop": "Ujian Berhenti:", "result.time spent": "Waktu yang Terpakai:", "result.questions attempted": "Soal yang Dicoba:", "result.score": "Nilai:", "result.result": "Hasil:", "result.question number": "Nomor Soal", "result.question score": "Skor", "result.question review title": "Ulas soal ini", "result.click a question to review": "Klik pada nomor soal untuk melihat bagaimana jawaban Anda dinilai, dan jika tersedia, solusi lengkapnya juga.", "end.exam has finished": "Ujian telah selesai. Anda dapat menutup jendela ini.", "control.confirm leave": "Anda belum menyelesaikan ujian.", "control.not all questions answered": "Anda belum menjawab setiap soal pada ujian ini.", "control.not all questions submitted": "Anda telah mengubah satu atau lebih jawaban namun belum menyerahkannya. Silakan periksa dan serahkan tiap soal.", "control.confirm end": "Apakah Anda yakin ingin mengakhiri ujian? Setelah mengakhiri, Anda tidak dapat mengubah satu pun jawaban.", "control.confirm regen": "Apakah Anda ingin mengacak ulang soal ini? Jika Anda klik OK, semua jawaban dan nilai untuk soal ini akan hilang.", "control.confirm reveal": "Apakah Anda ingin melihat jawaban dari soal ini? Semua nilai yang telah Anda terima sebelumnya akan dikunci dan Anda tidak bisa menjawab soal ini setelahnya.", "control.proceed anyway": "Tetap lanjutkan?", "control.regen": "Coba soal lain yang mirip dengan soal ini", "control.submit answer": "Kirim jawaban", "control.submit all parts": "Kirim semua bagian", "control.submit again": "Kirim lagi", "control.submit": "Kirim", "control.previous": "Sebelumnya", "control.next": "Selanjutnya", "control.advice": "Saran", "control.reveal": "Tunjukkan jawaban", "control.total": "Total", "control.pause": "Hentikan sementara", "control.end exam": "Selesaikan Ujian", "control.back to results": "Kembali ke hasil ujian", "display.part.jme.error making maths": "Kesalahan dalam membuat tampilan matematika", "exam.xml.bad root": "Elemen dasar XML ujian haruslah 'exam'", "exam.changeQuestion.no questions": "Ujian ini tidak mengandung soal! Cek kesalahan pada file .exam.", "feedback.you were awarded": "Anda mendapatkan <strong>{{count,niceNumber}}</strong> $t(mark).", "feedback.taken away": "<strong>{{count,niceNumber}}</strong> $t(mark) $t(was) telah diambil.", "jme.shunt.not enough arguments": "Tidak cukup argumen untuk operasi <code>{{op}}</code>", "jme.shunt.no left bracket in function": "Tidak ada kurung buka yang sesuai dalam aplikasi fungsi atau tupel", "jme.shunt.no left square bracket": "Tidak ada kurung buka yang sesuai", "jme.shunt.no left bracket": "Tidak ada kurung buka yang sesuai", "jme.shunt.no right bracket": "Tidak ada kurung tutup yang sesuai", "jme.shunt.no right square bracket": "Tidak ada kurung siku tutup yang sesuai untuk mengakhiri daftar", "jme.shunt.missing operator": "Ekspresi tidak bisa diproses -- kehilangan sebuah operator.", "jme.typecheck.function maybe implicit multiplication": "Fungsi <code>{{name}}</code> tidak didefinisikan. Apakah maksud Anda adalah <code>{{first}}*{{possibleOp}}(...)</code>?", "jme.typecheck.function not defined": "Fungsi <code>{{op}}</code> tidak didefinisikan. Apakah <code>{{op}}</code> suatu variabel sehingga maksud Anda adalah <code>{{suggestion}}*(...)</code>?", "jme.typecheck.op not defined": "Operasi '{{op}}' tidak terdefinisi.", "jme.typecheck.no right type definition": "Tidak ada definisi  '{{op}}' dari tipe yang benar.", "jme.typecheck.no right type unbound name": "Variabel <code>{{name}}</code> tidak terdefinisi.", "jme.typecheck.map not on enumerable": "Operasi <code>map</code> harus bekerja pada list atau range, bukan {{type}}", "jme.evaluate.undefined variable": "Variabel {{name}} tidak didefinisikan", "jme.thtml.not html": "Meneruskan nilai non-HTML ke konstruktor THTML.", "jme.func.switch.no default case": "Tidak ada default case untuk Switch statement", "jme.func.listval.invalid index": "Indeks list {{index}} tidak valid untuk list dengan ukuran {{size}}", "jme.func.listval.not a list": "Objek tidak dapat disubskribkan", "jme.func.matrix.invalid row type": "Tidak dapat membuat matriks dari baris bertipe {{type}}", "jme.func.except.continuous range": "Tidak dapat menggunakan operator 'except' untuk range kontinu.", "jme.matrix.reports bad size": "Ukuran matriks salah - pasti terdapat kesalahan pada fungsi konstruktor", "jme.texsubvars.no right bracket": "Tidak ada <code>]</code> dalam argumen {{op}}.", "jme.texsubvars.missing parameter": "Parameter hilang dalam {{op}}: {{parameter}}", "jme.texsubvars.no right brace": "Tidak ada <code>}</code> dalam {{op}}", "jme.user javascript.error": "Kesalahan di fungsi javascript buatan pengguna <code>{{name}}</code>: {{-message}}", "jme.variables.error making function": "Kesalahan dalam membuat fungsi <code>{{name}}</code>: {{-message}}", "jme.variables.syntax error in function definition": "Kesalahan sintaks pada definisi fungsi", "jme.variables.variable not defined": "Variabel <code>{{name}}</code> tidak didefinisikan.", "jme.variables.empty definition": "Definisi dari variabel <code>{{name}}</code> kosong.", "jme.variables.circular reference": "Referensi variabel sirkular dalam definisi <code>{{name}}</code>", "jme.variables.error computing dependency": "Error dalam menghitung variabel referensi <code>{{name}}</code>", "jme.variables.error evaluating variable": "Error dalam mengevaluasi variabel {{name}}: {{-message}}", "jme.variables.question took too many runs to generate variables": "Serangkaian variabel pertanyaan yang valid tidak dibuat tepat waktu.", "jme.display.unknown token type": "Tidak dapat membuat teks dari token bertipe {{type}}", "jme.display.collectRuleset.no sets": "Tidak ada set yang diberikan ke collectRuleset!", "jme.display.collectRuleset.set not defined": "Ruleset {{name}} tidak didefinisikan", "jme.display.simplifyTree.no scope given": "Numbas.jme.display.simplifyTree harus diberi Scope", "math.precround.complex": "Tidak dapat melakukan pembulatan ke bilangan kompleks dengan tempat desimal ini", "math.siground.complex": "Tidak dapat melakukan pembulatan ke bilangan kompleks dengan angka penting ini", "math.combinations.complex": "Tidak dapat menghitung kombinasi dari bilangan kompleks", "math.permutations.complex": "Tidak dapat menghitung permutasi dari bilangan kompleks", "math.gcf.complex": "Tidak dapat menghitung FPB dari bilangan kompleks", "math.lcm.complex": "Tidak dapat menghitung KPK dari bilangan kompleks", "math.lt.order complex numbers": "Tidak dapat mengurutkan bilangan kompleks", "math.choose.empty selection": "Pilihan kosong diberikan ke fungsi random", "matrixmath.abs.non-square": "Tidak dapat menghitung determinan matriks yang tidak berbentuk persegi.", "matrixmath.abs.too big": "Maaf, belum bisa menghitung determinan matriks berukuran lebih besar dari 3x3.", "matrixmath.mul.different sizes": "Tidak dapat mengalikan matriks dengan ukuran berbeda.", "vectormath.cross.not 3d": "Hanya bisa menentukan perkalian silang untuk vektor berdimensi tiga.", "vectormath.dot.matrix too big": "Tidak dapat menghitung hasil kali titik untuk matriks yang tidak berukuran $ 1 \\times N $ atau $ N \\times 1 $.", "vectormath.cross.matrix too big": "Tidak dapat menghitung hasil kali silang untuk matriks yang tidak berukuran $1 \\times N$ atau $N \\times 1$.", "part.with steps answer prompt": "Jawaban: ", "part.script.error": "Kesalahan pada bagian {{path}} skrip kustom {{script}}: {{-message}}", "part.marking.steps no matter": "Karena Anda menerima nilai penuh untuk bagian tersebut, jawaban Anda atas langkah-langkah tersebut tidak dihitung.", "part.marking.used variable replacements": "Bagian ini dinilai menggunakan jawaban Anda pada bagian sebelumnya.", "part.marking.variable replacement part not answered": "Anda harus menjawab {{part}} terlebih dahulu.", "part.marking.resubmit because of variable replacement": "Penilaian pada bagian ini bergantung pada jawaban Anda untuk bagian lain, yang telah Anda ubah. Kirimkan bagian ini lagi untuk memperbarui skor Anda.", "part.marking.not submitted": "Tidak ada jawaban yang terkirim.", "part.marking.did not answer": "Anda tidak menjawab pertanyaan ini.", "part.marking.nothing entered": "Anda tidak memasukkan jawaban.", "part.marking.incorrect": "Jawaban Anda salah.", "part.marking.correct": "Jawaban Anda benar.", "part.marking.uncaught error": "Kesalahan ketika menilai: {{-message}}", "part.correct answer": "Jawaban yang diharapkan:", "part.missing type attribute": "{{part}}: Atribut tipe bagian hilang", "part.unknown type": "{{part}}: Tipe bagian {{type}} tidak dikenal", "part.setting not present": "Properti '{{property}}' tidak diatur", "part.jme.answer missing": "Jawaban yang benar tidak ditemukan", "part.jme.answer too long": "Jawaban Anda terlalu panjang.", "part.jme.answer too short": "Jawaban Anda terlalu pendek.", "part.jme.answer invalid": "Jawaban Anda bukanlah ekspresi matematika yang valid.<br/>{{-message}}.", "part.jme.marking.correct": "Jawaban Anda benar secara numerik.", "part.jme.must-have bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.must-have one": "Jawaban Anda harus mengandung: {{strings}}", "part.jme.must-have several": "Jawaban Anda harus mengandung seluruhnya dari: {{strings}}", "part.jme.not-allowed bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.not-allowed one": "Jawaban Anda tidak boleh mengandung: {{strings}}", "part.jme.not-allowed several": "Jawaban Anda tidak boleh mengandung salah satu dari: {{strings}}", "part.jme.unexpected variable name": "Jawaban Anda diinterpretasikan dengan menggunakan nama variabel <code>{{name}}</code> yang tidak diharapkan.", "part.jme.unexpected variable name suggestion": "Jawaban Anda diinterpretasikan dengan menggunakan nama variabel <code>{{name}}</code> yang tidak diharapkan. Apakah maksud Anda adalah <code>{{suggestion}}</code>?", "part.patternmatch.display answer missing": "Jawaban tampilan hilang", "part.patternmatch.correct except case": "Jawaban Anda benar, kecuali besar kecilnya huruf.", "part.numberentry.correct except decimal": "Jawaban Anda berada dalam rentang yang benar, namun bilangan desimal tidak diperbolehkan.", "part.numberentry.correct except fraction": "Jawaban Anda berada dalam rentang yang benar, namun bentuk pecahan tidak dibolehkan.", "part.numberentry.answer invalid": "Anda tidak memasukkan angka yang sesuai.", "part.numberentry.answer not integer": "Jawaban Anda tidak sesuai. Anda harus memasukkan bilangan bulat, bukan desimal.", "part.numberentry.answer not integer or decimal": "Jawaban Anda tidak sesuai. Anda harus memasukkan bilangan asli atau desimal.", "part.numberentry.zero sig fig": "Bagian ini diatur untuk membulatkan jawaban siswa menjadi nol angka penting, yang tidak ada artinya.", "part.mcq.options def not a list": "Ekspresi yang mendefinisikan {{properties}} bukan merupakan suatu list.", "part.mcq.marking matrix string empty": "Ekspresi matriks penilaian kustom kosong.", "part.mcq.choices missing": "Definisi dari pilihan-pilihan tidak ada", "part.mcq.matrix not a number": "Bagian {{part}} pada sel matriks penilaian ({{row}},{{column}}) bukan merupakan angka", "part.mcq.wrong number of choices": "Anda memilih jumlah pilihan yang salah.", "part.mcq.no choices selected": "Tidak ada pilihan yang terpilih.", "part.mcq.matrix not a list": "Matriks penilaian yang didefinisikan oleh ekspresi JME bukan merupakan suatu list, dan seharusnya merupakan list.", "part.mcq.matrix wrong type": "Elemen bertipe '{{type}}' yang tidak valid digunakan dalam matriks penilaian.", "part.mcq.matrix mix of numbers and lists": "Campuran dari angka dan list digunakan dalam matriks penilaian.", "part.mcq.matrix wrong size": "Matriks penilaian memiliki ukuran yang salah.", "part.mcq.correct choice": "Anda telah memilih jawaban yang benar.", "part.matrix.answer invalid": "Jawaban Anda tidak sesuai.", "part.matrix.invalid cell": "Satu atau lebih sel pada jawaban Anda kosong atau tidak sesuai.", "part.matrix.some incorrect": "Satu atau lebih sel pada jawaban Anda salah, namun Anda telah mendapatkan nilai untuk sel lainnya.", "part.matrix.empty": "Anda belum memasukkan jawaban.", "part.matrix.empty cell": "Satu atau lebih sel pada jawaban Anda kosong.", "part.matrix.size mismatch": "Penulis soal belum mengizinkan siswa untuk menentukan dimensi jawaban mereka, tetapi jawaban yang benar adalah {{correct_dimensions}} sedangkan input jawaban adalah {{input_dimensions}}", "part.gapfill.feedback header": "<strong>{{name}}</strong>", "part.extension.not implemented": "Bagian ini belum mengimplementasikan metode <code>{{name}}</code>.", "question.loaded name mismatch": "Tidak dapat melanjutkan upaya ini - paket telah berubah sejak sesi terakhir.", "question.error": "Soal {{number}}: {{-message}}", "question.preamble.error": "Kesalahan dalam pembukaan: {{-message}}", "question.preamble.syntax error": "Kesalahan sintaks pada pembukaan", "question.unsupported part type": "Tipe bagian tidak didukung", "question.header": "Soal {{number}}", "question.submit part": "Kirim bagian", "question.show steps": "Tunjukkan langkah", "question.show steps penalty": "Anda akan kehilangan <strong>{{count,niceNumber}}</strong> $t(mark).", "question.show steps no penalty": "Skor Anda tidak akan terpengaruh.", "question.show steps already penalised": "Anda telah melihat langkah penyelesaian. Anda dapat melihatnya lagi tanpa hukuman lebih lanjut.", "question.hide steps": "Sembunyikan langkah", "question.hide steps no penalty": "Skor Anda tidak akan terpengaruh.", "question.advice": "Saran", "question.no such part": "Tidak dapat menemukan bagian {{path}}", "question.can not submit": "Tidak dapat mengirim jawaban - periksa bila ada kesalahan.", "question.answer submitted": "Jawaban terkirim", "question.score feedback.show": "Tunjukkan umpan balik", "question.score feedback.hide": "Sembunyikan umpan balik", "question.score feedback.answered": "Terjawab", "question.score feedback.unanswered": "Tidak terjawab", "question.score feedback.correct": "Jawaban Anda benar", "question.score feedback.partial": "Jawaban Anda benar sebagian", "question.score feedback.wrong": "Jawaban Anda salah", "question.selector.unsubmitted changes": "Perubahan belum dikirim.", "timing.no accumulator": "tidak ada akumulator waktu {{name}}", "timing.time remaining": "Waktu tersisa:", "xml.could not load": "Tidak dapat memuat dokumen XML: {{-message}}", "xml.property not number": "Properti {{name}} harus merupakan angka, tetapi bukan ({{value}}), pada node {{element}}", "xml.property not boolean": "Properti {{name}} harus merupakan boolean, tetapi bukan ({{value}}), pada node {{element}}", "scorm.error initialising": "Kesalahan dalam menginisialisasi protokol SCORM: {{-message}}", "scorm.failed save": "<p>Permintaan untuk menyimpan data ke server gagal. Tekan <b>OK</b> untuk mencoba lagi.</p>\n<p>Jika Anda mendapatkan pesan ini berulang kali, periksa koneksi internet Anda atau gunakan perangkat yang lain. Jawaban yang Anda kirimkan sebelumnya telah berhasil disimpan dan akan dipulihkan jika Anda melanjutkan sesi ini di perangkat lain.</p>\n<p>Jika pesan ini terus muncul dan Anda tidak dapat menyimpan <em>jawaban</em> apa pun, harap hubungi dosen atau guru Anda.</p>", "scorm.no exam suspend data": "Gagal untuk melanjutkan: tidak ada data penundaan ujian.", "scorm.error loading suspend data": "Terjadi kesalahan saat memuat data penangguhan: {{-message}}", "scorm.error loading question": "Terjadi kesalahan saat memuat soal {{number}}: {{-message}}", "scorm.no question suspend data": "Tidak ada pertanyaan yang menangguhkan data", "scorm.error loading part": "Terjadi kesalahan saat memuat bagian {{part}}: {{-message}}", "scorm.no part suspend data": "Tidak ada bagian yang menangguhkan data", "util.product.non list": "Meneruskan non-list ke <code>Numbas.util.product</code>", "mark": "nilai", "was": "adalah", "part": "bagian", "gap": "celah", "step": "langkah", "jme.substituteTree.undefined variable": "Variabel tidak didefinisikan: <code>{{name}}</code>", "jme.user javascript.returned undefined": "Fungsi javascript buatan pengguna <code>{{name}}</code> mengembalikan <code>undefined</code>.", "part.marking.steps change": "Anda mendapatkan <strong>{{count,niceNumber}}</strong> $t(mark) atas jawaban Anda untuk langkah-langkah tersebut.", "part.marking.total score": "Anda mendapatkan <strong>{{count,niceNumber}}</strong> $t(mark) untuk bagian ini.", "part.numberentry.precision type.dp": "angka di belakang koma", "part.numberentry.precision type.dp_plural": "angka di belakang koma", "part.numberentry.precision type.sigfig": "angka penting", "part.numberentry.precision type.sigfig_plural": "angka penting", "part.numberentry.give your answer to precision": "Bulatkan jawaban Anda ke {{count,niceNumber}} {{precisionType}}.", "question.unsubmitted changes": "Anda telah mengubah jawaban Anda, tetapi belum mengirimkannya. Harap periksa jawaban Anda, lalu tekan tombol <strong>Kirim jawaban</strong>.", "question.unsubmitted changes_plural": "Anda telah mengubah jawaban Anda, tetapi belum mengirimkannya. Silakan periksa jawaban Anda untuk setiap bagian dan kemudian tekan tombol <strong>Kirim semua bagian</strong>.", "util.equality not defined for type": "Kesamaan tidak didefinisikan untuk tipe {{type}}", "mark_plural": "nilai", "was_plural": "adalah", "die.script not loaded": "Numbas tidak dapat memulai karena file <code>{{file}}</code> tidak dimuat. Periksa apakah file tersebut termasuk dalam <code>scripts.js</code>.", "math.combinations.n less than zero": "Tidak bisa menghitung kombinasi: n lebih kecil dari nol", "math.combinations.k less than zero": "Tidak bisa menghitung kombinasi: k lebih kecil dari nol", "math.combinations.n less than k": "Tidak bisa menghitung kombinasi: n lebih kecil dari k", "math.permutations.n less than zero": "Tidak bisa menghitung permutasi: n lebih kecil dari nol", "math.permutations.k less than zero": "Tidak bisa menghitung permutasi: k lebih kecil dari nol", "math.permutations.n less than k": "Tidak bisa menghitung permutasi: n lebih kecil dari k", "part.numberentry.give your answer to precision_0": "Bulatkan jawaban Anda ke bilangan bulat terdekat.", "mathjax.error": "Kesalahan pemrosesan MathJax: {{-message}}", "mathjax.error with context": "Kesalahan pemrosesan MathJax dalam {{-context}}: {{-message}}", "exam.introduction": "Pengantar ujian", "exam.feedback": "Pesan umpan balik ujian", "jme.tokenise.keypair key not a string": "Key dari dictionary haruslah berupa string, bukan {{type}}.", "jme.shunt.list mixed argument types": "Tidak dapat mem-parse {{mode}}: campuran dari dictionary dan list element", "jme.func.listval.key not in dict": "Dictionary tidak mengandung key <code>{{key}}</code>", "part.prompt": "segera", "part.feedback": "umpan balik", "part.numberentry.answer not reduced": "Jawaban Anda tidak dalam bentuk paling sederhana.", "part.numberentry.give your answer as a reduced fraction": "Buat jawaban Anda ke dalam bentuk paling sederhana.", "part.numberentry.negative decimal places": "Bagian ini disiapkan untuk membulatkan jawaban siswa ke-n angka di belakang koma, dengan n bilangan negatif, yang tidak ada artinya.", "part.mcq.choices": "pilihan", "part.mcq.answers": "jawaban", "part.mcq.matrix cell empty": "Bagian {{part}} pada sel matriks penilaian ({{row}},{{column}}) kosong", "part.mcq.matrix jme error": "Bagian {{part}} pada sel matriks penilaian ({{row}},{{column}}) memberikan error JME: {{-error}}", "question.statement": "Pernyataan", "ruleset.circular reference": "Referensi sirkular dalam definisi ruleset <code>{{name}}</code>", "ruleset.set not defined": "Ruleset {{name}} tidak didefinisikan", "jme.evaluate.no scope given": "Numbas.jme.evaluate harus diberikan Scope", "answer.number.not a number": "Jawaban Anda bukan angka yang valid.", "answer.number.fractions not allowed": "Anda tidak diperbolehkan memasukkan pecahan.", "answer.jme.invalid expression": "{{-message}}", "answer.matrix.fractions not allowed": "Anda tidak diperbolehkan memasukkan pecahan.", "answer.matrix.some cell not a number": "Satu atau lebih sel pada jawaban Anda bukan angka yang valid.", "exam.enter password": "Kata Sandi:", "exam.password.correct": "Kata sandi benar. Anda dapat memulai ujian.", "exam.password.incorrect": "Kata sandi salah.", "frontpage.scorm.lms not connected": "Ujian ini berjalan di moda mandiri. Jawaban dan nilai Anda tidak akan tersimpan!", "result.question review": "Ulasan", "control.confirm regen no marks": "Apakah Anda ingin mengacak ulang soal ini?", "control.confirm reveal no marks": "Apakah Anda ingin melihat jawaban dari soal ini?", "jme.tokenise.invalid near": "Ekspresi tidak valid: <code>{{expression}}</code> pada posisi {{position}} dekat <code>{{nearby}}</code>", "jme.tokenise.number.object not complex": "Objek yang tidak valid diteruskan ke konstruktor bilangan.", "jme.subvars.null substitution": "Substitusi variabel kosong: <code>$t(left brace){{str}}$t(right brace)", "jme.type.type already registered": "Tipe data {{type}} sudah terdaftar sehingga tidak dapat didaftarkan lagi.", "jme.type.no cast method": "Tidak dapat mengonversi dari {{from}} ke {{to}}.", "jme.display.simplifyTree.empty expression": "Pernyataan kosong", "jme.display.simplifyTree.stuck in a loop": "Penyederhana terjebak dalam loop: <code>{{expr}}</code>", "math.niceNumber.undefined": "Mengharapkan suatu bilangan, tetapi mendapatkan <code>undefined</code>", "math.rangeToList.zero step size": "Tidak dapat mengonversi range dengan step berukuran nol ke suatu list.", "part.error": "{{path}}: {{-message}}", "part.marking.revealed steps": "Anda membuka langkahnya.", "part.marking.maximum scaled down": "Skor maksimum yang bisa Anda dapatkan pada bagian ini adalah <strong>{{count,niceNumber}}</strong> $t(mark). Skor Anda akan diskalakan ke bawah.", "part.marking.minimum score applied": "Skor minimum untuk bagian ini adalah <strong>{{score,niceNumber}}</strong>.", "part.marking.maximum score applied": "Skor maksimum untuk bagian ini adalah <strong>{{score,niceNumber}}</strong>.", "part.marking.error in marking script": "Terdapat kesalahan algoritma penilaian pada bagian ini. Harap laporkan hal ini. {{-message}}", "part.marking.no result after replacement": "Bagian ini tidak bisa dinilai dari jawaban Anda pada bagian sebelumnya.", "part.marking.missing required note": "Algoritma penilaian tidak mendefinisikan catatan <code>{{note}}</code>", "marking.apply.not a list": "Argumen pertama untuk <code>apply</code> haruslah merupakan list, dan bukan", "marking.apply marking script.script not found": "Skrip penilaian <code>{{name}}</code> tidak ditemukan", "marking.note.compilation error": "Kesalahan dalam mengompilasi catatan <code>{{name}}</code>: {{-message}}", "marking.note.error evaluating note": "Kesalahan dalam mengevaluasi catatan <code>{{name}}</code> - {{-message}}", "marking.note.invalid definition": "Definisi catatan tidak valid: <code>{{source}}</code>. {{-hint}}", "marking.note.invalid definition.missing colon": "Anda mungkin lupa titik dua setelah nama dan deskripsi", "marking.note.invalid definition.description missing closing bracket": "Anda mungkin lupa kurung tutup", "marking.note.empty expression": "Catatan <code>{{name}}</code> kosong.", "marking.script.error parsing notes": "Kesalahan dalam mem-parse skrip penilaian: {{-message}}", "part.feedback out of date": "Umpan balik ini berdasarkan jawaban yang terakhir dikirim. Kirim jawaban yang telah diubah untuk mendapatkan umpan balik baru.", "part.jme.invalid value generator expression": "Ekspresi pembuat nilai tidak valid untuk variabel <code>{{name}}</code>: {{-message}}", "part.mcq.incorrect choice": "Anda telah memilih jawaban yang salah.", "part.matrix.not all cells same precision": "Anda belum memberikan presisi yang sama untuk tiap sel pada jawaban Anda.", "part.gapfill.error marking gap": "Kesalahan dalam menilai {{name}}: {{-message}}", "part.custom.empty setting": "Tidak ada nilai yang diberikan.", "part.custom.unrecognised input type": "Jenis pengaturan <code>{{input_type}}</code> tidak dikenal", "part.custom.error evaluating input option": "Kesalahan dalam mengevaluasi opsi input <code>{{option}}</code>: {{-error}}", "part.custom.input option missing": "Definisi dari opsi input <code>{{option}}</code> hilang.", "part.custom.error evaluating setting": "Kesalahan dalam mengevaluasi pengaturan <code>{{setting}}</code>: {{-error}}", "question.error creating question": "Kesalahan dalam membuat pertanyaan {{number}}: {{-message}}", "question.score feedback.not marked": "Tidak dinilai", "question.score feedback.partially answered": "Dijawab sebagian", "question.score feedback.score total": "{{marksString}}", "question.score feedback.score actual": "Skor: {{scoreString}}", "question.score feedback.score total actual": "Skor: {{score,niceNumber}}/{{marks,niceNumber}}", "variable.error in variable definition": "Kesalahan dalam definisi variabel <code>{{name}}</code>", "left brace": "{", "right brace": "}", "extension.not found": "Tidak dapat memuat ekstensi<code>{{name}}</code>.", "control.toggle navigation menu": "Ganti menu navigasi", "part.input title": "Jawaban untuk bagian {{name}}", "part.correct answer title": "Jawaban yang diharapkan untuk bagian {{name}}", "part.jme.must-match.failed": "Jawaban Anda tidak dalam bentuk yang tepat.", "control.submit part.confirm remove next parts": "<p>Satu atau lebih bagian berikutnya bergantung pada jawaban Anda untuk bagian ini. Mengirimkan bagian ini lagi akan membatalkan bagian-bagian tersebut dan menghapusnya dari pertanyaan terkait. Hal ini tidak dapat dibatalkan.</p>\n<p> Apakah Anda ingin mengirimkan bagian ini lagi? </p>", "control.back to menu": "Kembali ke menu", "display.error making html": "Kesalahan membuat HTML di {{contextDescription}}: {{-message}}", "jme.subvars.error compiling": "{{-message}} pada <code>{{expression}}</code>", "jme.variables.empty name": "Variabel soal belum diberi nama.", "jme.calculus.unknown derivative": "Tidak mengetahui bagaimana membedakan <code>{{tree}}</code>", "math.order complex numbers": "Tidak dapat mengurutkan bilangan kompleks", "menu.choose a question": "Pilih soal.", "part.choose next part.answered": "Apa yang akan Anda lakukan selanjutnya?", "part.choose next part.unanswered": "Atau, Anda bisa:", "part.choose next part.will be locked": "(Bagian ini akan dikunci)", "part.reached dead end": "Tidak ada lagi yang bisa dilakukan sekarang.", "part.next part.penalty amount": "(kehilangan {{count}} $t(mark))", "part.marking.counts towards objective": "Bagian ini diperhitungkan dalam tujuan <strong>\u201c{{objective}}\u201d</strong>.", "part.numberentry.answer not integer or decimal or fraction": "Jawaban Anda tidak sesuai. Anda harus memasukkan bilangan asli, desimal, atau pecahan.", "question": "Soal", "question.progress": "Perkembangan soal:", "question.score feedback.unattempted": "Tidak dicoba", "question.score feedback.attempted": "Dicoba", "question.score feedback.score actual.plain": "{{scoreString}}", "question.score feedback.score total actual.plain": "{{score,niceNumber}}/{{marks,niceNumber}}", "question.objectives": "Tujuan", "question.penalties": "Hukuman", "question.back to previous part": "Kembali ke bagian sebelumnya", "end.print": "Cetak transkrip ujian Anda", "math.shuffle_together.lists not all the same length": "Tidak semua list memiliki panjang sama.", "jme.parse signature.invalid signature string": "String untuk signature fungsi tidak valid: {{str}}", "part.custom.expected answer has wrong type": "Jawaban yang diharapkan untuk bagian ini memiliki tipe yang salah. Seharusnya <code>{{shouldbe}}</code>.", "part.custom.input option has wrong type": "Jenis setelan input jawaban <code>{{option}}</code> salah. Seharusnya <code>{{shouldbe}}</code>.", "matrix input.size control legend": "Ukuran", "matrix input.rows": "Baris", "matrix input.columns": "Kolom", "part.jme.error checking numerically": "Terdapat kesalahan saat memeriksa jawaban Anda secara numerik: {{-message}}", "part.gapfill.cyclic adaptive marking": "Terdapat cycle dalam penilaian adaptif untuk bagian ini: <strong>{{name1}}</strong> bergantung pada <strong>{{name2}}</strong>, yang pada akhirnya bergantung lagi pada <strong>{{name1}}</strong>.", "modal.style.background colour": "Warna latar belakang", "modal.style.text colour": "Warna teks", "modal.style.text size": "Ukuran teks", "modal.style.explanation": "Gunakan kontrol ini untuk mengubah tampilan ujian.", "modal.style.reset to defaults": "Reset ke default", "modal.style.text size preview": "Mayoritas teks akan sebesar ini.", "control.style options": "Pengaturan Tampilan", "part.marking.partially correct": "Jawaban Anda benar sebagian.", "part.marking.error in adaptive marking": "Terdapat kesalahan dalam penilaian adaptif pada bagian ini. Harap laporkan hal ini. {{-message}}", "page.skip to content": "Lewati ke konten", "result.learning objective": "Tujuan pembelajaran", "jme.interpreted as": "diinterpretasikan sebagai", "jme.script.note.compilation error": "Kesalahan dalam mengompilasi catatan <code>{{name}}</code>: {{-message}}", "jme.script.note.error evaluating note": "Kesalahan dalam mengevaluasi catatan <code>{{name}}</code> - {{-message}}", "jme.script.note.invalid definition": "Definisi catatan tidak benar: <code>{{source}}</code>. {{-hint}}", "jme.script.note.invalid definition.missing colon": "Anda mungkin melupakan titik dua setelah nama dan deskripsi", "jme.script.note.invalid definition.description missing closing bracket": "Anda mungkin melupakan kurung tutup", "jme.script.note.empty expression": "Catatan <code>{{name}}</code> kosong.", "jme.script.error parsing notes": "Kesalahan dalam mem-parse skrip penilaian: {{-message}}", "matrix input.cell label": "Baris {{row}}, kolom {{column}}", "control.move to next question": "Beralih ke soal selanjutnya", "diagnostic.use retry": "Gunakan satu percobaan dan coba topik ini lagi.", "diagnostic.move to next topic": "Beralih ke topik selanjutnya.", "diagnostic.next step question": "Apa yang ingin Anda lakukan selanjutnya?", "diagnostic.now assessing topic": "Sekarang melakukan asesmen {{current_topic}}", "diagnostic.one retry left": "Anda memiliki 1 percobaan lagi", "diagnostic.retries left": "Anda memiliki {{retries}} percobaan lagi.", "diagnostic.percentage completed": "Anda menyelesaikan <strong>{{percentage}}%</strong> dari ujian.", "diagnostic.test is over": "Ujian berakhir.", "diagnostic.passed all lo": "Anda lulus seluruh tujuan pembelajaran.", "diagnostic.more work on lo": "Anda perlu melakukan beberapa pekerjaan lagi pada tujuan pembelajaran berikut: {{los}}.", "diagnostic.move to next question in topic": "Beralih pada pertanyaan selanjutnya dalam topik ini.", "diagnostic.complete": "Selesai!", "diagnostic.studying topic": "Mempelajari {{topic}}", "display.answer widget.unknown widget type": "Jenis widget jawaban <code>{{name}}</code> tidak dikenali.", "jme.shunt.expected argument before comma": "Diharapkan untuk melihat sesuatu di antara kurung buka dan tanda koma", "part.waiting for pre submit": "Jawaban Anda sedang dinilai. Mohon tunggu.", "diagnostic.end test": "Akhir ujian.", "page.no stylesheet loaded": "File stylesheet halaman belum dimuat.", "modal.confirm": "Konfirmasi", "modal.alert": "Peringatan", "suspend.resumed header": "Upaya dilanjutkan", "jme.vector.value not an array of numbers": "Mencoba membuat vektor menggunakan nilai yang bukan merupakan array dari bilangan.", "jme.matrix.value not the right type": "Mencoba membuat vektor menggunakan nilai dengan tipe yang salah.", "jme.subvars.html inserted twice": "Suatu nilai HTML disematkan dua kali. Pertimbangkan untuk mendefinisikan fungsi untuk menghasilkan nilai baru setiap kali digunakan.", "jme.variables.invalid function language": "Bahasa <code>{{language}}</code> tidak valid.", "jme.variables.duplicate definition": "Terdapat lebih dari satu definisi untuk variabel <code>{{name}}</code>.", "math.random_integer_partition.invalid k": "Ukuran partisi harus berada di antara 1 dan {{n}}.", "part.marking.parameter already in scope": "Terdapat variabel bernama <code>{{name}}</code>, yang juga merupakan nama dari suatu parameter penilaian. Harap namai ulang variabel tersebut.", "part.marking.adaptive variable replacement refers to self": "Bagian ini mengacu pada dirinya sendiri dalam pengganti variabel untuk penilaian adaptif.", "part.marking.adaptive variable replacement refers to nothing": "Bagian ini mengandung pengganti variabel yang tidak valid untuk penilaian adaptif.", "part.numberentry.display answer wrong type": "Tampilan jawaban untuk bagian ini merupakan nilai bertipe <code>{{got_type}}</code>, yang seharusnya <code>{{want_type}}</code>.", "part.matrix.invalid type in prefilled": "Terdapat nilai tidak valid bertipe <code>{{n}}</code> dalam array sel yang diisi sebelumnya.", "diagnostic.make a choice": "Silakan pilih", "matrixmath.not square": "Operasi ini hanya berlaku untuk matriks persegi.", "matrixmath.not invertible": "Operasi ini hanya berlaku untuk matriks terinverskan.", "matrixmath.gauss-jordan elimination.not enough columns": "Setidaknya banyaknya kolom harus sebanyak baris.", "question.required extension not available": "Soal ini memerlukan ekstensi <code>{{-extension}}</code> yang tidak tersedia.", "util.formatNumberNotation.unrecognised syntax": "Sintaks pemformatan bilangan <code>{{syntax}}</code> tidak dikenal.", "worksheet.number of exams to generate": "Banyaknya lembar kerja yang akan dibuat", "worksheet.starting with id": "Dimulai dengan ID", "worksheet.show exam id": "Tampilkan ID lembar kerja?", "worksheet.page break between questions": "Pisah halaman antar soal?", "worksheet.page margins": "Margin halaman (mm)", "worksheet.text size": "Ukuran teks (pt)", "worksheet.generate": "Buat", "worksheet.generating exams": "Membuat lembar kerja", "worksheet.sheet id": "ID Lembar Kerja:", "worksheet.print single": "Print lembar kerja ini", "worksheet.print several": "Print lembar kerja ini", "worksheet.answer sheets": "Lembar kerja untuk jawaban", "worksheet.question sheets": "Lembar kerja untuk soal", "worksheet.reconfigure": "Buat lembar kerja berbeda", "worksheet.show sheet": "Pratinjau lembar kerja dengan ID:", "accessibility statement": "Accessibility statement and guide to adapting Numbas to your needs.", "exam.enter your name": "Your name:", "exam.attempt download security warning": "This exam is configured to allow you to download your data, but it is not running in a secure browser context. You will not be able to download your exam data. Contact your lecturer or teacher for help.", "result.download exam object": "Download your exam data", "control.return to question": "Return to the question", "control.show introduction": "Introduction", "analysis.header": "Analyse attempt data", "analysis.help.upload files": "Upload attempt data files that your students have given you.", "analysis.help.file input label": "Choose student attempt data files, or drag files onto this window.", "analysis.table.total": "Exam totals", "analysis.table.question": "Exam and question totals", "analysis.table.all": "All details", "analysis.student name.anonymous": "No name given", "analysis.expected": "Expected results", "analysis.start time": "Start time", "analysis.maximum": "Maximum Marks", "analysis.file": "File", "analysis.download": "Download", "analysis.delete": "Delete", "analysis.view results": "View results", "analysis.upload files": "Upload files", "analysis.upload more": "Upload more files", "analysis.attempt data": "Attempt data", "analysis.select format": "Select data to show", "analysis.download this table": "Download this table", "analysis.student": "Student Results", "analysis.question key": "Question Key", "analysis.question name": "Question Name", "analysis.group": "Group", "analysis.question": "Question", "analysis.part": "Part", "analysis.gap": "Gap", "analysis.record type": "Record Type", "analysis.score": "Score", "analysis.marks available": "Marks Available", "analysis.percentage": "Percentage", "analysis.answer": "Answer", "analysis.student name": "Student Name", "analysis.summary.no files": "You have not uploaded any files yet.", "analysis.summary.no decrypted files": "You have not uploaded any valid files yet.", "analysis.summary.one file": "One attempt.", "analysis.summary.several files": "{{num_files,niceNumber}} attempts.", "analysis.not secure context": "This page must be opened in a secure browser context. A secure context is either a page served over HTTPS, or a file loaded from your device.", "jme.shunt.pipe right hand takes no arguments": "The expression on the right-hand side of the pipe operator must be a function application.", "question.explore.no parts defined": "There are no parts defined in this question.", "answer": "answer", "worksheet.answersheet show question content": "Show question content in answer sheets?", "modal.confirm end exam": "Write <code>{{endConfirmation}}</code> in the box to confirm:", "modal.end exam button": "End exam", "lightbox.zoom in on image": "Zoom in on this image", "exam.progress": "Progress", "exam.questions answered": "{{numAnsweredQuestions}} of {{numQuestions}} questions answered.", "result.question marks available": "Marks Available", "result.question answered": "Answered?", "control.confirm end.correct": "You may now end the exam.", "control.confirm end.incorrect": "This is not the expected text.", "control.confirm end.password": "end", "jme.typecheck.for in name wrong type": "The name in a <code>for</code> statement must be a name or list of names, not {{type}}.", "jme.makeFast.no fast definition of function": "The function <code>{{name}}</code> here isn't defined in a way that can be made fast.", "part.show feedback": "Show feedback", "part.hide feedback": "Hide feedback", "part.feedback title": "Feedback for {{name}}.", "part.jme.must-match.warning": "Your answer is not in the expected form: {{-message}}", "part.numberentry.write your answer as a fraction": "Write your answer as a fraction.", "question.nav.label": "Question controls", "question.answer saved": "Answer saved", "question.all answers saved": "All answers saved", "analysis.back to results": "Back to results", "analysis.review": "Review", "analysis.review this": "Review this attempt", "analysis.reviewing attempt": "Reviewing attempt by {{student_name}}.", "part.there is new feedback": "The feedback has changed.", "modal.style.colour scheme": "Colour scheme", "modal.style.automatic colour scheme": "Automatic", "modal.style.light colour scheme": "Light", "modal.style.dark colour scheme": "Dark", "modal.style.custom colour scheme": "Custom", "modal.style.main font": "Main font", "modal.style.font.sans serif": "Sans serif", "modal.style.font.serif": "Serif", "modal.style.font.monospace": "Monospace", "modal.style.spacing": "Spacing", "modal.style.font weight": "Font weight", "modal.style.forced colours warning": "Your browser has overridden the colours used in this page because of a setting such as high contrast mode. Changes to the colour scheme settings here might not have any effect.", "modal.style.text preview": "Most text will look like this.", "modal.style.more options": "More options", "modal.style.main colour": "Main (brand) colour", "modal.style.primary colour": "Primary button colour", "modal.style.link colour": "Link colour", "modal.style.success colour": "Success/correct colour", "modal.style.info colour": "Info colour", "modal.style.warning colour": "Warning colour", "modal.style.danger colour": "Danger/incorrect colour", "modal.style.muted colour": "Muted text colour", "modal.style.highlight colour": "Highlight colour", "exam.error loading exam definition": "There was an error while loading the exam definition: {{text}}", "exam.no exam definition": "No exam definition was given.", "jme.typecheck.wrong arguments for anonymous function": "Wrong number of arguments for this anonymous function.", "worksheet.top": "Top", "worksheet.left": "Left", "worksheet.bottom": "Bottom", "worksheet.right": "Right"}}, "fr-fr": {"translation": {"page.loading": "Chargement...", "page.saving": "<p>Sauvegarde en cours.</p>\n<p>Cela peut prendre quelques instants.</p>", "mathjax.math processing error": "\"{{-message}}\" lors du rendu tex de <code>{{expression}}</code>", "die.numbas failed": "Numbas a \u00e9chou\u00e9", "die.sorry": "D\u00e9sol\u00e9, Numbas a rencontr\u00e9 une erreur fatale et ne peut continuer. Une description de l'erreur suit ci-dessous.", "die.error": "Erreur", "modal.ok": "OK", "modal.cancel": "Annuler", "exam.exam name": "Nom de l'examen:", "exam.random seed": "Identificateur de session:", "exam.student name": "Nom de l'\u00e9tudiant:", "exam.number of questions": "Nombre de questions:", "exam.marks available": "Points disponibles:", "exam.pass percentage": "Pourcentage de r\u00e9ussite:", "exam.time allowed": "Temps accord\u00e9:", "exam.passed": "R\u00e9ussi", "exam.failed": "\u00c9chou\u00e9", "exam.review header": "R\u00e9vision:", "frontpage.start": "D\u00e9marrer", "suspend.paused header": "En pause", "suspend.exam suspended": "L'examen a \u00e9t\u00e9 suspendu. Appuyez sur <em>Reprendre</em> pour continuer.", "suspend.you can resume": "Vous pourrez reprendre cette session la prochaine fois que vous d\u00e9marrerez cette activit\u00e9.", "suspend.resume": "Reprendre", "result.exit": "Quitter l'examen", "result.print": "Imprimer le r\u00e9sum\u00e9 des notes", "result.exam summary": "Sommaire de l'examen", "result.performance summary": "Sommaire du rendement", "result.exam start": "D\u00e9but de l'examen:", "result.exam stop": "Fin de l'examen:", "result.time spent": "Temps \u00e9coul\u00e9:", "result.questions attempted": "Questions essay\u00e9es:", "result.score": "Note:", "result.result": "R\u00e9sultat:", "result.question number": "Num\u00e9ro de la question", "result.question score": "Note", "result.question review title": "R\u00e9viser cette question", "result.click a question to review": "Cliquez sur le num\u00e9ro d'une question pour voir comment vos r\u00e9ponses ont \u00e9t\u00e9 not\u00e9es et, le cas \u00e9ch\u00e9ant, les solutions compl\u00e8tes.", "end.exam has finished": "L'examen est termin\u00e9. Vous pouvez d\u00e9sormais fermer cette fen\u00eatre.", "control.confirm leave": "Vous n'avez pas termin\u00e9 l'examen.", "control.not all questions answered": "Vous n'avez pas compl\u00e9t\u00e9 toutes les questions de cet examen.", "control.not all questions submitted": "Vous avez modifi\u00e9 une ou plusieurs r\u00e9ponses et ne les avez pas soumises. S.V.P., v\u00e9rifiez que chaque question a \u00e9t\u00e9 soumise.", "control.confirm end": "\u00cates-vous certain de vouloir terminer l'examen? Apr\u00e8s avoir termin\u00e9 l'examen, vous ne pourrez plus changer vos r\u00e9ponses.", "control.confirm regen": "Voulez-vous \u00e0 nouveau r\u00e9partir au hasard cette question? Si vous cliquez OK, toutes vos r\u00e9ponses et vos r\u00e9sultats pour la question en cours seront perdus.", "control.confirm reveal": "Voulez-vous r\u00e9v\u00e9ler la r\u00e9ponse \u00e0 cette question? Tous les points re\u00e7us jusqu'\u00e0 maintenant seront verrouill\u00e9s et vous ne pourrez pas r\u00e9pondre \u00e0 cette question plus tard.", "control.proceed anyway": "Continuer malgr\u00e9 tout?", "control.regen": "Essayer une autre question comme celle-ci", "control.submit answer": "Soumettre la r\u00e9ponse", "control.submit all parts": "Soumettre toutes les parties", "control.submit again": "Soumettre \u00e0 nouveau", "control.submit": "Soumettre", "control.previous": "Pr\u00e9c\u00e9dent", "control.next": "Suivant", "control.advice": "Conseil", "control.reveal": "R\u00e9v\u00e9ler les r\u00e9ponses", "control.total": "Total", "control.pause": "Pause", "control.end exam": "Fin de l'examen", "control.back to results": "Retourner aux r\u00e9sultats", "display.part.jme.error making maths": "Erreur lors de l'affichage des math\u00e9matiques", "exam.xml.bad root": "L'\u00e9l\u00e9ment racine de l'examen XML devrait \u00eatre 'exam'", "exam.changeQuestion.no questions": "Cet examen ne contient aucune question! V\u00e9rifiez s'il y a des erreurs dans le fichier .exam.", "feedback.you were awarded": "Vous avez re\u00e7u <strong>{{count,niceNumber}}</strong> $t(mark).", "feedback.taken away": "<strong>{{count,niceNumber}}</strong> $t(mark) $t(was) retir\u00e9(s).", "jme.tokenise.invalid": "Expression invalide: <code>{{expression}}</code>", "jme.shunt.not enough arguments": "Pas assez d'arguments pour l'op\u00e9ration {{op}}", "jme.shunt.no left bracket in function": "Aucun crochet ouvrant correspondant dans la fonction ou le tuple", "jme.shunt.no left square bracket": "Aucun crochet ouvrant correspondant", "jme.shunt.no left bracket": "Aucun crochet ouvrant correspondant", "jme.shunt.no right bracket": "Aucun crochet fermant correspondant", "jme.shunt.no right square bracket": "Aucun crochet fermant correspondant pour fermer la liste", "jme.shunt.missing operator": "L'expression ne peut \u00eatre \u00e9valu\u00e9e -- il manque un op\u00e9rateur.", "jme.typecheck.function maybe implicit multiplication": "L'op\u00e9ration {{name}} n'est pas d\u00e9finie. Voulez-vous dire <br/><code>{{first}}*{{possibleOp}}(...)</code>?", "jme.typecheck.function not defined": "L'op\u00e9ration <code>{{op}}</code> n'est pas d\u00e9finie. Est-ce que <code>{{op}}</code> est une variable et voulez-vous dire <code>{{suggestion}}*(...)</code>?", "jme.typecheck.op not defined": "L'op\u00e9ration '{{op}}' n'est pas d\u00e9finie.", "jme.typecheck.no right type definition": "Aucune d\u00e9finition du bon type trouv\u00e9e pour '{{op}}'.", "jme.typecheck.no right type unbound name": "La variable <code>{{name}}</code> n'est pas d\u00e9finie.", "jme.typecheck.map not on enumerable": "L'op\u00e9ration <code>map</code> doit travailler sur une liste ou une plage, pas sur {{type}}", "jme.evaluate.undefined variable": "La variable {{name}} n'est pas d\u00e9finie", "jme.thtml.not html": "Valeur non-HTML pass\u00e9e dans le constructeur THTML.", "jme.func.switch.no default case": "Aucun cas par d\u00e9faut pour l'instruction Switch.", "jme.func.listval.invalid index": "Indice de liste {{index}} invalide sur une liste de taille {{size}}", "jme.func.listval.not a list": "L'objet n'est pas indi\u00e7able\n", "jme.func.matrix.invalid row type": "Impossible de construire une matrice \u00e0 partir de rang\u00e9es de type {{type}}", "jme.func.except.continuous range": "Impossible d'utiliser l'op\u00e9rateur 'except' sur une plage continue.", "jme.matrix.reports bad size": "La matrice d\u00e9clare sa taille incorrectement - doit \u00eatre une erreur dans la fonction du constructeur", "jme.texsubvars.no right bracket": "Aucun <code>]</code> dans les arguments {{op}}.", "jme.texsubvars.missing parameter": "Param\u00e8tre manquant dans {{op}}: {{parameter}}", "jme.texsubvars.no right brace": "Aucun <code>}</code> dans {{op}}", "jme.user javascript.error": "Erreur dans la fonction javascript d\u00e9finie par l'utilisateur <code>{{name}}</code>: {{-message}}", "jme.variables.error making function": "Erreur lors de la cr\u00e9ation de la fonction <code>{{name}}</code>: {{-message}}", "jme.variables.syntax error in function definition": "Erreur de syntaxe dans la d\u00e9finition de la fonction", "jme.variables.variable not defined": "La variable <code>{{name}}</code> n'est pas d\u00e9finie.", "jme.variables.empty definition": "La d\u00e9finition de la variable <code>{{name}}</code> est vide.", "jme.variables.circular reference": "R\u00e9f\u00e9rence circulaire \u00e0 une variable dans la d\u00e9finition de <code>{{name}}</code>", "jme.variables.error computing dependency": "Erreur lors du calcul de la variable r\u00e9f\u00e9renc\u00e9e <code>{{name}}</code>", "jme.variables.error evaluating variable": "Erreur lors de l'\u00e9valuation de la variable {{name}}: {{-message}}", "jme.variables.question took too many runs to generate variables": "Un ensemble valide de variables de question n'a pas \u00e9t\u00e9 g\u00e9n\u00e9r\u00e9 \u00e0 temps.", "jme.display.unknown token type": "Impossible de textifier le jeton de type {{type}}", "jme.display.collectRuleset.no sets": "Aucun ensemble n'a \u00e9t\u00e9 fourni \u00e0 collectRuleset!", "jme.display.collectRuleset.set not defined": "L'ensemble de r\u00e8gles {{name}} n'a pas \u00e9t\u00e9 d\u00e9fini", "jme.display.simplifyTree.no scope given": "Numbas.jme.display.simplifyTree doit avoir une port\u00e9e", "math.precround.complex": "Impossible d'arrondir un nombre complexe de d\u00e9cimales", "math.siground.complex": "Impossible d'arrondir \u00e0 un nombre complexe de chiffres significatifs", "math.combinations.complex": "Impossible de calculer des combinaisons de nombres complexes", "math.permutations.complex": "Impossible de calculer des permutation de nombres complexes.", "math.gcf.complex": "Impossible de calculer le plus grand facteur commun de nombres complexes.", "math.lcm.complex": "Impossible de calculer le plus petit commun multiple de nombres complexes.", "math.lt.order complex numbers": "Impossible de trier des nombres complexes.", "math.choose.empty selection": "S\u00e9lection vide donn\u00e9e \u00e0 la fonction al\u00e9atoire", "matrixmath.abs.non-square": "Impossible de calculer le d\u00e9terminant d'une matrice qui n'est pas carr\u00e9e.", "matrixmath.abs.too big": "D\u00e9sol\u00e9, impossible pour l'instant de calculer le d\u00e9terminant d'une matrice plus grande que 3x3.", "matrixmath.mul.different sizes": "Impossible de multiplier des matrices de tailles diff\u00e9rentes.", "vectormath.cross.not 3d": "Ne peut prendre que le produit vectoriel de vecteurs tridimensionnels.", "vectormath.dot.matrix too big": "Impossible de calculer le produit scalaire d'une matrice qui n'est pas $1 \\times N$ ou $N \\times 1$.", "vectormath.cross.matrix too big": "Impossible de calculer le produit vectoriel d'une matrice qui n'est pas $1 \\times N$ ou $N \\times 1$.", "part.with steps answer prompt": "R\u00e9ponse: ", "part.script.error": "Erreur dans {{path}} du script personnalis\u00e9 {{script}}: {{-message}}\n", "part.marking.steps no matter": "Parce que vous avez re\u00e7u toutes les points pour la partie, vos r\u00e9ponses aux \u00e9tapes ne sont pas compt\u00e9es.", "part.marking.revealed steps no penalty": "Vous avez r\u00e9v\u00e9l\u00e9 les \u00e9tapes.", "part.marking.used variable replacements": "Cette partie a \u00e9t\u00e9 not\u00e9e en utilisant vos r\u00e9ponses aux parties pr\u00e9c\u00e9dentes.", "part.marking.variable replacement part not answered": "Vous devez r\u00e9pondre \u00e0 {{part}} en premier.", "part.marking.resubmit because of variable replacement": "Le note de cette partie d\u00e9pend de vos r\u00e9ponses \u00e0 d'autres parties que vous avez modifi\u00e9es. Soumettez \u00e0 nouveau cette partie pour mettre \u00e0 jour votre r\u00e9sultat.", "part.marking.not submitted": "Aucune r\u00e9ponse soumise", "part.marking.did not answer": "Vous n'avez pas r\u00e9pondu \u00e0 cette question.", "part.marking.nothing entered": "Vous n'avez pas entr\u00e9 de r\u00e9ponse.", "part.marking.incorrect": "Votre r\u00e9ponse est incorrecte.", "part.marking.correct": "Votre r\u00e9ponse est correcte.", "part.marking.uncaught error": "Erreur lors de l'\u00e9valuation de la {{part}}: {{-message}}", "part.marking.no result": "Cette partie n'a pas pu \u00eatre \u00e9valu\u00e9e.", "part.correct answer": "R\u00e9ponse attendue:", "part.missing type attribute": "{{part}}: Attribut de type de partie manquant", "part.unknown type": "{{part}}: Type de partie {{type}} non reconnu", "part.setting not present": "Propri\u00e9t\u00e9 '{{property}}' non d\u00e9finie", "part.jme.answer missing": "R\u00e9ponse correcte manquante", "part.jme.answer too long": "Votre r\u00e9ponse est trop longue.", "part.jme.answer too short": "Votre r\u00e9ponse est trop courte.", "part.jme.answer invalid": "Votre r\u00e9ponse n'est pas une expression math\u00e9matique valide.<br/>{{-message}}.", "part.jme.marking.correct": "Votre r\u00e9ponse est num\u00e9riquement correcte.", "part.jme.must-have bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.must-have one": "Votre r\u00e9ponse doit contenir: {{strings}}", "part.jme.must-have several": "Votre r\u00e9ponse doit contenir tous les \u00e9l\u00e9ments suivants: {{strings}}", "part.jme.not-allowed bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.not-allowed one": "Votre r\u00e9ponse ne doit pas contenir: {{strings}}", "part.jme.not-allowed several": "Votre r\u00e9ponse ne doit pas contenir: {{strings}}", "part.jme.unexpected variable name": "Votre r\u00e9ponse a \u00e9t\u00e9 interpr\u00e9t\u00e9e pour utiliser le nom de variable inattendue <code> {{name}} </ code>.", "part.jme.unexpected variable name suggestion": "Votre r\u00e9ponse a \u00e9t\u00e9 interpr\u00e9t\u00e9e pour utiliser le nom de variable inattendue <code> {{name}} </ code>. Voulez-vous dire <code>{{suggestion}}</code>?", "part.patternmatch.display answer missing": "L'affichage de la r\u00e9ponse est manquant", "part.patternmatch.correct except case": "Votre r\u00e9ponse est correcte, \u00e0 l\u2019exception des majuscules et minuscules.", "part.numberentry.correct except decimal": "Votre r\u00e9ponse se situe dans la plage autoris\u00e9e, mais les nombres d\u00e9cimaux ne sont pas autoris\u00e9s.", "part.numberentry.correct except fraction": "Votre r\u00e9ponse se situe dans la plage autoris\u00e9e, mais les fractions ne sont pas autoris\u00e9es.", "part.numberentry.answer invalid": "Vous n'avez pas entr\u00e9 un nombre valide.", "part.numberentry.answer not integer": "Votre r\u00e9ponse est invalide. Vous devez entrer un nombre entier, pas un nombre d\u00e9cimal.", "part.numberentry.answer not integer or decimal": "Votre r\u00e9ponse est invalide. Vous devez entrer un nombre entier ou un nombre d\u00e9cimal.", "part.numberentry.zero sig fig": "Cette partie est r\u00e9gl\u00e9e pour arrondir la r\u00e9ponse de l'\u00e9tudiant \u00e0 z\u00e9ro chiffre significatif, ce qui n'a aucun sens.", "part.mcq.options def not a list": "L'expression d\u00e9finissant les {{properties}} n'est pas une liste.", "part.mcq.marking matrix string empty": "L'expression de la matrice d'\u00e9valuation personnalis\u00e9e est vide.", "part.mcq.choices missing": "La d\u00e9finition des choix est manquante", "part.mcq.matrix not a number": "Partie {{part}} de la cellule ({{row}},{{column}}) de la matrice d'\u00e9valuation ne donne pas un nombre", "part.mcq.wrong number of choices": "Vous avez s\u00e9lectionn\u00e9 le mauvais nombre de choix.", "part.mcq.no choices selected": "Aucun choix s\u00e9lectionn\u00e9.", "part.mcq.matrix not a list": "La matrice d'\u00e9valuation, d\u00e9finie par l'expression JME, n'est pas une liste mais elle devrait l'\u00eatre.", "part.mcq.matrix wrong type": "\u00c9l\u00e9ment de type '{{type}}' invalide utilis\u00e9 dans la matrice d'\u00e9valuation.\n", "part.mcq.matrix mix of numbers and lists": "M\u00e9lange de nombres et de listes utilis\u00e9s dans la matrice d'\u00e9valuation.", "part.mcq.matrix wrong size": "La matrice d'\u00e9valuation est de la mauvaise taille.", "part.mcq.correct choice": "Vous avez choisi la bonne r\u00e9ponse.", "part.matrix.answer invalid": "Votre r\u00e9ponse n'est pas valide.", "part.matrix.invalid cell": "Une ou plusieurs cellules de votre r\u00e9ponse sont vides ou invalides.", "part.matrix.some incorrect": "Une ou plusieurs cellules de votre r\u00e9ponse sont incorrectes, mais vous avez re\u00e7u des points pour le reste.", "part.matrix.empty": "Vous n'avez pas entr\u00e9 de r\u00e9ponse.", "part.matrix.empty cell": "Une ou plusieurs cellules de votre r\u00e9ponse sont vides.\n", "part.matrix.size mismatch": "L'auteur de la question n'a pas permis \u00e0 l'\u00e9tudiant de d\u00e9cider des dimensions de sa r\u00e9ponse, mais la bonne r\u00e9ponse est {{correct_dimensions}} alors que la r\u00e9ponse est {{input_dimensions}}", "part.gapfill.feedback header": "<strong>\u00c9cart {{index}}</strong>", "part.extension.not implemented": "La partie n'a pas impl\u00e9ment\u00e9 la m\u00e9thode <code>{{name}}</code>.", "question.loaded name mismatch": "Impossible de reprendre cette tentative - le progiciel a chang\u00e9 depuis la derni\u00e8re session.", "question.error": "Question {{number}}: {{-message}}", "question.preamble.error": "Erreur dans le pr\u00e9ambule: {{-message}}", "question.preamble.syntax error": "Erreur de syntaxe dans le pr\u00e9ambule", "question.unsupported part type": "Type de partie non support\u00e9", "question.header": "Question {{number}}", "question.submit part": "Soumettre la partie", "question.show steps": "Afficher les \u00e9tapes", "question.show steps penalty": "Vous allez perdre <strong>{{count,niceNumber}}</strong> $t(mark).\n", "question.show steps no penalty": "Votre r\u00e9sultat ne sera pas influenc\u00e9.", "question.show steps already penalised": "Vous avez d\u00e9j\u00e0 affich\u00e9 les \u00e9tapes. Vous pouvez les afficher de nouveau sans p\u00e9nalit\u00e9s.", "question.hide steps": "Masquer les \u00e9tapes", "question.hide steps no penalty": "Votre r\u00e9sultat ne sera pas influenc\u00e9.", "question.advice": "Conseil", "question.no such part": "Impossible de trouver la partie {{path}}", "question.can not submit": "Impossible de soumettre la r\u00e9ponse - v\u00e9rifiez s'il y a des erreurs.", "question.answer submitted": "R\u00e9ponse envoy\u00e9e.", "question.score feedback.show": "Afficher la r\u00e9troaction", "question.score feedback.hide": "Masquer la r\u00e9troaction", "question.score feedback.answered total actual": "R\u00e9sultat: {{score,niceNumber}}/{{marks,niceNumber}}", "question.score feedback.answered total": "{{marksString}}. R\u00e9pondu.\n", "question.score feedback.answered actual": "R\u00e9sultat: {{scoreString}}", "question.score feedback.answered": "R\u00e9pondu.", "question.score feedback.unanswered": "Sans r\u00e9ponse.", "question.score feedback.unanswered total": "{{marksString}}.", "question.score feedback.correct": "Votre r\u00e9ponse est correcte.", "question.score feedback.partial": "Votre r\u00e9ponse est partiellement correcte", "question.score feedback.wrong": "Votre r\u00e9ponse est incorrecte", "question.selector.unsubmitted changes": "Modifications non soumises.", "timing.no accumulator": "pas d'accumulateur de chronom\u00e9trage {{name}}", "timing.time remaining": "Temps restant:", "xml.could not load": "Impossible de charger un document XML: {{-message}}", "xml.property not number": "La propri\u00e9t\u00e9 {{name}} devrait \u00eatre un nombre, mais n'en est pas un ({{value}}), dans le noeud {{element}}", "xml.property not boolean": "La propri\u00e9t\u00e9 {{name}} devrait \u00eatre un bool\u00e9en, mais n'en est pas un ({{value}}), dans le noeud {{element}}", "xml.error in variable definition": "Erreur dans la d\u00e9finition de la variable <code>{{name}}</code>", "scorm.error initialising": "Erreur lord de l'initialisation du protocole SCORM: {{-message}}", "scorm.failed save": "<p>La requ\u00eate pour sauvegarder les donn\u00e9es sur le serveur a \u00e9chou\u00e9. Appuyez sur <b>OK</b> pour essayer \u00e0 nouveau.</p>\n<p>Si vous rencontrez ce message de fa\u00e7on r\u00e9p\u00e9t\u00e9e, v\u00e9rifiez votre connexion internet ou utilisez un autre ordinateur. Les r\u00e9ponses que vous avez envoy\u00e9es pr\u00e9c\u00e9demment ont \u00e9t\u00e9 sauvegard\u00e9es avec succ\u00e8s et seront restaur\u00e9es si vous reprenez cette session sur un autre ordinateur.</p>\n<p>Si ce message apparait constamment et que vous ne pouvez sauvegarder <em>aucune</em> de vos r\u00e9ponses, merci de contacter votre professeur.</p>", "scorm.no exam suspend data": "Impossible de reprendre: pas de donn\u00e9es d'examen en suspend.", "scorm.error loading suspend data": "Erreur lors du chargement des donn\u00e9es en suspend: {{-message}}", "scorm.error loading question": "Erreur lors du chargement de la question {{number}}: {{-message}}", "scorm.no question suspend data": "Pas de donn\u00e9es en suspend pour la question", "scorm.error loading part": "Erreur lors du chargement de la partie {{part}}: {{-message}}", "scorm.no part suspend data": "Pas de donn\u00e9es en suspend pour la partie", "util.product.non list": "Un \u00e9l\u00e9ment qui n'est pas une liste a \u00e9t\u00e9 pass\u00e9 \u00e0 <code>Numbas.util.product</code>\n", "mark": "point", "was": "\u00e9tait", "part": "partie", "gap": "\u00e9cart", "step": "\u00e9tape", "jme.substituteTree.undefined variable": "Variable non d\u00e9finie: <code>{{name}}</code>", "jme.user javascript.returned undefined": "La fonction javascript d\u00e9finie par l'utilisateur <code>{{name}}</ code> a renvoy\u00e9 <code>undefined</ code>.", "part.marking.steps change": "Vous avez re\u00e7u <strong>{{count,niceNumber}}</strong> $t(mark) pour vos r\u00e9ponses aux \u00e9tapes.", "part.marking.revealed steps with penalty": "Vous avez affich\u00e9 les \u00e9tapes. Le r\u00e9sultat maximum pour cette partie est <strong>{{count,niceNumber}}</strong> $t(mark). Vos r\u00e9sultats seront r\u00e9duits en cons\u00e9quence.", "part.marking.total score": "Vous avez obtenu <strong>{{count,niceNumber}}</strong> $t(mark) pour cette partie.", "part.numberentry.precision type.dp": "d\u00e9cimale", "part.numberentry.precision type.dp_plural": "d\u00e9cimales", "part.numberentry.precision type.sigfig": "Chiffre significatif", "part.numberentry.precision type.sigfig_plural": "Chiffres significatifs", "part.numberentry.give your answer to precision": "Arrondissez votre r\u00e9ponse \u00e0 {{count,niceNumber}} {{precisionType}}.", "question.unsubmitted changes": "Vous avez apport\u00e9 une modification \u00e0 votre r\u00e9ponse mais vous ne l'avez pas soumise. Veuillez v\u00e9rifier votre r\u00e9ponse, puis appuyez sur le bouton <strong>Envoyer la r\u00e9ponse</strong>.", "question.unsubmitted changes_plural": "Vous avez apport\u00e9 des modifications \u00e0 vos r\u00e9ponses, mais vous ne les avez pas soumises. Veuillez v\u00e9rifier vos r\u00e9ponses \u00e0 chaque partie, puis appuyez sur le bouton <strong>Envoyer toutes les parties</strong>.", "util.equality not defined for type": "L'\u00e9galit\u00e9 n'est pas d\u00e9finie pour le {{type}}", "mark_plural": "points", "was_plural": "\u00e9taient", "die.script not loaded": "Numbas n'a pas pu d\u00e9marrer car le fichier <code>{{file}}</code> n'a pas \u00e9t\u00e9 charg\u00e9. V\u00e9rifiez qu'il est inclus dans <code>scripts.js</code>.\n", "math.combinations.n less than zero": "Impossible de calculer les combinaisons: n est plus petit que z\u00e9ro", "math.combinations.k less than zero": "Impossible de calculer les combinaisons: k est plus petit que z\u00e9ro", "math.combinations.n less than k": "Impossible de calculer les combinaisons: n est plus petit que k", "math.permutations.n less than zero": "Impossible de calculer les permutations: n est plus petit que z\u00e9ro", "math.permutations.k less than zero": "Impossible de calculer les permutations: k est plus petit que z\u00e9ro", "math.permutations.n less than k": "Impossible de calculer les permutations: n est plus petit k", "part.numberentry.give your answer to precision_0": "Arrondissez votre r\u00e9ponse \u00e0 l'entier le plus pr\u00e8s.", "mathjax.error": "Erreur de traitement MathJax: {{-message}}", "mathjax.error with context": "Erreur de traitement MathJax dans {{context}}: {{-message}}", "exam.introduction": "Introduction \u00e0 l'examen", "exam.feedback": "Message de r\u00e9troaction \u00e0 l'examen", "jme.tokenise.keypair key not a string": "La cl\u00e9 du dictionnaire doit \u00eatre une cha\u00eene, pas {{type}}.\n", "jme.shunt.list mixed argument types": "Impossible d'analyser {{mode}}: m\u00e9lange de dictionnaire et d'\u00e9l\u00e9ments de liste", "jme.func.listval.key not in dict": "Le dictionnaire ne contient pas la cl\u00e9 <code>{{key}}</code>", "part.prompt": "invite", "part.feedback": "r\u00e9troaction", "part.numberentry.answer not reduced": "Votre r\u00e9ponse n'est pas simplifi\u00e9e.", "part.numberentry.give your answer as a reduced fraction": "Simplifiez votre r\u00e9ponse.", "part.numberentry.negative decimal places": "Cette partie est r\u00e9gl\u00e9e pour arrondir la r\u00e9ponse de l'\u00e9tudiant \u00e0 un nombre n\u00e9gatif de d\u00e9cimales, ce qui n'a aucun sens.", "part.mcq.choices": "choix", "part.mcq.answers": "r\u00e9ponses", "part.mcq.matrix cell empty": "La partie {{part}} de la cellule ({{row}},{{column}}) de la matrice d'\u00e9valuation est vide", "part.mcq.matrix jme error": "La partie {{part}} de la cellule ({{row}},{{column}}) de la matrice d'\u00e9valuation retourne une erreur JME: {{error}}\n", "question.statement": "D\u00e9claration", "ruleset.circular reference": "R\u00e9f\u00e9rence circulaire dans la d\u00e9finition du jeu de r\u00e8gles <code>{{name}}</code>", "ruleset.set not defined": "Le jeu de r\u00e8gles {{name}} n'a pas \u00e9t\u00e9 d\u00e9fini", "jme.evaluate.no scope given": "Numbas.jme.evaluate doit avoir une port\u00e9e", "answer.number.not a number": "La r\u00e9ponse doit \u00eatre un nombre.", "answer.number.fractions not allowed": "La r\u00e9ponse ne doit pas \u00eatre une fraction.", "answer.jme.invalid expression": "{{-message}}", "answer.matrix.fractions not allowed": "La r\u00e9ponse ne doit pas \u00eatre une fraction.", "answer.matrix.some cell not a number": "Une ou plusieurs cellule ne contient pas un nombre valide.", "exam.enter password": "Mot de passe:", "exam.password.correct": "Mot de passe correct. Vous pouvez commencer l'examen.", "exam.password.incorrect": "Mot de passe incorrect.", "frontpage.scorm.lms not connected": "L'examen est en mode d\u00e9connect\u00e9. Les r\u00e9ponses et les notes ne seront pas sauvegard\u00e9es ! ", "result.question review": "R\u00e9viser", "control.confirm regen no marks": "Souhaitez-vous recommencer cette question avec d'autres valeurs al\u00e9atoires ?", "control.confirm reveal no marks": "Souhaitez vous d\u00e9voiler la r\u00e9ponse de cette question ?", "jme.tokenise.invalid near": "expression invalide:  <code>{{expression}}</code> \u00e0  la position {{position}} pr\u00e8s de <code>{{nearby}}</code>", "jme.tokenise.number.object not complex": "Objet invalide a \u00e9t\u00e9 pass\u00e9 \u00e0 le constructeur de nombres.", "jme.subvars.null substitution": "Variable vide substitution: <code>$t(left brace){{str}}$t(right brace)", "jme.type.type already registered": "Le type de donn\u00e9e {{type}} a d\u00e9j\u00e0 \u00e9t\u00e9 enregistr\u00e9, alors il ne peut pas \u00eatre r\u00e9enregistr\u00e9.", "jme.type.no cast method": "Impossible de convertir automatiquement {{from}} en {{to}}.", "jme.display.simplifyTree.empty expression": "Expression vide", "jme.display.simplifyTree.stuck in a loop": "Simplifier est coinc\u00e9 dans une boucle : <code>{{expr}}</code>", "math.niceNumber.undefined": "Un nombre \u00e9tait attendu, mais on a eu <code>undefined</code>", "math.rangeToList.zero step size": "Ne peut pas convertir un intervalle avec une taille de pas de z\u00e9ro en une liste.", "part.error": "{{path}}: {{-message}}", "part.marking.revealed steps": "Vous avez affich\u00e9 les indices.", "part.marking.maximum scaled down": "Le score maximal de cette partie est <strong>{{count,niceNumber}}</strong> $t(mark). Vos r\u00e9sultats vont \u00eatre abaiss\u00e9 en cons\u00e9quence.", "part.marking.minimum score applied": "Le score minimal de cette partie est <strong>{{score,niceNumber}}</strong>.", "part.marking.maximum score applied": "Le score maximal de cette partie est <strong>{{score,niceNumber}}</strong>.", "part.marking.error in marking script": "Il y a une erreur dans l'algorithme de notation de cette partie. Merci de le signaler. {{-message}}", "part.marking.no result after replacement": "Cette partie ne peut pas \u00eatre \u00e9valu\u00e9e avec les r\u00e9ponses des pr\u00e9c\u00e9dentes parties.", "part.marking.missing required note": "L'algorithme de notation ne d\u00e9finit pas la note <code>{{note}}</code>", "marking.apply.not a list": "Le premier argument \u00e0 <code>appliquer</code> doit \u00eatre une liste, mais il n'est pas", "marking.apply marking script.script not found": "Le script de marquage <code>{{name}}</code> est introuvable", "marking.note.compilation error": "Erreur lors de la compilation de la note <code>{{name}}</code>: {{-message}}", "marking.note.error evaluating note": "Erreur lors de l'\u00e9valuation de la note <code>{{name}}</code> - {{-message}}", "marking.note.invalid definition": "D\u00e9finition de note invalide: <code>{{source}}</code>. {{-hint}}", "marking.note.invalid definition.missing colon": "Il vous manque peut-\u00eatre deux points apr\u00e8s le nom et la description", "marking.note.invalid definition.description missing closing bracket": "Il vous manque peut-\u00eatre une parenth\u00e8se fermante", "marking.note.empty expression": "La note <code>{{name}}</code> est vide.", "marking.script.error parsing notes": "Erreur lors de l'analyse du script de marquage: {{- message}}", "part.feedback out of date": "Cette r\u00e9action est bas\u00e9e sur votre derni\u00e8re r\u00e9ponse soumise. Soumettez votre r\u00e9ponse modifi\u00e9e pour obtenir une nouvelle r\u00e9action.", "part.jme.invalid value generator expression": "Invalid value generator expression for variable <code>{{name}}</code>: {{-message}}", "part.mcq.incorrect choice": "Vous avez choisi une mauvaise r\u00e9ponse.", "part.matrix.not all cells same precision": "You have not given every cell in your answer to the same precision.", "part.gapfill.error marking gap": "Erreur lors du marquage de {{name}}\u00a0: {{-message}}", "part.custom.empty setting": "Aucune valeur donn\u00e9e.", "part.custom.unrecognised input type": "Unrecognised setting type <code>{{input_type}}</code>", "part.custom.error evaluating input option": "Error evaluating input option <code>{{option}}</code>: {{-error}}", "part.custom.input option missing": "Definition of input option <code>{{option}}</code> is missing.", "part.custom.error evaluating setting": "Error evaluating setting <code>{{setting}}</code>: {{-error}}", "question.error creating question": "Error while creating question {{number}}: {{-message}}", "question.score feedback.not marked": "Non not\u00e9", "question.score feedback.partially answered": "R\u00e9ponse incompl\u00e8te", "question.score feedback.score total": "{{marksString}}", "question.score feedback.score actual": "Note: {{scoreString}}", "question.score feedback.score total actual": "Note: {{score,niceNumber}}/{{marks,niceNumber}}", "variable.error in variable definition": "Erreur dans la d\u00e9finition de la variable <code>{{name}}</code>", "left brace": "{", "right brace": "}", "extension.not found": "Impossible de charger l'extension <code>{{name}}</code>.", "control.toggle navigation menu": "Basculer sur le menu de navigation", "part.input title": "R\u00e9ponse pour la partie {{name}}", "part.correct answer title": "R\u00e9ponse attendue pour la partie {{name}}", "part.jme.must-match.failed": "Your answer is not in the expected form.", "question.score feedback.none": "", "control.submit part.confirm remove next parts": "<p>One or more subsequent parts depend on your answer to this part. Submitting this part again will invalidate those parts, and remove them from the question. This cannot be undone.</p>\n<p>Would you like to submit this part again?</p>", "control.back to menu": "Retour au menu", "display.error making html": "Error making HTML in {{contextDescription}}: {{-message}}", "jme.subvars.error compiling": "{{-message}} dans <code>{{expression}}</code>", "jme.variables.empty name": "Une variable de la question n'a pas de nom.", "jme.calculus.unknown derivative": "Impossible de diff\u00e9rencier <code>{{tree}}</code>", "math.order complex numbers": "Impossible d'ordonner des nombres complexes", "menu.choose a question": "Choisissez une question.", "part.choose next part.answered": "Que souhaitez-vous faire \u00e0 pr\u00e9sent ?", "part.choose next part.unanswered": "Sinon, vous pourriez : ", "part.choose next part.will be locked": "(Cette partie sera verrouill\u00e9e)", "part.reached dead end": "Il n'y a rien de plus \u00e0 faire ici.", "part.next part.penalty amount": "(perdre {{count}}\u00a0$t(mark))", "part.marking.counts towards objective": "This part counts towards the objective <strong>\u201c{{objective}}\u201d</strong>.", "part.numberentry.answer not integer or decimal or fraction": "Votre r\u00e9ponse n'est pas valide. Il faut saisir un nombre entier, un nombre d\u00e9cimal ou une fraction.", "question": "Question", "question.progress": "Avancement :", "question.score feedback.unattempted": "Non tent\u00e9", "question.score feedback.attempted": "Tent\u00e9", "question.score feedback.score actual.plain": "{{scoreString}}", "question.score feedback.score total actual.plain": "{{score,niceNumber}}/{{marks,niceNumber}}", "question.objectives": "Objectifs", "question.penalties": "P\u00e9nalit\u00e9s", "question.back to previous part": "Revenir \u00e0 la partie pr\u00e9c\u00e9dente", "end.print": "Imprimez votre relev\u00e9 de notes d'examen", "math.shuffle_together.lists not all the same length": "Toutes les listes n'ont pas la m\u00eame longueur.", "jme.parse signature.invalid signature string": "Invalid function signature string: {{str}}", "part.custom.expected answer has wrong type": "The expected answer for this part has the wrong type. It should be <code>{{shouldbe}}</code>.", "part.custom.input option has wrong type": "The answer input setting <code>{{option}}</code> has the wrong type. It should be <code>{{shouldbe}}</code>.", "matrix input.size control legend": "Taille", "matrix input.rows": "Lignes", "matrix input.columns": "Colonnes", "part.jme.error checking numerically": "There was an error numerically checking your answer: {{-message}}", "part.gapfill.cyclic adaptive marking": "There is a cycle in the adaptive marking for this part: <strong>{{name1}}</strong> relies on <strong>{{name2}}</strong>, which eventually relies on <strong>{{name1}}</strong>.", "modal.style.background colour": "Couleur de l'arri\u00e8re-plan", "modal.style.text colour": "Couleur du texte", "modal.style.text size": "Taille du texte", "modal.style.explanation": "Utilisez ces bouton pour changer l'apparence du test.", "modal.style.reset to defaults": "Remettre aux valeurs par d\u00e9faut.", "modal.style.text size preview": "La plupart des textes seront de cette taille.", "control.style options": "Options d'affichage", "part.marking.partially correct": "Votre r\u00e9ponse est en partie juste.", "part.marking.error in adaptive marking": "There was an error in the adaptive marking for this part. Please report this. {{-message}}", "page.skip to content": "Aller au contenu", "result.learning objective": "Objectif d'apprentissage", "jme.interpreted as": "interpr\u00e9t\u00e9 comme", "jme.script.note.compilation error": "Error compiling note <code>{{name}}</code>: {{-message}}", "jme.script.note.error evaluating note": "Error evaluating note <code>{{name}}</code> - {{-message}}", "jme.script.note.invalid definition": "Invalid note definition: <code>{{source}}</code>. {{-hint}}", "jme.script.note.invalid definition.missing colon": "You might be missing a colon after the name and description", "jme.script.note.invalid definition.description missing closing bracket": "Il doit manquer une parenth\u00e8se fermante ", "jme.script.note.empty expression": "La note <code>{{name}}</code> est vide.", "jme.script.error parsing notes": "Error parsing marking script: {{- message}}", "matrix input.cell label": "Ligne {{row}}, colonne {{column}}", "control.move to next question": "Passer \u00e0 la question suivante", "diagnostic.use retry": "Utilisez une nouvelle tentative et essayez \u00e0 nouveau ce sujet.", "diagnostic.move to next topic": "Passez au sujet suivant.", "diagnostic.next step question": "Que voulez-vous faire ensuite ?", "diagnostic.now assessing topic": "Now assessing {{current_topic}}", "diagnostic.one retry left": "Il vous reste 1\u00a0essai", "diagnostic.retries left": "Il vous reste {{retries}} essais.", "diagnostic.percentage completed": "Vous avez termin\u00e9 <strong>{{percentage}}\u00a0%</strong> de l'\u00e9preuve.", "diagnostic.test is over": "L'\u00e9preuve est termin\u00e9e.", "diagnostic.passed all lo": "Vous avez r\u00e9ussi tous les objectifs d'apprentissage.", "diagnostic.more work on lo": "You need to do some more work on the following learning objectives: {{los}}.", "diagnostic.move to next question in topic": "Passez \u00e0 la question suivante du sujet.", "diagnostic.complete": "Termin\u00e9 !", "diagnostic.studying topic": "\u00c9tudier {{topic}}", "display.answer widget.unknown widget type": "The answer widget type <code>{{name}}</code> is not recognised.", "jme.shunt.expected argument before comma": "Expected to see something between the opening bracket and the comma", "part.waiting for pre submit": "Votre r\u00e9ponse est en cours d'\u00e9valuation. Patientez s'il vous pla\u00eet.", "diagnostic.end test": "Terminez l'\u00e9preuve.", "page.no stylesheet loaded": "Le fichier de feuille de style de la page n'a pas \u00e9t\u00e9 charg\u00e9.", "modal.confirm": "Confirmer", "modal.alert": "Alerte", "suspend.resumed header": "La tentative a repris", "jme.vector.value not an array of numbers": "Tried to construct a vector using a value that is not an array of numbers.", "jme.matrix.value not the right type": "Tried to construct a vector using a value of the wrong type.", "jme.subvars.html inserted twice": "An HTML value has been embedded twice. Consider defining a function to generate a new value each time it is used.", "jme.variables.invalid function language": "The language <code>{{language}}</code> is not valid.", "jme.variables.duplicate definition": "There is more than one definition of the variable <code>{{name}}</code>.", "math.random_integer_partition.invalid k": "The size of the partition must be between 1 and {{n}}.", "part.marking.parameter already in scope": "There is a variable named <code>{{name}}</code>, which is also the name of a marking parameter. Please rename the variable.", "part.marking.adaptive variable replacement refers to self": "This part refers to itself in a variable replacement for adaptive marking.", "part.marking.adaptive variable replacement refers to nothing": "This part contains an invalid variable replacement for adaptive marking.", "part.numberentry.display answer wrong type": "The display answer for this part is a value of type <code>{{got_type}}</code>, but should be a <code>{{want_type}}</code>.", "part.matrix.invalid type in prefilled": "There is an invalid value of type <code>{{n}}</code> in the array of pre-filled cells.", "diagnostic.make a choice": "Faites un choix", "matrixmath.not square": "This operation only works on a square matrix.", "matrixmath.not invertible": "This operation only works on an invertible matrix.", "matrixmath.gauss-jordan elimination.not enough columns": "There must be at least as many columns as rows.", "question.required extension not available": "This question requires the extension <code>{{-extension}}</code> but it is not available.", "util.formatNumberNotation.unrecognised syntax": "The number formatting syntax <code>{{syntax}}</code> is not recognised.", "worksheet.number of exams to generate": "Nombre de feuilles \u00e0 g\u00e9n\u00e9rer", "worksheet.starting with id": "Commen\u00e7ant par ID", "worksheet.show exam id": "Afficher l'ID de la feuille\u00a0?", "worksheet.page break between questions": "Sauts de page entre les questions ?", "worksheet.page margins": "Marges de la page (mm)", "worksheet.text size": "Taille du texte (pt)", "worksheet.generate": "G\u00e9n\u00e9rer", "worksheet.generating exams": "G\u00e9n\u00e9ration des feuilles", "worksheet.sheet id": "Identifiant de la feuille\u00a0ID:", "worksheet.print single": "Imprimer cette feuille", "worksheet.print several": "Imprimez ces feuilles", "worksheet.answer sheets": "Feuilles-r\u00e9ponses", "worksheet.question sheets": "Feuilles-questions", "worksheet.reconfigure": "G\u00e9n\u00e9rer des feuilles diff\u00e9rentes", "worksheet.show sheet": "Pr\u00e9visualisez la feuille avec ID\u00a0:", "accessibility statement": "Accessibility statement and guide to adapting Numbas to your needs.", "exam.enter your name": "Your name:", "exam.attempt download security warning": "This exam is configured to allow you to download your data, but it is not running in a secure browser context. You will not be able to download your exam data. Contact your lecturer or teacher for help.", "result.download exam object": "Download your exam data", "control.return to question": "Return to the question", "control.show introduction": "Introduction", "analysis.header": "Analyse attempt data", "analysis.help.upload files": "Upload attempt data files that your students have given you.", "analysis.help.file input label": "Choose student attempt data files, or drag files onto this window.", "analysis.table.total": "Exam totals", "analysis.table.question": "Exam and question totals", "analysis.table.all": "All details", "analysis.student name.anonymous": "No name given", "analysis.expected": "Expected results", "analysis.start time": "Start time", "analysis.maximum": "Maximum Marks", "analysis.file": "File", "analysis.download": "Download", "analysis.delete": "Delete", "analysis.view results": "View results", "analysis.upload files": "Upload files", "analysis.upload more": "Upload more files", "analysis.attempt data": "Attempt data", "analysis.select format": "Select data to show", "analysis.download this table": "Download this table", "analysis.student": "Student Results", "analysis.question key": "Question Key", "analysis.question name": "Question Name", "analysis.group": "Group", "analysis.question": "Question", "analysis.part": "Part", "analysis.gap": "Gap", "analysis.record type": "Record Type", "analysis.score": "Score", "analysis.marks available": "Marks Available", "analysis.percentage": "Percentage", "analysis.answer": "Answer", "analysis.student name": "Student Name", "analysis.summary.no files": "You have not uploaded any files yet.", "analysis.summary.no decrypted files": "You have not uploaded any valid files yet.", "analysis.summary.one file": "One attempt.", "analysis.summary.several files": "{{num_files,niceNumber}} attempts.", "analysis.not secure context": "This page must be opened in a secure browser context. A secure context is either a page served over HTTPS, or a file loaded from your device.", "jme.shunt.pipe right hand takes no arguments": "The expression on the right-hand side of the pipe operator must be a function application.", "question.explore.no parts defined": "There are no parts defined in this question.", "answer": "answer", "worksheet.answersheet show question content": "Show question content in answer sheets?", "modal.confirm end exam": "Write <code>{{endConfirmation}}</code> in the box to confirm:", "modal.end exam button": "End exam", "lightbox.zoom in on image": "Zoom in on this image", "exam.progress": "Progress", "exam.questions answered": "{{numAnsweredQuestions}} of {{numQuestions}} questions answered.", "result.question marks available": "Marks Available", "result.question answered": "Answered?", "control.confirm end.correct": "You may now end the exam.", "control.confirm end.incorrect": "This is not the expected text.", "control.confirm end.password": "end", "jme.typecheck.for in name wrong type": "The name in a <code>for</code> statement must be a name or list of names, not {{type}}.", "jme.makeFast.no fast definition of function": "The function <code>{{name}}</code> here isn't defined in a way that can be made fast.", "part.show feedback": "Show feedback", "part.hide feedback": "Hide feedback", "part.feedback title": "Feedback for {{name}}.", "part.jme.must-match.warning": "Your answer is not in the expected form: {{-message}}", "part.numberentry.write your answer as a fraction": "Write your answer as a fraction.", "question.nav.label": "Question controls", "question.answer saved": "Answer saved", "question.all answers saved": "All answers saved", "analysis.back to results": "Back to results", "analysis.review": "Review", "analysis.review this": "Review this attempt", "analysis.reviewing attempt": "Reviewing attempt by {{student_name}}.", "part.there is new feedback": "The feedback has changed.", "modal.style.colour scheme": "Colour scheme", "modal.style.automatic colour scheme": "Automatic", "modal.style.light colour scheme": "Light", "modal.style.dark colour scheme": "Dark", "modal.style.custom colour scheme": "Custom", "modal.style.main font": "Main font", "modal.style.font.sans serif": "Sans serif", "modal.style.font.serif": "Serif", "modal.style.font.monospace": "Monospace", "modal.style.spacing": "Spacing", "modal.style.font weight": "Font weight", "modal.style.forced colours warning": "Your browser has overridden the colours used in this page because of a setting such as high contrast mode. Changes to the colour scheme settings here might not have any effect.", "modal.style.text preview": "Most text will look like this.", "modal.style.more options": "More options", "modal.style.main colour": "Main (brand) colour", "modal.style.primary colour": "Primary button colour", "modal.style.link colour": "Link colour", "modal.style.success colour": "Success/correct colour", "modal.style.info colour": "Info colour", "modal.style.warning colour": "Warning colour", "modal.style.danger colour": "Danger/incorrect colour", "modal.style.muted colour": "Muted text colour", "modal.style.highlight colour": "Highlight colour", "exam.error loading exam definition": "There was an error while loading the exam definition: {{text}}", "exam.no exam definition": "No exam definition was given.", "jme.typecheck.wrong arguments for anonymous function": "Wrong number of arguments for this anonymous function.", "worksheet.top": "Top", "worksheet.left": "Left", "worksheet.bottom": "Bottom", "worksheet.right": "Right"}}, "vi-vn": {"translation": {"page.loading": "\u0110ang t\u1ea3i...", "page.saving": "<p>\u0110ang l\u01b0u.</p>\n<p>Vi\u1ec7c n\u00e0y c\u00f3 th\u1ec3 m\u1ea5t v\u00e0i gi\u00e2y.</p>", "mathjax.math processing error": "\"{{-message}}\" khi hi\u1ec3n th\u1ecb c\u00f4ng th\u1ee9c to\u00e1n <code>{{expression}}</code>", "die.numbas failed": "Numbas b\u1ecb s\u1eadp", "die.sorry": "Xin l\u1ed7i, Numbas \u0111\u00e3 g\u1eb7p ph\u1ea3i m\u1ed9t l\u1ed7i n\u00ean n\u00f3 kh\u00f4ng th\u1ec3 ti\u1ebfp t\u1ee5c. L\u1ed7i n\u00e0y \u0111\u01b0\u1ee3c m\u00f4 t\u1ea3 nh\u01b0 d\u01b0\u1edbi \u0111\u00e2y.", "die.error": "L\u1ed7i", "modal.ok": "OK", "modal.cancel": "H\u1ee7y b\u1ecf", "exam.exam name": "T\u00ean b\u00e0i ki\u1ec3m tra:", "exam.random seed": "ID c\u1ee7a phi\u00ean:", "exam.student name": "T\u00ean h\u1ecdc vi\u00ean:", "exam.number of questions": "S\u1ed1 l\u01b0\u1ee3ng c\u00e2u h\u1ecfi:", "exam.marks available": "\u0110i\u1ec3m thi c\u00f3 s\u1eb5n:", "exam.pass percentage": "T\u1ec9 l\u1ec7 \u0111\u1ed7 (%):", "exam.time allowed": "Th\u1eddi gian cho ph\u00e9p:", "exam.passed": "\u0110\u1eadu", "exam.failed": "Tr\u01b0\u1ee3t", "exam.review header": "Xem l\u1ea1i: ", "frontpage.start": "B\u1eaft \u0111\u1ea7u", "suspend.paused header": "T\u1ea1m d\u1eebng", "suspend.exam suspended": "B\u00e0i thi b\u1ecb t\u1ea1m ng\u01b0ng. Nh\u1ea5n <em>Ti\u1ebfp t\u1ee5c</em> \u0111\u1ec3 ti\u1ebfp t\u1ee5c.", "suspend.you can resume": "B\u1ea1n s\u1ebd c\u00f3 th\u1ec3 ti\u1ebfp t\u1ee5c phi\u00ean n\u00e0y khi b\u1ea1n ti\u1ebfp t\u1ee5c ho\u1ea1t \u0111\u1ed9ng n\u00e0y v\u00e0o l\u1ea7n t\u1edbi.", "suspend.resume": "Ti\u1ebfp t\u1ee5c", "result.exit": "Tho\u00e1t kh\u1ecfi b\u00e0i ki\u1ec3m tra", "result.print": "In b\u1ea3n t\u00f3m t\u1eaft k\u1ebft qu\u1ea3 n\u00e0y", "result.exam summary": "T\u00f3m t\u1eaft b\u00e0i ki\u1ec3m tra", "result.performance summary": "T\u00f3m t\u1eaft th\u00e0nh t\u00edch", "result.exam start": "Th\u1eddi gian b\u1eaft \u0111\u1ea7u:", "result.exam stop": "Th\u1eddi gian k\u1ebft th\u00fac:", "result.time spent": "Th\u1eddi gian l\u00e0m b\u00e0i:", "result.questions attempted": "C\u00e2u h\u1ecfi \u0111\u00e3 l\u00e0m:", "result.score": "\u0110i\u1ec3m:", "result.result": "K\u1ebft qu\u1ea3:", "result.question number": "C\u00e2u h\u1ecfi s\u1ed1", "result.question score": "\u0110i\u1ec3m", "result.question review title": "Xem l\u1ea1i c\u00e2u h\u1ecfi n\u00e0y", "result.click a question to review": "B\u1ea5m v\u00e0o c\u00e2u h\u1ecfi \u0111\u1ec3 xem c\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n \u0111\u01b0\u1ee3c ch\u1ea5m nh\u01b0 th\u1ebf n\u00e0o v\u00e0 l\u1eddi gi\u1ea3i \u0111\u1ea7y \u0111\u1ee7 (n\u1ebfu c\u00f3).", "end.exam has finished": "B\u00e0i ki\u1ec3m tra \u0111\u00e3 k\u1ebft th\u00fac. B\u00e2y gi\u1edd b\u1ea1n c\u00f3 th\u1ec3 \u0111\u00f3ng c\u1eeda s\u1ed5 n\u00e0y.", "control.confirm leave": "B\u1ea1n ch\u01b0a ho\u00e0n th\u00e0nh b\u00e0i ki\u1ec3m tra.", "control.not all questions answered": "B\u1ea1n ch\u01b0a ho\u00e0n th\u00e0nh m\u1ecdi c\u00e2u h\u1ecfi trong b\u00e0i ki\u1ec3m tra n\u00e0y.", "control.not all questions submitted": "B\u1ea1n \u0111\u00e3 thay \u0111\u1ed5i m\u1ed9t ho\u1eb7c nhi\u1ec1u c\u00e2u tr\u1ea3 l\u1eddi nh\u01b0ng kh\u00f4ng g\u1eedi ch\u00fang. Vui l\u00f2ng ki\u1ec3m tra t\u1eebng c\u00e2u h\u1ecfi \u0111\u00e3 \u0111\u01b0\u1ee3c g\u1eedi.", "control.confirm end": "B\u1ea1n c\u00f3 ch\u1eafc ch\u1eafn mu\u1ed1n k\u1ebft th\u00fac b\u00e0i ki\u1ec3m tra? Sau khi b\u1ea1n k\u1ebft th\u00fac b\u00e0i ki\u1ec3m tra, b\u1ea1n s\u1ebd kh\u00f4ng th\u1ec3 thay \u0111\u1ed5i b\u1ea5t k\u1ef3 c\u00e2u tr\u1ea3 l\u1eddi n\u00e0o c\u1ee7a m\u00ecnh.", "control.confirm regen": "B\u1ea1n c\u00f3 mu\u1ed1n th\u1eed m\u1ed9t phi\u00ean b\u1ea3n m\u1edbi (ng\u1eabu nhi\u00ean) c\u1ee7a c\u00e2u h\u1ecfi n\u00e0y? N\u1ebfu b\u1ea1n b\u1ea5m v\u00e0o OK, t\u1ea5t c\u1ea3 c\u00e1c c\u00e2u tr\u1ea3 l\u1eddi v\u00e0 \u0111i\u1ec3m cho c\u00e2u h\u1ecfi hi\u1ec7n t\u1ea1i c\u1ee7a b\u1ea1n s\u1ebd b\u1ecb m\u1ea5t.", "control.confirm reveal": "B\u1ea1n c\u00f3 mu\u1ed1n xem c\u00e2u tr\u1ea3 l\u1eddi cho c\u00e2u h\u1ecfi n\u00e0y? T\u1ea5t c\u1ea3 s\u1ed1 \u0111i\u1ec3m b\u1ea1n nh\u1eadn \u0111\u01b0\u1ee3c cho \u0111\u1ebfn nay s\u1ebd b\u1ecb kh\u00f3a v\u00e0 b\u1ea1n s\u1ebd kh\u00f4ng th\u1ec3 tr\u1ea3 l\u1eddi c\u00e2u h\u1ecfi n\u00e0y v\u1ec1 sau.", "control.proceed anyway": "D\u00f9 sao v\u1eabn ti\u1ebfn h\u00e0nh?", "control.regen": "H\u00e3y th\u1eed m\u1ed9t c\u00e2u h\u1ecfi kh\u00e1c nh\u01b0 c\u00e2u n\u00e0y", "control.submit answer": "G\u1eedi c\u00e2u tr\u1ea3 l\u1eddi", "control.submit all parts": "G\u1eedi t\u1ea5t c\u1ea3 c\u00e1c ph\u1ea7n", "control.submit again": "G\u1eedi l\u1ea1i", "control.submit": "G\u1eedi", "control.previous": "Tr\u01b0\u1edbc", "control.next": "Sau", "control.advice": "G\u1ee3i \u00fd", "control.reveal": "Xem c\u00e2u tr\u1ea3 l\u1eddi", "control.total": "T\u1ed5ng \u0111i\u1ec3m", "control.pause": "T\u1ea1m d\u1eebng", "control.end exam": "K\u1ebft th\u00fac b\u00e0i ki\u1ec3m tra", "control.back to results": "Quay tr\u1edf l\u1ea1i k\u1ebft qu\u1ea3", "display.part.jme.error making maths": "L\u1ed7i hi\u1ec3n th\u1ecb to\u00e1n h\u1ecdc", "exam.xml.bad root": "Ph\u1ea7n t\u1eed g\u1ed1c c\u1ee7a b\u00e0i ki\u1ec3m tra XML ph\u1ea3i l\u00e0 'exam'", "exam.changeQuestion.no questions": "B\u00e0i ki\u1ec3m tra n\u00e0y kh\u00f4ng c\u00f3 c\u00e2u h\u1ecfi! Ki\u1ec3m tra t\u1ec7p .exam \u0111\u1ec3 xem l\u1ed7i.", "feedback.you were awarded": "B\u1ea1n \u0111\u01b0\u1ee3c <strong>{{count,niceNumber}}</strong> $t(\u0111i\u1ec3m).", "feedback.taken away": "B\u1ea1n b\u1ecb tr\u1eeb <strong>{{count,niceNumber}}</strong> $t(\u0111i\u1ec3m).", "jme.tokenise.invalid": "Bi\u1ec3u th\u1ee9c kh\u00f4ng h\u1ee3p l\u1ec7: <code>{{expression}}</code>", "jme.shunt.not enough arguments": "Kh\u00f4ng \u0111\u1ee7 \u0111\u1ed1i s\u1ed1 cho ph\u00e9p to\u00e1n <code>{{op}}</code>", "jme.shunt.no left bracket in function": "Kh\u00f4ng c\u00f3 d\u1ea5u ngo\u1eb7c tr\u00e1i trong h\u00e0m \u1ee9ng d\u1ee5ng ho\u1eb7c b\u1ed9 d\u1eef li\u1ec7u", "jme.shunt.no left square bracket": "Kh\u00f4ng c\u00f3 d\u1ea5u ngo\u1eb7c tr\u00e1i", "jme.shunt.no left bracket": "Kh\u00f4ng c\u00f3 d\u1ea5u ngo\u1eb7c tr\u00e1i", "jme.shunt.no right bracket": "Kh\u00f4ng c\u00f3 d\u1ea5u ngo\u1eb7c ph\u1ea3i", "jme.shunt.no right square bracket": "Kh\u00f4ng c\u00f3 d\u1ea5u ngo\u1eb7c vu\u00f4ng ph\u1ea3i \u1edf cu\u1ed1i danh s\u00e1ch", "jme.shunt.missing operator": "Bi\u1ec3u th\u1ee9c kh\u00f4ng th\u1ec3 t\u00ednh to\u00e1n -- thi\u1ebfu m\u1ed9t ph\u00e9p to\u00e1n.", "jme.typecheck.function maybe implicit multiplication": "H\u00e0m <code>{{name}}</code> kh\u00f4ng \u0111\u01b0\u1ee3c \u0111\u1ecbnh ngh\u0129a. \n\u00dd b\u1ea1n l\u00e0 <code>{{first}}*{{possibleOp}}(...)</code>?", "jme.typecheck.function not defined": "H\u00e0m <code>{{op}}</code> kh\u00f4ng \u0111\u01b0\u1ee3c \u0111\u1ecbnh ngh\u0129a. C\u00f3 ph\u1ea3i <code>{{op}}</code> l\u00e0 bi\u1ebfn kh\u00f4ng, v\u00e0 \u00fd b\u1ea1n l\u00e0 <code>{{suggestion}}*(...)</code>?", "jme.typecheck.op not defined": "Ph\u00e9p to\u00e1n '{{op}}' ch\u01b0a \u0111\u01b0\u1ee3c \u0111\u1ecbnh ngh\u0129a", "jme.typecheck.no right type definition": "Kh\u00f4ng t\u00ecm th\u1ea5y \u0111\u1ecbnh ngh\u0129a '{{op}}' \u0111\u00fang ki\u1ec3u.", "jme.typecheck.no right type unbound name": "Bi\u1ebfn <code>{{name}}</code> kh\u00f4ng \u0111\u01b0\u1ee3c \u0111\u1ecbnh ngh\u0129a.", "jme.typecheck.map not on enumerable": "<code>map</code> ph\u00e9p to\u00e1n ph\u1ea3i \u0111\u01b0\u1ee3c ho\u1ea1t \u0111\u1ed9ng tr\u00ean m\u1ed9t danh s\u00e1ch ho\u1eb7c m\u1ed9t mi\u1ec1n, kh\u00f4ng tr\u00ean {{type}}", "jme.evaluate.undefined variable": "Bi\u1ebfn {{name}} kh\u00f4ng \u0111\u01b0\u1ee3c \u0111\u1ecbnh ngh\u0129a", "jme.thtml.not html": "\u0110\u00e3 chuy\u1ec3n m\u1ed9t gi\u00e1 tr\u1ecb kh\u00f4ng ph\u1ea3i HTML v\u00e0o h\u00e0m t\u1ea1o THTML.", "jme.func.switch.no default case": "Kh\u00f4ng c\u00f3 tr\u01b0\u1eddng h\u1ee3p m\u1eb7c \u0111\u1ecbnh cho c\u00e2u l\u1ec7nh Switch", "jme.func.listval.invalid index": "Index c\u1ee7a danh s\u00e1ch kh\u00f4ng h\u1ee3p l\u1ec7 {{index}} trong k\u00edch th\u01b0\u1edbc {{size}}", "jme.func.listval.not a list": "\u0110\u1ed1i t\u01b0\u1ee3ng kh\u00f4ng \u0111\u01b0\u1ee3c \u0111\u0103ng k\u00fd", "jme.func.matrix.invalid row type": "Kh\u00f4ng th\u1ec3 t\u1ea1o m\u1ed9t ma tr\u1eadn t\u1eeb nh\u1eefng h\u00e0ng thu\u1ed9c ki\u1ec3u {{type}}", "jme.func.except.continuous range": "Kh\u00f4ng th\u1ec3 s\u1eed d\u1ee5ng to\u00e1n t\u1eed 'except' trong ph\u1ea1m vi li\u00ean t\u1ee5c.", "jme.matrix.reports bad size": "K\u00edch th\u01b0\u1edbc c\u1ee7a ma tr\u1eadn kh\u00f4ng ch\u00ednh x\u00e1c - c\u00f3 m\u1ed9t l\u1ed7i trong h\u00e0m x\u00e2y d\u1ef1ng", "jme.texsubvars.no right bracket": "Kh\u00f4ng kh\u1edbp <code>]</code> trong c\u00e1c \u0111\u1ed1i s\u1ed1 {{op}}.", "jme.texsubvars.missing parameter": "Thi\u1ebfu tham s\u1ed1 trong {{op}}: {{parameter}}", "jme.texsubvars.no right brace": "Kh\u00f4ng kh\u1edbp <code>]</code> trong {{op}}", "jme.user javascript.error": "L\u1ed7i trong h\u00e0m javascript do ng\u01b0\u1eddi d\u00f9ng \u0111\u1ecbnh ngh\u0129a <code> {{name}} </code>: {{-message}}", "jme.variables.error making function": "L\u1ed7i t\u1ea1o h\u00e0m <code> {{name}} </code>: {{-message}}", "jme.variables.syntax error in function definition": "L\u1ed7i c\u00fa ph\u00e1p trong \u0111\u1ecbnh ngh\u0129a h\u00e0m", "jme.variables.variable not defined": "Bi\u1ebfn <code>{{name}}</code> ch\u01b0a \u0111\u01b0\u1ee3c \u0111\u1ecbnh ngh\u0129a.", "jme.variables.empty definition": "\u0110\u1ecbnh ngh\u0129a c\u1ee7a bi\u1ebfn <code>{{name}}</code> c\u00f2n tr\u1ed1ng.", "jme.variables.circular reference": "Tham chi\u1ebfu v\u00f2ng tr\u00f2n c\u00e1c bi\u1ebfn theo \u0111\u1ecbnh ngh\u0129a c\u1ee7a <code> {{name}} </code>", "jme.variables.error computing dependency": "L\u1ed7i t\u00ednh to\u00e1n bi\u1ebfn tham chi\u1ebfu <code> {{name}} </code>", "jme.variables.error evaluating variable": "L\u1ed7i t\u00ednh gi\u00e1 tr\u1ecb bi\u1ebfn {{name}}: {{-message}}", "jme.variables.question took too many runs to generate variables": "M\u1ed9t t\u1eadp h\u1ee3p c\u00e1c c\u00e2u h\u1ecfi h\u1ee3p l\u1ec7 kh\u00f4ng \u0111\u01b0\u1ee3c t\u1ea1o ra k\u1ecbp th\u1eddi.", "jme.display.unknown token type": "Kh\u00f4ng th\u1ec3 x\u00e1c \u0111\u1ecbnh lo\u1ea1i m\u00e3 th\u00f4ng b\u00e1o {{type}}", "jme.display.collectRuleset.no sets": "Kh\u00f4ng c\u00f3 t\u1eadp n\u00e0o \u0111\u01b0\u1ee3c \u0111\u01b0a ra \u0111\u1ec3 collectRuleset!", "jme.display.collectRuleset.set not defined": "B\u1ed9 quy t\u1eafc {{name}} ch\u01b0a \u0111\u01b0\u1ee3c x\u00e1c \u0111\u1ecbnh", "jme.display.simplifyTree.no scope given": "Numbas.jme.display.simplifyTree ph\u1ea3i \u0111\u01b0\u1ee3c cung c\u1ea5p Scope", "math.precround.complex": "Kh\u00f4ng th\u1ec3 l\u00e0m tr\u00f2n s\u1ed1 ph\u1ee9c", "math.siground.complex": "Kh\u00f4ng th\u1ec3 l\u00e0m tr\u00f2n m\u1ed9t s\u1ed1 ph\u1ee9c", "math.combinations.complex": "Kh\u00f4ng th\u1ec3 t\u00ednh s\u1ed1 t\u1ed5 h\u1ee3p c\u1ee7a c\u00e1c s\u1ed1 ph\u1ee9c", "math.permutations.complex": "Kh\u00f4ng th\u1ec3 t\u00ednh s\u1ed1 ho\u00e1n v\u1ecb c\u1ee7a s\u1ed1 ph\u1ee9c", "math.gcf.complex": "Kh\u00f4ng th\u1ec3 t\u00ecm UCLN c\u1ee7a s\u1ed1 ph\u1ee9c", "math.lcm.complex": "Kh\u00f4ng th\u1ec3 t\u00ecm b\u1ed9i chung nh\u1ecf nh\u1ea5t c\u1ee7a c\u00e1c s\u1ed1 ph\u1ee9c", "math.lt.order complex numbers": "Kh\u00f4ng th\u1ec3 s\u1eafp x\u1ebfp th\u1ee9 t\u1ef1 c\u00e1c s\u1ed1 ph\u1ee9c", "math.choose.empty selection": "L\u1ef1a ch\u1ecdn tr\u1ed1ng cho h\u00e0m ng\u1eabu nhi\u00ean", "matrixmath.abs.non-square": "Kh\u00f4ng th\u1ec3 t\u00ednh \u0111\u1ecbnh th\u1ee9c c\u1ee7a m\u1ed9t ma tr\u1eadn kh\u00f4ng vu\u00f4ng.", "matrixmath.abs.too big": "Xin l\u1ed7i, kh\u00f4ng th\u1ec3 t\u00ednh \u0111\u1ecbnh th\u1ee9c c\u1ee7a m\u1ed9t ma tr\u1eadn c\u00f3 k\u00edch th\u01b0\u1edbc l\u1edbn h\u01a1n 3x3.", "matrixmath.mul.different sizes": "Kh\u00f4ng th\u1ec3 nh\u00e2n c\u00e1c ma tr\u1eadn c\u00f3 k\u00edch th\u01b0\u1edbc kh\u00e1c nhau.", "vectormath.cross.not 3d": "Ch\u1ec9 c\u00f3 th\u1ec3 l\u1ea5y t\u00edch ch\u00e9o c\u1ee7a c\u00e1c v\u00e9c-t\u01a1 3 chi\u1ec1u.", "vectormath.dot.matrix too big": "Kh\u00f4ng th\u1ec3 t\u00ednh t\u00edch v\u00f4 h\u01b0\u1edbng c\u1ee7a ma tr\u1eadn kh\u00f4ng ph\u1ea3i l\u00e0 $ 1 \\times N $ ho\u1eb7c $ N \\times 1 $.", "vectormath.cross.matrix too big": "Kh\u00f4ng th\u1ec3 t\u00ednh t\u00edch ch\u00e9o c\u1ee7a ma tr\u1eadn kh\u00f4ng ph\u1ea3i l\u00e0 $ 1 \\times N $ ho\u1eb7c $ N \\times 1 $.", "part.with steps answer prompt": "Tr\u1ea3 l\u1eddi:", "part.script.error": "L\u1ed7i trong ph\u1ea7n {{path}} t\u1eadp l\u1ec7nh t\u00f9y ch\u1ec9nh {{script}}: {{-message}}", "part.marking.steps no matter": "V\u00ec b\u1ea1n \u0111\u00e3 nh\u1eadn \u0111\u01b0\u1ee3c \u0111\u1ee7 \u0111i\u1ec3m cho ph\u1ea7n n\u00e0y, c\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n cho c\u00e1c b\u01b0\u1edbc kh\u00f4ng \u0111\u01b0\u1ee3c t\u00ednh.", "part.marking.revealed steps no penalty": "You revealed the steps.", "part.marking.used variable replacements": "Ph\u1ea7n \u0111\u01b0\u1ee3c \u0111\u00e1nh d\u1ea5u s\u1eed d\u1ee5ng c\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a ph\u1ea7n tr\u01b0\u1edbc.", "part.marking.variable replacement part not answered": "B\u1ea1n ph\u1ea3i tr\u1ea3 l\u1eddi {{part}} tr\u01b0\u1edbc.", "part.marking.resubmit because of variable replacement": "Vi\u1ec7c ch\u1ea5m \u0111i\u1ec3m ph\u1ea7n n\u00e0y ph\u1ee5 thu\u1ed9c v\u00e0o c\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n cho c\u00e1c ph\u1ea7n kh\u00e1c m\u00e0 b\u1ea1n \u0111\u00e3 thay \u0111\u1ed5i. G\u1eedi ph\u1ea7n n\u00e0y m\u1ed9t l\u1ea7n n\u1eefa \u0111\u1ec3 c\u1eadp nh\u1eadt \u0111i\u1ec3m s\u1ed1 c\u1ee7a b\u1ea1n.", "part.marking.not submitted": "Kh\u00f4ng c\u00e2u tr\u1ea3 l\u1eddi n\u00e0o \u0111\u01b0\u1ee3c g\u1eedi.", "part.marking.did not answer": "B\u1ea1n ch\u01b0a tr\u1ea3 l\u1eddi c\u00e2u h\u1ecfi n\u00e0y.", "part.marking.nothing entered": "B\u1ea1n ch\u01b0a nh\u1eadp c\u00e2u tr\u1ea3 l\u1eddi.", "part.marking.incorrect": "C\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n ch\u01b0a \u0111\u00fang.", "part.marking.correct": "C\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n \u0111\u00fang.", "part.marking.uncaught error": "L\u1ed7i khi \u0111\u00e1nh d\u1ea5u: {{-message}}", "part.marking.no result": "This part could not be marked.", "part.correct answer": "C\u00e2u tr\u1ea3 l\u1eddi d\u1ef1 ki\u1ebfn:", "part.missing type attribute": "{{part}}: Thi\u1ebfu thu\u1ed9c t\u00ednh ki\u1ec3u", "part.unknown type": "{{part}}: Kh\u00f4ng nh\u1eadn d\u1ea1ng \u0111\u01b0\u1ee3c ki\u1ec3u {{type}}", "part.setting not present": "Thu\u1ed9c t\u00ednh '{{property}}' ch\u01b0a \u0111\u01b0\u1ee3c \u0111\u1eb7t", "part.jme.answer missing": "C\u00e2u tr\u1ea3 l\u1eddi \u0111\u00fang b\u1ecb thi\u1ebfu", "part.jme.answer too long": "C\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n qu\u00e1 d\u00e0i.", "part.jme.answer too short": "C\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n qu\u00e1 ng\u1eafn.", "part.jme.answer invalid": "C\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n kh\u00f4ng ph\u1ea3i l\u00e0 bi\u1ec3u th\u1ee9c to\u00e1n h\u1ecdc h\u1ee3p l\u1ec7. <br/> {{-message}}.", "part.jme.marking.correct": "C\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n c\u00f3 gi\u00e1 tr\u1ecb \u0111\u00fang.", "part.jme.must-have bits": "<span class = \"monospace\"> {{string}} </span>", "part.jme.must-have one": "C\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n ph\u1ea3i c\u00f3: {{strings}}", "part.jme.must-have several": "C\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n ph\u1ea3i ch\u1ee9a t\u1ea5t c\u1ea3: {{strings}}", "part.jme.not-allowed bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.not-allowed one": "C\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n kh\u00f4ng \u0111\u01b0\u1ee3c ch\u1ee9a: {{strings}}", "part.jme.not-allowed several": "C\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n kh\u00f4ng \u0111\u01b0\u1ee3c ch\u1ee9a b\u1ea5t k\u1ef3 t\u1eeb n\u00e0o trong: {{strings}}", "part.jme.unexpected variable name": "C\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n \u0111\u00e3 \u0111\u01b0\u1ee3c di\u1ec5n gi\u1ea3i \u0111\u1ec3 s\u1eed d\u1ee5ng t\u00ean bi\u1ebfn kh\u00f4ng mong mu\u1ed1n <code> {{name}} </code>.", "part.jme.unexpected variable name suggestion": "C\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n \u0111\u00e3 \u0111\u01b0\u1ee3c di\u1ec5n gi\u1ea3i \u0111\u1ec3 s\u1eed d\u1ee5ng t\u00ean bi\u1ebfn kh\u00f4ng mong mu\u1ed1n <code> {{name}} </code>. \u00dd c\u1ee7a b\u1ea1n l\u00e0 <code> {{suggest}} </code>?", "part.patternmatch.display answer missing": "Hi\u1ec3n th\u1ecb c\u00e2u tr\u1ea3 l\u1eddi thi\u1ebfu.", "part.patternmatch.correct except case": "C\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n l\u00e0 ch\u00ednh x\u00e1c, ngo\u1ea1i tr\u1eeb tr\u01b0\u1eddng h\u1ee3p.", "part.numberentry.correct except decimal": "C\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n n\u1eb1m trong ph\u1ea1m vi cho ph\u00e9p, nh\u01b0ng s\u1ed1 th\u1eadp ph\u00e2n kh\u00f4ng \u0111\u01b0\u1ee3c ch\u1ea5p nh\u1eadn.", "part.numberentry.correct except fraction": "C\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n n\u1eb1m trong ph\u1ea1m vi cho ph\u00e9p, nh\u01b0ng ph\u00e2n s\u1ed1 kh\u00f4ng \u0111\u01b0\u1ee3c ch\u1ea5p nh\u1eadn.", "part.numberentry.answer invalid": "B\u1ea1n ch\u01b0a nh\u1eadp m\u1ed9t s\u1ed1 h\u1ee3p l\u1ec7.", "part.numberentry.answer not integer": "C\u00e2u tr\u1ea3 l\u1eddi kh\u00f4ng h\u1ee3p l\u1ec7. B\u1ea1n ph\u1ea3i nh\u1eadp m\u1ed9t s\u1ed1 nguy\u00ean, kh\u00f4ng ph\u1ea3i m\u1ed9t s\u1ed1 th\u1eadp ph\u00e2n.", "part.numberentry.answer not integer or decimal": "C\u00e2u tr\u1ea3 l\u1eddi kh\u00f4ng h\u1ee3p l\u1ec7. B\u1ea1n ph\u1ea3i nh\u1eadp m\u1ed9t s\u1ed1 nguy\u00ean ho\u1eb7c m\u1ed9t s\u1ed1 th\u1eadp ph\u00e2n.", "part.numberentry.zero sig fig": "Ph\u1ea7n n\u00e0y \u0111\u01b0\u1ee3c thi\u1ebft l\u1eadp \u0111\u1ec3 l\u00e0m tr\u00f2n c\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a h\u1ecdc sinh th\u00e0nh c\u00e1c s\u1ed1 nguy\u00ean.", "part.mcq.options def not a list": "Bi\u1ec3u th\u1ee9c x\u00e1c \u0111\u1ecbnh {{property}} kh\u00f4ng ph\u1ea3i l\u00e0 danh s\u00e1ch.", "part.mcq.marking matrix string empty": "Bi\u1ec3u th\u1ee9c ma tr\u1eadn \u0111\u00e1nh d\u1ea5u t\u00f9y ch\u1ec9nh c\u00f2n tr\u1ed1ng.", "part.mcq.choices missing": "\u0110\u1ecbnh ngh\u0129a c\u1ee7a s\u1ef1 l\u1ef1a ch\u1ecdn \u0111ang thi\u1ebfu", "part.mcq.matrix not a number": "Ph\u1ea7n {{part}} \u00f4 ma tr\u1eadn \u0111\u00e1nh d\u1ea5u ({{row}}, {{col}}) kh\u00f4ng c\u00f3 k\u1ebft qu\u1ea3 l\u00e0 m\u1ed9t s\u1ed1", "part.mcq.wrong number of choices": "B\u1ea1n \u0111\u00e3 ch\u1ecdn sai s\u1ed1 l\u01b0\u1ee3ng c\u00e2u tr\u1ea3 l\u1eddi.", "part.mcq.no choices selected": "Kh\u00f4ng c\u00f3 c\u00e2u tr\u1ea3 l\u1eddi n\u00e0o \u0111\u01b0\u1ee3c ch\u1ecdn.", "part.mcq.matrix not a list": "Ma tr\u1eadn \u0111\u00e1nh d\u1ea5u, \u0111\u01b0\u1ee3c x\u00e1c \u0111\u1ecbnh b\u1edfi bi\u1ec3u th\u1ee9c JME, kh\u00f4ng ph\u1ea3i l\u00e0 m\u1ed9t danh s\u00e1ch nh\u01b0ng n\u00f3 n\u00ean nh\u01b0 v\u1eady.", "part.mcq.matrix wrong type": "Ph\u1ea7n t\u1eed c\u1ee7a lo\u1ea1i kh\u00f4ng h\u1ee3p l\u1ec7 '{{type}}' \u0111\u01b0\u1ee3c s\u1eed d\u1ee5ng trong ma tr\u1eadn \u0111\u00e1nh d\u1ea5u.", "part.mcq.matrix mix of numbers and lists": "Tr\u1ed9n l\u1eabn s\u1ed1 v\u00e0 danh s\u00e1ch \u0111ang \u0111\u01b0\u1ee3c s\u1eed d\u1ee5ng trong ma tr\u1eadn \u0111\u00e1nh d\u1ea5u.", "part.mcq.matrix wrong size": "Ma tr\u1eadn \u0111\u00e1nh d\u1ea5u c\u00f3 k\u00edch th\u01b0\u1edbc sai.", "part.mcq.correct choice": "B\u1ea1n \u0111\u00e3 ch\u1ecdn m\u1ed9t c\u00e2u tr\u1ea3 l\u1eddi \u0111\u00fang.", "part.matrix.answer invalid": "C\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n kh\u00f4ng h\u1ee3p l\u1ec7.", "part.matrix.invalid cell": "M\u1ed9t ho\u1eb7c nhi\u1ec1u \u00f4 trong c\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n tr\u1ed1ng ho\u1eb7c kh\u00f4ng h\u1ee3p l\u1ec7.", "part.matrix.some incorrect": "M\u1ed9t ho\u1eb7c nhi\u1ec1u \u00f4 trong c\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n l\u00e0 kh\u00f4ng ch\u00ednh x\u00e1c, nh\u01b0ng b\u1ea1n \u0111\u00e3 \u0111\u01b0\u1ee3c ch\u1ea5m \u0111i\u1ec3m cho ph\u1ea7n c\u00f2n l\u1ea1i.", "part.matrix.empty": "B\u1ea1n ch\u01b0a nh\u1eadp c\u00e2u tr\u1ea3 l\u1eddi.", "part.matrix.empty cell": "M\u1ed9t ho\u1eb7c nhi\u1ec1u \u00f4 trong c\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n tr\u1ed1ng.", "part.matrix.size mismatch": "T\u00e1c gi\u1ea3 c\u00e2u h\u1ecfi \u0111\u00e3 kh\u00f4ng cho ph\u00e9p h\u1ecdc sinh quy\u1ebft \u0111\u1ecbnh k\u00edch th\u01b0\u1edbc c\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a h\u1ecd, nh\u01b0ng c\u00e2u tr\u1ea3 l\u1eddi \u0111\u00fang l\u00e0 {{correct_dimensions}} trong khi \u0111\u1ea7u v\u00e0o c\u00e2u tr\u1ea3 l\u1eddi l\u00e0 {{input_dimensions}}", "part.gapfill.feedback header": "<strong> {{name}} </strong>", "part.extension.not implemented": "M\u1ed9t ph\u1ea7n ch\u01b0a \u0111\u01b0\u1ee3c tri\u1ec3n khai theo c\u00e1ch <code> {{name}} </code>.", "question.loaded name mismatch": "Kh\u00f4ng th\u1ec3 ti\u1ebfp t\u1ee5c n\u1ed7 l\u1ef1c n\u00e0y - th\u01b0 vi\u1ec7n \u0111\u00e3 thay \u0111\u1ed5i k\u1ec3 t\u1eeb phi\u00ean tr\u01b0\u1edbc.", "question.error": "C\u00e2u h\u1ecfi {{number}}: {{-message}}", "question.preamble.error": "L\u1ed7i trong preamble: {{-message}}", "question.preamble.syntax error": "L\u1ed7i c\u00fa ph\u00e1p trong preamble", "question.unsupported part type": "Ki\u1ec3u kh\u00f4ng \u0111\u01b0\u1ee3c h\u1ed7 tr\u1ee3", "question.header": "C\u00e2u h\u1ecfi {{number}}", "question.submit part": "G\u1eedi m\u1ed9t ph\u1ea7n", "question.show steps": "Xem c\u00e1c b\u01b0\u1edbc", "question.show steps penalty": "B\u1ea1n s\u1ebd m\u1ea5t <strong>{{count,niceNumber}}</strong> $t(mark).", "question.show steps no penalty": "\u0110i\u1ec3m c\u1ee7a b\u1ea1n s\u1ebd kh\u00f4ng b\u1ecb \u1ea3nh h\u01b0\u1edfng.", "question.show steps already penalised": "B\u1ea1n \u0111\u00e3 hi\u1ec3n th\u1ecb c\u00e1c b\u01b0\u1edbc. B\u1ea1n c\u00f3 th\u1ec3 hi\u1ec3n th\u1ecb l\u1ea1i ch\u00fang m\u00e0 kh\u00f4ng b\u1ecb ph\u1ea1t th\u00eam.", "question.hide steps": "\u1ea8n c\u00e1c b\u01b0\u1edbc", "question.hide steps no penalty": "\u0110i\u1ec3m c\u1ee7a b\u1ea1n s\u1ebd kh\u00f4ng b\u1ecb \u1ea3nh h\u01b0\u1edfng.", "question.advice": "G\u1ee3i \u00fd", "question.no such part": "Kh\u00f4ng t\u00ecm th\u1ea5y ph\u1ea7n {{path}}", "question.can not submit": "Kh\u00f4ng th\u1ec3 g\u1eedi c\u00e2u tr\u1ea3 l\u1eddi - ki\u1ec3m tra l\u1ed7i.", "question.answer submitted": "C\u00e2u tr\u1ea3 l\u1eddi \u0111\u00e3 \u0111\u01b0\u1ee3c g\u1eedi", "question.score feedback.show": "Xem ph\u1ea3n h\u1ed3i", "question.score feedback.hide": "\u1ea8n ph\u1ea3n h\u1ed3i", "question.score feedback.answered total actual": "Score: {{score,niceNumber}}/{{marks,niceNumber}}", "question.score feedback.answered total": "{{marksString}}. Answered.", "question.score feedback.answered actual": "Score: {{scoreString}}", "question.score feedback.answered": "\u0110\u00e3 tr\u1ea3 l\u1eddi", "question.score feedback.unanswered": "Ch\u01b0a tr\u1ea3 l\u1eddi", "question.score feedback.unanswered total": "{{marksString}}.", "question.score feedback.correct": "C\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n ch\u01b0a \u0111\u00fang", "question.score feedback.partial": "C\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n \u0111\u00fang m\u1ed9t ph\u1ea7n", "question.score feedback.wrong": "C\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n ch\u01b0a \u0111\u00fang", "question.selector.unsubmitted changes": "Thay \u0111\u1ed5i ch\u01b0a \u0111\u01b0\u1ee3c g\u1eedi \u0111i.", "timing.no accumulator": "kh\u00f4ng c\u00f3 b\u1ed9 \u0111\u1ebfm th\u1eddi gian {{name}}", "timing.time remaining": "Th\u1eddi gian c\u00f2n l\u1ea1i:", "xml.could not load": "Kh\u00f4ng th\u1ec3 n\u1ea1p t\u1ec7p XML: {{-message}}", "xml.property not number": "Thu\u1ed9c t\u00ednh {{name}} ph\u1ea3i l\u00e0 m\u1ed9t s\u1ed1, nh\u01b0ng kh\u00f4ng ph\u1ea3i ({{value}}), trong n\u00fat {{element}}", "xml.property not boolean": "Thu\u1ed9c t\u00ednh {{name}} ph\u1ea3i l\u00e0 ki\u1ec3u boolean, nh\u01b0ng kh\u00f4ng ph\u1ea3i ({{value}}), trong n\u00fat {{element}}", "xml.error in variable definition": "Error in definition of variable <code>{{name}}</code>", "scorm.error initialising": "L\u1ed7i kh\u1edfi t\u1ea1o giao th\u1ee9c SCORM: {{-message}}", "scorm.failed save": "<p> Y\u00eau c\u1ea7u l\u01b0u d\u1eef li\u1ec7u v\u00e0o m\u00e1y ch\u1ee7 kh\u00f4ng th\u00e0nh c\u00f4ng. Nh\u1ea5n <b> OK </b> \u0111\u1ec3 th\u1eed l\u1ea1i. </P>\n<p> N\u1ebfu b\u1ea1n nh\u1eadn \u0111\u01b0\u1ee3c th\u00f4ng b\u00e1o n\u00e0y nhi\u1ec1u l\u1ea7n, h\u00e3y ki\u1ec3m tra k\u1ebft n\u1ed1i internet c\u1ee7a b\u1ea1n ho\u1eb7c s\u1eed d\u1ee5ng m\u00e1y t\u00ednh kh\u00e1c. C\u00e1c c\u00e2u tr\u1ea3 l\u1eddi \u0111\u00e3 g\u1eedi tr\u01b0\u1edbc \u0111\u00f3 c\u1ee7a b\u1ea1n \u0111\u00e3 \u0111\u01b0\u1ee3c l\u01b0u th\u00e0nh c\u00f4ng v\u00e0 s\u1ebd \u0111\u01b0\u1ee3c kh\u00f4i ph\u1ee5c n\u1ebfu b\u1ea1n ti\u1ebfp t\u1ee5c phi\u00ean n\u00e0y tr\u00ean m\u1ed9t m\u00e1y t\u00ednh kh\u00e1c. </P>\n<p> N\u1ebfu th\u00f4ng b\u00e1o n\u00e0y xu\u1ea5t hi\u1ec7n li\u00ean t\u1ee5c v\u00e0 b\u1ea1n kh\u00f4ng th\u1ec3 l\u01b0u <em> b\u1ea5t k\u1ef3 c\u00e2u tr\u1ea3 l\u1eddi </em> n\u00e0o, vui l\u00f2ng li\u00ean h\u1ec7 v\u1edbi gi\u1ea3ng vi\u00ean ho\u1eb7c gi\u00e1o vi\u00ean c\u1ee7a b\u1ea1n. </p>", "scorm.no exam suspend data": "Kh\u00f4ng th\u1ec3 ti\u1ebfp t\u1ee5c: kh\u00f4ng c\u00f3 b\u00e0i thi n\u00e0o \u0111ang l\u00e0m d\u1edf.", "scorm.error loading suspend data": "L\u1ed7i khi t\u1ea3i d\u1eef li\u1ec7u t\u1ea1m ng\u01b0ng: {{-message}}", "scorm.error loading question": "L\u1ed7i khi n\u1ea1p c\u00e2u h\u1ecfi {{number}}: {{-message}}", "scorm.no question suspend data": "Kh\u00f4ng c\u00f3 c\u00e2u h\u1ecfi \u0111ang l\u00e0m d\u1edf", "scorm.error loading part": "L\u1ed7i khi t\u1ea3i ph\u1ea7n {{part}}: {{-message}}", "scorm.no part suspend data": "Kh\u00f4ng c\u00f3 ph\u1ea7n d\u1eef li\u1ec7u \u0111ang l\u00e0m d\u1edf", "util.product.non list": "\u0110\u00e3 v\u01b0\u1ee3t qua m\u1ed9t danh s\u00e1ch kh\u00f4ng ph\u1ea3i l\u00e0 <code> Numbas.util.product </code>", "mark": "\u0111\u00e1nh d\u1ea5u", "was": "\u0111\u00e3 l\u00e0", "part": "ph\u1ea7n", "gap": "l\u1ed7 h\u1ed5ng", "step": "b\u01b0\u1edbc", "jme.substituteTree.undefined variable": "Bi\u1ebfn kh\u00f4ng x\u00e1c \u0111\u1ecbnh <code>{{name}}</code>", "jme.user javascript.returned undefined": "H\u00e0m javascript do ng\u01b0\u1eddi d\u00f9ng \u0111\u1ecbnh ngh\u0129a <code> {{name}} </code> \u0111\u01b0\u1ee3c tr\u1ea3 v\u1ec1 <code> kh\u00f4ng x\u00e1c \u0111\u1ecbnh </code>.", "part.marking.steps change": "B\u1ea1n \u0111\u00e3 \u0111\u01b0\u1ee3c trao <strong> {{Count, beautifulNumber}} </strong> $ t (\u0111\u00e1nh d\u1ea5u) cho c\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n cho c\u00e1c b\u01b0\u1edbc.", "part.marking.revealed steps with penalty": "You revealed the steps. The maximum you can score for this part is <strong>{{count,niceNumber}}</strong> $t(mark). Your scores will be scaled down accordingly.", "part.marking.total score": "B\u1ea1n \u0111\u00e3 ghi <strong> {{Count, beautifulNumber}} </strong> $ t (mark) cho ph\u1ea7n n\u00e0y.", "part.numberentry.precision type.dp": "v\u1ecb tr\u00ed d\u1ea5u th\u1eadp ph\u00e2n", "part.numberentry.precision type.dp_plural": "d\u1ea5u ph\u1ea3y th\u1eadp ph\u00e2n", "part.numberentry.precision type.sigfig": "ch\u1eef s\u1ed1 c\u00f3 ngh\u0129a", "part.numberentry.precision type.sigfig_plural": "ch\u1eef s\u1ed1 c\u00f3 ngh\u0129a", "part.numberentry.give your answer to precision": "L\u00e0m tr\u00f2n k\u1ebft qu\u1ea3 \u0111\u1ebfn {{count,niceNumber}} {{precisionType}}.", "question.unsubmitted changes": "B\u1ea1n \u0111\u00e3 th\u1ef1c hi\u1ec7n m\u1ed9t thay \u0111\u1ed5i cho c\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n nh\u01b0ng ch\u01b0a g\u1eedi n\u00f3. Vui l\u00f2ng ki\u1ec3m tra c\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n v\u00e0 sau \u0111\u00f3 nh\u1ea5n n\u00fat <strong> G\u1eedi c\u00e2u tr\u1ea3 l\u1eddi </strong>.", "question.unsubmitted changes_plural": "B\u1ea1n \u0111\u00e3 th\u1ef1c hi\u1ec7n thay \u0111\u1ed5i cho c\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n nh\u01b0ng ch\u01b0a g\u1eedi ch\u00fang. Vui l\u00f2ng ki\u1ec3m tra c\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n cho t\u1eebng ph\u1ea7n v\u00e0 sau \u0111\u00f3 nh\u1ea5n n\u00fat <strong> G\u1eedi t\u1ea5t c\u1ea3 c\u00e1c ph\u1ea7n </strong>.", "util.equality not defined for type": "So s\u00e1nh ch\u01b0a \u0111\u01b0\u1ee3c \u0111\u1ecbnh ngh\u0129a cho ki\u1ec3u {{type}}", "mark_plural": "\u0111\u00e1nh d\u1ea5u", "was_plural": "\u0111\u00e3 l\u00e0", "die.script not loaded": "Numbas kh\u00f4ng th\u1ec3 b\u1eaft \u0111\u1ea7u v\u00ec t\u1ec7p <code> {{file}} </code> kh\u00f4ng \u0111\u01b0\u1ee3c t\u1ea3i. Ki\u1ec3m tra xem n\u00f3 c\u00f3 trong <code>scripts.js</code> kh\u00f4ng.", "math.combinations.n less than zero": "Kh\u00f4ng th\u1ec3 t\u00ednh s\u1ed1 t\u1ed5 h\u1ee3p: n \u0111ang nh\u1ecf h\u01a1n 0", "math.combinations.k less than zero": "Kh\u00f4ng th\u1ec3 t\u00ednh s\u1ed1 t\u1ed5 h\u1ee3p: k \u0111ang nh\u1ecf h\u01a1n 0", "math.combinations.n less than k": "Kh\u00f4ng th\u1ec3 t\u00ednh s\u1ed1 t\u1ed5 h\u1ee3p: n \u0111ang nh\u1ecf h\u01a1n k", "math.permutations.n less than zero": "Kh\u00f4ng th\u1ec3 t\u00ednh s\u1ed1 ho\u00e1n v\u1ecb: n \u0111ang nh\u1ecf h\u01a1n 0", "math.permutations.k less than zero": "Kh\u00f4ng th\u1ec3 t\u00ednh s\u1ed1 ho\u00e1n v\u1ecb: k \u0111ang nh\u1ecf h\u01a1n 0", "math.permutations.n less than k": "Kh\u00f4ng th\u1ec3 t\u00ednh s\u1ed1 ho\u00e1n v\u1ecb: n \u0111ang nh\u1ecf h\u01a1n k", "part.numberentry.give your answer to precision_0": "L\u00e0m tr\u00f2n k\u1ebft qu\u1ea3 \u0111\u1ebfn s\u1ed1 nguy\u00ean g\u1ea7n nh\u1ea5t", "mathjax.error": "L\u1ed7i khi th\u1ef1c thi MathJax: {{-message}}", "mathjax.error with context": "L\u1ed7i th\u1ef1c thi MathJax trong {{-context}}: {{-message}}", "exam.introduction": "Gi\u1edbi thi\u1ec7u b\u00e0i thi", "exam.feedback": "Th\u00f4ng b\u00e1o ph\u1ea3n h\u1ed3i b\u00e0i ki\u1ec3m tra", "jme.tokenise.keypair key not a string": "Dictionary key ph\u1ea3i l\u00e0 string, kh\u00f4ng th\u1ec3 l\u00e0 {{type}}.", "jme.shunt.list mixed argument types": "Kh\u00f4ng th\u1ec3 th\u1ef1c thi {{mode}}: k\u1ebft h\u1ee3p c\u00e1c y\u1ebfu t\u1ed1 t\u1eeb \u0111i\u1ec3n v\u00e0 danh s\u00e1ch", "jme.func.listval.key not in dict": "T\u1eeb \u0111i\u1ec3n kh\u00f4ng ch\u1ee9a kh\u00f3a <code> {{key}} </code>", "part.prompt": "l\u1eddi nh\u1eafc", "part.feedback": "ph\u1ea3n h\u1ed3i", "part.numberentry.answer not reduced": "C\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n ch\u01b0a \u0111\u01b0\u1ee3c r\u00fat g\u1ecdn", "part.numberentry.give your answer as a reduced fraction": "R\u00fat g\u1ecdn c\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n", "part.numberentry.negative decimal places": "Ph\u1ea7n n\u00e0y \u0111\u01b0\u1ee3c thi\u1ebft l\u1eadp \u0111\u1ec3 l\u00e0m tr\u00f2n c\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a h\u1ecdc sinh cho m\u1ed9t s\u1ed1 th\u1eadp ph\u00e2n \u00e2m, kh\u00f4ng c\u00f3 ngh\u0129a.", "part.mcq.choices": "c\u00e1c l\u1ef1a ch\u1ecdn", "part.mcq.answers": "c\u00e1c c\u00e2u tr\u1ea3 l\u1eddi", "part.mcq.matrix cell empty": "Ph\u1ea7n {{part}} \u00f4 ma tr\u1eadn \u0111\u00e1nh d\u1ea5u ({{row}}, {{col}}) tr\u1ed1ng", "part.mcq.matrix jme error": "Ph\u1ea7n {{part}} \u00f4 ma tr\u1eadn \u0111\u00e1nh d\u1ea5u ({{row}}, {{col}}) \u0111\u01b0a ra l\u1ed7i JME: {{-error}}", "question.statement": "Tuy\u00ean b\u1ed1", "ruleset.circular reference": "Tham chi\u1ebfu tr\u00f2n trong \u0111\u1ecbnh ngh\u0129a c\u1ee7a quy t\u1eafc <code> {{name}} </code>", "ruleset.set not defined": "Quy t\u1eafc {{name}} ch\u01b0a \u0111\u01b0\u1ee3c x\u00e1c \u0111\u1ecbnh", "jme.evaluate.no scope given": "Numbas.jme.evalu ph\u1ea3i \u0111\u01b0\u1ee3c cung c\u1ea5p Scope", "answer.number.not a number": "C\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n kh\u00f4ng ph\u1ea3i l\u00e0 m\u1ed9t s\u1ed1 h\u1ee3p l\u1ec7.", "answer.number.fractions not allowed": "B\u1ea1n kh\u00f4ng th\u1ec3 nh\u1eadp m\u1ed9t ph\u00e2n s\u1ed1.", "answer.jme.invalid expression": "{{-message}}", "answer.matrix.fractions not allowed": "B\u1ea1n kh\u00f4ng th\u1ec3 nh\u1eadp m\u1ed9t ph\u00e2n s\u1ed1.", "answer.matrix.some cell not a number": "M\u1ed9t ho\u1eb7c nhi\u1ec1u \u00f4 trong c\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n kh\u00f4ng ph\u1ea3i l\u00e0 s\u1ed1 h\u1ee3p l\u1ec7.", "exam.enter password": "M\u1eadt kh\u1ea9u:", "exam.password.correct": "M\u1eadt kh\u1ea9u \u0111\u00fang. B\u1ea1n c\u00f3 th\u1ec3 b\u1eaft \u0111\u1ea7u thi.", "exam.password.incorrect": "M\u1eadt kh\u1ea9u ch\u01b0a \u0111\u00fang.", "frontpage.scorm.lms not connected": "B\u00e0i ki\u1ec3m tra n\u00e0y \u0111ang ch\u1ea1y trong ch\u1ebf \u0111\u1ed9 \u0111\u1ed9c l\u1eadp. C\u00e2u tr\u1ea3 l\u1eddi v\u00e0 \u0111i\u1ec3m c\u1ee7a b\u1ea1n s\u1ebd kh\u00f4ng \u0111\u01b0\u1ee3c l\u01b0u!", "result.question review": "Xem l\u1ea1i", "control.confirm regen no marks": "B\u1ea1n c\u00f3 mu\u1ed1n t\u1ea1o ng\u1eabu nhi\u00ean l\u1ea1i c\u00e2u h\u1ecfi n\u00e0y?", "control.confirm reveal no marks": "B\u1ea1n c\u00f3 mu\u1ed1n xem c\u00e2u tr\u1ea3 l\u1eddi cho c\u00e2u h\u1ecfi n\u00e0y?", "jme.tokenise.invalid near": "Bi\u1ec3u th\u1ee9c kh\u00f4ng h\u1ee3p l\u1ec7: <code> {{bi\u1ec3u th\u1ee9c}} </code> t\u1ea1i v\u1ecb tr\u00ed {{v\u1ecb tr\u00ed}} g\u1ea7n <code> {{l\u00e2n c\u1eadn}} </code>", "jme.tokenise.number.object not complex": "M\u1ed9t \u0111\u1ed1i t\u01b0\u1ee3ng kh\u00f4ng h\u1ee3p l\u1ec7 \u0111\u00e3 \u0111\u01b0\u1ee3c chuy\u1ec3n v\u00e0o h\u00e0m t\u1ea1o s\u1ed1.", "jme.subvars.null substitution": "Thay th\u1ebf bi\u1ebfn r\u1ed7ng: <code>$t(left brace){{str}}$t(right brace)", "jme.type.type already registered": "Ki\u1ec3u d\u1eef li\u1ec7u {{type}} \u0111\u00e3 \u0111\u01b0\u1ee3c \u0111\u0103ng k\u00fd n\u00ean kh\u00f4ng th\u1ec3 \u0111\u0103ng k\u00fd l\u1ea1i.", "jme.type.no cast method": "Kh\u00f4ng th\u1ec3 t\u1ef1 \u0111\u1ed9ng chuy\u1ec3n \u0111\u1ed5i t\u1eeb {{from}} sang {{to}}.", "jme.display.simplifyTree.empty expression": "Bi\u1ec3u th\u1ee9c \u0111ang tr\u1ed1ng", "jme.display.simplifyTree.stuck in a loop": "B\u1ed9 khu\u1ebfch \u0111\u1ea1i b\u1ecb k\u1eb9t trong m\u1ed9t v\u00f2ng l\u1eb7p: <code> {{expr}} </code>", "math.niceNumber.undefined": "\u0110\u00e3 mong \u0111\u1ee3i m\u1ed9t s\u1ed1, nh\u01b0ng c\u00f3 <code> kh\u00f4ng x\u00e1c \u0111\u1ecbnh </code>", "math.rangeToList.zero step size": "Kh\u00f4ng th\u1ec3 chuy\u1ec3n \u0111\u1ed5i m\u1ed9t kho\u1ea3ng v\u1edbi b\u01b0\u1edbc nh\u1ea3y b\u1eb1ng 0 th\u00e0nh m\u1ed9t danh s\u00e1ch.", "part.error": "{{path}}: {{-message}}", "part.marking.revealed steps": "B\u1ea1n \u0111\u00e3 xem c\u00e1c b\u01b0\u1edbc gi\u1ea3i.", "part.marking.maximum scaled down": "M\u1ee9c t\u1ed1i \u0111a b\u1ea1n c\u00f3 th\u1ec3 ghi cho ph\u1ea7n n\u00e0y l\u00e0 <strong> {{Count, beautifulNumber}} </strong> $ t (mark). \u0110i\u1ec3m s\u1ed1 c\u1ee7a b\u1ea1n s\u1ebd \u0111\u01b0\u1ee3c thu nh\u1ecf l\u1ea1i cho ph\u00f9 h\u1ee3p.", "part.marking.minimum score applied": "\u0110i\u1ec3m t\u1ed1i thi\u1ec3u cho ph\u1ea7n n\u00e0y l\u00e0 <strong> {{\u0111i\u1ec3m, NiceNumber}} </strong>.", "part.marking.maximum score applied": "\u0110i\u1ec3m t\u1ed1i \u0111a cho ph\u1ea7n n\u00e0y l\u00e0 <strong> {{\u0111i\u1ec3m, NiceNumber}} </strong>.", "part.marking.error in marking script": "C\u00f3 m\u1ed9t l\u1ed7i trong thu\u1eadt to\u00e1n \u0111\u00e1nh d\u1ea5u c\u1ee7a ph\u1ea7n n\u00e0y. H\u00e3y b\u00e1o c\u00e1o \u0111i\u1ec1u n\u00e0y. {{-message}}", "part.marking.no result after replacement": "Ph\u1ea7n n\u00e0y kh\u00f4ng th\u1ec3 \u0111\u01b0\u1ee3c \u0111\u00e1nh d\u1ea5u b\u1eb1ng c\u00e1ch s\u1eed d\u1ee5ng c\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n cho c\u00e1c ph\u1ea7n tr\u01b0\u1edbc.", "part.marking.missing required note": "Thu\u1eadt to\u00e1n \u0111\u00e1nh d\u1ea5u kh\u00f4ng x\u00e1c \u0111\u1ecbnh ghi ch\u00fa <code> {{note}} </code>", "marking.apply.not a list": "\u0110\u1ed1i s\u1ed1 \u0111\u1ea7u ti\u00ean \u0111\u1ec3 <code> \u00e1p d\u1ee5ng </code> ph\u1ea3i l\u00e0 m\u1ed9t danh s\u00e1ch v\u00e0 kh\u00f4ng", "marking.apply marking script.script not found": "Kh\u00f4ng t\u00ecm th\u1ea5y t\u1eadp l\u1ec7nh \u0111\u00e1nh d\u1ea5u <code> {{name}} </code>", "marking.note.compilation error": "L\u1ed7i bi\u00ean d\u1ecbch ghi ch\u00fa <code> {{name}} </code>: {{-message}}", "marking.note.error evaluating note": "L\u1ed7i \u0111\u00e1nh gi\u00e1 ghi ch\u00fa <code> {{name}} </code> - {{-message}}", "marking.note.invalid definition": "\u0110\u1ecbnh ngh\u0129a ghi ch\u00fa kh\u00f4ng h\u1ee3p l\u1ec7: <code> {{source}} </code>. {{-d\u1ea5u}}", "marking.note.invalid definition.missing colon": "B\u1ea1n c\u00f3 th\u1ec3 thi\u1ebfu d\u1ea5u hai ch\u1ea5m sau t\u00ean v\u00e0 m\u00f4 t\u1ea3", "marking.note.invalid definition.description missing closing bracket": "B\u1ea1n c\u00f3 th\u1ec3 \u0111ang thi\u1ebfu m\u1ed9t d\u1ea5u \u0111\u00f3ng ngo\u1eb7c", "marking.note.empty expression": "Ghi ch\u00fa <code> {{name}} </code> tr\u1ed1ng.", "marking.script.error parsing notes": "L\u1ed7i ph\u00e2n t\u00edch c\u00fa ph\u00e1p \u0111\u00e1nh d\u1ea5u t\u1eadp l\u1ec7nh: {{- message}}", "part.feedback out of date": "Ph\u1ea3n h\u1ed3i n\u00e0y d\u1ef1a tr\u00ean c\u00e2u tr\u1ea3 l\u1eddi cu\u1ed1i c\u00f9ng c\u1ee7a b\u1ea1n. G\u1eedi c\u00e2u tr\u1ea3 l\u1eddi m\u1edbi c\u1ee7a b\u1ea1n \u0111\u1ec3 nh\u1eadn \u0111\u01b0\u1ee3c ph\u1ea3n h\u1ed3i c\u1eadp nh\u1eadt.", "part.jme.invalid value generator expression": "Bi\u1ec3u th\u1ee9c t\u1ea1o gi\u00e1 tr\u1ecb kh\u00f4ng h\u1ee3p l\u1ec7 cho bi\u1ebfn <code> {{name}} </code>: {{-message}}", "part.mcq.incorrect choice": "B\u1ea1n \u0111\u00e3 ch\u1ecdn m\u1ed9t c\u00e2u tr\u1ea3 l\u1eddi kh\u00f4ng ch\u00ednh x\u00e1c.", "part.matrix.not all cells same precision": "C\u00e1c \u00f4 trong c\u00e2u tr\u1ea3 l\u1eddi ch\u01b0a c\u00f3 c\u00f9ng \u0111\u1ed9 ch\u00ednh x\u00e1c.", "part.gapfill.error marking gap": "L\u1ed7i \u0111\u00e1nh d\u1ea5u {{name}}: {{-message}}", "part.custom.empty setting": "Kh\u00f4ng c\u00f3 gi\u00e1 tr\u1ecb \u0111\u01b0\u1ee3c \u0111\u01b0a ra.", "part.custom.unrecognised input type": "Lo\u1ea1i c\u00e0i \u0111\u1eb7t kh\u00f4ng \u0111\u01b0\u1ee3c nh\u1eadn d\u1ea1ng <code> {{input_type}} </code>", "part.custom.error evaluating input option": "L\u1ed7i \u0111\u00e1nh gi\u00e1 t\u00f9y ch\u1ecdn \u0111\u1ea7u v\u00e0o <code> {{t\u00f9y ch\u1ecdn}} </code>: {{-error}}", "part.custom.input option missing": "\u0110\u1ecbnh ngh\u0129a t\u00f9y ch\u1ecdn \u0111\u1ea7u v\u00e0o <code> {{t\u00f9y ch\u1ecdn}} </code> b\u1ecb thi\u1ebfu.", "part.custom.error evaluating setting": "L\u1ed7i \u0111\u00e1nh gi\u00e1 c\u00e0i \u0111\u1eb7t <code> {{settings}} </code>: {{-error}}", "question.error creating question": "L\u1ed7i khi t\u1ea1o c\u00e2u h\u1ecfi {{number}}: {{-message}}", "question.score feedback.not marked": "Ch\u01b0a \u0111\u01b0\u1ee3c \u0111\u00e1nh d\u1ea5u.", "question.score feedback.partially answered": "\u0110\u01b0\u1ee3c tr\u1ea3 l\u1eddi m\u1ed9t ph\u1ea7n", "question.score feedback.score total": "{{marksString}}", "question.score feedback.score actual": "\u0110i\u1ec3m: {{scoreString}}", "question.score feedback.score total actual": "\u0110i\u1ec3m: {{score,niceNumber}}/{{marks,niceNumber}}", "variable.error in variable definition": "L\u1ed7i trong \u0111\u1ecbnh ngh\u0129a bi\u1ebfn <code>{{name}}</code>", "left brace": "{", "right brace": "}", "extension.not found": "Kh\u00f4ng th\u1ec3 t\u1ea3i ti\u1ec7n \u00edch m\u1edf r\u1ed9ng <code> {{name}} </code>.", "control.toggle navigation menu": "\u1ea8n hi\u1ec7n menu \u0111i\u1ec1u h\u01b0\u1edbng", "part.input title": "C\u00e2u tr\u1ea3 l\u1eddi cho ph\u1ea7n {{name}}", "part.correct answer title": "C\u00e2u tr\u1ea3 l\u1eddi d\u1ef1 ki\u1ebfn \u200b\u200bcho ph\u1ea7n {{name}}", "part.jme.must-match.failed": "C\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n ch\u01b0a \u0111\u00fang \u0111\u1ecbnh d\u1ea1ng.", "question.score feedback.none": "question.score feedback.none", "control.submit part.confirm remove next parts": "<p> M\u1ed9t ho\u1eb7c nhi\u1ec1u ph\u1ea7n ti\u1ebfp theo ph\u1ee5 thu\u1ed9c v\u00e0o c\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n cho ph\u1ea7n n\u00e0y. Vi\u1ec7c g\u1eedi l\u1ea1i ph\u1ea7n n\u00e0y s\u1ebd l\u00e0m m\u1ea5t hi\u1ec7u l\u1ef1c c\u1ee7a c\u00e1c ph\u1ea7n \u0111\u00f3 v\u00e0 x\u00f3a ch\u00fang kh\u1ecfi c\u00e2u h\u1ecfi. Thao t\u00e1c n\u00e0y kh\u00f4ng th\u1ec3 ho\u00e0n t\u00e1c. </p>\n<p> B\u1ea1n c\u00f3 mu\u1ed1n g\u1eedi ph\u1ea7n n\u00e0y m\u1ed9t l\u1ea7n n\u1eefa kh\u00f4ng? </p>", "control.back to menu": "Quay tr\u1edf l\u1ea1i menu", "display.error making html": "L\u1ed7i khi t\u1ea1o HTML trong {{contextDescription}}: {{-message}}", "jme.subvars.error compiling": "{{-message}} trong <code>{{expression}}</code>", "jme.variables.empty name": "A question variable has not been given a name.", "jme.calculus.unknown derivative": "Kh\u00f4ng bi\u1ebft l\u00e0m th\u1ebf n\u00e0o \u0111\u1ec3 ph\u00e2n bi\u1ec7t <code> {{tree}} </code>", "math.order complex numbers": "Kh\u00f4ng th\u1ec3 s\u1eafp x\u1ebfp th\u1ee9 t\u1ef1 s\u1ed1 ph\u1ee9c", "menu.choose a question": "Ch\u1ecdn m\u1ed9t c\u00e2u h\u1ecfi.", "part.choose next part.answered": "B\u1ea1n mu\u1ed1n l\u00e0m g\u00ec ti\u1ebfp theo?", "part.choose next part.unanswered": "Ho\u1eb7c, b\u1ea1n c\u00f3 th\u1ec3:", "part.choose next part.will be locked": "(Ph\u1ea7n n\u00e0y s\u1ebd b\u1ecb kh\u00f3a)", "part.reached dead end": "Kh\u00f4ng c\u00f2n g\u00ec \u0111\u1ec3 l\u00e0m n\u1eefa.", "part.next part.penalty amount": "(thua cu\u1ed9c {{count}} $t(mark))", "part.marking.counts towards objective": "This part counts towards the objective <strong>\u201c{{objective}}\u201d</strong>.", "part.numberentry.answer not integer or decimal or fraction": "C\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n kh\u00f4ng h\u1ee3p l\u1ec7. B\u1ea1n ph\u1ea3i nh\u1eadp m\u1ed9t s\u1ed1 nguy\u00ean, m\u1ed9t s\u1ed1 th\u1eadp ph\u00e2n ho\u1eb7c m\u1ed9t ph\u00e2n s\u1ed1.", "question": "C\u00e2u h\u1ecfi", "question.progress": "Ti\u1ebfn \u0111\u1ed9 c\u00e2u h\u1ecfi:", "question.score feedback.unattempted": "Ch\u01b0a n\u1ed7 l\u1ef1c", "question.score feedback.attempted": "C\u00f3 n\u1ed7 l\u1ef1c", "question.score feedback.score actual.plain": "{{scoreString}}", "question.score feedback.score total actual.plain": "{{score,niceNumber}}/{{marks,niceNumber}}", "question.objectives": "Objectives", "question.penalties": "Penalties", "question.back to previous part": "Quay l\u1ea1i ph\u1ea7n tr\u01b0\u1edbc", "end.print": "In b\u1ea3ng \u0111i\u1ec3m b\u00e0i ki\u1ec3m tra c\u1ee7a b\u1ea1n", "math.shuffle_together.lists not all the same length": "Not all lists are the same length.", "jme.parse signature.invalid signature string": "Ch\u1eef k\u00fd h\u00e0m kh\u00f4ng h\u1ee3p l\u1ec7: {{str}}", "part.custom.expected answer has wrong type": "C\u00e2u tr\u1ea3 l\u1eddi mong \u0111\u1ee3i c\u1ee7a ph\u1ea7n n\u00e0y b\u1ecb sai th\u1ec3 lo\u1ea1i. N\u00f3 n\u00ean l\u00e0 <code>{{shouldbe}}</code>.", "part.custom.input option has wrong type": "C\u00e0i \u0111\u1eb7t ph\u1ea7n nh\u1eadp c\u1ee7a c\u00e2u tr\u1ea3 l\u1eddi <code> {{option}} </code> kh\u00f4ng \u0111\u00fang lo\u1ea1i. N\u00f3 ph\u1ea3i l\u00e0 <code> {{shouldbe}}</code>.", "matrix input.size control legend": "K\u00edch th\u01b0\u1edbc", "matrix input.rows": "Rows", "matrix input.columns": "Columns", "part.jme.error checking numerically": "There was an error numerically checking your answer: {{-message}}", "part.gapfill.cyclic adaptive marking": "There is a cycle in the adaptive marking for this part: <strong>{{name1}}</strong> relies on <strong>{{name2}}</strong>, which eventually relies on <strong>{{name1}}</strong>.", "modal.style.background colour": "M\u00e0u n\u1ec1n", "modal.style.text colour": "M\u00e0u v\u0103n b\u1ea3n", "modal.style.text size": "C\u1ee1 v\u0103n b\u1ea3n", "modal.style.explanation": "S\u1eed d\u1ee5ng c\u00e1c c\u1ea5u h\u00ecnh n\u00e0y \u0111\u1ec3 thay \u0111\u1ed5i giao di\u1ec7n c\u1ee7a b\u00e0i ki\u1ec3m tra.", "modal.style.reset to defaults": "\u0110\u1eb7t tr\u1edf v\u1ec1 m\u1eb7c \u0111\u1ecbnh", "modal.style.text size preview": "H\u1ea7u h\u1ebft v\u0103n b\u1ea3n s\u1ebd l\u1edbn nh\u01b0 th\u1ebf n\u00e0y.", "control.style options": "T\u00f9y ch\u1ecdn hi\u1ec3n th\u1ecb", "part.marking.partially correct": "C\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n \u0111\u00fang m\u1ed9t ph\u1ea7n.", "part.marking.error in adaptive marking": "There was an error in the adaptive marking for this part. Please report this. {{-message}}", "page.skip to content": "Chuy\u1ec3n \u0111\u1ebfn ph\u1ea7n n\u1ed9i dung", "result.learning objective": "M\u1ee5c ti\u00eau h\u1ecdc t\u1eadp", "jme.interpreted as": "interpreted as", "jme.script.note.compilation error": "Error compiling note <code>{{name}}</code>: {{-message}}", "jme.script.note.error evaluating note": "Error evaluating note <code>{{name}}</code> - {{-message}}", "jme.script.note.invalid definition": "Invalid note definition: <code>{{source}}</code>. {{-hint}}", "jme.script.note.invalid definition.missing colon": "B\u1ea1n c\u00f3 th\u1ec3 thi\u1ebfu d\u1ea5u hai ch\u1ea5m sau t\u00ean v\u00e0 m\u00f4 t\u1ea3", "jme.script.note.invalid definition.description missing closing bracket": "B\u1ea1n c\u00f3 th\u1ec3 thi\u1ebfu d\u1ea5u ngo\u1eb7c \u0111\u00f3ng", "jme.script.note.empty expression": "Ghi ch\u00fa <code> {{name}} </code> tr\u1ed1ng.", "jme.script.error parsing notes": "Error parsing marking script: {{- message}}", "matrix input.cell label": "H\u00e0ng {{row}}, c\u1ed9t {{column}}", "control.move to next question": "Chuy\u1ec3n sang c\u00e2u h\u1ecfi ti\u1ebfp theo", "diagnostic.use retry": "S\u1eed d\u1ee5ng m\u1ed9t l\u1ea7n th\u1eed l\u1ea1i v\u00e0 th\u1eed l\u1ea1i ch\u1ee7 \u0111\u1ec1 n\u00e0y.", "diagnostic.move to next topic": "Chuy\u1ec3n sang ch\u1ee7 \u0111\u1ec1 ti\u1ebfp theo.", "diagnostic.next step question": "B\u1ea1n mu\u1ed1n l\u00e0m g\u00ec ti\u1ebfp theo?", "diagnostic.now assessing topic": "Hi\u1ec7n \u0111ang \u0111\u00e1nh gi\u00e1 {{current_topic}}", "diagnostic.one retry left": "B\u1ea1n c\u00f2n 1 l\u1ea7n th\u1eed l\u1ea1i", "diagnostic.retries left": "B\u1ea1n c\u00f2n {{retries}} l\u1ea7n th\u1eed l\u1ea1i.", "diagnostic.percentage completed": "B\u1ea1n \u0111\u00e3 ho\u00e0n th\u00e0nh <strong>{{percentage}}%</strong> b\u00e0i ki\u1ec3m tra.", "diagnostic.test is over": "B\u00e0i ki\u1ec3m tra k\u1ebft th\u00fac.", "diagnostic.passed all lo": "B\u1ea1n \u0111\u00e3 ho\u00e0n th\u00e0nh t\u1ea5t c\u1ea3 c\u00e1c m\u1ee5c ti\u00eau h\u1ecdc t\u1eadp.", "diagnostic.more work on lo": "B\u1ea1n c\u1ea7n n\u1ed7 l\u1ef1c h\u01a1n \u0111\u1ed1i v\u1edbi c\u00e1c m\u1ee5c ti\u00eau h\u1ecdc t\u1eadp sau: {{los}}.", "diagnostic.move to next question in topic": "Chuy\u1ec3n sang c\u00e2u h\u1ecfi ti\u1ebfp theo trong ch\u1ee7 \u0111\u1ec1.", "diagnostic.complete": "Ho\u00e0n th\u00e0nh!", "diagnostic.studying topic": "\u0110ang nghi\u00ean c\u1ee9u {{topic}}", "display.answer widget.unknown widget type": "The answer widget type <code>{{name}}</code> is not recognised.", "jme.shunt.expected argument before comma": "Mong mu\u1ed1n c\u00f3 m\u1ed9t c\u00e1i g\u00ec \u0111\u00f3 gi\u1eefa d\u1ea5u ngo\u1eb7c m\u1edf v\u00e0 d\u1ea5u ph\u1ea9y", "part.waiting for pre submit": "C\u00e2u tr\u1ea3 l\u1eddi c\u1ee7a b\u1ea1n \u0111ang \u0111\u01b0\u1ee3c ch\u1ea5m \u0111i\u1ec3m. Vui l\u00f2ng ch\u1edd.", "diagnostic.end test": "K\u1ebft th\u00fac b\u00e0i ki\u1ec3m tra.", "page.no stylesheet loaded": "The page's stylesheet file has not loaded.", "modal.confirm": "Confirm", "modal.alert": "Alert", "suspend.resumed header": "Attempt resumed", "jme.vector.value not an array of numbers": "Tried to construct a vector using a value that is not an array of numbers.", "jme.matrix.value not the right type": "Tried to construct a vector using a value of the wrong type.", "jme.subvars.html inserted twice": "An HTML value has been embedded twice. Consider defining a function to generate a new value each time it is used.", "jme.variables.invalid function language": "The language <code>{{language}}</code> is not valid.", "jme.variables.duplicate definition": "There is more than one definition of the variable <code>{{name}}</code>.", "math.random_integer_partition.invalid k": "The size of the partition must be between 1 and {{n}}.", "part.marking.parameter already in scope": "There is a variable named <code>{{name}}</code>, which is also the name of a marking parameter. Please rename the variable.", "part.marking.adaptive variable replacement refers to self": "This part refers to itself in a variable replacement for adaptive marking.", "part.marking.adaptive variable replacement refers to nothing": "This part contains an invalid variable replacement for adaptive marking.", "part.numberentry.display answer wrong type": "The display answer for this part is a value of type <code>{{got_type}}</code>, but should be a <code>{{want_type}}</code>.", "part.matrix.invalid type in prefilled": "There is an invalid value of type <code>{{n}}</code> in the array of pre-filled cells.", "diagnostic.make a choice": "Make a choice", "matrixmath.not square": "This operation only works on a square matrix.", "matrixmath.not invertible": "This operation only works on an invertible matrix.", "matrixmath.gauss-jordan elimination.not enough columns": "There must be at least as many columns as rows.", "question.required extension not available": "This question requires the extension <code>{{-extension}}</code> but it is not available.", "util.formatNumberNotation.unrecognised syntax": "The number formatting syntax <code>{{syntax}}</code> is not recognised.", "worksheet.number of exams to generate": "Number of sheets to generate", "worksheet.starting with id": "Starting with ID", "worksheet.show exam id": "Show sheet ID?", "worksheet.page break between questions": "Page breaks between questions?", "worksheet.page margins": "Page margins (mm)", "worksheet.text size": "Text size (pt)", "worksheet.generate": "Generate", "worksheet.generating exams": "Generating sheets", "worksheet.sheet id": "Sheet ID:", "worksheet.print single": "Print this sheet", "worksheet.print several": "Print these sheets", "worksheet.answer sheets": "Answer sheets", "worksheet.question sheets": "Question sheets", "worksheet.reconfigure": "Generate different sheets", "worksheet.show sheet": "Preview the sheet with ID:", "accessibility statement": "Accessibility statement and guide to adapting Numbas to your needs.", "exam.enter your name": "Your name:", "exam.attempt download security warning": "This exam is configured to allow you to download your data, but it is not running in a secure browser context. You will not be able to download your exam data. Contact your lecturer or teacher for help.", "result.download exam object": "Download your exam data", "control.return to question": "Return to the question", "control.show introduction": "Introduction", "analysis.header": "Analyse attempt data", "analysis.help.upload files": "Upload attempt data files that your students have given you.", "analysis.help.file input label": "Choose student attempt data files, or drag files onto this window.", "analysis.table.total": "Exam totals", "analysis.table.question": "Exam and question totals", "analysis.table.all": "All details", "analysis.student name.anonymous": "No name given", "analysis.expected": "Expected results", "analysis.start time": "Start time", "analysis.maximum": "Maximum Marks", "analysis.file": "File", "analysis.download": "Download", "analysis.delete": "Delete", "analysis.view results": "View results", "analysis.upload files": "Upload files", "analysis.upload more": "Upload more files", "analysis.attempt data": "Attempt data", "analysis.select format": "Select data to show", "analysis.download this table": "Download this table", "analysis.student": "Student Results", "analysis.question key": "Question Key", "analysis.question name": "Question Name", "analysis.group": "Group", "analysis.question": "Question", "analysis.part": "Part", "analysis.gap": "Gap", "analysis.record type": "Record Type", "analysis.score": "Score", "analysis.marks available": "Marks Available", "analysis.percentage": "Percentage", "analysis.answer": "Answer", "analysis.student name": "Student Name", "analysis.summary.no files": "You have not uploaded any files yet.", "analysis.summary.no decrypted files": "You have not uploaded any valid files yet.", "analysis.summary.one file": "One attempt.", "analysis.summary.several files": "{{num_files,niceNumber}} attempts.", "analysis.not secure context": "This page must be opened in a secure browser context. A secure context is either a page served over HTTPS, or a file loaded from your device.", "jme.shunt.pipe right hand takes no arguments": "The expression on the right-hand side of the pipe operator must be a function application.", "question.explore.no parts defined": "There are no parts defined in this question.", "answer": "answer", "worksheet.answersheet show question content": "Show question content in answer sheets?", "modal.confirm end exam": "Write <code>{{endConfirmation}}</code> in the box to confirm:", "modal.end exam button": "End exam", "lightbox.zoom in on image": "Zoom in on this image", "exam.progress": "Progress", "exam.questions answered": "{{numAnsweredQuestions}} of {{numQuestions}} questions answered.", "result.question marks available": "Marks Available", "result.question answered": "Answered?", "control.confirm end.correct": "You may now end the exam.", "control.confirm end.incorrect": "This is not the expected text.", "control.confirm end.password": "end", "jme.typecheck.for in name wrong type": "The name in a <code>for</code> statement must be a name or list of names, not {{type}}.", "jme.makeFast.no fast definition of function": "The function <code>{{name}}</code> here isn't defined in a way that can be made fast.", "part.show feedback": "Show feedback", "part.hide feedback": "Hide feedback", "part.feedback title": "Feedback for {{name}}.", "part.jme.must-match.warning": "Your answer is not in the expected form: {{-message}}", "part.numberentry.write your answer as a fraction": "Write your answer as a fraction.", "question.nav.label": "Question controls", "question.answer saved": "Answer saved", "question.all answers saved": "All answers saved", "analysis.back to results": "Back to results", "analysis.review": "Review", "analysis.review this": "Review this attempt", "analysis.reviewing attempt": "Reviewing attempt by {{student_name}}.", "part.there is new feedback": "The feedback has changed.", "modal.style.colour scheme": "Colour scheme", "modal.style.automatic colour scheme": "Automatic", "modal.style.light colour scheme": "Light", "modal.style.dark colour scheme": "Dark", "modal.style.custom colour scheme": "Custom", "modal.style.main font": "Main font", "modal.style.font.sans serif": "Sans serif", "modal.style.font.serif": "Serif", "modal.style.font.monospace": "Monospace", "modal.style.spacing": "Spacing", "modal.style.font weight": "Font weight", "modal.style.forced colours warning": "Your browser has overridden the colours used in this page because of a setting such as high contrast mode. Changes to the colour scheme settings here might not have any effect.", "modal.style.text preview": "Most text will look like this.", "modal.style.more options": "More options", "modal.style.main colour": "Main (brand) colour", "modal.style.primary colour": "Primary button colour", "modal.style.link colour": "Link colour", "modal.style.success colour": "Success/correct colour", "modal.style.info colour": "Info colour", "modal.style.warning colour": "Warning colour", "modal.style.danger colour": "Danger/incorrect colour", "modal.style.muted colour": "Muted text colour", "modal.style.highlight colour": "Highlight colour", "exam.error loading exam definition": "There was an error while loading the exam definition: {{text}}", "exam.no exam definition": "No exam definition was given.", "jme.typecheck.wrong arguments for anonymous function": "Wrong number of arguments for this anonymous function.", "worksheet.top": "Top", "worksheet.left": "Left", "worksheet.bottom": "Bottom", "worksheet.right": "Right"}}, "pt-br": {"translation": {"page.loading": "Carregando...", "page.saving": "<p>Salvando.</p>\n<p>Isso pode demorar alguns segundos.</p>", "mathjax.math processing error": "\"{{-message}}\" quando \"texificando\" <code>{{expression}}</ code>", "die.numbas failed": "Numbas falhou", "die.sorry": "Desculpe, Numbas encontrou um erro; portanto, n\u00e3o pode continuar. Abaixo est\u00e1 uma descri\u00e7\u00e3o do erro.", "die.error": "Erro", "modal.ok": "OK", "modal.cancel": "Cancelar", "exam.exam name": "Nome do Exame:", "exam.random seed": "ID de sess\u00e3o:", "exam.student name": "Nome do Estudante:", "exam.number of questions": "N\u00famero de perguntas:", "exam.marks available": "Notas dispon\u00edveis:", "exam.pass percentage": "Percentual de aprova\u00e7\u00e3o:", "exam.time allowed": "Tempo Permitido:", "exam.passed": "Aprovado", "exam.failed": "Reprovado", "exam.review header": "Avalia\u00e7\u00e3o:", "frontpage.start": "Iniciar", "suspend.paused header": "Pausado", "suspend.exam suspended": "O exame foi interrompido. Pressione <em>Continuar<em> para continuar.", "suspend.you can resume": "Voc\u00ea poder\u00e1 continuar essa sess\u00e3o na pr\u00f3xima vez que iniciar esta atividade.", "suspend.resume": "Continuar", "result.exit": "Sair", "result.print": "Imprimir relat\u00f3rio de resultados", "result.exam summary": "Relat\u00f3rio do exame", "result.performance summary": "Relat\u00f3rio de performance", "result.exam start": "Iniciar exame", "result.exam stop": "Parar exame", "result.time spent": "Tempo gasto", "result.questions attempted": "Quest\u00f5es tentadas:", "result.score": "Pontua\u00e7\u00e3o", "result.result": "Resultado", "result.question number": "N\u00famero da quest\u00e3o", "result.question score": "Pontua\u00e7\u00e3o", "result.question review title": "Revisar esta quest\u00e3o", "result.click a question to review": "Clique no n\u00famero de uma quest\u00e3o para ver como as suas respostas foram avaliadas e, quando dispon\u00edvel, as solu\u00e7\u00f5es completas.", "end.exam has finished": "O exame terminou. Voc\u00ea pode fechar esta janela.", "control.confirm leave": "Voc\u00ea n\u00e3o concluiu o exame.", "control.not all questions answered": "Voc\u00ea n\u00e3o completou todas as quest\u00f5es neste exame.", "control.not all questions submitted": "Voc\u00ea modificou uma ou mais respostas mas n\u00e3o as enviou. Por favor, verifique se cada uma das quest\u00f5es foi enviada.", "control.confirm end": "Tem certeza que quer finalizar o exame? Ap\u00f3s finalizar o exame n\u00e3o ser\u00e1 poss\u00edvel modificar as suas respostas.", "control.confirm regen": "Gostaria de re-aleatorizar esta quest\u00e3o? Se clicar OK, todas as suas respostas e pontua\u00e7\u00f5es para a quest\u00e3o atual ser\u00e3o perdidos.", "control.confirm reveal": "Gostaria de revelar a resposta para esta quest\u00e3o? Pontos recebidos at\u00e9 o momento ser\u00e3o bloqueados e voc\u00ea n\u00e3o poder\u00e1 responder essa quest\u00e3o mais tarde.", "control.proceed anyway": "Prosseguir mesmo assim?", "control.regen": "Tentar outra quest\u00e3o similar a esta", "control.submit answer": "Enviar resposta", "control.submit all parts": "Enviar todas as partes", "control.submit again": "Enviar novamente", "control.submit": "Enviar", "control.previous": "Anterior", "control.next": "Pr\u00f3ximo", "control.advice": "Dica", "control.reveal": "Revelar respostas", "control.total": "Total", "control.pause": "Pausar", "control.end exam": "Finalizar exame", "control.back to results": "Retornar aos resultados", "display.part.jme.error making maths": "Erro ao criar o display matem\u00e1tico", "exam.xml.bad root": "Elemento raiz do XML do exame deve ser 'exam'", "exam.changeQuestion.no questions": "Este exame n\u00e3o cont\u00e9m quest\u00f5es! Verifique se h\u00e1 erros no arquivo .exam.", "feedback.you were awarded": "Voc\u00ea recebeu <strong>{{count,niceNumber}}</strong> $t(mark).", "feedback.taken away": "<strong>{{count,niceNumber}}</strong> $t(mark) $t(was) tirados.", "jme.tokenise.invalid": "Express\u00e3o inv\u00e1lida: <code>{{expression}}</code>", "jme.shunt.not enough arguments": "Poucos argumentos para a opera\u00e7\u00e3o {{op}}", "jme.shunt.no left bracket in function": "Sem abertura de colchete correspondete na aplica\u00e7\u00e3o da fun\u00e7\u00e3o ou tupla", "jme.shunt.no left square bracket": "Sem abertura de colchete correspondente", "jme.shunt.no left bracket": "Sem abertura de colchete correspondente", "jme.shunt.no right bracket": "Sem fechamento de colchete correspondente", "jme.shunt.no right square bracket": "Sem fechamento de colchete para finalizar a lista", "jme.shunt.missing operator": "Express\u00e3o n\u00e3o pode ser avaliada -- fata um operador.", "jme.typecheck.function maybe implicit multiplication": "Opera\u00e7\u00e3o {{name}} n\u00e3o definida. Voc\u00ea quis dizer <code>{{first}}*{{possibleOp}}(...)</code>?", "jme.typecheck.function not defined": "Opera\u00e7\u00e3o <code>{{op}}</code> n\u00e3o definida. <code>{{op}}</code> \u00e9 uma vari\u00e1vel e voc\u00ea quis dizer <code>{{sugestion}}*(...)</code>?", "jme.typecheck.op not defined": "Opera\u00e7\u00e3o '{{op}}' n\u00e3o definida.", "jme.typecheck.no right type definition": "Nenhuma defini\u00e7\u00e3o de '{{op}}' do tipo correto foi encontrada.", "jme.typecheck.no right type unbound name": "Vari\u00e1vel <code>{{name}}</code> n\u00e3o definida.", "jme.typecheck.map not on enumerable": "Opera\u00e7\u00e3o <code>map</code> funciona em uma listas ou intervalo, n\u00e3o {{type}}", "jme.evaluate.undefined variable": "Vari\u00e1vel {{name}} n\u00e3o definida.", "jme.thtml.not html": "Passou um valor n\u00e3o-HTML para o contrutor THTML.", "jme.func.switch.no default case": "Sem caso default no Switch", "jme.func.listval.invalid index": "\u00cdndice {{index}} inv\u00e1lido em lista de tamanho {{size}}", "jme.func.listval.not a list": "Objeto n\u00e3o \u00e9 subscriptable", "jme.func.matrix.invalid row type": "N\u00e3o \u00e9 poss\u00edvel construir uma matriz a partir de linhas do tipo {{type}}", "jme.func.except.continuous range": "N\u00e3o \u00e9 poss\u00edvel usar o operador 'except' em intervalos cont\u00ednuos.", "jme.matrix.reports bad size": "Matriz informa tamanho incorretamente - deve ser um erro na fun\u00e7\u00e3o construtor", "jme.texsubvars.no right bracket": "Sem <code>]</code> correspondente nos argumentos de {{op}}.", "jme.texsubvars.missing parameter": "Faltando par\u00e2metro em {{op}}: {{parameter}}", "jme.texsubvars.no right brace": "Sem <code>}</code> correspondente em {{op}}", "jme.user javascript.error": "Erro na fun\u00e7\u00e3o definida pelo usu\u00e1rio <code>{{name}}</code>: {{-message}}", "jme.variables.error making function": "Erro ao criar fun\u00e7\u00e3o <code>{{name}}</code>: {{-message}}", "jme.variables.syntax error in function definition": "Erro de sintaxe ao definir fun\u00e7\u00e3o", "jme.variables.variable not defined": "Vari\u00e1vel <code>{{name}}</code> n\u00e3o definida.", "jme.variables.empty definition": "Defini\u00e7\u00e3o da vari\u00e1vel <code>{{name}}</code> est\u00e1 vazia.", "jme.variables.circular reference": "Refer\u00eancia circular na defini\u00e7\u00e3o da vari\u00e1vel <code>{{name}}</code>", "jme.variables.error computing dependency": "Erro ao computar vari\u00e1vel referenciada <code>{{name}}</code>", "jme.variables.error evaluating variable": "Erro ao avaliar a vari\u00e1vel {{name}}: {{-message}}", "jme.variables.question took too many runs to generate variables": "Um conjunto v\u00e1lido de vari\u00e1veis de quest\u00e3o n\u00e3o foi gerado a tempo.", "jme.display.unknown token type": "N\u00e3o \u00e9 poss\u00edvel \"texificar\" tipo {{type}}", "jme.display.collectRuleset.no sets": "Nenhum conjunto passado para collectRuleset!", "jme.display.collectRuleset.set not defined": "Conjunto de regras {{name}} n\u00e3o foi definido", "jme.display.simplifyTree.no scope given": "Numbas.jme.display.simplifyTree deve receber um Scope", "math.precround.complex": "N\u00e3o \u00e9 poss\u00edvel arredondar para um n\u00famero complexo de casas decimais", "math.siground.complex": "N\u00e3o \u00e9 poss\u00edvel arredondar para um n\u00famero complexo de algarismos significantes.", "math.combinations.complex": "N\u00e3o \u00e9 poss\u00edvel calcular combina\u00e7\u00f5es de n\u00fameros complexos", "math.permutations.complex": "N\u00e3o \u00e9 poss\u00edvel calcular permuta\u00e7\u00f5es de n\u00fameros complexos", "math.gcf.complex": "N\u00e3o \u00e9 poss\u00edvel calcular o MDC de n\u00fameros complexos", "math.lcm.complex": "N\u00e3o \u00e9 poss\u00edvel calcular o MMC de n\u00fameros complexos", "math.lt.order complex numbers": "N\u00e3o \u00e9 poss\u00edvel ordenar n\u00fameros complexos", "math.choose.empty selection": "Sele\u00e7\u00e3o vazia passada a um fun\u00e7\u00e3o rand\u00f4mica", "matrixmath.abs.non-square": "N\u00e3o \u00e9 poss\u00edvel calcular a determinante de uma matriz n\u00e3o quadrada", "matrixmath.abs.too big": "Desculpe, ainda n\u00e3o \u00e9 poss\u00edvel calcular a determinante de matrizes maiores que 3x3", "matrixmath.mul.different sizes": "N\u00e3o \u00e9 poss\u00edvel multiplicar matrizes de diferentes tamanhos", "vectormath.cross.not 3d": "Poss\u00edvel calcular produto vetorial apenas para vetores tridimensionais.", "vectormath.dot.matrix too big": "N\u00e3o \u00e9 poss\u00edvel calcular o produto escalar de uma matriz que n\u00e3o seja $1 \\times N$ ou $N \\times 1$.", "vectormath.cross.matrix too big": "N\u00e3o \u00e9 poss\u00edvel calcular o produto vetorial de uma matriz que n\u00e3o seja  $1 \\times N$ ou $N \\times 1$.", "part.with steps answer prompt": "Resposta:", "part.script.error": "Erro na part {{path}}, no script {{script}}: {{-message}}", "part.marking.steps no matter": "Como voc\u00ea recebeu nota m\u00e1xima para esta parte, suas respostas para os passos n\u00e3o foram contabilizadas.", "part.marking.revealed steps no penalty": "Voc\u00ea revelou os passos.", "part.marking.used variable replacements": "Esta parte foi pontuada usando suas respostas para as partes anteriores.", "part.marking.variable replacement part not answered": "Voc\u00ea deve responder {{part}} primeiro", "part.marking.resubmit because of variable replacement": "A avalia\u00e7\u00e3o desta parte depende das duas respostas em outras partes, as quais voc\u00ea modificou. Envie esta parte novamente para atualizar sua pontua\u00e7\u00e3o.", "part.marking.not submitted": "Sem resposta enviada", "part.marking.did not answer": "Voc\u00ea n\u00e3o respondeu esta quest\u00e3o", "part.marking.nothing entered": "Voc\u00ea n\u00e3o inseriu uma resposta.", "part.marking.incorrect": "Sua resposta est\u00e1 incorreta.", "part.marking.correct": "Sua resposta est\u00e1 correta.", "part.marking.uncaught error": "Erro ao avaliar: {{-message}}", "part.marking.no result": "Esta part n\u00e3o p\u00f4de ser avaliada.", "part.correct answer": "Resposta esperada:", "part.missing type attribute": "{{part}}: Faltando atributo de tipo de parte", "part.unknown type": "{{part}}: Tipo de parte n\u00e3o reconhecido {{type}}", "part.setting not present": "Propriedade '{{property}}' n\u00e3o atribuida", "part.jme.answer missing": "Resposta correta n\u00e3o encontrada", "part.jme.answer too long": "Sua resposta \u00e9 muito longa.", "part.jme.answer too short": "Sua resposta \u00e9 muito curta.", "part.jme.answer invalid": "Sua resposta n\u00e3o \u00e9 uma express\u00e3o matem\u00e1tica v\u00e1lida.\n<br/>{{-message}}", "part.jme.marking.correct": "Sua resposta est\u00e1 numericamente correta.", "part.jme.must-have bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.must-have one": "Sua resposta deve conter: {{strings}}", "part.jme.must-have several": "Sua resposta deve conter todos de: {{strings}}", "part.jme.not-allowed bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.not-allowed one": "Sua resposta n\u00e3o deve conter: {{strings}}", "part.jme.not-allowed several": "Sua resposta n\u00e3o deve conter nenhum de: {{strings}}", "part.jme.unexpected variable name": "Sua resposta usa uma vari\u00e1vel de nome inesperado <code>{{name}}</code>.", "part.jme.unexpected variable name suggestion": "Sua resposta usa uma vari\u00e1vel de nome inesperado <code>{{name}}</code>. Voc\u00ea quis dizer <code>{{suggestion}}</code>?", "part.patternmatch.display answer missing": "Resposta para exibi\u00e7\u00e3o n\u00e3o encontrada", "part.patternmatch.correct except case": "Sua resposta est\u00e1 correta, exceto a formata\u00e7\u00e3o.", "part.numberentry.correct except decimal": "Sua resposta est\u00e1 na faixa permitida, por\u00e9m n\u00fameros decimais n\u00e3o s\u00e3o permitidos.", "part.numberentry.correct except fraction": "Sua resposta est\u00e1 na faixa permitida, mas fra\u00e7\u00f5es n\u00e3o s\u00e3o permitidas.", "part.numberentry.answer invalid": "Voc\u00ea n\u00e3o inseriu um n\u00famero valido.", "part.numberentry.answer not integer": "Sua resposta \u00e9 inv\u00e1lida. Voc\u00ea deve inserir um n\u00famero inteiro, n\u00e3o um decimal.", "part.numberentry.answer not integer or decimal": "Sua resposta \u00e9 inv\u00e1lida. Voc\u00ea deve inserir um inteiro ou um decimal.", "part.numberentry.zero sig fig": "Esta parte est\u00e1 configurada para arredondar a resposta do estudante para zero algarismos significantes, o que n\u00e3o faz sentido.", "part.mcq.options def not a list": "A express\u00e3o que define {{properties}} n\u00e3o \u00e9 uma lista.", "part.mcq.marking matrix string empty": "A express\u00e3o customizada da matriz de pontua\u00e7\u00e3o est\u00e1 vazia.", "part.mcq.choices missing": "Defini\u00e7\u00e3o das alternativas n\u00e3o encontrada", "part.mcq.matrix not a number": "A c\u00e9lula da matriz de pontua\u00e7\u00e3o ({{row}},{{column}}) da parte {{part}} n\u00e3o cont\u00e9m um n\u00famero.", "part.mcq.wrong number of choices": "Voc\u00ea selecionou o n\u00famero errado de op\u00e7\u00f5es.", "part.mcq.no choices selected": "Nenhuma op\u00e7\u00e3o selecionada.", "part.mcq.matrix not a list": "Matriz de pontua\u00e7\u00e3o, definida por uma express\u00e3o JME, n\u00e3o \u00e9 uma lista mas deveria ser.", "part.mcq.matrix wrong type": "Elemento de tipo inv\u00e1lido '{{type}}' usado na matriz de pontua\u00e7\u00e3o.", "part.mcq.matrix mix of numbers and lists": "Mix de n\u00fameros e listas usados na matriz de pontua\u00e7\u00e3o.", "part.mcq.matrix wrong size": "Matriz de pontua\u00e7\u00e3o tem ", "part.mcq.correct choice": "Voc\u00ea escolheu a resposta correta.", "part.matrix.answer invalid": "Sua resposta n\u00e3o \u00e9 v\u00e1lida.", "part.matrix.invalid cell": "Uma ou mais c\u00e9lulas da sua resposta est\u00e3o vazias ou s\u00e3o inv\u00e1lidas.", "part.matrix.some incorrect": "Uma ou mais c\u00e9lulas da sua resposta est\u00e3o incorretas, mas voc\u00ea recebeu pontua\u00e7\u00e3o pelo resto.", "part.matrix.empty": "Voc\u00ea n\u00e3o inseriu uma resposta.", "part.matrix.empty cell": "Uma ou mais c\u00e9lulas da sua resposta est\u00e3o vazias.", "part.matrix.size mismatch": "O autor da quest\u00e3o n\u00e3o permitiu que o estudante decida as dimens\u00f5es da resposta, mas a resposta correta \u00e9 {{correct_dimension}} enquanto a resposta inserida \u00e9 {{input_dimensions}}", "part.gapfill.feedback header": "<strong> Lacuna {{index}}</strong>", "part.extension.not implemented": "Parte n\u00e3o implementou o m\u00e9todo <code>{{name}}</code>.", "question.loaded name mismatch": "N\u00e3o \u00e9 poss\u00edvel retomar esta tentativa - o pacote modificou desde a \u00faltima sess\u00e3o.", "question.error": "Quest\u00e3o {{number}}: {{-message}}", "question.preamble.error": "Erro no pre\u00e2mbulo: {{-message}}", "question.preamble.syntax error": "Erro de sintaxe no pre\u00e2mbulo", "question.unsupported part type": "Tipo de parte n\u00e3o suportado.", "question.header": "Quest\u00e3o {{number}}", "question.submit part": "Enviar parte", "question.show steps": "Mostrar passos", "question.show steps penalty": "Voc\u00ea perder\u00e1 <strong>{{count,niceNumber}}</strong> $t(mark).", "question.show steps no penalty": "Sua pontua\u00e7\u00e3o n\u00e3o ser\u00e1 afetada.", "question.show steps already penalised": "Voc\u00ea j\u00e1 mostrou passos. Voc\u00ea pode mostr\u00e1-los novamente sem mais penalidades.", "question.hide steps": "Ocultar passos", "question.hide steps no penalty": "Sua pontua\u00e7\u00e3o n\u00e3o ser\u00e1 afetada.", "question.advice": "Dica", "question.no such part": "N\u00e3o foi poss\u00edvel encontrar parte {{path}}", "question.can not submit": "N\u00e3o foi poss\u00edvel enviar a resposta - verifique se h\u00e1 erros", "question.answer submitted": "Resposta enviada", "question.score feedback.show": "Mostrar feedback", "question.score feedback.hide": "Ocultar feedback", "question.score feedback.answered total actual": "Pontua\u00e7\u00e3o: {{score,niceNumber}}/{{marks,niceNumber}}", "question.score feedback.answered total": "{{marksString}}. Respondida.", "question.score feedback.answered actual": "Pontua\u00e7\u00e3o: {{scoreString}}", "question.score feedback.answered": "Respondida.", "question.score feedback.unanswered": "N\u00e3o respondida.", "question.score feedback.unanswered total": "{{marksString}}.", "question.score feedback.correct": "Sua resposta est\u00e1 correta.", "question.score feedback.partial": "Sua resposta est\u00e1 parcialmente correta.", "question.score feedback.wrong": "Sua resposta est\u00e1 incorreta", "question.selector.unsubmitted changes": "Modifica\u00e7\u00f5es n\u00e3o enviadas", "timing.no accumulator": "sem acumulador de tempo {{name}}", "timing.time remaining": "Tempo restante:", "xml.could not load": "N\u00e3o foi poss\u00edvel carregar um documento XML: {{-message}}", "xml.property not number": "Propriedade {{name}} deve ser um n\u00famero, mas n\u00e3o \u00e9 ({{value}}), no n\u00f3 {{element}}", "xml.property not boolean": "Propriedade {{name}} deve ser um booleano, mas n\u00e3o \u00e9 ({{value}}), no n\u00f3 {{element}}", "xml.error in variable definition": "Erro na defini\u00e7\u00e3o da vari\u00e1vel <code>{{name}}</code>", "scorm.error initialising": "Erro ao inicializar protocolo SCORM: {{-message}}", "scorm.failed save": "<p> A requisi\u00e7\u00e3o para salvar dados falhou. Pressione <b>OK</b> para tentar novamente.</p>\n<p>Se receber essa mensagem repetidas vezes, verifique sua conex\u00e3o a internet ou use outro computador. Suas respostas previamente enviadas foram salvas com sucesso e ser\u00e3o restauradas caso continue esta sess\u00e3o em outro computador.</p>\n<p>Se esta mensagem aparecer persistentemente e voc\u00ea n\u00e3o conseguir salvar <em>nenhuma</em> resposta, por favor, contate o professor.</p>", "scorm.no exam suspend data": "Falha ao retomar: sem dados suspensos do exame.", "scorm.error loading suspend data": "Erro ao carregar dados suspensos: {{-message}}", "scorm.error loading question": "Erro ao carregar quest\u00e4o {{number}}: {{-message}}", "scorm.no question suspend data": "Sem dados suspensos de quest\u00e3o", "scorm.error loading part": "Erro ao carregar parte {{part}}: {{-message}}", "scorm.no part suspend data": "Sem dados suspensos de parte", "util.product.non list": "Passou uma n\u00e3o-lista para <code>Numbas.util.product</code>", "mark": "ponto", "was": "era", "part": "parte", "gap": "lacuna", "step": "passo", "jme.substituteTree.undefined variable": "Vari\u00e1vel indifinida <code>{{name}}</code>", "jme.user javascript.returned undefined": "Fun\u00e7\u00e3o javascript definida pelo usu\u00e1rio <code>{{name}}</code> retornou <code>undefined</code>.", "part.marking.steps change": "Voc\u00ea recebeu <strong>{{count,niceNumber}}</strong> $t(mark) por suas respostas aos passos.", "part.marking.revealed steps with penalty": "Voc\u00ea revelou passos. A pontua\u00e7\u00e3o m\u00e1xima que pode obter por esta parte \u00e9 <strong>{{count,niceNumber}}</strong> $t(mark). Suas pontua\u00e7\u00f5es ser\u00e3o ajustadas de acordo.", "part.marking.total score": "Voc\u00ea pontuou <strong>{{count,niceNumber}}</strong> $t(mark) por esta parte.", "part.numberentry.precision type.dp": "casa decimal", "part.numberentry.precision type.dp_plural": "casas decimais", "part.numberentry.precision type.sigfig": "algarismo significativo", "part.numberentry.precision type.sigfig_plural": "algarismos significativos", "part.numberentry.give your answer to precision": "Arredonde sua resposta para {{count,niceNumber}} {{precisionType}}", "question.unsubmitted changes": "Voc\u00ea modificou a sua resposta mas n\u00e3o a enviou. Por favor, verifique sua resposta e pressione o bot\u00e3o <strong>Enviar resposta</stron>.", "question.unsubmitted changes_plural": "Voc\u00ea modificou suas respostas mas n\u00e3o as enviou. Por favor, verifique suas respostas e pressione o bot\u00e3o <strong>Enviar todas as partes</stron>.", "util.equality not defined for type": "Igualdade n\u00e3o definida para o tipo {{type}}", "mark_plural": "pontos", "was_plural": "eram", "die.script not loaded": "Numbas n\u00e3o iniciou pois o arquivo <code>{{file}}</code> n\u00e3o foi carregado. Verifique se est\u00e1 incluso em <code>script.js</code>.", "math.combinations.n less than zero": "Imposs\u00edvel calcular combina\u00e7\u00f5es: n \u00e9 menor que zero", "math.combinations.k less than zero": "Imposs\u00edvel calcular combina\u00e7\u00f5es: k \u00e9 menor que zero", "math.combinations.n less than k": "Imposs\u00edvel calcular combina\u00e7\u00f5es: n \u00e9 menor que k", "math.permutations.n less than zero": "Imposs\u00edvel calcular permuta\u00e7\u00f5es: n \u00e9 menor que zero", "math.permutations.k less than zero": "Imposs\u00edvel calcular permuta\u00e7\u00f5es: k \u00e9 menor que zero", "math.permutations.n less than k": "Imposs\u00edvel calcular permuta\u00e7\u00f5es: n \u00e9 menor que k", "part.numberentry.give your answer to precision_0": "Arredonde sua resposta para o inteiro mais pr\u00f3ximo.", "mathjax.error": "Erro de processamento MathJax: {{-message}}", "mathjax.error with context": "Erro de processamento MathJax em {{context}}: {{-message}}", "exam.introduction": "Introdu\u00e7\u00e3o do exame", "exam.feedback": "Mensagem de feedback do exame", "jme.tokenise.keypair key not a string": "Chave do dicion\u00e1rio deve ser uma string, n\u00e3o {{type}}.", "jme.shunt.list mixed argument types": "N\u00e3o \u00e9 poss\u00edvel analisar {{mode}}: mix de dicion\u00e1rio e lista", "jme.func.listval.key not in dict": "Dicion\u00e1rio n\u00e3o cont\u00e9m a chave <code>{{key}}</code>", "part.prompt": "aviso", "part.feedback": "feedback", "part.numberentry.answer not reduced": "Sua respostar n\u00e3o est\u00e1 reduzida para os menores termos.", "part.numberentry.give your answer as a reduced fraction": "Reduza a sua resposta para os menores termos.", "part.numberentry.negative decimal places": "Esta parte est\u00e1 configurada para arredondar a resposta do estudante para um n\u00famero negativo de algarismos significantes, o que n\u00e3o faz sentido.", "part.mcq.choices": "op\u00e7\u00f5es", "part.mcq.answers": "respostas", "part.mcq.matrix cell empty": "A c\u00e9lula da matriz de pontua\u00e7\u00e3o ({{row}},{{column}}) da parte {{part}} est\u00e1 vazia.", "part.mcq.matrix jme error": "A c\u00e9lula da matriz de pontua\u00e7\u00e3o ({{row}},{{column}}) da parte {{part}} possui erro JME: {{error}}.", "question.statement": "Enunciado", "ruleset.circular reference": "Refer\u00eancia circular na defini\u00e7\u00e3o do conjunto de regras <code>{{name}}</code>", "ruleset.set not defined": "Conjunto de regras {{name}} n\u00e3o foi definido", "jme.evaluate.no scope given": "Numbas.jme.evaluate deve receber um Scope", "answer.number.not a number": "A resposta  n\u00e3o \u00e9 um n\u00famero v\u00e1lido.", "answer.number.fractions not allowed": "N\u00e3o pode inserir uma  fra\u00e7\u00e3o.", "answer.jme.invalid expression": "{{-mensagem}}", "answer.matrix.fractions not allowed": "N\u00e3o pode inserir fra\u00e7\u00f5es.", "answer.matrix.some cell not a number": "Uma ou mais partes  da resposta  n\u00e3o \u00e9 um n\u00famero v\u00e1lido.", "exam.enter password": "Palavra-passe:", "exam.password.correct": "A palavra-passe est\u00e1 correcta. Pode come\u00e7ar o exame.", "exam.password.incorrect": "A palavra-passe est\u00e1 incorrecta.", "frontpage.scorm.lms not connected": "This exam is running in standalone mode. Your answers and marks will not be saved!", "result.question review": "Revisar", "control.confirm regen no marks": "Gostaria de re-aleatorizar esta quest\u00e3o?", "control.confirm reveal no marks": "Would you like to reveal the answer to this question?", "jme.tokenise.invalid near": "Invalid expression: <code>{{expression}}</code> at position {{position}} near <code>{{nearby}}</code>", "jme.tokenise.number.object not complex": "Invalid object passed into number constructor.", "jme.subvars.null substitution": "Empty variable substitution: <code>$t(left brace){{str}}$t(right brace)", "jme.type.type already registered": "The data type {{type}} has already been registered so can't be registered again.", "jme.type.no cast method": "Can't automatically convert from {{from}} to {{to}}.", "jme.display.simplifyTree.empty expression": "Expression is empty", "jme.display.simplifyTree.stuck in a loop": "Simplifier is stuck in a loop: <code>{{expr}}</code>", "math.niceNumber.undefined": "Was expecting a number, but got <code>undefined</code>", "math.rangeToList.zero step size": "Can't convert a range with step size zero to a list.", "part.error": "{{path}}: {{-message}}", "part.marking.revealed steps": "You revealed the steps.", "part.marking.maximum scaled down": "The maximum you can score for this part is <strong>{{count,niceNumber}}</strong> $t(mark). Your scores will be scaled down accordingly.", "part.marking.minimum score applied": "The minimum score for this part is <strong>{{score,niceNumber}}</strong>.", "part.marking.maximum score applied": "The maximum score for this part is <strong>{{score,niceNumber}}</strong>.", "part.marking.error in marking script": "There was an error in this part's marking algorithm. Please report this. {{-message}}", "part.marking.no result after replacement": "This part could not be marked using your answers to previous parts.", "part.marking.missing required note": "The marking algorithm does not define the note <code>{{note}}</code>", "marking.apply.not a list": "The first argument to <code>apply</code> must be a list, and isn't", "marking.apply marking script.script not found": "Marking script <code>{{name}}</code> not found", "marking.note.compilation error": "Error compiling note <code>{{name}}</code>: {{-message}}", "marking.note.error evaluating note": "Error evaluating note <code>{{name}}</code> - {{-message}}", "marking.note.invalid definition": "Invalid note definition: <code>{{source}}</code>. {{-hint}}", "marking.note.invalid definition.missing colon": "You might be missing a colon after the name and description", "marking.note.invalid definition.description missing closing bracket": "You might be missing a closing bracket", "marking.note.empty expression": "The note <code>{{name}}</code> is empty.", "marking.script.error parsing notes": "Error parsing marking script: {{- message}}", "part.feedback out of date": "This feedback is based on your last submitted answer. Save your changed answer to get updated feedback.", "part.jme.invalid value generator expression": "Invalid value generator expression for variable <code>{{name}}</code>: {{-message}}", "part.mcq.incorrect choice": "You chose an incorrect answer.", "part.matrix.not all cells same precision": "You have not given every cell in your answer to the same precision.", "part.gapfill.error marking gap": "Error marking {{name}}: {{-message}}", "part.custom.empty setting": "No value given.", "part.custom.unrecognised input type": "Unrecognised setting type <code>{{input_type}}</code>", "part.custom.error evaluating input option": "Error evaluating input option <code>{{option}}</code>: {{-error}}", "part.custom.input option missing": "Definition of input option <code>{{option}}</code> is missing.", "part.custom.error evaluating setting": "Error evaluating setting <code>{{setting}}</code>: {{-error}}", "question.error creating question": "Error while creating question {{number}}: {{-message}}", "question.score feedback.not marked": "Not marked", "question.score feedback.partially answered": "Resposta parcial", "question.score feedback.score total": "{{marksString}}", "question.score feedback.score actual": "Pontua\u00e7\u00e3o: {{scoreString}}", "question.score feedback.score total actual": "Pontua\u00e7\u00e3o: {{scoreString}}", "variable.error in variable definition": "Error in definition of variable <code>{{name}}</code>", "left brace": "{", "right brace": "}", "extension.not found": "Couldn't load the extension <code>{{name}}</code>.", "control.toggle navigation menu": "Toggle the navigation menu", "part.input title": "Answer for part {{name}}", "part.correct answer title": "Expected answer for part {{name}}", "part.jme.must-match.failed": "Your answer is not in the expected form.", "question.score feedback.none": "", "control.submit part.confirm remove next parts": "<p>One or more subsequent parts depend on your answer to this part. Submitting this part again will invalidate those parts, and remove them from the question. This cannot be undone.</p>\n<p>Would you like to submit this part again?</p>", "control.back to menu": "Go back to the menu", "display.error making html": "Error making HTML in {{contextDescription}}: {{-message}}", "jme.subvars.error compiling": "{{-message}} in <code>{{expression}}</code>", "jme.variables.empty name": "A question variable has not been given a name.", "jme.calculus.unknown derivative": "Don't know how to differentiate <code>{{tree}}</code>", "math.order complex numbers": "Can't order complex numbers", "menu.choose a question": "Choose a question.", "part.choose next part.answered": "What do you want to do next?", "part.choose next part.unanswered": "Or, you could:", "part.choose next part.will be locked": "(This part will be locked)", "part.reached dead end": "There's nothing more to do from here.", "part.next part.penalty amount": "(lose {{count}} $t(mark))", "part.marking.counts towards objective": "This part counts towards the objective <strong>\u201c{{objective}}\u201d</strong>.", "part.numberentry.answer not integer or decimal or fraction": "Your answer is invalid. You must enter an integer, a decimal or a fraction.", "question": "Pergunta", "question.progress": "Question progress", "question.score feedback.unattempted": "Not attempted", "question.score feedback.attempted": "Attempted", "question.score feedback.score actual.plain": "{{scoreString}}", "question.score feedback.score total actual.plain": "{{score,niceNumber}}/{{marks,niceNumber}}", "question.objectives": "Objectives", "question.penalties": "Penalties", "question.back to previous part": "Go back to the previous part", "end.print": "Print your exam transcript", "math.shuffle_together.lists not all the same length": "Not all lists are the same length.", "jme.parse signature.invalid signature string": "Invalid function signature string: {{str}}", "part.custom.expected answer has wrong type": "The expected answer for this part has the wrong type. It should be <code>{{shouldbe}}</code>.", "part.custom.input option has wrong type": "The answer input setting <code>{{option}}</code> has the wrong type. It should be <code>{{shouldbe}}</code>.", "matrix input.size control legend": "Tamanho", "matrix input.rows": "Linhas", "matrix input.columns": "Colunas", "part.jme.error checking numerically": "There was an error numerically checking your answer: {{-message}}", "part.gapfill.cyclic adaptive marking": "There is a cycle in the adaptive marking for this part: <strong>{{name1}}</strong> relies on <strong>{{name2}}</strong>, which eventually relies on <strong>{{name1}}</strong>.", "modal.style.background colour": "Background colour", "modal.style.text colour": "Text colour", "modal.style.text size": "Text size", "modal.style.explanation": "Use these controls to change the appearance of the exam.", "modal.style.reset to defaults": "Reset to defaults", "modal.style.text size preview": "Most text will be this big.", "control.style options": "Display options", "part.marking.partially correct": "Your answer is partially correct.", "part.marking.error in adaptive marking": "There was an error in the adaptive marking for this part. Please report this. {{-message}}", "page.skip to content": "Skip to content", "result.learning objective": "Learning objective", "jme.interpreted as": "interpreted as", "jme.script.note.compilation error": "Error compiling note <code>{{name}}</code>: {{-message}}", "jme.script.note.error evaluating note": "Error evaluating note <code>{{name}}</code> - {{-message}}", "jme.script.note.invalid definition": "Invalid note definition: <code>{{source}}</code>. {{-hint}}", "jme.script.note.invalid definition.missing colon": "You might be missing a colon after the name and description", "jme.script.note.invalid definition.description missing closing bracket": "You might be missing a closing bracket", "jme.script.note.empty expression": "The note <code>{{name}}</code> is empty.", "jme.script.error parsing notes": "Error parsing marking script: {{- message}}", "matrix input.cell label": "Row {{row}}, column {{column}}", "control.move to next question": "Move to the next question", "diagnostic.use retry": "Use one retry and try this topic again.", "diagnostic.move to next topic": "Move on to the next topic.", "diagnostic.next step question": "What would you like to do next?", "diagnostic.now assessing topic": "Now assessing {{current_topic}}", "diagnostic.one retry left": "You have 1 retry left", "diagnostic.retries left": "You have {{retries}} retries left.", "diagnostic.percentage completed": "You've completed <strong>{{percentage}}%</strong> of the test.", "diagnostic.test is over": "The test is over.", "diagnostic.passed all lo": "You have passed all learning objectives.", "diagnostic.more work on lo": "You need to do some more work on the following learning objectives: {{los}}.", "diagnostic.move to next question in topic": "Move on to the next question in topic.", "diagnostic.complete": "Complete!", "diagnostic.studying topic": "Studying {{topic}}", "display.answer widget.unknown widget type": "The answer widget type <code>{{name}}</code> is not recognised.", "jme.shunt.expected argument before comma": "Expected to see something between the opening bracket and the comma", "part.waiting for pre submit": "Your answer is being marked. Please wait.", "diagnostic.end test": "End the test.", "page.no stylesheet loaded": "The page's stylesheet file has not loaded.", "modal.confirm": "Confirm", "modal.alert": "Alert", "suspend.resumed header": "Attempt resumed", "jme.vector.value not an array of numbers": "Tried to construct a vector using a value that is not an array of numbers.", "jme.matrix.value not the right type": "Tried to construct a vector using a value of the wrong type.", "jme.subvars.html inserted twice": "An HTML value has been embedded twice. Consider defining a function to generate a new value each time it is used.", "jme.variables.invalid function language": "The language <code>{{language}}</code> is not valid.", "jme.variables.duplicate definition": "There is more than one definition of the variable <code>{{name}}</code>.", "math.random_integer_partition.invalid k": "The size of the partition must be between 1 and {{n}}.", "part.marking.parameter already in scope": "There is a variable named <code>{{name}}</code>, which is also the name of a marking parameter. Please rename the variable.", "part.marking.adaptive variable replacement refers to self": "This part refers to itself in a variable replacement for adaptive marking.", "part.marking.adaptive variable replacement refers to nothing": "This part contains an invalid variable replacement for adaptive marking.", "part.numberentry.display answer wrong type": "The display answer for this part is a value of type <code>{{got_type}}</code>, but should be a <code>{{want_type}}</code>.", "part.matrix.invalid type in prefilled": "There is an invalid value of type <code>{{n}}</code> in the array of pre-filled cells.", "diagnostic.make a choice": "Make a choice", "matrixmath.not square": "This operation only works on a square matrix.", "matrixmath.not invertible": "This operation only works on an invertible matrix.", "matrixmath.gauss-jordan elimination.not enough columns": "There must be at least as many columns as rows.", "question.required extension not available": "This question requires the extension <code>{{-extension}}</code> but it is not available.", "util.formatNumberNotation.unrecognised syntax": "The number formatting syntax <code>{{syntax}}</code> is not recognised.", "worksheet.number of exams to generate": "Number of sheets to generate", "worksheet.starting with id": "Starting with ID", "worksheet.show exam id": "Show sheet ID?", "worksheet.page break between questions": "Page breaks between questions?", "worksheet.page margins": "Page margins (mm)", "worksheet.text size": "Text size (pt)", "worksheet.generate": "Generate", "worksheet.generating exams": "Generating sheets", "worksheet.sheet id": "Sheet ID:", "worksheet.print single": "Print this sheet", "worksheet.print several": "Print these sheets", "worksheet.answer sheets": "Answer sheets", "worksheet.question sheets": "Question sheets", "worksheet.reconfigure": "Generate different sheets", "worksheet.show sheet": "Preview the sheet with ID:", "accessibility statement": "Accessibility statement and guide to adapting Numbas to your needs.", "exam.enter your name": "Your name:", "exam.attempt download security warning": "This exam is configured to allow you to download your data, but it is not running in a secure browser context. You will not be able to download your exam data. Contact your lecturer or teacher for help.", "result.download exam object": "Download your exam data", "control.return to question": "Return to the question", "control.show introduction": "Introduction", "analysis.header": "Analyse attempt data", "analysis.help.upload files": "Upload attempt data files that your students have given you.", "analysis.help.file input label": "Choose student attempt data files, or drag files onto this window.", "analysis.table.total": "Exam totals", "analysis.table.question": "Exam and question totals", "analysis.table.all": "All details", "analysis.student name.anonymous": "No name given", "analysis.expected": "Expected results", "analysis.start time": "Start time", "analysis.maximum": "Maximum Marks", "analysis.file": "File", "analysis.download": "Download", "analysis.delete": "Delete", "analysis.view results": "View results", "analysis.upload files": "Upload files", "analysis.upload more": "Upload more files", "analysis.attempt data": "Attempt data", "analysis.select format": "Select data to show", "analysis.download this table": "Download this table", "analysis.student": "Student Results", "analysis.question key": "Question Key", "analysis.question name": "Question Name", "analysis.group": "Group", "analysis.question": "Question", "analysis.part": "Part", "analysis.gap": "Gap", "analysis.record type": "Record Type", "analysis.score": "Score", "analysis.marks available": "Marks Available", "analysis.percentage": "Percentage", "analysis.answer": "Answer", "analysis.student name": "Student Name", "analysis.summary.no files": "You have not uploaded any files yet.", "analysis.summary.no decrypted files": "You have not uploaded any valid files yet.", "analysis.summary.one file": "One attempt.", "analysis.summary.several files": "{{num_files,niceNumber}} attempts.", "analysis.not secure context": "This page must be opened in a secure browser context. A secure context is either a page served over HTTPS, or a file loaded from your device.", "jme.shunt.pipe right hand takes no arguments": "The expression on the right-hand side of the pipe operator must be a function application.", "question.explore.no parts defined": "There are no parts defined in this question.", "answer": "answer", "worksheet.answersheet show question content": "Show question content in answer sheets?", "modal.confirm end exam": "Write <code>{{endConfirmation}}</code> in the box to confirm:", "modal.end exam button": "End exam", "lightbox.zoom in on image": "Zoom in on this image", "exam.progress": "Progress", "exam.questions answered": "{{numAnsweredQuestions}} of {{numQuestions}} questions answered.", "result.question marks available": "Marks Available", "result.question answered": "Answered?", "control.confirm end.correct": "You may now end the exam.", "control.confirm end.incorrect": "This is not the expected text.", "control.confirm end.password": "end", "jme.typecheck.for in name wrong type": "The name in a <code>for</code> statement must be a name or list of names, not {{type}}.", "jme.makeFast.no fast definition of function": "The function <code>{{name}}</code> here isn't defined in a way that can be made fast.", "part.show feedback": "Show feedback", "part.hide feedback": "Hide feedback", "part.feedback title": "Feedback for {{name}}.", "part.jme.must-match.warning": "Your answer is not in the expected form: {{-message}}", "part.numberentry.write your answer as a fraction": "Write your answer as a fraction.", "question.nav.label": "Question controls", "question.answer saved": "Answer saved", "question.all answers saved": "All answers saved", "analysis.back to results": "Back to results", "analysis.review": "Review", "analysis.review this": "Review this attempt", "analysis.reviewing attempt": "Reviewing attempt by {{student_name}}.", "part.there is new feedback": "The feedback has changed.", "modal.style.colour scheme": "Colour scheme", "modal.style.automatic colour scheme": "Automatic", "modal.style.light colour scheme": "Light", "modal.style.dark colour scheme": "Dark", "modal.style.custom colour scheme": "Custom", "modal.style.main font": "Main font", "modal.style.font.sans serif": "Sans serif", "modal.style.font.serif": "Serif", "modal.style.font.monospace": "Monospace", "modal.style.spacing": "Spacing", "modal.style.font weight": "Font weight", "modal.style.forced colours warning": "Your browser has overridden the colours used in this page because of a setting such as high contrast mode. Changes to the colour scheme settings here might not have any effect.", "modal.style.text preview": "Most text will look like this.", "modal.style.more options": "More options", "modal.style.main colour": "Main (brand) colour", "modal.style.primary colour": "Primary button colour", "modal.style.link colour": "Link colour", "modal.style.success colour": "Success/correct colour", "modal.style.info colour": "Info colour", "modal.style.warning colour": "Warning colour", "modal.style.danger colour": "Danger/incorrect colour", "modal.style.muted colour": "Muted text colour", "modal.style.highlight colour": "Highlight colour", "exam.error loading exam definition": "There was an error while loading the exam definition: {{text}}", "exam.no exam definition": "No exam definition was given.", "jme.typecheck.wrong arguments for anonymous function": "Wrong number of arguments for this anonymous function.", "worksheet.top": "Top", "worksheet.left": "Left", "worksheet.bottom": "Bottom", "worksheet.right": "Right"}}, "sv-se": {"translation": {"page.loading": "Laddar...", "page.saving": "<p>Sparar.</p>\n<p>Detta kan ta n\u00e5gra sekunder</p>", "mathjax.math processing error": "\"{{-message}}\" n\u00e4r Numbas konverterade <code>{{expression}}</code> till LaTeX", "die.numbas failed": "Numbas kunde inte starta", "die.sorry": "Numbas fick ett fel som stannade programmet. Underliggande finns en beskrivning av felet.", "die.error": "Fel", "modal.ok": "OK", "modal.cancel": "Avbryt", "exam.exam name": "Namn p\u00e5 prov:", "exam.random seed": "Sessions ID:", "exam.student name": "Studentens namn", "exam.number of questions": "Antal fr\u00e5gor:", "exam.marks available": "Po\u00e4ng tillg\u00e4ngliga", "exam.pass percentage": "Godk\u00e4nd percentil", "exam.time allowed": "Till\u00e5ten tid", "exam.passed": "Godk\u00e4nd", "exam.failed": "Icke godk\u00e4nt", "exam.review header": "Granska:", "frontpage.start": "Start", "suspend.paused header": "Pausad", "suspend.exam suspended": "Provet har pausats. Tryck p\u00e5 <em>\u00c5teruppta</em> f\u00f6r att forts\u00e4tta.", "suspend.you can resume": "Du kan forts\u00e4tta n\u00e4sta g\u00e5ng du startar denna aktivitet", "suspend.resume": "\u00c5teruppta", "result.exit": "Avsluta prov", "result.print": "Skriv ut resultat-statistik", "result.exam summary": "Summering prov", "result.performance summary": "Summering utfall", "result.exam start": "Prov start:", "result.exam stop": "Prov avslutning:", "result.time spent": "Tid anv\u00e4nt:", "result.questions attempted": "Fr\u00e5gor ", "result.score": "Po\u00e4ng", "result.result": "Resultat:", "result.question number": "Fr\u00e5ga", "result.question score": "Resultat", "result.question review title": "\u00c5terkoppla denna fr\u00e5ga", "result.click a question to review": "Klicka p\u00e5 en fr\u00e5ga f\u00f6r att se hur dina svar \u00e4r bed\u00f6mda samt f\u00f6r att se eventuella l\u00f6sningar.", "end.exam has finished": "Provet har nu avslutats. Du kan st\u00e4nga detta f\u00f6nster.", "control.confirm leave": "Du har inte avslutat ditt prov", "control.not all questions answered": "Du har inte svarat p\u00e5 alla fr\u00e5gor i detta prov.", "control.not all questions submitted": "Du har \u00e4ndrat n\u00e5got eller n\u00e5gra svar men inte sparat dina \u00e4ndringar. Var sn\u00e4ll och se s\u00e5 alla fr\u00e5gor har sparats.", "control.confirm end": "\u00c4r du s\u00e4ker p\u00e5 att du vill avsluta ditt prov? Efter du avslutat kan du inte g\u00e5 tillbaka och \u00e4ndra dina svar.", "control.confirm regen": "Vill du skapa en ny version av fr\u00e5gan? Om du v\u00e4ljer OK f\u00f6rsvinner alla dina eventuella svar och po\u00e4ng samt fr\u00e5gan \u00e4ndras.", "control.confirm reveal": "Vill du visa svaret till denna fr\u00e5ga? Eventuella po\u00e4ng kommer l\u00e5sas samt du kommer inte kunna svara p\u00e5 fr\u00e5gan i efterhand. ", "control.proceed anyway": "Forts\u00e4tt \u00e4nd\u00e5?", "control.regen": "F\u00f6rs\u00f6k med en liknande fr\u00e5ga", "control.submit answer": "Skicka in svar", "control.submit all parts": "Skicka in alla delar", "control.submit again": "Skicka in igen", "control.submit": "Skicka", "control.previous": "F\u00f6reg\u00e5ende", "control.next": "N\u00e4sta", "control.advice": "R\u00e5d", "control.reveal": "Visa svar", "control.total": "Totalpo\u00e4ng", "control.pause": "Paus", "control.end exam": "Avsluta prov", "control.back to results": "G\u00e5 tillbaka till resultat", "display.part.jme.error making maths": "Ett fel uppstod n\u00e4r math display skulle skapas", "exam.xml.bad root": "Rot elementet av prov XML borde vara 'prov'", "exam.changeQuestion.no questions": "Detta prov inneh\u00e5ller inga fr\u00e5gor! Unders\u00f6k .exam filen f\u00f6r att hitta fel.", "feedback.you were awarded": "Du har f\u00e5tt <strong>{{count,niceNumber}}</strong> $t(mark).", "feedback.taken away": "<strong>{{count,niceNumber}}</strong> $t(mark) $t(was) tagits bort", "jme.tokenise.invalid": "Ogiltigt uttryck: <code>{{expression}}</code>", "jme.shunt.not enough arguments": "inte nog med argument f\u00f6r operationen {{op}}", "jme.shunt.no left bracket in function": "Ingen matchande v\u00e4nster-bracket i funktion eller serie", "jme.shunt.no left square bracket": "Ingen matchande v\u00e4nster-bracket", "jme.shunt.no left bracket": "Ingen matchande v\u00e4nster-bracket", "jme.shunt.no right bracket": "Ingen matchande h\u00f6ger-bracket", "jme.shunt.no right square bracket": "Ingen matchande h\u00f6ger-klammer som avslutar listan", "jme.shunt.missing operator": "Uttrycket kan inte utv\u00e4rderas -- Det saknas en operator.", "jme.typecheck.function maybe implicit multiplication": "Operationen {{name}} \u00e4r inte definierad. Menade du <br/><code>{{first}}*{{possibleOp}}(...)</code>?", "jme.typecheck.function not defined": "Operationen {{op}} \u00e4r inte definierad. Menade du <br/><code>{{suggestion}}*(...)</code>?", "jme.typecheck.op not defined": "Operationen {{op}} \u00e4r inte definierad.", "jme.typecheck.no right type definition": "Ingen definition av '{{op}}' av korrekt typ hittades", "jme.typecheck.no right type unbound name": "Variabeln <code>{{name}}</code> \u00e4r inte definierad.", "jme.typecheck.map not on enumerable": "<code>map</code> operationen m\u00e5ste arbeta med en lista eller ett intervall, inte {{type}}", "jme.evaluate.undefined variable": "Variabeln {{name}} \u00e4r odefinierad", "jme.thtml.not html": "THTML konstrukt\u00f6ren fick ett icke HTML v\u00e4rde.", "jme.func.switch.no default case": "Inget standardstatement fall f\u00f6r switch uttryck", "jme.func.listval.invalid index": "index f\u00f6r lista \u00e4r ogiltig  {{index}} p\u00e5 lista av storlek {{size}}", "jme.func.listval.not a list": "Objektet \u00e4r inte prenumereringsbart", "jme.func.matrix.invalid row type": "Kan inte konstruera ett matrix fr\u00e5n rader av text {{type}}", "jme.func.except.continuous range": "Kan inte anv\u00e4nda 'except' operatorn p\u00e5 kontinuerliga intervall", "jme.matrix.reports bad size": "Matrisen visar fel storlek - det m\u00e5ste vara n\u00e5got fel i funtionen som konstruerade den.", "jme.texsubvars.no right bracket": "Inga matchande <code>]</code> i {{op}} argumenten.", "jme.texsubvars.missing parameter": "Parameter saknas i {{op}}: {{parameter}}", "jme.texsubvars.no right brace": "Ingen matchande <code>}</code> in {{op}}", "jme.user javascript.error": "Fel i anv\u00e4ndar-definierad javascript funktion <code>{{name}}</code>: {{-message}}", "jme.variables.error making function": "Del uppstod n\u00e4r funktionen skapades <code>{{name}}</code>: {{-message}}", "jme.variables.syntax error in function definition": "Syntaxfel i funktionens definition", "jme.variables.variable not defined": "Variable <code>{{name}}</code> is not defined.", "jme.variables.empty definition": "Variabelns definition <code>{{name}}</code> \u00e4r tom.", "jme.variables.circular reference": "Circular variable reference in definition of <code>{{name}}</code>", "jme.variables.error computing dependency": "Fel i ber\u00e4kning av referens variabeln <code>{{name}}</code>", "jme.variables.error evaluating variable": "Fel i evaluering av variabel {{name}}: {{-message}}", "jme.variables.question took too many runs to generate variables": "Ett set av fr\u00e5gevariabler kunde inte genereras i tid", "jme.display.unknown token type": "Kan inte textifiera token skrift {{type}}", "jme.display.collectRuleset.no sets": "Inga sets \u00e4r givna till Samlade-Regler-setet!", "jme.display.collectRuleset.set not defined": "Regelsamlingen {{name}} har inte definierats", "jme.display.simplifyTree.no scope given": "Numbas.jme.display.simplify Tr\u00e4det m\u00e5ste ges en omfattning", "math.precround.complex": "Kan inte avrunda till ett komplext tal", "math.siground.complex": "Kan inte avrunda till ett komplext tal av signifikant v\u00e4rde", "math.combinations.complex": "Kan inte ber\u00e4kna kombinationen av komplexa tal", "math.permutations.complex": "Kan inte ber\u00e4kna permutationer av komplexa tal", "math.gcf.complex": "Kan inte ber\u00e4kna SGF av komplexa tal ", "math.lcm.complex": "Kan inte ber\u00e4kna MGM av komplexa tal ", "math.lt.order complex numbers": "Kan inte sortera complexa tal", "math.choose.empty selection": "tom sektion given till random-funktionen ", "matrixmath.abs.non-square": "Kan inte ber\u00e4kna determinanten av en matrix som inte \u00e4r fyrkantig", "matrixmath.abs.too big": "Urs\u00e4kta, systemet kan inte ber\u00e4kna determinanter av matriser st\u00f6rre \u00e4n 3x3.", "matrixmath.mul.different sizes": "Kan inte multiplicera matriser av olika storlekar.", "vectormath.cross.not 3d": "Kan bara ta kryssprodukten av tre dimensions vektorer.", "vectormath.dot.matrix too big": "Kan inte ber\u00e4kna dot produkten av en matrix som inte \u00e4r $1 \\times N$ eller $N \\times 1$.", "vectormath.cross.matrix too big": "Kan inte ber\u00e4kna kors produkten av en matrix som inte \u00e4r $1 \\g\u00e5nger N$ eller $N \\g\u00e5nger 1$.", "part.with steps answer prompt": "Svar: ", "part.script.error": "Fel i del {{path}} script {{script}}: {{-message}}", "part.marking.steps no matter": "Eftersom du redan r\u00e4ttat delen kan du inte f\u00e5 po\u00e4ng f\u00f6r nya svar.", "part.marking.revealed steps no penalty": "Du har \u00f6ppnat steg l\u00f6sningen.", "part.marking.used variable replacements": "Denna del r\u00e4ttades enligt dina tidigare svar", "part.marking.variable replacement part not answered": "Du m\u00e5ste svara p\u00e5 {{part}} f\u00f6rst", "part.marking.resubmit because of variable replacement": "Denna dels po\u00e4ng beror p\u00e5 svar p\u00e5 andra delar, dessa svar \u00e4r \u00e4ndrade s\u00e5 spara igen f\u00f6r att uppdatera din po\u00e4ng.", "part.marking.not submitted": "Inget svar inskickat", "part.marking.did not answer": "Du svarade inte p\u00e5 denna fr\u00e5ga", "part.marking.nothing entered": "Du har inte skrivit in ett svar.", "part.marking.incorrect": "Ditt svar \u00e4r inkorrekt.", "part.marking.correct": "Ditt svar \u00e4r korrekt.", "part.marking.uncaught error": "Fel i skapning av {{part}}: {{-message}}", "part.marking.no result": "Denna del kunde inte r\u00e4ttas", "part.correct answer": "F\u00f6rv\u00e4ntat svar.", "part.missing type attribute": "{{part}}: det saknas en attribut", "part.unknown type": " {{part}}: Icke-l\u00e4sbar del {{type}}", "part.setting not present": "Egenskap '{{property}}' inte satt", "part.jme.answer missing": "Korrekt svar saknas", "part.jme.answer too long": "Ditt svar \u00e4r f\u00f6r l\u00e5ngt", "part.jme.answer too short": "Ditt svar \u00e4r f\u00f6r kort", "part.jme.answer invalid": "Ditt svar \u00e4r inte ett korrekt matematiskt uttryck.<br/>{{-message}}.", "part.jme.marking.correct": "Ditt svar \u00e4r numeriskt korrekt.", "part.jme.must-have bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.must-have one": "Ditt svar m\u00e5ste inneh\u00e5lla: {{strings}}", "part.jme.must-have several": "Ditt svar m\u00e5ste inneh\u00e5lla alla: {{strings}}", "part.jme.not-allowed bits": " <span class=\"monospace\">{{string}}</span>", "part.jme.not-allowed one": "Ditt svar kan inte inneh\u00e5lla: {{strings}}", "part.jme.not-allowed several": "Ditt svar kan inte inneh\u00e5lla n\u00e5gon av: {{strings}}", "part.jme.unexpected variable name": "Ditt svar verkar inneh\u00e5lla en ogiltigt variabelnamn <code>{{name}}</code>.", "part.jme.unexpected variable name suggestion": "Ditt svar verkar inneh\u00e5lla en ogiltigt variabelnamn <code>{{name}}</code>. Menade du <code>{{suggestion}}</code>?", "part.patternmatch.display answer missing": "visat svar saknas", "part.patternmatch.correct except case": "Ditt svar \u00e4r korrekt, f\u00f6rutom enheten.", "part.numberentry.correct except decimal": "Ditt svar \u00e4r i r\u00e4tt intervall, dock till\u00e5ts inte decimaler.", "part.numberentry.correct except fraction": "Ditt svar \u00e4r i r\u00e4tt intervall, dock till\u00e5ts inte br\u00e5k.", "part.numberentry.answer invalid": "Du skrev inte en siffra", "part.numberentry.answer not integer": "Ditt svar \u00e4r felaktigt. Du m\u00e5ste anv\u00e4nda positiva heltal, inte decimaler.", "part.numberentry.answer not integer or decimal": "Ditt svar \u00e4r felaktigt. Du m\u00e5ste anv\u00e4nda heltal eller decimaler.", "part.numberentry.zero sig fig": "Du kan inte avrunda ett svar till noll v\u00e4rdesiffror.", "part.mcq.options def not a list": "Uttrycket som definierar {{properties}} \u00e4r inte en lista.", "part.mcq.marking matrix string empty": "R\u00e4ttnings-matrisen \u00e4r tom", "part.mcq.choices missing": "Definition av valen saknas.", "part.mcq.matrix not a number": "Del {{part}} r\u00e4ttnings-matrisen cell ({{row}},{{column}}) blir inget nummer", "part.mcq.wrong number of choices": "Du valde fel antal svar.", "part.mcq.no choices selected": "Inga svar \u00e4r valda.", "part.mcq.matrix not a list": "R\u00e4ttnings-matrisen, definierad av JME uttrycket, m\u00e5ste vara en lista.", "part.mcq.matrix wrong type": "Element av ogiltig typ '{{type}}' anv\u00e4nds i r\u00e4ttnings-matrisen", "part.mcq.matrix mix of numbers and lists": "En blandning av siffror och listor anv\u00e4nds i r\u00e4ttnings-matrisen.", "part.mcq.matrix wrong size": "Matrisen \u00e4r av fel storlek", "part.mcq.correct choice": "Du valde r\u00e4tt svar.", "part.matrix.answer invalid": "Ditt svar \u00e4r inte giltigt.", "part.matrix.invalid cell": "En eller fler av cellerna i ditt svar \u00e4r tomma eller ogiltiga.", "part.matrix.some incorrect": "En eller flera av cellerna i ditt svar \u00e4r inte korrekta, Du har dock f\u00e5tt po\u00e4ng f\u00f6r resterande svar.", "part.matrix.empty": "Du har inte skrivit n\u00e5got svar.", "part.matrix.empty cell": "En eller fler celler i ditt svar \u00e4r tomma.", "part.matrix.size mismatch": "Dimensionerna p\u00e5 svaret \u00e4r inte \u00f6ppna f\u00f6r \u00e4ndring, det korrekta dimensionen \u00e4r: \n{{correct_dimensions}} medan svaret givet \u00e4r: {{input_dimensions}}", "part.gapfill.feedback header": "<strong>Mellanrum {{index}}</strong>", "part.extension.not implemented": "Delen har inte anv\u00e4nt <code>{{name}}</code> metoden.", "question.loaded name mismatch": "Kan inte \u00e5teruppta detta f\u00f6rs\u00f6k. objektet har \u00e4ndrats sen du arbetade med det sist.", "question.error": "Fr\u00e5ga {{number}}: {{-message}}", "question.preamble.error": "Fel i inledningen: {{-message}}", "question.preamble.syntax error": "Syntax error i inledningen", "question.unsupported part type": "Ogiltig del typ", "question.header": "Fr\u00e5ga {{number}}", "question.submit part": "Spara del", "question.show steps": "Visa steg", "question.show steps penalty": "Du kommer f\u00f6rlora <strong>{{count,niceNumber}}</strong> $t(mark).", "question.show steps no penalty": "Dina po\u00e4ng kommer inte p\u00e5verkas.", "question.show steps already penalised": "Du har redan visat stegen. Du kan nu \u00f6ppna dem igen utan po\u00e4ngavdrag.", "question.hide steps": "G\u00f6m steg", "question.hide steps no penalty": "Dina po\u00e4nk kommer inte p\u00e5verkas.", "question.advice": "Tips", "question.no such part": "Kan inte hitta del {{path}}", "question.can not submit": "Kan inte spara svar - unders\u00f6k eventuella fel", "question.answer submitted": "Svar sparat", "question.score feedback.show": "Visa Feedback", "question.score feedback.hide": "G\u00f6m feedback", "question.score feedback.answered total actual": "Po\u00e4ng: {{score,niceNumber}}/{{marks,niceNumber}}", "question.score feedback.answered total": "{{marksString}}. Besvarade.", "question.score feedback.answered actual": "Po\u00e4ng: {{scoreString}}", "question.score feedback.answered": "Besvarad.", "question.score feedback.unanswered": "Obesvarad.", "question.score feedback.unanswered total": "{{marksString}}.", "question.score feedback.correct": "Ditt svar \u00e4r korrekt", "question.score feedback.partial": "Ditt svar \u00e4r delvis korrekt", "question.score feedback.wrong": "Ditt svar \u00e4r inkorrekt", "question.selector.unsubmitted changes": "Ickesparade \u00e4ndringar", "timing.no accumulator": "Ingen tids-ackumulator {{name}}", "timing.time remaining": "Tid Kvar:", "xml.could not load": "Kunder inte ladda XML dokumentet: {{-message}}", "xml.property not number": "Egenskap {{name}} borde vara en siffra, men \u00e4r det inte ({{value}}), i node {{element}}", "xml.property not boolean": "Egenskap {{name}} borde vara en boolean, men \u00e4r det inte ({{value}}), i node {{element}}", "xml.error in variable definition": "En variabel kunde inte definieras:  <code>{{name}}</code>", "scorm.error initialising": "Ett fel uppstod i SCORM protokollet: {{-message}}", "scorm.failed save": "<p> Sparnings-f\u00f6rfr\u00e5gan har misslyckats. Tryck <b>OK</b> f\u00f6r att f\u00f6rs\u00f6ka igen.</p> <p>Om du vid upprepade tillf\u00e4llen f\u00e5r detta meddelande kontrollera din internet-uppkoppling eller dator. Dina f\u00f6reg\u00e5ende sparade svar \u00e4r kvar och kan laddas p\u00e5 en annan dator</p>\n<p>Om du har problem med att svara <em>any</em> var sn\u00e4ll och kontakta din l\u00e4rare f\u00f6r hj\u00e4lp.</p>", "scorm.no exam suspend data": "Kunde inte \u00e5teruppta: ingen tidigare data hittades.", "scorm.error loading suspend data": "Fel i laddning av data: {{-message}}", "scorm.error loading question": "Ett fel uppstod vid laddning av fr\u00e5ga {{number}}: {{-message}}", "scorm.no question suspend data": "Ingen fr\u00e5ga, stannar", "scorm.error loading part": "Fel i laddning av del {{part}}: {{-message}}", "scorm.no part suspend data": "Ingen del, stannar", "util.product.non list": "En icke lista gavs till <code>Numbas.util.product</code>", "mark": "po\u00e4ng", "was": "var", "part": "del", "gap": "mellanrum", "step": "steg", "jme.substituteTree.undefined variable": "Odefinierad variabel: <code>{{name}}</code>", "jme.user javascript.returned undefined": "Javascript funktionen gav en felkod: <code>{{name}}</code> returned <code>undefined</code>.", "part.marking.steps change": "Du fick  <strong>{{count,niceNumber}}</strong> $t(mark) po\u00e4ng f\u00f6r dina svar p\u00e5 denna del.", "part.marking.revealed steps with penalty": "Du visade l\u00f6snings-stegen. Ditt maxpo\u00e4ng f\u00f6r denna delen \u00e4r <strong>{{count,niceNumber}}</strong> $t(mark). Dina po\u00e4ng skalas ned p\u00e5 grund av l\u00f6sningen.", "part.marking.total score": "Du fick <strong>{{count,niceNumber}}</strong> $t(mark) po\u00e4ng f\u00f6r denna del.", "part.numberentry.precision type.dp": "decimal", "part.numberentry.precision type.dp_plural": "decimaler", "part.numberentry.precision type.sigfig": "v\u00e4rdesiffra", "part.numberentry.precision type.sigfig_plural": "v\u00e4rdesiffror", "part.numberentry.give your answer to precision": "Avrunda ditt svar till {{count,niceNumber}} {{precisionType}}.", "question.unsubmitted changes": "Du har \u00e4ndrat svaret men inte sparat. Var sn\u00e4ll och <strong>Spara svar</strong> button. efter att du kontrollerat svaret.", "question.unsubmitted changes_plural": "Du har \u00e4ndrat svaret men inte sparat. Var sn\u00e4ll och <strong>Spara svar</strong> button. efter att du kontrollerat svaret till alla delar.", "util.equality not defined for type": "Ekvivalens \u00e4r inte definierat f\u00f6r text {{type}}", "mark_plural": "po\u00e4ng", "was_plural": "var", "die.script not loaded": "Numbas kunde inte startas f\u00f6r att filen <code>{{file}}</code> inte laddades. Unders\u00f6k om filen finns med i <code>scripts.js</code>.", "math.combinations.n less than zero": "Kan inte ber\u00e4kna kombinationer: n \u00e4r mindre \u00e4n 0", "math.combinations.k less than zero": "Kan inte ber\u00e4kna kombinationer: k \u00e4r mindre \u00e4n 0", "math.combinations.n less than k": "Kan inte ber\u00e4kna kombinationer: n \u00e4r mindre \u00e4n k", "math.permutations.n less than zero": "Kan inte ber\u00e4kna permutationer: n \u00e4r mindre \u00e4n 0", "math.permutations.k less than zero": "Kan inte ber\u00e4kna permutationer: k \u00e4r mindre \u00e4n 0", "math.permutations.n less than k": "Kan inte ber\u00e4kna permutationer: n \u00e4r mindre \u00e4n k", "part.numberentry.give your answer to precision_0": "Avrunda dina svar till n\u00e4rmsta heltal.", "mathjax.error": "Ett fel uppstod i MathJax {{context}}: {{-message}}", "mathjax.error with context": "MathJax felaktig process i {{context}}: {{-message}}", "exam.introduction": "Prov introduktion", "exam.feedback": "Prov, feedback meddelande", "jme.tokenise.keypair key not a string": "Ordboks nyckeln b\u00f6r vara en \"string\" inte {{type}}.", "jme.shunt.list mixed argument types": "Kan inte f\u00f6rst\u00e5 {{mode}}: en mix av olika typer av element (listor och ord).", "jme.func.listval.key not in dict": "Ordboken inneh\u00e5ller inte nyckeln  <code>{{key}}</code>", "part.prompt": "Meddelande", "part.feedback": "Feedback", "part.numberentry.answer not reduced": "Ditt svar \u00e4r inte i enklaste form.", "part.numberentry.give your answer as a reduced fraction": "F\u00f6renkla ditt svar till enklaste form.", "part.numberentry.negative decimal places": "Denna del avrundar studentens svar till ett negativt nummer decimaler. Ogiltigt kommando.", "part.mcq.choices": "Val", "part.mcq.answers": "Svar", "part.mcq.matrix cell empty": "Del {{part}} r\u00e4ttnings-matris cell ({{row}},{{column}}) \u00e4r tom.", "part.mcq.matrix jme error": "Del {{part}} r\u00e4ttnings-matris cell ({{row}},{{column}}) ger ett JME error: {{error}}", "question.statement": "P\u00e5st\u00e5ende", "ruleset.circular reference": "Cirkul\u00e4r referens i definitionen av regelsetet <code>{{name}}</code>", "ruleset.set not defined": "refelsetet {{name}} har inte definierats", "jme.evaluate.no scope given": "Numbas.jme.evaluate m\u00e5ste ges ett intervall.", "answer.number.not a number": "Your answer is not a valid number.", "answer.number.fractions not allowed": "You may not enter a fraction.", "answer.jme.invalid expression": "{{-message}}", "answer.matrix.fractions not allowed": "You may not enter fractions.", "answer.matrix.some cell not a number": "One or more of the cells in your answer is not a valid number.", "exam.enter password": "Password:", "exam.password.correct": "This password is correct. You can start the exam.", "exam.password.incorrect": "This password is incorrect.", "frontpage.scorm.lms not connected": "This exam is running in standalone mode. Your answers and marks will not be saved!", "result.question review": "Review", "control.confirm regen no marks": "Would you like to re-randomise this question?", "control.confirm reveal no marks": "Would you like to reveal the answer to this question?", "jme.tokenise.invalid near": "Invalid expression: <code>{{expression}}</code> at position {{position}} near <code>{{nearby}}</code>", "jme.tokenise.number.object not complex": "Invalid object passed into number constructor.", "jme.subvars.null substitution": "Empty variable substitution: <code>$t(left brace){{str}}$t(right brace)", "jme.type.type already registered": "The data type {{type}} has already been registered so can't be registered again.", "jme.type.no cast method": "Can't automatically convert from {{from}} to {{to}}.", "jme.display.simplifyTree.empty expression": "Expression is empty", "jme.display.simplifyTree.stuck in a loop": "Simplifier is stuck in a loop: <code>{{expr}}</code>", "math.niceNumber.undefined": "Was expecting a number, but got <code>undefined</code>", "math.rangeToList.zero step size": "Can't convert a range with step size zero to a list.", "part.error": "{{path}}: {{-message}}", "part.marking.revealed steps": "You revealed the steps.", "part.marking.maximum scaled down": "The maximum you can score for this part is <strong>{{count,niceNumber}}</strong> $t(mark). Your scores will be scaled down accordingly.", "part.marking.minimum score applied": "The minimum score for this part is <strong>{{score,niceNumber}}</strong>.", "part.marking.maximum score applied": "The maximum score for this part is <strong>{{score,niceNumber}}</strong>.", "part.marking.error in marking script": "There was an error in this part's marking algorithm. Please report this. {{-message}}", "part.marking.no result after replacement": "This part could not be marked using your answers to previous parts.", "part.marking.missing required note": "The marking algorithm does not define the note <code>{{note}}</code>", "marking.apply.not a list": "The first argument to <code>apply</code> must be a list, and isn't", "marking.apply marking script.script not found": "Marking script <code>{{name}}</code> not found", "marking.note.compilation error": "Error compiling note <code>{{name}}</code>: {{-message}}", "marking.note.error evaluating note": "Error evaluating note <code>{{name}}</code> - {{-message}}", "marking.note.invalid definition": "Invalid note definition: <code>{{source}}</code>. {{-hint}}", "marking.note.invalid definition.missing colon": "You might be missing a colon after the name and description", "marking.note.invalid definition.description missing closing bracket": "You might be missing a closing bracket", "marking.note.empty expression": "The note <code>{{name}}</code> is empty.", "marking.script.error parsing notes": "Error parsing marking script: {{- message}}", "part.feedback out of date": "This feedback is based on your last submitted answer. Save your changed answer to get updated feedback.", "part.jme.invalid value generator expression": "Invalid value generator expression for variable <code>{{name}}</code>: {{-message}}", "part.mcq.incorrect choice": "You chose an incorrect answer.", "part.matrix.not all cells same precision": "You have not given every cell in your answer to the same precision.", "part.gapfill.error marking gap": "Error marking {{name}}: {{-message}}", "part.custom.empty setting": "No value given.", "part.custom.unrecognised input type": "Unrecognised setting type <code>{{input_type}}</code>", "part.custom.error evaluating input option": "Error evaluating input option <code>{{option}}</code>: {{-error}}", "part.custom.input option missing": "Definition of input option <code>{{option}}</code> is missing.", "part.custom.error evaluating setting": "Error evaluating setting <code>{{setting}}</code>: {{-error}}", "question.error creating question": "Error while creating question {{number}}: {{-message}}", "question.score feedback.not marked": "Not marked", "question.score feedback.partially answered": "Partially answered", "question.score feedback.score total": "{{marksString}}", "question.score feedback.score actual": "Po\u00e4ng: {{scoreString}}", "question.score feedback.score total actual": "Po\u00e4ng: {{score,niceNumber}}/{{marks,niceNumber}}", "variable.error in variable definition": "Error in definition of variable <code>{{name}}</code>", "left brace": "{", "right brace": "}", "extension.not found": "Couldn't load the extension <code>{{name}}</code>.", "control.toggle navigation menu": "Toggle the navigation menu", "part.input title": "Answer for part {{name}}", "part.correct answer title": "Expected answer for part {{name}}", "part.jme.must-match.failed": "Your answer is not in the expected form.", "question.score feedback.none": "", "control.submit part.confirm remove next parts": "<p>One or more subsequent parts depend on your answer to this part. Submitting this part again will invalidate those parts, and remove them from the question. This cannot be undone.</p>\n<p>Would you like to submit this part again?</p>", "control.back to menu": "Go back to the menu", "display.error making html": "Error making HTML in {{contextDescription}}: {{-message}}", "jme.subvars.error compiling": "{{-message}} in <code>{{expression}}</code>", "jme.variables.empty name": "A question variable has not been given a name.", "jme.calculus.unknown derivative": "Don't know how to differentiate <code>{{tree}}</code>", "math.order complex numbers": "Can't order complex numbers", "menu.choose a question": "Choose a question.", "part.choose next part.answered": "What do you want to do next?", "part.choose next part.unanswered": "Or, you could:", "part.choose next part.will be locked": "(This part will be locked)", "part.reached dead end": "There's nothing more to do from here.", "part.next part.penalty amount": "(lose {{count}} $t(mark))", "part.marking.counts towards objective": "This part counts towards the objective <strong>\u201c{{objective}}\u201d</strong>.", "part.numberentry.answer not integer or decimal or fraction": "Your answer is invalid. You must enter an integer, a decimal or a fraction.", "question": "Question", "question.progress": "Question progress", "question.score feedback.unattempted": "Not attempted", "question.score feedback.attempted": "Attempted", "question.score feedback.score actual.plain": "{{scoreString}}", "question.score feedback.score total actual.plain": "{{score,niceNumber}}/{{marks,niceNumber}}", "question.objectives": "Objectives", "question.penalties": "Penalties", "question.back to previous part": "Go back to the previous part", "end.print": "Print your exam transcript", "math.shuffle_together.lists not all the same length": "Not all lists are the same length.", "jme.parse signature.invalid signature string": "Invalid function signature string: {{str}}", "part.custom.expected answer has wrong type": "The expected answer for this part has the wrong type. It should be <code>{{shouldbe}}</code>.", "part.custom.input option has wrong type": "The answer input setting <code>{{option}}</code> has the wrong type. It should be <code>{{shouldbe}}</code>.", "matrix input.size control legend": "Size", "matrix input.rows": "Rows", "matrix input.columns": "Columns", "part.jme.error checking numerically": "There was an error numerically checking your answer: {{-message}}", "part.gapfill.cyclic adaptive marking": "There is a cycle in the adaptive marking for this part: <strong>{{name1}}</strong> relies on <strong>{{name2}}</strong>, which eventually relies on <strong>{{name1}}</strong>.", "modal.style.background colour": "Background colour", "modal.style.text colour": "Text colour", "modal.style.text size": "Text size", "modal.style.explanation": "Use these controls to change the appearance of the exam.", "modal.style.reset to defaults": "Reset to defaults", "modal.style.text size preview": "Most text will be this big.", "control.style options": "Display options", "part.marking.partially correct": "Your answer is partially correct.", "part.marking.error in adaptive marking": "There was an error in the adaptive marking for this part. Please report this. {{-message}}", "page.skip to content": "Skip to content", "result.learning objective": "Learning objective", "jme.interpreted as": "interpreted as", "jme.script.note.compilation error": "Error compiling note <code>{{name}}</code>: {{-message}}", "jme.script.note.error evaluating note": "Error evaluating note <code>{{name}}</code> - {{-message}}", "jme.script.note.invalid definition": "Invalid note definition: <code>{{source}}</code>. {{-hint}}", "jme.script.note.invalid definition.missing colon": "You might be missing a colon after the name and description", "jme.script.note.invalid definition.description missing closing bracket": "You might be missing a closing bracket", "jme.script.note.empty expression": "The note <code>{{name}}</code> is empty.", "jme.script.error parsing notes": "Error parsing marking script: {{- message}}", "matrix input.cell label": "Row {{row}}, column {{column}}", "control.move to next question": "Move to the next question", "diagnostic.use retry": "Use one retry and try this topic again.", "diagnostic.move to next topic": "Move on to the next topic.", "diagnostic.next step question": "What would you like to do next?", "diagnostic.now assessing topic": "Now assessing {{current_topic}}", "diagnostic.one retry left": "You have 1 retry left", "diagnostic.retries left": "You have {{retries}} retries left.", "diagnostic.percentage completed": "You've completed <strong>{{percentage}}%</strong> of the test.", "diagnostic.test is over": "The test is over.", "diagnostic.passed all lo": "You have passed all learning objectives.", "diagnostic.more work on lo": "You need to do some more work on the following learning objectives: {{los}}.", "diagnostic.move to next question in topic": "Move on to the next question in topic.", "diagnostic.complete": "Complete!", "diagnostic.studying topic": "Studying {{topic}}", "display.answer widget.unknown widget type": "The answer widget type <code>{{name}}</code> is not recognised.", "jme.shunt.expected argument before comma": "Expected to see something between the opening bracket and the comma", "part.waiting for pre submit": "Your answer is being marked. Please wait.", "diagnostic.end test": "End the test.", "page.no stylesheet loaded": "The page's stylesheet file has not loaded.", "modal.confirm": "Confirm", "modal.alert": "Alert", "suspend.resumed header": "Attempt resumed", "jme.vector.value not an array of numbers": "Tried to construct a vector using a value that is not an array of numbers.", "jme.matrix.value not the right type": "Tried to construct a vector using a value of the wrong type.", "jme.subvars.html inserted twice": "An HTML value has been embedded twice. Consider defining a function to generate a new value each time it is used.", "jme.variables.invalid function language": "The language <code>{{language}}</code> is not valid.", "jme.variables.duplicate definition": "There is more than one definition of the variable <code>{{name}}</code>.", "math.random_integer_partition.invalid k": "The size of the partition must be between 1 and {{n}}.", "part.marking.parameter already in scope": "There is a variable named <code>{{name}}</code>, which is also the name of a marking parameter. Please rename the variable.", "part.marking.adaptive variable replacement refers to self": "This part refers to itself in a variable replacement for adaptive marking.", "part.marking.adaptive variable replacement refers to nothing": "This part contains an invalid variable replacement for adaptive marking.", "part.numberentry.display answer wrong type": "The display answer for this part is a value of type <code>{{got_type}}</code>, but should be a <code>{{want_type}}</code>.", "part.matrix.invalid type in prefilled": "There is an invalid value of type <code>{{n}}</code> in the array of pre-filled cells.", "diagnostic.make a choice": "Make a choice", "matrixmath.not square": "This operation only works on a square matrix.", "matrixmath.not invertible": "This operation only works on an invertible matrix.", "matrixmath.gauss-jordan elimination.not enough columns": "There must be at least as many columns as rows.", "question.required extension not available": "This question requires the extension <code>{{-extension}}</code> but it is not available.", "util.formatNumberNotation.unrecognised syntax": "The number formatting syntax <code>{{syntax}}</code> is not recognised.", "worksheet.number of exams to generate": "Number of sheets to generate", "worksheet.starting with id": "Starting with ID", "worksheet.show exam id": "Show sheet ID?", "worksheet.page break between questions": "Page breaks between questions?", "worksheet.page margins": "Page margins (mm)", "worksheet.text size": "Text size (pt)", "worksheet.generate": "Generate", "worksheet.generating exams": "Generating sheets", "worksheet.sheet id": "Sheet ID:", "worksheet.print single": "Print this sheet", "worksheet.print several": "Print these sheets", "worksheet.answer sheets": "Answer sheets", "worksheet.question sheets": "Question sheets", "worksheet.reconfigure": "Generate different sheets", "worksheet.show sheet": "Preview the sheet with ID:", "accessibility statement": "Accessibility statement and guide to adapting Numbas to your needs.", "exam.enter your name": "Your name:", "exam.attempt download security warning": "This exam is configured to allow you to download your data, but it is not running in a secure browser context. You will not be able to download your exam data. Contact your lecturer or teacher for help.", "result.download exam object": "Download your exam data", "control.return to question": "Return to the question", "control.show introduction": "Introduction", "analysis.header": "Analyse attempt data", "analysis.help.upload files": "Upload attempt data files that your students have given you.", "analysis.help.file input label": "Choose student attempt data files, or drag files onto this window.", "analysis.table.total": "Exam totals", "analysis.table.question": "Exam and question totals", "analysis.table.all": "All details", "analysis.student name.anonymous": "No name given", "analysis.expected": "Expected results", "analysis.start time": "Start time", "analysis.maximum": "Maximum Marks", "analysis.file": "File", "analysis.download": "Download", "analysis.delete": "Delete", "analysis.view results": "View results", "analysis.upload files": "Upload files", "analysis.upload more": "Upload more files", "analysis.attempt data": "Attempt data", "analysis.select format": "Select data to show", "analysis.download this table": "Download this table", "analysis.student": "Student Results", "analysis.question key": "Question Key", "analysis.question name": "Question Name", "analysis.group": "Group", "analysis.question": "Question", "analysis.part": "Part", "analysis.gap": "Gap", "analysis.record type": "Record Type", "analysis.score": "Score", "analysis.marks available": "Marks Available", "analysis.percentage": "Percentage", "analysis.answer": "Answer", "analysis.student name": "Student Name", "analysis.summary.no files": "You have not uploaded any files yet.", "analysis.summary.no decrypted files": "You have not uploaded any valid files yet.", "analysis.summary.one file": "One attempt.", "analysis.summary.several files": "{{num_files,niceNumber}} attempts.", "analysis.not secure context": "This page must be opened in a secure browser context. A secure context is either a page served over HTTPS, or a file loaded from your device.", "jme.shunt.pipe right hand takes no arguments": "The expression on the right-hand side of the pipe operator must be a function application.", "question.explore.no parts defined": "There are no parts defined in this question.", "answer": "answer", "worksheet.answersheet show question content": "Show question content in answer sheets?", "modal.confirm end exam": "Write <code>{{endConfirmation}}</code> in the box to confirm:", "modal.end exam button": "End exam", "lightbox.zoom in on image": "Zoom in on this image", "exam.progress": "Progress", "exam.questions answered": "{{numAnsweredQuestions}} of {{numQuestions}} questions answered.", "result.question marks available": "Marks Available", "result.question answered": "Answered?", "control.confirm end.correct": "You may now end the exam.", "control.confirm end.incorrect": "This is not the expected text.", "control.confirm end.password": "end", "jme.typecheck.for in name wrong type": "The name in a <code>for</code> statement must be a name or list of names, not {{type}}.", "jme.makeFast.no fast definition of function": "The function <code>{{name}}</code> here isn't defined in a way that can be made fast.", "part.show feedback": "Show feedback", "part.hide feedback": "Hide feedback", "part.feedback title": "Feedback for {{name}}.", "part.jme.must-match.warning": "Your answer is not in the expected form: {{-message}}", "part.numberentry.write your answer as a fraction": "Write your answer as a fraction.", "question.nav.label": "Question controls", "question.answer saved": "Answer saved", "question.all answers saved": "All answers saved", "analysis.back to results": "Back to results", "analysis.review": "Review", "analysis.review this": "Review this attempt", "analysis.reviewing attempt": "Reviewing attempt by {{student_name}}.", "part.there is new feedback": "The feedback has changed.", "modal.style.colour scheme": "Colour scheme", "modal.style.automatic colour scheme": "Automatic", "modal.style.light colour scheme": "Light", "modal.style.dark colour scheme": "Dark", "modal.style.custom colour scheme": "Custom", "modal.style.main font": "Main font", "modal.style.font.sans serif": "Sans serif", "modal.style.font.serif": "Serif", "modal.style.font.monospace": "Monospace", "modal.style.spacing": "Spacing", "modal.style.font weight": "Font weight", "modal.style.forced colours warning": "Your browser has overridden the colours used in this page because of a setting such as high contrast mode. Changes to the colour scheme settings here might not have any effect.", "modal.style.text preview": "Most text will look like this.", "modal.style.more options": "More options", "modal.style.main colour": "Main (brand) colour", "modal.style.primary colour": "Primary button colour", "modal.style.link colour": "Link colour", "modal.style.success colour": "Success/correct colour", "modal.style.info colour": "Info colour", "modal.style.warning colour": "Warning colour", "modal.style.danger colour": "Danger/incorrect colour", "modal.style.muted colour": "Muted text colour", "modal.style.highlight colour": "Highlight colour", "exam.error loading exam definition": "There was an error while loading the exam definition: {{text}}", "exam.no exam definition": "No exam definition was given.", "jme.typecheck.wrong arguments for anonymous function": "Wrong number of arguments for this anonymous function.", "worksheet.top": "Top", "worksheet.left": "Left", "worksheet.bottom": "Bottom", "worksheet.right": "Right"}}, "nl-nl": {"translation": {"page.loading": "Bezig met laden ...", "page.saving": "<p>Gegevens worden opgeslagen.</p>\n<p>Dit kan even duren.</p>", "mathjax.math processing error": "MathJax processing error: \"{{-message}}\" when texifying <code>{{expression}}</code>", "die.numbas failed": "Er is een fout opgetreden", "die.sorry": "Sorry, Numbas is een probleem tegengekomen en kan niet verder. Hieronder staat een beschrijving van het probleem.", "die.error": "Probleem", "modal.ok": "OK", "modal.cancel": "Annuleren", "exam.exam name": "Naam toets:", "exam.random seed": "Sessie ID:", "exam.student name": "Naam:", "exam.number of questions": "Aantal vragen:", "exam.marks available": "Punten beschikbaar:", "exam.pass percentage": "Geslaagd bij:", "exam.time allowed": "Beschikbare tijd:", "exam.passed": "Gehaald", "exam.failed": "Niet gehaald", "exam.review header": "Review: ", "frontpage.start": "Start", "suspend.paused header": "Gepauzeerd", "suspend.exam suspended": "De toets is onderbroken. Klik Ga door om verder te gaan.", "suspend.you can resume": "Volgende keer dat je de toets opstart kun je verder gaan met deze sessie.", "suspend.resume": "Ga door", "result.exit": "Exit", "result.print": "Print dit overzicht van de resultaten", "result.exam summary": "Overzicht toets", "result.performance summary": "Overzicht van de resultaten", "result.exam start": "Start toets:", "result.exam stop": "Einde toets:", "result.time spent": "Tijd:", "result.questions attempted": "Aantal gemaakte vragen:", "result.score": "Score:", "result.result": "Resultaat:", "result.question number": "Vraag nummer", "result.question score": "Score", "result.question review title": "Review deze vraag", "result.click a question to review": "Klik op een vraagnummer om je scores en eventuele uitwerkingen te zien.", "end.exam has finished": "De toets is afgelopen. Je kunt het venster sluiten.", "control.confirm leave": "De toets is nog niet af.", "control.not all questions answered": "Je hebt niet alle vragen in deze toets beantwoord.", "control.not all questions submitted": "Je hebt een of meer antwoorden gewijzigd maar ze nog niet verzonden. Controleer of elk antwoord verzonden is.", "control.confirm end": "Weet je zeker dat je de toets wilt afsluiten? Als je afsluit kun je geen antwoorden meer wijzigen.", "control.confirm regen": "Weet je zeker dat je deze vraag opnieuw wilt maken? Als je opnieuw begint verlies je je huidige score.", "control.confirm reveal": "Wil je het antwoord zien? Je verliest je huidige score en kunt de vraag later niet opnieuw beantwoorden.", "control.proceed anyway": "Doorgaan?", "control.regen": "Probeer soortgelijke vraag opnieuw", "control.submit answer": "Verzend", "control.submit all parts": "Verzend alles", "control.submit again": "Verzend opnieuw", "control.submit": "Verzend", "control.previous": "Vorige", "control.next": "Volgende", "control.advice": "Uitwerking", "control.reveal": "Laat zien", "control.total": "Totaal", "control.pause": "Pauze", "control.end exam": "Klaar", "control.back to results": "Ga terug naar resultaten", "control.move to next question": "Move to the next question", "display.part.jme.error making maths": "Fout tijdens weergeven formule", "exam.xml.bad root": "Top element van toets XML moet 'exam' zijn", "exam.changeQuestion.no questions": "Deze toets bevat geen vragen! Controleer het .exam bestand op fouten.", "feedback.you were awarded": "Je hebt <strong>{{count}}</strong> $t(mark) gekregen.", "feedback.taken away": "<strong>{{count,niceNumber}}</strong> $t(mark) $t(was) afgenomen.", "jme.tokenise.invalid": "Ongeldige formule: <code>{{expression}}</code>", "jme.shunt.not enough arguments": "Te weinig argumenten voor operator <code>{{op}}</code>", "jme.shunt.no left bracket in function": "Linker haakje ontbreekt in functieaanroep of rijtje", "jme.shunt.no left square bracket": "Linker haakje ontbreekt", "jme.shunt.no left bracket": "Linker haakje ontbreekt", "jme.shunt.no right bracket": "Rechter haakje ontbreekt", "jme.shunt.no right square bracket": "Rechter blokhaak ontbreekt aan het einde van de lijst", "jme.shunt.missing operator": "Formule kan niet berekend worden -- operator ontbreekt.", "jme.typecheck.function maybe implicit multiplication": "Operator {{name}} is niet gedefinieerd. Bedoel je <br/><code>{{first}}*{{possibleOp}}(...)</code>?", "jme.typecheck.function not defined": "Operator {{op}} is niet gedefinieerd. Bedoel je <br/><code>{{suggestion}}*(...)</code>?", "jme.typecheck.op not defined": "Operator '{{op}}' is niet gedefinieerd.", "jme.typecheck.no right type definition": "Geen definitie van '{{op}}' bekend van het juiste type.", "jme.typecheck.no right type unbound name": "Variabele <code>{{name}}</code> is niet gedefinieerd.", "jme.typecheck.map not on enumerable": "<code>map</code> operator vereist een list of range, niet {{type}}", "jme.evaluate.undefined variable": "Variabele {{name}} is niet gedefinieerd", "jme.thtml.not html": "Passed een niet-HTML waarde in de THML constructor", "jme.func.switch.no default case": "Default case van Switch statement ontbreekt", "jme.func.listval.invalid index": "Ongeldige list index {{index}} voor een list van grootte {{size}}", "jme.func.listval.not a list": "Object is niet indexeerbaar", "jme.func.matrix.invalid row type": "Een matrix kan niet gemaakt worden uit rijen van type {{type}}", "jme.func.except.continuous range": "De 'except' operator kan niet gebruikt worden op continue intervallen.", "jme.matrix.reports bad size": "Matrix afmetingen incorrect - moet fout in constructor functie zijn", "jme.texsubvars.no right bracket": "Geen passend <code>]</code> in {{op}} argumenten.", "jme.texsubvars.missing parameter": "Ontbrekende parameter in {{op}}: {{parameter}}", "jme.texsubvars.no right brace": "Geen passend <code>}</code> in {{op}}", "jme.user javascript.error": "Fout in javascript functie <code>{{name}}</code>: {{-message}}", "jme.variables.error making function": "Fout tijdens genereren functie <code>{{name}}</code>: {{-message}}", "jme.variables.syntax error in function definition": "Syntax error in functie definitie", "jme.variables.variable not defined": "Variabele <code>{{name}}</code> is niet gedefinieerd.", "jme.variables.empty definition": "Definitie van variabele <code>{{name}}</code> is leeg.", "jme.variables.circular reference": "Circulaire verwijzing naar variabele in <code>{{name}}</code>", "jme.variables.error computing dependency": "Fout tijdens het berekenen van variabele <code>{{name}}</code>", "jme.variables.error evaluating variable": "Fout bij berekenen variabele {{name}}: {{-message}}", "jme.variables.question took too many runs to generate variables": "Een geldige set vraag variabelen werd niet gegenereerd in de tijd.", "jme.display.unknown token type": "Kan token van type {{type}} niet naar TeX vertalen", "jme.display.collectRuleset.no sets": "Geen sets opgegeven aan collectRuleset!", "jme.display.collectRuleset.set not defined": "Ruleset {{name}} is niet gedefinieerd", "jme.display.simplifyTree.no scope given": "Numbas.jme.display.simplifyTree vereist een Scope", "math.precround.complex": "Afronden op een complex aantal decimalen is niet mogelijk", "math.siground.complex": "Afronden op een complex aantal significante cijfers is niet mogelijk", "math.combinations.complex": "Combinaties van complexe getallen kunnen niet berekend worden", "math.permutations.complex": "Permutaties van complexe getallen kunnen niet berekend worden", "math.gcf.complex": "De GGD van complexe getallen kan niet berekend worden", "math.lcm.complex": "Het KGV van complexe getallen kan niet berekend worden", "math.lt.order complex numbers": "Complexe getallen kunnen niet geordend worden", "math.choose.empty selection": "Een lege selectie gebruikt voor de random functie", "matrixmath.abs.non-square": "De determinant van een niet-vierkante matrix kan niet berekend worden.", "matrixmath.abs.too big": "Sorry, de determinant van een matrix groter dan 3x3 kan nog niet berekend worden.", "matrixmath.mul.different sizes": "Matrices van verschillende grootte kunnen niet vermenigvuldigd worden.", "vectormath.cross.not 3d": "Uitproduct is alleen te berekenen voor 3D-vectoren.", "vectormath.dot.matrix too big": "Het inproduct is alleen te berekenen voor een matrix met afmetingen $1 \\times N$ of $N \\times 1$.", "vectormath.cross.matrix too big": "Uitproduct is alleen te berekenen voor een matrix met afmetingen $1 \\times N$ of $N \\times 1$.", "part.with steps answer prompt": "Antwoord: ", "part.script.error": "Fout in deel {{path}} custom script {{script}}: {{-message}}", "part.marking.steps no matter": "Omdat je al alle punten voor dit onderdeel hebt leveren de afzonderlijke stappen geen punten meer op.", "part.marking.revealed steps no penalty": "Je hebt de stappen bekeken.", "part.marking.used variable replacements": "Dit deel is gescoord op basis van eerder gegeven antwoorden.", "part.marking.variable replacement part not answered": "Je moet {{part}} eerst beantwoorden", "part.marking.resubmit because of variable replacement": "De score van dit onderdeel hangt af van eerder gegeven antwoorden die je nu hebt gewijzigd. Verzend dit deel opnieuw om je score te updaten.", "part.marking.not submitted": "Geen antwoord verzonden", "part.marking.did not answer": "Je hebt deze vraag niet beantwoord.", "part.marking.nothing entered": "Je hebt geen antwoord ingevuld.", "part.marking.incorrect": "Je antwoord is fout.", "part.marking.correct": "Je antwoord is juist.", "part.marking.uncaught error": "Fout bij scoren van {{part}}: {{-message}}", "part.marking.no result": "Dit onderdeel kan niet gescoord worden.", "part.correct answer": "Goede antwoord:", "part.missing type attribute": "{{part}}: Onderdeeltype ontbreekt", "part.unknown type": "{{part}}: Onderdeeltype {{type}} is onbekend", "part.setting not present": "Property '{{property}}' onbekend", "part.jme.answer missing": "Goede antwoord voor een JME onderdeel ontbreekt (%s)", "part.jme.answer too long": "Je antwoord is te lang.", "part.jme.answer too short": "Je antwoord is te kort.", "part.jme.answer invalid": "Je antwoord is geen geldige formule.<br/>{{-message}}", "part.jme.marking.correct": "Je antwoord heeft de juiste waarde.", "part.jme.must-have bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.must-have one": "In je antwoord moet {{strings}} voorkomen", "part.jme.must-have several": "In je antwoord moet {{string}} allemaal voorkomen", "part.jme.not-allowed bits": "<span class=\"monospace\">{{string}}</span>", "part.jme.not-allowed one": "In je antwoord mag {{strings}} niet voorkomen", "part.jme.not-allowed several": "In je antwoord mag niets uit {{strings}} voorkomen", "part.jme.unexpected variable name": "In je antwoord gebruik je de onbekende variabelenaam <code>{{name}}</code>.", "part.jme.unexpected variable name suggestion": "In je antwoord gebruik je de onbekende variabelenaam <code>{{name}}</code>. Bedoel je <code>{{suggestion}}</code>?", "part.patternmatch.display answer missing": "Display answer ontbreekt in het Pattern Match onderdeel (%s)", "part.patternmatch.correct except case": "Je antwoord is goed, maar je haalt hoofd- en kleine letters door elkaar.", "part.numberentry.correct except decimal": "Je antwoord is goed maar mag geen decimaal getal zijn.", "part.numberentry.correct except fraction": "Je antwoord klopt maar breuken zijn niet toegestaan.", "part.numberentry.answer invalid": "Je hebt geen geldig getal ingevoerd.", "part.numberentry.answer not integer": "Je antwoord is niet correct. Vul een geheel getal in, geen decimaal getal.", "part.numberentry.answer not integer or decimal": "Je antwoord is niet correct. Vul een geheel getal of decimaal getal in.", "part.numberentry.zero sig fig": "Dit deel is zo gemaakt dat een antwoord op 0 significante cijfers wordt afgerond wat zinloos is.", "part.mcq.options def not a list": "De expressie die {{properties}} definieert is geen list.", "part.mcq.marking matrix string empty": "De score-matrix is leeg.", "part.mcq.choices missing": "Antwoordmogelijkheden voor Multiple Response onderdeel (%s) ontbreken", "part.mcq.matrix not a number": "In onderdeel {{part}} specificeert cel {{row}},{{column}} van de scorematrix geen getal", "part.mcq.wrong number of choices": "Je hebt het verkeerde aantal mogelijkheden geselecteerd.", "part.mcq.no choices selected": "Geen keuze gemaakt.", "part.mcq.matrix not a list": "De scorematrix van een Multiple Response onderdeel, gegeven door een JME expressie, moet een list zijn maar is het niet.", "part.mcq.matrix wrong type": "Element heeft een verkeerd type '{{type}}' in de score matrix.", "part.mcq.matrix mix of numbers and lists": "Mix van getallen en lijsten gebruikt in de score matrix.", "part.mcq.matrix wrong size": "Score matrix heeft de verkeerde afmetingen.", "part.mcq.correct choice": "Je hebt het juiste antwoord gekozen.", "part.matrix.answer invalid": "Je antwoord is ongeldig.", "part.matrix.invalid cell": "E\u00e9n of meer van de cellen in antwoord leeg of ongeldig.", "part.matrix.some incorrect": "Een of meer van de cellen in uw antwoord onjuist is, maar je hebt gekregen cijfers voor de rest.", "part.matrix.empty": "U hebt geen antwoord ingevuld", "part.matrix.empty cell": "E\u00e9n of meer van de cellen in antwoord leeg.", "part.matrix.size mismatch": "De vraagauteur heeft de leerling niet toegestaan de afmetingen van zijn antwoord te bepalen, maar het juiste antwoord is {{correct_dimensions}} terwijl de antwoordinvoer {{input_dimensions}} is", "part.gapfill.feedback header": "<strong>Vak {{index}}</strong>", "part.extension.not implemented": "Deel heeft de methode <code>{{name}}</code> niet ge\u00efmplementeerd.", "question.loaded name mismatch": "Kan deze poging niet hervatten - de toets is veranderd sinds de vorige sessie.", "question.error": "Vraag {{number}}: {{-message}}", "question.preamble.error": "Fout in preamble: {{-message}}", "question.preamble.syntax error": "Syntaxisfout in preamble", "question.unsupported part type": "Onbekend onderdeeltype", "question.header": "Vraag {{number}}", "question.submit part": "Verzend antwoord", "question.show steps": "Laat stappen zien", "question.show steps penalty": "Je verliest <strong>{{count}}</strong> $t(mark).", "question.show steps no penalty": "Je score verandert niet.", "question.show steps already penalised": "Je hebt de stappen al bekeken. Je kunt ze nog eens bekijken zonder opnieuw punten te verliezen.", "question.hide steps": "Verberg stappen", "question.hide steps no penalty": "Je score blijft ongewijzigd.", "question.advice": "Uitwerking", "question.no such part": "Onderdeel {{path}} ontbreekt", "question.can not submit": "Antwoord kan niet verzonden worden - controleer op fouten.", "question.answer submitted": "Antwoord verzonden", "question.score feedback.show": "Geef feedback", "question.score feedback.hide": "Verberg", "question.score feedback.answered total actual": "Score: {{score}}/{{marks}}", "question.score feedback.answered total": "{{marksString}}. Beantwoord.", "question.score feedback.answered actual": "Score: {{scoreString}}", "question.score feedback.answered": "Beantwoord.", "question.score feedback.unanswered": "Niet gemaakt.", "question.score feedback.unanswered total": "{{marksString}}.", "question.score feedback.correct": "Je antwoord is goed", "question.score feedback.partial": "Je antwoord is gedeeltelijk goed", "question.score feedback.wrong": "Je antwoord is fout", "question.selector.unsubmitted changes": "Niet verzonden wijzigingen.", "timing.no accumulator": "no timing accumulator {{name}}", "timing.time remaining": "Resterende tijd: %s", "xml.could not load": "Het XML document kan niet geladen worden: {{-message}}", "xml.property not number": "Property {{name}} moet een getal zijn maar is het niet ({{value}}), in node {{element}}", "xml.property not boolean": "Property {{name}} moet een boolean zijn maar is het niet ({{value}}), in node {{element}}", "xml.error in variable definition": "Fout in definitie van variabele <code>{{name}}</code>", "scorm.error initialising": "Fout initialiseren SCORM protocol: {{-message}}", "scorm.failed save": "De gegevens konden niet opgeslagen worden op de server. Je huidige sessie, inclusief antwoorden en scores, is misschien niet bewaard. Je kunt het nog eens proberen. Mail naar <a href=\"mailto:numbas@ncl.ac.uk\">numbas@ncl.ac.uk</a> als dit vaker voorkomt.", "scorm.no exam suspend data": "Hervatten mislukt: geen toetsgegevens gevonden.", "scorm.error loading suspend data": "Fout bij het laden van toetsgegevens: {{-message}}", "scorm.error loading question": "Fout bij het laden van vraag {{number}}: {{-message}}", "scorm.no question suspend data": "Geen vraaggegevens", "scorm.error loading part": "Fout bij laden van deel {{part}}: {{-message}}", "scorm.no part suspend data": "Geen gegevens van dit deel gevonden", "util.product.non list": "Er werd een niet-lijst doorgegeven aan <code>Numbas.util.product</code>", "mark": "punt", "was": "was", "part": "deel", "gap": "gat", "step": "stap", "jme.substituteTree.undefined variable": "Ongedefinieerde variabele: <code>{{name}}</code>", "jme.user javascript.returned undefined": "User-defined javascript function <code>{{name}}</code> returned <code>undefined</code>.", "part.marking.steps change": "Je krijgt <strong>{{count}}</strong> $t(mark) voor je antwoorden bij de verschillende stappen", "part.marking.revealed steps with penalty": "Je hebt de stappen bekeken. Het maximale aantal $t(mark) voor dit onderdeel is <strong>{{count}}</strong>. Je score wordt verlaagd.", "part.marking.total score": "Je krijgt <strong>{{count}}</strong> $t(mark) voor dit onderdeel.", "part.numberentry.precision type.dp": "decimaal", "part.numberentry.precision type.dp_plural": "decimalen", "part.numberentry.precision type.sigfig": "significant cijfer", "part.numberentry.precision type.sigfig_plural": "significante cijfers", "part.numberentry.give your answer to precision": "Rond af op {{count}} {{precisionType}}. ", "question.unsubmitted changes": "Je hebt je antwoord veranderd maar nog niet verzonden. Controleer je antwoord en klik op <strong>Verzend</strong>.", "question.unsubmitted changes_plural": "Je hebt antwoorden gewijzigd maar niet verzonden. Controleer alle antwoorden en klik op <strong>Verzend alles</strong>.", "util.equality not defined for type": "Gelijkheid niet gedefinieerd voor type {{type}}", "mark_plural": "punten", "was_plural": "waren", "die.script not loaded": "Numbas kon niet starten omdat bestand <code>{{file}}</code> niet geladen is. Controleer of het inbegrepen is in <code>scripts.js</code>.", "math.combinations.n less than zero": "Kan geen combinaties berekenen: n is kleiner dan nul", "math.combinations.k less than zero": "Kan geen combinaties berekenen: k is kleiner dan nul", "math.combinations.n less than k": "Kan geen combinaties berekenen: n is kleiner dan k", "math.permutations.n less than zero": "Kan geen permutaties berekenen: n is kleiner dan nul", "math.permutations.k less than zero": "Kan geen permutaties berekenen: k is kleiner dan nul", "math.permutations.n less than k": "Kan geen permutaties berekenen: n is kleiner dan k", "part.numberentry.give your answer to precision_0": "Rond je antwoord af tot op het dichtbij geheel", "mathjax.error": "MathJax verwerkingsprobleem: {{-message}}", "mathjax.error with context": "MathJax verwerkingsprobleem in {{-context}}: {{-message}}", "exam.introduction": "Examenintroductie", "exam.feedback": "Feedbackbericht voor het examen", "jme.tokenise.keypair key not a string": "De key van een dictionary moet een string zijn, niet {{type}}.", "jme.shunt.list mixed argument types": "{{mode}} kan niet worden geparsed: het is een mix van dictionary- en lijst-elementen", "jme.func.listval.key not in dict": "De dictionary bevat de key <code>{{key}}</code> niet", "part.prompt": "Vraagstelling", "part.feedback": "feedback", "part.numberentry.answer not reduced": "Je antwoord is nog niet onvereenvoudigbaar.", "part.numberentry.give your answer as a reduced fraction": "Vereenvoudig je antwoord. ", "part.numberentry.negative decimal places": "Dit deel is ingesteld zodat de antwoorden van studenten worden afgerond tot op een negatief aantal decimale plaatsen, wat geen betekenis heeft.", "part.mcq.choices": "keuzes", "part.mcq.answers": "antwoorden", "part.mcq.matrix cell empty": "Cell ({{row}},{{column}}) van de scorematrix van deel {{part}} is leeg", "part.mcq.matrix jme error": "Cell ({{row}},{{column}}) van de scorematrix van deel {{part}} geeft een JME error: {{-error}}", "question.statement": "Statement", "ruleset.circular reference": "Circulaire referentie in de definitie van ruleset <code>{{name}}</code>", "ruleset.set not defined": "Ruleset {{name}} is niet gedefinieerd", "jme.evaluate.no scope given": "Numbas.jme.evaluate moet een Scope meekrijgen", "answer.number.not a number": "Jouw antwoord is geen geldig getal.", "answer.number.fractions not allowed": "Je mag geen breuk ingeven.", "answer.jme.invalid expression": "{{-message}}", "answer.matrix.fractions not allowed": "Je mag geen breuken ingeven.", "answer.matrix.some cell not a number": "Een of meer cellen in jouw antwoord is geen geldig getal.", "exam.enter password": "Paswoord:", "exam.password.correct": "Het paswoord is correct. Je kan de toets starten.", "exam.password.incorrect": "Het paswoord is incorrect.", "frontpage.scorm.lms not connected": "Deze toets wordt in de zelfstandige modus uitgevoerd. Jouw antwoorden en cijfers worden niet opgeslagen!", "result.question review": "Review", "control.confirm regen no marks": "Weet je zeker dat je deze vraag opnieuw wilt maken?", "control.confirm reveal no marks": "Wil je het antwoord op deze vraag zien?", "jme.tokenise.invalid near": "Ongeldige uitdrukking: <code>{{expression}}</code> op positie {{position}} bij <code>{{nearby}}</code>", "jme.tokenise.number.object not complex": "Ongeldig object meegegeven aan de number constructor.", "jme.subvars.null substitution": "Substitutie van een lege variabele: <code>$t(left brace){{str}}$t(right brace)", "jme.type.type already registered": "Het datatype {{type}} is reeds geregistreerd en kan niet opnieuw worden geregistreerd.", "jme.type.no cast method": "Kan niet automatisch converteren van {{from}} naar {{to}}.", "jme.display.simplifyTree.empty expression": "Uitdrukking is leeg", "jme.display.simplifyTree.stuck in a loop": "De simplifier zit vast in een lus: <code>{{expr}}</code>", "math.niceNumber.undefined": "Verwachtte een getal, maar kreeg <code>undefined</code>", "math.rangeToList.zero step size": "Een range met stapgrootte 0 kan niet naar een lijst worden omgezet.", "part.error": "{{path}}: {{-message}}", "part.marking.revealed steps": "Je liet de stappen zien.", "part.marking.maximum scaled down": "De maximale score voor dit deel is <strong>{{count,niceNumber}}</strong> $t(mark). Je scores zullen overeenkomstig geschaald worden.", "part.marking.minimum score applied": "De minimale score voor dit deel is <strong>{{score,niceNumber}}</strong>.", "part.marking.maximum score applied": "De maximale score voor dit deel is <strong>{{score,niceNumber}}</strong>.", "part.marking.error in marking script": "Er was een probleem in het score-algoritme van dit deel. Meld dit aub. {{-message}}", "part.marking.no result after replacement": "Het is onmogelijk om punten toe te kennen aan dit deel gebruikmakend van je antwoorden op vorige delen.", "part.marking.missing required note": "Het score-algoritme definieert de note <code>{{note}}</code> niet", "marking.apply.not a list": "Het eerste argument van <code>apply</code> moet een lijst zijn, en dat is niet zo", "marking.apply marking script.script not found": "Marking script <code>{{name}}</code> niet gevonden", "marking.note.compilation error": "Error bij het compileren van note <code>{{name}}</code> - {{-message}}", "marking.note.error evaluating note": "Error bij het evalueren van note <code>{{name}}</code> - {{-message}}", "marking.note.invalid definition": "Ongeldige note-definitie: <code>{{source}}</code>. {{-hint}}", "marking.note.invalid definition.missing colon": "Mogelijk mis je een dubbelpunt naar de naam en beschrijving", "marking.note.invalid definition.description missing closing bracket": "Mogelijk mis je een sluitingshaakje", "marking.note.empty expression": "De note <code>{{name}}</code> is leeg.", "marking.script.error parsing notes": "Probleem bij het parsen van het scorescript: {{- message}}", "part.feedback out of date": "Deze feedback is gebaseerd op jouw laatst ingediende antwoord. Dien jouw gewijzigde antwoord in om bijgewerkte feedback te krijgen.", "part.jme.invalid value generator expression": "Ongeldige waardegeneratoruitdrukking voor variabele <code>{{name}}</code>: {{-message}}", "part.mcq.incorrect choice": "Je koos een incorrect antwoord.", "part.matrix.not all cells same precision": "Niet alle cellen in je antwoord heb je met dezelfde precisie ingegeven.", "part.gapfill.error marking gap": "Probleem bij het scoren van {{name}}: {{-message}}", "part.custom.empty setting": "Geen waarde gegeven.", "part.custom.unrecognised input type": "Onbekend instellingtype <code>{{input_type}}</code>", "part.custom.error evaluating input option": "Probleem bij het evalueren van inputoptie <code>{{option}}</code>: {{-error}}", "part.custom.input option missing": "De definitie van inputoptie <code>{{option}}</code> mist.", "part.custom.error evaluating setting": "Probleem bij het evalueren van instelling <code>{{setting}}</code>: {{-error}}", "question.error creating question": "Probleem bij het maken van vraag {{number}}: {{-message}}", "question.score feedback.not marked": "Nog niet gescoord", "question.score feedback.partially answered": "Gedeeltelijk beantwoord", "question.score feedback.score total": "{{marksString}}", "question.score feedback.score actual": "Score: {{scoreString}}", "question.score feedback.score total actual": "Score: {{score,niceNumber}}/{{marks,niceNumber}}", "variable.error in variable definition": "Probleem in de definite van variabele <code>{{name}}</code>", "left brace": "{", "right brace": "}", "extension.not found": "Kon de extensie <code>{{name}}</code> niet laden.", "control.toggle navigation menu": "Toggle het navigatiemenu", "part.input title": "Antwoord voor deel {{name}}", "part.correct answer title": "Verwachte antwoord voor deel {{name}}", "part.jme.must-match.failed": "Jouw antwoord is niet in de juiste vorm.", "question.score feedback.none": "", "control.submit part.confirm remove next parts": "<p>Een of meer volgende onderdelen zijn afhankelijk van jouw antwoord op dit onderdeel. Door dit onderdeel opnieuw in te dienen, worden die onderdelen ongeldig en worden ze uit de vraag verwijderd. Dit kan niet ongedaan gemaakt worden.</p>\n<p>Wil je dit onderdeel opnieuw indienen?</p>", "control.back to menu": "Ga terug naar het menu", "display.error making html": "Probleem bij het maken van HTML in {{contextDescription}}: {{-message}}", "jme.subvars.error compiling": "{{-message}} in <code>{{expression}}</code>", "jme.variables.empty name": "Er is een vraagvariabele waaraan geen naam is gegeven.", "jme.calculus.unknown derivative": "Weet niet hoe <code>{{tree}}</code> kan worden afgeleid", "math.order complex numbers": "Complexe getallen kunnen niet worden geordend", "menu.choose a question": "Kies een vraag.", "part.choose next part.answered": "Wat wil je hierna doen?", "part.choose next part.unanswered": "Of, je kan:", "part.choose next part.will be locked": "(Dit deel zal niet toegankelijk zijn)", "part.reached dead end": "Er is niets meer om te doen vanaf hier.", "part.next part.penalty amount": "(verlies {{count}} $t(mark))", "part.marking.counts towards objective": "Dit deel telt mee voor het doel <strong>\u201c{{objective}}\u201d</strong>.", "part.numberentry.answer not integer or decimal or fraction": "Je antwoord is ongeldig. Je moet een geheel getal, kommagetal of breuk ingeven.", "question": "Vraag", "question.progress": "Vraagvooruitgang:", "question.score feedback.unattempted": "Niet geprobeerd", "question.score feedback.attempted": "Geprobeerd", "question.score feedback.score actual.plain": "{{scoreString}}", "question.score feedback.score total actual.plain": "{{score,niceNumber}}/{{marks,niceNumber}}", "question.objectives": "Doelen", "question.penalties": "Bestraffingen", "question.back to previous part": "Ga terug naar het vorige onderdeel", "end.print": "Print het afschrift van je examen", "math.shuffle_together.lists not all the same length": "Niet alle lijsten hebben dezelfde lengte.", "jme.parse signature.invalid signature string": "Ongeldige functiesignatuur: {{str}}", "part.custom.expected answer has wrong type": "Het verwachte antwoord voor dit deel heeft het verkeerde type. Het zou <code>{{shouldbe}}</code> moeten zijn.", "part.custom.input option has wrong type": "De antwoordinvoerinstelling <code>{{option}}</code> heeft het verkeerde type. Het zou <code>{{shouldbe}}</code> moeten zijn.", "matrix input.size control legend": "Grootte", "matrix input.rows": "Rijen", "matrix input.columns": "Kolommen", "part.jme.error checking numerically": "Er was een probleem bij het numeriek controleren van je antwoord: {{message}}", "part.gapfill.cyclic adaptive marking": "Er is een lus in de adaptieve scoring van dit deel: <strong>{{name1}}</strong> berust op <strong>{{name2}}</strong>, die uiteindelijk berust op <strong>{{name1}}</strong>.", "modal.style.background colour": "Achtergrondkleur", "modal.style.text colour": "Tekstkleur", "modal.style.text size": "Tekstgrootte", "modal.style.explanation": "Gebruik dit om het uitzicht van het examen te veranderen.", "modal.style.reset to defaults": "Herstel standaardwaarden", "modal.style.text size preview": "De meeste tekst zal deze grootte hebben.", "control.style options": "Weergave-opties", "part.marking.partially correct": "Je antwoord is gedeeltelijk juist.", "part.marking.error in adaptive marking": "Er was een probleem in de adaptieve scoring voor dit deel. Meld dit aub. {{-message}}", "page.skip to content": "Ga door naar inhoud", "result.learning objective": "Leerdoel", "jme.interpreted as": "Ge\u00efnterpreteerd als", "jme.script.note.compilation error": "Error bij het compileren van note <code>{{name}}</code>: {{-message}}", "jme.script.note.error evaluating note": "Error bij het evalueren van note <code>{{name}}</code>: {{-message}}", "jme.script.note.invalid definition": "Ongeldige note definitie: <code>{{source}}</code>. {{-hint}}", "jme.script.note.invalid definition.missing colon": "Mogelijk mis je een dubbelpunt na de naam en beschrijving", "jme.script.note.invalid definition.description missing closing bracket": "Mogelijk mis je een sluitingshaakje", "jme.script.note.empty expression": "De note <code>{{name}}</code> is leeg.", "jme.script.error parsing notes": "Error bij het parsen van het scorescript: {{- message}}", "matrix input.cell label": "Rij {{row}}, kolom {{column}}", "diagnostic.use retry": "Use one retry and try this topic again.", "diagnostic.move to next topic": "Move on to the next topic.", "diagnostic.next step question": "What would you like to do next?", "diagnostic.now assessing topic": "Now assessing {{current_topic}}", "diagnostic.one retry left": "You have 1 retry left", "diagnostic.retries left": "You have {{retries}} retries left.", "diagnostic.percentage completed": "You've completed <strong>{{percentage}}%</strong> of the test.", "diagnostic.test is over": "The test is over.", "diagnostic.passed all lo": "You have passed all learning objectives.", "diagnostic.more work on lo": "You need to do some more work on the following learning objectives: {{los}}.", "diagnostic.move to next question in topic": "Move on to the next question in topic.", "diagnostic.complete": "Complete!", "diagnostic.studying topic": "Studying {{topic}}", "display.answer widget.unknown widget type": "The answer widget type <code>{{name}}</code> is not recognised.", "jme.shunt.expected argument before comma": "Expected to see something between the opening bracket and the comma", "part.waiting for pre submit": "Your answer is being marked. Please wait.", "diagnostic.end test": "End the test.", "page.no stylesheet loaded": "The page's stylesheet file has not loaded.", "modal.confirm": "Confirm", "modal.alert": "Alert", "suspend.resumed header": "Attempt resumed", "jme.vector.value not an array of numbers": "Tried to construct a vector using a value that is not an array of numbers.", "jme.matrix.value not the right type": "Tried to construct a vector using a value of the wrong type.", "jme.subvars.html inserted twice": "An HTML value has been embedded twice. Consider defining a function to generate a new value each time it is used.", "jme.variables.invalid function language": "The language <code>{{language}}</code> is not valid.", "jme.variables.duplicate definition": "There is more than one definition of the variable <code>{{name}}</code>.", "math.random_integer_partition.invalid k": "The size of the partition must be between 1 and {{n}}.", "part.marking.parameter already in scope": "There is a variable named <code>{{name}}</code>, which is also the name of a marking parameter. Please rename the variable.", "part.marking.adaptive variable replacement refers to self": "This part refers to itself in a variable replacement for adaptive marking.", "part.marking.adaptive variable replacement refers to nothing": "This part contains an invalid variable replacement for adaptive marking.", "part.numberentry.display answer wrong type": "The display answer for this part is a value of type <code>{{got_type}}</code>, but should be a <code>{{want_type}}</code>.", "part.matrix.invalid type in prefilled": "There is an invalid value of type <code>{{n}}</code> in the array of pre-filled cells.", "diagnostic.make a choice": "Make a choice", "matrixmath.not square": "This operation only works on a square matrix.", "matrixmath.not invertible": "This operation only works on an invertible matrix.", "matrixmath.gauss-jordan elimination.not enough columns": "There must be at least as many columns as rows.", "question.required extension not available": "This question requires the extension <code>{{-extension}}</code> but it is not available.", "util.formatNumberNotation.unrecognised syntax": "The number formatting syntax <code>{{syntax}}</code> is not recognised.", "worksheet.number of exams to generate": "Number of sheets to generate", "worksheet.starting with id": "Starting with ID", "worksheet.show exam id": "Show sheet ID?", "worksheet.page break between questions": "Page breaks between questions?", "worksheet.page margins": "Page margins (mm)", "worksheet.text size": "Text size (pt)", "worksheet.generate": "Generate", "worksheet.generating exams": "Generating sheets", "worksheet.sheet id": "Sheet ID:", "worksheet.print single": "Print this sheet", "worksheet.print several": "Print these sheets", "worksheet.answer sheets": "Answer sheets", "worksheet.question sheets": "Question sheets", "worksheet.reconfigure": "Generate different sheets", "worksheet.show sheet": "Preview the sheet with ID:", "accessibility statement": "Accessibility statement and guide to adapting Numbas to your needs.", "exam.enter your name": "Your name:", "exam.attempt download security warning": "This exam is configured to allow you to download your data, but it is not running in a secure browser context. You will not be able to download your exam data. Contact your lecturer or teacher for help.", "result.download exam object": "Download your exam data", "control.return to question": "Return to the question", "control.show introduction": "Introduction", "analysis.header": "Analyse attempt data", "analysis.help.upload files": "Upload attempt data files that your students have given you.", "analysis.help.file input label": "Choose student attempt data files, or drag files onto this window.", "analysis.table.total": "Exam totals", "analysis.table.question": "Exam and question totals", "analysis.table.all": "All details", "analysis.student name.anonymous": "No name given", "analysis.expected": "Expected results", "analysis.start time": "Start time", "analysis.maximum": "Maximum Marks", "analysis.file": "File", "analysis.download": "Download", "analysis.delete": "Delete", "analysis.view results": "View results", "analysis.upload files": "Upload files", "analysis.upload more": "Upload more files", "analysis.attempt data": "Attempt data", "analysis.select format": "Select data to show", "analysis.download this table": "Download this table", "analysis.student": "Student Results", "analysis.question key": "Question Key", "analysis.question name": "Question Name", "analysis.group": "Group", "analysis.question": "Question", "analysis.part": "Part", "analysis.gap": "Gap", "analysis.record type": "Record Type", "analysis.score": "Score", "analysis.marks available": "Marks Available", "analysis.percentage": "Percentage", "analysis.answer": "Answer", "analysis.student name": "Student Name", "analysis.summary.no files": "You have not uploaded any files yet.", "analysis.summary.no decrypted files": "You have not uploaded any valid files yet.", "analysis.summary.one file": "One attempt.", "analysis.summary.several files": "{{num_files,niceNumber}} attempts.", "analysis.not secure context": "This page must be opened in a secure browser context. A secure context is either a page served over HTTPS, or a file loaded from your device.", "jme.shunt.pipe right hand takes no arguments": "The expression on the right-hand side of the pipe operator must be a function application.", "question.explore.no parts defined": "There are no parts defined in this question.", "answer": "answer", "worksheet.answersheet show question content": "Show question content in answer sheets?", "modal.confirm end exam": "Write <code>{{endConfirmation}}</code> in the box to confirm:", "modal.end exam button": "End exam", "lightbox.zoom in on image": "Zoom in on this image", "exam.progress": "Progress", "exam.questions answered": "{{numAnsweredQuestions}} of {{numQuestions}} questions answered.", "result.question marks available": "Marks Available", "result.question answered": "Answered?", "control.confirm end.correct": "You may now end the exam.", "control.confirm end.incorrect": "This is not the expected text.", "control.confirm end.password": "end", "jme.typecheck.for in name wrong type": "The name in a <code>for</code> statement must be a name or list of names, not {{type}}.", "jme.makeFast.no fast definition of function": "The function <code>{{name}}</code> here isn't defined in a way that can be made fast.", "part.show feedback": "Show feedback", "part.hide feedback": "Hide feedback", "part.feedback title": "Feedback for {{name}}.", "part.jme.must-match.warning": "Your answer is not in the expected form: {{-message}}", "part.numberentry.write your answer as a fraction": "Write your answer as a fraction.", "question.nav.label": "Question controls", "question.answer saved": "Answer saved", "question.all answers saved": "All answers saved", "analysis.back to results": "Back to results", "analysis.review": "Review", "analysis.review this": "Review this attempt", "analysis.reviewing attempt": "Reviewing attempt by {{student_name}}.", "part.there is new feedback": "The feedback has changed.", "modal.style.colour scheme": "Colour scheme", "modal.style.automatic colour scheme": "Automatic", "modal.style.light colour scheme": "Light", "modal.style.dark colour scheme": "Dark", "modal.style.custom colour scheme": "Custom", "modal.style.main font": "Main font", "modal.style.font.sans serif": "Sans serif", "modal.style.font.serif": "Serif", "modal.style.font.monospace": "Monospace", "modal.style.spacing": "Spacing", "modal.style.font weight": "Font weight", "modal.style.forced colours warning": "Your browser has overridden the colours used in this page because of a setting such as high contrast mode. Changes to the colour scheme settings here might not have any effect.", "modal.style.text preview": "Most text will look like this.", "modal.style.more options": "More options", "modal.style.main colour": "Main (brand) colour", "modal.style.primary colour": "Primary button colour", "modal.style.link colour": "Link colour", "modal.style.success colour": "Success/correct colour", "modal.style.info colour": "Info colour", "modal.style.warning colour": "Warning colour", "modal.style.danger colour": "Danger/incorrect colour", "modal.style.muted colour": "Muted text colour", "modal.style.highlight colour": "Highlight colour", "exam.error loading exam definition": "There was an error while loading the exam definition: {{text}}", "exam.no exam definition": "No exam definition was given.", "jme.typecheck.wrong arguments for anonymous function": "Wrong number of arguments for this anonymous function.", "worksheet.top": "Top", "worksheet.left": "Left", "worksheet.bottom": "Bottom", "worksheet.right": "Right"}}}
        }
        });
        ;
Numbas.queueScript('marking_scripts', ['marking'], function() {
            Numbas.raw_marking_scripts = {"jme": "expand_juxtapositions_settings (Settings for the \"expand juxtapositions\" step):\n    [\n        \"singleLetterVariables\": settings[\"singleLetterVariables\"],\n        \"noUnknownFunctions\": not settings[\"allowUnknownFunctions\"],\n        \"implicitFunctionComposition\": settings[\"implicitFunctionComposition\"],\n        \"normaliseSubscripts\": true\n    ]\n\nstudentExpr_empty (If the student's answer is empty, don't try to mark it):\n    assert(trim(studentAnswer)<>\"\" and parse(studentAnswer)<>parse(\"\"),\n        warn(translate(\"part.marking.nothing entered\"));\n        fail(translate(\"part.marking.nothing entered\"))\n    )\n\nstudentExpr (The student's answer, parsed):\n    apply(studentExpr_empty);\n    try(\n        simplify(\n            expand_juxtapositions(parse(studentAnswer), expand_juxtapositions_settings),\n            'basic'\n        )\n    , message,\n        warn(translate(\"part.jme.answer invalid\",[\"message\":message]));\n        fail(translate(\"part.jme.answer invalid\",[\"message\":message]));\n        nothing\n    )\n\ncleanedStudentString (The student's answer as a string, cleaned up): string(studentExpr)\n\nscope_vars (Variables already defined in the scope):\n    definedvariables()\n\ncorrectExpr (The correct answer, parsed):\n    expand_juxtapositions(parse(settings[\"correctAnswer\"]), expand_juxtapositions_settings)\n\nstudentMatch (The result of matching the student's expression against the pattern):\n    scope_case_sensitive(match(studentExpr,settings[\"mustMatchPattern\"]),settings[\"caseSensitive\"])\n\ncorrectMatch (The result of matching the correct answer against the pattern):\n    scope_case_sensitive(match(correctExpr,settings[\"mustMatchPattern\"]),settings[\"caseSensitive\"])\n\ncompareName (The name of the matched group from each expression to compare): \n    settings[\"nameToCompare\"]\n\nformula_replacement_pattern: \"$v;lhs = ?;rhs\"\n\nformula_replacement:\n    if(is_formula,\n        string(\n            substitute(settings, expression(\"resultsequal(lhs, rhs, checkingType, checkingAccuracy)\"))\n        )\n    ,\n        \"lhs = rhs\"\n    )\n\nstudentCompare (The part of the student's expression to compare):\n    if(settings[\"mustMatchPattern\"]=\"\" or compareName=\"\",\n        replace(formula_replacement_pattern, formula_replacement, studentExpr)\n    ,\n        studentMatch[\"groups\"][compareName]\n    )\n\ncorrectCompare (The part of the correct expression to compare):\n    if(settings[\"mustMatchPattern\"]=\"\" or compareName=\"\",\n        replace(formula_replacement_pattern, formula_replacement, correctExpr)\n    ,\n        correctMatch[\"groups\"][compareName]\n    )\n\nfailNameToCompare (If comparing just a subexpression, stop marking if the student's expression doesn't have that subexpression):\n    assert(settings[\"mustMatchPattern\"]=\"\" or compareName=\"\" or (studentMatch[\"match\"] and compareName in studentMatch[\"groups\"]),\n        incorrect(settings[\"mustMatchMessage\"]);\n        end()\n    )\n\nstudentVariables (Variables used in the student's answer): \n    scope_case_sensitive(set(findvars(studentCompare)),settings[\"caseSensitive\"])\n\ncorrectVariables (Variables used in the correct answer):\n    scope_case_sensitive(set(findvars(correctCompare)),settings[\"caseSensitive\"])\n\nunexpectedVariables (Unexpected variables used in the student's answer):\n    let(uvars, filter(not (x in correctVariables),x,list(studentVariables)),\n        assert(not settings[\"checkVariableNames\"] or len(uvars)=0,\n            warn(translate(\"part.jme.unexpected variable name\",[\"name\":uvars[0]]));\n            feedback(translate(\"part.jme.unexpected variable name\",[\"name\":uvars[0]]))\n        );\n        uvars\n    )\n\nfailMinLength (Is the student's answer too short?):\n    assert(settings[\"minLength\"]=0 or len(cleanedStudentString)>=settings[\"minLength\"],\n        multiply_credit(settings[\"minLengthPC\"],settings[\"minLengthMessage\"]);\n        true\n    )\n\nfailMaxLength:\n    assert(settings[\"maxLength\"]=0 or len(cleanedStudentString)<=settings[\"maxLength\"],\n        multiply_credit(settings[\"maxLengthPC\"],settings[\"maxLengthMessage\"]);\n        true\n    )\n\nforbiddenStrings:\n    filter(x in cleanedStudentString, x, settings[\"notAllowed\"])\n\nforbiddenStringsPenalty:\n    assert(len(forbiddenStrings)=0,\n        translate(\n          if(len(settings[\"notAllowed\"])=1, 'part.jme.not-allowed one', 'part.jme.not-allowed several'),\n          [\"strings\":map(translate('part.jme.not-allowed bits',[\"string\":str]),str,forbiddenStrings)]\n        );\n        multiply_credit(settings[\"notAllowedPC\"],settings[\"notAllowedMessage\"]);\n        warn(settings[\"notAllowedMessage\"])\n    )\n\nrequiredStrings:\n    filter(not (x in cleanedStudentString), x, settings[\"mustHave\"])\n\nrequiredStringsPenalty:\n    assert(len(requiredStrings)=0,\n        translate(\n          if(len(settings[\"mustHave\"])=1, 'part.jme.must-have one', 'part.jme.must-have several'),\n          [\"strings\":map(translate('part.jme.must-have bits',[\"string\":str]),str,forbiddenStrings)]\n        );\n        multiply_credit(settings[\"mustHavePC\"],settings[\"mustHaveMessage\"]);\n        warn(settings[\"mustHaveMessage\"])\n    )\n\nvRange (The range to pick variable values from): \n    settings[\"vsetRangeStart\"]..settings[\"vsetRangeEnd\"] # 0\n\nanswerVariables (Variables used in either the correct answer or the student's answer):\n    correctVariables or studentVariables\n\nformula_match:\n  scope_case_sensitive(match(correctExpr,\"$v;lhs = ?;rhs\"),settings[\"caseSensitive\"])\n\nis_formula (Is the correct answer a formula of the form name = expression?):\n  formula_match[\"match\"]\n\nformula_variable (The variable on the left-hand side of the formula, if the correct answer is a formula):\n  try(string(formula_match[\"groups\"][\"lhs\"]),err,\"\")\n\nformula_expression (The right-hand side of the formula, if the correct answer is a formula):\n  formula_match[\"groups\"][\"rhs\"]\n\nformula_type (The type of value the formula produces, if the correct answer is a formula):\n  let(t,scope_case_sensitive(infer_type(formula_expression),settings[\"caseSensitive\"]),\n    if(t=\"name\",\"number\",t)\n  )\n\nvalue_generator_definitions:\n    dict([normalise_subscripts(k), v] for: [k,v] of: items(settings[\"valueGenerators\"]))\n\nvalue_generators (Expressions which generate values for each variable in the answer):\n    dict(map(\n        [\n          name,\n          get(\n            value_generator_definitions,\n            name,\n            if(is_formula and name=formula_variable,\n              exec(function(\"random\"),[formula_expression,default_value_generator[formula_type]])\n            ,\n              default_value_generator[get(variable_types,name,\"number\")]\n            )\n          )\n        ],\n        name,\n        answerVariables\n    ))\n\nvariable_types (Inferred types for each of the variables):\n    scope_case_sensitive(infer_variable_types(correctExpr),settings[\"caseSensitive\"])\n\ndefault_value_generator:\n    [\n        \"number\": expression(\"random(vRange)\"),\n        \"decimal\": expression(\"dec(random(vRange))\"),\n        \"integer\": expression(\"int(random(vRange))\"),\n        \"rational\": expression(\"rational(random(vRange))\"),\n        \"matrix\": expression(\"matrix(repeat(repeat(random(vRange),3),3))\"),\n        \"vector\": expression(\"vector(repeat(random(vRange),3))\"),\n        \"boolean\": expression(\"random(true,false)\"),\n        \"set\": expression(\"set(repeat(random(vRange),5))\")\n    ]\n\nvset (The sets of variable values to test against):\n    try(\n        repeat(\n            scope_case_sensitive(make_variables(value_generators,vRange),settings[\"caseSensitive\"]),\n            settings[\"vsetRangePoints\"]\n        ),\n        message,\n        warn(translate(\"part.jme.error checking numerically\",[\"message\":message]));\n        fail(translate(\"part.jme.error checking numerically\",[\"message\":message]));\n        []\n    )\n\nagree (Do the student's answer and the expected answer agree on each of the sets of variable values?):\n    apply(vset);\n    map(\n        try(\n            scope_case_sensitive(resultsequal(unset(question_definitions,eval(studentCompare,vars)),unset(question_definitions,eval(correctCompare,vars)),settings[\"checkingType\"],settings[\"checkingAccuracy\"]),settings[\"caseSensitive\"]),\n            message,\n            warn(translate(\"part.jme.answer invalid\",[\"message\":message]));\n            fail(translate(\"part.jme.answer invalid\",[\"message\":message]));\n            false\n        ),\n        vars,\n        vset\n    )\n\nnumFails (The number of times the student's answer and the expected answer disagree):\n    apply(agree);\n    len(filter(not x,x,agree))\n\nnumericallyCorrect (Is the student's answer numerically correct?):\n    apply(numFails);\n    if(numFails<settings[\"failureRate\"],\n        correct(translate(\"part.jme.marking.correct\"))\n    ,\n        incorrect()\n    )\n\nsameVars (Does the student use the same variables as the correct answer?):\n    // Removed, but still defined so that older questions with custom marking algorithms don't break\n    nothing\n\nstudentMatches (Does the student's answer match the required pattern?):\n    matches(studentExpr,settings[\"mustMatchPattern\"])\n\nmustMatchMessage:\n    if(settings[\"mustMatchMessage\"]=\"\",\n        translate(\"part.jme.must-match.failed\")\n    ,\n        translate(\"part.jme.must-match.warning\", [\"message\": settings[\"mustMatchMessage\"]])\n    )\n\nfailMatchPatternPrevent (Prevent submission if the student's answer doesn't match the required pattern):\n    assert(settings[\"mustMatchWarningTime\"]<>\"prevent\" or studentMatches,\n        warn(mustMatchMessage);\n        fail(mustMatchMessage)\n    )\n\nfailMatchPattern (Give feedback if the student's answer doesn't match the required pattern):\n    assert(settings[\"mustMatchPattern\"]=\"\" or studentMatches,\n        assert(settings[\"mustMatchWarningTime\"]<>\"input\",\n            warn(mustMatchMessage)\n        );\n        if(compareName=\"\",\n            multiply_credit(settings[\"mustMatchPC\"], mustMatchMessage)\n        ,\n            set_credit(0,mustMatchMessage)\n        );\n        true\n    )\n\nmark:\n    apply(studentExpr);\n    apply(failNameToCompare);\n    apply(unexpectedVariables);\n    apply(sameVars);\n    apply(failMatchPatternPrevent);\n    apply(numericallyCorrect);\n    apply(failMinLength);\n    apply(failMaxLength);\n    apply(forbiddenStringsPenalty);\n    apply(requiredStringsPenalty);\n    apply(failMatchPattern)\n\ninterpreted_answer (The student's answer, to be reused by other parts):\n    apply(studentExpr);\n    studentExpr\n\n", "gapfill": "marked_original_order (Mark the gaps in the original order, mainly to establish if every gap has a valid answer):\n    map(\n        mark_part(gap[\"path\"],studentAnswer),\n        [gap,studentAnswer],\n        zip(gaps,studentAnswer)\n    )\n\ninterpreted_answers (The interpreted answers for each gap, in the original order):\n    map(\n        res[\"values\"][\"interpreted_answer\"],\n        res,\n        marked_original_order\n    )\n\nanswers (The student's answers to each gap):\n    if(settings[\"sortAnswers\"],\n        sort(interpreted_answers)\n    ,\n        interpreted_answers\n    )\n\ngap_order:\n    if(settings[\"sortAnswers\"],\n        sort_destinations(interpreted_answers)\n    ,\n        gap_adaptive_order\n    )\n\nanswer_order:\n    if(settings[\"sortAnswers\"],\n        0..(len(studentAnswer)-1)\n    ,\n        gap_adaptive_order\n    )\n\ngap_feedback (Feedback on each of the gaps):\n    map(\n        try(\n            let(\n                answer, studentAnswer[answer_number],\n                result, submit_part(gaps[gap_number][\"path\"],answer),\n                gap, gaps[gap_number],\n                name, gap[\"name\"],\n                noFeedbackIcon, not gap[\"settings\"][\"showFeedbackIcon\"],\n                non_warning_feedback, filter(x[\"op\"]<>\"warning\",x,result[\"feedback\"]),\n                    assert(noFeedbackIcon,\n                        assert(name=\"\" or len(gaps)=1 or len(non_warning_feedback)=0,feedback(translate('part.gapfill.feedback header',[\"name\": name])))\n                    );\n                    concat_feedback(non_warning_feedback, if(marks>0,result[\"marks\"]/marks,1/len(gaps)), noFeedbackIcon);\n                    result\n            ),\n            err,\n            fail(translate(\"part.gapfill.error marking gap\",[\"name\": gaps[gap_number][\"name\"], \"message\": err]))\n        ),\n        [gap_number,answer_number],\n        zip(gap_order,answer_order)\n    )\n\nall_valid (Are the answers to all of the gaps valid?):\n  all(map(res[\"valid\"],res,marked_original_order))\n\nmark:\n  assert(all_valid or not settings[\"sortAnswers\"], fail(translate(\"question.can not submit\")));\n  apply(answers);\n  apply(gap_feedback)\n\ninterpreted_answer:\n  answers\n\npre_submit:\n    map(\n        let(\n            answer, studentAnswer[answer_number],\n            result, submit_part(gaps[gap_number][\"path\"],answer),\n            check_pre_submit(gaps[gap_number][\"path\"], answer, exec_path)\n        ),\n        [gap_number,answer_number],\n        zip(gap_order,answer_order)\n    )\n", "patternmatch": "regex_match (Match the student's answer with the correct answer, interpreted as a regular expression):\n  match_regex(settings[\"correctAnswer\"],studentAnswer,\"u\")\n\nregex_match_case_insensitive (Match the student's answer with the correct answer, interpreted as a case-insensitive regular expression):\n  match_regex(settings[\"correctAnswer\"],studentAnswer,\"iu\")\n\nexact_match (Is the student's answer exactly the correct answer?):\n  studentAnswer=settings[\"correctAnswer\"]\n\nexact_match_case_insensitive (Is the student's answer exactly the correct answer?):\n  lower(studentAnswer)=lower(settings[\"correctAnswer\"])\n\nmatches (Does the student's answer match the correct answer?):\n  switch(\n    settings[\"matchMode\"]=\"regex\", len(regex_match)>0,\n    settings[\"matchMode\"]=\"exact\", exact_match,\n    false\n  )\n      \nmatches_case_insensitive (Does the student's answer match the correct answer, ignoring case?):\n  switch(\n    settings[\"matchMode\"]=\"regex\", len(regex_match_case_insensitive)>0,\n    settings[\"matchMode\"]=\"exact\", exact_match_case_insensitive,\n    false\n  )\n     \n\nmark:\n  assert(settings[\"allowEmpty\"] or len(studentAnswer)>0,\n    warn(translate(\"part.marking.nothing entered\"));\n    fail(translate(\"part.marking.nothing entered\"))\n  );\n  if(settings[\"caseSensitive\"],\n    if(matches,\n      correct(),\n      if(matches_case_insensitive,\n        set_credit(settings[\"partialCredit\"],translate(\"part.patternmatch.correct except case\")),\n        incorrect()\n      )\n    )\n  ,\n    if(matches_case_insensitive,\n      correct()\n    ,\n      incorrect()\n    )\n  )\n\ninterpreted_answer (The student's answer, to be reused by other parts):\n  studentAnswer\n", "matrixentry": "rows (The number of rows in the student's answer): len(studentAnswer)\n\ncols (The number of columns in the student's answer): if(rows>0,len(studentAnswer[0]),0)\n\ncorrect_rows (The number of rows in the correct answer): len(settings['correctAnswer'])\n\ncorrect_cols (The number of columns in the correct answer): len(settings['correctAnswer'][0])\n\nnum_cells (The number of cells in the student's answer): rows*cols\n\ncell_indexes (The index of each cell in the student's answer): \n    product(list(0..rows-1), list(0..cols-1))\n\nstudent_cell_precisions (Precision the student used in each cell):\n    switch(\n        settings[\"precisionType\"]=\"dp\",\n            map(map(countdp(cell),cell,row), row, studentAnswer)\n        ,\n        settings[\"precisionType\"]=\"sigfig\",\n            map(map(countsigfigs(cell),cell,row), row, studentAnswer)\n        ,\n        map(map(0,cell,row),row,studentAnswer)\n    )\n\nall_same_precision:\n    if(len(student_cell_precisions)=0 or all(map(all(map(togivenprecision(c,settings[\"precisionType\"],student_cell_precisions[0][0],settings[\"strictPrecision\"]),c,row)),row,studentAnswer)),\n        true\n    ,\n        feedback(translate(\"part.matrix.not all cells same precision\"));\n        false\n    )\n\nstudentPrecision (Maximum precision student used, or the required precision, whichever's higher):\n    max(settings[\"precision\"], max(map(max(row),row,student_cell_precisions)))\n\nallowFractions: settings[\"allowFractions\"]\n\nallowedNotationStyles: [\"plain\",\"en\",\"si-en\"]\n\nstudentNumbers:\n    if(settings[\"allowFractions\"],\n        map(map(parsenumber_or_fraction(c, allowedNotationStyles), c, row), row, studentAnswer)\n    ,\n        map(map(parsenumber(c, allowedNotationStyles), c, row), row, studentAnswer)\n    )\n\nstudentMatrix (The student's answer, with each cell parsed to numbers): \n    matrix(studentNumbers)\n\nempty_cells (The positions of the cells in the student's answer which have been left empty):\n  filter(trim(studentAnswer[p[0]][p[1]])=\"\", p, cell_indexes)\n\nany_empty (Are any of the cells empty?):\n  assert(len(empty_cells)=0,\n    warn(translate(\"part.matrix.empty cell\"));\n    fail(translate(\"part.matrix.empty cell\"));\n    true\n  )\n\ninvalid_cells (The positions of the cells in the student's answer which can't be interpreted as numbers):\n    filter(isnan(studentNumbers[p[0]][p[1]]), p, cell_indexes)\n\nany_invalid (Are any of the cells invalid?):\n    assert(len(invalid_cells)=0,\n        warn(translate(\"part.matrix.invalid cell\"));\n        fail(translate(\"part.matrix.invalid cell\"));\n        true\n    )\n\nwrong_precision_cells (The indexes of the cells which are given to the wrong precision):\n    filter(not togivenprecision(studentAnswer[p[0]][p[1]], settings[\"precisionType\"], settings[\"precision\"], settings[\"strictPrecision\"]), p, cell_indexes)\n\nwrong_precision (Has every cell been given to the correct precision?):\n    assert(len(wrong_precision_cells)=0,\n        multiply_credit(settings['precisionPC'], settings[\"precisionMessage\"])\n    )\n\nwrong_size (Does the student's answer have the wrong dimensions?):\n    assert(rows=correct_rows and cols=correct_cols,\n        incorrect();\n        end()\n    )\n\nrounded_student_matrix (The student's answer, with each cell rounded to studentPrecision): \n    map(\n        switch(\n            settings[\"precisionType\"]=\"dp\",\n            precround(c, studentPrecision),\n            settings[\"precisionType\"]=\"sigfig\",\n            siground(c, studentPrecision),\n            c\n        ),\n        c,\n        studentMatrix\n    )\n\nrounded_correct_matrix:\n    map(\n        map(\n            switch(\n                settings[\"precisionType\"]=\"dp\",\n                precround(c, studentPrecision),\n                settings[\"precisionType\"]=\"sigfig\",\n                siground(c, studentPrecision),\n                c\n            ),\n            c,\n            row\n        ),\n        row,\n        settings[\"correctAnswer\"]\n    )\n\ncorrect_cells (The indexes of the cells which are correct):\n    filter(\n        if(p[0]<correct_rows and p[1]<correct_cols,\n            withintolerance(rounded_student_matrix[p[0]][p[1]], rounded_correct_matrix[p[0]][p[1]], settings['tolerance']),\n            false\n        ),\n        p, \n        cell_indexes\n    )\n\nmark:\n    apply(any_empty);\n    apply(any_invalid);\n    assert(settings[\"precisionType\"]=\"none\" and not settings[\"allowFractions\"], apply(all_same_precision));\n    apply(wrong_size);\n    if(len(correct_cells)=len(cell_indexes),\n        correct()\n    ,\n        if(settings['markPerCell'],\n            set_credit(len(correct_cells)/num_cells,translate('part.matrix.some incorrect',[\"count\":num_cells-len(correct_cells)]))\n        ,\n            incorrect();\n            end()\n        )\n    );\n    apply(wrong_precision)\n\ninterpreted_answer (The student's answer, to be reused by other parts):\n    apply(any_empty);\n    apply(any_invalid);\n    studentMatrix\n", "numberentry": "studentNumber (The student's answer, parsed as a number):\n    if(settings[\"allowFractions\"],\n        parsedecimal_or_fraction(studentAnswer,settings[\"notationStyles\"])\n    ,\n        parsedecimal(studentAnswer,settings[\"notationStyles\"])\n    )\n\nisInteger (Is the student's answer an integer?):\n    countdp(studentAnswer)=0\n\nisFraction (Is the student's answer a fraction?):\n    \"/\" in studentAnswer\n\nnumerator (The numerator of the student's answer, or 0 if not a fraction):\n    if(isFraction,\n        parsenumber(split(studentAnswer,\"/\")[0],settings[\"notationStyles\"])\n    ,\n        0\n    )\n\ndenominator (The numerator of the student's answer, or 0 if not a fraction):\n    if(isFraction,\n        parsenumber(split(studentAnswer,\"/\")[1],settings[\"notationStyles\"])\n    ,\n        0\n    )\n\ncancelled (Is the student's answer a cancelled fraction?):\n    assert(isFraction and gcd(numerator,denominator)=1,\n        assert(not settings[\"mustBeReduced\"],\n            multiply_credit(settings[\"mustBeReducedPC\"],translate(\"part.numberentry.answer not reduced\"))\n        );\n        false\n    )\n\ncleanedStudentAnswer:\n    cleannumber(studentAnswer, settings[\"notationStyles\"])\n\nstudent_is_scientific (Is the student's answer written in scientific notation?):\n    not isnan(matchnumber(studentAnswer, [\"scientific\"])[1])\n\nscientific_precision_offset (A number in scientific notation has 1 more significant digit than decimal places):\n    award(1,settings[\"precisionType\"]=\"dp\")\n\nstudentPrecision:\n    max(settings[\"precision\"],\n        switch(\n            student_is_scientific, countsigfigs(cleanedStudentAnswer)-scientific_precision_offset,\n            settings[\"precisionType\"]=\"dp\", max(settings[\"precision\"],countdp(cleanedStudentAnswer)),\n            settings[\"precisionType\"]=\"sigfig\", max(settings[\"precision\"],countsigfigs(cleanedStudentAnswer)),\n            0\n        )\n    )\n\nraw_minvalue:\n    switch(\n        student_is_scientific, siground(settings[\"minvalue\"],studentPrecision+scientific_precision_offset),\n        settings[\"precisionType\"]=\"dp\", precround(settings[\"minvalue\"],studentPrecision),\n        settings[\"precisionType\"]=\"sigfig\", siground(settings[\"minvalue\"],studentPrecision),\n        settings[\"minvalue\"]\n    )\n\nraw_maxvalue:\n    switch(\n        student_is_scientific, siground(settings[\"maxvalue\"],studentPrecision+scientific_precision_offset),\n        settings[\"precisionType\"]=\"dp\", precround(settings[\"maxvalue\"],studentPrecision),\n        settings[\"precisionType\"]=\"sigfig\", siground(settings[\"maxvalue\"],studentPrecision),\n        settings[\"maxvalue\"]\n    )\n\nminvalue: min(raw_minvalue,raw_maxvalue)\n\nmaxvalue: max(raw_minvalue,raw_maxvalue)\n\nvalidNumber (Is the student's answer a valid number?):\n    if(isnan(studentNumber),\n        warn(translate(\"part.numberentry.answer invalid\"));\n        fail(translate(\"part.numberentry.answer invalid\"))\n    ,\n        true\n    )\n\nnumberInRange (Is the student's number in the allowed range?):\n    if(studentNumber>=minvalue and studentNumber<=maxvalue,\n        correct()\n    ,\n        incorrect();\n        end()\n    )\n\ncorrectPrecision (Has the student's answer been given to the desired precision?):     \n    if(\n        if(student_is_scientific,\n            togivenprecision_scientific(studentAnswer,settings['precisionType'],settings['precision']),\n            togivenprecision(cleanedStudentAnswer,settings['precisionType'],settings['precision'],settings[\"strictPrecision\"])\n        )\n    ,\n        true\n    ,\n        multiply_credit(settings[\"precisionPC\"],settings[\"precisionMessage\"]);\n        false\n    )\n\nmark (Mark the student's answer):\n    apply(validNumber);\n    apply(numberInRange);\n    assert(numberInRange,end());\n    if(isFraction,\n        apply(cancelled)\n    ,\n        apply(correctPrecision)\n    )\n \ninterpreted_answer (The student's answer, to be reused by other parts):\n    apply(validNumber);\n    studentNumber\n\n", "multipleresponse": "numAnswers: len(settings[\"matrix\"])\n\nnumChoices: if(numAnswers=1,1,len(settings[\"matrix\"][0]))\n\nnumTicks (How many options did the student tick?):\n  sum(map(sum(map(if(x,1,0),x,row)),row,studentAnswer))\n\nwrongNumber:\n  assert(numTicks >= settings[\"minAnswers\"] and (settings[\"maxAnswers\"]=0 or numTicks<=settings[\"maxAnswers\"]),\n    if(settings[\"warningType\"]=\"prevent\",\n        fail(translate(\"part.mcq.wrong number of choices\"))\n    ,\n        incorrect(translate(\"part.mcq.wrong number of choices\"));\n        end()\n    )\n  )\n\ntick_indexes (Indexes of choice/answer pairs):\n    flatten(map(\n        map([x,y], x, shuffleAnswers),\n        y,\n        shuffleChoices\n    ))\n\nonly_ticked_score_ticks (The score for each choice/answer pair):\n  map(\n    if(studentAnswer[x][y],\n      let(distractor,settings[\"distractors\"][x][y], credit, if(marks=0,0,settings[\"matrix\"][x][y]/marks),\n        switch(\n          credit<>0,\n            if(not isnonemptyhtml(distractor),\n              add_credit(credit,translate(if(credit>0,'part.mcq.correct choice','part.mcq.incorrect choice')))\n            ,\n              add_credit(credit,distractor)\n            )\n          ,\n            if(isnonemptyhtml(distractor),negative_feedback(distractor),if(marks<>0,negative_feedback(translate('part.mcq.incorrect choice')),false))\n        );credit\n      )\n    ,\n      0\n    ),\n    [x,y],\n    tick_indexes\n  )\n\n\nlayout_tick_indexes (Indexes of choice/answer pairs shown in the layout):\n    filter(layout[tick[0]][tick[1]],tick,tick_indexes)\n\nbinary_score_ticks (Scores and feedback for each choice/answer pair, in the \"binary\" marking method):\n    let(\n        per_tick, 1/len(layout_tick_indexes),\n        scores,map(\n            let(distractor,settings[\"distractors\"][x][y],\n                should_tick, settings[\"matrix\"][x][y]>0,\n                if(studentAnswer[x][y]=should_tick,\n                    per_tick\n                ,\n                    assert(not isnonemptyhtml(distractor),negative_feedback(distractor));\n                    0\n                )\n            ),\n            [x,y],\n            layout_tick_indexes\n        ),\n        total, sum(scores),\n        switch(\n            total=1,correct(),\n            total=0 or settings[\"markingMethod\"]=\"all-or-nothing\",incorrect(),\n            set_credit(total,translate('part.marking.partially correct'))\n        )\n    )\n\nscore_ticks:\n    switch(\n        settings[\"markingMethod\"] in [\"score per matched cell\",\"all-or-nothing\"], apply(binary_score_ticks);binary_score_ticks,\n        apply(only_ticked_score_ticks);only_ticked_score_ticks\n    )\n\ntotal_score: \n    sum(score_ticks)\n\nmark:\n  assert(marks>0,correct());  // any answer is correct when 0 marks are available\n  assert(settings[\"markingMethod\"]<>\"sum ticked cells\" or numTicks>0,\n    warn(translate(\"part.marking.nothing entered\"));\n    fail(translate(\"part.marking.nothing entered\"))\n  );\n  apply(wrongNumber);\n  apply(score_ticks)\n\ninterpreted_answer (The student's answer, to be reused by other parts):\n  studentAnswer\n"};
        });
        ;
Numbas.queueScript('SCORM_API_wrapper', [], function(module) {
/* ===========================================================
pipwerks SCORM Wrapper for JavaScript
v1.1.20121005
Created by Philip Hutchison, January 2008
https://github.com/pipwerks/scorm-api-wrapper
Copyright (c) Philip Hutchison
MIT-style license: http://pipwerks.mit-license.org/
This wrapper works with both SCORM 1.2 and SCORM 2004.
Inspired by APIWrapper.js, created by the ADL and
Concurrent Technologies Corporation, distributed by
the ADL (http://www.adlnet.gov/scorm).
SCORM.API.find() and SCORM.API.get() functions based
on ADL code, modified by Mike Rustici
(http://www.scorm.com/resources/apifinder/SCORMAPIFinder.htm),
further modified by Philip Hutchison
=============================================================== */
var pipwerks = {};                                  //pipwerks 'namespace' helps ensure no conflicts with possible other "SCORM" variables
pipwerks.UTILS = {};                                //For holding UTILS functions
pipwerks.debug = { isActive: false };                //Enable (true) or disable (false) for debug mode
pipwerks.SCORM = {                                  //Define the SCORM object
    version:    null,                               //Store SCORM version.
    handleCompletionStatus: true,                   //Whether or not the wrapper should automatically handle the initial completion status
    handleExitMode: true,                           //Whether or not the wrapper should automatically handle the exit mode
    API:        { handle: null,
                  isFound: false },                 //Create API child object
    connection: { isActive: false },                //Create connection child object
    data:       { completionStatus: null,
                  exitStatus: null },               //Create data child object
    debug:      {}                                  //Create debug child object
};
/* --------------------------------------------------------------------------------
   pipwerks.SCORM.isAvailable
   A simple function to allow Flash ExternalInterface to confirm
   presence of JS wrapper before attempting any LMS communication.
   Parameters: none
   Returns:    Boolean (true)
----------------------------------------------------------------------------------- */
pipwerks.SCORM.isAvailable = function() {
    return true;
};
// ------------------------------------------------------------------------- //
// --- SCORM.API functions ------------------------------------------------- //
// ------------------------------------------------------------------------- //
/* -------------------------------------------------------------------------
   pipwerks.SCORM.API.find(window)
   Looks for an object named API in parent and opener windows
   Parameters: window (the browser window object).
   Returns:    Object if API is found, null if no API found
---------------------------------------------------------------------------- */
pipwerks.SCORM.API.find = function(win) {
    var API = null,
        findAttempts = 0,
        findAttemptLimit = 500,
        errorGettingAPI = false,
        traceMsgPrefix = "SCORM.API.find",
        trace = pipwerks.UTILS.trace,
        scorm = pipwerks.SCORM;
    try {
        while (!errorGettingAPI &&
               (!win.API && !win.API_1484_11) &&
               (win.parent) &&
               (win.parent != win) &&
               (findAttempts <= findAttemptLimit)) {
                    findAttempts++;
                    win = win.parent;
        }
    } catch(e) {
        errorGettingAPI = e;
    }
    try {
        if(scorm.version) {                                            //If SCORM version is specified by user, look for specific API
            switch(scorm.version) {
                case "2004" :
                    if(win.API_1484_11) {
                        API = win.API_1484_11;
                    } else {
                        trace(traceMsgPrefix + ": SCORM version 2004 was specified by user, but API_1484_11 cannot be found.");
                    }
                    break;
                case "1.2" :
                    if(win.API) {
                        API = win.API;
                    } else {
                        trace(traceMsgPrefix + ": SCORM version 1.2 was specified by user, but API cannot be found.");
                    }
                    break;
            }
        } else {                                                    //If SCORM version not specified by user, look for APIs
            if(win.API_1484_11) {                                    //SCORM 2004-specific API.
                scorm.version = "2004";                                //Set version
                API = win.API_1484_11;
            } else if(win.API) {                                        //SCORM 1.2-specific API
                scorm.version = "1.2";                                //Set version
                API = win.API;
            }
        }
    } catch(e) {
        errorGettingAPI = e;
    }
    if(API) {
        trace(traceMsgPrefix + ": API found. Version: " + scorm.version);
        trace("API: " + API);
    } else {
        trace(traceMsgPrefix + ": Error finding API. \nFind attempts: " + findAttempts + ". \nFind attempt limit: " + findAttemptLimit + ". \nError getting window parent: " + errorGettingAPI);
    }
    return API;
};
/* -------------------------------------------------------------------------
   pipwerks.SCORM.API.get()
   Looks for an object named API, first in the current window's frame
   hierarchy and then, if necessary, in the current window's opener window
   hierarchy (if there is an opener window).
   Parameters:  None.
   Returns:     Object if API found, null if no API found
---------------------------------------------------------------------------- */
pipwerks.SCORM.API.get = function() {
    var API = null,
        win = window,
        scorm = pipwerks.SCORM,
        find = scorm.API.find,
        trace = pipwerks.UTILS.trace;
    try {
        API = find(win);

        if(!API && win.parent && win.parent != win) {
            API = find(win.parent);
        }
        if(!API && win.top.opener) {
            API = find(win.top.opener);
        }
        //Special handling for Plateau
        //Thanks to Joseph Venditti for the patch
        if(!API && win.top.opener && win.top.opener.document) {
            API = find(win.top.opener.document);
        }
    } catch {}
    if(API) {
        scorm.API.isFound = true;
    } else {
        trace("API.get failed: Can't find the API!");
    }
    return API;
};
/* -------------------------------------------------------------------------
   pipwerks.SCORM.API.getHandle()
   Returns the handle to API object if it was previously set
   Parameters:  None.
   Returns:     Object (the pipwerks.SCORM.API.handle variable).
---------------------------------------------------------------------------- */
pipwerks.SCORM.API.getHandle = function() {
    var API = pipwerks.SCORM.API;
    if(!API.handle && !API.isFound) {
        API.handle = API.get();
    }
    return API.handle;
};
// ------------------------------------------------------------------------- //
// --- pipwerks.SCORM.connection functions --------------------------------- //
// ------------------------------------------------------------------------- //
/* -------------------------------------------------------------------------
   pipwerks.SCORM.connection.initialize()
   Tells the LMS to initiate the communication session.
   Parameters:  None
   Returns:     Boolean
---------------------------------------------------------------------------- */
pipwerks.SCORM.connection.initialize = function() {
    var success = false,
        scorm = pipwerks.SCORM,
        completionStatus = scorm.data.completionStatus,
        trace = pipwerks.UTILS.trace,
        makeBoolean = pipwerks.UTILS.StringToBoolean,
        debug = scorm.debug,
        traceMsgPrefix = "SCORM.connection.initialize ";
    trace("connection.initialize called.");
    scorm.connection.isActive = false;
    if(!scorm.connection.isActive) {
        scorm.API.handle = null;
        scorm.API.isFound = false;
        var API = scorm.API.getHandle(),
            errorCode = 0;
        if(API) {
            switch(scorm.version) {
                case "1.2" : success = makeBoolean(API.LMSInitialize("")); break;
                case "2004": success = makeBoolean(API.Initialize("")); break;
            }
            if(success) {
                //Double-check that connection is active and working before returning 'true' boolean
                errorCode = debug.getCode();
                if(errorCode !== null && errorCode === 0) {
                    scorm.connection.isActive = true;
                    if(scorm.handleCompletionStatus) {
                        //Automatically set new launches to incomplete
                        completionStatus = scorm.status("get");
                        if(completionStatus) {
                            switch(completionStatus) {
                                //Both SCORM 1.2 and 2004
                                case "not attempted": scorm.status("set", "incomplete"); break;
                                //SCORM 2004 only
                                case "unknown" : scorm.status("set", "incomplete"); break;
                                //Additional options, presented here in case you'd like to use them
                                //case "completed"  : break;
                                //case "incomplete" : break;
                                //case "passed"     : break;    //SCORM 1.2 only
                                //case "failed"     : break;    //SCORM 1.2 only
                                //case "browsed"    : break;    //SCORM 1.2 only
                            }
                        }
                    }
                } else {
                    success = false;
                    trace(traceMsgPrefix + "failed. \nError code: " + errorCode + " \nError info: " + debug.getInfo(errorCode));
                }
            } else {
                errorCode = debug.getCode();
                if(errorCode !== null && errorCode !== 0) {
                    trace(traceMsgPrefix + "failed. \nError code: " + errorCode + " \nError info: " + debug.getInfo(errorCode));
                } else {
                    trace(traceMsgPrefix + "failed: No response from server.");
                }
            }
        } else {
            trace(traceMsgPrefix + "failed: API is null.");
        }
    } else {
          trace(traceMsgPrefix + "aborted: Connection already active.");
     }
     return success;
};
/* -------------------------------------------------------------------------
   pipwerks.SCORM.connection.terminate()
   Tells the LMS to terminate the communication session
   Parameters:  None
   Returns:     Boolean
---------------------------------------------------------------------------- */
pipwerks.SCORM.connection.terminate = function() {
    var success = false,
        scorm = pipwerks.SCORM,
        exitStatus = scorm.data.exitStatus,
        completionStatus = scorm.data.completionStatus,
        trace = pipwerks.UTILS.trace,
        makeBoolean = pipwerks.UTILS.StringToBoolean,
        debug = scorm.debug,
        traceMsgPrefix = "SCORM.connection.terminate ";
    if(scorm.connection.isActive) {
        var API = scorm.API.getHandle(),
            errorCode = 0;
        if(API) {
             if(scorm.handleExitMode && !exitStatus) {
                if(completionStatus !== "completed" && completionStatus !== "passed") {
                    switch(scorm.version) {
                        case "1.2" : success = scorm.set("cmi.core.exit", "suspend"); break;
                        case "2004": success = scorm.set("cmi.exit", "suspend"); break;
                    }
                } else {
                    switch(scorm.version) {
                        case "1.2" : success = scorm.set("cmi.core.exit", "logout"); break;
                        case "2004": success = scorm.set("cmi.exit", "normal"); break;
                    }
                }
            }
            switch(scorm.version) {
                case "1.2" : success = makeBoolean(API.LMSFinish("")); break;
                case "2004": success = makeBoolean(API.Terminate("")); break;
            }
            if(success) {
                scorm.connection.isActive = false;
            } else {
                errorCode = debug.getCode();
                trace(traceMsgPrefix + "failed. \nError code: " + errorCode + " \nError info: " + debug.getInfo(errorCode));
            }
        } else {
            trace(traceMsgPrefix + "failed: API is null.");
        }
    } else {
        trace(traceMsgPrefix + "aborted: Connection already terminated.");
    }
    return success;
};
// ------------------------------------------------------------------------- //
// --- pipwerks.SCORM.data functions --------------------------------------- //
// ------------------------------------------------------------------------- //
/* -------------------------------------------------------------------------
   pipwerks.SCORM.data.get(parameter)
   Requests information from the LMS.
   Parameter: parameter (string, name of the SCORM data model element)
   Returns:   string (the value of the specified data model element)
---------------------------------------------------------------------------- */
pipwerks.SCORM.data.get = function(parameter) {
    var value = null,
        scorm = pipwerks.SCORM,
        trace = pipwerks.UTILS.trace,
        debug = scorm.debug,
        traceMsgPrefix = "SCORM.data.get(" + parameter + ") ";
    if(scorm.connection.isActive) {
        var API = scorm.API.getHandle(),
            errorCode = 0;
          if(API) {
            switch(scorm.version) {
                case "1.2" : value = API.LMSGetValue(parameter); break;
                case "2004": value = API.GetValue(parameter); break;
            }
            errorCode = debug.getCode();
            //GetValue returns an empty string on errors
            //If value is an empty string, check errorCode to make sure there are no errors
            if(value !== "" || errorCode === 0) {
                //GetValue is successful.
                //If parameter is lesson_status/completion_status or exit status, let's
                //grab the value and cache it so we can check it during connection.terminate()
                switch(parameter) {
                    case "cmi.core.lesson_status":
                    case "cmi.completion_status" : scorm.data.completionStatus = value; break;
                    case "cmi.core.exit":
                    case "cmi.exit" : scorm.data.exitStatus = value; break;
                }
            } else {
                trace(traceMsgPrefix + "failed. \nError code: " + errorCode + "\nError info: " + debug.getInfo(errorCode));
            }
        } else {
            trace(traceMsgPrefix + "failed: API is null.");
        }
    } else {
        trace(traceMsgPrefix + "failed: API connection is inactive.");
    }
    trace(traceMsgPrefix + " value: " + value);
    return String(value);
};
/* -------------------------------------------------------------------------
   pipwerks.SCORM.data.set()
   Tells the LMS to assign the value to the named data model element.
   Also stores the SCO's completion status in a variable named
   pipwerks.SCORM.data.completionStatus. This variable is checked whenever
   pipwerks.SCORM.connection.terminate() is invoked.
   Parameters: parameter (string). The data model element
               value (string). The value for the data model element
   Returns:    Boolean
---------------------------------------------------------------------------- */
pipwerks.SCORM.data.set = function(parameter, value) {
    var success = false,
        scorm = pipwerks.SCORM,
        trace = pipwerks.UTILS.trace,
        makeBoolean = pipwerks.UTILS.StringToBoolean,
        debug = scorm.debug,
        traceMsgPrefix = "SCORM.data.set(" + parameter + ") ";
    if(scorm.connection.isActive) {
        var API = scorm.API.getHandle(),
            errorCode = 0;
        if(API) {
            switch(scorm.version) {
                case "1.2" : success = makeBoolean(API.LMSSetValue(parameter, value)); break;
                case "2004": success = makeBoolean(API.SetValue(parameter, value)); break;
            }
            if(success) {
                if(parameter === "cmi.core.lesson_status" || parameter === "cmi.completion_status") {
                    scorm.data.completionStatus = value;
                }
            } else {
                trace(traceMsgPrefix + "failed. \nError code: " + errorCode + ". \nError info: " + debug.getInfo(errorCode));
            }
        } else {
            trace(traceMsgPrefix + "failed: API is null.");
        }
    } else {
        trace(traceMsgPrefix + "failed: API connection is inactive.");
    }
    return success;
};
/* -------------------------------------------------------------------------
   pipwerks.SCORM.data.save()
   Instructs the LMS to persist all data to this point in the session
   Parameters: None
   Returns:    Boolean
---------------------------------------------------------------------------- */
pipwerks.SCORM.data.save = function() {
    var success = false,
        scorm = pipwerks.SCORM,
        trace = pipwerks.UTILS.trace,
        makeBoolean = pipwerks.UTILS.StringToBoolean,
        traceMsgPrefix = "SCORM.data.save failed";
    if(scorm.connection.isActive) {
        var API = scorm.API.getHandle();
        if(API) {
            switch(scorm.version) {
                case "1.2" : success = makeBoolean(API.LMSCommit("")); break;
                case "2004": success = makeBoolean(API.Commit("")); break;
            }
        } else {
            trace(traceMsgPrefix + ": API is null.");
        }
    } else {
        trace(traceMsgPrefix + ": API connection is inactive.");
    }
    return success;
};
pipwerks.SCORM.status = function(action, status) {
    var success = false,
        scorm = pipwerks.SCORM,
        trace = pipwerks.UTILS.trace,
        traceMsgPrefix = "SCORM.getStatus failed",
        cmi = "";
    if(action !== null) {
        switch(scorm.version) {
            case "1.2" : cmi = "cmi.core.lesson_status"; break;
            case "2004": cmi = "cmi.completion_status"; break;
        }
        switch(action) {
            case "get": success = scorm.data.get(cmi); break;
            case "set": if(status !== null) {
                            success = scorm.data.set(cmi, status);
                        } else {
                            success = false;
                            trace(traceMsgPrefix + ": status was not specified.");
                        }
                        break;
            default : success = false;
                        trace(traceMsgPrefix + ": no valid action was specified.");
        }
    } else {
        trace(traceMsgPrefix + ": action was not specified.");
    }
    return success;
};
// ------------------------------------------------------------------------- //
// --- pipwerks.SCORM.debug functions -------------------------------------- //
// ------------------------------------------------------------------------- //
/* -------------------------------------------------------------------------
   pipwerks.SCORM.debug.getCode
   Requests the error code for the current error state from the LMS
   Parameters: None
   Returns:    Integer (the last error code).
---------------------------------------------------------------------------- */
pipwerks.SCORM.debug.getCode = function() {
    var scorm = pipwerks.SCORM,
        API = scorm.API.getHandle(),
        trace = pipwerks.UTILS.trace,
        code = 0;
    if(API) {
        switch(scorm.version) {
            case "1.2" : code = parseInt(API.LMSGetLastError(), 10); break;
            case "2004": code = parseInt(API.GetLastError(), 10); break;
        }
    } else {
        trace("SCORM.debug.getCode failed: API is null.");
    }
    return code;
};
/* -------------------------------------------------------------------------
   pipwerks.SCORM.debug.getInfo()
   "Used by a SCO to request the textual description for the error code
   specified by the value of [errorCode]."
   Parameters: errorCode (integer).
   Returns:    String.
----------------------------------------------------------------------------- */
pipwerks.SCORM.debug.getInfo = function(errorCode) {
    var scorm = pipwerks.SCORM,
        API = scorm.API.getHandle(),
        trace = pipwerks.UTILS.trace,
        result = "";
    if(API) {
        switch(scorm.version) {
            case "1.2" : result = API.LMSGetErrorString(errorCode.toString()); break;
            case "2004": result = API.GetErrorString(errorCode.toString()); break;
        }
    } else {
        trace("SCORM.debug.getInfo failed: API is null.");
    }
    return String(result);
};
/* -------------------------------------------------------------------------
   pipwerks.SCORM.debug.getDiagnosticInfo
   "Exists for LMS specific use. It allows the LMS to define additional
   diagnostic information through the API Instance."
   Parameters: errorCode (integer).
   Returns:    String (Additional diagnostic information about the given error code).
---------------------------------------------------------------------------- */
pipwerks.SCORM.debug.getDiagnosticInfo = function(errorCode) {
    var scorm = pipwerks.SCORM,
        API = scorm.API.getHandle(),
        trace = pipwerks.UTILS.trace,
        result = "";
    if(API) {
        switch(scorm.version) {
            case "1.2" : result = API.LMSGetDiagnostic(errorCode); break;
            case "2004": result = API.GetDiagnostic(errorCode); break;
        }
    } else {
        trace("SCORM.debug.getDiagnosticInfo failed: API is null.");
    }
    return String(result);
};
// ------------------------------------------------------------------------- //
// --- Shortcuts! ---------------------------------------------------------- //
// ------------------------------------------------------------------------- //
// Because nobody likes typing verbose code.
pipwerks.SCORM.init = pipwerks.SCORM.connection.initialize;
pipwerks.SCORM.get = pipwerks.SCORM.data.get;
pipwerks.SCORM.set = pipwerks.SCORM.data.set;
pipwerks.SCORM.save = pipwerks.SCORM.data.save;
pipwerks.SCORM.quit = pipwerks.SCORM.connection.terminate;
// ------------------------------------------------------------------------- //
// --- pipwerks.UTILS functions -------------------------------------------- //
// ------------------------------------------------------------------------- //
/* -------------------------------------------------------------------------
   pipwerks.UTILS.StringToBoolean()
   Converts 'boolean strings' into actual valid booleans.
   (Most values returned from the API are the strings "true" and "false".)
   Parameters: String
   Returns:    Boolean
---------------------------------------------------------------------------- */
pipwerks.UTILS.StringToBoolean = function(value) {
    var t = typeof value;
    switch(t) {
       //typeof new String("true") === "object", so handle objects as string via fall-through.
       //See https://github.com/pipwerks/scorm-api-wrapper/issues/3
       case "object":
       case "string": return (/(true|1)/i).test(value);
       case "number": return !!value;
       case "boolean": return value;
       case "undefined": return null;
       default: return false;
    }
};
/* -------------------------------------------------------------------------
   pipwerks.UTILS.trace()
   Displays error messages when in debug mode.
   Parameters: msg (string)
   Return:     None
---------------------------------------------------------------------------- */
pipwerks.UTILS.trace = function(msg) {
     if(pipwerks.debug.isActive) {
        if(window.console && window.console.log) {
            console.log(msg);
        } else {
            //alert(msg);
        }
     }
};
module.exports.pipwerks = pipwerks
});
;
Numbas.queueScript('analysis-display', ['base', 'download', 'util', 'csv', 'display-base'], function() {
    Numbas.analysis = {};

    /**
     * Given max scores for an item for each attempt, describe it as either `undefined`, `"varies"` or a number.
     * The max score is `undefined` if no attempt used this item.
     * It's `"varies"` if there are two or more attempts with different values.
     * It's a number if every attempt that uses this item has the same value.
     *
     * @param {Array.<number>} scores
     * @returns {string}
     */
    const describe_max_score = Numbas.analysis.describe_max_score = function(scores) {
        const unique_scores = new Set(scores.filter((s) => s !== undefined));
        if(unique_scores.size == 0) {
            return undefined;
        } else if(unique_scores.size > 1) {
            return 'varies';
        } else {
            return Array.from(unique_scores)[0];
        }
    }

    /**
     * A file uploaded by the user. It should be an attempt data file produced by the Numbas exam runtime.
     * This file is decoded, decrypted and parsed as JSON. The file is considered succesfully processed if all of these steps succeed.
     */
    class AttemptFile {
        constructor(file, vm) {
            this.file = file;
            this.vm = vm;
            this.content = Knockout.observable(null);
            this.error = Knockout.observable(null);
            this.raw_text = Knockout.observable(null);

            this.status = Knockout.computed(() => {
                if(this.error()) {
                    return 'error';
                } else if(this.content() && this.attempt_grouped_questions()) {
                    return 'processed';
                } else {
                    return 'processing';
                }
            });
            this.status_class = Knockout.computed(() => {
                return {
                    'error': 'danger',
                    'processed': '',
                    'processing': 'info'
                }[this.status()] || '';
            });

            this.student_name = Knockout.computed(() => this.content()?.student_name || R('analysis.student name.anonymous'));

            this.start_time = Knockout.computed(() => new Date(this.content()?.start));

            this.score = Knockout.computed(() => this.content()?.score || 0);
            this.max_score = Knockout.computed(() => this.content()?.max_score || 0);

            this.score_percentage = Knockout.computed(() => this.score() / this.max_score());

            this.download_url = window.URL.createObjectURL(this.file);

            /** For each question defined in the exam, an object relating it to the group and question index it appeared at in this attempt.
             */
            this.attempt_grouped_questions = Knockout.computed(() => {
                const exam_object = this.vm.exam_object;
                const content = this.content();
                if(!(content && exam_object)) {
                    return;
                }

                const attempt_grouped_questions = exam_object.question_groups.map(() => []);

                const question_order = content.questionGroupOrder.flatMap((i, s) => content.questionSubsets[s].map((n) => {
                    return {group_number:i, question_number:n}
                })).map((d, i) => {
                    d.data = content.questions[i]; return d
                });
                question_order.forEach(({group_number, question_number, data}) => {
                    attempt_grouped_questions[group_number][question_number] = data;
                });

                return attempt_grouped_questions;
            }, this);

            /** Data for the row corresponding to this attempt in the question totals-only data table.
             */
            this.question_table_row = Knockout.computed(() => {
                const exam_object = this.vm.exam_object;
                if(!(this.content() && exam_object)) {
                    return [];
                }
                const attempt_grouped_questions = this.attempt_grouped_questions();

                const row = [this.student_name(), this.score()];

                exam_object.question_groups.forEach((g, gi) => {
                    g.questions.forEach((q, qi) => {
                        const attempt_question = attempt_grouped_questions[gi][qi];
                        row.push(attempt_question?.score);
                    });
                });

                return row;
            });

            /** Data for the row corresponding to this attempt in the full data table.
             */
            this.full_table_row = Knockout.computed(() => {
                const exam_object = this.vm.exam_object;
                if(!(this.content() && exam_object)) {
                    return [];
                }
                const attempt_grouped_questions = this.attempt_grouped_questions();

                const row = [this.student_name(), this.score()];

                exam_object.question_groups.forEach((g, gi) => {
                    g.questions.forEach((q, qi) => {
                        const attempt_question = attempt_grouped_questions[gi][qi];

                        row.push(attempt_question?.score);

                        if(q.partsMode == 'explore') {
                   