Numbas.addExtension('numbertheory',['jme'],function(numbertheory) {
  var scope = numbertheory.scope;
  var TNum = Numbas.jme.types.TNum;
  var TList = Numbas.jme.types.TList;
  var TString = Numbas.jme.types.TString;
  var TMatrix = Numbas.jme.types.TMatrix;
  var TExpression = Numbas.jme.types.TExpression;


  var funcObj = Numbas.jme.funcObj;

  /** CONSTANTS **/

  /* We hard-code a list of primes, as this should be enough to cover most of our usage.*/
  const primes = [2, 3, 5, 7, 11, 13, 17, 19, 23, 29, 31, 37, 41, 43, 47, 53, 59, 61, 67, 71, 73, 79, 83, 89, 97, 101, 103, 107, 109, 113, 127, 131, 137, 139, 149, 151, 157, 163, 167, 173, 179, 181, 191, 193, 197, 199, 211, 223, 227, 229, 233, 239, 241, 251, 257, 263, 269, 271, 277, 281, 283, 293, 307, 311, 313, 317, 331, 337, 347, 349, 353, 359, 367, 373, 379, 383, 389, 397, 401, 409, 419, 421, 431, 433, 439, 443, 449, 457, 461, 463, 467, 479, 487, 491, 499, 503, 509, 521, 523, 541, 547, 557, 563, 569, 571, 577, 587, 593, 599, 601, 607, 613, 617, 619, 631, 641, 643, 647, 653, 659, 661, 673, 677, 683, 691, 701, 709, 719, 727, 733, 739, 743, 751, 757, 761, 769, 773, 787, 797, 809, 811, 821, 823, 827, 829, 839, 853, 857, 859, 863, 877, 881, 883, 887, 907, 911, 919, 929, 937, 941, 947, 953, 967, 971, 977, 983, 991, 997, 1009, 1013, 1019, 1021, 1031, 1033, 1039, 1049, 1051, 1061, 1063, 1069, 1087, 1091, 1093, 1097, 1103, 1109, 1117, 1123, 1129, 1151, 1153, 1163, 1171, 1181, 1187, 1193, 1201, 1213, 1217, 1223, 1229, 1231, 1237, 1249, 1259, 1277, 1279, 1283, 1289, 1291, 1297, 1301, 1303, 1307, 1319, 1321, 1327, 1361, 1367, 1373, 1381, 1399, 1409, 1423, 1427, 1429, 1433, 1439, 1447, 1451, 1453, 1459, 1471, 1481, 1483, 1487, 1489, 1493, 1499, 1511, 1523, 1531, 1543, 1549, 1553, 1559, 1567, 1571, 1579, 1583, 1597, 1601, 1607, 1609, 1613, 1619, 1621, 1627, 1637, 1657, 1663, 1667, 1669, 1693, 1697, 1699, 1709, 1721, 1723, 1733, 1741, 1747, 1753, 1759, 1777, 1783, 1787, 1789, 1801, 1811, 1823, 1831, 1847, 1861, 1867, 1871, 1873, 1877, 1879, 1889, 1901, 1907, 1913, 1931, 1933, 1949, 1951, 1973, 1979, 1987, 1993, 1997, 1999, 2003, 2011, 2017, 2027, 2029, 2039, 2053, 2063, 2069, 2081, 2083, 2087, 2089, 2099, 2111, 2113, 2129, 2131, 2137, 2141, 2143, 2153, 2161, 2179, 2203, 2207, 2213, 2221, 2237, 2239, 2243, 2251, 2267, 2269, 2273, 2281, 2287, 2293, 2297, 2309, 2311, 2333, 2339, 2341, 2347, 2351, 2357, 2371, 2377, 2381, 2383, 2389, 2393, 2399, 2411, 2417, 2423, 2437, 2441, 2447, 2459, 2467, 2473, 2477, 2503, 2521, 2531, 2539, 2543, 2549, 2551, 2557, 2579, 2591, 2593, 2609, 2617, 2621, 2633, 2647, 2657, 2659, 2663, 2671, 2677, 2683, 2687, 2689, 2693, 2699, 2707, 2711, 2713, 2719, 2729, 2731, 2741, 2749, 2753, 2767, 2777, 2789, 2791, 2797, 2801, 2803, 2819, 2833, 2837, 2843, 2851, 2857, 2861, 2879, 2887, 2897, 2903, 2909, 2917, 2927, 2939, 2953, 2957, 2963, 2969, 2971, 2999, 3001, 3011, 3019, 3023, 3037, 3041, 3049, 3061, 3067, 3079, 3083, 3089, 3109, 3119, 3121, 3137, 3163, 3167, 3169, 3181, 3187, 3191, 3203, 3209, 3217, 3221, 3229, 3251, 3253, 3257, 3259, 3271, 3299, 3301, 3307, 3313, 3319, 3323, 3329, 3331, 3343, 3347, 3359, 3361, 3371, 3373, 3389, 3391, 3407, 3413, 3433, 3449, 3457, 3461, 3463, 3467, 3469, 3491, 3499, 3511, 3517, 3527, 3529, 3533, 3539, 3541, 3547, 3557, 3559, 3571, 3581, 3583, 3593, 3607, 3613, 3617, 3623, 3631, 3637, 3643, 3659, 3671, 3673, 3677, 3691, 3697, 3701, 3709, 3719, 3727, 3733, 3739, 3761, 3767, 3769, 3779, 3793, 3797, 3803, 3821, 3823, 3833, 3847, 3851, 3853, 3863, 3877, 3881, 3889, 3907, 3911, 3917, 3919, 3923, 3929, 3931, 3943, 3947, 3967, 3989, 4001, 4003, 4007, 4013, 4019, 4021, 4027, 4049, 4051, 4057, 4073, 4079, 4091, 4093, 4099, 4111, 4127, 4129, 4133, 4139, 4153, 4157, 4159, 4177, 4201, 4211, 4217, 4219, 4229, 4231, 4241, 4243, 4253, 4259, 4261, 4271, 4273, 4283, 4289, 4297, 4327, 4337, 4339, 4349, 4357, 4363, 4373, 4391, 4397, 4409, 4421, 4423, 4441, 4447, 4451, 4457, 4463, 4481, 4483, 4493, 4507, 4513, 4517, 4519, 4523, 4547, 4549, 4561, 4567, 4583, 4591, 4597, 4603, 4621, 4637, 4639, 4643, 4649, 4651, 4657, 4663, 4673, 4679, 4691, 4703, 4721, 4723, 4729, 4733, 4751, 4759, 4783, 4787, 4789, 4793, 4799, 4801, 4813, 4817, 4831, 4861, 4871, 4877, 4889, 4903, 4909, 4919, 4931, 4933, 4937, 4943, 4951, 4957, 4967, 4969, 4973, 4987, 4993, 4999]
  
  /* The S-box is used for Rijndael rounds, in the AES algorithm.*/
  const Sbox = [[0x63, 0x7c, 0x77, 0x7b, 0xf2, 0x6b, 0x6f, 0xc5, 0x30, 0x01, 0x67, 0x2b, 0xfe, 0xd7, 0xab, 0x76],
                [0xca, 0x82, 0xc9, 0x7d, 0xfa, 0x59, 0x47, 0xf0, 0xad, 0xd4, 0xa2, 0xaf, 0x9c, 0xa4, 0x72, 0xc0],
                [0xb7, 0xfd, 0x93, 0x26, 0x36, 0x3f, 0xf7, 0xcc, 0x34, 0xa5, 0xe5, 0xf1, 0x71, 0xd8, 0x31, 0x15],
                [0x04, 0xc7, 0x23, 0xc3, 0x18, 0x96, 0x05, 0x9a, 0x07, 0x12, 0x80, 0xe2, 0xeb, 0x27, 0xb2, 0x75],
                [0x09, 0x83, 0x2c, 0x1a, 0x1b, 0x6e, 0x5a, 0xa0, 0x52, 0x3b, 0xd6, 0xb3, 0x29, 0xe3, 0x2f, 0x84],
                [0x53, 0xd1, 0x00, 0xed, 0x20, 0xfc, 0xb1, 0x5b, 0x6a, 0xcb, 0xbe, 0x39, 0x4a, 0x4c, 0x58, 0xcf],
                [0xd0, 0xef, 0xaa, 0xfb, 0x43, 0x4d, 0x33, 0x85, 0x45, 0xf9, 0x02, 0x7f, 0x50, 0x3c, 0x9f, 0xa8],
                [0x51, 0xa3, 0x40, 0x8f, 0x92, 0x9d, 0x38, 0xf5, 0xbc, 0xb6, 0xda, 0x21, 0x10, 0xff, 0xf3, 0xd2],
                [0xcd, 0x0c, 0x13, 0xec, 0x5f, 0x97, 0x44, 0x17, 0xc4, 0xa7, 0x7e, 0x3d, 0x64, 0x5d, 0x19, 0x73],
                [0x60, 0x81, 0x4f, 0xdc, 0x22, 0x2a, 0x90, 0x88, 0x46, 0xee, 0xb8, 0x14, 0xde, 0x5e, 0x0b, 0xdb],
                [0xe0, 0x32, 0x3a, 0x0a, 0x49, 0x06, 0x24, 0x5c, 0xc2, 0xd3, 0xac, 0x62, 0x91, 0x95, 0xe4, 0x79],
                [0xe7, 0xc8, 0x37, 0x6d, 0x8d, 0xd5, 0x4e, 0xa9, 0x6c, 0x56, 0xf4, 0xea, 0x65, 0x7a, 0xae, 0x08],
                [0xba, 0x78, 0x25, 0x2e, 0x1c, 0xa6, 0xb4, 0xc6, 0xe8, 0xdd, 0x74, 0x1f, 0x4b, 0xbd, 0x8b, 0x8a],
                [0x70, 0x3e, 0xb5, 0x66, 0x48, 0x03, 0xf6, 0x0e, 0x61, 0x35, 0x57, 0xb9, 0x86, 0xc1, 0x1d, 0x9e],
                [0xe1, 0xf8, 0x98, 0x11, 0x69, 0xd9, 0x8e, 0x94, 0x9b, 0x1e, 0x87, 0xe9, 0xce, 0x55, 0x28, 0xdf],
                [0x8c, 0xa1, 0x89, 0x0d, 0xbf, 0xe6, 0x42, 0x68, 0x41, 0x99, 0x2d, 0x0f, 0xb0, 0x54, 0xbb, 0x16 ]];

    /* The inverse S-box inverts the operation of the S-box substitution.*/
  const Sinv = [[0x52, 0x09, 0x6a, 0xd5, 0x30, 0x36, 0xa5, 0x38, 0xbf, 0x40, 0xa3, 0x9e, 0x81, 0xf3, 0xd7, 0xfb],
                [0x7c, 0xe3, 0x39, 0x82, 0x9b, 0x2f, 0xff, 0x87, 0x34, 0x8e, 0x43, 0x44, 0xc4, 0xde, 0xe9, 0xcb],
                [0x54, 0x7b, 0x94, 0x32, 0xa6, 0xc2, 0x23, 0x3d, 0xee, 0x4c, 0x95, 0x0b, 0x42, 0xfa, 0xc3, 0x4e],
                [0x08, 0x2e, 0xa1, 0x66, 0x28, 0xd9, 0x24, 0xb2, 0x76, 0x5b, 0xa2, 0x49, 0x6d, 0x8b, 0xd1, 0x25],
                [0x72, 0xf8, 0xf6, 0x64, 0x86, 0x68, 0x98, 0x16, 0xd4, 0xa4, 0x5c, 0xcc, 0x5d, 0x65, 0xb6, 0x92],
                [0x6c, 0x70, 0x48, 0x50, 0xfd, 0xed, 0xb9, 0xda, 0x5e, 0x15, 0x46, 0x57, 0xa7, 0x8d, 0x9d, 0x84],
                [0x90, 0xd8, 0xab, 0x00, 0x8c, 0xbc, 0xd3, 0x0a, 0xf7, 0xe4, 0x58, 0x05, 0xb8, 0xb3, 0x45, 0x06],
                [0xd0, 0x2c, 0x1e, 0x8f, 0xca, 0x3f, 0x0f, 0x02, 0xc1, 0xaf, 0xbd, 0x03, 0x01, 0x13, 0x8a, 0x6b],
                [0x3a, 0x91, 0x11, 0x41, 0x4f, 0x67, 0xdc, 0xea, 0x97, 0xf2, 0xcf, 0xce, 0xf0, 0xb4, 0xe6, 0x73],
                [0x96, 0xac, 0x74, 0x22, 0xe7, 0xad, 0x35, 0x85, 0xe2, 0xf9, 0x37, 0xe8, 0x1c, 0x75, 0xdf, 0x6e],
                [0x47, 0xf1, 0x1a, 0x71, 0x1d, 0x29, 0xc5, 0x89, 0x6f, 0xb7, 0x62, 0x0e, 0xaa, 0x18, 0xbe, 0x1b],
                [0xfc, 0x56, 0x3e, 0x4b, 0xc6, 0xd2, 0x79, 0x20, 0x9a, 0xdb, 0xc0, 0xfe, 0x78, 0xcd, 0x5a, 0xf4],
                [0x1f, 0xdd, 0xa8, 0x33, 0x88, 0x07, 0xc7, 0x31, 0xb1, 0x12, 0x10, 0x59, 0x27, 0x80, 0xec, 0x5f],
                [0x60, 0x51, 0x7f, 0xa9, 0x19, 0xb5, 0x4a, 0x0d, 0x2d, 0xe5, 0x7a, 0x9f, 0x93, 0xc9, 0x9c, 0xef],
                [0xa0, 0xe0, 0x3b, 0x4d, 0xae, 0x2a, 0xf5, 0xb0, 0xc8, 0xeb, 0xbb, 0x3c, 0x83, 0x53, 0x99, 0x61],
                [0x17, 0x2b, 0x04, 0x7e, 0xba, 0x77, 0xd6, 0x26, 0xe1, 0x69, 0x14, 0x63, 0x55, 0x21, 0x0c, 0x7d ]];
  
  /* rcon is part of AES, for key schedule. */
  const rcon = [0x00,0x01,0x02,0x04,0x08,0x10,0x20,0x40,0x80,0x1B,0x36]
  
  const rinv =  [[ 14, 11, 13, 9 ], [ 9, 14, 11, 13 ], [ 13, 9, 14, 11 ], [ 11, 13, 9, 14 ]]
  
  /* Initial permutation matrix for DES encryption.*/
  const DES_IP = [[58, 50, 42, 34, 26, 18, 10, 2],
                  [60, 52, 44, 36, 28, 20, 12, 4],
                  [62, 54, 46, 38, 30, 22, 14, 6],
                  [64, 56, 48, 40, 32, 24, 16, 8],
                  [57, 49, 41, 33, 25, 17, 9, 1],
                  [59, 51, 43, 35, 27, 19, 11, 3],
                  [61, 53, 45, 37, 29, 21, 13, 5],
                  [63, 55, 47, 39, 31, 23, 15, 7]] 
  
  const DES_FP = [[40, 8, 48, 16, 56, 24, 64, 32],
                  [39, 7, 47, 15, 55, 23, 63, 31],
                  [38, 6, 46, 14, 54, 22, 62, 30],
                  [37, 5, 45, 13, 53, 21, 61, 29],
                  [36, 4, 44, 12, 52, 20, 60, 28],
                  [35, 3, 43, 11, 51, 19, 59, 27],
                  [34, 2, 42, 10, 50, 18, 58, 26],
                  [33, 1, 41, 9, 49, 17, 57, 25]]
  
  /* Sboxes for S-DES */
  const SDES_Sbox0 = [[1,0,3,2],[3,2,1,0],[0,2,1,3],[3,1,3,1]]
  const SDES_Sbox1 = [[0,1,2,3],[2,0,1,3],[3,0,1,0],[2,1,0,3]]

  /* Sboxes for S-AES */
  const SAES_Sbox = [[0b1001,0b0100,0b1010,0b1011],[0b1101,0b0001,0b1000,0b0101],[0b0110,0b0010,0b0000,0b0011],[0b1100,0b1110,0b1111,0b0111]]

  /* Sboxes for mini-AES */
  const miniAES_Sbox = [[0b1110, 0b0100, 0b1101, 0b0001], [0b0010, 0b1111, 0b1011, 0b1000], [0b0011, 0b1010, 0b0110, 0b1100], [0b0101, 0b1001, 0b0000, 0b0111]]
  const miniAES_inverse_Sbox = [[0b1110, 0b0011,0b0100,0b1000],[0b0001,0b1100,0b1010,0b1111],[0b0111,0b1101,0b1001,0b0110],[0b1011,0b0010,0b0000,0b0101]]

  
  var fn_primelist = new funcObj('primelist',[TNum,TNum],TNum,
                                                   primelist,
                                                   {unwrapValues:true,
                                                    description: "List of Prime numbers"})
  fn_primelist.description = "Give a list of primes between the first and the second number, up to 5000."
  numbertheory.scope.addFunction(fn_primelist)
  
  numbertheory.scope.addFunction(new funcObj('powmod',[TNum,TNum,TNum],TNum,
                                                   powmod,
                                                   {unwrapValues:true}))
  
  numbertheory.scope.addFunction(new funcObj('primitiveroots',[TNum,TNum],TList,
                                                   primitiveRoots,
                                                   {unwrapValues:true}))
  
  numbertheory.scope.addFunction(new funcObj('discretelog',[TNum,TNum,TNum],TNum,
                                                   discretelog,
                                                   {unwrapValues:true}))
  
  numbertheory.scope.addFunction(new funcObj('invert',[TNum,TNum],TNum,
                                                 invert,
                                                 {unwrapValues:true}))
  
  numbertheory.scope.addFunction(new funcObj('bezout',[TNum,TNum],TList,
                                                 bezout,
                                                 {unwrapValues:true}))
  
  numbertheory.scope.addFunction(new funcObj('coprime',[TNum],TNum,
                                               coprime,
                                               {unwrapValues:true}))
  
  numbertheory.scope.addFunction(new funcObj('polyrepr',[TNum],TExpression,
                                               polyrepr,
                                               {unwrapValues:true}))
  
  numbertheory.scope.addFunction(new funcObj('fmod',[TNum,TNum],TNum,
                                               fmod,
                                               {unwrapValues:true}))
  
  numbertheory.scope.addFunction(new funcObj('fmul',[TNum,TNum],TNum,
                                               fmul,
                                               {unwrapValues:true}))
  numbertheory.scope.addFunction(new funcObj('fadd',[TNum,TNum],TNum,
                                               fadd,
                                               {unwrapValues:true}))
  numbertheory.scope.addFunction(new funcObj('fdiv',[TNum,TNum],TNum,
                                               fdiv,
                                               {unwrapValues:true}))
  
  numbertheory.scope.addFunction(new funcObj('eccadd',[TList,TList,TList],TList,
                                             eccadd,
                                               {unwrapValues:true}))
  
  /* Export AES functions */

  numbertheory.scope.addFunction(new funcObj('hexmatrixdisplay',[TList],TString,
                                             Hexmatrixdisplay,
                                               {unwrapValues:true}))
  numbertheory.scope.addFunction(new funcObj('mask',[TMatrix,TMatrix],TMatrix,
                                             mask,
                                               {unwrapValues:true}))
  numbertheory.scope.addFunction(new funcObj('bytesub',[TNum],TNum,
                                             bytesub,
                                               {unwrapValues:true}))
  numbertheory.scope.addFunction(new funcObj('invbytesub',[TNum],TNum,
                                             invbytesub,
                                               {unwrapValues:true}))
  numbertheory.scope.addFunction(new funcObj('shiftrow',[TMatrix],TMatrix,
                                             shiftrow,
                                               {unwrapValues:true}))
  numbertheory.scope.addFunction(new funcObj('invshiftrow',[TMatrix],TMatrix,
                                             invshiftrow,
                                               {unwrapValues:true}))
  numbertheory.scope.addFunction(new funcObj('mixcolumns',[TMatrix],TMatrix,
                                             mixcolumns,
                                               {unwrapValues:true}))
  numbertheory.scope.addFunction(new funcObj('invmixcolumns',[TMatrix],TMatrix,
                                             invmixcolumns,
                                               {unwrapValues:true}))
  numbertheory.scope.addFunction(new funcObj('addroundkey',[TMatrix,TMatrix],TMatrix,
                                             addroundkey,
                                             {unwrapValues:true}))
  numbertheory.scope.addFunction(new funcObj('rijndael',[TMatrix,TMatrix],TMatrix,
                                             rijndael,
                                             {unwrapValues:true}))
  numbertheory.scope.addFunction(new funcObj('invrijndael',[TMatrix,TMatrix],TMatrix,
                                             invrijndael,
                                             {unwrapValues:true}))
  numbertheory.scope.addFunction(new funcObj('rotword',[TMatrix],TMatrix,
                                             rotword,
                                               {unwrapValues:true}))
  numbertheory.scope.addFunction(new funcObj('subword',[TMatrix],TMatrix,
                                             subword,
                                               {unwrapValues:true}))
  numbertheory.scope.addFunction(new funcObj('rcon',[TMatrix,TMatrix,TNum],TMatrix,
                                             rconstep,
                                               {unwrapValues:true}))
  numbertheory.scope.addFunction(new funcObj('keyschedule',[TMatrix,TNum],TMatrix,
                                             keyschedule,
                                               {unwrapValues:true}))
  numbertheory.scope.addFunction(new funcObj('rconv',[TNum],TNum,
                                             x =>rcon[x],
                                               {unwrapValues:true}))
  
  /* Export Mini-AES functions */
  numbertheory.scope.addFunction(new funcObj('substitutenibble',[TNum],TNum,
                                             miniAES_substitutenibble,
                                               {unwrapValues:true}))
    numbertheory.scope.addFunction(new funcObj('inversenibble',[TNum],TNum,
                                             miniAES_inversesubstitutenibble,
                                               {unwrapValues:true}))
    numbertheory.scope.addFunction(new funcObj('miniAES_mixcolums',[TMatrix],TMatrix,
                                             miniAES_mixcolumns,
                                               {unwrapValues:true}))
  
  /* Export S-DES functions */
  numbertheory.scope.addFunction(new funcObj('permute',[TNum,TList],[TNum],permute,{unwrapValues: true}))
  numbertheory.scope.addFunction(new funcObj('sdes_permute_initial',[TNum],[TNum],sdes_permute_initial,{unwrapValues: true}))
  numbertheory.scope.addFunction(new funcObj('sdes_permute_final',[TNum],[TNum],sdes_permute_final,{unwrapValues: true}))
  numbertheory.scope.addFunction(new funcObj('sdes_getsubkeys',[TNum],[TNum,TNum],sdes_getsubkeys,{unwrapValues: true}))
  numbertheory.scope.addFunction(new funcObj('splitbinary',[TNum,TNum],[TNum,TNum],splitbinary,{unwrapValues: true}))
  numbertheory.scope.addFunction(new funcObj('expand',[TNum,TNum,TNum],[TNum],expand,{unwrapValues: true}))
  numbertheory.scope.addFunction(new funcObj('sdes_expand',[TNum],[TNum],sdes_expand,{unwrapValues: true}))
  numbertheory.scope.addFunction(new funcObj('sdes_addsubkey',[TNum,TNum],[TNum],sdes_addsubkey,{unwrapValues: true}))
  numbertheory.scope.addFunction(new funcObj('sdes_sboxsub',[TNum],[TNum],sdes_sboxsub,{unwrapValues: true}))
  numbertheory.scope.addFunction(new funcObj('sdes_permutesub',[TNum],[TNum],sdes_permutesub,{unwrapValues: true}))
  numbertheory.scope.addFunction(new funcObj('binaryxor',[TNum,TNum],[TNum],binaryxor,{unwrapValues: true}))
  numbertheory.scope.addFunction(new funcObj('sdes_feistel',[TNum,TNum],[TNum],sdes_feistel,{unwrapValues: true}))






  
  /**
  * Generate a list of primes.
  */
  function primelist(m,n){return primes.filter(x => x >= m && x<=n);}
  
  /** Basic Mod arithmetic functions **/
  /**
   * Compute k mod p
   */
  function mod(k,p){
    var r=k%p
    if(r<0){return p+r}else{return r}
  }


  /**
  * Solve $x^y (mod p)$
  **/
  function powmod(x,y,p){
    if(y==0) return 1;
    r = powmod(x,Math.trunc(y/2),p);
    r1 = (r*r)%p
    if(y%2 == 0) { r2 = 1} else {r2 = x%p}
    return (r1*r2)%p
  }
  
  /**
  * Get the first n primitive roots of p
  */
  function primitiveRoots(p,n){
    /* Consider only the numbers such that x^((p-1)/2) = -1 (mod p)) */
    let l = [...Array(p).keys()].filter(x => powmod(x,(p-1)/2,p) == p-1);
    let v = []
    for(let z of l){
      if(z>1){
        let i=0;
        for(i=2; i<p-1 && powmod(z,i,p) != 1; i++);
        if(i==p-1) { v.push(z)}
        if(v.length >= n) return v
       }
    }
    return v
  }
  
  /*
  * Solve the discrete logarithm problem of finding x in the equation k^x= y (mod p).
  * Not a particularly smart algorithm. Will definitely fail for moderately large values.
  * Will return 0 if there is no such value.
  */
  function discretelog(k,y,p){
    for(var i=1;i<p-1;i++){
      if(powmod(k,i,p) == y) return i;
    }
    return 0;
  }
  
 function extendedEuclidean(a,b,x,y){
   if (b==0) return [a,x]
   m = Math.floor(a/b)
   return extendedEuclidean(b,a-b*m,y,x-y*m)
 }
  
  /*
  * Return the Bézout coefficients of two numbers
  */
  function bezout(a,b){
    return extendedEuclidean(a,b,0,1)[1]
  }
  
  /*
  * Find the inverse of k mod p.
  */
  function invert(x,p){
    k = mod(x,p)
    var v = mod(bezout(p,k),p)
    if((v*k)%p == 1) return v
    else return new TNum("Infinity")
  }
  
  /*
  * Find the gcd of two numbers
  */
  function gcd(a,b){
    if(b==0) return a
    return gcd(b,a%b)
  }
  
  /*
  * Find the lcm of two numbers
  */
  function lcm(a,b){
    return a*b/gcd(a,b)
  }
  
  /*
  * Find coprimes to p that are less than p.
  */
  function coprime(p) {
    return [...Array(p).keys()].filter(x => gcd(p,x) == 1);
  }
  
  /*
  * Find the Euler totient function of pq, where p and q are primes.
  */
  function totient(p,q){
    return (p-1)*(q-1)
  }

  /*
  * Find the Carmichael totient function of pq, where p and q are primes.
  */
  function carmichael(p,q){
    return gcd(p-1,q-1)
  }
 
  /*** Finite Fields ***/
  /**
   * Get the polynomial representation of a number in ℤ₂[x].
   */
  function polyrepr(x){
    var expr = "0"
    for(i=0;2**i<x+1;i++){
      if((x>>i)&1){expr = "x^(" + i +")+" + expr}
    }
    return new TExpression(expr)
  }
  
  /*
   * fmod = mod operation in the field ℤ₂[x]
   */
  function fmod(a,r){
    var m,n;
    var b=a;
    for(m=0;2**(m+1)<=r;m++);
    for(n=0;2**(n+1)<=b;n++);
    if(n<m){return b}
    for(var i=n;i>=m;i--){
     if(b&(2**i)){b = b^(r << (i-m)) }
    }
    return b
  }

  /*
  * fmul - multiplication in the field ℤ₂[x] (mod r).
  */
  function fmul(a,n){
  /* Start by multiplying the two numbers.*/
  var c = 0;//a*n;
  for(var i=0;2**i<=n;i++){
    if(n&(2**i)) {c ^= (a<<i)}
  }
  return c
  }
  
  /**
   * fdiv - division of a by b in ℤ₂[x]
   */
  function fdiv(a,b){
    if(a==0){return 0}
    if(b==0){return -1}
    var m,n;
    for(m=0;2**(m+1)<=b;m++);
    for(n=0;2**(n+1)<=a;n++);
    if(n<m){return 0}
    return 2**(n-m)+fdiv(a^(b<<(n-m)),b)
  }
  
  /**
   * fadd - addition (and difference) of a and bin ℤ₂[x]
   */
  function fadd(a,b){return a^b}

  
  /*** Elliptic Curves ***/
  
  /** 
  * Add points on an elliptic curve.
  */
  function eccadd(p1,p2, [a,b,c,m]){
    var v,s,x,y
    if(p2[0]!= p1[0]){
    v = invert(p2[0]-p1[0],m)
    s = mod(v*(p2[1]-p1[1]),m)
    x = s*s-p1[0]-p2[0]
    y = s*(p1[0]-x)-p1[1]
    }
    else {
      if(p2[0] == -p1[0]){
        return [p1[0],0] // I don't think that this is correct??
      }
      else {
        v = invert(2*p1[1],m)
        s = mod(v*(3*p1[0]*p1[0]+b),m)
        x = s*s-p1[0]-p2[0]
        y = s*(p1[0]-x)-p1[1]
      }
    }
    return [mod(x,m),mod(y,m)]
  }
  
  /** AES **/
  
  /*
  * Display Hex Matrix in Latex form
  */
  function Hexmatrixdisplay(M){
    var txt = "\\begin{pmatrix}"
    txt += M.map(r => r.join(" & ")).join("\\\\");
    txt += "\\end{pmatrix}"
    return txt
  }
  
  /*
   * create an empty matrix of the same size as `M`
   */
  function emptyMatrix(M){
    var A = M
    if(M.value){A = M.value}
    var B = Array(A.rows).fill().map(()=>Array(A.columns).fill())
    B.rows = A.rows
    B.columns = A.columns
    return B
  }
  
  /*
  * mask - perform a pairwise x-or operation of two matrices.
  */
  function mask(P,M){
    var A = P
    if(P.value){A = P.value}
    var B = emptyMatrix(A)
    for(i=0;i<A.rows;i++){
      for(j=0;j<A.columns;j++){
        B[i][j] = A[i][j] ^ M[i][j]
      }
    }
    return new TMatrix(B);
  }
  
  /*
  * Transformation of a byte `a` to a byte `b` by means of a S-box.
  */
  function sboxsubstitute(a,Sbox){
    /* Get the dimension of the box */
    var nr = Sbox.length
    /* We split the byte into two parts.*/
    var r = (a>>Math.log2(nr)); /* left part */
    var c = (a&(nr-1)); /*  right part */
    return Sbox[r][c]
  }
  /*
  * ByteSub is the transformation of a byte `a` to a byte `b` by means of the S-box.
  * It is the first step of the Rijndael round.
  */
  function bytesub(a){ return sboxsubstitute(a,Sbox) }
  /*
  * InvByteSub is the transformation of a byte `a` to a byte `b` by means of the inverse S-box.
  * It is the first step of the Rijndael round.
  */
  function invbytesub(a){ return sboxsubstitute(a,Sinv) }
  
  /*
  * ShiftRow shifts the elements in the row to the left, as follows:
  * 1. First row (row 0) is shifted by 0 element;
  * 2. Second row (row 1) is shifted by 1 element;
  * 3. Third row (row 2) is shifted by 2 elements;
  * 4. Fourth row (row 3) is shifted by 3 elements.
  *
  * This is the second step in the Rijndael round.
  */
  function shiftrow(M){
    var A = M
    if(M.value){A = M.value}
    var B = emptyMatrix(A)
    for(var i=0;i<A.rows;i++){
      for(var j=0;j<A.columns;j++){
        B[i][j] = A[i][(j+i)%(A.columns)]
      }
    }
    return new TMatrix(B)
  }
  
  /*
  * InvShiftRow shifts the elements in the row to the left, as follows:
  * 1. First row (row 0) is shifted by 0 element;
  * 2. Second row (row 1) is shifted by 1 element;
  * 3. Third row (row 2) is shifted by 2 elements;
  * 4. Fourth row (row 3) is shifted by 3 elements.
  *
  * This is the second step in the Rijndael round.
  */
  function invshiftrow(M){
    var A = M
    if(M.value){A = M.value}
    var B = emptyMatrix(A)
    for(var i=0;i<A.rows;i++){
      for(var j=0;j<A.columns;j++){
        B[i][j] = A[i][(j+4-i)%4]
      }
    }
    return new TMatrix(B)
  }
  
  /*
  * rmul - multiplication in the Rijndael group.
  */
  function rmul(a,n){
  /* Start by multiplying the two numbers.*/
  var c = 0;//a*n;
  for(var i=0;i<8;i++){
    if(n&(2**i)) {c ^= (a<<i)}
  }
  /* Then apply the modulo by x^8+x^4+x^2+x */
  for(var i=0;i<8;i++){
    if(c&(2**(i+8))){c = c^(283 << i) }
  }
  return c
  }
  
  /*
  * rmatmul - matrix multiplication in the Rijndael group.
  */
  function rmatmul(M,P){
    var A = P
    if(P.value){A = P.value}
     var B = emptyMatrix(A)
     for(var i=0;i<A.rows;i++){
       for(var j=0;j<A.columns;j++){
         for(var k=0;k<A.rows;k++){
           B[i][j] ^= rmul(A[k][j],M[i][k])
         }
       }
     }
     return new TMatrix(B);
  }
  
  /*
  * MixColumns - apply the MixColumns transformation in the AES encryption standard.
  */
  function mixcolumns(A){
    const R = [[2,3,1,1],[1,2,3,1],[1,1,2,3],[3,1,1,2]]
    return rmatmul(R,A)
  }
  
  /*
  * MixColumns - apply the inverse MixColumns transformation in the AES encryption standard.
  */
  function invmixcolumns(A){
    const R = [[14,11,13,9],[9,14,11,13],[13,9,14,11],[11,13,9,14]]
    return rmatmul(R,A)
  }
  
  /*
  * AddRoundKey - X-Or two matrices, one with a message and the other the round key.
  * 
  * @param {Matrix} P - the message to apply the roundkey on.
  * @param {Matrix} M - round key.
  * @return {Matrix} - the output of the round key.
  */
  function addroundkey(P,M){
    var A = P
    if(P.value){A = P.value}
    var B = emptyMatrix(A)
    for(var i=0;i<A.rows;i++){
      for(var j=0;j<A.columns;j++){
        B[i][j] = A[i][j] ^ M[i][j]
      }
    }
    return new TMatrix(B)
  }
  
  /*
  * Rijndael Round - a full round of Rijndael for AES.
  * 
  * @param {Matrix} A - the message to apply the round on.
  * @param {Matrix} M - encryption key.
  * @return {Matrix} - the output of the round.
  */
  function rijndael(A,M){
    var B = A.map(x=>x.map(bytesub))
    B.rows = A.rows; B.columns = A.columns
    return addroundkey(mixcolumns(shiftrow(B)),M)
  }
  
  /*
  * Inverse Rijndael Round - for decryption.
  */
  function invrijndael(A,M){
     var B = invshiftrow(invmixcolumns(addroundkey(A,M)));
     var C = B.map(x=>x.map(invbytesub))
     C.rows = B.rows
     C.columns = B.columns
     return C
  }
  
  /*
  * RotWord - Rotate the bytes of the last column.
  * First step of the AES Key Schedule for 128 bits keys.
  */
  function rotword(A){
    var B = Array(A.length).fill()
    for(i=0;i<4;i++){
      B[i] = A[(i+1)%4][3]
    }
    return [B];
  }
  
  /*
  * SubWord - apply the S-box transform to the column.
  */
  function subword(b){
    return [b[0].map(bytesub)];
  }
  
  /*
  * rconstep - compute the new key from the old key and the mask
  * obtained from subword and rotword.
  */
  function rconstep(A,b,s){
    var B = emptyMatrix(A)
    for(var i=0;i<A.rows;i++){
      B[i][0] = A[i][0] ^ b[0][i]// ^ rcon[s]
    }
    B[0][0] = B[0][0] ^ rcon[s]
    for(var i=0;i<A.rows;i++){
      for(var j=1;j<A.columns;j++){
        B[i][j] = A[i][j] ^ B[i][j-1]
      }
    }
    return new TMatrix(B)
  }
  
  /* 
  * keyschedule - apply one step of the key schedule for AES
  */
  function keyschedule(A,s){
    return rconstep(A,subword(rotword(A)),s)
  }
  
  /*** END AES CODE ***/
  function miniAES_substitutenibble(a){ return sboxsubstitute(a,miniAES_Sbox) }
  function miniAES_inversesubstitutenibble(a){ return sboxsubstitute(a,miniAES_inverse_Sbox) }

  
    /*
  * rmul - multiplication in the Rijndael group.
  */
  function miniAES_rmul(a,n){
  /* Start by multiplying the two numbers.*/
  var c = 0;//a*n;
  for(var i=0;i<4;i++){
    if(n&(2**i)) {c ^= (a<<i)}
  }
  /* Then apply the modulo by x^3+x^2+1*/
  for(var i=0;i<4;i++){
    if(c&(2**(i+4))){c = c^(19 << i) }
  }
  return c
  }
  
  /*
  * rmatmul - matrix multiplication in the Rijndael group.
  */
  function miniAES_rmatmul(M,P){
    var A = P
    if(P.value){A = P.value}
     var B = emptyMatrix(A)
     for(var i=0;i<A.rows;i++){
       for(var j=0;j<A.columns;j++){
         for(var k=0;k<A.rows;k++){
           B[i][j] ^= miniAES_rmul(A[k][j],M[i][k])
         }
       }
     }
     return new TMatrix(B);
  }
  
  
     /*
  * MixColumns - apply the MixColumns transformation in the AES encryption standard.
  */
  function miniAES_mixcolumns(A){
    const R = [[3,2],[2,3]];
    return miniAES_rmatmul(R,A)
  }
  
  /*
  * MixColumns - apply the inverse MixColumns transformation in the AES encryption standard.
  */
  function miniAES_invmixcolumns(A){
    const R = [[7,6],[6,7]]
    return rmatmul(R,A)
  }

  /*** S-AES ***/

  /* S-AES shares many components with AES, with fewer rounds and smaller keys. */

  /*
  * SubstituteNibble is the transformation of a byte `a` to a byte `b` by means of the S-box
  * for S-AES.
  */
  function substitutenibble(a){ return sboxsubstitute(a,SAES_Sbox) }
  

  

  /* Run a full S-AES round on the message */
  function saes_round(A,M){
    var B = A.map(x=>x.map(substitutenibble))
    B.rows = A.rows; B.columns = A.columns
    return addroundkey(mixcolumns(shiftrow(B)),M)
  }

  
  /*** S-DES CODE ***/
  function permute(m,p){
    var result = 0;
    var l = 2**(Math.max.apply(0,p))
    for (var x of p)
    { 
      result <<= 1
      if ( m & (l>>x))
          result |= 1;
    }
    return result
  }

  function sdes_permute_initial(m) {return permute(m,[1,5,2,0,3,7,4,6])}
  function sdes_permute_final(m) {return permute(m,[3,0,2,4,6,1,7,5])}
  function sdes_getsubkeys(k) {
    var p = permute(k,[2,4,1,6,3,9,0,8,7,5])
    /* Subkey 1 */
    var p1 = permute(permute(p,[1,2,3,4,0,6,7,8,9,5]),[5,2,6,3,7,4,9,8])
    var p2 = permute(permute(p,[3,4,0,1,2,8,9,5,6,7]),[5,2,6,3,7,4,9,8])
    return [p1,p2]
    }

  function expand(m,k,l){return m<<l|k}
  
  function splitbinary(m,nbits){
    var r = m >> nbits
    var l = m - (r<<nbits)
    return [r,l]
  }
  function sdes_expand(m){return expand(permute(m,[3,0,1,2]),permute(m,[1,2,3,0]),4)}
  function sdes_addsubkey(m,k){return m^k}
  function sdes_getsboxvalue(m,s){
    var r = ((m>>2)&2) | (m&1) // Row comes from outer bits
    var c = (m>>1) & (2**2 -1)     // Column comes from inner bits
    return s[r][c]
  }
  function sdes_sboxsub(m){
    /* Start by splitting into left and right parts */
    [l,r] = splitbinary(m,4)
    return expand(sdes_getsboxvalue(l,SDES_Sbox0),sdes_getsboxvalue(r,SDES_Sbox1),2)
  }
  function sdes_permutesub(m) { return permute(m,[1,3,2,0])}
  function binaryxor(x,y){return x^y}
  function sdes_feistel(m,k){
    var [mleft,mright] = splitbinary(m,4)
    var step1 = sdes_expand(mright)
    var step2 = sdes_addsubkey(step1,k)
    var step3 = sdes_sboxsub(step2)
    var step4 = sdes_permutesub(step3)
    var step5 = binaryxor(mleft,step4)
    return expand(step5,mright,4)
  }
  /*** END S-DES CODE ***/
});