Numbas.addExtension('elliptic-curves',['base','util','jme','display'],function(ellipticCurves) {
  
    var ellipticcurvesScope = ellipticCurves.scope;

    var parse = Numbas.jme.parse;
    var funcObj = Numbas.jme.funcObj;
    var TBool = Numbas.jme.types.TBool;
    var TNum = Numbas.jme.types.TNum;
    var TString = Numbas.jme.types.TString;
    var TList = Numbas.jme.types.TList;
    var TExpr = Numbas.jme.types.TExpression;

    // Define the constructor for a new data type representing an elliptic curve.
    // `a`, `b` and `c` are the coefficients in the formula y^2 = ax^3+bx+c.
    function TEcc(parameters) {
      this.value = parameters;
      this.a = parameters['a']
      this.b = parameters['b']
      this.c = parameters['c']
      this.m = parameters['m']
    }
    TEcc.prototype.type = 'ecc';
  
    function TEccFromTree(expr,modulo){
      var scope = Numbas.jme.builtinScope;
      var rule = '(y^2)`? = '+
                   '('+
                   '(`+- $n);a * x^3 `| x^3;a:1 `| -x^3;a:-1'+
                   ' `|  '+
                   '(`+- $n);b * x `| x;b:1 `| -x;b:-1'+
                   ' `| '+
                   '(`+- $n);c' + ')`+ + $z'
      const pattern_elliptic_curve = new Numbas.jme.rules.Rule(rule,null,'acl','curve1');
      var m = pattern_elliptic_curve.match(expr,scope)
      if(!m){throw(new Error('Not an elliptic curve'))}
      curve = {
        'a': m['a'] ? m['a'].tok : 0,
        'b': m['b'] ? m['b'].tok : 0,
        'c': m['c'] ? m['c'].tok : 0,
        'm' : modulo}
      return curve
    }
     
    // Code to render a Elliptic Curve as LaTeX
    Numbas.jme.display.typeToTeX.ecc = function(thing,tok,texArgs,settings) {
        var out = 'y^2 ='
        if(tok.a==-1){out += '-'}
        else if (tok.a!=1){out+= tok.a}
        out += 'x^3'
        if (tok.b!=0){
          if (tok.b<0) {out += '-'}
          if (tok.b>0) {out += '+'}
          if (Math.abs(tok.b) != 1) {out += Math.abs(tok.b)}
          out += 'x'
        }
        if (tok.c!=0){
          if (tok.c<0) {out += '-'}
          if (tok.c>0) {out += '+'}
          out += Math.abs(tok.c)
        }
        if(tok.m != undefined){
          out += "\\pmod{"+tok.m+"}"
        }
        return out;
    }

    // Code to render an elliptic curve as a string
    const curvetotext = Numbas.jme.typeToDisplayString.ecc = function(tok) {
        var out = 'y^2 = ' + tok.a + '*x^3+' + tok.b + '*x+' + tok.c;
        return out;
    }
  
    Numbas.jme.display.typeToJME.ecc = function(tree,tok,bits,settings) {
        var out = 'curve(' + tok.a + ', ' + tok.b + ', ' + tok.c + ', ' + tok.m +')';
        return out;
    }
  
    // We also define a constructor for points on a curve.
    function TEccpoint(parameters){
      if(parameters.value == "Infinity" || parameters == "Infinity" || (parameters["x"] == 0 && parameters["y"]==0)){
        this.value = "Infinity"
        this.x = 0
        this.y = 0 
        this.curve = parameters["curve"]
      } else {
      this.value = parameters}
      this.x = parameters['x']
      this.y = parameters['y']
      this.curve = parameters['curve']
    }
  
    TEccpoint.prototype.type = 'eccpoint';  
  
    // Code to render a Elliptic Curve as LaTeX
    Numbas.jme.display.typeToTeX.eccpoint = function(thing,tok,texArgs,settings) {
        if(isinfinity(tok)) { return "\\infty"}
        var out = '(' + tok.x + ', ' + tok.y +')'
        return out;
    }

    // Code to render a chemical formula as a JME expression
    Numbas.jme.typeToDisplayString.eccpoint = function(tok) {
        if(isinfinity(tok)) { return "∞"}
        var out = '(' + tok.x + ', ' + tok.y +')'
        return out;
    }
    
    Numbas.jme.display.typeToJME.eccpoint = function(tree,tok,bits,settings) {      
        if(isinfinity(tok)) { return "point(\"Infinity\")"}
        var curve = 'curve(' + tok.curve.a + ', ' + tok.curve.b + ', ' + tok.curve.c + ', ' + tok.curve.m +')';
        var out = 'point(' + tok.x + ', ' + tok.y +',' + curve + ")"
        return out
    }
  
    function eccequal(c1,c2){
      if (c1.m == undefined){
        if(c2.m == undefined){
           return c1.a == c2.a && c1.b == c2.b && c1.c==c2.c
        }
        else{return false}
      }
      return c1.a == c2.a && c1.b == c2.b && c1.c==c2.c && c1.m==c2.m
    }
  
    Numbas.util.equalityTests['ecc'] = function(c1,c2) {
      if (c1.m == undefined){
        if(c2.m == undefined){
           return c1.a == c2.a && c1.b == c2.b && c1.c==c2.c
        }
        else{return false}
      }
      return c1.a == c2.a && c1.b == c2.b && c1.c==c2.c && c1.m==c2.m
    };
  
      Numbas.util.equalityTests['eccpoint'] = function(p1,p2) {
      if(isinfinity(p1)) {return isinfinity(p2)}
      var c1 = p1.curve
      var c2 = p2.curve
      if( c1.a == c2.a && c1.b == c2.b && c1.c==c2.c && c1.m==c2.m){
      return p1.x == p2.x && p1.y == p2.y}
      else{return false}
    };
  
    ellipticcurvesScope.addFunction(new funcObj('=',[TEcc,TEcc],TEcc,function(c1,c2) {
      if (c1.m == undefined){
        if(c2.m == undefined){
           return c1.a == c2.a && c1.b == c2.b && c1.c==c2.c
        }
        else{return false}
      }
      return c1.a == c2.a && c1.b == c2.b && c1.c==c2.c && c1.m==c2.m
    }));
  
    ellipticcurvesScope.addFunction(new funcObj('=',[TEccpoint,TEccpoint],TEccpoint,function(p1,p2) {
      if(isinfinity(p1)) {return isinfinity(p2)}
      var c1 = p1.curve
      var c2 = p2.curve
      if( c1.a == c2.a && c1.b == c2.b && c1.c==c2.c && c1.m==c2.m){
      return p1.x == p2.x && p1.y == p2.y}
      else{return false}
    }));

    ellipticcurvesScope.addFunction(new funcObj('curve',[TNum,TNum,TNum],TEcc,function(a,b,c){
      return {'a': a, 'b': b, 'c': c}
    }));
  
    ellipticcurvesScope.addFunction(new funcObj('curve',[TNum,TNum,TNum,TNum],TEcc,function(a,b,c,m){
    return {'a': a, 'b': b, 'c': c, 'm': m}
    }));
    
   ellipticcurvesScope.addFunction(new funcObj('point',[TNum,TNum,TEcc],TEccpoint,function(x,y,c){
      return {'x': x, 'y': y, 'curve': c}
    }));
  
   ellipticcurvesScope.addFunction(new funcObj('point',[TList,TEcc],TEccpoint,function(x,c){
      return {'x': x[0], 'y': x[1], 'curve': c}
    }));
  
    ellipticcurvesScope.addFunction(new funcObj('point',[TString,TEcc],TEccpoint,function(x,c){
      if(x!== "Infinity"){throw(new Error("Unrecognized Input"));}
      return {'value': "Infinity", 'x': 0, 'y': 0, 'curve': c}
    }));
  
      ellipticcurvesScope.addFunction(new funcObj('point',[TString],TEccpoint,function(x){
      if(x!== "Infinity"){throw(new Error("Unrecognized Input"));}
      return {'value': "Infinity", 'x': 0, 'y': 0}
    }));
  
   ellipticcurvesScope.addFunction(new funcObj('logme',[TNum],TList,function(c){
     console.log(c)
     return [c]
   }));

   function getpoints(c){
      var pts = []
      for(var x=0;x<c.m;x++){
        for(var y=0;y<c.m;y++){
            if(x!=0 || y!=0){
        	var z = mod(c.a*x*x*x+c.b*x+c.c,c.m)
        	if(mod(y*y,c.m) == z){pts.push(new TEccpoint({'x':x,'y':y,'curve':c}))}
            }
        }}
     return pts
    }
  
   ellipticcurvesScope.addFunction(new funcObj('toexpr',[TEcc],TExpr,function(c){
     return curvetotext(c)
   }));
                                               
   ellipticcurvesScope.addFunction(new funcObj('getpoints',[TEcc],TList,getpoints));
  
   ellipticcurvesScope.addFunction(new funcObj('get',[TEccpoint,TString],TNum,function(c,v){
      if(isinfinity(c)) { return "Infinity"}
      return c[v]
    }));
  
    ellipticcurvesScope.addFunction(new funcObj('coordinates',[TEccpoint],TList,function(c){
      if(isinfinity(c)) { return [new TNum("Infinity"),new TNum("Infinity")]}
      var cc = [new TNum(c["x"]),new TNum(c["y"])]
      return cc
    }));
  
    ellipticcurvesScope.addFunction(new funcObj('List',[TEccpoint],TList,function(c){
      if(isinfinity(c)) { return [new TNum("Infinity"),new TNum("Infinity")]}
      var cc = [new TNum(c["x"]),new TNum(c["y"])]
      return cc
    }));
  
    ellipticcurvesScope.addFunction(new funcObj('on',[TEccpoint,TEcc],TBool,function(p,c){
      var x = p["x"]
      var y = p["y"]
      var lhs = mod(y*y,c.m)
      var rhs = mod(c.a*x**3+c.b*x+c.c,c.m)
      return lhs==rhs
    }));
  
    ellipticcurvesScope.addFunction(new funcObj('on',[TList,TEcc],TBool,function(p,c){
      var x = p[0]
      var y = p[1]
      var lhs = mod(y*y,c.m)
      var rhs = mod(c.a*x**3+c.b*x+c.c,c.m)
      return lhs==rhs
    }));
  
  
  /**
  * Compute k mod p
  */
  function mod(k,p){
    if(p==undefined) return k;
    var r=k%p
    if(r<0){return p+r}else{return r}
  }
  
   function extendedEuclidean(a,b,x,y){
   if (b==0) return [a,x]
   m = Math.floor(a/b)
   return extendedEuclidean(b,a-b*m,y,x-y*m)
   }
  
  /**
  * Test if a point is at infinity.
  **/
  function isinfinity(p){
    return p.value == "Infinity" || p == "Infinity" || (p.x==0 && p.y==0)
  }
  
 /**
  * Find the gcd of two numbers
  */
  function gcd(a,b){
    if(b==0) return a
    return gcd(b,a%b)
  }
  
  /*
  * Find the inverse of k mod p.
  */
  function invert(x,p){
    var k=mod(x,p)
    if(gcd(k,p) != 1) {return "Infinity"} //Infinite.
    var x = extendedEuclidean(p,k,0,1)[1]
    return mod(x,p)
  }
   /** 
  * Add points on an elliptic curve.
  */
  function eccadd(p1,p2){
    if(isinfinity(p1)) { return p2}
    if(isinfinity(p2)) { return p1}
    //if(!eccequal(p1.curve,p2.curve)){ return TEccpoint(0,0)}
    var v,s,x,y
    var m = p1.curve.m
    if(p2.x!= p1.x){
    v = invert(p2.x-p1.x,m)
    //if(v=="Infinity"){return {"value":"Infinity","curve":p1.curve}}
    s = mod(v*(p2.y-p1.y),m)
    x = s*s-p1.x-p2.x
    y = s*(p1.x-x)-p1.y
    }
    else {
      if(p1.y != p2.y){return {"value":"Infinity", "x":0, "y":0 ,"curve":p1.curve};}
      v = invert(2*p1.y,m)
      if(v=="Infinity"){return {"value":"Infinity", "x":0, "y":0 ,"curve":p1.curve}}
      s = mod(v*(3*p1.x*p1.x+p1.curve.b),m)
      x = s*s-p1.x-p2.x
      y = s*(p1.x-x)-p1.y
    }
    return {'x': mod(x,m),'y':mod(y,m),'curve':p1.curve}
  }
  
  function negate(p){
    if(isinfinity(p)){return p}
    return {"x":p.x, "y": p.curve.m - p.y, "curve": p.curve}
  }
  
  ellipticcurvesScope.addFunction(new funcObj('-u',[TEccpoint],TEccpoint,negate))

  ellipticcurvesScope.addFunction(new funcObj('+',[TEccpoint,TEccpoint],TEccpoint,eccadd))
  ellipticcurvesScope.addFunction(new funcObj('-',[TEccpoint,TEccpoint],TEccpoint,function(p1,p2){
    return eccadd(p1,negate(p2))
  }))

  ellipticcurvesScope.addFunction(new funcObj('*',[TNum,TEccpoint],TEccpoint,function(n,p){
    var p1 = p
    for(var i=1;i<n;i++){
       p1 = eccadd(p1,p)
    }
    return p1
  }))
  
  ellipticcurvesScope.addFunction(new funcObj('order',[TEccpoint],TNum,function(p){
    var p1 = p
    var i=0
    const m=getpoints(p.curve).length
    while( !isinfinity(p1) && i<m+1){
       p1 = eccadd(p1,p)
      i=i+1
    }
    if(i==m){return "Infinity"};
    return i+1
  }))

  
});