(function() {
/** Parse a number with uncertainty attached, e.g. 0.0005(2).
 * The bit in brackets shows the uncertainty in the last digit - in the example above, the number is +/- 0.0002.
 */
function parse_uncertain_number(str) {
    return parseFloat(str.replace(/\(.*\)$/,''));
}

function encode_number(digit_glyphs) {
    return function(n) {
        return (n+'').replace(/\d/g,function(d){ d=parseInt(d); return digit_glyphs[d]; });
    }
}
var superscript = encode_number('⁰¹²³⁴⁵⁶⁷⁸⁹');
var subscript = encode_number('₀₁₂₃₄₅₆₇₈₉');

Numbas.addExtension('chemistry',['math','jme','jme-display'],function(chemistry) {
    var jme = Numbas.jme;

    var types = chemistry.types = {};

    chemistry.isotope_data = isotope_data;
    chemistry.periodic_table_data = periodic_table_data;
    chemistry.thermodynamic_data = thermodynamic_data;
    chemistry.parse_formula = parse_formula;

    /** Get data corresponding to an element with the given symbol
     * @param {String} symbol
     * @returns {periodic_table_data}
     */
    var element_with_symbol = chemistry.element_with_symbol = function(symbol) {
        return periodic_table_data.find(function(d) { return d.symbol == symbol; });
    }

    /** Get data corresponding to an element with the given name
     * @param {String} name
     * @returns {periodic_table_data}
     */
    var element_with_name = chemistry.element_with_name = function(name) {
        return periodic_table_data.find(function(d) { return d.name.toLowerCase() == name.toLowerCase(); });
    }

    /** Get data corresponding to an element with the given atomic number
     * @param {Number} number
     * @returns {periodic_table_data}
     */
    var element_by_number = chemistry.element_by_number = function(number) {
        return periodic_table_data.find(function(d) { return d.atomicNumber == number; });
    }

    /** Get data corresponding to a particular isotope
     * @param {Number} atomic_number
     * @param {Number} mass_number
     * @returns {isotope_data}
     */
    var get_isotope = chemistry.get_isotope = function(atomic_number,mass_number) {
        var atom = isotope_data[atomic_number];
        if(!atom) {
            throw(new Error("No data for atom number "+atomic_number));
        }
        if(mass_number===undefined) {
            var sorted_by_abundance = Object.values(atom).sort(function(a,b) {
                a = a['Isotopic Composition'];
                b = b['Isotopic Composition'];
                if(!a || !b) {
                    return a=='' ? b=='' ? 0 : 1 : -1;
                }
                a = parse_uncertain_number(a);
                b = parse_uncertain_number(b);
                return a>b ? -1 : a<b ? 1 : 0;
            });
            var most_common = sorted_by_abundance[0];
            return most_common;
        } else {
            var isotope = atom[mass_number];
            if(!isotope) {
                throw(new Error("No data for isotope of atom number "+atomic_number+" with mass number "+mass_number));
            }
            return isotope;
        }
    }

    /** Representation of an isotope of a particular element
     * @param {periodic_table_data} data
     * @param {isotope_data} [isotope] - if not given, the most common isotope will be used
     */
    var Atom = chemistry.Atom = function(data,isotope) {
        this.data = data;
        this.symbol = this.data.symbol;
        this.atomic_number = data.atomicNumber;
        this.isotope = get_isotope(this.atomic_number,isotope);
        this.isotope_specified = isotope!==undefined;
        this.mass_number = this.isotope['Mass Number'];
        this.relative_atomic_mass = parse_uncertain_number(this.isotope_specified ? this.isotope['Relative Atomic Mass'] : this.data.atomicMass);
    }
    Atom.prototype = {
        /** An English name for the atom
         * @returns {String}
         */
        name: function() {
            var name = this.data.name.toLowerCase();
            if(this.isotope_specified) {
                name += '-'+this.mass_number;
            }
            return name;
        },

        /** Mass numbers of common isotopes of this element
         * @returns {Array.<Number>}
         */
        isotopes: function() {
            return Object.keys(isotope_data[this.atomic_number]).map(function(m){return parseInt(m)});
        },

        /** A TeX representation of this atom.
         * If no isotope was specified, just the symbol is returned
         * @returns {TeX}
         */
        tex: function() {
            if(this.isotope_specified) {
                return '{}^{'+this.mass_number+'} \\mathrm{'+this.symbol+'}';
            } else {
                return '\\mathrm{'+this.symbol+'}';
            }
        },
        /** JME representation of this atom.
         * @returns {JME}
         */
        jme: function() {
            if(this.isotope_specified) {
                return 'atom("'+this.symbol+'",'+this.mass_number+')';
            } else {
                return 'atom("'+this.symbol+'")';
            }
        },
        /** String representation of this atom. If a mass number was specified, it's included as a superscript.
         * @returns {String}
         */
        string: function() {
            if(this.isotope_specified) {
                return superscript(this.mass_number)+this.symbol;
            } else {
                return this.symbol;
            }
        },
        /** Relative abundance of this isotope. 
         * If no isotope was specified, returns 1.
         * @returns {Number}
         */
        abundance: function() {
            if(this.isotope_specified) {
                return parse_uncertain_number(this.isotope['Isotopic Composition'] || '0');
            } else {
                return 1;
            }
        }
    }

    var TAtom = types.TAtom = function(atom) {
        this.value = atom;
    }
    TAtom.prototype = {
        type: 'atom',
    }

    jme.display.typeToTeX.atom = function(thing,tok,texArgs,settings) {
        return tok.value.tex();
    }
    jme.display.typeToJME.atom = function(tree,tok,bits,settings) {
        return tok.value.jme();
    }

    /** A chemical formula.
     * @see parse_formula
     */
    var Formula = chemistry.Formula = function(formula) {
        this.formula = formula;
    }
    Formula.prototype = {
        tex: function() {
            return formula_to_tex(this.formula);
        },
        jme: function() {
            return 'formula("'+Numbas.jme.escape(formula_to_string(this.formula))+'")';
        },
        plain_string: function() {
            return formula_to_string(this.formula);
        },
        string: function() {
            return display_formula(this.formula);
        },
        atom_counts: function() {
            return count_atoms(this.formula);
        },
        mass: function() {
            var t = 0;
            Object.entries(this.atom_counts()).forEach(function(c) {
                var symbol = c[0];
                var n = c[1];
                var atom = element_with_symbol(symbol);
                if(!atom) {
                    throw(new Error("Unknown element symbol "+symbol));
                }
                var atomicMass = parse_uncertain_number(atom.atomicMass);
                t += n * atomicMass;
            });
            return t;
        }
    }

    var TFormula = types.TFormula = function(formula) {
        this.value = formula;
    }
    TFormula.prototype = {
        type: 'chemical_formula'
    }

    jme.display.typeToTeX.chemical_formula = function(thing,tok,texArgs,settings) {
        return tok.value.tex();
    }
    jme.display.typeToJME.chemical_formula = function(tree,tok,bits,settings) {
        return tok.value.jme();
    }

	var funcObj = jme.funcObj;
	var TString = jme.types.TString;
	var TNum = jme.types.TNum;
	var TList = jme.types.TList;
	var TBool = jme.types.TBool;
    var TDict = jme.types.TDict;

    function addFunction(name,deps,outtype,fn,options) {
        var a = chemistry.scope.addFunction(new funcObj(name,deps,outtype,fn,options));
    };

    chemistry.scope.setVariable('periodic_table',jme.wrapValue([null].concat(periodic_table_data)));
    chemistry.scope.setVariable('isotope_data',jme.wrapValue(isotope_data));
    chemistry.scope.setVariable('thermodynamic_data',jme.wrapValue(thermodynamic_data));

    addFunction('atom',[TString],TAtom,function(name) {
        var data = element_with_symbol(name) || element_with_name(name);
        return new Atom(data);
    });
    addFunction('atom',[TNum],TAtom,function(number) {
        return new Atom(element_by_number(number));
    });
    addFunction('atom',[TString,TNum],TAtom,function(name,isotope) {
        var data = element_with_symbol(name) || element_with_name(name);
        return new Atom(data,isotope);
    });
    addFunction('atom',[TNum,TNum],TAtom,function(number,isotope) {
        return new Atom(element_by_number(number,isotope));
    });
    addFunction('name',[TAtom],TString,function(atom) {
        return atom.name();
    });
    addFunction('symbol',[TAtom],TString,function(atom) {
        return atom.symbol;
    });
    addFunction('string',[TAtom],TString,function(atom) {
        return atom.string();
    });
    addFunction('listval',[TAtom,TString],'?',function(atom,key) {
        return atom.data[key] || atom.isotope[key];
    },{unwrapValues:true});
    addFunction('atomic_number',[TAtom],TNum,function(atom) {
        return atom.atomic_number;
    });
    addFunction('mass_number',[TAtom],TNum,function(atom) {
        return atom.mass_number;
    });
    addFunction('neutrons',[TAtom],TNum,function(atom) {
        return atom.mass_number - atom.atomic_number;
    });
    addFunction('relative_mass',[TAtom],TNum,function(atom) {
        return atom.relative_atomic_mass;
    });
    addFunction('isotopes',[TAtom],TList,function(atom) {
        return atom.isotopes().map(function(m){return new TNum(m)});
    });
    addFunction('abundance',[TAtom],TNum,function(atom) {
        return atom.abundance();
    });

    addFunction('formula',[TString],TFormula,function(str) {
        return new Formula(parse_formula(str));
    });

    addFunction('string',[TFormula],TString,function(formula) {
        return formula.string();
    });
    addFunction('plain_string',[TFormula],TString,function(formula) {
        return formula.plain_string();
    });

    addFunction('atom_counts',[TFormula],TDict,function(formula) {
        return jme.wrapValue(formula.atom_counts());
    },{unwrapValues:true});

    addFunction('mass',[TFormula],TNum,function(formula) {
        return formula.mass();
    })

    addFunction('thermodynamic_data',[TString,TString],TDict,function(name,state) {
        var data = thermodynamic_data.find(function(d) {
            return (d.name==name || d.formula==name) && d.state==state;
        })
        return jme.wrapValue(data);
    },{unwrapValues: true});
});
var thermodynamic_data = [
  {
"name":"aluminium","formula":"Al","state":"g","Hfg":326,"Gfg":286,"Smg":164.4,"Cpm":"n/a"
},
{"name":"aluminium","formula":"Al","state":"c","Hfg":0,"Gfg":0,"Smg":28.3,"Cpm":24.33 
},
{"name":"aluminium oxide","formula":"Al2O3","state":"c","Hfg":-1676,"Gfg":-1583,"Smg":50.9,"Cpm":78.99 
},
{"name":"barium","formula":"Ba","state":"c","Hfg":0,"Gfg":0,"Smg":67,"Cpm":26.35
},
{"name":"barium chloride","formula":"BaCl2","state":"c","Hfg":-860.1,"Gfg":-810.9,"Smg":126,"Cpm":"n/a" 
},
{"name":"barium sulfate","formula":"BaSO4","state":"c","Hfg":-1465,"Gfg":-1353,"Smg":132.2,"Cpm":101.75 
},
{"name":"boron","formula":"B","state":"c","Hfg":0,"Gfg":0,"Smg":5.86,"Cpm":11.97 
},
{"name":"boron","formula":"B","state":"amorphous","Hfg":3.77,"Gfg":3.56,"Smg":6.53,"Cpm":"n/a" 
},
{"name":"boron trifluoride","formula":"BF3","state":"g","Hfg":-1137,"Gfg":-1120,"Smg":254.0,"Cpm":50.46 
},
{"name":"diborane","formula":"B2H6","state":"g","Hfg":35.56,"Gfg":86.6,"Smg":232.0,"Cpm":56.4 
},
{"name":"bromine","formula":"Br","state":"g","Hfg":111.9,"Gfg":82.43,"Smg":174.9,"Cpm":20.78 
},
{"name":"bromine","formula":"Br2","state":"g","Hfg":30.7,"Gfg":9.15,"Smg":245.3,"Cpm":35.98 
},
{"name":"bromine","formula":"Br2","state":"l","Hfg":0,"Gfg":0,"Smg":152.2,"Cpm":"n/a" 
},
{"name":"hydrogen bromide","formula":"HBr","state":"g","Hfg":-36.40,"Gfg":-53.43,"Smg":198.6,"Cpm":29.12 
},
{"name":"calcium","formula":"Ca","state":"c","Hfg":0,"Gfg":0,"Smg":41.6,"Cpm":26.38 
},
{"name":"calcium fluoride","formula":"CaF","state":"c","Hfg":-1214.6,"Gfg":-1162.4,"Smg":68.91,"Cpm":87.03 
},
{"name":"calcium carbonate","formula":"CaCO3","state":"c","Hfg":-1206,"Gfg":-1129,"Smg":92.9,"Cpm":81.88 
},
{"name":"calcium chloride","formula":"CaCl2","state":"c","Hfg":-795,"Gfg":-750.2,"Smg":113.8,"Cpm":73.47 
},
{"name":"calcium oxide","formula":"CaO","state":"c","Hfg":-635.6,"Gfg":-604.2,"Smg":39.7,"Cpm":42.80 
},
{"name":"calcium hydroxide","formula":"Ca(OH)2","state":"c","Hfg":-986.6,"Gfg":-896.8,"Smg":76.1,"Cpm":84.52 
},
{"name":"carbon","formula":"C","state":"g","Hfg":716.7,"Gfg":671.3,"Smg":158.0,"Cpm":20.83 
},
{"name":"carbon","formula":"C","state":"graphite","Hfg":0,"Gfg":0,"Smg":5.7,"Cpm":8.64 
},
{"name":"carbon","formula":"C","state":"diamond","Hfg":1.896,"Gfg":2.868,"Smg":2.44,"Cpm":6.06 
},
{"name":"carbon dioxide","formula":"CO2","state":"g","Hfg":-393.5,"Gfg":-394.4,"Smg":213.6,"Cpm":37.13 
},
{"name":"carbon disulfide","formula":"CS2","state":"l","Hfg":89.70,"Gfg":65.27,"Smg":151.3,"Cpm":75.73 
},
{"name":"carbon disulfide","formula":"CS2","state":"g","Hfg":117.4,"Gfg":67.15,"Smg":237.7,"Cpm":"n/a" 
},
{"name":"chlorine","formula":"Cl","state":"g","Hfg":121.7,"Gfg":105.7,"Smg":165.1,"Cpm":"n/a" 
},
{"name":"chlorine","formula":"Cl2","state":"g","Hfg":0,"Gfg":0,"Smg":233.0,"Cpm":33.93 
},
{"name":"hydrogen chloride","formula":"HCl","state":"g","Hfg":-92.30,"Gfg":-95.27,"Smg":186.8,"Cpm":29.12 
},
{"name":"copper","formula":"Cu","state":"c","Hfg":0,"Gfg":0,"Smg":33.3,"Cpm":24.46 
},
{"name":"copper(II) oxide","formula":"CuO","state":"c","Hfg":-155.2,"Gfg":-127.2,"Smg":43.5,"Cpm":44.35 
},
{"name":"copper(I) oxide","formula":"Cu2O","state":"c","Hfg":-166.7,"Gfg":-146.4,"Smg":100.8,"Cpm":69.87 
},
{"name":"anhydrous copper sulfate","formula":"CuSO4","state":"c","Hfg":-770,"Gfg":-661.9,"Smg":113.4,"Cpm":"n/a" 
},
{"name":"hydrated copper sulfate","formula":"CuSO4.H2O","state":"c","Hfg":-1084,"Gfg":-917.1,"Smg":149.8,"Cpm":"n/a" 
},
{"name":"copper sulfide","formula":"CuS","state":"c","Hfg":-48.5,"Gfg":-48.9,"Smg":66.5,"Cpm":47.82 
},
{"name":"fluorine","formula":"F","state":"g","Hfg":78.99,"Gfg":61.59,"Smg":158.6,"Cpm":"n/a" 
},
{"name":"fluorine","formula":"F2","state":"g","Hfg":0,"Gfg":0,"Smg":202.7,"Cpm":31.46 
},
{"name":"hydrogen fluoride","formula":"HF","state":"g","Hfg":-271.1,"Gfg":-273.2,"Smg":173.7,"Cpm":29.07 
},
{"name":"hydrogen","formula":"H","state":"g","Hfg":218.0,"Gfg":203.3,"Smg":114.6,"Cpm":"n/a" 
},
{"name":"hydrogen","formula":"H2","state":"g","Hfg":0,"Gfg":0,"Smg":130.6,"Cpm":28.84 
},
{"name":"water","formula":"H2O","state":"g","Hfg":-241.8,"Gfg":-228.6,"Smg":188.7,"Cpm":33.58 
},
{"name":"water","formula":"H2O","state":"l","Hfg":-285.8,"Gfg":-237.2,"Smg":69.9,"Cpm":75.30 
},
{"name":"hydrogen peroxide","formula":"H2O2","state":"l","Hfg":-187.8,"Gfg":-120.4,"Smg":109.6,"Cpm":"n/a" 
},
{"name":"iodine","formula":"I","state":"g","Hfg":106.8,"Gfg":70.29,"Smg":180.7,"Cpm":20.79 
},
{"name":"iodine","formula":"I2","state":"c","Hfg":0,"Gfg":0,"Smg":116.1,"Cpm":54.98 
},
{"name":"iodine","formula":"I2","state":"g","Hfg":62.43,"Gfg":19.37,"Smg":260.6,"Cpm":36.86 
},
{"name":"hydrogen iodide","formula":"HI","state":"g","Hfg":26.48,"Gfg":2.13,"Smg":206.5,"Cpm":29.16 
},
{"name":"iron","formula":"Fe","state":"c","Hfg":0,"Gfg":0,"Smg":27.2,"Cpm":25.23 
},
{"name":"iron(II) oxide","formula":"FeO","state":"c","Hfg":-266.5,"Gfg":-244.3,"Smg":54.0,"Cpm":"n/a" 
},
{"name":"iron(III) oxide","formula":"Fe2O3","state":"c","Hfg":-822.2,"Gfg":-741.0,"Smg":90.0,"Cpm":104.60 
},
{"name":"iron(IV) oxide","formula":"Fe3O4","state":"c","Hfg":-1117.1,"Gfg":-1014.2,"Smg":146.41,"Cpm":"n/a" 
},
{"name":"magnesium","formula":"Mg","state":"c","Hfg":0,"Gfg":0,"Smg":32.5,"Cpm":23.89 
},
{"name":"magnesium chloride","formula":"MgCl2","state":"c","Hfg":-641.8,"Gfg":-592.3,"Smg":89.5,"Cpm":"n/a" 
},
{"name":"magnesium oxide","formula":"MgO","state":"c","Hfg":-601.8,"Gfg":-596.6,"Smg":26.8,"Cpm":37.78 
},
{"name":"nickel","formula":"Ni","state":"c","Hfg":0,"Gfg":0,"Smg":30.1,"Cpm":25.98 
},
{"name":"nickel oxide","formula":"NiO","state":"c","Hfg":-244.3,"Gfg":-216.3,"Smg":38.6,"Cpm":44.35 
},
{"name":"nitrogen","formula":"N","state":"g","Hfg":472.7,"Gfg":455.6,"Smg":153.2,"Cpm":"n/a" 
},
{"name":"nitrogen","formula":"N2","state":"g","Hfg":0,"Gfg":0,"Smg":191.5,"Cpm":29.12 
},
{"name":"ammonia","formula":"NH3","state":"g","Hfg":-46.11,"Gfg":-16.48,"Smg":192.3,"Cpm":35.66 
},
{"name":"nitric oxide","formula":"NO","state":"g","Hfg":90.25,"Gfg":86.57,"Smg":210.7,"Cpm":29.83 
},
{"name":"nitrous oxide","formula":"N2O","state":"g","Hfg":82.05,"Gfg":104.2,"Smg":219.7,"Cpm":38.71 
},
{"name":"nitrogen dioxide","formula":"NO2","state":"g","Hfg":33.18,"Gfg":51.30,"Smg":240.0,"Cpm":37.94 
},
{"name":"dinitrogen tetroxide","formula":"N2O4","state":"g","Hfg":9.16,"Gfg":97.82,"Smg":304.2,"Cpm":79.08 
},
{"name":"oxygen","formula":"O","state":"g","Hfg":242.2,"Gfg":231.8,"Smg":161.0,"Cpm":"n/a" 
},
{"name":"oxygen","formula":"O2","state":"g","Hfg":0,"Gfg":0,"Smg":41.1,"Cpm":23.22 
},
{"name":"ozone","formula":"O3","state":"g","Hfg":142.7,"Gfg":163.2,"Smg":238.8,"Cpm":38.16 
},
{"name":"phosporous","formula":"P","state":"c,white","Hfg":0,"Gfg":0,"Smg":41.1,"Cpm":23.22 
},
{"name":"phosphorous","formula":"P","state":"c,red","Hfg":-18.4,"Gfg":"n/a","Smg":"n/a","Cpm":"n/a" 
},
{"name":"phosphorous trichloride","formula":"PCl3","state":"g","Hfg":-287.0,"Gfg":-267.8,"Smg":311.7,"Cpm":"n/a" 
},
{"name":"phosphorous pentachloride","formula":"PCl5","state":"g","Hfg":-374.9,"Gfg":-305.4,"Smg":-364.5,"Cpm":"n/a" 
},
{"name":"potassium","formula":"K","state":"c","Hfg":0,"Gfg":0,"Smg":64.2,"Cpm":29.16 
},
{"name":"silicon","formula":"Si","state":"c","Hfg":0,"Gfg":0,"Smg":18.8,"Cpm":19.87 
},
{"name":"silane","formula":"SiH4","state":"g","Hfg":34.31,"Gfg":56.9,"Smg":204.5,"Cpm":42.84 
},
{"name":"silicon dioxide","formula":"SiO2","state":"c","Hfg":-910.9,"Gfg":-856.7,"Smg":41.8,"Cpm":44.43
},
{"name":"silicon tetrafluoride","formula":"SiF4","state":"g","Hfg":-1548.0,"Gfg":-1506.3,"Smg":284.5,"Cpm":76.15 
},
{"name":"silver","formula":"Ag","state":"c","Hfg":0,"Gfg":0,"Smg":18.8,"Cpm":19.87 
},
{"name":"silver chloride","formula":"AgCl","state":"c","Hfg":-127.1,"Gfg":-109.8,"Smg":96.2,"Cpm":50.79 
},
{"name":"silver dioxide","formula":"AgO2","state":"c","Hfg":-30.57,"Gfg":-10.83,"Smg":121.7,"Cpm":65.56 
},
{"name":"sodium","formula":"Na","state":"c","Hfg":0,"Gfg":0,"Smg":51.2,"Cpm":28.41 
},
{"name":"sodium chloride","formula":"NaCl","state":"c","Hfg":-411.0,"Gfg":-384.0,"Smg":72.4,"Cpm":49.71 
},
{"name":"sodium oxided","formula":"Na2O","state":"c","Hfg":-415.9,"Gfg":-376.6,"Smg":72.8,"Cpm":68.20 
},
{"name":"sodium iodide","formula":"NaI","state":"c","Hfg":-287.9,"Gfg":-268.4,"Smg":102.5,"Cpm":54.39 
},
{"name":"sulfur","formula":"S","state":"g","Hfg":278.8,"Gfg":238.3,"Smg":167.7,"Cpm":"n/a" 
},
{"name":"sulfur","formula":"S","state":"rhombic,c","Hfg":0,"Gfg":0,"Smg":31.8,"Cpm":22.59 
},
{"name":"sulfur","formula":"S","state":"monoclinic,c","Hfg":0.29,"Gfg":0.05,"Smg":32.6,"Cpm":23.64 
},
{"name":"sulfide","formula":"S2","state":"g","Hfg":128.4,"Gfg":79.08,"Smg":228.1,"Cpm":"n/a" 
},
{"name":"hydrogen sulfide","formula":"H2S","state":"g","Hfg":-20.63,"Gfg":-33.56,"Smg":205.7,"Cpm":33.97 
},
{"name":"sulfur dioxide","formula":"SO2","state":"g","Hfg":-296.8,"Gfg":-300.2,"Smg":248.1,"Cpm":39.79 
},
{"name":"sulfur trioxide","formula":"SO3","state":"g","Hfg":-395.2,"Gfg":-370.4,"Smg":256.2,"Cpm":50.63 
},
{"name":"titanium","formula":"Ti","state":"c","Hfg":0,"Gfg":0,"Smg":30.3,"Cpm":25.15 
},
{"name":"titanium tetrachloride","formula":"TiCl4","state":"l","Hfg":-750.2,"Gfg":674.5,"Smg":252.7,"Cpm":156.90 
},
{"name":"titanium dioxide","formula":"TiO2","state":"c","Hfg":-912.1,"Gfg":-852.8,"Smg":50.2,"Cpm":55.06 
},
{"name":"zinc","formula":"Zn","state":"c","Hfg":0,"Gfg":0,"Smg":41.6,"Cpm":25.06 
},
{"name":"zinc oxide","formula":"ZnO","state":"c","Hfg":-348.0,"Gfg":-318.2,"Smg":43.9,"Cpm":40.25 
},
{"name":"zinc sulfide","formula":"ZnS","state":"c","Hfg":-202.9,"Gfg":-198.3,"Smg":57.7,"Cpm": 45.19
},
{"name":"methane","formula":"CH4","state":"g","Hfg":-74.81,"Gfg":-50.75,"Smg":186.1,"Cpm":35.71 
},
{"name":"ethyne","formula":"C2H2","state":"g","Hfg":-74.81,"Gfg":-50.75,"Smg":186.1,"Cpm":35.71 
},
{"name":"ethene","formula":"C2H4","state":"g","Hfg":52.28,"Gfg":68.12,"Smg":219.5,"Cpm":43.55 
},
{"name":"ethane","formula":"C2H6","state":"g","Hfg":-84.67,"Gfg":-32.89,"Smg":229.5,"Cpm":52.66 
},
{"name":"propane","formula":"C3H8","state":"g","Hfg":-103.8,"Gfg":-23.49,"Smg":269.9,"Cpm":"n/a" 
},
{"name":"n-butane","formula":"C4H10","state":"g","Hfg":-124.7,"Gfg":-15.71,"Smg":310.0,"Cpm":"n/a" 
},
{"name":"iso-butane","formula":"C4H10","state":"g","Hfg":-131.6,"Gfg":-17.97,"Smg":294.6,"Cpm":"n/a"
},
{"name":"benzene","formula":"C6H6","state":"l","Hfg":82.93,"Gfg":129.7,"Smg":269.2,"Cpm":108.16 
},
{"name":"carbon tetrachloride","formula":"CCl4","state":"l","Hfg":-135.4,"Gfg":-65.27,"Smg":216.4,"Cpm":83.51 
},
{"name":"chloroform","formula":"CHCl3","state":"l","Hfg":-134.5,"Gfg":-73.72,"Smg":201.7,"Cpm":"n/a" 
},
{"name":"chloromethane","formula":"CH3Cl","state":"g","Hfg":-80.83,"Gfg":-57.40,"Smg":234.5,"Cpm":"n/a" 
},
{"name":"formaldehyde","formula":"HCHO","state":"g","Hfg":-117.2,"Gfg":-113.0,"Smg":218.8,"Cpm":"n/a" 
},
{"name":"methanol","formula":"CH3OH","state":"g","Hfg":-200.7,"Gfg":-162.0,"Smg":239.7,"Cpm":"n/a" 
},
{"name":"methanol","formula":"CH3OH","state":"l","Hfg":-238.7,"Gfg":-166.4,"Smg":126.8,"Cpm":81.6 
},
{"name":"ethanol","formula":"C2H5OH","state":"l","Hfg":-277.6,"Gfg":-174.8,"Smg":160.7,"Cpm":111.5 
},
{"name":"ethanol","formula":"C2H5OH","state":"g","Hfg":-235.3,"Gfg":-168.6,"Smg":282.0,"Cpm":82.33 
},
{"name":"ethanoic acid","formula":"CH3COOH","state":"l","Hfg":-487.0,"Gfg":-382.5,"Smg":159.8,"Cpm":"n/a" 
},
{"name":"hydrogen cyanide","formula":"HCN","state":"g","Hfg":130.5,"Gfg":120.1,"Smg":201.8,"Cpm":35.90 
}
];
// from https://github.com/andrejewski/periodic-table
// which is itself based on https://web.archive.org/web/20161203095654/http://php.scripts.psu.edu/djh300/cmpsc221/p3s11-pt-data.htm
var periodic_table_data = [
{"atomicNumber":1,"symbol":"H","name":"Hydrogen","atomicMass":"1.00794(4)","cpkHexColor":"FFFFFF","electronicConfiguration":"1s1","electronegativity":2.2,"atomicRadius":37,"ionRadius":"","vanDerWaalsRadius":120,"ionizationEnergy":1312,"electronAffinity":-73,"oxidationStates":"-1, 1","standardState":"gas","bondingType":"diatomic","meltingPoint":14,"boilingPoint":20,"density":0.0000899,"groupBlock":"nonmetal","yearDiscovered":1766},
{"atomicNumber":2,"symbol":"He","name":"Helium","atomicMass":"4.002602(2)","cpkHexColor":"D9FFFF","electronicConfiguration":"1s2","electronegativity":"","atomicRadius":32,"ionRadius":"","vanDerWaalsRadius":140,"ionizationEnergy":2372,"electronAffinity":0,"oxidationStates":"","standardState":"gas","bondingType":"atomic","meltingPoint":"","boilingPoint":4,"density":0.0001785,"groupBlock":"noble gas","yearDiscovered":1868},
{"atomicNumber":3,"symbol":"Li","name":"Lithium","atomicMass":"6.941(2)","cpkHexColor":"CC80FF","electronicConfiguration":"[He] 2s1","electronegativity":0.98,"atomicRadius":134,"ionRadius":"76 (+1)","vanDerWaalsRadius":182,"ionizationEnergy":520,"electronAffinity":-60,"oxidationStates":1,"standardState":"solid","bondingType":"metallic","meltingPoint":454,"boilingPoint":1615,"density":0.535,"groupBlock":"alkali metal","yearDiscovered":1817},
{"atomicNumber":4,"symbol":"Be","name":"Beryllium","atomicMass":"9.012182(3)","cpkHexColor":"C2FF00","electronicConfiguration":"[He] 2s2","electronegativity":1.57,"atomicRadius":90,"ionRadius":"45 (+2)","vanDerWaalsRadius":"","ionizationEnergy":900,"electronAffinity":0,"oxidationStates":2,"standardState":"solid","bondingType":"metallic","meltingPoint":1560,"boilingPoint":2743,"density":1.848,"groupBlock":"alkaline earth metal","yearDiscovered":1798},
{"atomicNumber":5,"symbol":"B","name":"Boron","atomicMass":"10.811(7)","cpkHexColor":"FFB5B5","electronicConfiguration":"[He] 2s2 2p1","electronegativity":2.04,"atomicRadius":82,"ionRadius":"27 (+3)","vanDerWaalsRadius":"","ionizationEnergy":801,"electronAffinity":-27,"oxidationStates":"1, 2, 3","standardState":"solid","bondingType":"covalent network","meltingPoint":2348,"boilingPoint":4273,"density":2.46,"groupBlock":"metalloid","yearDiscovered":1807},
{"atomicNumber":6,"symbol":"C","name":"Carbon","atomicMass":"12.0107(8)","cpkHexColor":909090,"electronicConfiguration":"[He] 2s2 2p2","electronegativity":2.55,"atomicRadius":77,"ionRadius":"16 (+4)","vanDerWaalsRadius":170,"ionizationEnergy":1087,"electronAffinity":-154,"oxidationStates":"-4, -3, -2, -1, 1, 2, 3, 4","standardState":"solid","bondingType":"covalent network","meltingPoint":3823,"boilingPoint":4300,"density":2.26,"groupBlock":"nonmetal","yearDiscovered":"Ancient"},
{"atomicNumber":7,"symbol":"N","name":"Nitrogen","atomicMass":"14.0067(2)","cpkHexColor":"3050F8","electronicConfiguration":"[He] 2s2 2p3","electronegativity":3.04,"atomicRadius":75,"ionRadius":"146 (-3)","vanDerWaalsRadius":155,"ionizationEnergy":1402,"electronAffinity":-7,"oxidationStates":"-3, -2, -1, 1, 2, 3, 4, 5","standardState":"gas","bondingType":"diatomic","meltingPoint":63,"boilingPoint":77,"density":0.001251,"groupBlock":"nonmetal","yearDiscovered":1772},
{"atomicNumber":8,"symbol":"O","name":"Oxygen","atomicMass":"15.9994(3)","cpkHexColor":"FF0D0D","electronicConfiguration":"[He] 2s2 2p4","electronegativity":3.44,"atomicRadius":73,"ionRadius":"140 (-2)","vanDerWaalsRadius":152,"ionizationEnergy":1314,"electronAffinity":-141,"oxidationStates":"-2, -1, 1, 2","standardState":"gas","bondingType":"diatomic","meltingPoint":55,"boilingPoint":90,"density":0.001429,"groupBlock":"nonmetal","yearDiscovered":1774},
{"atomicNumber":9,"symbol":"F","name":"Fluorine","atomicMass":"18.9984032(5)","cpkHexColor":9e+51,"electronicConfiguration":"[He] 2s2 2p5","electronegativity":3.98,"atomicRadius":71,"ionRadius":"133 (-1)","vanDerWaalsRadius":147,"ionizationEnergy":1681,"electronAffinity":-328,"oxidationStates":-1,"standardState":"gas","bondingType":"atomic","meltingPoint":54,"boilingPoint":85,"density":0.001696,"groupBlock":"halogen","yearDiscovered":1670},
{"atomicNumber":10,"symbol":"Ne","name":"Neon","atomicMass":"20.1797(6)","cpkHexColor":"B3E3F5","electronicConfiguration":"[He] 2s2 2p6","electronegativity":"","atomicRadius":69,"ionRadius":"","vanDerWaalsRadius":154,"ionizationEnergy":2081,"electronAffinity":0,"oxidationStates":"","standardState":"gas","bondingType":"atomic","meltingPoint":25,"boilingPoint":27,"density":0.0009,"groupBlock":"noble gas","yearDiscovered":1898},
{"atomicNumber":11,"symbol":"Na","name":"Sodium","atomicMass":"22.98976928(2)","cpkHexColor":"AB5CF2","electronicConfiguration":"[Ne] 3s1","electronegativity":0.93,"atomicRadius":154,"ionRadius":"102 (+1)","vanDerWaalsRadius":227,"ionizationEnergy":496,"electronAffinity":-53,"oxidationStates":"-1, 1","standardState":"solid","bondingType":"metallic","meltingPoint":371,"boilingPoint":1156,"density":0.968,"groupBlock":"alkali metal","yearDiscovered":1807},
{"atomicNumber":12,"symbol":"Mg","name":"Magnesium","atomicMass":"24.3050(6)","cpkHexColor":"8AFF00","electronicConfiguration":"[Ne] 3s2","electronegativity":1.31,"atomicRadius":130,"ionRadius":"72 (+2)","vanDerWaalsRadius":173,"ionizationEnergy":738,"electronAffinity":0,"oxidationStates":"1, 2","standardState":"solid","bondingType":"metallic","meltingPoint":923,"boilingPoint":1363,"density":1.738,"groupBlock":"alkaline earth metal","yearDiscovered":1808},
{"atomicNumber":13,"symbol":"Al","name":"Aluminum","atomicMass":"26.9815386(8)","cpkHexColor":"BFA6A6","electronicConfiguration":"[Ne] 3s2 3p1","electronegativity":1.61,"atomicRadius":118,"ionRadius":"53.5 (+3)","vanDerWaalsRadius":"","ionizationEnergy":578,"electronAffinity":-43,"oxidationStates":"1, 3","standardState":"solid","bondingType":"metallic","meltingPoint":933,"boilingPoint":2792,"density":2.7,"groupBlock":"metal","yearDiscovered":"Ancient"},
{"atomicNumber":14,"symbol":"Si","name":"Silicon","atomicMass":"28.0855(3)","cpkHexColor":"F0C8A0","electronicConfiguration":"[Ne] 3s2 3p2","electronegativity":1.9,"atomicRadius":111,"ionRadius":"40 (+4)","vanDerWaalsRadius":210,"ionizationEnergy":787,"electronAffinity":-134,"oxidationStates":"-4, -3, -2, -1, 1, 2, 3, 4","standardState":"solid","bondingType":"metallic","meltingPoint":1687,"boilingPoint":3173,"density":2.33,"groupBlock":"metalloid","yearDiscovered":1854},
{"atomicNumber":15,"symbol":"P","name":"Phosphorus","atomicMass":"30.973762(2)","cpkHexColor":"FF8000","electronicConfiguration":"[Ne] 3s2 3p3","electronegativity":2.19,"atomicRadius":106,"ionRadius":"44 (+3)","vanDerWaalsRadius":180,"ionizationEnergy":1012,"electronAffinity":-72,"oxidationStates":"-3, -2, -1, 1, 2, 3, 4, 5","standardState":"solid","bondingType":"covalent network","meltingPoint":317,"boilingPoint":554,"density":1.823,"groupBlock":"nonmetal","yearDiscovered":1669},
{"atomicNumber":16,"symbol":"S","name":"Sulfur","atomicMass":"32.065(5)","cpkHexColor":"FFFF30","electronicConfiguration":"[Ne] 3s2 3p4","electronegativity":2.58,"atomicRadius":102,"ionRadius":"184 (-2)","vanDerWaalsRadius":180,"ionizationEnergy":1000,"electronAffinity":-200,"oxidationStates":"-2, -1, 1, 2, 3, 4, 5, 6","standardState":"solid","bondingType":"covalent network","meltingPoint":388,"boilingPoint":718,"density":1.96,"groupBlock":"nonmetal","yearDiscovered":"Ancient"},
{"atomicNumber":17,"symbol":"Cl","name":"Chlorine","atomicMass":"35.453(2)","cpkHexColor":"1FF01F","electronicConfiguration":"[Ne] 3s2 3p5","electronegativity":3.16,"atomicRadius":99,"ionRadius":"181 (-1)","vanDerWaalsRadius":175,"ionizationEnergy":1251,"electronAffinity":-349,"oxidationStates":"-1, 1, 2, 3, 4, 5, 6, 7","standardState":"gas","bondingType":"covalent network","meltingPoint":172,"boilingPoint":239,"density":0.003214,"groupBlock":"halogen","yearDiscovered":1774},
{"atomicNumber":18,"symbol":"Ar","name":"Argon","atomicMass":"39.948(1)","cpkHexColor":"80D1E3","electronicConfiguration":"[Ne] 3s2 3p6","electronegativity":"","atomicRadius":97,"ionRadius":"","vanDerWaalsRadius":188,"ionizationEnergy":1521,"electronAffinity":0,"oxidationStates":"","standardState":"gas","bondingType":"atomic","meltingPoint":84,"boilingPoint":87,"density":0.001784,"groupBlock":"noble gas","yearDiscovered":1894},
{"atomicNumber":19,"symbol":"K","name":"Potassium","atomicMass":"39.0983(1)","cpkHexColor":"8F40D4","electronicConfiguration":"[Ar] 4s1","electronegativity":0.82,"atomicRadius":196,"ionRadius":"138 (+1)","vanDerWaalsRadius":275,"ionizationEnergy":419,"electronAffinity":-48,"oxidationStates":1,"standardState":"solid","bondingType":"metallic","meltingPoint":337,"boilingPoint":1032,"density":0.856,"groupBlock":"alkali metal","yearDiscovered":1807},
{"atomicNumber":20,"symbol":"Ca","name":"Calcium","atomicMass":"40.078(4)","cpkHexColor":"3DFF00","electronicConfiguration":"[Ar] 4s2","electronegativity":1,"atomicRadius":174,"ionRadius":"100 (+2)","vanDerWaalsRadius":"","ionizationEnergy":590,"electronAffinity":-2,"oxidationStates":2,"standardState":"solid","bondingType":"metallic","meltingPoint":1115,"boilingPoint":1757,"density":1.55,"groupBlock":"alkaline earth metal","yearDiscovered":"Ancient"},
{"atomicNumber":21,"symbol":"Sc","name":"Scandium","atomicMass":"44.955912(6)","cpkHexColor":"E6E6E6","electronicConfiguration":"[Ar] 3d1 4s2","electronegativity":1.36,"atomicRadius":144,"ionRadius":"74.5 (+3)","vanDerWaalsRadius":"","ionizationEnergy":633,"electronAffinity":-18,"oxidationStates":"1, 2, 3","standardState":"solid","bondingType":"metallic","meltingPoint":1814,"boilingPoint":3103,"density":2.985,"groupBlock":"transition metal","yearDiscovered":1876},
{"atomicNumber":22,"symbol":"Ti","name":"Titanium","atomicMass":"47.867(1)","cpkHexColor":"BFC2C7","electronicConfiguration":"[Ar] 3d2 4s2","electronegativity":1.54,"atomicRadius":136,"ionRadius":"86 (+2)","vanDerWaalsRadius":"","ionizationEnergy":659,"electronAffinity":-8,"oxidationStates":"-1, 2, 3, 4","standardState":"solid","bondingType":"metallic","meltingPoint":1941,"boilingPoint":3560,"density":4.507,"groupBlock":"transition metal","yearDiscovered":1791},
{"atomicNumber":23,"symbol":"V","name":"Vanadium","atomicMass":"50.9415(1)","cpkHexColor":"A6A6AB","electronicConfiguration":"[Ar] 3d3 4s2","electronegativity":1.63,"atomicRadius":125,"ionRadius":"79 (+2)","vanDerWaalsRadius":"","ionizationEnergy":651,"electronAffinity":-51,"oxidationStates":"-1, 2, 3, 4","standardState":"solid","bondingType":"metallic","meltingPoint":2183,"boilingPoint":3680,"density":6.11,"groupBlock":"transition metal","yearDiscovered":1803},
{"atomicNumber":24,"symbol":"Cr","name":"Chromium","atomicMass":"51.9961(6)","cpkHexColor":"8A99C7","electronicConfiguration":"[Ar] 3d5 4s1","electronegativity":1.66,"atomicRadius":127,"ionRadius":"80 (+2*)","vanDerWaalsRadius":"","ionizationEnergy":653,"electronAffinity":-64,"oxidationStates":"-2, -1, 1, 2, 3, 4, 5, 6","standardState":"solid","bondingType":"metallic","meltingPoint":2180,"boilingPoint":2944,"density":7.14,"groupBlock":"transition metal","yearDiscovered":"Ancient"},
{"atomicNumber":25,"symbol":"Mn","name":"Manganese","atomicMass":"54.938045(5)","cpkHexColor":"9C7AC7","electronicConfiguration":"[Ar] 3d5 4s2","electronegativity":1.55,"atomicRadius":139,"ionRadius":"67 (+2)","vanDerWaalsRadius":"","ionizationEnergy":717,"electronAffinity":0,"oxidationStates":"-3, -2, -1, 1, 2, 3, 4, 5, 6, 7","standardState":"solid","bondingType":"metallic","meltingPoint":1519,"boilingPoint":2334,"density":7.47,"groupBlock":"transition metal","yearDiscovered":1774},
{"atomicNumber":26,"symbol":"Fe","name":"Iron","atomicMass":"55.845(2)","cpkHexColor":"E06633","electronicConfiguration":"[Ar] 3d6 4s2","electronegativity":1.83,"atomicRadius":125,"ionRadius":"78 (+2*)","vanDerWaalsRadius":"","ionizationEnergy":763,"electronAffinity":-16,"oxidationStates":"-2, -1, 1, 2, 3, 4, 5, 6","standardState":"solid","bondingType":"metallic","meltingPoint":1811,"boilingPoint":3134,"density":7.874,"groupBlock":"transition metal","yearDiscovered":"Ancient"},
{"atomicNumber":27,"symbol":"Co","name":"Cobalt","atomicMass":"58.933195(5)","cpkHexColor":"F090A0","electronicConfiguration":"[Ar] 3d7 4s2","electronegativity":1.88,"atomicRadius":126,"ionRadius":"74.5 (+2*)","vanDerWaalsRadius":"","ionizationEnergy":760,"electronAffinity":-64,"oxidationStates":"-1, 1, 2, 3, 4, 5","standardState":"solid","bondingType":"metallic","meltingPoint":1768,"boilingPoint":3200,"density":8.9,"groupBlock":"transition metal","yearDiscovered":"Ancient"},
{"atomicNumber":28,"symbol":"Ni","name":"Nickel","atomicMass":"58.6934(4)","cpkHexColor":"50D050","electronicConfiguration":"[Ar] 3d8 4s2","electronegativity":1.91,"atomicRadius":121,"ionRadius":"69 (+2)","vanDerWaalsRadius":163,"ionizationEnergy":737,"electronAffinity":-112,"oxidationStates":"-1, 1, 2, 3, 4","standardState":"solid","bondingType":"metallic","meltingPoint":1728,"boilingPoint":3186,"density":8.908,"groupBlock":"transition metal","yearDiscovered":1751},
{"atomicNumber":29,"symbol":"Cu","name":"Copper","atomicMass":"63.546(3)","cpkHexColor":"C88033","electronicConfiguration":"[Ar] 3d10 4s1","electronegativity":1.9,"atomicRadius":138,"ionRadius":"77 (+1)","vanDerWaalsRadius":140,"ionizationEnergy":746,"electronAffinity":-118,"oxidationStates":"1, 2, 3, 4","standardState":"solid","bondingType":"metallic","meltingPoint":1358,"boilingPoint":3200,"density":8.92,"groupBlock":"transition metal","yearDiscovered":"Ancient"},
{"atomicNumber":30,"symbol":"Zn","name":"Zinc","atomicMass":"65.38(2)","cpkHexColor":"7D80B0","electronicConfiguration":"[Ar] 3d10 4s2","electronegativity":1.65,"atomicRadius":131,"ionRadius":"74 (+2)","vanDerWaalsRadius":139,"ionizationEnergy":906,"electronAffinity":0,"oxidationStates":2,"standardState":"solid","bondingType":"metallic","meltingPoint":693,"boilingPoint":1180,"density":7.14,"groupBlock":"transition metal","yearDiscovered":1746},
{"atomicNumber":31,"symbol":"Ga","name":"Gallium","atomicMass":"69.723(1)","cpkHexColor":"C28F8F","electronicConfiguration":"[Ar] 3d10 4s2 4p1","electronegativity":1.81,"atomicRadius":126,"ionRadius":"62 (+3)","vanDerWaalsRadius":187,"ionizationEnergy":579,"electronAffinity":-29,"oxidationStates":"1, 2, 3","standardState":"solid","bondingType":"metallic","meltingPoint":303,"boilingPoint":2477,"density":5.904,"groupBlock":"metal","yearDiscovered":1875},
{"atomicNumber":32,"symbol":"Ge","name":"Germanium","atomicMass":"72.64(1)","cpkHexColor":"668F8F","electronicConfiguration":"[Ar] 3d10 4s2 4p2","electronegativity":2.01,"atomicRadius":122,"ionRadius":"73 (+2)","vanDerWaalsRadius":"","ionizationEnergy":762,"electronAffinity":-119,"oxidationStates":"-4, 1, 2, 3, 4","standardState":"solid","bondingType":"metallic","meltingPoint":1211,"boilingPoint":3093,"density":5.323,"groupBlock":"metalloid","yearDiscovered":1886},
{"atomicNumber":33,"symbol":"As","name":"Arsenic","atomicMass":"74.92160(2)","cpkHexColor":"BD80E3","electronicConfiguration":"[Ar] 3d10 4s2 4p3","electronegativity":2.18,"atomicRadius":119,"ionRadius":"58 (+3)","vanDerWaalsRadius":185,"ionizationEnergy":947,"electronAffinity":-78,"oxidationStates":"-3, 2, 3, 5","standardState":"solid","bondingType":"metallic","meltingPoint":1090,"boilingPoint":887,"density":5.727,"groupBlock":"metalloid","yearDiscovered":"Ancient"},
{"atomicNumber":34,"symbol":"Se","name":"Selenium","atomicMass":"78.96(3)","cpkHexColor":"FFA100","electronicConfiguration":"[Ar] 3d10 4s2 4p4","electronegativity":2.55,"atomicRadius":116,"ionRadius":"198 (-2)","vanDerWaalsRadius":190,"ionizationEnergy":941,"electronAffinity":-195,"oxidationStates":"-2, 2, 4, 6","standardState":"solid","bondingType":"metallic","meltingPoint":494,"boilingPoint":958,"density":4.819,"groupBlock":"nonmetal","yearDiscovered":1817},
{"atomicNumber":35,"symbol":"Br","name":"Bromine","atomicMass":"79.904(1)","cpkHexColor":"A62929","electronicConfiguration":"[Ar] 3d10 4s2 4p5","electronegativity":2.96,"atomicRadius":114,"ionRadius":"196 (-1)","vanDerWaalsRadius":185,"ionizationEnergy":1140,"electronAffinity":-325,"oxidationStates":"-1, 1, 3, 4, 5, 7","standardState":"liquid","bondingType":"covalent network","meltingPoint":266,"boilingPoint":332,"density":3.12,"groupBlock":"halogen","yearDiscovered":1826},
{"atomicNumber":36,"symbol":"Kr","name":"Krypton","atomicMass":"83.798(2)","cpkHexColor":"5CB8D1","electronicConfiguration":"[Ar] 3d10 4s2 4p6","electronegativity":"","atomicRadius":110,"ionRadius":"","vanDerWaalsRadius":202,"ionizationEnergy":1351,"electronAffinity":0,"oxidationStates":2,"standardState":"gas","bondingType":"atomic","meltingPoint":116,"boilingPoint":120,"density":0.00375,"groupBlock":"noble gas","yearDiscovered":1898},
{"atomicNumber":37,"symbol":"Rb","name":"Rubidium","atomicMass":"85.4678(3)","cpkHexColor":"702EB0","electronicConfiguration":"[Kr] 5s1","electronegativity":0.82,"atomicRadius":211,"ionRadius":"152 (+1)","vanDerWaalsRadius":"","ionizationEnergy":403,"electronAffinity":-47,"oxidationStates":1,"standardState":"solid","bondingType":"metallic","meltingPoint":312,"boilingPoint":961,"density":1.532,"groupBlock":"alkali metal","yearDiscovered":1861},
{"atomicNumber":38,"symbol":"Sr","name":"Strontium","atomicMass":"87.62(1)","cpkHexColor":"00FF00","electronicConfiguration":"[Kr] 5s2","electronegativity":0.95,"atomicRadius":192,"ionRadius":"118 (+2)","vanDerWaalsRadius":"","ionizationEnergy":550,"electronAffinity":-5,"oxidationStates":2,"standardState":"solid","bondingType":"metallic","meltingPoint":1050,"boilingPoint":1655,"density":2.63,"groupBlock":"alkaline earth metal","yearDiscovered":1790},
{"atomicNumber":39,"symbol":"Y","name":"Yttrium","atomicMass":"88.90585(2)","cpkHexColor":"94FFFF","electronicConfiguration":"[Kr] 4d1 5s2","electronegativity":1.22,"atomicRadius":162,"ionRadius":"90 (+3)","vanDerWaalsRadius":"","ionizationEnergy":600,"electronAffinity":-30,"oxidationStates":"1, 2, 3","standardState":"solid","bondingType":"metallic","meltingPoint":1799,"boilingPoint":3618,"density":4.472,"groupBlock":"transition metal","yearDiscovered":1794},
{"atomicNumber":40,"symbol":"Zr","name":"Zirconium","atomicMass":"91.224(2)","cpkHexColor":"94E0E0","electronicConfiguration":"[Kr] 4d2 5s2","electronegativity":1.33,"atomicRadius":148,"ionRadius":"72 (+4)","vanDerWaalsRadius":"","ionizationEnergy":640,"electronAffinity":-41,"oxidationStates":"1, 2, 3, 4","standardState":"solid","bondingType":"metallic","meltingPoint":2128,"boilingPoint":4682,"density":6.511,"groupBlock":"transition metal","yearDiscovered":1789},
{"atomicNumber":41,"symbol":"Nb","name":"Niobium","atomicMass":"92.90638(2)","cpkHexColor":"73C2C9","electronicConfiguration":"[Kr] 4d4 5s1","electronegativity":1.6,"atomicRadius":137,"ionRadius":"72 (+3)","vanDerWaalsRadius":"","ionizationEnergy":652,"electronAffinity":-86,"oxidationStates":"-1, 2, 3, 4, 5","standardState":"solid","bondingType":"metallic","meltingPoint":2750,"boilingPoint":5017,"density":8.57,"groupBlock":"transition metal","yearDiscovered":1801},
{"atomicNumber":42,"symbol":"Mo","name":"Molybdenum","atomicMass":"95.96(2)","cpkHexColor":"54B5B5","electronicConfiguration":"[Kr] 4d5 5s1","electronegativity":2.16,"atomicRadius":145,"ionRadius":"69 (+3)","vanDerWaalsRadius":"","ionizationEnergy":684,"electronAffinity":-72,"oxidationStates":"-2, -1, 1, 2, 3, 4, 5, 6","standardState":"solid","bondingType":"metallic","meltingPoint":2896,"boilingPoint":4912,"density":10.28,"groupBlock":"transition metal","yearDiscovered":1778},
{"atomicNumber":43,"symbol":"Tc","name":"Technetium","atomicMass":"98","cpkHexColor":"3B9E9E","electronicConfiguration":"[Kr] 4d5 5s2","electronegativity":1.9,"atomicRadius":156,"ionRadius":"64.5 (+4)","vanDerWaalsRadius":"","ionizationEnergy":702,"electronAffinity":-53,"oxidationStates":"-3, -1, 1, 2, 3, 4, 5, 6, 7","standardState":"solid","bondingType":"metallic","meltingPoint":2430,"boilingPoint":4538,"density":11.5,"groupBlock":"transition metal","yearDiscovered":1937},
{"atomicNumber":44,"symbol":"Ru","name":"Ruthenium","atomicMass":"101.07(2)","cpkHexColor":"248F8F","electronicConfiguration":"[Kr] 4d7 5s1","electronegativity":2.2,"atomicRadius":126,"ionRadius":"68 (+3)","vanDerWaalsRadius":"","ionizationEnergy":710,"electronAffinity":-101,"oxidationStates":"-2, 1, 2, 3, 4, 5, 6, 7, 8","standardState":"solid","bondingType":"metallic","meltingPoint":2607,"boilingPoint":4423,"density":12.37,"groupBlock":"transition metal","yearDiscovered":1827},
{"atomicNumber":45,"symbol":"Rh","name":"Rhodium","atomicMass":"102.90550(2)","cpkHexColor":"0A7D8C","electronicConfiguration":"[Kr] 4d8 5s1","electronegativity":2.28,"atomicRadius":135,"ionRadius":"66.5 (+3)","vanDerWaalsRadius":"","ionizationEnergy":720,"electronAffinity":-110,"oxidationStates":"-1, 1, 2, 3, 4, 5, 6","standardState":"solid","bondingType":"metallic","meltingPoint":2237,"boilingPoint":3968,"density":12.45,"groupBlock":"transition metal","yearDiscovered":1803},
{"atomicNumber":46,"symbol":"Pd","name":"Palladium","atomicMass":"106.42(1)","cpkHexColor":6985,"electronicConfiguration":"[Kr] 4d10","electronegativity":2.2,"atomicRadius":131,"ionRadius":"59 (+1)","vanDerWaalsRadius":163,"ionizationEnergy":804,"electronAffinity":-54,"oxidationStates":"2, 4","standardState":"solid","bondingType":"metallic","meltingPoint":1828,"boilingPoint":3236,"density":12.023,"groupBlock":"transition metal","yearDiscovered":1803},
{"atomicNumber":47,"symbol":"Ag","name":"Silver","atomicMass":"107.8682(2)","cpkHexColor":"C0C0C0","electronicConfiguration":"[Kr] 4d10 5s1","electronegativity":1.93,"atomicRadius":153,"ionRadius":"115 (+1)","vanDerWaalsRadius":172,"ionizationEnergy":731,"electronAffinity":-126,"oxidationStates":"1, 2, 3","standardState":"solid","bondingType":"metallic","meltingPoint":1235,"boilingPoint":2435,"density":10.49,"groupBlock":"transition metal","yearDiscovered":"Ancient"},
{"atomicNumber":48,"symbol":"Cd","name":"Cadmium","atomicMass":"112.411(8)","cpkHexColor":"FFD98F","electronicConfiguration":"[Kr] 4d10 5s2","electronegativity":1.69,"atomicRadius":148,"ionRadius":"95 (+2)","vanDerWaalsRadius":158,"ionizationEnergy":868,"electronAffinity":0,"oxidationStates":2,"standardState":"solid","bondingType":"metallic","meltingPoint":594,"boilingPoint":1040,"density":8.65,"groupBlock":"transition metal","yearDiscovered":1817},
{"atomicNumber":49,"symbol":"In","name":"Indium","atomicMass":"114.818(3)","cpkHexColor":"A67573","electronicConfiguration":"[Kr] 4d10 5s2 5p1","electronegativity":1.78,"atomicRadius":144,"ionRadius":"80 (+3)","vanDerWaalsRadius":193,"ionizationEnergy":558,"electronAffinity":-29,"oxidationStates":"1, 2, 3","standardState":"solid","bondingType":"metallic","meltingPoint":430,"boilingPoint":2345,"density":7.31,"groupBlock":"metal","yearDiscovered":1863},
{"atomicNumber":50,"symbol":"Sn","name":"Tin","atomicMass":"118.710(7)","cpkHexColor":668080,"electronicConfiguration":"[Kr] 4d10 5s2 5p2","electronegativity":1.96,"atomicRadius":141,"ionRadius":"112 (+2)","vanDerWaalsRadius":217,"ionizationEnergy":709,"electronAffinity":-107,"oxidationStates":"-4, 2, 4","standardState":"solid","bondingType":"metallic","meltingPoint":505,"boilingPoint":2875,"density":7.31,"groupBlock":"metal","yearDiscovered":"Ancient"},
{"atomicNumber":51,"symbol":"Sb","name":"Antimony","atomicMass":"121.760(1)","cpkHexColor":"9E63B5","electronicConfiguration":"[Kr] 4d10 5s2 5p3","electronegativity":2.05,"atomicRadius":138,"ionRadius":"76 (+3)","vanDerWaalsRadius":"","ionizationEnergy":834,"electronAffinity":-103,"oxidationStates":"-3, 3, 5","standardState":"solid","bondingType":"metallic","meltingPoint":904,"boilingPoint":1860,"density":6.697,"groupBlock":"metalloid","yearDiscovered":"Ancient"},
{"atomicNumber":52,"symbol":"Te","name":"Tellurium","atomicMass":"127.60(3)","cpkHexColor":"D47A00","electronicConfiguration":"[Kr] 4d10 5s2 5p4","electronegativity":2.1,"atomicRadius":135,"ionRadius":"221 (-2)","vanDerWaalsRadius":206,"ionizationEnergy":869,"electronAffinity":-190,"oxidationStates":"-2, 2, 4, 5, 6","standardState":"solid","bondingType":"metallic","meltingPoint":723,"boilingPoint":1261,"density":6.24,"groupBlock":"metalloid","yearDiscovered":1782},
{"atomicNumber":53,"symbol":"I","name":"Iodine","atomicMass":"126.90447(3)","cpkHexColor":940094,"electronicConfiguration":"[Kr] 4d10 5s2 5p5","electronegativity":2.66,"atomicRadius":133,"ionRadius":"220 (-1)","vanDerWaalsRadius":198,"ionizationEnergy":1008,"electronAffinity":-295,"oxidationStates":"-1, 1, 3, 5, 7","standardState":"solid","bondingType":"covalent network","meltingPoint":387,"boilingPoint":457,"density":4.94,"groupBlock":"halogen","yearDiscovered":1811},
{"atomicNumber":54,"symbol":"Xe","name":"Xenon","atomicMass":"131.293(6)","cpkHexColor":"429EB0","electronicConfiguration":"[Kr] 4d10 5s2 5p6","electronegativity":"","atomicRadius":130,"ionRadius":"48 (+8)","vanDerWaalsRadius":216,"ionizationEnergy":1170,"electronAffinity":0,"oxidationStates":"2, 4, 6, 8","standardState":"gas","bondingType":"atomic","meltingPoint":161,"boilingPoint":165,"density":0.0059,"groupBlock":"noble gas","yearDiscovered":1898},
{"atomicNumber":55,"symbol":"Cs","name":"Cesium","atomicMass":"132.9054519(2)","cpkHexColor":"57178F","electronicConfiguration":"[Xe] 6s1","electronegativity":0.79,"atomicRadius":225,"ionRadius":"167 (+1)","vanDerWaalsRadius":"","ionizationEnergy":376,"electronAffinity":-46,"oxidationStates":1,"standardState":"solid","bondingType":"metallic","meltingPoint":302,"boilingPoint":944,"density":1.879,"groupBlock":"alkali metal","yearDiscovered":1860},
{"atomicNumber":56,"symbol":"Ba","name":"Barium","atomicMass":"137.327(7)","cpkHexColor":"00C900","electronicConfiguration":"[Xe] 6s2","electronegativity":0.89,"atomicRadius":198,"ionRadius":"135 (+2)","vanDerWaalsRadius":"","ionizationEnergy":503,"electronAffinity":-14,"oxidationStates":2,"standardState":"solid","bondingType":"metallic","meltingPoint":1000,"boilingPoint":2143,"density":3.51,"groupBlock":"alkaline earth metal","yearDiscovered":1808},
{"atomicNumber":57,"symbol":"La","name":"Lanthanum","atomicMass":"138.90547(7)","cpkHexColor":"70D4FF","electronicConfiguration":"[Xe] 5d1 6s2","electronegativity":1.1,"atomicRadius":169,"ionRadius":"103.2 (+3)","vanDerWaalsRadius":"","ionizationEnergy":538,"electronAffinity":-48,"oxidationStates":"2, 3","standardState":"solid","bondingType":"metallic","meltingPoint":1193,"boilingPoint":3737,"density":6.146,"groupBlock":"lanthanoid","yearDiscovered":1839},
{"atomicNumber":58,"symbol":"Ce","name":"Cerium","atomicMass":"140.116(1)","cpkHexColor":"FFFFC7","electronicConfiguration":"[Xe] 4f1 5d1 6s2","electronegativity":1.12,"atomicRadius":"","ionRadius":"102 (+3)","vanDerWaalsRadius":"","ionizationEnergy":534,"electronAffinity":-50,"oxidationStates":"2, 3, 4","standardState":"solid","bondingType":"metallic","meltingPoint":1071,"boilingPoint":3633,"density":6.689,"groupBlock":"lanthanoid","yearDiscovered":1803},
{"atomicNumber":59,"symbol":"Pr","name":"Praseodymium","atomicMass":"140.90765(2)","cpkHexColor":"D9FFC7","electronicConfiguration":"[Xe] 4f3 6s2","electronegativity":1.13,"atomicRadius":"","ionRadius":"99 (+3)","vanDerWaalsRadius":"","ionizationEnergy":527,"electronAffinity":-50,"oxidationStates":"2, 3, 4","standardState":"solid","bondingType":"metallic","meltingPoint":1204,"boilingPoint":3563,"density":6.64,"groupBlock":"lanthanoid","yearDiscovered":1885},
{"atomicNumber":60,"symbol":"Nd","name":"Neodymium","atomicMass":"144.242(3)","cpkHexColor":"C7FFC7","electronicConfiguration":"[Xe] 4f4 6s2","electronegativity":1.14,"atomicRadius":"","ionRadius":"129 (+2)","vanDerWaalsRadius":"","ionizationEnergy":533,"electronAffinity":-50,"oxidationStates":"2, 3","standardState":"solid","bondingType":"metallic","meltingPoint":1294,"boilingPoint":3373,"density":7.01,"groupBlock":"lanthanoid","yearDiscovered":1885},
{"atomicNumber":61,"symbol":"Pm","name":"Promethium","atomicMass":"145","cpkHexColor":"A3FFC7","electronicConfiguration":"[Xe] 4f5 6s2","electronegativity":1.13,"atomicRadius":"","ionRadius":"97 (+3)","vanDerWaalsRadius":"","ionizationEnergy":540,"electronAffinity":-50,"oxidationStates":3,"standardState":"solid","bondingType":"metallic","meltingPoint":1373,"boilingPoint":3273,"density":7.264,"groupBlock":"lanthanoid","yearDiscovered":1947},
{"atomicNumber":62,"symbol":"Sm","name":"Samarium","atomicMass":"150.36(2)","cpkHexColor":"8FFFC7","electronicConfiguration":"[Xe] 4f6 6s2","electronegativity":1.17,"atomicRadius":"","ionRadius":"122 (+2)","vanDerWaalsRadius":"","ionizationEnergy":545,"electronAffinity":-50,"oxidationStates":"2, 3","standardState":"solid","bondingType":"metallic","meltingPoint":1345,"boilingPoint":2076,"density":7.353,"groupBlock":"lanthanoid","yearDiscovered":1853},
{"atomicNumber":63,"symbol":"Eu","name":"Europium","atomicMass":"151.964(1)","cpkHexColor":"61FFC7","electronicConfiguration":"[Xe] 4f7 6s2","electronegativity":1.2,"atomicRadius":"","ionRadius":"117 (+2)","vanDerWaalsRadius":"","ionizationEnergy":547,"electronAffinity":-50,"oxidationStates":"2, 3","standardState":"solid","bondingType":"metallic","meltingPoint":1095,"boilingPoint":1800,"density":5.244,"groupBlock":"lanthanoid","yearDiscovered":1901},
{"atomicNumber":64,"symbol":"Gd","name":"Gadolinium","atomicMass":"157.25(3)","cpkHexColor":"45FFC7","electronicConfiguration":"[Xe] 4f7 5d1 6s2","electronegativity":1.2,"atomicRadius":"","ionRadius":"93.8 (+3)","vanDerWaalsRadius":"","ionizationEnergy":593,"electronAffinity":-50,"oxidationStates":"1, 2, 3","standardState":"solid","bondingType":"metallic","meltingPoint":1586,"boilingPoint":3523,"density":7.901,"groupBlock":"lanthanoid","yearDiscovered":1880},
{"atomicNumber":65,"symbol":"Tb","name":"Terbium","atomicMass":"158.92535(2)","cpkHexColor":"30FFC7","electronicConfiguration":"[Xe] 4f9 6s2","electronegativity":1.2,"atomicRadius":"","ionRadius":"92.3 (+3)","vanDerWaalsRadius":"","ionizationEnergy":566,"electronAffinity":-50,"oxidationStates":"1, 3, 4","standardState":"solid","bondingType":"metallic","meltingPoint":1629,"boilingPoint":3503,"density":8.219,"groupBlock":"lanthanoid","yearDiscovered":1843},
{"atomicNumber":66,"symbol":"Dy","name":"Dysprosium","atomicMass":"162.500(1)","cpkHexColor":"1FFFC7","electronicConfiguration":"[Xe] 4f10 6s2","electronegativity":1.22,"atomicRadius":"","ionRadius":"107 (+2)","vanDerWaalsRadius":"","ionizationEnergy":573,"electronAffinity":-50,"oxidationStates":"2, 3","standardState":"solid","bondingType":"metallic","meltingPoint":1685,"boilingPoint":2840,"density":8.551,"groupBlock":"lanthanoid","yearDiscovered":1886},
{"atomicNumber":67,"symbol":"Ho","name":"Holmium","atomicMass":"164.93032(2)","cpkHexColor":"00FF9C","electronicConfiguration":"[Xe] 4f11 6s2","electronegativity":1.23,"atomicRadius":"","ionRadius":"90.1 (+3)","vanDerWaalsRadius":"","ionizationEnergy":581,"electronAffinity":-50,"oxidationStates":3,"standardState":"solid","bondingType":"metallic","meltingPoint":1747,"boilingPoint":2973,"density":8.795,"groupBlock":"lanthanoid","yearDiscovered":1878},
{"atomicNumber":68,"symbol":"Er","name":"Erbium","atomicMass":"167.259(3)","cpkHexColor":0,"electronicConfiguration":"[Xe] 4f12 6s2","electronegativity":1.24,"atomicRadius":"","ionRadius":"89 (+3)","vanDerWaalsRadius":"","ionizationEnergy":589,"electronAffinity":-50,"oxidationStates":3,"standardState":"solid","bondingType":"metallic","meltingPoint":1770,"boilingPoint":3141,"density":9.066,"groupBlock":"lanthanoid","yearDiscovered":1842},
{"atomicNumber":69,"symbol":"Tm","name":"Thulium","atomicMass":"168.93421(2)","cpkHexColor":"00D452","electronicConfiguration":"[Xe] 4f13 6s2","electronegativity":1.25,"atomicRadius":"","ionRadius":"103 (+2)","vanDerWaalsRadius":"","ionizationEnergy":597,"electronAffinity":-50,"oxidationStates":"2, 3","standardState":"solid","bondingType":"metallic","meltingPoint":1818,"boilingPoint":2223,"density":9.321,"groupBlock":"lanthanoid","yearDiscovered":1879},
{"atomicNumber":70,"symbol":"Yb","name":"Ytterbium","atomicMass":"173.054(5)","cpkHexColor":"00BF38","electronicConfiguration":"[Xe] 4f14 6s2","electronegativity":1.1,"atomicRadius":"","ionRadius":"102 (+2)","vanDerWaalsRadius":"","ionizationEnergy":603,"electronAffinity":-50,"oxidationStates":"2, 3","standardState":"solid","bondingType":"metallic","meltingPoint":1092,"boilingPoint":1469,"density":6.57,"groupBlock":"lanthanoid","yearDiscovered":1878},
{"atomicNumber":71,"symbol":"Lu","name":"Lutetium","atomicMass":"174.9668(1)","cpkHexColor":"00AB24","electronicConfiguration":"[Xe] 4f14 5d1 6s2","electronegativity":1.27,"atomicRadius":160,"ionRadius":"86.1 (+3)","vanDerWaalsRadius":"","ionizationEnergy":524,"electronAffinity":-50,"oxidationStates":3,"standardState":"solid","bondingType":"metallic","meltingPoint":1936,"boilingPoint":3675,"density":9.841,"groupBlock":"lanthanoid","yearDiscovered":1907},
{"atomicNumber":72,"symbol":"Hf","name":"Hafnium","atomicMass":"178.49(2)","cpkHexColor":"4DC2FF","electronicConfiguration":"[Xe] 4f14 5d2 6s2","electronegativity":1.3,"atomicRadius":150,"ionRadius":"71 (+4)","vanDerWaalsRadius":"","ionizationEnergy":659,"electronAffinity":0,"oxidationStates":"2, 3, 4","standardState":"solid","bondingType":"metallic","meltingPoint":2506,"boilingPoint":4876,"density":13.31,"groupBlock":"transition metal","yearDiscovered":1923},
{"atomicNumber":73,"symbol":"Ta","name":"Tantalum","atomicMass":"180.94788(2)","cpkHexColor":"4DA6FF","electronicConfiguration":"[Xe] 4f14 5d3 6s2","electronegativity":1.5,"atomicRadius":138,"ionRadius":"72 (+3)","vanDerWaalsRadius":"","ionizationEnergy":761,"electronAffinity":-31,"oxidationStates":"-1, 2, 3, 4, 5","standardState":"solid","bondingType":"metallic","meltingPoint":3290,"boilingPoint":5731,"density":16.65,"groupBlock":"transition metal","yearDiscovered":1802},
{"atomicNumber":74,"symbol":"W","name":"Tungsten","atomicMass":"183.84(1)","cpkHexColor":"2194D6","electronicConfiguration":"[Xe] 4f14 5d4 6s2","electronegativity":2.36,"atomicRadius":146,"ionRadius":"66 (+4)","vanDerWaalsRadius":"","ionizationEnergy":770,"electronAffinity":-79,"oxidationStates":"-2, -1, 1, 2, 3, 4, 5, 6","standardState":"solid","bondingType":"metallic","meltingPoint":3695,"boilingPoint":5828,"density":19.25,"groupBlock":"transition metal","yearDiscovered":1783},
{"atomicNumber":75,"symbol":"Re","name":"Rhenium","atomicMass":"186.207(1)","cpkHexColor":"267DAB","electronicConfiguration":"[Xe] 4f14 5d5 6s2","electronegativity":1.9,"atomicRadius":159,"ionRadius":"63 (+4)","vanDerWaalsRadius":"","ionizationEnergy":760,"electronAffinity":-15,"oxidationStates":"-3, -1, 1, 2, 3, 4, 5, 6, 7","standardState":"solid","bondingType":"metallic","meltingPoint":3459,"boilingPoint":5869,"density":21.02,"groupBlock":"transition metal","yearDiscovered":1925},
{"atomicNumber":76,"symbol":"Os","name":"Osmium","atomicMass":"190.23(3)","cpkHexColor":266696,"electronicConfiguration":"[Xe] 4f14 5d6 6s2","electronegativity":2.2,"atomicRadius":128,"ionRadius":"63 (+4)","vanDerWaalsRadius":"","ionizationEnergy":840,"electronAffinity":-106,"oxidationStates":"-2, -1, 1, 2, 3, 4, 5, 6, 7, 8","standardState":"solid","bondingType":"metallic","meltingPoint":3306,"boilingPoint":5285,"density":22.61,"groupBlock":"transition metal","yearDiscovered":1803},
{"atomicNumber":77,"symbol":"Ir","name":"Iridium","atomicMass":"192.217(3)","cpkHexColor":175487,"electronicConfiguration":"[Xe] 4f14 5d7 6s2","electronegativity":2.2,"atomicRadius":137,"ionRadius":"68 (+3)","vanDerWaalsRadius":"","ionizationEnergy":880,"electronAffinity":-151,"oxidationStates":"-3, -1, 1, 2, 3, 4, 5, 6","standardState":"solid","bondingType":"metallic","meltingPoint":2739,"boilingPoint":4701,"density":22.65,"groupBlock":"transition metal","yearDiscovered":1803},
{"atomicNumber":78,"symbol":"Pt","name":"Platinum","atomicMass":"195.084(9)","cpkHexColor":"D0D0E0","electronicConfiguration":"[Xe] 4f14 5d9 6s1","electronegativity":2.28,"atomicRadius":128,"ionRadius":"86 (+2)","vanDerWaalsRadius":175,"ionizationEnergy":870,"electronAffinity":-205,"oxidationStates":"2, 4, 5, 6","standardState":"solid","bondingType":"metallic","meltingPoint":2041,"boilingPoint":4098,"density":21.09,"groupBlock":"transition metal","yearDiscovered":"Ancient"},
{"atomicNumber":79,"symbol":"Au","name":"Gold","atomicMass":"196.966569(4)","cpkHexColor":"FFD123","electronicConfiguration":"[Xe] 4f14 5d10 6s1","electronegativity":2.54,"atomicRadius":144,"ionRadius":"137 (+1)","vanDerWaalsRadius":166,"ionizationEnergy":890,"electronAffinity":-223,"oxidationStates":"-1, 1, 2, 3, 5","standardState":"solid","bondingType":"metallic","meltingPoint":1337,"boilingPoint":3129,"density":19.3,"groupBlock":"transition metal","yearDiscovered":"Ancient"},
{"atomicNumber":80,"symbol":"Hg","name":"Mercury","atomicMass":"200.59(2)","cpkHexColor":"B8B8D0","electronicConfiguration":"[Xe] 4f14 5d10 6s2","electronegativity":2,"atomicRadius":149,"ionRadius":"119 (+1)","vanDerWaalsRadius":155,"ionizationEnergy":1007,"electronAffinity":0,"oxidationStates":"1, 2, 4","standardState":"liquid","bondingType":"metallic","meltingPoint":234,"boilingPoint":630,"density":13.534,"groupBlock":"transition metal","yearDiscovered":"Ancient"},
{"atomicNumber":81,"symbol":"Tl","name":"Thallium","atomicMass":"204.3833(2)","cpkHexColor":"A6544D","electronicConfiguration":"[Xe] 4f14 5d10 6s2 6p1","electronegativity":2.04,"atomicRadius":148,"ionRadius":"150 (+1)","vanDerWaalsRadius":196,"ionizationEnergy":589,"electronAffinity":-19,"oxidationStates":"1, 3","standardState":"solid","bondingType":"metallic","meltingPoint":577,"boilingPoint":1746,"density":11.85,"groupBlock":"metal","yearDiscovered":1861},
{"atomicNumber":82,"symbol":"Pb","name":"Lead","atomicMass":"207.2(1)","cpkHexColor":575961,"electronicConfiguration":"[Xe] 4f14 5d10 6s2 6p2","electronegativity":2.33,"atomicRadius":147,"ionRadius":"119 (+2)","vanDerWaalsRadius":202,"ionizationEnergy":716,"electronAffinity":-35,"oxidationStates":"-4, 2, 4","standardState":"solid","bondingType":"metallic","meltingPoint":601,"boilingPoint":2022,"density":11.34,"groupBlock":"metal","yearDiscovered":"Ancient"},
{"atomicNumber":83,"symbol":"Bi","name":"Bismuth","atomicMass":"208.98040(1)","cpkHexColor":"9E4FB5","electronicConfiguration":"[Xe] 4f14 5d10 6s2 6p3","electronegativity":2.02,"atomicRadius":146,"ionRadius":"103 (+3)","vanDerWaalsRadius":"","ionizationEnergy":703,"electronAffinity":-91,"oxidationStates":"-3, 3, 5","standardState":"solid","bondingType":"metallic","meltingPoint":544,"boilingPoint":1837,"density":9.78,"groupBlock":"metal","yearDiscovered":"Ancient"},
{"atomicNumber":84,"symbol":"Po","name":"Polonium","atomicMass":"209","cpkHexColor":"AB5C00","electronicConfiguration":"[Xe] 4f14 5d10 6s2 6p4","electronegativity":2,"atomicRadius":"","ionRadius":"94 (+4)","vanDerWaalsRadius":"","ionizationEnergy":812,"electronAffinity":-183,"oxidationStates":"-2, 2, 4, 6","standardState":"solid","bondingType":"metallic","meltingPoint":527,"boilingPoint":1235,"density":9.196,"groupBlock":"metalloid","yearDiscovered":1898},
{"atomicNumber":85,"symbol":"At","name":"Astatine","atomicMass":"210","cpkHexColor":"754F45","electronicConfiguration":"[Xe] 4f14 5d10 6s2 6p5","electronegativity":2.2,"atomicRadius":"","ionRadius":"62 (+7)","vanDerWaalsRadius":"","ionizationEnergy":920,"electronAffinity":-270,"oxidationStates":"-1, 1, 3, 5","standardState":"solid","bondingType":"covalent network","meltingPoint":575,"boilingPoint":"","density":"","groupBlock":"halogen","yearDiscovered":1940},
{"atomicNumber":86,"symbol":"Rn","name":"Radon","atomicMass":"222","cpkHexColor":428296,"electronicConfiguration":"[Xe] 4f14 5d10 6s2 6p6","electronegativity":"","atomicRadius":145,"ionRadius":"","vanDerWaalsRadius":"","ionizationEnergy":1037,"electronAffinity":"","oxidationStates":2,"standardState":"gas","bondingType":"atomic","meltingPoint":202,"boilingPoint":211,"density":0.00973,"groupBlock":"noble gas","yearDiscovered":1900},
{"atomicNumber":87,"symbol":"Fr","name":"Francium","atomicMass":"223","cpkHexColor":420066,"electronicConfiguration":"[Rn] 7s1","electronegativity":0.7,"atomicRadius":"","ionRadius":"180 (+1)","vanDerWaalsRadius":"","ionizationEnergy":380,"electronAffinity":"","oxidationStates":1,"standardState":"solid","bondingType":"metallic","meltingPoint":"","boilingPoint":"","density":"","groupBlock":"alkali metal","yearDiscovered":1939},
{"atomicNumber":88,"symbol":"Ra","name":"Radium","atomicMass":"226","cpkHexColor":"007D00","electronicConfiguration":"[Rn] 7s2","electronegativity":0.9,"atomicRadius":"","ionRadius":"148 (+2)","vanDerWaalsRadius":"","ionizationEnergy":509,"electronAffinity":"","oxidationStates":2,"standardState":"solid","bondingType":"metallic","meltingPoint":973,"boilingPoint":2010,"density":5,"groupBlock":"alkaline earth metal","yearDiscovered":1898},
{"atomicNumber":89,"symbol":"Ac","name":"Actinium","atomicMass":"227","cpkHexColor":"70ABFA","electronicConfiguration":"[Rn] 6d1 7s2","electronegativity":1.1,"atomicRadius":"","ionRadius":"112 (+3)","vanDerWaalsRadius":"","ionizationEnergy":499,"electronAffinity":"","oxidationStates":3,"standardState":"solid","bondingType":"metallic","meltingPoint":1323,"boilingPoint":3473,"density":10.07,"groupBlock":"actinoid","yearDiscovered":1899},
{"atomicNumber":90,"symbol":"Th","name":"Thorium","atomicMass":"232.03806(2)","cpkHexColor":"00BAFF","electronicConfiguration":"[Rn] 6d2 7s2","electronegativity":1.3,"atomicRadius":"","ionRadius":"94 (+4)","vanDerWaalsRadius":"","ionizationEnergy":587,"electronAffinity":"","oxidationStates":"2, 3, 4","standardState":"solid","bondingType":"metallic","meltingPoint":2023,"boilingPoint":5093,"density":11.724,"groupBlock":"actinoid","yearDiscovered":1828},
{"atomicNumber":91,"symbol":"Pa","name":"Protactinium","atomicMass":"231.03588(2)","cpkHexColor":"00A1FF","electronicConfiguration":"[Rn] 5f2 6d1 7s2","electronegativity":1.5,"atomicRadius":"","ionRadius":"104 (+3)","vanDerWaalsRadius":"","ionizationEnergy":568,"electronAffinity":"","oxidationStates":"3, 4, 5","standardState":"solid","bondingType":"metallic","meltingPoint":1845,"boilingPoint":4273,"density":15.37,"groupBlock":"actinoid","yearDiscovered":1913},
{"atomicNumber":92,"symbol":"U","name":"Uranium","atomicMass":"238.02891(3)","cpkHexColor":"008FFF","electronicConfiguration":"[Rn] 5f3 6d1 7s2","electronegativity":1.38,"atomicRadius":"","ionRadius":"102.5 (+3)","vanDerWaalsRadius":186,"ionizationEnergy":598,"electronAffinity":"","oxidationStates":"3, 4, 5, 6","standardState":"solid","bondingType":"metallic","meltingPoint":1408,"boilingPoint":4200,"density":19.05,"groupBlock":"actinoid","yearDiscovered":1789},
{"atomicNumber":93,"symbol":"Np","name":"Neptunium","atomicMass":"237","cpkHexColor":"0080FF","electronicConfiguration":"[Rn] 5f4 6d1 7s2","electronegativity":1.36,"atomicRadius":"","ionRadius":"110 (+2)","vanDerWaalsRadius":"","ionizationEnergy":605,"electronAffinity":"","oxidationStates":"3, 4, 5, 6, 7","standardState":"solid","bondingType":"metallic","meltingPoint":917,"boilingPoint":4273,"density":20.45,"groupBlock":"actinoid","yearDiscovered":1940},
{"atomicNumber":94,"symbol":"Pu","name":"Plutonium","atomicMass":"244","cpkHexColor":"006BFF","electronicConfiguration":"[Rn] 5f6 7s2","electronegativity":1.28,"atomicRadius":"","ionRadius":"100 (+3)","vanDerWaalsRadius":"","ionizationEnergy":585,"electronAffinity":"","oxidationStates":"3, 4, 5, 6, 7","standardState":"solid","bondingType":"metallic","meltingPoint":913,"boilingPoint":3503,"density":19.816,"groupBlock":"actinoid","yearDiscovered":1940},
{"atomicNumber":95,"symbol":"Am","name":"Americium","atomicMass":"243","cpkHexColor":"545CF2","electronicConfiguration":"[Rn] 5f7 7s2","electronegativity":1.3,"atomicRadius":"","ionRadius":"126 (+2)","vanDerWaalsRadius":"","ionizationEnergy":578,"electronAffinity":"","oxidationStates":"2, 3, 4, 5, 6","standardState":"solid","bondingType":"metallic","meltingPoint":1449,"boilingPoint":2284,"density":"","groupBlock":"actinoid","yearDiscovered":1944},
{"atomicNumber":96,"symbol":"Cm","name":"Curium","atomicMass":"247","cpkHexColor":"785CE3","electronicConfiguration":"[Rn] 5f7 6d1 7s2","electronegativity":1.3,"atomicRadius":"","ionRadius":"97 (+3)","vanDerWaalsRadius":"","ionizationEnergy":581,"electronAffinity":"","oxidationStates":"3, 4","standardState":"solid","bondingType":"metallic","meltingPoint":1618,"boilingPoint":3383,"density":13.51,"groupBlock":"actinoid","yearDiscovered":1944},
{"atomicNumber":97,"symbol":"Bk","name":"Berkelium","atomicMass":"247","cpkHexColor":"8A4FE3","electronicConfiguration":"[Rn] 5f9 7s2","electronegativity":1.3,"atomicRadius":"","ionRadius":"96 (+3)","vanDerWaalsRadius":"","ionizationEnergy":601,"electronAffinity":"","oxidationStates":"3, 4","standardState":"solid","bondingType":"metallic","meltingPoint":1323,"boilingPoint":"","density":14.78,"groupBlock":"actinoid","yearDiscovered":1949},
{"atomicNumber":98,"symbol":"Cf","name":"Californium","atomicMass":"251","cpkHexColor":"A136D4","electronicConfiguration":"[Rn] 5f10 7s2","electronegativity":1.3,"atomicRadius":"","ionRadius":"95 (+3)","vanDerWaalsRadius":"","ionizationEnergy":608,"electronAffinity":"","oxidationStates":"2, 3, 4","standardState":"solid","bondingType":"metallic","meltingPoint":1173,"boilingPoint":"","density":15.1,"groupBlock":"actinoid","yearDiscovered":1950},
{"atomicNumber":99,"symbol":"Es","name":"Einsteinium","atomicMass":"252","cpkHexColor":"B31FD4","electronicConfiguration":"[Rn] 5f11 7s2","electronegativity":1.3,"atomicRadius":"","ionRadius":"","vanDerWaalsRadius":"","ionizationEnergy":619,"electronAffinity":"","oxidationStates":"2, 3","standardState":"solid","bondingType":"","meltingPoint":1133,"boilingPoint":"","density":"","groupBlock":"actinoid","yearDiscovered":1952},
{"atomicNumber":100,"symbol":"Fm","name":"Fermium","atomicMass":"257","cpkHexColor":"B31FBA","electronicConfiguration":"[Rn] 5f12 7s2","electronegativity":1.3,"atomicRadius":"","ionRadius":"","vanDerWaalsRadius":"","ionizationEnergy":627,"electronAffinity":"","oxidationStates":"2, 3","standardState":"","bondingType":"","meltingPoint":1800,"boilingPoint":"","density":"","groupBlock":"actinoid","yearDiscovered":1952},
{"atomicNumber":101,"symbol":"Md","name":"Mendelevium","atomicMass":"258","cpkHexColor":"B30DA6","electronicConfiguration":"[Rn] 5f13 7s2","electronegativity":1.3,"atomicRadius":"","ionRadius":"","vanDerWaalsRadius":"","ionizationEnergy":635,"electronAffinity":"","oxidationStates":"2, 3","standardState":"","bondingType":"","meltingPoint":1100,"boilingPoint":"","density":"","groupBlock":"actinoid","yearDiscovered":1955},
{"atomicNumber":102,"symbol":"No","name":"Nobelium","atomicMass":"259","cpkHexColor":"BD0D87","electronicConfiguration":"[Rn] 5f14 7s2","electronegativity":1.3,"atomicRadius":"","ionRadius":"","vanDerWaalsRadius":"","ionizationEnergy":642,"electronAffinity":"","oxidationStates":"2, 3","standardState":"","bondingType":"","meltingPoint":1100,"boilingPoint":"","density":"","groupBlock":"actinoid","yearDiscovered":1957},
{"atomicNumber":103,"symbol":"Lr","name":"Lawrencium","atomicMass":"262","cpkHexColor":"C70066","electronicConfiguration":"[Rn] 5f14 7s2 7p1","electronegativity":1.3,"atomicRadius":"","ionRadius":"","vanDerWaalsRadius":"","ionizationEnergy":"","electronAffinity":"","oxidationStates":3,"standardState":"","bondingType":"","meltingPoint":1900,"boilingPoint":"","density":"","groupBlock":"transition metal","yearDiscovered":1961},
{"atomicNumber":104,"symbol":"Rf","name":"Rutherfordium","atomicMass":"267","cpkHexColor":"CC0059","electronicConfiguration":"[Rn] 5f14 6d2 7s2","electronegativity":"","atomicRadius":"","ionRadius":"","vanDerWaalsRadius":"","ionizationEnergy":"","electronAffinity":"","oxidationStates":4,"standardState":"","bondingType":"","meltingPoint":"","boilingPoint":"","density":"","groupBlock":"transition metal","yearDiscovered":1969},
{"atomicNumber":105,"symbol":"Db","name":"Dubnium","atomicMass":"268","cpkHexColor":"D1004F","electronicConfiguration":"[Rn] 5f14 6d3 7s2","electronegativity":"","atomicRadius":"","ionRadius":"","vanDerWaalsRadius":"","ionizationEnergy":"","electronAffinity":"","oxidationStates":"","standardState":"","bondingType":"","meltingPoint":"","boilingPoint":"","density":"","groupBlock":"transition metal","yearDiscovered":1967},
{"atomicNumber":106,"symbol":"Sg","name":"Seaborgium","atomicMass":"271","cpkHexColor":"D90045","electronicConfiguration":"[Rn] 5f14 6d4 7s2","electronegativity":"","atomicRadius":"","ionRadius":"","vanDerWaalsRadius":"","ionizationEnergy":"","electronAffinity":"","oxidationStates":"","standardState":"","bondingType":"","meltingPoint":"","boilingPoint":"","density":"","groupBlock":"transition metal","yearDiscovered":1974},
{"atomicNumber":107,"symbol":"Bh","name":"Bohrium","atomicMass":"272","cpkHexColor":"E00038","electronicConfiguration":"[Rn] 5f14 6d5 7s2","electronegativity":"","atomicRadius":"","ionRadius":"","vanDerWaalsRadius":"","ionizationEnergy":"","electronAffinity":"","oxidationStates":"","standardState":"","bondingType":"","meltingPoint":"","boilingPoint":"","density":"","groupBlock":"transition metal","yearDiscovered":1976},
{"atomicNumber":108,"symbol":"Hs","name":"Hassium","atomicMass":"270","cpkHexColor":"E6002E","electronicConfiguration":"[Rn] 5f14 6d6 7s2","electronegativity":"","atomicRadius":"","ionRadius":"","vanDerWaalsRadius":"","ionizationEnergy":"","electronAffinity":"","oxidationStates":"","standardState":"","bondingType":"","meltingPoint":"","boilingPoint":"","density":"","groupBlock":"transition metal","yearDiscovered":1984},
{"atomicNumber":109,"symbol":"Mt","name":"Meitnerium","atomicMass":"276","cpkHexColor":"EB0026","electronicConfiguration":"[Rn] 5f14 6d7 7s2","electronegativity":"","atomicRadius":"","ionRadius":"","vanDerWaalsRadius":"","ionizationEnergy":"","electronAffinity":"","oxidationStates":"","standardState":"","bondingType":"","meltingPoint":"","boilingPoint":"","density":"","groupBlock":"transition metal","yearDiscovered":1982},
{"atomicNumber":110,"symbol":"Ds","name":"Darmstadtium","atomicMass":"281","cpkHexColor":"","electronicConfiguration":"[Rn] 5f14 6d9 7s1","electronegativity":"","atomicRadius":"","ionRadius":"","vanDerWaalsRadius":"","ionizationEnergy":"","electronAffinity":"","oxidationStates":"","standardState":"","bondingType":"","meltingPoint":"","boilingPoint":"","density":"","groupBlock":"transition metal","yearDiscovered":1994},
{"atomicNumber":111,"symbol":"Rg","name":"Roentgenium","atomicMass":"280","cpkHexColor":"","electronicConfiguration":"[Rn] 5f14 6d10 7s1","electronegativity":"","atomicRadius":"","ionRadius":"","vanDerWaalsRadius":"","ionizationEnergy":"","electronAffinity":"","oxidationStates":"","standardState":"","bondingType":"","meltingPoint":"","boilingPoint":"","density":"","groupBlock":"transition metal","yearDiscovered":1994},
{"atomicNumber":112,"symbol":"Cn","name":"Copernicium","atomicMass":"285","cpkHexColor":"","electronicConfiguration":"[Rn] 5f14 6d10 7s2","electronegativity":"","atomicRadius":"","ionRadius":"","vanDerWaalsRadius":"","ionizationEnergy":"","electronAffinity":"","oxidationStates":"","standardState":"","bondingType":"","meltingPoint":"","boilingPoint":"","density":"","groupBlock":"transition metal","yearDiscovered":1996},
{"atomicNumber":113,"symbol":"Nh","name":"Nihonium","atomicMass":"284","cpkHexColor":"","electronicConfiguration":"[Rn] 5f14 6d10 7s2 7p1","electronegativity":"","atomicRadius":"","ionRadius":"","vanDerWaalsRadius":"","ionizationEnergy":"","electronAffinity":"","oxidationStates":"","standardState":"","bondingType":"","meltingPoint":"","boilingPoint":"","density":"","groupBlock":"post-transition metal","yearDiscovered":2003},
{"atomicNumber":114,"symbol":"Fl","name":"Flerovium","atomicMass":"289","cpkHexColor":"","electronicConfiguration":"[Rn] 5f14 6d10 7s2 7p2","electronegativity":"","atomicRadius":"","ionRadius":"","vanDerWaalsRadius":"","ionizationEnergy":"","electronAffinity":"","oxidationStates":"","standardState":"","bondingType":"","meltingPoint":"","boilingPoint":"","density":"","groupBlock":"post-transition metal","yearDiscovered":1998},
{"atomicNumber":115,"symbol":"Mc","name":"Moscovium","atomicMass":"288","cpkHexColor":"","electronicConfiguration":"[Rn] 5f14 6d10 7s2 7p3","electronegativity":"","atomicRadius":"","ionRadius":"","vanDerWaalsRadius":"","ionizationEnergy":"","electronAffinity":"","oxidationStates":"","standardState":"","bondingType":"","meltingPoint":"","boilingPoint":"","density":"","groupBlock":"post-transition metal","yearDiscovered":2003},
{"atomicNumber":116,"symbol":"Lv","name":"Livermorium","atomicMass":"293","cpkHexColor":"","electronicConfiguration":"[Rn] 5f14 6d10 7s2 7p4","electronegativity":"","atomicRadius":"","ionRadius":"","vanDerWaalsRadius":"","ionizationEnergy":"","electronAffinity":"","oxidationStates":"","standardState":"","bondingType":"","meltingPoint":"","boilingPoint":"","density":"","groupBlock":"post-transition metal","yearDiscovered":2000},
{"atomicNumber":117,"symbol":"Ts","name":"Tennessine","atomicMass":"294","cpkHexColor":"","electronicConfiguration":"[Rn] 5f14 6d10 7s2 7p5","electronegativity":"","atomicRadius":"","ionRadius":"","vanDerWaalsRadius":"","ionizationEnergy":"","electronAffinity":"","oxidationStates":"","standardState":"","bondingType":"","meltingPoint":"","boilingPoint":"","density":"","groupBlock":"post-transition metal","yearDiscovered":2010},
{"atomicNumber":118,"symbol":"Og","name":"Oganesson","atomicMass":"294","cpkHexColor":"","electronicConfiguration":"[Rn] 5f14 6d10 7s2 7p6","electronegativity":"","atomicRadius":"","ionRadius":"","vanDerWaalsRadius":"","ionizationEnergy":"","electronAffinity":"","oxidationStates":"","standardState":"","bondingType":"","meltingPoint":"","boilingPoint":"","density":"","groupBlock":"noble gas","yearDiscovered":2002}
]
// from https://www.nist.gov/pml/atomic-weights-and-isotopic-compositions-relative-atomic-masses
var isotope_data = {"21": {"36": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "44.955908(5)", "Atomic Number": "21", "Atomic Symbol": "Sc", "Relative Atomic Mass": "36.01648(32#)", "Mass Number": "36"}, "37": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "44.955908(5)", "Atomic Number": "21", "Atomic Symbol": "Sc", "Relative Atomic Mass": "37.00374(32#)", "Mass Number": "37"}, "38": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "44.955908(5)", "Atomic Number": "21", "Atomic Symbol": "Sc", "Relative Atomic Mass": "37.99512(21#)", "Mass Number": "38"}, "39": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "44.955908(5)", "Atomic Number": "21", "Atomic Symbol": "Sc", "Relative Atomic Mass": "38.984785(26)", "Mass Number": "39"}, "40": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "44.955908(5)", "Atomic Number": "21", "Atomic Symbol": "Sc", "Relative Atomic Mass": "39.9779673(30)", "Mass Number": "40"}, "41": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "44.955908(5)", "Atomic Number": "21", "Atomic Symbol": "Sc", "Relative Atomic Mass": "40.969251105(88)", "Mass Number": "41"}, "42": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "44.955908(5)", "Atomic Number": "21", "Atomic Symbol": "Sc", "Relative Atomic Mass": "41.96551653(18)", "Mass Number": "42"}, "43": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "44.955908(5)", "Atomic Number": "21", "Atomic Symbol": "Sc", "Relative Atomic Mass": "42.9611505(20)", "Mass Number": "43"}, "44": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "44.955908(5)", "Atomic Number": "21", "Atomic Symbol": "Sc", "Relative Atomic Mass": "43.9594029(19)", "Mass Number": "44"}, "45": {"Isotopic Composition": "1", "Notes": " ", "Standard Atomic Weight": "44.955908(5)", "Atomic Number": "21", "Atomic Symbol": "Sc", "Relative Atomic Mass": "44.95590828(77)", "Mass Number": "45"}, "46": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "44.955908(5)", "Atomic Number": "21", "Atomic Symbol": "Sc", "Relative Atomic Mass": "45.95516826(78)", "Mass Number": "46"}, "47": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "44.955908(5)", "Atomic Number": "21", "Atomic Symbol": "Sc", "Relative Atomic Mass": "46.9524037(21)", "Mass Number": "47"}, "48": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "44.955908(5)", "Atomic Number": "21", "Atomic Symbol": "Sc", "Relative Atomic Mass": "47.9522236(53)", "Mass Number": "48"}, "49": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "44.955908(5)", "Atomic Number": "21", "Atomic Symbol": "Sc", "Relative Atomic Mass": "48.9500146(29)", "Mass Number": "49"}, "50": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "44.955908(5)", "Atomic Number": "21", "Atomic Symbol": "Sc", "Relative Atomic Mass": "49.952176(16)", "Mass Number": "50"}, "51": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "44.955908(5)", "Atomic Number": "21", "Atomic Symbol": "Sc", "Relative Atomic Mass": "50.953592(21)", "Mass Number": "51"}, "52": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "44.955908(5)", "Atomic Number": "21", "Atomic Symbol": "Sc", "Relative Atomic Mass": "51.95688(15)", "Mass Number": "52"}, "53": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "44.955908(5)", "Atomic Number": "21", "Atomic Symbol": "Sc", "Relative Atomic Mass": "52.95909(29)", "Mass Number": "53"}, "54": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "44.955908(5)", "Atomic Number": "21", "Atomic Symbol": "Sc", "Relative Atomic Mass": "53.96393(39)", "Mass Number": "54"}, "55": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "44.955908(5)", "Atomic Number": "21", "Atomic Symbol": "Sc", "Relative Atomic Mass": "54.96782(50)", "Mass Number": "55"}, "56": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "44.955908(5)", "Atomic Number": "21", "Atomic Symbol": "Sc", "Relative Atomic Mass": "55.97345(43#)", "Mass Number": "56"}, "57": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "44.955908(5)", "Atomic Number": "21", "Atomic Symbol": "Sc", "Relative Atomic Mass": "56.97777(54#)", "Mass Number": "57"}, "58": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "44.955908(5)", "Atomic Number": "21", "Atomic Symbol": "Sc", "Relative Atomic Mass": "57.98403(64#)", "Mass Number": "58"}, "59": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "44.955908(5)", "Atomic Number": "21", "Atomic Symbol": "Sc", "Relative Atomic Mass": "58.98894(64#)", "Mass Number": "59"}, "60": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "44.955908(5)", "Atomic Number": "21", "Atomic Symbol": "Sc", "Relative Atomic Mass": "59.99565(75#)", "Mass Number": "60"}, "61": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "44.955908(5)", "Atomic Number": "21", "Atomic Symbol": "Sc", "Relative Atomic Mass": "61.00100(86#)", "Mass Number": "61"}}, "104": {"256": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "104", "Atomic Symbol": "Rf", "Relative Atomic Mass": "256.101152(19)", "Mass Number": "256"}, "257": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "104", "Atomic Symbol": "Rf", "Relative Atomic Mass": "257.102918(12)", "Mass Number": "257"}, "258": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "104", "Atomic Symbol": "Rf", "Relative Atomic Mass": "258.103428(34)", "Mass Number": "258"}, "259": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "104", "Atomic Symbol": "Rf", "Relative Atomic Mass": "259.105596(78#)", "Mass Number": "259"}, "260": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "104", "Atomic Symbol": "Rf", "Relative Atomic Mass": "260.10644(22#)", "Mass Number": "260"}, "261": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "104", "Atomic Symbol": "Rf", "Relative Atomic Mass": "261.108773(54)", "Mass Number": "261"}, "262": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "104", "Atomic Symbol": "Rf", "Relative Atomic Mass": "262.10992(24#)", "Mass Number": "262"}, "263": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "104", "Atomic Symbol": "Rf", "Relative Atomic Mass": "263.11249(20#)", "Mass Number": "263"}, "264": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "104", "Atomic Symbol": "Rf", "Relative Atomic Mass": "264.11388(39#)", "Mass Number": "264"}, "265": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "104", "Atomic Symbol": "Rf", "Relative Atomic Mass": "265.11668(39#)", "Mass Number": "265"}, "266": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "104", "Atomic Symbol": "Rf", "Relative Atomic Mass": "266.11817(50#)", "Mass Number": "266"}, "267": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "104", "Atomic Symbol": "Rf", "Relative Atomic Mass": "267.12179(62#)", "Mass Number": "267"}, "268": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "104", "Atomic Symbol": "Rf", "Relative Atomic Mass": "268.12397(77#)", "Mass Number": "268"}, "253": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "104", "Atomic Symbol": "Rf", "Relative Atomic Mass": "253.10044(44#)", "Mass Number": "253"}, "254": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "104", "Atomic Symbol": "Rf", "Relative Atomic Mass": "254.10005(30#)", "Mass Number": "254"}, "255": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "104", "Atomic Symbol": "Rf", "Relative Atomic Mass": "255.10127(12#)", "Mass Number": "255"}}, "11": {"32": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "22.98976928(2)", "Atomic Number": "11", "Atomic Symbol": "Na", "Relative Atomic Mass": "32.02019(13)", "Mass Number": "32"}, "33": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "22.98976928(2)", "Atomic Number": "11", "Atomic Symbol": "Na", "Relative Atomic Mass": "33.02573(64#)", "Mass Number": "33"}, "34": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "22.98976928(2)", "Atomic Number": "11", "Atomic Symbol": "Na", "Relative Atomic Mass": "34.03359(54#)", "Mass Number": "34"}, "35": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "22.98976928(2)", "Atomic Number": "11", "Atomic Symbol": "Na", "Relative Atomic Mass": "35.04062(63#)", "Mass Number": "35"}, "36": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "22.98976928(2)", "Atomic Number": "11", "Atomic Symbol": "Na", "Relative Atomic Mass": "36.04929(64#)", "Mass Number": "36"}, "37": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "22.98976928(2)", "Atomic Number": "11", "Atomic Symbol": "Na", "Relative Atomic Mass": "37.05705(65#)", "Mass Number": "37"}, "18": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "22.98976928(2)", "Atomic Number": "11", "Atomic Symbol": "Na", "Relative Atomic Mass": "18.02688(12)", "Mass Number": "18"}, "19": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "22.98976928(2)", "Atomic Number": "11", "Atomic Symbol": "Na", "Relative Atomic Mass": "19.013880(11)", "Mass Number": "19"}, "20": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "22.98976928(2)", "Atomic Number": "11", "Atomic Symbol": "Na", "Relative Atomic Mass": "20.0073544(12)", "Mass Number": "20"}, "21": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "22.98976928(2)", "Atomic Number": "11", "Atomic Symbol": "Na", "Relative Atomic Mass": "20.99765469(30)", "Mass Number": "21"}, "22": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "22.98976928(2)", "Atomic Number": "11", "Atomic Symbol": "Na", "Relative Atomic Mass": "21.99443741(18)", "Mass Number": "22"}, "23": {"Isotopic Composition": "1", "Notes": " ", "Standard Atomic Weight": "22.98976928(2)", "Atomic Number": "11", "Atomic Symbol": "Na", "Relative Atomic Mass": "22.9897692820(19)", "Mass Number": "23"}, "24": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "22.98976928(2)", "Atomic Number": "11", "Atomic Symbol": "Na", "Relative Atomic Mass": "23.990962950(38)", "Mass Number": "24"}, "25": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "22.98976928(2)", "Atomic Number": "11", "Atomic Symbol": "Na", "Relative Atomic Mass": "24.9899540(13)", "Mass Number": "25"}, "26": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "22.98976928(2)", "Atomic Number": "11", "Atomic Symbol": "Na", "Relative Atomic Mass": "25.9926346(38)", "Mass Number": "26"}, "27": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "22.98976928(2)", "Atomic Number": "11", "Atomic Symbol": "Na", "Relative Atomic Mass": "26.9940765(40)", "Mass Number": "27"}, "28": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "22.98976928(2)", "Atomic Number": "11", "Atomic Symbol": "Na", "Relative Atomic Mass": "27.998939(11)", "Mass Number": "28"}, "29": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "22.98976928(2)", "Atomic Number": "11", "Atomic Symbol": "Na", "Relative Atomic Mass": "29.0028771(79)", "Mass Number": "29"}, "30": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "22.98976928(2)", "Atomic Number": "11", "Atomic Symbol": "Na", "Relative Atomic Mass": "30.0090979(51)", "Mass Number": "30"}, "31": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "22.98976928(2)", "Atomic Number": "11", "Atomic Symbol": "Na", "Relative Atomic Mass": "31.013163(25)", "Mass Number": "31"}}, "93": {"219": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[237]", "Atomic Number": "93", "Atomic Symbol": "Np", "Relative Atomic Mass": "219.03143(21#)", "Mass Number": "219"}, "220": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[237]", "Atomic Number": "93", "Atomic Symbol": "Np", "Relative Atomic Mass": "220.03254(21#)", "Mass Number": "220"}, "221": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[237]", "Atomic Number": "93", "Atomic Symbol": "Np", "Relative Atomic Mass": "221.03204(21#)", "Mass Number": "221"}, "222": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[237]", "Atomic Number": "93", "Atomic Symbol": "Np", "Relative Atomic Mass": "222.03330(21#)", "Mass Number": "222"}, "223": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[237]", "Atomic Number": "93", "Atomic Symbol": "Np", "Relative Atomic Mass": "223.03285(21#)", "Mass Number": "223"}, "224": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[237]", "Atomic Number": "93", "Atomic Symbol": "Np", "Relative Atomic Mass": "224.03422(21#)", "Mass Number": "224"}, "225": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[237]", "Atomic Number": "93", "Atomic Symbol": "Np", "Relative Atomic Mass": "225.033911(77)", "Mass Number": "225"}, "226": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[237]", "Atomic Number": "93", "Atomic Symbol": "Np", "Relative Atomic Mass": "226.035188(95#)", "Mass Number": "226"}, "227": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[237]", "Atomic Number": "93", "Atomic Symbol": "Np", "Relative Atomic Mass": "227.034957(78)", "Mass Number": "227"}, "228": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[237]", "Atomic Number": "93", "Atomic Symbol": "Np", "Relative Atomic Mass": "228.036067(54)", "Mass Number": "228"}, "229": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[237]", "Atomic Number": "93", "Atomic Symbol": "Np", "Relative Atomic Mass": "229.036264(93)", "Mass Number": "229"}, "230": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[237]", "Atomic Number": "93", "Atomic Symbol": "Np", "Relative Atomic Mass": "230.037828(55)", "Mass Number": "230"}, "231": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[237]", "Atomic Number": "93", "Atomic Symbol": "Np", "Relative Atomic Mass": "231.038245(54)", "Mass Number": "231"}, "232": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[237]", "Atomic Number": "93", "Atomic Symbol": "Np", "Relative Atomic Mass": "232.04011(11#)", "Mass Number": "232"}, "233": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[237]", "Atomic Number": "93", "Atomic Symbol": "Np", "Relative Atomic Mass": "233.040741(55)", "Mass Number": "233"}, "234": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[237]", "Atomic Number": "93", "Atomic Symbol": "Np", "Relative Atomic Mass": "234.0428953(91)", "Mass Number": "234"}, "235": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[237]", "Atomic Number": "93", "Atomic Symbol": "Np", "Relative Atomic Mass": "235.0440635(21)", "Mass Number": "235"}, "236": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[237]", "Atomic Number": "93", "Atomic Symbol": "Np", "Relative Atomic Mass": "236.046570(54)", "Mass Number": "236"}, "237": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[237]", "Atomic Number": "93", "Atomic Symbol": "Np", "Relative Atomic Mass": "237.0481736(19)", "Mass Number": "237"}, "238": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[237]", "Atomic Number": "93", "Atomic Symbol": "Np", "Relative Atomic Mass": "238.0509466(19)", "Mass Number": "238"}, "239": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[237]", "Atomic Number": "93", "Atomic Symbol": "Np", "Relative Atomic Mass": "239.0529392(22)", "Mass Number": "239"}, "240": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[237]", "Atomic Number": "93", "Atomic Symbol": "Np", "Relative Atomic Mass": "240.056165(18)", "Mass Number": "240"}, "241": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[237]", "Atomic Number": "93", "Atomic Symbol": "Np", "Relative Atomic Mass": "241.058253(76)", "Mass Number": "241"}, "242": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[237]", "Atomic Number": "93", "Atomic Symbol": "Np", "Relative Atomic Mass": "242.06164(21)", "Mass Number": "242"}, "243": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[237]", "Atomic Number": "93", "Atomic Symbol": "Np", "Relative Atomic Mass": "243.064280(34#)", "Mass Number": "243"}, "244": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[237]", "Atomic Number": "93", "Atomic Symbol": "Np", "Relative Atomic Mass": "244.06785(32#)", "Mass Number": "244"}, "245": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[237]", "Atomic Number": "93", "Atomic Symbol": "Np", "Relative Atomic Mass": "245.07080(43#)", "Mass Number": "245"}}, "101": {"256": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "101", "Atomic Symbol": "Md", "Relative Atomic Mass": "256.09389(13#)", "Mass Number": "256"}, "257": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "101", "Atomic Symbol": "Md", "Relative Atomic Mass": "257.0955424(29)", "Mass Number": "257"}, "258": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "101", "Atomic Symbol": "Md", "Relative Atomic Mass": "258.0984315(50)", "Mass Number": "258"}, "259": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "101", "Atomic Symbol": "Md", "Relative Atomic Mass": "259.10051(22#)", "Mass Number": "259"}, "260": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "101", "Atomic Symbol": "Md", "Relative Atomic Mass": "260.10365(34#)", "Mass Number": "260"}, "261": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "101", "Atomic Symbol": "Md", "Relative Atomic Mass": "261.10583(62#)", "Mass Number": "261"}, "262": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "101", "Atomic Symbol": "Md", "Relative Atomic Mass": "262.10910(45#)", "Mass Number": "262"}, "245": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "101", "Atomic Symbol": "Md", "Relative Atomic Mass": "245.08081(33#)", "Mass Number": "245"}, "246": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "101", "Atomic Symbol": "Md", "Relative Atomic Mass": "246.08171(28#)", "Mass Number": "246"}, "247": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "101", "Atomic Symbol": "Md", "Relative Atomic Mass": "247.08152(22#)", "Mass Number": "247"}, "248": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "101", "Atomic Symbol": "Md", "Relative Atomic Mass": "248.08282(26#)", "Mass Number": "248"}, "249": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "101", "Atomic Symbol": "Md", "Relative Atomic Mass": "249.08291(22#)", "Mass Number": "249"}, "250": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "101", "Atomic Symbol": "Md", "Relative Atomic Mass": "250.08441(32#)", "Mass Number": "250"}, "251": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "101", "Atomic Symbol": "Md", "Relative Atomic Mass": "251.084774(20)", "Mass Number": "251"}, "252": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "101", "Atomic Symbol": "Md", "Relative Atomic Mass": "252.08643(14#)", "Mass Number": "252"}, "253": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "101", "Atomic Symbol": "Md", "Relative Atomic Mass": "253.087144(34#)", "Mass Number": "253"}, "254": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "101", "Atomic Symbol": "Md", "Relative Atomic Mass": "254.08959(11#)", "Mass Number": "254"}, "255": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "101", "Atomic Symbol": "Md", "Relative Atomic Mass": "255.0910841(73)", "Mass Number": "255"}}, "82": {"192": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "191.975775(13)", "Mass Number": "192"}, "193": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "192.976173(53)", "Mass Number": "193"}, "194": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "193.974012(19)", "Mass Number": "194"}, "195": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "194.974543(25)", "Mass Number": "195"}, "196": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "195.972774(15)", "Mass Number": "196"}, "197": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "196.9734312(60)", "Mass Number": "197"}, "198": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "197.972034(16)", "Mass Number": "198"}, "199": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "198.972913(11)", "Mass Number": "199"}, "200": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "199.971819(12)", "Mass Number": "200"}, "201": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "200.972883(23)", "Mass Number": "201"}, "202": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "201.9721520(40)", "Mass Number": "202"}, "203": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "202.9733911(71)", "Mass Number": "203"}, "204": {"Isotopic Composition": "0.014(1)", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "203.9730440(13)", "Mass Number": "204"}, "205": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "204.9744822(13)", "Mass Number": "205"}, "206": {"Isotopic Composition": "0.241(1)", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "205.9744657(13)", "Mass Number": "206"}, "207": {"Isotopic Composition": "0.221(1)", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "206.9758973(13)", "Mass Number": "207"}, "208": {"Isotopic Composition": "0.524(1)", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "207.9766525(13)", "Mass Number": "208"}, "209": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "208.9810905(19)", "Mass Number": "209"}, "210": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "209.9841889(16)", "Mass Number": "210"}, "211": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "210.9887371(28)", "Mass Number": "211"}, "212": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "211.9918977(23)", "Mass Number": "212"}, "213": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "212.9965629(72)", "Mass Number": "213"}, "214": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "213.9998059(25)", "Mass Number": "214"}, "215": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "215.00474(11#)", "Mass Number": "215"}, "216": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "216.00803(21#)", "Mass Number": "216"}, "217": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "217.01314(32#)", "Mass Number": "217"}, "218": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "218.01659(32#)", "Mass Number": "218"}, "219": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "219.02177(43#)", "Mass Number": "219"}, "220": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "220.02541(43#)", "Mass Number": "220"}, "178": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "178.003831(26)", "Mass Number": "178"}, "179": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "179.002201(81)", "Mass Number": "179"}, "180": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "179.997928(15)", "Mass Number": "180"}, "181": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "180.996653(81)", "Mass Number": "181"}, "182": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "181.992672(13)", "Mass Number": "182"}, "183": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "182.991872(30)", "Mass Number": "183"}, "184": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "183.988136(14)", "Mass Number": "184"}, "185": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "184.987610(17)", "Mass Number": "185"}, "186": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "185.984238(12)", "Mass Number": "186"}, "187": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "186.9839109(55)", "Mass Number": "187"}, "188": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "187.980875(11)", "Mass Number": "188"}, "189": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "188.980807(37)", "Mass Number": "189"}, "190": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "189.978082(13)", "Mass Number": "190"}, "191": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "207.2(1)", "Atomic Number": "82", "Atomic Symbol": "Pb", "Relative Atomic Mass": "190.978276(41)", "Mass Number": "191"}}, "17": {"28": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[35.446,35.457]", "Atomic Number": "17", "Atomic Symbol": "Cl", "Relative Atomic Mass": "28.02954(64#)", "Mass Number": "28"}, "29": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[35.446,35.457]", "Atomic Number": "17", "Atomic Symbol": "Cl", "Relative Atomic Mass": "29.01478(43#)", "Mass Number": "29"}, "30": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[35.446,35.457]", "Atomic Number": "17", "Atomic Symbol": "Cl", "Relative Atomic Mass": "30.00477(21#)", "Mass Number": "30"}, "31": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[35.446,35.457]", "Atomic Number": "17", "Atomic Symbol": "Cl", "Relative Atomic Mass": "30.992414(54)", "Mass Number": "31"}, "32": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[35.446,35.457]", "Atomic Number": "17", "Atomic Symbol": "Cl", "Relative Atomic Mass": "31.98568464(60)", "Mass Number": "32"}, "33": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[35.446,35.457]", "Atomic Number": "17", "Atomic Symbol": "Cl", "Relative Atomic Mass": "32.97745199(42)", "Mass Number": "33"}, "34": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[35.446,35.457]", "Atomic Number": "17", "Atomic Symbol": "Cl", "Relative Atomic Mass": "33.973762485(52)", "Mass Number": "34"}, "35": {"Isotopic Composition": "0.7576(10)", "Notes": "m", "Standard Atomic Weight": "[35.446,35.457]", "Atomic Number": "17", "Atomic Symbol": "Cl", "Relative Atomic Mass": "34.968852682(37)", "Mass Number": "35"}, "36": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[35.446,35.457]", "Atomic Number": "17", "Atomic Symbol": "Cl", "Relative Atomic Mass": "35.968306809(38)", "Mass Number": "36"}, "37": {"Isotopic Composition": "0.2424(10)", "Notes": "m", "Standard Atomic Weight": "[35.446,35.457]", "Atomic Number": "17", "Atomic Symbol": "Cl", "Relative Atomic Mass": "36.965902602(55)", "Mass Number": "37"}, "38": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[35.446,35.457]", "Atomic Number": "17", "Atomic Symbol": "Cl", "Relative Atomic Mass": "37.96801044(11)", "Mass Number": "38"}, "39": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[35.446,35.457]", "Atomic Number": "17", "Atomic Symbol": "Cl", "Relative Atomic Mass": "38.9680082(19)", "Mass Number": "39"}, "40": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[35.446,35.457]", "Atomic Number": "17", "Atomic Symbol": "Cl", "Relative Atomic Mass": "39.970415(34)", "Mass Number": "40"}, "41": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[35.446,35.457]", "Atomic Number": "17", "Atomic Symbol": "Cl", "Relative Atomic Mass": "40.970685(74)", "Mass Number": "41"}, "42": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[35.446,35.457]", "Atomic Number": "17", "Atomic Symbol": "Cl", "Relative Atomic Mass": "41.97325(15)", "Mass Number": "42"}, "43": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[35.446,35.457]", "Atomic Number": "17", "Atomic Symbol": "Cl", "Relative Atomic Mass": "42.97389(10)", "Mass Number": "43"}, "44": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[35.446,35.457]", "Atomic Number": "17", "Atomic Symbol": "Cl", "Relative Atomic Mass": "43.97787(20)", "Mass Number": "44"}, "45": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[35.446,35.457]", "Atomic Number": "17", "Atomic Symbol": "Cl", "Relative Atomic Mass": "44.98029(11)", "Mass Number": "45"}, "46": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[35.446,35.457]", "Atomic Number": "17", "Atomic Symbol": "Cl", "Relative Atomic Mass": "45.98517(17)", "Mass Number": "46"}, "47": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[35.446,35.457]", "Atomic Number": "17", "Atomic Symbol": "Cl", "Relative Atomic Mass": "46.98916(43#)", "Mass Number": "47"}, "48": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[35.446,35.457]", "Atomic Number": "17", "Atomic Symbol": "Cl", "Relative Atomic Mass": "47.99564(54#)", "Mass Number": "48"}, "49": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[35.446,35.457]", "Atomic Number": "17", "Atomic Symbol": "Cl", "Relative Atomic Mass": "49.00123(64#)", "Mass Number": "49"}, "50": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[35.446,35.457]", "Atomic Number": "17", "Atomic Symbol": "Cl", "Relative Atomic Mass": "50.00905(64#)", "Mass Number": "50"}, "51": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[35.446,35.457]", "Atomic Number": "17", "Atomic Symbol": "Cl", "Relative Atomic Mass": "51.01554(75#)", "Mass Number": "51"}}, "46": {"128": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "127.95183(64#)", "Mass Number": "128"}, "91": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "90.95032(54#)", "Mass Number": "91"}, "92": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "91.94088(54#)", "Mass Number": "92"}, "93": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "92.93651(43#)", "Mass Number": "93"}, "94": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "93.9290376(46)", "Mass Number": "94"}, "95": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "94.9248898(33)", "Mass Number": "95"}, "96": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "95.9182151(45)", "Mass Number": "96"}, "97": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "96.9164720(52)", "Mass Number": "97"}, "98": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "97.9126983(51)", "Mass Number": "98"}, "99": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "98.9117748(54)", "Mass Number": "99"}, "100": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "99.908505(19)", "Mass Number": "100"}, "101": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "100.9082864(49)", "Mass Number": "101"}, "102": {"Isotopic Composition": "0.0102(1)", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "101.9056022(28)", "Mass Number": "102"}, "103": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "102.9060809(27)", "Mass Number": "103"}, "104": {"Isotopic Composition": "0.1114(8)", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "103.9040305(14)", "Mass Number": "104"}, "105": {"Isotopic Composition": "0.2233(8)", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "104.9050796(12)", "Mass Number": "105"}, "106": {"Isotopic Composition": "0.2733(3)", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "105.9034804(12)", "Mass Number": "106"}, "107": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "106.9051282(13)", "Mass Number": "107"}, "108": {"Isotopic Composition": "0.2646(9)", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "107.9038916(12)", "Mass Number": "108"}, "109": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "108.9059504(12)", "Mass Number": "109"}, "110": {"Isotopic Composition": "0.1172(9)", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "109.90517220(75)", "Mass Number": "110"}, "111": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "110.90768968(86)", "Mass Number": "111"}, "112": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "111.9073297(70)", "Mass Number": "112"}, "113": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "112.9102610(75)", "Mass Number": "113"}, "114": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "113.9103686(75)", "Mass Number": "114"}, "115": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "114.913659(15)", "Mass Number": "115"}, "116": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "115.9142970(77)", "Mass Number": "116"}, "117": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "116.9179547(78)", "Mass Number": "117"}, "118": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "117.9190667(27)", "Mass Number": "118"}, "119": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "118.9233402(89)", "Mass Number": "119"}, "120": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "119.9245511(25)", "Mass Number": "120"}, "121": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "120.9289503(36)", "Mass Number": "121"}, "122": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "121.930632(21)", "Mass Number": "122"}, "123": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "122.93514(21#)", "Mass Number": "123"}, "124": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "123.93714(32#)", "Mass Number": "124"}, "125": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "124.94179(43#)", "Mass Number": "125"}, "126": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "125.94416(54#)", "Mass Number": "126"}, "127": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "106.42(1)", "Atomic Number": "46", "Atomic Symbol": "Pd", "Relative Atomic Mass": "126.94907(54#)", "Mass Number": "127"}}, "115": {"288": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "115", "Atomic Symbol": "Mc", "Relative Atomic Mass": "288.19274(62#)", "Mass Number": "288"}, "289": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "115", "Atomic Symbol": "Mc", "Relative Atomic Mass": "289.19363(89#)", "Mass Number": "289"}, "290": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "115", "Atomic Symbol": "Mc", "Relative Atomic Mass": "290.19598(73#)", "Mass Number": "290"}, "291": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "115", "Atomic Symbol": "Uup", "Relative Atomic Mass": "291.19707(88#)", "Mass Number": "291"}, "287": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "115", "Atomic Symbol": "Mc", "Relative Atomic Mass": "287.19070(52#)", "Mass Number": "287"}}, "18": {"30": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "39.948(1)", "Atomic Number": "18", "Atomic Symbol": "Ar", "Relative Atomic Mass": "30.02307(54#)", "Mass Number": "30"}, "31": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "39.948(1)", "Atomic Number": "18", "Atomic Symbol": "Ar", "Relative Atomic Mass": "31.01212(22#)", "Mass Number": "31"}, "32": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "39.948(1)", "Atomic Number": "18", "Atomic Symbol": "Ar", "Relative Atomic Mass": "31.9976378(19)", "Mass Number": "32"}, "33": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "39.948(1)", "Atomic Number": "18", "Atomic Symbol": "Ar", "Relative Atomic Mass": "32.98992555(43)", "Mass Number": "33"}, "34": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "39.948(1)", "Atomic Number": "18", "Atomic Symbol": "Ar", "Relative Atomic Mass": "33.980270090(83)", "Mass Number": "34"}, "35": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "39.948(1)", "Atomic Number": "18", "Atomic Symbol": "Ar", "Relative Atomic Mass": "34.97525759(80)", "Mass Number": "35"}, "36": {"Isotopic Composition": "0.003336(21)", "Notes": "g,r", "Standard Atomic Weight": "39.948(1)", "Atomic Number": "18", "Atomic Symbol": "Ar", "Relative Atomic Mass": "35.967545105(28)", "Mass Number": "36"}, "37": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "39.948(1)", "Atomic Number": "18", "Atomic Symbol": "Ar", "Relative Atomic Mass": "36.96677633(22)", "Mass Number": "37"}, "38": {"Isotopic Composition": "0.000629(7)", "Notes": "g,r", "Standard Atomic Weight": "39.948(1)", "Atomic Number": "18", "Atomic Symbol": "Ar", "Relative Atomic Mass": "37.96273211(21)", "Mass Number": "38"}, "39": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "39.948(1)", "Atomic Number": "18", "Atomic Symbol": "Ar", "Relative Atomic Mass": "38.9643130(54)", "Mass Number": "39"}, "40": {"Isotopic Composition": "0.996035(25)", "Notes": "g,r", "Standard Atomic Weight": "39.948(1)", "Atomic Number": "18", "Atomic Symbol": "Ar", "Relative Atomic Mass": "39.9623831237(24)", "Mass Number": "40"}, "41": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "39.948(1)", "Atomic Number": "18", "Atomic Symbol": "Ar", "Relative Atomic Mass": "40.96450057(37)", "Mass Number": "41"}, "42": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "39.948(1)", "Atomic Number": "18", "Atomic Symbol": "Ar", "Relative Atomic Mass": "41.9630457(62)", "Mass Number": "42"}, "43": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "39.948(1)", "Atomic Number": "18", "Atomic Symbol": "Ar", "Relative Atomic Mass": "42.9656361(57)", "Mass Number": "43"}, "44": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "39.948(1)", "Atomic Number": "18", "Atomic Symbol": "Ar", "Relative Atomic Mass": "43.9649238(17)", "Mass Number": "44"}, "45": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "39.948(1)", "Atomic Number": "18", "Atomic Symbol": "Ar", "Relative Atomic Mass": "44.96803973(55)", "Mass Number": "45"}, "46": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "39.948(1)", "Atomic Number": "18", "Atomic Symbol": "Ar", "Relative Atomic Mass": "45.968083(44)", "Mass Number": "46"}, "47": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "39.948(1)", "Atomic Number": "18", "Atomic Symbol": "Ar", "Relative Atomic Mass": "46.972935(96)", "Mass Number": "47"}, "48": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "39.948(1)", "Atomic Number": "18", "Atomic Symbol": "Ar", "Relative Atomic Mass": "47.97591(32#)", "Mass Number": "48"}, "49": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "39.948(1)", "Atomic Number": "18", "Atomic Symbol": "Ar", "Relative Atomic Mass": "48.98190(43#)", "Mass Number": "49"}, "50": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "39.948(1)", "Atomic Number": "18", "Atomic Symbol": "Ar", "Relative Atomic Mass": "49.98613(54#)", "Mass Number": "50"}, "51": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "39.948(1)", "Atomic Number": "18", "Atomic Symbol": "Ar", "Relative Atomic Mass": "50.99370(64#)", "Mass Number": "51"}, "52": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "39.948(1)", "Atomic Number": "18", "Atomic Symbol": "Ar", "Relative Atomic Mass": "51.99896(64#)", "Mass Number": "52"}, "53": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "39.948(1)", "Atomic Number": "18", "Atomic Symbol": "Ar", "Relative Atomic Mass": "53.00729(75#)", "Mass Number": "53"}}, "27": {"64": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "63.935811(21)", "Mass Number": "64"}, "65": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "64.9364621(22)", "Mass Number": "65"}, "66": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "65.939443(15)", "Mass Number": "66"}, "67": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "66.9406096(69)", "Mass Number": "67"}, "68": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "67.94426(16)", "Mass Number": "68"}, "69": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "68.94614(20)", "Mass Number": "69"}, "70": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "69.94963(32)", "Mass Number": "70"}, "71": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "70.95237(50)", "Mass Number": "71"}, "72": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "71.95729(43#)", "Mass Number": "72"}, "73": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "72.96039(54#)", "Mass Number": "73"}, "74": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "73.96515(64#)", "Mass Number": "74"}, "75": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "74.96876(75#)", "Mass Number": "75"}, "76": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "75.97413(86#)", "Mass Number": "76"}, "47": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "47.01057(86#)", "Mass Number": "47"}, "48": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "48.00093(86#)", "Mass Number": "48"}, "49": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "48.98891(75#)", "Mass Number": "49"}, "50": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "49.98091(64#)", "Mass Number": "50"}, "51": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "50.970647(52)", "Mass Number": "51"}, "52": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "51.96351(21#)", "Mass Number": "52"}, "53": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "52.9542041(19)", "Mass Number": "53"}, "54": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "53.94845987(54)", "Mass Number": "54"}, "55": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "54.94199720(57)", "Mass Number": "55"}, "56": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "55.93983880(63)", "Mass Number": "56"}, "57": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "56.93629057(66)", "Mass Number": "57"}, "58": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "57.9357521(13)", "Mass Number": "58"}, "59": {"Isotopic Composition": "1", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "58.93319429(56)", "Mass Number": "59"}, "60": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "59.93381630(56)", "Mass Number": "60"}, "61": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "60.93247662(95)", "Mass Number": "61"}, "62": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "61.934059(20)", "Mass Number": "62"}, "63": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "58.933194(4)", "Atomic Number": "27", "Atomic Symbol": "Co", "Relative Atomic Mass": "62.933600(20)", "Mass Number": "63"}}, "33": {"64": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "63.95743(33#)", "Mass Number": "64"}, "65": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "64.949611(91)", "Mass Number": "65"}, "66": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "65.9441488(61)", "Mass Number": "66"}, "67": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "66.93925111(48)", "Mass Number": "67"}, "68": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "67.9367741(20)", "Mass Number": "68"}, "69": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "68.932246(34)", "Mass Number": "69"}, "70": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "69.930926(54)", "Mass Number": "70"}, "71": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "70.9271138(45)", "Mass Number": "71"}, "72": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "71.9267523(44)", "Mass Number": "72"}, "73": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "72.9238291(41)", "Mass Number": "73"}, "74": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "73.9239286(18)", "Mass Number": "74"}, "75": {"Isotopic Composition": "1", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "74.92159457(95)", "Mass Number": "75"}, "76": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "75.92239202(95)", "Mass Number": "76"}, "77": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "76.9206476(18)", "Mass Number": "77"}, "78": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "77.921828(11)", "Mass Number": "78"}, "79": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "78.9209484(58)", "Mass Number": "79"}, "80": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "79.9224746(36)", "Mass Number": "80"}, "81": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "80.9221323(29)", "Mass Number": "81"}, "82": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "81.9247412(46)", "Mass Number": "82"}, "83": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "82.9252069(30)", "Mass Number": "83"}, "84": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "83.9293033(34)", "Mass Number": "84"}, "85": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "84.9321637(33)", "Mass Number": "85"}, "86": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "85.9367015(37)", "Mass Number": "86"}, "87": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "86.9402917(32)", "Mass Number": "87"}, "88": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "87.94555(21#)", "Mass Number": "88"}, "89": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "88.94976(32#)", "Mass Number": "89"}, "90": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "89.95563(64#)", "Mass Number": "90"}, "91": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "90.96039(64#)", "Mass Number": "91"}, "92": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "91.96674(75#)", "Mass Number": "92"}, "60": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "59.99388(43#)", "Mass Number": "60"}, "61": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "60.98112(32#)", "Mass Number": "61"}, "62": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "61.97361(32#)", "Mass Number": "62"}, "63": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "74.921595(6)", "Atomic Number": "33", "Atomic Symbol": "As", "Relative Atomic Mass": "62.96390(21#)", "Mass Number": "63"}}, "109": {"265": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "109", "Atomic Symbol": "Mt", "Relative Atomic Mass": "265.13600(48#)", "Mass Number": "265"}, "266": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "109", "Atomic Symbol": "Mt", "Relative Atomic Mass": "266.13737(33#)", "Mass Number": "266"}, "267": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "109", "Atomic Symbol": "Mt", "Relative Atomic Mass": "267.13719(54#)", "Mass Number": "267"}, "268": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "109", "Atomic Symbol": "Mt", "Relative Atomic Mass": "268.13865(25#)", "Mass Number": "268"}, "269": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "109", "Atomic Symbol": "Mt", "Relative Atomic Mass": "269.13882(50#)", "Mass Number": "269"}, "270": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "109", "Atomic Symbol": "Mt", "Relative Atomic Mass": "270.14033(18#)", "Mass Number": "270"}, "271": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "109", "Atomic Symbol": "Mt", "Relative Atomic Mass": "271.14074(35#)", "Mass Number": "271"}, "272": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "109", "Atomic Symbol": "Mt", "Relative Atomic Mass": "272.14341(52#)", "Mass Number": "272"}, "273": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "109", "Atomic Symbol": "Mt", "Relative Atomic Mass": "273.14440(52#)", "Mass Number": "273"}, "274": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "109", "Atomic Symbol": "Mt", "Relative Atomic Mass": "274.14724(38#)", "Mass Number": "274"}, "275": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "109", "Atomic Symbol": "Mt", "Relative Atomic Mass": "275.14882(50#)", "Mass Number": "275"}, "276": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "109", "Atomic Symbol": "Mt", "Relative Atomic Mass": "276.15159(59#)", "Mass Number": "276"}, "277": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "109", "Atomic Symbol": "Mt", "Relative Atomic Mass": "277.15327(82#)", "Mass Number": "277"}, "278": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "109", "Atomic Symbol": "Mt", "Relative Atomic Mass": "278.15631(68#)", "Mass Number": "278"}, "279": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "109", "Atomic Symbol": "Mt", "Relative Atomic Mass": "279.15808(72#)", "Mass Number": "279"}}, "92": {"217": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "238.02891(3)", "Atomic Number": "92", "Atomic Symbol": "U", "Relative Atomic Mass": "217.02466(11#)", "Mass Number": "217"}, "218": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "238.02891(3)", "Atomic Number": "92", "Atomic Symbol": "U", "Relative Atomic Mass": "218.023523(20)", "Mass Number": "218"}, "219": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "238.02891(3)", "Atomic Number": "92", "Atomic Symbol": "U", "Relative Atomic Mass": "219.024999(55)", "Mass Number": "219"}, "220": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "238.02891(3)", "Atomic Number": "92", "Atomic Symbol": "U", "Relative Atomic Mass": "220.02462(11#)", "Mass Number": "220"}, "221": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "238.02891(3)", "Atomic Number": "92", "Atomic Symbol": "U", "Relative Atomic Mass": "221.02628(11#)", "Mass Number": "221"}, "222": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "238.02891(3)", "Atomic Number": "92", "Atomic Symbol": "U", "Relative Atomic Mass": "222.02600(11#)", "Mass Number": "222"}, "223": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "238.02891(3)", "Atomic Number": "92", "Atomic Symbol": "U", "Relative Atomic Mass": "223.027739(76)", "Mass Number": "223"}, "224": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "238.02891(3)", "Atomic Number": "92", "Atomic Symbol": "U", "Relative Atomic Mass": "224.027605(27)", "Mass Number": "224"}, "225": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "238.02891(3)", "Atomic Number": "92", "Atomic Symbol": "U", "Relative Atomic Mass": "225.029391(13)", "Mass Number": "225"}, "226": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "238.02891(3)", "Atomic Number": "92", "Atomic Symbol": "U", "Relative Atomic Mass": "226.029339(14)", "Mass Number": "226"}, "227": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "238.02891(3)", "Atomic Number": "92", "Atomic Symbol": "U", "Relative Atomic Mass": "227.031157(18)", "Mass Number": "227"}, "228": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "238.02891(3)", "Atomic Number": "92", "Atomic Symbol": "U", "Relative Atomic Mass": "228.031371(15)", "Mass Number": "228"}, "229": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "238.02891(3)", "Atomic Number": "92", "Atomic Symbol": "U", "Relative Atomic Mass": "229.0335063(64)", "Mass Number": "229"}, "230": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "238.02891(3)", "Atomic Number": "92", "Atomic Symbol": "U", "Relative Atomic Mass": "230.0339401(51)", "Mass Number": "230"}, "231": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "238.02891(3)", "Atomic Number": "92", "Atomic Symbol": "U", "Relative Atomic Mass": "231.0362939(32)", "Mass Number": "231"}, "232": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "238.02891(3)", "Atomic Number": "92", "Atomic Symbol": "U", "Relative Atomic Mass": "232.0371563(23)", "Mass Number": "232"}, "233": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "238.02891(3)", "Atomic Number": "92", "Atomic Symbol": "U", "Relative Atomic Mass": "233.0396355(29)", "Mass Number": "233"}, "234": {"Isotopic Composition": "0.000054(5)", "Notes": "g,m", "Standard Atomic Weight": "238.02891(3)", "Atomic Number": "92", "Atomic Symbol": "U", "Relative Atomic Mass": "234.0409523(19)", "Mass Number": "234"}, "235": {"Isotopic Composition": "0.007204(6)", "Notes": "g,m", "Standard Atomic Weight": "238.02891(3)", "Atomic Number": "92", "Atomic Symbol": "U", "Relative Atomic Mass": "235.0439301(19)", "Mass Number": "235"}, "236": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "238.02891(3)", "Atomic Number": "92", "Atomic Symbol": "U", "Relative Atomic Mass": "236.0455682(19)", "Mass Number": "236"}, "237": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "238.02891(3)", "Atomic Number": "92", "Atomic Symbol": "U", "Relative Atomic Mass": "237.0487304(20)", "Mass Number": "237"}, "238": {"Isotopic Composition": "0.992742(10)", "Notes": "g,m", "Standard Atomic Weight": "238.02891(3)", "Atomic Number": "92", "Atomic Symbol": "U", "Relative Atomic Mass": "238.0507884(20)", "Mass Number": "238"}, "239": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "238.02891(3)", "Atomic Number": "92", "Atomic Symbol": "U", "Relative Atomic Mass": "239.0542935(20)", "Mass Number": "239"}, "240": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "238.02891(3)", "Atomic Number": "92", "Atomic Symbol": "U", "Relative Atomic Mass": "240.0565934(57)", "Mass Number": "240"}, "241": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "238.02891(3)", "Atomic Number": "92", "Atomic Symbol": "U", "Relative Atomic Mass": "241.06033(32#)", "Mass Number": "241"}, "242": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "238.02891(3)", "Atomic Number": "92", "Atomic Symbol": "U", "Relative Atomic Mass": "242.06293(22#)", "Mass Number": "242"}, "243": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "238.02891(3)", "Atomic Number": "92", "Atomic Symbol": "U", "Relative Atomic Mass": "243.06699(43#)", "Mass Number": "243"}}, "2": {"3": {"Isotopic Composition": "0.00000134(3)", "Notes": "g,r", "Standard Atomic Weight": "4.002602(2)", "Atomic Number": "2", "Atomic Symbol": "He", "Relative Atomic Mass": "3.0160293201(25)", "Mass Number": "3"}, "4": {"Isotopic Composition": "0.99999866(3)", "Notes": "g,r", "Standard Atomic Weight": "4.002602(2)", "Atomic Number": "2", "Atomic Symbol": "He", "Relative Atomic Mass": "4.00260325413(6)", "Mass Number": "4"}, "5": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "4.002602(2)", "Atomic Number": "2", "Atomic Symbol": "He", "Relative Atomic Mass": "5.012057(21)", "Mass Number": "5"}, "6": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "4.002602(2)", "Atomic Number": "2", "Atomic Symbol": "He", "Relative Atomic Mass": "6.018885891(57)", "Mass Number": "6"}, "7": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "4.002602(2)", "Atomic Number": "2", "Atomic Symbol": "He", "Relative Atomic Mass": "7.0279907(81)", "Mass Number": "7"}, "8": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "4.002602(2)", "Atomic Number": "2", "Atomic Symbol": "He", "Relative Atomic Mass": "8.033934390(95)", "Mass Number": "8"}, "9": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "4.002602(2)", "Atomic Number": "2", "Atomic Symbol": "He", "Relative Atomic Mass": "9.043946(50)", "Mass Number": "9"}, "10": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "4.002602(2)", "Atomic Number": "2", "Atomic Symbol": "He", "Relative Atomic Mass": "10.05279(11)", "Mass Number": "10"}}, "47": {"128": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "127.94106(32#)", "Mass Number": "128"}, "129": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "128.94395(32#)", "Mass Number": "129"}, "130": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "129.95070(36#)", "Mass Number": "130"}, "93": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "92.95033(54#)", "Mass Number": "93"}, "94": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "93.94373(69#)", "Mass Number": "94"}, "95": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "94.93602(43#)", "Mass Number": "95"}, "96": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "95.930744(97)", "Mass Number": "96"}, "97": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "96.92397(12)", "Mass Number": "97"}, "98": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "97.921560(35)", "Mass Number": "98"}, "99": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "98.9176458(67)", "Mass Number": "99"}, "100": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "99.9161154(54)", "Mass Number": "100"}, "101": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "100.9126840(52)", "Mass Number": "101"}, "102": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "101.9117047(88)", "Mass Number": "102"}, "103": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "102.9089631(41)", "Mass Number": "103"}, "104": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "103.9086239(45)", "Mass Number": "104"}, "105": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "104.9065256(49)", "Mass Number": "105"}, "106": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "105.9066636(32)", "Mass Number": "106"}, "107": {"Isotopic Composition": "0.51839(8)", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "106.9050916(26)", "Mass Number": "107"}, "108": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "107.9059503(26)", "Mass Number": "108"}, "109": {"Isotopic Composition": "0.48161(8)", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "108.9047553(14)", "Mass Number": "109"}, "110": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "109.9061102(14)", "Mass Number": "110"}, "111": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "110.9052959(16)", "Mass Number": "111"}, "112": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "111.9070486(26)", "Mass Number": "112"}, "113": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "112.906573(18)", "Mass Number": "113"}, "114": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "113.9088230(49)", "Mass Number": "114"}, "115": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "114.908767(20)", "Mass Number": "115"}, "116": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "115.9113868(35)", "Mass Number": "116"}, "117": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "116.911774(15)", "Mass Number": "117"}, "118": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "117.9145955(27)", "Mass Number": "118"}, "119": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "118.915570(16)", "Mass Number": "119"}, "120": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "119.9187848(48)", "Mass Number": "120"}, "121": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "120.920125(13)", "Mass Number": "121"}, "122": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "121.923664(41)", "Mass Number": "122"}, "123": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "122.925337(33)", "Mass Number": "123"}, "124": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "123.92893(27)", "Mass Number": "124"}, "125": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "124.93105(64)", "Mass Number": "125"}, "126": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "125.93475(21#)", "Mass Number": "126"}, "127": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "107.8682(2)", "Atomic Number": "47", "Atomic Symbol": "Ag", "Relative Atomic Mass": "126.93711(21#)", "Mass Number": "127"}}, "35": {"67": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "66.96465(54#)", "Mass Number": "67"}, "68": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "67.95873(33#)", "Mass Number": "68"}, "69": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "68.950497(40)", "Mass Number": "69"}, "70": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "69.944792(16)", "Mass Number": "70"}, "71": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "70.9393422(58)", "Mass Number": "71"}, "72": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "71.9365886(72)", "Mass Number": "72"}, "73": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "72.9316715(78)", "Mass Number": "73"}, "74": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "73.9299102(63)", "Mass Number": "74"}, "75": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "74.9258105(46)", "Mass Number": "75"}, "76": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "75.924542(10)", "Mass Number": "76"}, "77": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "76.9213792(30)", "Mass Number": "77"}, "78": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "77.9211459(38)", "Mass Number": "78"}, "79": {"Isotopic Composition": "0.5069(7)", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "78.9183376(14)", "Mass Number": "79"}, "80": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "79.9185298(14)", "Mass Number": "80"}, "81": {"Isotopic Composition": "0.4931(7)", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "80.9162897(14)", "Mass Number": "81"}, "82": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "81.9168032(14)", "Mass Number": "82"}, "83": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "82.9151756(41)", "Mass Number": "83"}, "84": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "83.916496(28)", "Mass Number": "84"}, "85": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "84.9156458(33)", "Mass Number": "85"}, "86": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "85.9188054(33)", "Mass Number": "86"}, "87": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "86.9206740(34)", "Mass Number": "87"}, "88": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "87.9240833(34)", "Mass Number": "88"}, "89": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "88.9267046(35)", "Mass Number": "89"}, "90": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "89.9312928(36)", "Mass Number": "90"}, "91": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "90.9343986(38)", "Mass Number": "91"}, "92": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "91.9396316(72)", "Mass Number": "92"}, "93": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "92.94313(48)", "Mass Number": "93"}, "94": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "93.94890(43#)", "Mass Number": "94"}, "95": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "94.95301(21#)", "Mass Number": "95"}, "96": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "95.95903(32#)", "Mass Number": "96"}, "97": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "96.96344(43#)", "Mass Number": "97"}, "98": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[79.901,79.907]", "Atomic Number": "35", "Atomic Symbol": "Br", "Relative Atomic Mass": "97.96946(43#)", "Mass Number": "98"}}, "107": {"260": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "107", "Atomic Symbol": "Bh", "Relative Atomic Mass": "260.12166(26#)", "Mass Number": "260"}, "261": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "107", "Atomic Symbol": "Bh", "Relative Atomic Mass": "261.12145(22#)", "Mass Number": "261"}, "262": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "107", "Atomic Symbol": "Bh", "Relative Atomic Mass": "262.12297(33#)", "Mass Number": "262"}, "263": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "107", "Atomic Symbol": "Bh", "Relative Atomic Mass": "263.12292(33#)", "Mass Number": "263"}, "264": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "107", "Atomic Symbol": "Bh", "Relative Atomic Mass": "264.12459(19#)", "Mass Number": "264"}, "265": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "107", "Atomic Symbol": "Bh", "Relative Atomic Mass": "265.12491(25#)", "Mass Number": "265"}, "266": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "107", "Atomic Symbol": "Bh", "Relative Atomic Mass": "266.12679(18#)", "Mass Number": "266"}, "267": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "107", "Atomic Symbol": "Bh", "Relative Atomic Mass": "267.12750(28#)", "Mass Number": "267"}, "268": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "107", "Atomic Symbol": "Bh", "Relative Atomic Mass": "268.12969(41#)", "Mass Number": "268"}, "269": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "107", "Atomic Symbol": "Bh", "Relative Atomic Mass": "269.13042(40#)", "Mass Number": "269"}, "270": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "107", "Atomic Symbol": "Bh", "Relative Atomic Mass": "270.13336(31#)", "Mass Number": "270"}, "271": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "107", "Atomic Symbol": "Bh", "Relative Atomic Mass": "271.13526(48#)", "Mass Number": "271"}, "272": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "107", "Atomic Symbol": "Bh", "Relative Atomic Mass": "272.13826(58#)", "Mass Number": "272"}, "273": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "107", "Atomic Symbol": "Bh", "Relative Atomic Mass": "273.14024(80#)", "Mass Number": "273"}, "274": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "107", "Atomic Symbol": "Bh", "Relative Atomic Mass": "274.14355(65#)", "Mass Number": "274"}, "275": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "107", "Atomic Symbol": "Bh", "Relative Atomic Mass": "275.14567(64#)", "Mass Number": "275"}}, "57": {"128": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "127.915592(58)", "Mass Number": "128"}, "129": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "128.912694(23)", "Mass Number": "129"}, "130": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "129.912369(28)", "Mass Number": "130"}, "131": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "130.910070(30)", "Mass Number": "131"}, "132": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "131.910119(39)", "Mass Number": "132"}, "133": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "132.908218(30)", "Mass Number": "133"}, "134": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "133.908514(21)", "Mass Number": "134"}, "135": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "134.906984(10)", "Mass Number": "135"}, "136": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "135.907635(57)", "Mass Number": "136"}, "137": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "136.9064504(18)", "Mass Number": "137"}, "138": {"Isotopic Composition": "0.0008881(71)", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "137.9071149(37)", "Mass Number": "138"}, "139": {"Isotopic Composition": "0.9991119(71)", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "138.9063563(24)", "Mass Number": "139"}, "140": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "139.9094806(24)", "Mass Number": "140"}, "141": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "140.9109660(48)", "Mass Number": "141"}, "142": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "141.9140909(69)", "Mass Number": "142"}, "143": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "142.9160795(79)", "Mass Number": "143"}, "144": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "143.919646(14)", "Mass Number": "144"}, "145": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "144.921808(13)", "Mass Number": "145"}, "146": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "145.925875(36)", "Mass Number": "146"}, "147": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "146.928418(12)", "Mass Number": "147"}, "148": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "147.932679(21)", "Mass Number": "148"}, "149": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "148.93535(21)", "Mass Number": "149"}, "150": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "149.93947(21#)", "Mass Number": "150"}, "151": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "150.94232(21#)", "Mass Number": "151"}, "152": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "151.94682(32#)", "Mass Number": "152"}, "153": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "152.95036(32#)", "Mass Number": "153"}, "154": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "153.95517(43#)", "Mass Number": "154"}, "155": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "154.95901(43#)", "Mass Number": "155"}, "116": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "115.95630(23#)", "Mass Number": "116"}, "117": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "116.94999(32#)", "Mass Number": "117"}, "118": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "117.94673(32#)", "Mass Number": "118"}, "119": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "118.94099(32#)", "Mass Number": "119"}, "120": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "119.93807(32#)", "Mass Number": "120"}, "121": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "120.93315(32#)", "Mass Number": "121"}, "122": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "121.93071(32#)", "Mass Number": "122"}, "123": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "122.92630(21#)", "Mass Number": "123"}, "124": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "123.924574(61)", "Mass Number": "124"}, "125": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "124.920816(28)", "Mass Number": "125"}, "126": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "125.919513(97)", "Mass Number": "126"}, "127": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "138.90547(7)", "Atomic Number": "57", "Atomic Symbol": "La", "Relative Atomic Mass": "126.916375(28)", "Mass Number": "127"}}, "78": {"192": {"Isotopic Composition": "0.00782(24)", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "191.9610387(32)", "Mass Number": "192"}, "193": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "192.9629824(21)", "Mass Number": "193"}, "194": {"Isotopic Composition": "0.3286(40)", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "193.9626809(10)", "Mass Number": "194"}, "195": {"Isotopic Composition": "0.3378(24)", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "194.9647917(10)", "Mass Number": "195"}, "196": {"Isotopic Composition": "0.2521(34)", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "195.96495209(99)", "Mass Number": "196"}, "197": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "196.96734069(94)", "Mass Number": "197"}, "198": {"Isotopic Composition": "0.07356(130)", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "197.9678949(23)", "Mass Number": "198"}, "199": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "198.9705952(24)", "Mass Number": "199"}, "200": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "199.971443(22)", "Mass Number": "200"}, "201": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "200.974513(54)", "Mass Number": "201"}, "202": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "201.975639(27)", "Mass Number": "202"}, "203": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "202.97893(21#)", "Mass Number": "203"}, "204": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "203.98076(21#)", "Mass Number": "204"}, "205": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "204.98608(32#)", "Mass Number": "205"}, "206": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "205.98966(32#)", "Mass Number": "206"}, "166": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "165.99486(54#)", "Mass Number": "166"}, "167": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "166.99269(33#)", "Mass Number": "167"}, "168": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "167.98813(17)", "Mass Number": "168"}, "169": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "168.98657(22#)", "Mass Number": "169"}, "170": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "169.982496(20)", "Mass Number": "170"}, "171": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "170.981245(78)", "Mass Number": "171"}, "172": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "171.977351(12)", "Mass Number": "172"}, "173": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "172.976443(60)", "Mass Number": "173"}, "174": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "173.972820(11)", "Mass Number": "174"}, "175": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "174.972410(19)", "Mass Number": "175"}, "176": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "175.968938(14)", "Mass Number": "176"}, "177": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "176.968470(16)", "Mass Number": "177"}, "178": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "177.965650(11)", "Mass Number": "178"}, "179": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "178.9653590(86)", "Mass Number": "179"}, "180": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "179.963032(12)", "Mass Number": "180"}, "181": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "180.963098(16)", "Mass Number": "181"}, "182": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "181.961172(14)", "Mass Number": "182"}, "183": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "182.961597(17)", "Mass Number": "183"}, "184": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "183.959915(17)", "Mass Number": "184"}, "185": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "184.960614(28)", "Mass Number": "185"}, "186": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "185.959351(23)", "Mass Number": "186"}, "187": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "186.960617(26)", "Mass Number": "187"}, "188": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "187.9593889(61)", "Mass Number": "188"}, "189": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "188.960831(12)", "Mass Number": "189"}, "190": {"Isotopic Composition": "0.00012(2)", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "189.9599297(63)", "Mass Number": "190"}, "191": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "195.084(9)", "Atomic Number": "78", "Atomic Symbol": "Pt", "Relative Atomic Mass": "190.9616729(53)", "Mass Number": "191"}}, "7": {"10": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[14.00643,14.00728]", "Atomic Number": "7", "Atomic Symbol": "N", "Relative Atomic Mass": "10.04165(43)", "Mass Number": "10"}, "11": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[14.00643,14.00728]", "Atomic Number": "7", "Atomic Symbol": "N", "Relative Atomic Mass": "11.026091(50)", "Mass Number": "11"}, "12": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[14.00643,14.00728]", "Atomic Number": "7", "Atomic Symbol": "N", "Relative Atomic Mass": "12.0186132(11)", "Mass Number": "12"}, "13": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[14.00643,14.00728]", "Atomic Number": "7", "Atomic Symbol": "N", "Relative Atomic Mass": "13.00573861(29)", "Mass Number": "13"}, "14": {"Isotopic Composition": "0.99636(20)", "Notes": " ", "Standard Atomic Weight": "[14.00643,14.00728]", "Atomic Number": "7", "Atomic Symbol": "N", "Relative Atomic Mass": "14.00307400443(20)", "Mass Number": "14"}, "15": {"Isotopic Composition": "0.00364(20)", "Notes": " ", "Standard Atomic Weight": "[14.00643,14.00728]", "Atomic Number": "7", "Atomic Symbol": "N", "Relative Atomic Mass": "15.00010889888(64)", "Mass Number": "15"}, "16": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[14.00643,14.00728]", "Atomic Number": "7", "Atomic Symbol": "N", "Relative Atomic Mass": "16.0061019(25)", "Mass Number": "16"}, "17": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[14.00643,14.00728]", "Atomic Number": "7", "Atomic Symbol": "N", "Relative Atomic Mass": "17.008449(16)", "Mass Number": "17"}, "18": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[14.00643,14.00728]", "Atomic Number": "7", "Atomic Symbol": "N", "Relative Atomic Mass": "18.014078(20)", "Mass Number": "18"}, "19": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[14.00643,14.00728]", "Atomic Number": "7", "Atomic Symbol": "N", "Relative Atomic Mass": "19.017022(18)", "Mass Number": "19"}, "20": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[14.00643,14.00728]", "Atomic Number": "7", "Atomic Symbol": "N", "Relative Atomic Mass": "20.023366(60)", "Mass Number": "20"}, "21": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[14.00643,14.00728]", "Atomic Number": "7", "Atomic Symbol": "N", "Relative Atomic Mass": "21.02711(10)", "Mass Number": "21"}, "22": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[14.00643,14.00728]", "Atomic Number": "7", "Atomic Symbol": "N", "Relative Atomic Mass": "22.03439(21)", "Mass Number": "22"}, "23": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[14.00643,14.00728]", "Atomic Number": "7", "Atomic Symbol": "N", "Relative Atomic Mass": "23.04114(32#)", "Mass Number": "23"}, "24": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[14.00643,14.00728]", "Atomic Number": "7", "Atomic Symbol": "N", "Relative Atomic Mass": "24.05039(43#)", "Mass Number": "24"}, "25": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[14.00643,14.00728]", "Atomic Number": "7", "Atomic Symbol": "N", "Relative Atomic Mass": "25.06010(54#)", "Mass Number": "25"}}, "59": {"128": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "127.928791(32)", "Mass Number": "128"}, "129": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "128.925095(32)", "Mass Number": "129"}, "130": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "129.923590(69)", "Mass Number": "130"}, "131": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "130.920235(50)", "Mass Number": "131"}, "132": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "131.919255(61)", "Mass Number": "132"}, "133": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "132.916331(13)", "Mass Number": "133"}, "134": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "133.915697(22)", "Mass Number": "134"}, "135": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "134.913112(13)", "Mass Number": "135"}, "136": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "135.912677(12)", "Mass Number": "136"}, "137": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "136.9106792(87)", "Mass Number": "137"}, "138": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "137.910754(15)", "Mass Number": "138"}, "139": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "138.9089408(85)", "Mass Number": "139"}, "140": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "139.9090803(69)", "Mass Number": "140"}, "141": {"Isotopic Composition": "1", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "140.9076576(23)", "Mass Number": "141"}, "142": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "141.9100496(23)", "Mass Number": "142"}, "143": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "142.9108228(24)", "Mass Number": "143"}, "144": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "143.9133109(32)", "Mass Number": "144"}, "145": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "144.9145182(78)", "Mass Number": "145"}, "146": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "145.917680(37)", "Mass Number": "146"}, "147": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "146.919008(17)", "Mass Number": "147"}, "148": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "147.922130(16)", "Mass Number": "148"}, "149": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "148.923736(11)", "Mass Number": "149"}, "150": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "149.9266765(97)", "Mass Number": "150"}, "151": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "150.928309(13)", "Mass Number": "151"}, "152": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "151.931553(20)", "Mass Number": "152"}, "153": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "152.933904(13)", "Mass Number": "153"}, "154": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "153.93753(16)", "Mass Number": "154"}, "155": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "154.940509(18)", "Mass Number": "155"}, "156": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "155.94464(32#)", "Mass Number": "156"}, "157": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "156.94789(43#)", "Mass Number": "157"}, "158": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "157.95241(43#)", "Mass Number": "158"}, "159": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "158.95589(54#)", "Mass Number": "159"}, "121": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "120.95532(54#)", "Mass Number": "121"}, "122": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "121.95175(54#)", "Mass Number": "122"}, "123": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "122.94596(43#)", "Mass Number": "123"}, "124": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "123.94294(43#)", "Mass Number": "124"}, "125": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "124.93770(32#)", "Mass Number": "125"}, "126": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "125.93524(21#)", "Mass Number": "126"}, "127": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "140.90766(2)", "Atomic Number": "59", "Atomic Symbol": "Pr", "Relative Atomic Mass": "126.93071(21#)", "Mass Number": "127"}}, "58": {"128": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "127.918911(30)", "Mass Number": "128"}, "129": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "128.918102(30)", "Mass Number": "129"}, "130": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "129.914736(30)", "Mass Number": "130"}, "131": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "130.914429(35)", "Mass Number": "131"}, "132": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "131.911464(22)", "Mass Number": "132"}, "133": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "132.911520(18)", "Mass Number": "133"}, "134": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "133.908928(22)", "Mass Number": "134"}, "135": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "134.909161(11)", "Mass Number": "135"}, "136": {"Isotopic Composition": "0.00185(2)", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "135.90712921(41)", "Mass Number": "136"}, "137": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "136.90776236(45)", "Mass Number": "137"}, "138": {"Isotopic Composition": "0.00251(2)", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "137.905991(11)", "Mass Number": "138"}, "139": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "138.9066551(78)", "Mass Number": "139"}, "140": {"Isotopic Composition": "0.88450(51)", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "139.9054431(23)", "Mass Number": "140"}, "141": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "140.9082807(23)", "Mass Number": "141"}, "142": {"Isotopic Composition": "0.11114(51)", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "141.9092504(29)", "Mass Number": "142"}, "143": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "142.9123921(29)", "Mass Number": "143"}, "144": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "143.9136529(34)", "Mass Number": "144"}, "145": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "144.917265(36)", "Mass Number": "145"}, "146": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "145.918802(18)", "Mass Number": "146"}, "147": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "146.9226899(92)", "Mass Number": "147"}, "148": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "147.924424(12)", "Mass Number": "148"}, "149": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "148.928427(11)", "Mass Number": "149"}, "150": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "149.930384(13)", "Mass Number": "150"}, "151": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "150.934272(19)", "Mass Number": "151"}, "152": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "151.93660(21#)", "Mass Number": "152"}, "153": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "152.94093(21#)", "Mass Number": "153"}, "154": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "153.94380(32#)", "Mass Number": "154"}, "155": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "154.94855(43#)", "Mass Number": "155"}, "156": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "155.95183(43#)", "Mass Number": "156"}, "157": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "156.95705(54#)", "Mass Number": "157"}, "119": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "118.95271(54#)", "Mass Number": "119"}, "120": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "119.94654(54#)", "Mass Number": "120"}, "121": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "120.94335(43#)", "Mass Number": "121"}, "122": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "121.93787(43#)", "Mass Number": "122"}, "123": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "122.93528(32#)", "Mass Number": "123"}, "124": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "123.93031(32#)", "Mass Number": "124"}, "125": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "124.92844(21#)", "Mass Number": "125"}, "126": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "125.923971(30)", "Mass Number": "126"}, "127": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "140.116(1)", "Atomic Number": "58", "Atomic Symbol": "Ce", "Relative Atomic Mass": "126.922727(31)", "Mass Number": "127"}}, "68": {"142": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "141.97010(54#)", "Mass Number": "142"}, "143": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "142.96662(43#)", "Mass Number": "143"}, "144": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "143.96070(21#)", "Mass Number": "144"}, "145": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "144.95805(21#)", "Mass Number": "145"}, "146": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "145.9524184(72)", "Mass Number": "146"}, "147": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "146.949964(41)", "Mass Number": "147"}, "148": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "147.944735(11)", "Mass Number": "148"}, "149": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "148.942306(30)", "Mass Number": "149"}, "150": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "149.937916(18)", "Mass Number": "150"}, "151": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "150.937449(18)", "Mass Number": "151"}, "152": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "151.935057(10)", "Mass Number": "152"}, "153": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "152.935080(10)", "Mass Number": "153"}, "154": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "153.9327908(55)", "Mass Number": "154"}, "155": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "154.9332159(67)", "Mass Number": "155"}, "156": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "155.931067(26)", "Mass Number": "156"}, "157": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "156.931949(27)", "Mass Number": "157"}, "158": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "157.929893(27)", "Mass Number": "158"}, "159": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "158.9306918(42)", "Mass Number": "159"}, "160": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "159.929077(26)", "Mass Number": "160"}, "161": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "160.9300046(96)", "Mass Number": "161"}, "162": {"Isotopic Composition": "0.00139(5)", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "161.9287884(20)", "Mass Number": "162"}, "163": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "162.9300408(53)", "Mass Number": "163"}, "164": {"Isotopic Composition": "0.01601(3)", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "163.9292088(20)", "Mass Number": "164"}, "165": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "164.9307345(21)", "Mass Number": "165"}, "166": {"Isotopic Composition": "0.33503(36)", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "165.9302995(22)", "Mass Number": "166"}, "167": {"Isotopic Composition": "0.22869(9)", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "166.9320546(22)", "Mass Number": "167"}, "168": {"Isotopic Composition": "0.26978(18)", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "167.9323767(22)", "Mass Number": "168"}, "169": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "168.9345968(22)", "Mass Number": "169"}, "170": {"Isotopic Composition": "0.14910(36)", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "169.9354702(26)", "Mass Number": "170"}, "171": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "170.9380357(26)", "Mass Number": "171"}, "172": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "171.9393619(47)", "Mass Number": "172"}, "173": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "172.94240(21#)", "Mass Number": "173"}, "174": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "173.94423(32#)", "Mass Number": "174"}, "175": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "174.94777(43#)", "Mass Number": "175"}, "176": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "175.94994(43#)", "Mass Number": "176"}, "177": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "167.259(3)", "Atomic Number": "68", "Atomic Symbol": "Er", "Relative Atomic Mass": "176.95399(54#)", "Mass Number": "177"}}, "111": {"272": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "111", "Atomic Symbol": "Rg", "Relative Atomic Mass": "272.15327(25#)", "Mass Number": "272"}, "273": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "111", "Atomic Symbol": "Rg", "Relative Atomic Mass": "273.15313(56#)", "Mass Number": "273"}, "274": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "111", "Atomic Symbol": "Rg", "Relative Atomic Mass": "274.15525(19#)", "Mass Number": "274"}, "275": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "111", "Atomic Symbol": "Rg", "Relative Atomic Mass": "275.15594(56#)", "Mass Number": "275"}, "276": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "111", "Atomic Symbol": "Rg", "Relative Atomic Mass": "276.15833(68#)", "Mass Number": "276"}, "277": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "111", "Atomic Symbol": "Rg", "Relative Atomic Mass": "277.15907(61#)", "Mass Number": "277"}, "278": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "111", "Atomic Symbol": "Rg", "Relative Atomic Mass": "278.16149(38#)", "Mass Number": "278"}, "279": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "111", "Atomic Symbol": "Rg", "Relative Atomic Mass": "279.16272(51#)", "Mass Number": "279"}, "280": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "111", "Atomic Symbol": "Rg", "Relative Atomic Mass": "280.16514(61#)", "Mass Number": "280"}, "281": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "111", "Atomic Symbol": "Rg", "Relative Atomic Mass": "281.16636(89#)", "Mass Number": "281"}, "282": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "111", "Atomic Symbol": "Rg", "Relative Atomic Mass": "282.16912(72#)", "Mass Number": "282"}, "283": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "111", "Atomic Symbol": "Rg", "Relative Atomic Mass": "283.17054(79#)", "Mass Number": "283"}}, "8": {"12": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[15.99903,15.99977]", "Atomic Number": "8", "Atomic Symbol": "O", "Relative Atomic Mass": "12.034262(26)", "Mass Number": "12"}, "13": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[15.99903,15.99977]", "Atomic Number": "8", "Atomic Symbol": "O", "Relative Atomic Mass": "13.024815(10)", "Mass Number": "13"}, "14": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[15.99903,15.99977]", "Atomic Number": "8", "Atomic Symbol": "O", "Relative Atomic Mass": "14.00859636(12)", "Mass Number": "14"}, "15": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[15.99903,15.99977]", "Atomic Number": "8", "Atomic Symbol": "O", "Relative Atomic Mass": "15.00306562(53)", "Mass Number": "15"}, "16": {"Isotopic Composition": "0.99757(16)", "Notes": " ", "Standard Atomic Weight": "[15.99903,15.99977]", "Atomic Number": "8", "Atomic Symbol": "O", "Relative Atomic Mass": "15.99491461957(17)", "Mass Number": "16"}, "17": {"Isotopic Composition": "0.00038(1)", "Notes": " ", "Standard Atomic Weight": "[15.99903,15.99977]", "Atomic Number": "8", "Atomic Symbol": "O", "Relative Atomic Mass": "16.99913175650(69)", "Mass Number": "17"}, "18": {"Isotopic Composition": "0.00205(14)", "Notes": " ", "Standard Atomic Weight": "[15.99903,15.99977]", "Atomic Number": "8", "Atomic Symbol": "O", "Relative Atomic Mass": "17.99915961286(76)", "Mass Number": "18"}, "19": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[15.99903,15.99977]", "Atomic Number": "8", "Atomic Symbol": "O", "Relative Atomic Mass": "19.0035780(28)", "Mass Number": "19"}, "20": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[15.99903,15.99977]", "Atomic Number": "8", "Atomic Symbol": "O", "Relative Atomic Mass": "20.00407535(95)", "Mass Number": "20"}, "21": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[15.99903,15.99977]", "Atomic Number": "8", "Atomic Symbol": "O", "Relative Atomic Mass": "21.008655(13)", "Mass Number": "21"}, "22": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[15.99903,15.99977]", "Atomic Number": "8", "Atomic Symbol": "O", "Relative Atomic Mass": "22.009966(61)", "Mass Number": "22"}, "23": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[15.99903,15.99977]", "Atomic Number": "8", "Atomic Symbol": "O", "Relative Atomic Mass": "23.015696(97)", "Mass Number": "23"}, "24": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[15.99903,15.99977]", "Atomic Number": "8", "Atomic Symbol": "O", "Relative Atomic Mass": "24.01986(12)", "Mass Number": "24"}, "25": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[15.99903,15.99977]", "Atomic Number": "8", "Atomic Symbol": "O", "Relative Atomic Mass": "25.02936(12)", "Mass Number": "25"}, "26": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[15.99903,15.99977]", "Atomic Number": "8", "Atomic Symbol": "O", "Relative Atomic Mass": "26.03729(17)", "Mass Number": "26"}, "27": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[15.99903,15.99977]", "Atomic Number": "8", "Atomic Symbol": "O", "Relative Atomic Mass": "27.04772(54#)", "Mass Number": "27"}, "28": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[15.99903,15.99977]", "Atomic Number": "8", "Atomic Symbol": "O", "Relative Atomic Mass": "28.05591(75#)", "Mass Number": "28"}}, "94": {"228": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[244]", "Atomic Number": "94", "Atomic Symbol": "Pu", "Relative Atomic Mass": "228.038732(33)", "Mass Number": "228"}, "229": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[244]", "Atomic Number": "94", "Atomic Symbol": "Pu", "Relative Atomic Mass": "229.040144(55)", "Mass Number": "229"}, "230": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[244]", "Atomic Number": "94", "Atomic Symbol": "Pu", "Relative Atomic Mass": "230.039650(16)", "Mass Number": "230"}, "231": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[244]", "Atomic Number": "94", "Atomic Symbol": "Pu", "Relative Atomic Mass": "231.041102(28)", "Mass Number": "231"}, "232": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[244]", "Atomic Number": "94", "Atomic Symbol": "Pu", "Relative Atomic Mass": "232.041185(19)", "Mass Number": "232"}, "233": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[244]", "Atomic Number": "94", "Atomic Symbol": "Pu", "Relative Atomic Mass": "233.042998(54)", "Mass Number": "233"}, "234": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[244]", "Atomic Number": "94", "Atomic Symbol": "Pu", "Relative Atomic Mass": "234.0433174(75)", "Mass Number": "234"}, "235": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[244]", "Atomic Number": "94", "Atomic Symbol": "Pu", "Relative Atomic Mass": "235.045286(22)", "Mass Number": "235"}, "236": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[244]", "Atomic Number": "94", "Atomic Symbol": "Pu", "Relative Atomic Mass": "236.0460581(23)", "Mass Number": "236"}, "237": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[244]", "Atomic Number": "94", "Atomic Symbol": "Pu", "Relative Atomic Mass": "237.0484098(24)", "Mass Number": "237"}, "238": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[244]", "Atomic Number": "94", "Atomic Symbol": "Pu", "Relative Atomic Mass": "238.0495601(19)", "Mass Number": "238"}, "239": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[244]", "Atomic Number": "94", "Atomic Symbol": "Pu", "Relative Atomic Mass": "239.0521636(19)", "Mass Number": "239"}, "240": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[244]", "Atomic Number": "94", "Atomic Symbol": "Pu", "Relative Atomic Mass": "240.0538138(19)", "Mass Number": "240"}, "241": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[244]", "Atomic Number": "94", "Atomic Symbol": "Pu", "Relative Atomic Mass": "241.0568517(19)", "Mass Number": "241"}, "242": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[244]", "Atomic Number": "94", "Atomic Symbol": "Pu", "Relative Atomic Mass": "242.0587428(20)", "Mass Number": "242"}, "243": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[244]", "Atomic Number": "94", "Atomic Symbol": "Pu", "Relative Atomic Mass": "243.0620036(34)", "Mass Number": "243"}, "244": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[244]", "Atomic Number": "94", "Atomic Symbol": "Pu", "Relative Atomic Mass": "244.0642053(56)", "Mass Number": "244"}, "245": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[244]", "Atomic Number": "94", "Atomic Symbol": "Pu", "Relative Atomic Mass": "245.067826(15)", "Mass Number": "245"}, "246": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[244]", "Atomic Number": "94", "Atomic Symbol": "Pu", "Relative Atomic Mass": "246.070205(16)", "Mass Number": "246"}, "247": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[244]", "Atomic Number": "94", "Atomic Symbol": "Pu", "Relative Atomic Mass": "247.07419(21#)", "Mass Number": "247"}}, "9": {"14": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "18.998403163(6)", "Atomic Number": "9", "Atomic Symbol": "F", "Relative Atomic Mass": "14.034315(44)", "Mass Number": "14"}, "15": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "18.998403163(6)", "Atomic Number": "9", "Atomic Symbol": "F", "Relative Atomic Mass": "15.018043(67)", "Mass Number": "15"}, "16": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "18.998403163(6)", "Atomic Number": "9", "Atomic Symbol": "F", "Relative Atomic Mass": "16.0114657(89)", "Mass Number": "16"}, "17": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "18.998403163(6)", "Atomic Number": "9", "Atomic Symbol": "F", "Relative Atomic Mass": "17.00209524(27)", "Mass Number": "17"}, "18": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "18.998403163(6)", "Atomic Number": "9", "Atomic Symbol": "F", "Relative Atomic Mass": "18.00093733(50)", "Mass Number": "18"}, "19": {"Isotopic Composition": "1", "Notes": " ", "Standard Atomic Weight": "18.998403163(6)", "Atomic Number": "9", "Atomic Symbol": "F", "Relative Atomic Mass": "18.99840316273(92)", "Mass Number": "19"}, "20": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "18.998403163(6)", "Atomic Number": "9", "Atomic Symbol": "F", "Relative Atomic Mass": "19.999981252(31)", "Mass Number": "20"}, "21": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "18.998403163(6)", "Atomic Number": "9", "Atomic Symbol": "F", "Relative Atomic Mass": "20.9999489(19)", "Mass Number": "21"}, "22": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "18.998403163(6)", "Atomic Number": "9", "Atomic Symbol": "F", "Relative Atomic Mass": "22.002999(13)", "Mass Number": "22"}, "23": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "18.998403163(6)", "Atomic Number": "9", "Atomic Symbol": "F", "Relative Atomic Mass": "23.003557(54)", "Mass Number": "23"}, "24": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "18.998403163(6)", "Atomic Number": "9", "Atomic Symbol": "F", "Relative Atomic Mass": "24.008115(78)", "Mass Number": "24"}, "25": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "18.998403163(6)", "Atomic Number": "9", "Atomic Symbol": "F", "Relative Atomic Mass": "25.012199(81)", "Mass Number": "25"}, "26": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "18.998403163(6)", "Atomic Number": "9", "Atomic Symbol": "F", "Relative Atomic Mass": "26.020038(83)", "Mass Number": "26"}, "27": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "18.998403163(6)", "Atomic Number": "9", "Atomic Symbol": "F", "Relative Atomic Mass": "27.02644(20)", "Mass Number": "27"}, "28": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "18.998403163(6)", "Atomic Number": "9", "Atomic Symbol": "F", "Relative Atomic Mass": "28.03534(21)", "Mass Number": "28"}, "29": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "18.998403163(6)", "Atomic Number": "9", "Atomic Symbol": "F", "Relative Atomic Mass": "29.04254(54#)", "Mass Number": "29"}, "30": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "18.998403163(6)", "Atomic Number": "9", "Atomic Symbol": "F", "Relative Atomic Mass": "30.05165(64#)", "Mass Number": "30"}, "31": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "18.998403163(6)", "Atomic Number": "9", "Atomic Symbol": "F", "Relative Atomic Mass": "31.05971(56#)", "Mass Number": "31"}}, "6": {"8": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[12.0096,12.0116]", "Atomic Number": "6", "Atomic Symbol": "C", "Relative Atomic Mass": "8.037643(20)", "Mass Number": "8"}, "9": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[12.0096,12.0116]", "Atomic Number": "6", "Atomic Symbol": "C", "Relative Atomic Mass": "9.0310372(23)", "Mass Number": "9"}, "10": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[12.0096,12.0116]", "Atomic Number": "6", "Atomic Symbol": "C", "Relative Atomic Mass": "10.01685331(42)", "Mass Number": "10"}, "11": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[12.0096,12.0116]", "Atomic Number": "6", "Atomic Symbol": "C", "Relative Atomic Mass": "11.0114336(10)", "Mass Number": "11"}, "12": {"Isotopic Composition": "0.9893(8)", "Notes": " ", "Standard Atomic Weight": "[12.0096,12.0116]", "Atomic Number": "6", "Atomic Symbol": "C", "Relative Atomic Mass": "12.0000000(00)", "Mass Number": "12"}, "13": {"Isotopic Composition": "0.0107(8)", "Notes": " ", "Standard Atomic Weight": "[12.0096,12.0116]", "Atomic Number": "6", "Atomic Symbol": "C", "Relative Atomic Mass": "13.00335483507(23)", "Mass Number": "13"}, "14": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[12.0096,12.0116]", "Atomic Number": "6", "Atomic Symbol": "C", "Relative Atomic Mass": "14.0032419884(40)", "Mass Number": "14"}, "15": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[12.0096,12.0116]", "Atomic Number": "6", "Atomic Symbol": "C", "Relative Atomic Mass": "15.01059926(86)", "Mass Number": "15"}, "16": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[12.0096,12.0116]", "Atomic Number": "6", "Atomic Symbol": "C", "Relative Atomic Mass": "16.0147013(38)", "Mass Number": "16"}, "17": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[12.0096,12.0116]", "Atomic Number": "6", "Atomic Symbol": "C", "Relative Atomic Mass": "17.022577(19)", "Mass Number": "17"}, "18": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[12.0096,12.0116]", "Atomic Number": "6", "Atomic Symbol": "C", "Relative Atomic Mass": "18.026751(32)", "Mass Number": "18"}, "19": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[12.0096,12.0116]", "Atomic Number": "6", "Atomic Symbol": "C", "Relative Atomic Mass": "19.03480(11)", "Mass Number": "19"}, "20": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[12.0096,12.0116]", "Atomic Number": "6", "Atomic Symbol": "C", "Relative Atomic Mass": "20.04032(26)", "Mass Number": "20"}, "21": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[12.0096,12.0116]", "Atomic Number": "6", "Atomic Symbol": "C", "Relative Atomic Mass": "21.04900(43#)", "Mass Number": "21"}, "22": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[12.0096,12.0116]", "Atomic Number": "6", "Atomic Symbol": "C", "Relative Atomic Mass": "22.05753(26)", "Mass Number": "22"}, "23": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[12.0096,12.0116]", "Atomic Number": "6", "Atomic Symbol": "C", "Relative Atomic Mass": "23.0689(11#)", "Mass Number": "23"}}, "73": {"192": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "191.97514(43#)", "Mass Number": "192"}, "155": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "154.97424(54#)", "Mass Number": "155"}, "156": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "155.97203(32#)", "Mass Number": "156"}, "157": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "156.96818(17)", "Mass Number": "157"}, "158": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "157.96654(22#)", "Mass Number": "158"}, "159": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "158.963023(21)", "Mass Number": "159"}, "160": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "159.961488(79)", "Mass Number": "160"}, "161": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "160.958452(27)", "Mass Number": "161"}, "162": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "161.957294(56)", "Mass Number": "162"}, "163": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "162.954337(41)", "Mass Number": "163"}, "164": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "163.953534(30)", "Mass Number": "164"}, "165": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "164.950781(15)", "Mass Number": "165"}, "166": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "165.950512(30)", "Mass Number": "166"}, "167": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "166.948093(30)", "Mass Number": "167"}, "168": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "167.948047(30)", "Mass Number": "168"}, "169": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "168.946011(30)", "Mass Number": "169"}, "170": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "169.946175(30)", "Mass Number": "170"}, "171": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "170.944476(30)", "Mass Number": "171"}, "172": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "171.944895(30)", "Mass Number": "172"}, "173": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "172.943750(30)", "Mass Number": "173"}, "174": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "173.944454(30)", "Mass Number": "174"}, "175": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "174.943737(30)", "Mass Number": "175"}, "176": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "175.944857(33)", "Mass Number": "176"}, "177": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "176.9444795(38)", "Mass Number": "177"}, "178": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "177.945678(56#)", "Mass Number": "178"}, "179": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "178.9459366(21)", "Mass Number": "179"}, "180": {"Isotopic Composition": "0.0001201(32)", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "179.9474648(24)", "Mass Number": "180"}, "181": {"Isotopic Composition": "0.9998799(32)", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "180.9479958(20)", "Mass Number": "181"}, "182": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "181.9501519(20)", "Mass Number": "182"}, "183": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "182.9513726(20)", "Mass Number": "183"}, "184": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "183.954008(28)", "Mass Number": "184"}, "185": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "184.955559(15)", "Mass Number": "185"}, "186": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "185.958551(64)", "Mass Number": "186"}, "187": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "186.960386(71)", "Mass Number": "187"}, "188": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "187.963916(71)", "Mass Number": "188"}, "189": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "188.96583(32#)", "Mass Number": "189"}, "190": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "189.96939(21#)", "Mass Number": "190"}, "191": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "180.94788(2)", "Atomic Number": "73", "Atomic Symbol": "Ta", "Relative Atomic Mass": "190.97156(32#)", "Mass Number": "191"}}, "85": {"192": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "192.003152(35)", "Mass Number": "192"}, "193": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "192.999927(23)", "Mass Number": "193"}, "194": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "193.999236(29)", "Mass Number": "194"}, "195": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "194.9962685(98)", "Mass Number": "195"}, "196": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "195.995800(33)", "Mass Number": "196"}, "197": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "196.993189(55)", "Mass Number": "197"}, "198": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "197.992784(54#)", "Mass Number": "198"}, "199": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "198.9905277(58)", "Mass Number": "199"}, "200": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "199.990351(26)", "Mass Number": "200"}, "201": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "200.9884171(88)", "Mass Number": "201"}, "202": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "201.988630(30)", "Mass Number": "202"}, "203": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "202.986943(11)", "Mass Number": "203"}, "204": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "203.987251(24)", "Mass Number": "204"}, "205": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "204.986076(16)", "Mass Number": "205"}, "206": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "205.986657(16)", "Mass Number": "206"}, "207": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "206.985800(13)", "Mass Number": "207"}, "208": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "207.9866133(96)", "Mass Number": "208"}, "209": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "208.9861702(55)", "Mass Number": "209"}, "210": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "209.9871479(83)", "Mass Number": "210"}, "211": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "210.9874966(30)", "Mass Number": "211"}, "212": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "211.9907377(26)", "Mass Number": "212"}, "213": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "212.9929370(53)", "Mass Number": "213"}, "214": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "213.9963721(46)", "Mass Number": "214"}, "215": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "214.9986528(73)", "Mass Number": "215"}, "216": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "216.0024236(39)", "Mass Number": "216"}, "217": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "217.0047192(55)", "Mass Number": "217"}, "218": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "218.008695(12)", "Mass Number": "218"}, "219": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "219.0111618(42)", "Mass Number": "219"}, "220": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "220.015433(15)", "Mass Number": "220"}, "221": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "221.018017(15)", "Mass Number": "221"}, "222": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "222.022494(17)", "Mass Number": "222"}, "223": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "223.025151(15)", "Mass Number": "223"}, "224": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "224.029749(24)", "Mass Number": "224"}, "225": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "225.03263(32#)", "Mass Number": "225"}, "226": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "226.03716(32#)", "Mass Number": "226"}, "227": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "227.04024(32#)", "Mass Number": "227"}, "228": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "228.04475(43#)", "Mass Number": "228"}, "229": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "229.04812(43#)", "Mass Number": "229"}, "191": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[210]", "Atomic Number": "85", "Atomic Symbol": "At", "Relative Atomic Mass": "191.004148(17)", "Mass Number": "191"}}, "37": {"71": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "70.96532(54#)", "Mass Number": "71"}, "72": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "71.95908(54#)", "Mass Number": "72"}, "73": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "72.95053(11#)", "Mass Number": "73"}, "74": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "73.9442659(32)", "Mass Number": "74"}, "75": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "74.9385732(13)", "Mass Number": "75"}, "76": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "75.9350730(10)", "Mass Number": "76"}, "77": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "76.9304016(14)", "Mass Number": "77"}, "78": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "77.9281419(35)", "Mass Number": "78"}, "79": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "78.9239899(23)", "Mass Number": "79"}, "80": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "79.9225164(20)", "Mass Number": "80"}, "81": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "80.9189939(53)", "Mass Number": "81"}, "82": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "81.9182090(32)", "Mass Number": "82"}, "83": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "82.9151142(25)", "Mass Number": "83"}, "84": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "83.9143752(24)", "Mass Number": "84"}, "85": {"Isotopic Composition": "0.7217(2)", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "84.9117897379(54)", "Mass Number": "85"}, "86": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "85.91116743(21)", "Mass Number": "86"}, "87": {"Isotopic Composition": "0.2783(2)", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "86.9091805310(60)", "Mass Number": "87"}, "88": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "87.91131559(17)", "Mass Number": "88"}, "89": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "88.9122783(59)", "Mass Number": "89"}, "90": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "89.9147985(70)", "Mass Number": "90"}, "91": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "90.9165372(84)", "Mass Number": "91"}, "92": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "91.9197284(66)", "Mass Number": "92"}, "93": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "92.9220393(84)", "Mass Number": "93"}, "94": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "93.9263948(22)", "Mass Number": "94"}, "95": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "94.929260(22)", "Mass Number": "95"}, "96": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "95.9341334(36)", "Mass Number": "96"}, "97": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "96.9371771(21)", "Mass Number": "97"}, "98": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "97.9416869(37)", "Mass Number": "98"}, "99": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "98.94503(12)", "Mass Number": "99"}, "100": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "99.95003(21#)", "Mass Number": "100"}, "101": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "100.95404(23#)", "Mass Number": "101"}, "102": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "101.95952(32#)", "Mass Number": "102"}, "103": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "85.4678(3)", "Atomic Number": "37", "Atomic Symbol": "Rb", "Relative Atomic Mass": "102.96392(43#)", "Mass Number": "103"}}, "19": {"32": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "39.0983(1)", "Atomic Number": "19", "Atomic Symbol": "K", "Relative Atomic Mass": "32.02265(54#)", "Mass Number": "32"}, "33": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "39.0983(1)", "Atomic Number": "19", "Atomic Symbol": "K", "Relative Atomic Mass": "33.00756(21#)", "Mass Number": "33"}, "34": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "39.0983(1)", "Atomic Number": "19", "Atomic Symbol": "K", "Relative Atomic Mass": "33.99869(32#)", "Mass Number": "34"}, "35": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "39.0983(1)", "Atomic Number": "19", "Atomic Symbol": "K", "Relative Atomic Mass": "34.98800541(55)", "Mass Number": "35"}, "36": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "39.0983(1)", "Atomic Number": "19", "Atomic Symbol": "K", "Relative Atomic Mass": "35.98130201(37)", "Mass Number": "36"}, "37": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "39.0983(1)", "Atomic Number": "19", "Atomic Symbol": "K", "Relative Atomic Mass": "36.97337589(10)", "Mass Number": "37"}, "38": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "39.0983(1)", "Atomic Number": "19", "Atomic Symbol": "K", "Relative Atomic Mass": "37.96908112(21)", "Mass Number": "38"}, "39": {"Isotopic Composition": "0.932581(44)", "Notes": " ", "Standard Atomic Weight": "39.0983(1)", "Atomic Number": "19", "Atomic Symbol": "K", "Relative Atomic Mass": "38.9637064864(49)", "Mass Number": "39"}, "40": {"Isotopic Composition": "0.000117(1)", "Notes": " ", "Standard Atomic Weight": "39.0983(1)", "Atomic Number": "19", "Atomic Symbol": "K", "Relative Atomic Mass": "39.963998166(60)", "Mass Number": "40"}, "41": {"Isotopic Composition": "0.067302(44)", "Notes": " ", "Standard Atomic Weight": "39.0983(1)", "Atomic Number": "19", "Atomic Symbol": "K", "Relative Atomic Mass": "40.9618252579(41)", "Mass Number": "41"}, "42": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "39.0983(1)", "Atomic Number": "19", "Atomic Symbol": "K", "Relative Atomic Mass": "41.96240231(11)", "Mass Number": "42"}, "43": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "39.0983(1)", "Atomic Number": "19", "Atomic Symbol": "K", "Relative Atomic Mass": "42.96073470(44)", "Mass Number": "43"}, "44": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "39.0983(1)", "Atomic Number": "19", "Atomic Symbol": "K", "Relative Atomic Mass": "43.96158699(45)", "Mass Number": "44"}, "45": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "39.0983(1)", "Atomic Number": "19", "Atomic Symbol": "K", "Relative Atomic Mass": "44.96069149(56)", "Mass Number": "45"}, "46": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "39.0983(1)", "Atomic Number": "19", "Atomic Symbol": "K", "Relative Atomic Mass": "45.96198159(78)", "Mass Number": "46"}, "47": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "39.0983(1)", "Atomic Number": "19", "Atomic Symbol": "K", "Relative Atomic Mass": "46.9616616(15)", "Mass Number": "47"}, "48": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "39.0983(1)", "Atomic Number": "19", "Atomic Symbol": "K", "Relative Atomic Mass": "47.96534119(83)", "Mass Number": "48"}, "49": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "39.0983(1)", "Atomic Number": "19", "Atomic Symbol": "K", "Relative Atomic Mass": "48.96821075(86)", "Mass Number": "49"}, "50": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "39.0983(1)", "Atomic Number": "19", "Atomic Symbol": "K", "Relative Atomic Mass": "49.9723800(83)", "Mass Number": "50"}, "51": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "39.0983(1)", "Atomic Number": "19", "Atomic Symbol": "K", "Relative Atomic Mass": "50.975828(14)", "Mass Number": "51"}, "52": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "39.0983(1)", "Atomic Number": "19", "Atomic Symbol": "K", "Relative Atomic Mass": "51.98224(43#)", "Mass Number": "52"}, "53": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "39.0983(1)", "Atomic Number": "19", "Atomic Symbol": "K", "Relative Atomic Mass": "52.98746(54#)", "Mass Number": "53"}, "54": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "39.0983(1)", "Atomic Number": "19", "Atomic Symbol": "K", "Relative Atomic Mass": "53.99463(64#)", "Mass Number": "54"}, "55": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "39.0983(1)", "Atomic Number": "19", "Atomic Symbol": "K", "Relative Atomic Mass": "55.00076(75#)", "Mass Number": "55"}, "56": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "39.0983(1)", "Atomic Number": "19", "Atomic Symbol": "K", "Relative Atomic Mass": "56.00851(86#)", "Mass Number": "56"}}, "41": {"81": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "80.94960(43#)", "Mass Number": "81"}, "82": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "81.94396(32#)", "Mass Number": "82"}, "83": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "82.93729(32)", "Mass Number": "83"}, "84": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "83.93449(32#)", "Mass Number": "84"}, "85": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "84.9288458(44)", "Mass Number": "85"}, "86": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "85.9257828(59)", "Mass Number": "86"}, "87": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "86.9206937(73)", "Mass Number": "87"}, "88": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "87.918222(61)", "Mass Number": "88"}, "89": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "88.913445(25)", "Mass Number": "89"}, "90": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "89.9112584(38)", "Mass Number": "90"}, "91": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "90.9069897(37)", "Mass Number": "91"}, "92": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "91.9071881(26)", "Mass Number": "92"}, "93": {"Isotopic Composition": "1", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "92.9063730(20)", "Mass Number": "93"}, "94": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "93.9072788(20)", "Mass Number": "94"}, "95": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "94.90683240(71)", "Mass Number": "95"}, "96": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "95.9080973(35)", "Mass Number": "96"}, "97": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "96.9080959(19)", "Mass Number": "97"}, "98": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "97.9103265(58)", "Mass Number": "98"}, "99": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "98.911613(13)", "Mass Number": "99"}, "100": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "99.9143276(88)", "Mass Number": "100"}, "101": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "100.9153103(42)", "Mass Number": "101"}, "102": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "101.9180772(35)", "Mass Number": "102"}, "103": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "102.9194572(44)", "Mass Number": "103"}, "104": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "103.9228925(37)", "Mass Number": "104"}, "105": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "104.9249465(45)", "Mass Number": "105"}, "106": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "105.9289317(46)", "Mass Number": "106"}, "107": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "106.9315937(87)", "Mass Number": "107"}, "108": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "107.9360748(88)", "Mass Number": "108"}, "109": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "108.93922(56)", "Mass Number": "109"}, "110": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "109.94403(21#)", "Mass Number": "110"}, "111": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "110.94753(32#)", "Mass Number": "111"}, "112": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "111.95247(32#)", "Mass Number": "112"}, "113": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "112.95651(43#)", "Mass Number": "113"}, "114": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "113.96201(54#)", "Mass Number": "114"}, "115": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "92.90637(2)", "Atomic Number": "41", "Atomic Symbol": "Nb", "Relative Atomic Mass": "114.96634(54#)", "Mass Number": "115"}}, "64": {"133": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "132.96133(54#)", "Mass Number": "133"}, "134": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "133.95566(43#)", "Mass Number": "134"}, "135": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "134.95245(43#)", "Mass Number": "135"}, "136": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "135.94730(32#)", "Mass Number": "136"}, "137": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "136.94502(32#)", "Mass Number": "137"}, "138": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "137.94025(21#)", "Mass Number": "138"}, "139": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "138.93813(21#)", "Mass Number": "139"}, "140": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "139.933674(30)", "Mass Number": "140"}, "141": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "140.932126(21)", "Mass Number": "141"}, "142": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "141.928116(30)", "Mass Number": "142"}, "143": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "142.92675(22)", "Mass Number": "143"}, "144": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "143.922963(30)", "Mass Number": "144"}, "145": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "144.921713(21)", "Mass Number": "145"}, "146": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "145.9183188(46)", "Mass Number": "146"}, "147": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "146.9191014(25)", "Mass Number": "147"}, "148": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "147.9181215(21)", "Mass Number": "148"}, "149": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "148.9193481(38)", "Mass Number": "149"}, "150": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "149.9186644(66)", "Mass Number": "150"}, "151": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "150.9203560(35)", "Mass Number": "151"}, "152": {"Isotopic Composition": "0.0020(1)", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "151.9197995(18)", "Mass Number": "152"}, "153": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "152.9217580(18)", "Mass Number": "153"}, "154": {"Isotopic Composition": "0.0218(3)", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "153.9208741(17)", "Mass Number": "154"}, "155": {"Isotopic Composition": "0.1480(12)", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "154.9226305(17)", "Mass Number": "155"}, "156": {"Isotopic Composition": "0.2047(9)", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "155.9221312(17)", "Mass Number": "156"}, "157": {"Isotopic Composition": "0.1565(2)", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "156.9239686(17)", "Mass Number": "157"}, "158": {"Isotopic Composition": "0.2484(7)", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "157.9241123(17)", "Mass Number": "158"}, "159": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "158.9263970(17)", "Mass Number": "159"}, "160": {"Isotopic Composition": "0.2186(19)", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "159.9270624(18)", "Mass Number": "160"}, "161": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "160.9296775(21)", "Mass Number": "161"}, "162": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "161.9309930(45)", "Mass Number": "162"}, "163": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "162.9341769(90)", "Mass Number": "163"}, "164": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "163.93583(21#)", "Mass Number": "164"}, "165": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "164.93936(32#)", "Mass Number": "165"}, "166": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "165.94146(64#)", "Mass Number": "166"}, "167": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "166.94545(43#)", "Mass Number": "167"}, "168": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "167.94808(43#)", "Mass Number": "168"}, "169": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "157.25(3)", "Atomic Number": "64", "Atomic Symbol": "Gd", "Relative Atomic Mass": "168.95260(54#)", "Mass Number": "169"}}, "67": {"140": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "139.96859(54#)", "Mass Number": "140"}, "141": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "140.96311(54#)", "Mass Number": "141"}, "142": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "141.96001(54#)", "Mass Number": "142"}, "143": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "142.95486(43#)", "Mass Number": "143"}, "144": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "143.9521097(91)", "Mass Number": "144"}, "145": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "144.9472674(80)", "Mass Number": "145"}, "146": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "145.9449935(71)", "Mass Number": "146"}, "147": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "146.9401423(54)", "Mass Number": "147"}, "148": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "147.937744(90)", "Mass Number": "148"}, "149": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "148.933803(16)", "Mass Number": "149"}, "150": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "149.933498(15)", "Mass Number": "150"}, "151": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "150.9316983(89)", "Mass Number": "151"}, "152": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "151.931724(14)", "Mass Number": "152"}, "153": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "152.9302064(56)", "Mass Number": "153"}, "154": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "153.9306068(89)", "Mass Number": "154"}, "155": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "154.929104(19)", "Mass Number": "155"}, "156": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "155.929706(64)", "Mass Number": "156"}, "157": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "156.928254(25)", "Mass Number": "157"}, "158": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "157.928946(29)", "Mass Number": "158"}, "159": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "158.9277197(36)", "Mass Number": "159"}, "160": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "159.928737(16)", "Mass Number": "160"}, "161": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "160.9278615(30)", "Mass Number": "161"}, "162": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "161.9291023(39)", "Mass Number": "162"}, "163": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "162.9287410(20)", "Mass Number": "163"}, "164": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "163.9302403(25)", "Mass Number": "164"}, "165": {"Isotopic Composition": "1", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "164.9303288(21)", "Mass Number": "165"}, "166": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "165.9322909(21)", "Mass Number": "166"}, "167": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "166.9331385(59)", "Mass Number": "167"}, "168": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "167.935522(32)", "Mass Number": "168"}, "169": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "168.936878(22)", "Mass Number": "169"}, "170": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "169.939625(54)", "Mass Number": "170"}, "171": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "170.94147(64)", "Mass Number": "171"}, "172": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "171.94473(21#)", "Mass Number": "172"}, "173": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "172.94702(32#)", "Mass Number": "173"}, "174": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "173.95095(32#)", "Mass Number": "174"}, "175": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "164.93033(2)", "Atomic Number": "67", "Atomic Symbol": "Ho", "Relative Atomic Mass": "174.95362(43#)", "Mass Number": "175"}}, "113": {"278": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "113", "Atomic Symbol": "Nh", "Relative Atomic Mass": "278.17058(20#)", "Mass Number": "278"}, "279": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "113", "Atomic Symbol": "Nh", "Relative Atomic Mass": "279.17095(75#)", "Mass Number": "279"}, "280": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "113", "Atomic Symbol": "Nh", "Relative Atomic Mass": "280.17293(75#)", "Mass Number": "280"}, "281": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "113", "Atomic Symbol": "Nh", "Relative Atomic Mass": "281.17348(75#)", "Mass Number": "281"}, "282": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "113", "Atomic Symbol": "Nh", "Relative Atomic Mass": "282.17567(39#)", "Mass Number": "282"}, "283": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "113", "Atomic Symbol": "Nh", "Relative Atomic Mass": "283.17657(52#)", "Mass Number": "283"}, "284": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "113", "Atomic Symbol": "Nh", "Relative Atomic Mass": "284.17873(62#)", "Mass Number": "284"}, "285": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "113", "Atomic Symbol": "Nh", "Relative Atomic Mass": "285.17973(89#)", "Mass Number": "285"}, "286": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "113", "Atomic Symbol": "Nh", "Relative Atomic Mass": "286.18221(72#)", "Mass Number": "286"}, "287": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "113", "Atomic Symbol": "Nh", "Relative Atomic Mass": "287.18339(81#)", "Mass Number": "287"}}, "97": {"234": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "97", "Atomic Symbol": "Bk", "Relative Atomic Mass": "234.05727(15#)", "Mass Number": "234"}, "235": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "97", "Atomic Symbol": "Bk", "Relative Atomic Mass": "235.05658(43#)", "Mass Number": "235"}, "236": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "97", "Atomic Symbol": "Bk", "Relative Atomic Mass": "236.05748(43#)", "Mass Number": "236"}, "237": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "97", "Atomic Symbol": "Bk", "Relative Atomic Mass": "237.05710(24#)", "Mass Number": "237"}, "238": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "97", "Atomic Symbol": "Bk", "Relative Atomic Mass": "238.05820(27#)", "Mass Number": "238"}, "239": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "97", "Atomic Symbol": "Bk", "Relative Atomic Mass": "239.05824(22#)", "Mass Number": "239"}, "240": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "97", "Atomic Symbol": "Bk", "Relative Atomic Mass": "240.05976(16#)", "Mass Number": "240"}, "241": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "97", "Atomic Symbol": "Bk", "Relative Atomic Mass": "241.06016(22#)", "Mass Number": "241"}, "242": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "97", "Atomic Symbol": "Bk", "Relative Atomic Mass": "242.06198(22#)", "Mass Number": "242"}, "243": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "97", "Atomic Symbol": "Bk", "Relative Atomic Mass": "243.0630078(51)", "Mass Number": "243"}, "244": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "97", "Atomic Symbol": "Bk", "Relative Atomic Mass": "244.065181(16)", "Mass Number": "244"}, "245": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "97", "Atomic Symbol": "Bk", "Relative Atomic Mass": "245.0663618(24)", "Mass Number": "245"}, "246": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "97", "Atomic Symbol": "Bk", "Relative Atomic Mass": "246.068673(64)", "Mass Number": "246"}, "247": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "97", "Atomic Symbol": "Bk", "Relative Atomic Mass": "247.0703073(59)", "Mass Number": "247"}, "248": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "97", "Atomic Symbol": "Bk", "Relative Atomic Mass": "248.073088(76#)", "Mass Number": "248"}, "249": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "97", "Atomic Symbol": "Bk", "Relative Atomic Mass": "249.0749877(27)", "Mass Number": "249"}, "250": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "97", "Atomic Symbol": "Bk", "Relative Atomic Mass": "250.0783167(42)", "Mass Number": "250"}, "251": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "97", "Atomic Symbol": "Bk", "Relative Atomic Mass": "251.080762(12)", "Mass Number": "251"}, "252": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "97", "Atomic Symbol": "Bk", "Relative Atomic Mass": "252.08431(22#)", "Mass Number": "252"}, "253": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "97", "Atomic Symbol": "Bk", "Relative Atomic Mass": "253.08688(39#)", "Mass Number": "253"}, "254": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "97", "Atomic Symbol": "Bk", "Relative Atomic Mass": "254.09060(32#)", "Mass Number": "254"}}, "88": {"201": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "201.01271(11#)", "Mass Number": "201"}, "202": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "202.009760(26)", "Mass Number": "202"}, "203": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "203.009304(86)", "Mass Number": "203"}, "204": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "204.006492(16)", "Mass Number": "204"}, "205": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "205.006268(76)", "Mass Number": "205"}, "206": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "206.003828(19)", "Mass Number": "206"}, "207": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "207.003799(59)", "Mass Number": "207"}, "208": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "208.001841(17)", "Mass Number": "208"}, "209": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "209.001990(54)", "Mass Number": "209"}, "210": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "210.000494(16)", "Mass Number": "210"}, "211": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "211.0008932(85)", "Mass Number": "211"}, "212": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "211.999787(12)", "Mass Number": "212"}, "213": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "213.000384(22)", "Mass Number": "213"}, "214": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "214.0000997(56)", "Mass Number": "214"}, "215": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "215.0027204(82)", "Mass Number": "215"}, "216": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "216.0035334(94)", "Mass Number": "216"}, "217": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "217.0063207(92)", "Mass Number": "217"}, "218": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "218.007141(12)", "Mass Number": "218"}, "219": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "219.0100855(89)", "Mass Number": "219"}, "220": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "220.0110259(89)", "Mass Number": "220"}, "221": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "221.0139177(50)", "Mass Number": "221"}, "222": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "222.0153748(49)", "Mass Number": "222"}, "223": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "223.0185023(27)", "Mass Number": "223"}, "224": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "224.0202120(23)", "Mass Number": "224"}, "225": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "225.0236119(32)", "Mass Number": "225"}, "226": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "226.0254103(25)", "Mass Number": "226"}, "227": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "227.0291783(25)", "Mass Number": "227"}, "228": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "228.0310707(26)", "Mass Number": "228"}, "229": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "229.034942(16)", "Mass Number": "229"}, "230": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "230.037055(11)", "Mass Number": "230"}, "231": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "231.041027(12)", "Mass Number": "231"}, "232": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "232.0434753(98)", "Mass Number": "232"}, "233": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "233.047582(17)", "Mass Number": "233"}, "234": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "234.050342(33)", "Mass Number": "234"}, "235": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[226]", "Atomic Number": "88", "Atomic Symbol": "Ra", "Relative Atomic Mass": "235.05497(32#)", "Mass Number": "235"}}, "71": {"150": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "149.97355(54#)", "Mass Number": "150"}, "151": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "150.96768(43#)", "Mass Number": "151"}, "152": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "151.96412(21#)", "Mass Number": "152"}, "153": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "152.95875(17)", "Mass Number": "153"}, "154": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "153.95736(22#)", "Mass Number": "154"}, "155": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "154.954321(21)", "Mass Number": "155"}, "156": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "155.953033(79)", "Mass Number": "156"}, "157": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "156.950127(16)", "Mass Number": "157"}, "158": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "157.949316(16)", "Mass Number": "158"}, "159": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "158.946636(40)", "Mass Number": "159"}, "160": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "159.946033(61)", "Mass Number": "160"}, "161": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "160.943572(30)", "Mass Number": "161"}, "162": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "161.943283(81)", "Mass Number": "162"}, "163": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "162.941179(30)", "Mass Number": "163"}, "164": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "163.941339(30)", "Mass Number": "164"}, "165": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "164.939407(28)", "Mass Number": "165"}, "166": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "165.939859(32)", "Mass Number": "166"}, "167": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "166.938270(34)", "Mass Number": "167"}, "168": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "167.938736(42)", "Mass Number": "168"}, "169": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "168.9376441(39)", "Mass Number": "169"}, "170": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "169.938478(18)", "Mass Number": "170"}, "171": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "170.9379170(27)", "Mass Number": "171"}, "172": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "171.9390891(30)", "Mass Number": "172"}, "173": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "172.9389340(23)", "Mass Number": "173"}, "174": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "173.9403409(23)", "Mass Number": "174"}, "175": {"Isotopic Composition": "0.97401(13)", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "174.9407752(20)", "Mass Number": "175"}, "176": {"Isotopic Composition": "0.02599(13)", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "175.9426897(20)", "Mass Number": "176"}, "177": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "176.9437615(20)", "Mass Number": "177"}, "178": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "177.9459580(29)", "Mass Number": "178"}, "179": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "178.9473309(57)", "Mass Number": "179"}, "180": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "179.949888(76)", "Mass Number": "180"}, "181": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "180.95191(17)", "Mass Number": "181"}, "182": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "181.95504(21#)", "Mass Number": "182"}, "183": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "182.957363(98)", "Mass Number": "183"}, "184": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "183.96091(32#)", "Mass Number": "184"}, "185": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "174.9668(1)", "Atomic Number": "71", "Atomic Symbol": "Lu", "Relative Atomic Mass": "184.96362(32#)", "Mass Number": "185"}}, "14": {"22": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[28.084,28.086]", "Atomic Number": "14", "Atomic Symbol": "Si", "Relative Atomic Mass": "22.03579(54#)", "Mass Number": "22"}, "23": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[28.084,28.086]", "Atomic Number": "14", "Atomic Symbol": "Si", "Relative Atomic Mass": "23.02544(54#)", "Mass Number": "23"}, "24": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[28.084,28.086]", "Atomic Number": "14", "Atomic Symbol": "Si", "Relative Atomic Mass": "24.011535(21)", "Mass Number": "24"}, "25": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[28.084,28.086]", "Atomic Number": "14", "Atomic Symbol": "Si", "Relative Atomic Mass": "25.004109(11)", "Mass Number": "25"}, "26": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[28.084,28.086]", "Atomic Number": "14", "Atomic Symbol": "Si", "Relative Atomic Mass": "25.99233384(11)", "Mass Number": "26"}, "27": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[28.084,28.086]", "Atomic Number": "14", "Atomic Symbol": "Si", "Relative Atomic Mass": "26.98670481(15)", "Mass Number": "27"}, "28": {"Isotopic Composition": "0.92223(19)", "Notes": " ", "Standard Atomic Weight": "[28.084,28.086]", "Atomic Number": "14", "Atomic Symbol": "Si", "Relative Atomic Mass": "27.97692653465(44)", "Mass Number": "28"}, "29": {"Isotopic Composition": "0.04685(8)", "Notes": " ", "Standard Atomic Weight": "[28.084,28.086]", "Atomic Number": "14", "Atomic Symbol": "Si", "Relative Atomic Mass": "28.97649466490(52)", "Mass Number": "29"}, "30": {"Isotopic Composition": "0.03092(11)", "Notes": " ", "Standard Atomic Weight": "[28.084,28.086]", "Atomic Number": "14", "Atomic Symbol": "Si", "Relative Atomic Mass": "29.973770136(23)", "Mass Number": "30"}, "31": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[28.084,28.086]", "Atomic Number": "14", "Atomic Symbol": "Si", "Relative Atomic Mass": "30.975363194(46)", "Mass Number": "31"}, "32": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[28.084,28.086]", "Atomic Number": "14", "Atomic Symbol": "Si", "Relative Atomic Mass": "31.97415154(32)", "Mass Number": "32"}, "33": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[28.084,28.086]", "Atomic Number": "14", "Atomic Symbol": "Si", "Relative Atomic Mass": "32.97797696(75)", "Mass Number": "33"}, "34": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[28.084,28.086]", "Atomic Number": "14", "Atomic Symbol": "Si", "Relative Atomic Mass": "33.978576(15)", "Mass Number": "34"}, "35": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[28.084,28.086]", "Atomic Number": "14", "Atomic Symbol": "Si", "Relative Atomic Mass": "34.984583(41)", "Mass Number": "35"}, "36": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[28.084,28.086]", "Atomic Number": "14", "Atomic Symbol": "Si", "Relative Atomic Mass": "35.986695(77)", "Mass Number": "36"}, "37": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[28.084,28.086]", "Atomic Number": "14", "Atomic Symbol": "Si", "Relative Atomic Mass": "36.992921(89)", "Mass Number": "37"}, "38": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[28.084,28.086]", "Atomic Number": "14", "Atomic Symbol": "Si", "Relative Atomic Mass": "37.995523(75)", "Mass Number": "38"}, "39": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[28.084,28.086]", "Atomic Number": "14", "Atomic Symbol": "Si", "Relative Atomic Mass": "39.002491(97)", "Mass Number": "39"}, "40": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[28.084,28.086]", "Atomic Number": "14", "Atomic Symbol": "Si", "Relative Atomic Mass": "40.00583(25)", "Mass Number": "40"}, "41": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[28.084,28.086]", "Atomic Number": "14", "Atomic Symbol": "Si", "Relative Atomic Mass": "41.01301(40)", "Mass Number": "41"}, "42": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[28.084,28.086]", "Atomic Number": "14", "Atomic Symbol": "Si", "Relative Atomic Mass": "42.01778(54#)", "Mass Number": "42"}, "43": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[28.084,28.086]", "Atomic Number": "14", "Atomic Symbol": "Si", "Relative Atomic Mass": "43.02480(64#)", "Mass Number": "43"}, "44": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[28.084,28.086]", "Atomic Number": "14", "Atomic Symbol": "Si", "Relative Atomic Mass": "44.03061(64#)", "Mass Number": "44"}, "45": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[28.084,28.086]", "Atomic Number": "14", "Atomic Symbol": "Si", "Relative Atomic Mass": "45.03995(75#)", "Mass Number": "45"}}, "83": {"192": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "191.985469(33)", "Mass Number": "192"}, "193": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "192.982960(10)", "Mass Number": "193"}, "194": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "193.982785(54#)", "Mass Number": "194"}, "195": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "194.9806488(57)", "Mass Number": "195"}, "196": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "195.980667(26)", "Mass Number": "196"}, "197": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "196.9788651(89)", "Mass Number": "197"}, "198": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "197.979206(30)", "Mass Number": "198"}, "199": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "198.977673(11)", "Mass Number": "199"}, "200": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "199.978131(24)", "Mass Number": "200"}, "201": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "200.977010(16)", "Mass Number": "201"}, "202": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "201.977734(17)", "Mass Number": "202"}, "203": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "202.976893(14)", "Mass Number": "203"}, "204": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "203.9778361(99)", "Mass Number": "204"}, "205": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "204.9773867(55)", "Mass Number": "205"}, "206": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "205.9784993(82)", "Mass Number": "206"}, "207": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "206.9784710(26)", "Mass Number": "207"}, "208": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "207.9797425(25)", "Mass Number": "208"}, "209": {"Isotopic Composition": "1", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "208.9803991(16)", "Mass Number": "209"}, "210": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "209.9841207(16)", "Mass Number": "210"}, "211": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "210.9872697(59)", "Mass Number": "211"}, "212": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "211.9912860(21)", "Mass Number": "212"}, "213": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "212.9943851(56)", "Mass Number": "213"}, "214": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "213.998712(12)", "Mass Number": "214"}, "215": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "215.001770(16)", "Mass Number": "215"}, "216": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "216.006306(12)", "Mass Number": "216"}, "217": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "217.009372(19)", "Mass Number": "217"}, "218": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "218.014188(29)", "Mass Number": "218"}, "219": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "219.01748(21#)", "Mass Number": "219"}, "220": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "220.02235(32#)", "Mass Number": "220"}, "221": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "221.02587(32#)", "Mass Number": "221"}, "222": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "222.03078(32#)", "Mass Number": "222"}, "223": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "223.03450(43#)", "Mass Number": "223"}, "224": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "224.03947(43#)", "Mass Number": "224"}, "184": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "184.001275(84)", "Mass Number": "184"}, "185": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "184.997600(87#)", "Mass Number": "185"}, "186": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "185.996644(65)", "Mass Number": "186"}, "187": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "186.993147(11)", "Mass Number": "187"}, "188": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "187.992287(22)", "Mass Number": "188"}, "189": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "188.989195(22)", "Mass Number": "189"}, "190": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "189.988622(24)", "Mass Number": "190"}, "191": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "208.98040(1)", "Atomic Number": "83", "Atomic Symbol": "Bi", "Relative Atomic Mass": "190.9857866(80)", "Mass Number": "191"}}, "116": {"289": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "116", "Atomic Symbol": "Lv", "Relative Atomic Mass": "289.19816(57#)", "Mass Number": "289"}, "290": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "116", "Atomic Symbol": "Lv", "Relative Atomic Mass": "290.19864(71#)", "Mass Number": "290"}, "291": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "116", "Atomic Symbol": "Lv", "Relative Atomic Mass": "291.20108(66#)", "Mass Number": "291"}, "292": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "116", "Atomic Symbol": "Lv", "Relative Atomic Mass": "292.20174(91#)", "Mass Number": "292"}, "293": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "116", "Atomic Symbol": "Lv", "Relative Atomic Mass": "293.20449(60#)", "Mass Number": "293"}}, "81": {"192": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "191.972225(34)", "Mass Number": "192"}, "193": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "192.9705020(72)", "Mass Number": "193"}, "194": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "193.971081(15)", "Mass Number": "194"}, "195": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "194.969774(12)", "Mass Number": "195"}, "196": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "195.970481(13)", "Mass Number": "196"}, "197": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "196.969576(18)", "Mass Number": "197"}, "198": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "197.970483(86)", "Mass Number": "198"}, "199": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "198.969877(30)", "Mass Number": "199"}, "200": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "199.9709633(62)", "Mass Number": "200"}, "201": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "200.970822(15)", "Mass Number": "201"}, "202": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "201.972102(15)", "Mass Number": "202"}, "203": {"Isotopic Composition": "0.2952(1)", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "202.9723446(14)", "Mass Number": "203"}, "204": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "203.9738639(13)", "Mass Number": "204"}, "205": {"Isotopic Composition": "0.7048(1)", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "204.9744278(14)", "Mass Number": "205"}, "206": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "205.9761106(15)", "Mass Number": "206"}, "207": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "206.9774197(59)", "Mass Number": "207"}, "208": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "207.9820190(21)", "Mass Number": "208"}, "209": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "208.9853594(86)", "Mass Number": "209"}, "210": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "209.990074(12)", "Mass Number": "210"}, "211": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "210.993475(45)", "Mass Number": "211"}, "212": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "211.99834(22#)", "Mass Number": "212"}, "213": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "213.001915(29)", "Mass Number": "213"}, "214": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "214.00694(21#)", "Mass Number": "214"}, "215": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "215.01064(32#)", "Mass Number": "215"}, "216": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "216.01580(32#)", "Mass Number": "216"}, "217": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "217.01966(43#)", "Mass Number": "217"}, "218": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "218.02479(43#)", "Mass Number": "218"}, "176": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "176.000624(81)", "Mass Number": "176"}, "177": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "176.996431(25)", "Mass Number": "177"}, "178": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "177.99485(11#)", "Mass Number": "178"}, "179": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "178.991111(43)", "Mass Number": "179"}, "180": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "179.990057(64)", "Mass Number": "180"}, "181": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "180.9862600(98)", "Mass Number": "181"}, "182": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "181.985713(63)", "Mass Number": "182"}, "183": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "182.982193(10)", "Mass Number": "183"}, "184": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "183.981886(22)", "Mass Number": "184"}, "185": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "184.978789(22)", "Mass Number": "185"}, "186": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "185.978651(24)", "Mass Number": "186"}, "187": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "186.9759063(88)", "Mass Number": "187"}, "188": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "187.976021(32)", "Mass Number": "188"}, "189": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "188.973588(12)", "Mass Number": "189"}, "190": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "189.973828(54#)", "Mass Number": "190"}, "191": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[204.382,204.385]", "Atomic Number": "81", "Atomic Symbol": "Tl", "Relative Atomic Mass": "190.9717842(79)", "Mass Number": "191"}}, "105": {"256": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "105", "Atomic Symbol": "Db", "Relative Atomic Mass": "256.10789(26#)", "Mass Number": "256"}, "257": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "105", "Atomic Symbol": "Db", "Relative Atomic Mass": "257.10758(22#)", "Mass Number": "257"}, "258": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "105", "Atomic Symbol": "Db", "Relative Atomic Mass": "258.10928(33#)", "Mass Number": "258"}, "259": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "105", "Atomic Symbol": "Db", "Relative Atomic Mass": "259.109492(57)", "Mass Number": "259"}, "260": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "105", "Atomic Symbol": "Db", "Relative Atomic Mass": "260.11130(10#)", "Mass Number": "260"}, "261": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "105", "Atomic Symbol": "Db", "Relative Atomic Mass": "261.11192(12#)", "Mass Number": "261"}, "262": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "105", "Atomic Symbol": "Db", "Relative Atomic Mass": "262.11407(15#)", "Mass Number": "262"}, "263": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "105", "Atomic Symbol": "Db", "Relative Atomic Mass": "263.11499(18#)", "Mass Number": "263"}, "264": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "105", "Atomic Symbol": "Db", "Relative Atomic Mass": "264.11741(25#)", "Mass Number": "264"}, "265": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "105", "Atomic Symbol": "Db", "Relative Atomic Mass": "265.11861(24#)", "Mass Number": "265"}, "266": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "105", "Atomic Symbol": "Db", "Relative Atomic Mass": "266.12103(30#)", "Mass Number": "266"}, "267": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "105", "Atomic Symbol": "Db", "Relative Atomic Mass": "267.12247(44#)", "Mass Number": "267"}, "268": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "105", "Atomic Symbol": "Db", "Relative Atomic Mass": "268.12567(57#)", "Mass Number": "268"}, "269": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "105", "Atomic Symbol": "Db", "Relative Atomic Mass": "269.12791(73#)", "Mass Number": "269"}, "270": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "105", "Atomic Symbol": "Db", "Relative Atomic Mass": "270.13136(64#)", "Mass Number": "270"}, "255": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "105", "Atomic Symbol": "Db", "Relative Atomic Mass": "255.10707(45#)", "Mass Number": "255"}}, "25": {"64": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "63.9538494(38)", "Mass Number": "64"}, "65": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "64.9560198(40)", "Mass Number": "65"}, "66": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "65.960547(12)", "Mass Number": "66"}, "67": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "66.96424(43#)", "Mass Number": "67"}, "68": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "67.96962(54#)", "Mass Number": "68"}, "69": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "68.97366(64#)", "Mass Number": "69"}, "70": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "69.97937(75#)", "Mass Number": "70"}, "71": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "70.98368(75#)", "Mass Number": "71"}, "44": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "44.00715(54#)", "Mass Number": "44"}, "45": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "44.99449(43#)", "Mass Number": "45"}, "46": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "45.98609(43#)", "Mass Number": "46"}, "47": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "46.975775(34)", "Mass Number": "47"}, "48": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "47.96852(18)", "Mass Number": "48"}, "49": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "48.959595(11)", "Mass Number": "49"}, "50": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "49.95423778(95)", "Mass Number": "50"}, "51": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "50.94820847(94)", "Mass Number": "51"}, "52": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "51.9455639(20)", "Mass Number": "52"}, "53": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "52.94128889(68)", "Mass Number": "53"}, "54": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "53.9403576(12)", "Mass Number": "54"}, "55": {"Isotopic Composition": "1", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "54.93804391(48)", "Mass Number": "55"}, "56": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "55.93890369(49)", "Mass Number": "56"}, "57": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "56.9382861(16)", "Mass Number": "57"}, "58": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "57.9400666(29)", "Mass Number": "58"}, "59": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "58.9403911(25)", "Mass Number": "59"}, "60": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "59.9431366(25)", "Mass Number": "60"}, "61": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "60.9444525(25)", "Mass Number": "61"}, "62": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "61.94795(16#)", "Mass Number": "62"}, "63": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "54.938044(3)", "Atomic Number": "25", "Atomic Symbol": "Mn", "Relative Atomic Mass": "62.9496647(40)", "Mass Number": "63"}}, "26": {"64": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "63.9409878(54)", "Mass Number": "64"}, "65": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "64.9450115(73)", "Mass Number": "65"}, "66": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "65.9462500(44)", "Mass Number": "66"}, "67": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "66.95054(23)", "Mass Number": "67"}, "68": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "67.95295(39)", "Mass Number": "68"}, "69": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "68.95807(43#)", "Mass Number": "69"}, "70": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "69.96102(54#)", "Mass Number": "70"}, "71": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "70.96672(64#)", "Mass Number": "71"}, "72": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "71.96983(75#)", "Mass Number": "72"}, "73": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "72.97572(75#)", "Mass Number": "73"}, "74": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "73.97935(86#)", "Mass Number": "74"}, "45": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "45.01442(43#)", "Mass Number": "45"}, "46": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "46.00063(54#)", "Mass Number": "46"}, "47": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "46.99185(54#)", "Mass Number": "47"}, "48": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "47.98023(43#)", "Mass Number": "48"}, "49": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "48.973429(26)", "Mass Number": "49"}, "50": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "49.962975(64)", "Mass Number": "50"}, "51": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "50.9568410(96)", "Mass Number": "51"}, "52": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "51.9481131(70)", "Mass Number": "52"}, "53": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "52.9453064(18)", "Mass Number": "53"}, "54": {"Isotopic Composition": "0.05845(35)", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "53.93960899(53)", "Mass Number": "54"}, "55": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "54.93829199(51)", "Mass Number": "55"}, "56": {"Isotopic Composition": "0.91754(36)", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "55.93493633(49)", "Mass Number": "56"}, "57": {"Isotopic Composition": "0.02119(10)", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "56.93539284(49)", "Mass Number": "57"}, "58": {"Isotopic Composition": "0.00282(4)", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "57.93327443(53)", "Mass Number": "58"}, "59": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "58.93487434(54)", "Mass Number": "59"}, "60": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "59.9340711(37)", "Mass Number": "60"}, "61": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "60.9367462(28)", "Mass Number": "61"}, "62": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "61.9367918(30)", "Mass Number": "62"}, "63": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "55.845(2)", "Atomic Number": "26", "Atomic Symbol": "Fe", "Relative Atomic Mass": "62.9402727(46)", "Mass Number": "63"}}, "29": {"64": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "63.92976434(56)", "Mass Number": "64"}, "65": {"Isotopic Composition": "0.3085(15)", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "64.92778970(71)", "Mass Number": "65"}, "66": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "65.92886903(72)", "Mass Number": "66"}, "67": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "66.9277303(13)", "Mass Number": "67"}, "68": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "67.9296109(17)", "Mass Number": "68"}, "69": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "68.9294293(15)", "Mass Number": "69"}, "70": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "69.9323921(12)", "Mass Number": "70"}, "71": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "70.9326768(16)", "Mass Number": "71"}, "72": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "71.9358203(15)", "Mass Number": "72"}, "73": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "72.9366744(21)", "Mass Number": "73"}, "74": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "73.9398749(66)", "Mass Number": "74"}, "75": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "74.9415226(25)", "Mass Number": "75"}, "76": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "75.9452750(72)", "Mass Number": "76"}, "77": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "76.94792(16#)", "Mass Number": "77"}, "78": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "77.95223(54)", "Mass Number": "78"}, "79": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "78.95502(43#)", "Mass Number": "79"}, "80": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "79.96089(64#)", "Mass Number": "80"}, "81": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "80.96587(86#)", "Mass Number": "81"}, "82": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "81.97244(86#)", "Mass Number": "82"}, "52": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "51.99671(86#)", "Mass Number": "52"}, "53": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "52.98459(86#)", "Mass Number": "53"}, "54": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "53.97666(54#)", "Mass Number": "54"}, "55": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "54.96604(17)", "Mass Number": "55"}, "56": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "55.95895(21#)", "Mass Number": "56"}, "57": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "56.94921250(66)", "Mass Number": "57"}, "58": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "57.94453305(70)", "Mass Number": "58"}, "59": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "58.93949748(67)", "Mass Number": "59"}, "60": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "59.9373645(18)", "Mass Number": "60"}, "61": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "60.9334576(10)", "Mass Number": "61"}, "62": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "61.93259541(75)", "Mass Number": "62"}, "63": {"Isotopic Composition": "0.6915(15)", "Notes": "r", "Standard Atomic Weight": "63.546(3)", "Atomic Number": "29", "Atomic Symbol": "Cu", "Relative Atomic Mass": "62.92959772(56)", "Mass Number": "63"}}, "1": {"1": {"Isotopic Composition": "0.999885(70)", "Notes": "m", "Standard Atomic Weight": "[1.00784,1.00811]", "Atomic Number": "1", "Atomic Symbol": "H", "Relative Atomic Mass": "1.00782503223(9)", "Mass Number": "1"}, "2": {"Isotopic Composition": "0.000115(70)", "Notes": "m", "Standard Atomic Weight": "[1.00784,1.00811]", "Atomic Number": "1", "Atomic Symbol": "D", "Relative Atomic Mass": "2.01410177812(12)", "Mass Number": "2"}, "3": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[1.00784,1.00811]", "Atomic Number": "1", "Atomic Symbol": "T", "Relative Atomic Mass": "3.0160492779(24)", "Mass Number": "3"}, "4": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[1.00784,1.00811]", "Atomic Number": "1", "Atomic Symbol": "H", "Relative Atomic Mass": "4.02643(11)", "Mass Number": "4"}, "5": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[1.00784,1.00811]", "Atomic Number": "1", "Atomic Symbol": "H", "Relative Atomic Mass": "5.035311(96)", "Mass Number": "5"}, "6": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[1.00784,1.00811]", "Atomic Number": "1", "Atomic Symbol": "H", "Relative Atomic Mass": "6.04496(27)", "Mass Number": "6"}, "7": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[1.00784,1.00811]", "Atomic Number": "1", "Atomic Symbol": "H", "Relative Atomic Mass": "7.0527(11#)", "Mass Number": "7"}}, "5": {"6": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[10.806,10.821]", "Atomic Number": "5", "Atomic Symbol": "B", "Relative Atomic Mass": "6.0508(22#)", "Mass Number": "6"}, "7": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[10.806,10.821]", "Atomic Number": "5", "Atomic Symbol": "B", "Relative Atomic Mass": "7.029712(27)", "Mass Number": "7"}, "8": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[10.806,10.821]", "Atomic Number": "5", "Atomic Symbol": "B", "Relative Atomic Mass": "8.0246073(11)", "Mass Number": "8"}, "9": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[10.806,10.821]", "Atomic Number": "5", "Atomic Symbol": "B", "Relative Atomic Mass": "9.01332965(97)", "Mass Number": "9"}, "10": {"Isotopic Composition": "0.199(7)", "Notes": "m", "Standard Atomic Weight": "[10.806,10.821]", "Atomic Number": "5", "Atomic Symbol": "B", "Relative Atomic Mass": "10.01293695(41)", "Mass Number": "10"}, "11": {"Isotopic Composition": "0.801(7)", "Notes": "m", "Standard Atomic Weight": "[10.806,10.821]", "Atomic Number": "5", "Atomic Symbol": "B", "Relative Atomic Mass": "11.00930536(45)", "Mass Number": "11"}, "12": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[10.806,10.821]", "Atomic Number": "5", "Atomic Symbol": "B", "Relative Atomic Mass": "12.0143527(14)", "Mass Number": "12"}, "13": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[10.806,10.821]", "Atomic Number": "5", "Atomic Symbol": "B", "Relative Atomic Mass": "13.0177802(12)", "Mass Number": "13"}, "14": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[10.806,10.821]", "Atomic Number": "5", "Atomic Symbol": "B", "Relative Atomic Mass": "14.025404(23)", "Mass Number": "14"}, "15": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[10.806,10.821]", "Atomic Number": "5", "Atomic Symbol": "B", "Relative Atomic Mass": "15.031088(23)", "Mass Number": "15"}, "16": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[10.806,10.821]", "Atomic Number": "5", "Atomic Symbol": "B", "Relative Atomic Mass": "16.039842(26)", "Mass Number": "16"}, "17": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[10.806,10.821]", "Atomic Number": "5", "Atomic Symbol": "B", "Relative Atomic Mass": "17.04699(18)", "Mass Number": "17"}, "18": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[10.806,10.821]", "Atomic Number": "5", "Atomic Symbol": "B", "Relative Atomic Mass": "18.05566(18)", "Mass Number": "18"}, "19": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[10.806,10.821]", "Atomic Number": "5", "Atomic Symbol": "B", "Relative Atomic Mass": "19.06310(43#)", "Mass Number": "19"}, "20": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[10.806,10.821]", "Atomic Number": "5", "Atomic Symbol": "B", "Relative Atomic Mass": "20.07207(75#)", "Mass Number": "20"}, "21": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[10.806,10.821]", "Atomic Number": "5", "Atomic Symbol": "B", "Relative Atomic Mass": "21.08129(97#)", "Mass Number": "21"}}, "32": {"64": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "63.9416899(40)", "Mass Number": "64"}, "65": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "64.9393681(23)", "Mass Number": "65"}, "66": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "65.9338621(26)", "Mass Number": "66"}, "67": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "66.9327339(50)", "Mass Number": "67"}, "68": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "67.9280953(20)", "Mass Number": "68"}, "69": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "68.9279645(14)", "Mass Number": "69"}, "70": {"Isotopic Composition": "0.2057(27)", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "69.92424875(90)", "Mass Number": "70"}, "71": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "70.92495233(90)", "Mass Number": "71"}, "72": {"Isotopic Composition": "0.2745(32)", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "71.922075826(81)", "Mass Number": "72"}, "73": {"Isotopic Composition": "0.0775(12)", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "72.923458956(61)", "Mass Number": "73"}, "74": {"Isotopic Composition": "0.3650(20)", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "73.921177761(13)", "Mass Number": "74"}, "75": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "74.922858370(55)", "Mass Number": "75"}, "76": {"Isotopic Composition": "0.0773(12)", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "75.921402726(19)", "Mass Number": "76"}, "77": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "76.923549843(57)", "Mass Number": "77"}, "78": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "77.9228529(38)", "Mass Number": "78"}, "79": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "78.925360(40)", "Mass Number": "79"}, "80": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "79.9253508(22)", "Mass Number": "80"}, "81": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "80.9288329(22)", "Mass Number": "81"}, "82": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "81.9297740(24)", "Mass Number": "82"}, "83": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "82.9345391(26)", "Mass Number": "83"}, "84": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "83.9375751(34)", "Mass Number": "84"}, "85": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "84.9429697(40)", "Mass Number": "85"}, "86": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "85.94658(32#)", "Mass Number": "86"}, "87": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "86.95268(43#)", "Mass Number": "87"}, "88": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "87.95691(54#)", "Mass Number": "88"}, "89": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "88.96379(64#)", "Mass Number": "89"}, "90": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "89.96863(75#)", "Mass Number": "90"}, "58": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "57.99172(43#)", "Mass Number": "58"}, "59": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "58.98249(32#)", "Mass Number": "59"}, "60": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "59.97036(21#)", "Mass Number": "60"}, "61": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "60.96379(32#)", "Mass Number": "61"}, "62": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "61.95502(15#)", "Mass Number": "62"}, "63": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "72.630(8)", "Atomic Number": "32", "Atomic Symbol": "Ge", "Relative Atomic Mass": "62.949628(40)", "Mass Number": "63"}}, "108": {"263": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "108", "Atomic Symbol": "Hs", "Relative Atomic Mass": "263.12852(14#)", "Mass Number": "263"}, "264": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "108", "Atomic Symbol": "Hs", "Relative Atomic Mass": "264.128357(31)", "Mass Number": "264"}, "265": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "108", "Atomic Symbol": "Hs", "Relative Atomic Mass": "265.129793(26)", "Mass Number": "265"}, "266": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "108", "Atomic Symbol": "Hs", "Relative Atomic Mass": "266.130046(42)", "Mass Number": "266"}, "267": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "108", "Atomic Symbol": "Hs", "Relative Atomic Mass": "267.13167(10#)", "Mass Number": "267"}, "268": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "108", "Atomic Symbol": "Hs", "Relative Atomic Mass": "268.13186(30#)", "Mass Number": "268"}, "269": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "108", "Atomic Symbol": "Hs", "Relative Atomic Mass": "269.13375(13#)", "Mass Number": "269"}, "270": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "108", "Atomic Symbol": "Hs", "Relative Atomic Mass": "270.13429(27#)", "Mass Number": "270"}, "271": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "108", "Atomic Symbol": "Hs", "Relative Atomic Mass": "271.13717(32#)", "Mass Number": "271"}, "272": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "108", "Atomic Symbol": "Hs", "Relative Atomic Mass": "272.13850(55#)", "Mass Number": "272"}, "273": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "108", "Atomic Symbol": "Hs", "Relative Atomic Mass": "273.14168(40#)", "Mass Number": "273"}, "274": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "108", "Atomic Symbol": "Hs", "Relative Atomic Mass": "274.14330(63#)", "Mass Number": "274"}, "275": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "108", "Atomic Symbol": "Hs", "Relative Atomic Mass": "275.14667(63#)", "Mass Number": "275"}, "276": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "108", "Atomic Symbol": "Hs", "Relative Atomic Mass": "276.14846(86#)", "Mass Number": "276"}, "277": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "108", "Atomic Symbol": "Hs", "Relative Atomic Mass": "277.15190(58#)", "Mass Number": "277"}}, "66": {"138": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "137.96250(43#)", "Mass Number": "138"}, "139": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "138.95959(54#)", "Mass Number": "139"}, "140": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "139.95402(54#)", "Mass Number": "140"}, "141": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "140.95128(32#)", "Mass Number": "141"}, "142": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "141.94619(78#)", "Mass Number": "142"}, "143": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "142.943994(14)", "Mass Number": "143"}, "144": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "143.9392695(77)", "Mass Number": "144"}, "145": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "144.9374740(70)", "Mass Number": "145"}, "146": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "145.9328445(72)", "Mass Number": "146"}, "147": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "146.9310827(95)", "Mass Number": "147"}, "148": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "147.927157(10)", "Mass Number": "148"}, "149": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "148.927322(10)", "Mass Number": "149"}, "150": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "149.9255933(48)", "Mass Number": "150"}, "151": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "150.9261916(38)", "Mass Number": "151"}, "152": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "151.9247253(51)", "Mass Number": "152"}, "153": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "152.9257724(45)", "Mass Number": "153"}, "154": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "153.9244293(80)", "Mass Number": "154"}, "155": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "154.925759(10)", "Mass Number": "155"}, "156": {"Isotopic Composition": "0.00056(3)", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "155.9242847(17)", "Mass Number": "156"}, "157": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "156.9254707(57)", "Mass Number": "157"}, "158": {"Isotopic Composition": "0.00095(3)", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "157.9244159(31)", "Mass Number": "158"}, "159": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "158.9257470(22)", "Mass Number": "159"}, "160": {"Isotopic Composition": "0.02329(18)", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "159.9252046(20)", "Mass Number": "160"}, "161": {"Isotopic Composition": "0.18889(42)", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "160.9269405(20)", "Mass Number": "161"}, "162": {"Isotopic Composition": "0.25475(36)", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "161.9268056(20)", "Mass Number": "162"}, "163": {"Isotopic Composition": "0.24896(42)", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "162.9287383(20)", "Mass Number": "163"}, "164": {"Isotopic Composition": "0.28260(54)", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "163.9291819(20)", "Mass Number": "164"}, "165": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "164.9317105(20)", "Mass Number": "165"}, "166": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "165.9328139(21)", "Mass Number": "166"}, "167": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "166.935661(65)", "Mass Number": "167"}, "168": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "167.93713(15)", "Mass Number": "168"}, "169": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "168.94031(32)", "Mass Number": "169"}, "170": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "169.94239(21#)", "Mass Number": "170"}, "171": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "170.94612(32#)", "Mass Number": "171"}, "172": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "171.94846(32#)", "Mass Number": "172"}, "173": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "162.500(1)", "Atomic Number": "66", "Atomic Symbol": "Dy", "Relative Atomic Mass": "172.95283(43#)", "Mass Number": "173"}}, "112": {"276": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "112", "Atomic Symbol": "Cn", "Relative Atomic Mass": "276.16141(64#)", "Mass Number": "276"}, "277": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "112", "Atomic Symbol": "Cn", "Relative Atomic Mass": "277.16364(15#)", "Mass Number": "277"}, "278": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "112", "Atomic Symbol": "Cn", "Relative Atomic Mass": "278.16416(47#)", "Mass Number": "278"}, "279": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "112", "Atomic Symbol": "Cn", "Relative Atomic Mass": "279.16654(50#)", "Mass Number": "279"}, "280": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "112", "Atomic Symbol": "Cn", "Relative Atomic Mass": "280.16715(63#)", "Mass Number": "280"}, "281": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "112", "Atomic Symbol": "Cn", "Relative Atomic Mass": "281.16975(42#)", "Mass Number": "281"}, "282": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "112", "Atomic Symbol": "Cn", "Relative Atomic Mass": "282.17050(70#)", "Mass Number": "282"}, "283": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "112", "Atomic Symbol": "Cn", "Relative Atomic Mass": "283.17327(65#)", "Mass Number": "283"}, "284": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "112", "Atomic Symbol": "Cn", "Relative Atomic Mass": "284.17416(91#)", "Mass Number": "284"}, "285": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "112", "Atomic Symbol": "Cn", "Relative Atomic Mass": "285.17712(60#)", "Mass Number": "285"}}, "118": {"293": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "118", "Atomic Symbol": "Og", "Relative Atomic Mass": "293.21356(78#)", "Mass Number": "293"}, "294": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "118", "Atomic Symbol": "Og", "Relative Atomic Mass": "294.21392(71#)", "Mass Number": "294"}, "295": {"Isotopic Composition": "", "Standard Atomic Weight": " ", "Atomic Number": "118", "Atomic Symbol": "Og", "Relative Atomic Mass": "295.21624(69#)", "Mass Number": "295"}}, "31": {"64": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "63.9368404(15)", "Mass Number": "64"}, "65": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "64.93273459(88)", "Mass Number": "65"}, "66": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "65.9315894(34)", "Mass Number": "66"}, "67": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "66.9282025(13)", "Mass Number": "67"}, "68": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "67.9279805(16)", "Mass Number": "68"}, "69": {"Isotopic Composition": "0.60108(9)", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "68.9255735(13)", "Mass Number": "69"}, "70": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "69.9260219(13)", "Mass Number": "70"}, "71": {"Isotopic Composition": "0.39892(9)", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "70.92470258(87)", "Mass Number": "71"}, "72": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "71.92636747(88)", "Mass Number": "72"}, "73": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "72.9251747(18)", "Mass Number": "73"}, "74": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "73.9269457(32)", "Mass Number": "74"}, "75": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "74.9265002(26)", "Mass Number": "75"}, "76": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "75.9288276(21)", "Mass Number": "76"}, "77": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "76.9291543(26)", "Mass Number": "77"}, "78": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "77.9316088(20)", "Mass Number": "78"}, "79": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "78.9328523(20)", "Mass Number": "79"}, "80": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "79.9364208(31)", "Mass Number": "80"}, "81": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "80.9381338(35)", "Mass Number": "81"}, "82": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "81.9431765(26)", "Mass Number": "82"}, "83": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "82.9471203(28)", "Mass Number": "83"}, "84": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "83.95246(43#)", "Mass Number": "84"}, "85": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "84.95699(32#)", "Mass Number": "85"}, "86": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "85.96301(75#)", "Mass Number": "86"}, "87": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "86.96824(86#)", "Mass Number": "87"}, "56": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "55.99536(64#)", "Mass Number": "56"}, "57": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "56.98320(32#)", "Mass Number": "57"}, "58": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "57.97478(21#)", "Mass Number": "58"}, "59": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "58.96353(18#)", "Mass Number": "59"}, "60": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "59.95729(21#)", "Mass Number": "60"}, "61": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "60.949399(41)", "Mass Number": "61"}, "62": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "61.94419025(75)", "Mass Number": "62"}, "63": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "69.723(1)", "Atomic Number": "31", "Atomic Symbol": "Ga", "Relative Atomic Mass": "62.9392942(14)", "Mass Number": "63"}}, "72": {"153": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "152.97069(54#)", "Mass Number": "153"}, "154": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "153.96486(54#)", "Mass Number": "154"}, "155": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "154.96311(32#)", "Mass Number": "155"}, "156": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "155.95935(17)", "Mass Number": "156"}, "157": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "156.95824(21#)", "Mass Number": "157"}, "158": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "157.954801(19)", "Mass Number": "158"}, "159": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "158.953996(18)", "Mass Number": "159"}, "160": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "159.950691(11)", "Mass Number": "160"}, "161": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "160.950278(24)", "Mass Number": "161"}, "162": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "161.9472148(97)", "Mass Number": "162"}, "163": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "162.947113(27)", "Mass Number": "163"}, "164": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "163.944371(17)", "Mass Number": "164"}, "165": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "164.944567(30)", "Mass Number": "165"}, "166": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "165.942180(30)", "Mass Number": "166"}, "167": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "166.942600(30)", "Mass Number": "167"}, "168": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "167.940568(30)", "Mass Number": "168"}, "169": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "168.941259(30)", "Mass Number": "169"}, "170": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "169.939609(30)", "Mass Number": "170"}, "171": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "170.940492(31)", "Mass Number": "171"}, "172": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "171.939450(26)", "Mass Number": "172"}, "173": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "172.940513(30)", "Mass Number": "173"}, "174": {"Isotopic Composition": "0.0016(1)", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "173.9400461(28)", "Mass Number": "174"}, "175": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "174.9415092(29)", "Mass Number": "175"}, "176": {"Isotopic Composition": "0.0526(7)", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "175.9414076(22)", "Mass Number": "176"}, "177": {"Isotopic Composition": "0.1860(9)", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "176.9432277(20)", "Mass Number": "177"}, "178": {"Isotopic Composition": "0.2728(7)", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "177.9437058(20)", "Mass Number": "178"}, "179": {"Isotopic Composition": "0.1362(2)", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "178.9458232(20)", "Mass Number": "179"}, "180": {"Isotopic Composition": "0.3508(16)", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "179.9465570(20)", "Mass Number": "180"}, "181": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "180.9491083(20)", "Mass Number": "181"}, "182": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "181.9505612(68)", "Mass Number": "182"}, "183": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "182.953530(32)", "Mass Number": "183"}, "184": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "183.955446(43)", "Mass Number": "184"}, "185": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "184.958862(98)", "Mass Number": "185"}, "186": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "185.960897(59)", "Mass Number": "186"}, "187": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "186.96477(32#)", "Mass Number": "187"}, "188": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "187.96685(32#)", "Mass Number": "188"}, "189": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "178.49(2)", "Atomic Number": "72", "Atomic Symbol": "Hf", "Relative Atomic Mass": "188.97084(32#)", "Mass Number": "189"}}, "43": {"85": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "84.95058(43#)", "Mass Number": "85"}, "86": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "85.94493(32#)", "Mass Number": "86"}, "87": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "86.9380672(45)", "Mass Number": "87"}, "88": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "87.93378(16)", "Mass Number": "88"}, "89": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "88.9276487(41)", "Mass Number": "89"}, "90": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "89.9240739(11)", "Mass Number": "90"}, "91": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "90.9184254(25)", "Mass Number": "91"}, "92": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "91.9152698(33)", "Mass Number": "92"}, "93": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "92.9102460(14)", "Mass Number": "93"}, "94": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "93.9096536(44)", "Mass Number": "94"}, "95": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "94.9076536(55)", "Mass Number": "95"}, "96": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "95.9078680(55)", "Mass Number": "96"}, "97": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "96.9063667(40)", "Mass Number": "97"}, "98": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "97.9072124(36)", "Mass Number": "98"}, "99": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "98.9062508(10)", "Mass Number": "99"}, "100": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "99.9076539(15)", "Mass Number": "100"}, "101": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "100.907309(26)", "Mass Number": "101"}, "102": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "101.9092097(99)", "Mass Number": "102"}, "103": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "102.909176(11)", "Mass Number": "103"}, "104": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "103.911425(27)", "Mass Number": "104"}, "105": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "104.911655(38)", "Mass Number": "105"}, "106": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "105.914358(13)", "Mass Number": "106"}, "107": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "106.9154606(93)", "Mass Number": "107"}, "108": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "107.9184957(94)", "Mass Number": "108"}, "109": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "108.920256(10)", "Mass Number": "109"}, "110": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "109.923744(10)", "Mass Number": "110"}, "111": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "110.925901(11)", "Mass Number": "111"}, "112": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "111.9299458(60)", "Mass Number": "112"}, "113": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "112.9325690(36)", "Mass Number": "113"}, "114": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "113.93691(11#)", "Mass Number": "114"}, "115": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "114.93998(21#)", "Mass Number": "115"}, "116": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "115.94476(32#)", "Mass Number": "116"}, "117": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "116.94806(43#)", "Mass Number": "117"}, "118": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "117.95299(43#)", "Mass Number": "118"}, "119": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "118.95666(54#)", "Mass Number": "119"}, "120": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[98]", "Atomic Number": "43", "Atomic Symbol": "Tc", "Relative Atomic Mass": "119.96187(54#)", "Mass Number": "120"}}, "76": {"192": {"Isotopic Composition": "0.4078(19)", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "191.9614770(29)", "Mass Number": "192"}, "193": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "192.9641479(29)", "Mass Number": "193"}, "194": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "193.9651772(30)", "Mass Number": "194"}, "195": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "194.968318(65)", "Mass Number": "195"}, "196": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "195.969641(43)", "Mass Number": "196"}, "197": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "196.97283(21#)", "Mass Number": "197"}, "198": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "197.97441(21#)", "Mass Number": "198"}, "199": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "198.97801(21#)", "Mass Number": "199"}, "200": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "199.97984(32#)", "Mass Number": "200"}, "201": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "200.98364(32#)", "Mass Number": "201"}, "202": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "201.98595(43#)", "Mass Number": "202"}, "161": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "160.98903(43#)", "Mass Number": "161"}, "162": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "161.98443(54#)", "Mass Number": "162"}, "163": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "162.98241(32#)", "Mass Number": "163"}, "164": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "163.97802(17)", "Mass Number": "164"}, "165": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "164.97660(22#)", "Mass Number": "165"}, "166": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "165.972692(20)", "Mass Number": "166"}, "167": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "166.971549(78)", "Mass Number": "167"}, "168": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "167.967808(12)", "Mass Number": "168"}, "169": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "168.967018(27)", "Mass Number": "169"}, "170": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "169.963578(11)", "Mass Number": "170"}, "171": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "170.963174(19)", "Mass Number": "171"}, "172": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "171.960017(14)", "Mass Number": "172"}, "173": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "172.959808(16)", "Mass Number": "173"}, "174": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "173.957064(11)", "Mass Number": "174"}, "175": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "174.956945(13)", "Mass Number": "175"}, "176": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "175.954806(30)", "Mass Number": "176"}, "177": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "176.954966(17)", "Mass Number": "177"}, "178": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "177.953254(15)", "Mass Number": "178"}, "179": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "178.953817(18)", "Mass Number": "179"}, "180": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "179.952375(17)", "Mass Number": "180"}, "181": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "180.953247(27)", "Mass Number": "181"}, "182": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "181.952110(23)", "Mass Number": "182"}, "183": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "182.953125(53)", "Mass Number": "183"}, "184": {"Isotopic Composition": "0.0002(1)", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "183.9524885(14)", "Mass Number": "184"}, "185": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "184.9540417(14)", "Mass Number": "185"}, "186": {"Isotopic Composition": "0.0159(3)", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "185.9538350(16)", "Mass Number": "186"}, "187": {"Isotopic Composition": "0.0196(2)", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "186.9557474(16)", "Mass Number": "187"}, "188": {"Isotopic Composition": "0.1324(8)", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "187.9558352(16)", "Mass Number": "188"}, "189": {"Isotopic Composition": "0.1615(5)", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "188.9581442(17)", "Mass Number": "189"}, "190": {"Isotopic Composition": "0.2626(2)", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "189.9584437(17)", "Mass Number": "190"}, "191": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "190.23(3)", "Atomic Number": "76", "Atomic Symbol": "Os", "Relative Atomic Mass": "190.9609264(17)", "Mass Number": "191"}}, "12": {"19": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[24.304,24.307]", "Atomic Number": "12", "Atomic Symbol": "Mg", "Relative Atomic Mass": "19.034169(54)", "Mass Number": "19"}, "20": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[24.304,24.307]", "Atomic Number": "12", "Atomic Symbol": "Mg", "Relative Atomic Mass": "20.018850(29)", "Mass Number": "20"}, "21": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[24.304,24.307]", "Atomic Number": "12", "Atomic Symbol": "Mg", "Relative Atomic Mass": "21.011716(18)", "Mass Number": "21"}, "22": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[24.304,24.307]", "Atomic Number": "12", "Atomic Symbol": "Mg", "Relative Atomic Mass": "21.99957065(34)", "Mass Number": "22"}, "23": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[24.304,24.307]", "Atomic Number": "12", "Atomic Symbol": "Mg", "Relative Atomic Mass": "22.99412421(74)", "Mass Number": "23"}, "24": {"Isotopic Composition": "0.7899(4)", "Notes": " ", "Standard Atomic Weight": "[24.304,24.307]", "Atomic Number": "12", "Atomic Symbol": "Mg", "Relative Atomic Mass": "23.985041697(14)", "Mass Number": "24"}, "25": {"Isotopic Composition": "0.1000(1)", "Notes": " ", "Standard Atomic Weight": "[24.304,24.307]", "Atomic Number": "12", "Atomic Symbol": "Mg", "Relative Atomic Mass": "24.985836976(50)", "Mass Number": "25"}, "26": {"Isotopic Composition": "0.1101(3)", "Notes": " ", "Standard Atomic Weight": "[24.304,24.307]", "Atomic Number": "12", "Atomic Symbol": "Mg", "Relative Atomic Mass": "25.982592968(31)", "Mass Number": "26"}, "27": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[24.304,24.307]", "Atomic Number": "12", "Atomic Symbol": "Mg", "Relative Atomic Mass": "26.984340624(53)", "Mass Number": "27"}, "28": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[24.304,24.307]", "Atomic Number": "12", "Atomic Symbol": "Mg", "Relative Atomic Mass": "27.9838767(22)", "Mass Number": "28"}, "29": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[24.304,24.307]", "Atomic Number": "12", "Atomic Symbol": "Mg", "Relative Atomic Mass": "28.988617(12)", "Mass Number": "29"}, "30": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[24.304,24.307]", "Atomic Number": "12", "Atomic Symbol": "Mg", "Relative Atomic Mass": "29.9904629(37)", "Mass Number": "30"}, "31": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[24.304,24.307]", "Atomic Number": "12", "Atomic Symbol": "Mg", "Relative Atomic Mass": "30.9966480(33)", "Mass Number": "31"}, "32": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[24.304,24.307]", "Atomic Number": "12", "Atomic Symbol": "Mg", "Relative Atomic Mass": "31.9991102(34)", "Mass Number": "32"}, "33": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[24.304,24.307]", "Atomic Number": "12", "Atomic Symbol": "Mg", "Relative Atomic Mass": "33.0053271(31)", "Mass Number": "33"}, "34": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[24.304,24.307]", "Atomic Number": "12", "Atomic Symbol": "Mg", "Relative Atomic Mass": "34.008935(31)", "Mass Number": "34"}, "35": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[24.304,24.307]", "Atomic Number": "12", "Atomic Symbol": "Mg", "Relative Atomic Mass": "35.01679(19)", "Mass Number": "35"}, "36": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[24.304,24.307]", "Atomic Number": "12", "Atomic Symbol": "Mg", "Relative Atomic Mass": "36.02188(49)", "Mass Number": "36"}, "37": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[24.304,24.307]", "Atomic Number": "12", "Atomic Symbol": "Mg", "Relative Atomic Mass": "37.03037(54#)", "Mass Number": "37"}, "38": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[24.304,24.307]", "Atomic Number": "12", "Atomic Symbol": "Mg", "Relative Atomic Mass": "38.03658(54#)", "Mass Number": "38"}, "39": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[24.304,24.307]", "Atomic Number": "12", "Atomic Symbol": "Mg", "Relative Atomic Mass": "39.04538(55#)", "Mass Number": "39"}, "40": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[24.304,24.307]", "Atomic Number": "12", "Atomic Symbol": "Mg", "Relative Atomic Mass": "40.05218(64#)", "Mass Number": "40"}}, "69": {"144": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "143.97628(43#)", "Mass Number": "144"}, "145": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "144.97039(21#)", "Mass Number": "145"}, "146": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "145.96684(21#)", "Mass Number": "146"}, "147": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "146.9613799(73)", "Mass Number": "147"}, "148": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "147.958384(11)", "Mass Number": "148"}, "149": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "148.95289(32#)", "Mass Number": "149"}, "150": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "149.95009(21#)", "Mass Number": "150"}, "151": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "150.945488(21)", "Mass Number": "151"}, "152": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "151.944422(79)", "Mass Number": "152"}, "153": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "152.942040(16)", "Mass Number": "153"}, "154": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "153.941570(15)", "Mass Number": "154"}, "155": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "154.939210(11)", "Mass Number": "155"}, "156": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "155.938992(16)", "Mass Number": "156"}, "157": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "156.936944(28)", "Mass Number": "157"}, "158": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "157.936980(27)", "Mass Number": "158"}, "159": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "158.934975(30)", "Mass Number": "159"}, "160": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "159.935263(37)", "Mass Number": "160"}, "161": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "160.933549(30)", "Mass Number": "161"}, "162": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "161.934002(28)", "Mass Number": "162"}, "163": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "162.9326592(62)", "Mass Number": "163"}, "164": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "163.933544(26)", "Mass Number": "164"}, "165": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "164.9324431(26)", "Mass Number": "165"}, "166": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "165.933561(13)", "Mass Number": "166"}, "167": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "166.9328562(25)", "Mass Number": "167"}, "168": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "167.9341774(27)", "Mass Number": "168"}, "169": {"Isotopic Composition": "1", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "168.9342179(22)", "Mass Number": "169"}, "170": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "169.9358060(22)", "Mass Number": "170"}, "171": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "170.9364339(24)", "Mass Number": "171"}, "172": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "171.9384055(62)", "Mass Number": "172"}, "173": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "172.9396084(53)", "Mass Number": "173"}, "174": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "173.942173(48)", "Mass Number": "174"}, "175": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "174.943841(54)", "Mass Number": "175"}, "176": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "175.94700(11)", "Mass Number": "176"}, "177": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "176.94904(32#)", "Mass Number": "177"}, "178": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "177.95264(43#)", "Mass Number": "178"}, "179": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "168.93422(2)", "Atomic Number": "69", "Atomic Symbol": "Tm", "Relative Atomic Mass": "178.95534(54#)", "Mass Number": "179"}}, "4": {"5": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "9.0121831(5)", "Atomic Number": "4", "Atomic Symbol": "Be", "Relative Atomic Mass": "5.0399(22#)", "Mass Number": "5"}, "6": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "9.0121831(5)", "Atomic Number": "4", "Atomic Symbol": "Be", "Relative Atomic Mass": "6.0197264(58)", "Mass Number": "6"}, "7": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "9.0121831(5)", "Atomic Number": "4", "Atomic Symbol": "Be", "Relative Atomic Mass": "7.016928717(76)", "Mass Number": "7"}, "8": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "9.0121831(5)", "Atomic Number": "4", "Atomic Symbol": "Be", "Relative Atomic Mass": "8.005305102(37)", "Mass Number": "8"}, "9": {"Isotopic Composition": "1", "Notes": " ", "Standard Atomic Weight": "9.0121831(5)", "Atomic Number": "4", "Atomic Symbol": "Be", "Relative Atomic Mass": "9.012183065(82)", "Mass Number": "9"}, "10": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "9.0121831(5)", "Atomic Number": "4", "Atomic Symbol": "Be", "Relative Atomic Mass": "10.013534695(86)", "Mass Number": "10"}, "11": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "9.0121831(5)", "Atomic Number": "4", "Atomic Symbol": "Be", "Relative Atomic Mass": "11.02166108(26)", "Mass Number": "11"}, "12": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "9.0121831(5)", "Atomic Number": "4", "Atomic Symbol": "Be", "Relative Atomic Mass": "12.0269221(20)", "Mass Number": "12"}, "13": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "9.0121831(5)", "Atomic Number": "4", "Atomic Symbol": "Be", "Relative Atomic Mass": "13.036135(11)", "Mass Number": "13"}, "14": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "9.0121831(5)", "Atomic Number": "4", "Atomic Symbol": "Be", "Relative Atomic Mass": "14.04289(14)", "Mass Number": "14"}, "15": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "9.0121831(5)", "Atomic Number": "4", "Atomic Symbol": "Be", "Relative Atomic Mass": "15.05342(43#)", "Mass Number": "15"}, "16": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "9.0121831(5)", "Atomic Number": "4", "Atomic Symbol": "Be", "Relative Atomic Mass": "16.06167(18)", "Mass Number": "16"}}, "102": {"256": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "102", "Atomic Symbol": "No", "Relative Atomic Mass": "256.0942829(84)", "Mass Number": "256"}, "257": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "102", "Atomic Symbol": "No", "Relative Atomic Mass": "257.0968878(74)", "Mass Number": "257"}, "258": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "102", "Atomic Symbol": "No", "Relative Atomic Mass": "258.09821(11#)", "Mass Number": "258"}, "259": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "102", "Atomic Symbol": "No", "Relative Atomic Mass": "259.10103(11#)", "Mass Number": "259"}, "260": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "102", "Atomic Symbol": "No", "Relative Atomic Mass": "260.10264(22#)", "Mass Number": "260"}, "261": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "102", "Atomic Symbol": "No", "Relative Atomic Mass": "261.10570(22#)", "Mass Number": "261"}, "262": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "102", "Atomic Symbol": "No", "Relative Atomic Mass": "262.10746(39#)", "Mass Number": "262"}, "263": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "102", "Atomic Symbol": "No", "Relative Atomic Mass": "263.11071(53#)", "Mass Number": "263"}, "264": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "102", "Atomic Symbol": "No", "Relative Atomic Mass": "264.11273(70#)", "Mass Number": "264"}, "248": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "102", "Atomic Symbol": "No", "Relative Atomic Mass": "248.08655(24#)", "Mass Number": "248"}, "249": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "102", "Atomic Symbol": "No", "Relative Atomic Mass": "249.08780(30#)", "Mass Number": "249"}, "250": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "102", "Atomic Symbol": "No", "Relative Atomic Mass": "250.08756(22#)", "Mass Number": "250"}, "251": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "102", "Atomic Symbol": "No", "Relative Atomic Mass": "251.08894(12#)", "Mass Number": "251"}, "252": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "102", "Atomic Symbol": "No", "Relative Atomic Mass": "252.088967(10)", "Mass Number": "252"}, "253": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "102", "Atomic Symbol": "No", "Relative Atomic Mass": "253.0905641(75)", "Mass Number": "253"}, "254": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "102", "Atomic Symbol": "No", "Relative Atomic Mass": "254.090956(11)", "Mass Number": "254"}, "255": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "102", "Atomic Symbol": "No", "Relative Atomic Mass": "255.093191(16)", "Mass Number": "255"}}, "99": {"256": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "99", "Atomic Symbol": "Es", "Relative Atomic Mass": "256.09360(11#)", "Mass Number": "256"}, "257": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "99", "Atomic Symbol": "Es", "Relative Atomic Mass": "257.09598(44#)", "Mass Number": "257"}, "258": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "99", "Atomic Symbol": "Es", "Relative Atomic Mass": "258.09952(32#)", "Mass Number": "258"}, "239": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "99", "Atomic Symbol": "Es", "Relative Atomic Mass": "239.06823(32#)", "Mass Number": "239"}, "240": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "99", "Atomic Symbol": "Es", "Relative Atomic Mass": "240.06892(43#)", "Mass Number": "240"}, "241": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "99", "Atomic Symbol": "Es", "Relative Atomic Mass": "241.06856(24#)", "Mass Number": "241"}, "242": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "99", "Atomic Symbol": "Es", "Relative Atomic Mass": "242.06957(28#)", "Mass Number": "242"}, "243": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "99", "Atomic Symbol": "Es", "Relative Atomic Mass": "243.06951(22#)", "Mass Number": "243"}, "244": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "99", "Atomic Symbol": "Es", "Relative Atomic Mass": "244.07088(20#)", "Mass Number": "244"}, "245": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "99", "Atomic Symbol": "Es", "Relative Atomic Mass": "245.07125(22#)", "Mass Number": "245"}, "246": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "99", "Atomic Symbol": "Es", "Relative Atomic Mass": "246.07290(24#)", "Mass Number": "246"}, "247": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "99", "Atomic Symbol": "Es", "Relative Atomic Mass": "247.073622(21)", "Mass Number": "247"}, "248": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "99", "Atomic Symbol": "Es", "Relative Atomic Mass": "248.075471(56#)", "Mass Number": "248"}, "249": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "99", "Atomic Symbol": "Es", "Relative Atomic Mass": "249.076411(32#)", "Mass Number": "249"}, "250": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "99", "Atomic Symbol": "Es", "Relative Atomic Mass": "250.07861(11#)", "Mass Number": "250"}, "251": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "99", "Atomic Symbol": "Es", "Relative Atomic Mass": "251.0799936(67)", "Mass Number": "251"}, "252": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "99", "Atomic Symbol": "Es", "Relative Atomic Mass": "252.082980(54)", "Mass Number": "252"}, "253": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "99", "Atomic Symbol": "Es", "Relative Atomic Mass": "253.0848257(27)", "Mass Number": "253"}, "254": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "99", "Atomic Symbol": "Es", "Relative Atomic Mass": "254.0880222(45)", "Mass Number": "254"}, "255": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "99", "Atomic Symbol": "Es", "Relative Atomic Mass": "255.090275(12)", "Mass Number": "255"}}, "15": {"24": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "30.973761998(5)", "Atomic Number": "15", "Atomic Symbol": "P", "Relative Atomic Mass": "24.03577(54#)", "Mass Number": "24"}, "25": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "30.973761998(5)", "Atomic Number": "15", "Atomic Symbol": "P", "Relative Atomic Mass": "25.02119(43#)", "Mass Number": "25"}, "26": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "30.973761998(5)", "Atomic Number": "15", "Atomic Symbol": "P", "Relative Atomic Mass": "26.01178(21#)", "Mass Number": "26"}, "27": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "30.973761998(5)", "Atomic Number": "15", "Atomic Symbol": "P", "Relative Atomic Mass": "26.999224(28)", "Mass Number": "27"}, "28": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "30.973761998(5)", "Atomic Number": "15", "Atomic Symbol": "P", "Relative Atomic Mass": "27.9923266(12)", "Mass Number": "28"}, "29": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "30.973761998(5)", "Atomic Number": "15", "Atomic Symbol": "P", "Relative Atomic Mass": "28.98180079(60)", "Mass Number": "29"}, "30": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "30.973761998(5)", "Atomic Number": "15", "Atomic Symbol": "P", "Relative Atomic Mass": "29.97831375(34)", "Mass Number": "30"}, "31": {"Isotopic Composition": "1", "Notes": " ", "Standard Atomic Weight": "30.973761998(5)", "Atomic Number": "15", "Atomic Symbol": "P", "Relative Atomic Mass": "30.97376199842(70)", "Mass Number": "31"}, "32": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "30.973761998(5)", "Atomic Number": "15", "Atomic Symbol": "P", "Relative Atomic Mass": "31.973907643(42)", "Mass Number": "32"}, "33": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "30.973761998(5)", "Atomic Number": "15", "Atomic Symbol": "P", "Relative Atomic Mass": "32.9717257(12)", "Mass Number": "33"}, "34": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "30.973761998(5)", "Atomic Number": "15", "Atomic Symbol": "P", "Relative Atomic Mass": "33.97364589(87)", "Mass Number": "34"}, "35": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "30.973761998(5)", "Atomic Number": "15", "Atomic Symbol": "P", "Relative Atomic Mass": "34.9733141(20)", "Mass Number": "35"}, "36": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "30.973761998(5)", "Atomic Number": "15", "Atomic Symbol": "P", "Relative Atomic Mass": "35.978260(14)", "Mass Number": "36"}, "37": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "30.973761998(5)", "Atomic Number": "15", "Atomic Symbol": "P", "Relative Atomic Mass": "36.979607(41)", "Mass Number": "37"}, "38": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "30.973761998(5)", "Atomic Number": "15", "Atomic Symbol": "P", "Relative Atomic Mass": "37.984252(93)", "Mass Number": "38"}, "39": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "30.973761998(5)", "Atomic Number": "15", "Atomic Symbol": "P", "Relative Atomic Mass": "38.986227(98)", "Mass Number": "39"}, "40": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "30.973761998(5)", "Atomic Number": "15", "Atomic Symbol": "P", "Relative Atomic Mass": "39.99133(12)", "Mass Number": "40"}, "41": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "30.973761998(5)", "Atomic Number": "15", "Atomic Symbol": "P", "Relative Atomic Mass": "40.994654(86)", "Mass Number": "41"}, "42": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "30.973761998(5)", "Atomic Number": "15", "Atomic Symbol": "P", "Relative Atomic Mass": "42.00108(23)", "Mass Number": "42"}, "43": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "30.973761998(5)", "Atomic Number": "15", "Atomic Symbol": "P", "Relative Atomic Mass": "43.00502(40)", "Mass Number": "43"}, "44": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "30.973761998(5)", "Atomic Number": "15", "Atomic Symbol": "P", "Relative Atomic Mass": "44.01121(54#)", "Mass Number": "44"}, "45": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "30.973761998(5)", "Atomic Number": "15", "Atomic Symbol": "P", "Relative Atomic Mass": "45.01645(64#)", "Mass Number": "45"}, "46": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "30.973761998(5)", "Atomic Number": "15", "Atomic Symbol": "P", "Relative Atomic Mass": "46.02446(75#)", "Mass Number": "46"}, "47": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "30.973761998(5)", "Atomic Number": "15", "Atomic Symbol": "P", "Relative Atomic Mass": "47.03139(86#)", "Mass Number": "47"}}, "74": {"192": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "191.96817(21#)", "Mass Number": "192"}, "193": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "192.97178(21#)", "Mass Number": "193"}, "194": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "193.97367(32#)", "Mass Number": "194"}, "157": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "156.97884(43#)", "Mass Number": "157"}, "158": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "157.97456(54#)", "Mass Number": "158"}, "159": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "158.97264(32#)", "Mass Number": "159"}, "160": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "159.96846(17)", "Mass Number": "160"}, "161": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "160.96720(21#)", "Mass Number": "161"}, "162": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "161.963499(19)", "Mass Number": "162"}, "163": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "162.962524(57)", "Mass Number": "163"}, "164": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "163.958961(11)", "Mass Number": "164"}, "165": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "164.958281(27)", "Mass Number": "165"}, "166": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "165.955031(10)", "Mass Number": "166"}, "167": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "166.954805(20)", "Mass Number": "167"}, "168": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "167.951806(14)", "Mass Number": "168"}, "169": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "168.951779(17)", "Mass Number": "169"}, "170": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "169.949232(14)", "Mass Number": "170"}, "171": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "170.949451(30)", "Mass Number": "171"}, "172": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "171.947292(30)", "Mass Number": "172"}, "173": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "172.947689(30)", "Mass Number": "173"}, "174": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "173.946079(30)", "Mass Number": "174"}, "175": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "174.946717(30)", "Mass Number": "175"}, "176": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "175.945634(30)", "Mass Number": "176"}, "177": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "176.946643(30)", "Mass Number": "177"}, "178": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "177.945883(16)", "Mass Number": "178"}, "179": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "178.947077(16)", "Mass Number": "179"}, "180": {"Isotopic Composition": "0.0012(1)", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "179.9467108(20)", "Mass Number": "180"}, "181": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "180.9481978(51)", "Mass Number": "181"}, "182": {"Isotopic Composition": "0.2650(16)", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "181.94820394(91)", "Mass Number": "182"}, "183": {"Isotopic Composition": "0.1431(4)", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "182.95022275(90)", "Mass Number": "183"}, "184": {"Isotopic Composition": "0.3064(2)", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "183.95093092(94)", "Mass Number": "184"}, "185": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "184.95341897(99)", "Mass Number": "185"}, "186": {"Isotopic Composition": "0.2843(19)", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "185.9543628(17)", "Mass Number": "186"}, "187": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "186.9571588(17)", "Mass Number": "187"}, "188": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "187.9584862(36)", "Mass Number": "188"}, "189": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "188.961763(44)", "Mass Number": "189"}, "190": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "189.963091(42)", "Mass Number": "190"}, "191": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "183.84(1)", "Atomic Number": "74", "Atomic Symbol": "W", "Relative Atomic Mass": "190.966531(48)", "Mass Number": "191"}}, "84": {"192": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "191.991336(12)", "Mass Number": "192"}, "193": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "192.991026(37)", "Mass Number": "193"}, "194": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "193.988186(14)", "Mass Number": "194"}, "195": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "194.988126(41)", "Mass Number": "195"}, "196": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "195.985526(14)", "Mass Number": "196"}, "197": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "196.985660(53)", "Mass Number": "197"}, "198": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "197.983389(19)", "Mass Number": "198"}, "199": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "198.983667(25)", "Mass Number": "199"}, "200": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "199.981799(15)", "Mass Number": "200"}, "201": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "200.9822598(63)", "Mass Number": "201"}, "202": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "201.980758(16)", "Mass Number": "202"}, "203": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "202.9814161(93)", "Mass Number": "203"}, "204": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "203.980310(12)", "Mass Number": "204"}, "205": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "204.981203(22)", "Mass Number": "205"}, "206": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "205.9804740(43)", "Mass Number": "206"}, "207": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "206.9815938(72)", "Mass Number": "207"}, "208": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "207.9812461(19)", "Mass Number": "208"}, "209": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "208.9824308(20)", "Mass Number": "209"}, "210": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "209.9828741(13)", "Mass Number": "210"}, "211": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "210.9866536(14)", "Mass Number": "211"}, "212": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "211.9888684(13)", "Mass Number": "212"}, "213": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "212.9928576(33)", "Mass Number": "213"}, "214": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "213.9952017(16)", "Mass Number": "214"}, "215": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "214.9994201(27)", "Mass Number": "215"}, "216": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "216.0019152(23)", "Mass Number": "216"}, "217": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "217.0063182(67)", "Mass Number": "217"}, "218": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "218.0089735(25)", "Mass Number": "218"}, "219": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "219.013614(17)", "Mass Number": "219"}, "220": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "220.016386(19)", "Mass Number": "220"}, "221": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "221.021228(21)", "Mass Number": "221"}, "222": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "222.024140(43)", "Mass Number": "222"}, "223": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "223.02907(21#)", "Mass Number": "223"}, "224": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "224.03211(21#)", "Mass Number": "224"}, "225": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "225.03707(32#)", "Mass Number": "225"}, "226": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "226.04031(43#)", "Mass Number": "226"}, "227": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "227.04539(43#)", "Mass Number": "227"}, "186": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "186.004393(35)", "Mass Number": "186"}, "187": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "187.003041(34)", "Mass Number": "187"}, "188": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "187.999416(21)", "Mass Number": "188"}, "189": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "188.998473(24)", "Mass Number": "189"}, "190": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "189.995101(14)", "Mass Number": "190"}, "191": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[209]", "Atomic Number": "84", "Atomic Symbol": "Po", "Relative Atomic Mass": "190.9945585(76)", "Mass Number": "191"}}, "60": {"128": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "127.93525(21#)", "Mass Number": "128"}, "129": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "128.93310(22#)", "Mass Number": "129"}, "130": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "129.928506(30)", "Mass Number": "130"}, "131": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "130.927248(30)", "Mass Number": "131"}, "132": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "131.923321(26)", "Mass Number": "132"}, "133": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "132.922348(50)", "Mass Number": "133"}, "134": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "133.918790(13)", "Mass Number": "134"}, "135": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "134.918181(21)", "Mass Number": "135"}, "136": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "135.914976(13)", "Mass Number": "136"}, "137": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "136.914562(13)", "Mass Number": "137"}, "138": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "137.911950(12)", "Mass Number": "138"}, "139": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "138.911954(30)", "Mass Number": "139"}, "140": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "139.909550(28)", "Mass Number": "140"}, "141": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "140.9096147(38)", "Mass Number": "141"}, "142": {"Isotopic Composition": "0.27152(40)", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "141.9077290(20)", "Mass Number": "142"}, "143": {"Isotopic Composition": "0.12174(26)", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "142.9098200(20)", "Mass Number": "143"}, "144": {"Isotopic Composition": "0.23798(19)", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "143.9100930(20)", "Mass Number": "144"}, "145": {"Isotopic Composition": "0.08293(12)", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "144.9125793(20)", "Mass Number": "145"}, "146": {"Isotopic Composition": "0.17189(32)", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "145.9131226(20)", "Mass Number": "146"}, "147": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "146.9161061(20)", "Mass Number": "147"}, "148": {"Isotopic Composition": "0.05756(21)", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "147.9168993(26)", "Mass Number": "148"}, "149": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "148.9201548(26)", "Mass Number": "149"}, "150": {"Isotopic Composition": "0.05638(28)", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "149.9209022(18)", "Mass Number": "150"}, "151": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "150.9238403(18)", "Mass Number": "151"}, "152": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "151.924692(26)", "Mass Number": "152"}, "153": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "152.9277180(29)", "Mass Number": "153"}, "154": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "153.92948(12)", "Mass Number": "154"}, "155": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "154.9331357(98)", "Mass Number": "155"}, "156": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "155.93508(21)", "Mass Number": "156"}, "157": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "156.939386(27)", "Mass Number": "157"}, "158": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "157.94197(32#)", "Mass Number": "158"}, "159": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "158.94653(43#)", "Mass Number": "159"}, "160": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "159.94940(43#)", "Mass Number": "160"}, "161": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "160.95428(54#)", "Mass Number": "161"}, "124": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "123.95220(54#)", "Mass Number": "124"}, "125": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "124.94890(43#)", "Mass Number": "125"}, "126": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "125.94311(32#)", "Mass Number": "126"}, "127": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "144.242(3)", "Atomic Number": "60", "Atomic Symbol": "Nd", "Relative Atomic Mass": "126.94038(32#)", "Mass Number": "127"}}, "90": {"208": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "208.017900(36)", "Mass Number": "208"}, "209": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "209.017753(93)", "Mass Number": "209"}, "210": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "210.015094(20)", "Mass Number": "210"}, "211": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "211.014929(80)", "Mass Number": "211"}, "212": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "212.012988(17)", "Mass Number": "212"}, "213": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "213.013009(76)", "Mass Number": "213"}, "214": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "214.011500(17)", "Mass Number": "214"}, "215": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "215.0117248(95)", "Mass Number": "215"}, "216": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "216.011056(13)", "Mass Number": "216"}, "217": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "217.013117(22)", "Mass Number": "217"}, "218": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "218.013276(11)", "Mass Number": "218"}, "219": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "219.015537(54)", "Mass Number": "219"}, "220": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "220.015748(24)", "Mass Number": "220"}, "221": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "221.018184(10)", "Mass Number": "221"}, "222": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "222.018469(13)", "Mass Number": "222"}, "223": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "223.0208119(99)", "Mass Number": "223"}, "224": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "224.021464(11)", "Mass Number": "224"}, "225": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "225.0239514(55)", "Mass Number": "225"}, "226": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "226.0249034(50)", "Mass Number": "226"}, "227": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "227.0277042(27)", "Mass Number": "227"}, "228": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "228.0287413(23)", "Mass Number": "228"}, "229": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "229.0317627(30)", "Mass Number": "229"}, "230": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "230.0331341(19)", "Mass Number": "230"}, "231": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "231.0363046(19)", "Mass Number": "231"}, "232": {"Isotopic Composition": "1", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "232.0380558(21)", "Mass Number": "232"}, "233": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "233.0415823(21)", "Mass Number": "233"}, "234": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "234.0436014(37)", "Mass Number": "234"}, "235": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "235.047255(14)", "Mass Number": "235"}, "236": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "236.049657(15)", "Mass Number": "236"}, "237": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "237.053629(17)", "Mass Number": "237"}, "238": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "238.05650(30#)", "Mass Number": "238"}, "239": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "232.0377(4)", "Atomic Number": "90", "Atomic Symbol": "Th", "Relative Atomic Mass": "239.06077(43#)", "Mass Number": "239"}}, "28": {"64": {"Isotopic Composition": "0.009255(19)", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "63.92796682(58)", "Mass Number": "64"}, "65": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "64.93008517(60)", "Mass Number": "65"}, "66": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "65.9291393(15)", "Mass Number": "66"}, "67": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "66.9315694(31)", "Mass Number": "67"}, "68": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "67.9318688(32)", "Mass Number": "68"}, "69": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "68.9356103(40)", "Mass Number": "69"}, "70": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "69.9364313(23)", "Mass Number": "70"}, "71": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "70.9405190(24)", "Mass Number": "71"}, "72": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "71.9417859(24)", "Mass Number": "72"}, "73": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "72.9462067(26)", "Mass Number": "73"}, "74": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "73.94798(43#)", "Mass Number": "74"}, "75": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "74.95250(32#)", "Mass Number": "75"}, "76": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "75.95533(54#)", "Mass Number": "76"}, "77": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "76.96055(54#)", "Mass Number": "77"}, "78": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "77.96336(86#)", "Mass Number": "78"}, "79": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "78.97025(86#)", "Mass Number": "79"}, "48": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "48.01769(54#)", "Mass Number": "48"}, "49": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "49.00770(86#)", "Mass Number": "49"}, "50": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "49.99474(86#)", "Mass Number": "50"}, "51": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "50.98611(86#)", "Mass Number": "51"}, "52": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "51.97480(75#)", "Mass Number": "52"}, "53": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "52.968190(27)", "Mass Number": "53"}, "54": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "53.957892(54)", "Mass Number": "54"}, "55": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "54.95133063(85)", "Mass Number": "55"}, "56": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "55.94212855(57)", "Mass Number": "56"}, "57": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "56.93979218(71)", "Mass Number": "57"}, "58": {"Isotopic Composition": "0.68077(19)", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "57.93534241(52)", "Mass Number": "58"}, "59": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "58.93434620(52)", "Mass Number": "59"}, "60": {"Isotopic Composition": "0.26223(15)", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "59.93078588(52)", "Mass Number": "60"}, "61": {"Isotopic Composition": "0.011399(13)", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "60.93105557(52)", "Mass Number": "61"}, "62": {"Isotopic Composition": "0.036346(40)", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "61.92834537(55)", "Mass Number": "62"}, "63": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "58.6934(4)", "Atomic Number": "28", "Atomic Symbol": "Ni", "Relative Atomic Mass": "62.92966963(56)", "Mass Number": "63"}}, "70": {"148": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "147.96758(64#)", "Mass Number": "148"}, "149": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "148.96436(54#)", "Mass Number": "149"}, "150": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "149.95852(43#)", "Mass Number": "150"}, "151": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "150.95540(32)", "Mass Number": "151"}, "152": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "151.95027(17)", "Mass Number": "152"}, "153": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "152.94932(21#)", "Mass Number": "153"}, "154": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "153.946396(19)", "Mass Number": "154"}, "155": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "154.945783(18)", "Mass Number": "155"}, "156": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "155.942825(11)", "Mass Number": "156"}, "157": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "156.942645(12)", "Mass Number": "157"}, "158": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "157.9398705(86)", "Mass Number": "158"}, "159": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "158.940055(19)", "Mass Number": "159"}, "160": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "159.937557(17)", "Mass Number": "160"}, "161": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "160.937907(17)", "Mass Number": "161"}, "162": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "161.935774(17)", "Mass Number": "162"}, "163": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "162.936340(17)", "Mass Number": "163"}, "164": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "163.934495(17)", "Mass Number": "164"}, "165": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "164.935270(28)", "Mass Number": "165"}, "166": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "165.9338747(78)", "Mass Number": "166"}, "167": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "166.9349530(47)", "Mass Number": "167"}, "168": {"Isotopic Composition": "0.00123(3)", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "167.9338896(22)", "Mass Number": "168"}, "169": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "168.9351825(22)", "Mass Number": "169"}, "170": {"Isotopic Composition": "0.02982(39)", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "169.9347664(22)", "Mass Number": "170"}, "171": {"Isotopic Composition": "0.1409(14)", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "170.9363302(22)", "Mass Number": "171"}, "172": {"Isotopic Composition": "0.2168(13)", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "171.9363859(22)", "Mass Number": "172"}, "173": {"Isotopic Composition": "0.16103(63)", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "172.9382151(22)", "Mass Number": "173"}, "174": {"Isotopic Composition": "0.32026(80)", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "173.9388664(22)", "Mass Number": "174"}, "175": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "174.9412808(22)", "Mass Number": "175"}, "176": {"Isotopic Composition": "0.12996(83)", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "175.9425764(24)", "Mass Number": "176"}, "177": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "176.9452656(24)", "Mass Number": "177"}, "178": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "177.946651(11)", "Mass Number": "178"}, "179": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "178.95004(21#)", "Mass Number": "179"}, "180": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "179.95212(32#)", "Mass Number": "180"}, "181": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "173.054(5)", "Atomic Number": "70", "Atomic Symbol": "Yb", "Relative Atomic Mass": "180.95589(32#)", "Mass Number": "181"}}, "34": {"64": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "63.97109(54#)", "Mass Number": "64"}, "65": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "64.96440(64#)", "Mass Number": "65"}, "66": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "65.95559(32#)", "Mass Number": "66"}, "67": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "66.949994(72)", "Mass Number": "67"}, "68": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "67.94182524(53)", "Mass Number": "68"}, "69": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "68.9394148(16)", "Mass Number": "69"}, "70": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "69.9335155(17)", "Mass Number": "70"}, "71": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "70.9322094(30)", "Mass Number": "71"}, "72": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "71.9271405(21)", "Mass Number": "72"}, "73": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "72.9267549(80)", "Mass Number": "73"}, "74": {"Isotopic Composition": "0.0089(4)", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "73.922475934(15)", "Mass Number": "74"}, "75": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "74.922522870(78)", "Mass Number": "75"}, "76": {"Isotopic Composition": "0.0937(29)", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "75.919213704(17)", "Mass Number": "76"}, "77": {"Isotopic Composition": "0.0763(16)", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "76.919914154(67)", "Mass Number": "77"}, "78": {"Isotopic Composition": "0.2377(28)", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "77.91730928(20)", "Mass Number": "78"}, "79": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "78.91849929(24)", "Mass Number": "79"}, "80": {"Isotopic Composition": "0.4961(41)", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "79.9165218(13)", "Mass Number": "80"}, "81": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "80.9179930(14)", "Mass Number": "81"}, "82": {"Isotopic Composition": "0.0873(22)", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "81.9166995(15)", "Mass Number": "82"}, "83": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "82.9191186(36)", "Mass Number": "83"}, "84": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "83.9184668(21)", "Mass Number": "84"}, "85": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "84.9222608(28)", "Mass Number": "85"}, "86": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "85.9243117(27)", "Mass Number": "86"}, "87": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "86.9286886(24)", "Mass Number": "87"}, "88": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "87.9314175(36)", "Mass Number": "88"}, "89": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "88.9366691(40)", "Mass Number": "89"}, "90": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "89.94010(35)", "Mass Number": "90"}, "91": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "90.94596(54#)", "Mass Number": "91"}, "92": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "91.94984(64#)", "Mass Number": "92"}, "93": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "92.95629(86#)", "Mass Number": "93"}, "94": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "93.96049(86#)", "Mass Number": "94"}, "95": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "78.971(8)", "Atomic Number": "34", "Atomic Symbol": "Se", "Relative Atomic Mass": "94.96730(86#)", "Mass Number": "95"}}, "50": {"128": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "127.910507(19)", "Mass Number": "128"}, "129": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "128.913465(21)", "Mass Number": "129"}, "130": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "129.9139738(23)", "Mass Number": "130"}, "131": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "130.9170450(65)", "Mass Number": "131"}, "132": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "131.9178267(31)", "Mass Number": "132"}, "133": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "132.9239134(26)", "Mass Number": "133"}, "134": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "133.9286821(35)", "Mass Number": "134"}, "135": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "134.9349086(33)", "Mass Number": "135"}, "136": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "135.93999(43#)", "Mass Number": "136"}, "137": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "136.94655(54#)", "Mass Number": "137"}, "138": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "137.95184(64#)", "Mass Number": "138"}, "99": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "98.94853(54#)", "Mass Number": "99"}, "100": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "99.93850(32)", "Mass Number": "100"}, "101": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "100.93526(32)", "Mass Number": "101"}, "102": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "101.93029(11)", "Mass Number": "102"}, "103": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "102.928105(76)", "Mass Number": "103"}, "104": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "103.9231052(62)", "Mass Number": "104"}, "105": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "104.9212684(43)", "Mass Number": "105"}, "106": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "105.9169574(55)", "Mass Number": "106"}, "107": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "106.9157137(57)", "Mass Number": "107"}, "108": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "107.9118943(58)", "Mass Number": "108"}, "109": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "108.9112921(85)", "Mass Number": "109"}, "110": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "109.907845(15)", "Mass Number": "110"}, "111": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "110.9077401(58)", "Mass Number": "111"}, "112": {"Isotopic Composition": "0.0097(1)", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "111.90482387(61)", "Mass Number": "112"}, "113": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "112.9051757(18)", "Mass Number": "113"}, "114": {"Isotopic Composition": "0.0066(1)", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "113.9027827(10)", "Mass Number": "114"}, "115": {"Isotopic Composition": "0.0034(1)", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "114.903344699(16)", "Mass Number": "115"}, "116": {"Isotopic Composition": "0.1454(9)", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "115.90174280(10)", "Mass Number": "116"}, "117": {"Isotopic Composition": "0.0768(7)", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "116.90295398(52)", "Mass Number": "117"}, "118": {"Isotopic Composition": "0.2422(9)", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "117.90160657(54)", "Mass Number": "118"}, "119": {"Isotopic Composition": "0.0859(4)", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "118.90331117(78)", "Mass Number": "119"}, "120": {"Isotopic Composition": "0.3258(9)", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "119.90220163(97)", "Mass Number": "120"}, "121": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "120.9042426(10)", "Mass Number": "121"}, "122": {"Isotopic Composition": "0.0463(3)", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "121.9034438(26)", "Mass Number": "122"}, "123": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "122.9057252(26)", "Mass Number": "123"}, "124": {"Isotopic Composition": "0.0579(5)", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "123.9052766(11)", "Mass Number": "124"}, "125": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "124.9077864(11)", "Mass Number": "125"}, "126": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "125.907659(11)", "Mass Number": "126"}, "127": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "118.710(7)", "Atomic Number": "50", "Atomic Symbol": "Sn", "Relative Atomic Mass": "126.910390(11)", "Mass Number": "127"}}, "53": {"128": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "127.9058086(39)", "Mass Number": "128"}, "129": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "128.9049837(34)", "Mass Number": "129"}, "130": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "129.9066702(34)", "Mass Number": "130"}, "131": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "130.90612630(69)", "Mass Number": "131"}, "132": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "131.9079935(44)", "Mass Number": "132"}, "133": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "132.9077970(50)", "Mass Number": "133"}, "134": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "133.9097588(59)", "Mass Number": "134"}, "135": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "134.9100488(58)", "Mass Number": "135"}, "136": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "135.914604(15)", "Mass Number": "136"}, "137": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "136.9180282(90)", "Mass Number": "137"}, "138": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "137.9227264(64)", "Mass Number": "138"}, "139": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "138.926506(31)", "Mass Number": "139"}, "140": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "139.93173(20)", "Mass Number": "140"}, "141": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "140.93569(21#)", "Mass Number": "141"}, "142": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "141.94120(40)", "Mass Number": "142"}, "143": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "142.94565(32#)", "Mass Number": "143"}, "144": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "143.95139(43#)", "Mass Number": "144"}, "145": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "144.95605(54#)", "Mass Number": "145"}, "107": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "106.94678(32#)", "Mass Number": "107"}, "108": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "107.94348(14)", "Mass Number": "108"}, "109": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "108.9380853(61)", "Mass Number": "109"}, "110": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "109.935089(54)", "Mass Number": "110"}, "111": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "110.9302692(51)", "Mass Number": "111"}, "112": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "111.928005(11)", "Mass Number": "112"}, "113": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "112.9236501(86)", "Mass Number": "113"}, "114": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "113.92185(32#)", "Mass Number": "114"}, "115": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "114.918048(31)", "Mass Number": "115"}, "116": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "115.91681(10)", "Mass Number": "116"}, "117": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "116.913648(28)", "Mass Number": "117"}, "118": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "117.913074(21)", "Mass Number": "118"}, "119": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "118.910074(30)", "Mass Number": "119"}, "120": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "119.910087(16)", "Mass Number": "120"}, "121": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "120.9074051(58)", "Mass Number": "121"}, "122": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "121.9075888(56)", "Mass Number": "122"}, "123": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "122.9055885(40)", "Mass Number": "123"}, "124": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "123.9062090(26)", "Mass Number": "124"}, "125": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "124.9046294(16)", "Mass Number": "125"}, "126": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "125.9056233(41)", "Mass Number": "126"}, "127": {"Isotopic Composition": "1", "Notes": " ", "Standard Atomic Weight": "126.90447(3)", "Atomic Number": "53", "Atomic Symbol": "I", "Relative Atomic Mass": "126.9044719(39)", "Mass Number": "127"}}, "63": {"130": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "129.96369(54#)", "Mass Number": "130"}, "131": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "130.95784(43#)", "Mass Number": "131"}, "132": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "131.95467(43#)", "Mass Number": "132"}, "133": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "132.94929(32#)", "Mass Number": "133"}, "134": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "133.94640(32#)", "Mass Number": "134"}, "135": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "134.94187(21#)", "Mass Number": "135"}, "136": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "135.93962(21#)", "Mass Number": "136"}, "137": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "136.93546(21#)", "Mass Number": "137"}, "138": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "137.933709(30)", "Mass Number": "138"}, "139": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "138.929792(14)", "Mass Number": "139"}, "140": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "139.928088(55)", "Mass Number": "140"}, "141": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "140.924932(14)", "Mass Number": "141"}, "142": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "141.923442(32)", "Mass Number": "142"}, "143": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "142.920299(12)", "Mass Number": "143"}, "144": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "143.918820(12)", "Mass Number": "144"}, "145": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "144.9162726(36)", "Mass Number": "145"}, "146": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "145.9172110(65)", "Mass Number": "146"}, "147": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "146.9167527(31)", "Mass Number": "147"}, "148": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "147.918089(11)", "Mass Number": "148"}, "149": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "148.9179378(44)", "Mass Number": "149"}, "150": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "149.9197077(68)", "Mass Number": "150"}, "151": {"Isotopic Composition": "0.4781(6)", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "150.9198578(18)", "Mass Number": "151"}, "152": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "151.9217522(18)", "Mass Number": "152"}, "153": {"Isotopic Composition": "0.5219(6)", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "152.9212380(18)", "Mass Number": "153"}, "154": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "153.9229870(19)", "Mass Number": "154"}, "155": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "154.9229011(19)", "Mass Number": "155"}, "156": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "155.9247605(59)", "Mass Number": "156"}, "157": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "156.9254334(46)", "Mass Number": "157"}, "158": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "157.927799(11)", "Mass Number": "158"}, "159": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "158.9291001(47)", "Mass Number": "159"}, "160": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "159.931851(10)", "Mass Number": "160"}, "161": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "160.933664(11)", "Mass Number": "161"}, "162": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "161.936989(65)", "Mass Number": "162"}, "163": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "162.939196(76)", "Mass Number": "163"}, "164": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "163.94274(22#)", "Mass Number": "164"}, "165": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "164.94559(35#)", "Mass Number": "165"}, "166": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "165.94962(32#)", "Mass Number": "166"}, "167": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "151.964(1)", "Atomic Number": "63", "Atomic Symbol": "Eu", "Relative Atomic Mass": "166.95289(43#)", "Mass Number": "167"}}, "56": {"128": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "127.9083420(56)", "Mass Number": "128"}, "129": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "128.908681(11)", "Mass Number": "129"}, "130": {"Isotopic Composition": "0.00106(1)", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "129.9063207(28)", "Mass Number": "130"}, "131": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "130.9069410(28)", "Mass Number": "131"}, "132": {"Isotopic Composition": "0.00101(1)", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "131.9050611(11)", "Mass Number": "132"}, "133": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "132.9060074(11)", "Mass Number": "133"}, "134": {"Isotopic Composition": "0.02417(18)", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "133.90450818(30)", "Mass Number": "134"}, "135": {"Isotopic Composition": "0.06592(12)", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "134.90568838(29)", "Mass Number": "135"}, "136": {"Isotopic Composition": "0.07854(24)", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "135.90457573(29)", "Mass Number": "136"}, "137": {"Isotopic Composition": "0.11232(24)", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "136.90582714(30)", "Mass Number": "137"}, "138": {"Isotopic Composition": "0.71698(42)", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "137.90524700(31)", "Mass Number": "138"}, "139": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "138.90884110(31)", "Mass Number": "139"}, "140": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "139.9106057(85)", "Mass Number": "140"}, "141": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "140.9144033(57)", "Mass Number": "141"}, "142": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "141.9164324(64)", "Mass Number": "142"}, "143": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "142.9206253(74)", "Mass Number": "143"}, "144": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "143.9229549(77)", "Mass Number": "144"}, "145": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "144.9275184(91)", "Mass Number": "145"}, "146": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "145.930284(22)", "Mass Number": "146"}, "147": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "146.935304(21)", "Mass Number": "147"}, "148": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "147.938171(68)", "Mass Number": "148"}, "149": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "148.94308(21#)", "Mass Number": "149"}, "150": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "149.94605(32#)", "Mass Number": "150"}, "151": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "150.95127(32#)", "Mass Number": "151"}, "152": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "151.95481(43#)", "Mass Number": "152"}, "153": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "152.96036(43#)", "Mass Number": "153"}, "114": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "113.95066(12)", "Mass Number": "114"}, "115": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "114.94737(54#)", "Mass Number": "115"}, "116": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "115.94128(32#)", "Mass Number": "116"}, "117": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "116.93814(21)", "Mass Number": "117"}, "118": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "117.93306(21#)", "Mass Number": "118"}, "119": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "118.93066(21)", "Mass Number": "119"}, "120": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "119.92605(32)", "Mass Number": "120"}, "121": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "120.92405(15)", "Mass Number": "121"}, "122": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "121.919904(30)", "Mass Number": "122"}, "123": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "122.918781(13)", "Mass Number": "123"}, "124": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "123.915094(13)", "Mass Number": "124"}, "125": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "124.914472(12)", "Mass Number": "125"}, "126": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "125.911250(13)", "Mass Number": "126"}, "127": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "137.327(7)", "Atomic Number": "56", "Atomic Symbol": "Ba", "Relative Atomic Mass": "126.911091(12)", "Mass Number": "127"}}, "87": {"199": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "199.007259(45)", "Mass Number": "199"}, "200": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "200.006586(63)", "Mass Number": "200"}, "201": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "201.003867(77)", "Mass Number": "201"}, "202": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "202.003320(55#)", "Mass Number": "202"}, "203": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "203.0009407(67)", "Mass Number": "203"}, "204": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "204.000652(26)", "Mass Number": "204"}, "205": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "204.9985939(84)", "Mass Number": "205"}, "206": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "205.998666(30)", "Mass Number": "206"}, "207": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "206.996946(19)", "Mass Number": "207"}, "208": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "207.997138(12)", "Mass Number": "208"}, "209": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "208.995955(16)", "Mass Number": "209"}, "210": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "209.996422(16)", "Mass Number": "210"}, "211": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "210.995556(13)", "Mass Number": "211"}, "212": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "211.9962257(94)", "Mass Number": "212"}, "213": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "212.9961860(55)", "Mass Number": "213"}, "214": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "213.9989713(93)", "Mass Number": "214"}, "215": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "215.0003418(76)", "Mass Number": "215"}, "216": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "216.0031899(45)", "Mass Number": "216"}, "217": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "217.0046323(70)", "Mass Number": "217"}, "218": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "218.0075787(51)", "Mass Number": "218"}, "219": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "219.0092524(76)", "Mass Number": "219"}, "220": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "220.0123277(44)", "Mass Number": "220"}, "221": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "221.0142552(54)", "Mass Number": "221"}, "222": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "222.017552(23)", "Mass Number": "222"}, "223": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "223.0197360(25)", "Mass Number": "223"}, "224": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "224.023398(14)", "Mass Number": "224"}, "225": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "225.025573(13)", "Mass Number": "225"}, "226": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "226.029566(13)", "Mass Number": "226"}, "227": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "227.031869(14)", "Mass Number": "227"}, "228": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "228.035823(14)", "Mass Number": "228"}, "229": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "229.038298(15)", "Mass Number": "229"}, "230": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "230.042416(17)", "Mass Number": "230"}, "231": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "231.045158(27)", "Mass Number": "231"}, "232": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "232.04937(17#)", "Mass Number": "232"}, "233": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[223]", "Atomic Number": "87", "Atomic Symbol": "Fr", "Relative Atomic Mass": "233.05264(32#)", "Mass Number": "233"}}, "110": {"267": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "110", "Atomic Symbol": "Ds", "Relative Atomic Mass": "267.14377(15#)", "Mass Number": "267"}, "268": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "110", "Atomic Symbol": "Ds", "Relative Atomic Mass": "268.14348(32#)", "Mass Number": "268"}, "269": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "110", "Atomic Symbol": "Ds", "Relative Atomic Mass": "269.144752(34)", "Mass Number": "269"}, "270": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "110", "Atomic Symbol": "Ds", "Relative Atomic Mass": "270.144584(52)", "Mass Number": "270"}, "271": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "110", "Atomic Symbol": "Ds", "Relative Atomic Mass": "271.14595(10#)", "Mass Number": "271"}, "272": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "110", "Atomic Symbol": "Ds", "Relative Atomic Mass": "272.14602(44#)", "Mass Number": "272"}, "273": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "110", "Atomic Symbol": "Ds", "Relative Atomic Mass": "273.14856(14#)", "Mass Number": "273"}, "274": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "110", "Atomic Symbol": "Ds", "Relative Atomic Mass": "274.14941(42#)", "Mass Number": "274"}, "275": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "110", "Atomic Symbol": "Ds", "Relative Atomic Mass": "275.15203(45#)", "Mass Number": "275"}, "276": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "110", "Atomic Symbol": "Ds", "Relative Atomic Mass": "276.15303(59#)", "Mass Number": "276"}, "277": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "110", "Atomic Symbol": "Ds", "Relative Atomic Mass": "277.15591(41#)", "Mass Number": "277"}, "278": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "110", "Atomic Symbol": "Ds", "Relative Atomic Mass": "278.15704(67#)", "Mass Number": "278"}, "279": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "110", "Atomic Symbol": "Ds", "Relative Atomic Mass": "279.16010(64#)", "Mass Number": "279"}, "280": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "110", "Atomic Symbol": "Ds", "Relative Atomic Mass": "280.16131(89#)", "Mass Number": "280"}, "281": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "110", "Atomic Symbol": "Ds", "Relative Atomic Mass": "281.16451(59#)", "Mass Number": "281"}}, "36": {"69": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "68.96518(43#)", "Mass Number": "69"}, "70": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "69.95604(21#)", "Mass Number": "70"}, "71": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "70.95027(14)", "Mass Number": "71"}, "72": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "71.9420924(86)", "Mass Number": "72"}, "73": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "72.9392892(71)", "Mass Number": "73"}, "74": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "73.9330840(22)", "Mass Number": "74"}, "75": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "74.9309457(87)", "Mass Number": "75"}, "76": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "75.9259103(43)", "Mass Number": "76"}, "77": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "76.9246700(21)", "Mass Number": "77"}, "78": {"Isotopic Composition": "0.00355(3)", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "77.92036494(76)", "Mass Number": "78"}, "79": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "78.9200829(38)", "Mass Number": "79"}, "80": {"Isotopic Composition": "0.02286(10)", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "79.91637808(75)", "Mass Number": "80"}, "81": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "80.9165912(15)", "Mass Number": "81"}, "82": {"Isotopic Composition": "0.11593(31)", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "81.91348273(94)", "Mass Number": "82"}, "83": {"Isotopic Composition": "0.11500(19)", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "82.91412716(32)", "Mass Number": "83"}, "84": {"Isotopic Composition": "0.56987(15)", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "83.9114977282(44)", "Mass Number": "84"}, "85": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "84.9125273(21)", "Mass Number": "85"}, "86": {"Isotopic Composition": "0.17279(41)", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "85.9106106269(41)", "Mass Number": "86"}, "87": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "86.91335476(26)", "Mass Number": "87"}, "88": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "87.9144479(28)", "Mass Number": "88"}, "89": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "88.9178355(23)", "Mass Number": "89"}, "90": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "89.9195279(20)", "Mass Number": "90"}, "91": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "90.9238063(24)", "Mass Number": "91"}, "92": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "91.9261731(29)", "Mass Number": "92"}, "93": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "92.9311472(27)", "Mass Number": "93"}, "94": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "93.934140(13)", "Mass Number": "94"}, "95": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "94.939711(20)", "Mass Number": "95"}, "96": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "95.943017(22)", "Mass Number": "96"}, "97": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "96.94909(14)", "Mass Number": "97"}, "98": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "97.95243(32#)", "Mass Number": "98"}, "99": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "98.95839(54#)", "Mass Number": "99"}, "100": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "99.96237(43#)", "Mass Number": "100"}, "101": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "83.798(2)", "Atomic Number": "36", "Atomic Symbol": "Kr", "Relative Atomic Mass": "100.96873(54#)", "Mass Number": "101"}}, "3": {"3": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[6.938,6.997]", "Atomic Number": "3", "Atomic Symbol": "Li", "Relative Atomic Mass": "3.0308(21#)", "Mass Number": "3"}, "4": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[6.938,6.997]", "Atomic Number": "3", "Atomic Symbol": "Li", "Relative Atomic Mass": "4.02719(23)", "Mass Number": "4"}, "5": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[6.938,6.997]", "Atomic Number": "3", "Atomic Symbol": "Li", "Relative Atomic Mass": "5.012538(54)", "Mass Number": "5"}, "6": {"Isotopic Composition": "0.0759(4)", "Notes": "m", "Standard Atomic Weight": "[6.938,6.997]", "Atomic Number": "3", "Atomic Symbol": "Li", "Relative Atomic Mass": "6.0151228874(16)", "Mass Number": "6"}, "7": {"Isotopic Composition": "0.9241(4)", "Notes": "m", "Standard Atomic Weight": "[6.938,6.997]", "Atomic Number": "3", "Atomic Symbol": "Li", "Relative Atomic Mass": "7.0160034366(45)", "Mass Number": "7"}, "8": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[6.938,6.997]", "Atomic Number": "3", "Atomic Symbol": "Li", "Relative Atomic Mass": "8.022486246(50)", "Mass Number": "8"}, "9": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[6.938,6.997]", "Atomic Number": "3", "Atomic Symbol": "Li", "Relative Atomic Mass": "9.02679019(20)", "Mass Number": "9"}, "10": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[6.938,6.997]", "Atomic Number": "3", "Atomic Symbol": "Li", "Relative Atomic Mass": "10.035483(14)", "Mass Number": "10"}, "11": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[6.938,6.997]", "Atomic Number": "3", "Atomic Symbol": "Li", "Relative Atomic Mass": "11.04372358(66)", "Mass Number": "11"}, "12": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[6.938,6.997]", "Atomic Number": "3", "Atomic Symbol": "Li", "Relative Atomic Mass": "12.052517(16)", "Mass Number": "12"}, "13": {"Isotopic Composition": "", "Notes": "m", "Standard Atomic Weight": "[6.938,6.997]", "Atomic Number": "3", "Atomic Symbol": "Li", "Relative Atomic Mass": "13.06263(38)", "Mass Number": "13"}}, "51": {"128": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "127.909146(21)", "Mass Number": "128"}, "129": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "128.909147(23)", "Mass Number": "129"}, "130": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "129.911662(15)", "Mass Number": "130"}, "131": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "130.9119888(23)", "Mass Number": "131"}, "132": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "131.9145077(29)", "Mass Number": "132"}, "133": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "132.9152732(34)", "Mass Number": "133"}, "134": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "133.9205357(18)", "Mass Number": "134"}, "135": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "134.9251851(31)", "Mass Number": "135"}, "136": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "135.9307459(68)", "Mass Number": "136"}, "137": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "136.93555(32)", "Mass Number": "137"}, "138": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "137.94145(32#)", "Mass Number": "138"}, "139": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "138.94655(43#)", "Mass Number": "139"}, "140": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "139.95283(64#)", "Mass Number": "140"}, "103": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "102.93969(32#)", "Mass Number": "103"}, "104": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "103.93648(13)", "Mass Number": "104"}, "105": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "104.931276(23)", "Mass Number": "105"}, "106": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "105.9286380(80)", "Mass Number": "106"}, "107": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "106.9241506(45)", "Mass Number": "107"}, "108": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "107.9222267(59)", "Mass Number": "108"}, "109": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "108.9181411(57)", "Mass Number": "109"}, "110": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "109.9168543(64)", "Mass Number": "110"}, "111": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "110.9132182(95)", "Mass Number": "111"}, "112": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "111.912400(19)", "Mass Number": "112"}, "113": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "112.909375(18)", "Mass Number": "113"}, "114": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "113.909290(23)", "Mass Number": "114"}, "115": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "114.906598(17)", "Mass Number": "115"}, "116": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "115.9067931(55)", "Mass Number": "116"}, "117": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "116.9048415(91)", "Mass Number": "117"}, "118": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "117.9055321(32)", "Mass Number": "118"}, "119": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "118.9039455(83)", "Mass Number": "119"}, "120": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "119.9050794(77)", "Mass Number": "120"}, "121": {"Isotopic Composition": "0.5721(5)", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "120.9038120(30)", "Mass Number": "121"}, "122": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "121.9051699(30)", "Mass Number": "122"}, "123": {"Isotopic Composition": "0.4279(5)", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "122.9042132(23)", "Mass Number": "123"}, "124": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "123.9059350(23)", "Mass Number": "124"}, "125": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "124.9052530(28)", "Mass Number": "125"}, "126": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "125.907253(34)", "Mass Number": "126"}, "127": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "121.760(1)", "Atomic Number": "51", "Atomic Symbol": "Sb", "Relative Atomic Mass": "126.9069243(55)", "Mass Number": "127"}}, "23": {"64": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "50.9415(1)", "Atomic Number": "23", "Atomic Symbol": "V", "Relative Atomic Mass": "63.98264(43#)", "Mass Number": "64"}, "65": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "50.9415(1)", "Atomic Number": "23", "Atomic Symbol": "V", "Relative Atomic Mass": "64.98750(54#)", "Mass Number": "65"}, "66": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "50.9415(1)", "Atomic Number": "23", "Atomic Symbol": "V", "Relative Atomic Mass": "65.99398(64#)", "Mass Number": "66"}, "40": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "50.9415(1)", "Atomic Number": "23", "Atomic Symbol": "V", "Relative Atomic Mass": "40.01276(43#)", "Mass Number": "40"}, "41": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "50.9415(1)", "Atomic Number": "23", "Atomic Symbol": "V", "Relative Atomic Mass": "41.00021(32#)", "Mass Number": "41"}, "42": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "50.9415(1)", "Atomic Number": "23", "Atomic Symbol": "V", "Relative Atomic Mass": "41.99182(32#)", "Mass Number": "42"}, "43": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "50.9415(1)", "Atomic Number": "23", "Atomic Symbol": "V", "Relative Atomic Mass": "42.980766(46)", "Mass Number": "43"}, "44": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "50.9415(1)", "Atomic Number": "23", "Atomic Symbol": "V", "Relative Atomic Mass": "43.97411(20)", "Mass Number": "44"}, "45": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "50.9415(1)", "Atomic Number": "23", "Atomic Symbol": "V", "Relative Atomic Mass": "44.9657748(86)", "Mass Number": "45"}, "46": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "50.9415(1)", "Atomic Number": "23", "Atomic Symbol": "V", "Relative Atomic Mass": "45.96019878(36)", "Mass Number": "46"}, "47": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "50.9415(1)", "Atomic Number": "23", "Atomic Symbol": "V", "Relative Atomic Mass": "46.95490491(36)", "Mass Number": "47"}, "48": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "50.9415(1)", "Atomic Number": "23", "Atomic Symbol": "V", "Relative Atomic Mass": "47.9522522(11)", "Mass Number": "48"}, "49": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "50.9415(1)", "Atomic Number": "23", "Atomic Symbol": "V", "Relative Atomic Mass": "48.94851180(96)", "Mass Number": "49"}, "50": {"Isotopic Composition": "0.00250(4)", "Notes": " ", "Standard Atomic Weight": "50.9415(1)", "Atomic Number": "23", "Atomic Symbol": "V", "Relative Atomic Mass": "49.94715601(95)", "Mass Number": "50"}, "51": {"Isotopic Composition": "0.99750(4)", "Notes": " ", "Standard Atomic Weight": "50.9415(1)", "Atomic Number": "23", "Atomic Symbol": "V", "Relative Atomic Mass": "50.94395704(94)", "Mass Number": "51"}, "52": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "50.9415(1)", "Atomic Number": "23", "Atomic Symbol": "V", "Relative Atomic Mass": "51.94477301(95)", "Mass Number": "52"}, "53": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "50.9415(1)", "Atomic Number": "23", "Atomic Symbol": "V", "Relative Atomic Mass": "52.9443367(34)", "Mass Number": "53"}, "54": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "50.9415(1)", "Atomic Number": "23", "Atomic Symbol": "V", "Relative Atomic Mass": "53.946439(16)", "Mass Number": "54"}, "55": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "50.9415(1)", "Atomic Number": "23", "Atomic Symbol": "V", "Relative Atomic Mass": "54.94724(10)", "Mass Number": "55"}, "56": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "50.9415(1)", "Atomic Number": "23", "Atomic Symbol": "V", "Relative Atomic Mass": "55.95048(19)", "Mass Number": "56"}, "57": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "50.9415(1)", "Atomic Number": "23", "Atomic Symbol": "V", "Relative Atomic Mass": "56.95252(24)", "Mass Number": "57"}, "58": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "50.9415(1)", "Atomic Number": "23", "Atomic Symbol": "V", "Relative Atomic Mass": "57.95672(14)", "Mass Number": "58"}, "59": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "50.9415(1)", "Atomic Number": "23", "Atomic Symbol": "V", "Relative Atomic Mass": "58.95939(17)", "Mass Number": "59"}, "60": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "50.9415(1)", "Atomic Number": "23", "Atomic Symbol": "V", "Relative Atomic Mass": "59.96431(24)", "Mass Number": "60"}, "61": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "50.9415(1)", "Atomic Number": "23", "Atomic Symbol": "V", "Relative Atomic Mass": "60.96725(96)", "Mass Number": "61"}, "62": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "50.9415(1)", "Atomic Number": "23", "Atomic Symbol": "V", "Relative Atomic Mass": "61.97265(32#)", "Mass Number": "62"}, "63": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "50.9415(1)", "Atomic Number": "23", "Atomic Symbol": "V", "Relative Atomic Mass": "62.97639(43#)", "Mass Number": "63"}}, "98": {"256": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "98", "Atomic Symbol": "Cf", "Relative Atomic Mass": "256.09344(34#)", "Mass Number": "256"}, "237": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "98", "Atomic Symbol": "Cf", "Relative Atomic Mass": "237.062198(94)", "Mass Number": "237"}, "238": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "98", "Atomic Symbol": "Cf", "Relative Atomic Mass": "238.06149(32#)", "Mass Number": "238"}, "239": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "98", "Atomic Symbol": "Cf", "Relative Atomic Mass": "239.06253(23#)", "Mass Number": "239"}, "240": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "98", "Atomic Symbol": "Cf", "Relative Atomic Mass": "240.062256(20)", "Mass Number": "240"}, "241": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "98", "Atomic Symbol": "Cf", "Relative Atomic Mass": "241.06369(18#)", "Mass Number": "241"}, "242": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "98", "Atomic Symbol": "Cf", "Relative Atomic Mass": "242.063754(14)", "Mass Number": "242"}, "243": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "98", "Atomic Symbol": "Cf", "Relative Atomic Mass": "243.06548(12#)", "Mass Number": "243"}, "244": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "98", "Atomic Symbol": "Cf", "Relative Atomic Mass": "244.0660008(31)", "Mass Number": "244"}, "245": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "98", "Atomic Symbol": "Cf", "Relative Atomic Mass": "245.0680487(30)", "Mass Number": "245"}, "246": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "98", "Atomic Symbol": "Cf", "Relative Atomic Mass": "246.0688055(22)", "Mass Number": "246"}, "247": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "98", "Atomic Symbol": "Cf", "Relative Atomic Mass": "247.070965(16)", "Mass Number": "247"}, "248": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "98", "Atomic Symbol": "Cf", "Relative Atomic Mass": "248.0721851(57)", "Mass Number": "248"}, "249": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "98", "Atomic Symbol": "Cf", "Relative Atomic Mass": "249.0748539(23)", "Mass Number": "249"}, "250": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "98", "Atomic Symbol": "Cf", "Relative Atomic Mass": "250.0764062(22)", "Mass Number": "250"}, "251": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "98", "Atomic Symbol": "Cf", "Relative Atomic Mass": "251.0795886(48)", "Mass Number": "251"}, "252": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "98", "Atomic Symbol": "Cf", "Relative Atomic Mass": "252.0816272(56)", "Mass Number": "252"}, "253": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "98", "Atomic Symbol": "Cf", "Relative Atomic Mass": "253.0851345(67)", "Mass Number": "253"}, "254": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "98", "Atomic Symbol": "Cf", "Relative Atomic Mass": "254.087324(13)", "Mass Number": "254"}, "255": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "98", "Atomic Symbol": "Cf", "Relative Atomic Mass": "255.09105(22#)", "Mass Number": "255"}}, "80": {"171": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "171.00353(33#)", "Mass Number": "171"}, "172": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "171.99881(17)", "Mass Number": "172"}, "173": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "172.99709(22#)", "Mass Number": "173"}, "174": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "173.992865(21)", "Mass Number": "174"}, "175": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "174.991441(78)", "Mass Number": "175"}, "176": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "175.987361(14)", "Mass Number": "176"}, "177": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "176.986277(81)", "Mass Number": "177"}, "178": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "177.982484(12)", "Mass Number": "178"}, "179": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "178.981831(29)", "Mass Number": "179"}, "180": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "179.978260(14)", "Mass Number": "180"}, "181": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "180.977819(17)", "Mass Number": "181"}, "182": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "181.974689(11)", "Mass Number": "182"}, "183": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "182.9744448(76)", "Mass Number": "183"}, "184": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "183.971714(11)", "Mass Number": "184"}, "185": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "184.971899(17)", "Mass Number": "185"}, "186": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "185.969362(13)", "Mass Number": "186"}, "187": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "186.969814(15)", "Mass Number": "187"}, "188": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "187.967567(12)", "Mass Number": "188"}, "189": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "188.968195(34)", "Mass Number": "189"}, "190": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "189.966323(17)", "Mass Number": "190"}, "191": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "190.967157(24)", "Mass Number": "191"}, "192": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "191.965635(17)", "Mass Number": "192"}, "193": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "192.966653(17)", "Mass Number": "193"}, "194": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "193.9654491(31)", "Mass Number": "194"}, "195": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "194.966721(25)", "Mass Number": "195"}, "196": {"Isotopic Composition": "0.0015(1)", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "195.9658326(32)", "Mass Number": "196"}, "197": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "196.9672128(35)", "Mass Number": "197"}, "198": {"Isotopic Composition": "0.0997(20)", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "197.96676860(52)", "Mass Number": "198"}, "199": {"Isotopic Composition": "0.1687(22)", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "198.96828064(46)", "Mass Number": "199"}, "200": {"Isotopic Composition": "0.2310(19)", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "199.96832659(47)", "Mass Number": "200"}, "201": {"Isotopic Composition": "0.1318(9)", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "200.97030284(69)", "Mass Number": "201"}, "202": {"Isotopic Composition": "0.2986(26)", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "201.97064340(69)", "Mass Number": "202"}, "203": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "202.9728728(18)", "Mass Number": "203"}, "204": {"Isotopic Composition": "0.0687(15)", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "203.97349398(53)", "Mass Number": "204"}, "205": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "204.9760734(39)", "Mass Number": "205"}, "206": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "205.977514(22)", "Mass Number": "206"}, "207": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "206.982300(32)", "Mass Number": "207"}, "208": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "207.985759(33)", "Mass Number": "208"}, "209": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "208.99072(16#)", "Mass Number": "209"}, "210": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "209.99424(21#)", "Mass Number": "210"}, "211": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "210.99933(21#)", "Mass Number": "211"}, "212": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "212.00296(32#)", "Mass Number": "212"}, "213": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "213.00823(32#)", "Mass Number": "213"}, "214": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "214.01200(43#)", "Mass Number": "214"}, "215": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "215.01740(43#)", "Mass Number": "215"}, "216": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "200.592(3)", "Atomic Number": "80", "Atomic Symbol": "Hg", "Relative Atomic Mass": "216.02132(43#)", "Mass Number": "216"}}, "114": {"288": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "114", "Atomic Symbol": "Fl", "Relative Atomic Mass": "288.18757(91#)", "Mass Number": "288"}, "289": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "114", "Atomic Symbol": "Fl", "Relative Atomic Mass": "289.19042(60#)", "Mass Number": "289"}, "285": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "114", "Atomic Symbol": "Fl", "Relative Atomic Mass": "285.18364(47#)", "Mass Number": "285"}, "286": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "114", "Atomic Symbol": "Fl", "Relative Atomic Mass": "286.18423(71#)", "Mass Number": "286"}, "287": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "114", "Atomic Symbol": "Fl", "Relative Atomic Mass": "287.18678(66#)", "Mass Number": "287"}}, "45": {"89": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "88.95058(39#)", "Mass Number": "89"}, "90": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "89.94422(43#)", "Mass Number": "90"}, "91": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "90.93688(43#)", "Mass Number": "91"}, "92": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "91.9323677(47)", "Mass Number": "92"}, "93": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "92.9259128(28)", "Mass Number": "93"}, "94": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "93.9217305(36)", "Mass Number": "94"}, "95": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "94.9158979(42)", "Mass Number": "95"}, "96": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "95.914453(11)", "Mass Number": "96"}, "97": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "96.911329(38)", "Mass Number": "97"}, "98": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "97.910708(13)", "Mass Number": "98"}, "99": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "98.9081282(73)", "Mass Number": "99"}, "100": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "99.908117(19)", "Mass Number": "100"}, "101": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "100.9061606(63)", "Mass Number": "101"}, "102": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "101.9068374(50)", "Mass Number": "102"}, "103": {"Isotopic Composition": "1", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "102.9054980(26)", "Mass Number": "103"}, "104": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "103.9066492(26)", "Mass Number": "104"}, "105": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "104.9056885(27)", "Mass Number": "105"}, "106": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "105.9072868(58)", "Mass Number": "106"}, "107": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "106.906748(13)", "Mass Number": "107"}, "108": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "107.908714(15)", "Mass Number": "108"}, "109": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "108.9087488(43)", "Mass Number": "109"}, "110": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "109.911079(19)", "Mass Number": "110"}, "111": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "110.9116423(74)", "Mass Number": "111"}, "112": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "111.914403(47)", "Mass Number": "112"}, "113": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "112.9154393(77)", "Mass Number": "113"}, "114": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "113.918718(77)", "Mass Number": "114"}, "115": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "114.9203116(78)", "Mass Number": "115"}, "116": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "115.924059(76)", "Mass Number": "116"}, "117": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "116.9260354(95)", "Mass Number": "117"}, "118": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "117.930340(26)", "Mass Number": "118"}, "119": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "118.932557(10)", "Mass Number": "119"}, "120": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "119.93686(21#)", "Mass Number": "120"}, "121": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "120.93942(32#)", "Mass Number": "121"}, "122": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "121.94399(32#)", "Mass Number": "122"}, "123": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "122.94685(43#)", "Mass Number": "123"}, "124": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "123.95151(43#)", "Mass Number": "124"}, "125": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "124.95469(54#)", "Mass Number": "125"}, "126": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "102.90550(2)", "Atomic Number": "45", "Atomic Symbol": "Rh", "Relative Atomic Mass": "125.95946(54#)", "Mass Number": "126"}}, "54": {"128": {"Isotopic Composition": "0.019102(8)", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "127.9035310(11)", "Mass Number": "128"}, "129": {"Isotopic Composition": "0.264006(82)", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "128.9047808611(60)", "Mass Number": "129"}, "130": {"Isotopic Composition": "0.040710(13)", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "129.903509349(10)", "Mass Number": "130"}, "131": {"Isotopic Composition": "0.212324(30)", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "130.90508406(24)", "Mass Number": "131"}, "132": {"Isotopic Composition": "0.269086(33)", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "131.9041550856(56)", "Mass Number": "132"}, "133": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "132.9059108(26)", "Mass Number": "133"}, "134": {"Isotopic Composition": "0.104357(21)", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "133.90539466(90)", "Mass Number": "134"}, "135": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "134.9072278(45)", "Mass Number": "135"}, "136": {"Isotopic Composition": "0.088573(44)", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "135.907214484(11)", "Mass Number": "136"}, "137": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "136.91155778(11)", "Mass Number": "137"}, "138": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "137.9141463(30)", "Mass Number": "138"}, "139": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "138.9187922(23)", "Mass Number": "139"}, "140": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "139.9216458(25)", "Mass Number": "140"}, "141": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "140.9267872(31)", "Mass Number": "141"}, "142": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "141.9299731(29)", "Mass Number": "142"}, "143": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "142.9353696(50)", "Mass Number": "143"}, "144": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "143.9389451(57)", "Mass Number": "144"}, "145": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "144.944720(12)", "Mass Number": "145"}, "146": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "145.948518(26)", "Mass Number": "146"}, "147": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "146.95426(21#)", "Mass Number": "147"}, "148": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "147.95813(21#)", "Mass Number": "148"}, "109": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "108.95043(32)", "Mass Number": "109"}, "110": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "109.94426(11)", "Mass Number": "110"}, "111": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "110.941607(93)", "Mass Number": "111"}, "112": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "111.9355590(89)", "Mass Number": "112"}, "113": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "112.9332217(73)", "Mass Number": "113"}, "114": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "113.927980(12)", "Mass Number": "114"}, "115": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "114.926294(13)", "Mass Number": "115"}, "116": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "115.921581(14)", "Mass Number": "116"}, "117": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "116.920359(11)", "Mass Number": "117"}, "118": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "117.916179(11)", "Mass Number": "118"}, "119": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "118.915411(11)", "Mass Number": "119"}, "120": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "119.911784(13)", "Mass Number": "120"}, "121": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "120.911453(11)", "Mass Number": "121"}, "122": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "121.908368(12)", "Mass Number": "122"}, "123": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "122.908482(10)", "Mass Number": "123"}, "124": {"Isotopic Composition": "0.000952(3)", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "123.9058920(19)", "Mass Number": "124"}, "125": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "124.9063944(20)", "Mass Number": "125"}, "126": {"Isotopic Composition": "0.000890(2)", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "125.9042983(38)", "Mass Number": "126"}, "127": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "131.293(6)", "Atomic Number": "54", "Atomic Symbol": "Xe", "Relative Atomic Mass": "126.9051829(44)", "Mass Number": "127"}}, "96": {"232": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "96", "Atomic Symbol": "Cm", "Relative Atomic Mass": "232.04982(22#)", "Mass Number": "232"}, "233": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "96", "Atomic Symbol": "Cm", "Relative Atomic Mass": "233.050770(77)", "Mass Number": "233"}, "234": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "96", "Atomic Symbol": "Cm", "Relative Atomic Mass": "234.050160(20)", "Mass Number": "234"}, "235": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "96", "Atomic Symbol": "Cm", "Relative Atomic Mass": "235.05154(22#)", "Mass Number": "235"}, "236": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "96", "Atomic Symbol": "Cm", "Relative Atomic Mass": "236.051374(20)", "Mass Number": "236"}, "237": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "96", "Atomic Symbol": "Cm", "Relative Atomic Mass": "237.052869(76)", "Mass Number": "237"}, "238": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "96", "Atomic Symbol": "Cm", "Relative Atomic Mass": "238.053081(13)", "Mass Number": "238"}, "239": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "96", "Atomic Symbol": "Cm", "Relative Atomic Mass": "239.054910(58)", "Mass Number": "239"}, "240": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "96", "Atomic Symbol": "Cm", "Relative Atomic Mass": "240.0555297(24)", "Mass Number": "240"}, "241": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "96", "Atomic Symbol": "Cm", "Relative Atomic Mass": "241.0576532(23)", "Mass Number": "241"}, "242": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "96", "Atomic Symbol": "Cm", "Relative Atomic Mass": "242.0588360(19)", "Mass Number": "242"}, "243": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "96", "Atomic Symbol": "Cm", "Relative Atomic Mass": "243.0613893(22)", "Mass Number": "243"}, "244": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "96", "Atomic Symbol": "Cm", "Relative Atomic Mass": "244.0627528(19)", "Mass Number": "244"}, "245": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "96", "Atomic Symbol": "Cm", "Relative Atomic Mass": "245.0654915(22)", "Mass Number": "245"}, "246": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "96", "Atomic Symbol": "Cm", "Relative Atomic Mass": "246.0672238(22)", "Mass Number": "246"}, "247": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "96", "Atomic Symbol": "Cm", "Relative Atomic Mass": "247.0703541(47)", "Mass Number": "247"}, "248": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "96", "Atomic Symbol": "Cm", "Relative Atomic Mass": "248.0723499(56)", "Mass Number": "248"}, "249": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "96", "Atomic Symbol": "Cm", "Relative Atomic Mass": "249.0759548(56)", "Mass Number": "249"}, "250": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "96", "Atomic Symbol": "Cm", "Relative Atomic Mass": "250.078358(12)", "Mass Number": "250"}, "251": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "96", "Atomic Symbol": "Cm", "Relative Atomic Mass": "251.082286(24)", "Mass Number": "251"}, "252": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "96", "Atomic Symbol": "Cm", "Relative Atomic Mass": "252.08487(32#)", "Mass Number": "252"}}, "24": {"64": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "51.9961(6)", "Atomic Number": "24", "Atomic Symbol": "Cr", "Relative Atomic Mass": "63.96408(32#)", "Mass Number": "64"}, "65": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "51.9961(6)", "Atomic Number": "24", "Atomic Symbol": "Cr", "Relative Atomic Mass": "64.96996(32#)", "Mass Number": "65"}, "66": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "51.9961(6)", "Atomic Number": "24", "Atomic Symbol": "Cr", "Relative Atomic Mass": "65.97366(54#)", "Mass Number": "66"}, "67": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "51.9961(6)", "Atomic Number": "24", "Atomic Symbol": "Cr", "Relative Atomic Mass": "66.98016(54#)", "Mass Number": "67"}, "68": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "51.9961(6)", "Atomic Number": "24", "Atomic Symbol": "Cr", "Relative Atomic Mass": "67.98403(75#)", "Mass Number": "68"}, "42": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "51.9961(6)", "Atomic Number": "24", "Atomic Symbol": "Cr", "Relative Atomic Mass": "42.00670(43#)", "Mass Number": "42"}, "43": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "51.9961(6)", "Atomic Number": "24", "Atomic Symbol": "Cr", "Relative Atomic Mass": "42.99753(43#)", "Mass Number": "43"}, "44": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "51.9961(6)", "Atomic Number": "24", "Atomic Symbol": "Cr", "Relative Atomic Mass": "43.98536(32#)", "Mass Number": "44"}, "45": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "51.9961(6)", "Atomic Number": "24", "Atomic Symbol": "Cr", "Relative Atomic Mass": "44.979050(38)", "Mass Number": "45"}, "46": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "51.9961(6)", "Atomic Number": "24", "Atomic Symbol": "Cr", "Relative Atomic Mass": "45.968359(21)", "Mass Number": "46"}, "47": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "51.9961(6)", "Atomic Number": "24", "Atomic Symbol": "Cr", "Relative Atomic Mass": "46.9628974(75)", "Mass Number": "47"}, "48": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "51.9961(6)", "Atomic Number": "24", "Atomic Symbol": "Cr", "Relative Atomic Mass": "47.9540291(79)", "Mass Number": "48"}, "49": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "51.9961(6)", "Atomic Number": "24", "Atomic Symbol": "Cr", "Relative Atomic Mass": "48.9513333(25)", "Mass Number": "49"}, "50": {"Isotopic Composition": "0.04345(13)", "Notes": " ", "Standard Atomic Weight": "51.9961(6)", "Atomic Number": "24", "Atomic Symbol": "Cr", "Relative Atomic Mass": "49.94604183(94)", "Mass Number": "50"}, "51": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "51.9961(6)", "Atomic Number": "24", "Atomic Symbol": "Cr", "Relative Atomic Mass": "50.94476502(94)", "Mass Number": "51"}, "52": {"Isotopic Composition": "0.83789(18)", "Notes": " ", "Standard Atomic Weight": "51.9961(6)", "Atomic Number": "24", "Atomic Symbol": "Cr", "Relative Atomic Mass": "51.94050623(63)", "Mass Number": "52"}, "53": {"Isotopic Composition": "0.09501(17)", "Notes": " ", "Standard Atomic Weight": "51.9961(6)", "Atomic Number": "24", "Atomic Symbol": "Cr", "Relative Atomic Mass": "52.94064815(62)", "Mass Number": "53"}, "54": {"Isotopic Composition": "0.02365(7)", "Notes": " ", "Standard Atomic Weight": "51.9961(6)", "Atomic Number": "24", "Atomic Symbol": "Cr", "Relative Atomic Mass": "53.93887916(61)", "Mass Number": "54"}, "55": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "51.9961(6)", "Atomic Number": "24", "Atomic Symbol": "Cr", "Relative Atomic Mass": "54.94083843(64)", "Mass Number": "55"}, "56": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "51.9961(6)", "Atomic Number": "24", "Atomic Symbol": "Cr", "Relative Atomic Mass": "55.9406531(20)", "Mass Number": "56"}, "57": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "51.9961(6)", "Atomic Number": "24", "Atomic Symbol": "Cr", "Relative Atomic Mass": "56.9436130(20)", "Mass Number": "57"}, "58": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "51.9961(6)", "Atomic Number": "24", "Atomic Symbol": "Cr", "Relative Atomic Mass": "57.94435(22)", "Mass Number": "58"}, "59": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "51.9961(6)", "Atomic Number": "24", "Atomic Symbol": "Cr", "Relative Atomic Mass": "58.94859(26)", "Mass Number": "59"}, "60": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "51.9961(6)", "Atomic Number": "24", "Atomic Symbol": "Cr", "Relative Atomic Mass": "59.95008(23)", "Mass Number": "60"}, "61": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "51.9961(6)", "Atomic Number": "24", "Atomic Symbol": "Cr", "Relative Atomic Mass": "60.95442(14)", "Mass Number": "61"}, "62": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "51.9961(6)", "Atomic Number": "24", "Atomic Symbol": "Cr", "Relative Atomic Mass": "61.95610(16)", "Mass Number": "62"}, "63": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "51.9961(6)", "Atomic Number": "24", "Atomic Symbol": "Cr", "Relative Atomic Mass": "62.96165(49)", "Mass Number": "63"}}, "30": {"64": {"Isotopic Composition": "0.4917(75)", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "63.92914201(71)", "Mass Number": "64"}, "65": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "64.92924077(71)", "Mass Number": "65"}, "66": {"Isotopic Composition": "0.2773(98)", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "65.92603381(94)", "Mass Number": "66"}, "67": {"Isotopic Composition": "0.0404(16)", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "66.92712775(96)", "Mass Number": "67"}, "68": {"Isotopic Composition": "0.1845(63)", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "67.92484455(98)", "Mass Number": "68"}, "69": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "68.9265507(10)", "Mass Number": "69"}, "70": {"Isotopic Composition": "0.0061(10)", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "69.9253192(21)", "Mass Number": "70"}, "71": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "70.9277196(28)", "Mass Number": "71"}, "72": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "71.9268428(23)", "Mass Number": "72"}, "73": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "72.9295826(20)", "Mass Number": "73"}, "74": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "73.9294073(27)", "Mass Number": "74"}, "75": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "74.9328402(21)", "Mass Number": "75"}, "76": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "75.9331150(16)", "Mass Number": "76"}, "77": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "76.9368872(21)", "Mass Number": "77"}, "78": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "77.9382892(21)", "Mass Number": "78"}, "79": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "78.9426381(24)", "Mass Number": "79"}, "80": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "79.9445529(28)", "Mass Number": "80"}, "81": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "80.9504026(54)", "Mass Number": "81"}, "82": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "81.95426(32#)", "Mass Number": "82"}, "83": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "82.96056(54#)", "Mass Number": "83"}, "84": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "83.96521(64#)", "Mass Number": "84"}, "85": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "84.97226(75#)", "Mass Number": "85"}, "54": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "53.99204(75#)", "Mass Number": "54"}, "55": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "54.98398(75#)", "Mass Number": "55"}, "56": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "55.97254(54#)", "Mass Number": "56"}, "57": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "56.96506(22#)", "Mass Number": "57"}, "58": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "57.954591(54)", "Mass Number": "58"}, "59": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "58.94931266(89)", "Mass Number": "59"}, "60": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "59.94184210(69)", "Mass Number": "60"}, "61": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "60.939507(17)", "Mass Number": "61"}, "62": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "61.93433397(73)", "Mass Number": "62"}, "63": {"Isotopic Composition": "", "Notes": "r", "Standard Atomic Weight": "65.38(2)", "Atomic Number": "30", "Atomic Symbol": "Zn", "Relative Atomic Mass": "62.9332115(17)", "Mass Number": "63"}}, "52": {"128": {"Isotopic Composition": "0.3174(8)", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "127.90446128(93)", "Mass Number": "128"}, "129": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "128.90659646(93)", "Mass Number": "129"}, "130": {"Isotopic Composition": "0.3408(62)", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "129.906222748(12)", "Mass Number": "130"}, "131": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "130.908522213(65)", "Mass Number": "131"}, "132": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "131.9085467(37)", "Mass Number": "132"}, "133": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "132.9109688(39)", "Mass Number": "133"}, "134": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "133.9113940(30)", "Mass Number": "134"}, "135": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "134.9165557(29)", "Mass Number": "135"}, "136": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "135.9201006(26)", "Mass Number": "136"}, "137": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "136.9255989(27)", "Mass Number": "137"}, "138": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "137.9294722(47)", "Mass Number": "138"}, "139": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "138.9353672(38)", "Mass Number": "139"}, "140": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "139.939499(30)", "Mass Number": "140"}, "141": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "140.94580(43#)", "Mass Number": "141"}, "142": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "141.95022(54#)", "Mass Number": "142"}, "143": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "142.95676(54#)", "Mass Number": "143"}, "105": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "104.94330(32)", "Mass Number": "105"}, "106": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "105.93750(11)", "Mass Number": "106"}, "107": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "106.935012(76)", "Mass Number": "107"}, "108": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "107.9293805(58)", "Mass Number": "108"}, "109": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "108.9273045(47)", "Mass Number": "109"}, "110": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "109.9224581(71)", "Mass Number": "110"}, "111": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "110.9210006(69)", "Mass Number": "111"}, "112": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "111.9167279(90)", "Mass Number": "112"}, "113": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "112.915891(30)", "Mass Number": "113"}, "114": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "113.912089(30)", "Mass Number": "114"}, "115": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "114.911902(30)", "Mass Number": "115"}, "116": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "115.908460(30)", "Mass Number": "116"}, "117": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "116.908646(14)", "Mass Number": "117"}, "118": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "117.905854(20)", "Mass Number": "118"}, "119": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "118.9064071(85)", "Mass Number": "119"}, "120": {"Isotopic Composition": "0.0009(1)", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "119.9040593(33)", "Mass Number": "120"}, "121": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "120.904944(28)", "Mass Number": "121"}, "122": {"Isotopic Composition": "0.0255(12)", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "121.9030435(16)", "Mass Number": "122"}, "123": {"Isotopic Composition": "0.0089(3)", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "122.9042698(16)", "Mass Number": "123"}, "124": {"Isotopic Composition": "0.0474(14)", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "123.9028171(16)", "Mass Number": "124"}, "125": {"Isotopic Composition": "0.0707(15)", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "124.9044299(16)", "Mass Number": "125"}, "126": {"Isotopic Composition": "0.1884(25)", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "125.9033109(16)", "Mass Number": "126"}, "127": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "127.60(3)", "Atomic Number": "52", "Atomic Symbol": "Te", "Relative Atomic Mass": "126.9052257(16)", "Mass Number": "127"}}, "62": {"128": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "127.95842(54#)", "Mass Number": "128"}, "129": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "128.95476(54#)", "Mass Number": "129"}, "130": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "129.94900(43#)", "Mass Number": "130"}, "131": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "130.94618(43#)", "Mass Number": "131"}, "132": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "131.94087(32#)", "Mass Number": "132"}, "133": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "132.93856(32#)", "Mass Number": "133"}, "134": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "133.93411(21#)", "Mass Number": "134"}, "135": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "134.93252(17)", "Mass Number": "135"}, "136": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "135.928276(13)", "Mass Number": "136"}, "137": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "136.926971(46)", "Mass Number": "137"}, "138": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "137.923244(13)", "Mass Number": "138"}, "139": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "138.922297(12)", "Mass Number": "139"}, "140": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "139.918995(13)", "Mass Number": "140"}, "141": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "140.9184816(92)", "Mass Number": "141"}, "142": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "141.9152044(36)", "Mass Number": "142"}, "143": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "142.9146353(33)", "Mass Number": "143"}, "144": {"Isotopic Composition": "0.0307(7)", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "143.9120065(21)", "Mass Number": "144"}, "145": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "144.9134173(21)", "Mass Number": "145"}, "146": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "145.9130470(35)", "Mass Number": "146"}, "147": {"Isotopic Composition": "0.1499(18)", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "146.9149044(19)", "Mass Number": "147"}, "148": {"Isotopic Composition": "0.1124(10)", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "147.9148292(19)", "Mass Number": "148"}, "149": {"Isotopic Composition": "0.1382(7)", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "148.9171921(18)", "Mass Number": "149"}, "150": {"Isotopic Composition": "0.0738(1)", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "149.9172829(18)", "Mass Number": "150"}, "151": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "150.9199398(18)", "Mass Number": "151"}, "152": {"Isotopic Composition": "0.2675(16)", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "151.9197397(18)", "Mass Number": "152"}, "153": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "152.9221047(18)", "Mass Number": "153"}, "154": {"Isotopic Composition": "0.2275(29)", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "153.9222169(20)", "Mass Number": "154"}, "155": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "154.9246477(20)", "Mass Number": "155"}, "156": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "155.925536(10)", "Mass Number": "156"}, "157": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "156.9284187(48)", "Mass Number": "157"}, "158": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "157.9299510(53)", "Mass Number": "158"}, "159": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "158.9332172(64)", "Mass Number": "159"}, "160": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "159.9353353(64)", "Mass Number": "160"}, "161": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "160.9391602(73)", "Mass Number": "161"}, "162": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "161.94146(21#)", "Mass Number": "162"}, "163": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "162.94555(32#)", "Mass Number": "163"}, "164": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "163.94836(32#)", "Mass Number": "164"}, "165": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "150.36(2)", "Atomic Number": "62", "Atomic Symbol": "Sm", "Relative Atomic Mass": "164.95297(43#)", "Mass Number": "165"}}, "117": {"291": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "117", "Atomic Symbol": "Ts", "Relative Atomic Mass": "291.20553(68#)", "Mass Number": "291"}, "292": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "117", "Atomic Symbol": "Ts", "Relative Atomic Mass": "292.20746(75#)", "Mass Number": "292"}, "293": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "117", "Atomic Symbol": "Ts", "Relative Atomic Mass": "293.20824(89#)", "Mass Number": "293"}, "294": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "117", "Atomic Symbol": "Uus", "Relative Atomic Mass": "294.21046(74#)", "Mass Number": "294"}}, "22": {"38": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "47.867(1)", "Atomic Number": "22", "Atomic Symbol": "Ti", "Relative Atomic Mass": "38.01145(32#)", "Mass Number": "38"}, "39": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "47.867(1)", "Atomic Number": "22", "Atomic Symbol": "Ti", "Relative Atomic Mass": "39.00236(22#)", "Mass Number": "39"}, "40": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "47.867(1)", "Atomic Number": "22", "Atomic Symbol": "Ti", "Relative Atomic Mass": "39.99050(17)", "Mass Number": "40"}, "41": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "47.867(1)", "Atomic Number": "22", "Atomic Symbol": "Ti", "Relative Atomic Mass": "40.983148(30)", "Mass Number": "41"}, "42": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "47.867(1)", "Atomic Number": "22", "Atomic Symbol": "Ti", "Relative Atomic Mass": "41.97304903(30)", "Mass Number": "42"}, "43": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "47.867(1)", "Atomic Number": "22", "Atomic Symbol": "Ti", "Relative Atomic Mass": "42.9685225(78)", "Mass Number": "43"}, "44": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "47.867(1)", "Atomic Number": "22", "Atomic Symbol": "Ti", "Relative Atomic Mass": "43.95968995(75)", "Mass Number": "44"}, "45": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "47.867(1)", "Atomic Number": "22", "Atomic Symbol": "Ti", "Relative Atomic Mass": "44.95812198(95)", "Mass Number": "45"}, "46": {"Isotopic Composition": "0.0825(3)", "Notes": " ", "Standard Atomic Weight": "47.867(1)", "Atomic Number": "22", "Atomic Symbol": "Ti", "Relative Atomic Mass": "45.95262772(35)", "Mass Number": "46"}, "47": {"Isotopic Composition": "0.0744(2)", "Notes": " ", "Standard Atomic Weight": "47.867(1)", "Atomic Number": "22", "Atomic Symbol": "Ti", "Relative Atomic Mass": "46.95175879(38)", "Mass Number": "47"}, "48": {"Isotopic Composition": "0.7372(3)", "Notes": " ", "Standard Atomic Weight": "47.867(1)", "Atomic Number": "22", "Atomic Symbol": "Ti", "Relative Atomic Mass": "47.94794198(38)", "Mass Number": "48"}, "49": {"Isotopic Composition": "0.0541(2)", "Notes": " ", "Standard Atomic Weight": "47.867(1)", "Atomic Number": "22", "Atomic Symbol": "Ti", "Relative Atomic Mass": "48.94786568(39)", "Mass Number": "49"}, "50": {"Isotopic Composition": "0.0518(2)", "Notes": " ", "Standard Atomic Weight": "47.867(1)", "Atomic Number": "22", "Atomic Symbol": "Ti", "Relative Atomic Mass": "49.94478689(39)", "Mass Number": "50"}, "51": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "47.867(1)", "Atomic Number": "22", "Atomic Symbol": "Ti", "Relative Atomic Mass": "50.94661065(65)", "Mass Number": "51"}, "52": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "47.867(1)", "Atomic Number": "22", "Atomic Symbol": "Ti", "Relative Atomic Mass": "51.9468930(76)", "Mass Number": "52"}, "53": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "47.867(1)", "Atomic Number": "22", "Atomic Symbol": "Ti", "Relative Atomic Mass": "52.94973(11)", "Mass Number": "53"}, "54": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "47.867(1)", "Atomic Number": "22", "Atomic Symbol": "Ti", "Relative Atomic Mass": "53.95105(13)", "Mass Number": "54"}, "55": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "47.867(1)", "Atomic Number": "22", "Atomic Symbol": "Ti", "Relative Atomic Mass": "54.95527(17)", "Mass Number": "55"}, "56": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "47.867(1)", "Atomic Number": "22", "Atomic Symbol": "Ti", "Relative Atomic Mass": "55.95791(15)", "Mass Number": "56"}, "57": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "47.867(1)", "Atomic Number": "22", "Atomic Symbol": "Ti", "Relative Atomic Mass": "56.96364(27)", "Mass Number": "57"}, "58": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "47.867(1)", "Atomic Number": "22", "Atomic Symbol": "Ti", "Relative Atomic Mass": "57.96660(43#)", "Mass Number": "58"}, "59": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "47.867(1)", "Atomic Number": "22", "Atomic Symbol": "Ti", "Relative Atomic Mass": "58.97247(43#)", "Mass Number": "59"}, "60": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "47.867(1)", "Atomic Number": "22", "Atomic Symbol": "Ti", "Relative Atomic Mass": "59.97603(54#)", "Mass Number": "60"}, "61": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "47.867(1)", "Atomic Number": "22", "Atomic Symbol": "Ti", "Relative Atomic Mass": "60.98245(64#)", "Mass Number": "61"}, "62": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "47.867(1)", "Atomic Number": "22", "Atomic Symbol": "Ti", "Relative Atomic Mass": "61.98651(75#)", "Mass Number": "62"}, "63": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "47.867(1)", "Atomic Number": "22", "Atomic Symbol": "Ti", "Relative Atomic Mass": "62.99375(75#)", "Mass Number": "63"}}, "55": {"128": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "127.9077487(58)", "Mass Number": "128"}, "129": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "128.9060657(49)", "Mass Number": "129"}, "130": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "129.9067093(90)", "Mass Number": "130"}, "131": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "130.9054649(53)", "Mass Number": "131"}, "132": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "131.9064339(21)", "Mass Number": "132"}, "133": {"Isotopic Composition": "1", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "132.9054519610(80)", "Mass Number": "133"}, "134": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "133.906718503(17)", "Mass Number": "134"}, "135": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "134.9059770(11)", "Mass Number": "135"}, "136": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "135.9073114(20)", "Mass Number": "136"}, "137": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "136.90708923(36)", "Mass Number": "137"}, "138": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "137.9110171(98)", "Mass Number": "138"}, "139": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "138.9133638(34)", "Mass Number": "139"}, "140": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "139.9172831(88)", "Mass Number": "140"}, "141": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "140.9200455(98)", "Mass Number": "141"}, "142": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "141.9242960(79)", "Mass Number": "142"}, "143": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "142.927349(24)", "Mass Number": "143"}, "144": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "143.932076(27)", "Mass Number": "144"}, "145": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "144.935527(12)", "Mass Number": "145"}, "146": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "145.940344(42)", "Mass Number": "146"}, "147": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "146.944156(57)", "Mass Number": "147"}, "148": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "147.94923(62)", "Mass Number": "148"}, "149": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "148.95302(21#)", "Mass Number": "149"}, "150": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "149.95833(32#)", "Mass Number": "150"}, "151": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "150.96258(43#)", "Mass Number": "151"}, "112": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "111.950309(93)", "Mass Number": "112"}, "113": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "112.9444291(93)", "Mass Number": "113"}, "114": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "113.941296(76)", "Mass Number": "114"}, "115": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "114.93591(32#)", "Mass Number": "115"}, "116": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "115.93337(11#)", "Mass Number": "116"}, "117": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "116.928617(67)", "Mass Number": "117"}, "118": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "117.926560(14)", "Mass Number": "118"}, "119": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "118.922377(15)", "Mass Number": "119"}, "120": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "119.920677(11)", "Mass Number": "120"}, "121": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "120.917227(15)", "Mass Number": "121"}, "122": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "121.916108(36)", "Mass Number": "122"}, "123": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "122.912996(13)", "Mass Number": "123"}, "124": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "123.9122578(89)", "Mass Number": "124"}, "125": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "124.9097280(83)", "Mass Number": "125"}, "126": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "125.909446(11)", "Mass Number": "126"}, "127": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "132.90545196(6)", "Atomic Number": "55", "Atomic Symbol": "Cs", "Relative Atomic Mass": "126.9074174(60)", "Mass Number": "127"}}, "49": {"128": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "127.92040(16)", "Mass Number": "128"}, "129": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "128.9218053(29)", "Mass Number": "129"}, "130": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "129.924977(41)", "Mass Number": "130"}, "131": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "130.9269715(29)", "Mass Number": "131"}, "132": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "131.933001(64)", "Mass Number": "132"}, "133": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "132.93831(21#)", "Mass Number": "133"}, "134": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "133.94454(32#)", "Mass Number": "134"}, "135": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "134.95005(43#)", "Mass Number": "135"}, "97": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "96.94934(54#)", "Mass Number": "97"}, "98": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "97.94214(21#)", "Mass Number": "98"}, "99": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "98.93411(21#)", "Mass Number": "99"}, "100": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "99.93096(20)", "Mass Number": "100"}, "101": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "100.92634(32#)", "Mass Number": "101"}, "102": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "101.9241071(49)", "Mass Number": "102"}, "103": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "102.9198819(98)", "Mass Number": "103"}, "104": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "103.9182145(62)", "Mass Number": "104"}, "105": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "104.914502(11)", "Mass Number": "105"}, "106": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "105.913464(13)", "Mass Number": "106"}, "107": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "106.910290(12)", "Mass Number": "107"}, "108": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "107.9096935(93)", "Mass Number": "108"}, "109": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "108.9071514(43)", "Mass Number": "109"}, "110": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "109.907170(12)", "Mass Number": "110"}, "111": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "110.9051085(38)", "Mass Number": "111"}, "112": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "111.9055377(46)", "Mass Number": "112"}, "113": {"Isotopic Composition": "0.0429(5)", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "112.90406184(91)", "Mass Number": "113"}, "114": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "113.90491791(94)", "Mass Number": "114"}, "115": {"Isotopic Composition": "0.9571(5)", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "114.903878776(12)", "Mass Number": "115"}, "116": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "115.90525999(24)", "Mass Number": "116"}, "117": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "116.9045157(52)", "Mass Number": "117"}, "118": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "117.9063566(83)", "Mass Number": "118"}, "119": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "118.9058507(78)", "Mass Number": "119"}, "120": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "119.907967(43)", "Mass Number": "120"}, "121": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "120.907851(29)", "Mass Number": "121"}, "122": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "121.910281(54)", "Mass Number": "122"}, "123": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "122.910434(21)", "Mass Number": "123"}, "124": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "123.913182(33)", "Mass Number": "124"}, "125": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "124.913605(29)", "Mass Number": "125"}, "126": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "125.916507(29)", "Mass Number": "126"}, "127": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "114.818(1)", "Atomic Number": "49", "Atomic Symbol": "In", "Relative Atomic Mass": "126.917446(23)", "Mass Number": "127"}}, "79": {"192": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "191.964814(17)", "Mass Number": "192"}, "193": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "192.9641373(93)", "Mass Number": "193"}, "194": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "193.9654178(23)", "Mass Number": "194"}, "195": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "194.9650352(15)", "Mass Number": "195"}, "196": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "195.9665699(32)", "Mass Number": "196"}, "197": {"Isotopic Composition": "1", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "196.96656879(71)", "Mass Number": "197"}, "198": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "197.96824242(70)", "Mass Number": "198"}, "199": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "198.96876528(70)", "Mass Number": "199"}, "200": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "199.970756(29)", "Mass Number": "200"}, "201": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "200.9716575(34)", "Mass Number": "201"}, "202": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "201.973856(25)", "Mass Number": "202"}, "203": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "202.9751544(33)", "Mass Number": "203"}, "204": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "203.97783(22#)", "Mass Number": "204"}, "205": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "204.97985(21#)", "Mass Number": "205"}, "206": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "205.98474(32#)", "Mass Number": "206"}, "207": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "206.98840(32#)", "Mass Number": "207"}, "208": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "207.99345(32#)", "Mass Number": "208"}, "209": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "208.99735(43#)", "Mass Number": "209"}, "210": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "210.00250(43#)", "Mass Number": "210"}, "169": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "168.99808(32#)", "Mass Number": "169"}, "170": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "169.99597(22#)", "Mass Number": "170"}, "171": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "170.991876(22)", "Mass Number": "171"}, "172": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "171.989942(81)", "Mass Number": "172"}, "173": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "172.986241(26)", "Mass Number": "173"}, "174": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "173.984717(95#)", "Mass Number": "174"}, "175": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "174.981304(42)", "Mass Number": "175"}, "176": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "175.980250(36)", "Mass Number": "176"}, "177": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "176.976870(11)", "Mass Number": "177"}, "178": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "177.976032(61)", "Mass Number": "178"}, "179": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "178.973174(13)", "Mass Number": "179"}, "180": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "179.972523(21)", "Mass Number": "180"}, "181": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "180.970079(21)", "Mass Number": "181"}, "182": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "181.969618(22)", "Mass Number": "182"}, "183": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "182.967591(10)", "Mass Number": "183"}, "184": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "183.967452(24)", "Mass Number": "184"}, "185": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "184.965790(28)", "Mass Number": "185"}, "186": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "185.965953(23)", "Mass Number": "186"}, "187": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "186.964543(24)", "Mass Number": "187"}, "188": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "187.965349(17)", "Mass Number": "188"}, "189": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "188.963948(22)", "Mass Number": "189"}, "190": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "189.964698(17)", "Mass Number": "190"}, "191": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "196.966569(5)", "Atomic Number": "79", "Atomic Symbol": "Au", "Relative Atomic Mass": "190.963702(40)", "Mass Number": "191"}}, "20": {"34": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "40.078(4)", "Atomic Number": "20", "Atomic Symbol": "Ca", "Relative Atomic Mass": "34.01487(32#)", "Mass Number": "34"}, "35": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "40.078(4)", "Atomic Number": "20", "Atomic Symbol": "Ca", "Relative Atomic Mass": "35.00514(21#)", "Mass Number": "35"}, "36": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "40.078(4)", "Atomic Number": "20", "Atomic Symbol": "Ca", "Relative Atomic Mass": "35.993074(43)", "Mass Number": "36"}, "37": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "40.078(4)", "Atomic Number": "20", "Atomic Symbol": "Ca", "Relative Atomic Mass": "36.98589785(68)", "Mass Number": "37"}, "38": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "40.078(4)", "Atomic Number": "20", "Atomic Symbol": "Ca", "Relative Atomic Mass": "37.97631922(21)", "Mass Number": "38"}, "39": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "40.078(4)", "Atomic Number": "20", "Atomic Symbol": "Ca", "Relative Atomic Mass": "38.97071081(64)", "Mass Number": "39"}, "40": {"Isotopic Composition": "0.96941(156)", "Notes": "g", "Standard Atomic Weight": "40.078(4)", "Atomic Number": "20", "Atomic Symbol": "Ca", "Relative Atomic Mass": "39.962590863(22)", "Mass Number": "40"}, "41": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "40.078(4)", "Atomic Number": "20", "Atomic Symbol": "Ca", "Relative Atomic Mass": "40.96227792(15)", "Mass Number": "41"}, "42": {"Isotopic Composition": "0.00647(23)", "Notes": "g", "Standard Atomic Weight": "40.078(4)", "Atomic Number": "20", "Atomic Symbol": "Ca", "Relative Atomic Mass": "41.95861783(16)", "Mass Number": "42"}, "43": {"Isotopic Composition": "0.00135(10)", "Notes": "g", "Standard Atomic Weight": "40.078(4)", "Atomic Number": "20", "Atomic Symbol": "Ca", "Relative Atomic Mass": "42.95876644(24)", "Mass Number": "43"}, "44": {"Isotopic Composition": "0.02086(110)", "Notes": "g", "Standard Atomic Weight": "40.078(4)", "Atomic Number": "20", "Atomic Symbol": "Ca", "Relative Atomic Mass": "43.95548156(35)", "Mass Number": "44"}, "45": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "40.078(4)", "Atomic Number": "20", "Atomic Symbol": "Ca", "Relative Atomic Mass": "44.95618635(39)", "Mass Number": "45"}, "46": {"Isotopic Composition": "0.00004(3)", "Notes": "g", "Standard Atomic Weight": "40.078(4)", "Atomic Number": "20", "Atomic Symbol": "Ca", "Relative Atomic Mass": "45.9536890(24)", "Mass Number": "46"}, "47": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "40.078(4)", "Atomic Number": "20", "Atomic Symbol": "Ca", "Relative Atomic Mass": "46.9545424(24)", "Mass Number": "47"}, "48": {"Isotopic Composition": "0.00187(21)", "Notes": "g", "Standard Atomic Weight": "40.078(4)", "Atomic Number": "20", "Atomic Symbol": "Ca", "Relative Atomic Mass": "47.95252276(13)", "Mass Number": "48"}, "49": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "40.078(4)", "Atomic Number": "20", "Atomic Symbol": "Ca", "Relative Atomic Mass": "48.95566274(23)", "Mass Number": "49"}, "50": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "40.078(4)", "Atomic Number": "20", "Atomic Symbol": "Ca", "Relative Atomic Mass": "49.9574992(17)", "Mass Number": "50"}, "51": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "40.078(4)", "Atomic Number": "20", "Atomic Symbol": "Ca", "Relative Atomic Mass": "50.960989(24)", "Mass Number": "51"}, "52": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "40.078(4)", "Atomic Number": "20", "Atomic Symbol": "Ca", "Relative Atomic Mass": "51.963217(64)", "Mass Number": "52"}, "53": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "40.078(4)", "Atomic Number": "20", "Atomic Symbol": "Ca", "Relative Atomic Mass": "52.96945(43#)", "Mass Number": "53"}, "54": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "40.078(4)", "Atomic Number": "20", "Atomic Symbol": "Ca", "Relative Atomic Mass": "53.97340(54#)", "Mass Number": "54"}, "55": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "40.078(4)", "Atomic Number": "20", "Atomic Symbol": "Ca", "Relative Atomic Mass": "54.98030(54#)", "Mass Number": "55"}, "56": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "40.078(4)", "Atomic Number": "20", "Atomic Symbol": "Ca", "Relative Atomic Mass": "55.98508(64#)", "Mass Number": "56"}, "57": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "40.078(4)", "Atomic Number": "20", "Atomic Symbol": "Ca", "Relative Atomic Mass": "56.99262(64#)", "Mass Number": "57"}, "58": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "40.078(4)", "Atomic Number": "20", "Atomic Symbol": "Ca", "Relative Atomic Mass": "57.99794(75#)", "Mass Number": "58"}}, "95": {"230": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "95", "Atomic Symbol": "Am", "Relative Atomic Mass": "230.04609(14#)", "Mass Number": "230"}, "231": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "95", "Atomic Symbol": "Am", "Relative Atomic Mass": "231.04556(32#)", "Mass Number": "231"}, "232": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "95", "Atomic Symbol": "Am", "Relative Atomic Mass": "232.04645(32#)", "Mass Number": "232"}, "233": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "95", "Atomic Symbol": "Am", "Relative Atomic Mass": "233.04644(11#)", "Mass Number": "233"}, "234": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "95", "Atomic Symbol": "Am", "Relative Atomic Mass": "234.04773(17#)", "Mass Number": "234"}, "235": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "95", "Atomic Symbol": "Am", "Relative Atomic Mass": "235.047908(56)", "Mass Number": "235"}, "236": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "95", "Atomic Symbol": "Am", "Relative Atomic Mass": "236.04943(12#)", "Mass Number": "236"}, "237": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "95", "Atomic Symbol": "Am", "Relative Atomic Mass": "237.049996(64#)", "Mass Number": "237"}, "238": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "95", "Atomic Symbol": "Am", "Relative Atomic Mass": "238.051985(54)", "Mass Number": "238"}, "239": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "95", "Atomic Symbol": "Am", "Relative Atomic Mass": "239.0530247(26)", "Mass Number": "239"}, "240": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "95", "Atomic Symbol": "Am", "Relative Atomic Mass": "240.055300(15)", "Mass Number": "240"}, "241": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "95", "Atomic Symbol": "Am", "Relative Atomic Mass": "241.0568293(19)", "Mass Number": "241"}, "242": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "95", "Atomic Symbol": "Am", "Relative Atomic Mass": "242.0595494(19)", "Mass Number": "242"}, "243": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "95", "Atomic Symbol": "Am", "Relative Atomic Mass": "243.0613813(24)", "Mass Number": "243"}, "244": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "95", "Atomic Symbol": "Am", "Relative Atomic Mass": "244.0642851(22)", "Mass Number": "244"}, "245": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "95", "Atomic Symbol": "Am", "Relative Atomic Mass": "245.0664548(34)", "Mass Number": "245"}, "246": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "95", "Atomic Symbol": "Am", "Relative Atomic Mass": "246.069775(20#)", "Mass Number": "246"}, "247": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "95", "Atomic Symbol": "Am", "Relative Atomic Mass": "247.07209(11#)", "Mass Number": "247"}, "248": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "95", "Atomic Symbol": "Am", "Relative Atomic Mass": "248.07575(22#)", "Mass Number": "248"}, "249": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "95", "Atomic Symbol": "Am", "Relative Atomic Mass": "249.07848(32#)", "Mass Number": "249"}}, "39": {"76": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "75.95856(54#)", "Mass Number": "76"}, "77": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "76.949781(65#)", "Mass Number": "77"}, "78": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "77.94361(43#)", "Mass Number": "78"}, "79": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "78.93735(48)", "Mass Number": "79"}, "80": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "79.9343561(67)", "Mass Number": "80"}, "81": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "80.9294556(58)", "Mass Number": "81"}, "82": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "81.9269314(59)", "Mass Number": "82"}, "83": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "82.922485(20)", "Mass Number": "83"}, "84": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "83.9206721(46)", "Mass Number": "84"}, "85": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "84.916433(20)", "Mass Number": "85"}, "86": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "85.914886(15)", "Mass Number": "86"}, "87": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "86.9108761(17)", "Mass Number": "87"}, "88": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "87.9095016(20)", "Mass Number": "88"}, "89": {"Isotopic Composition": "1", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "88.9058403(24)", "Mass Number": "89"}, "90": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "89.9071439(24)", "Mass Number": "90"}, "91": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "90.9072974(28)", "Mass Number": "91"}, "92": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "91.9089451(99)", "Mass Number": "92"}, "93": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "92.909578(11)", "Mass Number": "93"}, "94": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "93.9115906(69)", "Mass Number": "94"}, "95": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "94.9128161(74)", "Mass Number": "95"}, "96": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "95.9158968(69)", "Mass Number": "96"}, "97": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "96.9182741(75)", "Mass Number": "97"}, "98": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "97.9223821(88)", "Mass Number": "98"}, "99": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "98.9241480(74)", "Mass Number": "99"}, "100": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "99.927715(12)", "Mass Number": "100"}, "101": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "100.9301477(79)", "Mass Number": "101"}, "102": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "101.9343277(44)", "Mass Number": "102"}, "103": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "102.937243(12)", "Mass Number": "103"}, "104": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "103.94196(43#)", "Mass Number": "104"}, "105": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "104.94544(54#)", "Mass Number": "105"}, "106": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "105.95056(54#)", "Mass Number": "106"}, "107": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "106.95452(54#)", "Mass Number": "107"}, "108": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "107.95996(64#)", "Mass Number": "108"}, "109": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "88.90584(2)", "Atomic Number": "39", "Atomic Symbol": "Y", "Relative Atomic Mass": "108.96436(75#)", "Mass Number": "109"}}, "86": {"193": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "193.009708(27)", "Mass Number": "193"}, "194": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "194.006144(18)", "Mass Number": "194"}, "195": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "195.005422(54)", "Mass Number": "195"}, "196": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "196.002116(15)", "Mass Number": "196"}, "197": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "197.001585(38)", "Mass Number": "197"}, "198": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "197.998679(14)", "Mass Number": "198"}, "199": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "198.998390(68)", "Mass Number": "199"}, "200": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "199.995690(14)", "Mass Number": "200"}, "201": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "200.995628(53)", "Mass Number": "201"}, "202": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "201.993264(19)", "Mass Number": "202"}, "203": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "202.993388(25)", "Mass Number": "203"}, "204": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "203.991430(16)", "Mass Number": "204"}, "205": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "204.991719(54)", "Mass Number": "205"}, "206": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "205.990214(16)", "Mass Number": "206"}, "207": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "206.9907303(91)", "Mass Number": "207"}, "208": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "207.989635(12)", "Mass Number": "208"}, "209": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "208.990415(22)", "Mass Number": "209"}, "210": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "209.9896891(49)", "Mass Number": "210"}, "211": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "210.9906011(73)", "Mass Number": "211"}, "212": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "211.9907039(34)", "Mass Number": "212"}, "213": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "212.9938831(61)", "Mass Number": "213"}, "214": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "213.9953630(99)", "Mass Number": "214"}, "215": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "214.9987459(83)", "Mass Number": "215"}, "216": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "216.0002719(65)", "Mass Number": "216"}, "217": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "217.0039280(45)", "Mass Number": "217"}, "218": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "218.0056016(25)", "Mass Number": "218"}, "219": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "219.0094804(27)", "Mass Number": "219"}, "220": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "220.0113941(23)", "Mass Number": "220"}, "221": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "221.0155371(63)", "Mass Number": "221"}, "222": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "222.0175782(25)", "Mass Number": "222"}, "223": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "223.0218893(84)", "Mass Number": "223"}, "224": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "224.024096(11)", "Mass Number": "224"}, "225": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "225.028486(12)", "Mass Number": "225"}, "226": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "226.030861(11)", "Mass Number": "226"}, "227": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "227.035304(15)", "Mass Number": "227"}, "228": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "228.037835(19)", "Mass Number": "228"}, "229": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "229.042257(14)", "Mass Number": "229"}, "230": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "230.04514(21#)", "Mass Number": "230"}, "231": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[222]", "Atomic Number": "86", "Atomic Symbol": "Rn", "Relative Atomic Mass": "231.04987(32#)", "Mass Number": "231"}}, "103": {"256": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "103", "Atomic Symbol": "Lr", "Relative Atomic Mass": "256.098494(89)", "Mass Number": "256"}, "257": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "103", "Atomic Symbol": "Lr", "Relative Atomic Mass": "257.099418(47#)", "Mass Number": "257"}, "258": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "103", "Atomic Symbol": "Lr", "Relative Atomic Mass": "258.10176(11#)", "Mass Number": "258"}, "259": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "103", "Atomic Symbol": "Lr", "Relative Atomic Mass": "259.102902(76#)", "Mass Number": "259"}, "260": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "103", "Atomic Symbol": "Lr", "Relative Atomic Mass": "260.10550(13#)", "Mass Number": "260"}, "261": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "103", "Atomic Symbol": "Lr", "Relative Atomic Mass": "261.10688(22#)", "Mass Number": "261"}, "262": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "103", "Atomic Symbol": "Lr", "Relative Atomic Mass": "262.10961(22#)", "Mass Number": "262"}, "263": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "103", "Atomic Symbol": "Lr", "Relative Atomic Mass": "263.11136(30#)", "Mass Number": "263"}, "264": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "103", "Atomic Symbol": "Lr", "Relative Atomic Mass": "264.11420(47#)", "Mass Number": "264"}, "265": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "103", "Atomic Symbol": "Lr", "Relative Atomic Mass": "265.11619(65#)", "Mass Number": "265"}, "266": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "103", "Atomic Symbol": "Lr", "Relative Atomic Mass": "266.11983(56#)", "Mass Number": "266"}, "251": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "103", "Atomic Symbol": "Lr", "Relative Atomic Mass": "251.09418(32#)", "Mass Number": "251"}, "252": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "103", "Atomic Symbol": "Lr", "Relative Atomic Mass": "252.09526(26#)", "Mass Number": "252"}, "253": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "103", "Atomic Symbol": "Lr", "Relative Atomic Mass": "253.09509(22#)", "Mass Number": "253"}, "254": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "103", "Atomic Symbol": "Lr", "Relative Atomic Mass": "254.09648(32#)", "Mass Number": "254"}, "255": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "103", "Atomic Symbol": "Lr", "Relative Atomic Mass": "255.096562(19)", "Mass Number": "255"}}, "44": {"87": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "86.95069(43#)", "Mass Number": "87"}, "88": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "87.94160(32#)", "Mass Number": "88"}, "89": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "88.93762(32#)", "Mass Number": "89"}, "90": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "89.9303444(40)", "Mass Number": "90"}, "91": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "90.9267419(24)", "Mass Number": "91"}, "92": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "91.9202344(29)", "Mass Number": "92"}, "93": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "92.9171044(22)", "Mass Number": "93"}, "94": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "93.9113429(34)", "Mass Number": "94"}, "95": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "94.910406(10)", "Mass Number": "95"}, "96": {"Isotopic Composition": "0.0554(14)", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "95.90759025(49)", "Mass Number": "96"}, "97": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "96.9075471(30)", "Mass Number": "97"}, "98": {"Isotopic Composition": "0.0187(3)", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "97.9052868(69)", "Mass Number": "98"}, "99": {"Isotopic Composition": "0.1276(14)", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "98.9059341(11)", "Mass Number": "99"}, "100": {"Isotopic Composition": "0.1260(7)", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "99.9042143(11)", "Mass Number": "100"}, "101": {"Isotopic Composition": "0.1706(2)", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "100.9055769(12)", "Mass Number": "101"}, "102": {"Isotopic Composition": "0.3155(14)", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "101.9043441(12)", "Mass Number": "102"}, "103": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "102.9063186(12)", "Mass Number": "103"}, "104": {"Isotopic Composition": "0.1862(27)", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "103.9054275(28)", "Mass Number": "104"}, "105": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "104.9077476(28)", "Mass Number": "105"}, "106": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "105.9073291(58)", "Mass Number": "106"}, "107": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "106.9099720(93)", "Mass Number": "107"}, "108": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "107.9101880(93)", "Mass Number": "108"}, "109": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "108.9133260(96)", "Mass Number": "109"}, "110": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "109.9140407(96)", "Mass Number": "110"}, "111": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "110.917570(10)", "Mass Number": "111"}, "112": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "111.918809(10)", "Mass Number": "112"}, "113": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "112.922844(39)", "Mass Number": "113"}, "114": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "113.9246136(38)", "Mass Number": "114"}, "115": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "114.928820(71)", "Mass Number": "115"}, "116": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "115.9312192(40)", "Mass Number": "116"}, "117": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "116.93610(63)", "Mass Number": "117"}, "118": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "117.93853(32#)", "Mass Number": "118"}, "119": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "118.94357(32#)", "Mass Number": "119"}, "120": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "119.94631(43#)", "Mass Number": "120"}, "121": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "120.95164(43#)", "Mass Number": "121"}, "122": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "121.95447(54#)", "Mass Number": "122"}, "123": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "122.95989(54#)", "Mass Number": "123"}, "124": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "101.07(2)", "Atomic Number": "44", "Atomic Symbol": "Ru", "Relative Atomic Mass": "123.96305(64#)", "Mass Number": "124"}}, "61": {"128": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "127.94870(32#)", "Mass Number": "128"}, "129": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "128.94323(32#)", "Mass Number": "129"}, "130": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "129.94053(21#)", "Mass Number": "130"}, "131": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "130.93567(21#)", "Mass Number": "131"}, "132": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "131.93384(16#)", "Mass Number": "132"}, "133": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "132.929782(54)", "Mass Number": "133"}, "134": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "133.928353(62)", "Mass Number": "134"}, "135": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "134.924823(70)", "Mass Number": "135"}, "136": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "135.923585(77)", "Mass Number": "136"}, "137": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "136.920480(14)", "Mass Number": "137"}, "138": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "137.919548(30)", "Mass Number": "138"}, "139": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "138.916800(15)", "Mass Number": "139"}, "140": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "139.916040(38)", "Mass Number": "140"}, "141": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "140.913555(15)", "Mass Number": "141"}, "142": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "141.912890(25)", "Mass Number": "142"}, "143": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "142.9109383(34)", "Mass Number": "143"}, "144": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "143.9125964(34)", "Mass Number": "144"}, "145": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "144.9127559(33)", "Mass Number": "145"}, "146": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "145.9147024(48)", "Mass Number": "146"}, "147": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "146.9151450(19)", "Mass Number": "147"}, "148": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "147.9174819(63)", "Mass Number": "148"}, "149": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "148.9183423(27)", "Mass Number": "149"}, "150": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "149.920991(22)", "Mass Number": "150"}, "151": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "150.9212175(51)", "Mass Number": "151"}, "152": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "151.923506(28)", "Mass Number": "152"}, "153": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "152.9241567(97)", "Mass Number": "153"}, "154": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "153.926472(48)", "Mass Number": "154"}, "155": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "154.9281370(51)", "Mass Number": "155"}, "156": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "155.9311175(39)", "Mass Number": "156"}, "157": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "156.9331214(75)", "Mass Number": "157"}, "158": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "157.936565(14)", "Mass Number": "158"}, "159": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "158.939287(11)", "Mass Number": "159"}, "160": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "159.94310(32#)", "Mass Number": "160"}, "161": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "160.94607(32#)", "Mass Number": "161"}, "162": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "161.95022(43#)", "Mass Number": "162"}, "163": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "162.95357(54#)", "Mass Number": "163"}, "126": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "125.95792(54#)", "Mass Number": "126"}, "127": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[145]", "Atomic Number": "61", "Atomic Symbol": "Pm", "Relative Atomic Mass": "126.95192(43#)", "Mass Number": "127"}}, "42": {"83": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "82.94988(43#)", "Mass Number": "83"}, "84": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "83.94149(43#)", "Mass Number": "84"}, "85": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "84.938261(17)", "Mass Number": "85"}, "86": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "85.9311748(40)", "Mass Number": "86"}, "87": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "86.9281962(31)", "Mass Number": "87"}, "88": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "87.9219678(41)", "Mass Number": "88"}, "89": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "88.9194682(42)", "Mass Number": "89"}, "90": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "89.9139309(38)", "Mass Number": "90"}, "91": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "90.9117453(67)", "Mass Number": "91"}, "92": {"Isotopic Composition": "0.1453(30)", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "91.90680796(84)", "Mass Number": "92"}, "93": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "92.90680958(84)", "Mass Number": "93"}, "94": {"Isotopic Composition": "0.0915(9)", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "93.90508490(48)", "Mass Number": "94"}, "95": {"Isotopic Composition": "0.1584(11)", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "94.90583877(47)", "Mass Number": "95"}, "96": {"Isotopic Composition": "0.1667(15)", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "95.90467612(47)", "Mass Number": "96"}, "97": {"Isotopic Composition": "0.0960(14)", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "96.90601812(49)", "Mass Number": "97"}, "98": {"Isotopic Composition": "0.2439(37)", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "97.90540482(49)", "Mass Number": "98"}, "99": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "98.90770851(52)", "Mass Number": "99"}, "100": {"Isotopic Composition": "0.0982(31)", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "99.9074718(11)", "Mass Number": "100"}, "101": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "100.9103414(11)", "Mass Number": "101"}, "102": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "101.9102834(91)", "Mass Number": "102"}, "103": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "102.913079(10)", "Mass Number": "103"}, "104": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "103.9137344(98)", "Mass Number": "104"}, "105": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "104.916969(10)", "Mass Number": "105"}, "106": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "105.918259(10)", "Mass Number": "106"}, "107": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "106.922106(10)", "Mass Number": "107"}, "108": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "107.924033(10)", "Mass Number": "108"}, "109": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "108.928424(12)", "Mass Number": "109"}, "110": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "109.930704(26)", "Mass Number": "110"}, "111": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "110.935654(14)", "Mass Number": "111"}, "112": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "111.93831(21#)", "Mass Number": "112"}, "113": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "112.94335(32#)", "Mass Number": "113"}, "114": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "113.94653(32#)", "Mass Number": "114"}, "115": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "114.95196(43#)", "Mass Number": "115"}, "116": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "115.95545(54#)", "Mass Number": "116"}, "117": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "95.95(1)", "Atomic Number": "42", "Atomic Symbol": "Mo", "Relative Atomic Mass": "116.96117(54#)", "Mass Number": "117"}}, "10": {"32": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "20.1797(6)", "Atomic Number": "10", "Atomic Symbol": "Ne", "Relative Atomic Mass": "32.03972(54#)", "Mass Number": "32"}, "33": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "20.1797(6)", "Atomic Number": "10", "Atomic Symbol": "Ne", "Relative Atomic Mass": "33.04938(64#)", "Mass Number": "33"}, "34": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "20.1797(6)", "Atomic Number": "10", "Atomic Symbol": "Ne", "Relative Atomic Mass": "34.05673(55#)", "Mass Number": "34"}, "16": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "20.1797(6)", "Atomic Number": "10", "Atomic Symbol": "Ne", "Relative Atomic Mass": "16.025750(22)", "Mass Number": "16"}, "17": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "20.1797(6)", "Atomic Number": "10", "Atomic Symbol": "Ne", "Relative Atomic Mass": "17.01771396(38)", "Mass Number": "17"}, "18": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "20.1797(6)", "Atomic Number": "10", "Atomic Symbol": "Ne", "Relative Atomic Mass": "18.00570870(39)", "Mass Number": "18"}, "19": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "20.1797(6)", "Atomic Number": "10", "Atomic Symbol": "Ne", "Relative Atomic Mass": "19.00188091(17)", "Mass Number": "19"}, "20": {"Isotopic Composition": "0.9048(3)", "Notes": "g,m", "Standard Atomic Weight": "20.1797(6)", "Atomic Number": "10", "Atomic Symbol": "Ne", "Relative Atomic Mass": "19.9924401762(17)", "Mass Number": "20"}, "21": {"Isotopic Composition": "0.0027(1)", "Notes": "g,m", "Standard Atomic Weight": "20.1797(6)", "Atomic Number": "10", "Atomic Symbol": "Ne", "Relative Atomic Mass": "20.993846685(41)", "Mass Number": "21"}, "22": {"Isotopic Composition": "0.0925(3)", "Notes": "g,m", "Standard Atomic Weight": "20.1797(6)", "Atomic Number": "10", "Atomic Symbol": "Ne", "Relative Atomic Mass": "21.991385114(18)", "Mass Number": "22"}, "23": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "20.1797(6)", "Atomic Number": "10", "Atomic Symbol": "Ne", "Relative Atomic Mass": "22.99446691(11)", "Mass Number": "23"}, "24": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "20.1797(6)", "Atomic Number": "10", "Atomic Symbol": "Ne", "Relative Atomic Mass": "23.99361065(55)", "Mass Number": "24"}, "25": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "20.1797(6)", "Atomic Number": "10", "Atomic Symbol": "Ne", "Relative Atomic Mass": "24.997789(48)", "Mass Number": "25"}, "26": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "20.1797(6)", "Atomic Number": "10", "Atomic Symbol": "Ne", "Relative Atomic Mass": "26.000515(20)", "Mass Number": "26"}, "27": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "20.1797(6)", "Atomic Number": "10", "Atomic Symbol": "Ne", "Relative Atomic Mass": "27.007553(70)", "Mass Number": "27"}, "28": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "20.1797(6)", "Atomic Number": "10", "Atomic Symbol": "Ne", "Relative Atomic Mass": "28.01212(10)", "Mass Number": "28"}, "29": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "20.1797(6)", "Atomic Number": "10", "Atomic Symbol": "Ne", "Relative Atomic Mass": "29.01975(11)", "Mass Number": "29"}, "30": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "20.1797(6)", "Atomic Number": "10", "Atomic Symbol": "Ne", "Relative Atomic Mass": "30.02473(30)", "Mass Number": "30"}, "31": {"Isotopic Composition": "", "Notes": "g,m", "Standard Atomic Weight": "20.1797(6)", "Atomic Number": "10", "Atomic Symbol": "Ne", "Relative Atomic Mass": "31.0331(17)", "Mass Number": "31"}}, "100": {"256": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "100", "Atomic Symbol": "Fm", "Relative Atomic Mass": "256.0917745(78)", "Mass Number": "256"}, "257": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "100", "Atomic Symbol": "Fm", "Relative Atomic Mass": "257.0951061(69)", "Mass Number": "257"}, "258": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "100", "Atomic Symbol": "Fm", "Relative Atomic Mass": "258.09708(22#)", "Mass Number": "258"}, "259": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "100", "Atomic Symbol": "Fm", "Relative Atomic Mass": "259.10060(30#)", "Mass Number": "259"}, "260": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "100", "Atomic Symbol": "Fm", "Relative Atomic Mass": "260.10281(55#)", "Mass Number": "260"}, "241": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "100", "Atomic Symbol": "Fm", "Relative Atomic Mass": "241.07421(32#)", "Mass Number": "241"}, "242": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "100", "Atomic Symbol": "Fm", "Relative Atomic Mass": "242.07343(43#)", "Mass Number": "242"}, "243": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "100", "Atomic Symbol": "Fm", "Relative Atomic Mass": "243.07446(23#)", "Mass Number": "243"}, "244": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "100", "Atomic Symbol": "Fm", "Relative Atomic Mass": "244.07404(22#)", "Mass Number": "244"}, "245": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "100", "Atomic Symbol": "Fm", "Relative Atomic Mass": "245.07535(21#)", "Mass Number": "245"}, "246": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "100", "Atomic Symbol": "Fm", "Relative Atomic Mass": "246.075350(17)", "Mass Number": "246"}, "247": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "100", "Atomic Symbol": "Fm", "Relative Atomic Mass": "247.07694(12#)", "Mass Number": "247"}, "248": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "100", "Atomic Symbol": "Fm", "Relative Atomic Mass": "248.0771865(92)", "Mass Number": "248"}, "249": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "100", "Atomic Symbol": "Fm", "Relative Atomic Mass": "249.0789275(68)", "Mass Number": "249"}, "250": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "100", "Atomic Symbol": "Fm", "Relative Atomic Mass": "250.0795210(86)", "Mass Number": "250"}, "251": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "100", "Atomic Symbol": "Fm", "Relative Atomic Mass": "251.081540(16)", "Mass Number": "251"}, "252": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "100", "Atomic Symbol": "Fm", "Relative Atomic Mass": "252.0824671(61)", "Mass Number": "252"}, "253": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "100", "Atomic Symbol": "Fm", "Relative Atomic Mass": "253.0851846(37)", "Mass Number": "253"}, "254": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "100", "Atomic Symbol": "Fm", "Relative Atomic Mass": "254.0868544(30)", "Mass Number": "254"}, "255": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "100", "Atomic Symbol": "Fm", "Relative Atomic Mass": "255.0899640(52)", "Mass Number": "255"}}, "77": {"192": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "191.9626002(21)", "Mass Number": "192"}, "193": {"Isotopic Composition": "0.627(2)", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "192.9629216(21)", "Mass Number": "193"}, "194": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "193.9650735(21)", "Mass Number": "194"}, "195": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "194.9659747(21)", "Mass Number": "195"}, "196": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "195.968397(41)", "Mass Number": "196"}, "197": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "196.969655(22)", "Mass Number": "197"}, "198": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "197.97228(21#)", "Mass Number": "198"}, "199": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "198.973805(44)", "Mass Number": "199"}, "200": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "199.97680(21#)", "Mass Number": "200"}, "201": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "200.97864(21#)", "Mass Number": "201"}, "202": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "201.98199(32#)", "Mass Number": "202"}, "203": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "202.98423(43#)", "Mass Number": "203"}, "204": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "203.98960(43#)", "Mass Number": "204"}, "164": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "163.99191(34#)", "Mass Number": "164"}, "165": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "164.98750(18#)", "Mass Number": "165"}, "166": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "165.98566(22#)", "Mass Number": "166"}, "167": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "166.981666(20)", "Mass Number": "167"}, "168": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "167.979907(80)", "Mass Number": "168"}, "169": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "168.976298(27)", "Mass Number": "169"}, "170": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "169.974922(95#)", "Mass Number": "170"}, "171": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "170.971640(42)", "Mass Number": "171"}, "172": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "171.970607(35)", "Mass Number": "172"}, "173": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "172.967506(12)", "Mass Number": "173"}, "174": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "173.966861(30)", "Mass Number": "174"}, "175": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "174.964150(13)", "Mass Number": "175"}, "176": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "175.963650(22)", "Mass Number": "176"}, "177": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "176.961301(21)", "Mass Number": "177"}, "178": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "177.961082(21)", "Mass Number": "178"}, "179": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "178.959120(10)", "Mass Number": "179"}, "180": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "179.959229(23)", "Mass Number": "180"}, "181": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "180.957625(28)", "Mass Number": "181"}, "182": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "181.958076(23)", "Mass Number": "182"}, "183": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "182.956840(26)", "Mass Number": "183"}, "184": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "183.957476(30)", "Mass Number": "184"}, "185": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "184.956698(30)", "Mass Number": "185"}, "186": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "185.957944(18)", "Mass Number": "186"}, "187": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "186.957542(30)", "Mass Number": "187"}, "188": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "187.958828(10)", "Mass Number": "188"}, "189": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "188.958715(14)", "Mass Number": "189"}, "190": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "189.9605412(21)", "Mass Number": "190"}, "191": {"Isotopic Composition": "0.373(2)", "Notes": " ", "Standard Atomic Weight": "192.217(3)", "Atomic Number": "77", "Atomic Symbol": "Ir", "Relative Atomic Mass": "190.9605893(21)", "Mass Number": "191"}}, "38": {"73": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "72.96570(43#)", "Mass Number": "73"}, "74": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "73.95617(11#)", "Mass Number": "74"}, "75": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "74.94995(24)", "Mass Number": "75"}, "76": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "75.941763(37)", "Mass Number": "76"}, "77": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "76.9379455(85)", "Mass Number": "77"}, "78": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "77.9321800(80)", "Mass Number": "78"}, "79": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "78.9297077(90)", "Mass Number": "79"}, "80": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "79.9245175(37)", "Mass Number": "80"}, "81": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "80.9232114(34)", "Mass Number": "81"}, "82": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "81.9183999(64)", "Mass Number": "82"}, "83": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "82.9175544(73)", "Mass Number": "83"}, "84": {"Isotopic Composition": "0.0056(1)", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "83.9134191(13)", "Mass Number": "84"}, "85": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "84.9129320(30)", "Mass Number": "85"}, "86": {"Isotopic Composition": "0.0986(1)", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "85.9092606(12)", "Mass Number": "86"}, "87": {"Isotopic Composition": "0.0700(1)", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "86.9088775(12)", "Mass Number": "87"}, "88": {"Isotopic Composition": "0.8258(1)", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "87.9056125(12)", "Mass Number": "88"}, "89": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "88.9074511(12)", "Mass Number": "89"}, "90": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "89.9077300(28)", "Mass Number": "90"}, "91": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "90.9101954(61)", "Mass Number": "91"}, "92": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "91.9110382(37)", "Mass Number": "92"}, "93": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "92.9140242(81)", "Mass Number": "93"}, "94": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "93.9153556(18)", "Mass Number": "94"}, "95": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "94.9193529(63)", "Mass Number": "95"}, "96": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "95.9217066(93)", "Mass Number": "96"}, "97": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "96.9263740(36)", "Mass Number": "97"}, "98": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "97.9286888(40)", "Mass Number": "98"}, "99": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "98.9328907(38)", "Mass Number": "99"}, "100": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "99.935770(10)", "Mass Number": "100"}, "101": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "100.940352(86)", "Mass Number": "101"}, "102": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "101.943791(75)", "Mass Number": "102"}, "103": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "102.94909(21#)", "Mass Number": "103"}, "104": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "103.95265(32#)", "Mass Number": "104"}, "105": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "104.95855(54#)", "Mass Number": "105"}, "106": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "105.96265(64#)", "Mass Number": "106"}, "107": {"Isotopic Composition": "", "Notes": "g,r", "Standard Atomic Weight": "87.62(1)", "Atomic Number": "38", "Atomic Symbol": "Sr", "Relative Atomic Mass": "106.96897(75#)", "Mass Number": "107"}}, "65": {"135": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "134.96476(43#)", "Mass Number": "135"}, "136": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "135.96129(54#)", "Mass Number": "136"}, "137": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "136.95602(54#)", "Mass Number": "137"}, "138": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "137.95312(32#)", "Mass Number": "138"}, "139": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "138.94833(32#)", "Mass Number": "139"}, "140": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "139.94581(86)", "Mass Number": "140"}, "141": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "140.94145(11)", "Mass Number": "141"}, "142": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "141.93928(75)", "Mass Number": "142"}, "143": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "142.935137(55)", "Mass Number": "143"}, "144": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "143.933045(30)", "Mass Number": "144"}, "145": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "144.92882(10)", "Mass Number": "145"}, "146": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "145.927253(48)", "Mass Number": "146"}, "147": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "146.9240548(87)", "Mass Number": "147"}, "148": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "147.924282(14)", "Mass Number": "148"}, "149": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "148.9232535(41)", "Mass Number": "149"}, "150": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "149.9236649(80)", "Mass Number": "150"}, "151": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "150.9231096(46)", "Mass Number": "151"}, "152": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "151.924083(43)", "Mass Number": "152"}, "153": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "152.9234424(44)", "Mass Number": "153"}, "154": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "153.924685(49)", "Mass Number": "154"}, "155": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "154.923511(11)", "Mass Number": "155"}, "156": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "155.9247552(43)", "Mass Number": "156"}, "157": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "156.9240330(18)", "Mass Number": "157"}, "158": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "157.9254209(20)", "Mass Number": "158"}, "159": {"Isotopic Composition": "1", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "158.9253547(19)", "Mass Number": "159"}, "160": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "159.9271756(19)", "Mass Number": "160"}, "161": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "160.9275778(20)", "Mass Number": "161"}, "162": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "161.929495(39)", "Mass Number": "162"}, "163": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "162.9306547(47)", "Mass Number": "163"}, "164": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "163.93336(11)", "Mass Number": "164"}, "165": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "164.93498(21#)", "Mass Number": "165"}, "166": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "165.937860(75)", "Mass Number": "166"}, "167": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "166.93996(21#)", "Mass Number": "167"}, "168": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "167.94340(32#)", "Mass Number": "168"}, "169": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "168.94597(32#)", "Mass Number": "169"}, "170": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "169.94984(43#)", "Mass Number": "170"}, "171": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "158.92535(2)", "Atomic Number": "65", "Atomic Symbol": "Tb", "Relative Atomic Mass": "170.95273(54#)", "Mass Number": "171"}}, "106": {"258": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "106", "Atomic Symbol": "Sg", "Relative Atomic Mass": "258.11298(44#)", "Mass Number": "258"}, "259": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "106", "Atomic Symbol": "Sg", "Relative Atomic Mass": "259.11440(13#)", "Mass Number": "259"}, "260": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "106", "Atomic Symbol": "Sg", "Relative Atomic Mass": "260.114384(22)", "Mass Number": "260"}, "261": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "106", "Atomic Symbol": "Sg", "Relative Atomic Mass": "261.115949(20)", "Mass Number": "261"}, "262": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "106", "Atomic Symbol": "Sg", "Relative Atomic Mass": "262.116337(38)", "Mass Number": "262"}, "263": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "106", "Atomic Symbol": "Sg", "Relative Atomic Mass": "263.11829(10#)", "Mass Number": "263"}, "264": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "106", "Atomic Symbol": "Sg", "Relative Atomic Mass": "264.11893(30#)", "Mass Number": "264"}, "265": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "106", "Atomic Symbol": "Sg", "Relative Atomic Mass": "265.12109(13#)", "Mass Number": "265"}, "266": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "106", "Atomic Symbol": "Sg", "Relative Atomic Mass": "266.12198(26#)", "Mass Number": "266"}, "267": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "106", "Atomic Symbol": "Sg", "Relative Atomic Mass": "267.12436(30#)", "Mass Number": "267"}, "268": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "106", "Atomic Symbol": "Sg", "Relative Atomic Mass": "268.12539(50#)", "Mass Number": "268"}, "269": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "106", "Atomic Symbol": "Sg", "Relative Atomic Mass": "269.12863(39#)", "Mass Number": "269"}, "270": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "106", "Atomic Symbol": "Sg", "Relative Atomic Mass": "270.13043(60#)", "Mass Number": "270"}, "271": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "106", "Atomic Symbol": "Sg", "Relative Atomic Mass": "271.13393(63#)", "Mass Number": "271"}, "272": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "106", "Atomic Symbol": "Sg", "Relative Atomic Mass": "272.13589(83#)", "Mass Number": "272"}, "273": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": " ", "Atomic Number": "106", "Atomic Symbol": "Sg", "Relative Atomic Mass": "273.13958(54#)", "Mass Number": "273"}}, "40": {"78": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "77.95566(54#)", "Mass Number": "78"}, "79": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "78.94948(43#)", "Mass Number": "79"}, "80": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "79.9404(16)", "Mass Number": "80"}, "81": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "80.93731(18)", "Mass Number": "81"}, "82": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "81.93135(22#)", "Mass Number": "82"}, "83": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "82.9292421(69)", "Mass Number": "83"}, "84": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "83.9233269(59)", "Mass Number": "84"}, "85": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "84.9214444(69)", "Mass Number": "85"}, "86": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "85.9162972(38)", "Mass Number": "86"}, "87": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "86.9148180(45)", "Mass Number": "87"}, "88": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "87.9102213(58)", "Mass Number": "88"}, "89": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "88.9088814(37)", "Mass Number": "89"}, "90": {"Isotopic Composition": "0.5145(40)", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "89.9046977(20)", "Mass Number": "90"}, "91": {"Isotopic Composition": "0.1122(5)", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "90.9056396(20)", "Mass Number": "91"}, "92": {"Isotopic Composition": "0.1715(8)", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "91.9050347(20)", "Mass Number": "92"}, "93": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "92.9064699(20)", "Mass Number": "93"}, "94": {"Isotopic Composition": "0.1738(28)", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "93.9063108(20)", "Mass Number": "94"}, "95": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "94.9080385(19)", "Mass Number": "95"}, "96": {"Isotopic Composition": "0.0280(9)", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "95.9082714(21)", "Mass Number": "96"}, "97": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "96.9109512(21)", "Mass Number": "97"}, "98": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "97.9127289(93)", "Mass Number": "98"}, "99": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "98.916667(11)", "Mass Number": "99"}, "100": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "99.9180006(89)", "Mass Number": "100"}, "101": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "100.9214480(91)", "Mass Number": "101"}, "102": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "101.9231409(97)", "Mass Number": "102"}, "103": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "102.927191(10)", "Mass Number": "103"}, "104": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "103.929436(10)", "Mass Number": "104"}, "105": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "104.934008(13)", "Mass Number": "105"}, "106": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "105.93676(21#)", "Mass Number": "106"}, "107": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "106.94174(32#)", "Mass Number": "107"}, "108": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "107.94487(43#)", "Mass Number": "108"}, "109": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "108.95041(54#)", "Mass Number": "109"}, "110": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "109.95396(64#)", "Mass Number": "110"}, "111": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "110.95968(75#)", "Mass Number": "111"}, "112": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "91.224(2)", "Atomic Number": "40", "Atomic Symbol": "Zr", "Relative Atomic Mass": "111.96370(75#)", "Mass Number": "112"}}, "91": {"212": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "212.023203(80)", "Mass Number": "212"}, "213": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "213.021109(76)", "Mass Number": "213"}, "214": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "214.020918(82)", "Mass Number": "214"}, "215": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "215.019183(78)", "Mass Number": "215"}, "216": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "216.019109(57)", "Mass Number": "216"}, "217": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "217.018325(56)", "Mass Number": "217"}, "218": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "218.020059(20)", "Mass Number": "218"}, "219": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "219.019904(55)", "Mass Number": "219"}, "220": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "220.021705(55#)", "Mass Number": "220"}, "221": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "221.021875(55)", "Mass Number": "221"}, "222": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "222.023784(78#)", "Mass Number": "222"}, "223": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "223.023963(76)", "Mass Number": "223"}, "224": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "224.0256176(82)", "Mass Number": "224"}, "225": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "225.026131(76)", "Mass Number": "225"}, "226": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "226.027948(12)", "Mass Number": "226"}, "227": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "227.0288054(80)", "Mass Number": "227"}, "228": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "228.0310517(47)", "Mass Number": "228"}, "229": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "229.0320972(38)", "Mass Number": "229"}, "230": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "230.0345410(35)", "Mass Number": "230"}, "231": {"Isotopic Composition": "1", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "231.0358842(24)", "Mass Number": "231"}, "232": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "232.0385917(83)", "Mass Number": "232"}, "233": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "233.0402472(22)", "Mass Number": "233"}, "234": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "234.0433072(51)", "Mass Number": "234"}, "235": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "235.045399(15)", "Mass Number": "235"}, "236": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "236.048668(15)", "Mass Number": "236"}, "237": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "237.051023(14)", "Mass Number": "237"}, "238": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "238.054637(17)", "Mass Number": "238"}, "239": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "239.05726(21#)", "Mass Number": "239"}, "240": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "240.06098(32#)", "Mass Number": "240"}, "241": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "231.03588(2)", "Atomic Number": "91", "Atomic Symbol": "Pa", "Relative Atomic Mass": "241.06408(43#)", "Mass Number": "241"}}, "13": {"21": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "26.9815385(7)", "Atomic Number": "13", "Atomic Symbol": "Al", "Relative Atomic Mass": "21.02897(43#)", "Mass Number": "21"}, "22": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "26.9815385(7)", "Atomic Number": "13", "Atomic Symbol": "Al", "Relative Atomic Mass": "22.01954(43#)", "Mass Number": "22"}, "23": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "26.9815385(7)", "Atomic Number": "13", "Atomic Symbol": "Al", "Relative Atomic Mass": "23.00724435(37)", "Mass Number": "23"}, "24": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "26.9815385(7)", "Atomic Number": "13", "Atomic Symbol": "Al", "Relative Atomic Mass": "23.9999489(12)", "Mass Number": "24"}, "25": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "26.9815385(7)", "Atomic Number": "13", "Atomic Symbol": "Al", "Relative Atomic Mass": "24.99042810(51)", "Mass Number": "25"}, "26": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "26.9815385(7)", "Atomic Number": "13", "Atomic Symbol": "Al", "Relative Atomic Mass": "25.986891904(69)", "Mass Number": "26"}, "27": {"Isotopic Composition": "1", "Notes": " ", "Standard Atomic Weight": "26.9815385(7)", "Atomic Number": "13", "Atomic Symbol": "Al", "Relative Atomic Mass": "26.98153853(11)", "Mass Number": "27"}, "28": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "26.9815385(7)", "Atomic Number": "13", "Atomic Symbol": "Al", "Relative Atomic Mass": "27.98191021(13)", "Mass Number": "28"}, "29": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "26.9815385(7)", "Atomic Number": "13", "Atomic Symbol": "Al", "Relative Atomic Mass": "28.9804565(10)", "Mass Number": "29"}, "30": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "26.9815385(7)", "Atomic Number": "13", "Atomic Symbol": "Al", "Relative Atomic Mass": "29.982960(15)", "Mass Number": "30"}, "31": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "26.9815385(7)", "Atomic Number": "13", "Atomic Symbol": "Al", "Relative Atomic Mass": "30.983945(22)", "Mass Number": "31"}, "32": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "26.9815385(7)", "Atomic Number": "13", "Atomic Symbol": "Al", "Relative Atomic Mass": "31.988085(13)", "Mass Number": "32"}, "33": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "26.9815385(7)", "Atomic Number": "13", "Atomic Symbol": "Al", "Relative Atomic Mass": "32.990909(81)", "Mass Number": "33"}, "34": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "26.9815385(7)", "Atomic Number": "13", "Atomic Symbol": "Al", "Relative Atomic Mass": "33.996705(74)", "Mass Number": "34"}, "35": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "26.9815385(7)", "Atomic Number": "13", "Atomic Symbol": "Al", "Relative Atomic Mass": "34.999764(75)", "Mass Number": "35"}, "36": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "26.9815385(7)", "Atomic Number": "13", "Atomic Symbol": "Al", "Relative Atomic Mass": "36.00639(11)", "Mass Number": "36"}, "37": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "26.9815385(7)", "Atomic Number": "13", "Atomic Symbol": "Al", "Relative Atomic Mass": "37.01053(13)", "Mass Number": "37"}, "38": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "26.9815385(7)", "Atomic Number": "13", "Atomic Symbol": "Al", "Relative Atomic Mass": "38.01740(27)", "Mass Number": "38"}, "39": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "26.9815385(7)", "Atomic Number": "13", "Atomic Symbol": "Al", "Relative Atomic Mass": "39.02254(54#)", "Mass Number": "39"}, "40": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "26.9815385(7)", "Atomic Number": "13", "Atomic Symbol": "Al", "Relative Atomic Mass": "40.03003(54#)", "Mass Number": "40"}, "41": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "26.9815385(7)", "Atomic Number": "13", "Atomic Symbol": "Al", "Relative Atomic Mass": "41.03638(64#)", "Mass Number": "41"}, "42": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "26.9815385(7)", "Atomic Number": "13", "Atomic Symbol": "Al", "Relative Atomic Mass": "42.04384(64#)", "Mass Number": "42"}, "43": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "26.9815385(7)", "Atomic Number": "13", "Atomic Symbol": "Al", "Relative Atomic Mass": "43.05147(75#)", "Mass Number": "43"}}, "16": {"26": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[32.059,32.076]", "Atomic Number": "16", "Atomic Symbol": "S", "Relative Atomic Mass": "26.02907(64#)", "Mass Number": "26"}, "27": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[32.059,32.076]", "Atomic Number": "16", "Atomic Symbol": "S", "Relative Atomic Mass": "27.01828(43#)", "Mass Number": "27"}, "28": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[32.059,32.076]", "Atomic Number": "16", "Atomic Symbol": "S", "Relative Atomic Mass": "28.00437(17)", "Mass Number": "28"}, "29": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[32.059,32.076]", "Atomic Number": "16", "Atomic Symbol": "S", "Relative Atomic Mass": "28.996611(54)", "Mass Number": "29"}, "30": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[32.059,32.076]", "Atomic Number": "16", "Atomic Symbol": "S", "Relative Atomic Mass": "29.98490703(40)", "Mass Number": "30"}, "31": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[32.059,32.076]", "Atomic Number": "16", "Atomic Symbol": "S", "Relative Atomic Mass": "30.97955701(25)", "Mass Number": "31"}, "32": {"Isotopic Composition": "0.9499(26)", "Notes": " ", "Standard Atomic Weight": "[32.059,32.076]", "Atomic Number": "16", "Atomic Symbol": "S", "Relative Atomic Mass": "31.9720711744(14)", "Mass Number": "32"}, "33": {"Isotopic Composition": "0.0075(2)", "Notes": " ", "Standard Atomic Weight": "[32.059,32.076]", "Atomic Number": "16", "Atomic Symbol": "S", "Relative Atomic Mass": "32.9714589098(15)", "Mass Number": "33"}, "34": {"Isotopic Composition": "0.0425(24)", "Notes": " ", "Standard Atomic Weight": "[32.059,32.076]", "Atomic Number": "16", "Atomic Symbol": "S", "Relative Atomic Mass": "33.967867004(47)", "Mass Number": "34"}, "35": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[32.059,32.076]", "Atomic Number": "16", "Atomic Symbol": "S", "Relative Atomic Mass": "34.969032310(43)", "Mass Number": "35"}, "36": {"Isotopic Composition": "0.0001(1)", "Notes": " ", "Standard Atomic Weight": "[32.059,32.076]", "Atomic Number": "16", "Atomic Symbol": "S", "Relative Atomic Mass": "35.96708071(20)", "Mass Number": "36"}, "37": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[32.059,32.076]", "Atomic Number": "16", "Atomic Symbol": "S", "Relative Atomic Mass": "36.97112551(21)", "Mass Number": "37"}, "38": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[32.059,32.076]", "Atomic Number": "16", "Atomic Symbol": "S", "Relative Atomic Mass": "37.9711633(77)", "Mass Number": "38"}, "39": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[32.059,32.076]", "Atomic Number": "16", "Atomic Symbol": "S", "Relative Atomic Mass": "38.975134(54)", "Mass Number": "39"}, "40": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[32.059,32.076]", "Atomic Number": "16", "Atomic Symbol": "S", "Relative Atomic Mass": "39.9754826(43)", "Mass Number": "40"}, "41": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[32.059,32.076]", "Atomic Number": "16", "Atomic Symbol": "S", "Relative Atomic Mass": "40.9795935(44)", "Mass Number": "41"}, "42": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[32.059,32.076]", "Atomic Number": "16", "Atomic Symbol": "S", "Relative Atomic Mass": "41.9810651(30)", "Mass Number": "42"}, "43": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[32.059,32.076]", "Atomic Number": "16", "Atomic Symbol": "S", "Relative Atomic Mass": "42.9869076(53)", "Mass Number": "43"}, "44": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[32.059,32.076]", "Atomic Number": "16", "Atomic Symbol": "S", "Relative Atomic Mass": "43.9901188(56)", "Mass Number": "44"}, "45": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[32.059,32.076]", "Atomic Number": "16", "Atomic Symbol": "S", "Relative Atomic Mass": "44.99572(74)", "Mass Number": "45"}, "46": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[32.059,32.076]", "Atomic Number": "16", "Atomic Symbol": "S", "Relative Atomic Mass": "46.00004(54#)", "Mass Number": "46"}, "47": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[32.059,32.076]", "Atomic Number": "16", "Atomic Symbol": "S", "Relative Atomic Mass": "47.00795(54#)", "Mass Number": "47"}, "48": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[32.059,32.076]", "Atomic Number": "16", "Atomic Symbol": "S", "Relative Atomic Mass": "48.01370(64#)", "Mass Number": "48"}, "49": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[32.059,32.076]", "Atomic Number": "16", "Atomic Symbol": "S", "Relative Atomic Mass": "49.02276(72#)", "Mass Number": "49"}}, "75": {"192": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "191.966088(82)", "Mass Number": "192"}, "193": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "192.967541(41)", "Mass Number": "193"}, "194": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "193.97076(21#)", "Mass Number": "194"}, "195": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "194.97254(32#)", "Mass Number": "195"}, "196": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "195.97580(32#)", "Mass Number": "196"}, "197": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "196.97799(32#)", "Mass Number": "197"}, "198": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "197.98160(43#)", "Mass Number": "198"}, "159": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "158.98418(54#)", "Mass Number": "159"}, "160": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "159.98182(32#)", "Mass Number": "160"}, "161": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "160.97757(17)", "Mass Number": "161"}, "162": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "161.97584(22#)", "Mass Number": "162"}, "163": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "162.972080(20)", "Mass Number": "163"}, "164": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "163.970453(79)", "Mass Number": "164"}, "165": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "164.967103(27)", "Mass Number": "165"}, "166": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "165.965761(78)", "Mass Number": "166"}, "167": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "166.962595(44#)", "Mass Number": "167"}, "168": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "167.961573(33)", "Mass Number": "168"}, "169": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "168.958766(12)", "Mass Number": "169"}, "170": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "169.958220(28)", "Mass Number": "170"}, "171": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "170.955716(30)", "Mass Number": "171"}, "172": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "171.955420(42)", "Mass Number": "172"}, "173": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "172.953243(30)", "Mass Number": "173"}, "174": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "173.953115(30)", "Mass Number": "174"}, "175": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "174.951381(30)", "Mass Number": "175"}, "176": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "175.951623(30)", "Mass Number": "176"}, "177": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "176.950328(30)", "Mass Number": "177"}, "178": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "177.950989(30)", "Mass Number": "178"}, "179": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "178.949989(26)", "Mass Number": "179"}, "180": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "179.950792(23)", "Mass Number": "180"}, "181": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "180.950058(14)", "Mass Number": "181"}, "182": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "181.95121(11)", "Mass Number": "182"}, "183": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "182.9508196(86)", "Mass Number": "183"}, "184": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "183.9525228(47)", "Mass Number": "184"}, "185": {"Isotopic Composition": "0.3740(2)", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "184.9529545(13)", "Mass Number": "185"}, "186": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "185.9549856(13)", "Mass Number": "186"}, "187": {"Isotopic Composition": "0.6260(2)", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "186.9557501(16)", "Mass Number": "187"}, "188": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "187.9581115(16)", "Mass Number": "188"}, "189": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "188.9592260(89)", "Mass Number": "189"}, "190": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "189.961744(76)", "Mass Number": "190"}, "191": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "186.207(1)", "Atomic Number": "75", "Atomic Symbol": "Re", "Relative Atomic Mass": "190.963122(11)", "Mass Number": "191"}}, "48": {"128": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "127.9278129(78)", "Mass Number": "128"}, "129": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "128.93182(21#)", "Mass Number": "129"}, "130": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "129.93394(18)", "Mass Number": "130"}, "131": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "130.94060(21#)", "Mass Number": "131"}, "132": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "131.94604(21#)", "Mass Number": "132"}, "133": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "132.95285(32#)", "Mass Number": "133"}, "95": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "94.94994(54#)", "Mass Number": "95"}, "96": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "95.94034(43#)", "Mass Number": "96"}, "97": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "96.93510(32#)", "Mass Number": "97"}, "98": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "97.927389(56)", "Mass Number": "98"}, "99": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "98.9249258(17)", "Mass Number": "99"}, "100": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "99.9203488(18)", "Mass Number": "100"}, "101": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "100.9185862(16)", "Mass Number": "101"}, "102": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "101.9144820(18)", "Mass Number": "102"}, "103": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "102.9134165(19)", "Mass Number": "103"}, "104": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "103.9098564(18)", "Mass Number": "104"}, "105": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "104.9094639(15)", "Mass Number": "105"}, "106": {"Isotopic Composition": "0.0125(6)", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "105.9064599(12)", "Mass Number": "106"}, "107": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "106.9066121(18)", "Mass Number": "107"}, "108": {"Isotopic Composition": "0.0089(3)", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "107.9041834(12)", "Mass Number": "108"}, "109": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "108.9049867(17)", "Mass Number": "109"}, "110": {"Isotopic Composition": "0.1249(18)", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "109.90300661(61)", "Mass Number": "110"}, "111": {"Isotopic Composition": "0.1280(12)", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "110.90418287(61)", "Mass Number": "111"}, "112": {"Isotopic Composition": "0.2413(21)", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "111.90276287(60)", "Mass Number": "112"}, "113": {"Isotopic Composition": "0.1222(12)", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "112.90440813(45)", "Mass Number": "113"}, "114": {"Isotopic Composition": "0.2873(42)", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "113.90336509(43)", "Mass Number": "114"}, "115": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "114.90543751(77)", "Mass Number": "115"}, "116": {"Isotopic Composition": "0.0749(18)", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "115.90476315(17)", "Mass Number": "116"}, "117": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "116.9072260(11)", "Mass Number": "117"}, "118": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "117.906922(21)", "Mass Number": "118"}, "119": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "118.909847(40)", "Mass Number": "119"}, "120": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "119.9098681(40)", "Mass Number": "120"}, "121": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "120.9129637(21)", "Mass Number": "121"}, "122": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "121.9134591(25)", "Mass Number": "122"}, "123": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "122.9168925(29)", "Mass Number": "123"}, "124": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "123.9176574(32)", "Mass Number": "124"}, "125": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "124.9212576(31)", "Mass Number": "125"}, "126": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "125.9224291(27)", "Mass Number": "126"}, "127": {"Isotopic Composition": "", "Notes": "g", "Standard Atomic Weight": "112.414(4)", "Atomic Number": "48", "Atomic Symbol": "Cd", "Relative Atomic Mass": "126.926472(14)", "Mass Number": "127"}}, "89": {"206": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "206.014452(77#)", "Mass Number": "206"}, "207": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "207.011966(54)", "Mass Number": "207"}, "208": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "208.011550(60)", "Mass Number": "208"}, "209": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "209.009495(54)", "Mass Number": "209"}, "210": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "210.009436(62)", "Mass Number": "210"}, "211": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "211.007732(57)", "Mass Number": "211"}, "212": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "212.007813(55)", "Mass Number": "212"}, "213": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "213.006609(56)", "Mass Number": "213"}, "214": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "214.006918(16)", "Mass Number": "214"}, "215": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "215.006475(13)", "Mass Number": "215"}, "216": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "216.008743(12)", "Mass Number": "216"}, "217": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "217.009344(12)", "Mass Number": "217"}, "218": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "218.011642(54)", "Mass Number": "218"}, "219": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "219.012421(54)", "Mass Number": "219"}, "220": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "220.0147549(66)", "Mass Number": "220"}, "221": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "221.015592(54)", "Mass Number": "221"}, "222": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "222.0178442(56)", "Mass Number": "222"}, "223": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "223.0191377(77)", "Mass Number": "223"}, "224": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "224.0217232(45)", "Mass Number": "224"}, "225": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "225.0232300(53)", "Mass Number": "225"}, "226": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "226.0260984(36)", "Mass Number": "226"}, "227": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "227.0277523(25)", "Mass Number": "227"}, "228": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "228.0310215(27)", "Mass Number": "228"}, "229": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "229.032956(13)", "Mass Number": "229"}, "230": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "230.036327(17)", "Mass Number": "230"}, "231": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "231.038393(14)", "Mass Number": "231"}, "232": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "232.042034(14)", "Mass Number": "232"}, "233": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "233.044346(14)", "Mass Number": "233"}, "234": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "234.048139(15)", "Mass Number": "234"}, "235": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "235.050840(15)", "Mass Number": "235"}, "236": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "236.054988(41)", "Mass Number": "236"}, "237": {"Isotopic Composition": "", "Notes": " ", "Standard Atomic Weight": "[227]", "Atomic Number": "89", "Atomic Symbol": "Ac", "Relative Atomic Mass": "237.05827(43#)", "Mass Number": "237"}}};function get_token(str) {
    var token_types = [
        {re: /^\(/, type:'('},
        {re: /^\)/, type:')'},
        {re: /^\d+/, type: 'number'},
        {re: /^[A-Z][a-z]?/, type: 'symbol'}
    ];
    for(var i=0;i<token_types.length;i++) {
        var token_type = token_types[i];
        var m = token_type.re.exec(str);
        if(m) {
            return {str: m[0], type: token_type.type};
        }
    }
}

function tokenise(str) {
    var tokens = [];
    var m;
    var re_token = /^(\(|\)|\d+|[A-Z][a-z]?)/;
    var pos = 0;
    while(pos<str.length) {
        var token = get_token(str.slice(pos));
        if(!token) {
            throw(new Error("Couldn't parse formula "+str+": unexpected character "+str.slice(pos)+" at position "+pos));
        }
        tokens.push(token);
        pos += token.str.length;
    }
    return tokens;
}

/**
 * Data Formula:
 * {type: 'stoichiometry', formula: bracket, n: Number}
 * {type: 'formula', bits: [simple|numbered_bracket]}
 * {type: 'simple', bits: [element]}
 * {type: 'bracket', n: Number, middle: formula},
 * {type: 'element', n: Number, symbol: String}
 */

function parse_formula(str) {
    var tokens = tokenise(str);
    var res = stoichiometry(tokens) || formula(tokens);
    if(res.length==tokens.length) {
        return res.formula;
    }
}

function stoichiometry(tokens) {
    if(tokens[0].type=='number') {
        var n = parseInt(tokens[0].str);
        var bracketed = bracket(tokens.slice(1));
        if(bracketed) {
            return {formula: {type: 'stoichiometry', formula: bracketed.formula.middle, n: n}, length: bracketed.length+1};
        }
        var f = formula(tokens.slice(1));
        if(f) {
            return {formula: {type: 'stoichiometry', formula: f.formula, n: n}, length: f.length+1};
        }
    }
}

function formula(tokens) {
    var i = 0;
    var bits = [];
    while(i<tokens.length) {
        var ctokens = tokens.slice(i);
        var bit = simple(ctokens) || numbered_bracket(ctokens) || bracket(ctokens);
        if(!bit) {
            break;
        }
        i += bit.length;
        var f = bit.formula;
        if(bit.formula.type=='bracket' && bit.formula.n==1) {
            f = bit.formula.middle;
        }
        bits.push(f);
    }
    if(bits.length) {
        return {formula: {type: 'formula', bits: bits}, length: i};
    }
}

function simple(tokens) {
    var bits = [];
    var i = 0;
    while(i<tokens.length) {
        var bit = element(tokens.slice(i));
        if(!bit) {
            break;
        }
        bits.push(bit.formula);
        i += bit.length;
    }
    if(bits.length) {
        return {formula: {type: 'simple', bits: bits}, length: i};
    }
}

function numbered_bracket(tokens) {
    var bracketed = bracket(tokens);
    if(!bracketed) {
        return;
    }
    if(tokens.length>=bracketed.length+1 && tokens[bracketed.length].type=='number') {
        return {formula: {type: 'bracket', n: parseInt(tokens[bracketed.length].str), middle: bracketed.formula.middle}, length: bracketed.length+1};
    }
}

function bracket(tokens) {
    if(tokens.length<1 || tokens[0].type!='(') {
        return;
    }
    var middle = formula(tokens.slice(1));
    if(!middle) {
        return;
    }
    if(tokens.length<middle.length+2 || tokens[middle.length+1].type!=')') {
        return;
    }
    return {formula: {type: 'bracket', n: 1, middle: middle.formula}, length: middle.length+2};
}

function element(tokens) {
    if(tokens.length==0 || tokens[0].type!='symbol') {
        return ;
    }
    if(tokens.length>=2 && tokens[1].type=='number') {
        return {formula: {type: 'element', n: parseInt(tokens[1].str), symbol: tokens[0].str}, length: 2};
    } else {
        return {formula: {type: 'element', n: 1, symbol: tokens[0].str}, length: 1};
    }
}


function count_atoms(formula) {
    var d = {};
    switch(formula.type) {
        case 'stoichiometry':
            Object.entries(count_atoms(formula.formula)).forEach(function(e) {
                d[e[0]] = formula.n * e[1];
            });
            break;
        case 'formula': 
            formula.bits.forEach(function(b) {
                Object.entries(count_atoms(b)).forEach(function(e) {
                    d[e[0]] = (d[e[0]] || 0) + e[1];
                });
            });
            break;
        case 'simple':
            formula.bits.forEach(function(b) {
                d[b.symbol] = (d[b.symbol] || 0) + b.n;
            });
            break;
        case 'bracket':
            Object.entries(count_atoms(formula.middle)).forEach(function(e) {
                d[e[0]] = formula.n * e[1];
            });
            break;
        case 'element':
            d[formula.symbol] = formula.n;
            break;
        default:
            throw(new Error("Not a formula"));
    }
    return d;
}

function formula_renderer(sub,symbol) {
    sub = sub || function(n){return n+'';};
    symbol = symbol || function(n){return n+'';};
    var render = function(formula,use_subscript) {
        switch(formula.type) {
            case 'stoichiometry':
                return formula.n+'('+render(formula.formula)+')';
            case 'formula': 
                return formula.bits.map(render).join('');
            case 'simple':
                return formula.bits.map(render).join('');
            case 'bracket':
                return '('+render(formula.middle)+')' + (formula.n==1 ? '' : sub(formula.n));
            case 'element':
                return symbol(formula.symbol) + (formula.n==1 ? '' : sub(formula.n));
            default:
                throw(new Error("Not a formula"));
        }
    }
    return render;
}
var display_formula = formula_renderer(subscript);
var formula_to_string = formula_renderer();
var formula_to_tex = formula_renderer(function(n){ return '_{'+n+'}'; }, function(s){return '\\mathrm{'+s+'}'});

})();