/*
    Numbas JSXGraph extension.
    Copyright 2010-2021 Newcastle University
*/

Numbas.addExtension('jsxgraph',['display','util','jme'],function(jsxgraph) {
    var jme = Numbas.jme;
    var util = Numbas.util;
    var math = Numbas.math;

    var jme = Numbas.jme;
    var sig = jme.signature;

    var types = jme.types;
    var funcObj = jme.funcObj;
    var TString = types.TString;
    var TFunc = types.TFunc;
    var TVector = types.TVector;
    var TNum = types.TNum;
    var TDict = types.TDict;
    var TList = types.TList
    var TBool = types.TBool;
    var THTML = types.THTML;
    var TNothing = types.TNothing;
    var TExpression = types.TExpression;

    var object_property_getters = [
        ['value','Value', TNum],
        ['radius','Radius', TNum],
        ['has_point_sector','hasPointSector',TBool,[sig.type('number'),sig.type('number')]],
        ['area','Area',TNum],
        ['bounds','bounds'],
        ['max_x','maxX',TNum],
        ['min_x','minX',TNum],
        ['attribute','getAttribute',null,[sig.type('string')]],
        ['name','getName',TString],
        ['type','getType',TString],
        ['parents','getParents'],
        ['width','W',TNum],
        ['angle','getAngle',TNum],
        ['rise','getRise',TNum],
        ['slope','getSlope',TNum],
        ['length','L',TNum],
        ['bounding_box','boundingBox'],
        ['perimeter','Perimeter',TNum],
        ['size','getSize',TNum]
    ];

    var boards = 0;
    var boardholder;
    Numbas.util.document_ready(function() {
        boardholder = document.createElement('div');
        boardholder.id = 'jsxgraphholder';
        boardholder.setAttribute('class','invisible');
        boardholder.setAttribute('style','height:0; overflow-y: hidden;');
        document.body.appendChild(boardholder);
    });

    jsxgraph.default_options = {
        boundingBox:[-5,5,5,-5],
        showCopyright:false,
        showNavigation:false,
        axis:true
    };

    jsxgraph.makeBoard = function(width,height,options) {
        width = width || '600px';
        height = height || '600px';
        options = Numbas.util.extend_object(
            {},
            jsxgraph.default_options,
            options
        );

        var div = document.createElement('div');
        div.style.margin='0 auto';
        div.id = 'jsxgraph'+(boards++);
        div.style.width = width;
        div.style.height = height;
        div.classList.add('jsxgraph-board');
        boardholder.appendChild(div);
        div.board = JXG.JSXGraph.initBoard(div,options);

        var attached_interval = setInterval(function() {
            if(div.parentElement!=boardholder) {
                div.board.fullUpdate();
                clearInterval(attached_interval);
            }
        },10);

        return div;
    }

    jsxgraph.makeBoardPromise = function(width,height,options) {
        width = width || '600px';
        height = height || '600px';
        options = Numbas.util.extend_object(
            {},
            jsxgraph.default_options,
            options
        );

        var div = document.createElement('div');
        div.style.margin='0 auto';
        div.id = 'jsxgraph'+(boards++);
        div.style.width = width;
        div.style.height = height;
        div.classList.add('jsxgraph-board');

        var promise = new Promise(function(resolve,reject) {
            var attached_interval = setInterval(function() {
                var p = div;
                while(p && !(p.nodeType == p.DOCUMENT_NODE || p.nodeType == p.DOCUMENT_FRAGMENT_NODE)) {
                    p = p.parentNode;
                }
                if(p) {
                    clearInterval(attached_interval);
                    var board = div.board = JXG.JSXGraph.initBoard(div,options);
                    board.fullUpdate();
                    resolve(board);
                }
            },10);
        });

        return {element: div, promise: promise};
    }

    var not_initialised_error = false;

    var TJSXGraphBoard = jsxgraph.TJSXGraphBoard = function(width,height,options,question,immediate) {
        var jb = this;
        this.width = width;
        this.height = height;
        this.options = options || {};
        this.question = question;
        this.init_callbacks = [];
        this.init_actions = [];
        this.has_interacted = false;
        var res;
        if(immediate) {
            res = jsxgraph.makeBoard(width+'px',height+'px',options);
            this.boardPromise = Promise.resolve(res.board);
            this.element = res;
        } else {
            res = jsxgraph.makeBoardPromise(width+'px',height+'px',options);
            this.boardPromise = res.promise;
            this.element = res.element;
        }
        this.boardPromise.then(function(board) {
            jb.board = board;
            jb.init_callbacks.forEach(function(fn) {
                fn(jb.board);
            });
            function interact() {
                jb.has_interacted = true;
            }
            jb.board.on('down', interact);
            jb.board.on('keymove', interact);
            if(question) {
                question.signals.on('revealed',function() {
                    jb.lockBoard();
                });
                question.signals.on('ready', function() {
                    question.allParts().map(function(p) {
                        jb.linkToPart(p);
                    });
                    question.events.on('add part',function(p) {
                        jb.linkToPart(p);
                    });
                });
            }
        });
        this.cache = {};
    }
    TJSXGraphBoard.prototype = {
        toJSON: function() {
            return {
                width: this.width,
                height: this.height,
                options: this.options,
                board: this.board ? JXG.Dump.toJessie(this.board) : null
            }
        },

        getBoard: function() {
            if(!this.board) {
                not_initialised_error = true;
                throw(new Error("The JSXGraph board hasn't initialised yet."));
            }
            return this.board;
        },

        getObject: function(id) {
            var board = this.getBoard();
            var object = board.objectsList.find(function(object) { return object.id==id; });
            if(!object) {
                throw(new Error("There's no object with the ID "+id+" in this JSXGraph board."));
            }
            return object;
        },

        lockBoard: function() {
            var board = this.getBoard();

            board.attr.drag.enabled = false;
        },

        linkToPart: function(p) {
            var jb = this;
            if(!p.markingScript) {
                return;
            }
            var setting_answer = false;
            var reading_answer = false;

            var auto_submit_timeout = null;
            function debounce_auto_submit() {
                if(!jb.has_interacted) {
                    return;
                }
                if(auto_submit_timeout) {
                    clearTimeout(auto_submit_timeout);
                }
                if(!(p.question && p.question.exam && p.question.exam.settings.autoSubmit)) {
                    return;
                }
                auto_submit_timeout = setTimeout(function() {
                    let pp = p;
                    while(pp.isGap) {
                        pp = pp.parentPart;
                    }
                    pp.submit()
                }, 1000);
            }

            function bind_inputs() {
                if(setting_answer || !p.hasStagedAnswer()) {
                    return;
                }
                if(!jb.board) {
                    return;
                }
                reading_answer = true;

                var dirty = p.isDirty;
                var dirties = [];
                var pp = p.parentPart;
                while(pp) {
                    dirties.push(pp.isDirty);
                    pp = pp.parentPart;
                }
                var stagedAnswer = p.stagedAnswer;
                var os = p.studentAnswer;
                p.setStudentAnswer();
                var raw_answer = p.rawStudentAnswerAsJME();

                var errors = [];
                not_initialised_error = false;
                try {
                    var scope = new Numbas.jme.Scope([p.getScope()]);
                    var res = p.markingScript.evaluate_note('jxg_input',scope,p.marking_parameters(raw_answer));
                    for(var x in res.scope.state_errors) {
                        errors.push(res.scope.state_errors[x]);
                    }
                    if(errors.length) {
                        throw(new Error("There was an error linking this part to a diagram."));
                    }
                    var bindings = res.value;
                    bindings.value.forEach(function(b) {
                        b.fn();
                    });
                } catch(e) {
                    if(!not_initialised_error) {
                        p.giveWarning(e);
                        errors.forEach(function(err) {
                            p.giveWarning(err);
                        });
                    }
                }

                var warnings = p.warnings.slice();
                p.storeAnswer(os,true);
                p.setStudentAnswer();
                p.storeAnswer(stagedAnswer,true);
                p.setDirty(dirty);
                if(!dirty) {
                    var pp = p.parentPart;
                    for(var i=0;i<dirties.length;i++) {
                        pp.setDirty(dirties[i]);
                        pp = pp.parentPart;
                    }
                }
                p.setWarnings(warnings);
            }
            function bind_outputs() {
                if(reading_answer) {
                    return;
                }
                if(!jb.board) {
                    return;
                }
                setting_answer = true;
                var errors = [];
                try {
                    var scope = new Numbas.jme.Scope([p.getScope()]);
                    var res = p.markingScript.evaluate_note('jxg_output', scope);
                    var error = res.scope.state_errors['jxg_output'];
                    for(var x in res.scope.state_errors) {
                        errors.push(res.scope.state_errors[x]);
                    }
                    if(errors.length) {
                        throw(new Error("There was an error linking this part to a diagram."));
                    }
                    var answer = res.value;
                    if(!Numbas.util.eq(answer, jme.wrapValue(p.stagedAnswer))) {
                        var uanswer = jme.unwrapValue(answer);
                        var ostagedAnswer = p.stagedAnswer;
                        p.storeAnswer(uanswer);
                        p.display.restoreAnswer(uanswer);
                        if(ostagedAnswer === undefined) {
                            p.setDirty(false);
                        } else {
                            debounce_auto_submit();
                        }
                    }
                } catch(e) {
                    if(!not_initialised_error) {
                        p.removeWarnings();
                        p.giveWarning(e);
                        errors.forEach(function(err) {
                            p.giveWarning(err);
                        });
                    }
                }
                setting_answer = false;
            }
            if(p.markingScript.notes['jxg_input']!==undefined && !p.jsxgraph_input_linked) {
                p.jsxgraph_input_linked = true;
                p.events.on('storeAnswer', bind_inputs);
                bind_inputs();
            }
            if(p.markingScript.notes['jxg_output']!==undefined) {
                this.board.on('update', bind_outputs);
                this.board.on('down',function() {
                    reading_answer = false;
                });
                bind_outputs();
            } else if(p.type=='extension') {
                var has_interacted = false;
                this.board.on('up', () => { has_interacted = true; });
                this.board.on('keymove', () => { has_interacted = true; });
                this.board.on('update', () => {
                    if(has_interacted) {
                        if(p.display && p.display.html && p.display.html.contains(this.element)) {
                            p.setDirty(true);
                            debounce_auto_submit();
                        }
                    }
                });
            }
        },

        when_board: function(fn) {
            if(this.board) {
                fn(this.board);
            } else {
                this.init_callbacks.push(fn);
            }
        },

        run_jessiecode: function(code) {
            this.init_actions.push({
                type: 'jessiecode',
                code: code
            });
            this.when_board(function(board) {
                board.jc.parse(code);
            });
        },

        /** Add a list or dictionary of objects to the diagram.
         * @param {Numbas.jme.token} tobjects - A list or dictionary of definitions.
         * @param {Numbas.jme.Scope} scope
         */
        add_jme_objects: function(tobjects,scope) {
            this.init_actions.push({
                type: 'jme_objects',
                objects: tobjects
            });
            this.when_board(function(board) {
                var objects = tobjects.value;
                if(tobjects.type=='dict') {
                    var nobjects = [];
                    Object.keys(objects).forEach(function(name) {
                        var od = objects[name].value.slice();
                        if(od[2].type=='nothing') {
                            od[2] = new TDict({});
                        }
                        var options = od[2];
                        options.value.name = new TString(name);
                        nobjects.push(od);
                    });
                    objects = nobjects;
                } else {
                    objects = objects.map(function(od) { return od.value; });
                }
                objects.forEach(function(od) {
                    var name = jme.unwrapValue(od[0]);
                    var parents = od[1].value.map(function(bit) {
                        if(jme.isType(bit,'expression')) {
                            var expr = jme.castToType(bit,'expression');
                            return make_function_plotter(board, expr.tree, scope);
                        } else {
                            return jme.unwrapValue(bit);
                        }
                    });
                    var options = od[2].type=='nothing' ? {} : jme.unwrapValue(od[2]);
                    var obj = board.create.apply(board,[name,parents,options]);
                    var events = od[3].type=='nothing' ? {} : jme.unwrapValue(od[3]);
                    Object.keys(events).forEach(function(ename) {
                        var code = events[ename];
                        obj.on(ename,function() {
                            board.jc.parse(code);
                        });
                    });
                });
            });
        },

        run_init_actions: function(actions, scope) {
            var jb = this;
            actions.forEach(function(a) {
                switch(a.value.type.value) {
                    case 'jessiecode':
                        jb.run_jessiecode(a.value.code.value);
                        break;
                    case 'jme_objects':
                        jb.add_jme_objects(a.value.objects, scope);
                        break;
                }
            });
        }
    }

    jme.registerType(
        TJSXGraphBoard,
        'jsxgraphboard',
        {
            'html': function(v) {
                return new THTML(v.element);
            }
        }
    );
    jme.display.registerType(
        TJSXGraphBoard,
        {
            tex: function(v) {
                return '\\text{JSXGraph board}';
            },
            jme: function(v) {
                if(v.tok._to_jme) {
                    throw(new Numbas.Error("A JSXGraph board refers to itself in its own definition."));
                }
                v.tok._to_jme = true;
                var s;
                if(v.tok.board) {
                    s = jme.display.treeToJME({
                        tok: new TFunc('resume_jsxgraph_board'),
                        args: [{tok:jme.wrapValue(v.tok.toJSON())}]
                    });
                } else {
                    s = jme.display.treeToJME({
                        tok: new TFunc('resume_jsxgraph_board_init'),
                        args: [
                            {tok: jme.wrapValue(v.tok.width)},
                            {tok: jme.wrapValue(v.tok.height)},
                            {tok: jme.wrapValue(v.tok.options)},
                            {
                                tok: new TList(v.tok.init_actions.map(function(a) {
                                    switch(a.type) {
                                        case 'jessiecode':
                                            return new TDict({type: jme.wrapValue(a.type), code: jme.wrapValue(a.code)});
                                        case 'jme_objects':
                                            return new TDict({type: jme.wrapValue(a.type), objects: a.objects})
                                    }
                                }))
                            }
                        ]
                    },{wrapexpressions:true});
                }
                v.tok._to_jme = false;
                return s;
            },
            displayString: function(v) {
                return 'JSXGraph board';
            }
        }
    );

    Numbas.util.equalityTests['jsxgraphboard'] = function(a,b) {
        return a==b;
    };

    var TJSXGraphObject = jsxgraph.TJSXGraphObject = function(board,id) {
        this.board = board;
        this.id = id;
    }
    TJSXGraphObject.prototype = {
        get: function() {
            return this.board.getObject(this.id);
        }
    }

    var TJSXGraphAction = jsxgraph.TJSXGraphAction = function(fn) {
        this.fn = fn;
    }

    var make_function_plotter = jsxgraph.make_function_plotter = function(board,expr,scope) {
        if(typeof(expr)=='string') {
            expr = jme.compile(expr);
        }
        var freevars = jme.findvars(expr).filter(function(n) { return !scope.getVariable(n) });
        freevars.sort();

        var vars_in_board = freevars.map(function(n) { return board.objects[n] && board.objects[n].Value; });
        var has_vars_in_board = vars_in_board.some(x => x);
        var nscope = new Numbas.jme.Scope([scope]);

        try {
            var fn = Numbas.jme.makeFast(expr, scope, freevars);

            if(has_vars_in_board) {
                return function() {
                    var args = new Array(freevars.length);
                    let num_in_board = 0;
                    for(let i=0;i<freevars.length;i++) {
                        if(vars_in_board[i]) {
                            num_in_board += 1;
                            args[i] = board.objects[freevars[i]].Value();
                        } else {
                            args[i] = arguments[i-num_in_board];
                        }
                    }
                    return fn(...args);
                }
            } else {
                return fn;
            }
        } catch(e) {
            function eval() {
                var params = {};
                var args = Array.from(arguments);
                freevars.forEach(function(freevar, i) {
                    if(vars_in_board[i]) {
                        params[freevar] = board.objects[freevar].Value();
                    } else {
                        params[freevar] = jme.wrapValue(args[i]);
                    }
                });
                scope.cache_signatures = true;
                return jme.unwrapValue(scope.evaluate(expr,params));
            }
            return eval;
        }
    }

    var sig_jsxgraph_object = sig.list(
        sig.type('string'),
        sig.type('list'),
        sig.label('attributes',sig.optional(sig.type('dict'))),
        sig.label('events',sig.optional(sig.type('dict')))
    );

    var sig_jme_objects =
        sig.label('objects',sig.or(
            sig.dict(sig_jsxgraph_object),
            sig.listof(sig_jsxgraph_object)
        ))
    ;

    var sig_jsxgraph = sig.sequence(
        sig.label('width',sig.type('number')),
        sig.label('height',sig.type('number')),
        sig.optional(sig.label('boundingBox',sig.list(
            sig.type('number'),
            sig.type('number'),
            sig.type('number'),
            sig.type('number')
        ))),
        sig_jme_objects,
        sig.optional(sig.label('options',sig.type('dict')))
    );
    jsxgraph.scope.addFunction(new funcObj('jsxgraph',[sig_jsxgraph],TJSXGraphBoard,null,{
        random: false,
        evaluate: function(args,scope) {
            var m = sig_jsxgraph(args.filter(function(t){ return t.type!='nothing' }));
            var argdict = {};
            m.forEach(function(d,i) {
                if(!d.missing) {
                    argdict[d.name] = args[i];
                }
            });
            var width = argdict.width.value;
            var height = argdict.height.value;
            var options = argdict.options ? jme.unwrapValue(argdict.options) : {};
            if(argdict.boundingBox) {
                options.boundingBox = jme.unwrapValue(argdict.boundingBox);
            }

            var jb = new TJSXGraphBoard(width,height,options,scope.question);
            jb.add_jme_objects(argdict.objects,scope);

            return jb;
        }
    }));

    var sig_jessiecode = sig.sequence(
        sig.label('width',sig.type('number')),
        sig.label('height',sig.type('number')),
        sig.optional(sig.label('boundingBox',sig.list(
            sig.type('number'),
            sig.type('number'),
            sig.type('number'),
            sig.type('number')
        ))),
        sig.label('script',sig.type('string')),
        sig.optional(sig.label('options',sig.type('dict')))
    );
    jsxgraph.scope.addFunction(new funcObj('jessiecode',[sig_jessiecode],TJSXGraphBoard,null, {
        random: false,
        evaluate: function(args,scope) {
            var m = sig_jessiecode(args.filter(function(t){ return t.type!='nothing' }));
            var argdict = {};
            m.forEach(function(d,i) {
                if(!d.missing) {
                    argdict[d.name] = args[i];
                }
            });
            var width = argdict.width.value;
            var height = argdict.height.value;
            var options = argdict.options ? jme.unwrapValue(argdict.options) : {};
            if(argdict.boundingBox) {
                options.boundingBox = jme.unwrapValue(argdict.boundingBox);
            }

            var jb = new TJSXGraphBoard(width,height,options,scope.question);
            jb.run_jessiecode(argdict.script.value);

            return jb;
        }
    }));

    jsxgraph.scope.addFunction(new funcObj('resume_jsxgraph_board',[TDict],TJSXGraphBoard,null, {
        random: false,
        evaluate: function(args,scope) {
            var argdict = jme.unwrapValue(args[0]);
            var width = argdict.width;
            var height = argdict.height;
            var options = argdict.options || {};
            var code = argdict.board;
            options.grid = false;
            options.axis = false;

            var jb = new TJSXGraphBoard(width,height,options,scope.question,true);
            jb.boardPromise.then(function(board) {
                board.jc.parse(code);
            });

            return jb;
        }
    }));

    jsxgraph.scope.addFunction(new funcObj('resume_jsxgraph_board_init',[TNum, TNum, TDict, TList], TJSXGraphBoard, null, {
        random: false,
        evaluate: function(args,scope) {
            var width = jme.unwrapValue(args[0]);
            var height = jme.unwrapValue(args[1]);
            var options = jme.unwrapValue(args[2]);
            var actions = args[3].value;

            var jb = new TJSXGraphBoard(width,height,options,scope.question,true);
            jb.run_init_actions(actions, scope);

            return jb;
        }
    }));

    jsxgraph.scope.addFunction(new funcObj('listval',[TJSXGraphBoard,TString], TJSXGraphObject, null, {
        random: false,
        evaluate: function(args,scope) {
            return new TJSXGraphObject(args[0],args[1].value);
        }
    }));

    object_property_getters.forEach(function(def) {
        var jme_name = def[0];
        var jxg_name = def[1];
        var outtype = def[2];
        var sig = def[3] || [];
        jsxgraph.scope.addFunction(new funcObj('jxg_'+jme_name, [TJSXGraphObject].concat(sig), outtype, null, {
            random: false,
            evaluate: function(args,scope) {
                var object = args[0].get();
                var args = args.slice(1).map(function(arg) { return jme.unwrapValue(arg); });
                var res = object[jxg_name].apply(object,args);
                return outtype ? new outtype(res) : jme.wrapValue(res);
            }
        }));
    });

    jsxgraph.scope.addFunction(new funcObj('jxg_position',[TJSXGraphObject], TVector, null, {
        random: false,
        evaluate: function(args,scope) {
            var object = args[0].get();
            var x = object.X();
            var y = object.Y();
            return new TVector([x,y]);
        }
    }));

    jsxgraph.scope.addFunction(new funcObj('jxg_position_at',[TJSXGraphObject, TNum], TVector, null, {
        random: false,
        evaluate: function(args,scope) {
            var object = args[0].get();
            var t = args[1].value;
            var x = object.X(t);
            var y = object.Y(t);
            return new TVector([x,y]);
        }
    }));

    function has_point(tobject,x,y) {
        var c = new JXG.Coords(JXG.COORDS_BY_USER, [x,y], tobject.board.board);
        var obj = tobject.get();
        return new TBool(obj.hasPoint(c.scrCoords[1], c.scrCoords[2]));
    }

    jsxgraph.scope.addFunction(new funcObj('jxg_has_point',[TJSXGraphObject, TVector], TBool, null, {
        random: false,
        evaluate: function(args,scope) {
            return has_point(args[0], args[1].value[0], args[1].value[1]);
        }
    }));

    jsxgraph.scope.addFunction(new funcObj('jxg_has_point',[TJSXGraphObject, TNum, TNum], TBool, null, {
        random: false,
        evaluate: function(args,scope) {
            return has_point(args[0], args[1].value, args[2].value);
        }
    }));

    jsxgraph.scope.addFunction(new funcObj('jxg_distance',[TJSXGraphObject, TJSXGraphObject], TNum, null, {
        random: false,
        evaluate: function(args,scope) {
            var a = args[0].get();
            var b = args[1].get();
            return new TNum(a.Dist(b));
        }
    }));

    function make_action(fn) {
        return function() {
            var t = this;
            var args = arguments;
            return new TJSXGraphAction(function() {
                return fn.apply(t,args);
            });
        }
    }

    jsxgraph.scope.addFunction(new funcObj('jxg_set_position',[TJSXGraphObject, TVector], TJSXGraphAction, null, {
        random: false,
        evaluate: make_action(function(args,scope) {
            var tboard = args[0].board;
            tboard.boardPromise.then(function(board) {
                var object = args[0].get();
                var coords = jme.unwrapValue(args[1]);
                if(coords.every(function(z) { return !isNaN(z); })) {
                    object.setPosition(JXG.COORDS_BY_USER, coords);
                    board.update();
                }
            });
        })
    }));

    jsxgraph.scope.addFunction(new funcObj('jxg_show',[TJSXGraphObject, TBool], TJSXGraphAction, null, {
        random: false,
        evaluate: make_action(function(args,scope) {
            var tboard = args[0].board;
            tboard.boardPromise.then(function() {
                var object = args[0].get();
                var show = args[1].value;
                if(show) {
                    object.show();
                } else {
                    object.hide();
                }
            });
        })
    }));

    jsxgraph.scope.addFunction(new funcObj('jxg_set',[TJSXGraphObject, TString, sig.multiple(sig.anything())], TJSXGraphAction, null, {
        random: false,
        evaluate: make_action(function(args,scope) {
            var tboard = args[0].board;
            tboard.boardPromise.then(function(board) {
                var object = args[0].get();
                var name = Numbas.util.capitalise(args[1].value);
                var jargs = args.slice(2).map(function(a) {
                    var v = jme.unwrapValue(a);
                    if(v.toNumber) {
                        v = v.toNumber();
                    }
                    return v;
                });
                object['set'+name].apply(object,jargs);
                board.update();
            });
        })
    }));

    jsxgraph.scope.addFunction(new funcObj('jxg_set_attribute',[TJSXGraphObject, TDict], TJSXGraphAction, null, {
        random: false,
        evaluate: make_action(function(args,scope) {
            var tboard = args[0].board;
            tboard.boardPromise.then(function() {
                var object = args[0].get();
                var attributes = jme.unwrapValue(args[1]);
                object.setAttribute(attributes);
            });
        })
    }));

    jsxgraph.scope.addFunction(new funcObj('jxg_set_term',[TJSXGraphObject, TExpression], TJSXGraphAction, null, {
        random: false,
        evaluate: make_action(function(args,scope) {
            var tboard = args[0].board;
            tboard.boardPromise.then(function(board) {
                var object = args[0].get();
                var expr = args[1];
                var plotter = make_function_plotter(board, expr.tree, scope);
                if(object.elType == 'implicitcurve') {
                    object.f = plotter;
                    object.updateDataArray();
                } else {
                    object.generateTerm('x','x', plotter);
                }
                board.update();
            });
        })
    }));

    jsxgraph.scope.addFunction(new funcObj('jxg_add_objects',[TJSXGraphBoard, sig_jme_objects], TJSXGraphBoard, null, {
        random: false,
        evaluate: function(args,scope) {
            args[0].add_jme_objects(args[1],scope);
            return args[0];
        }
    }));

    jsxgraph.scope.addFunction(new funcObj('jxg_run_jessiecode',[TJSXGraphBoard, TString], TJSXGraphBoard, null, {
        random: false,
        evaluate: function(args,scope) {
            var code = args[1].value;
            args[0].run_jessiecode(code);
            return args[0];
        }
    }));
});

/*
    JSXGraph 1.12.0

    Copyright 2008-2025
        Matthias Ehmann,
        Michael Gerhaeuser,
        Carsten Miller,
        Bianca Valentin,
        Andreas Walter,
        Alfred Wassermann,
        Peter Wilfahrt

    This file is part of JSXGraph.

    JSXGraph is free software dual licensed under the GNU LGPL or MIT License.

    You can redistribute it and/or modify it under the terms of the

      * GNU Lesser General Public License as published by
        the Free Software Foundation, either version 3 of the License, or
        (at your option) any later version
      OR
      * MIT License: https://github.com/jsxgraph/jsxgraph/blob/master/LICENSE.MIT

    JSXGraph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License and
    the MIT License along with JSXGraph. If not, see <https://www.gnu.org/licenses/>
    and <https://opensource.org/licenses/MIT/>.
*/

/**
 * UTF-8 Decoder by Bjoern Hoehrmann
 *
 * Copyright (c) 2008-2009 Bjoern Hoehrmann <bjoern@hoehrmann.de>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this
 * software and associated documentation files (the "Software"), to deal in the Software
 * without restriction, including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
!function(t,e){"object"==typeof exports&&"object"==typeof module?module.exports=e():"function"==typeof define&&define.amd?define([],e):"object"==typeof exports?exports.JXG=e():t.JXG=e()}("undefined"!=typeof self?self:this,(function(){return function(){"use strict";var __webpack_modules__={351:function(t,e,i){var s,r=i(765),o="1.12.0";s={version:o,licenseText:"JSXGraph v"+o+" © jsxgraph.org",licenseLogo:"data:image/png;base64,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",COORDS_BY_USER:1,COORDS_BY_SCREEN:2,OBJECT_TYPE_ARC:1,OBJECT_TYPE_ARROW:2,OBJECT_TYPE_AXIS:3,OBJECT_TYPE_AXISPOINT:4,OBJECT_TYPE_TICKS:5,OBJECT_TYPE_CIRCLE:6,OBJECT_TYPE_CONIC:7,OBJECT_TYPE_CURVE:8,OBJECT_TYPE_GLIDER:9,OBJECT_TYPE_IMAGE:10,OBJECT_TYPE_LINE:11,OBJECT_TYPE_POINT:12,OBJECT_TYPE_SLIDER:13,OBJECT_TYPE_CAS:14,OBJECT_TYPE_GXTCAS:15,OBJECT_TYPE_POLYGON:16,OBJECT_TYPE_SECTOR:17,OBJECT_TYPE_TEXT:18,OBJECT_TYPE_ANGLE:19,OBJECT_TYPE_INTERSECTION:20,OBJECT_TYPE_TURTLE:21,OBJECT_TYPE_VECTOR:22,OBJECT_TYPE_OPROJECT:23,OBJECT_TYPE_GRID:24,OBJECT_TYPE_TANGENT:25,OBJECT_TYPE_HTMLSLIDER:26,OBJECT_TYPE_CHECKBOX:27,OBJECT_TYPE_INPUT:28,OBJECT_TYPE_BUTTON:29,OBJECT_TYPE_TRANSFORMATION:30,OBJECT_TYPE_FOREIGNOBJECT:31,OBJECT_TYPE_VIEW3D:32,OBJECT_TYPE_POINT3D:33,OBJECT_TYPE_LINE3D:34,OBJECT_TYPE_PLANE3D:35,OBJECT_TYPE_CURVE3D:36,OBJECT_TYPE_SURFACE3D:37,OBJECT_TYPE_MEASUREMENT:38,OBJECT_TYPE_INTERSECTION_LINE3D:39,OBJECT_TYPE_SPHERE3D:40,OBJECT_TYPE_CIRCLE3D:41,OBJECT_TYPE_INTERSECTION_CIRCLE3D:42,OBJECT_TYPE_TEXT3D:43,OBJECT_TYPE_FACE3D:44,OBJECT_TYPE_POLYHEDRON3D:45,OBJECT_TYPE_POLYGON3D:46,OBJECT_CLASS_POINT:1,OBJECT_CLASS_LINE:2,OBJECT_CLASS_CIRCLE:3,OBJECT_CLASS_CURVE:4,OBJECT_CLASS_AREA:5,OBJECT_CLASS_OTHER:6,OBJECT_CLASS_TEXT:7,OBJECT_CLASS_3D:8},r.Z.extendConstants(r.Z,s),e.Z=s},705:function(t,e,i){var s=i(765),r=i(351),o=i(88),n=i(109),a=i(275);s.Z.Coords=function(t,e,i,s){this.board=i,this.usrCoords=[],this.scrCoords=[],this.emitter=!n.Z.exists(s)||s,this.emitter&&o.Z.eventify(this),this.setCoordinates(t,e,!1,!0)},s.Z.extend(s.Z.Coords.prototype,{normalizeUsrCoords:function(){Math.abs(this.usrCoords[0])>a.Z.eps&&(this.usrCoords[1]/=this.usrCoords[0],this.usrCoords[2]/=this.usrCoords[0],this.usrCoords[0]=1)},usr2screen:function(t){var e=Math.round,i=this.board,s=this.usrCoords,r=i.origin.scrCoords;!0===t?(this.scrCoords[0]=e(s[0]),this.scrCoords[1]=e(s[0]*r[1]+s[1]*i.unitX),this.scrCoords[2]=e(s[0]*r[2]-s[2]*i.unitY)):(this.scrCoords[0]=s[0],this.scrCoords[1]=s[0]*r[1]+s[1]*i.unitX,this.scrCoords[2]=s[0]*r[2]-s[2]*i.unitY)},screen2usr:function(){var t=this.board.origin.scrCoords,e=this.scrCoords,i=this.board;this.usrCoords[0]=1,this.usrCoords[1]=(e[1]-t[1])/i.unitX,this.usrCoords[2]=(t[2]-e[2])/i.unitY},distance:function(t,e){var i,s,o=this.usrCoords,n=this.scrCoords;return t===r.Z.COORDS_BY_USER?(i=e.usrCoords,(s=o[0]-i[0])*s>a.Z.eps*a.Z.eps?Number.POSITIVE_INFINITY:a.Z.hypot(o[1]-i[1],o[2]-i[2])):(i=e.scrCoords,a.Z.hypot(n[1]-i[1],n[2]-i[2]))},setCoordinates:function(t,e,i,s){var o=this.usrCoords,n=this.scrCoords,a=[o[0],o[1],o[2]],h=[n[0],n[1],n[2]];return t===r.Z.COORDS_BY_USER?(2===e.length?(o[0]=1,o[1]=e[0],o[2]=e[1]):(o[0]=e[0],o[1]=e[1],o[2]=e[2],this.normalizeUsrCoords()),this.usr2screen(i)):(2===e.length?(n[1]=e[0],n[2]=e[1]):(n[1]=e[1],n[2]=e[2]),this.screen2usr()),!this.emitter||s||h[1]===n[1]&&h[2]===n[2]||this.triggerEventHandlers(["update"],[a,h]),this},copy:function(t,e){return void 0===e&&(e=0),this[t].slice(e)},isReal:function(){return!isNaN(this.usrCoords[1]+this.usrCoords[2])&&Math.abs(this.usrCoords[0])>a.Z.eps},__evt__update:function(t,e){},__evt:function(){}}),e.Z=s.Z.Coords},218:function(t,e,i){var s=i(765),r=i(275),o=i(922),n=i(336),a=i(309),h=i(705),l=i(351),c=i(109);s.Z.CoordsElement=function(t,e){var i;for(c.Z.exists(t)||(t=[1,0,0]),i=0;i<t.length;++i)t[i]=parseFloat(t[i]);this.coords=new h.Z(l.Z.COORDS_BY_USER,t,this.board),this.initialCoords=new h.Z(l.Z.COORDS_BY_USER,t,this.board),this.actualCoords=new h.Z(l.Z.COORDS_BY_USER,t,this.board),this.position=null,this.isConstrained=!1,this.onPolygon=!1,this.slideObject=null,this.slideObjects=[],this.needsUpdateFromParent=!0,this.groups=[],this.Xjc=null,this.Yjc=null,this.methodMap=c.Z.deepCopy(this.methodMap,{move:"moveTo",moveTo:"moveTo",moveAlong:"moveAlong",visit:"visit",glide:"makeGlider",makeGlider:"makeGlider",intersect:"makeIntersection",makeIntersection:"makeIntersection",X:"X",Y:"Y",Coords:"Coords",free:"free",setPosition:"setGliderPosition",setGliderPosition:"setGliderPosition",addConstraint:"addConstraint",dist:"Dist",Dist:"Dist",onPolygon:"onPolygon",startAnimation:"startAnimation",stopAnimation:"stopAnimation"}),c.Z.exists(this.element)&&this.addAnchor(t,e),this.isDraggable=!0},s.Z.extend(s.Z.CoordsElement.prototype,{updateConstraint:function(){return this},updateCoords:function(t){return this.needsUpdate?(c.Z.exists(t)||(t=!1),this.evalVisProp("frozen")||this.updateConstraint(),this.type===l.Z.OBJECT_TYPE_GLIDER&&(this.isConstrained&&(t=!1),t?this.updateGliderFromParent():this.updateGlider()),this.updateTransform(t),this):this},updateGlider:function(){var t,e,i,s,n,a,d,u,p,f,_,m,g,b,v,Z,C,y,P,E,x=2*Math.PI,O=!1,M=this.slideObject,w=[];if(this.needsUpdateFromParent=!1,M.elementClass===l.Z.OBJECT_CLASS_CIRCLE)this.evalVisProp("isgeonext")&&(x=1),b=o.Z.projectPointToCircle(this,M,this.board),v=o.Z.rad([M.center.X()+1,M.center.Y()],M.center,this)/x;else if(M.elementClass===l.Z.OBJECT_CLASS_LINE){if(this.onPolygon){if(s=M.point1.coords.usrCoords,e=(n=M.point2.coords.usrCoords)[t=1]-s[t],Math.abs(e)<r.Z.eps&&(e=n[t=2]-s[t]),d=(o.Z.projectPointToLine(this,M,this.board).usrCoords[t]-s[t])/e,a=M.parentPolygon,d<0){for(t=0;t<a.borders.length;t++)if(M===a.borders[t]){M=a.borders[(t-1+a.borders.length)%a.borders.length];break}}else if(d>1)for(t=0;t<a.borders.length;t++)if(M===a.borders[t]){M=a.borders[(t+1+a.borders.length)%a.borders.length];break}M.id!==this.slideObject.id&&(this.slideObject=M)}s=M.point1.coords,n=M.point2.coords,(e=s.distance(l.Z.COORDS_BY_USER,n))<r.Z.eps?(b=s,O=!0,v=0):(b=o.Z.projectPointToLine(this,M,this.board),s=s.usrCoords.slice(0),n=n.usrCoords.slice(0),Math.abs(n[0])<r.Z.eps?(e=n[t=1],Math.abs(e)<r.Z.eps&&(e=n[t=2]),v=((e=(b.usrCoords[t]-s[t])/e)>=0?1:-1)*(e=Math.abs(e))/(e+1)):Math.abs(s[0])<r.Z.eps?(e=s[t=1],Math.abs(e)<r.Z.eps&&(e=s[t=2]),v=(e=(b.usrCoords[t]-n[t])/e)<0?(1-2*e)/(1-e):1/(e+1)):(e=n[t=1]-s[t],Math.abs(e)<r.Z.eps&&(e=n[t=2]-s[t]),v=(b.usrCoords[t]-s[t])/e)),null!==(C=this.findClosestSnapValue(v))?(v=(this.evalVisProp("snapvalues")[C]-this._smin)/(this._smax-this._smin),this.update(!0)):(Z=this.evalVisProp("snapwidth"))>0&&Math.abs(this._smax-this._smin)>=r.Z.eps&&(i=(v=Math.max(Math.min(v,1),0))*(this._smax-this._smin),v=((i=Math.round(i/Z)*Z+this._smin)-this._smin)/(this._smax-this._smin),this.update(!0)),s=M.point1.coords,!M.evalVisProp("straightfirst")&&Math.abs(s.usrCoords[0])>r.Z.eps&&v<0&&(b=s,O=!0,v=0),n=M.point2.coords,!M.evalVisProp("straightlast")&&Math.abs(n.usrCoords[0])>r.Z.eps&&v>1&&(b=n,O=!0,v=1)}else if(M.type===l.Z.OBJECT_TYPE_TURTLE)this.updateConstraint(),b=(y=o.Z.projectPointToTurtle(this,M,this.board))[0],v=y[1];else if(M.elementClass===l.Z.OBJECT_CLASS_CURVE)if(M.type===l.Z.OBJECT_TYPE_ARC||M.type===l.Z.OBJECT_TYPE_SECTOR)b=o.Z.projectPointToCircle(this,M,this.board),u=o.Z.rad(M.radiuspoint,M.center,this),p=0,f=o.Z.rad(M.radiuspoint,M.center,M.anglepoint),v=u,("minor"===(Z=M.evalVisProp("selection"))&&f>Math.PI||"major"===Z&&f<Math.PI)&&(p=f,f=2*Math.PI),(u<p||u>f)&&(v=f,(u<p&&u>.5*p||u>f&&u>.5*f+Math.PI)&&(v=p),this.needsUpdateFromParent=!0,this.updateGliderFromParent()),x=f-p,this.visProp.isgeonext&&(x=1),Math.abs(x)>r.Z.eps&&(v/=x);else if(this.updateConstraint(),M.transformations.length>0){for(E=!1,(y=M.getTransformationSource())[0]&&(E=y[0],w.push(M),w.push(y[1]));y[0]&&c.Z.exists(y[1]._transformationSource);)y=y[1].getTransformationSource(),w.push(y[1]);if(P=this.coords.usrCoords,E){for(t=0;t<w.length;t++)w[t].updateTransformMatrix(),g=r.Z.inverse(w[t].transformMat),P=r.Z.matVecMult(g,P);for(_=new h.Z(l.Z.COORDS_BY_USER,P,this.board).usrCoords,P=(m=o.Z.projectCoordsToCurve(_[1],_[2],this.position||0,w[w.length-1],this.board))[0].usrCoords,t=w.length-2;t>=0;t--)P=r.Z.matVecMult(w[t].transformMat,P);m[0]=new h.Z(l.Z.COORDS_BY_USER,P,this.board)}else M.updateTransformMatrix(),g=r.Z.inverse(M.transformMat),P=r.Z.matVecMult(g,P),_=new h.Z(l.Z.COORDS_BY_USER,P,this.board).usrCoords,m=o.Z.projectCoordsToCurve(_[1],_[2],this.position||0,M,this.board);b=m[0],v=m[1]}else b=(y=o.Z.projectPointToCurve(this,M,this.board))[0],v=y[1];else c.Z.isPoint(M)&&(b=o.Z.projectPointToPoint(this,M,this.board),v=this.position);this.coords.setCoordinates(l.Z.COORDS_BY_USER,b.usrCoords,O),this.position=v},findClosestSnapValue:function(t){var e,i,s,o,n=null;if(s=this.evalVisProp("snapvalues"),o=this.evalVisProp("snapvaluedistance"),c.Z.isArray(s)&&Math.abs(this._smax-this._smin)>=r.Z.eps&&o>0)for(e=0;e<s.length;e++)(i=Math.abs(t*(this._smax-this._smin)+this._smin-s[e]))<o&&(o=i,n=e);return n},updateGliderFromParent:function(){var t,e,i,s,n,a,d,u,p,f,_,m,g=this.slideObject,b=[],v=2*Math.PI;if(this.needsUpdateFromParent){if(g.elementClass===l.Z.OBJECT_CLASS_CIRCLE)i=g.Radius(),this.evalVisProp("isgeonext")&&(v=1),n=[g.center.X()+i*Math.cos(this.position*v),g.center.Y()+i*Math.sin(this.position*v)];else if(g.elementClass===l.Z.OBJECT_CLASS_LINE)t=g.point1.coords.usrCoords,e=g.point2.coords.usrCoords,0===t[0]&&0===t[1]&&0===t[2]||0===e[0]&&0===e[1]&&0===e[2]?n=[0,0,0]:Math.abs(e[0])<r.Z.eps?(s=Math.min(Math.abs(this.position),1-r.Z.eps),s/=1-s,this.position<0&&(s=-s),n=[t[0]+s*e[0],t[1]+s*e[1],t[2]+s*e[2]]):Math.abs(t[0])<r.Z.eps?(s=Math.max(this.position,r.Z.eps),s=(s=Math.min(s,2-r.Z.eps))>1?(s-1)/(s-2):(1-s)/s,n=[e[0]+s*t[0],e[1]+s*t[1],e[2]+s*t[2]]):(s=this.position,n=[t[0]+s*(e[0]-t[0]),t[1]+s*(e[1]-t[1]),t[2]+s*(e[2]-t[2])]);else if(g.type===l.Z.OBJECT_TYPE_TURTLE)this.coords.setCoordinates(l.Z.COORDS_BY_USER,[g.Z(this.position),g.X(this.position),g.Y(this.position)]),this.updateConstraint(),n=o.Z.projectPointToTurtle(this,g,this.board)[0].usrCoords;else if(g.elementClass===l.Z.OBJECT_CLASS_CURVE){for(u=!1,(a=g.getTransformationSource())[0]&&(u=a[0],b.push(g),b.push(a[1]));a[0]&&c.Z.exists(a[1]._transformationSource);)a=a[1].getTransformationSource(),b.push(a[1]);if(u?this.coords.setCoordinates(l.Z.COORDS_BY_USER,[b[b.length-1].Z(this.position),b[b.length-1].X(this.position),b[b.length-1].Y(this.position)]):this.coords.setCoordinates(l.Z.COORDS_BY_USER,[g.Z(this.position),g.X(this.position),g.Y(this.position)]),g.type===l.Z.OBJECT_TYPE_ARC||g.type===l.Z.OBJECT_TYPE_SECTOR)p=o.Z.rad([g.center.X()+1,g.center.Y()],g.center,g.radiuspoint),f=0,m=o.Z.rad(g.radiuspoint,g.center,g.anglepoint),("minor"===g.visProp.selection&&m>Math.PI||"major"===g.visProp.selection&&m<Math.PI)&&(f=m,m=2*Math.PI),v=m-f,this.evalVisProp("isgeonext")&&(v=1),((_=this.position*v)<f||_>m)&&(((_=m)<f&&_>.5*f||_>m&&_>.5*m+Math.PI)&&(_=f),this.position=_,Math.abs(v)>r.Z.eps&&(this.position/=v)),i=g.Radius(),n=[g.center.X()+i*Math.cos(this.position*v+p),g.center.Y()+i*Math.sin(this.position*v+p)];else if(this.updateConstraint(),u)for(n=o.Z.projectPointToCurve(this,b[b.length-1],this.board)[0].usrCoords,d=b.length-2;d>=0;d--)n=new h.Z(l.Z.COORDS_BY_USER,r.Z.matVecMult(b[d].transformMat,n),this.board).usrCoords;else n=o.Z.projectPointToCurve(this,g,this.board)[0].usrCoords}else c.Z.isPoint(g)&&(n=o.Z.projectPointToPoint(this,g,this.board).usrCoords);this.coords.setCoordinates(l.Z.COORDS_BY_USER,n,!1)}else this.needsUpdateFromParent=!0},updateRendererGeneric:function(t){return this.needsUpdate&&this.board.renderer?(this.visPropCalc.visible&&(this.isReal=!isNaN(this.coords.usrCoords[1]+this.coords.usrCoords[2]),this.isReal=Math.abs(this.coords.usrCoords[0])>r.Z.eps&&this.isReal,this.isReal||this.updateVisibility(!1)),this.visPropCalc.visible&&this.board.renderer[t](this),this.hasLabel&&this.visPropCalc.visible&&this.label&&this.label.visPropCalc.visible&&this.isReal&&(this.label.update(),this.board.renderer.updateText(this.label)),this.setDisplayRendNode(),this.needsUpdate=!1,this):this},X:function(){return this.coords.usrCoords[1]},Y:function(){return this.coords.usrCoords[2]},Z:function(){return this.coords.usrCoords[0]},Coords:function(t){return t?this.coords.usrCoords.slice():this.coords.usrCoords.slice(1)},XEval:function(){return this.coords.usrCoords[1]},YEval:function(){return this.coords.usrCoords[2]},ZEval:function(){return this.coords.usrCoords[0]},Dist:function(t){return this.isReal&&t.isReal?this.coords.distance(l.Z.COORDS_BY_USER,t.coords):NaN},snapToGrid:function(t){return this.handleSnapToGrid(t)},handleSnapToPoints:function(t){var e,i,s,r,n,a,h,d,u=0,p=1/0,f=null,_=this.evalVisProp("ignoredsnaptopoints"),m=!1;if(r=this.board.objectsList.length,_&&(h=_.length),this.evalVisProp("snaptopoints")||t){for(n=this.evalVisProp("attractorunit"),a=this.evalVisProp("attractordistance"),e=0;e<r;e++){if(i=this.board.objectsList[e],_){for(m=!1,d=0;d<h;d++)if(i===this.board.select(_[d])){m=!0;break}if(m)continue}c.Z.isPoint(i)&&i!==this&&i.visPropCalc.visible&&(s=o.Z.projectPointToPoint(this,i,this.board),(u="screen"===n?s.distance(l.Z.COORDS_BY_SCREEN,this.coords):s.distance(l.Z.COORDS_BY_USER,this.coords))<a&&u<p&&(p=u,f=s))}null!==f&&this.coords.setCoordinates(l.Z.COORDS_BY_USER,f.usrCoords)}return this},snapToPoints:function(t){return this.handleSnapToPoints(t)},handleAttractors:function(){var t,e,i,s,r=0,n=this.evalVisProp("attractorunit"),a=this.evalVisProp("attractordistance"),d=this.evalVisProp("snatchdistance"),u=this.evalVisProp("attractors"),p=u.length;if(0!==a){for(t=0;t<p;t++)if(e=this.board.select(u[t]),c.Z.exists(e)&&e!==this){if(c.Z.isPoint(e)?i=o.Z.projectPointToPoint(this,e,this.board):e.elementClass===l.Z.OBJECT_CLASS_LINE?(s=o.Z.projectCoordsToSegment(this.coords.usrCoords,e.point1.coords.usrCoords,e.point2.coords.usrCoords),i=!e.evalVisProp("straightfirst")&&s[1]<0?e.point1.coords:!e.evalVisProp("straightlast")&&s[1]>1?e.point2.coords:new h.Z(l.Z.COORDS_BY_USER,s[0],this.board)):e.elementClass===l.Z.OBJECT_CLASS_CIRCLE?i=o.Z.projectPointToCircle(this,e,this.board):e.elementClass===l.Z.OBJECT_CLASS_CURVE?i=o.Z.projectPointToCurve(this,e,this.board)[0]:e.type===l.Z.OBJECT_TYPE_TURTLE?i=o.Z.projectPointToTurtle(this,e,this.board)[0]:e.type===l.Z.OBJECT_TYPE_POLYGON&&(i=new h.Z(l.Z.COORDS_BY_USER,o.Z.projectCoordsToPolygon(this.coords.usrCoords,e),this.board)),(r="screen"===n?i.distance(l.Z.COORDS_BY_SCREEN,this.coords):i.distance(l.Z.COORDS_BY_USER,this.coords))<a){this.type===l.Z.OBJECT_TYPE_GLIDER&&(e===this.slideObject||this.slideObject&&this.onPolygon&&this.slideObject.parentPolygon===e)||this.makeGlider(e);break}r>=d&&(e===this.slideObject||this.slideObject&&this.onPolygon&&this.slideObject.parentPolygon===e)&&this.popSlideObject()}return this}},setPositionDirectly:function(t,e){var i,s,o,n,c,d=this.coords;if(this.relativeCoords)return s=new h.Z(t,e,this.board),this.evalVisProp("islabel")?(o=a.Z.subtract(s.scrCoords,d.scrCoords),this.relativeCoords.scrCoords[1]+=o[1],this.relativeCoords.scrCoords[2]+=o[2]):(o=a.Z.subtract(s.usrCoords,d.usrCoords),this.relativeCoords.usrCoords[1]+=o[1],this.relativeCoords.usrCoords[2]+=o[2]),this;if(this.coords.setCoordinates(t,e),this.handleSnapToGrid(),this.handleSnapToPoints(),this.handleAttractors(),this.actualCoords.setCoordinates(l.Z.COORDS_BY_USER,this.coords.usrCoords),this.transformations.length>0){for(t===l.Z.COORDS_BY_SCREEN?c=new h.Z(t,e,this.board).usrCoords:(2===e.length&&(e=[1].concat(e)),c=e),n=[[1,0,0],[0,1,0],[0,0,1]],i=0;i<this.transformations.length;i++)n=r.Z.matMatMult(this.transformations[i].matrix,n);c=r.Z.matVecMult(r.Z.inverse(n),c),this.initialCoords.setCoordinates(l.Z.COORDS_BY_USER,c),this.elementClass!==l.Z.OBJECT_CLASS_POINT&&this.coords.setCoordinates(l.Z.COORDS_BY_USER,c)}return this.prepareUpdate().update(),this.board.isSuspendedUpdate&&this.type===l.Z.OBJECT_TYPE_GLIDER&&this.updateGlider(),this},setPositionByTransform:function(t,e){var i;return e=new h.Z(t,e,this.board),i=this.board.create("transform",e.usrCoords.slice(1),{type:"translate"}),this.transformations.length>0&&this.transformations[this.transformations.length-1].isNumericMatrix?this.transformations[this.transformations.length-1].melt(i):this.addTransform(this,i),this.prepareUpdate().update(),this},setPosition:function(t,e){return this.setPositionDirectly(t,e)},setGliderPosition:function(t){return this.type===l.Z.OBJECT_TYPE_GLIDER&&(this.position=t,this.board.update()),this},makeGlider:function(t){var e,i,r,o=this.board.select(t),n=!1;if(o.type===l.Z.OBJECT_TYPE_POLYGON){for(e=Number.MAX_VALUE,i=0;i<o.borders.length;i++)(r=s.Z.Math.Geometry.distPointLine(this.coords.usrCoords,o.borders[i].stdform))<e&&(e=r,t=o.borders[i]);o=this.board.select(t),n=!0}if(!c.Z.exists(o))throw new Error("JSXGraph: slide object undefined.");if(o.type===l.Z.OBJECT_TYPE_TICKS)throw new Error("JSXGraph: gliders on ticks are not possible.");return this.slideObject=this.board.select(t),this.slideObjects.push(this.slideObject),this.addParents(t),this.type=l.Z.OBJECT_TYPE_GLIDER,this.elType="glider",this.visProp.snapwidth=-1,this.slideObject.addChild(this),this.isDraggable=!0,this.onPolygon=n,this.generatePolynomial=function(){return this.slideObject.generatePolynomial(this)},this.updateGlider(),this.needsUpdateFromParent=!0,this.updateGliderFromParent(),this},popSlideObject:function(){this.slideObjects.length>0&&(this.slideObjects.pop(),this.slideObject.removeChild(this),0===this.slideObjects.length?(this.type=this._org_type,this.type===l.Z.OBJECT_TYPE_POINT?this.elType="point":this.elementClass===l.Z.OBJECT_CLASS_TEXT?this.elType="text":this.type===l.Z.OBJECT_TYPE_IMAGE?this.elType="image":this.type===l.Z.OBJECT_TYPE_FOREIGNOBJECT&&(this.elType="foreignobject"),this.slideObject=null):this.slideObject=this.slideObjects[this.slideObjects.length-1])},free:function(){var t,e;if(this.type!==l.Z.OBJECT_TYPE_GLIDER){if(this.transformations.length=0,delete this.updateConstraint,this.isConstrained=!1,this.isDraggable)return;this.isDraggable=!0,this.elementClass===l.Z.OBJECT_CLASS_POINT&&(this.type=l.Z.OBJECT_TYPE_POINT,this.elType="point"),this.XEval=function(){return this.coords.usrCoords[1]},this.YEval=function(){return this.coords.usrCoords[2]},this.ZEval=function(){return this.coords.usrCoords[0]},this.Xjc=null,this.Yjc=null}for(t in this.board.objects)this.board.objects.hasOwnProperty(t)&&(e=this.board.objects[t]).descendants&&(delete e.descendants[this.id],delete e.childElements[this.id],this.hasLabel&&(delete e.descendants[this.label.id],delete e.childElements[this.label.id]));this.ancestors={},this.parents=[],this.slideObject=null,this.slideObjects=[],this.elementClass===l.Z.OBJECT_CLASS_POINT?(this.type=l.Z.OBJECT_TYPE_POINT,this.elType="point"):this.elementClass===l.Z.OBJECT_CLASS_TEXT?(this.type=this._org_type,this.elType="text"):this.elementClass===l.Z.OBJECT_CLASS_OTHER&&(this.type=this._org_type,this.elType="image")},addConstraint:function(t){var e,i,s=[],r=["X","Y"],o=function(t){return function(){return t}},n=function(t){return function(){return t.Value()}};for(this.elementClass===l.Z.OBJECT_CLASS_POINT&&(this.type=l.Z.OBJECT_TYPE_CAS),this.isDraggable=!1,e=0;e<t.length;e++)i=t[e],c.Z.isString(i)?(s[e]=this.board.jc.snippet(i,!0,null,!0),this.addParentsFromJCFunctions([s[e]]),2===t.length&&(this[r[e]+"jc"]=t[e])):c.Z.isFunction(i)?s[e]=i:c.Z.isNumber(i)?s[e]=o(i):c.Z.isObject(i)&&c.Z.isFunction(i.Value)&&(s[e]=n(i)),s[e].origin=i;return 1===t.length?this.updateConstraint=function(){var t=s[0]();return c.Z.isArray(t)?this.coords.setCoordinates(l.Z.COORDS_BY_USER,t):this.coords=t,this}:2===t.length?(this.XEval=s[0],this.YEval=s[1],this.addParents([s[0].origin,s[1].origin]),this.updateConstraint=function(){return this.coords.setCoordinates(l.Z.COORDS_BY_USER,[this.XEval(),this.YEval()]),this}):(this.ZEval=s[0],this.XEval=s[1],this.YEval=s[2],this.addParents([s[0].origin,s[1].origin,s[2].origin]),this.updateConstraint=function(){return this.coords.setCoordinates(l.Z.COORDS_BY_USER,[this.ZEval(),this.XEval(),this.YEval()]),this}),this.isConstrained=!0,this.prepareUpdate().update(),this.board.isSuspendedUpdate||(this.updateVisibility().updateRenderer(),this.hasLabel&&this.label.fullUpdate()),this},addAnchor:function(t,e){this.relativeCoords=e?new h.Z(l.Z.COORDS_BY_SCREEN,t.slice(0,2),this.board):new h.Z(l.Z.COORDS_BY_USER,t,this.board),this.element.addChild(this),e&&this.addParents(this.element),this.XEval=function(){var t,e,i;return this.evalVisProp("islabel")?(i=this.evalVisProp("offset"),t=parseFloat(i[0]),e=this.element.getLabelAnchor(),new h.Z(l.Z.COORDS_BY_SCREEN,[t+this.relativeCoords.scrCoords[1]+e.scrCoords[1],0],this.board).usrCoords[1]):(e=this.element.getTextAnchor(),this.relativeCoords.usrCoords[1]+e.usrCoords[1])},this.YEval=function(){var t,e,i;return this.evalVisProp("islabel")?(i=this.evalVisProp("offset"),t=-parseFloat(i[1]),e=this.element.getLabelAnchor(),new h.Z(l.Z.COORDS_BY_SCREEN,[0,t+this.relativeCoords.scrCoords[2]+e.scrCoords[2]],this.board).usrCoords[2]):(e=this.element.getTextAnchor(),this.relativeCoords.usrCoords[2]+e.usrCoords[2])},this.ZEval=c.Z.createFunction(1,this.board,""),this.updateConstraint=function(){this.coords.setCoordinates(l.Z.COORDS_BY_USER,[this.ZEval(),this.XEval(),this.YEval()])},this.isConstrained=!0,this.updateConstraint()},updateTransform:function(t){var e,i;if(0===this.transformations.length||null===this.baseElement)return this;for(this.transformations[0].update(),e=this===this.baseElement?this.transformations[0].apply(this,"self"):this.transformations[0].apply(this.baseElement),i=1;i<this.transformations.length;i++)this.transformations[i].update(),e=r.Z.matVecMult(this.transformations[i].matrix,e);return this.actualCoords.setCoordinates(l.Z.COORDS_BY_USER,e),this},addTransform:function(t,e){var i,s=c.Z.isArray(e)?e:[e],r=s.length;for(0===this.transformations.length&&(this.baseElement=t),i=0;i<r;i++)this.transformations.push(s[i]);return this},startAnimation:function(t,e,i,s){var r=c.Z.evaluate(t),o=c.Z.evaluate(e),n=this;return i=c.Z.evaluate(i)||250,s="undefined"!==(s=c.Z.evaluate(s))?s:-1,this.type!==l.Z.OBJECT_TYPE_GLIDER||c.Z.exists(this.intervalCode)||0===s||(this.roundsCount=0,this.intervalCode=window.setInterval((function(){n._anim(r,o,s)}),i),c.Z.exists(this.intervalCount)||(this.intervalCount=0)),this},stopAnimation:function(){return c.Z.exists(this.intervalCode)&&(window.clearInterval(this.intervalCode),delete this.intervalCode),this},moveAlong:function(t,e,i){i=i||{};var s,r,o,a,h,d=[],u=[],p=e/this.board.attr.animationdelay,f=function(e,i){return function(){return t[e][i]}};if(c.Z.isArray(t)){for(o=t.length,s=0;s<o;s++)c.Z.isPoint(t[s])?u[s]=t[s]:u[s]={elementClass:l.Z.OBJECT_CLASS_POINT,X:f(s,0),Y:f(s,1)};if(0===(e=e||0))return this.setPosition(l.Z.COORDS_BY_USER,[u[u.length-1].X(),u[u.length-1].Y()]),this.board.update(this);if(!c.Z.exists(i.interpolate)||i.interpolate)for(r=n.Z.Neville(u),s=0;s<p;s++)d[s]=[],d[s][0]=r[0]((p-s)/p*r[3]()),d[s][1]=r[1]((p-s)/p*r[3]());else{for(o=t.length-1,s=0;s<p;++s)h=s/p*o-(a=Math.floor(s/p*o)),d[s]=[],d[s][0]=(1-h)*u[a].X()+h*u[a+1].X(),d[s][1]=(1-h)*u[a].Y()+h*u[a+1].Y();d.push([u[o].X(),u[o].Y()]),d.reverse()}this.animationPath=d}else c.Z.isFunction(t)&&(this.animationPath=t,this.animationStart=(new Date).getTime());return this.animationCallback=i.callback,this.board.addAnimation(this),this},moveTo:function(t,e,i){i=i||{},t=new h.Z(l.Z.COORDS_BY_USER,t,this.board);var s,o=this.board.attr.animationdelay,n=Math.ceil(e/o),a=[],d=this.coords.usrCoords[1],u=this.coords.usrCoords[2],p=t.usrCoords[1]-d,f=t.usrCoords[2]-u,_=function(t){var e=t/n;if(i.effect){if("<>"===i.effect)return Math.pow(Math.sin(e*Math.PI/2),2);if("<"===i.effect)return e*e*e;if(">"===i.effect)return 1-Math.pow(1-e,3);if("=="===i.effect)return t/n;throw new Error("valid effects are '==', '<>', '>', and '<'.")}return t/n};if(!c.Z.exists(e)||0===e||Math.abs(t.usrCoords[0]-this.coords.usrCoords[0])>r.Z.eps)return this.setPosition(l.Z.COORDS_BY_USER,t.usrCoords),this.board.update(this);if(!c.Z.exists(i.callback)&&Math.abs(p)<r.Z.eps&&Math.abs(f)<r.Z.eps)return this;for(s=n;s>=0;s--)a[n-s]=[t.usrCoords[0],d+p*_(s),u+f*_(s)];return this.animationPath=a,this.animationCallback=i.callback,this.board.addAnimation(this),this},visit:function(t,e,i){t=new h.Z(l.Z.COORDS_BY_USER,t,this.board);var s,r,o,n=this.board.attr.animationdelay,a=[],d=this.coords.usrCoords[1],u=this.coords.usrCoords[2],p=t.usrCoords[1]-d,f=t.usrCoords[2]-u,_=function(t){var e=t<o/2?2*t/o:2*(o-t)/o;if(i.effect){if("<>"===i.effect)return Math.pow(Math.sin(e*Math.PI/2),2);if("<"===i.effect)return e*e*e;if(">"===i.effect)return 1-Math.pow(1-e,3);if("=="===i.effect)return e;throw new Error("valid effects are '==', '<>', '>', and '<'.")}return e};for(c.Z.isNumber(i)?i={repeat:i}:(i=i||{},c.Z.exists(i.repeat)||(i.repeat=1)),o=Math.ceil(e/(n*i.repeat)),r=0;r<i.repeat;r++)for(s=o;s>=0;s--)a[r*(o+1)+o-s]=[t.usrCoords[0],d+p*_(s),u+f*_(s)];return this.animationPath=a,this.animationCallback=i.callback,this.board.addAnimation(this),this},moveAlongES6:function(t,e,i){return new Promise(((s,r)=>{c.Z.exists(i)&&c.Z.exists(i.callback)?i.callback=s:i={callback:s},this.moveAlong(t,e,i)}))},moveToES6:function(t,e,i){return new Promise(((s,r)=>{c.Z.exists(i)&&c.Z.exists(i.callback)?i.callback=s:i={callback:s},this.moveTo(t,e,i)}))},visitES6:function(t,e,i){return new Promise(((s,r)=>{c.Z.exists(i)&&c.Z.exists(i.callback)?i.callback=s:i={callback:s},this.visit(t,e,i)}))},_anim:function(t,e,i){var s,r,n,a,h,c,d,u,p;return this.intervalCount+=1,this.intervalCount>e&&(this.intervalCount=0,this.roundsCount+=1,i>0&&this.roundsCount>=i)?(this.roundsCount=0,this.stopAnimation()):(this.slideObject.elementClass===l.Z.OBJECT_CLASS_LINE?(d=this.slideObject.point1.coords.scrCoords,u=this.slideObject.point2.coords.scrCoords,s=Math.round((u[1]-d[1])*this.intervalCount/e),r=Math.round((u[2]-d[2])*this.intervalCount/e),t>0?a=this.slideObject.point1:(a=this.slideObject.point2,s*=-1,r*=-1),this.coords.setCoordinates(l.Z.COORDS_BY_SCREEN,[a.coords.scrCoords[1]+s,a.coords.scrCoords[2]+r])):this.slideObject.elementClass===l.Z.OBJECT_CLASS_CURVE?(h=t>0?(this.slideObject.maxX()-this.slideObject.minX())*this.intervalCount/e+this.slideObject.minX():-(this.slideObject.maxX()-this.slideObject.minX())*this.intervalCount/e+this.slideObject.maxX(),this.coords.setCoordinates(l.Z.COORDS_BY_USER,[this.slideObject.X(h),this.slideObject.Y(h)]),p=o.Z.projectPointToCurve(this,this.slideObject,this.board),this.coords=p[0],this.position=p[1]):this.slideObject.elementClass===l.Z.OBJECT_CLASS_CIRCLE&&(n=2*Math.PI,n*=t<0?this.intervalCount/e:(e-this.intervalCount)/e,c=this.slideObject.Radius(),this.coords.setCoordinates(l.Z.COORDS_BY_USER,[this.slideObject.center.coords.usrCoords[1]+c*Math.cos(n),this.slideObject.center.coords.usrCoords[2]+c*Math.sin(n)])),this.board.update(this),this)},getTextAnchor:function(){return this.coords},getLabelAnchor:function(){return this.coords},getParents:function(){var t=[this.Z(),this.X(),this.Y()];return 0!==this.parents.length&&(t=this.parents),this.type===l.Z.OBJECT_TYPE_GLIDER&&(t=[this.X(),this.Y(),this.slideObject.id]),t}}),s.Z.CoordsElement.create=function(t,e,i,s,r,o){var n,a,h=!1;for(a=0;a<i.length;a++)(c.Z.isFunction(i[a])||c.Z.isString(i[a]))&&(h=!0);if(h)(n=new t(e,[0,0],s,r,o)).addConstraint(i);else if(c.Z.isNumber(i[0])&&c.Z.isNumber(i[1]))n=new t(e,i,s,r,o),c.Z.exists(s.slideobject)?n.makeGlider(s.slideobject):n.baseElement=n,n.isDraggable=!0;else{if(!c.Z.isObject(i[0])||!c.Z.isTransformationOrArray(i[1]))return!1;(n=new t(e,[0,0],s,r,o)).addTransform(i[0],i[1]),n.isDraggable=!1}return n.handleSnapToGrid(),n.handleSnapToPoints(),n.handleAttractors(),n.addParents(i),n},e.Z=s.Z.CoordsElement},958:function(t,e,i){var s=i(765),r=i(351),o=i(705),n=i(275),a=i(309),h=i(766),l=i(88),c=i(327),d=i(109);s.Z.GeometryElement=function(t,e,i,s){var o,n,a;if(this.needsUpdate=!0,this.isDraggable=!1,this.isReal=!0,this.childElements={},this.hasLabel=!1,this.highlighted=!1,this.notExistingParents={},this.traces={},this.numTraces=0,this.transformations=[],this.baseElement=null,this.descendants={},this.ancestors={},this.parents=[],this.symbolic={},this.rendNode=null,this.elType="",this.dump=!0,this.subs={},this.inherits=[],this._pos=-1,this.stdform=[1,0,0,0,1,1,0,0],this.methodMap={setLabel:"setLabel",label:"label",setName:"setName",getName:"getName",Name:"getName",addTransform:"addTransform",setProperty:"setAttribute",setAttribute:"setAttribute",addChild:"addChild",animate:"animate",on:"on",off:"off",trigger:"trigger",addTicks:"addTicks",removeTicks:"removeTicks",removeAllTicks:"removeAllTicks",Bounds:"bounds"},this.quadraticform=[[1,0,0],[0,1,0],[0,0,1]],this.visProp={},this.visPropCalc={visible:!1},l.Z.eventify(this),this.mouseover=!1,this.lastDragTime=new Date,this.view=null,arguments.length>0){for(n in this.board=t,this.type=i,this._org_type=i,this.elementClass=s||r.Z.OBJECT_CLASS_OTHER,this.id=e.id,o=e.name,d.Z.exists(o)||(o=this.board.generateName(this)),""!==o&&(this.board.elementsByName[o]=this),this.name=o,this.needsRegularUpdate=e.needsregularupdate,d.Z.clearVisPropOld(this),a=this.resolveShortcuts(e))a.hasOwnProperty(n)&&this._set(n,a[n]);this.visProp.draft=a.draft&&a.draft.draft}},s.Z.extend(s.Z.GeometryElement.prototype,{addChild:function(t){var e,i;for(e in this.childElements[t.id]=t,this.addDescendants(t),t.ancestors[this.id]=this,this.descendants)if(this.descendants.hasOwnProperty(e))for(i in this.descendants[e].ancestors[this.id]=this,this.ancestors)this.ancestors.hasOwnProperty(i)&&(this.descendants[e].ancestors[this.ancestors[i].id]=this.ancestors[i]);for(e in this.ancestors)if(this.ancestors.hasOwnProperty(e))for(i in this.descendants)this.descendants.hasOwnProperty(i)&&(this.ancestors[e].descendants[this.descendants[i].id]=this.descendants[i]);return this},addDescendants:function(t){var e;for(e in this.descendants[t.id]=t,t.childElements)t.childElements.hasOwnProperty(e)&&this.addDescendants(t.childElements[e]);return this},addParents:function(t){var e,i,s;for(i=(s=d.Z.isArray(t)?t:arguments).length,e=0;e<i;++e)d.Z.exists(s[e])&&(d.Z.isId(this.board,s[e])?this.parents.push(s[e]):d.Z.exists(s[e].id)&&this.parents.push(s[e].id));this.parents=d.Z.uniqueArray(this.parents)},setParents:function(t){this.parents=[],this.addParents(t)},addParentsFromJCFunctions:function(t){var e,i,s;for(e=0;e<t.length;e++)for(i in t[e].deps)s=t[e].deps[i],this.addParents(s),s.addChild(this);return this},removeChild:function(t){return delete this.childElements[t.id],this.removeDescendants(t),delete t.ancestors[this.id],this},removeDescendants:function(t){var e;for(e in delete this.descendants[t.id],t.childElements)t.childElements.hasOwnProperty(e)&&this.removeDescendants(t.childElements[e]);return this},countChildren:function(){var t,e,i=0;for(t in e=this.childElements)e.hasOwnProperty(t)&&t.indexOf("Label")<0&&i++;return i},getName:function(){return this.name},addTransform:function(t){return this},draggable:function(){return this.isDraggable&&!this.evalVisProp("fixed")&&this.type!==r.Z.OBJECT_TYPE_GLIDER},setPosition:function(t,e){var i,s,o,n,a=[];if(!d.Z.exists(this.parents))return this;for(o=this.parents.length,s=0;s<o;++s)if(i=this.board.select(this.parents[s]),d.Z.isPoint(i)){if(!i.draggable())return this;a.push(i)}if(3===e.length&&(e=e.slice(1)),n=this.board.create("transform",e,{type:"translate"}),(o=a.length)>0){if(n.applyOnce(a),d.Z.exists(this.view)&&"view3d"===this.view.elType)for(s=0;s<this.parents.length;++s)i=this.view.select(this.parents[s]),d.Z.exists(i.setPosition2D)&&i.setPosition2D(n)}else this.transformations.length>0&&this.transformations[this.transformations.length-1].isNumericMatrix?this.transformations[this.transformations.length-1].melt(n):this.addTransform(n);for(s=0;s<o;++s)a[s].type===r.Z.OBJECT_TYPE_GLIDER&&a[s].updateGlider();return this},setPositionDirectly:function(t,e,i){var s=new o.Z(t,e,this.board,!1),n=new o.Z(t,i,this.board,!1),h=a.Z.subtract(s.usrCoords,n.usrCoords);return this.setPosition(r.Z.COORDS_BY_USER,h),this},generatePolynomial:function(){return[]},animate:function(t,e,i){i=i||{};var s,r,o,n=this.board.attr.animationdelay,a=Math.ceil(e/n),h=this,l=function(t,e,i){var s,r,n,l,d;for(s=c.Z.rgb2hsv(t),n=((r=c.Z.rgb2hsv(e))[0]-s[0])/a,l=(r[1]-s[1])/a,d=(r[2]-s[2])/a,h.animationData[i]=[],o=0;o<a;o++)h.animationData[i][a-o-1]=c.Z.hsv2rgb(s[0]+(o+1)*n,s[1]+(o+1)*l,s[2]+(o+1)*d)},u=function(t,e,i,s){var r,n;if(t=parseFloat(t),e=parseFloat(e),!isNaN(t)&&!isNaN(e))for(n=(e-t)/a,h.animationData[i]=[],o=0;o<a;o++)r=t+(o+1)*n,h.animationData[i][a-o-1]=s?Math.floor(r):r};for(s in this.animationData={},t)if(t.hasOwnProperty(s))switch(r=s.toLowerCase()){case"strokecolor":case"fillcolor":l(this.visProp[r],t[s],r);break;case"size":if(!d.Z.isPoint(this))break;u(this.visProp[r],t[s],r,!0);break;case"strokeopacity":case"strokewidth":case"fillopacity":u(this.visProp[r],t[s],r,!1)}return this.animationCallback=i.callback,this.board.addAnimation(this),this},update:function(){return this.evalVisProp("trace")&&this.cloneToBackground(),this},updateRenderer:function(){return this},fullUpdate:function(t){return this.prepareUpdate().update().updateVisibility(t).updateRenderer()},setDisplayRendNode:function(t){var e,i,s,r,o;if(void 0===t&&(t=this.visPropCalc.visible),t===this.visPropOld.visible)return this;for(this.board.renderer.display(this,t),i=this.inherits.length,s=0;s<i;s++)if(o=this.inherits[s],d.Z.isArray(o))for(r=o.length,e=0;e<r;e++)d.Z.exists(o[e])&&d.Z.exists(o[e].rendNode)&&"inherit"===o[e].evalVisProp("visible")&&o[e].setDisplayRendNode(t);else d.Z.exists(o)&&d.Z.exists(o.rendNode)&&"inherit"===o.evalVisProp("visible")&&o.setDisplayRendNode(t);return this.hasLabel&&d.Z.exists(this.label)&&d.Z.exists(this.label.rendNode)&&"inherit"===this.label.evalVisProp("visible")&&this.label.setDisplayRendNode(t),this},hide:function(){return this.setAttribute({visible:!1}),this},hideElement:function(){return this.hide(),this},show:function(){return this.setAttribute({visible:!0}),this},showElement:function(){return this.show(),this},updateVisibility:function(t){var e,i,s,r,o,n;if(this.needsUpdate){if(d.Z.exists(this.view)&&!1===this.view.evalVisProp("visible"))this.visPropCalc.visible=!1;else for(void 0!==t?this.visPropCalc.visible=t:(n=this.evalVisProp("visible"),d.Z.exists(this.hiddenByParent)&&this.hiddenByParent&&(n=!1),"inherit"!==n&&(this.visPropCalc.visible=n)),i=this.inherits.length,s=0;s<i;s++)if(o=this.inherits[s],d.Z.isArray(o))for(r=o.length,e=0;e<r;e++)d.Z.exists(o[e])&&"inherit"===o[e].evalVisProp("visible")&&o[e].prepareUpdate().updateVisibility(this.visPropCalc.visible);else d.Z.exists(o)&&"inherit"===o.evalVisProp("visible")&&o.prepareUpdate().updateVisibility(this.visPropCalc.visible);d.Z.exists(this.label)&&d.Z.exists(this.label.visProp)&&this.label.evalVisProp("visible")&&this.label.prepareUpdate().updateVisibility(this.visPropCalc.visible)}return this},_set:function(t,e){var i;if(t=t.toLocaleLowerCase(),this.visProp.hasOwnProperty(t)&&t.indexOf("color")>=0&&d.Z.isString(e)&&9===e.length&&"#"===e.charAt(0))e=c.Z.rgba2rgbo(e),this.visProp[t]=e[0],this.visProp[t.replace("color","opacity")]=e[1];else if(null===e||!d.Z.isObject(e)||d.Z.exists(e.id)||d.Z.exists(e.name))this.visProp[t]=e;else for(i in this.visProp[t]={},e)e.hasOwnProperty(i)&&(this.visProp[t][i.toLocaleLowerCase()]=e[i])},resolveShortcuts:function(t){var e,i,s,r=["traceattributes","traceAttributes"];for(e in h.Z.shortcuts)if(h.Z.shortcuts.hasOwnProperty(e)){if(d.Z.exists(t[e]))for(i=0;i<h.Z.shortcuts[e].length;i++)d.Z.exists(t[h.Z.shortcuts[e][i]])||(t[h.Z.shortcuts[e][i]]=t[e]);for(s=0;s<r.length;s++)d.Z.isObject(t[r[s]])&&(t[r[s]]=this.resolveShortcuts(t[r[s]]))}return t},setLabel:function(t){this.hasLabel||this.setAttribute({withlabel:!0}),this.setLabelText(t)},setLabelText:function(t){return d.Z.exists(this.label)&&(t=t.replace(/</g,"&lt;").replace(/>/g,"&gt;"),this.label.setText(t)),this},setName:function(t){t=t.replace(/</g,"&lt;").replace(/>/g,"&gt;"),"slider"!==this.elType&&this.setLabelText(t),this.setAttribute({name:t})},setProperty:function(){s.Z.deprecated("setProperty()","setAttribute()"),this.setAttribute.apply(this,arguments)},setAttribute:function(t){var e,i,o,n,a,l,u,p,f,_={};for(e=0;e<arguments.length;e++)l=arguments[e],d.Z.isString(l)?(p=l.split(":"),_[d.Z.trim(p[0])]=d.Z.trim(p[1])):d.Z.isArray(l)?_[l[0]]=l[1]:s.Z.extend(_,l);for(e in _=this.resolveShortcuts(_))if(_.hasOwnProperty(e)){if(n=e.replace(/\s+/g,"").toLowerCase(),a=_[e],d.Z.isObject(a)&&d.Z.exists(this.visProp[n])){if(!d.Z.isObject(this.visProp[n])&&null!==a&&d.Z.isObject(a)&&(this.visProp[n]={}),d.Z.mergeAttr(this.visProp[n],a),this.type===r.Z.OBJECT_TYPE_TICKS&&d.Z.exists(this.labels))for(o=this.labels.length,i=0;i<o;i++)this.labels[i].setAttribute(a);else if(d.Z.exists(this[n]))if(d.Z.isArray(this[n]))for(i=0;i<this[n].length;i++)this[n][i].setAttribute(a);else this[n].setAttribute(a);else f=null,this.triggerEventHandlers(["attribute:"+n],[f,a,this]);continue}switch(f=this.visProp[n],n){case"checked":d.Z.exists(this.rendNodeTag)&&(this.rendNodeCheckbox.checked=!!a);break;case"disabled":d.Z.exists(this.rendNodeTag)&&(this.rendNodeTag.disabled=!!a);break;case"face":d.Z.isPoint(this)&&(this.visProp.face=a,this.board.renderer.changePointStyle(this));break;case"generatelabelvalue":this.type===r.Z.OBJECT_TYPE_TICKS&&d.Z.isFunction(a)&&(this.generateLabelValue=a);break;case"gradient":this.visProp.gradient=a,this.board.renderer.setGradient(this);break;case"gradientsecondcolor":a=c.Z.rgba2rgbo(a),this.visProp.gradientsecondcolor=a[0],this.visProp.gradientsecondopacity=a[1],this.board.renderer.updateGradient(this);break;case"gradientsecondopacity":this.visProp.gradientsecondopacity=a,this.board.renderer.updateGradient(this);break;case"infoboxtext":d.Z.isString(a)?this.infoboxText=a:this.infoboxText=!1;break;case"labelcolor":u=(a=c.Z.rgba2rgbo(a))[1],a=a[0],0===u&&d.Z.exists(this.label)&&this.hasLabel&&this.label.hideElement(),d.Z.exists(this.label)&&this.hasLabel&&(this.label.visProp.strokecolor=a,this.board.renderer.setObjectStrokeColor(this.label,a,u)),this.elementClass===r.Z.OBJECT_CLASS_TEXT&&(this.visProp.strokecolor=a,this.visProp.strokeopacity=u,this.board.renderer.setObjectStrokeColor(this,a,u));break;case"layer":this.board.renderer.setLayer(this,this.eval(a)),this._set(n,a);break;case"maxlength":d.Z.exists(this.rendNodeTag)&&(this.rendNodeTag.maxlength=!!a);break;case"name":f=this.name,delete this.board.elementsByName[this.name],this.name=a,this.board.elementsByName[this.name]=this;break;case"needsregularupdate":this.needsRegularUpdate=!("false"===a||!1===a),this.board.renderer.setBuffering(this,this.needsRegularUpdate?"auto":"static");break;case"onpolygon":this.type===r.Z.OBJECT_TYPE_GLIDER&&(this.onPolygon=!!a);break;case"radius":this.type!==r.Z.OBJECT_TYPE_ANGLE&&this.type!==r.Z.OBJECT_TYPE_SECTOR||this.setRadius(a);break;case"rotate":(this.elementClass===r.Z.OBJECT_CLASS_TEXT&&"internal"===this.evalVisProp("display")||this.type===r.Z.OBJECT_TYPE_IMAGE)&&this.addRotation(a);break;case"tabindex":d.Z.exists(this.rendNode)&&(this.rendNode.setAttribute("tabindex",a),this._set(n,a));break;case"trace":"false"===a||!1===a?(this.clearTrace(),this.visProp.trace=!1):this.visProp.trace="pause"!==a;break;case"visible":this.visProp.visible="false"!==a&&("true"===a||a),this.setDisplayRendNode(this.evalVisProp("visible")),this.evalVisProp("visible")&&d.Z.exists(this.updateSize)&&this.updateSize();break;case"withlabel":this.visProp.withlabel=a,this.evalVisProp("withlabel")?(this.label||this.createLabel(),this.label.setAttribute({visible:"inherit"})):this.label&&this.hasLabel&&this.label.setAttribute({visible:!1}),this.hasLabel=a;break;case"straightfirst":case"straightlast":for(i in this._set(n,a),this.childElements)this.childElements.hasOwnProperty(i)&&"glider"===this.childElements[i].elType&&this.childElements[i].fullUpdate();break;default:d.Z.exists(this.visProp[n])&&(!s.Z.Validator[n]||s.Z.Validator[n]&&s.Z.Validator[n](a)||s.Z.Validator[n]&&d.Z.isFunction(a)&&s.Z.Validator[n](a(this)))?(a=(!a.toLowerCase||"false"!==a.toLowerCase())&&a,this._set(n,a)):n in h.Z.shortcuts||s.Z.warn("attribute '"+n+"' does not accept type '"+typeof a+"' of value "+a+".")}this.triggerEventHandlers(["attribute:"+n],[f,a,this])}return this.triggerEventHandlers(["attribute"],[_,this]),this.evalVisProp("needsregularupdate")?this.board.update(this):this.board.fullUpdate(),this.elementClass===r.Z.OBJECT_CLASS_TEXT&&this.updateSize(),this},getProperty:function(){s.Z.deprecated("getProperty()","getAttribute()"),this.getProperty.apply(this,arguments)},getAttribute:function(t){var e;switch(t=t.toLowerCase()){case"needsregularupdate":e=this.needsRegularUpdate;break;case"labelcolor":e=this.label.visProp.strokecolor;break;case"infoboxtext":e=this.infoboxText;break;case"withlabel":e=this.hasLabel;break;default:e=this.visProp[t]}return e},evalVisProp:function(t){var e,i,r,o;if(-1===(t=t.toLowerCase()).indexOf("."))e=this.visProp[t];else for(o=(i=t.split(".")).length,e=this.visProp,r=0;r<o;r++)e=e[i[r]];return s.Z.isFunction(e)?!0===this.visProp.islabel&&d.Z.exists(this.visProp.anchor)?null!==this.visProp.anchor.visProp.element3d?e(this.visProp.anchor.visProp.element3d):e(this.visProp.anchor):s.Z.exists(this.visProp.element3d)?e(this.visProp.element3d):e(this):e},eval:function(t){return s.Z.isFunction(t)?!0===this.visProp.islabel&&d.Z.exists(this.visProp.anchor)?null!==this.visProp.anchor.visProp.element3d?t(this.visProp.anchor.visProp.element3d):t(this.visProp.anchor):null!==this.visProp.element3d?t(this.visProp.element3d):t(this):t},setDash:function(t){return this.setAttribute({dash:t}),this},prepareUpdate:function(){return this.needsUpdate=!0,this},remove:function(){return this.board.renderer.remove(this.rendNode),this.hasLabel&&this.board.renderer.remove(this.board.renderer.getElementById(this.label.id)),this},getTextAnchor:function(){return new o.Z(r.Z.COORDS_BY_USER,[0,0],this.board)},getLabelAnchor:function(){return new o.Z(r.Z.COORDS_BY_USER,[0,0],this.board)},setArrow:function(t,e){return this.visProp.firstarrow=t,this.visProp.lastarrow=e,e&&(this.type=r.Z.OBJECT_TYPE_VECTOR,this.elType="arrow"),this.prepareUpdate().update().updateVisibility().updateRenderer(),this},createGradient:function(){var t=this.evalVisProp("gradient");"linear"!==t&&"radial"!==t||this.board.renderer.setGradient(this)},createLabel:function(){var t,e=this;return s.Z.elements.text?((t=d.Z.deepCopy(this.visProp.label,null)).id=this.id+"Label",t.isLabel=!0,t.anchor=this,t.priv=this.visProp.priv,this.visProp.withlabel&&(this.label=s.Z.elements.text(this.board,[0,0,function(){return d.Z.isFunction(e.name)?e.name(e):e.name}],t),this.label.needsUpdate=!0,this.label.dump=!1,this.label.fullUpdate(),this.hasLabel=!0)):s.Z.debug("JSXGraph: Can't create label: text element is not available. Make sure you include base/text"),this},highlight:function(t){return t=d.Z.def(t,!1),!this.evalVisProp("highlight")||this.highlighted&&!t||(this.highlighted=!0,this.board.highlightedObjects[this.id]=this,this.board.renderer.highlight(this)),this},noHighlight:function(){return this.highlighted&&(this.highlighted=!1,delete this.board.highlightedObjects[this.id],this.board.renderer.noHighlight(this)),this},clearTrace:function(){var t;for(t in this.traces)this.traces.hasOwnProperty(t)&&this.board.renderer.remove(this.traces[t]);return this.numTraces=0,this},cloneToBackground:function(){return this},bounds:function(){return[0,0,0,0]},normalize:function(){return this.stdform=n.Z.normalize(this.stdform),this},toJSON:function(){var t,e,i=['{"name":',this.name];for(e in i.push(', "id":'+this.id),t=[],this.visProp)this.visProp.hasOwnProperty(e)&&d.Z.exists(this.visProp[e])&&t.push('"'+e+'":'+this.visProp[e]);return i.push(', "visProp":{'+t.toString()+"}"),i.push("}"),i.join("")},addRotation:function(t){var e,i,s,o,n,a=this;return this.elementClass!==r.Z.OBJECT_CLASS_TEXT&&this.type!==r.Z.OBJECT_TYPE_IMAGE||0===t||(e=this.board.create("transform",[function(){return-a.X()},function(){return-a.Y()}],{type:"translate"}),i=this.board.create("transform",[function(){return a.X()},function(){return a.Y()}],{type:"translate"}),s=this.board.create("transform",[function(){return a.board.unitX/a.board.unitY},function(){return 1}],{type:"scale"}),o=this.board.create("transform",[function(){return a.board.unitY/a.board.unitX},function(){return 1}],{type:"scale"}),n=this.board.create("transform",[function(){return a.eval(t)*Math.PI/180}],{type:"rotate"}),e.bindTo(this),s.bindTo(this),n.bindTo(this),o.bindTo(this),i.bindTo(this)),this},highlightStrokeColor:function(t){return s.Z.deprecated("highlightStrokeColor()","setAttribute()"),this.setAttribute({highlightStrokeColor:t}),this},strokeColor:function(t){return s.Z.deprecated("strokeColor()","setAttribute()"),this.setAttribute({strokeColor:t}),this},strokeWidth:function(t){return s.Z.deprecated("strokeWidth()","setAttribute()"),this.setAttribute({strokeWidth:t}),this},fillColor:function(t){return s.Z.deprecated("fillColor()","setAttribute()"),this.setAttribute({fillColor:t}),this},highlightFillColor:function(t){return s.Z.deprecated("highlightFillColor()","setAttribute()"),this.setAttribute({highlightFillColor:t}),this},labelColor:function(t){return s.Z.deprecated("labelColor()","setAttribute()"),this.setAttribute({labelColor:t}),this},dash:function(t){return s.Z.deprecated("dash()","setAttribute()"),this.setAttribute({dash:t}),this},visible:function(t){return s.Z.deprecated("visible()","setAttribute()"),this.setAttribute({visible:t}),this},shadow:function(t){return s.Z.deprecated("shadow()","setAttribute()"),this.setAttribute({shadow:t}),this},getType:function(){return this.elType},getParents:function(){return d.Z.isArray(this.parents)?this.parents:[]},snapToGrid:function(){return this},snapToPoints:function(){return this},getAttributes:function(){var t,e=d.Z.deepCopy(this.visProp),i=[],s=i.length;for(e.id=this.id,e.name=this.name,t=0;t<s;t++)delete e[i[t]];return e},hasPoint:function(t,e){return!1},addTicks:function(t){return""!==t.id&&d.Z.exists(t.id)||(t.id=this.id+"_ticks_"+(this.ticks.length+1)),this.board.renderer.drawTicks(t),this.ticks.push(t),t.id},removeAllTicks:function(){var t;if(d.Z.exists(this.ticks)){for(t=this.ticks.length-1;t>=0;t--)this.removeTicks(this.ticks[t]);this.ticks=[],this.board.update()}},removeTicks:function(t){var e,i;if(d.Z.exists(this.defaultTicks)&&this.defaultTicks===t&&(this.defaultTicks=null),d.Z.exists(this.ticks))for(e=this.ticks.length-1;e>=0;e--)if(this.ticks[e]===t){if(this.board.removeObject(this.ticks[e]),this.ticks[e].ticks)for(i=0;i<this.ticks[e].ticks.length;i++)d.Z.exists(this.ticks[e].labels[i])&&this.board.removeObject(this.ticks[e].labels[i]);delete this.ticks[e];break}},getSnapSizes:function(){var t,e,i;return t=this.evalVisProp("snapsizex"),e=this.evalVisProp("snapsizey"),t<=0&&this.board.defaultAxes&&this.board.defaultAxes.x.defaultTicks&&(t=(i=this.board.defaultAxes.x.defaultTicks).ticksDelta*(i.evalVisProp("minorticks")+1)),e<=0&&this.board.defaultAxes&&this.board.defaultAxes.y.defaultTicks&&(e=(i=this.board.defaultAxes.y.defaultTicks).ticksDelta*(i.evalVisProp("minorticks")+1)),[t,e]},handleSnapToGrid:function(t,e){var i,o,n,a,h,l,c,u,p,f,_,m=this.evalVisProp("attracttogrid"),g=this.evalVisProp("attractorunit"),b=this.evalVisProp("attractordistance");return!d.Z.exists(this.coords)||this.evalVisProp("fixed")||(this.evalVisProp("snaptogrid")||m||!0===t)&&(i=this.coords.usrCoords[1],o=this.coords.usrCoords[2],f=(p=this.getSnapSizes())[0],_=p[1],f>0&&_>0&&(u=this.board.getBoundingBox(),n=Math.round(i/f)*f,a=Math.round(o/_)*_,h=new s.Z.Coords(r.Z.COORDS_BY_USER,[n,a],this.board),(!m||h.distance("screen"===g?r.Z.COORDS_BY_SCREEN:r.Z.COORDS_BY_USER,this.coords)<b)&&(i=n,o=a,e||(l=Math.min(u[0],u[2]),c=Math.max(u[0],u[2]),i<l&&i>l-f?i+=f:i>c&&i<c+f&&(i-=f),l=Math.min(u[1],u[3]),c=Math.max(u[1],u[3]),o<l&&o>l-_?o+=_:o>c&&o<c+_&&(o-=_)),this.coords.setCoordinates(r.Z.COORDS_BY_USER,[i,o])))),this},getBoundingBox:function(){var t,e,i,s,o,n,a=[1/0,1/0,-1/0,-1/0];if(this.type===r.Z.OBJECT_TYPE_POLYGON){if((e=this.vertices.length-1)<=0)return a;for(t=0;t<e;t++)i=this.vertices[t].X(),a[0]=i<a[0]?i:a[0],a[2]=i>a[2]?i:a[2],i=this.vertices[t].Y(),a[1]=i<a[1]?i:a[1],a[3]=i>a[3]?i:a[3]}else if(this.elementClass===r.Z.OBJECT_CLASS_CIRCLE)s=this.center.X(),o=this.center.Y(),a=[s-this.radius,o+this.radius,s+this.radius,o-this.radius];else if(this.elementClass===r.Z.OBJECT_CLASS_CURVE){if(0===(e=this.points.length))return a;for(t=0;t<e;t++)i=this.points[t].usrCoords[1],a[0]=i<a[0]?i:a[0],a[2]=i>a[2]?i:a[2],i=this.points[t].usrCoords[2],a[1]=i<a[1]?i:a[1],a[3]=i>a[3]?i:a[3]}else this.elementClass===r.Z.OBJECT_CLASS_POINT?(s=this.X(),o=this.Y(),a=[s-(n=this.evalVisProp("size"))/this.board.unitX,o-n/this.board.unitY,s+n/this.board.unitX,o+n/this.board.unitY]):this.elementClass===r.Z.OBJECT_CLASS_LINE&&(i=this.point1.coords.usrCoords[1],a[0]=i<a[0]?i:a[0],a[2]=i>a[2]?i:a[2],i=this.point1.coords.usrCoords[2],a[1]=i<a[1]?i:a[1],a[3]=i>a[3]?i:a[3],i=this.point2.coords.usrCoords[1],a[0]=i<a[0]?i:a[0],a[2]=i>a[2]?i:a[2],i=this.point2.coords.usrCoords[2],a[1]=i<a[1]?i:a[1],a[3]=i>a[3]?i:a[3]);return a},addEvent:s.Z.shortcut(s.Z.GeometryElement.prototype,"on"),removeEvent:s.Z.shortcut(s.Z.GeometryElement.prototype,"off"),formatNumberLocale:function(t,e){var i,s,r,o={},n={maximumfractiondigits:"maximumFractionDigits",minimumfractiondigits:"minimumFractionDigits",compactdisplay:"compactDisplay",currencydisplay:"currencyDisplay",currencysign:"currencySign",localematcher:"localeMatcher",numberingsystem:"numberingSystem",signdisplay:"signDisplay",unitdisplay:"unitDisplay",usegrouping:"useGrouping",roundingmode:"roundingMode",roundingpriority:"roundingPriority",roundingincrement:"roundingIncrement",trailingzerodisplay:"trailingZeroDisplay",minimumintegerdigits:"minimumIntegerDigits",minimumsignificantdigits:"minimumSignificantDigits",maximumsignificantdigits:"maximumSignificantDigits"};if(d.Z.exists(Intl)&&this.useLocale()){for(r in i=this.evalVisProp("intl.locale")||this.eval(this.board.attr.intl.locale),s=this.evalVisProp("intl.options")||{})s.hasOwnProperty(r)&&(n.hasOwnProperty(r)?o[n[r]]=this.eval(s[r]):o[r]=this.eval(s[r]));return r="maximumfractiondigits",d.Z.exists(s[r])||(o[n[r]]=e),Intl.NumberFormat(i,o).format(t)}return t},useLocale:function(){var t;return!(!d.Z.exists(this.visProp.intl)||!d.Z.exists(this.visProp.intl.enabled))&&(!0===(t=this.evalVisProp("intl.enabled"))||"inherit"===t&&!0===this.eval(this.board.attr.intl.enabled))},__evt__over:function(t){},__evt__mouseover:function(t){},__evt__out:function(t){},__evt__mouseout:function(t){},__evt__move:function(t){},__evt__mousemove:function(t){},__evt__drag:function(t){},__evt__mousedrag:function(t){},__evt__pendrag:function(t){},__evt__touchdrag:function(t){},__evt__keydrag:function(t){},__evt__down:function(t){},__evt__mousedown:function(t){},__evt__pendown:function(t){},__evt__touchdown:function(t){},__evt__click:function(t){},__evt__dblclick:function(t){},__evt__mouseclick:function(t){},__evt__mousedblclick:function(t){},__evt__pointerclick:function(t){},__evt__pointerdblclick:function(t){},__evt__up:function(t){},__evt__mouseup:function(t){},__evt__penup:function(t){},__evt__touchup:function(t){},__evt__attribute:function(t,e){},__evt__attribute_:function(t,e,i){},__evt:function(){}}),e.Z=s.Z.GeometryElement},573:function(t,e,i){var s=i(765),r=i(351),o=i(958),n=i(632),a=i(503),h=i(109),l=i(275),c=i(218),d=function(){this._val=parseFloat(this.rendNodeRange.value),this.rendNodeOut.value=this.rendNodeRange.value,this.board.update()};s.Z.Text=function(t,e,i,s){var o;this.constructor(t,i,r.Z.OBJECT_TYPE_TEXT,r.Z.OBJECT_CLASS_TEXT),this.element=this.board.select(i.anchor),this.coordsConstructor(e,this.evalVisProp("islabel")),this.content="",this.plaintext="",this.plaintextOld=null,this.orgText="",this.needsSizeUpdate=!1,this.hiddenByParent=!1,this.size=[1,1],this.id=this.board.setId(this,"T"),this.board.renderer.drawText(this),this.board.finalizeAdding(this),o=this.visProp.visible,this.visProp.visible=!0,this.setText(s),this.visProp.visible=o,h.Z.isString(this.content)&&this.notifyParents(this.content),this.elType="text",this.methodMap=h.Z.deepCopy(this.methodMap,{setText:"setTextJessieCode",move:"setCoords",Size:"getSize",setAutoPosition:"setAutoPosition"})},s.Z.Text.prototype=new o.Z,h.Z.copyPrototypeMethods(s.Z.Text,c.Z,"coordsConstructor"),s.Z.extend(s.Z.Text.prototype,{hasPoint:function(t,e){var i,s,r,o,n,a,c,d;return h.Z.isObject(this.evalVisProp("precision"))?(c=this.board._inputDevice,d=this.evalVisProp("precision."+c)):d=this.board.options.precision.hasPoint,this.transformations.length>0&&(t=(i=l.Z.matVecMult(l.Z.inverse(this.board.renderer.joinTransforms(this,this.transformations)),[1,t,e]))[1],e=i[2]),s=(i="right"===(n=this.getAnchorX())?this.coords.scrCoords[1]-this.size[0]:"middle"===n?this.coords.scrCoords[1]-.5*this.size[0]:this.coords.scrCoords[1])+this.size[0],r=(o="top"===(a=this.getAnchorY())?this.coords.scrCoords[2]+this.size[1]:"middle"===a?this.coords.scrCoords[2]+.5*this.size[1]:this.coords.scrCoords[2])-this.size[1],"all"===this.evalVisProp("dragarea")?t>=i-d&&t<s+d&&e>=r-d&&e<=o+d:e>=r-d&&e<=o+d&&(t>=i-d&&t<=i+2*d||t>=s-2*d&&t<=s+d)},_createFctUpdateText:function(t){var e,i,s,r,o,n,a=this.evalVisProp("parse"),l=this.evalVisProp("usemathjax"),c=this.evalVisProp("usekatex"),d=!1;if(this.orgText=t,h.Z.isFunction(t))this.updateText=function(){r=t().toString(),this.plaintext=!a||l||c?r:this.replaceSub(this.replaceSup(this.convertGeonextAndSketchometry2CSS(r,!1)))};else if(h.Z.isNumber(t)&&this.evalVisProp("formatnumber")?this.evalVisProp("tofraction")?this.content=l?"\\("+h.Z.toFraction(t,!0)+"\\)":h.Z.toFraction(t,c):(s=this.evalVisProp("digits"),this.useLocale()?this.content=this.formatNumberLocale(t,s):this.content=h.Z.toFixed(t,s)):h.Z.isString(t)&&a?(this.evalVisProp("useasciimathml")?this.content="'`"+t+"`'":l||c?(this.content=this.valueTagToJessieCode(t),h.Z.isArray(this.content)||(this.content=this.content.replace(/\\/g,"\\\\"))):this.content=this.poorMansTeX(this.valueTagToJessieCode(t)),d=!0):this.content=t,d){if(h.Z.isArray(this.content)){for(n=this,o=0;o<this.content.length;o++)if('"'!==this.content[o][0])for(i in this.content[o]=this.board.jc.snippet(this.content[o],!0,"",!1),this.content[o].deps)this.addParents(this.content[o].deps[i]),this.content[o].deps[i].addChild(this);e=function(){var t,e,i=n.evalVisProp("digits"),s="";for(t=0;t<n.content.length;t++)h.Z.isFunction(n.content[t])?(e=n.content[t](),e=n.useLocale()?n.formatNumberLocale(e,i):h.Z.toFixed(e,i)):'"'===(e=n.content[t])[0]&&'"'===e[e.length-1]&&(e=e.slice(1,-1)),s+=e;return s}}else for(i in(e=this.board.jc.snippet(this.content,!0,"",!1)).deps)this.addParents(e.deps[i]),e.deps[i].addChild(this);this.updateText=function(){this.plaintext=this.unescapeTicks(e())}}else this.updateText=function(){this.plaintext=this.content}},_setText:function(t){return this._createFctUpdateText(t),this.updateText(),this.fullUpdate(),this.board.infobox&&this.id===this.board.infobox.id||this.updateSize(),this},setTextJessieCode:function(t){var e;return this.visProp.castext=t,e=h.Z.isFunction(t)?function(){return h.Z.sanitizeHTML(t())}:h.Z.isNumber(t)?t:h.Z.sanitizeHTML(t),this._setText(e)},setText:function(t){return this._setText(t)},updateSize:function(){var t,e,i,s=this.evalVisProp("display");return a.Z.isBrowser&&"no"!==this.board.renderer.type?(i=this.rendNode,"html"===s||"vml"===this.board.renderer.type?h.Z.exists(i.offsetWidth)?(e=this,window.setTimeout((function(){e.size=[i.offsetWidth,i.offsetHeight],e.needsUpdate=!0,e.updateRenderer()}),0)):this.size=this.crudeSizeEstimate():"internal"===s&&("svg"===this.board.renderer.type?(e=this,window.setTimeout((function(){try{t=i.getBBox(),e.size=[t.width,t.height],e.needsUpdate=!0,e.updateRenderer()}catch(t){}}),0)):"canvas"===this.board.renderer.type&&(this.size=this.crudeSizeEstimate())),this):this},crudeSizeEstimate:function(){var t=parseFloat(this.evalVisProp("fontsize"));return[t*this.plaintext.length*.45,.9*t]},utf8_decode:function(t){return t.replace(/&#x(\w+);/g,(function(t,e){return String.fromCharCode(parseInt(e,16))}))},replaceSub:function(t){if(!t.indexOf)return t;for(var e,i=t.indexOf("_{");i>=0;)(e=(t=t.slice(0,i)+t.slice(i).replace(/_\{/,"<sub>")).indexOf("}",i+4))>=0&&(t=t.slice(0,e)+t.slice(e).replace(/\}/,"</sub>")),i=t.indexOf("_{");for(i=t.indexOf("_");i>=0;)i=(t=t.slice(0,i)+t.slice(i).replace(/_(.?)/,"<sub>$1</sub>")).indexOf("_");return t},replaceSup:function(t){if(!t.indexOf)return t;for(var e,i=t.indexOf("^{");i>=0;)(e=(t=t.slice(0,i)+t.slice(i).replace(/\^\{/,"<sup>")).indexOf("}",i+4))>=0&&(t=t.slice(0,e)+t.slice(e).replace(/\}/,"</sup>")),i=t.indexOf("^{");for(i=t.indexOf("^");i>=0;)i=(t=t.slice(0,i)+t.slice(i).replace(/\^(.?)/,"<sup>$1</sup>")).indexOf("^");return t},getSize:function(){return this.size},setCoords:function(t,e){var i,s,o;return h.Z.isArray(t)&&t.length>1&&(e=t[1],t=t[0]),this.evalVisProp("islabel")&&h.Z.exists(this.element)?(s=(t-(i=this.element.getLabelAnchor()).usrCoords[1])*this.board.unitX,o=-(e-i.usrCoords[2])*this.board.unitY,this.relativeCoords.setCoordinates(r.Z.COORDS_BY_SCREEN,[s,o])):this.coords.setCoordinates(r.Z.COORDS_BY_USER,[t,e]),this.fullUpdate(),this},update:function(t){return this.needsUpdate?(this.updateCoords(t),this.updateText(),"internal"===this.evalVisProp("display")&&h.Z.isString(this.plaintext)&&(this.plaintext=this.utf8_decode(this.plaintext)),this.checkForSizeUpdate(),this.needsSizeUpdate&&this.updateSize(),this):this},checkForSizeUpdate:function(){this.board.infobox&&this.id===this.board.infobox.id?this.needsSizeUpdate=!1:(this.needsSizeUpdate=this.plaintextOld!==this.plaintext,this.needsSizeUpdate&&(this.plaintextOld=this.plaintext))},updateRenderer:function(){return this.evalVisProp("autoposition")&&this.setAutoPosition().updateConstraint(),this.updateRendererGeneric("updateText")},expandShortMath:function(t){return t.replace(/([)0-9.])\s*([(a-zA-Z_])/g,"$1*$2")},generateTerm:function(t,e,i){var s,r,o,a,l='""';if(o=(t=(t=(t=(t=(t=(t=(t=(t=(t=(t=(t=t||"").replace(/\r/g,"")).replace(/\n/g,"")).replace(/"/g,"'")).replace(/'/g,"\\'")).replace(/&amp;arc;/g,"&ang;")).replace(/<arc\s*\/>/g,"&ang;")).replace(/&lt;arc\s*\/&gt;/g,"&ang;")).replace(/&lt;sqrt\s*\/&gt;/g,"&radic;")).replace(/&lt;value&gt;/g,"<value>")).replace(/&lt;\/value&gt;/g,"</value>")).indexOf("<value>"),a=t.indexOf("</value>"),o>=0)for(;o>=0;)l+=' + "'+this.replaceSub(this.replaceSup(t.slice(0,o)))+'"',r=(r=t.slice(o+7,a)).replace(/\s+/g,""),!0===e&&(r=this.expandShortMath(r)),(s=(s=(s=i?r:n.Z.geonext2JS(r,this.board)).replace(/\\"/g,"'")).replace(/\\'/g,"'")).indexOf("toFixed")<0&&h.Z.isNumber(h.Z.bind(this.board.jc.snippet(s,!0,"",!1),this)())?l+="+("+s+").toFixed("+this.evalVisProp("digits")+")":l+="+("+s+")",o=(t=t.slice(a+8)).indexOf("<value>"),a=t.indexOf("</value>");return l+=' + "'+this.replaceSub(this.replaceSup(t))+'"',l=(l=(l=this.convertGeonextAndSketchometry2CSS(l)).replace(/&amp;/g,"&")).replace(/"/g,"'")},valueTagToJessieCode:function(t){var e,i,s,r,o=[],n='"';if(s=(t=(t=(t=(t=(t=t||"").replace(/\r/g,"")).replace(/\n/g,"")).replace(/&lt;value&gt;/g,"<value>")).replace(/&lt;\/value&gt;/g,"</value>")).indexOf("<value>"),r=t.indexOf("</value>"),s>=0)for(;s>=0;)o.push(n+this.escapeTicks(t.slice(0,s))+n),i=(i=t.slice(s+7,r)).replace(/\s+/g,""),e=(e=i=this.expandShortMath(i)).replace(/\\"/g,"'").replace(/\\'/g,"'"),o.push("("+e+")"),s=(t=t.slice(r+8)).indexOf("<value>"),r=t.indexOf("</value>");for(o.push(n+this.escapeTicks(t)+n),s=0;s<o.length;s++)o[s]=o[s].replace(/&amp;/g,"&");return o},poorMansTeX:function(t){var e,i;if(h.Z.isArray(t)){for(i=[],e=0;e<t.length;e++)i.push(this.poorMansTeX(t[e]));return i}return t=t.replace(/<arc\s*\/*>/g,"&ang;").replace(/&lt;arc\s*\/*&gt;/g,"&ang;").replace(/<sqrt\s*\/*>/g,"&radic;").replace(/&lt;sqrt\s*\/*&gt;/g,"&radic;"),this.convertGeonextAndSketchometry2CSS(this.replaceSub(this.replaceSup(t)),!0)},escapeTicks:function(t){return t.replace(/"/g,"%22").replace(/'/g,"%27")},unescapeTicks:function(t){return t.replace(/%22/g,'"').replace(/%27/g,"'")},convertGeonext2CSS:function(t){return h.Z.isString(t)&&(t=(t=(t=(t=t.replace(/(<|&lt;)overline(>|&gt;)/g,"<span style=text-decoration:overline;>")).replace(/(<|&lt;)\/overline(>|&gt;)/g,"</span>")).replace(/(<|&lt;)arrow(>|&gt;)/g,"<span style=text-decoration:overline;>")).replace(/(<|&lt;)\/arrow(>|&gt;)/g,"</span>")),t},convertSketchometry2CSS:function(t,e){var i='<span class="sketcho sketcho-inherit sketcho-',s='"></span>';return h.Z.isString(t)&&(e&&(i=this.escapeTicks(i),s=this.escapeTicks(s)),t=(t=t.replace(/(<|&lt;)sketchofont(>|&gt;)/g,i)).replace(/(<|&lt;)\/sketchofont(>|&gt;)/g,s)),t},convertGeonextAndSketchometry2CSS:function(t,e){return t=this.convertGeonext2CSS(t),t=this.convertSketchometry2CSS(t,e)},notifyParents:function(t){var e,i=null;t=(t=t.replace(/&lt;value&gt;/g,"<value>")).replace(/&lt;\/value&gt;/g,"</value>");do{null!==(i=(e=/<value>([\w\s*/^\-+()[\],<>=!]+)<\/value>/).exec(t))&&(n.Z.findDependencies(this,i[1],this.board),t=(t=t.slice(i.index)).replace(e,""))}while(null!==i);return this},getParents:function(){var t;return t=void 0!==this.relativeCoords?[this.relativeCoords.usrCoords[1],this.relativeCoords.usrCoords[2],this.orgText]:[this.Z(),this.X(),this.Y(),this.orgText],0!==this.parents.length&&(t=this.parents),t},bounds:function(){var t=this.coords.usrCoords;return this.evalVisProp("islabel")||0===this.board.unitY||0===this.board.unitX?[0,0,0,0]:[t[1],t[2]+this.size[1]/this.board.unitY,t[1]+this.size[0]/this.board.unitX,t[2]]},getAnchorX:function(){var t=this.evalVisProp("anchorx");if("auto"===t)switch(this.visProp.position){case"top":case"bot":return"middle";case"rt":case"lrt":case"urt":return"left";default:return"right"}return t},getAnchorY:function(){var t=this.evalVisProp("anchory");if("auto"===t)switch(this.visProp.position){case"top":case"ulft":case"urt":return"bottom";case"bot":case"lrt":case"llft":return"top";default:return"middle"}return t},getNumberOfConflicts:function(t,e,i,s,o){o=o||[];var n,a,h,l,c,d,u,p,f=0,_=this.board.options.precision.hasPoint;for(this.board.options.precision.hasPoint=.3*(i+s),n=0;n<this.board.objectsList.length;n++)(a=this.board.objectsList[n]).visPropCalc.visible&&a!==this&&-1===o.indexOf(a.id)&&!0!==a.evalVisProp("ignoreforlabelautoposition")&&(l=a.visProp.hasinnerpoints,a.visProp.hasinnerpoints=!1,a.visProp.islabel||a.elementClass===r.Z.OBJECT_CLASS_POINT?(c=(h=a.coords.scrCoords)[1],d=h[2],u=a.size[0],p=a.size[1],a.visProp.islabel?("top"===a.visProp.anchory?d+=p/2:d-=p/2,"left"===a.visProp.anchorx?c+=u/2:c-=u/2):(u=0,p=0),Math.abs(c-t)<(i+u)/2&&Math.abs(d-e)<(s+p)/2&&f++):a.hasPoint(t,e)&&f++,a.visProp.hasinnerpoints=l);return this.board.options.precision.hasPoint=_,f},calculateScore:function(t,e){var i,s,r,o,n,a,l=0,c=[0,0],d=this.getSize()[0],u=this.getSize()[1],p=this.evalVisProp("offset"),f=this.board.getBoundingBox();return this.evalVisProp("islabel")&&h.Z.exists(this.element)?(a=this.element.getLabelAnchor().scrCoords,r=Math.cos(e),o=Math.sin(e),i=a[1]+t*r,s=a[2]-t*o,r<0?(c[0]=i-d,i-=d/2):(c[0]=i+d,i+=d/2),o<0?(c[1]=s+u,s+=u/2):(c[1]=s-u,s-=u/2),(c[0]<0||c[0]>(f[2]-f[0])*this.board.unitX||c[1]<0||c[1]>(f[1]-f[3])*this.board.unitY)&&(l-=1),l-=this.getNumberOfConflicts(i,s,d,u,h.Z.evaluate(this.visProp.autopositionwhitelist)),l-=t/this.evalVisProp("autopositionmindistance")/10-.1,(n=Math.atan2(p[1],p[0]))<0&&(n+=2*Math.PI),l-=(r>0&&Math.cos(n)>0&&Math.abs(e-n)>Math.PI?Math.abs(e-n-2*Math.PI):Math.abs(e-n))/10):0},setAutoPosition:function(){var t,e,i,s,r,o,n,a,h,l,c,d=-1/0,u=this.evalVisProp("autopositionmindistance"),p=this.evalVisProp("autopositionmaxdistance"),f=this.evalVisProp("offset");if(this===this.board.infobox||!this.element||!this.visPropCalc.visible||!this.evalVisProp("islabel"))return this;if(l=Math.sqrt(f[0]*f[0]+f[1]*f[1]),c=Math.atan2(f[1],f[0]),0===this.calculateScore(l,c))return this;for(t=u,i=(p-u)/3;p-t>-.01;){for(s=0;s<60;s++)if(e=s/60*2*Math.PI,(n=this.calculateScore(t,e))>d&&(o=e,r=t,d=n),0===d){t=p;break}t+=i}return a=Math.cos(o),h=Math.sin(o),this.visProp.anchorx=a<0?"right":"left",this.visProp.anchory=h<0?"top":"bottom",this.visProp.offset=[r*a,r*h],this}}),s.Z.createText=function(t,e,i){var r,o=h.Z.copyAttributes(i,t.options,"text"),n=e.slice(0,-1),a=e[e.length-1];if(o.anchor=o.parent||o.anchor,!(r=c.Z.create(s.Z.Text,t,n,o,a)))throw new Error("JSXGraph: Can't create text with parent types '"+typeof e[0]+"' and '"+typeof e[1]+"'.\nPossible parent types: [x,y], [z,x,y], [element,transformation]");return 0!==o.rotate&&r.addRotation(o.rotate),r},s.Z.registerElement("text",s.Z.createText),s.Z.createHTMLSlider=function(t,e,i){var r,o,n=h.Z.copyAttributes(i,t.options,"htmlslider");if(2!==e.length||2!==e[0].length||3!==e[1].length)throw new Error("JSXGraph: Can't create htmlslider with parent types '"+typeof e[0]+"' and '"+typeof e[1]+"'.\nPossible parents are: [[x,y], [min, start, max]]");n.anchor=n.parent||n.anchor,n.fixed=n.fixed||!0,o=[e[0][0],e[0][1],'<form style="display:inline"><input type="range" /><span></span><input type="text" /></form>'],(r=s.Z.createText(t,o,n)).type=h.Z.OBJECT_TYPE_HTMLSLIDER,r.rendNodeForm=r.rendNode.childNodes[0],r.rendNodeRange=r.rendNodeForm.childNodes[0],r.rendNodeRange.min=e[1][0],r.rendNodeRange.max=e[1][2],r.rendNodeRange.step=n.step,r.rendNodeRange.value=e[1][1],r.rendNodeLabel=r.rendNodeForm.childNodes[1],r.rendNodeLabel.id=r.rendNode.id+"_label",n.withlabel&&(r.rendNodeLabel.innerHTML=r.name+"="),r.rendNodeOut=r.rendNodeForm.childNodes[2],r.rendNodeOut.value=e[1][1];try{r.rendNodeForm.id=r.rendNode.id+"_form",r.rendNodeRange.id=r.rendNode.id+"_range",r.rendNodeOut.id=r.rendNode.id+"_out"}catch(t){s.Z.debug(t)}return r.rendNodeRange.style.width=n.widthrange+"px",r.rendNodeRange.style.verticalAlign="middle",r.rendNodeOut.style.width=n.widthout+"px",r._val=e[1][1],s.Z.supportsVML()?a.Z.addEvent(r.rendNodeForm,"change",d,r):a.Z.addEvent(r.rendNodeForm,"input",d,r),r.Value=function(){return this._val},r},s.Z.registerElement("htmlslider",s.Z.createHTMLSlider),e.Z=s.Z.Text},765:function(t,e){var i={};try{"object"!=typeof JXG||JXG.extend||(i=JXG)}catch(t){}i.extend=function(t,e,i,s){var r;for(r in i=i||!1,s=s||!1,e)(!i||i&&e.hasOwnProperty(r))&&(t[s?r.toLowerCase():r]=e[r])},i.defineConstant=function(t,e,s,r){(r=r||!1)&&i.exists(t[e])||Object.defineProperty(t,e,{value:s,writable:!1,enumerable:!0,configurable:!1})},i.extendConstants=function(t,e,i,s){var r,o;for(r in i=i||!1,s=s||!1,e)(!i||i&&e.hasOwnProperty(r))&&(o=s?r.toUpperCase():r,this.defineConstant(t,o,e[r]))},i.extend(i,{boards:{},readers:{},elements:{},registerElement:function(t,e){t=t.toLowerCase(),this.elements[t]=e},registerReader:function(t,e){var i,s;for(i=0;i<e.length;i++)s=e[i].toLowerCase(),"function"!=typeof this.readers[s]&&(this.readers[s]=t)},shortcut:function(t,e){return function(){return t[e].apply(this,arguments)}},getRef:function(t,e){return i.deprecated("JXG.getRef()","Board.select()"),t.select(e)},getReference:function(t,e){return i.deprecated("JXG.getReference()","Board.select()"),t.select(e)},getBoardByContainerId:function(t){var e;for(e in JXG.boards)if(JXG.boards.hasOwnProperty(e)&&JXG.boards[e].container===t)return JXG.boards[e];return null},deprecated:function(t,e){var s=t+" is deprecated.";e&&(s+=" Please use "+e+" instead."),i.warn(s)},warn:function(t){"object"==typeof window&&window.console&&console.warn?console.warn("WARNING:",t):"object"==typeof document&&document.getElementById("warning")&&(document.getElementById("debug").innerHTML+="WARNING: "+t+"<br />")},debugInt:function(t){var e,i;for(e=0;e<arguments.length;e++)i=arguments[e],"object"==typeof window&&window.console&&console.log?console.log(i):"object"==typeof document&&document.getElementById("debug")&&(document.getElementById("debug").innerHTML+=i+"<br/>")},debugWST:function(t){var e=new Error;i.debugInt.apply(this,arguments),e&&e.stack&&(i.debugInt("stacktrace"),i.debugInt(e.stack.split("\n").slice(1).join("\n")))},debugLine:function(t){var e=new Error;i.debugInt.apply(this,arguments),e&&e.stack&&i.debugInt("Called from",e.stack.split("\n").slice(2,3).join("\n"))},debug:function(t){i.debugInt.apply(this,arguments)},themes:{}}),e.Z=i},922:function(t,e,i){var s=i(765),r=i(351),o=i(705),n=i(275),a=i(309),h=i(336),l=i(109),c=i(421);n.Z.Geometry={},s.Z.extend(n.Z.Geometry,{angle:function(t,e,i){var r,o,n,a,h=[],l=[],c=[];return s.Z.deprecated("Geometry.angle()","Geometry.rad()"),t.coords?(h[0]=t.coords.usrCoords[1],h[1]=t.coords.usrCoords[2]):(h[0]=t[0],h[1]=t[1]),e.coords?(l[0]=e.coords.usrCoords[1],l[1]=e.coords.usrCoords[2]):(l[0]=e[0],l[1]=e[1]),i.coords?(c[0]=i.coords.usrCoords[1],c[1]=i.coords.usrCoords[2]):(c[0]=i[0],c[1]=i[1]),r=h[0]-l[0],o=h[1]-l[1],n=c[0]-l[0],a=c[1]-l[1],Math.atan2(r*a-o*n,r*n+o*a)},trueAngle:function(t,e,i){return 57.29577951308232*this.rad(t,e,i)},rad:function(t,e,i){var s,r,o,n,a,h,l;return t.coords?(s=t.coords.usrCoords[1],r=t.coords.usrCoords[2]):(s=t[0],r=t[1]),e.coords?(o=e.coords.usrCoords[1],n=e.coords.usrCoords[2]):(o=e[0],n=e[1]),i.coords?(a=i.coords.usrCoords[1],h=i.coords.usrCoords[2]):(a=i[0],h=i[1]),(l=Math.atan2(h-n,a-o)-Math.atan2(r-n,s-o))<0&&(l+=6.283185307179586),l},angleBisector:function(t,e,i,s){var n,a,h,c,d,u=t.coords.usrCoords,p=e.coords.usrCoords,f=i.coords.usrCoords;return l.Z.exists(s)||(s=t.board),0===p[0]?new o.Z(r.Z.COORDS_BY_USER,[1,.5*(u[1]+f[1]),.5*(u[2]+f[2])],s):(c=u[1]-p[1],d=u[2]-p[2],n=Math.atan2(d,c),c=f[1]-p[1],d=f[2]-p[2],h=.5*(n+(a=Math.atan2(d,c))),n>a&&(h+=Math.PI),c=Math.cos(h)+p[1],d=Math.sin(h)+p[2],new o.Z(r.Z.COORDS_BY_USER,[1,c,d],s))},reflection:function(t,e,i){var s,n,a,h,c,d,u=e.coords.usrCoords,p=t.point1.coords.usrCoords,f=t.point2.coords.usrCoords;return l.Z.exists(i)||(i=e.board),h=f[1]-p[1],c=f[2]-p[2],s=u[1]-p[1],d=(h*(u[2]-p[2])-c*s)/(h*h+c*c),n=u[1]+2*d*c,a=u[2]-2*d*h,new o.Z(r.Z.COORDS_BY_USER,[n,a],i)},rotation:function(t,e,i,s){var n,a,h,c,d,u,p=e.coords.usrCoords,f=t.coords.usrCoords;return l.Z.exists(s)||(s=e.board),n=p[1]-f[1],a=p[2]-f[2],d=n*(h=Math.cos(i))-a*(c=Math.sin(i))+f[1],u=n*c+a*h+f[2],new o.Z(r.Z.COORDS_BY_USER,[d,u],s)},perpendicular:function(t,e,i){var s,a,h,c,d,u=t.point1.coords.usrCoords,p=t.point2.coords.usrCoords,f=e.coords.usrCoords;return l.Z.exists(i)||(i=e.board),e===t.point1?(s=u[1]+p[2]-u[2],a=u[2]-p[1]+u[1],d=u[0]*p[0],Math.abs(d)<n.Z.eps&&(s=p[2],a=-p[1]),c=[d,s,a],h=!0):e===t.point2?(s=p[1]+u[2]-p[2],a=p[2]-u[1]+p[1],d=u[0]*p[0],Math.abs(d)<n.Z.eps&&(s=u[2],a=-u[1]),c=[d,s,a],h=!1):Math.abs(n.Z.innerProduct(f,t.stdform,3))<n.Z.eps?(s=f[1]+p[2]-f[2],a=f[2]-p[1]+f[1],d=p[0],Math.abs(d)<n.Z.eps&&(s=p[2],a=-p[1]),h=!0,Math.abs(d)>n.Z.eps&&Math.abs(s-f[1])<n.Z.eps&&Math.abs(a-f[2])<n.Z.eps&&(s=f[1]+u[2]-f[2],a=f[2]-u[1]+f[1],h=!1),c=[d,s,a]):(c=[0,t.stdform[1],t.stdform[2]],c=n.Z.crossProduct(c,f),c=n.Z.crossProduct(c,t.stdform),h=!0),[new o.Z(r.Z.COORDS_BY_USER,c,i),h]},circumcenterMidpoint:function(){s.Z.deprecated("Geometry.circumcenterMidpoint()","Geometry.circumcenter()"),this.circumcenter.apply(this,arguments)},circumcenter:function(t,e,i,s){var a,h,c,d,u=t.coords.usrCoords,p=e.coords.usrCoords,f=i.coords.usrCoords;return l.Z.exists(s)||(s=t.board),a=[p[0]-u[0],-p[2]+u[2],p[1]-u[1]],h=[.5*(u[0]+p[0]),.5*(u[1]+p[1]),.5*(u[2]+p[2])],c=n.Z.crossProduct(a,h),a=[f[0]-p[0],-f[2]+p[2],f[1]-p[1]],h=[.5*(p[0]+f[0]),.5*(p[1]+f[1]),.5*(p[2]+f[2])],d=n.Z.crossProduct(a,h),new o.Z(r.Z.COORDS_BY_USER,n.Z.crossProduct(c,d),s)},distance:function(t,e,i){var s,r=0;for(i||(i=Math.min(t.length,e.length)),s=0;s<i;s++)r+=(t[s]-e[s])*(t[s]-e[s]);return Math.sqrt(r)},affineDistance:function(t,e,i){var s;return(s=this.distance(t,e,i))>n.Z.eps&&(Math.abs(t[0])<n.Z.eps||Math.abs(e[0])<n.Z.eps)?1/0:s},affineRatio:function(t,e,i){var s;return l.Z.exists(t.usrCoords)&&(t=t.usrCoords),l.Z.exists(e.usrCoords)&&(e=e.usrCoords),l.Z.exists(i.usrCoords)&&(i=i.usrCoords),s=e[1]-t[1],Math.abs(s)>n.Z.eps?(i[1]-t[1])/s:(i[2]-t[2])/(e[2]-t[2])},sortVertices:function(t){for(var e,i=c.Z.each(t,c.Z.coordsArray),s=i.length,r=null;i[0][0]===i[s-1][0]&&i[0][1]===i[s-1][1]&&i[0][2]===i[s-1][2];)r=i.pop(),s--;return e=i[0],i.sort((function(t,i){return(t[2]===e[2]&&t[1]===e[1]?-1/0:Math.atan2(t[2]-e[2],t[1]-e[1]))-(i[2]===e[2]&&i[1]===e[1]?-1/0:Math.atan2(i[2]-e[2],i[1]-e[1]))})),null!==r&&i.push(r),i},signedTriangle:function(t,e,i){var s=c.Z.coordsArray(t),r=c.Z.coordsArray(e),o=c.Z.coordsArray(i);return.5*((r[1]-s[1])*(o[2]-s[2])-(r[2]-s[2])*(o[1]-s[1]))},signedPolygon:function(t,e){var i,s,r=0,o=c.Z.each(t,c.Z.coordsArray);for(void 0===e&&(e=!0),e?o.unshift(o[o.length-1]):o=this.sortVertices(o),s=o.length,i=1;i<s;i++)r+=o[i-1][1]*o[i][2]-o[i][1]*o[i-1][2];return.5*r},GrahamScan:function(t){var e,i,s,r,o,a,h,d,u,p,f,_,m,g,b,v,Z,C,y,P,E,x,O,M=n.Z.eps*n.Z.eps,w=this,S=[],T=[],N=c.Z.each(t,c.Z.coordsArray);if(0===(O=N.length))return[];if(O>1024)for(h=o=N[0][1],d=a=N[0][2],_=p=N[0][1]-N[0][2],f=u=N[0][1]+N[0][2],m=0,g=0,b=0,v=0,C=0,P=0,Z=0,y=0,e=1;e<O;e++)(E=N[e][1])<o?(o=E,m=e):E>h&&(h=E,g=e),(E=N[e][2])<a?(a=E,b=e):E>d&&(d=E,v=e),(E=N[e][1]-N[e][2])<p?(p=E,C=e):E>_&&(_=E,P=e),(E=N[e][1]+N[e][2])<u?(u=E,Z=e):E>f&&(f=E,y=e);for(e=0;e<O;e++)x=N[e],(O<=1024||e in[m,g,b,v,Z,C,y,P]||m!==C&&this.signedTriangle(N[m],N[C],x)>=-M||C!==v&&this.signedTriangle(N[C],N[v],x)>=-M||v!==y&&this.signedTriangle(N[v],N[y],x)>=-M||y!==g&&this.signedTriangle(N[y],N[g],x)>=-M||g!==P&&this.signedTriangle(N[g],N[P],x)>=-M||P!==b&&this.signedTriangle(N[P],N[b],x)>=-M||b!==Z&&this.signedTriangle(N[b],N[Z],x)>=-M||Z!==m&&this.signedTriangle(N[Z],N[m],x)>=-M)&&S.push({i:e,c:x});for(O=S.length,r=0,o=S[0].c[1],a=S[0].c[2],e=1;e<O;e++)(S[e].c[2]<a||S[e].c[2]===a&&S[e].c[1]<o)&&(o=S[e].c[1],a=S[e].c[2],r=e);for(S=l.Z.swap(S,r,0),s=S[0].c,S.sort((function(t,e){var i=t.c,r=e.c,o=w.signedTriangle(s,i,r);return 0===o?n.Z.hypot(i[1]-s[1],i[2]-s[2])-n.Z.hypot(r[1]-s[1],r[2]-s[2]):-o})),i=0,e=0;e<O;e++){for(;i>1&&this.signedTriangle(T[i-2].c,T[i-1].c,S[e].c)<=0;)T.pop(),i--;T.push(S[e]),i++}return T},convexHull:function(t,e){var i,s,r=[];for(s=this.GrahamScan(t),i=0;i<s.length;i++)e?r.push(s[i].c):r.push(t[s[i].i]);return r},isConvex:function(t){var e,i,s,o,a,h,d,u,p,f,_,m=n.Z.eps*n.Z.eps,g=0;if(l.Z.isArray(t)?e=c.Z.each(t,c.Z.coordsArray):l.Z.exists(t.type)&&t.type===r.Z.OBJECT_TYPE_POLYGON&&(e=c.Z.each(t.vertices,c.Z.coordsArray)),0===(i=e.length))return!0;if(i<3)return!0;for(_=null,o=e[i-2][1],a=e[i-2][2],d=e[i-1][1],u=e[i-1][2],p=Math.atan2(u-a,d-o),s=0;s<i;s++)if(o=d,a=u,h=p,d=e[s][1],u=e[s][2],o!==d||a!==u){if((f=(p=Math.atan2(u-a,d-o))-h)<=-Math.PI?f+=2*Math.PI:f>Math.PI&&(f-=2*Math.PI),null===_){if(0===f)continue;_=f>0?1:-1}else if(_*f<-m)return!1;g+=f}return Math.abs(g/(2*Math.PI))-1<m},calcStraight:function(t,e,i,s){var o,a,h,c,d,u,p,f,_,m;if(l.Z.exists(s)||(s=10),u=t.evalVisProp("straightfirst"),p=t.evalVisProp("straightlast"),Math.abs(e.scrCoords[0])<n.Z.eps&&(u=!0),Math.abs(i.scrCoords[0])<n.Z.eps&&(p=!0),(u||p)&&((f=[])[0]=t.stdform[0]-t.stdform[1]*t.board.origin.scrCoords[1]/t.board.unitX+t.stdform[2]*t.board.origin.scrCoords[2]/t.board.unitY,f[1]=t.stdform[1]/t.board.unitX,f[2]=-t.stdform[2]/t.board.unitY,!isNaN(f[0]+f[1]+f[2]))){if(!1,!1,o=!u&&Math.abs(e.usrCoords[0])>=n.Z.eps&&e.scrCoords[1]>=0&&e.scrCoords[1]<=t.board.canvasWidth&&e.scrCoords[2]>=0&&e.scrCoords[2]<=t.board.canvasHeight,a=!p&&Math.abs(i.usrCoords[0])>=n.Z.eps&&i.scrCoords[1]>=0&&i.scrCoords[1]<=t.board.canvasWidth&&i.scrCoords[2]>=0&&i.scrCoords[2]<=t.board.canvasHeight,c=(h=this.meetLineBoard(f,t.board,s))[0],d=h[1],!o&&!a){if(!u&&p&&!this.isSameDirection(e,i,c)&&!this.isSameDirection(e,i,d))return;if(u&&!p&&!this.isSameDirection(i,e,c)&&!this.isSameDirection(i,e,d))return}o?a||(m=this.isSameDir(e,i,c,d)?d:c):a?_=this.isSameDir(e,i,c,d)?c:d:this.isSameDir(e,i,c,d)?(_=c,m=d):(m=c,_=d),_&&e.setCoordinates(r.Z.COORDS_BY_USER,_.usrCoords),m&&i.setCoordinates(r.Z.COORDS_BY_USER,m.usrCoords)}},calcLineDelimitingPoints:function(t,e,i){var s,o,a,h,l,c,d,u,p,f,_;if(l=t.evalVisProp("straightfirst"),c=t.evalVisProp("straightlast"),Math.abs(e.scrCoords[0])<n.Z.eps&&(l=!0),Math.abs(i.scrCoords[0])<n.Z.eps&&(c=!0),(d=[])[0]=t.stdform[0]-t.stdform[1]*t.board.origin.scrCoords[1]/t.board.unitX+t.stdform[2]*t.board.origin.scrCoords[2]/t.board.unitY,d[1]=t.stdform[1]/t.board.unitX,d[2]=-t.stdform[2]/t.board.unitY,!isNaN(d[0]+d[1]+d[2])){if(f=!l,_=!c,o=t.board.getBoundingBox(),t.getSlope()>=0?(a=this.projectPointToLine({coords:{usrCoords:[1,o[2],o[1]]}},t,t.board),h=this.projectPointToLine({coords:{usrCoords:[1,o[0],o[3]]}},t,t.board)):(a=this.projectPointToLine({coords:{usrCoords:[1,o[0],o[1]]}},t,t.board),h=this.projectPointToLine({coords:{usrCoords:[1,o[2],o[3]]}},t,t.board)),!f&&!_){if(!l&&!c){if(s=e.distance(r.Z.COORDS_BY_USER,i),Math.abs(e.distance(r.Z.COORDS_BY_USER,a)+a.distance(r.Z.COORDS_BY_USER,i)-s)>n.Z.eps)return;if(Math.abs(e.distance(r.Z.COORDS_BY_USER,h)+h.distance(r.Z.COORDS_BY_USER,i)-s)>n.Z.eps)return}if(!l&&c&&!this.isSameDirection(e,i,a)&&!this.isSameDirection(e,i,h))return;if(l&&!c&&!this.isSameDirection(i,e,a)&&!this.isSameDirection(i,e,h))return}f?_||(p=this.isSameDir(e,i,a,h)?h:a):_?u=this.isSameDir(e,i,a,h)?a:h:this.isSameDir(e,i,a,h)?(u=a,p=h):(p=a,u=h),u&&e.setCoordinates(r.Z.COORDS_BY_USER,u.usrCoords),p&&i.setCoordinates(r.Z.COORDS_BY_USER,p.usrCoords)}},calcLabelQuadrant:function(t){return t<0&&(t+=2*Math.PI),["rt","urt","top","ulft","lft","llft","lrt"][Math.floor((t+Math.PI/8)/(Math.PI/4))%8]},isSameDir:function(t,e,i,s){var r=e.usrCoords[1]-t.usrCoords[1],o=e.usrCoords[2]-t.usrCoords[2],a=s.usrCoords[1]-i.usrCoords[1],h=s.usrCoords[2]-i.usrCoords[2];return Math.abs(e.usrCoords[0])<n.Z.eps&&(r=e.usrCoords[1],o=e.usrCoords[2]),Math.abs(t.usrCoords[0])<n.Z.eps&&(r=-t.usrCoords[1],o=-t.usrCoords[2]),r*a+o*h>=0},isSameDirection:function(t,e,i){var s,r,o,a,h=!1;return s=e.usrCoords[1]-t.usrCoords[1],r=e.usrCoords[2]-t.usrCoords[2],o=i.usrCoords[1]-t.usrCoords[1],a=i.usrCoords[2]-t.usrCoords[2],Math.abs(s)<n.Z.eps&&(s=0),Math.abs(r)<n.Z.eps&&(r=0),Math.abs(o)<n.Z.eps&&(o=0),Math.abs(a)<n.Z.eps&&(a=0),(s>=0&&o>=0||s<=0&&o<=0)&&(h=r>=0&&a>=0||r<=0&&a<=0),h},det3p:function(t,e,i){var s,r,o;return s=l.Z.isPoint(t)?t.Coords(!0):t,r=l.Z.isPoint(e)?e.Coords(!0):e,o=l.Z.isPoint(i)?i.Coords(!0):i,(s[1]-o[1])*(r[2]-o[2])-(r[1]-o[1])*(s[2]-o[2])},windingNumber:function(t,e,i){var s,r,o,a,h,c,d=0,u=e.length,p=t[1],f=t[2],_=0;if(0===u)return 0;if((i=i||!1)&&(_=1),isNaN(p)||isNaN(f))return 1;if(l.Z.exists(e[0].coords)?(s=e[0].coords,r=e[u-1].coords):(s=e[0],r=e[u-1]),s.usrCoords[1]===p&&s.usrCoords[2]===f)return 1;for(c=0;c<u-_;c++)if(l.Z.exists(e[c].coords)?(r=e[c].coords.usrCoords,o=e[(c+1)%u].coords.usrCoords):(r=e[c].usrCoords,o=e[(c+1)%u].usrCoords),!(0===r[0]||0===o[0]||isNaN(r[1])||isNaN(o[1])||isNaN(r[2])||isNaN(o[2]))){if(o[2]===f){if(o[1]===p)return 1;if(r[2]===f&&o[1]>p==r[1]<p)return 0}if(r[2]<f!=o[2]<f)if(h=2*(o[2]>r[2]?1:0)-1,r[1]>=p)if(o[1]>p)d+=h;else{if(0===(a=this.det3p(r,o,t)))return 0;a>0+n.Z.eps==o[2]>r[2]&&(d+=h)}else o[1]>p&&(a=this.det3p(r,o,t))>0+n.Z.eps==o[2]>r[2]&&(d+=h)}return d},pnpoly:function(t,e,i,s,n){var a,h,c,d,u,p,f,_,m=i,g=!1;for(s===r.Z.COORDS_BY_USER?(p=(_=new o.Z(r.Z.COORDS_BY_USER,[t,e],n)).scrCoords[1],f=_.scrCoords[2]):(p=t,f=e),a=0,h=(u=i.length)-2;a<u-1;h=a++)c=l.Z.exists(m[a].coords)?m[a].coords:m[a],d=l.Z.exists(m[h].coords)?m[h].coords:m[h],c.scrCoords[2]>f!=d.scrCoords[2]>f&&p<(d.scrCoords[1]-c.scrCoords[1])*(f-c.scrCoords[2])/(d.scrCoords[2]-c.scrCoords[2])+c.scrCoords[1]&&(g=!g);return g},intersectionFunction:function(t,e,i,a,h,c){var d,u,p=this;return d=e.elementClass===r.Z.OBJECT_CLASS_CURVE&&(e.type===r.Z.OBJECT_TYPE_ARC||e.type===r.Z.OBJECT_TYPE_SECTOR),u=i.elementClass===r.Z.OBJECT_CLASS_CURVE&&(i.type===r.Z.OBJECT_TYPE_ARC||i.type===r.Z.OBJECT_TYPE_SECTOR),e.elementClass!==r.Z.OBJECT_CLASS_CURVE&&i.elementClass!==r.Z.OBJECT_CLASS_CURVE||e.elementClass!==r.Z.OBJECT_CLASS_CURVE&&e.elementClass!==r.Z.OBJECT_CLASS_CIRCLE||i.elementClass!==r.Z.OBJECT_CLASS_CURVE&&i.elementClass!==r.Z.OBJECT_CLASS_CIRCLE?e.elementClass===r.Z.OBJECT_CLASS_CURVE&&!d&&i.elementClass===r.Z.OBJECT_CLASS_LINE||i.elementClass===r.Z.OBJECT_CLASS_CURVE&&!u&&e.elementClass===r.Z.OBJECT_CLASS_LINE?function(){return p.meetCurveLine(e,i,a,e.board,l.Z.evaluate(c))}:e.type===r.Z.OBJECT_TYPE_POLYGON||i.type===r.Z.OBJECT_TYPE_POLYGON?e.elementClass===r.Z.OBJECT_CLASS_LINE?function(){var t,s=e.evalVisProp("straightfirst"),r=e.evalVisProp("straightlast"),o=i.evalVisProp("straightfirst"),n=i.evalVisProp("straightlast");return t=!(l.Z.evaluate(c)||s&&r&&o&&n),p.meetPolygonLine(i,e,a,e.board,t)}:i.elementClass===r.Z.OBJECT_CLASS_LINE?function(){var t,s=e.evalVisProp("straightfirst"),r=e.evalVisProp("straightlast"),o=i.evalVisProp("straightfirst"),n=i.evalVisProp("straightlast");return t=!(l.Z.evaluate(c)||s&&r&&o&&n),p.meetPolygonLine(e,i,a,e.board,t)}:function(){return p.meetPathPath(e,i,a,e.board)}:e.elementClass===r.Z.OBJECT_CLASS_LINE&&i.elementClass===r.Z.OBJECT_CLASS_LINE?function(){var t,s,n=e.evalVisProp("straightfirst"),h=e.evalVisProp("straightlast"),d=i.evalVisProp("straightfirst"),u=i.evalVisProp("straightlast");return l.Z.evaluate(c)||n&&h&&d&&u?p.meet(e.stdform,i.stdform,a,e.board):(t=p.meetSegmentSegment(e.point1.coords.usrCoords,e.point2.coords.usrCoords,i.point1.coords.usrCoords,i.point2.coords.usrCoords),s=!n&&t[1]<0||!h&&t[1]>1||!d&&t[2]<0||!u&&t[2]>1?[0,NaN,NaN]:t[0],new o.Z(r.Z.COORDS_BY_USER,s,e.board))}:function(){var t,h,f,_=p.meet(e.stdform,i.stdform,a,e.board),m=!0;return l.Z.evaluate(c)?_:(e.elementClass!==r.Z.OBJECT_CLASS_LINE||(t=e.evalVisProp("straightfirst"),h=e.evalVisProp("straightlast"),t&&h||(f=p.affineRatio(e.point1.coords,e.point2.coords,_),!(!h&&f>1+n.Z.eps||!t&&f<0-n.Z.eps))))&&(i.elementClass!==r.Z.OBJECT_CLASS_LINE||(t=i.evalVisProp("straightfirst"),h=i.evalVisProp("straightlast"),t&&h||(f=p.affineRatio(i.point1.coords,i.point2.coords,_),!(!h&&f>1+n.Z.eps||!t&&f<0-n.Z.eps))))?d&&((m=p.coordsOnArc(e,_))&&u&&(m=p.coordsOnArc(i,_)),!m)?new o.Z(s.Z.COORDS_BY_USER,[0,NaN,NaN],e.board):_:new o.Z(s.Z.COORDS_BY_USER,[0,NaN,NaN],e.board)}:function(){return p.meetCurveCurve(e,i,a,h,e.board)}},otherIntersectionFunction:function(t,e,i,o){var n,a,h,c=this;return a=t[0],h=t[1],n=a.board,function(){var t,d,u,p,f=e.length,_=l.Z.evaluate(o);for(t=f;t>=0;t--){for(a.elementClass===r.Z.OBJECT_CLASS_CIRCLE&&[r.Z.OBJECT_CLASS_CIRCLE,r.Z.OBJECT_CLASS_LINE].indexOf(h.elementClass)>=0?u=c.meet(a.stdform,h.stdform,t,n):a.elementClass===r.Z.OBJECT_CLASS_CURVE&&[r.Z.OBJECT_CLASS_CURVE,r.Z.OBJECT_CLASS_CIRCLE].indexOf(h.elementClass)>=0?u=c.meetCurveCurve(a,h,t,0,n,"segment"):a.elementClass===r.Z.OBJECT_CLASS_CURVE&&h.elementClass===r.Z.OBJECT_CLASS_LINE&&(u=l.Z.exists(a.dataX)?s.Z.Math.Geometry.meetCurveLine(a,h,t,a.board,l.Z.evaluate(i)):s.Z.Math.Geometry.meetCurveLineContinuous(a,h,t,a.board)),p=!1,d=0;!p&&d<f;d++)u.distance(s.Z.COORDS_BY_USER,e[d].coords)<_&&(p=!0);if(!p)return u}return u}},coordsOnArc:function(t,e){var i=this.rad(t.radiuspoint,t.center,e.usrCoords.slice(1)),s=0,r=this.rad(t.radiuspoint,t.center,t.anglepoint),o=t.evalVisProp("selection");return("minor"===o&&r>Math.PI||"major"===o&&r<Math.PI)&&(s=r,r=2*Math.PI),!(i<s||i>r)},meet:function(t,e,i,s){var r=n.Z.eps;return Math.abs(t[3])<r&&Math.abs(e[3])<r?this.meetLineLine(t,e,i,s):Math.abs(t[3])>=r&&Math.abs(e[3])<r?this.meetLineCircle(e,t,i,s):Math.abs(t[3])<r&&Math.abs(e[3])>=r?this.meetLineCircle(t,e,i,s):this.meetCircleCircle(t,e,i,s)},meetLineBoard:function(t,e,i){var s,a,h,c,d=[];for(l.Z.exists(i)||(i=0),d[0]=n.Z.crossProduct(t,[i,0,1]),d[1]=n.Z.crossProduct(t,[i,1,0]),d[2]=n.Z.crossProduct(t,[-i-e.canvasHeight,0,1]),d[3]=n.Z.crossProduct(t,[-i-e.canvasWidth,1,0]),h=0;h<4;h++)if(Math.abs(d[h][0])>n.Z.eps){for(c=2;c>0;c--)d[h][c]/=d[h][0];d[h][0]=1}return Math.abs(d[1][0])<n.Z.eps?(s=d[0],a=d[2]):Math.abs(d[0][0])<n.Z.eps?(s=d[1],a=d[3]):d[1][2]<0?(s=d[0],a=d[3][2]>e.canvasHeight?d[2]:d[3]):d[1][2]>e.canvasHeight?(s=d[2],a=d[3][2]<0?d[0]:d[3]):(s=d[1],a=d[3][2]<0?d[0]:d[3][2]>e.canvasHeight?d[2]:d[3]),[new o.Z(r.Z.COORDS_BY_SCREEN,s.slice(1),e),new o.Z(r.Z.COORDS_BY_SCREEN,a.slice(1),e)]},meetLineLine:function(t,e,i,s){var a=isNaN(t[5]+e[5])?[0,0,0]:n.Z.crossProduct(t,e);return Math.abs(a[0])<1e-14&&(a[0]=0),new o.Z(r.Z.COORDS_BY_USER,a,s)},meetLineCircle:function(t,e,i,s){var a,h,c,d,u,p,f,_,m;return e[4]<n.Z.eps?Math.abs(n.Z.innerProduct([1,e[6],e[7]],t,3))<n.Z.eps?new o.Z(r.Z.COORDS_BY_USER,e.slice(6,8),s):new o.Z(r.Z.COORDS_BY_USER,[NaN,NaN],s):(c=e[0],h=e.slice(1,3),a=e[3],d=t[0],u=t.slice(1,3),p=a,(_=(f=h[0]*u[1]-h[1]*u[0])*f-4*p*(a*d*d-(h[0]*u[0]+h[1]*u[1])*d+c))>-n.Z.eps*n.Z.eps?(m=[(-f+(_=Math.sqrt(Math.abs(_))))/(2*p),(-f-_)/(2*p)],0===l.Z.evaluate(i)?new o.Z(r.Z.COORDS_BY_USER,[-m[0]*-u[1]-d*u[0],-m[0]*u[0]-d*u[1]],s):new o.Z(r.Z.COORDS_BY_USER,[-m[1]*-u[1]-d*u[0],-m[1]*u[0]-d*u[1]],s)):new o.Z(r.Z.COORDS_BY_USER,[0,0,0],s))},meetCircleCircle:function(t,e,i,s){var a;return t[4]<n.Z.eps?Math.abs(this.distance(t.slice(6,2),e.slice(6,8))-e[4])<n.Z.eps?new o.Z(r.Z.COORDS_BY_USER,t.slice(6,8),s):new o.Z(r.Z.COORDS_BY_USER,[0,0,0],s):e[4]<n.Z.eps?Math.abs(this.distance(e.slice(6,2),t.slice(6,8))-t[4])<n.Z.eps?new o.Z(r.Z.COORDS_BY_USER,e.slice(6,8),s):new o.Z(r.Z.COORDS_BY_USER,[0,0,0],s):(a=[e[3]*t[0]-t[3]*e[0],e[3]*t[1]-t[3]*e[1],e[3]*t[2]-t[3]*e[2],0,1,1/0,1/0,1/0],a=n.Z.normalize(a),this.meetLineCircle(a,t,i,s))},meetCurveCurve:function(t,e,i,s,n,a){var c;return c=l.Z.exists(a)&&"newton"===a?h.Z.generalizedNewton(t,e,l.Z.evaluate(i),s):3===t.bezierDegree||3===e.bezierDegree?this.meetBezierCurveRedBlueSegments(t,e,i):this.meetCurveRedBlueSegments(t,e,i),new o.Z(r.Z.COORDS_BY_USER,c,n)},meetCurveLine:function(t,e,i,s,o){var n,a;return l.Z.exists(s)||(s=t.board),t.elementClass===r.Z.OBJECT_CLASS_CURVE?(n=t,a=e):(n=e,a=t),this.meetCurveLineDiscrete(n,a,i,s,!o)},meetCurveLineContinuous:function(t,e,i,s,a){var l,c,d,u,p,f,_,m,g,b,v,Z,C,y=n.Z.eps,P=n.Z.eps;for(u=this.meetCurveLineDiscrete(t,e,i,s,a),p=u.usrCoords[1],f=u.usrCoords[2],l=function(e){var i,s;return e>t.maxX()||e<t.minX()?1/0:(i=t.X(e)-p)*i+(s=t.Y(e)-f)*s},c=function(i){var s=e.stdform[0]+e.stdform[1]*t.X(i)+e.stdform[2]*t.Y(i);return s*s},50,m=(t.maxX()-t.minX())/50,g=t.minX(),v=1e-4,b=NaN,Z=0;Z<50&&(d=h.Z.root(l,[Math.max(g,t.minX()),Math.min(g+m,t.maxX())]),!((C=Math.abs(l(d)))<=v&&(b=d,(v=C)<y)));Z++)g+=m;return d=b,C=c(d=h.Z.root(c,[Math.max(d-m,t.minX()),Math.min(d+m,t.maxX())])),_=isNaN(C)||Math.abs(C)>P?0:1,new o.Z(r.Z.COORDS_BY_USER,[_,t.X(d),t.Y(d)],s)},meetCurveLineDiscrete:function(t,e,i,s,a){var h,c,d,u,p,f,_,m=l.Z.evaluate(i),g=e.point1.coords.usrCoords,b=e.point2.coords.usrCoords,v=0,Z=t.numberPoints,C=e.evalVisProp("straightfirst"),y=e.evalVisProp("straightlast");for(f=new o.Z(r.Z.COORDS_BY_USER,[0,NaN,NaN],s),0===g[0]?g=[1,b[1]+e.stdform[2],b[2]-e.stdform[1]]:0===b[0]&&(b=[1,g[1]+e.stdform[2],g[2]-e.stdform[1]]),u=t.points[0].usrCoords,h=1;h<Z;h+=t.bezierDegree)if(d=u.slice(0),u=t.points[h].usrCoords,this.distance(d,u)>n.Z.eps)for(_=3===t.bezierDegree?this.meetBeziersegmentBeziersegment([t.points[h-1].usrCoords.slice(1),t.points[h].usrCoords.slice(1),t.points[h+1].usrCoords.slice(1),t.points[h+2].usrCoords.slice(1)],[g.slice(1),b.slice(1)],a):[this.meetSegmentSegment(d,u,g,b)],c=0;c<_.length;c++)if(0<=(p=_[c])[1]&&p[1]<=1){if(v===m)return a&&(!C&&p[2]<0||!y&&p[2]>1)?f:f=new o.Z(r.Z.COORDS_BY_USER,p[0],s);v+=1}return f},meetCurveRedBlueSegments:function(t,e,i){var s,r,o,n,a,h,c,d,u,p=l.Z.evaluate(i),f=0,_=e.numberPoints,m=t.numberPoints;if(_<=1||m<=1)return[0,NaN,NaN];for(s=1;s<m;s++)for(o=t.points[s-1].usrCoords,n=t.points[s].usrCoords,d=Math.min(o[1],n[1]),u=Math.max(o[1],n[1]),h=e.points[0].usrCoords,r=1;r<_;r++)if(a=h,h=e.points[r].usrCoords,Math.min(a[1],h[1])<u&&Math.max(a[1],h[1])>d&&(c=this.meetSegmentSegment(o,n,a,h))[1]>=0&&c[2]>=0&&(c[1]<1&&c[2]<1||s===m-1&&1===c[1]||r===_-1&&1===c[2])){if(f===p)return c[0];f++}return[0,NaN,NaN]},meetSegmentSegment:function(t,e,i,s){var r,o,a,h=n.Z.crossProduct(t,e),l=n.Z.crossProduct(i,s),c=n.Z.crossProduct(h,l);return Math.abs(c[0])<n.Z.eps?[c,1/0,1/0]:(c[1]/=c[0],c[2]/=c[0],c[0]/=c[0],a=t[o=Math.abs(e[1]-e[0]*t[1])<n.Z.eps?2:1]/t[0],r=(c[o]-a)/(0!==e[0]?e[o]/e[0]-a:e[o]),a=i[o=Math.abs(s[1]-s[0]*i[1])<n.Z.eps?2:1]/i[0],[c,r,(c[o]-a)/(0!==s[0]?s[o]/s[0]-a:s[o])])},meetPathPath:function(t,e,i,a){var h,c,d,u,p=l.Z.evaluate(i);return(d=(h=s.Z.Math.Clip._getPath(t,a)).length)>0&&this.distance(h[0].coords.usrCoords,h[d-1].coords.usrCoords,3)<n.Z.eps&&h.pop(),(d=(c=s.Z.Math.Clip._getPath(e,a)).length)>0&&this.distance(c[0].coords.usrCoords,c[d-1].coords.usrCoords,3)<n.Z.eps*n.Z.eps&&c.pop(),i<0||s.Z.Math.Clip.isEmptyCase(h,c,"intersection")?new o.Z(r.Z.COORDS_BY_USER,[0,0,0],a):(s.Z.Math.Clip.makeDoublyLinkedList(h),s.Z.Math.Clip.makeDoublyLinkedList(c),p<(u=s.Z.Math.Clip.findIntersections(h,c,a)[0]).length?u[p].coords:new o.Z(r.Z.COORDS_BY_USER,[0,0,0],a))},meetPolygonLine:function(t,e,i,s,n){var a,h,c,d=l.Z.evaluate(i),u=[0,0,0],p=t.borders.length,f=[];for(a=0;a<p;a++)c=t.borders[a],h=this.meetSegmentSegment(c.point1.coords.usrCoords,c.point2.coords.usrCoords,e.point1.coords.usrCoords,e.point2.coords.usrCoords),(!n||h[2]>=0&&h[2]<1)&&h[1]>=0&&h[1]<1&&f.push(h[0]);return d>=0&&d<f.length&&(u=f[d]),new o.Z(r.Z.COORDS_BY_USER,u,s)},_bezierSplit:function(t){var e,i,s,r,o,n;return e=[.5*(t[0][0]+t[1][0]),.5*(t[0][1]+t[1][1])],i=[.5*(t[1][0]+t[2][0]),.5*(t[1][1]+t[2][1])],s=[.5*(t[2][0]+t[3][0]),.5*(t[2][1]+t[3][1])],n=[.5*((r=[.5*(e[0]+i[0]),.5*(e[1]+i[1])])[0]+(o=[.5*(i[0]+s[0]),.5*(i[1]+s[1])])[0]),.5*(r[1]+o[1])],[[t[0],e,r,n],[n,o,s,t[3]]]},_bezierBbox:function(t){var e=[];return 4===t.length?(e[0]=Math.min(t[0][0],t[1][0],t[2][0],t[3][0]),e[1]=Math.max(t[0][1],t[1][1],t[2][1],t[3][1]),e[2]=Math.max(t[0][0],t[1][0],t[2][0],t[3][0]),e[3]=Math.min(t[0][1],t[1][1],t[2][1],t[3][1])):(e[0]=Math.min(t[0][0],t[1][0]),e[1]=Math.max(t[0][1],t[1][1]),e[2]=Math.max(t[0][0],t[1][0]),e[3]=Math.min(t[0][1],t[1][1])),e},_bezierOverlap:function(t,e){return t[2]>=e[0]&&t[0]<=e[2]&&t[1]>=e[3]&&t[3]<=e[1]},_bezierListConcat:function(t,e,i,s){var r,o=l.Z.exists(s),n=0,a=e.length,h=t.length;for(h>0&&a>0&&(1===t[h-1][1]&&0===e[0][1]||o&&1===t[h-1][2]&&0===e[0][2])&&(n=1),r=n;r<a;r++)o&&(e[r][2]*=.5,e[r][2]+=s),e[r][1]*=.5,e[r][1]+=i,t.push(e[r])},_bezierMeetSubdivision:function(t,e,i){var s,r,o,n,a,h,l,c,d,u,p,f,_=[];return r=this._bezierBbox(e),s=this._bezierBbox(t),this._bezierOverlap(r,s)?i<5?(h=(o=this._bezierSplit(t))[0],l=o[1],n=(o=this._bezierSplit(e))[0],a=o[1],this._bezierListConcat(_,this._bezierMeetSubdivision(h,n,i+1),0,0),this._bezierListConcat(_,this._bezierMeetSubdivision(h,a,i+1),0,.5),this._bezierListConcat(_,this._bezierMeetSubdivision(l,n,i+1),.5,0),this._bezierListConcat(_,this._bezierMeetSubdivision(l,a,i+1),.5,.5),_):(p=[1].concat(t[0]),f=[1].concat(t[3]),d=[1].concat(e[0]),u=[1].concat(e[3]),(c=this.meetSegmentSegment(p,f,d,u))[1]>=0&&c[2]>=0&&c[1]<=1&&c[2]<=1?[c]:[]):[]},_bezierLineMeetSubdivision:function(t,e,i,s){var r,o,n,a,h,l,c,d,u,p,f=[];return r=this._bezierBbox(e),o=this._bezierBbox(t),s&&!this._bezierOverlap(o,r)?[]:i<5?(a=(n=this._bezierSplit(t))[0],h=n[1],this._bezierListConcat(f,this._bezierLineMeetSubdivision(a,e,i+1),0),this._bezierListConcat(f,this._bezierLineMeetSubdivision(h,e,i+1),.5),f):(u=[1].concat(t[0]),p=[1].concat(t[3]),c=[1].concat(e[0]),d=[1].concat(e[1]),(l=this.meetSegmentSegment(u,p,c,d))[1]>=0&&l[1]<=1&&(!s||l[2]>=0&&l[2]<=1)?[l]:[])},meetBeziersegmentBeziersegment:function(t,e,i){var s,r,o;for((s=4===t.length&&4===e.length?this._bezierMeetSubdivision(t,e,0):this._bezierLineMeetSubdivision(t,e,0,i)).sort((function(t,e){return 1e7*(t[1]-e[1])+(t[2]-e[2])})),r=[],o=0;o<s.length;o++)0!==o&&s[o][1]===s[o-1][1]&&s[o][2]===s[o-1][2]||r.push(s[o]);return r},meetBezierCurveRedBlueSegments:function(t,e,i){var s,o,a,h,c,d,u,p,f,_,m,g,b,v=l.Z.evaluate(i),Z=0,C=0,y=[];if(e.numberPoints<e.bezierDegree+1||t.numberPoints<t.bezierDegree+1)return[0,NaN,NaN];for(1===t.bezierDegree&&3===e.bezierDegree&&(d=t,t=e,e=d),g=e.numberPoints-e.bezierDegree,b=t.numberPoints-t.bezierDegree,t.type===r.Z.OBJECT_TYPE_SECTOR&&(Z=3,b-=3),e.type===r.Z.OBJECT_TYPE_SECTOR&&(C=3,g-=3),o=Z;o<b;o+=t.bezierDegree)for(u=[(s=t.points)[o].usrCoords.slice(1),s[o+1].usrCoords.slice(1)],3===t.bezierDegree&&(u[2]=s[o+2].usrCoords.slice(1),u[3]=s[o+3].usrCoords.slice(1)),f=this._bezierBbox(u),a=C;a<g;a+=e.bezierDegree)if(p=[(s=e.points)[a].usrCoords.slice(1),s[a+1].usrCoords.slice(1)],3===e.bezierDegree&&(p[2]=s[a+2].usrCoords.slice(1),p[3]=s[a+3].usrCoords.slice(1)),_=this._bezierBbox(p),this._bezierOverlap(f,_)){if(0===(m=this.meetBeziersegmentBeziersegment(u,p)).length)continue;for(h=0;h<m.length;h++)(c=m[h])[1]<-n.Z.eps||c[1]>1+n.Z.eps||c[2]<-n.Z.eps||c[2]>1+n.Z.eps||y.push(c);if(y.length>v)return y[v][0]}return y.length>v?y[v][0]:[0,NaN,NaN]},bezierSegmentEval:function(t,e){var i,s,r,o=1-t;return s=0,r=0,s+=(i=o*o*o)*e[0][0],r+=i*e[0][1],s+=(i=3*t*o*o)*e[1][0],r+=i*e[1][1],s+=(i=3*t*t*o)*e[2][0],r+=i*e[2][1],[1,s+=(i=t*t*t)*e[3][0],r+=i*e[3][1]]},bezierArc:function(t,e,i,s,r){var o,a,h,c,d,u,p,f,_,m,g,b,v,Z,C,y,P,E,x=e[1],O=e[2],M=e[0],w=[],S=[];for(d=this.distance(e,t),x/=M,O/=M,u=this.rad(t.slice(1),e.slice(1),i.slice(1)),-1===r&&(u=2*Math.PI-u),f=u/4,(o=t)[1]/=o[0],o[2]/=o[0],o[0]/=o[0],c=o.slice(0),s?(w=[x,x+.333*(o[1]-x),x+.666*(o[1]-x),o[1]],S=[O,O+.333*(o[2]-O),O+.666*(o[2]-O),o[2]]):(w=[o[1]],S=[o[2]]);u>n.Z.eps;)u>f?(p=f,u-=f):(p=u,u=0),E=[[1,0,0],[x*(1-(_=Math.cos(r*p)))+O*(m=Math.sin(r*p)),_,-m],[O*(1-_)-x*m,m,_]],c=[(y=n.Z.matVecMult(E,o))[0]/y[0],y[1]/y[0],y[2]/y[0]],g=o[1]-x,b=o[2]-O,v=c[1]-x,Z=c[2]-O,P=n.Z.hypot(g+v,b+Z),C=Math.abs(Z-b)>n.Z.eps?(g+v)*(d/P-.5)/(Z-b)*8/3:(b+Z)*(d/P-.5)/(g-v)*8/3,a=[1,o[1]-C*b,o[2]+C*g],h=[1,c[1]+C*Z,c[2]-C*v],l.Z.concat(w,[a[1],h[1],c[1]]),l.Z.concat(S,[a[2],h[2],c[2]]),o=c.slice(0);return s&&(l.Z.concat(w,[c[1]+.333*(x-c[1]),c[1]+.666*(x-c[1]),x]),l.Z.concat(S,[c[2]+.333*(O-c[2]),c[2]+.666*(O-c[2]),O])),[w,S]},projectPointToCircle:function(t,e,i){var s,a,h,c,d,u=e.center.coords.usrCoords;return l.Z.exists(i)||(i=t.board),l.Z.isPoint(t)?(s=t.coords.distance(r.Z.COORDS_BY_USER,e.center.coords),a=t.coords.usrCoords):(s=t.distance(r.Z.COORDS_BY_USER,e.center.coords),a=t.usrCoords),Math.abs(s)<n.Z.eps&&(s=n.Z.eps),d=e.Radius()/s,h=u[1]+d*(a[1]-u[1]),c=u[2]+d*(a[2]-u[2]),new o.Z(r.Z.COORDS_BY_USER,[h,c],i)},projectPointToLine:function(t,e,i){var s,a=[0,e.stdform[1],e.stdform[2]];return l.Z.exists(i)||(i=l.Z.exists(t.coords)?t.board:e.board),s=l.Z.exists(t.coords)?t.coords.usrCoords:t.usrCoords,a=n.Z.crossProduct(a,s),new o.Z(r.Z.COORDS_BY_USER,n.Z.crossProduct(a,e.stdform),i)},projectCoordsToSegment:function(t,e,i){var s,r=[i[1]-e[1],i[2]-e[2]],o=[t[1]-e[1],t[2]-e[2]];return Math.abs(r[0])<n.Z.eps&&Math.abs(r[1])<n.Z.eps?[e,0]:(s=n.Z.innerProduct(o,r),[[1,(s/=n.Z.innerProduct(r,r))*r[0]+e[1],s*r[1]+e[2]],s])},projectCoordsToBeziersegment:function(t,e,i){var r;return r=s.Z.Math.Numerics.fminbr((function(s){var r=[1,e.X(i+s),e.Y(i+s)];return r[1]-=t[1],r[2]-=t[2],r[1]*r[1]+r[2]*r[2]}),[0,1]),[[1,e.X(r+i),e.Y(r+i)],r]},projectPointToCurve:function(t,e,i){l.Z.exists(i)||(i=t.board);var s=t.X(),r=t.Y(),o=t.position;return l.Z.exists(o)||(o="functiongraph"===e.evalVisProp("curvetype")?s:0),this.projectCoordsToCurve(s,r,o,e,i)},projectCoordsToCurve:function(t,e,i,s,n){var a,c,d,u,p,f,_,m,g,b,v,Z,C,y,P,E,x,O,M,w,S,T,N,A,D,k=Number.POSITIVE_INFINITY;if(l.Z.exists(n)||(n=s.board),"plot"===s.evalVisProp("curvetype")){if(i=0,p=k,a=0===s.numberPoints?[0,1,1]:[s.Z(0),s.X(0),s.Y(0)],s.numberPoints>1)for(m=[1,t,e],3===s.bezierDegree?u=0:v=[s.Z(0),s.X(0),s.Y(0)],d=0;d<s.numberPoints-1;d++)3===s.bezierDegree?C=this.projectCoordsToBeziersegment(m,s,u):(Z=[s.Z(d+1),s.X(d+1),s.Y(d+1)],C=this.projectCoordsToSegment(m,v,Z)),_=C[1],g=C[0],0<=_&&_<=1?(f=this.distance(g,m),b=d+_):_<0?(g=v,f=this.distance(v,m),b=d):_>1&&d===s.numberPoints-2&&(g=Z,f=this.distance(g,m),b=s.numberPoints-1),f<p&&(p=f,i=b,a=g),3===s.bezierDegree?(u++,d+=2):v=Z;c=new o.Z(r.Z.COORDS_BY_USER,a,n)}else{for(A=s.minX(),D=s.maxX(),T=A,N=D,"functiongraph"===s.evalVisProp("curvetype")&&(O=Math.abs(e-s.Y(t)),isNaN(O)||(T=t-O,N=t+O)),y=function(i){var r,o;return i<A||i>s.maxX_glob?1/0:(r=t-s.X(i))*r+(o=e-s.Y(i))*o},x=y(i),50,M=(N-T)/50,P=T,d=0;d<50;d++)((E=y(P))<x||x===1/0||isNaN(x))&&(i=P,x=E),P+=M;for(w=M,d=0;d<20&&isNaN(y(i-w));d++,w*=.5);for(isNaN(y(i-w))&&(w=0),S=M,d=0;d<20&&isNaN(y(i+S));d++,S*=.5);isNaN(y(i+S))&&(S=0),i=(i=(i=h.Z.fminbr(y,[Math.max(i-w,T),Math.min(i+S,N)]))<A?A:i)>D?D:i,c=new o.Z(r.Z.COORDS_BY_USER,[s.X(i),s.Y(i)],n)}return[s.updateTransform(c),i]},projectCoordsToPolygon:function(t,e){var i,r,o,n,a,h=e.vertices.length,l=1/0;for(i=0;i<h-1;i++)0<=(o=s.Z.Math.Geometry.projectCoordsToSegment(t,e.vertices[i].coords.usrCoords,e.vertices[i+1].coords.usrCoords))[1]&&o[1]<=1?(r=s.Z.Math.Geometry.distance(o[0],t,3),n=o[0]):o[1]<0?(r=s.Z.Math.Geometry.distance(e.vertices[i].coords.usrCoords,t,3),n=e.vertices[i].coords.usrCoords):(r=s.Z.Math.Geometry.distance(e.vertices[i+1].coords.usrCoords,t,3),n=e.vertices[i+1].coords.usrCoords),r<l&&(a=n.slice(0),l=r);return a},projectPointToTurtle:function(t,e,i){var s,n,a,h,c,d,u,p,f,_,m=0,g=0,b=Number.POSITIVE_INFINITY,v=e.objects.length;for(l.Z.exists(i)||(i=t.board),c=0;c<v;c++)(u=e.objects[c]).elementClass===r.Z.OBJECT_CLASS_CURVE&&(s=(f=this.projectPointToCurve(t,u))[0],_=f[1],(d=this.distance(s.usrCoords,t.coords.usrCoords))<b&&(a=s.usrCoords[1],h=s.usrCoords[2],n=_,b=d,p=u,g=m),m+=u.numberPoints);return s=new o.Z(r.Z.COORDS_BY_USER,[a,h],i),[p.updateTransform(s),n+g]},projectPointToPoint:function(t,e){return e.coords},projectPointToBoard:function(t,e){var i,s,r,o=e||t.board,a=[[1,1,0,0,3,0,1],[-1,2,1,0,1,2,1],[-1,1,2,2,1,2,3],[1,2,3,0,3,2,3]],h=t.coords||t,l=o.getBoundingBox();for(i=0;i<4;i++)(r=a[i])[0]*h.usrCoords[r[1]]<r[0]*l[r[2]]&&((s=n.Z.crossProduct([1,l[r[3]],l[r[4]]],[1,l[r[5]],l[r[6]]]))[3]=0,s=n.Z.normalize(s),h=this.projectPointToLine({coords:h},{stdform:s},o));return h},distPointLine:function(t,e){var i,s=e[1],r=e[2],o=e[0];return Math.abs(s)+Math.abs(r)<n.Z.eps?Number.POSITIVE_INFINITY:(i=s*t[1]+r*t[2]+o,s*=s,r*=r,Math.abs(i)/Math.sqrt(s+r))},distPointSegment:function(t,e,i){var s,r,o,a,h,l,c=n.Z.eps*n.Z.eps,d=1e6;return s=(s=t[1]-e[1])===1/0?d:s===-1/0?-d:s,r=(r=t[2]-e[2])===1/0?d:r===-1/0?-d:r,(h=(o=(o=i[1]-e[1])===1/0?d:o===-1/0?-d:o)*o+(a=(a=i[2]-e[2])===1/0?d:a===-1/0?-d:a)*a)>c&&((l=(s*o+r*a)/h)<0?l=0:l>1&&(l=1),s-=l*o,r-=l*a),n.Z.hypot(s,r)},intersectionFunction3D:function(t,e,i){var s,o=this;return e.type===r.Z.OBJECT_TYPE_PLANE3D?i.type===r.Z.OBJECT_TYPE_PLANE3D?s=t.intersectionPlanePlane(e,i):i.type===r.Z.OBJECT_TYPE_SPHERE3D&&(s=o.meetPlaneSphere(e,i)):e.type===r.Z.OBJECT_TYPE_SPHERE3D&&(i.type===r.Z.OBJECT_TYPE_PLANE3D?s=o.meetPlaneSphere(i,e):i.type===r.Z.OBJECT_TYPE_SPHERE3D&&(s=o.meetSphereSphere(e,i))),s},meet3Planes:function(t,e,i,s,r,o){var a,h,l,c,d,u=[1,0,0,0];for(a=n.Z.crossProduct(r.slice(1),t.slice(1)),h=n.Z.crossProduct(t.slice(1),i.slice(1)),l=n.Z.crossProduct(i.slice(1),r.slice(1)),c=n.Z.innerProduct(t.slice(1),l,3),d=0;d<3;d++)u[d+1]=(e*l[d]+s*a[d]+o*h[d])/c;return u},meetPlanePlane:function(t,e,i,s){var r,o,a,h;return a=t.slice(1),h=e.slice(1),r=n.Z.crossProduct(a,h),a=i.slice(1),h=s.slice(1),o=n.Z.crossProduct(a,h),(h=n.Z.crossProduct(r,o)).unshift(0),h},meetPlaneSphere:function(t,e){var i=function(){return n.Z.innerProduct(t.normal,e.center.coords,4)-t.d};return[function(){return n.Z.axpy(-i(),t.normal,e.center.coords)},t.normal,function(){var t=e.Radius(),s=i();return Math.sqrt(t*t-s*s)}]},meetSphereSphere:function(t,e){var i=function(){var i=t.center.distance(e.center),s=t.Radius(),r=e.Radius();return(s-r)*(s+r)/(i*i)};return[function(){var s=i();return[1,.5*((1-s)*t.center.coords[1]+(1+s)*e.center.coords[1]),.5*((1-s)*t.center.coords[2]+(1+s)*e.center.coords[2]),.5*((1-s)*t.center.coords[3]+(1+s)*e.center.coords[3])]},function(){return a.Z.subtract(e.center.coords,t.center.coords)},function(){var s=t.center.distance(e.center),r=t.Radius(),o=e.Radius(),n=i(),a=.5*(r*r+o*o-.5*s*s*(1+n*n));return Math.sqrt(a)}]},_paramsOutOfRange:function(t,e,i){return t[0]<e[0]||t[0]>e[1]||t.length>1&&(t[1]<i[0]||t[1]>i[1])},projectCoordsToParametric:function(t,e,i,s){var r,o,a,h,c,d=.01*Math.random()+.5,u=2*i;return 1===i?r=.1*((h=[l.Z.evaluate(e.range[0]),l.Z.evaluate(e.range[1])])[1]-h[0]):2===i&&(h=[l.Z.evaluate(e.range_u[0]),l.Z.evaluate(e.range_u[1])],c=[l.Z.evaluate(e.range_v[0]),l.Z.evaluate(e.range_v[1])],r=.1*Math.min(h[1]-h[0],c[1]-c[0])),o=r/5e6,a=function(i,s,r,o){var n=[e.X.apply(e,r),e.Y.apply(e,r),e.Z.apply(e,r)],a=t[0]-n[0],l=t[1]-n[1],d=t[2]-n[2];return s>=2&&(o[0]=r[0]-h[0],o[1]=-r[0]+h[1]),s>=4&&(o[2]=r[1]-c[0],o[3]=-r[1]+c[1]),a*a+l*l+d*d},0===s.length&&(s[0]=d*(h[0]+h[1]),2===i&&(s[1]=d*(c[0]+c[1]))),n.Z.Nlp.FindMinimum(a,i,0,s,r,o,0,200),t=[e.X.apply(e,s),e.Y.apply(e,s),e.Z.apply(e,s)],this._paramsOutOfRange(s,h,c)&&(s[0]=d*(h[0]+h[1]),2===i&&(s[1]=d*(c[0]+c[1])),n.Z.Nlp.FindMinimum(a,i,u,s,r,o,0,200)),[1,e.X.apply(e,s),e.Y.apply(e,s),e.Z.apply(e,s)]},project3DTo3DPlane:function(t,e,i){var s,r;return i=i||[0,0,0],s=n.Z.norm(e),r=(n.Z.innerProduct(t,e,3)-n.Z.innerProduct(i,e,3))/s,n.Z.axpy(-r,e,t)},getPlaneBounds:function(t,e,i,s,r){var o,n,a,l,c;return t[2]+e[0]!==0?(a=[[t[0],e[0]],[t[1],e[1]]],l=[s-i[0],s-i[1]],o=(c=h.Z.Gauss(a,l))[0],n=c[1],l=[r-i[0],r-i[1]],[o,(c=h.Z.Gauss(a,l))[0],n,c[1]]):null},reuleauxPolygon:function(t,e){var i,s=2*Math.PI,r=s/e,o=(e-1)/2,a=0,h=function(h,l){return function(c,d){var u=(c%s+s)%s,p=Math.floor(u/r)%e;return d||(a=t[0].Dist(t[o]),i=n.Z.Geometry.rad([t[0].X()+1,t[0].Y()],t[0],t[o%e])),isNaN(p)?p:(u=.5*u+p*r*.5+i,t[p][h]()+a*Math[l](u))}};return[h("X","cos"),h("Y","sin"),0,s]}}),e.Z=n.Z.Geometry},785:function(t,e,i){var s=i(765),r=i(275),o=i(109);s.Z.Math.DoubleBits=function(){var t,e,i,s,r,o,n,a,h=new Float64Array(1),l=new Uint32Array(h.buffer);void 0!==Float64Array&&(h[0]=1,1072693248===l[1]?(t=function(t){return h[0]=t,[l[0],l[1]]},e=function(t,e){return l[0]=t,l[1]=e,h[0]},i=function(t){return h[0]=t,l[0]},s=function(t){return h[0]=t,l[1]},this.doubleBits=t,this.pack=e,this.lo=i,this.hi=s):1072693248===l[0]&&(r=function(t){return h[0]=t,[l[1],l[0]]},o=function(t,e){return l[1]=t,l[0]=e,h[0]},n=function(t){return h[0]=t,l[1]},a=function(t){return h[0]=t,l[0]},this.doubleBits=r,this.pack=o,this.lo=n,this.hi=a))},s.Z.extend(s.Z.Math.DoubleBits.prototype,{sign:function(t){return this.hi(t)>>>31},exponent:function(t){return(this.hi(t)<<1>>>21)-1023},fraction:function(t){var e=this.lo(t),i=this.hi(t),s=1048575&i;return 2146435072&i&&(s+=1<<20),[e,s]},denormalized:function(t){return!(2146435072&this.hi(t))}});var n=new s.Z.Math.DoubleBits,a=function(t,e){if(void 0!==t&&void 0!==e){if(r.Z.IntervalArithmetic.isInterval(t)){if(!r.Z.IntervalArithmetic.isSingleton(t))throw new TypeError("JXG.Math.IntervalArithmetic: interval `lo` must be a singleton");this.lo=t.lo}else this.lo=t;if(r.Z.IntervalArithmetic.isInterval(e)){if(!r.Z.IntervalArithmetic.isSingleton(e))throw new TypeError("JXG.Math.IntervalArithmetic: interval `hi` must be a singleton");this.hi=e.hi}else this.hi=e}else{if(void 0!==t)return Array.isArray(t)?new a(t[0],t[1]):new a(t,t);this.lo=this.hi=0}};s.Z.extend(a.prototype,{print:function(){console.log("[",this.lo,this.hi,"]")},set:function(t,e){return this.lo=t,this.hi=e,this},bounded:function(t,e){return this.set(r.Z.IntervalArithmetic.prev(t),r.Z.IntervalArithmetic.next(e))},boundedSingleton:function(t){return this.bounded(t,t)},assign:function(t,e){if("number"!=typeof t||"number"!=typeof e)throw new TypeError("JXG.Math.Interval#assign: arguments must be numbers");return isNaN(t)||isNaN(e)||t>e?this.setEmpty():this.set(t,e)},setEmpty:function(){return this.set(Number.POSITIVE_INFINITY,Number.NEGATIVE_INFINITY)},setWhole:function(){return this.set(Number.NEGATIVE_INFINITY,Number.POSITIVE_INFINITY)},open:function(t,e){return this.assign(r.Z.IntervalArithmetic.next(t),r.Z.IntervalArithmetic.prev(e))},halfOpenLeft:function(t,e){return this.assign(r.Z.IntervalArithmetic.next(t),e)},halfOpenRight:function(t,e){return this.assign(t,r.Z.IntervalArithmetic.prev(e))},toArray:function(){return[this.lo,this.hi]},clone:function(){return(new a).set(this.lo,this.hi)}}),s.Z.Math.IntervalArithmetic={Interval:function(t,e){return new a(t,e)},isInterval:function(t){return null!==t&&"object"==typeof t&&"number"==typeof t.lo&&"number"==typeof t.hi},isSingleton:function(t){return t.lo===t.hi},add:function(t,e){return o.Z.isNumber(t)&&(t=this.Interval(t)),o.Z.isNumber(e)&&(e=this.Interval(e)),new a(this.addLo(t.lo,e.lo),this.addHi(t.hi,e.hi))},sub:function(t,e){return o.Z.isNumber(t)&&(t=this.Interval(t)),o.Z.isNumber(e)&&(e=this.Interval(e)),new a(this.subLo(t.lo,e.hi),this.subHi(t.hi,e.lo))},mul:function(t,e){var i,s,r,n,h;return o.Z.isNumber(t)&&(t=this.Interval(t)),o.Z.isNumber(e)&&(e=this.Interval(e)),this.isEmpty(t)||this.isEmpty(e)?this.EMPTY.clone():(i=t.lo,s=t.hi,r=e.lo,n=e.hi,h=new a,i<0?s>0?r<0?n>0?(h.lo=Math.min(this.mulLo(i,n),this.mulLo(s,r)),h.hi=Math.max(this.mulHi(i,r),this.mulHi(s,n))):(h.lo=this.mulLo(s,r),h.hi=this.mulHi(i,r)):n>0?(h.lo=this.mulLo(i,n),h.hi=this.mulHi(s,n)):(h.lo=0,h.hi=0):r<0?n>0?(h.lo=this.mulLo(i,n),h.hi=this.mulHi(i,r)):(h.lo=this.mulLo(s,n),h.hi=this.mulHi(i,r)):n>0?(h.lo=this.mulLo(i,n),h.hi=this.mulHi(s,r)):(h.lo=0,h.hi=0):s>0?r<0?n>0?(h.lo=this.mulLo(s,r),h.hi=this.mulHi(s,n)):(h.lo=this.mulLo(s,r),h.hi=this.mulHi(i,n)):n>0?(h.lo=this.mulLo(i,r),h.hi=this.mulHi(s,n)):(h.lo=0,h.hi=0):(h.lo=0,h.hi=0),h)},div:function(t,e){return o.Z.isNumber(t)&&(t=this.Interval(t)),o.Z.isNumber(e)&&(e=this.Interval(e)),this.isEmpty(t)||this.isEmpty(e)?this.EMPTY.clone():this.zeroIn(e)?0!==e.lo?0!==e.hi?this.divZero(t):this.divNegative(t,e.lo):0!==e.hi?this.divPositive(t,e.hi):this.EMPTY.clone():this.divNonZero(t,e)},positive:function(t){return new a(t.lo,t.hi)},negative:function(t){return o.Z.isNumber(t)?new a(-t):new a(-t.hi,-t.lo)},isEmpty:function(t){return t.lo>t.hi},isWhole:function(t){return t.lo===-1/0&&t.hi===1/0},zeroIn:function(t){return this.hasValue(t,0)},hasValue:function(t,e){return!this.isEmpty(t)&&(t.lo<=e&&e<=t.hi)},hasInterval:function(t,e){return!!this.isEmpty(t)||!this.isEmpty(e)&&e.lo<=t.lo&&t.hi<=e.hi},intervalsOverlap:function(t,e){return!this.isEmpty(t)&&!this.isEmpty(e)&&(t.lo<=e.lo&&e.lo<=t.hi||e.lo<=t.lo&&t.lo<=e.hi)},divNonZero:function(t,e){var i=t.lo,s=t.hi,r=e.lo,o=e.hi,n=new a;return s<0?o<0?(n.lo=this.divLo(s,r),n.hi=this.divHi(i,o)):(n.lo=this.divLo(i,r),n.hi=this.divHi(s,o)):i<0?o<0?(n.lo=this.divLo(s,o),n.hi=this.divHi(i,o)):(n.lo=this.divLo(i,r),n.hi=this.divHi(s,r)):o<0?(n.lo=this.divLo(s,o),n.hi=this.divHi(i,r)):(n.lo=this.divLo(i,o),n.hi=this.divHi(s,r)),n},divPositive:function(t,e){return 0===t.lo&&0===t.hi?t:this.zeroIn(t)?this.WHOLE:t.hi<0?new a(Number.NEGATIVE_INFINITY,this.divHi(t.hi,e)):new a(this.divLo(t.lo,e),Number.POSITIVE_INFINITY)},divNegative:function(t,e){return 0===t.lo&&0===t.hi?t:this.zeroIn(t)?this.WHOLE:t.hi<0?new a(this.divLo(t.hi,e),Number.POSITIVE_INFINITY):new a(Number.NEGATIVE_INFINITY,this.divHi(t.lo,e))},divZero:function(t){return 0===t.lo&&0===t.hi?t:this.WHOLE},fmod:function(t,e){var i,s;return o.Z.isNumber(t)&&(t=this.Interval(t)),o.Z.isNumber(e)&&(e=this.Interval(e)),this.isEmpty(t)||this.isEmpty(e)?this.EMPTY.clone():(i=t.lo<0?e.lo:e.hi,s=(s=t.lo/i)<0?Math.ceil(s):Math.floor(s),this.sub(t,this.mul(e,new a(s))))},multiplicativeInverse:function(t){return o.Z.isNumber(t)&&(t=this.Interval(t)),this.isEmpty(t)?this.EMPTY.clone():this.zeroIn(t)?0!==t.lo?0!==t.hi?this.WHOLE:new a(Number.NEGATIVE_INFINITY,this.divHi(1,t.lo)):0!==t.hi?new a(this.divLo(1,t.hi),Number.POSITIVE_INFINITY):this.EMPTY.clone():new a(this.divLo(1,t.hi),this.divHi(1,t.lo))},pow:function(t,e){var i,s;if(o.Z.isNumber(t)&&(t=this.Interval(t)),this.isEmpty(t))return this.EMPTY.clone();if(this.isInterval(e)){if(!this.isSingleton(e))return this.EMPTY.clone();e=e.lo}return 0===e?0===t.lo&&0===t.hi?this.EMPTY.clone():this.ONE.clone():e<0?this.pow(this.multiplicativeInverse(t),-e):e%1==0?t.hi<0?(i=this.powLo(-t.hi,e),s=this.powHi(-t.lo,e),1==(1&e)?new a(-s,-i):new a(i,s)):t.lo<0?1==(1&e)?new a(-this.powLo(-t.lo,e),this.powHi(t.hi,e)):new a(0,this.powHi(Math.max(-t.lo,t.hi),e)):new a(this.powLo(t.lo,e),this.powHi(t.hi,e)):(console.warn("power is not an integer, you should use nth-root instead, returning an empty interval"),this.EMPTY.clone())},sqrt:function(t){return o.Z.isNumber(t)&&(t=this.Interval(t)),this.nthRoot(t,2)},nthRoot:function(t,e){var i,s,r,n,h;if(o.Z.isNumber(t)&&(t=this.Interval(t)),this.isEmpty(t)||e<0)return this.EMPTY.clone();if(this.isInterval(e)){if(!this.isSingleton(e))return this.EMPTY.clone();e=e.lo}return i=1/e,t.hi<0?e%1==0&&1==(1&e)?(s=this.powHi(-t.lo,i),r=this.powLo(-t.hi,i),new a(-s,-r)):this.EMPTY.clone():t.lo<0?(n=this.powHi(t.hi,i),e%1==0&&1==(1&e)?(h=-this.powHi(-t.lo,i),new a(h,n)):new a(0,n)):new a(this.powLo(t.lo,i),this.powHi(t.hi,i))},exp:function(t){return o.Z.isNumber(t)&&(t=this.Interval(t)),this.isEmpty(t)?this.EMPTY.clone():new a(this.expLo(t.lo),this.expHi(t.hi))},log:function(t){var e;return o.Z.isNumber(t)&&(t=this.Interval(t)),this.isEmpty(t)?this.EMPTY.clone():(e=t.lo<=0?Number.NEGATIVE_INFINITY:this.logLo(t.lo),new a(e,this.logHi(t.hi)))},ln:function(t){return this.log(t)},log10:function(t){return this.isEmpty(t)?this.EMPTY.clone():this.div(this.log(t),this.log(new a(10,10)))},log2:function(t){return this.isEmpty(t)?this.EMPTY.clone():this.div(this.log(t),this.log(new a(2,2)))},hull:function(t,e){var i=this.isEmpty(t),s=this.isEmpty(e);return i&&s?this.EMPTY.clone():i?e.clone():s?t.clone():new a(Math.min(t.lo,e.lo),Math.max(t.hi,e.hi))},intersection:function(t,e){var i,s;return this.isEmpty(t)||this.isEmpty(e)?this.EMPTY.clone():(i=Math.max(t.lo,e.lo))<=(s=Math.min(t.hi,e.hi))?new a(i,s):this.EMPTY.clone()},union:function(t,e){if(!this.intervalsOverlap(t,e))throw new Error("Interval#unions do not overlap");return new a(Math.min(t.lo,e.lo),Math.max(t.hi,e.hi))},difference:function(t,e){if(this.isEmpty(t)||this.isWhole(e))return this.EMPTY.clone();if(this.intervalsOverlap(t,e)){if(t.lo<e.lo&&e.hi<t.hi)throw new Error("Interval.difference: difference creates multiple intervals");return e.lo<=t.lo&&e.hi===1/0||e.hi>=t.hi&&e.lo===-1/0?this.EMPTY.clone():e.lo<=t.lo?(new a).halfOpenLeft(e.hi,t.hi):(new a).halfOpenRight(t.lo,e.lo)}return t.clone()},width:function(t){return this.isEmpty(t)?0:this.subHi(t.hi,t.lo)},abs:function(t){return o.Z.isNumber(t)&&(t=this.Interval(t)),this.isEmpty(t)?this.EMPTY.clone():t.lo>=0?t.clone():t.hi<=0?this.negative(t):new a(0,Math.max(-t.lo,t.hi))},max:function(t,e){var i=this.isEmpty(t),s=this.isEmpty(e);return i&&s?this.EMPTY.clone():i?e.clone():s?t.clone():new a(Math.max(t.lo,e.lo),Math.max(t.hi,e.hi))},min:function(t,e){var i=this.isEmpty(t),s=this.isEmpty(e);return i&&s?this.EMPTY.clone():i?e.clone():s?t.clone():new a(Math.min(t.lo,e.lo),Math.min(t.hi,e.hi))},onlyInfinity:function(t){return!isFinite(t.lo)&&t.lo===t.hi},_handleNegative:function(t){var e;return t.lo<0&&(t.lo===-1/0?(t.lo=0,t.hi=1/0):(e=Math.ceil(-t.lo/this.piTwiceLow),t.lo+=this.piTwiceLow*e,t.hi+=this.piTwiceLow*e)),t},cos:function(t){var e,i,s,r,o,n,h,l;return this.isEmpty(t)||this.onlyInfinity(t)?this.EMPTY.clone():(e=(new a).set(t.lo,t.hi),this._handleNegative(e),i=this.PI_TWICE,s=this.fmod(e,i),this.width(s)>=i.lo?new a(-1,1):s.lo>=this.piHigh?(r=this.cos(this.sub(s,this.PI)),this.negative(r)):(o=s.lo,n=s.hi,h=this.cosLo(n),l=this.cosHi(o),n<=this.piLow?new a(h,l):n<=i.lo?new a(-1,Math.max(h,l)):new a(-1,1)))},sin:function(t){return this.isEmpty(t)||this.onlyInfinity(t)?this.EMPTY.clone():this.cos(this.sub(t,this.PI_HALF))},tan:function(t){var e,i,s;return this.isEmpty(t)||this.onlyInfinity(t)?this.EMPTY.clone():(e=(new a).set(t.lo,t.hi),this._handleNegative(e),s=this.PI,(i=this.fmod(e,s)).lo>=this.piHalfLow&&(i=this.sub(i,s)),i.lo<=-this.piHalfLow||i.hi>=this.piHalfLow?this.WHOLE.clone():new a(this.tanLo(i.lo),this.tanHi(i.hi)))},asin:function(t){var e,i;return this.isEmpty(t)||t.hi<-1||t.lo>1?this.EMPTY.clone():(e=t.lo<=-1?-this.piHalfHigh:this.asinLo(t.lo),i=t.hi>=1?this.piHalfHigh:this.asinHi(t.hi),new a(e,i))},acos:function(t){var e,i;return this.isEmpty(t)||t.hi<-1||t.lo>1?this.EMPTY.clone():(e=t.hi>=1?0:this.acosLo(t.hi),i=t.lo<=-1?this.piHigh:this.acosHi(t.lo),new a(e,i))},acot:function(t){return this.isEmpty(t)?this.EMPTY.clone():new a(this.acotLo(t.lo),this.acotHi(t.hi))},atan:function(t){return this.isEmpty(t)?this.EMPTY.clone():new a(this.atanLo(t.lo),this.atanHi(t.hi))},sinh:function(t){return this.isEmpty(t)?this.EMPTY.clone():new a(this.sinhLo(t.lo),this.sinhHi(t.hi))},cosh:function(t){return this.isEmpty(t)?this.EMPTY.clone():t.hi<0?new a(this.coshLo(t.hi),this.coshHi(t.lo)):t.lo>=0?new a(this.coshLo(t.lo),this.coshHi(t.hi)):new a(1,this.coshHi(-t.lo>t.hi?t.lo:t.hi))},tanh:function(t){return this.isEmpty(t)?this.EMPTY.clone():new a(this.tanhLo(t.lo),this.tanhHi(t.hi))},equal:function(t,e){return this.isEmpty(t)?this.isEmpty(e):!this.isEmpty(e)&&t.lo===e.lo&&t.hi===e.hi},notEqual:function(t,e){return this.isEmpty(t)?!this.isEmpty(e):this.isEmpty(e)||t.hi<e.lo||t.lo>e.hi},lt:function(t,e){return o.Z.isNumber(t)&&(t=this.Interval(t)),o.Z.isNumber(e)&&(e=this.Interval(e)),!this.isEmpty(t)&&!this.isEmpty(e)&&t.hi<e.lo},gt:function(t,e){return o.Z.isNumber(t)&&(t=this.Interval(t)),o.Z.isNumber(e)&&(e=this.Interval(e)),!this.isEmpty(t)&&!this.isEmpty(e)&&t.lo>e.hi},leq:function(t,e){return o.Z.isNumber(t)&&(t=this.Interval(t)),o.Z.isNumber(e)&&(e=this.Interval(e)),!this.isEmpty(t)&&!this.isEmpty(e)&&t.hi<=e.lo},geq:function(t,e){return o.Z.isNumber(t)&&(t=this.Interval(t)),o.Z.isNumber(e)&&(e=this.Interval(e)),!this.isEmpty(t)&&!this.isEmpty(e)&&t.lo>=e.hi},piLow:3.141592653589793,piHigh:3.1415926535897936,piHalfLow:1.5707963267948966,piHalfHigh:1.5707963267948968,piTwiceLow:6.283185307179586,piTwiceHigh:6.283185307179587,identity:function(t){return t},_prev:function(t){return t===1/0?t:this.nextafter(t,-1/0)},_next:function(t){return t===-1/0?t:this.nextafter(t,1/0)},prev:function(t){return this._prev(t)},next:function(t){return this._next(t)},toInteger:function(t){return t<0?Math.ceil(t):Math.floor(t)},addLo:function(t,e){return this.prev(t+e)},addHi:function(t,e){return this.next(t+e)},subLo:function(t,e){return this.prev(t-e)},subHi:function(t,e){return this.next(t-e)},mulLo:function(t,e){return this.prev(t*e)},mulHi:function(t,e){return this.next(t*e)},divLo:function(t,e){return this.prev(t/e)},divHi:function(t,e){return this.next(t/e)},intLo:function(t){return this.toInteger(this.prev(t))},intHi:function(t){return this.toInteger(this.next(t))},logLo:function(t){return this.prev(Math.log(t))},logHi:function(t){return this.next(Math.log(t))},expLo:function(t){return this.prev(Math.exp(t))},expHi:function(t){return this.next(Math.exp(t))},sinLo:function(t){return this.prev(Math.sin(t))},sinHi:function(t){return this.next(Math.sin(t))},cosLo:function(t){return this.prev(Math.cos(t))},cosHi:function(t){return this.next(Math.cos(t))},tanLo:function(t){return this.prev(Math.tan(t))},tanHi:function(t){return this.next(Math.tan(t))},asinLo:function(t){return this.prev(Math.asin(t))},asinHi:function(t){return this.next(Math.asin(t))},acosLo:function(t){return this.prev(Math.acos(t))},acosHi:function(t){return this.next(Math.acos(t))},acotLo:function(t){return this.prev(r.Z.acot(t))},acotHi:function(t){return this.next(r.Z.acot(t))},atanLo:function(t){return this.prev(Math.atan(t))},atanHi:function(t){return this.next(Math.atan(t))},sinhLo:function(t){return this.prev(r.Z.sinh(t))},sinhHi:function(t){return this.next(r.Z.sinh(t))},coshLo:function(t){return this.prev(r.Z.cosh(t))},coshHi:function(t){return this.next(r.Z.cosh(t))},tanhLo:function(t){return this.prev(r.Z.tanh(t))},tanhHi:function(t){return this.next(r.Z.tanh(t))},sqrtLo:function(t){return this.prev(Math.sqrt(t))},sqrtHi:function(t){return this.next(Math.sqrt(t))},powLo:function(t,e){var i;if(e%1!=0)return this.prev(Math.pow(t,e));for(i=1==(1&e)?t:1,e>>=1;e>0;)t=this.mulLo(t,t),1==(1&e)&&(i=this.mulLo(t,i)),e>>=1;return i},powHi:function(t,e){var i;if(e%1!=0)return this.next(Math.pow(t,e));for(i=1==(1&e)?t:1,e>>=1;e>0;)t=this.mulHi(t,t),1==(1&e)&&(i=this.mulHi(t,i)),e>>=1;return i},disable:function(){this.next=this.prev=this.identity},enable:function(){this.prev=function(t){return this._prev(t)},this.next=function(t){return this._next(t)}},SMALLEST_DENORM:Math.pow(2,-1074),UINT_MAX:-1>>>0,nextafter:function(t,e){var i,s;return isNaN(t)||isNaN(e)?NaN:t===e?t:0===t?e<0?-this.SMALLEST_DENORM:this.SMALLEST_DENORM:(s=n.hi(t),i=n.lo(t),e>t==t>0?i===this.UINT_MAX?(s+=1,i=0):i+=1:0===i?(i=this.UINT_MAX,s-=1):i-=1,n.pack(i,s))}},s.Z.Math.IntervalArithmetic.PI=new a(r.Z.IntervalArithmetic.piLow,r.Z.IntervalArithmetic.piHigh),s.Z.Math.IntervalArithmetic.PI_HALF=new a(r.Z.IntervalArithmetic.piHalfLow,r.Z.IntervalArithmetic.piHalfHigh),s.Z.Math.IntervalArithmetic.PI_TWICE=new a(r.Z.IntervalArithmetic.piTwiceLow,r.Z.IntervalArithmetic.piTwiceHigh),s.Z.Math.IntervalArithmetic.ZERO=new a(0),s.Z.Math.IntervalArithmetic.ONE=new a(1),s.Z.Math.IntervalArithmetic.WHOLE=(new a).setWhole(),s.Z.Math.IntervalArithmetic.EMPTY=(new a).setEmpty(),e.Z=s.Z.Math.IntervalArithmetic},275:function(t,e,i){var s,r=i(765),o=i(109),n=function(t){var e,i;return t.memo||(e={},i=Array.prototype.join,t.memo=function(){var r=i.call(arguments);return e[r]!==s?e[r]:e[r]=t.apply(this,arguments)}),t.memo};r.Z.Math={eps:1e-6,relDif:function(t,e){var i=Math.abs(t),s=Math.abs(e);return 0===(s=Math.max(i,s))?0:Math.abs(t-e)/s},mod:function(t,e){return t-Math.floor(t/e)*e},wrap:function(t,e,i){return e+this.mod(t-e,i-e)},clamp:function(t,e,i){return Math.min(Math.max(t,e),i)},wrapAndClamp:function(t,e,i,s){var r=.5*(e+i),o=.5*s;return this.clamp(this.wrap(t,r-o,r+o),e,i)},vector:function(t,e){var i,s;for(e=e||0,i=[],s=0;s<t;s++)i[s]=e;return i},matrix:function(t,e,i){var s,r,o;for(i=i||0,e=e||t,s=[],r=0;r<t;r++)for(s[r]=[],o=0;o<e;o++)s[r][o]=i;return s},identity:function(t,e){var i,r;for(e===s&&"number"!=typeof e&&(e=t),i=this.matrix(t,e),r=0;r<Math.min(t,e);r++)i[r][r]=1;return i},frustum:function(t,e,i,s,r,o){var n=this.matrix(4,4);return n[0][0]=2*r/(e-t),n[0][1]=0,n[0][2]=(e+t)/(e-t),n[0][3]=0,n[1][0]=0,n[1][1]=2*r/(s-i),n[1][2]=(s+i)/(s-i),n[1][3]=0,n[2][0]=0,n[2][1]=0,n[2][2]=-(o+r)/(o-r),n[2][3]=-o*r*2/(o-r),n[3][0]=0,n[3][1]=0,n[3][2]=-1,n[3][3]=0,n},projection:function(t,e,i,s){var r=i*Math.tan(t/2),o=r*e;return this.frustum(-o,o,-r,r,i,s)},matVecMult:function(t,e){var i,s,r,o=t.length,n=e.length,a=[];if(3===n)for(i=0;i<o;i++)a[i]=t[i][0]*e[0]+t[i][1]*e[1]+t[i][2]*e[2];else for(i=0;i<o;i++){for(r=0,s=0;s<n;s++)r+=t[i][s]*e[s];a[i]=r}return a},vecMatMult:function(t,e){var i,s,r,o=e.length,n=t.length,a=[];if(3===n)for(i=0;i<o;i++)a[i]=t[0]*e[0][i]+t[1]*e[1][i]+t[2]*e[2][i];else for(i=0;i<n;i++){for(r=0,s=0;s<o;s++)r+=t[s]*e[s][i];a[i]=r}return a},matMatMult:function(t,e){var i,s,r,o,n=t.length,a=n>0?e[0].length:0,h=e.length,l=this.matrix(n,a);for(i=0;i<n;i++)for(s=0;s<a;s++){for(r=0,o=0;o<h;o++)r+=t[i][o]*e[o][s];l[i][s]=r}return l},matNumberMult:function(t,e){var i,s,r=t.length,o=r>0?t[0].length:0,n=this.matrix(r,o);for(i=0;i<r;i++)for(s=0;s<o;s++)n[i][s]=t[i][s]*e;return n},matMatAdd:function(t,e){var i,s,r=t.length,o=r>0?t[0].length:0,n=this.matrix(r,o);for(i=0;i<r;i++)for(s=0;s<o;s++)n[i][s]=t[i][s]+e[i][s];return n},transpose:function(t){var e,i,s,r,o;for(r=t.length,o=t.length>0?t[0].length:0,e=this.matrix(o,r),i=0;i<o;i++)for(s=0;s<r;s++)e[i][s]=t[s][i];return e},inverse:function(t){var e,i,s,o,n,a,h,l=this.eps*this.eps,c=t.length,d=[],u=[],p=[];for(e=0;e<c;e++){for(d[e]=[],i=0;i<c;i++)d[e][i]=t[e][i];u[e]=e}for(i=0;i<c;i++){for(a=Math.abs(d[i][i]),o=i,e=i+1;e<c;e++)Math.abs(d[e][i])>a&&(a=Math.abs(d[e][i]),o=e);if(a<=l)return r.Z.warn("JXG.Math.inverse: singular matrix"),[];if(o>i){for(s=0;s<c;s++)h=d[i][s],d[i][s]=d[o][s],d[o][s]=h;h=u[i],u[i]=u[o],u[o]=h}for(n=1/d[i][i],e=0;e<c;e++)d[e][i]*=n;for(d[i][i]=n,s=0;s<c;s++)if(s!==i){for(e=0;e<c;e++)e!==i&&(d[e][s]-=d[e][i]*d[i][s]);d[i][s]=-n*d[i][s]}}for(e=0;e<c;e++){for(s=0;s<c;s++)p[u[s]]=d[e][s];for(s=0;s<c;s++)d[e][s]=p[s]}return d},trace:function(t){var e,i,s=0;if(t.length!==(i=t.length>0?t[0].length:0))return NaN;for(e=0;e<i;e++)s+=t[e][e];return s},innerProduct:function(t,e,i){var r,n=0;for(i!==s&&o.Z.isNumber(i)||(i=t.length),r=0;r<i;r++)n+=t[r]*e[r];return n},crossProduct:function(t,e){return[t[1]*e[2]-t[2]*e[1],t[2]*e[0]-t[0]*e[2],t[0]*e[1]-t[1]*e[0]]},norm:function(t,e){var i,r=0;for(e!==s&&o.Z.isNumber(e)||(e=t.length),i=0;i<e;i++)r+=t[i]*t[i];return Math.sqrt(r)},axpy:function(t,e,i){var s,r=e.length,o=[];for(s=0;s<r;s++)o[s]=t*e[s]+i[s];return o},factorial:n((function(t){return t<0?NaN:0===(t=Math.floor(t))||1===t?1:t*this.factorial(t-1)})),binomial:n((function(t,e){var i,s;if(e>t||e<0)return NaN;if(e=Math.round(e),t=Math.round(t),0===e||e===t)return 1;for(i=1,s=0;s<e;s++)i*=t-s,i/=s+1;return i})),cosh:Math.cosh||function(t){return.5*(Math.exp(t)+Math.exp(-t))},sinh:Math.sinh||function(t){return.5*(Math.exp(t)-Math.exp(-t))},acosh:Math.acosh||function(t){return Math.log(t+Math.sqrt(t*t-1))},asinh:Math.asinh||function(t){return t===-1/0?t:Math.log(t+Math.sqrt(t*t+1))},cot:function(t){return 1/Math.tan(t)},acot:function(t){return(t>=0?.5:-.5)*Math.PI-Math.atan(t)},nthroot:function(t,e){var i=1/e;return e<=0||Math.floor(e)!==e?NaN:0===t?0:t>0?Math.exp(i*Math.log(t)):e%2==1?-Math.exp(i*Math.log(-t)):NaN},cbrt:Math.cbrt||function(t){return this.nthroot(t,3)},pow:function(t,e){return 0===t?0===e?1:0:Math.floor(e)===e?Math.pow(t,e):t>0?Math.exp(e*Math.log(t)):NaN},ratpow:function(t,e,i){var s;return 0===e?1:0===i?NaN:(s=this.gcd(e,i),this.nthroot(this.pow(t,e/s),i/s))},log10:function(t){return Math.log(t)/Math.log(10)},log2:function(t){return Math.log(t)/Math.log(2)},log:function(t,e){return void 0!==e&&o.Z.isNumber(e)?Math.log(t)/Math.log(e):Math.log(t)},sign:Math.sign||function(t){return 0===(t=+t)||isNaN(t)?t:t>0?1:-1},squampow:function(t,e){var i;if(Math.floor(e)===e){for(i=1,e<0&&(t=1/t,e*=-1);0!==e;)1&e&&(i*=t),e>>=1,t*=t;return i}return this.pow(t,e)},gcd:function(t,e){var i;if(t=Math.abs(t),e=Math.abs(e),!o.Z.isNumber(t)||!o.Z.isNumber(e))return NaN;for(e>t&&(i=t,t=e,e=i);;){if(0===(t%=e))return e;if(0===(e%=t))return t}},lcm:function(t,e){var i;return o.Z.isNumber(t)&&o.Z.isNumber(e)?0!==(i=t*e)?i/this.gcd(t,e):0:NaN},roundToStep:function(t,e,i,s){var n,a,h=t;if(!o.Z.exists(e)&&!o.Z.exists(i)&&!o.Z.exists(s))return h;if(r.Z.exists(s)&&(h=Math.min(h,s)),r.Z.exists(i)&&(h=Math.max(h,i)),a=i||0,r.Z.exists(e)){if(n=(h-a)/e,Number.isInteger(n))return h;h=a+(n=Math.round(n))*e}return r.Z.exists(s)&&(h=Math.min(h,s)),r.Z.exists(i)&&(h=Math.max(h,i)),h},erf:function(t){return this.ProbFuncs.erf(t)},erfc:function(t){return this.ProbFuncs.erfc(t)},erfi:function(t){return this.ProbFuncs.erfi(t)},ndtr:function(t){return this.ProbFuncs.ndtr(t)},ndtri:function(t){return this.ProbFuncs.ndtri(t)},hypot:function(){var t,e,i,s;for(e=arguments.length,t=0,s=0;t<e;t++)s+=(i=arguments[t])*i;return Math.sqrt(s)},hstep:function(t){return t>0?1:t<0?0:.5},gamma:function(t){var e,i,s,r,o,n=[1,676.5203681218851,-1259.1392167224028,771.3234287776531,-176.6150291621406,12.507343278686905,-.13857109526572012,9984369578019572e-21,1.5056327351493116e-7];if(t<.5)i=Math.PI/(Math.sin(Math.PI*t)*this.gamma(1-t));else{for(t-=1,e=n[0],o=n.length,r=1;r<o;r++)e+=n[r]/(t+r);s=t+7+.5,i=Math.sqrt(2*Math.PI)*Math.pow(s,t+.5)*Math.exp(-s)*e}return i},lt:function(t,e){return t<e},leq:function(t,e){return t<=e},gt:function(t,e){return t>e},geq:function(t,e){return t>=e},eq:function(t,e){return t===e},neq:function(t,e){return t!==e},and:function(t,e){return t&&e},not:function(t){return!t},or:function(t,e){return t||e},xor:function(t,e){return(t||e)&&!(t&&e)},decToFraction:function(t,e){var i,s,r,n,a,h,l,c,d,u=0;for(e=o.Z.def(e,.001),s=(t=1e-12*Math.round(1e12*t))<0?-1:1,t=Math.abs(t),i=Math.floor(t),t-=Math.floor(t),h=1,n=a=r=0,d=0,l=c=1;t-Math.floor(t)>e&&u<20;)t=1/(t-r),n=h+(r=Math.floor(t))*a,l=d+r*c,h=a,d=c,a=n,c=l,u++;return[s,i,n,l]},normalize:function(t){var e,i,s=2*t[3],r=t[4]/s;return t[5]=r,t[6]=-t[1]/s,t[7]=-t[2]/s,isFinite(r)?Math.abs(r)>=1?(t[0]=(t[6]*t[6]+t[7]*t[7]-r*r)/(2*r),t[1]=-t[6]/r,t[2]=-t[7]/r,t[3]=1/(2*r),t[4]=1):(i=r<=0?-1:1,t[0]=i*(t[6]*t[6]+t[7]*t[7]-r*r)*.5,t[1]=-i*t[6],t[2]=-i*t[7],t[3]=i/2,t[4]=i*r):(e=this.hypot(t[1],t[2]),t[0]/=e,t[1]/=e,t[2]/=e,t[3]=0,t[4]=1),t},toGL:function(t){var e,i,s;if(e="function"==typeof Float32Array?new Float32Array(16):new Array(16),4!==t.length&&4!==t[0].length)return e;for(i=0;i<4;i++)for(s=0;s<4;s++)e[i+4*s]=t[i][s];return e},Vieta:function(t){var e,i,s,r=t.length,o=[];for(o=t.slice(),e=1;e<r;++e){for(s=o[e],o[e]*=o[e-1],i=e-1;i>=1;--i)o[i]+=o[i-1]*s;o[0]+=s}return o}},e.Z=r.Z.Math},336:function(t,e,i){var s=i(765),r=i(109),o=i(503),n=i(275),a={rk4:{s:4,A:[[0,0,0,0],[.5,0,0,0],[0,.5,0,0],[0,0,1,0]],b:[1/6,1/3,1/3,1/6],c:[0,.5,.5,1]},heun:{s:2,A:[[0,0],[1,0]],b:[.5,.5],c:[0,1]},euler:{s:1,A:[[0]],b:[1],c:[0]}};n.Z.Numerics={Gauss:function(t,e){var i,s,o,a,h,l=n.Z.eps,c=t.length>0?t[0].length:0;if(c!==e.length||c!==t.length)throw new Error("JXG.Math.Numerics.Gauss: Dimensions don't match. A must be a square matrix and b must be of the same length as A.");for(a=[],h=e.slice(0,c),i=0;i<c;i++)a[i]=t[i].slice(0,c);for(s=0;s<c;s++){for(i=c-1;i>s;i--)if(Math.abs(a[i][s])>l)if(Math.abs(a[s][s])<l)r.Z.swap(a,i,s),r.Z.swap(h,i,s);else for(a[i][s]/=a[s][s],h[i]-=a[i][s]*h[s],o=s+1;o<c;o++)a[i][o]-=a[i][s]*a[s][o];if(Math.abs(a[s][s])<l)throw new Error("JXG.Math.Numerics.Gauss(): The given matrix seems to be singular.")}return this.backwardSolve(a,h,!0),h},backwardSolve:function(t,e,i){var s,r,o,n,a;for(s=i?e:e.slice(0,e.length),r=t.length,o=t.length>0?t[0].length:0,n=r-1;n>=0;n--){for(a=o-1;a>n;a--)s[n]-=t[n][a]*s[a];s[n]/=t[n][n]}return s},gaussBareiss:function(t){var e,i,s,r,o,a,h,l,c,d=n.Z.eps;if((h=t.length)<=0)return 0;for(t[0].length<h&&(h=t[0].length),l=[],r=0;r<h;r++)l[r]=t[r].slice(0,h);for(i=1,s=1,e=0;e<h-1;e++){if(a=l[e][e],Math.abs(a)<d){for(r=e+1;r<h&&!(Math.abs(l[r][e])>=d);r++);if(r===h)return 0;for(o=e;o<h;o++)c=l[r][o],l[r][o]=l[e][o],l[e][o]=c;s=-s,a=l[e][e]}for(r=e+1;r<h;r++)for(o=e+1;o<h;o++)c=a*l[r][o]-l[r][e]*l[e][o],l[r][o]=c/i;i=a}return s*l[h-1][h-1]},det:function(t){return 2===t.length&&2===t[0].length?t[0][0]*t[1][1]-t[1][0]*t[0][1]:this.gaussBareiss(t)},Jacobi:function(t){var e,i,s,r,o,a,h,l,c,d=n.Z.eps*n.Z.eps,u=0,p=t.length,f=[[0,0,0],[0,0,0],[0,0,0]],_=[[0,0,0],[0,0,0],[0,0,0]],m=0;for(e=0;e<p;e++){for(i=0;i<p;i++)f[e][i]=0,_[e][i]=t[e][i],u+=Math.abs(_[e][i]);f[e][e]=1}if(1===p)return[_,f];if(u<=0)return[_,f];u/=p*p;do{for(l=0,c=0,i=1;i<p;i++)for(e=0;e<i;e++)if((r=Math.abs(_[e][i]))>c&&(c=r),l+=r,r>=d){for(r=.5*Math.atan2(2*_[e][i],_[e][e]-_[i][i]),o=Math.sin(r),a=Math.cos(r),s=0;s<p;s++)h=_[s][e],_[s][e]=a*h+o*_[s][i],_[s][i]=-o*h+a*_[s][i],h=f[s][e],f[s][e]=a*h+o*f[s][i],f[s][i]=-o*h+a*f[s][i];for(_[e][e]=a*_[e][e]+o*_[i][e],_[i][i]=-o*_[e][i]+a*_[i][i],_[e][i]=0,s=0;s<p;s++)_[e][s]=_[s][e],_[i][s]=_[s][i]}m+=1}while(Math.abs(l)/u>d&&m<2e3);return[_,f]},NewtonCotes:function(t,e,i){var s,o,n,a=0,h=i&&r.Z.isNumber(i.number_of_nodes)?i.number_of_nodes:28,l={trapez:!0,simpson:!0,milne:!0},c=i&&i.integration_type&&l.hasOwnProperty(i.integration_type)&&l[i.integration_type]?i.integration_type:"milne",d=(t[1]-t[0])/h;switch(c){case"trapez":for(a=.5*(e(t[0])+e(t[1])),s=t[0],o=0;o<h-1;o++)a+=e(s+=d);a*=d;break;case"simpson":if(h%2>0)throw new Error("JSXGraph:  INT_SIMPSON requires config.number_of_nodes dividable by 2.");for(n=h/2,a=e(t[0])+e(t[1]),s=t[0],o=0;o<n-1;o++)a+=2*e(s+=2*d);for(s=t[0]-d,o=0;o<n;o++)a+=4*e(s+=2*d);a*=d/3;break;default:if(h%4>0)throw new Error("JSXGraph: Error in INT_MILNE: config.number_of_nodes must be a multiple of 4");for(n=.25*h,a=7*(e(t[0])+e(t[1])),s=t[0],o=0;o<n-1;o++)a+=14*e(s+=4*d);for(s=t[0]-3*d,o=0;o<n;o++)a+=32*(e(s+=4*d)+e(s+2*d));for(s=t[0]-2*d,o=0;o<n;o++)a+=12*e(s+=4*d);a*=2*d/45}return a},Romberg:function(t,e,i){var s,o,n,a,h,l,c,d,u=[],p=0,f=1/0,_=i&&r.Z.isNumber(i.max_iterations)?i.max_iterations:20,m=i&&r.Z.isNumber(i.eps)?i.eps:i.eps||1e-7;for(s=t[0],n=(o=t[1])-s,h=1,u[0]=.5*n*(e(s)+e(o)),l=0;l<_;++l){for(a=0,n*=.5,h*=2,d=1,c=1;c<h;c+=2)a+=e(s+c*n);for(u[l+1]=.5*u[l]+a*n,p=u[l+1],c=l-1;c>=0;--c)d*=4,u[c]=u[c+1]+(u[c+1]-u[c])/(d-1),p=u[c];if(Math.abs(p-f)<m*Math.abs(p))break;f=p}return p},GaussLegendre:function(t,e,i){var s,o,n,a,h,l,c,d,u=0,p=[],f=[],_=i&&r.Z.isNumber(i.n)?i.n:12;if(_>18&&(_=18),p[2]=[.5773502691896257],f[2]=[1],p[4]=[.33998104358485626,.8611363115940526],f[4]=[.6521451548625461,.34785484513745385],p[6]=[.2386191860831969,.6612093864662645,.932469514203152],f[6]=[.46791393457269104,.3607615730481386,.17132449237917036],p[8]=[.1834346424956498,.525532409916329,.7966664774136267,.9602898564975363],f[8]=[.362683783378362,.31370664587788727,.22238103445337448,.10122853629037626],p[10]=[.14887433898163122,.4333953941292472,.6794095682990244,.8650633666889845,.9739065285171717],f[10]=[.29552422471475287,.26926671930999635,.21908636251598204,.1494513491505806,.06667134430868814],p[12]=[.1252334085114689,.3678314989981802,.5873179542866175,.7699026741943047,.9041172563704749,.9815606342467192],f[12]=[.24914704581340277,.2334925365383548,.20316742672306592,.16007832854334622,.10693932599531843,.04717533638651183],p[14]=[.10805494870734367,.31911236892788974,.5152486363581541,.6872929048116855,.827201315069765,.9284348836635735,.9862838086968123],f[14]=[.2152638534631578,.2051984637212956,.18553839747793782,.15720316715819355,.12151857068790319,.08015808715976021,.03511946033175186],p[16]=[.09501250983763744,.2816035507792589,.45801677765722737,.6178762444026438,.755404408355003,.8656312023878318,.9445750230732326,.9894009349916499],f[16]=[.1894506104550685,.18260341504492358,.16915651939500254,.14959598881657674,.12462897125553388,.09515851168249279,.062253523938647894,.027152459411754096],p[18]=[.0847750130417353,.2518862256915055,.41175116146284263,.5597708310739475,.6916870430603532,.8037049589725231,.8926024664975557,.9558239495713977,.9915651684209309],f[18]=[.1691423829631436,.16427648374583273,.15468467512626524,.14064291467065065,.12255520671147846,.10094204410628717,.07642573025488905,.0497145488949698,.02161601352648331],p[3]=[0,.7745966692414834],f[3]=[.8888888888888888,.5555555555555556],p[5]=[0,.5384693101056831,.906179845938664],f[5]=[.5688888888888889,.47862867049936647,.23692688505618908],p[7]=[0,.4058451513773972,.7415311855993945,.9491079123427585],f[7]=[.4179591836734694,.3818300505051189,.27970539148927664,.1294849661688697],p[9]=[0,.3242534234038089,.6133714327005904,.8360311073266358,.9681602395076261],f[9]=[.3302393550012598,.31234707704000286,.26061069640293544,.1806481606948574,.08127438836157441],p[11]=[0,.26954315595234496,.5190961292068118,.7301520055740494,.8870625997680953,.978228658146057],f[11]=[.2729250867779006,.26280454451024665,.23319376459199048,.18629021092773426,.1255803694649046,.05566856711617366],p[13]=[0,.2304583159551348,.44849275103644687,.6423493394403402,.8015780907333099,.9175983992229779,.9841830547185881],f[13]=[.2325515532308739,.22628318026289723,.2078160475368885,.17814598076194574,.13887351021978725,.09212149983772845,.04048400476531588],p[15]=[0,.20119409399743451,.3941513470775634,.5709721726085388,.7244177313601701,.8482065834104272,.937273392400706,.9879925180204854],f[15]=[.2025782419255613,.19843148532711158,.1861610000155622,.16626920581699392,.13957067792615432,.10715922046717194,.07036604748810812,.03075324199611727],p[17]=[0,.17848418149584785,.3512317634538763,.5126905370864769,.6576711592166907,.7815140038968014,.8802391537269859,.9506755217687678,.9905754753144174],f[17]=[.17944647035620653,.17656270536699264,.16800410215645004,.15404576107681028,.13513636846852548,.11188384719340397,.08503614831717918,.0554595293739872,.02414830286854793],s=t[0],o=t[1],a=_+1>>1,c=p[_],d=f[_],l=.5*(o-s),h=.5*(o+s),!0&_)for(u=d[0]*e(h),n=1;n<a;++n)u+=d[n]*(e(h+l*c[n])+e(h-l*c[n]));else for(u=0,n=0;n<a;++n)u+=d[n]*(e(h+l*c[n])+e(h-l*c[n]));return l*u},_rescale_error:function(t,e,i){var s,r,o=2220446049250313e-31;return t=Math.abs(t),0!==i&&0!==t&&(t=(s=Math.pow(200*t/i,1.5))<1?i*s:i),e>20041683600089728e-310&&(r=50*o*e)>t&&(t=r),t},_gaussKronrod:function(t,e,i,s,r,o,n){var a,h,l,c,d,u,p,f,_,m,g,b=t[0],v=t[1],Z=.5*(b+v),C=.5*(v-b),y=Math.abs(C),P=e(Z),E=0,x=P*o[i-1],O=Math.abs(x),M=0,w=[],S=[];for(i%2==0&&(E=P*r[i/2-1]),a=Math.floor((i-1)/2),d=0;d<a;d++)g=(_=e(Z-(f=C*s[u=2*d+1])))+(m=e(Z+f)),w[u]=_,S[u]=m,E+=r[d]*g,x+=o[u]*g,O+=o[u]*(Math.abs(_)+Math.abs(m));for(a=Math.floor(i/2),d=0;d<a;d++)_=e(Z-(f=C*s[p=2*d])),m=e(Z+f),w[p]=_,S[p]=m,x+=o[p]*(_+m),O+=o[p]*(Math.abs(_)+Math.abs(m));for(l=.5*x,M=o[i-1]*Math.abs(P-l),d=0;d<i-1;d++)M+=o[d]*(Math.abs(w[d]-l)+Math.abs(S[d]-l));return c=(x-E)*C,O*=y,M*=y,h=x*=C,n.abserr=this._rescale_error(c,O,M),n.resabs=O,n.resasc=M,h},GaussKronrod15:function(t,e,i){return this._gaussKronrod(t,e,8,[.9914553711208126,.9491079123427585,.8648644233597691,.7415311855993945,.5860872354676911,.4058451513773972,.20778495500789848,0],[.1294849661688697,.27970539148927664,.3818300505051189,.4179591836734694],[.022935322010529224,.06309209262997856,.10479001032225019,.14065325971552592,.1690047266392679,.19035057806478542,.20443294007529889,.20948214108472782],i)},GaussKronrod21:function(t,e,i){return this._gaussKronrod(t,e,11,[.9956571630258081,.9739065285171717,.9301574913557082,.8650633666889845,.7808177265864169,.6794095682990244,.5627571346686047,.4333953941292472,.2943928627014602,.14887433898163122,0],[.06667134430868814,.1494513491505806,.21908636251598204,.26926671930999635,.29552422471475287],[.011694638867371874,.032558162307964725,.054755896574351995,.07503967481091996,.0931254545836976,.10938715880229764,.12349197626206584,.13470921731147334,.14277593857706009,.14773910490133849,.1494455540029169],i)},GaussKronrod31:function(t,e,i){return this._gaussKronrod(t,e,16,[.9980022986933971,.9879925180204854,.9677390756791391,.937273392400706,.8972645323440819,.8482065834104272,.790418501442466,.7244177313601701,.650996741297417,.5709721726085388,.4850818636402397,.3941513470775634,.29918000715316884,.20119409399743451,.1011420669187175,0],[.03075324199611727,.07036604748810812,.10715922046717194,.13957067792615432,.16626920581699392,.1861610000155622,.19843148532711158,.2025782419255613],[.005377479872923349,.015007947329316122,.02546084732671532,.03534636079137585,.04458975132476488,.05348152469092809,.06200956780067064,.06985412131872826,.07684968075772038,.08308050282313302,.08856444305621176,.09312659817082532,.09664272698362368,.09917359872179196,.10076984552387559,.10133000701479154],i)},_workspace:function(t,e){return{limit:e,size:0,nrmax:0,i:0,alist:[t[0]],blist:[t[1]],rlist:[0],elist:[0],order:[0],level:[0],qpsrt:function(){var t,e,i,s,r,o=this.size-1,n=this.limit,a=this.nrmax,h=this.order[a];if(o<2)return this.order[0]=0,this.order[1]=1,void(this.i=h);for(t=this.elist[h];a>0&&t>this.elist[this.order[a-1]];)this.order[a]=this.order[a-1],a--;for(r=o<n/2+2?o:n-o+1,i=a+1;i<r&&t<this.elist[this.order[i]];)this.order[i-1]=this.order[i],i++;for(this.order[i-1]=h,e=this.elist[o],s=r-1;s>i-2&&e>=this.elist[this.order[s]];)this.order[s+1]=this.order[s],s--;this.order[s+1]=o,h=this.order[a],this.i=h,this.nrmax=a},set_initial_result:function(t,e){this.size=1,this.rlist[0]=t,this.elist[0]=e},update:function(t,e,i,s,r,o,n,a){var h=this.i,l=this.size,c=this.level[this.i]+1;a>s?(this.alist[h]=r,this.rlist[h]=n,this.elist[h]=a,this.level[h]=c,this.alist[l]=t,this.blist[l]=e,this.rlist[l]=i,this.elist[l]=s,this.level[l]=c):(this.blist[h]=e,this.rlist[h]=i,this.elist[h]=s,this.level[h]=c,this.alist[l]=r,this.blist[l]=o,this.rlist[l]=n,this.elist[l]=a,this.level[l]=c),this.size++,c>this.maximum_level&&(this.maximum_level=c),this.qpsrt()},retrieve:function(){var t=this.i;return{a:this.alist[t],b:this.blist[t],r:this.rlist[t],e:this.elist[t]}},sum_results:function(){var t,e=this.size,i=0;for(t=0;t<e;t++)i+=this.rlist[t];return i},subinterval_too_small:function(t,e,i){var s=1.0000000000000222*(Math.abs(e)+22250738585072014e-321);return Math.abs(t)<=s&&Math.abs(i)<=s}}},Qag:function(t,e,i){var o,a,h,l,c,d,u,p,f,_,m,g,b,v,Z,C,y,P,E=this._workspace(t,1e3),x=i&&r.Z.isNumber(i.limit)?i.limit:15,O=i&&r.Z.isNumber(i.epsrel)?i.epsrel:1e-7,M=i&&r.Z.isNumber(i.epsabs)?i.epsabs:1e-7,w=i&&r.Z.isFunction(i.q)?i.q:this.GaussKronrod15,S={},T=0,N=0,A=0,D=0,k=0,R=0,L=0,B=0,I=0,j=0;if(x>E.limit&&s.Z.warn("iteration limit exceeds available workspace"),M<=0&&(O<50*n.Z.eps||O<5e-29)&&s.Z.warn("tolerance cannot be acheived with given epsabs and epsrel"),h=w.apply(this,[t,e,S]),l=S.abserr,c=S.resabs,d=S.resasc,E.set_initial_result(h,l),u=Math.max(M,O*Math.abs(h)),l<=11102230246251565e-30*c&&l>u)return h,s.Z.warn("cannot reach tolerance because of roundoff error on first attempt"),-1/0;if(l<=u&&l!==d||0===l)return h;if(1===x)return h,s.Z.warn("a maximum of one iteration was insufficient"),-1/0;o=h,a=l,T=1;do{k=0,R=0,L=0,B=0,I=0,j=0,g=(y=E.retrieve()).a,b=y.b,v=y.r,Z=y.e,p=g,_=f=.5*(g+b),m=b,k=w.apply(this,[[p,f],e,S]),B=S.abserr,C=S.resasc,R=w.apply(this,[[_,m],e,S]),a+=(j=B+(I=S.abserr))-Z,o+=(L=k+R)-v,C!==B&&S.resasc!==I&&(P=v-L,Math.abs(P)<=1e-5*Math.abs(L)&&j>=.99*Z&&N++,T>=10&&j>Z&&A++),a>(u=Math.max(M,O*Math.abs(o)))&&((N>=6||A>=20)&&(D=2),E.subinterval_too_small(p,_,m)&&(D=3)),E.update(p,f,k,B,_,m,R,I),g=(y=E.retrieve()).a_i,b=y.b_i,v=y.r_i,Z=y.e_i,T++}while(T<x&&!D&&a>u);return E.sum_results()},I:function(t,e){return this.Qag(t,e,{q:this.GaussKronrod15,limit:15,epsrel:1e-7,epsabs:1e-7})},Newton:function(t,e,i){var s,o=0,a=n.Z.eps,h=t.apply(i,[e]);for(r.Z.isArray(e)&&(e=e[0]);o<50&&Math.abs(h)>a;)s=this.D(t,i)(e),Math.abs(s)>a?e-=h/s:e+=.2*Math.random()-1,h=t.apply(i,[e]),o+=1;return e},root:function(t,e,i){return this.chandrupatla(t,e,i)},generalizedNewton:function(t,e,i,s){var r,o,a,h,l,c,d,u,p,f,_,m,g,b,v=0;for(this.generalizedNewton.t1memo?(r=this.generalizedNewton.t1memo,o=this.generalizedNewton.t2memo):(r=i,o=s),f=(d=t.X(r)-e.X(o))*d+(u=t.Y(r)-e.Y(o))*u,_=this.D(t.X,t),m=this.D(e.X,e),g=this.D(t.Y,t),b=this.D(e.Y,e);f>n.Z.eps&&v<10;)a=_(r),h=-m(o),l=g(r),r-=((c=-b(o))*d-h*u)/(p=a*c-h*l),o-=(a*u-l*d)/p,f=(d=t.X(r)-e.X(o))*d+(u=t.Y(r)-e.Y(o))*u,v+=1;return this.generalizedNewton.t1memo=r,this.generalizedNewton.t2memo=o,Math.abs(r)<Math.abs(o)?[t.X(r),t.Y(r)]:[e.X(o),e.Y(o)]},Neville:function(t){var e=[],i=function(i){return function(s,r){var o,a,h,l=n.Z.binomial,c=t.length,d=c-1,u=0,p=0;if(!r)for(h=1,o=0;o<c;o++)e[o]=l(d,o)*h,h*=-1;for(a=s,o=0;o<c;o++){if(0===a)return t[o][i]();h=e[o]/a,a-=1,u+=t[o][i]()*h,p+=h}return u/p}};return[i("X"),i("Y"),0,function(){return t.length-1}]},splineDef:function(t,e){var i,s,r,o=Math.min(t.length,e.length),n=[],a=[],h=[],l=[],c=[],d=[];if(2===o)return[0,0];for(s=0;s<o;s++)i={X:t[s],Y:e[s]},h.push(i);for(h.sort((function(t,e){return t.X-e.X})),s=0;s<o;s++)t[s]=h[s].X,e[s]=h[s].Y;for(s=0;s<o-1;s++)l.push(t[s+1]-t[s]);for(s=0;s<o-2;s++)c.push(6*(e[s+2]-e[s+1])/l[s+1]-6*(e[s+1]-e[s])/l[s]);for(n.push(2*(l[0]+l[1])),a.push(c[0]),s=0;s<o-3;s++)r=l[s+1]/n[s],n.push(2*(l[s+1]+l[s+2])-r*l[s+1]),a.push(c[s+1]-r*a[s]);for(d[o-3]=a[o-3]/n[o-3],s=o-4;s>=0;s--)d[s]=(a[s]-l[s+1]*d[s+1])/n[s];for(s=o-3;s>=0;s--)d[s+1]=d[s];return d[0]=0,d[o-1]=0,d},splineEval:function(t,e,i,s){var o,n,a,h,l,c,d,u=Math.min(e.length,i.length),p=1,f=!1,_=[];for(r.Z.isArray(t)?(p=t.length,f=!0):t=[t],o=0;o<p;o++){if(t[o]<e[0]||e[o]>e[u-1])return NaN;for(n=1;n<u&&!(t[o]<=e[n]);n++);a=i[n-=1],h=(i[n+1]-i[n])/(e[n+1]-e[n])-(e[n+1]-e[n])/6*(s[n+1]+2*s[n]),l=s[n]/2,c=(s[n+1]-s[n])/(6*(e[n+1]-e[n])),d=t[o]-e[n],_.push(a+(h+(l+c*d)*d)*d)}return f?_:_[0]},generatePolynomialTerm:function(t,e,i,s){var o,n=[];for(o=e;o>=0;o--)r.Z.concat(n,["(",t[o].toPrecision(s),")"]),o>1?r.Z.concat(n,["*",i,"<sup>",o,"<","/sup> + "]):1===o&&r.Z.concat(n,["*",i," + "]);return n.join("")},lagrangePolynomial:function(t){var e=[],i=this,s=function(i,s){var r,o,n,a,h=t.length,l=0,c=0;if(!s)for(r=0;r<h;r++){for(e[r]=1,n=t[r].X(),o=0;o<h;o++)o!==r&&(e[r]*=n-t[o].X());e[r]=1/e[r]}for(r=0;r<h;r++){if(i===(n=t[r].X()))return t[r].Y();c+=a=e[r]/(i-n),l+=a*t[r].Y()}return l/c};return s.getTerm=function(e,s,r){return i.lagrangePolynomialTerm(t,e,s,r)()},s.getCoefficients=function(){return i.lagrangePolynomialCoefficients(t)()},s},lagrangePolynomialTerm:function(t,e,i,r){var a=this;return function(){var h,l,c,d,u,p=t.length,f=!0;for(i=i||"x",void 0===r&&(r=" * "),l=p-1,h=a.lagrangePolynomialCoefficients(t)(),c="",d=0;d<h.length;d++)u=h[d],Math.abs(u)<n.Z.eps||(s.Z.exists(e)&&(u=o.Z._round10(u,-e)),f?(c+=u>0?u:"-"+-u,f=!1):c+=u>0?" + "+u:" - "+-u,l-d>1?c+=r+i+"^"+(l-d):l-d==1&&(c+=r+i));return c}},lagrangePolynomialCoefficients:function(t){return function(){var e,i,s,r,o=t.length,a=[],h=[],l=[];for(i=0;i<o;i++)l[i]=0;for(e=0;e<o;e++){for(s=t[e].Y(),r=t[e].X(),a=[],i=0;i<o;i++)i!==e&&(s/=r-t[i].X(),a.push(t[i].X()));for(h=[1].concat(n.Z.Vieta(a)),i=0;i<h.length;i++)l[i]+=(i%2==1?-1:1)*h[i]*s}return l}},_initCubicPoly:function(t,e,i,s){return[t,i,-3*t+3*e-2*i-s,2*t-2*e+i+s]},CardinalSpline:function(t,e,i){var s,o,a,h,l=[],c=this;return h=r.Z.isFunction(e)?e:function(){return e},void 0===i&&(i="uniform"),o=function(e){return function(r,o){var d,u,p,f,_,m,g,b,v,Z;if(t.length<2)return NaN;if(!o)for(a=h(),p={X:function(){return 2*t[0].X()-t[1].X()},Y:function(){return 2*t[0].Y()-t[1].Y()},Dist:function(t){var e=this.X()-t.X(),i=this.Y()-t.Y();return n.Z.hypot(e,i)}},f={X:function(){return 2*t[t.length-1].X()-t[t.length-2].X()},Y:function(){return 2*t[t.length-1].Y()-t[t.length-2].Y()},Dist:function(t){var e=this.X()-t.X(),i=this.Y()-t.Y();return n.Z.hypot(e,i)}},Z=(s=[p].concat(t,[f])).length,l[e]=[],d=0;d<Z-3;d++)"centripetal"===i?(g=s[d].Dist(s[d+1]),b=s[d+2].Dist(s[d+1]),v=s[d+3].Dist(s[d+2]),g=Math.sqrt(g),b=Math.sqrt(b),v=Math.sqrt(v),b<n.Z.eps&&(b=1),g<n.Z.eps&&(g=b),v<n.Z.eps&&(v=b),_=(s[d+1][e]()-s[d][e]())/g-(s[d+2][e]()-s[d][e]())/(b+g)+(s[d+2][e]()-s[d+1][e]())/b,m=(s[d+2][e]()-s[d+1][e]())/b-(s[d+3][e]()-s[d+1][e]())/(v+b)+(s[d+3][e]()-s[d+2][e]())/v,_*=b,m*=b,l[e][d]=c._initCubicPoly(s[d+1][e](),s[d+2][e](),a*_,a*m)):l[e][d]=c._initCubicPoly(s[d+1][e](),s[d+2][e](),a*(s[d+2][e]()-s[d][e]()),a*(s[d+3][e]()-s[d+1][e]()));return isNaN(r)?NaN:(Z=t.length,r<=0?t[0][e]():r>=Z?t[Z-1][e]():(d=Math.floor(r))===r?t[d][e]():(r-=d,void 0===(u=l[e][d])?NaN:((u[3]*r+u[2])*r+u[1])*r+u[0]))}},[o("X"),o("Y"),0,function(){return t.length-1}]},CatmullRomSpline:function(t,e){return this.CardinalSpline(t,.5,e)},regressionPolynomial:function(t,e,i){var s,o,a,h,l,c,d="";if(r.Z.isPoint(t)&&r.Z.isFunction(t.Value))o=function(){return t.Value()};else if(r.Z.isFunction(t))o=t;else{if(!r.Z.isNumber(t))throw new Error("JSXGraph: Can't create regressionPolynomial from degree of type'"+typeof t+"'.");o=function(){return t}}if(3===arguments.length&&r.Z.isArray(e)&&r.Z.isArray(i))l=0;else if(2===arguments.length&&r.Z.isArray(e)&&e.length>0&&r.Z.isPoint(e[0]))l=1;else{if(!(2===arguments.length&&r.Z.isArray(e)&&e.length>0&&e[0].usrCoords&&e[0].scrCoords))throw new Error("JSXGraph: Can't create regressionPolynomial. Wrong parameters.");l=2}return(c=function(t,c){var u,p,f,_,m,g,b,v,Z,C=e.length;if(Z=Math.floor(o()),!c){if(1===l)for(a=[],h=[],u=0;u<C;u++)a[u]=e[u].X(),h[u]=e[u].Y();if(2===l)for(a=[],h=[],u=0;u<C;u++)a[u]=e[u].usrCoords[1],h[u]=e[u].usrCoords[2];if(0===l)for(a=[],h=[],u=0;u<C;u++)r.Z.isFunction(e[u])?a.push(e[u]()):a.push(e[u]),r.Z.isFunction(i[u])?h.push(i[u]()):h.push(i[u]);for(f=[],p=0;p<C;p++)f.push([1]);for(u=1;u<=Z;u++)for(p=0;p<C;p++)f[p][u]=f[p][u-1]*a[p];m=h,_=n.Z.transpose(f),g=n.Z.matMatMult(_,f),b=n.Z.matVecMult(_,m),s=n.Z.Numerics.Gauss(g,b),d=n.Z.Numerics.generatePolynomialTerm(s,Z,"x",3)}for(v=s[Z],u=Z-1;u>=0;u--)v=v*t+s[u];return v}).getTerm=function(){return d},c},bezier:function(t){var e,i,s=function(s){return function(r,o){var n=3*Math.floor(r),a=r%1,h=1-a;return o||(i=3*Math.floor((t.length-1)/3),e=Math.floor(i/3)),r<0?t[0][s]():r>=e?t[i][s]():isNaN(r)?NaN:h*h*(h*t[n][s]()+3*a*t[n+1][s]())+(3*h*t[n+2][s]()+a*t[n+3][s]())*a*a}};return[s("X"),s("Y"),0,function(){return Math.floor(t.length/3)}]},bspline:function(t,e){var i,s=function(s){return function(r,o){var n,a,h,l,c=t.length,d=c-1,u=e;if(d<=0)return NaN;if(d+2<=u&&(u=d+1),r<=0)return t[0][s]();if(r>=d-u+2)return t[d][s]();for(h=Math.floor(r)+u-1,i=function(t,e){var i,s=[];for(i=0;i<t+e+1;i++)s[i]=i<e?0:i<=t?i-e+1:t-e+2;return s}(d,u),l=function(t,e,i,s){var r,o,n,a,h,l=[];for(e[s]<=t&&t<e[s+1]?l[s]=1:l[s]=0,r=2;r<=i;r++)for(o=s-r+1;o<=s;o++)n=o<=s-r+1||o<0?0:l[o],a=o>=s?0:l[o+1],h=e[o+r-1]-e[o],l[o]=0===h?0:(t-e[o])/h*n,0!=(h=e[o+r]-e[o+1])&&(l[o]+=(e[o+r]-t)/h*a);return l}(r,i,u,h),n=0,a=h-u+1;a<=h;a++)a<c&&a>=0&&(n+=t[a][s]()*l[a]);return n}};return[s("X"),s("Y"),0,function(){return t.length-1}]},D:function(t,e){return r.Z.exists(e)?function(i,s){var r=1e-5;return(t.apply(e,[i+r,s])-t.apply(e,[i-r,s]))/2e-5}:function(e,i){var s=1e-5;return(t(e+s,i)-t(e-s,i))/2e-5}},_riemannValue:function(t,e,i,s){var r,o,n,a;if(s<0&&("trapezoidal"!==i&&(t+=s),s*=-1,"lower"===i?i="upper":"upper"===i&&(i="lower")),a=.01*s,"right"===i)r=e(t+s);else if("middle"===i)r=e(t+.5*s);else if("left"===i||"trapezoidal"===i)r=e(t);else if("lower"===i){for(r=e(t),n=t+a;n<=t+s;n+=a)(o=e(n))<r&&(r=o);(o=e(t+s))<r&&(r=o)}else if("upper"===i){for(r=e(t),n=t+a;n<=t+s;n+=a)(o=e(n))>r&&(r=o);(o=e(t+s))>r&&(r=o)}else r="random"===i?e(t+s*Math.random()):"simpson"===i?(e(t)+4*e(t+.5*s)+e(t+s))/6:e(t);return r},riemann:function(t,e,i,s,o){var n,a,h,l,c,d,u,p,f,_,m,g,b,v,Z=[],C=[],y=s,P=0;if(r.Z.isArray(t)?(v=t[0],b=t[1]):b=t,(e=Math.floor(e))<=0)return[Z,C,P];for(a=(o-s)/e,n=0;n<e;n++){if("simpson"===i){for(P+=this._riemannValue(y,b,i,a)*a,m=.5*a,u=b(y),p=b(y+m),l=((f=b(y+2*m))+u-2*p)/(m*m)*.5,c=(f-u)/(2*m),d=p,h=0;h<30;h++)_=h*a/30-m,Z.push(y+_+m),C.push(l*_*_+c*_+d);y+=a,g=f}else g=this._riemannValue(y,b,i,a),Z.push(y),C.push(g),y+=a,"trapezoidal"===i?(P+=.5*(g+(f=b(y)))*a,g=f):P+=g*a,Z.push(y),C.push(g);Z.push(y),C.push(g)}for(n=0;n<e;n++){if("simpson"===i&&v){for(P-=this._riemannValue(y,v,i,-a)*a,m=.5*a,u=v(y),p=v(y-m),l=((f=v(y-2*m))+u-2*p)/(m*m)*.5,c=(f-u)/(2*m),d=p,h=0;h<30;h++)_=h*a/30-m,Z.push(y-_-m),C.push(l*_*_+c*_+d);y-=a,g=f}else g=v?this._riemannValue(y,v,i,-a):0,Z.push(y),C.push(g),y-=a,v&&("trapezoidal"===i?(P-=.5*(g+(f=v(y)))*a,g=f):P-=g*a);Z.push(y),C.push(g),Z.push(y),C.push(b(y))}return[Z,C,P]},riemannsum:function(t,e,i,r,o){return s.Z.deprecated("Numerics.riemannsum()","Numerics.riemann()[2]"),this.riemann(t,e,i,r,o)[2]},rungeKutta:function(t,e,i,s,o){var n,h,l,c,d,u,p=[],f=[],_=(i[1]-i[0])/s,m=i[0],g=e.length,b=[],v=0;for(r.Z.isString(t)&&(t=a[t]||a.euler),u=t.s,p=e.slice(),h=0;h<=s;h++){for(b[v]=p.slice(),v++,c=[],l=0;l<u;l++){for(n=0;n<g;n++)f[n]=0;for(d=0;d<l;d++)for(n=0;n<g;n++)f[n]+=t.A[l][d]*_*c[d][n];for(n=0;n<g;n++)f[n]+=p[n];c.push(o(m+t.c[l]*_,f))}for(n=0;n<g;n++)f[n]=0;for(d=0;d<u;d++)for(n=0;n<g;n++)f[n]+=t.b[d]*c[d][n];for(n=0;n<g;n++)p[n]=p[n]+_*f[n];m+=_}return b},maxIterationsRoot:80,maxIterationsMinimize:500,findBracket:function(t,e,i){var s,o,n,a,h,l,c,d,u,p;if(r.Z.isArray(e))return e;for(s=e,n=t.call(i,s),p=(a=[s-.1*(o=0===s?1:s),s+.1*o,s-1,s+1,s-.5*o,s+.5*o,s-.6*o,s+.6*o,s-1*o,s+1*o,s-2*o,s+2*o,s-5*o,s+5*o,s-10*o,s+10*o,s-50*o,s+50*o,s-100*o,s+100*o]).length,u=0;u<p&&(h=a[u],!(n*(l=t.call(i,h))<=0));u++);return h<s&&(c=s,s=h,h=c,d=n,n=l,l=d),[s,n,h,l]},fzero:function(t,e,i){var s,o,a,h,l,c,d,u,p,f,_,m,g,b,v,Z,C=n.Z.eps,y=this.maxIterationsRoot,P=0;if(r.Z.isArray(e)){if(e.length<2)throw new Error("JXG.Math.Numerics.fzero: length of array x0 has to be at least two.");s=(u=this.findDomain(t,e,i))[0],o=u[1],h=t.call(i,s),l=t.call(i,o)}else s=(d=this.findBracket(t,e,i))[0],h=d[1],o=d[2],l=d[3];if(Math.abs(h)<=C)return s;if(Math.abs(l)<=C)return o;if(h*l>0)return r.Z.isArray(e)?this.fminbr(t,[s,o],i):this.Newton(t,s,i);for(a=s,c=h;P<y;){if(p=o-s,Math.abs(c)<Math.abs(l)&&(s=o,o=a,a=s,h=l,l=c,c=h),g=2*C*Math.abs(o)+.5*C,Z=.5*(a-o),Math.abs(Z)<=g||Math.abs(l)<=C)return o;Math.abs(p)>=g&&Math.abs(h)>Math.abs(l)&&(m=a-o,s===a?(b=m*(f=l/h),v=1-f):(b=(_=l/h)*(m*(v=h/c)*(v-(f=l/c))-(o-s)*(f-1)),v=(v-1)*(f-1)*(_-1)),b>0?v=-v:b=-b,b<.75*m*v-.5*Math.abs(g*v)&&b<Math.abs(p*v*.5)&&(Z=b/v)),Math.abs(Z)<g&&(Z=Z>0?g:-g),s=o,h=l,o+=Z,((l=t.call(i,o))>0&&c>0||l<0&&c<0)&&(a=s,c=h),P++}return o},chandrupatla:function(t,e,i){var s,o,a,h,l,c,d,u,p,f,_,m,g,b,v,Z,C,y,P,E,x=0,O=this.maxIterationsRoot,M=1+.001*Math.random(),w=.5*M,S=n.Z.eps;if(r.Z.isArray(e)){if(e.length<2)throw new Error("JXG.Math.Numerics.fzero: length of array x0 has to be at least two.");s=e[0],a=t.call(i,s),o=e[1],h=t.call(i,o)}else s=(l=this.findBracket(t,e,i))[0],a=l[1],o=l[2],h=l[3];if(a*h>0)return r.Z.isArray(e)?this.fminbr(t,[s,o],i):this.Newton(t,s,i);c=s,d=o,f=a,_=h;do{if(p=c+w*(d-c),g=t.call(i,p),Math.sign(g)===Math.sign(f)?(u=c,c=p,m=f,f=g):(u=d,d=c,m=_,_=f),b=c=p,v=f=g,Math.abs(_)<Math.abs(f)&&(b=d,v=_),(Z=(2*S*Math.abs(b)+5e-6)/Math.abs(d-c))>.5||0===v)break;C=(c-d)/(u-d),y=(f-_)/(m-_),P=1-Math.sqrt(1-C),E=Math.sqrt(C),(w=P<y&&y<E?f/(_-f)*(m/(_-m))+f/(m-f)*(_/(m-_))*((u-c)/(d-c)):.5*M)<Z&&(w=Z),w>1-Z&&(w=1-Z),x++}while(x<=O);return b},findDomain:function(t,e,i,s){var o,n,a,h,l,c,d=1-1/1.61803398875;if(void 0===s&&(s=!0),!r.Z.isArray(e)||e.length<2)throw new Error("JXG.Math.Numerics.findDomain: length of array x0 has to be at least two.");if(o=(l=e.slice())[0],n=l[1],h=t.call(i,o),isNaN(h)){for(c=0;n-o>.001&&c<20;)a=(n-o)*d+o,h=t.call(i,a),isNaN(h)?o=a:n=a,c++;l[0]=s?o:n}if(o=l[0],n=l[1],h=t.call(i,n),isNaN(h)){for(c=0;n-o>.001&&c<20;)a=n-(n-o)*d,h=t.call(i,a),isNaN(h)?n=a:o=a,c++;l[1]=s?n:o}return l},fminbr:function(t,e,i){var s,o,a,h,l,c,d,u,p,f,_,m,g,b,v,Z,C,y=.5*(3-Math.sqrt(5)),P=n.Z.eps,E=n.Z.eps,x=this.maxIterationsMinimize,O=0;if(!r.Z.isArray(e)||e.length<2)throw new Error("JXG.Math.Numerics.fminbr: length of array x0 has to be at least two.");for(a=h=(s=(p=this.findDomain(t,e,i))[0])+y*((o=p[1])-s),l=h,c=d=t.call(i,h),u=d;O<x;){if(f=o-s,_=.5*(s+o),m=E*Math.abs(a)+P/3,Math.abs(a-_)+.5*f<=2*m)return a;g=y*(a<_?o-a:s-a),Math.abs(a-l)>=m&&(b=(a-h)*(v=(a-h)*(c-u))-(a-l)*(Z=(a-l)*(c-d)),(v=2*(v-Z))>0?b=-b:v=-v,Math.abs(b)<Math.abs(g*v)&&b>v*(s-a+2*m)&&b<v*(o-a-2*m)&&(g=b/v)),Math.abs(g)<m&&(g=g>0?m:-m),Z=a+g,(C=t.call(i,Z))<=c?(Z<a?o=a:s=a,h=l,l=a,a=Z,d=u,u=c,c=C):(Z<a?s=Z:o=Z,C<=u||l===a?(h=l,l=Z,d=u,u=C):(C<=d||h===a||h===l)&&(h=Z,d=C)),O+=1}return a},glomin:function(t,e){var i,s,r,o,a,h,l,c,d,u,p,f,_,m,g,b,v,Z,C,y,P,E,x,O,M,w,S,T,N=1e7,A=n.Z.eps,D=n.Z.eps*n.Z.eps,k=n.Z.eps*n.Z.eps*n.Z.eps;if(M=e[0],w=e[1],S=t(M)<t(w)?M:w,T=i=w,s=M,P=v=t(w),v<(b=C=t(M))?b=v:T=M,w<=M)return b;for(d=.5*(1+16*k)*N,c=3,o=s-(g=S<=M||w<=S?.5*(M+w):S),l=9/11,(Z=t(g))<b&&(T=g,b=Z);;){if(h=g-i,O=w-s,u=_=(a=s-i)*a*(E=C-Z)-o*o*(x=C-v),p=f=2*(o*x-a*E),c<1e6||C<=b)for(;p*(_*(P-C)+O*p*(C-b+A))<O*d*_*(O*p-_)&&(y=t(r=s+_/p))<b&&(T=r,b=y),p=1,!(O<=(_=1e-5*(w-M)*(c=1611*c%1048576))););else for(p=1,_=1e-5*(w-M)*(c=1611*c%1048576);_<O;)p*(_*(P-C)+O*p*(C-b+A))<O*d*_*(O*p-_)&&(y=t(r=s+_/p))<b&&(T=r,b=y),p=1,_=1e-5*(w-M)*(c=1611*c%1048576);for(u=(l=.5*(1+l))*(u+2*(_=d*o*a*h)*(m=Math.sqrt((C-b+A)/d))),_=(_=-.5*(o+(E+2.01*D)/(o*d)))<m||o<0?s+m:s+_,r=0<u*(p+=.5*f)?s+u/p:_;(w<=(r=Math.max(r,_))?(r=w,y=P):y=t(r),y<b&&(T=r,b=y),o=r-s,!(r<=_))&&(u=2*(C-y)/(N*o),!((1+9*k)*o<=Math.abs(u)))&&!(.5*d*(o*o+u*u)<=C-b+(y-b)+2*A);)r=.5*(s+r),l*=.9;if(w<=r)break;i=g,g=s,s=r,v=Z,Z=C,C=y}return[T,b]},polzeros:function(t,e,i,r,o){var n,a,h,l=[],c=[],d=[],u=function(t,e,i){var r,o,n=t.length-1;if(i=i||!1)for(o=s.Z.C.mult(n,t[n]),r=n-1;r>0;r--)o.mult(e),o.add(s.Z.C.mult(t[r],r));else for(o=s.Z.C.copy(t[n]),r=n-1;r>=0;r--)o.mult(e),o.add(t[r]);return o},p=function(t,e,i){var r,o,n=t.length-1;if(i=i||!1)for(o=s.Z.C.mult(n,t[0]),r=n-1;r>0;r--)o.mult(e),o.add(s.Z.C.mult(t[n-r],r));else for(o=s.Z.C.copy(t[0]),r=n-1;r>=0;r--)o.mult(e),o.add(t[n-r]);return o},f=function(t,e){var i,s,r=t.length-1;for(s=t[r],i=r-1;i>=0;i--)s=s*e+t[i];return s};for(i=i||Number.EPSILON,r=r||30,a=t.length,s.Z.isNumber(e)&&e>=0&&e<a-1&&(a=e+1),n=0;n<a;n++)l.push(new s.Z.Complex(t[n]));for(n=0;n<a;n++)if(0!==l[n].real||0!==l[n].imaginary){h=n;break}for(n=0;n<h;n++)c.push(new s.Z.Complex(0));for(n=(a=(l=l.slice(h)).length)-1;n>=0&&(0===l[n].real&&0===l[n].imaginary);n--)l.pop();if(0===(a=l.length))return[];if(o)for(n=0;n<a-1;n++)d.push(new s.Z.Complex(o[n]));else d=function(t){var e,i,r,o,n=t.length-1,a=2*Math.PI/n,h=Math.PI/n*.5,l=[];for((r=s.Z.C.mult(-1,t[n-1])).div(s.Z.C.mult(n,t[n])),o=s.Z.C.div(u(t,r),t[n]),0===(i=Math.pow(s.Z.C.abs(o),1/n))&&(i=1),e=0;e<n;e++)t=new s.Z.Complex(i*Math.cos(a*e+h),i*Math.sin(a*e+h)),l[e]=s.Z.C.add(r,t);return l}(l);return function(t,e,i,r){var o,n,a,h,l,c,d,_,m,g=[],b=[],v=0,Z=r.length;for(n=0;n<Z;n++)g.push(!1);for(n=0;n<t.length;n++)b.push(s.Z.C.abs(t[n])*(4*n+1));for(o=0;o<i&&v<Z;o++)for(n=0;n<Z;n++)if(!g[n])if(c=u(t,r[n]),l=s.Z.C.abs(r[n]),s.Z.C.abs(c)<e*f(b,l)){if(g[n]=!0,++v===Z)break}else{for(l>1?(h=s.Z.C.div(1,r[n]),(m=p(t,h,!0)).div(p(t,h)),m.mult(h),c=s.Z.C.sub(Z,m),c=s.Z.C.div(r[n],c)):c.div(u(t,r[n],!0)),d=new s.Z.Complex(0),a=0;a<Z;a++)a!==n&&(_=s.Z.C.sub(r[n],r[a]),_=s.Z.C.div(1,_),d.add(_));d.mult(c),d=s.Z.C.sub(1,d),c.div(d),r[n].sub(c)}return o}(l,i,r,d),(d=c.concat(d)).sort((function(t,e){return t.real<e.real?-1:t.real>e.real?1:0})),d},RamerDouglasPeucker:function(t,e){var i,s,r,o=[],a=[],h=function(t,e,i,s,r){var o=function(t,e,i){var s,r,o,a,h,l,c,d,u,p,f,_=n.Z.eps*n.Z.eps,m=1e4,g=0,b=e;if(i-e<2)return[-1,0];if(o=t[e].scrCoords,a=t[i].scrCoords,isNaN(o[1])||isNaN(o[2]))return[NaN,e];if(isNaN(a[1])||isNaN(a[2]))return[NaN,i];for(r=e+1;r<i;r++){if(h=t[r].scrCoords,isNaN(h[1])||isNaN(h[2]))return[NaN,r];l=(l=(l=h[1]-o[1])==1/0?m:l)===-1/0?-m:l,c=(c=(c=h[2]-o[2])==1/0?m:c)===-1/0?-m:c,(p=(d=(d=(d=a[1]-o[1])==1/0?m:d)===-1/0?-m:d)*d+(u=(u=(u=a[2]-o[2])==1/0?m:u)===-1/0?-m:u)*u)>_?((f=(l*d+c*u)/p)<0?f=0:f>1&&(f=1),s=(l-=f*d)*l+(c-=f*u)*c):(f=0,s=l*l+c*c),s>g&&(g=s,b=r)}return[Math.sqrt(g),b]}(t,e,i),a=o[1];if(isNaN(o[0])){h(t,e,a-1,s,r),r.push(t[a]);do{++a}while(a<=i&&isNaN(t[a].scrCoords[1]+t[a].scrCoords[2]));a<=i&&r.push(t[a]),h(t,a+1,i,s,r)}else o[0]>s?(h(t,e,a,s,r),h(t,a,i,s,r)):r.push(t[i])};for(r=t.length,i=0;;){for(;i<r&&isNaN(t[i].scrCoords[1]+t[i].scrCoords[2]);)i+=1;for(s=i+1;s<r&&!isNaN(t[s].scrCoords[1]+t[s].scrCoords[2]);)s+=1;if(s--,i<r&&s>i&&((a=[])[0]=t[i],h(t,i,s,e,a),o=o.concat(a)),i>=r)break;s<r-1&&o.push(t[s+1]),i=s+1}return o},RamerDouglasPeuker:function(t,e){return s.Z.deprecated("Numerics.RamerDouglasPeuker()","Numerics.RamerDouglasPeucker()"),this.RamerDouglasPeucker(t,e)},Visvalingam:function(t,e){var i,r,o,n,a,h,l,c,d,u=[],p=[],f=[];if((r=t.length)<=2)return t;for(u[0]={used:!0,lft:null,node:null},a=0,i=1;i<r-1;i++)o=Math.abs(s.Z.Math.Numerics.det([t[i-1].usrCoords,t[i].usrCoords,t[i+1].usrCoords])),isNaN(o)||(d={v:o,idx:i},p.push(d),u[i]={used:!0,lft:a,node:d},u[a].rt=i,a=i);for(u[r-1]={used:!0,rt:null,lft:a,node:null},u[a].rt=r-1,n=-1/0;p.length>e;)p.sort((function(t,e){return e.v-t.v})),u[i=p.pop().idx].used=!1,n=u[i].node.v,a=u[i].lft,h=u[i].rt,u[a].rt=h,u[h].lft=a,null!==(l=u[a].lft)&&(o=Math.abs(s.Z.Math.Numerics.det([t[l].usrCoords,t[a].usrCoords,t[h].usrCoords])),u[a].node.v=o>=n?o:n),null!==(c=u[h].rt)&&(o=Math.abs(s.Z.Math.Numerics.det([t[a].usrCoords,t[h].usrCoords,t[c].usrCoords])),u[h].node.v=o>=n?o:n);f=[t[i=0]];do{i=u[i].rt,f.push(t[i])}while(null!==u[i].rt);return f}},e.Z=n.Z.Numerics},309:function(t,e,i){var s=i(765),r=i(275),o=i(109);r.Z.Statistics={sum:function(t){var e,i=t.length,s=0;for(e=0;e<i;e++)s+=t[e];return s},prod:function(t){var e,i=t.length,s=1;for(e=0;e<i;e++)s*=t[e];return s},mean:function(t){return t.length>0?this.sum(t)/t.length:0},median:function(t){var e,i;return t.length>0?(ArrayBuffer.isView(t)?(e=new Float64Array(t)).sort():(e=t.slice(0)).sort((function(t,e){return t-e})),1&(i=e.length)?e[parseInt(.5*i,10)]:.5*(e[.5*i-1]+e[.5*i])):0},percentile:function(t,e){var i,s,r,n,a,h=[];if(t.length>0){for(ArrayBuffer.isView(t)?(i=new Float64Array(t)).sort():(i=t.slice(0)).sort((function(t,e){return t-e})),s=i.length,n=o.Z.isArray(e)?e:[e],r=0;r<n.length;r++)a=s*n[r]*.01,parseInt(a,10)===a?h.push(.5*(i[a-1]+i[a])):h.push(i[parseInt(a,10)]);return o.Z.isArray(e)?h:h[0]}return 0},variance:function(t){var e,i,s,r=t.length;if(r>1){for(e=this.mean(t),i=0,s=0;s<r;s++)i+=(t[s]-e)*(t[s]-e);return i/(t.length-1)}return 0},sd:function(t){return Math.sqrt(this.variance(t))},weightedMean:function(t,e){if(t.length!==e.length)throw new Error("JSXGraph error (Math.Statistics.weightedMean): Array dimension mismatch.");return t.length>0?this.mean(this.multiply(t,e)):0},max:function(t){return Math.max.apply(this,t)},min:function(t){return Math.min.apply(this,t)},range:function(t){return[this.min(t),this.max(t)]},abs:function(t){var e,i,s;if(o.Z.isArray(t))if(t.map)s=t.map(Math.abs);else for(i=t.length,s=[],e=0;e<i;e++)s[e]=Math.abs(t[e]);else s=ArrayBuffer.isView(t)?t.map(Math.abs):Math.abs(t);return s},add:function(t,e){var i,s,r=[];if(t=o.Z.evalSlider(t),e=o.Z.evalSlider(e),o.Z.isArray(t)&&o.Z.isNumber(e))for(s=t.length,i=0;i<s;i++)r[i]=t[i]+e;else if(o.Z.isNumber(t)&&o.Z.isArray(e))for(s=e.length,i=0;i<s;i++)r[i]=t+e[i];else if(o.Z.isArray(t)&&o.Z.isArray(e))for(s=Math.min(t.length,e.length),i=0;i<s;i++)r[i]=t[i]+e[i];else r=t+e;return r},div:function(t,e){var i,s,r=[];if(t=o.Z.evalSlider(t),e=o.Z.evalSlider(e),o.Z.isArray(t)&&o.Z.isNumber(e))for(s=t.length,i=0;i<s;i++)r[i]=t[i]/e;else if(o.Z.isNumber(t)&&o.Z.isArray(e))for(s=e.length,i=0;i<s;i++)r[i]=t/e[i];else if(o.Z.isArray(t)&&o.Z.isArray(e))for(s=Math.min(t.length,e.length),i=0;i<s;i++)r[i]=t[i]/e[i];else r=t/e;return r},divide:function(){s.Z.deprecated("Statistics.divide()","Statistics.div()"),r.Z.Statistics.div.apply(r.Z.Statistics,arguments)},mod:function(t,e,i){var s,n,a=[],h=function(t,e){return t%e};if((i=o.Z.def(i,!1))&&(h=r.Z.mod),t=o.Z.evalSlider(t),e=o.Z.evalSlider(e),o.Z.isArray(t)&&o.Z.isNumber(e))for(n=t.length,s=0;s<n;s++)a[s]=h(t[s],e);else if(o.Z.isNumber(t)&&o.Z.isArray(e))for(n=e.length,s=0;s<n;s++)a[s]=h(t,e[s]);else if(o.Z.isArray(t)&&o.Z.isArray(e))for(n=Math.min(t.length,e.length),s=0;s<n;s++)a[s]=h(t[s],e[s]);else a=h(t,e);return a},multiply:function(t,e){var i,s,r=[];if(t=o.Z.evalSlider(t),e=o.Z.evalSlider(e),o.Z.isArray(t)&&o.Z.isNumber(e))for(s=t.length,i=0;i<s;i++)r[i]=t[i]*e;else if(o.Z.isNumber(t)&&o.Z.isArray(e))for(s=e.length,i=0;i<s;i++)r[i]=t*e[i];else if(o.Z.isArray(t)&&o.Z.isArray(e))for(s=Math.min(t.length,e.length),i=0;i<s;i++)r[i]=t[i]*e[i];else r=t*e;return r},subtract:function(t,e){var i,s,r=[];if(t=o.Z.evalSlider(t),e=o.Z.evalSlider(e),o.Z.isArray(t)&&o.Z.isNumber(e))for(s=t.length,i=0;i<s;i++)r[i]=t[i]-e;else if(o.Z.isNumber(t)&&o.Z.isArray(e))for(s=e.length,i=0;i<s;i++)r[i]=t-e[i];else if(o.Z.isArray(t)&&o.Z.isArray(e))for(s=Math.min(t.length,e.length),i=0;i<s;i++)r[i]=t[i]-e[i];else r=t-e;return r},TheilSenRegression:function(t){var e,i,s=[],o=[],n=[];for(e=0;e<t.length;e++){for(o.length=0,i=0;i<t.length;i++)Math.abs(t[i].usrCoords[1]-t[e].usrCoords[1])>r.Z.eps&&(o[i]=(t[i].usrCoords[2]-t[e].usrCoords[2])/(t[i].usrCoords[1]-t[e].usrCoords[1]));s[e]=this.median(o),n.push(t[e].usrCoords[2]-s[e]*t[e].usrCoords[1])}return[this.median(n),this.median(s),-1]},generateGaussian:function(t,e){var i,s,r;if(this.hasSpare)return this.hasSpare=!1,this.spare*e+t;do{r=(i=2*Math.random()-1)*i+(s=2*Math.random()-1)*s}while(r>=1||0===r);return r=Math.sqrt(-2*Math.log(r)/r),this.spare=s*r,this.hasSpare=!0,t+e*i*r},randomNormal:function(t,e){return this.generateGaussian(t,e)},randomUniform:function(t,e){return Math.random()*(e-t)+t},randomExponential:function(t){var e;if(t<=0)return NaN;do{e=Math.random()}while(0===e);return-Math.log(e)/t},randomGamma:function(t,e,i){var s,r,o,n,a,h;if(t<=0)return NaN;if(e=e||1,i=i||0,1===t)return e*this.randomExponential(1)+i;if(t<1){a=Math.E/(t+Math.E);do{s=Math.random();do{r=Math.random()}while(0===r);s<a?(o=Math.pow(r,1/t),h=Math.exp(-o)):(o=1-Math.log(r),h=Math.pow(o,t-1)),s=Math.random()}while(s>=h);return e*o+i}do{n=Math.tan(Math.PI*Math.random()),(o=Math.sqrt(2*t-1)*n+t-1)>0&&(r=Math.random())}while(o<=0||r>(1+n*n)*Math.exp((t-1)*Math.log(o/(t-1))-Math.sqrt(2*t-1)*n));return e*o+i},randomBeta:function(t,e){var i;return t<=0||e<=0?NaN:(i=this.randomGamma(t))/(i+this.randomGamma(e))},randomChisquare:function(t){return t<=0?NaN:2*this.randomGamma(.5*t)},randomF:function(t,e){return t<=0||e<=0?NaN:this.randomChisquare(t)*e/(this.randomChisquare(e)*t)},randomT:function(t){var e,i;return t<=0?NaN:(e=this.randomNormal(0,1),i=this.randomChisquare(t),e/Math.sqrt(i/t))},randomBinomial:function(t,e){var i,s,r,o,n;if(e<0||e>1||t<0)return NaN;if(0===e)return 0;if(1===e)return t;if(0===t)return 0;if(1===t)return Math.random()<e?1:0;if(e>.5)return t-this.randomBinomial(t,1-e);if(t<100){if(i=-1,s=0,0===(r=Math.log(1-e)))return 0;do{i+=1,s+=Math.floor(Math.log(Math.random())/r)+1}while(s<t)}else n=t-(o=1+Math.floor(.5*t))+1,i=(i=this.randomBeta(o,n))>=e?this.randomBinomial(o-1,e/i):o+this.randomBinomial(n-1,(e-i)/(1-i));return i},randomGeometric:function(t){var e;return t<0||t>1?NaN:(e=Math.random(),Math.ceil(Math.log(e)/Math.log(1-t)))},randomPoisson:function(t){var e,i,s=Math.exp(-t),r=0,o=1;if(t<=0)return NaN;if(t<10){do{o*=Math.random(),r+=1}while(o>s);e=r-1}else r=Math.floor(7/8*t),e=(i=this.randomGamma(r))<t?r+this.randomPoisson(t-i):this.randomBinomial(r-1,t/i);return e},randomPareto:function(t,e){var i=Math.random();return t<=0||e<=0?NaN:e*Math.pow(1-i,-1/t)},randomHypergeometric:function(t,e,i){var s,r,o,n=t+e-i,a=Math.min(t,e),h=a;if(t<1||e<1||i>t+e)return NaN;for(s=i;h*s>0;)r=Math.random(),h-=Math.floor(r+h/(n+s)),s-=1;return o=a-h,t<=e?o:i-o},histogram:function(t,e){var i,r,o,n,a,h,l,c,d,u=[],p=[],f=0;for(h=e.bins||10,!1===(c=e.range||!1)?(n=Math.min.apply(null,t),a=Math.max.apply(null,t)):(n=c[0],a=c[1]),l=h>0?(a-n)/(h-1):0,i=0;i<h;i++)u.push(0),p.push(n+i*l);for(r=t.length,i=0;i<r;i++)(o=Math.floor((t[i]-n)/l))>=0&&o<h?u[o]+=1:f+=1;if(e.density)for(d=s.Z.Math.Statistics.sum(u)+f,i=0;i<h;i++)u[i]/=d*l;if(e.cumulative){if(e.density)for(i=0;i<h;i++)u[i]*=l;for(i=1;i<h;i++)u[i]+=u[i-1]}return[u,p]}},e.Z=r.Z.Statistics},766:function(t,e,i){var s=i(765),r=i(351),o=i(275),n=i(327),a=i(109);s.Z.Options={jc:{enabled:!0,compile:!0},board:{animationDelay:35,axis:!1,boundingBox:[-5,5,5,-5],browserPan:!1,clickDelay:600,dblClickSuppressClick:!1,defaultAxes:{x:{name:"x",fixed:!0,needsRegularUpdate:!1,ticks:{label:{visible:"inherit",anchorX:"middle",anchorY:"top",fontSize:12,offset:[0,-3]},tickEndings:[0,1],majorTickEndings:[1,1],drawZero:!1,visible:"inherit"}},y:{name:"y",fixed:!0,needsRegularUpdate:!1,ticks:{label:{visible:"inherit",anchorX:"right",anchorY:"middle",fontSize:12,offset:[-6,0]},tickEndings:[1,0],majorTickEndings:[1,1],drawZero:!1,visible:"inherit"}}},document:!1,drag:{enabled:!0},fullscreen:{symbol:'<svg height="1em" width="1em" version="1.1" viewBox="10 10 18 18"><path fill="#666" d="m 10,16 2,0 0,-4 4,0 0,-2 L 10,10 l 0,6 0,0 z"></path><path fill="#666" d="m 20,10 0,2 4,0 0,4 2,0 L 26,10 l -6,0 0,0 z"></path><path fill="#666" d="m 24,24 -4,0 0,2 L 26,26 l 0,-6 -2,0 0,4 0,0 z"></path><path fill="#666" d="M 12,20 10,20 10,26 l 6,0 0,-2 -4,0 0,-4 0,0 z"></path></svg>',scale:.85,id:null},ignoreLabels:!0,intl:{enabled:!1},keepAspectRatio:!1,keyboard:{enabled:!0,dx:10,dy:10,panShift:!0,panCtrl:!1},logging:{enabled:!1},minimizeReflow:"none",maxBoundingBox:[-1/0,1/0,1/0,-1/0],maxFrameRate:40,maxNameLength:1,moveTarget:null,offsetX:0,offsetY:0,pan:{enabled:!0,needShift:!0,needTwoFingers:!1},registerEvents:!0,renderer:"auto",resize:{enabled:!0,throttle:10},screenshot:{scale:1,type:"png",symbol:"⌘",css:"background-color:#eeeeee; opacity:1.0; border:2px solid black; border-radius:10px; text-align:center",cssButton:"padding: 4px 10px; border: solid #356AA0 1px; border-radius: 5px; position: absolute; right: 2ex; top: 2ex; background-color: rgba(255, 255, 255, 0.3);"},selection:{enabled:!1,name:"selectionPolygon",needShift:!1,needCtrl:!0,fillColor:"#ffff00",visible:!1,withLines:!1,vertices:{visible:!1}},showClearTraces:!1,showCopyright:!0,showFullscreen:!1,showInfobox:!0,showLogo:!1,showNavigation:!0,showReload:!1,showScreenshot:!1,showZoom:!0,takeFirst:!1,takeSizeFromFile:!1,theme:"default",title:"",zoom:{enabled:!0,factorX:1.25,factorY:1.25,wheel:!0,needShift:!0,center:"auto",min:1e-4,max:1e4,pinch:!0,pinchHorizontal:!0,pinchVertical:!0,pinchSensitivity:7},zoomX:1,zoomY:1},navbar:{strokeColor:"#333333",fillColor:"transparent",highlightFillColor:"#aaaaaa",padding:"2px",position:"absolute",fontSize:"14px",cursor:"pointer",zIndex:"100",right:"5px",bottom:"5px"},elements:{aria:{enabled:!1,label:"",live:"assertive"},cssClass:"",highlightCssClass:"",dash:0,dashScale:!1,draft:{draft:!1,strokeColor:"#565656",fillColor:"#565656",strokeOpacity:.8,fillOpacity:.8,strokeWidth:1},dragToTopOfLayer:!1,element3D:null,fillColor:n.Z.palette.red,fillOpacity:1,fixed:!1,frozen:!1,gradient:null,gradientAngle:0,gradientCX:.5,gradientCY:.5,gradientEndOffset:1,gradientFX:.5,gradientFY:.5,gradientFR:0,gradientR:.5,gradientSecondColor:"#ffffff",gradientSecondOpacity:1,gradientStartOffset:0,highlight:!0,highlightFillColor:"none",highlightFillOpacity:1,highlightStrokeColor:"#c3d9ff",highlightStrokeOpacity:1,highlightStrokeWidth:2,isLabel:!1,layer:0,lineCap:"butt",needsRegularUpdate:!0,nonnegativeOnly:!1,precision:"inherit",priv:!1,rotatable:!0,scalable:!0,shadow:{enabled:!1,color:[0,0,0],opacity:1,blur:3,blend:.1,offset:[5,5]},snapToGrid:!1,strokeColor:n.Z.palette.blue,strokeOpacity:1,strokeWidth:2,tabindex:-1,trace:!1,traceAttributes:{},transitionDuration:100,transitionProperties:["fill","fill-opacity","stroke","stroke-opacity","stroke-width"],visible:!0,withLabel:!1,ignoreForLabelAutoposition:!1},ticks:{generateLabelText:null,generateLabelValue:null,drawLabels:!1,label:{tabindex:null,layer:7,highlight:!1,autoPosition:!1},beautifulScientificTickLabels:!1,useUnicodeMinus:!0,anchor:"left",drawZero:!1,insertTicks:!1,minTicksDistance:10,minorHeight:4,majorHeight:10,tickEndings:[1,1],majorTickEndings:[1,1],ignoreInfiniteTickEndings:!0,minorTicks:4,ticksPerLabel:!1,scale:1,scaleSymbol:"",labels:[],maxLabelLength:5,precision:3,digits:3,ticksDistance:1,face:"|",strokeOpacity:1,strokeWidth:1,strokeColor:"#000000",highlightStrokeColor:"#888888",fillColor:"none",highlightFillColor:"none",visible:"inherit",includeBoundaries:!1,type:"linear",intl:{enabled:"inherit",options:{}},minorTicksInArrow:!1,majorTicksInArrow:!0,labelInArrow:!0,minorTicksInMargin:!1,majorTicksInMargin:!0,labelInMargin:!0,ignoreForLabelAutoposition:!0},hatch:{drawLabels:!1,drawZero:!0,majorHeight:20,anchor:"middle",face:"|",strokeWidth:2,strokeColor:n.Z.palette.blue,ticksDistance:.2},precision:{touch:30,touchMax:100,mouse:4,pen:4,epsilon:1e-4,hasPoint:4},layer:{numlayers:20,unused9:19,unused8:18,unused7:17,unused6:16,unused5:15,unused4:14,unused3:13,unused2:12,unused1:11,unused0:10,text:9,point:9,glider:9,arc:8,line:7,circle:6,curve:5,turtle:5,polygon:3,sector:3,angle:3,integral:3,axis:2,ticks:2,grid:1,image:0,trace:0},angle:{withLabel:!0,radius:"auto",type:"sector",orthoType:"square",orthoSensitivity:1,fillColor:n.Z.palette.orange,highlightFillColor:n.Z.palette.orange,strokeColor:n.Z.palette.orange,fillOpacity:.3,highlightFillOpacity:.3,radiuspoint:{withLabel:!1,visible:!1,name:""},pointsquare:{withLabel:!1,visible:!1,name:""},dot:{visible:!1,strokeColor:"none",fillColor:"#000000",size:2,face:"o",withLabel:!1,name:""},label:{position:"top",offset:[0,0],strokeColor:n.Z.palette.blue},arc:{visible:!1,fillColor:"none"}},arc:{selection:"auto",hasInnerPoints:!1,label:{anchorX:"auto",anchorY:"auto"},firstArrow:!1,lastArrow:!1,fillColor:"none",highlightFillColor:"none",strokeColor:n.Z.palette.blue,highlightStrokeColor:"#c3d9ff",useDirection:!1,center:{},radiusPoint:{},anglePoint:{}},arrow:{firstArrow:!1,lastArrow:{type:1,highlightSize:6,size:6}},arrowparallel:{firstArrow:!1,lastArrow:{type:1,highlightSize:6,size:6}},axis:{name:"",needsRegularUpdate:!1,strokeWidth:1,lastArrow:{type:1,highlightSize:8,size:8},strokeColor:"#666666",highlightStrokeWidth:1,highlightStrokeColor:"#888888",position:"static",anchor:"",anchorDist:"10%",ticksAutoPos:!1,ticksAutoPosThreshold:"5%",withTicks:!0,straightFirst:!0,straightLast:!0,margin:-4,withLabel:!1,scalable:!1,ticks:{label:{offset:[4,-9],parse:!1,needsRegularUpdate:!1,display:"internal",visible:"inherit",layer:9},visible:"inherit",needsRegularUpdate:!1,strokeWidth:1,strokeColor:"#666666",highlightStrokeColor:"#888888",drawLabels:!0,drawZero:!1,insertTicks:!0,minTicksDistance:5,minorHeight:10,majorHeight:-1,tickEndings:[0,1],majorTickEndings:[1,1],minorTicks:4,ticksDistance:1,strokeOpacity:.25},point1:{needsRegularUpdate:!1,visible:!1},point2:{needsRegularUpdate:!1,visible:!1},tabindex:-1,label:{position:"lft",offset:[10,10]},ignoreForLabelAutoposition:!0},bisector:{strokeColor:"#000000",point:{visible:!1,fixed:!1,withLabel:!1,name:""}},bisectorlines:{line1:{strokeColor:"#000000"},line2:{strokeColor:"#000000"}},boxplot:{dir:"vertical",smallWidth:.5,strokeWidth:2,strokeColor:n.Z.palette.blue,fillColor:n.Z.palette.blue,fillOpacity:.2,highlightStrokeWidth:2,highlightStrokeColor:n.Z.palette.blue,highlightFillColor:n.Z.palette.blue,highlightFillOpacity:.1},button:{disabled:!1,display:"html"},cardinalspline:{createPoints:!0,isArrayOfCoordinates:!0,points:{strokeOpacity:.05,fillOpacity:.05,highlightStrokeOpacity:1,highlightFillOpacity:1,withLabel:!1,name:"",fixed:!1}},chart:{chartStyle:"line",colors:["#B02B2C","#3F4C6B","#C79810","#D15600","#FFFF88","#c3d9ff","#4096EE","#008C00"],highlightcolors:null,fillcolor:null,highlightonsector:!1,highlightbysize:!1,fillOpacity:.6,withLines:!1,label:{}},checkbox:{disabled:!1,checked:!1,display:"html"},circle:{hasInnerPoints:!1,fillColor:"none",highlightFillColor:"none",strokeColor:n.Z.palette.blue,highlightStrokeColor:"#c3d9ff",center:{visible:!1,withLabel:!1,fixed:!1,fillColor:n.Z.palette.red,strokeColor:n.Z.palette.red,highlightFillColor:"#c3d9ff",highlightStrokeColor:"#c3d9ff",layer:9,name:""},point2:{fillColor:n.Z.palette.red,strokeColor:n.Z.palette.red,highlightFillColor:"#c3d9ff",highlightStrokeColor:"#c3d9ff",layer:9,visible:!1,withLabel:!1,fixed:!1,name:""},label:{position:"urt"}},circumcircle:{fillColor:"none",highlightFillColor:"none",strokeColor:n.Z.palette.blue,highlightStrokeColor:"#c3d9ff",center:{visible:!1,fixed:!1,withLabel:!1,fillColor:n.Z.palette.red,strokeColor:n.Z.palette.red,highlightFillColor:"#c3d9ff",highlightStrokeColor:"#c3d9ff",name:""}},circumcirclearc:{fillColor:"none",highlightFillColor:"none",strokeColor:n.Z.palette.blue,highlightStrokeColor:"#c3d9ff",useDirection:!0,center:{visible:!1,withLabel:!1,fixed:!1,name:""}},circumcirclesector:{useDirection:!0,fillColor:n.Z.palette.yellow,highlightFillColor:n.Z.palette.yellow,fillOpacity:.3,highlightFillOpacity:.3,strokeColor:n.Z.palette.blue,highlightStrokeColor:"#c3d9ff",point:{visible:!1,fixed:!1,withLabel:!1,name:""}},comb:{frequency:.2,width:.4,angle:Math.PI/3,reverse:!1,point1:{visible:!1,withLabel:!1,fixed:!1,name:""},point2:{visible:!1,withLabel:!1,fixed:!1,name:""},strokeWidth:1,strokeColor:"#0000ff",fillColor:"none"},conic:{fillColor:"none",highlightFillColor:"none",strokeColor:n.Z.palette.blue,highlightStrokeColor:"#c3d9ff",foci:{fixed:!1,visible:!1,withLabel:!1,name:""},center:{visible:!1,withLabel:!1,name:""},point:{withLabel:!1,name:""},line:{visible:!1}},curve:{strokeWidth:1,strokeColor:n.Z.palette.blue,fillColor:"none",fixed:!0,curveType:null,doAdvancedPlot:!0,doAdvancedPlotOld:!1,firstArrow:!1,handDrawing:!1,label:{position:"lft"},lastArrow:!1,lineCap:"round",numberPointsHigh:1600,numberPointsLow:400,plotVersion:2,recursionDepthHigh:17,recursionDepthLow:15},foreignobject:{fixed:!0,visible:!0,needsRegularUpdate:!1,attractors:[],evaluateOnlyOnce:!1},functiongraph:{},glider:{label:{}},grid:{needsRegularUpdate:!1,hasGrid:!1,highlight:!1,gridX:null,gridY:null,majorStep:"auto",minorElements:0,forceSquare:!1,includeBoundaries:!1,size:void 0,face:void 0,margin:void 0,drawZero:void 0,polygonVertices:void 0,major:{size:5,face:"line",margin:0,drawZero:!0,polygonVertices:6},minor:{visible:"inherit",size:3,face:"point",margin:0,drawZero:!0,polygonVertices:6},snapToGrid:!1,strokeColor:"#c0c0c0",strokeWidth:1,strokeOpacity:.5,dash:0,theme:0,themes:[{},{forceSquare:"min",major:{face:"line"}},{major:{face:"line"},minor:{size:3,face:"point"},minorElements:"auto"},{major:{face:"line"},minor:{face:"line",strokeOpacity:.25},minorElements:"auto"},{major:{face:"line"},minor:{face:"+",size:"95%"},minorElements:"auto"},{major:{face:"+",size:10,strokeOpacity:1},minor:{face:"point",size:3},minorElements:"auto"},{major:{face:"circle",size:8,fillColor:"#c0c0c0"},minor:{face:"point",size:3},minorElements:"auto"}]},group:{needsRegularUpdate:!0},htmlslider:{widthRange:100,widthOut:34,step:.01,frozen:!0,isLabel:!1,strokeColor:"#000000",display:"html",anchorX:"left",anchorY:"middle",withLabel:!1},image:{imageString:null,fillOpacity:1,highlightFillOpacity:.6,cssClass:"JXGimage",highlightCssClass:"JXGimageHighlight",rotate:0,snapSizeX:1,snapSizeY:1,attractors:[]},implicitcurve:{margin:1,resolution_outer:5,resolution_inner:5,max_steps:1024,alpha_0:.05,tol_u0:o.Z.eps,tol_newton:1e-7,tol_cusp:.05,tol_progress:1e-4,qdt_box:.2,kappa_0:.2,delta_0:.05,h_initial:.1,h_critical:.001,h_max:.5,loop_dist:.09,loop_dir:.99,loop_detection:!0},incircle:{fillColor:"none",highlightFillColor:"none",strokeColor:n.Z.palette.blue,highlightStrokeColor:"#c3d9ff",center:{visible:!1,fixed:!1,withLabel:!1,fillColor:n.Z.palette.red,strokeColor:n.Z.palette.red,highlightFillColor:"#c3d9ff",highlightStrokeColor:"#c3d9ff",name:""}},inequality:{fillColor:n.Z.palette.red,fillOpacity:.2,strokeColor:"none",inverse:!1},infobox:{distanceX:-20,distanceY:25,intl:{enabled:"inherit",options:{}},fontSize:12,isLabel:!1,strokeColor:"#bbbbbb",display:"html",anchorX:"left",anchorY:"middle",cssClass:"JXGinfobox",rotate:0,visible:!0,parse:!1,transitionDuration:0,needsRegularUpdate:!1,tabindex:null,viewport:[0,0,0,0],ignoreForLabelAutoposition:!0},integral:{axis:"x",withLabel:!0,fixed:!0,strokeWidth:0,strokeOpacity:0,fillColor:n.Z.palette.red,fillOpacity:.3,highlightFillColor:n.Z.palette.red,highlightFillOpacity:.2,curveLeft:{visible:!0,withLabel:!1,color:n.Z.palette.red,fillOpacity:.8,layer:9},baseLeft:{visible:!1,fixed:!1,withLabel:!1,name:""},curveRight:{visible:!0,withLabel:!1,color:n.Z.palette.red,fillOpacity:.8,layer:9},baseRight:{visible:!1,fixed:!1,withLabel:!1,name:""},label:{fontSize:20,digits:4,intl:{enabled:!1,options:{}}}},input:{disabled:!1,maxlength:524288,display:"html"},intersection:{alwaysIntersect:!0},label:{visible:"inherit",strokeColor:"#000000",strokeOpacity:1,highlightStrokeOpacity:.666666,highlightStrokeColor:"#000000",fixed:!0,tabindex:null,position:"urt",distance:1.5,offset:[10,10],autoPosition:!1,autoPositionMinDistance:12,autoPositionMaxDistance:28,autoPositionWhitelist:[]},legend:{style:"vertical",labels:["1","2","3","4","5","6","7","8"],colors:["#B02B2C","#3F4C6B","#C79810","#D15600","#FFFF88","#c3d9ff","#4096EE","#008C00"],lineLength:1,strokeOpacity:[1],rowHeight:20,strokeWidth:5,frozen:!1},line:{firstArrow:!1,lastArrow:!1,margin:0,straightFirst:!0,straightLast:!0,fillColor:"none",highlightFillColor:"none",strokeColor:n.Z.palette.blue,highlightStrokeColor:"#c3d9ff",withTicks:!1,point1:{fillColor:n.Z.palette.red,strokeColor:n.Z.palette.red,highlightFillColor:"#c3d9ff",highlightStrokeColor:"#c3d9ff",layer:9,visible:!1,withLabel:!1,fixed:!1,name:""},point2:{fillColor:n.Z.palette.red,strokeColor:n.Z.palette.red,highlightFillColor:"#c3d9ff",highlightStrokeColor:"#c3d9ff",layer:9,visible:!1,withLabel:!1,fixed:!1,name:""},ticks:{drawLabels:!0,label:{offset:[4,-9]},drawZero:!1,insertTicks:!1,ticksDistance:1,minTicksDistance:50,minorHeight:4,majorHeight:-1,minorTicks:4,strokeOpacity:.3,visible:"inherit"},label:{position:"llft"},snapToGrid:!1,snapSizeX:1,snapSizeY:1,touchFirstPoint:!1,touchLastPoint:!1},locus:{translateToOrigin:!1,translateTo10:!1,stretch:!1,toOrigin:null,to10:null},measurement:{baseUnit:"",units:{},showPrefix:!0,showSuffix:!0,prefix:"",suffix:"",dim:null,formatCoords:function(t,e,i,s){return 1!==parseFloat(s)?"Infinit coords":"("+e+", "+i+")"},formatDirection:function(t,e,i){return"("+e+", "+i+")"}},metapostspline:{createPoints:!0,isArrayOfCoordinates:!0,points:{strokeOpacity:.5,fillOpacity:.5,highlightStrokeOpacity:1,highlightFillOpacity:1,withLabel:!1,name:"",fixed:!1}},mirrorelement:{fixed:!0,point:{},center:{},type:"Euclidean"},nonreflexangle:{},normal:{strokeColor:"#000000",point:{visible:!1,fixed:!1,withLabel:!1,name:""}},orthogonalprojection:{},otherintersection:{alwaysIntersect:!0,precision:.001},parallel:{strokeColor:"#000000",point:{visible:!1,fixed:!1,withLabel:!1,name:""},label:{position:"llft"}},parallelogram:{parallelpoint:{withLabel:!1,name:""}},parallelpoint:{},perpendicular:{strokeColor:"#000000",straightFirst:!0,straightLast:!0},perpendicularsegment:{strokeColor:"#000000",straightFirst:!1,straightLast:!1,point:{visible:!1,fixed:!0,withLabel:!1,name:""}},point:{withLabel:!0,label:{},style:5,face:"o",size:3,sizeUnit:"screen",strokeWidth:2,transitionProperties:["fill","fill-opacity","stroke","stroke-opacity","stroke-width","width","height","rx","ry"],fillColor:n.Z.palette.red,strokeColor:n.Z.palette.red,highlightFillColor:"#c3d9ff",highlightStrokeColor:"#c3d9ff",zoom:!1,showInfobox:"inherit",infoboxDigits:"auto",draft:!1,attractors:[],attractorUnit:"user",attractorDistance:0,snatchDistance:0,snapToGrid:!1,attractToGrid:!1,snapSizeX:1,snapSizeY:1,snapToPoints:!1,ignoredSnapToPoints:[]},polygon:{hasInnerPoints:!1,fillColor:n.Z.palette.yellow,highlightFillColor:n.Z.palette.yellow,fillOpacity:.3,highlightFillOpacity:.2,withLines:!0,borders:{withLabel:!1,strokeWidth:1,highlightStrokeWidth:1,layer:5,label:{position:"top"},visible:"inherit"},highlightByStrokeWidth:!1,vertices:{layer:9,withLabel:!1,name:"",strokeColor:n.Z.palette.red,fillColor:n.Z.palette.red,fixed:!1,visible:"inherit"},label:{offset:[0,0]}},polygonalchain:{fillColor:"none",highlightFillColor:"none"},prescribedangle:{anglePoint:{size:2,visible:!1,withLabel:!1}},reflection:{fixed:!0,center:{},type:"Euclidean"},reflexangle:{},regularpolygon:{hasInnerPoints:!1,fillColor:n.Z.palette.yellow,highlightFillColor:n.Z.palette.yellow,fillOpacity:.3,highlightFillOpacity:.2,withLines:!0,borders:{withLabel:!1,strokeWidth:1,highlightStrokeWidth:1,layer:5,label:{position:"top"}},vertices:{layer:9,withLabel:!0,strokeColor:n.Z.palette.red,fillColor:n.Z.palette.red,fixed:!1},label:{offset:[0,0]}},riemannsum:{withLabel:!1,fillOpacity:.3,fillColor:n.Z.palette.yellow},sector:{fillColor:n.Z.palette.yellow,highlightFillColor:n.Z.palette.yellow,fillOpacity:.3,highlightFillOpacity:.3,highlightOnSector:!1,highlightStrokeWidth:0,useDirection:!1,selection:"auto",arc:{visible:!1,fillColor:"none",withLabel:!1,name:"",center:{visible:!1,withLabel:!1,name:""},radiusPoint:{visible:!1,withLabel:!1,name:""},anglePoint:{visible:!1,withLabel:!1,name:""}},radiusPoint:{visible:!1,withLabel:!1},center:{visible:!1,withLabel:!1},anglePoint:{visible:!1,withLabel:!1},label:{offset:[0,0],anchorX:"auto",anchorY:"auto"}},segment:{label:{position:"top"}},semicircle:{center:{visible:!1,withLabel:!1,fixed:!1,fillColor:n.Z.palette.red,strokeColor:n.Z.palette.red,highlightFillColor:"#eeeeee",highlightStrokeColor:n.Z.palette.red,name:""}},slider:{snapWidth:-1,snapValues:[],snapValueDistance:0,precision:2,digits:2,intl:{enabled:"inherit",options:{}},firstArrow:!1,lastArrow:!1,withTicks:!0,withLabel:!0,suffixLabel:null,unitLabel:null,postLabel:null,layer:9,showInfobox:!1,name:"",visible:!0,strokeColor:"#000000",highlightStrokeColor:"#888888",fillColor:"#ffffff",highlightFillColor:"none",size:6,point1:{needsRegularUpdate:!1,showInfobox:!1,withLabel:!1,visible:!1,fixed:!0,frozen:"inherit",name:""},point2:{needsRegularUpdate:!1,showInfobox:!1,withLabel:!1,visible:!1,fixed:!0,frozen:"inherit",name:""},baseline:{needsRegularUpdate:!1,visible:"inherit",fixed:!0,scalable:!1,tabindex:null,name:"",strokeWidth:1,strokeColor:"#000000",highlightStrokeColor:"#888888"},ticks:{needsRegularUpdate:!1,fixed:!0,drawLabels:!1,digits:2,includeBoundaries:!0,drawZero:!0,label:{offset:[-4,-14],display:"internal"},minTicksDistance:30,insertTicks:!0,ticksDistance:1,minorHeight:4,majorHeight:5,minorTicks:0,strokeOpacity:1,strokeWidth:1,tickEndings:[0,1],majortickEndings:[0,1],strokeColor:"#000000",visible:"inherit"},highline:{strokeWidth:3,visible:"inherit",fixed:!0,tabindex:null,name:"",strokeColor:"#000000",highlightStrokeColor:"#888888"},label:{visible:"inherit",strokeColor:"#000000"},moveOnUp:!0},slopefield:{strokeWidth:.5,highlightStrokeWidth:.5,highlightStrokeColor:n.Z.palette.blue,highlightStrokeOpacity:.8,scale:1,arrowhead:{enabled:!1,size:5,angle:.125*Math.PI}},slopetriangle:{fillColor:n.Z.palette.red,fillOpacity:.4,highlightFillColor:n.Z.palette.red,highlightFillOpacity:.3,borders:{lastArrow:{type:1,size:6}},glider:{fixed:!0,visible:!1,withLabel:!1},baseline:{visible:!1,withLabel:!1,name:""},basepoint:{visible:!1,withLabel:!1,name:""},tangent:{visible:!1,withLabel:!1,name:""},toppoint:{visible:!1,withLabel:!1,name:""},label:{visible:!0,position:"first"}},smartlabelangle:{cssClass:"smart-label-solid smart-label-angle",highlightCssClass:"smart-label-solid smart-label-angle",anchorX:"left",anchorY:"middle",unit:"",prefix:"",suffix:"",measure:"deg",useMathJax:!0},smartlabelcircle:{cssClass:"smart-label-solid smart-label-circle",highlightCssClass:"smart-label-solid smart-label-circle",anchorX:"middle",useMathJax:!0,unit:"",prefix:"",suffix:"",measure:"radius"},smartlabelline:{cssClass:"smart-label-solid smart-label-line",highlightCssClass:"smart-label-solid smart-label-line",anchorX:"middle",useMathJax:!0,unit:"",measure:"length"},smartlabelpoint:{cssClass:"smart-label-solid smart-label-point",highlightCssClass:"smart-label-solid smart-label-point",anchorX:"middle",anchorY:"top",useMathJax:!0,dir:"row",unit:""},smartlabelpolygon:{cssClass:"smart-label-solid smart-label-polygon",highlightCssClass:"smart-label-solid smart-label-polygon",anchorX:"middle",useMathJax:!0,unit:"",measure:"area"},stepfunction:{},tangent:{},tangentto:{polar:{visible:!1,strokeWidth:1,dash:3},point:{visible:!1}},tapemeasure:{strokeColor:"#000000",strokeWidth:2,highlightStrokeColor:"#000000",withTicks:!0,withLabel:!0,rotate:0,precision:2,digits:2,point1:{visible:!0,strokeColor:"#000000",fillColor:"#ffffff",fillOpacity:0,highlightFillOpacity:.1,size:6,snapToPoints:!0,attractorUnit:"screen",attractorDistance:20,showInfobox:!1,withLabel:!1,name:""},point2:{visible:!0,strokeColor:"#000000",fillColor:"#ffffff",fillOpacity:0,highlightFillOpacity:.1,size:6,snapToPoints:!0,attractorUnit:"screen",attractorDistance:20,showInfobox:!1,withLabel:!1,name:""},ticks:{drawLabels:!1,drawZero:!0,insertTicks:!0,ticksDistance:.1,minorHeight:8,majorHeight:16,minorTicks:4,tickEndings:[0,1],majorTickEndings:[0,1],strokeOpacity:1,strokeWidth:1,strokeColor:"#000000",visible:"inherit",label:{anchorY:"top",anchorX:"middle",offset:[0,-10]}},label:{position:"top"}},text:{fontSize:12,fontUnit:"px",formatNumber:!1,digits:2,intl:{enabled:"inherit",options:{minimumFractionDigits:0,maximumFractionDigits:2}},parse:!0,useCaja:!1,isLabel:!1,strokeColor:"#000000",highlightStrokeColor:"#000000",highlightStrokeOpacity:.666666,cssDefaultStyle:"font-family: Arial, Helvetica, Geneva, sans-serif;",highlightCssDefaultStyle:"font-family: Arial, Helvetica, Geneva, sans-serif;",cssStyle:"",highlightCssStyle:"",transitionProperties:["color","opacity"],useASCIIMathML:!1,useMathJax:!1,useKatex:!1,katexMacros:{},toFraction:!1,display:"html",anchor:null,anchorX:"left",anchorY:"middle",cssClass:"JXGtext",highlightCssClass:"JXGtext",dragArea:"all",withLabel:!1,rotate:0,visible:!0,snapSizeX:1,snapSizeY:1,attractors:[]},tracecurve:{strokeColor:"#000000",fillColor:"none",numberPoints:100},turtle:{strokeWidth:1,fillColor:"none",strokeColor:"#000000",arrow:{strokeWidth:2,withLabel:!1,strokeColor:n.Z.palette.red,lastArrow:!0}},vectorfield:{strokeWidth:.5,highlightStrokeWidth:.5,highlightStrokeColor:n.Z.palette.blue,highlightStrokeOpacity:.8,scale:1,arrowhead:{enabled:!0,size:5,angle:.125*Math.PI}},shortcuts:{color:["strokeColor","fillColor"],opacity:["strokeOpacity","fillOpacity"],highlightColor:["highlightStrokeColor","highlightFillColor"],highlightOpacity:["highlightStrokeOpacity","highlightFillOpacity"],strokeWidth:["strokeWidth","highlightStrokeWidth"]}},s.Z.Validator=function(){var t,e=function(t){return a.Z.isString(t)},i=function(t){return a.Z.isNumber(t,!0,!1)},r=function(t){return Math.abs(t-Math.round(t))<o.Z.eps},n=function(t){return r(t)&&t>0},h=function(t){return t>0},l=function(t){return t>=0},c={},d={attractorDistance:l,color:e,display:function(t){return"html"===t||"internal"===t},doAdvancedPlot:!1,draft:!1,drawLabels:!1,drawZero:!1,face:function(t){return a.Z.exists(s.Z.normalizePointFace(t))},factor:i,fillColor:e,fillOpacity:i,firstArrow:!1,fontSize:r,dash:r,gridX:i,gridY:i,hasGrid:!1,highlightFillColor:e,highlightFillOpacity:i,highlightStrokeColor:e,highlightStrokeOpacity:i,insertTicks:!1,lastArrow:!1,layer:function(t){return r(t)&&t>=0},majorHeight:r,minorHeight:r,minorTicks:l,minTicksDistance:n,numberPointsHigh:n,numberPointsLow:n,opacity:i,radius:i,RDPsmoothing:!1,renderer:function(t){return"vml"===t||"svg"===t||"canvas"===t||"no"===t},right:function(t){return/^[0-9]+px$/.test(t)},showCopyright:!1,showInfobox:!1,showNavigation:!1,size:l,snapSizeX:h,snapSizeY:h,snapWidth:i,snapToGrid:!1,snatchDistance:l,straightFirst:!1,straightLast:!1,stretch:!1,strokeColor:e,strokeOpacity:i,strokeWidth:l,takeFirst:!1,takeSizeFromFile:!1,to10:!1,toOrigin:!1,translateTo10:!1,translateToOrigin:!1,useASCIIMathML:!1,useDirection:!1,useMathJax:!1,withLabel:!1,withTicks:!1,zoom:!1};for(t in d)d.hasOwnProperty(t)&&(c[t.toLowerCase()]=d[t]);return c}(),s.Z.normalizePointFace=function(t){return{cross:"x",x:"x",circle:"o",o:"o",square:"[]","[]":"[]",plus:"+","+":"+",divide:"|","|":"|",minus:"-","-":"-",diamond:"<>","<>":"<>",diamond2:"<<>>","<<>>":"<<>>",triangleup:"^",A:"^",a:"^","^":"^",triangledown:"v",v:"v",triangleleft:"<","<":"<",triangleright:">",">":">"}[t]},s.Z.useStandardOptions=function(t){var e,i,o,n,a=s.Z.Options,h=t.hasGrid;for(e in t.options.grid.hasGrid=a.grid.hasGrid,t.options.grid.gridX=a.grid.gridX,t.options.grid.gridY=a.grid.gridY,t.options.grid.gridColor=a.grid.gridColor,t.options.grid.gridOpacity=a.grid.gridOpacity,t.options.grid.gridDash=a.grid.gridDash,t.options.grid.snapToGrid=a.grid.snapToGrid,t.options.grid.snapSizeX=a.grid.SnapSizeX,t.options.grid.snapSizeY=a.grid.SnapSizeY,t.takeSizeFromFile=a.takeSizeFromFile,n=function(t,e){t.visProp.fillcolor=e.fillColor,t.visProp.highlightfillcolor=e.highlightFillColor,t.visProp.strokecolor=e.strokeColor,t.visProp.highlightstrokecolor=e.highlightStrokeColor},t.objects)if(t.objects.hasOwnProperty(e))if((o=t.objects[e]).elementClass===r.Z.OBJECT_CLASS_POINT)n(o,a.point);else if(o.elementClass===r.Z.OBJECT_CLASS_LINE)for(n(o,a.line),i=0;i<o.ticks.length;i++)o.ticks[i].majorTicks=a.line.ticks.majorTicks,o.ticks[i].minTicksDistance=a.line.ticks.minTicksDistance,o.ticks[i].visProp.minorheight=a.line.ticks.minorHeight,o.ticks[i].visProp.majorheight=a.line.ticks.majorHeight;else o.elementClass===r.Z.OBJECT_CLASS_CIRCLE?n(o,a.circle):o.type===r.Z.OBJECT_TYPE_ANGLE?n(o,a.angle):o.type===r.Z.OBJECT_TYPE_ARC?n(o,a.arc):o.type===r.Z.OBJECT_TYPE_POLYGON?n(o,a.polygon):o.type===r.Z.OBJECT_TYPE_CONIC?n(o,a.conic):o.type===r.Z.OBJECT_TYPE_CURVE?n(o,a.curve):o.type===r.Z.OBJECT_TYPE_SECTOR&&(o.arc.visProp.fillcolor=a.sector.fillColor,o.arc.visProp.highlightfillcolor=a.sector.highlightFillColor,o.arc.visProp.fillopacity=a.sector.fillOpacity,o.arc.visProp.highlightfillopacity=a.sector.highlightFillOpacity);t.fullUpdate(),h&&!t.hasGrid?t.removeGrids(t):!h&&t.hasGrid&&t.create("grid",[])},s.Z.useBlackWhiteOptions=function(t){var e=s.Z.Options;e.point.fillColor=n.Z.rgb2bw(e.point.fillColor),e.point.highlightFillColor=n.Z.rgb2bw(e.point.highlightFillColor),e.point.strokeColor=n.Z.rgb2bw(e.point.strokeColor),e.point.highlightStrokeColor=n.Z.rgb2bw(e.point.highlightStrokeColor),e.line.fillColor=n.Z.rgb2bw(e.line.fillColor),e.line.highlightFillColor=n.Z.rgb2bw(e.line.highlightFillColor),e.line.strokeColor=n.Z.rgb2bw(e.line.strokeColor),e.line.highlightStrokeColor=n.Z.rgb2bw(e.line.highlightStrokeColor),e.circle.fillColor=n.Z.rgb2bw(e.circle.fillColor),e.circle.highlightFillColor=n.Z.rgb2bw(e.circle.highlightFillColor),e.circle.strokeColor=n.Z.rgb2bw(e.circle.strokeColor),e.circle.highlightStrokeColor=n.Z.rgb2bw(e.circle.highlightStrokeColor),e.arc.fillColor=n.Z.rgb2bw(e.arc.fillColor),e.arc.highlightFillColor=n.Z.rgb2bw(e.arc.highlightFillColor),e.arc.strokeColor=n.Z.rgb2bw(e.arc.strokeColor),e.arc.highlightStrokeColor=n.Z.rgb2bw(e.arc.highlightStrokeColor),e.polygon.fillColor=n.Z.rgb2bw(e.polygon.fillColor),e.polygon.highlightFillColor=n.Z.rgb2bw(e.polygon.highlightFillColor),e.sector.fillColor=n.Z.rgb2bw(e.sector.fillColor),e.sector.highlightFillColor=n.Z.rgb2bw(e.sector.highlightFillColor),e.curve.strokeColor=n.Z.rgb2bw(e.curve.strokeColor),e.grid.gridColor=n.Z.rgb2bw(e.grid.gridColor),s.Z.useStandardOptions(t)},s.Z.Options.normalizePointFace=s.Z.normalizePointFace,e.Z=s.Z.Options},632:function(t,e,i){var s=i(765),r=i(351),o=i(109);s.Z.GeonextParser={replacePow:function(t){var e,i,s,r,o,n,a,h,l,c,d,u;for(c=(t=t.replace(/(\s*)\^(\s*)/g,"^")).indexOf("^"),r=-1;c>=0&&c<t.length-1;){if(r===c)throw new Error("JSXGraph: Error while parsing expression '"+t+"'");if(r=c,l=t.slice(0,c),d=t.slice(c+1),")"===l.charAt(l.length-1)){for(e=1,i=l.length-2;i>=0&&e>0;)")"===(s=l.charAt(i))?e++:"("===s&&(e-=1),i-=1;if(0!==e)throw new Error("JSXGraph: Missing '(' in expression");for(o="",a=l.substring(0,i+1),h=i;h>=0&&a.slice(h,h+1).match(/([\w.]+)/);)o=RegExp.$1+o,h-=1;o=(o+=l.substring(i+1,l.length)).replace(/([()+*%^\-/\][])/g,"\\$1")}else o="[\\w\\.]+";if(d.match(/^([\w.]*\()/)){for(e=1,i=RegExp.$1.length;i<d.length&&e>0;)")"===(s=d.charAt(i))?e-=1:"("===s&&(e+=1),i+=1;if(0!==e)throw new Error("JSXGraph: Missing ')' in expression");n=(n=d.substring(0,i)).replace(/([()+*%^\-/[\]])/g,"\\$1")}else n="[\\w\\.]+";u=new RegExp("("+o+")\\^("+n+")"),c=(t=t.replace(u,"pow($1,$2)")).indexOf("^")}return t},replaceIf:function(t){var e,i,s,r,o,n,a,h,l,c="",d=null,u=null,p=null;if((s=t.indexOf("If("))<0)return t;for(t=t.replace(/""/g,"0");s>=0;){for(e=t.slice(0,s),i=t.slice(s+3),o=1,r=0,n=-1,a=-1;r<i.length&&o>0;)")"===(h=i.charAt(r))?o-=1:"("===h?o+=1:","===h&&1===o&&(n<0?n=r:a=r),r+=1;if(l=i.slice(0,r-1),i=i.slice(r),n<0)return"";if(a<0)return"";d=l.slice(0,n),u=l.slice(n+1,a),p=l.slice(a+1),c+=e+"(("+(d=this.replaceIf(d))+")?("+(u=this.replaceIf(u))+"):("+(p=this.replaceIf(p))+"))",d=null,u=null,s=(t=i).indexOf("If(")}return c+=i},replaceNameById:function(t,e,i){var s,r,o,n,a=0,h=["X","Y","L","V"],l=function(t){return i?"$('"+t+"')":t};for(n=0;n<h.length;n++)for(a=t.indexOf(h[n]+"(");a>=0;)a>=0&&(s=t.indexOf(")",a+2))>=0&&(r=(r=t.slice(a+2,s)).replace(/\\(['"])?/g,"$1"),(o=e.elementsByName[r])&&(t=t.slice(0,a+2)+(i?"$('":"")+l(o.id)+t.slice(s))),s=t.indexOf(")",a+2),a=t.indexOf(h[n]+"(",s);for(a=t.indexOf("Dist(");a>=0;)a>=0&&(s=t.indexOf(",",a+5))>=0&&(r=(r=t.slice(a+5,s)).replace(/\\(['"])?/g,"$1"),(o=e.elementsByName[r])&&(t=t.slice(0,a+5)+l(o.id)+t.slice(s))),s=t.indexOf(",",a+5),a=t.indexOf(",",s),(s=t.indexOf(")",a+1))>=0&&(r=(r=t.slice(a+1,s)).replace(/\\(['"])?/g,"$1"),(o=e.elementsByName[r])&&(t=t.slice(0,a+1)+l(o.id)+t.slice(s))),s=t.indexOf(")",a+1),a=t.indexOf("Dist(",s);for(h=["Deg","Rad"],n=0;n<h.length;n++)for(a=t.indexOf(h[n]+"(");a>=0;)a>=0&&(s=t.indexOf(",",a+4))>=0&&(r=(r=t.slice(a+4,s)).replace(/\\(['"])?/g,"$1"),(o=e.elementsByName[r])&&(t=t.slice(0,a+4)+l(o.id)+t.slice(s))),s=t.indexOf(",",a+4),a=t.indexOf(",",s),(s=t.indexOf(",",a+1))>=0&&(r=(r=t.slice(a+1,s)).replace(/\\(['"])?/g,"$1"),(o=e.elementsByName[r])&&(t=t.slice(0,a+1)+l(o.id)+t.slice(s))),s=t.indexOf(",",a+1),a=t.indexOf(",",s),(s=t.indexOf(")",a+1))>=0&&(r=(r=t.slice(a+1,s)).replace(/\\(['"])?/g,"$1"),(o=e.elementsByName[r])&&(t=t.slice(0,a+1)+l(o.id)+t.slice(s))),s=t.indexOf(")",a+1),a=t.indexOf(h[n]+"(",s);return t},replaceIdByObj:function(t){var e=/(X|Y|L)\(([\w_]+)\)/g;return t=t.replace(e,"$('$2').$1()"),e=/(V)\(([\w_]+)\)/g,t=t.replace(e,"$('$2').Value()"),e=/(Dist)\(([\w_]+),([\w_]+)\)/g,t=t.replace(e,"dist($('$2'), $('$3'))"),e=/(Deg)\(([\w_]+),([ \w[\w_]+),([\w_]+)\)/g,t=t.replace(e,"deg($('$2'),$('$3'),$('$4'))"),e=/Rad\(([\w_]+),([\w_]+),([\w_]+)\)/g,t=t.replace(e,"rad($('$1'),$('$2'),$('$3'))"),e=/N\((.+)\)/g,t=t.replace(e,"($1)")},geonext2JS:function(t,e){var i,s,r,o=["Abs","ACos","ASin","ATan","Ceil","Cos","Exp","Factorial","Floor","Log","Max","Min","Random","Round","Sin","Sqrt","Tan","Trunc"],n=["abs","acos","asin","atan","ceil","cos","exp","factorial","floor","log","max","min","random","round","sin","sqrt","tan","ceil"];for(s=t=(t=(t=t.replace(/&lt;/g,"<")).replace(/&gt;/g,">")).replace(/&amp;/g,"&"),s=this.replaceNameById(s,e),s=this.replaceIf(s),s=this.replacePow(s),s=this.replaceIdByObj(s),r=0;r<o.length;r++)i=new RegExp(["(\\W|^)(",o[r],")"].join(""),"ig"),s=s.replace(i,["$1",n[r]].join(""));return s=(s=(s=(s=(s=s.replace(/True/g,"true")).replace(/False/g,"false")).replace(/fasle/g,"false")).replace(/Pi/g,"PI")).replace(/"/g,"'")},findDependencies:function(t,e,i){var s,n,a,h;for(n in o.Z.exists(i)||(i=t.board),s=i.elementsByName)s.hasOwnProperty(n)&&n!==t.name&&(s[n].elementClass===r.Z.OBJECT_CLASS_TEXT&&s[n].evalVisProp("islabel")||(h=(h=n.replace(/\[/g,"\\[")).replace(/\]/g,"\\]"),a=new RegExp("\\(([\\w\\[\\]'_ ]+,)*("+h+")(,[\\w\\[\\]'_ ]+)*\\)","g"),e.search(a)>=0&&s[n].addChild(t)))},gxt2jc:function(t,e){var i;return i=t=(t=(t=t.replace(/&lt;/g,"<")).replace(/&gt;/g,">")).replace(/&amp;/g,"&"),i=(i=(i=(i=this.replaceNameById(i,e,!0)).replace(/True/g,"true")).replace(/False/g,"false")).replace(/fasle/g,"false")}},e.Z=s.Z.GeonextParser},254:function(__unused_webpack_module,__webpack_exports__,__webpack_require__){var _jxg_js__WEBPACK_IMPORTED_MODULE_0__=__webpack_require__(765),_base_constants_js__WEBPACK_IMPORTED_MODULE_1__=__webpack_require__(351),_base_text_js__WEBPACK_IMPORTED_MODULE_2__=__webpack_require__(573),_math_math_js__WEBPACK_IMPORTED_MODULE_3__=__webpack_require__(275),_math_ia_js__WEBPACK_IMPORTED_MODULE_4__=__webpack_require__(785),_math_geometry_js__WEBPACK_IMPORTED_MODULE_5__=__webpack_require__(922),_math_statistics_js__WEBPACK_IMPORTED_MODULE_6__=__webpack_require__(309),_utils_type_js__WEBPACK_IMPORTED_MODULE_7__=__webpack_require__(109),_utils_env_js__WEBPACK_IMPORTED_MODULE_8__=__webpack_require__(503);Object.create||(Object.create=function(t,e){if("object"!=typeof t&&"function"!=typeof t)throw new TypeError("Object prototype may only be an Object: "+t);if(null===t)throw new Error("This browser's implementation of Object.create is a shim and doesn't support 'null' as the first argument.");if(void 0!==e)throw new Error("This browser's implementation of Object.create is a shim and doesn't support a second argument.");function i(){}return i.prototype=t,new i});var priv={modules:{math:_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z,"math/geometry":_math_geometry_js__WEBPACK_IMPORTED_MODULE_5__.Z,"math/statistics":_math_statistics_js__WEBPACK_IMPORTED_MODULE_6__.Z,"math/numerics":_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.Numerics}},r,_ccache;_jxg_js__WEBPACK_IMPORTED_MODULE_0__.Z.JessieCode=function(t,e){this.scope={id:0,hasChild:!0,args:[],locals:{},context:null,previous:null},this.scopes=[],this.scopes.push(this.scope),this.dpstack=[[]],this.pscope=0,this.propstack=[{}],this.propscope=0,this.lhs=[],this.isLHS=!1,this.warnLog="jcwarn",this.$log=[],this.builtIn=this.defineBuiltIn(),this.operands=this.getPossibleOperands(),this.board=null,this.forceValueCall=!1,this.lineToElement={},this.parCurLine=1,this.parCurColumn=0,this.line=1,this.col=1,_jxg_js__WEBPACK_IMPORTED_MODULE_0__.Z.CA&&(this.CA=new _jxg_js__WEBPACK_IMPORTED_MODULE_0__.Z.CA(this.node,this.createNode,this)),this.code="","string"==typeof t&&this.parse(t,e)},_jxg_js__WEBPACK_IMPORTED_MODULE_0__.Z.extend(_jxg_js__WEBPACK_IMPORTED_MODULE_0__.Z.JessieCode.prototype,{node:function(t,e,i){return{type:t,value:e,children:i}},createNode:function(t,e,i){var s,r=this.node(t,e,[]);for(s=2;s<arguments.length;s++)r.children.push(arguments[s]);return"node_const"===r.type&&_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isNumber(r.value)&&(r.isMath=!0),r.line=this.parCurLine,r.col=this.parCurColumn,r},pushScope:function(t){var e={args:t,locals:{},context:null,previous:this.scope};return this.scope.hasChild=!0,this.scope=e,e.id=this.scopes.push(e)-1,e},popScope:function(){var t=this.scope.previous;return this.scope=null!==t?t:this.scope,this.scope},getElementById:function(t){return this.board.objects[t]},log:function(){this.$log.push(arguments),"object"==typeof console&&console.log&&console.log.apply(console,arguments)},creator:(_ccache={},(r=function(t){var e,i;return"function"==typeof _ccache[this.board.id+t]?e=_ccache[this.board.id+t]:(i=this,(e=function(e,s){var r;return void 0===(r=_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(s)?s:{}).name&&void 0===r.id&&(r.name=0!==i.lhs[i.scope.id]?i.lhs[i.scope.id]:""),i.board.create(t,e,r)}).creator=!0,_ccache[this.board.id+t]=e),e}).clearCache=function(){_ccache={}},r),letvar:function(t,e){this.builtIn[t]&&this._warn('"'+t+'" is a predefined value.'),this.scope.locals[t]=e},isLocalVariable:function(t){for(var e=this.scope;null!==e;){if(_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(e.locals[t]))return e;e=e.previous}return null},isParameter:function(t){for(var e=this.scope;null!==e;){if(_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.indexOf(e.args,t)>-1)return e;e=e.previous}return null},isCreator:function(t){return!!_jxg_js__WEBPACK_IMPORTED_MODULE_0__.Z.elements[t]},isMathMethod:function(t){return"E"!==t&&!!Math[t]},isBuiltIn:function(t){return!!this.builtIn[t]},getvar:function(t,e,i){var s;if(e=_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.def(e,!1),null!==(s=this.isLocalVariable(t)))return s.locals[t];if("$board"===t||"EULER"===t||"PI"===t)return this.builtIn[t];if(i){if(this.isBuiltIn(t))return this.builtIn[t];if(this.isMathMethod(t))return Math[t];if(this.isCreator(t))return this.creator(t)}return e||(s=this.board.select(t))===t?void 0:s},resolve:function(t){for(var e=this.scope;null!==e;){if(_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(e.locals[t]))return e.locals[t];e=e.previous}},getvarJS:function(t,e,i){var s,r="";return e=_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.def(e,!1),i=_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.def(i,!1),null!==this.isParameter(t)?t:null===this.isLocalVariable(t)||i?this.isCreator(t)?"(function () { var a = Array.prototype.slice.call(arguments, 0), props = "+(i?"a.pop()":"{}")+"; return $jc$.board.create.apply($jc$.board, ['"+t+"'].concat([a, props])); })":(i&&this._error("Syntax error (attribute values are allowed with element creators only)"),this.isBuiltIn(t)?(r=this.builtIn[t].src||this.builtIn[t],_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isNumber(r)||r.match(/board\.select/)?r:(t=r.split(".").pop(),_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(this.board.mathLib)&&null!==(s=new RegExp("^Math."+t)).exec(r)?r.replace(s,"$jc$.board.mathLib."+t):_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(this.board.mathLibJXG)&&null!==(s=new RegExp("^JXG.Math.")).exec(r)?r.replace(s,"$jc$.board.mathLibJXG."):r)):this.isMathMethod(t)?"$jc$.board.mathLib."+t:e?"":(_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isId(this.board,t)?(r="$jc$.board.objects['"+t+"']","slider"===this.board.objects[t].elType&&(r+=".Value()")):_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isName(this.board,t)?(r="$jc$.board.elementsByName['"+t+"']","slider"===this.board.elementsByName[t].elType&&(r+=".Value()")):_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isGroup(this.board,t)&&(r="$jc$.board.groups['"+t+"']"),r)):"$jc$.resolve('"+t+"')"},makeMap:function(t){return t.isMap=!0,t},functionCodeJS:function(t){var e=t.children[0].join(", "),i="",s="";return"op_map"===t.value&&(i="{ return  ",s=" }"),"function ("+e+") {\nvar $oldscope$ = $jc$.scope;\n$jc$.scope = $jc$.scopes["+this.scope.id+"];\nvar r = (function () "+i+this.compile(t.children[1],!0)+s+")();\n$jc$.scope = $oldscope$;\nreturn r;\n}"},defineFunction:function(node){var fun,i,that=this,list=node.children[0],scope=this.pushScope(list),_that;if(this.board.options.jc.compile){for(this.isLHS=!1,i=0;i<list.length;i++)scope.locals[list[i]]=list[i];this.replaceNames(node.children[1]),fun=function(jc){var fun,str="var f = function($jc$) { return "+jc.functionCodeJS(node)+"}; f;";try{for(fun=eval(str)(jc),scope.argtypes=[],i=0;i<list.length;i++)scope.argtypes.push(that.resolveType(list[i],node));return fun}catch(t){return jc._warn("error compiling function\n\n"+str+"\n\n"+t.toString()),function(){}}}(this),this.popScope()}else fun=function(t,e,i){return function(){var s,r;for(r=e.scope,e.scope=e.scopes[i],s=0;s<t.length;s++)e.scope.locals[t[s]]=arguments[s];return s=e.execute(node.children[1]),e.scope=r,s}}(list,this,scope.id);return fun.node=node,fun.scope=scope,fun.toJS=fun.toString,fun.toString=(_that=this,function(){return _that.compile(_that.replaceIDs(_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.deepCopy(node)))}),fun.deps={},this.collectDependencies(node.children[1],node.children[0],fun.deps),fun},mergeAttributes:function(t){var e,i={};for(e=0;e<arguments.length;e++)i=_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.deepCopy(i,arguments[e],!0);return i},setProp:function(t,e,i){var s,r,o={};t.elementClass!==_base_constants_js__WEBPACK_IMPORTED_MODULE_1__.Z.OBJECT_CLASS_POINT||"X"!==e&&"Y"!==e?t.elementClass!==_base_constants_js__WEBPACK_IMPORTED_MODULE_1__.Z.OBJECT_CLASS_TEXT||"X"!==e&&"Y"!==e?t.type&&t.elementClass&&t.visProp?_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(t[t.methodMap[e]])&&"function"!=typeof t[t.methodMap[e]]?t[t.methodMap[e]]=i:(o[e]=i,t.setAttribute(o)):t[e]=i:("number"==typeof i?t[e]=function(){return i}:"function"==typeof i?(t.isDraggable=!1,t[e]=i):"string"==typeof i&&(t.isDraggable=!1,t[e]=_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.createFunction(i,this.board),t[e+"jc"]=i),t[e].origin=i,this.board.update()):(e=e.toLowerCase(),t.isDraggable&&"number"==typeof i?(s="x"===e?i:t.X(),r="y"===e?i:t.Y(),t.setPosition(_base_constants_js__WEBPACK_IMPORTED_MODULE_1__.Z.COORDS_BY_USER,[s,r])):!t.isDraggable||"function"!=typeof i&&"string"!=typeof i?t.isDraggable||(s="x"===e?i:t.XEval.origin,r="y"===e?i:t.YEval.origin,t.addConstraint([s,r])):(s="x"===e?i:t.coords.usrCoords[1],r="y"===e?i:t.coords.usrCoords[2],t.addConstraint([s,r])),this.board.update())},_genericParse:function(t,e,i,s){var r,o,n,a,h=t.replace(/\r\n/g,"\n").split("\n"),l=[];s||(this.code+=t+"\n"),_base_text_js__WEBPACK_IMPORTED_MODULE_2__.Z&&(o=_base_text_js__WEBPACK_IMPORTED_MODULE_2__.Z.prototype.setText,_base_text_js__WEBPACK_IMPORTED_MODULE_2__.Z.prototype.setText=_base_text_js__WEBPACK_IMPORTED_MODULE_2__.Z.prototype.setTextJessieCode);try{for(_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(i)||(i=!1),r=0;r<h.length;r++)i&&(h[r]=_jxg_js__WEBPACK_IMPORTED_MODULE_0__.Z.GeonextParser.geonext2JS(h[r],this.board)),l.push(h[r]);switch(t=l.join("\n"),n=parser.parse(t),this.CA&&(n=this.CA.expandDerivatives(n,null,n),n=this.CA.removeTrivialNodes(n)),e){case"parse":a=this.execute(n);break;case"manipulate":a=this.compile(n);break;case"getAst":a=n;break;default:a=!1}}catch(t){throw t}finally{_base_text_js__WEBPACK_IMPORTED_MODULE_2__.Z&&(_base_text_js__WEBPACK_IMPORTED_MODULE_2__.Z.prototype.setText=o)}return a},parse:function(t,e,i){return this._genericParse(t,"parse",e,i)},manipulate:function(t,e,i){return this._genericParse(t,"manipulate",e,i)},getAST:function(t,e,i){return this._genericParse(t,"getAst",e,i)},snippet:function(t,e,i,s,r){var o;return e=_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.def(e,!0),i=_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.def(i,""),s=_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.def(s,!1),this.forceValueCall=_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.def(r,!0),o=(e?" function ("+i+") { return ":"")+t+(e?"; }":"")+";",this.parse(o,s,!0)},replaceIDs:function(t){var e,i;if(t.replaced&&(i=this.board.objects[t.children[1][0].value],_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(i)&&""!==i.name&&(t.type="node_var",t.value=i.name,t.children.length=0,delete t.replaced)),_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isArray(t))for(e=0;e<t.length;e++)t[e]=this.replaceIDs(t[e]);if(t.children)for(e=t.children.length;e>0;e--)_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(t.children[e-1])&&(t.children[e-1]=this.replaceIDs(t.children[e-1]));return t},replaceNames:function(t,e){var i,s,r=!1;if(void 0!==e&&(r=e),s=t.value,"node_op"===t.type&&"op_lhs"===s&&1===t.children.length?this.isLHS=!0:"node_var"===t.type&&(this.isLHS?this.letvar(s,!0):!_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(this.getvar(s,!0))&&_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(this.board.elementsByName[s])&&(r&&"slider"!==this.board.elementsByName[s].elType&&(r=!1),t=this.createReplacementNode(t,r))),_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isArray(t))for(i=0;i<t.length;i++)t[i]=this.replaceNames(t[i],r);if(t.children)for(this.forceValueCall&&("op_execfun"===t.value&&"V"!==t.children[0].value&&"$"!==t.children[0].value&&(_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(Math[t.children[0].value])||_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z[t.children[0].value]))&&"node_var"===t.children[1][0].type||"op_return"===t.value&&1===t.children.length&&"node_var"===t.children[0].type)&&(r=!0),i=t.children.length;i>0;i--)_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(t.children[i-1])&&(t.children[i-1]=this.replaceNames(t.children[i-1],r));return"node_op"===t.type&&"op_lhs"===t.value&&1===t.children.length&&(this.isLHS=!1),t},createReplacementNode:function(t,e){var i=t.value,s=this.board.elementsByName[i];return(t=this.createNode("node_op","op_execfun",this.createNode("node_var",!0===e?"$value":"$"),[this.createNode("node_str",s.id)])).replaced=!0,t},collectDependencies:function(t,e,i){var s,r,o,n;if(_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isArray(t))for(n=t.length,s=0;s<n;s++)this.collectDependencies(t[s],e,i);else if(r=t.value,"node_var"===t.type&&e.indexOf(r)<0&&(o=this.getvar(r))&&o.visProp&&o.elType&&o.elementClass&&o.id&&(i[o.id]=o),"node_op"===t.type&&"op_execfun"===t.value&&t.children.length>1&&("$"===t.children[0].value||"$value"===t.children[0].value)&&t.children[1].length>0&&(i[o=t.children[1][0].value]=this.board.objects[o]),t.children)for(s=t.children.length;s>0;s--)_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(t.children[s-1])&&this.collectDependencies(t.children[s-1],e,i)},resolveProperty:function(t,e,i){return i=_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.def(i,!1),t&&t.methodMap&&(_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(t.subs)&&_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(t.subs[e])?t=t.subs:_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(t.methodMap[e])?e=t.methodMap[e]:(t=t.visProp,e=e.toLowerCase())),_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isFunction(t)&&this._error("Accessing function properties is not allowed."),_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(t)||this._error(t+" is not an object"),_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(t[e])||this._error("unknown property "+e),i&&"function"==typeof t[e]?function(){return t[e].apply(t,arguments)}:t[e]},resolveType:function(t,e){var i,s;if(_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isArray(e))for(i=0;i<e.length;i++)if("any"!==(s=this.resolveType(t,e[i])))return s;if("node_op"===e.type&&"op_execfun"===e.value&&"node_var"===e.children[0].type&&e.children[0].value===t)return"function";if("node_op"===e.type){for(i=0;i<e.children.length;i++)if("node_var"===e.children[0].type&&e.children[0].value===t&&("op_add"===e.value||"op_sub"===e.value||"op_mul"===e.value||"op_div"===e.value||"op_mod"===e.value||"op_exp"===e.value||"op_neg"===e.value))return"any";for(i=0;i<e.children.length;i++)if("any"!==(s=this.resolveType(t,e.children[i])))return s}return"any"},getLHS:function(t){var e;if("node_var"===t.type)e={o:this.scope.locals,what:t.value};else if("node_op"===t.type&&"op_property"===t.value)e={o:this.execute(t.children[0]),what:t.children[1]};else{if("node_op"!==t.type||"op_extvalue"!==t.value)throw new Error("Syntax error: Invalid left-hand side of assignment.");e={o:this.execute(t.children[0]),what:this.execute(t.children[1])}}return e},getLHSCompiler:function(t,e){var i;if("node_var"===t.type)i=t.value;else if("node_op"===t.type&&"op_property"===t.value)i=[this.compile(t.children[0],e),"'"+t.children[1]+"'"];else{if("node_op"!==t.type||"op_extvalue"!==t.value)throw new Error("Syntax error: Invalid left-hand side of assignment.");i=[this.compile(t.children[0],e),"node_const"===t.children[1].type?t.children[1].value:this.compile(t.children[1],e)]}return i},execute:function(t){var e,i,s,r,o,n,a,h,l,c,d=[];if(e=0,!t)return e;switch(this.line=t.line,this.col=t.col,t.type){case"node_op":switch(t.value){case"op_none":t.children[0]&&this.execute(t.children[0]),t.children[1]&&(e=this.execute(t.children[1]));break;case"op_assign":i=this.getLHS(t.children[0]),this.lhs[this.scope.id]=i.what,i.o.type&&i.o.elementClass&&i.o.methodMap&&"label"===i.what&&this._error("Left-hand side of assignment is read-only."),e=this.execute(t.children[1]),i.o!==this.scope.locals||_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isArray(i.o)&&"number"==typeof i.what?this.setProp(i.o,i.what,e):this.letvar(i.what,e),this.lhs[this.scope.id]=0;break;case"op_if":this.execute(t.children[0])&&(e=this.execute(t.children[1]));break;case"op_conditional":case"op_if_else":e=this.execute(t.children[0])?this.execute(t.children[1]):this.execute(t.children[2]);break;case"op_while":for(;this.execute(t.children[0]);)this.execute(t.children[1]);break;case"op_do":do{this.execute(t.children[0])}while(this.execute(t.children[1]));break;case"op_for":for(this.execute(t.children[0]);this.execute(t.children[1]);this.execute(t.children[2]))this.execute(t.children[3]);break;case"op_proplst":t.children[0]&&this.execute(t.children[0]),t.children[1]&&this.execute(t.children[1]);break;case"op_emptyobject":e={};break;case"op_proplst_val":this.propstack.push({}),this.propscope++,this.execute(t.children[0]),e=this.propstack[this.propscope],this.propstack.pop(),this.propscope--;break;case"op_prop":this.propstack[this.propscope][t.children[0]]=this.execute(t.children[1]);break;case"op_array":for(e=[],o=t.children[0].length,s=0;s<o;s++)e.push(this.execute(t.children[0][s]));break;case"op_extvalue":e=this.execute(t.children[0]),e="number"==typeof(s=this.execute(t.children[1]))&&Math.abs(Math.round(s)-s)<_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.eps?e[s]:undefined;break;case"op_return":if(0!==this.scope)return this.execute(t.children[0]);this._error("Unexpected return.");break;case"op_map":t.children[1].isMath||"node_var"===t.children[1].type||this._error("execute: In a map only function calls and mathematical expressions are allowed."),(h=this.defineFunction(t)).isMap=!0,e=h;break;case"op_function":(h=this.defineFunction(t)).isMap=!1,e=h;break;case"op_execfun":if(this.dpstack.push([]),this.pscope++,n=t.children[1],_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(t.children[2]))if(t.children[3])for(a=t.children[2],l={},s=0;s<a.length;s++)l=_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.deepCopy(l,this.execute(a[s]),!0);else l=this.execute(t.children[2]);for(t.children[0]._isFunctionName=!0,h=this.execute(t.children[0]),delete t.children[0]._isFunctionName,c=_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(h)&&_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(h.sc)?h.sc:this,!h.creator&&_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(t.children[2])&&this._error("Unexpected value. Only element creators are allowed to have a value after the function call."),s=0;s<n.length;s++)_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(h.scope)&&_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(h.scope.argtypes)&&"function"===h.scope.argtypes[s]?(n[s]._isFunctionName=!0,d[s]=this.execute(n[s]),delete n[s]._isFunctionName):d[s]=this.execute(n[s]),this.dpstack[this.pscope].push({line:t.children[1][s].line,col:t.children[1][s].ecol});if("function"!=typeof h||h.creator)if("function"==typeof h&&h.creator){r=this.line;try{for((e=h(d,l)).jcLineStart=r,e.jcLineEnd=t.eline,s=r;s<=t.line;s++)this.lineToElement[s]=e;e.debugParents=this.dpstack[this.pscope]}catch(t){this._error(t.toString())}}else this._error("Function '"+h+"' is undefined.");else e=h.apply(c,d);this.dpstack.pop(),this.pscope--;break;case"op_property":r=this.execute(t.children[0]),i=t.children[1],e=this.resolveProperty(r,i,!1),_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(e)&&["number","string","boolean"].indexOf(typeof e)<0&&(e.sc=r);break;case"op_use":this._warn("Use of the 'use' operator is deprecated."),this.use(t.children[0].toString());break;case"op_delete":this._warn("Use of the 'delete' operator is deprecated. Please use the remove() function."),i=this.getvar(t.children[0]),e=this.del(i);break;case"op_eq":e=this.execute(t.children[0])==this.execute(t.children[1]);break;case"op_neq":e=this.execute(t.children[0])!=this.execute(t.children[1]);break;case"op_approx":e=Math.abs(this.execute(t.children[0])-this.execute(t.children[1]))<_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.eps;break;case"op_gt":e=this.execute(t.children[0])>this.execute(t.children[1]);break;case"op_lt":e=this.execute(t.children[0])<this.execute(t.children[1]);break;case"op_geq":e=this.execute(t.children[0])>=this.execute(t.children[1]);break;case"op_leq":e=this.execute(t.children[0])<=this.execute(t.children[1]);break;case"op_or":e=this.execute(t.children[0])||this.execute(t.children[1]);break;case"op_and":e=this.execute(t.children[0])&&this.execute(t.children[1]);break;case"op_not":e=!this.execute(t.children[0]);break;case"op_add":e=this.add(this.execute(t.children[0]),this.execute(t.children[1]));break;case"op_sub":e=this.sub(this.execute(t.children[0]),this.execute(t.children[1]));break;case"op_div":e=this.div(this.execute(t.children[0]),this.execute(t.children[1]));break;case"op_mod":e=this.mod(this.execute(t.children[0]),this.execute(t.children[1]),!0);break;case"op_mul":e=this.mul(this.execute(t.children[0]),this.execute(t.children[1]));break;case"op_exp":e=this.pow(this.execute(t.children[0]),this.execute(t.children[1]));break;case"op_neg":e=this.neg(this.execute(t.children[0]))}break;case"node_var":e=this.getvar(t.value,!1,t._isFunctionName);break;case"node_const":e=null===t.value?null:Number(t.value);break;case"node_const_bool":e=t.value;break;case"node_str":e=t.value.replace(/\\(.)/g,"$1")}return e},compile:function(t,e){var i,s,r,o="";if(_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(e)||(e=!1),!t)return o;switch(t.type){case"node_op":switch(t.value){case"op_none":t.children[0]&&(o=this.compile(t.children[0],e)),t.children[1]&&(o+=this.compile(t.children[1],e));break;case"op_assign":e?(i=this.getLHSCompiler(t.children[0],e),_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isArray(i)?o="$jc$.setProp("+i[0]+", "+i[1]+", "+this.compile(t.children[1],e)+");\n":(this.isLocalVariable(i)!==this.scope&&(this.scope.locals[i]=!0),o="$jc$.scopes["+this.scope.id+"].locals['"+i+"'] = "+this.compile(t.children[1],e)+";\n")):o=(i=this.compile(t.children[0]))+" = "+this.compile(t.children[1],e)+";\n";break;case"op_if":o=" if ("+this.compile(t.children[0],e)+") "+this.compile(t.children[1],e);break;case"op_if_else":o=" if ("+this.compile(t.children[0],e)+")"+this.compile(t.children[1],e),o+=" else "+this.compile(t.children[2],e);break;case"op_conditional":o="(("+this.compile(t.children[0],e)+")?("+this.compile(t.children[1],e),o+="):("+this.compile(t.children[2],e)+"))";break;case"op_while":o=" while ("+this.compile(t.children[0],e)+") {\n"+this.compile(t.children[1],e)+"}\n";break;case"op_do":o=" do {\n"+this.compile(t.children[0],e)+"} while ("+this.compile(t.children[1],e)+");\n";break;case"op_for":o=" for ("+this.compile(t.children[0],e)+this.compile(t.children[1],e)+"; "+this.compile(t.children[2],e).slice(0,-2)+") {\n"+this.compile(t.children[3],e)+"\n}\n";break;case"op_proplst":t.children[0]&&(o=this.compile(t.children[0],e)+", "),o+=this.compile(t.children[1],e);break;case"op_prop":o=t.children[0]+": "+this.compile(t.children[1],e);break;case"op_emptyobject":o=e?"{}":"<< >>";break;case"op_proplst_val":o=this.compile(t.children[0],e);break;case"op_array":for(r=[],s=0;s<t.children[0].length;s++)r.push(this.compile(t.children[0][s],e));o="["+r.join(", ")+"]";break;case"op_extvalue":o=this.compile(t.children[0],e)+"["+this.compile(t.children[1],e)+"]";break;case"op_return":o=" return "+this.compile(t.children[0],e)+";\n";break;case"op_map":t.children[1].isMath||"node_var"===t.children[1].type||this._error("compile: In a map only function calls and mathematical expressions are allowed."),r=t.children[0],o=e?" $jc$.makeMap(function ("+r.join(", ")+") { return "+this.compile(t.children[1],e)+"; })":"map ("+r.join(", ")+") -> "+this.compile(t.children[1],e);break;case"op_function":r=t.children[0],this.pushScope(r),o=e?this.functionCodeJS(t):" function ("+r.join(", ")+") "+this.compile(t.children[1],e),this.popScope();break;case"op_execfunmath":console.log("op_execfunmath: TODO"),o="-1";break;case"op_execfun":if(t.children[2]){for(r=[],s=0;s<t.children[2].length;s++)r.push(this.compile(t.children[2][s],e));i=e?"$jc$.mergeAttributes("+r.join(", ")+")":r.join(", ")}for(t.children[0].withProps=!!t.children[2],r=[],s=0;s<t.children[1].length;s++)r.push(this.compile(t.children[1][s],e));o=this.compile(t.children[0],e)+"("+r.join(", ")+(t.children[2]&&e?", "+i:"")+")"+(t.children[2]&&!e?" "+i:""),e&&(o+="\n"),e&&"$"===t.children[0].value&&(o="$jc$.board.objects["+this.compile(t.children[1][0],e)+"]");break;case"op_property":o=e&&"X"!==t.children[1]&&"Y"!==t.children[1]?"$jc$.resolveProperty("+this.compile(t.children[0],e)+", '"+t.children[1]+"', true)":this.compile(t.children[0],e)+"."+t.children[1];break;case"op_use":this._warn("Use of the 'use' operator is deprecated."),o=e?"$jc$.use('":"use('",o+=t.children[0].toString()+"');";break;case"op_delete":this._warn("Use of the 'delete' operator is deprecated. Please use the remove() function."),o=e?"$jc$.del(":"remove(",o+=this.compile(t.children[0],e)+")";break;case"op_eq":o="("+this.compile(t.children[0],e)+" === "+this.compile(t.children[1],e)+")";break;case"op_neq":o="("+this.compile(t.children[0],e)+" !== "+this.compile(t.children[1],e)+")";break;case"op_approx":o="("+this.compile(t.children[0],e)+" ~= "+this.compile(t.children[1],e)+")";break;case"op_gt":o=e?"$jc$.gt("+this.compile(t.children[0],e)+", "+this.compile(t.children[1],e)+")":"("+this.compile(t.children[0],e)+" > "+this.compile(t.children[1],e)+")";break;case"op_lt":o=e?"$jc$.lt("+this.compile(t.children[0],e)+", "+this.compile(t.children[1],e)+")":"("+this.compile(t.children[0],e)+" < "+this.compile(t.children[1],e)+")";break;case"op_geq":o=e?"$jc$.geq("+this.compile(t.children[0],e)+", "+this.compile(t.children[1],e)+")":"("+this.compile(t.children[0],e)+" >= "+this.compile(t.children[1],e)+")";break;case"op_leq":o=e?"$jc$.leq("+this.compile(t.children[0],e)+", "+this.compile(t.children[1],e)+")":"("+this.compile(t.children[0],e)+" <= "+this.compile(t.children[1],e)+")";break;case"op_or":o="("+this.compile(t.children[0],e)+" || "+this.compile(t.children[1],e)+")";break;case"op_and":o="("+this.compile(t.children[0],e)+" && "+this.compile(t.children[1],e)+")";break;case"op_not":o="!("+this.compile(t.children[0],e)+")";break;case"op_add":o=e?"$jc$.add("+this.compile(t.children[0],e)+", "+this.compile(t.children[1],e)+")":"("+this.compile(t.children[0],e)+" + "+this.compile(t.children[1],e)+")";break;case"op_sub":o=e?"$jc$.sub("+this.compile(t.children[0],e)+", "+this.compile(t.children[1],e)+")":"("+this.compile(t.children[0],e)+" - "+this.compile(t.children[1],e)+")";break;case"op_div":o=e?"$jc$.div("+this.compile(t.children[0],e)+", "+this.compile(t.children[1],e)+")":"("+this.compile(t.children[0],e)+" / "+this.compile(t.children[1],e)+")";break;case"op_mod":o=e?"$jc$.mod("+this.compile(t.children[0],e)+", "+this.compile(t.children[1],e)+", true)":"("+this.compile(t.children[0],e)+" % "+this.compile(t.children[1],e)+")";break;case"op_mul":o=e?"$jc$.mul("+this.compile(t.children[0],e)+", "+this.compile(t.children[1],e)+")":"("+this.compile(t.children[0],e)+" * "+this.compile(t.children[1],e)+")";break;case"op_exp":o=e?"$jc$.pow("+this.compile(t.children[0],e)+", "+this.compile(t.children[1],e)+")":"("+this.compile(t.children[0],e)+"^"+this.compile(t.children[1],e)+")";break;case"op_neg":o=e?"$jc$.neg("+this.compile(t.children[0],e)+")":"(-"+this.compile(t.children[0],e)+")"}break;case"node_var":o=e?this.getvarJS(t.value,!1,t.withProps):t.value;break;case"node_const":case"node_const_bool":o=t.value;break;case"node_str":o="'"+t.value+"'"}return t.needsBrackets&&(o="{\n"+o+" }\n"),t.needsAngleBrackets&&(o=e?"{\n"+o+" }\n":"<< "+o+" >>\n"),o},getName:function(t,e){var i="";return _utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(t)&&_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(t.getName)?(i=t.getName(),_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(i)&&""!==i||!e||(i=t.id)):e&&(i=t.id),i},X:function(t){return t.X()},Y:function(t){return t.Y()},V:function(t){return t.Value()},L:function(t){return t.L()},area:function(t){return _utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(t)&&_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(t.Area)||this._error("Error: Can't calculate area."),t.Area()},perimeter:function(t){return _utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(t)&&_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(t.Perimeter)||this._error("Error: Can't calculate perimeter."),t.Perimeter()},dist:function(t,e){return _utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(t)&&_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(t.Dist)||this._error("Error: Can't calculate distance."),t.Dist(e)},radius:function(t){return _utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(t)&&_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(t.Radius)||this._error("Error: Can't calculate radius."),t.Radius()},slope:function(t){return _utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(t)&&_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(t.Slope)||this._error("Error: Can't calculate slope."),t.Slope()},add:function(t,e){var i,s,r;if(t=_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.evalSlider(t),e=_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.evalSlider(e),_math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.isInterval(t)||_math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.isInterval(e))r=_math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.add(t,e);else if(_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isArray(t)&&_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isArray(e))for(s=Math.min(t.length,e.length),r=[],i=0;i<s;i++)r[i]=t[i]+e[i];else _utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isNumber(t)&&_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isNumber(e)?r=t+e:_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isString(t)||_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isString(e)?r=t.toString()+e.toString():this._error("Operation + not defined on operands "+typeof t+" and "+typeof e);return r},sub:function(t,e){var i,s,r;if(t=_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.evalSlider(t),e=_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.evalSlider(e),_math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.isInterval(t)||_math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.isInterval(e))r=_math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.sub(t,e);else if(_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isArray(t)&&_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isArray(e))for(s=Math.min(t.length,e.length),r=[],i=0;i<s;i++)r[i]=t[i]-e[i];else _utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isNumber(t)&&_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isNumber(e)?r=t-e:this._error("Operation - not defined on operands "+typeof t+" and "+typeof e);return r},neg:function(t){var e,i,s;if(t=_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.evalSlider(t),_math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.isInterval(t))s=_math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.negative(t);else if(_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isArray(t))for(i=t.length,s=[],e=0;e<i;e++)s[e]=-t[e];else _utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isNumber(t)?s=-t:this._error("Unary operation - not defined on operand "+typeof t);return s},mul:function(t,e){var i,s,r;if(t=_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.evalSlider(t),e=_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.evalSlider(e),_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isArray(t)&&_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isNumber(e)&&(i=t,e=t=e),_math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.isInterval(t)||_math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.isInterval(e))r=_math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.mul(t,e);else if(_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isArray(t)&&_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isArray(e))s=Math.min(t.length,e.length),r=_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.innerProduct(t,e,s);else if(_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isNumber(t)&&_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isArray(e))for(s=e.length,r=[],i=0;i<s;i++)r[i]=t*e[i];else _utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isNumber(t)&&_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isNumber(e)?r=t*e:this._error("Operation * not defined on operands "+typeof t+" and "+typeof e);return r},div:function(t,e){var i,s,r;if(t=_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.evalSlider(t),e=_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.evalSlider(e),_math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.isInterval(t)||_math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.isInterval(e))r=_math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.div(t,e);else if(_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isArray(t)&&_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isNumber(e))for(s=t.length,r=[],i=0;i<s;i++)r[i]=t[i]/e;else _utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isNumber(t)&&_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isNumber(e)?r=t/e:this._error("Operation * not defined on operands "+typeof t+" and "+typeof e);return r},mod:function(t,e){var i,s,r;if(t=_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.evalSlider(t),e=_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.evalSlider(e),_math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.isInterval(t)||_math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.isInterval(e))return _math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.fmod(t,e);if(_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isArray(t)&&_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isNumber(e))for(s=t.length,r=[],i=0;i<s;i++)r[i]=_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.mod(t[i],e,!0);else _utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isNumber(t)&&_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.isNumber(e)?r=_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.mod(t,e,!0):this._error("Operation * not defined on operands "+typeof t+" and "+typeof e);return r},pow:function(t,e){return t=_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.evalSlider(t),e=_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.evalSlider(e),_math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.isInterval(t)||_math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.isInterval(e)?_math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.pow(t,e):_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.pow(t,e)},lt:function(t,e){return _math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.isInterval(t)||_math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.isInterval(e)?_math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.lt(t,e):t<e},leq:function(t,e){return _math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.isInterval(t)||_math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.isInterval(e)?_math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.leq(t,e):t<=e},gt:function(t,e){return _math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.isInterval(t)||_math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.isInterval(e)?_math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.gt(t,e):t>e},geq:function(t,e){return _math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.isInterval(t)||_math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.isInterval(e)?_math_ia_js__WEBPACK_IMPORTED_MODULE_4__.Z.geq(t,e):t>=e},randint:function(t,e,i){return _utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(i)||(i=1),Math.round(Math.random()*(e-t)/i)*i+t},DDD:function(t){console.log("Dummy derivative function. This should never appear!")},ifthen:function(t,e,i){return t?e:i},del:function(t){"object"==typeof t&&_jxg_js__WEBPACK_IMPORTED_MODULE_0__.Z.exists(t.type)&&_jxg_js__WEBPACK_IMPORTED_MODULE_0__.Z.exists(t.elementClass)&&this.board.removeObject(t)},eval:function(t){return _jxg_js__WEBPACK_IMPORTED_MODULE_0__.Z.evaluate(t)},use:function(t){var e,i,s=!1;if("string"==typeof t){for(e in _jxg_js__WEBPACK_IMPORTED_MODULE_0__.Z.boards)if(_jxg_js__WEBPACK_IMPORTED_MODULE_0__.Z.boards.hasOwnProperty(e)&&_jxg_js__WEBPACK_IMPORTED_MODULE_0__.Z.boards[e].container===t){i=_jxg_js__WEBPACK_IMPORTED_MODULE_0__.Z.boards[e],s=!0;break}}else i=t,s=!0;s?(this.board=i,this.builtIn.$board=i,this.builtIn.$board.src="$jc$.board"):this._error("Board '"+t+"' not found!")},findSymbol:function(t,e){var i,s;for(s=-1===(e=_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.def(e,-1))?this.scope:this.scopes[e];null!==s;){for(i in s.locals)if(s.locals.hasOwnProperty(i)&&s.locals[i]===t)return[i,s];s=s.previous}return[]},importModule:function(t){return priv.modules[t.toLowerCase()]},defineBuiltIn:function(){var t=this,e={PI:Math.PI,EULER:Math.E,D:t.DDD,X:t.X,Y:t.Y,V:t.V,Value:t.V,L:t.L,Length:t.L,acosh:_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.acosh,acot:_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.acot,asinh:_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.asinh,binomial:_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.binomial,cbrt:_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.cbrt,cosh:_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.cosh,cot:_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.cot,deg:_math_geometry_js__WEBPACK_IMPORTED_MODULE_5__.Z.trueAngle,A:t.area,area:t.area,Area:t.area,perimeter:t.perimeter,Perimeter:t.perimeter,dist:t.dist,Dist:t.dist,R:t.radius,radius:t.radius,Radius:t.radius,erf:_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.erf,erfc:_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.erfc,erfi:_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.erfi,factorial:_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.factorial,gcd:_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.gcd,lb:_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.log2,lcm:_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.lcm,ld:_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.log2,lg:_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.log10,ln:Math.log,log:_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.log,log10:_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.log10,log2:_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.log2,ndtr:_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.ndtr,ndtri:_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.ndtri,nthroot:_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.nthroot,pow:_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.pow,rad:_math_geometry_js__WEBPACK_IMPORTED_MODULE_5__.Z.rad,ratpow:_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.ratpow,trunc:_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.trunc,sinh:_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z.sinh,slope:t.slope,Slope:t.slope,randint:t.randint,IfThen:t.ifthen,import:t.importModule,eval:t.eval,use:t.use,remove:t.del,$:t.getElementById,$value:function(e){return t.getElementById(e).Value()},getName:t.getName,name:t.getName,$board:t.board,$log:t.log};return e.rad.sc=_math_geometry_js__WEBPACK_IMPORTED_MODULE_5__.Z,e.deg.sc=_math_geometry_js__WEBPACK_IMPORTED_MODULE_5__.Z,e.factorial.sc=_math_math_js__WEBPACK_IMPORTED_MODULE_3__.Z,e.X.src="$jc$.X",e.Y.src="$jc$.Y",e.V.src="$jc$.V",e.Value.src="$jc$.V",e.L.src="$jc$.L",e.Length.src="$jc$.L",e.acosh.src="JXG.Math.acosh",e.acot.src="JXG.Math.acot",e.asinh.src="JXG.Math.asinh",e.binomial.src="JXG.Math.binomial",e.cbrt.src="JXG.Math.cbrt",e.cot.src="JXG.Math.cot",e.cosh.src="JXG.Math.cosh",e.deg.src="JXG.Math.Geometry.trueAngle",e.erf.src="JXG.Math.erf",e.erfc.src="JXG.Math.erfc",e.erfi.src="JXG.Math.erfi",e.A.src="$jc$.area",e.area.src="$jc$.area",e.Area.src="$jc$.area",e.perimeter.src="$jc$.perimeter",e.Perimeter.src="$jc$.perimeter",e.dist.src="$jc$.dist",e.Dist.src="$jc$.dist",e.R.src="$jc$.radius",e.radius.src="$jc$.radius",e.Radius.src="$jc$.radius",e.factorial.src="JXG.Math.factorial",e.gcd.src="JXG.Math.gcd",e.lb.src="JXG.Math.log2",e.lcm.src="JXG.Math.lcm",e.ld.src="JXG.Math.log2",e.lg.src="JXG.Math.log10",e.ln.src="Math.log",e.log.src="JXG.Math.log",e.log10.src="JXG.Math.log10",e.log2.src="JXG.Math.log2",e.ndtr.src="JXG.Math.ndtr",e.ndtri.src="JXG.Math.ndtri",e.nthroot.src="JXG.Math.nthroot",e.pow.src="JXG.Math.pow",e.rad.src="JXG.Math.Geometry.rad",e.ratpow.src="JXG.Math.ratpow",e.trunc.src="JXG.trunc",e.sinh.src="JXG.Math.sinh",e.slope.src="$jc$.slope",e.Slope.src="$jc$.slope",e.randint.src="$jc$.randint",e.import.src="$jc$.importModule",e.eval.src="$jc$.eval",e.use.src="$jc$.use",e.remove.src="$jc$.del",e.IfThen.src="$jc$.ifthen",e.$.src="(function (n) { return $jc$.board.select(n); })",e.$value.src="(function (n) { return $jc$.board.select(n).Value(); })",e.getName.src="$jc$.getName",e.name.src="$jc$.getName",e.$board&&(e.$board.src="$jc$.board"),e.$log.src="$jc$.log",e=_jxg_js__WEBPACK_IMPORTED_MODULE_0__.Z.merge(e,t._addedBuiltIn)},_addedBuiltIn:{},addBuiltIn:function(t,e){if(_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(this.builtIn)){if(_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(this.builtIn[t]))return;this.builtIn[t]=e,this.builtIn[t].src="$jc$."+t}_utils_type_js__WEBPACK_IMPORTED_MODULE_7__.Z.exists(this._addedBuiltIn[t])||(this._addedBuiltIn[t]=e,this._addedBuiltIn[t].src="$jc$."+t,_jxg_js__WEBPACK_IMPORTED_MODULE_0__.Z.JessieCode.prototype[t]=e)},getPossibleOperands:function(){var t,e,i,s,r,o,n,a,h,l,c,d,u,p,f=["E"],_=this.builtIn||this.defineBuiltIn(),m=Math;for(u=function(t,e){return t.toLowerCase().localeCompare(e.toLowerCase())},p=function(t,e){var i=null;if("jc"===e)i=_[t];else{if("Math"!==e)return;i=m[t]}if(!(f.indexOf(t)>=0))return _jxg_js__WEBPACK_IMPORTED_MODULE_0__.Z.isFunction(i)?{name:t,type:"function",numParams:i.length,origin:e}:_jxg_js__WEBPACK_IMPORTED_MODULE_0__.Z.isNumber(i)?{name:t,type:"constant",value:i,origin:e}:void(t.startsWith("$")||void 0!==i&&console.error("undefined type",i))},t=Object.getOwnPropertyNames(_).sort(u),e=Object.getOwnPropertyNames(m).sort(u),i=[],s=0,r=0;s<t.length||r<e.length;)t[s]===e[r]?(o=p(e[r],"Math"),_jxg_js__WEBPACK_IMPORTED_MODULE_0__.Z.exists(o)&&i.push(o),s++,r++):!_jxg_js__WEBPACK_IMPORTED_MODULE_0__.Z.exists(e[r])||t[s].toLowerCase().localeCompare(e[r].toLowerCase())<0?(o=p(t[s],"jc"),_jxg_js__WEBPACK_IMPORTED_MODULE_0__.Z.exists(o)&&i.push(o),s++):(o=p(e[r],"Math"),_jxg_js__WEBPACK_IMPORTED_MODULE_0__.Z.exists(o)&&i.push(o),r++);for(h=[],l=[],c=[],d={},n=i.length,s=0;s<n;s++){switch((a=i[s]).type){case"function":h.push(a.name),"jc"===a.origin&&l.push(a.name);break;case"constant":c.push(a.name)}d[a.name]=a}return{all:d,list:i,functions:h,functions_jessiecode:l,constants:c}},_debug:function(t){"object"==typeof console?console.log(t):_utils_env_js__WEBPACK_IMPORTED_MODULE_8__.Z.isBrowser&&document&&null!==document.getElementById("debug")&&(document.getElementById("debug").innerHTML+=t+"<br />")},_error:function(t){var e=new Error("Error("+this.line+"): "+t);throw e.line=this.line,e},_warn:function(t){"object"==typeof console?console.log("Warning("+this.line+"): "+t):_utils_env_js__WEBPACK_IMPORTED_MODULE_8__.Z.isBrowser&&document&&null!==document.getElementById(this.warnLog)&&(document.getElementById(this.warnLog).innerHTML+="Warning("+this.line+"): "+t+"<br />")},_log:function(t){"object"!=typeof window&&"object"==typeof self&&self.postMessage?self.postMessage({type:"log",msg:"Log: "+t.toString()}):console.log("Log: ",arguments)}});var parser=function(){var t=function(t,e,i,s){for(i=i||{},s=t.length;s--;i[t[s]]=e);return i},e=[2,14],i=[1,13],s=[1,37],r=[1,14],o=[1,15],n=[1,21],a=[1,16],h=[1,17],l=[1,33],c=[1,18],d=[1,19],u=[1,12],p=[1,59],f=[1,60],_=[1,58],m=[1,46],g=[1,48],b=[1,49],v=[1,50],Z=[1,51],C=[1,52],y=[1,53],P=[1,54],E=[1,45],x=[1,38],O=[1,39],M=[5,7,8,14,15,16,17,19,20,21,23,26,27,50,51,58,65,74,75,76,77,78,79,80,82,91,93],w=[5,7,8,12,14,15,16,17,19,20,21,23,26,27,50,51,58,65,74,75,76,77,78,79,80,82,91,93],S=[8,10,16,32,34,35,37,39,41,42,43,45,46,47,48,50,51,53,54,55,57,64,65,66,83,86],T=[2,48],N=[1,72],A=[10,16,32,34,35,37,39,41,42,43,45,46,47,48,50,51,53,54,55,57,66,83,86],D=[1,78],k=[8,10,16,32,34,35,37,41,42,43,45,46,47,48,50,51,53,54,55,57,64,65,66,83,86],R=[1,82],L=[8,10,16,32,34,35,37,39,45,46,47,48,50,51,53,54,55,57,64,65,66,83,86],B=[1,83],I=[1,84],j=[1,85],Y=[8,10,16,32,34,35,37,39,41,42,43,50,51,53,54,55,57,64,65,66,83,86],V=[1,89],X=[1,90],U=[1,91],F=[1,92],z=[1,97],J=[8,10,16,32,34,35,37,39,41,42,43,45,46,47,48,53,54,55,57,64,65,66,83,86],G=[1,103],W=[1,104],H=[8,10,16,32,34,35,37,39,41,42,43,45,46,47,48,50,51,57,64,65,66,83,86],K=[1,105],$=[1,106],q=[1,107],Q=[1,126],tt=[1,139],et=[83,86],it=[1,150],st=[10,66,86],rt=[8,10,16,20,32,34,35,37,39,41,42,43,45,46,47,48,50,51,53,54,55,57,64,65,66,82,83,86],ot=[1,167],nt=[10,86],at={trace:function(){},yy:{},symbols_:{error:2,Program:3,StatementList:4,EOF:5,IfStatement:6,IF:7,"(":8,Expression:9,")":10,Statement:11,ELSE:12,LoopStatement:13,WHILE:14,FOR:15,";":16,DO:17,UnaryStatement:18,USE:19,IDENTIFIER:20,DELETE:21,ReturnStatement:22,RETURN:23,EmptyStatement:24,StatementBlock:25,"{":26,"}":27,ExpressionStatement:28,AssignmentExpression:29,ConditionalExpression:30,LeftHandSideExpression:31,"=":32,LogicalORExpression:33,"?":34,":":35,LogicalANDExpression:36,"||":37,EqualityExpression:38,"&&":39,RelationalExpression:40,"==":41,"!=":42,"~=":43,AdditiveExpression:44,"<":45,">":46,"<=":47,">=":48,MultiplicativeExpression:49,"+":50,"-":51,UnaryExpression:52,"*":53,"/":54,"%":55,ExponentExpression:56,"^":57,"!":58,MemberExpression:59,CallExpression:60,PrimaryExpression:61,FunctionExpression:62,MapExpression:63,".":64,"[":65,"]":66,BasicLiteral:67,ObjectLiteral:68,ArrayLiteral:69,NullLiteral:70,BooleanLiteral:71,StringLiteral:72,NumberLiteral:73,NULL:74,TRUE:75,FALSE:76,STRING:77,NUMBER:78,NAN:79,INFINITY:80,ElementList:81,"<<":82,">>":83,PropertyList:84,Property:85,",":86,PropertyName:87,Arguments:88,AttributeList:89,Attribute:90,FUNCTION:91,ParameterDefinitionList:92,MAP:93,"->":94,$accept:0,$end:1},terminals_:{2:"error",5:"EOF",7:"IF",8:"(",10:")",12:"ELSE",14:"WHILE",15:"FOR",16:";",17:"DO",19:"USE",20:"IDENTIFIER",21:"DELETE",23:"RETURN",26:"{",27:"}",32:"=",34:"?",35:":",37:"||",39:"&&",41:"==",42:"!=",43:"~=",45:"<",46:">",47:"<=",48:">=",50:"+",51:"-",53:"*",54:"/",55:"%",57:"^",58:"!",64:".",65:"[",66:"]",74:"NULL",75:"TRUE",76:"FALSE",77:"STRING",78:"NUMBER",79:"NAN",80:"INFINITY",82:"<<",83:">>",86:",",91:"FUNCTION",93:"MAP",94:"->"},productions_:[0,[3,2],[6,5],[6,7],[13,5],[13,9],[13,7],[18,2],[18,2],[22,2],[22,3],[24,1],[25,3],[4,2],[4,0],[11,1],[11,1],[11,1],[11,1],[11,1],[11,1],[11,1],[28,2],[9,1],[29,1],[29,3],[30,1],[30,5],[33,1],[33,3],[36,1],[36,3],[38,1],[38,3],[38,3],[38,3],[40,1],[40,3],[40,3],[40,3],[40,3],[44,1],[44,3],[44,3],[49,1],[49,3],[49,3],[49,3],[56,1],[56,3],[52,1],[52,2],[52,2],[52,2],[31,1],[31,1],[59,1],[59,1],[59,1],[59,3],[59,4],[61,1],[61,1],[61,1],[61,1],[61,3],[67,1],[67,1],[67,1],[67,1],[70,1],[71,1],[71,1],[72,1],[73,1],[73,1],[73,1],[69,2],[69,3],[68,2],[68,3],[84,1],[84,3],[85,3],[87,1],[87,1],[87,1],[60,2],[60,3],[60,2],[60,4],[60,3],[88,2],[88,3],[89,1],[89,3],[90,1],[90,1],[81,1],[81,3],[62,4],[62,5],[63,5],[63,6],[92,1],[92,3]],performAction:function(t,e,i,s,r,o,n){var a=o.length-1;switch(r){case 1:return o[a-1];case 2:this.$=ht.createNode(lt(n[a-4]),"node_op","op_if",o[a-2],o[a]);break;case 3:this.$=ht.createNode(lt(n[a-6]),"node_op","op_if_else",o[a-4],o[a-2],o[a]);break;case 4:this.$=ht.createNode(lt(n[a-4]),"node_op","op_while",o[a-2],o[a]);break;case 5:this.$=ht.createNode(lt(n[a-8]),"node_op","op_for",o[a-6],o[a-4],o[a-2],o[a]);break;case 6:this.$=ht.createNode(lt(n[a-6]),"node_op","op_do",o[a-5],o[a-2]);break;case 7:this.$=ht.createNode(lt(n[a-1]),"node_op","op_use",o[a]);break;case 8:this.$=ht.createNode(lt(n[a-1]),"node_op","op_delete",o[a]);break;case 9:this.$=ht.createNode(lt(n[a-1]),"node_op","op_return",void 0);break;case 10:this.$=ht.createNode(lt(n[a-2]),"node_op","op_return",o[a-1]);break;case 11:case 14:this.$=ht.createNode(lt(n[a]),"node_op","op_none");break;case 12:this.$=o[a-1],this.$.needsBrackets=!0;break;case 13:this.$=ht.createNode(lt(n[a-1]),"node_op","op_none",o[a-1],o[a]);break;case 15:case 16:case 17:case 18:case 19:case 20:case 21:case 23:case 24:case 26:case 28:case 30:case 32:case 36:case 41:case 44:case 48:case 50:case 52:case 54:case 55:case 56:case 58:case 62:case 81:case 84:case 85:case 86:this.$=o[a];break;case 22:case 65:case 93:this.$=o[a-1];break;case 25:this.$=ht.createNode(lt(n[a-2]),"node_op","op_assign",o[a-2],o[a]),this.$.isMath=!1;break;case 27:this.$=ht.createNode(lt(n[a-4]),"node_op","op_conditional",o[a-4],o[a-2],o[a]),this.$.isMath=!1;break;case 29:this.$=ht.createNode(lt(n[a-2]),"node_op","op_or",o[a-2],o[a]),this.$.isMath=!1;break;case 31:this.$=ht.createNode(lt(n[a-2]),"node_op","op_and",o[a-2],o[a]),this.$.isMath=!1;break;case 33:this.$=ht.createNode(lt(n[a-2]),"node_op","op_eq",o[a-2],o[a]),this.$.isMath=!1;break;case 34:this.$=ht.createNode(lt(n[a-2]),"node_op","op_neq",o[a-2],o[a]),this.$.isMath=!1;break;case 35:this.$=ht.createNode(lt(n[a-2]),"node_op","op_approx",o[a-2],o[a]),this.$.isMath=!1;break;case 37:this.$=ht.createNode(lt(n[a-2]),"node_op","op_lt",o[a-2],o[a]),this.$.isMath=!1;break;case 38:this.$=ht.createNode(lt(n[a-2]),"node_op","op_gt",o[a-2],o[a]),this.$.isMath=!1;break;case 39:this.$=ht.createNode(lt(n[a-2]),"node_op","op_leq",o[a-2],o[a]),this.$.isMath=!1;break;case 40:this.$=ht.createNode(lt(n[a-2]),"node_op","op_geq",o[a-2],o[a]),this.$.isMath=!1;break;case 42:this.$=ht.createNode(lt(n[a-2]),"node_op","op_add",o[a-2],o[a]),this.$.isMath=!0;break;case 43:this.$=ht.createNode(lt(n[a-2]),"node_op","op_sub",o[a-2],o[a]),this.$.isMath=!0;break;case 45:this.$=ht.createNode(lt(n[a-2]),"node_op","op_mul",o[a-2],o[a]),this.$.isMath=!0;break;case 46:this.$=ht.createNode(lt(n[a-2]),"node_op","op_div",o[a-2],o[a]),this.$.isMath=!0;break;case 47:this.$=ht.createNode(lt(n[a-2]),"node_op","op_mod",o[a-2],o[a]),this.$.isMath=!0;break;case 49:this.$=ht.createNode(lt(n[a-2]),"node_op","op_exp",o[a-2],o[a]),this.$.isMath=!0;break;case 51:this.$=ht.createNode(lt(n[a-1]),"node_op","op_not",o[a]),this.$.isMath=!1;break;case 53:this.$=ht.createNode(lt(n[a-1]),"node_op","op_neg",o[a]),this.$.isMath=!0;break;case 57:case 63:case 64:case 66:case 67:case 68:case 97:this.$=o[a],this.$.isMath=!1;break;case 59:case 91:this.$=ht.createNode(lt(n[a-2]),"node_op","op_property",o[a-2],o[a]),this.$.isMath=!0;break;case 60:case 90:this.$=ht.createNode(lt(n[a-3]),"node_op","op_extvalue",o[a-3],o[a-1]),this.$.isMath=!0;break;case 61:this.$=ht.createNode(lt(n[a]),"node_var",o[a]);break;case 69:this.$=o[a],this.$.isMath=!0;break;case 70:this.$=ht.createNode(lt(n[a]),"node_const",null);break;case 71:this.$=ht.createNode(lt(n[a]),"node_const_bool",!0);break;case 72:this.$=ht.createNode(lt(n[a]),"node_const_bool",!1);break;case 73:this.$=ht.createNode(lt(n[a]),"node_str",o[a].substring(1,o[a].length-1));break;case 74:this.$=ht.createNode(lt(n[a]),"node_const",parseFloat(o[a]));break;case 75:this.$=ht.createNode(lt(n[a]),"node_const",NaN);break;case 76:this.$=ht.createNode(lt(n[a]),"node_const",1/0);break;case 77:this.$=ht.createNode(lt(n[a-1]),"node_op","op_array",[]);break;case 78:this.$=ht.createNode(lt(n[a-2]),"node_op","op_array",o[a-1]);break;case 79:this.$=ht.createNode(lt(n[a-1]),"node_op","op_emptyobject",{}),this.$.needsAngleBrackets=!0;break;case 80:this.$=ht.createNode(lt(n[a-2]),"node_op","op_proplst_val",o[a-1]),this.$.needsAngleBrackets=!0;break;case 82:this.$=ht.createNode(lt(n[a-2]),"node_op","op_proplst",o[a-2],o[a]);break;case 83:this.$=ht.createNode(lt(n[a-2]),"node_op","op_prop",o[a-2],o[a]);break;case 87:case 89:this.$=ht.createNode(lt(n[a-1]),"node_op","op_execfun",o[a-1],o[a]),this.$.isMath=!0;break;case 88:this.$=ht.createNode(lt(n[a-2]),"node_op","op_execfun",o[a-2],o[a-1],o[a],!0),this.$.isMath=!1;break;case 92:this.$=[];break;case 94:case 98:case 104:this.$=[o[a]];break;case 95:case 99:case 105:this.$=o[a-2].concat(o[a]);break;case 96:this.$=ht.createNode(lt(n[a]),"node_var",o[a]),this.$.isMath=!0;break;case 100:this.$=ht.createNode(lt(n[a-3]),"node_op","op_function",[],o[a]),this.$.isMath=!1;break;case 101:this.$=ht.createNode(lt(n[a-4]),"node_op","op_function",o[a-2],o[a]),this.$.isMath=!1;break;case 102:this.$=ht.createNode(lt(n[a-4]),"node_op","op_map",[],o[a]);break;case 103:this.$=ht.createNode(lt(n[a-5]),"node_op","op_map",o[a-3],o[a])}},table:[t([5,7,8,14,15,16,17,19,20,21,23,26,50,51,58,65,74,75,76,77,78,79,80,82,91,93],e,{3:1,4:2}),{1:[3]},{5:[1,3],6:6,7:i,8:s,9:20,11:4,13:7,14:r,15:o,16:n,17:a,18:8,19:h,20:l,21:c,22:9,23:d,24:11,25:5,26:u,28:10,29:22,30:23,31:24,33:25,36:28,38:32,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{1:[2,1]},t(M,[2,13]),t(w,[2,15]),t(w,[2,16]),t(w,[2,17]),t(w,[2,18]),t(w,[2,19]),t(w,[2,20]),t(w,[2,21]),t([7,8,14,15,16,17,19,20,21,23,26,27,50,51,58,65,74,75,76,77,78,79,80,82,91,93],e,{4:61}),{8:[1,62]},{8:[1,63]},{8:[1,64]},{6:6,7:i,8:s,9:20,11:65,13:7,14:r,15:o,16:n,17:a,18:8,19:h,20:l,21:c,22:9,23:d,24:11,25:5,26:u,28:10,29:22,30:23,31:24,33:25,36:28,38:32,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{20:[1,66]},{20:[1,67]},{8:s,9:69,16:[1,68],20:l,29:22,30:23,31:24,33:25,36:28,38:32,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{16:[1,70]},t(w,[2,11]),t(S,[2,23]),t(S,[2,24]),t([8,10,16,34,35,37,39,41,42,43,45,46,47,48,50,51,53,54,55,64,65,66,83,86],T,{32:[1,71],57:N}),t([8,10,16,32,35,39,41,42,43,45,46,47,48,50,51,53,54,55,57,64,65,66,83,86],[2,26],{34:[1,73],37:[1,74]}),t(A,[2,54],{88:77,8:D,64:[1,75],65:[1,76]}),t(A,[2,55],{88:79,8:D,64:[1,81],65:[1,80]}),t(k,[2,28],{39:R}),t(S,[2,56]),t(S,[2,57]),t(S,[2,58]),t(L,[2,30],{41:B,42:I,43:j}),t(S,[2,61]),t(S,[2,62]),t(S,[2,63]),t(S,[2,64]),{8:s,9:86,20:l,29:22,30:23,31:24,33:25,36:28,38:32,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{8:[1,87]},{8:[1,88]},t(Y,[2,32],{45:V,46:X,47:U,48:F}),t(S,[2,66]),t(S,[2,67]),t(S,[2,68]),t(S,[2,69]),{20:z,72:98,73:99,77:Z,78:C,79:y,80:P,83:[1,93],84:94,85:95,87:96},{8:s,20:l,29:102,30:23,31:24,33:25,36:28,38:32,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,66:[1,100],67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,81:101,82:E,91:x,93:O},t(J,[2,36],{50:G,51:W}),t(S,[2,70]),t(S,[2,71]),t(S,[2,72]),t(S,[2,73]),t(S,[2,74]),t(S,[2,75]),t(S,[2,76]),t(H,[2,41],{53:K,54:$,55:q}),t(S,[2,44]),t(S,[2,50]),{8:s,20:l,31:109,50:p,51:f,52:108,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{8:s,20:l,31:109,50:p,51:f,52:110,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{8:s,20:l,31:109,50:p,51:f,52:111,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{6:6,7:i,8:s,9:20,11:4,13:7,14:r,15:o,16:n,17:a,18:8,19:h,20:l,21:c,22:9,23:d,24:11,25:5,26:u,27:[1,112],28:10,29:22,30:23,31:24,33:25,36:28,38:32,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{8:s,9:113,20:l,29:22,30:23,31:24,33:25,36:28,38:32,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{8:s,9:114,20:l,29:22,30:23,31:24,33:25,36:28,38:32,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{8:s,9:115,20:l,29:22,30:23,31:24,33:25,36:28,38:32,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{14:[1,116]},t(w,[2,7]),t(w,[2,8]),t(w,[2,9]),{16:[1,117]},t(w,[2,22]),{8:s,20:l,29:118,30:23,31:24,33:25,36:28,38:32,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{8:s,20:l,31:109,50:p,51:f,52:119,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{8:s,20:l,29:120,30:23,31:24,33:25,36:28,38:32,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{8:s,20:l,31:109,36:121,38:32,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{20:[1,122]},{8:s,9:123,20:l,29:22,30:23,31:24,33:25,36:28,38:32,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},t(S,[2,87],{89:124,90:125,68:127,20:Q,82:E}),{8:s,10:[1,128],20:l,29:102,30:23,31:24,33:25,36:28,38:32,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,81:129,82:E,91:x,93:O},t(S,[2,89]),{8:s,9:130,20:l,29:22,30:23,31:24,33:25,36:28,38:32,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{20:[1,131]},{8:s,20:l,31:109,38:132,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{8:s,20:l,31:109,40:133,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{8:s,20:l,31:109,40:134,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{8:s,20:l,31:109,40:135,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{10:[1,136]},{10:[1,137],20:tt,92:138},{10:[1,140],20:tt,92:141},{8:s,20:l,31:109,44:142,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{8:s,20:l,31:109,44:143,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{8:s,20:l,31:109,44:144,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{8:s,20:l,31:109,44:145,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},t(S,[2,79]),{83:[1,146],86:[1,147]},t(et,[2,81]),{35:[1,148]},{35:[2,84]},{35:[2,85]},{35:[2,86]},t(S,[2,77]),{66:[1,149],86:it},t(st,[2,98]),{8:s,20:l,31:109,49:151,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{8:s,20:l,31:109,49:152,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{8:s,20:l,31:109,50:p,51:f,52:153,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{8:s,20:l,31:109,50:p,51:f,52:154,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{8:s,20:l,31:109,50:p,51:f,52:155,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},t(S,[2,51]),t([8,10,16,32,34,35,37,39,41,42,43,45,46,47,48,50,51,53,54,55,64,65,66,83,86],T,{57:N}),t(S,[2,52]),t(S,[2,53]),t([5,7,8,10,12,14,15,16,17,19,20,21,23,26,27,32,34,35,37,39,41,42,43,45,46,47,48,50,51,53,54,55,57,58,64,65,66,74,75,76,77,78,79,80,82,83,86,91,93],[2,12]),{10:[1,156]},{10:[1,157]},{16:[1,158]},{8:[1,159]},t(w,[2,10]),t(S,[2,25]),t(S,[2,49]),{35:[1,160]},t(k,[2,29],{39:R}),t(S,[2,59]),{66:[1,161]},t([8,10,16,32,34,35,37,39,41,42,43,45,46,47,48,50,51,53,54,55,57,64,65,66,83],[2,88],{86:[1,162]}),t(S,[2,94]),t(S,[2,96]),t(S,[2,97]),t(rt,[2,92]),{10:[1,163],86:it},{66:[1,164]},t(S,[2,91]),t(L,[2,31],{41:B,42:I,43:j}),t(Y,[2,33],{45:V,46:X,47:U,48:F}),t(Y,[2,34],{45:V,46:X,47:U,48:F}),t(Y,[2,35],{45:V,46:X,47:U,48:F}),t(S,[2,65]),{25:165,26:u},{10:[1,166],86:ot},t(nt,[2,104]),{94:[1,168]},{10:[1,169],86:ot},t(J,[2,37],{50:G,51:W}),t(J,[2,38],{50:G,51:W}),t(J,[2,39],{50:G,51:W}),t(J,[2,40],{50:G,51:W}),t(S,[2,80]),{20:z,72:98,73:99,77:Z,78:C,79:y,80:P,85:170,87:96},{8:s,20:l,29:171,30:23,31:24,33:25,36:28,38:32,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},t(S,[2,78]),{8:s,20:l,29:172,30:23,31:24,33:25,36:28,38:32,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},t(H,[2,42],{53:K,54:$,55:q}),t(H,[2,43],{53:K,54:$,55:q}),t(S,[2,45]),t(S,[2,46]),t(S,[2,47]),{6:6,7:i,8:s,9:20,11:173,13:7,14:r,15:o,16:n,17:a,18:8,19:h,20:l,21:c,22:9,23:d,24:11,25:5,26:u,28:10,29:22,30:23,31:24,33:25,36:28,38:32,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{6:6,7:i,8:s,9:20,11:174,13:7,14:r,15:o,16:n,17:a,18:8,19:h,20:l,21:c,22:9,23:d,24:11,25:5,26:u,28:10,29:22,30:23,31:24,33:25,36:28,38:32,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{8:s,9:175,20:l,29:22,30:23,31:24,33:25,36:28,38:32,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{8:s,9:176,20:l,29:22,30:23,31:24,33:25,36:28,38:32,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{8:s,20:l,29:177,30:23,31:24,33:25,36:28,38:32,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},t(S,[2,60]),{20:Q,68:127,82:E,90:178},t(rt,[2,93]),t(S,[2,90]),t(S,[2,100]),{25:179,26:u},{20:[1,180]},{8:s,9:181,20:l,29:22,30:23,31:24,33:25,36:28,38:32,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{94:[1,182]},t(et,[2,82]),t(et,[2,83]),t(st,[2,99]),t(M,[2,2],{12:[1,183]}),t(w,[2,4]),{16:[1,184]},{10:[1,185]},t(S,[2,27]),t(S,[2,95]),t(S,[2,101]),t(nt,[2,105]),t(S,[2,102]),{8:s,9:186,20:l,29:22,30:23,31:24,33:25,36:28,38:32,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{6:6,7:i,8:s,9:20,11:187,13:7,14:r,15:o,16:n,17:a,18:8,19:h,20:l,21:c,22:9,23:d,24:11,25:5,26:u,28:10,29:22,30:23,31:24,33:25,36:28,38:32,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{8:s,9:188,20:l,29:22,30:23,31:24,33:25,36:28,38:32,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},{16:[1,189]},t(S,[2,103]),t(w,[2,3]),{10:[1,190]},t(w,[2,6]),{6:6,7:i,8:s,9:20,11:191,13:7,14:r,15:o,16:n,17:a,18:8,19:h,20:l,21:c,22:9,23:d,24:11,25:5,26:u,28:10,29:22,30:23,31:24,33:25,36:28,38:32,40:40,44:47,49:55,50:p,51:f,52:56,56:57,58:_,59:26,60:27,61:29,62:30,63:31,65:m,67:34,68:35,69:36,70:41,71:42,72:43,73:44,74:g,75:b,76:v,77:Z,78:C,79:y,80:P,82:E,91:x,93:O},t(w,[2,5])],defaultActions:{3:[2,1],97:[2,84],98:[2,85],99:[2,86]},parseError:function(t,e){if(!e.recoverable){var i=new Error(t);throw i.hash=e,i}this.trace(t)},parse:function(t){var e=this,i=[0],s=[null],r=[],o=this.table,n="",a=0,h=0,l=0,c=r.slice.call(arguments,1),d=Object.create(this.lexer),u={yy:{}};for(var p in this.yy)Object.prototype.hasOwnProperty.call(this.yy,p)&&(u.yy[p]=this.yy[p]);d.setInput(t,u.yy),u.yy.lexer=d,u.yy.parser=this,void 0===d.yylloc&&(d.yylloc={});var f=d.yylloc;r.push(f);var _=d.options&&d.options.ranges;"function"==typeof u.yy.parseError?this.parseError=u.yy.parseError:this.parseError=Object.getPrototypeOf(this).parseError;for(var m,g,b,v,Z,C,y,P,E,x=function(){var t;return"number"!=typeof(t=d.lex()||1)&&(t=e.symbols_[t]||t),t},O={};;){if(b=i[i.length-1],this.defaultActions[b]?v=this.defaultActions[b]:(null==m&&(m=x()),v=o[b]&&o[b][m]),void 0===v||!v.length||!v[0]){var M="";for(C in E=[],o[b])this.terminals_[C]&&C>2&&E.push("'"+this.terminals_[C]+"'");M=d.showPosition?"Parse error on line "+(a+1)+":\n"+d.showPosition()+"\nExpecting "+E.join(", ")+", got '"+(this.terminals_[m]||m)+"'":"Parse error on line "+(a+1)+": Unexpected "+(1==m?"end of input":"'"+(this.terminals_[m]||m)+"'"),this.parseError(M,{text:d.match,token:this.terminals_[m]||m,line:d.yylineno,loc:f,expected:E})}if(v[0]instanceof Array&&v.length>1)throw new Error("Parse Error: multiple actions possible at state: "+b+", token: "+m);switch(v[0]){case 1:i.push(m),s.push(d.yytext),r.push(d.yylloc),i.push(v[1]),m=null,g?(m=g,g=null):(h=d.yyleng,n=d.yytext,a=d.yylineno,f=d.yylloc,l>0&&l--);break;case 2:if(y=this.productions_[v[1]][1],O.$=s[s.length-y],O._$={first_line:r[r.length-(y||1)].first_line,last_line:r[r.length-1].last_line,first_column:r[r.length-(y||1)].first_column,last_column:r[r.length-1].last_column},_&&(O._$.range=[r[r.length-(y||1)].range[0],r[r.length-1].range[1]]),void 0!==(Z=this.performAction.apply(O,[n,h,a,u.yy,v[1],s,r].concat(c))))return Z;y&&(i=i.slice(0,-1*y*2),s=s.slice(0,-1*y),r=r.slice(0,-1*y)),i.push(this.productions_[v[1]][0]),s.push(O.$),r.push(O._$),P=o[i[i.length-2]][i[i.length-1]],i.push(P);break;case 3:return!0}}return!0}},ht={node:function(t,e,i){return{type:t,value:e,children:i}},createNode:function(t,e,i,s){var r,o=this.node(e,i,[]);for(r=3;r<arguments.length;r++)o.children.push(arguments[r]);return o.line=t[0],o.col=t[1],o.eline=t[2],o.ecol=t[3],o}},lt=function(t){return[t.first_line,t.first_column,t.last_line,t.last_column]},ct=function(){var t={EOF:1,parseError:function(t,e){if(!this.yy.parser)throw new Error(t);this.yy.parser.parseError(t,e)},setInput:function(t,e){return this.yy=e||this.yy||{},this._input=t,this._more=this._backtrack=this.done=!1,this.yylineno=this.yyleng=0,this.yytext=this.matched=this.match="",this.conditionStack=["INITIAL"],this.yylloc={first_line:1,first_column:0,last_line:1,last_column:0},this.options.ranges&&(this.yylloc.range=[0,0]),this.offset=0,this},input:function(){var t=this._input[0];return this.yytext+=t,this.yyleng++,this.offset++,this.match+=t,this.matched+=t,t.match(/(?:\r\n?|\n).*/g)?(this.yylineno++,this.yylloc.last_line++):this.yylloc.last_column++,this.options.ranges&&this.yylloc.range[1]++,this._input=this._input.slice(1),t},unput:function(t){var e=t.length,i=t.split(/(?:\r\n?|\n)/g);this._input=t+this._input,this.yytext=this.yytext.substr(0,this.yytext.length-e),this.offset-=e;var s=this.match.split(/(?:\r\n?|\n)/g);this.match=this.match.substr(0,this.match.length-1),this.matched=this.matched.substr(0,this.matched.length-1),i.length-1&&(this.yylineno-=i.length-1);var r=this.yylloc.range;return this.yylloc={first_line:this.yylloc.first_line,last_line:this.yylineno+1,first_column:this.yylloc.first_column,last_column:i?(i.length===s.length?this.yylloc.first_column:0)+s[s.length-i.length].length-i[0].length:this.yylloc.first_column-e},this.options.ranges&&(this.yylloc.range=[r[0],r[0]+this.yyleng-e]),this.yyleng=this.yytext.length,this},more:function(){return this._more=!0,this},reject:function(){return this.options.backtrack_lexer?(this._backtrack=!0,this):this.parseError("Lexical error on line "+(this.yylineno+1)+". You can only invoke reject() in the lexer when the lexer is of the backtracking persuasion (options.backtrack_lexer = true).\n"+this.showPosition(),{text:"",token:null,line:this.yylineno})},less:function(t){this.unput(this.match.slice(t))},pastInput:function(){var t=this.matched.substr(0,this.matched.length-this.match.length);return(t.length>20?"...":"")+t.substr(-20).replace(/\n/g,"")},upcomingInput:function(){var t=this.match;return t.length<20&&(t+=this._input.substr(0,20-t.length)),(t.substr(0,20)+(t.length>20?"...":"")).replace(/\n/g,"")},showPosition:function(){var t=this.pastInput(),e=new Array(t.length+1).join("-");return t+this.upcomingInput()+"\n"+e+"^"},test_match:function(t,e){var i,s,r;if(this.options.backtrack_lexer&&(r={yylineno:this.yylineno,yylloc:{first_line:this.yylloc.first_line,last_line:this.last_line,first_column:this.yylloc.first_column,last_column:this.yylloc.last_column},yytext:this.yytext,match:this.match,matches:this.matches,matched:this.matched,yyleng:this.yyleng,offset:this.offset,_more:this._more,_input:this._input,yy:this.yy,conditionStack:this.conditionStack.slice(0),done:this.done},this.options.ranges&&(r.yylloc.range=this.yylloc.range.slice(0))),(s=t[0].match(/(?:\r\n?|\n).*/g))&&(this.yylineno+=s.length),this.yylloc={first_line:this.yylloc.last_line,last_line:this.yylineno+1,first_column:this.yylloc.last_column,last_column:s?s[s.length-1].length-s[s.length-1].match(/\r?\n?/)[0].length:this.yylloc.last_column+t[0].length},this.yytext+=t[0],this.match+=t[0],this.matches=t,this.yyleng=this.yytext.length,this.options.ranges&&(this.yylloc.range=[this.offset,this.offset+=this.yyleng]),this._more=!1,this._backtrack=!1,this._input=this._input.slice(t[0].length),this.matched+=t[0],i=this.performAction.call(this,this.yy,this,e,this.conditionStack[this.conditionStack.length-1]),this.done&&this._input&&(this.done=!1),i)return i;if(this._backtrack){for(var o in r)this[o]=r[o];return!1}return!1},next:function(){if(this.done)return this.EOF;var t,e,i,s;this._input||(this.done=!0),this._more||(this.yytext="",this.match="");for(var r=this._currentRules(),o=0;o<r.length;o++)if((i=this._input.match(this.rules[r[o]]))&&(!e||i[0].length>e[0].length)){if(e=i,s=o,this.options.backtrack_lexer){if(!1!==(t=this.test_match(i,r[o])))return t;if(this._backtrack){e=!1;continue}return!1}if(!this.options.flex)break}return e?!1!==(t=this.test_match(e,r[s]))&&t:""===this._input?this.EOF:this.parseError("Lexical error on line "+(this.yylineno+1)+". Unrecognized text.\n"+this.showPosition(),{text:"",token:null,line:this.yylineno})},lex:function(){var t=this.next();return t||this.lex()},begin:function(t){this.conditionStack.push(t)},popState:function(){return this.conditionStack.length-1>0?this.conditionStack.pop():this.conditionStack[0]},_currentRules:function(){return this.conditionStack.length&&this.conditionStack[this.conditionStack.length-1]?this.conditions[this.conditionStack[this.conditionStack.length-1]].rules:this.conditions.INITIAL.rules},topState:function(t){return(t=this.conditionStack.length-1-Math.abs(t||0))>=0?this.conditionStack[t]:"INITIAL"},pushState:function(t){this.begin(t)},stateStackSize:function(){return this.conditionStack.length},options:{},performAction:function(t,e,i,s){switch(i){case 0:case 6:case 7:break;case 1:case 2:case 3:return 78;case 4:case 5:return 77;case 8:return 7;case 9:return 12;case 10:return 14;case 11:return 17;case 12:return 15;case 13:return 91;case 14:return 93;case 15:return 19;case 16:return 23;case 17:return 21;case 18:return 75;case 19:return 76;case 20:return 74;case 21:return 80;case 22:case 23:return 94;case 24:return 82;case 25:return 83;case 26:return 26;case 27:return 27;case 28:return 16;case 29:return"#";case 30:return 34;case 31:return 35;case 32:return 79;case 33:return 64;case 34:return 65;case 35:return 66;case 36:return 8;case 37:return 10;case 38:return 58;case 39:case 40:return 57;case 41:return 53;case 42:return 54;case 43:return 55;case 44:return 50;case 45:return 51;case 46:return 47;case 47:return 45;case 48:return 48;case 49:return 46;case 50:return 41;case 51:return 43;case 52:return 42;case 53:return 39;case 54:return 37;case 55:return 32;case 56:return 86;case 57:return 5;case 58:return 20;case 59:return"INVALID"}},rules:[/^(?:\s+)/,/^(?:[0-9]*\.?[0-9]+([eE][-+]?[0-9]+))/,/^(?:[0-9]+\.[0-9]*|[0-9]*\.[0-9]+\b)/,/^(?:[0-9]+)/,/^(?:"(\\["]|[^"])*")/,/^(?:'(\\[']|[^'])*')/,/^(?:\/\/.*)/,/^(?:\/\*(.|\n|\r)*?\*\/)/,/^(?:if\b)/,/^(?:else\b)/,/^(?:while\b)/,/^(?:do\b)/,/^(?:for\b)/,/^(?:function\b)/,/^(?:map\b)/,/^(?:use\b)/,/^(?:return\b)/,/^(?:delete\b)/,/^(?:true\b)/,/^(?:false\b)/,/^(?:null\b)/,/^(?:Infinity\b)/,/^(?:->)/,/^(?:=>)/,/^(?:<<)/,/^(?:>>)/,/^(?:\{)/,/^(?:\})/,/^(?:;)/,/^(?:#)/,/^(?:\?)/,/^(?::)/,/^(?:NaN\b)/,/^(?:\.)/,/^(?:\[)/,/^(?:\])/,/^(?:\()/,/^(?:\))/,/^(?:!)/,/^(?:\^)/,/^(?:\*\*)/,/^(?:\*)/,/^(?:\/)/,/^(?:%)/,/^(?:\+)/,/^(?:-)/,/^(?:<=)/,/^(?:<)/,/^(?:>=)/,/^(?:>)/,/^(?:==)/,/^(?:~=)/,/^(?:!=)/,/^(?:&&)/,/^(?:\|\|)/,/^(?:=)/,/^(?:,)/,/^(?:$)/,/^(?:[A-Za-z_\$][A-Za-z0-9_]*)/,/^(?:.)/],conditions:{INITIAL:{rules:[0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,49,50,51,52,53,54,55,56,57,58,59],inclusive:!0}}};return t}();function dt(){this.yy={}}return at.lexer=ct,dt.prototype=at,at.Parser=dt,new dt}();parser.yy.parseError=parser.parseError,__webpack_exports__.Z=_jxg_js__WEBPACK_IMPORTED_MODULE_0__.Z.JessieCode},327:function(t,e,i){var s=i(765),r=i(109),o=i(275),n={aliceblue:"f0f8ff",antiquewhite:"faebd7",aqua:"00ffff",aquamarine:"7fffd4",azure:"f0ffff",beige:"f5f5dc",bisque:"ffe4c4",black:"000000",blanchedalmond:"ffebcd",blue:"0000ff",blueviolet:"8a2be2",brown:"a52a2a",burlywood:"deb887",cadetblue:"5f9ea0",chartreuse:"7fff00",chocolate:"d2691e",coral:"ff7f50",cornflowerblue:"6495ed",cornsilk:"fff8dc",crimson:"dc143c",cyan:"00ffff",darkblue:"00008b",darkcyan:"008b8b",darkgoldenrod:"b8860b",darkgray:"a9a9a9",darkgreen:"006400",darkkhaki:"bdb76b",darkmagenta:"8b008b",darkolivegreen:"556b2f",darkorange:"ff8c00",darkorchid:"9932cc",darkred:"8b0000",darksalmon:"e9967a",darkseagreen:"8fbc8f",darkslateblue:"483d8b",darkslategray:"2f4f4f",darkturquoise:"00ced1",darkviolet:"9400d3",deeppink:"ff1493",deepskyblue:"00bfff",dimgray:"696969",dodgerblue:"1e90ff",feldspar:"d19275",firebrick:"b22222",floralwhite:"fffaf0",forestgreen:"228b22",fuchsia:"ff00ff",gainsboro:"dcdcdc",ghostwhite:"f8f8ff",gold:"ffd700",goldenrod:"daa520",gray:"808080",green:"008000",greenyellow:"adff2f",honeydew:"f0fff0",hotpink:"ff69b4",indianred:"cd5c5c",indigo:"4b0082",ivory:"fffff0",khaki:"f0e68c",lavender:"e6e6fa",lavenderblush:"fff0f5",lawngreen:"7cfc00",lemonchiffon:"fffacd",lightblue:"add8e6",lightcoral:"f08080",lightcyan:"e0ffff",lightgoldenrodyellow:"fafad2",lightgrey:"d3d3d3",lightgreen:"90ee90",lightpink:"ffb6c1",lightsalmon:"ffa07a",lightseagreen:"20b2aa",lightskyblue:"87cefa",lightslateblue:"8470ff",lightslategray:"778899",lightsteelblue:"b0c4de",lightyellow:"ffffe0",lime:"00ff00",limegreen:"32cd32",linen:"faf0e6",magenta:"ff00ff",maroon:"800000",mediumaquamarine:"66cdaa",mediumblue:"0000cd",mediumorchid:"ba55d3",mediumpurple:"9370d8",mediumseagreen:"3cb371",mediumslateblue:"7b68ee",mediumspringgreen:"00fa9a",mediumturquoise:"48d1cc",mediumvioletred:"c71585",midnightblue:"191970",mintcream:"f5fffa",mistyrose:"ffe4e1",moccasin:"ffe4b5",navajowhite:"ffdead",navy:"000080",oldlace:"fdf5e6",olive:"808000",olivedrab:"6b8e23",orange:"ffa500",orangered:"ff4500",orchid:"da70d6",palegoldenrod:"eee8aa",palegreen:"98fb98",paleturquoise:"afeeee",palevioletred:"d87093",papayawhip:"ffefd5",peachpuff:"ffdab9",peru:"cd853f",pink:"ffc0cb",plum:"dda0dd",powderblue:"b0e0e6",purple:"800080",red:"ff0000",rosybrown:"bc8f8f",royalblue:"4169e1",saddlebrown:"8b4513",salmon:"fa8072",sandybrown:"f4a460",seagreen:"2e8b57",seashell:"fff5ee",sienna:"a0522d",silver:"c0c0c0",skyblue:"87ceeb",slateblue:"6a5acd",slategray:"708090",snow:"fffafa",springgreen:"00ff7f",steelblue:"4682b4",tan:"d2b48c",teal:"008080",thistle:"d8bfd8",tomato:"ff6347",turquoise:"40e0d0",venetianred:"ae181e",violet:"ee82ee",violetred:"d02090",wheat:"f5deb3",white:"ffffff",whitesmoke:"f5f5f5",yellow:"ffff00",yellowgreen:"9acd32"},a=[{re:/^\s*rgba\(\s*(\d{1,3})\s*,\s*(\d{1,3})\s*,\s*(\d{1,3})\s*,\s*([\d.]{1,3})\s*\)\s*$/,example:["rgba(123, 234, 45, 0.5)","rgba(255,234,245,1.0)"],process:function(t){return[parseInt(t[1],10),parseInt(t[2],10),parseInt(t[3],10)]}},{re:/^\s*rgb\(\s*(\d{1,3})\s*,\s*(\d{1,3})\s*,\s*(\d{1,3})\s*\)\s*$/,example:["rgb(123, 234, 45)","rgb(255,234,245)"],process:function(t){return[parseInt(t[1],10),parseInt(t[2],10),parseInt(t[3],10)]}},{re:/^(\w{2})(\w{2})(\w{2})$/,example:["#00ff00","336699"],process:function(t){return[parseInt(t[1],16),parseInt(t[2],16),parseInt(t[3],16)]}},{re:/^(\w{1})(\w{1})(\w{1})$/,example:["#fb0","f0f"],process:function(t){return[parseInt(t[1]+t[1],16),parseInt(t[2]+t[2],16),parseInt(t[3]+t[3],16)]}}];s.Z.rgbParser=function(t,e,i){var s,o,h,l,c,d,u,p,f,_,m=t;if(!r.Z.exists(t))return[];if(r.Z.exists(e)&&r.Z.exists(i)&&(m=[t,e,i]),s=m,_=!1,r.Z.isArray(s)){for(d=0;d<3;d++)_=_||/\./.test(m[d].toString());for(d=0;d<3;d++)_=_&&m[d]>=0&&m[d]<=1;return _?[Math.ceil(255*m[0]),Math.ceil(255*m[1]),Math.ceil(255*m[2])]:m}for("string"==typeof m&&(s=m),"#"===s.charAt(0)&&(s=s.slice(1,7)),s=s.replace(/ /g,"").toLowerCase(),s=n[s]||s,d=0;d<a.length;d++)h=a[d].re,l=a[d].process,(c=h.exec(s))&&(u=(o=l(c))[0],p=o[1],f=o[2]);return isNaN(u)||isNaN(p)||isNaN(f)?[]:[u=u<0||isNaN(u)?0:u>255?255:u,p=p<0||isNaN(p)?0:p>255?255:p,f=f<0||isNaN(f)?0:f>255?255:f]},s.Z.isColor=function(t){var e=(new Option).style;return e.color=t,""!==e.color},s.Z.rgb2css=function(t,e,i){var r;return"rgb("+(r=s.Z.rgbParser(t,e,i))[0]+", "+r[1]+", "+r[2]+")"},s.Z.rgb2hex=function(t,e,i){var r,o,n;return o=(r=s.Z.rgbParser(t,e,i))[1],n=r[2],r=(r=r[0]).toString(16),o=o.toString(16),n=n.toString(16),1===r.length&&(r="0"+r),1===o.length&&(o="0"+o),1===n.length&&(n="0"+n),"#"+r+o+n},s.Z.hex2rgb=function(t){return s.Z.deprecated("JXG.hex2rgb()","JXG.rgb2css()"),s.Z.rgb2css(t)},s.Z.hsv2rgb=function(t,e,i){var s,r,n,a,h,l,c,d,u;if(t=(t%360+360)%360,0===e){if(!(isNaN(t)||t<o.Z.eps))return"#ffffff";s=i,r=i,n=i}else switch(l=t>=360?0:t,c=i*(1-e),d=i*(1-e*(a=(l/=60)-(h=Math.floor(l)))),u=i*(1-e*(1-a)),h){case 0:s=i,r=u,n=c;break;case 1:s=d,r=i,n=c;break;case 2:s=c,r=i,n=u;break;case 3:s=c,r=d,n=i;break;case 4:s=u,r=c,n=i;break;case 5:s=i,r=c,n=d}return["#",s=2===(s=Math.round(255*s).toString(16)).length?s:1===s.length?"0"+s:"00",r=2===(r=Math.round(255*r).toString(16)).length?r:1===r.length?"0"+r:"00",n=2===(n=Math.round(255*n).toString(16)).length?n:1===n.length?"0"+n:"00"].join("")},s.Z.rgb2hsv=function(t,e,i){var r,o,n,a,h,l,c,d,u,p,f,_,m;return o=(r=s.Z.rgbParser(t,e,i))[1],n=r[2],a=(r=r[0])/255,h=o/255,l=n/255,_=Math.max(r,o,n),d=(m=Math.min(r,o,n))/255,p=0,(f=c=_/255)>0&&(p=(f-d)/f),u=1/(c-d),p>0&&(_===r?u*=h-l:u=_===o?2+(l-a)*u:4+(a-h)*u),(u*=60)<0&&(u+=360),_===m&&(u=0),[u,p,f]},s.Z.rgb2LMS=function(t,e,i){var r,o,n,a,h,l,c,d=[[.05059983,.08585369,.0095242],[.01893033,.08925308,.01370054],[.00292202,.00975732,.07145979]];return o=(r=s.Z.rgbParser(t,e,i))[1],n=r[2],r=r[0],r=Math.pow(r,.476190476),o=Math.pow(o,.476190476),n=Math.pow(n,.476190476),(c=[a=r*d[0][0]+o*d[0][1]+n*d[0][2],h=r*d[1][0]+o*d[1][1]+n*d[1][2],l=r*d[2][0]+o*d[2][1]+n*d[2][2]]).l=a,c.m=h,c.s=l,c},s.Z.LMS2rgb=function(t,e,i){var s,r,o,n,a=[[30.830854,-29.832659,1.610474],[-6.481468,17.715578,-2.532642],[-.37569,-1.199062,14.273846]],h=function(t){for(var e=127,i=64;i>0;){if(Math.pow(e,.476190476)>t)e-=i;else{if(Math.pow(e+1,.476190476)>t)return e;e+=i}i/=2}return 254===e&&13.994955247<t?255:e};return s=t*a[0][0]+e*a[0][1]+i*a[0][2],r=t*a[1][0]+e*a[1][1]+i*a[1][2],o=t*a[2][0]+e*a[2][1]+i*a[2][2],(n=[s=h(s),r=h(r),o=h(o)]).r=s,n.g=r,n.b=o,n},s.Z.rgba2rgbo=function(t){var e;return 9===t.length&&"#"===t.charAt(0)?(e=parseInt(t.slice(7,9).toUpperCase(),16)/255,t=t.slice(0,7)):e=1,[t,e]},s.Z.rgbo2rgba=function(t,e){var i;return"none"===t||"transparent"===t?t:(1===(i=Math.round(255*e).toString(16)).length&&(i="0"+i),s.Z.rgb2hex(t)+i)},s.Z.rgb2bw=function(t){var e,i,r,o="0123456789ABCDEF";return"none"===t?t:(r=s.Z.rgbParser(t),e=Math.floor(.3*r[0]+.59*r[1]+.11*r[2]),t="#"+(i=o.charAt(e>>4&15)+o.charAt(15&e))+i+i)},s.Z.rgb2cb=function(t,e){var i,r,o,n,a,h,l,c,d,u,p,f="0123456789ABCDEF";if("none"===t)return t;switch(r=(a=s.Z.rgb2LMS(t))[0],o=a[1],n=a[2],e=e.toLowerCase()){case"protanopia":h=-.06150039994295001,l=.08277001656812001,c=-.013200141220000003,d=.05858939668799999,u=-.07934519995360001,p=.013289415272000003,r=n/o<.6903216543277437?-(l*o+c*n)/h:-(u*o+p*n)/d;break;case"tritanopia":h=-.00058973116217,l=.007690316482,c=-.01011703519052,d=.025495080838999994,u=-.0422740347,p=.017005316784,n=o/r<.8349489908460004?-(h*r+l*o)/c:-(d*r+u*o)/p;break;default:h=-.06150039994295001,l=.08277001656812001,c=-.013200141220000003,d=.05858939668799999,u=-.07934519995360001,p=.013289415272000003,o=n/r<.5763833686400911?-(h*r+c*n)/l:-(d*r+p*n)/u}return i=s.Z.LMS2rgb(r,o,n),t="#"+(f.charAt(i[0]>>4&15)+f.charAt(15&i[0])),t+=f.charAt(i[1]>>4&15)+f.charAt(15&i[1]),t+=f.charAt(i[2]>>4&15)+f.charAt(15&i[2])},s.Z.shadeColor=function(t,e){var i=s.Z.rgbParser(t),r=i[0],o=i[1],n=i[2];return r=(r=(r=parseInt(r+255*e))>0?r:0)<255?r:255,o=(o=(o=parseInt(o+255*e))>0?o:0)<255?o:255,n=(n=(n=parseInt(n+255*e))>0?n:0)<255?n:255,r=Math.round(r),o=Math.round(o),n=Math.round(n),s.Z.rgb2hex([r,o,n])},s.Z.lightenColor=function(t,e){return s.Z.shadeColor(t,e)},s.Z.darkenColor=function(t,e){return s.Z.shadeColor(t,-1*e)},s.Z.autoHighlight=function(t){var e=s.Z.rgba2rgbo(t),i=e[0],r=e[1];return"#"===t.charAt(0)?(r*=r<.3?1.8:.4,s.Z.rgbo2rgba(i,r)):t},s.Z.contrast=function(t,e,i,r){var o,n,a,h,l;return e=e||"#000000",i=i||"#ffffff",r=r||7,o=s.Z.rgbParser(t),n=s.Z.rgbParser("#000000"),l=(a=.2126*Math.pow(o[0]/255,2.2)+.7152*Math.pow(o[1]/255,2.2)+.0722*Math.pow(o[2]/255,2.2))>(h=.2126*Math.pow(n[0]/255,2.2)+.7152*Math.pow(n[1]/255,2.2)+.0722*Math.pow(n[2]/255,2.2))?Math.floor((a+.05)/(h+.05)):Math.floor((h+.05)/(a+.05)),(l-=1)>r?e:i},s.Z.setClassicColors=function(){s.Z.Options.elements.strokeColor="blue",s.Z.Options.elements.fillColor="red",s.Z.Options.hatch.strokeColor="blue",s.Z.Options.angle.fillColor="#ff7f00",s.Z.Options.angle.highlightFillColor="#ff7f00",s.Z.Options.angle.strokeColor="#ff7f00",s.Z.Options.angle.label.strokeColor="blue",s.Z.Options.arc.strokeColor="blue",s.Z.Options.circle.center.fillColor="red",s.Z.Options.circle.center.strokeColor="blue",s.Z.Options.circumcircle.strokeColor="blue",s.Z.Options.circumcircle.center.fillColor="red",s.Z.Options.circumcircle.center.strokeColor="blue",s.Z.Options.circumcirclearc.strokeColor="blue",s.Z.Options.circumcirclesector.strokeColor="blue",s.Z.Options.circumcirclesector.fillColor="green",s.Z.Options.circumcirclesector.highlightFillColor="green",s.Z.Options.conic.strokeColor="blue",s.Z.Options.curve.strokeColor="blue",s.Z.Options.incircle.strokeColor="blue",s.Z.Options.incircle.center.fillColor="red",s.Z.Options.incircle.center.strokeColor="blue",s.Z.Options.inequality.fillColor="red",s.Z.Options.integral.fillColor="red",s.Z.Options.integral.curveLeft.color="red",s.Z.Options.integral.curveRight.color="red",s.Z.Options.line.strokeColor="blue",s.Z.Options.point.fillColor="red",s.Z.Options.point.strokeColor="red",s.Z.Options.polygon.fillColor="green",s.Z.Options.polygon.highlightFillColor="green",s.Z.Options.polygon.vertices.strokeColor="red",s.Z.Options.polygon.vertices.fillColor="red",s.Z.Options.regularpolygon.fillColor="green",s.Z.Options.regularpolygon.highlightFillColor="green",s.Z.Options.regularpolygon.vertices.strokeColor="red",s.Z.Options.regularpolygon.vertices.fillColor="red",s.Z.Options.riemannsum.fillColor="yellow",s.Z.Options.sector.fillColor="green",s.Z.Options.sector.highlightFillColor="green",s.Z.Options.semicircle.center.fillColor="red",s.Z.Options.semicircle.center.strokeColor="blue",s.Z.Options.slopetriangle.fillColor="red",s.Z.Options.slopetriangle.highlightFillColor="red",s.Z.Options.turtle.arrow.strokeColor="blue"},s.Z.extend(s.Z,{paletteWong:{black:"#000000",orange:"#E69F00",skyblue:"#56B4E9",bluishgreen:"#009E73",yellow:"#F0E442",darkblue:"#0072B2",vermillion:"#D55E00",reddishpurple:"#CC79A7",blue:"#0072B2",red:"#D55E00",green:"#009E73",purple:"#CC79A7",white:"#ffffff"}}),s.Z.palette=s.Z.paletteWong,e.Z=s.Z},503:function(t,e,i){var s=i(765),r=i(109);s.Z.extendConstants(s.Z,{}),s.Z.extend(s.Z,{isTouchEvent:function(t){return s.Z.exists(t.touches)},isPointerEvent:function(t){return s.Z.exists(t.pointerId)},isMouseEvent:function(t){return!s.Z.isTouchEvent(t)&&!s.Z.isPointerEvent(t)},getNumberOfTouchPoints:function(t){var e=-1;return s.Z.isTouchEvent(t)&&(e=t.touches.length),e},isFirstTouch:function(t){var e=s.Z.getNumberOfTouchPoints(t);return s.Z.isPointerEvent(t)?t.isPrimary:1===e},isBrowser:"object"==typeof window&&"object"==typeof document,supportsES6:function(){try{return new Function("(a = 0) => a"),!0}catch(t){return!1}},supportsVML:function(){return this.isBrowser&&!!document.namespaces},supportsSVG:function(){return!!this.isBrowser&&(!!document.createElementNS&&!!document.createElementNS("http://www.w3.org/2000/svg","svg").createSVGRect)},supportsCanvas:function(){var t=!1;return this.isNode()&&(t=!0),t||this.isBrowser&&!!document.createElement("canvas").getContext},isNode:function(){return!this.isBrowser&&"undefined"!=typeof process&&-1!==process.release.name.search(/node|io.js/)},isWebWorker:function(){return!this.isBrowser&&"object"==typeof self&&"function"==typeof self.postMessage},supportsPointerEvents:function(){return!!(this.isBrowser&&window.navigator&&(window.PointerEvent||window.navigator.pointerEnabled||window.navigator.msPointerEnabled))},isTouchDevice:function(){return this.isBrowser&&void 0!==window.ontouchstart},isAndroid:function(){return r.Z.exists(navigator)&&navigator.userAgent.toLowerCase().indexOf("android")>-1},isWebkitAndroid:function(){return this.isAndroid()&&navigator.userAgent.indexOf(" AppleWebKit/")>-1},isApple:function(){return r.Z.exists(navigator)&&(navigator.userAgent.indexOf("iPad")>-1||navigator.userAgent.indexOf("iPhone")>-1)},isWebkitApple:function(){return this.isApple()&&navigator.userAgent.search(/Mobile\/[0-9A-Za-z.]*Safari/)>-1},isMetroApp:function(){return"object"==typeof window&&window.clientInformation&&window.clientInformation.appVersion&&window.clientInformation.appVersion.indexOf("MSAppHost")>-1},isMozilla:function(){return r.Z.exists(navigator)&&navigator.userAgent.toLowerCase().indexOf("mozilla")>-1&&-1===navigator.userAgent.toLowerCase().indexOf("apple")},isFirefoxOS:function(){return r.Z.exists(navigator)&&-1===navigator.userAgent.toLowerCase().indexOf("android")&&-1===navigator.userAgent.toLowerCase().indexOf("apple")&&navigator.userAgent.toLowerCase().indexOf("mobile")>-1&&navigator.userAgent.toLowerCase().indexOf("mozilla")>-1},isDesktop:function(){return!0},isMobile:function(){return!0},ieVersion:function(){var t,e,i=3;if("object"!=typeof document)return 0;e=(t=document.createElement("div")).getElementsByTagName("i");do{t.innerHTML="\x3c!--[if gt IE "+ ++i+"]><i></i><![endif]--\x3e"}while(e[0]);return i>4?i:void 0}(),getDimensions:function(t,e){var i,s,o,n,a,h,l,c,d,u=/\d+(\.\d*)?px/;if(!this.isBrowser||null===t)return{width:500,height:500};if(e=e||document,i=r.Z.isString(t)?e.getElementById(t):t,!r.Z.exists(i))throw new Error("\nJSXGraph: HTML container element '"+t+"' not found.");return"none"!==(s=i.style.display)&&null!==s?i.clientWidth>0&&i.clientHeight>0?{width:i.clientWidth,height:i.clientHeight}:(d=window.getComputedStyle?window.getComputedStyle(i):i.style,{width:u.test(d.width)?parseFloat(d.width):0,height:u.test(d.height)?parseFloat(d.height):0}):(n=(o=i.style).visibility,a=o.position,h=o.display,o.visibility="hidden",o.position="absolute",o.display="block",l=i.clientWidth,c=i.clientHeight,o.display=h,o.position=a,o.visibility=n,{width:l,height:c})},addEvent:function(t,e,i,s,o){var n=function(){return i.apply(s,arguments)};n.origin=i,"object"==typeof s&&r.Z.exists(s.BOARD_MODE_NONE)&&(s["x_internal"+e]=s["x_internal"+e]||[],s["x_internal"+e].push(n)),r.Z.exists(t)&&r.Z.exists(t.addEventListener)&&(o=o||!1,t.addEventListener(e,n,o)),r.Z.exists(t)&&r.Z.exists(t.attachEvent)&&t.attachEvent("on"+e,n)},removeEvent:function(t,e,i,o){var n;if(r.Z.exists(o))if(r.Z.exists(o["x_internal"+e]))if(r.Z.isArray(o["x_internal"+e]))if(-1!==(n=r.Z.indexOf(o["x_internal"+e],i,"origin"))){try{r.Z.exists(t)&&r.Z.exists(t.removeEventListener)&&t.removeEventListener(e,o["x_internal"+e][n],!1),r.Z.exists(t)&&r.Z.exists(t.detachEvent)&&t.detachEvent("on"+e,o["x_internal"+e][n])}catch(t){s.Z.debug("removeEvent: event not registered in browser: ("+e+" -- "+i+")")}o["x_internal"+e].splice(n,1)}else s.Z.debug("removeEvent: no such event function in internal list: "+i);else s.Z.debug("owner[x_internal + "+e+"] is not an array");else s.Z.debug("removeEvent: no such type: "+e);else s.Z.debug("no such owner")},removeAllEvents:function(t,e,i){var r;if(i["x_internal"+e]){for(r=i["x_internal"+e].length-1;r>=0;r--)s.Z.removeEvent(t,e,i["x_internal"+e][r].origin,i);i["x_internal"+e].length>0&&s.Z.debug("removeAllEvents: Not all events could be removed.")}},getPosition:function(t,e,i){var s,o,n,a=0,h=0;if(t||(t=window.event),i=i||document,n=t.touches,r.Z.exists(n)&&0===n.length&&(n=t.changedTouches),r.Z.exists(e)&&r.Z.exists(n))if(-1===e){for(o=n.length,s=0;s<o;s++)if(n[s]){t=n[s];break}}else t=n[e];return t.clientX&&(a=t.clientX,h=t.clientY),[a,h]},getOffset:function(t){var e,i=t,s=t,r=i.offsetLeft-i.scrollLeft,o=i.offsetTop-i.scrollTop;for(r=(e=this.getCSSTransform([r,o],i))[0],o=e[1],i=i.offsetParent;i;){for(r+=i.offsetLeft,o+=i.offsetTop,i.offsetParent&&(r+=i.clientLeft-i.scrollLeft,o+=i.clientTop-i.scrollTop),r=(e=this.getCSSTransform([r,o],i))[0],o=e[1],s=s.parentNode;s!==i;)r+=s.clientLeft-s.scrollLeft,o+=s.clientTop-s.scrollTop,r=(e=this.getCSSTransform([r,o],s))[0],o=e[1],s=s.parentNode;i=i.offsetParent}return[r,o]},getStyle:function(t,e){var i,r=t.ownerDocument;return r.defaultView&&r.defaultView.getComputedStyle?i=r.defaultView.getComputedStyle(t,null).getPropertyValue(e):t.currentStyle&&s.Z.ieVersion>=9?i=t.currentStyle[e]:t.style&&(e=e.replace(/-([a-z]|[0-9])/gi,(function(t,e){return e.toUpperCase()})),i=t.style[e]),i},getProp:function(t,e){var i=parseInt(this.getStyle(t,e),10);return isNaN(i)?0:i},getCSSTransform:function(t,e){var i,s,o,n,a,h,l,c=["transform","webkitTransform","MozTransform","msTransform","oTransform"];for(a=c.length,i=0,o="";i<a;i++)if(r.Z.exists(e.style[c[i]])){o=e.style[c[i]];break}if(""!==o&&(n=o.indexOf("("))>0){for(a=o.length,s=0,h=(l=o.substring(n+1,a-1).split(",")).length;s<h;s++)l[s]=parseFloat(l[s]);0===o.indexOf("matrix")?(t[0]+=l[4],t[1]+=l[5]):0===o.indexOf("translateX")?t[0]+=l[0]:0===o.indexOf("translateY")?t[1]+=l[0]:0===o.indexOf("translate")&&(t[0]+=l[0],t[1]+=l[1])}return r.Z.exists(e.style.zoom)&&""!==(o=e.style.zoom)&&(t[0]*=parseFloat(o),t[1]*=parseFloat(o)),t},getCSSTransformMatrix:function(t){var e,i,s,o,n,a,h,l,c=t.ownerDocument,d=["transform","webkitTransform","MozTransform","msTransform","oTransform"],u=[[1,0,0],[0,1,0],[0,0,1]];if(c.defaultView&&c.defaultView.getComputedStyle)s=(l=c.defaultView.getComputedStyle(t,null)).getPropertyValue("-webkit-transform")||l.getPropertyValue("-moz-transform")||l.getPropertyValue("-ms-transform")||l.getPropertyValue("-o-transform")||l.getPropertyValue("transform");else for(a=d.length,e=0,s="";e<a;e++)if(r.Z.exists(t.style[d[e]])){s=t.style[d[e]];break}if(""!==s&&(n=s.indexOf("("))>0){for(a=s.length,i=0,h=(o=s.substring(n+1,a-1).split(",")).length;i<h;i++)o[i]=parseFloat(o[i]);0===s.indexOf("matrix")?u=[[1,0,0],[0,o[0],o[1]],[0,o[2],o[3]]]:0===s.indexOf("scaleX")?u[1][1]=o[0]:0===s.indexOf("scaleY")?u[2][2]=o[0]:0===s.indexOf("scale")&&(u[1][1]=o[0],u[2][2]=o[1])}return r.Z.exists(t.style.zoom)&&""!==(s=t.style.zoom)&&(u[1][1]*=parseFloat(s),u[2][2]*=parseFloat(s)),u},timedChunk:function(t,e,i,s){var r=t.slice(),o=function(){var n=+new Date;do{e.call(i,r.shift())}while(r.length>0&&+new Date-n<300);r.length>0?window.setTimeout(o,1):s(t)};window.setTimeout(o,1)},scaleJSXGraphDiv:function(t,e,i,r){var o,n,a,h,l,c,d,u,p,f,_,m,g=r;for(n=(a=i.getElementById(t).getBoundingClientRect()).height,o=a.width,c=(u=i.getElementById(e))._cssFullscreenStore.w,_=(d=u._cssFullscreenStore.h)/c,c>o*g&&(d=(c=o*g)*_),d>n*g&&(c=(d=n*g)/_),h=c,p=_<n/o?o*g/c:n*g/d,f=.5*(n-(l=d)),m=0;m<4;m++)try{u.style.width=h+"px !important",u.style.height=l+"px !important",u.style.margin="0 auto",u.style.transform=u._cssFullscreenStore.transform+" matrix("+p+",0,0,"+p+",0,"+f+")";break}catch(t){s.Z.debug("JXG.scaleJSXGraphDiv:\n"+t)}4===m&&s.Z.debug("JXG.scaleJSXGraphDiv: Could not set any CSS property.")}}),e.Z=s.Z},88:function(t,e,i){var s=i(765),r=i(109);s.Z.EventEmitter={eventHandlers:{},suspended:{},trigger:function(t,e){var i,s,r,o,n,a;for(n=t.length,s=0;s<n;s++)if(o=this.eventHandlers[t[s]],!this.suspended[t[s]]){if(this.suspended[t[s]]=!0,o)for(a=o.length,i=0;i<a;i++)(r=o[i]).handler.apply(r.context,e);this.suspended[t[s]]=!1}return this},on:function(t,e,i){return r.Z.isArray(this.eventHandlers[t])||(this.eventHandlers[t]=[]),i=r.Z.def(i,this),this.eventHandlers[t].push({handler:e,context:i}),this},off:function(t,e){var i;return t&&r.Z.isArray(this.eventHandlers[t])?(e?((i=r.Z.indexOf(this.eventHandlers[t],e,"handler"))>-1&&this.eventHandlers[t].splice(i,1),0===this.eventHandlers[t].length&&delete this.eventHandlers[t]):delete this.eventHandlers[t],this):this},eventify:function(t){t.eventHandlers={clicks:0},t.on=this.on,t.off=this.off,t.triggerEventHandlers=this.trigger,t.trigger=this.trigger,t.suspended={}}},e.Z=s.Z.EventEmitter},421:function(t,e,i){var s=i(765),r=i(109),o=i(351),n=i(705);s.Z.Expect={each:function(t,e,i){var s,o,n=[];if(r.Z.exists(t.length))for(o=t.length,s=0;s<o;s++)n.push(e.call(this,t[s],i));return n},coords:function(t,e){var i=t;return t&&t.elementClass===o.Z.OBJECT_CLASS_POINT?i=t.coords:t.usrCoords&&t.scrCoords&&t.usr2screen&&(i=t),e&&(i=new n.Z(o.Z.COORDS_BY_USER,i.usrCoords,i.board)),i},coordsArray:function(t,e){var i;return(i=r.Z.isArray(t)?t:this.coords(t).usrCoords).length<3&&i.unshift(1),e&&(i=[i[0],i[1],i[2]]),i}},e.Z=s.Z.Expect},109:function(t,e,i){var s=i(765),r=i(351),o=i(275);s.Z.extend(s.Z,{isBoard:function(t){return null!==t&&"object"==typeof t&&this.isNumber(t.BOARD_MODE_NONE)&&this.isObject(t.objects)&&this.isObject(t.jc)&&this.isFunction(t.update)&&!!t.containerObj&&this.isString(t.id)},isId:function(t,e){return"string"==typeof e&&!!t.objects[e]},isName:function(t,e){return"string"==typeof e&&!!t.elementsByName[e]},isGroup:function(t,e){return"string"==typeof e&&!!t.groups[e]},isString:function(t){return"string"==typeof t},isNumber:function(t,e,i){var s="number"==typeof t||"[Object Number]"===Object.prototype.toString.call(t);return i=void 0===i||i,(e=e||!1)&&(s=s||""+parseFloat(t)===t),i||(s=s&&!isNaN(t)),s},isFunction:function(t){return"function"==typeof t},isArray:function(t){return Array.isArray?Array.isArray(t):null!==t&&"object"==typeof t&&"function"==typeof t.splice&&"function"==typeof t.join},isObject:function(t){return"object"==typeof t&&!this.isArray(t)},isDocumentOrFragment:function(t){return this.isObject(t)&&(9===t.nodeType||11===t.nodeType)},isPoint:function(t){return!(null===t||"object"!=typeof t||!this.exists(t.elementClass))&&t.elementClass===r.Z.OBJECT_CLASS_POINT},isPoint3D:function(t){return!(null===t||"object"!=typeof t||!this.exists(t.type))&&t.type===r.Z.OBJECT_TYPE_POINT3D},isPointType:function(t,e){var i,s;return!!this.isArray(e)||(!!(this.isFunction(e)&&(i=e(),this.isArray(i)&&i.length>1))||(s=t.select(e),this.isPoint(s)))},isPointType3D:function(t,e){var i,s;return!!(this.isArray(e)&&e.length>=3)||(!!(this.isFunction(e)&&(i=e(),this.isArray(i)&&i.length>=3))||(s=t.select(e),this.isPoint3D(s)))},isTransformationOrArray:function(t){if(null!==t){if(this.isArray(t)&&t.length>0)return this.isTransformationOrArray(t[0]);if("object"==typeof t)return t.type===r.Z.OBJECT_TYPE_TRANSFORMATION}return!1},isEmpty:function(t){return 0===Object.keys(t).length},exists:function(t,e){var i=!(null==t||null===t);return(e=e||!1)?i&&""!==t:i},def:function(t,e){return this.exists(t)?t:e},str2Bool:function(t){return!this.exists(t)||("boolean"==typeof t?t:!!this.isString(t)&&"true"===t.toLowerCase())},cssParse:function(t){var e=t;return this.isString(e)?(e='{"'+(e=(e=(e=(e=e.replace(/\s*;\s*$/g,"")).replace(/\s*;\s*/g,'","')).replace(/\s*:\s*/g,'":"')).trim())+'"}',JSON.parse(e)):{}},cssStringify:function(t){var e,i,s="";if(!this.isObject(t))return"";for(e in t)t.hasOwnProperty(e)&&(i=t[e],(this.isString(i)||this.isNumber(i))&&(s+=e+":"+i+"; "));return s=s.trim()},createEvalFunction:function(t,e,i){var s,r,o,n=[],a={};for(r=0;r<i;r++)for(o in n[r]=this.createFunction(e[r],t),n[r].deps)a[o]=n[r].deps;return(s=function(t){return n[t]()}).deps=a,s},createFunction:function(t,e,i,s){var r=null;return this.isString(t)?r=e.jc.snippet(t,!0,i,!1):this.isFunction(t)?(r=t).deps=this.isObject(t.deps)?t.deps:{}:(this.isNumber(t)||this.isArray(t))&&((r=function(){return t}).deps={}),null!==r&&(r.origin=t),r},providePoints:function(t,e,i,s,r){var o,n,a,h,l,c=0,d=[];for(this.isArray(e)||(e=[e]),a=e.length,this.exists(r)&&(c=r.length),0===c&&(h=this.copyAttributes(i,t.options,s)),o=0;o<a;++o)if(c>0&&(n=Math.min(o,c-1),h=this.copyAttributes(i,t.options,s,r[n].toLowerCase())),this.isArray(e[o])&&e[o].length>1?(d.push(t.create("point",e[o],h)),d[d.length-1]._is_new=!0):this.isFunction(e[o])?(l=e[o](),this.isArray(l)&&l.length>1&&(d.push(t.create("point",[e[o]],h)),d[d.length-1]._is_new=!0)):d.push(t.select(e[o])),!this.isPoint(d[o]))return!1;return d},providePoints3D:function(t,e,i,s,r){var o,n,a,h,l,c=0,d=[];for(this.isArray(e)||(e=[e]),a=e.length,this.exists(r)&&(c=r.length),0===c&&(h=this.copyAttributes(i,t.board.options,s)),o=0;o<a;++o){if(c>0&&(n=Math.min(o,c-1),h=this.copyAttributes(i,t.board.options,s,r[n])),this.isArray(e[o])&&e[o].length>0&&e[o].every((t=>this.isArray(t)&&this.isNumber(t[0]))))for(n=0;n<e[o].length;n++)d.push(t.create("point3d",e[o][n],h)),d[d.length-1]._is_new=!0;else this.isArray(e[o])&&e[o].every((t=>this.isNumber(t)||this.isFunction(t)))?(d.push(t.create("point3d",e[o],h)),d[d.length-1]._is_new=!0):this.isPoint3D(e[o])?d.push(e[o]):this.isFunction(e[o])?(l=e[o](),this.isArray(l)&&l.length>1&&(d.push(t.create("point3d",[e[o]],h)),d[d.length-1]._is_new=!0)):d.push(t.select(e[o]));if(!this.isPoint3D(d[o]))return!1}return d},bind:function(t,e){return function(){return t.apply(e,arguments)}},evaluate:function(t){return this.isFunction(t)?t():t},indexOf:function(t,e,i){var s,r=this.exists(i);if(Array.indexOf&&!r)return t.indexOf(e);for(s=0;s<t.length;s++)if(r&&t[s][i]===e||!r&&t[s]===e)return s;return-1},eliminateDuplicates:function(t){var e,i=t.length,s=[],r={};for(e=0;e<i;e++)r[t[e]]=0;for(e in r)r.hasOwnProperty(e)&&s.push(e);return s},swap:function(t,e,i){var s;return s=t[e],t[e]=t[i],t[i]=s,t},uniqueArray:function(t){var e,i,r,o=[];if(0===t.length)return[];for(e=0;e<t.length;e++)if(r=this.isArray(t[e]),this.exists(t[e]))for(i=e+1;i<t.length;i++)r&&s.Z.cmpArrays(t[e],t[i])?t[e]=[]:r||t[e]!==t[i]||(t[e]="");else t[e]="";for(i=0,e=0;e<t.length;e++)(r=this.isArray(t[e]))||""===t[e]?r&&0!==t[e].length&&(o[i]=t[e].slice(0),i++):(o[i]=t[e],i++);return t=o,o},isInArray:function(t,e){return s.Z.indexOf(t,e)>-1},coordsArrayToMatrix:function(t,e){var i,s=[],r=[];for(i=0;i<t.length;i++)e?(s.push(t[i].usrCoords[1]),r.push(t[i].usrCoords[2])):r.push([t[i].usrCoords[1],t[i].usrCoords[2]]);return e&&(r=[s,r]),r},cmpArrays:function(t,e){var i;if(t===e)return!0;if(t.length!==e.length)return!1;for(i=0;i<t.length;i++)if(this.isArray(t[i])&&this.isArray(e[i])){if(!this.cmpArrays(t[i],e[i]))return!1}else if(t[i]!==e[i])return!1;return!0},removeElementFromArray:function(t,e){var i;for(i=0;i<t.length;i++)if(t[i]===e)return t.splice(i,1),t;return t},trunc:function(t,e){return e=s.Z.def(e,0),this.toFixed(t,e)},_decimalAdjust:function(t,e,i){return void 0===i||0==+i?Math[t](e):(e=+e,i=+i,isNaN(e)||"number"!=typeof i||i%1!=0?NaN:(e=e.toString().split("e"),+((e=(e=Math[t](+(e[0]+"e"+(e[1]?+e[1]-i:-i)))).toString().split("e"))[0]+"e"+(e[1]?+e[1]+i:i))))},_round10:function(t,e){return this._decimalAdjust("round",t,e)},_floor10:function(t,e){return this._decimalAdjust("floor",t,e)},_ceil10:function(t,e){return this._decimalAdjust("ceil",t,e)},toFixed:function(t,e){return this._round10(t,-e).toFixed(e)},autoDigits:function(t){var e=Math.abs(t);return e>=.1?this.toFixed(t,2):e>=.01?this.toFixed(t,4):e>=1e-4?this.toFixed(t,6):t},parseNumber:function(t,e,i){var s;return this.isString(t)&&t.indexOf("%")>-1?(s=t.replace(/\s+%\s+/,""),parseFloat(s)*e*.01):this.isString(t)&&t.indexOf("fr")>-1?(s=t.replace(/\s+fr\s+/,""),parseFloat(s)*e):this.isString(t)&&t.indexOf("px")>-1?(s=t.replace(/\s+px\s+/,""),s=parseFloat(s),this.isFunction(i)?i(s):this.isNumber(i)?s*i:s):parseFloat(t)},parsePosition:function(t){var e,i,s="",r="";if(""!==(t=t.trim()))for(e=t.split(/[ ,]+/),i=0;i<e.length;i++)"left"===e[i]||"right"===e[i]?s=e[i]:r=e[i];return{side:s,pos:r}},keys:function(t,e){var i,s=[];for(i in t)e?t.hasOwnProperty(i)&&s.push(i):s.push(i);return s},clone:function(t){var e={};return e.prototype=t,e},cloneAndCopy:function(t,e){var i,s=function(){};for(i in s.prototype=t,e)s[i]=e[i];return s},merge:function(t,e){var i,s,r,o;for(i in e)if(e.hasOwnProperty(i))if(r=e[i],this.isArray(r))for(t[i]||(t[i]=[]),s=0;s<r.length;s++)o=e[i][s],"object"==typeof e[i][s]?t[i][s]=this.merge(t[i][s],o):t[i][s]=e[i][s];else"object"==typeof r?(t[i]||(t[i]={}),t[i]=this.merge(t[i],r)):("boolean"==typeof t&&(t={}),t[i]=r);return t},deepCopy:function(t,e,i){var s,r,o,n;if(i=i||!1,"object"!=typeof t||null===t)return t;if(this.isArray(t))for(s=[],r=0;r<t.length;r++)null!==(o=t[r])&&"object"==typeof o?this.exists(o.board)?s[r]=o.id:s[r]=this.deepCopy(o,{},i):s[r]=o;else{for(r in s={},t)t.hasOwnProperty(r)&&(n=i?r.toLowerCase():r,null!==(o=t[r])&&"object"==typeof o?this.exists(o.board)?s[n]=o.id:s[n]=this.deepCopy(o,{},i):s[n]=o);for(r in e)e.hasOwnProperty(r)&&(n=i?r.toLowerCase():r,null!==(o=e[r])&&"object"==typeof o?this.isArray(o)||!this.exists(s[n])?s[n]=this.deepCopy(o,{},i):s[n]=this.deepCopy(s[n],o,i):s[n]=o)}return s},mergeAttr:function(t,e,i,s){var r,o,n;for(r in i=i||!0,s=s||!1,e)e.hasOwnProperty(r)&&((o=i?r.toLowerCase():r)!==r&&t.hasOwnProperty(r)&&(t.hasOwnProperty(o)?this.mergeAttr(t[o],t[r],i):t[o]=t[r],delete t[r]),n=e[r],!this.isObject(n)||null===n||this.isDocumentOrFragment(n)||this.exists(n.board)||"string"==typeof n.valueOf()?s&&!this.exists(n)||(t[o]=n):(void 0!==t[o]&&null!==t[o]&&this.isObject(t[o])||(t[o]={}),this.mergeAttr(t[o],n,i)))},keysToLowerCase:function(t){var e,i,s=Object.keys(t),r=s.length,o={};if("object"!=typeof t)return t;for(;r--;)e=s[r],t.hasOwnProperty(e)&&("object"!=typeof(i=t[e])||null===i||this.isArray(i)||this.exists(i.nodeType)||this.exists(i.board)?o[e.toLowerCase()]=i:o[e.toLowerCase()]=this.keysToLowerCase(i));return o},copyAttributes:function(t,e,i){var r,o,n,a,h,l;for(r=(a=arguments.length)<3||{circle:1,curve:1,foreignobject:1,image:1,line:1,point:1,polygon:1,text:1,ticks:1,integral:1}[i]?s.Z.deepCopy(e.elements,null,!0):{},a<4&&this.exists(i)&&this.exists(e.layer[i])&&(r.layer=e.layer[i]),h=e,l=!0,n=2;n<a;n++){if(o=arguments[n],!this.exists(h[o])){l=!1;break}h=h[o]}for(l&&(r=s.Z.deepCopy(r,h,!0)),h="object"==typeof t?this.keysToLowerCase(t):{},l=!0,n=3;n<a;n++){if(o=arguments[n].toLowerCase(),!this.exists(h[o])){l=!1;break}h=h[o]}if(l&&this.mergeAttr(r,h,!0),"board"===arguments[2])return r;for(h=e,l=!0,n=2;n<a;n++){if(o=arguments[n],!this.exists(h[o])){l=!1;break}h=h[o]}return l&&this.exists(h.label)&&(r.label=s.Z.deepCopy(h.label,r.label,!0)),r.label=s.Z.deepCopy(e.label,r.label,!0),r},copyPrototypeMethods:function(t,e,i){var s;for(s in t.prototype[i]=e.prototype.constructor,e.prototype)e.prototype.hasOwnProperty(s)&&(t.prototype[s]=e.prototype[s])},getCloneObject:function(t){var e,i,s={};for(i in s.id=t.id+"T"+t.numTraces,t.numTraces+=1,s.coords=t.coords,e=this.deepCopy(t.visProp,t.visProp.traceattributes,!0),s.visProp={},e)e.hasOwnProperty(i)&&0!==i.indexOf("aria")&&0!==i.indexOf("highlight")&&0!==i.indexOf("attractor")&&"label"!==i&&"needsregularupdate"!==i&&"infoboxdigits"!==i&&(s.visProp[i]=t.eval(e[i]));return s.evalVisProp=function(t){return s.visProp[t]},s.eval=function(t){return t},s.visProp.layer=t.board.options.layer.trace,s.visProp.tabindex=null,s.visProp.highlight=!1,s.board=t.board,s.elementClass=t.elementClass,this.clearVisPropOld(s),s.visPropCalc={visible:t.evalVisProp("visible")},s},toJSON:function(t,e){var i,r,o,n;if(e=s.Z.def(e,!1),void 0!==JSON&&JSON.stringify&&!e)try{return JSON.stringify(t)}catch(t){}switch(typeof t){case"object":if(t){if(i=[],this.isArray(t)){for(o=0;o<t.length;o++)i.push(s.Z.toJSON(t[o],e));return"["+i.join(",")+"]"}for(r in t)if(t.hasOwnProperty(r)){try{n=s.Z.toJSON(t[r],e)}catch(t){n=""}e?i.push(r+":"+n):i.push('"'+r+'":'+n)}return"{"+i.join(",")+"} "}return"null";case"string":return"'"+t.replace(/(["'])/g,"\\$1")+"'";case"number":case"boolean":return t.toString()}return"0"},clearVisPropOld:function(t){return t.visPropOld={cssclass:"",cssdefaultstyle:"",cssstyle:"",fillcolor:"",fillopacity:"",firstarrow:!1,fontsize:-1,lastarrow:!1,left:-1e5,linecap:"",shadow:!1,strokecolor:"",strokeopacity:"",strokewidth:"",tabindex:-1e5,transitionduration:0,top:-1e5,visible:null},t},isInObject:function(t,e){var i;for(i in t)if(t.hasOwnProperty(i)&&t[i]===e)return!0;return!1},escapeHTML:function(t){return t.replace(/&/g,"&amp;").replace(/</g,"&lt;").replace(/>/g,"&gt;")},unescapeHTML:function(t){return t.replace(/<\/?[^>]+>/gi,"").replace(/&amp;/g,"&").replace(/&lt;/g,"<").replace(/&gt;/g,">")},capitalize:function(t){return t.charAt(0).toUpperCase()+t.substring(1).toLowerCase()},trimNumber:function(t){return"."!==(t=(t=t.replace(/^0+/,"")).replace(/0+$/,""))[t.length-1]&&","!==t[t.length-1]||(t=t.slice(0,-1)),"."!==t[0]&&","!==t[0]||(t="0"+t),t},filterElements:function(t,e){var i,s,r,o,n,a,h,l=t.length,c=[];if("function"!=typeof e&&"object"!=typeof e)return c;for(i=0;i<l;i++){if(h=!0,r=t[i],"object"==typeof e){for(s in e)if(e.hasOwnProperty(s)&&(o=s.toLowerCase(),n="function"==typeof r[s]?r[s]():r[s],a=r.visProp&&"function"==typeof r.visProp[o]?r.visProp[o]():r.visProp&&r.visProp[o],!(h="function"==typeof e[s]?e[s](n)||e[s](a):n===e[s]||a===e[s])))break}else"function"==typeof e&&(h=e(r));h&&c.push(r)}return c},trim:function(t){return t.replace(/^[\s\uFEFF\xA0]+|[\s\uFEFF\xA0]+$/g,"")},toFraction:function(t,e,i){var s=o.Z.decToFraction(t,i),r="";return 0===s[1]&&0===s[2]?r+="0":(s[0]<0&&(r+="-"),0===s[2]?r+=s[1]:1===s[2]&&1===s[3]||(0!==s[1]&&(r+=s[1]+" "),r+=!0===e?"\\frac{"+s[2]+"}{"+s[3]+"}":s[2]+"/"+s[3])),r},concat:function(t,e){var i,s=e.length;for(i=0;i<s;i++)t.push(e[i]);return t},sanitizeHTML:function(t,e){return"function"==typeof html_sanitize&&e?html_sanitize(t,(function(){}),(function(t){return t})):(t&&"string"==typeof t&&(t=t.replace(/</g,"&lt;").replace(/>/g,"&gt;")),t)},evalSlider:function(t){return t&&t.type===r.Z.OBJECT_TYPE_GLIDER&&"function"==typeof t.Value?t.Value():t},stack2jsxgraph:function(t){var e;return"["===(e=t.replace(/%pi/g,"PI").replace(/%e/g,"EULER").replace(/%phi/g,"1.618033988749895").replace(/%gamma/g,"0.5772156649015329").trim())[0]&&"]"===e[e.length-1]&&(e=e.slice(1,-1).split(/\s*,\s*/)),e}}),e.Z=s.Z}},__webpack_module_cache__={};function __webpack_require__(t){var e=__webpack_module_cache__[t];if(void 0!==e)return e.exports;var i=__webpack_module_cache__[t]={exports:{}};return __webpack_modules__[t](i,i.exports,__webpack_require__),i.exports}__webpack_require__.d=function(t,e){for(var i in e)__webpack_require__.o(e,i)&&!__webpack_require__.o(t,i)&&Object.defineProperty(t,i,{enumerable:!0,get:e[i]})},__webpack_require__.o=function(t,e){return Object.prototype.hasOwnProperty.call(t,e)};var __webpack_exports__={};return function(){__webpack_require__.d(__webpack_exports__,{default:function(){return ot}});var t=__webpack_require__(765),e=__webpack_require__(503),i=__webpack_require__(351),s=__webpack_require__(109);t.Z.XML={cleanWhitespace:function(t){for(var e=t.firstChild;s.Z.exists(e);)3!==e.nodeType||/\S/.test(e.nodeValue)?1===e.nodeType&&this.cleanWhitespace(e):t.removeChild(e),e=e.nextSibling},parse:function(t){var e,i;return i="function"==typeof DOMParser||"object"==typeof DOMParser?DOMParser:function(){this.parseFromString=function(t){var e;return"function"==typeof ActiveXObject&&(e=new ActiveXObject("MSXML.DomDocument")).loadXML(t),e}},e=(new i).parseFromString(t,"text/xml"),this.cleanWhitespace(e),e}};t.Z.XML;var r=__webpack_require__(88),o=(__webpack_require__(421),__webpack_require__(275));o.Z.ProbFuncs={MAXNUM:17014118346046923e22,SQRTH:.7071067811865476,SQRT2:1.4142135623730951,MAXLOG:708.3964185322641,P:[2.461969814735305e-10,.5641895648310689,7.463210564422699,48.63719709856814,196.5208329560771,526.4451949954773,934.5285271719576,1027.5518868951572,557.5353353693994],Q:[13.228195115474499,86.70721408859897,354.9377788878199,975.7085017432055,1823.9091668790973,2246.3376081871097,1656.6630919416134,557.5353408177277],R:[.5641895835477551,1.275366707599781,5.019050422511805,6.160210979930536,7.4097426995044895,2.9788666537210022],S:[2.2605286322011726,9.396035249380015,12.048953980809666,17.08144507475659,9.608968090632859,3.369076451000815],T:[9.604973739870516,90.02601972038427,2232.005345946843,7003.325141128051,55592.30130103949],U:[33.56171416475031,521.3579497801527,4594.323829709801,22629.000061389095,49267.39426086359],M:128,MINV:.0078125,expx2:function(t,e){var i,s,r,o;return t=Math.abs(t),e<0&&(t=-t),i=(r=this.MINV*Math.floor(this.M*t+.5))*r,s=2*r*(o=t-r)+o*o,e<0&&(i=-i,s=-s),i+s>this.MAXLOG?1/0:i=Math.exp(i)*Math.exp(s)},polevl:function(t,e,i){var r,o;if(s.Z.exists(e.reduce))return e.reduce((function(e,i){return e*t+i}),0);for(o=0,r=0;o<=i;o++)r=r*t+e[o];return r},p1evl:function(t,e,i){var r,o;if(s.Z.exists(e.reduce))return e.reduce((function(e,i){return e*t+i}),1);for(o=0,r=1;o<i;o++)r=r*t+e[o];return r},ndtr:function(t){var e,i,s;return e=t*this.SQRTH,(s=Math.abs(e))<1?i=.5+.5*this.erf(e):(i=.5*this.erfce(s),s=this.expx2(t,-1),i*=Math.sqrt(s),e>0&&(i=1-i)),i},_underflow:function(t){return console.log("erfc","UNDERFLOW"),t<0?2:0},erfc:function(t){var e,i,s,r,o;return(s=t<0?-t:t)<1?1-this.erf(t):(o=-t*t)<-this.MAXLOG?this._underflow(t):(o=this.expx2(t,-1),s<8?(e=this.polevl(s,this.P,8),i=this.p1evl(s,this.Q,8)):(e=this.polevl(s,this.R,5),i=this.p1evl(s,this.S,6)),r=o*e/i,t<0&&(r=2-r),0===r?this._underflow(t):r)},erfce:function(t){var e,i;return t<8?(e=this.polevl(t,this.P,8),i=this.p1evl(t,this.Q,8)):(e=this.polevl(t,this.R,5),i=this.p1evl(t,this.S,6)),e/i},erf:function(t){var e;return Math.abs(t)>1?1-this.erfc(t):(e=t*t,t*this.polevl(e,this.T,4)/this.p1evl(e,this.U,5))},s2pi:2.5066282746310007,P0:[-59.96335010141079,98.00107541859997,-56.67628574690703,13.931260938727968,-1.2391658386738125],Q0:[1.9544885833814176,4.676279128988815,86.36024213908905,-225.46268785411937,200.26021238006066,-82.03722561683334,15.90562251262117,-1.1833162112133],P1:[4.0554489230596245,31.525109459989388,57.16281922464213,44.08050738932008,14.684956192885803,2.1866330685079025,-.1402560791713545,-.03504246268278482,-.0008574567851546854],Q1:[15.779988325646675,45.39076351288792,41.3172038254672,15.04253856929075,2.504649462083094,-.14218292285478779,-.03808064076915783,-.0009332594808954574],P2:[3.2377489177694603,6.915228890689842,3.9388102529247444,1.3330346081580755,.20148538954917908,.012371663481782003,.00030158155350823543,26580697468673755e-22,6.239745391849833e-9],Q2:[6.02427039364742,3.6798356385616087,1.3770209948908132,.21623699359449663,.013420400608854318,.00032801446468212774,28924786474538068e-22,6.790194080099813e-9],ndtri:function(t){var e,i,s,r,o;return t<=0?-1/0:t>=1?1/0:(o=1,(i=t)>.8646647167633873&&(i=1-i,o=0),i>.1353352832366127?(e=(i-=.5)+i*((r=i*i)*this.polevl(r,this.P0,4)/this.p1evl(r,this.Q0,8)),e*=this.s2pi):(s=1/(e=Math.sqrt(-2*Math.log(i))),e=e-Math.log(e)/e-(e<8?s*this.polevl(s,this.P1,8)/this.p1evl(s,this.Q1,8):s*this.polevl(s,this.P2,8)/this.p1evl(s,this.Q2,8)),0!==o&&(e=-e),e))},erfi:function(t){return this.ndtri(.5*(t+1))*this.SQRTH}};o.Z.ProbFuncs;var n=__webpack_require__(785);o.Z.Extrapolate={upper:15,infty:1e4,wynnEps:function(t,e,i){var s,r,o,n,a;if(i[e]=t,0===e)a=t;else{for(o=0,s=e;s>0;s--)r=o,o=i[s-1],n=i[s]-o,Math.abs(n)<=1e-15?i[s-1]=1e20:(1,i[s-1]=1*r+1/n);a=i[e%2]}return a},aitken:function(t,e,i){var s,r,o,n,a,h;if(i[e]=t,e<2)s=t;else{for(n=e/2,a=1;a<=n;a++)r=i[(h=e-2*a)+2]-2*i[h+1]+i[h],Math.abs(r)<1e-15?i[h]=1e20:(o=i[h]-i[h+1],i[h]-=o*o/r);s=i[e%2]}return s},brezinski:function(t,e,i){var s,r,o,n,a,h,l,c;if(i[e]=t,e<3)s=t;else{for(h=e/3,c=e,l=1;l<=h;l++)o=i[(c-=3)+1]-i[c],n=i[c+2]-i[c+1],r=(a=i[c+3]-i[c+2])*(n-o)-o*(a-n),Math.abs(r)<1e-15?i[c]=1e20:i[c]=i[c+1]-o*n*(a-n)/r;s=i[e%3]}return s},iteration:function(t,e,i,s,r){var o,n,a,h,l=NaN,c=[],d="finite",u=e;for(r=r||0,o=1;o<=this.upper;o++){if(n=i(t+(u=0===r?e/(o+1):.5*u),!0),a=this[s](n,o-1,c),isNaN(a)){d="NaN";break}if(0!==n&&a/n>this.infty){l=a,d="infinite";break}if(h=a-l,Math.abs(h)<1e-7)break;l=a}return[l,d,1-(o-1)/this.upper]},levin:function(t,e,i,s,r,o){var n,a,h,l;if(l=1/(s+e),r[e]=t/i,o[e]=1/i,e>0&&(r[e-1]=r[e]-r[e-1],o[e-1]=o[e]-o[e-1],e>1))for(h=(s+e-1)*l,n=2;n<=e;n++)a=(s+e-n)*Math.pow(h,n-2)*l,r[e-n]=r[e-n+1]-a*r[e-n],o[e-n]=o[e-n+1]-a*o[e-n],l*=h;return Math.abs(o[0])<1e-15?1e20:r[0]/o[0]},iteration_levin:function(t,e,i,s){var r,o,n,a,h,l,c,d=NaN,u=[],p=[],f="finite",_=e;for(s=s||0,a=i(t+e,!0),r=1;r<=this.upper;r++){if(h=(o=i(t+(_=0===s?e/(r+1):.5*_),!0))-a,c="u"===(Math.abs(h)<1?"u":"t")?(1+r)*h:h,a=o,l=(n=this.levin(o,r-1,c,1,u,p))-d,isNaN(n)){f="NaN";break}if(0!==o&&n/o>this.infty){d=n,f="infinite";break}if(Math.abs(l)<1e-7)break;d=n}return[d,f,1-(r-1)/this.upper]},limit:function(t,e,i){return this.iteration_levin(t,e,i,0)}};var a=o.Z.Extrapolate,h=__webpack_require__(922);o.Z.Quadtree=function(t,e,i){e=e||{capacity:10,pointType:"coords"},this.config={},this.config.capacity=e.capacity||10,this.config.pointType=e.pointType||"coords",this.points=[],this.xlb=t[0],this.xub=t[2],this.ylb=t[3],this.yub=t[1],this.parent=i||null,this.northWest=null,this.northEast=null,this.southEast=null,this.southWest=null},s.Z.extend(o.Z.Quadtree.prototype,{contains:function(t,e){return this.xlb<t&&t<=this.xub&&this.ylb<e&&e<=this.yub},insert:function(t){switch(this.config.pointType){case"coords":if(!this.contains(t.usrCoords[1],t.usrCoords[2]))return!1;break;case"object":if(!this.contains(t.x,t.y))return!1}return this.points.length<this.config.capacity&&null===this.northWest?(this.points.push(t),!0):(null===this.northWest&&this.subdivide(),!!this.northWest.insert(t)||(!!this.northEast.insert(t)||(!!this.southEast.insert(t)||!!this.southWest.insert(t))))},subdivide:function(){var t=this.xlb+.5*(this.xub-this.xlb),e=this.ylb+.5*(this.yub-this.ylb);this.northWest=new o.Z.Quadtree([this.xlb,this.yub,t,e],this.config,this),this.northEast=new o.Z.Quadtree([t,this.yub,this.xub,e],this.config,this),this.southEast=new o.Z.Quadtree([this.xlb,e,t,this.ylb],this.config,this),this.southWest=new o.Z.Quadtree([t,e,this.xub,this.ylb],this.config,this)},_query:function(t,e){var i;if(this.contains(t,e)){if(null===this.northWest)return this;if(i=this.northWest._query(t,e))return i;if(i=this.northEast._query(t,e))return i;if(i=this.southEast._query(t,e))return i;if(i=this.southWest._query(t,e))return i}return!1},query:function(t,e){var i,r;return s.Z.exists(e)?(i=t,r=e):(i=t.usrCoords[1],r=t.usrCoords[2]),this._query(i,r)},hasPoint:function(t,e,i){var s,r,o;if(this.contains(t,e)){switch(o=this.points.length,this.config.pointType){case"coords":for(r=0;r<o;r++)if(h.Z.distance([t,e],this.points[r].usrCoords.slice(1),2)<i)return!0;break;case"object":for(r=0;r<o;r++)if(h.Z.distance([t,e],[this.points[r].x,this.points[r].y],2)<i)return!0}if(null===this.northWest)return!1;if(s=this.northWest.hasPoint(t,e,i))return s;if(s=this.northEast.hasPoint(t,e,i))return s;if(s=this.southEast.hasPoint(t,e,i))return s;if(s=this.southWest.hasPoint(t,e,i))return s}return!1},getAllPoints:function(){var t=[];return this.getAllPointsRecursive(t),t},getAllPointsRecursive(t){Array.prototype.push.apply(t,this.points.slice()),null!==this.northWest&&(this.northWest.getAllPointsRecursive(t),this.northEast.getAllPointsRecursive(t),this.southEast.getAllPointsRecursive(t),this.southWest.getAllPointsRecursive(t))}});var l=o.Z.Quadtree;o.Z.BoxQuadtree=function(t,e,i){var s,r,o,n;t--,this.depth=t,this.capacity=e,this.items=[],this.northWest=null,this.northEast=null,this.southEast=null,this.southWest=null,this.bbox=null,this.cx=null,this.cy=null,i&&(s=i[0],r=i[1],o=i[2],n=i[3],this.cx=.5*(s+o),this.cy=.5*(r+n),this.bbox=[s,r,o,n])},s.Z.extend(o.Z.BoxQuadtree.prototype,{insert:function(t){var e,i,s,r,o,n,a,h,l,c,d,u=[],p=[],f=[],_=[];if(null===this.bbox){for(i=t.length,s=n=1/0,o=r=-1/0,e=0;e<t.length;e++)s=(a=t[e]).xlb<s?a.xlb:s,r=a.yub>r?a.yub:r,o=a.xub>o?a.xub:o,n=a.ylb<n?a.ylb:n;this.cx=.5*(s+o),this.cy=.5*(r+n),this.bbox=[s,r,o,n]}else s=this.bbox[0],r=this.bbox[1],o=this.bbox[2],n=this.bbox[3];if(0===this.depth||this.items.length+t.length<this.capacity)return this.items=this.items.concat(t),this;for(i=t.length,e=0;e<i;e++)h=(a=t[e]).xlb<=this.cx&&a.yub>this.cy,c=a.xlb<=this.cx&&a.ylb<=this.cy,l=a.xub>this.cx&&a.yub>this.cy,d=a.xub>this.cx&&a.ylb<=this.cy,h&&l&&d&&c?this.items.push(a):(h&&u.push(a),c&&f.push(a),l&&p.push(a),d&&_.push(a));return this.subdivide(u,f,p,_,s,r,o,n),this},insertItem:function(t){var e,i,s,r,o,n,a,h,l=[],c=[],d=[],u=[];return null===this.bbox?(e=r=1/0,s=i=-1/0,e=t.xlb<e?t.xlb:e,i=t.yub>i?t.yub:i,s=t.xub>s?t.xub:s,r=t.ylb<r?t.ylb:r,this.cx=.5*(e+s),this.cy=.5*(i+r),this.bbox=[e,i,s,r]):(e=this.bbox[0],i=this.bbox[1],s=this.bbox[2],r=this.bbox[3]),0===this.depth||this.items.length+1<this.capacity?(this.items.push(t),this):(o=t.xlb<=this.cx&&t.yub>this.cy,a=t.xlb<=this.cx&&t.ylb<=this.cy,n=t.xub>this.cx&&t.yub>this.cy,h=t.xub>this.cx&&t.ylb<=this.cy,o&&n&&h&&a?this.items.push(t):(o&&l.push(t),a&&d.push(t),n&&c.push(t),h&&u.push(t)),this.subdivide(l,d,c,u,e,i,s,r),this)},subdivide:function(t,e,i,s,r,o,n,a){return t.length>0&&(null===this.northWest&&(this.northWest=new JXG.Math.BoxQuadtree(this.depth,this.capacity,[r,o,this.cx,this.cy])),this.northWest.insert(t)),e.length>0&&(null===this.southWest&&(this.southWest=new JXG.Math.BoxQuadtree(this.depth,this.capacity,[r,this.cy,this.cx,a])),this.southWest.insert(e)),i.length>0&&(null===this.northEast&&(this.northEast=new JXG.Math.BoxQuadtree(this.depth,this.capacity,[this.cx,o,n,this.cy])),this.northEast.insert(i)),s.length>0&&(null===this.southEast&&(this.southEast=new JXG.Math.BoxQuadtree(this.depth,this.capacity,[this.cx,this.cy,n,a])),this.southEast.insert(s)),this},find:function(t){var e,i,r,o=[];for(i=this.items.length,e=0;e<i;e++)r=this.items[e],t[2]>=r.xlb&&t[0]<=r.xub&&t[3]<=r.yub&&t[1]>=r.ylb&&o.push(this.items[e]);return null!==this.northWest&&t[0]<=this.cx&t[1]>=this.cy&&s.Z.concat(o,this.northWest.find(t)),null!==this.southWest&&t[0]<=this.cx&t[3]<=this.cy&&s.Z.concat(o,this.southWest.find(t)),null!==this.northEast&&t[2]>=this.cx&t[1]>=this.cy&&s.Z.concat(o,this.northEast.find(t)),null!==this.southEast&&t[2]>=this.cx&t[3]<=this.cy&&s.Z.concat(o,this.southEast.find(t)),o},analyzeTree:function(){var t,e={number_items:this.items.length,depth:1};return null!==this.northWest&&(t=this.northWest.analyzeTree(),e.number_items+=t.number_items,e.depth=Math.max(e.depth,1+t.depth)),null!==this.southWest&&(t=this.southWest.analyzeTree(),e.number_items+=t.number_items,e.depth=Math.max(e.depth,1+t.depth)),null!==this.northEast&&(t=this.northEast.analyzeTree(),e.number_items+=t.number_items,e.depth=Math.max(e.depth,1+t.depth)),null!==this.southEast&&(t=this.southEast.analyzeTree(),e.number_items+=t.number_items,e.depth=Math.max(e.depth,1+t.depth)),e},plot:function(){var t,e=[],i=[];return e.push(this.bbox[0]),i.push(this.bbox[3]),e.push(this.bbox[2]),i.push(this.bbox[3]),e.push(this.bbox[2]),i.push(this.bbox[1]),e.push(this.bbox[0]),i.push(this.bbox[1]),e.push(this.bbox[0]),i.push(this.bbox[3]),e.push(NaN),i.push(NaN),null!==this.northWest&&(t=this.northWest.plot(),s.Z.concat(e,t[0]),s.Z.concat(i,t[1])),null!==this.northEast&&(t=this.northEast.plot(),s.Z.concat(e,t[0]),s.Z.concat(i,t[1])),null!==this.southEast&&(t=this.southEast.plot(),s.Z.concat(e,t[0]),s.Z.concat(i,t[1])),null!==this.southWest&&(t=this.southWest.plot(),s.Z.concat(e,t[0]),s.Z.concat(i,t[1])),[e,i]}});var c=o.Z.BoxQuadtree,d=__webpack_require__(336);t.Z.Math.Nlp={arr:function(t){return new Float64Array(t)},arr2:function(t,e){for(var i=0,s=new Array(t);i<t;)s[i]=this.arr(e),i++;return s},arraycopy:function(t,e,i,s,r){for(var o=0;o<r;)i[o+s]=t[o+e],o++},lastNumberOfEvaluations:0,GetLastNumberOfEvaluations:function(){return this.lastNumberOfEvaluations},Normal:0,MaxIterationsReached:1,DivergingRoundingErrors:2,FindMinimum:function(t,e,i,s,r,o,n,a,h){var l,c,d=i+2,u=this.arr(e+1),p=this;return this.lastNumberOfEvaluations=0,h&&console.log("Experimental feature 'testForRoundingErrors' is activated."),u[0]=0,this.arraycopy(s,0,u,1,e),c=function(e,i,s,r){var o,n,a=p.arr(e);return p.lastNumberOfEvaluations=p.lastNumberOfEvaluations+1,p.arraycopy(s,1,a,0,e),o=p.arr(i),n=t(e,i,a,o),p.arraycopy(o,0,r,1,i),n},l=this.cobylb(c,e,i,d,u,r,o,n,a,h),this.arraycopy(u,1,s,0,e),l},cobylb:function(t,e,i,s,r,o,n,a,h,l){var c,d,u,p,f,_,m,g,b,v,Z,C,y,P,E,x,O,M,w,S,T,N,A,D,k,R,L,B,I,j=-1,Y=0,V=0,X=e+1,U=i+1,F=o,z=0,J=!1,G=!1,W=0,H=0,K=0,$=this.arr(1+s),q=this.arr2(1+e,1+X),Q=this.arr2(1+e,1+e),tt=this.arr2(1+s,1+X),et=this.arr2(1+e,1+U),it=this.arr(1+e),st=this.arr(1+e),rt=this.arr(1+e),ot=this.arr(1+e),nt=this.arr(1+e);for(a>=2&&console.log("The initial value of RHO is "+F+" and PARMU is set to zero."),g=0,_=1/F,d=1;d<=e;++d)q[d][X]=r[d],q[d][d]=F,Q[d][d]=_;b=X,v=!1;t:for(;;){if(g>=h&&g>0){j=this.MaxIterationsReached;break t}for(++g,Y=t(e,i,r,$),V=0,p=1;p<=i;++p)V=Math.max(V,-$[p]);if(g!==a-1&&3!==a||this.PrintIterationResult(g,Y,V,r,e,a),$[U]=Y,$[s]=V,Z=!0,!v){for(Z=!1,d=1;d<=s;++d)tt[d][b]=$[d];if(g<=X){if(b<=e)if(tt[U][X]<=Y)r[b]=q[b][X];else{for(q[b][X]=r[b],p=1;p<=s;++p)tt[p][b]=tt[p][X],tt[p][X]=$[p];for(p=1;p<=b;++p){for(q[b][p]=-F,_=0,d=p;d<=b;++d)_-=Q[d][p];Q[b][p]=_}}if(g<=e){r[b=g]+=F;continue t}}v=!0}e:for(;;){i:do{if(!Z){for(C=tt[U][X]+z*tt[s][X],y=X,u=1;u<=e;++u)(_=tt[U][u]+z*tt[s][u])<C?(y=u,C=_):_===C&&0===z&&tt[s][u]<tt[s][y]&&(y=u);if(y<=e){for(d=1;d<=s;++d)_=tt[d][X],tt[d][X]=tt[d][y],tt[d][y]=_;for(d=1;d<=e;++d){for(_=q[d][y],q[d][y]=0,q[d][X]+=_,m=0,p=1;p<=e;++p)q[d][p]-=_,m-=Q[p][d];Q[y][d]=m}}if(P=0,l)for(d=1;d<=e;++d)for(u=1;u<=e;++u)_=this.DOT_PRODUCT_ROW_COL(Q,d,q,u,1,e)-(d===u?1:0),P=Math.max(P,Math.abs(_));if(P>.1){j=this.DivergingRoundingErrors;break t}for(p=1;p<=U;++p){for($[p]=-tt[p][X],u=1;u<=e;++u)nt[u]=tt[p][u]+$[p];for(d=1;d<=e;++d)et[d][p]=(p===U?-1:1)*this.DOT_PRODUCT_ROW_COL(nt,-1,Q,d,1,e)}for(J=!0,W=.25*F,E=2.1*F,u=1;u<=e;++u){for(x=0,O=0,p=1;p<=e;++p)x+=Q[u][p]*Q[u][p],O+=q[p][u]*q[p][u];it[u]=1/Math.sqrt(x),st[u]=Math.sqrt(O),(it[u]<W||st[u]>E)&&(J=!1)}if(!v&&!J){for(b=0,_=E,u=1;u<=e;++u)st[u]>_&&(b=u,_=st[u]);if(0===b)for(u=1;u<=e;++u)it[u]<_&&(b=u,_=it[u]);for(_=.5*F*it[b],p=1;p<=e;++p)ot[p]=_*Q[b][p];for(M=0,w=0,c=0,p=1;p<=U;++p)c=this.DOT_PRODUCT_ROW_COL(ot,-1,et,p,1,e),p<U&&(_=tt[p][X],M=Math.max(M,-c-_),w=Math.max(w,c-_));for(S=z*(M-w)>2*c?-1:1,_=0,d=1;d<=e;++d)ot[d]=S*ot[d],q[d][b]=ot[d],_+=Q[b][d]*ot[d];for(p=1;p<=e;++p)Q[b][p]/=_;for(u=1;u<=e;++u){if(u!==b)for(_=this.DOT_PRODUCT_ROW_COL(Q,u,ot,-1,1,e),p=1;p<=e;++p)Q[u][p]-=_*Q[b][p];r[u]=q[u][X]+ot[u]}continue t}if(!(G=this.trstlp(e,i,et,$,F,ot))){for(_=0,p=1;p<=e;++p)_+=ot[p]*ot[p];if(_<.25*F*F){v=!0;break i}}for(c=0,T=0,$[U]=0,p=1;p<=U;++p)c=$[p]-this.DOT_PRODUCT_ROW_COL(ot,-1,et,p,1,e),p<U&&(T=Math.max(T,c));if(z<1.5*(N=(H=tt[s][X]-T)>0?c/H:0))for(z=2*N,a>=2&&console.log("Increase in PARMU to "+z),A=tt[U][X]+z*tt[s][X],u=1;u<=e;++u)if((_=tt[U][u]+z*tt[s][u])<A||_===A&&0===z&&tt[s][u]<tt[s][X])continue e;for(K=z*H-c,p=1;p<=e;++p)r[p]=q[p][X]+ot[p];v=!0;continue t}for(Z=!1,D=tt[U][X]+z*tt[s][X]-(Y+z*V),0===z&&Y===tt[U][X]&&(K=H,D=tt[s][X]-V),k=D<=0?1:0,b=0,u=1;u<=e;++u)(_=Math.abs(this.DOT_PRODUCT_ROW_COL(Q,u,ot,-1,1,e)))>k&&(b=u,k=_),rt[u]=_*it[u];for(R=1.1*F,f=0,u=1;u<=e;++u)if(rt[u]>=W||rt[u]>=it[u]){if(_=st[u],D>0){for(_=0,p=1;p<=e;++p)_+=Math.pow(ot[p]-q[p][u],2);_=Math.sqrt(_)}_>R&&(f=u,R=_)}if(f>0&&(b=f),0!==b){for(_=0,d=1;d<=e;++d)q[d][b]=ot[d],_+=Q[b][d]*ot[d];for(p=1;p<=e;++p)Q[b][p]/=_;for(u=1;u<=e;++u)if(u!==b)for(_=this.DOT_PRODUCT_ROW_COL(Q,u,ot,-1,1,e),p=1;p<=e;++p)Q[u][p]-=_*Q[b][p];for(p=1;p<=s;++p)tt[p][b]=$[p];if(D>0&&D>=.1*K)continue e}}while(0);if(J){if(F<=n){j=this.Normal;break t}if(L=0,B=0,(F*=.5)<=1.5*n&&(F=n),z>0){for(I=0,p=1;p<=U;++p){for(B=L=tt[p][X],d=1;d<=e;++d)L=Math.min(L,tt[p][d]),B=Math.max(B,tt[p][d]);p<=i&&L<.5*B&&(_=Math.max(B,0)-L,I=I<=0?_:Math.min(I,_))}0===I?z=0:B-L<z*I&&(z=(B-L)/I)}a>=2&&console.log("Reduction in RHO to "+F+"  and PARMU = "+z),2===a&&this.PrintIterationResult(g,tt[U][X],tt[s][X],this.COL(q,X),e,a)}else v=!1}}switch(j){case this.Normal:if(a>=1&&console.log("%nNormal return from subroutine COBYLA%n"),G)return a>=1&&this.PrintIterationResult(g,Y,V,r,e,a),j;break;case this.MaxIterationsReached:a>=1&&console.log("%nReturn from subroutine COBYLA because the MAXFUN limit has been reached.%n");break;case this.DivergingRoundingErrors:a>=1&&console.log("%nReturn from subroutine COBYLA because rounding errors are becoming damaging.%n")}for(p=1;p<=e;++p)r[p]=q[p][X];return Y=tt[U][X],V=tt[s][X],a>=1&&this.PrintIterationResult(g,Y,V,r,e,a),j},trstlp:function(t,e,i,s,r,n){var a,h,l,c,d,u,p,f,_,m,g,b,v,Z,C,y,P,E,x,O,M,w,S,T,N,A,D,k,R,L,B,I,j,Y=0,V=0,X=0,U=this.arr2(1+t,1+t),F=this.arr(2+e),z=this.arr(2+e),J=this.arr(1+t),G=this.arr(1+t),W=this.arr(2+e),H=this.arr(2+e),K=e,$=0;for(l=1;l<=t;++l)U[l][l]=1,n[l]=0;if(a=0,h=0,e>=1){for(c=1;c<=e;++c)s[c]>h&&(h=s[c],a=c);for(c=1;c<=e;++c)H[c]=c,z[c]=h-s[c]}d=!0;do{t:for(;;){(!d||d&&0===h)&&(a=K=e+1,H[K]=K,z[K]=0),d=!1,u=0,p=0,f=0,_=0;e:do{if(m=K===e?h:-this.DOT_PRODUCT_ROW_COL(n,-1,i,K,1,t),0===p||m<u?(u=m,V=$,p=3):$>V?(V=$,p=3):--p,0===p)break t;if(g=0,a<=$){if(a<$){b=H[a],v=z[a],c=a;do{for(y=H[C=c+1],P=this.DOT_PRODUCT(this.PART(this.COL(U,c),1,t),this.PART(this.COL(i,y),1,t)),Y=o.Z.hypot(P,F[C]),E=F[C]/Y,x=P/Y,F[C]=E*F[c],F[c]=Y,l=1;l<=t;++l)Y=E*U[l][C]+x*U[l][c],U[l][C]=E*U[l][c]-x*U[l][C],U[l][c]=Y;H[c]=y,z[c]=z[C],c=C}while(c<$);H[c]=b,z[c]=v}if(--$,K>e)for(Y=1/F[$],c=1;c<=t;++c)J[c]=Y*U[c][$];else for(Y=this.DOT_PRODUCT_ROW_COL(J,-1,U,$+1,1,t),c=1;c<=t;++c)J[c]-=Y*U[c][$+1]}else{for(y=H[a],c=1;c<=t;++c)G[c]=i[c][y];for(O=0,c=t;c>$;){for(P=0,M=0,l=1;l<=t;++l)P+=Y=U[l][c]*G[l],M+=Math.abs(Y);if(w=M+.1*Math.abs(P),S=M+.2*Math.abs(P),(M>=w||w>=S)&&(P=0),0===O)O=P;else for(C=c+1,E=P/(Y=o.Z.hypot(P,O)),x=O/Y,O=Y,l=1;l<=t;++l)Y=E*U[l][c]+x*U[l][C],U[l][C]=E*U[l][C]-x*U[l][c],U[l][c]=Y;--c}if(0===O){g=-1,c=$;do{for(T=0,N=0,l=1;l<=t;++l)T+=Y=U[l][c]*G[l],N+=Math.abs(Y);if(w=N+.1*Math.abs(T),S=N+.2*Math.abs(T),N<w&&w<S){if((Y=T/F[c])>0&&H[c]<=e&&(j=z[c]/Y,(g<0||j<g)&&(g=j)),c>=2)for(A=H[c],l=1;l<=t;++l)G[l]-=Y*i[l][A];W[c]=Y}else W[c]=0}while(--c>0);if(g<0)break t;for(c=1;c<=$;++c)z[c]=Math.max(0,z[c]-g*W[c]);if(a<$){b=H[a],v=z[a],c=a;do{for(A=H[C=c+1],P=this.DOT_PRODUCT(this.PART(this.COL(U,c),1,t),this.PART(this.COL(i,A),1,t)),Y=o.Z.hypot(P,F[C]),E=F[C]/Y,x=P/Y,F[C]=E*F[c],F[c]=Y,l=1;l<=t;++l)Y=E*U[l][C]+x*U[l][c],U[l][C]=E*U[l][c]-x*U[l][C],U[l][c]=Y;H[c]=A,z[c]=z[C],c=C}while(c<$);H[c]=b,z[c]=v}if(0===(Y=this.DOT_PRODUCT(this.PART(this.COL(U,$),1,t),this.PART(this.COL(i,y),1,t))))break t;F[$]=Y,z[a]=0,z[$]=g}else F[++$]=O,z[a]=z[$],z[$]=0;if(H[a]=H[$],H[$]=y,K>e&&y!==K){for(c=$-1,P=this.DOT_PRODUCT(this.PART(this.COL(U,c),1,t),this.PART(this.COL(i,y),1,t)),Y=o.Z.hypot(P,F[$]),E=F[$]/Y,x=P/Y,F[$]=E*F[c],F[c]=Y,l=1;l<=t;++l)Y=E*U[l][$]+x*U[l][c],U[l][$]=E*U[l][c]-x*U[l][$],U[l][c]=Y;H[$]=H[c],H[c]=y,Y=z[c],z[c]=z[$],z[$]=Y}if(K>e)for(Y=1/F[$],c=1;c<=t;++c)J[c]=Y*U[c][$];else for(y=H[$],Y=(this.DOT_PRODUCT_ROW_COL(J,-1,i,y,1,t)-1)/F[$],c=1;c<=t;++c)J[c]-=Y*U[c][$]}for(D=r*r,R=0,k=0,l=1;l<=t;++l)Math.abs(n[l])>=1e-6*r&&(D-=n[l]*n[l]),R+=n[l]*J[l],k+=J[l]*J[l];if(D<=0)break t;if(Y=Math.sqrt(k*D),Math.abs(R)>=1e-6*Y&&(Y=Math.sqrt(k*D+R*R)),f=_=D/(Y+R),K===e){if(S=f+.2*h,f>=(w=f+.1*h)||w>=S)break e;f=Math.min(f,h)}for(c=1;c<=t;++c)G[c]=n[c]+f*J[c];if(K===e)for(X=h,h=0,c=1;c<=$;++c)Y=s[y=H[c]]-this.DOT_PRODUCT_ROW_COL(G,-1,i,y,1,t),h=Math.max(h,Y);c=$;do{for(L=0,B=0,l=1;l<=t;++l)L+=Y=U[l][c]*G[l],B+=Math.abs(Y);if(w=B+.1*Math.abs(L),S=B+.2*Math.abs(L),(B>=w||w>=S)&&(L=0),W[c]=L/F[c],c>=2)for(y=H[c],l=1;l<=t;++l)G[l]-=W[c]*i[l][y]}while(c-- >=2);for(K>e&&(W[$]=Math.max(0,W[$])),c=1;c<=t;++c)G[c]=n[c]+f*J[c];if(K>$)for(c=$+1;c<=K;++c){for(Z=h-s[y=H[c]],I=h+Math.abs(s[y]),l=1;l<=t;++l)Z+=Y=i[l][y]*G[l],I+=Math.abs(Y);w=I+.1*Math.abs(Z),S=I+.2*Math.abs(Z),(I>=w||w>=S)&&(Z=0),W[c]=Z}for(g=1,a=0,c=1;c<=K;++c)W[c]<0&&(Y=z[c]/(z[c]-W[c]))<g&&(g=Y,a=c);for(Y=1-g,c=1;c<=t;++c)n[c]=Y*n[c]+g*G[c];for(c=1;c<=K;++c)z[c]=Math.max(0,Y*z[c]+g*W[c]);K===e&&(h=X+g*(h-X))}while(a>0);if(f===_)return!0}}while(K===e);return!1},PrintIterationResult:function(t,e,i,s,r,o){o>1&&console.log("NFVALS = "+t+"  F = "+e+"  MAXCV = "+i),o>1&&console.log("X = "+this.PART(s,1,r))},ROW:function(t,e){return t[e].slice()},COL:function(t,e){var i,s=t.length,r=[];for(i=0;i<s;++i)r[i]=t[i][e];return r},PART:function(t,e,i){return t.slice(e,i+1)},FORMAT:function(t){return t.join(",")},DOT_PRODUCT:function(t,e){var i,s=0,r=t.length;for(i=0;i<r;++i)s+=t[i]*e[i];return s},DOT_PRODUCT_ROW_COL:function(t,e,i,s,r,o){var n,a=0;if(-1===e)for(n=r;n<=o;++n)a+=t[n]*i[n][s];else if(-1===s)for(n=r;n<=o;++n)a+=t[e][n]*i[n];else for(n=r;n<=o;++n)a+=t[e][n]*i[n][s];return a}};t.Z.Math.Nlp;var u=__webpack_require__(705),p=__webpack_require__(309);o.Z.Plot={checkReal:function(t){var e,i,s=!1,r=t.length;for(e=0;e<r;e++)if(i=t[e].usrCoords,!isNaN(i[1])&&!isNaN(i[2])&&Math.abs(i[0])>o.Z.eps){s=!0;break}return s},updateParametricCurveNaive:function(t,e,s,r){var o,n,a=!1,h=(s-e)/r;for(o=0;o<r;o++)n=e+o*h,t.points[o].setCoordinates(i.Z.COORDS_BY_USER,[t.X(n,a),t.Y(n,a)],!1),t.points[o]._t=n,a=!0;return t},isSegmentOutside:function(t,e,i,s,r){return e<0&&s<0||e>r.canvasHeight&&s>r.canvasHeight||t<0&&i<0||t>r.canvasWidth&&i>r.canvasWidth},isDistOK:function(t,e,i,s){return Math.abs(t)<i&&Math.abs(e)<s&&!isNaN(t+e)},isSegmentDefined:function(t,e,i,s){return!(isNaN(t+e)&&isNaN(i+s))},updateParametricCurveOld:function(e,s,r){var n,a,h,l,c,d,p,f,_,m,g,b=!1,v=new u.Z(i.Z.COORDS_BY_USER,[0,0],e.board,!1),Z=[],C=[],y=[],P=[],E=!1,x=0,O=function(t,e,i){var s,r=i[1]-t[1],n=i[2]-t[2],a=e[0]-t[1],h=e[1]-t[2],l=a*a+h*h;return l>=o.Z.eps&&(s=(r*a+n*h)/l)>0&&(s<=1?(r-=s*a,n-=s*h):(r-=a,n-=h)),o.Z.hypot(r,n)};for(t.Z.deprecated("Curve.updateParametricCurveOld()"),e.board.updateQuality===e.board.BOARD_QUALITY_LOW?(_=15,m=10,g=10):(_=21,m=.7,g=.7),P[0]=r-s,n=1;n<_;n++)P[n]=.5*P[n-1];n=1,Z[0]=1,C[0]=0,a=s,v.setCoordinates(i.Z.COORDS_BY_USER,[e.X(a,b),e.Y(a,b)],!1),b=!0,c=v.scrCoords[1],d=v.scrCoords[2],a=r,v.setCoordinates(i.Z.COORDS_BY_USER,[e.X(a,b),e.Y(a,b)],!1),h=v.scrCoords[1],l=v.scrCoords[2],y[0]=[h,l],p=1,f=0,e.points=[],e.points[x++]=new u.Z(i.Z.COORDS_BY_SCREEN,[c,d],e.board,!1);do{for(E=this.isDistOK(h-c,l-d,m,g)||this.isSegmentOutside(c,d,h,l,e.board);f<_&&(!E||f<6)&&(f<=7||this.isSegmentDefined(c,d,h,l));)Z[p]=n,C[p]=f,y[p]=[h,l],p+=1,a=s+(n=2*n-1)*P[++f],v.setCoordinates(i.Z.COORDS_BY_USER,[e.X(a,b),e.Y(a,b)],!1,!0),h=v.scrCoords[1],l=v.scrCoords[2],E=this.isDistOK(h-c,l-d,m,g)||this.isSegmentOutside(c,d,h,l,e.board);x>1&&O(e.points[x-2].scrCoords,[h,l],e.points[x-1].scrCoords)<.015&&(x-=1),e.points[x]=new u.Z(i.Z.COORDS_BY_SCREEN,[h,l],e.board,!1),e.points[x]._t=a,x+=1,c=h,d=l,h=y[p-=1][0],l=y[p][1],f=C[p]+1,n=2*Z[p]}while(p>0&&x<5e5);return e.numberPoints=e.points.length,e},_insertPoint_v2:function(t,e,i){var s=!isNaN(this._lastCrds[1]+this._lastCrds[2]),r=!isNaN(e.scrCoords[1]+e.scrCoords[2]),o=t.board.canvasWidth,n=t.board.canvasHeight,a=500;(!(r=r&&e.scrCoords[1]>-500&&e.scrCoords[2]>-500&&e.scrCoords[1]<o+a&&e.scrCoords[2]<n+a)&&s||r&&(!s||Math.abs(e.scrCoords[1]-this._lastCrds[1])>.7||Math.abs(e.scrCoords[2]-this._lastCrds[2])>.7))&&(e._t=i,t.points.push(e),this._lastCrds=e.copy("scrCoords"))},neighborhood_isNaN_v2:function(t,e){var s,r,n=new u.Z(i.Z.COORDS_BY_USER,[0,0],t.board,!1);return s=e+o.Z.eps,n.setCoordinates(i.Z.COORDS_BY_USER,[t.X(s,!0),t.Y(s,!0)],!1),r=n.usrCoords,!!(isNaN(r[1]+r[2])||(s=e-o.Z.eps,n.setCoordinates(i.Z.COORDS_BY_USER,[t.X(s,!0),t.Y(s,!0)],!1),r=n.usrCoords,isNaN(r[1]+r[2])))},_borderCase:function(t,e,s,r,o,n,a,h){var l,c,d,p,f,_,m=null,g=!1;if(h<=1){if(c=new u.Z(i.Z.COORDS_BY_USER,[0,0],t.board,!1),isNaN(e[1]+e[2])&&!isNaN(r[1]+r[2])&&!this.neighborhood_isNaN_v2(t,o))return!1;if(isNaN(s[1]+s[2])&&!isNaN(r[1]+r[2])&&!this.neighborhood_isNaN_v2(t,n))return!1;if(isNaN(r[1]+r[2])&&(!isNaN(e[1]+e[2])||!isNaN(s[1]+s[2]))&&!this.neighborhood_isNaN_v2(t,a))return!1;p=0;do{if(isNaN(e[1]+e[2])&&!isNaN(r[1]+r[2]))f=o,_=a;else if(isNaN(s[1]+s[2])&&!isNaN(r[1]+r[2]))f=n,_=a;else if(isNaN(r[1]+r[2])&&!isNaN(s[1]+s[2]))f=a,_=n;else{if(!isNaN(r[1]+r[2])||isNaN(e[1]+e[2]))return!1;f=a,_=o}l=.5*(f+_),c.setCoordinates(i.Z.COORDS_BY_USER,[t.X(l,!0),t.Y(l,!0)],!1),d=c.usrCoords,(g=isNaN(d[1]+d[2]))?f=l:_=l,++p}while(g&&p<30);if(p<30&&(m=d.slice(),r=d.slice(),_=l),null!==m)return this._insertPoint_v2(t,new u.Z(i.Z.COORDS_BY_USER,m,t.board,!1)),!0}return!1},_plotRecursive_v2:function(t,e,s,r,o,n,a){var h,l,c,d,p,f=0,_=new u.Z(i.Z.COORDS_BY_USER,[0,0],t.board,!1);if(!(t.numberPoints>65536))return n<this.nanLevel&&this._isUndefined(t,e,s,r,o)||n<this.nanLevel&&this._isOutside(e,s,r,o,t.board)?this:(h=.5*(s+o),_.setCoordinates(i.Z.COORDS_BY_USER,[t.X(h,!0),t.Y(h,!0)],!1),l=_.scrCoords,this._borderCase(t,e,r,l,s,o,h,n)||(c=this._triangleDists(e,r,l),d=n<this.smoothLevel&&c[3]<a,p=n<=this.jumpLevel&&(isNaN(c[0])||isNaN(c[1])||isNaN(c[2]))||n<this.jumpLevel&&(c[2]>.99*c[0]||c[1]>.99*c[0]||c[0]===1/0||c[1]===1/0||c[2]===1/0),n<this.smoothLevel+2&&c[0]<.5*(c[1]+c[2])&&(f=0,d=!1),--n,p?this._insertPoint_v2(t,new u.Z(i.Z.COORDS_BY_SCREEN,[NaN,NaN],t.board,!1),h):n<=f||d?this._insertPoint_v2(t,_,h):(this._plotRecursive_v2(t,e,s,l,h,n,a),isNaN(_.scrCoords[1]+_.scrCoords[2])||this._insertPoint_v2(t,_,h),this._plotRecursive_v2(t,l,h,r,o,n,a))),this)},updateParametricCurve_v2:function(t,e,s){var r,o,n,a,h,l,c,d,p,f=!1,_=new u.Z(i.Z.COORDS_BY_USER,[0,0],t.board,!1),m=new u.Z(i.Z.COORDS_BY_USER,[0,0],t.board,!1);return t.board.updateQuality===t.board.BOARD_QUALITY_LOW?(h=t.evalVisProp("recursiondepthlow")||13,l=2,this.smoothLevel=h-6,this.jumpLevel=3):(h=t.evalVisProp("recursiondepthhigh")||17,l=2,this.smoothLevel=h-9,this.jumpLevel=2),this.nanLevel=h-4,t.points=[],"x"===this.xterm?(c=.3*((d=t.board.getBoundingBox())[2]-d[0]),r=Math.max(e,d[0]-c),o=Math.min(s,d[2]+c)):(r=e,o=s),_.setCoordinates(i.Z.COORDS_BY_USER,[t.X(r,f),t.Y(r,f)],!1),f=!0,m.setCoordinates(i.Z.COORDS_BY_USER,[t.X(o,f),t.Y(o,f)],!1),p=this._findStartPoint(t,_.scrCoords,r,m.scrCoords,o),_.setCoordinates(i.Z.COORDS_BY_SCREEN,p[0],!1),r=p[1],p=this._findStartPoint(t,m.scrCoords,o,_.scrCoords,r),m.setCoordinates(i.Z.COORDS_BY_SCREEN,p[0],!1),o=p[1],this._visibleArea=[r,o],n=_.copy("scrCoords"),a=m.copy("scrCoords"),_._t=r,t.points.push(_),this._lastCrds=_.copy("scrCoords"),this._plotRecursive_v2(t,n,r,a,o,h,l),m._t=o,t.points.push(m),t.numberPoints=t.points.length,t},_insertLimesPoint:function(t,e,s,r,n){var a,h,l;Math.abs(this._lastUsrCrds[1])===1/0&&Math.abs(n.left_x)===1/0||Math.abs(this._lastUsrCrds[2])===1/0&&Math.abs(n.left_y)===1/0||((a=new u.Z(i.Z.COORDS_BY_USER,[n.left_x,n.left_y],t.board))._t=s,t.points.push(a),isNaN(n.left_x)||isNaN(n.left_y)||isNaN(n.right_x)||isNaN(n.right_y)||!(Math.abs(n.left_x-n.right_x)>o.Z.eps||Math.abs(n.left_y-n.right_y)>o.Z.eps)||((h=new u.Z(i.Z.COORDS_BY_SCREEN,e,t.board))._t=s,t.points.push(h)),(l=new u.Z(i.Z.COORDS_BY_USER,[n.right_x,n.right_y],t.board))._t=s,t.points.push(l),this._lastScrCrds=l.copy("scrCoords"),this._lastUsrCrds=l.copy("usrCoords"))},_insertPoint:function(t,e,r,o,n){var a,h=!isNaN(this._lastScrCrds[1]+this._lastScrCrds[2]),l=!isNaN(e[1]+e[2]),c=t.board.canvasWidth,d=t.board.canvasHeight,p=500;s.Z.exists(n)?this._insertLimesPoint(t,e,r,o,n):(l=l&&e[1]>-500&&e[2]>-500&&e[1]<c+p&&e[2]<d+p,(h||l)&&(l&&h&&Math.abs(e[1]-this._lastScrCrds[1])<.8&&Math.abs(e[2]-this._lastScrCrds[2])<.8||Math.abs(e[1])===1/0&&Math.abs(this._lastUsrCrds[1])===1/0||Math.abs(e[2])===1/0&&Math.abs(this._lastUsrCrds[2])===1/0||((a=new u.Z(i.Z.COORDS_BY_SCREEN,e,t.board))._t=r,t.points.push(a),this._lastScrCrds=a.copy("scrCoords"),this._lastUsrCrds=a.copy("usrCoords"))))},_triangleDists:function(t,e,i){var s;return s=[t[0]*e[0],.5*(t[1]+e[1]),.5*(t[2]+e[2])],[h.Z.distance(t,e,3),h.Z.distance(t,i,3),h.Z.distance(i,e,3),h.Z.distance(i,s,3)]},_isUndefined:function(t,e,s,r,o){var n,a,h;if(!isNaN(e[1]+e[2])||!isNaN(r[1]+r[2]))return!1;for(h=new u.Z(i.Z.COORDS_BY_USER,[0,0],t.board,!1),a=0;a<20;++a)if(n=s+Math.random()*(o-s),h.setCoordinates(i.Z.COORDS_BY_USER,[t.X(n,!0),t.Y(n,!0)],!1),!isNaN(h.scrCoords[0]+h.scrCoords[1]+h.scrCoords[2]))return!1;return!0},_isOutside:function(t,e,i,s,r){var o=500,n=r.canvasWidth,a=r.canvasHeight;return!!(t[1]<-500&&i[1]<-500||t[2]<-500&&i[2]<-500||t[1]>n+o&&i[1]>n+o||t[2]>a+o&&i[2]>a+o)},_isOutsidePoint:function(t,e){var i=500,s=e.canvasWidth,r=e.canvasHeight;return!!(t[1]<-500||t[2]<-500||t[1]>s+i||t[2]>r+i)},_findStartPoint:function(t,e,i,s,r){return[e,i]},_getBorderPos:function(t,e,s,r,n,a,h){var l,c,d,p,f,_;if(c=new u.Z(i.Z.COORDS_BY_USER,[0,0],t.board,!1),p=0,isNaN(s[1]+s[2])&&!isNaN(n[1]+n[2]))_=e,f=r;else if(isNaN(h[1]+h[2])&&!isNaN(n[1]+n[2]))_=a,f=r;else if(isNaN(n[1]+n[2])&&!isNaN(h[1]+h[2]))_=r,f=a;else{if(!isNaN(n[1]+n[2])||isNaN(s[1]+s[2]))return!1;_=r,f=e}do{l=.5*(f+_),c.setCoordinates(i.Z.COORDS_BY_USER,[t.X(l,!0),t.Y(l,!0)],!1),d=c.usrCoords,isNaN(d[1]+d[2])?_=l:f=l,++p}while(p<30&&Math.abs(f-_)>o.Z.eps);return l},_getCuspPos:function(t,e,i){var s=[t.X(e,!0),t.Y(e,!0)],r=[t.X(i,!0),t.Y(i,!0)];return d.Z.fminbr((function(e){var i=[t.X(e,!0),t.Y(e,!0)];return-(o.Z.hypot(s[0]-i[0],s[1]-i[1])+o.Z.hypot(r[0]-i[0],r[1]-i[1]))}),[e,i],t)},_getJumpPos:function(t,e,i){return d.Z.fminbr((function(e){var i=o.Z.eps*o.Z.eps,s=[t.X(e,!0),t.Y(e,!0)],r=[t.X(e+i,!0),t.Y(e+i,!0)];return-Math.abs((r[1]-s[1])/(r[0]-s[0]))}),[e,i],t)},_getLimits:function(t,e){var i,s,r,o,n,h=2/(t.maxX()-t.minX());return s=(i=a.limit(e,-h,t.X))[0],"infinite"===i[1]&&(s=Math.sign(s)*(1/0)),o=(i=a.limit(e,-h,t.Y))[0],"infinite"===i[1]&&(o=Math.sign(o)*(1/0)),r=(i=a.limit(e,h,t.X))[0],"infinite"===i[1]&&(r=Math.sign(r)*(1/0)),n=(i=a.limit(e,h,t.Y))[0],"infinite"===i[1]&&(n=Math.sign(n)*(1/0)),{left_x:s,left_y:o,right_x:r,right_y:n,t:e}},_getLimes:function(t,e,i,s,r,o,n,a,h){var l;return"border"===a?l=this._getBorderPos(t,e,i,s,r,o,n):"cusp"===a?l=this._getCuspPos(t,e,o):"jump"===a&&(l=this._getJumpPos(t,e,o)),this._getLimits(t,l)},_plotNonRecursive:function(t,e,i,s,r,o){var n,a,h,l,c,d,u,p,f,_,m,g=null,b=!1,v="",Z=[],C=0;for(p=t.board.origin.scrCoords,Z[C++]=[e,i,s,r,o,1/0];C>0;){if(e=(m=Z[--C])[0],i=m[1],s=m[2],r=m[3],f=m[4],_=m[5],b=!1,v="",g=null,t.points.length>65536)return;if(f<this.nanLevel){if(this._isUndefined(t,e,i,s,r))continue;if(this._isOutside(e,i,s,r,t.board))continue}n=.5*(i+r),d=t.X(n,!0),u=t.Y(n,!0),a=[1,p[1]+d*t.board.unitX,p[2]-u*t.board.unitY],h=this._triangleDists(e,s,a),l=isNaN(e[1]+e[2]),c=isNaN(s[1]+s[2]),l&&!c||!l&&c?v="border":h[0]>.66*_||h[0]<this.cusp_threshold*(h[1]+h[2])||h[1]>5*h[2]||h[2]>5*h[1]?v="cusp":(h[2]>this.jump_threshold*h[0]||h[1]>this.jump_threshold*h[0]||h[0]===1/0||h[1]===1/0||h[2]===1/0)&&(v="jump"),b=""===v&&f<this.smoothLevel&&h[3]<this.smooth_threshold,f<this.testLevel&&!b&&(""===v?b=!0:g=this._getLimes(t,i,e,n,a,r,s,v,f)),null!==g?(a=[1,NaN,NaN],this._insertPoint(t,a,n,f,g)):f<=0||b?this._insertPoint(t,a,n,f,null):(Z[C++]=[a,n,s,r,f-1,h[0]],Z[C++]=[e,i,a,n,f-1,h[0]])}return this},updateParametricCurve_v3:function(t,e,s){var r,o,n,a,h,l,c,d,p=!1,f=new u.Z(i.Z.COORDS_BY_USER,[0,0],t.board,!1),_=new u.Z(i.Z.COORDS_BY_USER,[0,0],t.board,!1);return h=t.board.updateQuality===t.board.BOARD_QUALITY_LOW?t.evalVisProp("recursiondepthlow")||14:t.evalVisProp("recursiondepthhigh")||17,this.smoothLevel=7,this.nanLevel=h-4,this.testLevel=4,this.cusp_threshold=.5,this.jump_threshold=.99,this.smooth_threshold=2,t.points=[],"x"===t.xterm?(l=.3*((c=t.board.getBoundingBox())[2]-c[0]),r=Math.max(e,c[0]-l),o=Math.min(s,c[2]+l)):(r=e,o=s),f.setCoordinates(i.Z.COORDS_BY_USER,[t.X(r,p),t.Y(r,p)],!1),p=!0,_.setCoordinates(i.Z.COORDS_BY_USER,[t.X(o,p),t.Y(o,p)],!1),d=this._findStartPoint(t,f.scrCoords,r,_.scrCoords,o),f.setCoordinates(i.Z.COORDS_BY_SCREEN,d[0],!1),r=d[1],d=this._findStartPoint(t,_.scrCoords,o,f.scrCoords,r),_.setCoordinates(i.Z.COORDS_BY_SCREEN,d[0],!1),o=d[1],this._visibleArea=[r,o],n=f.copy("scrCoords"),a=_.copy("scrCoords"),f._t=r,t.points.push(f),this._lastScrCrds=f.copy("scrCoords"),this._lastUsrCrds=f.copy("usrCoords"),this._plotNonRecursive(t,n,r,a,o,h),_._t=o,t.points.push(_),t.numberPoints=t.points.length,t},_criticalInterval:function(t,e,i){var s,r,o,n,a,h,l,c=!1,d=-1/0,u=!1,f=!1,_=0,m=[],g=[],b=[];for(l=p.Z.abs(t),(n=p.Z.median(l))<1e-7?(n=1e-7,u=!0):n*=this.criticalThreshold,s=0;s<e;s++)l[s]>n?(b.push({i:s,v:t[s],group:_}),d=s,c||(c=!0)):c&&s>d+4&&(b.length>0&&m.push(b.slice(0)),b=[],c=!1,_++);for(c&&b.length>1&&m.push(b.slice(0)),u&&0===m.length&&(f=!0),r=0;r<m.length;r++)if(g[r]="point",!((o=m[r].length)<64)){for(h=0,a=Math.sign(m[r][0].v),s=1;s<o;s++)Math.sign(m[r][s].v)!==a&&(h++,a=Math.sign(m[r][s].v));6*h>o&&(g[r]="interval")}return{smooth:f,groups:m,types:g}},Component:function(){this.left_isNaN=!1,this.right_isNaN=!1,this.left_t=null,this.right_t=null,this.t_values=[],this.x_values=[],this.y_values=[],this.len=0},findComponents:function(t,e,i,s){var r,o,n,a,h,l,c=[],d=0,u=0,p=0,f=!1,_=!1;for(n=(i-e)/s,c[d]=new this.Component,l=c[d],r=0,o=e;r<=s;r++,o+=n)a=t.X(o,_),h=t.Y(o,_),isNaN(a)||isNaN(h)?++p>1&&f&&(l.right_isNaN=!0,l.right_t=o-n,l.len=u,f=!1,c[++d]=new this.Component,l=c[d],p=0):(f||(f=!0,u=0,p>0&&(l.left_t=o-n,l.left_isNaN=!0)),p=0,l.t_values[u]=o,l.x_values[u]=a,l.y_values[u]=h,u++),0===r&&(_=!0);return f?l.len=u:c.pop(),c},getPointType:function(t,e,i,s,r,o,n){var a=r[0],h=o[0],l=s.length,c={idx:e,t:i,x:a[e],y:h[e],type:"other"};return e<5?(c.type="borderleft",c.idx=0,c.t=s[0],c.x=a[0],c.y=h[0],c):e>n-6?(c.type="borderright",c.idx=l-1,c.t=s[l-1],c.x=a[l-1],c.y=h[l-1],c):c},newtonApprox:function(t,e,i,s,r){var o,n=0;for(o=s;o>0;o--)n=(n+r[o][t])*(e-(o-1)*i)/o;return n+r[0][t]},thiele:function(t,e,i,s,r){var o,n=0;for(o=r;o>1;o--)n=(t-i[s+o])/(e[o][s+1]-e[o-2][s+1]+n);return e[0][s+1]+(t-i[s+1])/(e[1][s+1]+n)},differenceMethodExperiments:function(t,e){var i,s,r,o,n,a,h,l,c,d,u,p=t.t_values,f=t.x_values,_=t.y_values,m=[],g=[],b=[],v=[],Z=[],C=[],y=[],P=[],E=0,x=[];for(n=p[1]-p[0],Z.push([]),C.push([]),y.push([]),P.push([]),r=_.length,i=0;i<r;i++)Z[0][i]=f[i],C[0][i]=_[i],y[0][i]=f[i],P[0][i]=_[i];for(Z.push([]),C.push([]),y.push([]),P.push([]),a=n,r=_.length-1,i=0;i<r;i++)m[i]=f[i+1]-f[i],g[i]=_[i+1]-_[i],b[i]=m[i],v[i]=g[i],Z[1][i]=m[i],C[1][i]=g[i],y[1][i]=a/m[i],P[1][i]=a/g[i];for(r--,o=Math.min(8,_.length-1),s=1;s<o;s++){for(Z.push([]),C.push([]),y.push([]),P.push([]),a*=n,i=0;i<r;i++)m[i]=m[i+1]-m[i],g[i]=g[i+1]-g[i],Z[s+1][i]=m[i],C[s+1][i]=g[i],y[s+1][i]=a/(y[s][i+1]-y[s][i])+y[s-1][i+1],P[s+1][i]=a/(P[s][i+1]-P[s][i])+P[s-1][i+1];if(!1===(u=this._criticalPoints(g,r,s))){console.log("Polynomial of degree",s),u=[];break}if(u.length>0&&++E>1&&s%2==0)break;r--}for(i=0;i<u.length;i++){for(l=-1/0,c=0;c<u[i].length;c++)(d=Math.abs(u[i][c].v))>l&&(l=d,h=c);h=Math.floor(u[i][h].i+s/2),x.push(this.getPointType(e,h,p,f,_,b,v,r+1))}return[x,Z,C,y,P]},getCenterOfCriticalInterval:function(t,e,i){var s,r,o,n,a,h=0,l=0,c=i[1]-i[0],d=[];for(s=-1/0,d=[],r=0;r<t.length;r++)(n=Math.abs(t[r].v))>s?(d=[r],s=n,o=r):s===n&&d.push(r);if(d.length>0&&(a=d.reduce((function(t,e){return t+e}),0)/d.length,o=Math.floor(a),a+=t[0].i),s<1/0){for(r=0;r<t.length;r++)h+=Math.abs(t[r].v)*t[r].i,l+=Math.abs(t[r].v);a=h/l}return a+=e/2,[t[o].i+e/2,a,i[Math.floor(a)]+c*(a-Math.floor(a))]},differenceMethod:function(t,e){var i,s,r,o,n,a,h,l,c,d,u=t.t_values,p=t.x_values,f=t.y_values,_=[],m=[],g=0,b=-1,v=-1,Z=[],C=[];for(r=f.length,_.push(new Float64Array(p)),m.push(new Float64Array(f)),r--,o=Math.min(12,r),s=0;s<o&&(_.push(new Float64Array(r)),m.push(new Float64Array(r)),_[s+1]=_[s].map((function(t,e,i){return i[e+1]-t})),m[s+1]=m[s].map((function(t,e,i){return i[e+1]-t})),!0===(l=this._criticalInterval(m[s+1],r,s)).smooth&&(v=s,Z=[]),h=this._criticalInterval(_[s+1],r,s),-1===b&&!0===h.smooth&&(b=s),!(v>=0));s++){if(l.groups.length>0&&++g>2&&(s+1)%2==0){Z=l.groups,d=l.types;break}r--}for(i=0;i<Z.length;i++)"interval"!==d[i]&&(a=this.getCenterOfCriticalInterval(Z[i],s+1,u),l[0],n=Math.floor(a[1]),c=a[2],C.push(this.getPointType(e,n,c,u,_,m,r+1)));return[C,_,m,b,v]},_insertPoint_v4:function(t,e,s,r){var o,n,a,h=null;t.points.length>0&&(h=t.points[t.points.length-1].scrCoords),o=new u.Z(i.Z.COORDS_BY_USER,e,t.board),null!==h&&(n=o.scrCoords[1]-h[1])*n+(a=o.scrCoords[2]-h[2])*a<.8*.8||(o._t=s,t.points.push(o))},getInterval:function(e,i,s){var r,o;return n.Z.disable(),r=n.Z.Interval(i,s),e.board.mathLib=n.Z,e.board.mathLibJXG=n.Z,o=e.Y(r,!0),e.board.mathLib=Math,e.board.mathLibJXG=t.Z.Math,o},sign:function(t){return t<0?-1:t>0?1:0},handleBorder:function(t,e,i,r,o){var n,a,h,l,c,d,u,p,f,_,m,g,b=i.idx;if(g=e.t_values[1]-e.t_values[0],"borderleft"===i.type?h=(a=n=e.left_isNaN?e.left_t:i.t-g)+g:"borderright"===i.type?a=(h=n=e.right_isNaN?e.right_t:i.t+g)-g:console.log("No bordercase!!!"),0!==(_=this.findComponents(t,a,h,32)).length){for("borderleft"===i.type&&(a=_[0].left_t,h=_[0].t_values[0],g=_[0].t_values[1]-_[0].t_values[0],n=a=null===a?h-g:a,l=this.getInterval(t,a,h),s.Z.isObject(l)&&(u=l.lo,p=l.hi,c=t.X(n,!0),d=o[1][b]<0?p:u,this._insertPoint_v4(t,[1,c,d],n))),m=_[0].t_values.length,f=0;f<m;f++)n=_[0].t_values[f],c=_[0].x_values[f],d=_[0].y_values[f],this._insertPoint_v4(t,[1,c,d],n);"borderright"===i.type&&(a=_[0].t_values[m-1],h=_[0].right_t,g=_[0].t_values[1]-_[0].t_values[0],n=h=null===h?a+g:h,l=this.getInterval(t,a,h),s.Z.isObject(l)&&(u=l.lo,p=l.hi,c=t.X(n,!0),d=o[1][b]>0?p:u,this._insertPoint_v4(t,[1,c,d],n)))}},_seconditeration_v4:function(t,e,i,s,r){var o,n,a,h,l,c,d,u,p,f,_,m,g;for(n=e.t_values[i.idx-2],a=e.t_values[i.idx+2],l=this.findComponents(t,n,a,64),d=0;d<l.length;d++){for(c=l[d],u=(h=this.differenceMethod(c,t))[0],f=h[1],_=h[2],m=0,p=0;p<=u.length;p++){for(g=p===u.length?c.len:u[p].idx,o=m;o<g;o++)isNaN(c.x_values[o])||isNaN(c.y_values[o])||this._insertPoint_v4(t,[1,c.x_values[o],c.y_values[o]],c.t_values[o]);p<u.length&&(this.handleSingularity(t,c,u[p],f,_),m=u[p].idx+1)}g=c.len,d<l.length-1&&this._insertPoint_v4(t,[1,NaN,NaN],c.right_t)}return this},_recurse_v4:function(t,e,i,s,r,n,a,h){var l,c,d=.5*(e+i),u=t.X(d,!0),p=t.Y(d,!0);0!==h?(l=(u-s)*t.board.unitX,c=(p-r)*t.board.unitY,o.Z.hypot(l,c)>2?this._recurse_v4(t,e,d,s,r,u,p,h-1):this._insertPoint_v4(t,[1,u,p],d),l=(u-n)*t.board.unitX,c=(p-a)*t.board.unitY,o.Z.hypot(l,c)>2?this._recurse_v4(t,d,i,u,p,n,a,h-1):this._insertPoint_v4(t,[1,u,p],d)):this._insertPoint_v4(t,[1,NaN,NaN],d)},handleSingularity:function(t,e,i,r,o){var n,a,h,l,c,d,u,p,f,_,m,g=i.idx,b=100;n=i.t,console.log("HandleSingularity at t =",n),a=e.t_values[g-5],h=e.t_values[g+5],l=this.getInterval(t,a,h),s.Z.isObject(l)?(p=l.lo,f=l.hi):o[0][g-1]<o[0][g+1]?(p=o[0][g-1],f=o[0][g+1]):(p=o[0][g+1],f=o[0][g-1]),u=t.X(n,!0),_=(o[0][g-3]-o[0][g-5])/(e.t_values[g-3]-e.t_values[g-5]),m=(o[0][g+3]-o[0][g+5])/(e.t_values[g+3]-e.t_values[g+5]),console.log(":::",_,m),_<-100?(this._insertPoint_v4(t,[1,u,p],n,!0),m<=b&&this._insertPoint_v4(t,[1,NaN,NaN],n)):_>b?(this._insertPoint_v4(t,[1,u,f],n),m>=-100&&this._insertPoint_v4(t,[1,NaN,NaN],n)):(p===-1/0&&(this._insertPoint_v4(t,[1,u,p],n,!0),this._insertPoint_v4(t,[1,NaN,NaN],n)),f===1/0&&(this._insertPoint_v4(t,[1,NaN,NaN],n),this._insertPoint_v4(t,[1,u,f],n,!0)),i.t<e.t_values[g]?(c=g-1,d=g):(c=g,d=g+1),a=e.t_values[c],h=e.t_values[d],this._recurse_v4(t,a,h,r[0][c],o[0][c],r[0][d],o[0][d],10)),m<-100?this._insertPoint_v4(t,[1,u,f],n):m>b&&this._insertPoint_v4(t,[1,u,p],n)},steps:1021,criticalThreshold:1e3,plot_v4:function(t,e,i,r){var o,n,a,h,l,c,u,p,f,_,m,g,b,v,Z,C=(i-e)/r,y=function(e){return t.Y(e,!0)},P=function(e){return-t.Y(e,!0)},E=.5*C;for(a=this.findComponents(t,e,i,r),h=0;h<a.length;h++){for(l=a[h],c=(f=this.differenceMethod(l,t))[0],_=f[1],m=f[2],0!==c.length&&"borderleft"===c[0].type||c.unshift({idx:0,t:l.t_values[0],x:l.x_values[0],y:l.y_values[0],type:"borderleft"}),"borderright"!==c[c.length-1].type&&(n=l.t_values.length,c.push({idx:n-1,t:l.t_values[n-1],x:l.x_values[n-1],y:l.y_values[n-1],type:"borderright"})),p=0,u=0;u<=c.length;u++){for(n=u===c.length?l.len:c[u].idx-1,o=p;o<n-2;o++)this._insertPoint_v4(t,[1,l.x_values[o],l.y_values[o]],l.t_values[o]),o>=p+3&&o<n-3&&m.length>3&&Math.abs(m[2][o])>.2*Math.abs(m[0][o])&&(g=l.t_values[o],E=.25*C,Z=this.getInterval(t,g,g+C),s.Z.isObject(Z)?m[2][o]>0?this._insertPoint_v4(t,[1,g+E,Z.lo],g+E):this._insertPoint_v4(t,[1,g+C-E,Z.hi],g+C-E):(b=d.Z.fminbr(y,[g,g+C]))<(v=d.Z.fminbr(P,[g,g+C]))?(this._insertPoint_v4(t,[1,t.X(b,!0),t.Y(b,!0)],b),this._insertPoint_v4(t,[1,t.X(v,!0),t.Y(v,!0)],v)):(this._insertPoint_v4(t,[1,t.X(v,!0),t.Y(v,!0)],v),this._insertPoint_v4(t,[1,t.X(b,!0),t.Y(b,!0)],b)));u<c.length&&(o=c[u].idx,"borderleft"===c[u].type||"borderright"===c[u].type?this.handleBorder(t,l,c[u],_,m):this._seconditeration_v4(t,l,c[u],_,m),p=c[u].idx+1+1)}n=l.len,h<a.length-1&&this._insertPoint_v4(t,[1,NaN,NaN],l.right_t)}},updateParametricCurve_v4:function(t,e,i){var s,r,o,n;"x"===t.xterm?(o=.3*((n=t.board.getBoundingBox())[2]-n[0]),s=Math.max(e,n[0]-o),r=Math.min(i,n[2]+o)):(s=e,r=i),t.points=[],this.plot_v4(t,s,r,this.steps),t.numberPoints=t.points.length},updateParametricCurve:function(t,e,i){return this.updateParametricCurve_v2(t,e,i)}};var f=o.Z.Plot;o.Z.ImplicitPlot=function(t,e,i,r,n){var a={resolution_out:5,resolution_in:5,max_steps:1024,alpha_0:.05,tol_u0:o.Z.eps,tol_newton:1e-7,tol_cusp:.05,tol_progress:1e-4,qdt_box:.2,kappa_0:.2,delta_0:.05,h_initial:.1,h_critical:.001,h_max:1,loop_dist:.09,loop_dir:.99,loop_detection:!0,unitX:10,unitY:10};this.config=s.Z.merge(a,e),this.f=i,this.dfx=null,this.dfy=null,s.Z.isFunction(r)?this.dfx=r:this.dfx=function(t,e){var i=o.Z.eps*o.Z.eps;return.5*(this.f(t+i,e)-this.f(t-i,e))/i},s.Z.isFunction(n)?this.dfy=n:this.dfy=function(t,e){var i=o.Z.eps*o.Z.eps;return.5*(this.f(t,e+i)-this.f(t,e-i))/i},this.bbox=t,this.qdt=new c(20,5,t),this.components=[]},s.Z.extend(o.Z.ImplicitPlot.prototype,{plot:function(){var t,e,i,s,r,n,a,h=[],l=[],c=[],d=0,u=this,p=function(e){return u.f(t,e)},f=function(e){return-u.f(t,e)},_=function(t){return u.f(t,e)},m=function(t){return-u.f(t,e)};for(i=Math.min(this.bbox[0],this.bbox[2])-o.Z.eps,s=Math.max(this.bbox[0],this.bbox[2]),r=Math.min(this.bbox[1],this.bbox[3])+o.Z.eps,n=Math.max(this.bbox[1],this.bbox[3]),a=this.config.resolution_out/this.config.unitX,a*=1+o.Z.eps,t=i;t<s;t+=a)!1!==(c=this.searchLine(p,f,t,[r,n],"vertical",d,h,l,20))&&(h=c[0],l=c[1],d=c[2]);for(a=this.config.resolution_out/this.config.unitY,a*=1+o.Z.eps,e=r;e<n;e+=a)!1!==(c=this.searchLine(_,m,e,[i,s],"horizontal",d,h,l,20))&&(h=c[0],l=c[1],d=c[2]);return[h,l,d]},searchLine:function(t,e,i,r,n,a,h,l,c){var u,p,f,_,m,g,b,v,Z,C,y,P,E,x,O=this.config.tol_u0,M=r[0],w=r[1];if(m=t(u=d.Z.fminbr(t,[M,w])),g=t(p=d.Z.fminbr(e,[M,w])),m<O&&g>-O){if(b=u,u=Math.min(b,p),p=Math.max(b,p),_=t(f=d.Z.fzero(t,[u,p])),Math.abs(_)>Math.max((g-m)*o.Z.eps,.001))return!1;if("vertical"===n?(Z=[1,i,f],x=this.config.resolution_in/this.config.unitY):(Z=[1,f,i],x=this.config.resolution_in/this.config.unitX),x*=1+o.Z.eps,this.curveContainsPoint(Z,h,l,2*x,this.config.qdt_box))false;else if((P=this.traceComponent(Z,1))[0].length>0){for(a>0&&(h.push(NaN),l.push(NaN)),E=h.length,y=P[0].length,C=1;C<y;C++)this.qdt.insertItem({xlb:Math.min(P[0][C-1],P[0][C]),xub:Math.max(P[0][C-1],P[0][C]),ylb:Math.min(P[1][C-1],P[1][C]),yub:Math.max(P[1][C-1],P[1][C]),idx1:E+C-1,idx2:E+C,comp:a});a++,s.Z.concat(h,P[0]),s.Z.concat(l,P[1])}return(v=f-.01*x)-M>x&&c>0&&!1!==(P=this.searchLine(t,e,i,[M,v],n,a,h,l,c-1))&&(h=P[0],l=P[1],a=P[2]),w-(v=f+.01*x)>x&&c>0&&!1!==(P=this.searchLine(t,e,i,[v,w],n,a,h,l,c-1))&&(h=P[0],l=P[1],a=P[2]),[h,l,a]}return!1},curveContainsPoint:function(t,e,i,s,r){var o,n,a,l=t[1],c=t[2];for(n=(a=this.qdt.find([l-r,c+r,l+r,c-r])).length,o=0;o<n;o++)if(h.Z.distPointSegment(t,[1,e[a[o].idx1],i[a[o].idx1]],[1,e[a[o].idx2],i[a[o].idx2]])<s)return!0;return!1},traceComponent:function(t){var e=[],i=[],s=[];return 0===(s=this.tracing(t,1)).length||(e=s[0],i=s[1]),s[2]||0===(s=this.tracing(t,-1)).length||(e=s[0].reverse().concat(e.slice(1)),i=s[1].reverse().concat(i.slice(1))),e.length>0&&e.length<6&&(e.push(e[e.length-1]),i.push(i[i.length-1])),[e,i]},tracing:function(t,e){var i,s,r,n,a,l,c,d,u,p,f,_,m,g,b,v,Z,C,y,P,E=[],x=[],O=[],M=0,w=0,S=!1,T=!1,N=!1,A=this.config.kappa_0,D=this.config.delta_0,k=this.config.alpha_0,R=this.config.h_initial,L=this.config.max_steps,B=e,I=[],j=[],Y=[];if(E=t.slice(1),I.push(E[0]),j.push(E[1]),!1===(r=this.tangent(E)))return[];this.dfx(E[0],E[1]),this.dfy(E[0],E[1]);do{if(!1===(r=this.tangent(E))){E=x.slice(),I.push(E[0]),j.push(E[1]);break}if(1===I.length)a=r.slice();else if(2===I.length)Y.push(I.length-1);else if(T&&I.length>2&&!N){if(h.Z.distPointSegment([1,E[0],E[1]],[1,I[0],j[0]],[1,I[1],j[1]])<this.config.loop_dist*R&&o.Z.innerProduct(r,a,2)>this.config.loop_dir){E=t.slice(1),I.push(E[0]),j.push(E[1]),S=!0;break}if(this.config.loop_detection){for(Z=I.length-1,C=Math.floor(o.Z.log2(Z)),y=0;y<=C&&!(h.Z.distPointSegment([1,E[0],E[1]],[1,I[Y[y]-1],j[Y[y]-1]],[1,I[Y[y]],j[Y[y]]])<this.config.loop_dist*R&&(n=this.tangent([I[Y[y]],j[Y[y]]]),o.Z.innerProduct(r,n,2)>this.config.loop_dir));y++);if(y<=C){S=!0;break}for(C=1,P=0,y=0;y<100&&(Z+1)%C==0;y++)C*=2,P++;Y[P]=Z}}x[0]=E[0]+R*B*r[0],x[1]=E[1]+R*B*r[1],w=0;do{_=(c=[this.dfx(x[0],x[1]),this.dfy(x[0],x[1])])[0]*c[0]+c[1]*c[1],d=this.f(x[0],x[1])/_,O[0]=x[0]-c[0]*d,O[1]=x[1]-c[1]*d,0===w?p=Math.abs(d)*Math.sqrt(_):1===w&&(f=Math.abs(d)*Math.sqrt(_)),x[0]=O[0],x[1]=O[1],w++}while(w<20&&Math.abs(this.f(x[0],x[1]))>this.config.tol_newton);if(g=p,m=w>1?f/p:0,n=this.tangent(x),u=o.Z.innerProduct(r,n,2),u=Math.max(-1,Math.min(1,u)),b=Math.acos(u),N=!1,h.Z.distance(E,x,2)<this.config.tol_progress){E=x.slice(),I.push(E[0]),j.push(E[1]);break}if(u<0)if(R>this.config.h_critical);else{if(N=!0,!this.isBifurcation(E,this.config.tol_cusp)){E=x.slice(),I.push(E[0]),j.push(E[1]);break}B*=-1,b=0}if(!N&&(v=Math.max(Math.sqrt(m/A),Math.sqrt(g/D),b/k),isNaN(v)&&(v=1),R/=v=Math.max(Math.min(v,2),.5),R=Math.min(this.config.h_max,R),v>=2)){if(++M>=3*L)break;T=!1}else E=x.slice(),I.push(E[0]),j.push(E[1]),T=!0,M++}while(M<L&&E[0]>=this.bbox[0]&&E[1]<=this.bbox[1]&&E[0]<=this.bbox[2]&&E[1]>=this.bbox[3]);return i=[I[(s=I.length)-2],j[s-2]],E[0]<this.bbox[0]&&(E[0]!==i[0]&&(l=(this.bbox[0]-i[0])/(E[0]-i[0]),E[1]!==i[1]&&(E[1]=i[1]+l*(E[1]-i[1]))),E[0]=this.bbox[0]),E[0]>this.bbox[2]&&(E[0]!==i[0]&&(l=(this.bbox[2]-i[0])/(E[0]-i[0]),E[1]!==i[1]&&(E[1]=i[1]+l*(E[1]-i[1]))),E[0]=this.bbox[2]),E[1]<this.bbox[3]&&(E[1]!==i[1]&&(l=(this.bbox[3]-i[1])/(E[1]-i[1]),E[0]!==i[0]&&(E[0]=i[0]+l*(E[0]-i[0]))),E[1]=this.bbox[3]),E[1]>this.bbox[1]&&(E[1]!==i[1]&&(l=(this.bbox[1]-i[1])/(E[1]-i[1]),E[0]!==i[0]&&(E[0]=i[0]+l*(E[0]-i[0]))),E[1]=this.bbox[1]),I[s-1]=E[0],j[s-1]=E[1],[I,j,S]},isBifurcation:function(t,e){var i,s,r,n,a,h,l,c,d,u,p=o.Z.eps*o.Z.eps*100;return i=t[0],s=t[1],r=.5*(this.dfx(i+p,s)-this.dfx(i-p,s))/p,n=.5*(this.dfx(i,s+p)-this.dfx(i,s-p))/p,a=.5*(this.dfy(i+p,s)-this.dfy(i-p,s))/p,u=(l=r+(h=.5*(this.dfy(i,s+p)-this.dfy(i,s-p))/p))*l-4*(r*h-n*a),c=.5*(l+Math.sqrt(u)),d=.5*(l-Math.sqrt(u)),Math.abs(c)>e&&Math.abs(d)>e},handleCriticalPoint:function(t,e,i,s){var r,o=Math.atan2(s*e[1],s*e[0]);return r=d.Z.root((function(e){var s=t[0]+i*Math.cos(e),r=t[1]+i*Math.sin(e);return this.f(s,r)}),o),[t[0]+i*Math.cos(r),t[1]+i*Math.sin(r)]},updateA:function(t,e,i){var s,r,o=[i[0]-e[0],i[1]-e[1]],n=this.f(i[0],i[1])-this.f(e[0],e[1]);return r=o[0]*o[0]+o[1]*o[1],s=n-(t[0]*o[0]+t[1]*o[1]),s/=r,t[0]+=s*o[0],t[1]+=s*o[1],t},tangent_A:function(t){var e=[-t[1],t[0]],i=o.Z.norm(e,2);return o.Z.eps,[e[0]/i,e[1]/i]},tangent:function(t){var e=[-this.dfy(t[0],t[1]),this.dfx(t[0],t[1])],i=o.Z.norm(e,2);return!(i<o.Z.eps*o.Z.eps)&&[e[0]/i,e[1]/i]}});var _=o.Z.ImplicitPlot;o.Z.Metapost={MP_ENDPOINT:0,MP_EXPLICIT:1,MP_GIVEN:2,MP_CURL:3,MP_OPEN:4,MP_END_CYCLE:5,UNITY:1,FRACTION_ONE:1,FRACTION_THREE:3,ONE_EIGHTY_DEG:Math.PI,THREE_SIXTY_DEG:2*Math.PI,EPS_SQ:1e-5*1e-5,make_choices:function(t){var e,i,s,r,o,n,a,h,l,c,d,u,p,f,_;a=t[0];do{if(!a)break;n=a.next,a.rtype>this.MP_EXPLICIT&&(a.x-n.x)*(a.x-n.x)+(a.y-n.y)*(a.y-n.y)<this.EPS_SQ&&(a.rtype=this.MP_EXPLICIT,a.ltype===this.MP_OPEN&&(a.ltype=this.MP_CURL,a.set_left_curl(this.UNITY)),n.ltype=this.MP_EXPLICIT,n.rtype===this.MP_OPEN&&(n.rtype=this.MP_CURL,n.set_right_curl(this.UNITY)),a.rx=a.x,n.lx=a.x,a.ry=a.y,n.ly=a.y),a=n}while(a!==t[0]);for(i=t[0];i.ltype===this.MP_OPEN&&i.rtype===this.MP_OPEN;)if((i=i.next)===t[0]){i.ltype=this.MP_END_CYCLE;break}for(a=i;a;){if(n=a.next,a.rtype>=this.MP_GIVEN){for(;n.ltype===this.MP_OPEN&&n.rtype===this.MP_OPEN;)n=n.next;for(s=0,h=a,o=t.length,u=[],p=[],f=[],_=[null];c=h.next,u.push(c.x-h.x),p.push(c.y-h.y),f.push(this.mp_pyth_add(u[s],p[s])),s>0&&(d=p[s-1]/f[s-1],l=u[s-1]/f[s-1],_.push(Math.atan2(p[s]*l-u[s]*d,u[s]*l+p[s]*d))),s++,(h=c)===n&&(o=s),!(s>=o&&h.ltype!==this.MP_END_CYCLE););s===o?_.push(0):_.push(_[1]),n.ltype===this.MP_OPEN&&((r=n.rx-n.x)*r+(e=n.ry-n.y)*e<this.EPS_SQ?(n.ltype=this.MP_CURL,n.set_left_curl(this.UNITY)):(n.ltype=this.MP_GIVEN,n.set_left_given(Math.atan2(e,r)))),a.rtype===this.MP_OPEN&&a.ltype===this.MP_EXPLICIT&&((r=a.x-a.lx)*r+(e=a.y-a.ly)*e<this.EPS_SQ?(a.rtype=this.MP_CURL,a.set_right_curl(this.UNITY)):(a.rtype=this.MP_GIVEN,a.set_right_given(Math.atan2(e,r)))),this.mp_solve_choices(a,n,o,u,p,f,_)}else a.rtype===this.MP_ENDPOINT&&(a.rx=a.x,a.ry=a.y,n.lx=n.x,n.ly=n.y);if((a=n)===i)break}},mp_solve_choices:function(t,e,i,s,r,o,n){var a,h,l,c,d,u,p,f,_,m,g,b,v,Z,C,y,P,E,x,O,M,w,S,T,N,A;for(d=o.length+1,m=new Array(d),_=new Array(d),l=new Array(d),C=new Array(d),N=0;N<d;N++)C[N]=l[N]=_[N]=m[N]=0;for(p=0,f=t,b=0;;){if(v=f.next,0===p)if(f.rtype===this.MP_GIVEN){if(v.ltype===this.MP_GIVEN)return a=Math.atan2(r[0],s[0]),O=(x=this.mp_n_sin_cos(t.right_given()-a))[0],M=x[1],S=(w=this.mp_n_sin_cos(e.left_given()-a))[0],T=w[1],void this.mp_set_controls(t,e,s[0],r[0],M,O,-T,S);l[0]=f.right_given()-Math.atan2(r[0],s[0]),l[0]=this.reduce_angle(l[0]),m[0]=0,_[0]=0}else if(f.rtype===this.MP_CURL){if(v.ltype===this.MP_CURL)return t.rtype=this.MP_EXPLICIT,e.ltype=this.MP_EXPLICIT,g=Math.abs(e.left_tension()),y=Math.abs(t.right_tension()),Z=this.UNITY/(3*y),t.rx=t.x+s[0]*Z,t.ry=t.y+r[0]*Z,Z=this.UNITY/(3*g),e.lx=e.x-s[0]*Z,void(e.ly=e.y-r[0]*Z);E=f.right_curl(),g=Math.abs(v.left_tension()),y=Math.abs(f.right_tension()),m[0]=this.mp_curl_ratio(E,y,g),l[0]=-n[1]*m[0],_[0]=0}else f.rtype===this.MP_OPEN&&(m[0]=0,l[0]=0,_[0]=this.FRACTION_ONE);else if(f.ltype===this.MP_END_CYCLE||f.ltype===this.MP_OPEN){if(a=this.UNITY/(3*Math.abs(b.right_tension())-this.UNITY),P=o[p]*(this.FRACTION_THREE-this.UNITY/Math.abs(b.right_tension())),c=this.UNITY/(3*Math.abs(v.left_tension())-this.UNITY),u=o[p-1]*(this.FRACTION_THREE-this.UNITY/Math.abs(v.left_tension())),P*=E=this.FRACTION_ONE-m[p-1]*a,(g=Math.abs(f.left_tension()))<(y=Math.abs(f.right_tension()))?P*=Math.pow(g/y,2):g>y&&(u*=Math.pow(y/g,2)),Z=u/(u+P),m[p]=Z*c,h=-n[p+1]*m[p],b.rtype===this.MP_CURL?(_[p]=0,l[p]=h-n[1]*(this.FRACTION_ONE-Z)):(Z=(this.FRACTION_ONE-Z)/E,h-=n[p]*Z,Z*=a,l[p]=h-l[p-1]*Z,_[p]=-_[p-1]*Z),f.ltype===this.MP_END_CYCLE){for(a=0,c=this.FRACTION_ONE;0===(p-=1)&&(p=i),a=l[p]-a*m[p],c=_[p]-c*m[p],p!==i;);for(a/=this.FRACTION_ONE-c,C[i]=a,l[0]=a,A=1;A<i;A++)l[A]=l[A]+a*_[A];break}}else{if(f.ltype===this.MP_CURL){E=f.left_curl(),g=Math.abs(f.left_tension()),y=Math.abs(b.right_tension()),Z=this.mp_curl_ratio(E,g,y),C[i]=-l[i-1]*Z/(this.FRACTION_ONE-Z*m[i-1]);break}if(f.ltype===this.MP_GIVEN){C[i]=f.left_given()-Math.atan2(r[i-1],s[i-1]),C[i]=this.reduce_angle(C[i]);break}}b=f,f=v,p+=1}for(p=i-1;p>-1;p--)C[p]=l[p]-C[p+1]*m[p];for(f=t,p=0;v=f.next,O=(x=this.mp_n_sin_cos(C[p]))[0],M=x[1],S=(w=this.mp_n_sin_cos(-n[p+1]-C[p+1]))[0],T=w[1],this.mp_set_controls(f,v,s[p],r[p],M,O,T,S),f=v,++p!==i;);},mp_n_sin_cos:function(t){return[Math.cos(t),Math.sin(t)]},mp_set_controls:function(t,e,i,s,r,o,n,a){var h,l,c,d,u;c=Math.abs(e.left_tension()),h=Math.abs(t.right_tension()),u=this.mp_velocity(r,o,n,a,h),l=this.mp_velocity(n,a,r,o,c),(t.right_tension()<0||e.left_tension()<0)&&(r>=0&&n>=0||r<=0&&n<=0)&&(d=Math.abs(r)*a+Math.abs(n)*o)>0&&(d*=1.00024414062,t.right_tension()<0&&this.mp_ab_vs_cd(Math.abs(n),this.FRACTION_ONE,u,d)<0&&(u=Math.abs(n)/d),e.left_tension()<0&&this.mp_ab_vs_cd(Math.abs(r),this.FRACTION_ONE,l,d)<0&&(l=Math.abs(r)/d)),t.rx=t.x+(i*o-s*r)*u,t.ry=t.y+(s*o+i*r)*u,e.lx=e.x-(i*a+s*n)*l,e.ly=e.y-(s*a-i*n)*l,t.rtype=this.MP_EXPLICIT,e.ltype=this.MP_EXPLICIT},mp_pyth_add:function(t,e){return o.Z.hypot(t,e)},mp_curl_ratio:function(t,e,i){var s=1/e,r=1/i;return Math.min(4,((3-s)*s*s*t+r*r*r)/(s*s*s*t+(3-r)*r*r))},mp_ab_vs_cd:function(t,e,i,s){return t*e==i*s?0:t*e>i*s?1:-1},mp_velocity:function(t,e,i,s,r){return Math.min(4,(2+Math.sqrt(2)*(t-i/16)*(i-t/16)*(e-s))/(1.5*r*(2+(Math.sqrt(5)-1)*e+(3-Math.sqrt(5))*s)))},reduce_angle:function(t){return Math.abs(t)>this.ONE_EIGHTY_DEG&&(t>0?t-=this.THREE_SIXTY_DEG:t+=this.THREE_SIXTY_DEG),t},makeknots:function(t,e){var i,s,r=[];for(s=t.length,i=0;i<s;i++)r.push({x:t[i][0],y:t[i][1],ltype:this.MP_OPEN,rtype:this.MP_OPEN,lx:!1,rx:!1,ly:e,ry:e,left_curl:function(){return this.lx||0},right_curl:function(){return this.rx||0},left_tension:function(){return this.ly||1},right_tension:function(){return this.ry||1},set_right_curl:function(t){this.rx=t||0},set_left_curl:function(t){this.lx=t||0}});for(s=r.length,i=0;i<s;i++)r[i].next=r[i+1]||r[i],r[i].set_right_given=r[i].set_right_curl,r[i].set_left_given=r[i].set_left_curl,r[i].right_given=r[i].right_curl,r[i].left_given=r[i].left_curl;return r[s-1].next=r[0],r},curve:function(t,e){var i,r,o,n,a,h,l=!1,c=[],d=[];for(n in a=1,(e=e||{tension:1,direction:{},curl:{},isClosed:!1}).hasOwnProperty("tension")&&(a=s.Z.evaluate(e.tension)),r=(i=this.makeknots(t,a)).length,s.Z.exists(e.isClosed)&&s.Z.evaluate(e.isClosed)&&(l=!0),l||(i[0].ltype=this.MP_ENDPOINT,i[0].rtype=this.MP_CURL,i[r-1].rtype=this.MP_ENDPOINT,i[r-1].ltype=this.MP_CURL),e)if(e.hasOwnProperty(n)){if(o=parseInt(n,10),isNaN(o)||o<0||o>=r)continue;if(h=e[o],s.Z.exists(h.type)&&"curl"===h.type)a=s.Z.evaluate(h.curl),0===o?(i[o].rtype=this.MP_CURL,i[o].set_right_curl(a)):o===r-1?(i[o].ltype=this.MP_CURL,i[o].set_left_curl(a)):(i[o].ltype=this.MP_CURL,i[o].rtype=this.MP_CURL,i[o].lx=a,i[o].rx=a);s.Z.exists(h.direction)&&(a=s.Z.evaluate(h.direction),s.Z.isArray(a)?(!1!==a[0]&&(i[o].lx=a[0]*Math.PI/180,i[o].ltype=this.MP_GIVEN),!1!==a[1]&&(i[o].rx=a[1]*Math.PI/180,i[o].rtype=this.MP_GIVEN)):(i[o].lx=a*Math.PI/180,i[o].rx=a*Math.PI/180,i[o].ltype=i[o].rtype=this.MP_GIVEN)),s.Z.exists(h.tension)&&(a=s.Z.evaluate(h.tension),s.Z.isArray(a)?(!1!==a[0]&&(i[o].ly=s.Z.evaluate(a[0])),!1!==a[1]&&(i[o].ry=s.Z.evaluate(a[1]))):(i[o].ly=a,i[o].ry=a))}for(this.make_choices(i),o=0;o<r-1;o++)c.push(i[o].x),c.push(i[o].rx),c.push(i[o+1].lx),d.push(i[o].y),d.push(i[o].ry),d.push(i[o+1].ly);return c.push(i[r-1].x),d.push(i[r-1].y),l&&(c.push(i[r-1].rx),d.push(i[r-1].ry),c.push(i[0].lx),d.push(i[0].ly),c.push(i[0].x),d.push(i[0].y)),[c,d]}};var m=o.Z.Metapost;o.Z.Clip={_isSeparator:function(t){return isNaN(t.coords.usrCoords[1])&&isNaN(t.coords.usrCoords[2])},makeDoublyLinkedList:function(t){var e,i=null,s=[],r=t.length;if(r>0)for(e=0;e<r;e++)this._isSeparator(t[e])?(t[e]._next=t[(e+1)%r],t[e]._prev=t[(r+e-1)%r]):(null===i&&(i=e,s.push(i)),this._isSeparator(t[(e+1)%r])||e===r-1?(t[e]._next=t[i],t[i]._prev=t[e],t[e]._end=!0,i=null):(t[e]._next=t[(e+1)%r],t[i]._prev=t[e]),this._isSeparator(t[(r+e-1)%r])||(t[e]._prev=t[(r+e-1)%r]));return s},Vertex:function(t,e,s,r,o,n){this.pos=e,this.intersection=!0,this.coords=t,this.elementClass=i.Z.OBJECT_CLASS_POINT,this.data={alpha:s,path:r,pathname:o,done:!1,type:n,idx:0},this.neighbour=null,this.entry_exit=!1},_addToList:function(t,e,s){var r=t.length,n=o.Z.eps*o.Z.eps;r>0&&Math.abs(t[r-1].coords.usrCoords[0]-e.usrCoords[0])<n&&Math.abs(t[r-1].coords.usrCoords[1]-e.usrCoords[1])<n&&Math.abs(t[r-1].coords.usrCoords[2]-e.usrCoords[2])<n||t.push({pos:s,intersection:!1,coords:e,elementClass:i.Z.OBJECT_CLASS_POINT})},sortIntersections:function(t){var e,i,s,r,o,n,a=[],h=t.length;for(e=0;e<h;e++)if(t[e].sort((function(t,e){return t.data.alpha>e.data.alpha?1:-1})),t[e].length>0){for(o=t[e].length-1,n=(r=(s=t[e][0]).data.path[s.pos])._next,e===h-1&&(r._end=!1),0===s.data.alpha&&"T"===s.data.type?(r.intersection=!0,r.data=s.data,r.neighbour=s.neighbour,r.neighbour.neighbour=r,r.entry_exit=!1,t[e][0]=r):(s._prev=r,s._prev._next=s),i=1;i<=o;i++)(s=t[e][i])._prev=t[e][i-1],s._prev._next=s;(s=t[e][o])._next=n,s._next._prev=s,e===h-1&&(s._end=!0),a=a.concat(t[e])}return a},_inbetween:function(t,e,i){var s,r=o.Z.eps*o.Z.eps,n=i[1]-e[1],a=i[2]-e[2],h=t[1]-e[1],l=t[2]-e[2];return 0===n&&0===a&&0===h&&0===l||(s=Math.abs(h)<r&&Math.abs(n)<r?l/a:h/n,Math.abs(s)<r&&(s=0),s)},_print_array:function(t){var e,i;for(e=0;e<t.length;e++)try{i="",t[e]._end&&(i=" end"),console.log(e,t[e].coords.usrCoords,t[e].data.type,"\t","prev",t[e]._prev.coords.usrCoords,"next",t[e]._next.coords.usrCoords+i)}catch(i){console.log(e,t[e].coords.usrCoords)}},_print_list:function(t){for(var e,i=0;i<100&&(e=t.data?t.data.alpha:"-",console.log("\t",t.coords.usrCoords,"\n\t\tis:",t.intersection,"end:",t._end,e,"\n\t\t-:",t._prev.coords.usrCoords,"\n\t\t+:",t._next.coords.usrCoords,"\n\t\tn:",t.intersection?t.neighbour.coords.usrCoords:"-"),!t._end);)t=t._next,i++},_noOverlap:function(t,e,i,s){var r,n,a,h,l,c=Math.sqrt(o.Z.eps),d=!1;for(r=0;r<3;r++)if(n=Math.min(t[r],e[r]),a=Math.max(t[r],e[r]),h=Math.min(i[r],s[r]),l=Math.max(i[r],s[r]),a<h-c||n>l+c){d=!0;break}return d},findIntersections:function(t,e,s){var r,n,a,l,c,d,p,f,_,m,g,b,v,Z=[],C=100*o.Z.eps,y=t.length,P=e.length,E=[],x=[],O=[],M=!1,w=!1,S=!1;for(n=0;n<P;n++)O.push([]);for(r=0;r<y;r++)if(x.push([]),this._isSeparator(t[r])||this._isSeparator(t[(r+1)%y]))M=!0;else{if(M&&r===y-1)break;for(l=t[r].coords.usrCoords,c=t[(r+1)%y].coords.usrCoords,n=0;n<P;n++)if(this._isSeparator(e[n])||this._isSeparator(e[(n+1)%P]))w=!0;else{if(w&&n===P-1)break;if(d=e[n].coords.usrCoords,p=e[(n+1)%P].coords.usrCoords,!this._noOverlap(l,c,d,p)&&(Z=h.Z.meetSegmentSegment(l,c,d,p),f=h.Z.distance(l,c,3),_=h.Z.distance(d,p,3),Z[1]*f>-C&&Z[1]<1-C/f&&Z[2]*_>-C&&Z[2]<1-C/_||Z[1]===1/0&&Z[2]===1/0&&o.Z.norm(Z[0],3)<C)){if(a=new u.Z(i.Z.COORDS_BY_USER,Z[0],s),g="X",Math.abs(Z[1])*f<C||Math.abs(Z[2])*_<C)g="T",Math.abs(Z[1])*f<C&&(Z[1]=0),Math.abs(Z[2])*_<C&&(Z[2]=0),a=0===Z[1]?new u.Z(i.Z.COORDS_BY_USER,l,s):new u.Z(i.Z.COORDS_BY_USER,d,s);else if(Z[1]===1/0&&Z[2]===1/0&&o.Z.norm(Z[0],3)<C){(m=this._inbetween(l,d,p))>=0&&m<1&&(g="T",a=new u.Z(i.Z.COORDS_BY_USER,l,s),Z[1]=0,Z[2]=m,b=new this.Vertex(a,r,Z[1],t,"S",g),v=new this.Vertex(a,n,Z[2],e,"C",g),b.neighbour=v,v.neighbour=b,x[r].push(b),O[n].push(v)),m=this._inbetween(d,l,c),h.Z.distance(l,d,3)>C&&m>=0&&m<1&&(g="T",a=new u.Z(i.Z.COORDS_BY_USER,d,s),Z[1]=m,Z[2]=0,b=new this.Vertex(a,r,Z[1],t,"S",g),v=new this.Vertex(a,n,Z[2],e,"C",g),b.neighbour=v,v.neighbour=b,x[r].push(b),O[n].push(v));continue}S,b=new this.Vertex(a,r,Z[1],t,"S",g),v=new this.Vertex(a,n,Z[2],e,"C",g),b.neighbour=v,v.neighbour=b,x[r].push(b),O[n].push(v)}}}for(E=this.sortIntersections(x),r=0;r<E.length;r++)E[r].data.idx=r,E[r].neighbour.data.idx=r;return[E,this.sortIntersections(O)]},_getPosition:function(t,e,i,s){var r=h.Z.det3p(t,e,i),o=h.Z.det3p(t,i,s);return h.Z.det3p(e,i,s)>=0?r>=0&&o>=0?"left":"right":r>=0||o>=0?"left":"right"},_classifyDegenerateIntersections:function(t){var e,i,r,n,a,l,c,d,u,p,f,_,m,g,b=!1;for(b,u=h.Z.det3p,c=0,t._tours=0;;){if(t.intersection&&"T"===t.data.type&&(e=t._next.coords.usrCoords,i=t._prev.coords.usrCoords,h.Z.distance(t.coords.usrCoords,e,3)<o.Z.eps&&(e=t._next._next.coords.usrCoords),h.Z.distance(t.coords.usrCoords,i,3)<o.Z.eps&&(i=t._prev._prev.coords.usrCoords),n=(a=t.neighbour)._prev.coords.usrCoords,r=a._next.coords.usrCoords,h.Z.distance(a.coords.usrCoords,r,3)<o.Z.eps&&(r=a._next._next.coords.usrCoords),h.Z.distance(a.coords.usrCoords,n,3)<o.Z.eps&&(n=a._prev._prev.coords.usrCoords),f=u(t.coords.usrCoords,i,n),_=u(t.coords.usrCoords,e,r),m=u(t.coords.usrCoords,i,r),g=u(t.coords.usrCoords,e,n),0===f&&0===_&&0===m&&0===g&&(t.coords.usrCoords[1]*=1+Math.random()*o.Z.eps,t.coords.usrCoords[2]*=1+Math.random()*o.Z.eps,a.coords.usrCoords[1]=t.coords.usrCoords[1],a.coords.usrCoords[2]=t.coords.usrCoords[2],f=u(t.coords.usrCoords,i,n),_=u(t.coords.usrCoords,e,r),m=u(t.coords.usrCoords,i,r),g=u(t.coords.usrCoords,e,n)),p=!1,0===f?h.Z.affineRatio(t.coords.usrCoords,i,n)<0&&(p=!0):0===_?h.Z.affineRatio(t.coords.usrCoords,e,r)<0&&(p=!0):0===m?h.Z.affineRatio(t.coords.usrCoords,i,r)>0&&(p=!0):0===g&&h.Z.affineRatio(t.coords.usrCoords,e,n)>0&&(p=!0),p&&(d=n,n=r,r=d,d=f,f=m,m=d,d=_,_=g,g=d),s.Z.exists(t.delayedStatus)||(t.delayedStatus=[]),0===f&&0===_?t.delayedStatus=["on","on"]:0===f?(l=this._getPosition(e,n,a.coords.usrCoords,r),t.delayedStatus=["on",l]):0===_?(l=this._getPosition(i,n,a.coords.usrCoords,r),t.delayedStatus=[l,"on"]):0===t.delayedStatus.length&&(this._getPosition(i,n,a.coords.usrCoords,r)!==this._getPosition(e,n,a.coords.usrCoords,r)?t.data.type="X":t.data.type="B")),s.Z.exists(t._tours)&&t._tours++,t._tours>3||t._end||c>1e3){c>1e3&&console.log("Clipping: _classifyDegenerateIntersections exit"),s.Z.exists(t._tours)&&delete t._tours;break}t.intersection&&c++,t=t._next}},_handleIntersectionChains:function(t){var e,i=0,s="Null",r=!1,o=!1,n=!1;for(n;!0===t.intersection&&("T"===t.data.type&&("on"!==t.delayedStatus[0]&&"on"===t.delayedStatus[1]?(r=!0,e=t,s=t.delayedStatus[0]):r&&"on"===t.delayedStatus[0]&&"on"===t.delayedStatus[1]?t.data.type="B":r&&"on"===t.delayedStatus[0]&&"on"!==t.delayedStatus[1]&&(r=!1,s===t.delayedStatus[1]?(e.data.type="DB",t.data.type="DB"):(e.data.type="DX",t.data.type="DX"))),i++),t._end&&(o=!0),!o||r;){if(i>1e3){console.log("Warning: _handleIntersectionChains: intersection chain reached maximum numbers of iterations");break}t=t._next}},_handleFullyDegenerateCase:function(t,e,s){var r,n,a,l,c,d,p,f,_,m,g,b,v,Z,C=[t,e];for(a=0;a<2;a++){for(m=(r=C[a]).length,f=0,Z=!0;f<m;f++)if(!r[f].intersection){Z=!1;break}if(Z)for(g=(n=C[(a+1)%2]).length,f=0;f<m;f++){for(c=r[f].coords.usrCoords,d=r[f]._next.coords.usrCoords,l=[.5*(c[0]+d[0]),.5*(c[1]+d[1]),.5*(c[2]+d[2])],_=0,b=!1;_<g;_++)if(Math.abs(h.Z.det3p(n[_].coords.usrCoords,n[(_+1)%g].coords.usrCoords,l))<o.Z.eps){b=!0;break}if(!b){p={pos:f,intersection:!1,coords:new u.Z(i.Z.COORDS_BY_USER,l,s),elementClass:i.Z.OBJECT_CLASS_POINT},v=r[f]._next,r[f]._next=p,p._prev=r[f],p._next=v,v._prev=p,r[f]._end&&(r[f]._end=!1,p._end=!0);break}}}},_getStatus:function(t,e){for(;t.intersection&&!t._end;)t=t._next;return[t,0===h.Z.windingNumber(t.coords.usrCoords,e)?"entry":"exit"]},markEntryExit:function(t,e,i){var r,o,n,a,h,l,c,d=null,u=0;for(l=i.length,h=0;h<l;h++)for(c=i[h],this._classifyDegenerateIntersections(t[c]),this._handleIntersectionChains(t[c]),o=(a=this._getStatus(t[c],e))[0],r=a[1],o._starter=!0,n=0,d=null,u=0;!0===o.intersection&&("X"===o.data.type&&1===u&&(d.entry_exit=r,"exit"===r&&(d.data.type="X"),u=2),"X"!==o.data.type&&"DB"!==o.data.type||(o.entry_exit=r,r="entry"===r?"exit":"entry"),"DX"===o.data.type&&(0===u?(d=o,u=1):1===u?(o.entry_exit=r,d.entry_exit=r,"exit"===r?d.data.type="X":o.data.type="X",r="entry"===r?"exit":"entry",d=null,u=0):2===u&&(o.entry_exit=r,o.data.type="X",r="entry"===r?"exit":"entry",d=null,u=0))),o=o._next,!(s.Z.exists(o._starter)||n>1e4);)n++},_stayOnPath:function(t,e){var i=!0;return t.intersection&&"B"!==t.data.type&&(i=e===t.entry_exit),i},_addVertex:function(t,e,i){return isNaN(e.coords.usrCoords[1])||isNaN(e.coords.usrCoords[2])||t.push(e),e.intersection&&e.data.done?(i&&console.log("Add last intersection point",e.coords.usrCoords,"on",e.data.pathname,e.entry_exit,e.data.type),!0):(e.intersection&&(e.data.done=!0,i&&console.log("Add intersection point",e.coords.usrCoords,"on",e.data.pathname,e.entry_exit,e.data.type)),!1)},tracing:function(t,e,i){var s,r,o,n,a=0,h=1e4,l=0,c=[],d=!1,u=!1;for(u;l<e.length&&a<h;)if((o=e[l]).data.done||"X"!==o.data.type)l++;else{u,c.length>0&&c.push([NaN,NaN]),n=o.data.idx,s=t,d=this._addVertex(c,o,u),r=o.entry_exit;do{if(d)break;if("intersection"===i&&"entry"===o.entry_exit||"union"===i&&"exit"===o.entry_exit||"difference"===i&&s===t==("exit"===o.entry_exit)){u;do{if(o=o._next,d=this._addVertex(c,o,u))break}while(this._stayOnPath(o,r));a++}else{u;do{if(o=o._prev,d=this._addVertex(c,o,u))break}while(this._stayOnPath(o,r));a++}if(d)break;if(!o.neighbour)return console.log("Tracing: emergency break - no neighbour!!!!!!!!!!!!!!!!!",a),[[0],[0]];if((o=o.neighbour).data.done)break;o.data.done=!0,r=o.entry_exit,s=o.data.path}while(o.data.idx!==n&&a<h);a>=h&&console.log("Tracing: stopping an infinite loop!",a),l++}return this._getCoordsArrays(c,!1)},isEmptyCase:function(t,e,i){return"intersection"===i&&(0===t.length||0===e.length)||("union"===i&&0===t.length&&0===e.length||"difference"===i&&0===t.length)},_getCoordsArrays:function(t,e){var i,s=[],r=[],o=t.length;for(i=0;i<o;i++)t[i].coords?(s.push(t[i].coords.usrCoords[1]),r.push(t[i].coords.usrCoords[2])):(s.push(t[i][0]),r.push(t[i][1]));return e&&o>0&&(t[0].coords?(s.push(t[0].coords.usrCoords[1]),r.push(t[0].coords.usrCoords[2])):(s.push(t[0][0]),r.push(t[0][1]))),[s,r]},handleEmptyIntersection:function(t,e,i){var r,o,n=!1,a=[];if(0===t.length)return a="union"===i?e:[],this._getCoordsArrays(a,!0);if(0===e.length)return a="intersection"===i?[]:t,this._getCoordsArrays(a,!0);if(t.length>0)for(r=t[0];r.intersection&&!(r=r._next)._end;);if(e.length>0)for(o=e[0];o.intersection&&!(o=o._next)._end;);return 0===h.Z.windingNumber(r.coords.usrCoords,e)?0!==h.Z.windingNumber(o.coords.usrCoords,t)?("union"===i?(s.Z.concat(a,t),a.push(t[0])):"difference"===i&&(s.Z.concat(a,t),a.push(t[0]),h.Z.signedPolygon(t)*h.Z.signedPolygon(e)>0&&a.reverse(),a.push([NaN,NaN])),"difference"!==i&&"intersection"!==i||(s.Z.concat(a,e),a.push(e[0]),n=!1)):"difference"===i?(s.Z.concat(a,t),n=!0):"union"===i&&(s.Z.concat(a,t),a.push(t[0]),a.push([NaN,NaN]),s.Z.concat(a,e),a.push(e[0])):"intersection"===i?(s.Z.concat(a,t),n=!0):"union"===i&&(s.Z.concat(a,e),a.push(e[0])),this._getCoordsArrays(a,n)},_countCrossingIntersections:function(t){var e,i=t.length,s=0;for(e=0;e<i;e++)"X"===t[e].data.type&&s++;return s},_getPath:function(t,e){var r,o,n,a,l,c,d,p=[];if(t.elementClass!==i.Z.OBJECT_CLASS_CURVE||t.type!==i.Z.OBJECT_TYPE_ARC&&t.type!==i.Z.OBJECT_TYPE_SECTOR){if(t.elementClass===i.Z.OBJECT_CLASS_CURVE&&s.Z.exists(t.points))for(o=t.numberPoints,r=0;r<o;r++)this._addToList(p,t.points[r],r);else if(t.type===i.Z.OBJECT_TYPE_POLYGON)for(r=0;r<t.vertices.length;r++)this._addToList(p,t.vertices[r].coords,r);else if(t.elementClass===i.Z.OBJECT_CLASS_CIRCLE)for(d=359,n=t.Radius(),a=2*Math.PI/d,r=0;r<=d;r++)this._addToList(p,new u.Z(i.Z.COORDS_BY_USER,[t.center.coords.usrCoords[0],t.center.coords.usrCoords[1]+Math.cos(r*a)*n,t.center.coords.usrCoords[2]+Math.sin(r*a)*n],e),r);else if(s.Z.isArray(t))for(o=t.length,r=0;r<o;r++)s.Z.exists(t[r].coords)?this._addToList(p,t[r].coords,r):s.Z.isArray(t[r])?this._addToList(p,new u.Z(i.Z.COORDS_BY_USER,t[r],e),r):s.Z.exists(t[r].usrCoords)&&this._addToList(p,t[r],r)}else{for(l=h.Z.rad(t.radiuspoint,t.center,t.anglepoint),d=Math.floor(180*l/Math.PI),n=t.Radius(),a=l/d,c=Math.atan2(t.radiuspoint.coords.usrCoords[2]-t.center.coords.usrCoords[2],t.radiuspoint.coords.usrCoords[1]-t.center.coords.usrCoords[1]),t.type===i.Z.OBJECT_TYPE_SECTOR&&this._addToList(p,t.center.coords,0),r=0;r<=d;r++)this._addToList(p,new u.Z(i.Z.COORDS_BY_USER,[t.center.coords.usrCoords[0],t.center.coords.usrCoords[1]+Math.cos(r*a+c)*n,t.center.coords.usrCoords[2]+Math.sin(r*a+c)*n],e),r+1);t.type===i.Z.OBJECT_TYPE_SECTOR&&this._addToList(p,t.center.coords,d+2)}return p},greinerHormann:function(t,e,i,s){var r,n,a,l,c=[],d=[];return(r=(c=this._getPath(t,s)).length)>0&&h.Z.distance(c[0].coords.usrCoords,c[r-1].coords.usrCoords,3)<o.Z.eps&&c.pop(),(r=(d=this._getPath(e,s)).length)>0&&h.Z.distance(d[0].coords.usrCoords,d[r-1].coords.usrCoords,3)<o.Z.eps*o.Z.eps&&d.pop(),this.isEmptyCase(c,d,i)?[[],[]]:(a=this.makeDoublyLinkedList(c),l=this.makeDoublyLinkedList(d),n=this.findIntersections(c,d,s)[0],this._handleFullyDegenerateCase(c,d,s),this.markEntryExit(c,d,a),this.markEntryExit(d,c,l),0===this._countCrossingIntersections(n)?this.handleEmptyIntersection(c,d,i):this.tracing(c,n,i))},union:function(t,e,i){return this.greinerHormann(t,e,"union",i)},intersection:function(t,e,i){return this.greinerHormann(t,e,"intersection",i)},difference:function(t,e,i){return this.greinerHormann(t,e,"difference",i)}};var g=o.Z.Clip;o.Z.Poly={},o.Z.Poly.Ring=function(t){this.vars=t},t.Z.extend(o.Z.Poly.Ring.prototype,{}),o.Z.Poly.Monomial=function(t,e,i){var r;if(!s.Z.exists(t))throw new Error("JSXGraph error: In JXG.Math.Poly.monomial missing parameter 'ring'.");for(s.Z.isArray(i)||(i=[]),r=(i=i.slice(0,t.vars.length)).length;r<t.vars.length;r++)i.push(0);this.ring=t,this.coefficient=e||0,this.exponents=s.Z.deepCopy(i)},t.Z.extend(o.Z.Poly.Monomial.prototype,{copy:function(){return new o.Z.Poly.Monomial(this.ring,this.coefficient,this.exponents)},print:function(){var t,e=[];for(t=0;t<this.ring.vars.length;t++)e.push(this.ring.vars[t]+"^"+this.exponents[t]);return this.coefficient+"*"+e.join("*")}}),o.Z.Poly.Polynomial=function(t,e){var i;if(!s.Z.exists(t))throw new Error("JSXGraph error: In JXG.Math.Poly.polynomial missing parameter 'ring'.");i=s.Z.exists(e)&&s.Z.isString(e)?void 0:[],this.ring=t,this.monomials=i},t.Z.extend(o.Z.Poly.Polynomial.prototype,{findSignature:function(t){var e;for(e=0;e<this.monomials.length;e++)if(s.Z.cmpArrays(this.monomials[e].exponents,t))return e;return-1},addSubMonomial:function(t,e){var i;(i=this.findSignature(t.exponents))>-1?this.monomials[i].coefficient+=e*t.coefficient:(t.coefficient*=e,this.monomials.push(t))},add:function(t){var e;if(!s.Z.exists(t)||t.ring!==this.ring)throw new Error("JSXGraph error: In JXG.Math.Poly.polynomial.add either summand is undefined or rings don't match.");if(s.Z.isArray(t.exponents))this.addSubMonomial(t,1);else for(e=0;e<t.monomials.length;e++)this.addSubMonomial(t.monomials[e],1)},sub:function(t){var e;if(!s.Z.exists(t)||t.ring!==this.ring)throw new Error("JSXGraph error: In JXG.Math.Poly.polynomial.sub either summand is undefined or rings don't match.");if(s.Z.isArray(t.exponents))this.addSubMonomial(t,-1);else for(e=0;e<t.monomials.length;e++)this.addSubMonomial(t.monomials[e],-1)},copy:function(){var t,e;for(e=new o.Z.Poly.Polynomial(this.ring),t=0;t<this.monomials.length;t++)e.monomials.push(this.monomials[t].copy());return e},print:function(){var t,e=[];for(t=0;t<this.monomials.length;t++)e.push("("+this.monomials[t].print()+")");return e.join("+")}});o.Z.Poly;t.Z.Complex=function(t,e){this.isComplex=!0,t&&t.isComplex&&(e=t.imaginary,t=t.real),this.real=t||0,this.imaginary=e||0},t.Z.extend(t.Z.Complex.prototype,{toString:function(){return this.real+" + "+this.imaginary+"i"},add:function(t){return s.Z.isNumber(t)?this.real+=t:(this.real+=t.real,this.imaginary+=t.imaginary),this},sub:function(t){return s.Z.isNumber(t)?this.real-=t:(this.real-=t.real,this.imaginary-=t.imaginary),this},mult:function(t){var e,i;return s.Z.isNumber(t)?(this.real*=t,this.imaginary*=t):(e=this.real,i=this.imaginary,this.real=e*t.real-i*t.imaginary,this.imaginary=e*t.imaginary+i*t.real),this},div:function(t){var e,i,r;if(s.Z.isNumber(t)){if(Math.abs(t)<Math.eps)return this.real=1/0,this.imaginary=1/0,this;this.real/=t,this.imaginary/=t}else{if(Math.abs(t.real)<Math.eps&&Math.abs(t.imaginary)<Math.eps)return this.real=1/0,this.imaginary=1/0,this;e=t.real*t.real+t.imaginary*t.imaginary,r=this.real,i=this.imaginary,this.real=(r*t.real+i*t.imaginary)/e,this.imaginary=(i*t.real-r*t.imaginary)/e}return this},conj:function(){return this.imaginary*=-1,this},abs:function(){var t=this.real,e=this.imaginary;return Math.sqrt(t*t+e*e)},angle:function(){return Math.atan2(this.imaginary,this.real)}}),t.Z.C={},t.Z.C.add=function(e,i){var s=new t.Z.Complex(e);return s.add(i),s},t.Z.C.sub=function(e,i){var s=new t.Z.Complex(e);return s.sub(i),s},t.Z.C.mult=function(e,i){var s=new t.Z.Complex(e);return s.mult(i),s},t.Z.C.div=function(e,i){var s=new t.Z.Complex(e);return s.div(i),s},t.Z.C.conj=function(e){var i=new t.Z.Complex(e);return i.conj(),i},t.Z.C.abs=function(e){return new t.Z.Complex(e).abs()},t.Z.C.angle=function(e){return new t.Z.Complex(e).angle()},t.Z.C.copy=function(e){return new t.Z.Complex(e)},t.Z.Complex.C=t.Z.C;var b=t.Z.Complex,v=__webpack_require__(766);t.Z.AbstractRenderer=function(){this.vOffsetText=0,this.enhancedRendering=!0,this.container=null,this.type="",this.supportsForeignObject=!1,this.dashArray=[[2,2],[5,5],[10,10],[20,20],[20,10,10,10],[20,5,10,5],[0,5]]},t.Z.extend(t.Z.AbstractRenderer.prototype,{_updateVisual:function(t,e,i){(i||this.enhancedRendering)&&(e=e||{},this.setObjectTransition(t),t.evalVisProp("draft")?this.setDraft(t):(e.stroke||(t.highlighted?(this.setObjectStrokeColor(t,t.evalVisProp("highlightstrokecolor"),t.evalVisProp("highlightstrokeopacity")),this.setObjectStrokeWidth(t,t.evalVisProp("highlightstrokewidth"))):(this.setObjectStrokeColor(t,t.evalVisProp("strokecolor"),t.evalVisProp("strokeopacity")),this.setObjectStrokeWidth(t,t.evalVisProp("strokewidth")))),e.fill||(t.highlighted?this.setObjectFillColor(t,t.evalVisProp("highlightfillcolor"),t.evalVisProp("highlightfillopacity")):this.setObjectFillColor(t,t.evalVisProp("fillcolor"),t.evalVisProp("fillopacity"))),e.dash||this.setDashStyle(t,t.visProp),e.shadow||this.setShadow(t),e.tabindex||this.setTabindex(t)),t.highlighted?this.setCssClass(t,t.evalVisProp("highlightcssclass")):this.setCssClass(t,t.evalVisProp("cssclass")),t.evalVisProp("aria.enabled")&&this.setARIA(t))},_getHighlighted:function(t){var e=!1;return s.Z.exists(t.board)&&s.Z.exists(t.board.highlightedObjects)||(e=!0),!e&&s.Z.exists(t.board.highlightedObjects[t.id])?"highlight":""},drawPoint:function(t){var e,i=v.Z.normalizePointFace(t.evalVisProp("face"));e="o"===i?"ellipse":"[]"===i?"rect":"path",t.rendNode=this.appendChildPrim(this.createPrim(e,t.id),t.evalVisProp("layer")),this.appendNodesToElement(t,e),this._updateVisual(t,{dash:!0,shadow:!0},!0),this.updatePoint(t)},updatePoint:function(t){var e,i=t.evalVisProp("size"),s=v.Z.normalizePointFace(t.evalVisProp("face")),r=t.evalVisProp("sizeunit"),o=t.evalVisProp("zoom");isNaN(t.coords.scrCoords[2]+t.coords.scrCoords[1])||("user"===r&&(i*=Math.sqrt(Math.abs(t.board.unitX*t.board.unitY))),e=0===(i*=t.board&&o?Math.sqrt(t.board.zoomX*t.board.zoomY):1)?0:i+1,"o"===s?this.updateEllipsePrim(t.rendNode,t.coords.scrCoords[1],t.coords.scrCoords[2],e,e):"[]"===s?this.updateRectPrim(t.rendNode,t.coords.scrCoords[1]-i,t.coords.scrCoords[2]-i,2*i,2*i):this.updatePathPrim(t.rendNode,this.updatePathStringPoint(t,i,s),t.board),this._updateVisual(t,{dash:!1,shadow:!1}),this.setShadow(t))},changePointStyle:function(t){var e=this.getElementById(t.id);s.Z.exists(e)&&this.remove(e),this.drawPoint(t),s.Z.clearVisPropOld(t),t.visPropCalc.visible||this.display(t,!1),t.evalVisProp("draft")&&this.setDraft(t)},drawLine:function(t){t.rendNode=this.appendChildPrim(this.createPrim("line",t.id),t.evalVisProp("layer")),this.appendNodesToElement(t,"lines"),this.updateLine(t)},updateLine:function(t){this._updateVisual(t),this.updatePathWithArrowHeads(t),this.setLineCap(t)},drawCurve:function(t){t.rendNode=this.appendChildPrim(this.createPrim("path",t.id),t.evalVisProp("layer")),this.appendNodesToElement(t,"path"),this.updateCurve(t)},updateCurve:function(t){this._updateVisual(t),this.updatePathWithArrowHeads(t),this.setLineCap(t)},updatePathWithArrowHeads:function(t,e){var s,r,o=e?"highlight":"";s=e&&t.evalVisProp("highlightstrokewidth")?Math.max(t.evalVisProp("highlightstrokewidth"),t.evalVisProp("strokewidth")):t.evalVisProp("strokewidth"),r=this.getArrowHeadData(t,s,o),this.makeArrows(t,r),t.elementClass===i.Z.OBJECT_CLASS_LINE?this.updateLineWithEndings(t,r):t.elementClass===i.Z.OBJECT_CLASS_CURVE&&this.updatePath(t),this.setArrowSize(t,r)},getArrowHeadData:function(t,e,r){var n,a,h,l,c=o.Z.eps,d=0,u=0,p=0,f=0,_=t.evalVisProp("firstarrow"),m=t.evalVisProp("lastarrow");return(_||m)&&(n=s.Z.exists(_.type)?t.eval(_.type):t.elementClass===i.Z.OBJECT_CLASS_LINE?1:7,a=s.Z.exists(m.type)?t.eval(m.type):t.elementClass===i.Z.OBJECT_CLASS_LINE?1:7,_&&(l=6,s.Z.exists(_.size)&&(l=t.eval(_.size)),""!==r&&s.Z.exists(_[r+"size"])&&(l=t.eval(_[r+"size"])),h=e*l,2===n?(h*=.5,c+=e*l):3===n?(h=e*l/3,c+=e):4===n||5===n||6===n?(h=e*l/1.5,c+=e*l):7===n?(h=0,l=10,c+=e):c+=e*l,d+=h,p=l),m&&(l=6,s.Z.exists(m.size)&&(l=t.eval(m.size)),""!==r&&s.Z.exists(m[r+"size"])&&(l=t.eval(m[r+"size"])),h=e*l,2===a?(h*=.5,c+=e*l):3===a?(h=e*l/3,c+=e):4===a||5===a||6===a?(h=e*l/1.5,c+=e*l):7===a?(h=0,l=10,c+=e):c+=e*l,u+=h,f=l)),t.visPropCalc.typeFirst=n,t.visPropCalc.typeLast=a,{evFirst:_,evLast:m,typeFirst:n,typeLast:a,offFirst:d,offLast:u,sizeFirst:p,sizeLast:f,showFirst:1,showLast:1,minLen:c,strokeWidth:e}},updateLineWithEndings:function(t,e){var s,r,o;return s=new u.Z(i.Z.COORDS_BY_USER,t.point1.coords.usrCoords,t.board),r=new u.Z(i.Z.COORDS_BY_USER,t.point2.coords.usrCoords,t.board),o=t.evalVisProp("margin"),h.Z.calcStraight(t,s,r,o),this.handleTouchpoints(t,s,r,e),this.getPositionArrowHead(t,s,r,e),this.updateLinePrim(t.rendNode,s.scrCoords[1],s.scrCoords[2],r.scrCoords[1],r.scrCoords[2],t.board),this},updatePath:function(t){return t.evalVisProp("handdrawing")?this.updatePathPrim(t.rendNode,this.updatePathStringBezierPrim(t),t.board):this.updatePathPrim(t.rendNode,this.updatePathStringPrim(t),t.board),this},getPositionArrowHead:function(t,e,s,r){var o,n,a,h,l;return(r.evFirst||r.evLast)&&(n=a=h=l=0,o=e.distance(i.Z.COORDS_BY_SCREEN,s),r.evFirst&&"vml"!==t.board.renderer.type&&(o>=r.minLen?(n=(s.scrCoords[1]-e.scrCoords[1])*r.offFirst/o,a=(s.scrCoords[2]-e.scrCoords[2])*r.offFirst/o):r.showFirst=0),r.evLast&&"vml"!==t.board.renderer.type&&(o>=r.minLen?(h=(s.scrCoords[1]-e.scrCoords[1])*r.offLast/o,l=(s.scrCoords[2]-e.scrCoords[2])*r.offLast/o):r.showLast=0),e.setCoordinates(i.Z.COORDS_BY_SCREEN,[e.scrCoords[1]+n,e.scrCoords[2]+a],!1,!0),s.setCoordinates(i.Z.COORDS_BY_SCREEN,[s.scrCoords[1]-h,s.scrCoords[2]-l],!1,!0)),this},handleTouchpoints:function(t,e,s,r){var o,n,a,h,l,c,d;return(r.evFirst||r.evLast)&&(a=h=l=c=d=0,o=t.point1.evalVisProp("size")+t.point1.evalVisProp("strokewidth"),n=t.point2.evalVisProp("size")+t.point2.evalVisProp("strokewidth"),r.evFirst&&t.evalVisProp("touchfirstpoint")&&t.point1.evalVisProp("visible")&&(a=e.distance(i.Z.COORDS_BY_SCREEN,s),h=(s.scrCoords[1]-e.scrCoords[1])*o/a,l=(s.scrCoords[2]-e.scrCoords[2])*o/a),r.evLast&&t.evalVisProp("touchlastpoint")&&t.point2.evalVisProp("visible")&&(a=e.distance(i.Z.COORDS_BY_SCREEN,s),c=(s.scrCoords[1]-e.scrCoords[1])*n/a,d=(s.scrCoords[2]-e.scrCoords[2])*n/a),e.setCoordinates(i.Z.COORDS_BY_SCREEN,[e.scrCoords[1]+h,e.scrCoords[2]+l],!1,!0),s.setCoordinates(i.Z.COORDS_BY_SCREEN,[s.scrCoords[1]-c,s.scrCoords[2]-d],!1,!0)),this},setArrowSize:function(t,e){return e.evFirst&&this._setArrowWidth(t.rendNodeTriangleStart,e.showFirst*e.strokeWidth,t.rendNode,e.sizeFirst),e.evLast&&this._setArrowWidth(t.rendNodeTriangleEnd,e.showLast*e.strokeWidth,t.rendNode,e.sizeLast),this},setLineCap:function(t){},drawTicks:function(t){t.rendNode=this.appendChildPrim(this.createPrim("path",t.id),t.evalVisProp("layer")),this.appendNodesToElement(t,"path")},updateTicks:function(t){},drawEllipse:function(t){t.rendNode=this.appendChildPrim(this.createPrim("ellipse",t.id),t.evalVisProp("layer")),this.appendNodesToElement(t,"ellipse"),this.updateEllipse(t)},updateEllipse:function(t){this._updateVisual(t);var e=t.Radius();Math.abs(t.center.coords.usrCoords[0])>o.Z.eps&&!isNaN(e+t.center.coords.scrCoords[1]+t.center.coords.scrCoords[2])&&e*t.board.unitX<2e6&&this.updateEllipsePrim(t.rendNode,t.center.coords.scrCoords[1],t.center.coords.scrCoords[2],e*t.board.unitX,e*t.board.unitY),this.setLineCap(t)},drawPolygon:function(t){t.rendNode=this.appendChildPrim(this.createPrim("polygon",t.id),t.evalVisProp("layer")),this.appendNodesToElement(t,"polygon"),this.updatePolygon(t)},updatePolygon:function(t){this._updateVisual(t,{stroke:!0,dash:!0}),this.updatePolygonPrim(t.rendNode,t)},displayCopyright:function(t,e){},displayLogo:function(t,e){},drawInternalText:function(t){},updateInternalText:function(t){},drawText:function(t){var i,r,o,n;"html"===t.evalVisProp("display")&&e.Z.isBrowser&&"no"!==this.type?((i=this.container.ownerDocument.createElement("div")).style.position="absolute",i.className=t.evalVisProp("cssclass"),o=t.evalVisProp("layer"),s.Z.exists(o)||(o=0),r=""===this.container.style.zIndex?0:parseInt(this.container.style.zIndex,10),i.style.zIndex=r+o,this.container.appendChild(i),i.setAttribute("id",this.container.id+"_"+t.id)):i=this.drawInternalText(t),t.rendNode=i,t.htmlStr="",t.visProp.islabel&&s.Z.exists(t.visProp.anchor)?(n=t.visProp.anchor.evalVisProp("visible"),t.prepareUpdate().updateVisibility(n)):t.prepareUpdate().updateVisibility(),this.updateText(t)},updateText:function(e){var i,r,o,n,a,h,l,c,d,u,p,f=e.plaintext;if(e.visPropCalc.visible)if(this.updateTextStyle(e,!1),"html"===e.evalVisProp("display")&&"no"!==this.type){if(isNaN(e.coords.scrCoords[1]+e.coords.scrCoords[2])||(r=e.coords.scrCoords[1],r=Math.abs(r)<1e6?r:1e6,"right"===(a=e.getAnchorX())?(i=e.board.canvasWidth-r,u="right"):"middle"===a?(i=r-.5*e.size[0],u="center"):(i=r,u="left"),e.visPropOld.left!==a+i&&("right"===a?(e.rendNode.style.right=i+"px",e.rendNode.style.left="auto"):(e.rendNode.style.left=i+"px",e.rendNode.style.right="auto"),e.visPropOld.left=a+i),r=e.coords.scrCoords[2]+this.vOffsetText,r=Math.abs(r)<1e6?r:1e6,"bottom"===(h=e.getAnchorY())?(i=e.board.canvasHeight-r,p="bottom"):"middle"===h?(i=r-.5*e.size[1],p="center"):(i=r,p="top"),e.visPropOld.top!==h+i&&("bottom"===h?(e.rendNode.style.top="auto",e.rendNode.style.bottom=i+"px"):(e.rendNode.style.bottom="auto",e.rendNode.style.top=i+"px"),e.visPropOld.top=h+i)),e.htmlStr!==f){try{e.type===s.Z.OBJECT_TYPE_BUTTON?e.rendNodeButton.innerHTML=f:e.type===s.Z.OBJECT_TYPE_CHECKBOX||e.type===s.Z.OBJECT_TYPE_INPUT?e.rendNodeLabel.innerHTML=f:e.rendNode.innerHTML=f}catch(t){o=e.rendNode.parentNode,e.rendNode.parentNode.removeChild(e.rendNode),e.rendNode.innerHTML=f,o.appendChild(e.rendNode)}if(e.htmlStr=f,e.evalVisProp("usemathjax"))try{MathJax.typeset?MathJax.typeset([e.rendNode]):MathJax.Hub.Queue(["Typeset",MathJax.Hub,e.rendNode])}catch(e){t.Z.debug("MathJax (not yet) loaded")}else if(e.evalVisProp("usekatex"))try{(n=0===e.rendNode.innerHTML.indexOf("<span")&&e.rendNode.innerHTML.indexOf("<label")>0&&(e.rendNode.innerHTML.indexOf("<checkbox")>0||e.rendNode.innerHTML.indexOf("<input")>0)?e.rendNodeLabel:0===e.rendNode.innerHTML.indexOf("<button")?e.rendNodeButton:e.rendNode)&&katex.render(f,n,{macros:e.evalVisProp("katexmacros"),throwOnError:!1})}catch(e){t.Z.debug("KaTeX not loaded (yet)")}else if(e.evalVisProp("useasciimathml"))try{AMprocessNode(e.rendNode,!1)}catch(e){t.Z.debug("AsciiMathML not loaded (yet)")}}0!==(l=e.evalVisProp("rotate"))&&(l*=Math.PI/180,c=Math.cos(l),d=Math.sin(l),e.rendNode.style.transform="matrix("+[c,-1*d,d,c,0,0].join(",")+")",e.rendNode.style["transform-origin"]=u+" "+p),this.transformRect(e,e.transformations)}else this.updateInternalText(e)},_css2js:function(t){var e,i,r,o,n,a=[],h=s.Z.trim(t).replace(/;$/,"").split(";");for(i=h.length,e=0;e<i;++e)""!==s.Z.trim(h[e])&&(n=h[e].split(":"),r=s.Z.trim(n[0].replace(/-([a-z])/gi,(function(t,e){return e.toUpperCase()}))),o=s.Z.trim(n[1]),a.push({key:r,val:o}));return a},updateTextStyle:function(t,i){var r,o,n,a,h,l,c,d,u,p=e.Z.isBrowser?t.visProp.display:"internal",f=["rendNode","rendNodeTag","rendNodeLabel"],_=f.length,m=t.evalVisProp("fontunit"),g=["cssdefaultstyle","cssstyle"],b=g.length;if(i?(n=t.evalVisProp("highlightstrokecolor"),o=t.evalVisProp("highlightstrokeopacity"),a=t.evalVisProp("highlightcssclass")):(n=t.evalVisProp("strokecolor"),o=t.evalVisProp("strokeopacity"),a=t.evalVisProp("cssclass")),"no"!==this.type&&("html"===p||"canvas"!==this.type)){for(d=0;d<b;d++)for(u=t.evalVisProp((i?"highlight":"")+g[d]),h=0;h<_;h++)if(s.Z.exists(t[f[h]])&&""!==u&&t.visPropOld[g[d]+"_"+h]!==u){for(c in l=this._css2js(u))l.hasOwnProperty(c)&&(t[f[h]].style[l[c].key]=l[c].val);t.visPropOld[g[d]+"_"+h]=u}if(r=t.evalVisProp("fontsize"),t.visPropOld.fontsize!==r){t.needsSizeUpdate=!0;try{for(h=0;h<_;h++)s.Z.exists(t[f[h]])&&(t[f[h]].style.fontSize=r+m)}catch(e){for(h=0;h<_;h++)s.Z.exists(t[f[h]])&&(t[f[h]].style.fontSize=r)}t.visPropOld.fontsize=r}}return this.setTabindex(t),this.setObjectTransition(t),"html"===p&&"no"!==this.type?(t.visPropOld.cssclass!==a&&(t.rendNode.className=a,t.visPropOld.cssclass=a,t.needsSizeUpdate=!0),this.setObjectStrokeColor(t,n,o)):this.updateInternalTextStyle(t,n,o),t.evalVisProp("aria.enabled")&&this.setARIA(t),this},updateInternalTextStyle:function(t,e,i){this.setObjectStrokeColor(t,e,i)},drawImage:function(t){},updateImage:function(t){this.updateRectPrim(t.rendNode,t.coords.scrCoords[1],t.coords.scrCoords[2]-t.size[1],t.size[0],t.size[1]),this.updateImageURL(t),this.transformRect(t,t.transformations),this._updateVisual(t,{stroke:!0,dash:!0},!0)},joinTransforms:function(t,e){var i,s=t.board.origin.scrCoords[1],r=t.board.origin.scrCoords[2],n=t.board.unitX,a=t.board.unitY,h=e.length,l=[[1,0,0],[-s/n,1/n,0],[r/a,0,-1/a]];for(i=0;i<h;i++)l=o.Z.matMatMult(e[i].matrix,l);return l=o.Z.matMatMult([[1,0,0],[s,n,0],[r,0,-a]],l)},transformRect:function(t,e){},updateImageURL:function(t){},updateImageStyle:function(t,e){t.rendNode.className=t.evalVisProp(e?"highlightcssclass":"cssclass")},drawForeignObject:function(t){},updateForeignObject:function(t){},appendChildPrim:function(t,e){},appendNodesToElement:function(t,e){},createPrim:function(t,e){return null},remove:function(t){},makeArrows:function(t,e){},_setArrowWidth:function(t,e,i){},updateEllipsePrim:function(t,e,i,s,r){},updateLinePrim:function(t,e,i,s,r,o){},updatePathPrim:function(t,e,i){},updatePathStringPoint:function(t,e,i){},updatePathStringPrim:function(t){},updatePathStringBezierPrim:function(t){},updatePolygonPrim:function(t,e){},updateRectPrim:function(t,e,i,s,r){},display:function(t,e){t&&(t.visPropOld.visible=e)},hide:function(t){},highlight:function(t,e){var s,r,o;if(this.setObjectTransition(t),!t.visProp.draft){if(t.type===i.Z.OBJECT_TYPE_POLYGON)for(this.setObjectFillColor(t,t.evalVisProp("highlightfillcolor"),t.evalVisProp("highlightfillopacity")),r=t.evalVisProp("highlightbystrokewidth"),s=0;s<t.borders.length;s++)this.highlight(t.borders[s],!r);else t.elementClass===i.Z.OBJECT_CLASS_TEXT?this.updateTextStyle(t,!0):t.type===i.Z.OBJECT_TYPE_IMAGE?(this.updateImageStyle(t,!0),this.setObjectFillColor(t,t.evalVisProp("highlightfillcolor"),t.evalVisProp("highlightfillopacity"))):(this.setObjectStrokeColor(t,t.evalVisProp("highlightstrokecolor"),t.evalVisProp("highlightstrokeopacity")),this.setObjectFillColor(t,t.evalVisProp("highlightfillcolor"),t.evalVisProp("highlightfillopacity")));!e&&t.evalVisProp("highlightstrokewidth")&&(o=Math.max(t.evalVisProp("highlightstrokewidth"),t.evalVisProp("strokewidth")),this.setObjectStrokeWidth(t,o),t.elementClass!==i.Z.OBJECT_CLASS_LINE&&t.elementClass!==i.Z.OBJECT_CLASS_CURVE||this.updatePathWithArrowHeads(t,!0))}return this.setCssClass(t,t.evalVisProp("highlightcssclass")),this},noHighlight:function(t){var e,s;if(this.setObjectTransition(t),!t.evalVisProp("draft")){if(t.type===i.Z.OBJECT_TYPE_POLYGON)for(this.setObjectFillColor(t,t.evalVisProp("fillcolor"),t.evalVisProp("fillopacity")),e=0;e<t.borders.length;e++)this.noHighlight(t.borders[e]);else t.elementClass===i.Z.OBJECT_CLASS_TEXT?this.updateTextStyle(t,!1):t.type===i.Z.OBJECT_TYPE_IMAGE?(this.updateImageStyle(t,!1),this.setObjectFillColor(t,t.evalVisProp("fillcolor"),t.evalVisProp("fillopacity"))):(this.setObjectStrokeColor(t,t.evalVisProp("strokecolor"),t.evalVisProp("strokeopacity")),this.setObjectFillColor(t,t.evalVisProp("fillcolor"),t.evalVisProp("fillopacity")));s=t.evalVisProp("strokewidth"),this.setObjectStrokeWidth(t,s),t.elementClass!==i.Z.OBJECT_CLASS_LINE&&t.elementClass!==i.Z.OBJECT_CLASS_CURVE||this.updatePathWithArrowHeads(t,!1)}return this.setCssClass(t,t.evalVisProp("cssclass")),this},removeDraft:function(t){this.setObjectTransition(t),t.type===i.Z.OBJECT_TYPE_POLYGON?this.setObjectFillColor(t,t.evalVisProp("fillcolor"),t.evalVisProp("fillopacity")):(t.type===i.Z.OBJECT_CLASS_POINT&&this.setObjectFillColor(t,t.evalVisProp("fillcolor"),t.evalVisProp("fillopacity")),this.setObjectStrokeColor(t,t.evalVisProp("strokecolor"),t.evalVisProp("strokeopacity")),this.setObjectStrokeWidth(t,t.evalVisProp("strokewidth")))},setARIA:function(t){},setBuffering:function(t,e){},setCssClass:function(t,e){},setDashStyle:function(t){},setDraft:function(t){if(t.evalVisProp("draft")){var e=t.board.options.elements.draft.color,s=t.board.options.elements.draft.opacity;this.setObjectTransition(t),t.type===i.Z.OBJECT_TYPE_POLYGON?this.setObjectFillColor(t,e,s):(t.elementClass===i.Z.OBJECT_CLASS_POINT?this.setObjectFillColor(t,e,s):this.setObjectFillColor(t,"none",0),this.setObjectStrokeColor(t,e,s),this.setObjectStrokeWidth(t,t.board.options.elements.draft.strokeWidth))}},setGradient:function(t){},setLayer:function(t,e){},setObjectFillColor:function(t,e,i){},setObjectStrokeColor:function(t,e,i){},setObjectStrokeWidth:function(t,e){},setObjectTransition:function(t,e){},setPropertyPrim:function(t,e,i){},setShadow:function(t){},setTabindex:function(t){},show:function(t){},updateGradient:function(t){},suspendRedraw:function(){},unsuspendRedraw:function(){},drawNavigationBar:function(t,i){var r,o,n=function(t){t||(t=window.event),t.stopPropagation?t.stopPropagation():t.cancelBubble=!0},a=function(i,a,h,l){var c;h=h||"",(c=r.createElement("span")).innerHTML=i,c.style.paddingLeft="7px",c.style.paddingRight="7px",void 0!==c.classList&&(c.classList.add("JXG_navigation_button"),c.classList.add("JXG_navigation_button_"+l)),c.setAttribute("id",h+"_navigation_"+l),c.setAttribute("aria-hidden","true"),o.appendChild(c),e.Z.addEvent(c,"click",(function(e){return s.Z.bind(a,t)(),!1}),t),e.Z.addEvent(c,"pointerup",n,t),e.Z.addEvent(c,"pointerdown",n,t),e.Z.addEvent(c,"pointerleave",n,t),e.Z.addEvent(c,"mouseup",n,t),e.Z.addEvent(c,"mousedown",n,t),e.Z.addEvent(c,"touchend",n,t),e.Z.addEvent(c,"touchstart",n,t)};e.Z.isBrowser&&"no"!==this.type&&(r=t.containerObj.ownerDocument,(o=r.createElement("div")).setAttribute("id",t.container+"_navigationbar"),o.style.color=i.strokecolor,o.style.backgroundColor=i.fillcolor,o.style.padding=i.padding,o.style.position=i.position,o.style.fontSize=i.fontsize,o.style.cursor=i.cursor,o.style.zIndex=i.zindex,t.containerObj.appendChild(o),o.style.right=i.right,o.style.bottom=i.bottom,void 0!==o.classList&&o.classList.add("JXG_navigation"),t.attr.showfullscreen&&a(t.attr.fullscreen.symbol,(function(){t.toFullscreen(t.attr.fullscreen.id)}),t.container,"fullscreen"),t.attr.showscreenshot&&a(t.attr.screenshot.symbol,(function(){window.setTimeout((function(){t.renderer.screenshot(t,"",!1)}),330)}),t.container,"screenshot"),t.attr.showreload&&a("↻",(function(){t.reload()}),t.container,"reload"),t.attr.showcleartraces&&a("⊗",(function(){t.clearTraces()}),t.container,"cleartraces"),t.attr.shownavigation&&(t.attr.showzoom&&(a("–",t.zoomOut,t.container,"out"),a("o",t.zoom100,t.container,"100"),a("+",t.zoomIn,t.container,"in")),a("←",t.clickLeftArrow,t.container,"left"),a("↓",t.clickUpArrow,t.container,"down"),a("↑",t.clickDownArrow,t.container,"up"),a("→",t.clickRightArrow,t.container,"right")))},getElementById:function(t){var e;return s.Z.exists(this.container)?(e=this.container.id+"_"+t,s.Z.exists(CSS)&&s.Z.exists(CSS.escape)&&(e=CSS.escape(e)),this.container.querySelector("#"+e)):""},removeToInsertLater:function(t){var e=t.parentNode,i=t.nextSibling;if(null!==e)return e.removeChild(t),function(){i?e.insertBefore(t,i):e.appendChild(t)}},resize:function(t,e){},createTouchpoints:function(t){},showTouchpoint:function(t){},hideTouchpoint:function(t){},updateTouchpoint:function(t,e){},dumpToDataURI:function(t){},dumpToCanvas:function(t,e,i,s){},screenshot:function(t){}});var Z=t.Z.AbstractRenderer,C=[0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,9,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,7,8,8,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,10,3,3,3,3,3,3,3,3,3,3,3,3,4,3,3,11,6,6,6,5,8,8,8,8,8,8,8,8,8,8,8,0,12,24,36,60,96,84,12,12,12,48,72,12,12,12,12,12,12,12,12,12,12,12,12,12,0,12,12,12,12,12,0,12,0,12,12,12,24,12,12,12,12,12,24,12,24,12,12,12,12,12,12,12,12,12,24,12,12,12,12,12,24,12,12,12,12,12,12,12,24,12,12,12,12,12,12,12,12,12,36,12,36,12,12,12,36,12,12,12,12,12,36,12,36,12,12,12,36,12,12,12,12,12,12,12,12,12,12];t.Z.Util=t.Z.Util||{},t.Z.Util.UTF8={encode:function(t){var e,i,s="",r=t.length;if(t=t.replace(/\r\n/g,"\n"),"function"==typeof unescape&&"function"==typeof encodeURIComponent)return unescape(encodeURIComponent(t));for(e=0;e<r;e++)(i=t.charCodeAt(e))<128?s+=String.fromCharCode(i):i>127&&i<2048?(s+=String.fromCharCode(i>>6|192),s+=String.fromCharCode(63&i|128)):(s+=String.fromCharCode(i>>12|224),s+=String.fromCharCode(i>>6&63|128),s+=String.fromCharCode(63&i|128));return s},decode:function(t){var e,i,s,r=0,o=0,n=0,a=[],h=t.length,l=[];for(e=0;e<h;e++)i=t.charCodeAt(e),s=C[i],o=0!==n?63&i|o<<6:255>>s&i,0===(n=C[256+n+s])&&(o>65535?a.push(55232+(o>>10),56320+(1023&o)):a.push(o),++r%1e4==0&&(l.push(String.fromCharCode.apply(null,a)),a=[]));return l.push(String.fromCharCode.apply(null,a)),l.join("")},asciiCharCodeAt:function(t,e){var i=t.charCodeAt(e);if(i>255)switch(i){case 8364:i=128;break;case 8218:i=130;break;case 402:i=131;break;case 8222:i=132;break;case 8230:i=133;break;case 8224:i=134;break;case 8225:i=135;break;case 710:i=136;break;case 8240:i=137;break;case 352:i=138;break;case 8249:i=139;break;case 338:i=140;break;case 381:i=142;break;case 8216:i=145;break;case 8217:i=146;break;case 8220:i=147;break;case 8221:i=148;break;case 8226:i=149;break;case 8211:i=150;break;case 8212:i=151;break;case 732:i=152;break;case 8482:i=153;break;case 353:i=154;break;case 8250:i=155;break;case 339:i=156;break;case 382:i=158;break;case 376:i=159}return i}};var y=t.Z.Util.UTF8,P="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/",E="=";t.Z.Util=t.Z.Util||{},t.Z.Util.Base64={_getByte:function(t,e){return 255&t.charCodeAt(e)},_getIndex:function(t,e){return P.indexOf(t.charAt(e))},encode:function(t){var e,i,s,r,o,n=[];for(r=(s=(o=y.encode(t)).length)%3,e=0;e<s-r;e+=3)i=this._getByte(o,e)<<16|this._getByte(o,e+1)<<8|this._getByte(o,e+2),n.push(P.charAt(i>>18),P.charAt(i>>12&63),P.charAt(i>>6&63),P.charAt(63&i));switch(r){case 1:i=this._getByte(o,s-1),n.push(P.charAt(i>>2),P.charAt(i<<4&63),E,E);break;case 2:i=this._getByte(o,s-2)<<8|this._getByte(o,s-1),n.push(P.charAt(i>>10),P.charAt(i>>4&63),P.charAt(i<<2&63),E)}return n.join("")},decode:function(t,e){var i,s,r,o,n,a,h=[],l=[];if((r=(i=t.replace(/[^A-Za-z0-9+/=]/g,"")).length)%4!=0)throw new Error("JSXGraph/utils/base64: Can't decode string (invalid input length).");for(i.charAt(r-1)===E&&(o=1,i.charAt(r-2)===E&&(o=2),r-=4),s=0;s<r;s+=4)n=this._getIndex(i,s)<<18|this._getIndex(i,s+1)<<12|this._getIndex(i,s+2)<<6|this._getIndex(i,s+3),l.push(n>>16,n>>8&255,255&n),s%1e4==0&&(h.push(String.fromCharCode.apply(null,l)),l=[]);switch(o){case 1:n=this._getIndex(i,r)<<12|this._getIndex(i,r+1)<<6|this._getIndex(i,r+2),l.push(n>>10,n>>2&255);break;case 2:n=this._getIndex(i,s)<<6|this._getIndex(i,s+1),l.push(n>>4)}return h.push(String.fromCharCode.apply(null,l)),a=h.join(""),e&&(a=y.decode(a)),a},decodeAsArray:function(t){var e,i=this.decode(t),s=[],r=i.length;for(e=0;e<r;e++)s[e]=i.charCodeAt(e);return s}};var x=t.Z.Util.Base64;t.Z.FileReader={handleRemoteFile:function(e,i,r,o,n,a){var h=!1;try{h=new XMLHttpRequest,"raw"===r.toLowerCase()?h.overrideMimeType("text/plain; charset="+n):h.overrideMimeType("text/xml; charset="+n)}catch(t){try{h=new ActiveXObject("Msxml2.XMLHTTP")}catch(t){try{h=new ActiveXObject("Microsoft.XMLHTTP")}catch(t){h=!1}}}if(h){h.open("GET",e,o),"raw"===r.toLowerCase()?this.cbp=function(){var t=h;4===t.readyState&&i(t.responseText)}:this.cbp=function(){var t=h,e="";4===t.readyState&&(e=!s.Z.exists(t.responseStream)||"PK"!==t.responseText.slice(0,2)&&31!==y.asciiCharCodeAt(t.responseText.slice(0,1),0)?t.responseText:x.decode(jxgBinFileReader(t)),this.parseString(e,i,r,a))},this.cb=s.Z.bind(this.cbp,this),h.onreadystatechange=this.cb;try{h.send(null)}catch(t){throw new Error("JSXGraph: A problem occurred while trying to read remote file '"+e+"'.")}}else t.Z.debug("AJAX not activated!")},handleLocalFile:function(t,e,i,r,o,n){s.Z.exists(r)||(r=!0),"raw"===i.toLowerCase()?this.cbp=function(t){e(t.target.result)}:this.cbp=function(t){var s=t.target.result;this.parseString(s,e,i,n)},this.cb=s.Z.bind(this.cbp,this);var a=new FileReader;a.onload=this.cb,"raw"===i.toLowerCase()?a.readAsText(t):a.readAsText(t,o)},parseFileContent:function(t,e,i,r,o,n){s.Z.isString(t)||void 0===FileReader?this.handleRemoteFile(t,e,i,r,o,n):this.handleLocalFile(t,e,i,r,o,n)},parseString:function(e,i,r,o){var n;if(r=r.toLowerCase(),n=t.Z.readers[r],s.Z.exists(n))new n(i,e).read();else if("jessiecode"!==r)throw new Error("JSXGraph: There is no reader available for '"+r+"'.");s.Z.isFunction(o)&&o(i)}},!e.Z.isMetroApp()&&e.Z.isBrowser&&"object"==typeof navigator&&/msie/i.test(navigator.userAgent)&&!/opera/i.test(navigator.userAgent)&&document&&document.write&&document.write('<script type="text/vbscript">\nFunction Base64Encode(inData)\n  Const Base64 = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/"\n  Dim cOut, sOut, I\n  For I = 1 To LenB(inData) Step 3\n    Dim nGroup, pOut, sGroup\n    nGroup = &H10000 * AscB(MidB(inData, I, 1)) + _\n      &H100 * MyASC(MidB(inData, I + 1, 1)) + MyASC(MidB(inData, I + 2, 1))\n    nGroup = Oct(nGroup)\n    nGroup = String(8 - Len(nGroup), "0") & nGroup\n    pOut = Mid(Base64, CLng("&o" & Mid(nGroup, 1, 2)) + 1, 1) + _\n      Mid(Base64, CLng("&o" & Mid(nGroup, 3, 2)) + 1, 1) + _\n      Mid(Base64, CLng("&o" & Mid(nGroup, 5, 2)) + 1, 1) + _\n      Mid(Base64, CLng("&o" & Mid(nGroup, 7, 2)) + 1, 1)\n    sOut = sOut + pOut\n  Next\n  Select Case LenB(inData) Mod 3\n    Case 1: \'8 bit final\n      sOut = Left(sOut, Len(sOut) - 2) + "=="\n    Case 2: \'16 bit final\n      sOut = Left(sOut, Len(sOut) - 1) + "="\n  End Select\n  Base64Encode = sOut\nEnd Function\n\nFunction MyASC(OneChar)\n  If OneChar = "" Then MyASC = 0 Else MyASC = AscB(OneChar)\nEnd Function\n\nFunction jxgBinFileReader(xhr)\n    Dim byteString\n    Dim b64String\n    Dim i\n    byteString = xhr.responseBody\n    ReDim byteArray(LenB(byteString))\n    For i = 1 To LenB(byteString)\n        byteArray(i-1) = AscB(MidB(byteString, i, 1))\n    Next\n    b64String = Base64Encode(byteString)\n    jxgBinFileReader = b64String\nEnd Function\n<\/script>\n');var O=t.Z.FileReader,M=__webpack_require__(632),w=__webpack_require__(254),S=__webpack_require__(327);t.Z.Composition=function(t){var e,i=this,r=["setAttribute","setParents","prepareUpdate","updateRenderer","update","fullUpdate","highlight","noHighlight"],o=function(t){return function(){var e;for(e in i.elements)i.elements.hasOwnProperty(e)&&s.Z.exists(i.elements[e][t])&&i.elements[e][t].apply(i.elements[e],arguments);return i}};for(e=0;e<r.length;e++)this[r[e]]=o(r[e]);for(e in this.elements={},this.objects=this.elements,this.elementsByName={},this.objectsList=[],this.groups={},this.methodMap={setAttribute:"setAttribute",setProperty:"setAttribute",setParents:"setParents",add:"add",remove:"remove",select:"select"},t)t.hasOwnProperty(e)&&this.add(e,t[e]);this.dump=!0,this.subs={}},t.Z.extend(t.Z.Composition.prototype,{add:function(t,e){return!(s.Z.exists(this[t])||!s.Z.exists(e))&&(s.Z.exists(e.id)?this.elements[e.id]=e:this.elements[t]=e,s.Z.exists(e.name)&&(this.elementsByName[e.name]=e),e.on("attribute:name",this.nameListener,this),this.objectsList.push(e),this[t]=e,this.methodMap[t]=e,!0)},remove:function(t){var e,i=!1;for(e in this.elements)if(this.elements.hasOwnProperty(e)&&this.elements[e].id===this[t].id){i=!0;break}return i&&(delete this.elements[this[t].id],delete this[t]),i},nameListener:function(t,e,i){delete this.elementsByName[t],this.elementsByName[e]=i},select:function(e){return s.Z.exists(t.Z.Board)?t.Z.Board.prototype.select.call(this,e):new t.Z.Composition},getParents:function(){return this.parents},getType:function(){return this.elType},getAttributes:function(){var t,e={};for(t in this.subs)this.subs.hasOwnProperty(t)&&(e[t]=this.subs[t].visProp);return this.attr}});var T=t.Z.Composition;t.Z.Board=function(o,n,a,h,l,c,d,u,p,f,_){if(this.BOARD_MODE_NONE=0,this.BOARD_MODE_DRAG=1,this.BOARD_MODE_MOVE_ORIGIN=2,this.BOARD_MODE_ZOOM=17,this.BOARD_QUALITY_LOW=1,this.BOARD_QUALITY_HIGH=2,s.Z.exists(_.document)&&!1!==_.document?this.document=_.document:e.Z.isBrowser&&(this.document=document),this.container="",this.id="",this.containerObj=null,s.Z.isString(o)?(this.container=o,this.containerObj=e.Z.isBrowser?this.document.getElementById(this.container):null):e.Z.isBrowser&&(this.containerObj=o,this.container=this.containerObj.getAttribute("id"),null===this.container&&(this.container="null"+parseInt(16777216*Math.random()).toString())),e.Z.isBrowser&&"no"!==n.type&&null===this.containerObj)throw new Error('\nJSXGraph: HTML container element "'+o+'" not found.');s.Z.exists(a)&&""!==a&&e.Z.isBrowser&&!s.Z.exists(this.document.getElementById(a))?this.id=a:this.id=this.generateId(),this.renderer=n,this.grids=[],this.options=s.Z.deepCopy(v.Z),this.attr=_,"default"!==this.attr.theme&&s.Z.exists(t.Z.themes[this.attr.theme])&&s.Z.mergeAttr(this.options,t.Z.themes[this.attr.theme],!0),this.dimension=2,this.jc=new w.Z,this.jc.use(this),this.origin={},this.origin.usrCoords=[1,0,0],this.origin.scrCoords=[1,h[0],h[1]],this.zoomX=l,this.zoomY=c,this.unitX=d*this.zoomX,this.unitY=u*this.zoomY,this.keepaspectratio=!1,this.canvasWidth=p,this.canvasHeight=f,r.Z.eventify(this),this.hooks=[],this.dependentBoards=[],this.inUpdate=!1,this.objects={},this.objectsList=[],this.groups={},this.animationObjects={},this.highlightedObjects={},this.numObjects=0,this.elementsByName={},this.mode=this.BOARD_MODE_NONE,this.updateQuality=this.BOARD_QUALITY_HIGH,this.isSuspendedRedraw=!1,this.calculateSnapSizes(),this.drag_dx=0,this.drag_dy=0,this.drag_position=[0,0],this.mouse={},this.touches=[],this.xmlString="",this.cPos=[],this.touchMoveLast=0,this.touchMoveLastId=1/0,this.positionAccessLast=0,this.downObjects=[],this.clickObjects={},this.focusObjects=[],(this.attr.showcopyright||this.attr.showlogo)&&this.renderer.displayLogo(i.Z.licenseLogo,parseInt(this.options.text.fontSize,10),this),this.attr.showcopyright&&this.renderer.displayCopyright(i.Z.licenseText,parseInt(this.options.text.fontSize,10)),this.needsFullUpdate=!1,this.reducedUpdate=!1,this.currentCBDef="none",this.geonextCompatibilityMode=!1,this.options.text.useASCIIMathML&&translateASCIIMath?init():this.options.text.useASCIIMathML=!1,this.hasMouseHandlers=!1,this.hasTouchHandlers=!1,this.hasPointerHandlers=!1,this.hasWheelHandlers=!1,this.hasMouseUp=!1,this.hasTouchEnd=!1,this.hasPointerUp=!1,this.resizeHandlers=[],this._drag_offset=[0,0],this._inputDevice="mouse",this._board_touches=[],this.selectingMode=!1,this.isSelecting=!1,this._isScrolling=!1,this._isResizing=!1,this._change3DView=!1,this.selectingBox=[[0,0],[0,0]],this.userLog=[],this.mathLib=Math,this.mathLibJXG=t.Z.Math,!0===this.attr.registerevents?this.attr.registerevents={fullscreen:!0,keyboard:!0,pointer:!0,resize:!0,wheel:!0}:"object"==typeof this.attr.registerevents&&(s.Z.exists(this.attr.registerevents.fullscreen)||(this.attr.registerevents.fullscreen=!0),s.Z.exists(this.attr.registerevents.keyboard)||(this.attr.registerevents.keyboard=!0),s.Z.exists(this.attr.registerevents.pointer)||(this.attr.registerevents.pointer=!0),s.Z.exists(this.attr.registerevents.resize)||(this.attr.registerevents.resize=!0),s.Z.exists(this.attr.registerevents.wheel)||(this.attr.registerevents.wheel=!0)),!1!==this.attr.registerevents&&(this.attr.registerevents.fullscreen&&this.addFullscreenEventHandlers(),this.attr.registerevents.keyboard&&this.addKeyboardEventHandlers(),this.attr.registerevents.pointer&&this.addEventHandlers(),this.attr.registerevents.resize&&this.addResizeEventHandlers(),this.attr.registerevents.wheel&&this.addWheelEventHandlers()),this.methodMap={update:"update",fullUpdate:"fullUpdate",on:"on",off:"off",trigger:"trigger",setAttribute:"setAttribute",setBoundingBox:"setBoundingBox",setView:"setBoundingBox",getBoundingBox:"getBoundingBox",BoundingBox:"getBoundingBox",getView:"getBoundingBox",View:"getBoundingBox",migratePoint:"migratePoint",colorblind:"emulateColorblindness",suspendUpdate:"suspendUpdate",unsuspendUpdate:"unsuspendUpdate",clearTraces:"clearTraces",left:"clickLeftArrow",right:"clickRightArrow",up:"clickUpArrow",down:"clickDownArrow",zoomIn:"zoomIn",zoomOut:"zoomOut",zoom100:"zoom100",zoomElements:"zoomElements",remove:"removeObject",removeObject:"removeObject"}},t.Z.extend(t.Z.Board.prototype,{generateName:function(t){var e,r,o=this.attr.maxnamelength,n="",a="",h=[],l="";if(t.type===i.Z.OBJECT_TYPE_TICKS)return"";for(e=s.Z.isPoint(t)||s.Z.isPoint3D(t)?["","A","B","C","D","E","F","G","H","I","J","K","L","M","N","O","P","Q","R","S","T","U","V","W","X","Y","Z"]:t.type===i.Z.OBJECT_TYPE_ANGLE?["","&alpha;","&beta;","&gamma;","&delta;","&epsilon;","&zeta;","&eta;","&theta;","&iota;","&kappa;","&lambda;","&mu;","&nu;","&xi;","&omicron;","&pi;","&rho;","&sigma;","&tau;","&upsilon;","&phi;","&chi;","&psi;","&omega;"]:["","a","b","c","d","e","f","g","h","i","j","k","l","m","n","o","p","q","r","s","t","u","v","w","x","y","z"],s.Z.isPoint(t)||s.Z.isPoint3D(t)||t.elementClass===i.Z.OBJECT_CLASS_LINE||t.type===i.Z.OBJECT_TYPE_ANGLE||(n=t.type===i.Z.OBJECT_TYPE_POLYGON?"P_{":t.elementClass===i.Z.OBJECT_CLASS_CIRCLE?"k_{":t.elementClass===i.Z.OBJECT_CLASS_TEXT?"t_{":"s_{",a="}"),r=0;r<o;r++)h[r]=0;for(;h[o-1]<e.length;){for(h[0]=1;h[0]<e.length;h[0]++){for(l=n,r=o;r>0;r--)l+=e[h[r-1]];if(!s.Z.exists(this.elementsByName[l+a]))return l+a}for(h[0]=e.length,r=1;r<o;r++)h[r-1]===e.length&&(h[r-1]=1,h[r]+=1)}return""},generateId:function(){for(var e=1;s.Z.exists(t.Z.boards["jxgBoard"+e]);)e=Math.round(16777216*Math.random());return"jxgBoard"+e},setId:function(t,e){var i,r=this.numObjects,o=t.id;if(this.numObjects+=1,""===o||!s.Z.exists(o))for(o=this.id+e+r;s.Z.exists(this.objects[o]);)i=Math.round(65535*Math.random()),o=this.id+e+r+"-"+i;return t.id=o,this.objects[o]=t,t._pos=this.objectsList.length,this.objectsList[this.objectsList.length]=t,o},finalizeAdding:function(t){!1===t.evalVisProp("visible")&&this.renderer.display(t,!1)},finalizeLabel:function(t){t.hasLabel&&!t.label.evalVisProp("islabel")&&!1===t.label.evalVisProp("visible")&&this.renderer.display(t.label,!1)},checkFrameRate:function(t){var e=!1,i=(new Date).getTime();return s.Z.exists(t.pointerId)&&this.touchMoveLastId!==t.pointerId&&(e=!0,this.touchMoveLastId=t.pointerId),!e&&(i-this.touchMoveLast)*this.attr.maxframerate>=1e3&&(e=!0),e&&(this.touchMoveLast=i),e},getCoordsTopLeftCorner:function(){var t,i,r,o,n,a=this.document.ownerDocument||this.document,h=a.documentElement||this.document.body.parentNode,l=a.body,c=this.containerObj;if(this.cPos.length>0&&(this.mode===this.BOARD_MODE_DRAG||this.mode===this.BOARD_MODE_MOVE_ORIGIN||(new Date).getTime()-this.positionAccessLast<1e3))return this.cPos;if(this.positionAccessLast=(new Date).getTime(),c.getBoundingClientRect){for(r=c.getBoundingClientRect(),o=1,n=c;n&&s.Z.exists(n.parentNode);)s.Z.exists(n.style)&&s.Z.exists(n.style.zoom)&&""!==n.style.zoom&&(o*=parseFloat(n.style.zoom)),n=n.parentNode;return(t=[r.left*o,r.top*o])[0]+=e.Z.getProp(c,"border-left-width"),t[1]+=e.Z.getProp(c,"border-top-width"),"vml"!==this.renderer.type&&(t[0]+=e.Z.getProp(c,"padding-left"),t[1]+=e.Z.getProp(c,"padding-top")),this.cPos=t.slice(),this.cPos}return t=e.Z.getOffset(c),i=this.document.documentElement.ownerDocument,!this.containerObj.currentStyle&&i.defaultView&&(t[0]+=e.Z.getProp(h,"margin-left"),t[1]+=e.Z.getProp(h,"margin-top"),t[0]+=e.Z.getProp(h,"border-left-width"),t[1]+=e.Z.getProp(h,"border-top-width"),t[0]+=e.Z.getProp(h,"padding-left"),t[1]+=e.Z.getProp(h,"padding-top")),l&&(t[0]+=e.Z.getProp(l,"left"),t[1]+=e.Z.getProp(l,"top")),"object"==typeof google&&google.translate&&(t[0]+=10,t[1]+=25),t[0]+=e.Z.getProp(c,"border-left-width"),t[1]+=e.Z.getProp(c,"border-top-width"),"vml"!==this.renderer.type&&(t[0]+=e.Z.getProp(c,"padding-left"),t[1]+=e.Z.getProp(c,"padding-top")),t[0]+=this.attr.offsetx,t[1]+=this.attr.offsety,this.cPos=t.slice(),this.cPos},getPointLoc:function(t,e){var i,r,o,n;return i=this.getBoundingBox(),2===(r=t).length&&r.unshift(void 0),o=[0,0],n=e||0,s.Z.isNumber(n)&&(n=[n,n,n,n]),r[1]>i[2]-n[1]&&(o[0]=1),r[1]<i[0]+n[3]&&(o[0]=-1),r[2]>i[1]-n[0]&&(o[1]=1),r[2]<i[3]+n[2]&&(o[1]=-1),o},getLocationOrigin:function(t){return this.getPointLoc([0,0],t)},getMousePosition:function(t,i){var s,r,n=this.getCoordsTopLeftCorner();return s=e.Z.getPosition(t,i,this.document),this.updateCSSTransforms(),r=[1,s[0]-n[0],s[1]-n[1]],(r=o.Z.matVecMult(this.cssTransMat,r))[1]/=r[0],r[2]/=r[0],[r[1],r[2]]},initMoveOrigin:function(t,e){this.drag_dx=t-this.origin.scrCoords[1],this.drag_dy=e-this.origin.scrCoords[2],this.mode=this.BOARD_MODE_MOVE_ORIGIN,this.updateQuality=this.BOARD_QUALITY_LOW},initMoveObject:function(t,e,r,o){var n,a,h,l=[],c=[],d=this.objectsList.length,u={visProp:{layer:-1e4}};for(this._shiftKey=r.shiftKey,this._ctrlKey=r.ctrlKey,a=0;a<d;a++)h=(n=this.objectsList[a]).hasPoint&&n.hasPoint(t,e),n.visPropCalc.visible&&h&&(n.triggerEventHandlers([o+"down","down"],[r]),this.downObjects.push(n)),h&&n.isDraggable&&n.visPropCalc.visible&&(this.geonextCompatibilityMode&&(s.Z.isPoint(n)||n.elementClass===i.Z.OBJECT_CLASS_TEXT)||!this.geonextCompatibilityMode)&&!n.evalVisProp("fixed")&&(n.visProp.layer>u.visProp.layer||n.visProp.layer===u.visProp.layer&&n.lastDragTime.getTime()>=u.lastDragTime.getTime())&&(this.attr.ignorelabels&&s.Z.exists(u.label)&&n===u.label||(u=n,l.push(u),s.Z.exists(u.coords)?u.elementClass===i.Z.OBJECT_CLASS_POINT||u.relativeCoords?c.push(p.Z.subtract(u.coords.scrCoords.slice(1),[t,e])):c.push(p.Z.subtract(u.actualCoords.scrCoords.slice(1),[t,e])):c.push([0,0])));return this.attr.drag.enabled&&l.length>0&&(this.mode=this.BOARD_MODE_DRAG),this.attr.takefirst?(l.length=1,this._drag_offset=c[0]):(l=l.slice(-1),this._drag_offset=c[c.length-1]),this._drag_offset||(this._drag_offset=[0,0]),"svg"===this.renderer.type&&s.Z.exists(l[0])&&l[0].evalVisProp("dragtotopoflayer")&&1===l.length&&s.Z.exists(l[0].rendNode)&&l[0].rendNode.parentNode.appendChild(l[0].rendNode),l.length>=1&&(l[0].highlight(!0),this.triggerEventHandlers(["mousehit","hit"],[r,l[0]])),l},moveObject:function(t,e,r,o,n){var a,h,l,c=new u.Z(i.Z.COORDS_BY_SCREEN,this.getScrCoordsOfMouse(t,e),this);r&&r.obj&&((a=r.obj).coords&&(h=a.coords.scrCoords.slice()),this.addLogEntry("drag",a,c.usrCoords.slice(1)),this.drag_position=[c.scrCoords[1],c.scrCoords[2]],this.drag_position=p.Z.add(this.drag_position,this._drag_offset),this._shiftKey=o.shiftKey,this._ctrlKey=o.ctrlKey,s.Z.exists(a.coords)?a.setPositionDirectly(i.Z.COORDS_BY_SCREEN,this.drag_position,[t,e]):(this.displayInfobox(!1),isNaN(r.targets[0].Xprev+r.targets[0].Yprev)||a.setPositionDirectly(i.Z.COORDS_BY_SCREEN,[c.scrCoords[1],c.scrCoords[2]],[r.targets[0].Xprev,r.targets[0].Yprev]),r.targets[0].Xprev=c.scrCoords[1],r.targets[0].Yprev=c.scrCoords[2]),s.Z.exists(a.coords)&&(a.prepareUpdate().update(!1).updateRenderer(),this.updateInfobox(a),a.prepareUpdate().update(!0).updateRenderer()),a.coords&&(l=a.coords.scrCoords),a.coords&&h[1]===l[1]&&h[2]===l[2]||(a.triggerEventHandlers([n+"drag","drag"],[o]),this.update(a)),a.highlight(!0),this.triggerEventHandlers(["mousehit","hit"],[o,a]),a.lastDragTime=new Date)},twoFingerMove:function(t,e,r){var o;s.Z.exists(t)&&s.Z.exists(t.obj)&&((o=t.obj).elementClass===i.Z.OBJECT_CLASS_LINE||o.type===i.Z.OBJECT_TYPE_POLYGON?this.twoFingerTouchObject(t.targets,o,e):o.elementClass===i.Z.OBJECT_CLASS_CIRCLE&&this.twoFingerTouchCircle(t.targets,o,e),r&&o.triggerEventHandlers(["touchdrag","drag"],[r]))},getTwoFingerTransform(t,e,s,r){var n,a,h,l,c,d,p,f,_,m,g,b,v,Z,C,y,P;return a=(n=new u.Z(i.Z.COORDS_BY_SCREEN,[t.Xprev,t.Yprev],this).usrCoords)[1],h=n[2],l=(n=new u.Z(i.Z.COORDS_BY_SCREEN,[e.Xprev,e.Yprev],this).usrCoords)[1],c=n[2],f=(n=new u.Z(i.Z.COORDS_BY_SCREEN,[t.X,t.Y],this).usrCoords)[1],_=n[2],Z=((b=(m=(n=new u.Z(i.Z.COORDS_BY_SCREEN,[e.X,e.Y],this).usrCoords)[1])-f)*(d=l-a)+(v=(g=n[2])-_)*(p=c-h))/(y=d*d+p*p),C=(v*d-b*p)/y,s||(Z/=P=o.Z.hypot(Z,C),C/=P),r||(C=0),[1,0,0,.5*(f+m-Z*(a+l)+C*(h+c)),Z,-C,.5*(_+g-C*(a+l)-Z*(h+c)),C,Z]},twoFingerTouchObject:function(t,e,r){var o,n,a,h,l,c=!1;if(s.Z.exists(t[0])&&s.Z.exists(t[1])&&!isNaN(t[0].Xprev+t[0].Yprev+t[1].Xprev+t[1].Yprev)){if(n=this.getTwoFingerTransform(t[0],t[1],e.evalVisProp("scalable"),e.evalVisProp("rotatable")),(o=this.create("transform",n,{type:"generic"})).update(),e.elementClass===i.Z.OBJECT_CLASS_LINE)a=[],e.point1.draggable()&&a.push(e.point1),e.point2.draggable()&&a.push(e.point2),o.applyOnce(a);else if(e.type===i.Z.OBJECT_TYPE_POLYGON){for(l=e.vertices.length-1,c=e.evalVisProp("snaptogrid")||e.evalVisProp("snaptopoints"),h=0;h<l&&!c;++h)c=(c=c||e.vertices[h].evalVisProp("snaptogrid")||e.vertices[h].evalVisProp("snaptopoints"))||!e.vertices[h].draggable();if(!c){for(a=[],h=0;h<l;++h)e.vertices[h].draggable()&&a.push(e.vertices[h]);o.applyOnce(a)}}this.update(),e.highlight(!0)}},twoFingerTouchCircle:function(t,e,r){var o,n,a,l,c,d,p,f,_,m,g;"pointCircle"!==e.method&&"pointLine"!==e.method&&s.Z.exists(t[0])&&s.Z.exists(t[1])&&!isNaN(t[0].Xprev+t[0].Yprev+t[1].Xprev+t[1].Yprev)&&(r===t[0].num?(o=t[1],n=t[0]):(o=t[0],n=t[1]),c=new u.Z(i.Z.COORDS_BY_SCREEN,[o.Xprev,o.Yprev],this).usrCoords,l=new u.Z(i.Z.COORDS_BY_SCREEN,[n.Xprev,n.Yprev],this).usrCoords,a=new u.Z(i.Z.COORDS_BY_SCREEN,[n.X,n.Y],this).usrCoords,p=h.Z.rad(l.slice(1),c.slice(1),a.slice(1)),f=this.create("transform",[-c[1],-c[2]],{type:"translate"}),_=this.create("transform",[p],{type:"rotate"}),f.melt(_),e.evalVisProp("scalable")&&(d=h.Z.distance(c,a)/h.Z.distance(c,l),m=this.create("transform",[d,d],{type:"scale"}),f.melt(m)),g=this.create("transform",[c[1],c[2]],{type:"translate"}),f.melt(g),e.center.draggable()&&f.applyOnce([e.center]),"twoPoints"===e.method?e.point2.draggable()&&f.applyOnce([e.point2]):"pointRadius"===e.method&&s.Z.isNumber(e.updateRadius.origin)&&e.setRadius(e.radius*d),this.update(e.center),e.highlight(!0))},highlightElements:function(t,e,i,r){var o,n,a,h={},l=this.objectsList.length;for(o=0;o<l;o++)a=(n=this.objectsList[o]).id,s.Z.exists(n.hasPoint)&&n.visPropCalc.visible&&n.hasPoint(t,e)&&(this.updateInfobox(n),s.Z.exists(this.highlightedObjects[a])||(h[a]=n,n.highlight(),this.triggerEventHandlers(["mousehit","hit"],[i,n,r])),n.mouseover?n.triggerEventHandlers(["mousemove","move"],[i]):(n.triggerEventHandlers(["mouseover","over"],[i]),n.mouseover=!0));for(o=0;o<l;o++)a=(n=this.objectsList[o]).id,n.mouseover&&(h[a]||(n.triggerEventHandlers(["mouseout","out"],[i]),n.mouseover=!1))},saveStartPos:function(e,r){var o,n,a=[];if(e.type===i.Z.OBJECT_TYPE_TICKS)a.push([1,NaN,NaN]);else if(e.elementClass===i.Z.OBJECT_CLASS_LINE)a.push(e.point1.coords.usrCoords),a.push(e.point2.coords.usrCoords);else if(e.elementClass===i.Z.OBJECT_CLASS_CIRCLE)a.push(e.center.coords.usrCoords),"twoPoints"===e.method&&a.push(e.point2.coords.usrCoords);else if(e.type===i.Z.OBJECT_TYPE_POLYGON)for(n=e.vertices.length-1,o=0;o<n;o++)a.push(e.vertices[o].coords.usrCoords);else if(e.type===i.Z.OBJECT_TYPE_SECTOR)a.push(e.point1.coords.usrCoords),a.push(e.point2.coords.usrCoords),a.push(e.point3.coords.usrCoords);else if(s.Z.isPoint(e)||e.type===i.Z.OBJECT_TYPE_GLIDER)a.push(e.coords.usrCoords);else if(e.elementClass===i.Z.OBJECT_CLASS_CURVE)e.points.length>0&&a.push(e.points[0].usrCoords);else try{a.push(e.coords.usrCoords)}catch(e){t.Z.debug("JSXGraph+ saveStartPos: obj.coords.usrCoords not available: "+e)}for(n=a.length,o=0;o<n;o++)r.Zstart.push(a[o][0]),r.Xstart.push(a[o][1]),r.Ystart.push(a[o][2])},mouseOriginMoveStart:function(t){var e,i;return(e=this._isRequiredKeyPressed(t,"pan"))&&(i=this.getMousePosition(t),this.initMoveOrigin(i[0],i[1])),e},mouseOriginMove:function(t){var e,i=this.mode===this.BOARD_MODE_MOVE_ORIGIN;return i&&(e=this.getMousePosition(t),this.moveOrigin(e[0],e[1],!0)),i},touchStartMoveOriginOneFinger:function(t){var e,i,s=t.touches;return(e=this.attr.pan.enabled&&!this.attr.pan.needtwofingers&&1===s.length)&&(i=this.getMousePosition(t,0),this.initMoveOrigin(i[0],i[1])),e},touchOriginMove:function(t){var e,i=this.mode===this.BOARD_MODE_MOVE_ORIGIN;return i&&(e=this.getMousePosition(t,0),this.moveOrigin(e[0],e[1],!0)),i},originMoveEnd:function(){this.updateQuality=this.BOARD_QUALITY_HIGH,this.mode=this.BOARD_MODE_NONE},suppressDefault:function(t){return s.Z.exists(t)&&t.preventDefault(),!1},addEventHandlers:function(){e.Z.supportsPointerEvents()?this.addPointerEventHandlers():(this.addMouseEventHandlers(),this.addTouchEventHandlers()),null!==this.containerObj&&e.Z.addEvent(this.containerObj,"contextmenu",this.suppressDefault,this)},removeEventHandlers:function(){(this.hasPointerHandlers||this.hasMouseHandlers||this.hasTouchHandlers)&&null!==this.containerObj&&e.Z.removeEvent(this.containerObj,"contextmenu",this.suppressDefault,this),this.removeMouseEventHandlers(),this.removeTouchEventHandlers(),this.removePointerEventHandlers(),this.removeFullscreenEventHandlers(),this.removeKeyboardEventHandlers(),this.removeResizeEventHandlers()},addResizeEventHandlers:function(){if(this.resizeHandlers=[],e.Z.isBrowser){try{this.startResizeObserver(),this.resizeHandlers.push("resizeobserver")}catch(t){e.Z.addEvent(window,"resize",this.resizeListener,this),this.startIntersectionObserver(),this.resizeHandlers.push("resize")}e.Z.addEvent(window,"scroll",this.scrollListener,this),this.resizeHandlers.push("scroll");try{window.matchMedia("print").addEventListener("change",this.printListenerMatch.bind(this)),window.matchMedia("screen").addEventListener("change",this.printListenerMatch.bind(this)),this.resizeHandlers.push("print")}catch(e){t.Z.debug("Error adding printListener",e)}}},removeResizeEventHandlers:function(){var t;if(this.resizeHandlers.length>0&&e.Z.isBrowser){for(t=0;t<this.resizeHandlers.length;t++)switch(this.resizeHandlers[t]){case"resizeobserver":s.Z.exists(this.resizeObserver)&&this.stopResizeObserver();break;case"resize":e.Z.removeEvent(window,"resize",this.resizeListener,this),s.Z.exists(this.intersectionObserver)&&this.stopIntersectionObserver();break;case"scroll":e.Z.removeEvent(window,"scroll",this.scrollListener,this);break;case"print":window.matchMedia("print").removeEventListener("change",this.printListenerMatch.bind(this),!1),window.matchMedia("screen").removeEventListener("change",this.printListenerMatch.bind(this),!1)}this.resizeHandlers=[]}},addPointerEventHandlers:function(){if(!this.hasPointerHandlers&&e.Z.isBrowser){var t=this.attr.movetarget||this.containerObj;window.navigator.msPointerEnabled?(e.Z.addEvent(this.containerObj,"MSPointerDown",this.pointerDownListener,this),e.Z.addEvent(t,"MSPointerMove",this.pointerMoveListener,this)):(e.Z.addEvent(this.containerObj,"pointerdown",this.pointerDownListener,this),e.Z.addEvent(t,"pointermove",this.pointerMoveListener,this),e.Z.addEvent(t,"pointerleave",this.pointerLeaveListener,this),e.Z.addEvent(t,"click",this.pointerClickListener,this),e.Z.addEvent(t,"dblclick",this.pointerDblClickListener,this)),null!==this.containerObj&&(this.containerObj.style.touchAction="none"),this.hasPointerHandlers=!0}},addMouseEventHandlers:function(){if(!this.hasMouseHandlers&&e.Z.isBrowser){var t=this.attr.movetarget||this.containerObj;e.Z.addEvent(this.containerObj,"mousedown",this.mouseDownListener,this),e.Z.addEvent(t,"mousemove",this.mouseMoveListener,this),e.Z.addEvent(t,"click",this.mouseClickListener,this),e.Z.addEvent(t,"dblclick",this.mouseDblClickListener,this),this.hasMouseHandlers=!0}},addTouchEventHandlers:function(t){if(!this.hasTouchHandlers&&e.Z.isBrowser){var i=this.attr.movetarget||this.containerObj;e.Z.addEvent(this.containerObj,"touchstart",this.touchStartListener,this),e.Z.addEvent(i,"touchmove",this.touchMoveListener,this),this.hasTouchHandlers=!0}},addWheelEventHandlers:function(){!this.hasWheelHandlers&&e.Z.isBrowser&&(e.Z.addEvent(this.containerObj,"mousewheel",this.mouseWheelListener,this),e.Z.addEvent(this.containerObj,"DOMMouseScroll",this.mouseWheelListener,this),this.hasWheelHandlers=!0)},addFullscreenEventHandlers:function(){var t,i=["fullscreenchange","mozfullscreenchange","webkitfullscreenchange","msfullscreenchange"],s=i.length;if(!this.hasFullscreenEventHandlers&&e.Z.isBrowser){for(t=0;t<s;t++)e.Z.addEvent(this.document,i[t],this.fullscreenListener,this);this.hasFullscreenEventHandlers=!0}},addKeyboardEventHandlers:function(){this.attr.keyboard.enabled&&!this.hasKeyboardHandlers&&e.Z.isBrowser&&(e.Z.addEvent(this.containerObj,"keydown",this.keyDownListener,this),e.Z.addEvent(this.containerObj,"focusin",this.keyFocusInListener,this),e.Z.addEvent(this.containerObj,"focusout",this.keyFocusOutListener,this),this.hasKeyboardHandlers=!0)},removeKeyboardEventHandlers:function(){this.hasKeyboardHandlers&&e.Z.isBrowser&&(e.Z.removeEvent(this.containerObj,"keydown",this.keyDownListener,this),e.Z.removeEvent(this.containerObj,"focusin",this.keyFocusInListener,this),e.Z.removeEvent(this.containerObj,"focusout",this.keyFocusOutListener,this),this.hasKeyboardHandlers=!1)},removeFullscreenEventHandlers:function(){var t,i=["fullscreenchange","mozfullscreenchange","webkitfullscreenchange","msfullscreenchange"],s=i.length;if(this.hasFullscreenEventHandlers&&e.Z.isBrowser){for(t=0;t<s;t++)e.Z.removeEvent(this.document,i[t],this.fullscreenListener,this);this.hasFullscreenEventHandlers=!1}},removePointerEventHandlers:function(){if(this.hasPointerHandlers&&e.Z.isBrowser){var t=this.attr.movetarget||this.containerObj;window.navigator.msPointerEnabled?(e.Z.removeEvent(this.containerObj,"MSPointerDown",this.pointerDownListener,this),e.Z.removeEvent(t,"MSPointerMove",this.pointerMoveListener,this)):(e.Z.removeEvent(this.containerObj,"pointerdown",this.pointerDownListener,this),e.Z.removeEvent(t,"pointermove",this.pointerMoveListener,this),e.Z.removeEvent(t,"pointerleave",this.pointerLeaveListener,this),e.Z.removeEvent(t,"click",this.pointerClickListener,this),e.Z.removeEvent(t,"dblclick",this.pointerDblClickListener,this)),this.hasWheelHandlers&&(e.Z.removeEvent(this.containerObj,"mousewheel",this.mouseWheelListener,this),e.Z.removeEvent(this.containerObj,"DOMMouseScroll",this.mouseWheelListener,this)),this.hasPointerUp&&(window.navigator.msPointerEnabled?e.Z.removeEvent(this.document,"MSPointerUp",this.pointerUpListener,this):(e.Z.removeEvent(this.document,"pointerup",this.pointerUpListener,this),e.Z.removeEvent(this.document,"pointercancel",this.pointerUpListener,this)),this.hasPointerUp=!1),this.hasPointerHandlers=!1}},removeMouseEventHandlers:function(){if(this.hasMouseHandlers&&e.Z.isBrowser){var t=this.attr.movetarget||this.containerObj;e.Z.removeEvent(this.containerObj,"mousedown",this.mouseDownListener,this),e.Z.removeEvent(t,"mousemove",this.mouseMoveListener,this),e.Z.removeEvent(t,"click",this.mouseClickListener,this),e.Z.removeEvent(t,"dblclick",this.mouseDblClickListener,this),this.hasMouseUp&&(e.Z.removeEvent(this.document,"mouseup",this.mouseUpListener,this),this.hasMouseUp=!1),this.hasWheelHandlers&&(e.Z.removeEvent(this.containerObj,"mousewheel",this.mouseWheelListener,this),e.Z.removeEvent(this.containerObj,"DOMMouseScroll",this.mouseWheelListener,this)),this.hasMouseHandlers=!1}},removeTouchEventHandlers:function(){if(this.hasTouchHandlers&&e.Z.isBrowser){var t=this.attr.movetarget||this.containerObj;e.Z.removeEvent(this.containerObj,"touchstart",this.touchStartListener,this),e.Z.removeEvent(t,"touchmove",this.touchMoveListener,this),this.hasTouchEnd&&(e.Z.removeEvent(this.document,"touchend",this.touchEndListener,this),this.hasTouchEnd=!1),this.hasTouchHandlers=!1}},clickLeftArrow:function(){return this.moveOrigin(this.origin.scrCoords[1]+.1*this.canvasWidth,this.origin.scrCoords[2]),this},clickRightArrow:function(){return this.moveOrigin(this.origin.scrCoords[1]-.1*this.canvasWidth,this.origin.scrCoords[2]),this},clickUpArrow:function(){return this.moveOrigin(this.origin.scrCoords[1],this.origin.scrCoords[2]-.1*this.canvasHeight),this},clickDownArrow:function(){return this.moveOrigin(this.origin.scrCoords[1],this.origin.scrCoords[2]+.1*this.canvasHeight),this},gestureChangeListener:function(t){var e,r,o,n,a,l,c,d,p,f,_,m,g,b=!1,v=this.attr.zoom.factorx,Z=this.attr.zoom.factory,C=!1;return this.mode!==this.BOARD_MODE_ZOOM||(t.preventDefault(),l=h.Z.distance([t.touches[0].clientX,t.touches[0].clientY],[t.touches[1].clientX,t.touches[1].clientY],2),void 0===t.scale&&(t.scale=l/this.prevDist),!!s.Z.exists(this.prevCoords)&&(r=[t.touches[0].clientX-this.prevCoords[0][0],t.touches[0].clientY-this.prevCoords[0][1]],o=[t.touches[1].clientX-this.prevCoords[1][0],t.touches[1].clientY-this.prevCoords[1][1]],r[0]*r[0]+r[1]*r[1]<100&&o[0]*o[0]+o[1]*o[1]<100||(n=h.Z.rad(r,[0,0],o),"pan"!==this.isPreviousGesture&&Math.abs(n)>.2*Math.PI&&Math.abs(n)<1.8*Math.PI&&(b=!0),"pan"===this.isPreviousGesture||b||(Math.abs(t.scale)<.77||Math.abs(t.scale)>1.3)&&(b=!0),a=t.scale/this.prevScale,this.prevScale=t.scale,this.prevCoords=[[t.touches[0].clientX,t.touches[0].clientY],[t.touches[1].clientX,t.touches[1].clientY]],e=new u.Z(i.Z.COORDS_BY_SCREEN,this.getMousePosition(t,0),this),this.attr.pan.enabled&&this.attr.pan.needtwofingers&&!b?(this.isPreviousGesture="pan",this.moveOrigin(e.scrCoords[1],e.scrCoords[2],!0)):this.attr.zoom.enabled&&Math.abs(a-1)<.5&&(C=!1,p=this.attr.zoom.center,(this.attr.zoom.pinchhorizontal||this.attr.zoom.pinchvertical)&&(f=Math.abs(t.touches[0].clientX-t.touches[1].clientX),_=Math.abs(t.touches[0].clientY-t.touches[1].clientY),c=Math.abs(Math.atan2(_,f)),d=Math.PI*this.attr.zoom.pinchsensitivity/90),!this.keepaspectratio&&this.attr.zoom.pinchhorizontal&&c<d?(this.attr.zoom.factorx=a,this.attr.zoom.factory=1,m=0,g=0,C=!0):!this.keepaspectratio&&this.attr.zoom.pinchvertical&&Math.abs(c-.5*Math.PI)<d?(this.attr.zoom.factorx=1,this.attr.zoom.factory=a,m=0,g=0,C=!0):this.attr.zoom.pinch&&(this.attr.zoom.factorx=a,this.attr.zoom.factory=a,m=e.usrCoords[1],g=e.usrCoords[2],C=!0),C&&("board"===p?this.zoomIn():this.zoomIn(m,g),this.attr.zoom.factorx=v,this.attr.zoom.factory=Z))),!1))},gestureStartListener:function(t){var e;return t.preventDefault(),this.prevScale=1,this.prevDist=h.Z.distance([t.touches[0].clientX,t.touches[0].clientY],[t.touches[1].clientX,t.touches[1].clientY],2),this.prevCoords=[[t.touches[0].clientX,t.touches[0].clientY],[t.touches[1].clientX,t.touches[1].clientY]],this.isPreviousGesture="none",e=this.getMousePosition(t,0),this.initMoveOrigin(e[0],e[1]),this.mode=this.BOARD_MODE_ZOOM,!1},_isRequiredKeyPressed:function(t,e){var i=this.attr[e];return!!i.enabled&&!(!(i.needshift&&t.shiftKey||!i.needshift&&!t.shiftKey)||!(i.needctrl&&t.ctrlKey||!i.needctrl&&!t.ctrlKey))},_isPointerRegistered:function(t){var e,i=this._board_touches.length;for(e=0;e<i;e++)if(this._board_touches[e].pointerId===t.pointerId)return!0;return!1},_pointerStorePosition:function(t){var e,i;for(e=0,i=!1;e<this._board_touches.length;e++)if(this._board_touches[e].pointerId===t.pointerId){this._board_touches[e].clientX=t.clientX,this._board_touches[e].clientY=t.clientY,i=!0;break}return!i&&this._board_touches.length<2&&this._board_touches.push({pointerId:t.pointerId,clientX:t.clientX,clientY:t.clientY}),this},_pointerRemoveTouches:function(t){var e;for(e=0;e<this._board_touches.length;e++)if(this._board_touches[e].pointerId===t.pointerId){this._board_touches.splice(e,1);break}return this},_pointerClearTouches:function(t){this._board_touches.length>0&&this.dehighlightAll(),this.updateQuality=this.BOARD_QUALITY_HIGH,this.mode=this.BOARD_MODE_NONE,this._board_touches=[],this.touches=[]},_getPointerInputDevice:function(t){if(e.Z.isBrowser){if("touch"===t.pointerType||window.navigator.msMaxTouchPoints&&window.navigator.msMaxTouchPoints>1)return"touch";if("mouse"===t.pointerType)return"mouse";if("pen"===t.pointerType)return"pen"}return"mouse"},pointerDownListener:function(t,i,r){var o,n,a,h,l,c,d,u,p,f,_;if(!i&&this._isPointerRegistered(t))return!1;if(null===s.Z.evaluate(this.attr.movetarget)&&s.Z.exists(t.target)&&s.Z.exists(t.target.releasePointerCapture)&&t.target.releasePointerCapture(t.pointerId),!i&&t.isPrimary,this.hasPointerUp||(window.navigator.msPointerEnabled?e.Z.addEvent(this.document,"MSPointerUp",this.pointerUpListener,this):(e.Z.addEvent(this.document,"pointerup",this.pointerUpListener,this),e.Z.addEvent(this.document,"pointercancel",this.pointerUpListener,this)),this.hasPointerUp=!0),this.hasMouseHandlers&&this.removeMouseEventHandlers(),this.hasTouchHandlers&&this.removeTouchEventHandlers(),this.document.selection&&s.Z.isFunction(this.document.selection.empty))this.document.selection.empty();else if(window.getSelection&&(c=window.getSelection()).removeAllRanges)try{c.removeAllRanges()}catch(t){}if(this._inputDevice=this._getPointerInputDevice(t),u=this._inputDevice,this.options.precision.hasPoint=this.options.precision[u],h=this.getMousePosition(t),this._testForSelection(t),this.selectingMode)return this._startSelecting(h),void this.triggerEventHandlers(["touchstartselecting","pointerstartselecting","startselecting"],[t]);if(this.attr.drag.enabled&&i?(l=[i],this.mode=this.BOARD_MODE_DRAG):l=this.initMoveObject(h[0],h[1],t,u),d={num:t.pointerId,X:h[0],Y:h[1],Xprev:NaN,Yprev:NaN,Xstart:[],Ystart:[],Zstart:[]},l.length>0){for(f=l[l.length-1],p=!1,o=0;o<this.touches.length;o++)if(this.touches[o].obj===f){n=o,a=this.touches[o].targets.push(d)-1,p=!0;break}p||(a=0,n=this.touches.push({obj:f,targets:[d]})-1),this.dehighlightAll(),f.highlight(!0),this.saveStartPos(f,this.touches[n].targets[a]),t&&t.preventDefault&&!r&&t.preventDefault()}return this.touches.length>0&&!r&&(t.preventDefault(),t.stopPropagation()),!!e.Z.isBrowser&&("touch"!==this._getPointerInputDevice(t)?this.mode===this.BOARD_MODE_NONE&&this.mouseOriginMoveStart(t):(this._pointerStorePosition(t),t.touches=this._board_touches,1===t.touches.length&&this.mode===this.BOARD_MODE_NONE&&this.touchStartMoveOriginOneFinger(t)||2!==t.touches.length||this.mode!==this.BOARD_MODE_NONE&&this.mode!==this.BOARD_MODE_MOVE_ORIGIN||(this.mode===this.BOARD_MODE_MOVE_ORIGIN&&this.originMoveEnd(),this.gestureStartListener(t))),_="none",this.mode!==this.BOARD_MODE_NONE||!0!==s.Z.evaluate(this.attr.browserpan)&&!0!==s.Z.evaluate(this.attr.browserpan.enabled)||!1!==s.Z.evaluate(this.attr.pan.enabled)&&!0!==s.Z.evaluate(this.attr.pan.needtwofingers)||(_="auto"),this.containerObj.style.touchAction=_,this.triggerEventHandlers(["touchstart","down","pointerdown","MSPointerDown"],[t]),!0)},_handleClicks:function(t,e){var i,r,o=this;if(this.selectingMode)t.stopPropagation();else{if(r=s.Z.evaluate(this.attr.clickdelay),s.Z.evaluate(this.attr.dblclicksuppressclick))this._preventSingleClick=!1,this._singleClickTimer=setTimeout((function(){if(!o._preventSingleClick)for(i in o.triggerEventHandlers(e,[t]),o.clickObjects)o.clickObjects.hasOwnProperty(i)&&(o.clickObjects[i].triggerEventHandlers(e,[t]),delete o.clickObjects[i])}),r);else{for(i in o.triggerEventHandlers(e,[t]),o.clickObjects)o.clickObjects.hasOwnProperty(i)&&o.clickObjects[i].triggerEventHandlers(e,[t]);setTimeout((function(){for(i in o.clickObjects)o.clickObjects.hasOwnProperty(i)&&delete o.clickObjects[i]}),r)}t.stopPropagation()}},_handleDblClicks:function(t,e){var i;if(this.selectingMode)t.stopPropagation();else{for(i in this._preventSingleClick=!0,clearTimeout(this._singleClickTimer),this.triggerEventHandlers(e,[t]),this.clickObjects)this.clickObjects.hasOwnProperty(i)&&(this.clickObjects[i].triggerEventHandlers(e,[t]),delete this.clickObjects[i]);t.stopPropagation()}},pointerClickListener:function(t){this._handleClicks(t,["click","pointerclick"])},pointerDblClickListener:function(t){this._handleDblClicks(t,["dblclick","pointerdblclick"])},mouseClickListener:function(t){this._handleClicks(t,["click","mouseclick"])},mouseDblClickListener:function(t){this._handleDblClicks(t,["dblclick","mousedblclick"])},pointerMoveListener:function(t){var e,i,r,o,n,a;if("touch"===this._getPointerInputDevice(t)&&!this._isPointerRegistered(t))return this.BOARD_MODE_NONE;if(!this.checkFrameRate(t))return!1;if(this.mode!==this.BOARD_MODE_DRAG&&(this.dehighlightAll(),this.displayInfobox(!1)),this.mode!==this.BOARD_MODE_NONE&&(t.preventDefault(),t.stopPropagation()),this.updateQuality=this.BOARD_QUALITY_LOW,this._inputDevice=this._getPointerInputDevice(t),a=this._inputDevice,this.options.precision.hasPoint=this.options.precision[a],o=.3333*this.options.precision.hasPoint,r=this.getMousePosition(t),null===s.Z.evaluate(this.attr.movetarget)&&r[0]<=o||r[1]<=o||r[0]>=this.canvasWidth-o||r[1]>=this.canvasHeight-o)return this.mode===this.BOARD_MODE_NONE;if(this.selectingMode)this._moveSelecting(r),this.triggerEventHandlers(["touchmoveselecting","moveselecting","pointermoveselecting"],[t,this.mode]);else if(!this.mouseOriginMove(t))if(this.mode===this.BOARD_MODE_DRAG){for(e=0;e<this.touches.length;e++)for(n=this.touches[e].targets,i=0;i<n.length;i++)if(n[i].num===t.pointerId){n[i].X=r[0],n[i].Y=r[1],1===n.length?this.moveObject(r[0],r[1],this.touches[e],t,a):2===n.length&&(this.twoFingerMove(this.touches[e],t.pointerId,t),n[i].Xprev=r[0],n[i].Yprev=r[1]);break}}else"touch"===this._getPointerInputDevice(t)&&(this._pointerStorePosition(t),2===this._board_touches.length&&(t.touches=this._board_touches,this.gestureChangeListener(t))),this.highlightElements(r[0],r[1],t,-1);return this.triggerEventHandlers(["pointermove","MSPointerMove","move"],[t,this.mode]),this.updateQuality=this.BOARD_QUALITY_HIGH,this.mode===this.BOARD_MODE_NONE},pointerUpListener:function(t){var i,r,o,n,a,h=!1;if(this.triggerEventHandlers(["touchend","up","pointerup","MSPointerUp"],[t]),this.displayInfobox(!1),t)for(i=0;i<this.touches.length;i++)for(a=this.touches[i].targets,r=0;r<a.length;r++)if(a[r].num===t.pointerId){a.splice(r,1),0===a.length&&this.touches.splice(i,1);break}if(this.originMoveEnd(),this.update(),this.selectingMode)this._stopSelecting(t),this.triggerEventHandlers(["touchstopselecting","pointerstopselecting","stopselecting"],[t]),this.stopSelectionMode();else for(i=this.downObjects.length-1;i>-1;i--){for(o=!1,r=0;r<this.touches.length;r++)this.touches[r].obj.id===this.downObjects[i].id&&(o=!0);o||(this.downObjects[i].triggerEventHandlers(["touchend","up","pointerup","MSPointerUp"],[t]),s.Z.exists(this.downObjects[i].coords)||(this.downObjects[i].snapToGrid(),this.downObjects[i].snapToPoints(),h=!0),n=this.downObjects[i].eventHandlers,(s.Z.exists(n.click)&&n.click.length>0||s.Z.exists(n.pointerclick)&&n.pointerclick.length>0||s.Z.exists(n.dblclick)&&n.dblclick.length>0||s.Z.exists(n.pointerdblclick)&&n.pointerdblclick.length>0)&&(this.clickObjects[this.downObjects[i].id]=this.downObjects[i]),this.downObjects.splice(i,1))}return this.hasPointerUp&&(window.navigator.msPointerEnabled?e.Z.removeEvent(this.document,"MSPointerUp",this.pointerUpListener,this):(e.Z.removeEvent(this.document,"pointerup",this.pointerUpListener,this),e.Z.removeEvent(this.document,"pointercancel",this.pointerUpListener,this)),this.hasPointerUp=!1),this._pointerClearTouches(t.pointerId),"touch"!==this._getPointerInputDevice(t)&&this.dehighlightAll(),h&&this.update(),!0},pointerLeaveListener:function(t){return this.displayInfobox(!1),this.dehighlightAll(),!0},touchStartListener:function(r){var o,n,a,h,l,c,d,u,p,f,_=this.options.precision.touch,m=r.touches;for(this.hasTouchEnd||(e.Z.addEvent(this.document,"touchend",this.touchEndListener,this),this.hasTouchEnd=!0),this.document.selection&&s.Z.isFunction(this.document.selection.empty)?this.document.selection.empty():window.getSelection&&window.getSelection().removeAllRanges(),this._inputDevice="touch",this.options.precision.hasPoint=this.options.precision.touch,o=0;o<m.length;o++)m[o].jxg_isused=!1;for(o=0;o<this.touches.length;o++)for(f=this.touches[o].targets,n=0;n<f.length;n++){f[n].num=-1,_=this.options.precision.touch;do{for(a=0;a<m.length;a++)if(Math.abs(Math.pow(m[a].screenX-f[n].X,2)+Math.pow(m[a].screenY-f[n].Y,2))<_*_){f[n].num=a,f[n].X=m[a].screenX,f[n].Y=m[a].screenY,m[a].jxg_isused=!0;break}_*=2}while(-1===f[n].num&&_<this.options.precision.touchMax);-1===f[n].num&&(t.Z.debug("i couldn't find a targettouches for target no "+n+" on "+this.touches[o].obj.name+" ("+this.touches[o].obj.id+"). Removed the target."),t.Z.debug("eps = "+_+", touchMax = "+v.Z.precision.touchMax),f.splice(o,1))}for(o=0;o<m.length;o++)if(!m[o].jxg_isused){if(h=this.getMousePosition(r,o),this.selectingMode)return this._startSelecting(h),this.triggerEventHandlers(["touchstartselecting","startselecting"],[r]),r.preventDefault(),r.stopPropagation(),this.options.precision.hasPoint=this.options.precision.mouse,this.touches.length>0;if(0!==(l=this.initMoveObject(h[0],h[1],r,"touch")).length)if(c=l[l.length-1],p={num:o,X:m[o].screenX,Y:m[o].screenY,Xprev:NaN,Yprev:NaN,Xstart:[],Ystart:[],Zstart:[]},s.Z.isPoint(c)||c.elementClass===i.Z.OBJECT_CLASS_TEXT||c.type===i.Z.OBJECT_TYPE_TICKS||c.type===i.Z.OBJECT_TYPE_IMAGE)u=[p],this.saveStartPos(c,u[0]),this.touches.push({obj:c,targets:u}),c.highlight(!0);else if(c.elementClass===i.Z.OBJECT_CLASS_LINE||c.elementClass===i.Z.OBJECT_CLASS_CIRCLE||c.elementClass===i.Z.OBJECT_CLASS_CURVE||c.type===i.Z.OBJECT_TYPE_POLYGON){for(d=!1,n=0;n<this.touches.length;n++)c.id===this.touches[n].obj.id&&(d=!0,1===this.touches[n].targets.length&&(this.saveStartPos(c,p),this.touches[n].targets.push(p)),m[o].jxg_isused=!0);d||(u=[p],this.saveStartPos(c,u[0]),this.touches.push({obj:c,targets:u}),c.highlight(!0))}m[o].jxg_isused=!0}return this.touches.length>0&&(r.preventDefault(),r.stopPropagation()),1===m.length&&this.mode===this.BOARD_MODE_NONE&&this.touchStartMoveOriginOneFinger(r)||2!==m.length||this.mode!==this.BOARD_MODE_NONE&&this.mode!==this.BOARD_MODE_MOVE_ORIGIN||(this.mode===this.BOARD_MODE_MOVE_ORIGIN&&this.originMoveEnd(),this.gestureStartListener(r)),this.options.precision.hasPoint=this.options.precision.mouse,this.triggerEventHandlers(["touchstart","down"],[r]),!1},touchMoveListener:function(t){var e,i,s,r,o=t.touches;if(!this.checkFrameRate(t))return!1;if(this.mode!==this.BOARD_MODE_NONE&&(t.preventDefault(),t.stopPropagation()),this.mode!==this.BOARD_MODE_DRAG&&(this.dehighlightAll(),this.displayInfobox(!1)),this._inputDevice="touch",this.options.precision.hasPoint=this.options.precision.touch,this.updateQuality=this.BOARD_QUALITY_LOW,this.selectingMode){for(e=0;e<o.length;e++)if(!o[e].jxg_isused){i=this.getMousePosition(t,e),this._moveSelecting(i),this.triggerEventHandlers(["touchmoves","moveselecting"],[t,this.mode]);break}}else if(!this.touchOriginMove(t))if(this.mode===this.BOARD_MODE_DRAG){for(e=0;e<this.touches.length;e++)if(1===(r=this.touches[e].targets).length){if(o[r[0].num]){if((i=this.getMousePosition(t,r[0].num))[0]<0||i[0]>this.canvasWidth||i[1]<0||i[1]>this.canvasHeight)return;r[0].X=i[0],r[0].Y=i[1],this.moveObject(i[0],i[1],this.touches[e],t,"touch")}}else if(2===r.length&&r[0].num>-1&&r[1].num>-1&&o[r[0].num]&&o[r[1].num]){if(i=this.getMousePosition(t,r[0].num),s=this.getMousePosition(t,r[1].num),i[0]<0||i[0]>this.canvasWidth||i[1]<0||i[1]>this.canvasHeight||s[0]<0||s[0]>this.canvasWidth||s[1]<0||s[1]>this.canvasHeight)return;r[0].X=i[0],r[0].Y=i[1],r[1].X=s[0],r[1].Y=s[1],this.twoFingerMove(this.touches[e],r[0].num,t),r[0].Xprev=i[0],r[0].Yprev=i[1],r[1].Xprev=s[0],r[1].Yprev=s[1]}}else 2===o.length&&this.gestureChangeListener(t),i=this.getMousePosition(t,0),this.highlightElements(i[0],i[1],t,-1);return this.mode!==this.BOARD_MODE_DRAG&&this.displayInfobox(!1),this.triggerEventHandlers(["touchmove","move"],[t,this.mode]),this.options.precision.hasPoint=this.options.precision.mouse,this.updateQuality=this.BOARD_QUALITY_HIGH,this.mode===this.BOARD_MODE_NONE},touchEndListener:function(t){var r,o,n,a,h,l,c=this.options.precision.touch,d=[],u=t&&t.touches,p=!1;if(this.triggerEventHandlers(["touchend","up"],[t]),this.displayInfobox(!1),this.selectingMode)this._stopSelecting(t),this.triggerEventHandlers(["touchstopselecting","stopselecting"],[t]),this.stopSelectionMode();else if(u&&u.length>0){for(r=0;r<this.touches.length;r++)d[r]=this.touches[r];for(this.touches.length=0,r=0;r<u.length;r++)u[r].jxg_isused=!1;for(r=0;r<d.length;r++){for(a=!1,h=0,l=d[r].targets,o=0;o<l.length;o++)for(l[o].found=!1,n=0;n<u.length;n++)if(Math.abs(Math.pow(u[n].screenX-l[o].X,2)+Math.pow(u[n].screenY-l[o].Y,2))<c*c){l[o].found=!0,l[o].num=n,l[o].X=u[n].screenX,l[o].Y=u[n].screenY,h+=1;break}if(s.Z.isPoint(d[r].obj)?a=l[0]&&l[0].found:d[r].obj.elementClass===i.Z.OBJECT_CLASS_LINE?a=l[0]&&l[0].found||l[1]&&l[1].found:d[r].obj.elementClass===i.Z.OBJECT_CLASS_CIRCLE&&(a=1===h||3===h),a)for(this.touches.push({obj:d[r].obj,targets:[]}),o=0;o<l.length;o++)l[o].found&&this.touches[this.touches.length-1].targets.push({num:l[o].num,X:l[o].screenX,Y:l[o].screenY,Xprev:NaN,Yprev:NaN,Xstart:l[o].Xstart,Ystart:l[o].Ystart,Zstart:l[o].Zstart});else d[r].obj.noHighlight()}}else this.touches.length=0;for(r=this.downObjects.length-1;r>-1;r--){for(a=!1,o=0;o<this.touches.length;o++)this.touches[o].obj.id===this.downObjects[r].id&&(a=!0);a||(this.downObjects[r].triggerEventHandlers(["touchup","up"],[t]),s.Z.exists(this.downObjects[r].coords)||(this.downObjects[r].snapToGrid(),this.downObjects[r].snapToPoints(),p=!0),this.downObjects.splice(r,1))}return u&&0!==u.length||(this.hasTouchEnd&&(e.Z.removeEvent(this.document,"touchend",this.touchEndListener,this),this.hasTouchEnd=!1),this.dehighlightAll(),this.updateQuality=this.BOARD_QUALITY_HIGH,this.originMoveEnd(),p&&this.update()),!0},mouseDownListener:function(t){var i,r,o;if(this.document.selection&&s.Z.isFunction(this.document.selection.empty)?this.document.selection.empty():window.getSelection&&window.getSelection().removeAllRanges(),!this.hasMouseUp)return e.Z.addEvent(this.document,"mouseup",this.mouseUpListener,this),this.hasMouseUp=!0,this._inputDevice="mouse",this.options.precision.hasPoint=this.options.precision.mouse,i=this.getMousePosition(t),this._testForSelection(t),this.selectingMode?(this._startSelecting(i),void this.triggerEventHandlers(["mousestartselecting","startselecting"],[t])):(0===(r=this.initMoveObject(i[0],i[1],t,"mouse")).length?(this.mode=this.BOARD_MODE_NONE,o=!0):(this.mouse={obj:null,targets:[{X:i[0],Y:i[1],Xprev:NaN,Yprev:NaN}]},this.mouse.obj=r[r.length-1],this.dehighlightAll(),this.mouse.obj.highlight(!0),this.mouse.targets[0].Xstart=[],this.mouse.targets[0].Ystart=[],this.mouse.targets[0].Zstart=[],this.saveStartPos(this.mouse.obj,this.mouse.targets[0]),t&&t.preventDefault?t.preventDefault():window.event&&(window.event.returnValue=!1)),this.mode===this.BOARD_MODE_NONE&&(o=this.mouseOriginMoveStart(t)),this.triggerEventHandlers(["mousedown","down"],[t]),o)},mouseMoveListener:function(t){var e;if(!this.checkFrameRate(t))return!1;e=this.getMousePosition(t),this.updateQuality=this.BOARD_QUALITY_LOW,this.mode!==this.BOARD_MODE_DRAG&&(this.dehighlightAll(),this.displayInfobox(!1)),this.selectingMode?(this._moveSelecting(e),this.triggerEventHandlers(["mousemoveselecting","moveselecting"],[t,this.mode])):this.mouseOriginMove(t)||(this.mode===this.BOARD_MODE_DRAG?this.moveObject(e[0],e[1],this.mouse,t,"mouse"):this.highlightElements(e[0],e[1],t,-1),this.triggerEventHandlers(["mousemove","move"],[t,this.mode])),this.updateQuality=this.BOARD_QUALITY_HIGH},mouseUpListener:function(t){var i;if(!1===this.selectingMode&&this.triggerEventHandlers(["mouseup","up"],[t]),this.updateQuality=this.BOARD_QUALITY_HIGH,this.mouse&&this.mouse.obj&&(s.Z.exists(this.mouse.obj.coords)||(this.mouse.obj.snapToGrid(this.mouse.targets[0]),this.mouse.obj.snapToPoints())),this.originMoveEnd(),this.dehighlightAll(),this.update(),this.selectingMode)this._stopSelecting(t),this.triggerEventHandlers(["mousestopselecting","stopselecting"],[t]),this.stopSelectionMode();else for(i=0;i<this.downObjects.length;i++)this.downObjects[i].triggerEventHandlers(["mouseup","up"],[t]);this.downObjects.length=0,this.hasMouseUp&&(e.Z.removeEvent(this.document,"mouseup",this.mouseUpListener,this),this.hasMouseUp=!1),this.mouse=null},mouseWheelListener:function(t){var e,s;return!(this.attr.zoom.enabled&&this.attr.zoom.wheel&&this._isRequiredKeyPressed(t,"zoom"))||(e=(t=t||window.event).detail?-t.detail:t.wheelDelta/40,s="board"===this.attr.zoom.center?[]:new u.Z(i.Z.COORDS_BY_SCREEN,this.getMousePosition(t),this).usrCoords,e>0?this.zoomIn(s[1],s[2]):this.zoomOut(s[1],s[2]),this.triggerEventHandlers(["mousewheel"],[t]),t.preventDefault(),!1)},keyDownListener:function(e){var r,o,n,a,h,l,c=e.target.id,d=0,u=0,p=s.Z.evaluate(this.attr.keyboard.dx)/this.unitX,f=s.Z.evaluate(this.attr.keyboard.dy)/this.unitY,_=!1,m=!0;return!(!this.attr.keyboard.enabled||""===c)&&((!(a=this.containerObj.shadowRoot||document).activeElement||"INPUT"!==(o=a.activeElement).tagName&&"textarea"!==o.tagName)&&(r=c.replace(this.containerObj.id+"_",""),o=this.select(r),s.Z.exists(o.coords)&&(l=o.coords.usrCoords.slice(1)),s.Z.evaluate(this.attr.keyboard.panshift)&&e.shiftKey||s.Z.evaluate(this.attr.keyboard.panctrl)&&e.ctrlKey?(!0===s.Z.evaluate(this.attr.zoom.enabled)&&(_=!0),38===e.keyCode?this.clickUpArrow():40===e.keyCode?this.clickDownArrow():37===e.keyCode?this.clickLeftArrow():39===e.keyCode?this.clickRightArrow():_&&171===e.keyCode?this.zoomIn():_&&173===e.keyCode?this.zoomOut():_&&79===e.keyCode?this.zoom100():m=!1):e.shiftKey||e.ctrlKey||(s.Z.exists(o.visProp)&&(s.Z.exists(o.visProp.snaptogrid)&&o.visProp.snaptogrid&&o.evalVisProp("snapsizex")&&o.evalVisProp("snapsizey")?(p=d=(n=o.getSnapSizes())[0],f=u=n[1]):s.Z.exists(o.visProp.attracttogrid)&&o.visProp.attracttogrid&&o.evalVisProp("attractordistance")&&o.evalVisProp("attractorunit")&&(u=d=1.1*o.evalVisProp("attractordistance"),"screen"===o.evalVisProp("attractorunit")&&(d/=this.unitX,u/=this.unitX),p=Math.max(d,p),f=Math.max(u,f))),38===e.keyCode?h=[0,f]:40===e.keyCode?h=[0,-f]:37===e.keyCode?h=[-p,0]:39===e.keyCode?h=[p,0]:m=!1,h&&o.isDraggable&&o.visPropCalc.visible&&(this.geonextCompatibilityMode&&(s.Z.isPoint(o)||o.elementClass===i.Z.OBJECT_CLASS_TEXT)||!this.geonextCompatibilityMode)&&!o.evalVisProp("fixed")&&(this.mode=this.BOARD_MODE_DRAG,s.Z.exists(o.coords)&&(h[0]+=l[0],h[1]+=l[1]),s.Z.exists(o.coords)?(o.setPosition(t.Z.COORDS_BY_USER,h),this.updateInfobox(o)):(this.displayInfobox(!1),o.setPositionDirectly(i.Z.COORDS_BY_USER,h,[0,0])),this.triggerEventHandlers(["keymove","move"],[e,this.mode]),o.triggerEventHandlers(["keydrag","drag"],[e]),this.mode=this.BOARD_MODE_NONE)),this.update(),m&&s.Z.exists(e.preventDefault)&&e.preventDefault(),m))},keyFocusInListener:function(t){var e,i,r=t.target.id;if(!this.attr.keyboard.enabled||""===r)return!1;e=r.replace(this.containerObj.id+"_",""),i=this.select(e),s.Z.exists(i.highlight)&&(i.highlight(!0),this.focusObjects=[e],i.triggerEventHandlers(["hit"],[t])),s.Z.exists(i.coords)&&this.updateInfobox(i)},keyFocusOutListener:function(t){if(!this.attr.keyboard.enabled)return!1;this.focusObjects=[],this.dehighlightAll(),this.displayInfobox(!1)},updateContainerDims:function(t,e){var i,r,o,n=t,a=e;return void 0===t&&(n=this.containerObj.offsetWidth,a=this.containerObj.offsetHeight),void 0===t&&window&&window.getComputedStyle&&(i=window.getComputedStyle(this.containerObj,null),r=parseFloat(i.getPropertyValue("border-left-width"))+parseFloat(i.getPropertyValue("border-right-width")),isNaN(r)||(n-=r),o=parseFloat(i.getPropertyValue("border-top-width"))+parseFloat(i.getPropertyValue("border-bottom-width")),isNaN(o)||(a-=o)),n<=0||a<=0||isNaN(n)||isNaN(a)?this:(isNaN(this.getBoundingBox()[0])&&this.setBoundingBox(this.attr.boundingbox,this.keepaspectratio,"keep"),s.Z.exists(this._prevDim)&&this._prevDim.w===n&&this._prevDim.h===a||(this.resizeContainer(n,a,!0),this._prevDim={w:n,h:a}),this)},startResizeObserver:function(){var i=this;e.Z.isBrowser&&this.attr.resize&&this.attr.resize.enabled&&(this.resizeObserver=new ResizeObserver((function(e){var s;i._isResizing||(i._isResizing=!0,s=e[0].contentRect,window.setTimeout((function(){try{i.updateContainerDims(s.width,s.height)}catch(e){t.Z.debug(e),i.stopResizeObserver()}finally{i._isResizing=!1}}),i.attr.resize.throttle))})),this.resizeObserver.observe(this.containerObj))},stopResizeObserver:function(){e.Z.isBrowser&&this.attr.resize&&this.attr.resize.enabled&&s.Z.exists(this.resizeObserver)&&this.resizeObserver.unobserve(this.containerObj)},resizeListener:function(){var t=this;e.Z.isBrowser&&this.attr.resize&&this.attr.resize.enabled&&(this._isScrolling||this._isResizing||(this._isResizing=!0,window.setTimeout((function(){t.updateContainerDims(),t._isResizing=!1}),this.attr.resize.throttle)))},scrollListener:function(t){var i=this;e.Z.isBrowser&&(this._isScrolling||(this._isScrolling=!0,window.setTimeout((function(){i._isScrolling=!1}),66)))},startIntersectionObserver:function(){var e=this;try{this.intersectionObserver=new IntersectionObserver((function(t){isNaN(e.getBoundingBox()[0])&&e.updateContainerDims()}),{root:null,rootMargin:"0px",threshold:.8}),this.intersectionObserver.observe(e.containerObj)}catch(e){t.Z.debug("JSXGraph: IntersectionObserver not available in this browser.")}},stopIntersectionObserver:function(){s.Z.exists(this.intersectionObserver)&&this.intersectionObserver.unobserve(this.containerObj)},printListener:function(t){this.updateContainerDims()},printListenerMatch:function(t){t.matches&&this.printListener()},initInfobox:function(t){var e=s.Z.copyAttributes(t,this.options,"infobox");return e.id=this.id+"_infobox",this.infobox=this.create("text",[0,0,"0,0"],e),this.infobox.dump=!1,this.displayInfobox(!1),this},updateInfobox:function(t){var e,i,r,o,n,a,h,l=t.evalVisProp("showinfobox");return!s.Z.evaluate(this.attr.showinfobox)&&"inherit"===l||!l||s.Z.isPoint(t)&&(r=t.coords.usrCoords[1],o=t.coords.usrCoords[2],a=this.infobox.evalVisProp("distancex"),h=this.infobox.evalVisProp("distancey"),this.infobox.setCoords(r+a/this.unitX,o+h/this.unitY),n=t.evalVisProp("infoboxdigits"),"string"!=typeof t.infoboxText?("auto"===n?this.infobox.useLocale()?(e=this.infobox.formatNumberLocale(r),i=this.infobox.formatNumberLocale(o)):(e=s.Z.autoDigits(r),i=s.Z.autoDigits(o)):s.Z.isNumber(n)?this.infobox.useLocale()?(e=this.infobox.formatNumberLocale(r,n),i=this.infobox.formatNumberLocale(o,n)):(e=s.Z.toFixed(r,n),i=s.Z.toFixed(o,n)):(e=r,i=o),this.highlightInfobox(e,i,t)):this.highlightCustomInfobox(t.infoboxText,t),this.displayInfobox(!0)),this},displayInfobox:function(t){return!t&&this.focusObjects.length>0&&this.select(this.focusObjects[0]).elementClass===i.Z.OBJECT_CLASS_POINT||this.infobox.hiddenByParent===t&&(this.infobox.hiddenByParent=!t,this.infobox.prepareUpdate().updateVisibility(t).updateRenderer()),this},showInfobox:function(t){return this.displayInfobox(t)},highlightInfobox:function(t,e,i){return this.highlightCustomInfobox("("+t+", "+e+")",i),this},highlightCustomInfobox:function(t,e){return this.infobox.setText(t),this},dehighlightAll:function(){var t,e,i={},s=!1;for(t in this.highlightedObjects)this.highlightedObjects.hasOwnProperty(t)&&(e=this.highlightedObjects[t],this.focusObjects.indexOf(t)<0?((this.hasMouseHandlers||this.hasPointerHandlers)&&e.noHighlight(),s=!0):i[t]=e);return this.highlightedObjects=i,"canvas"===this.renderer.type&&s&&(this.prepareUpdate(),this.renderer.suspendRedraw(this),this.updateRenderer(),this.renderer.unsuspendRedraw()),this},getScrCoordsOfMouse:function(t,e){return[t,e]},getUsrCoordsOfMouse:function(t){var s=this.getCoordsTopLeftCorner(),r=e.Z.getPosition(t,null,this.document),o=r[0]-s[0],n=r[1]-s[1];return new u.Z(i.Z.COORDS_BY_SCREEN,[o,n],this).usrCoords.slice(1)},getAllUnderMouse:function(t){var e=this.getAllObjectsUnderMouse(t);return e.push(this.getUsrCoordsOfMouse(t)),e},getAllObjectsUnderMouse:function(t){var i,s,r=this.getCoordsTopLeftCorner(),o=e.Z.getPosition(t,null,this.document),n=o[0]-r[0],a=o[1]-r[1],h=[],l=this.objectsList.length;for(i=0;i<l;i++)(s=this.objectsList[i]).visPropCalc.visible&&s.hasPoint&&s.hasPoint(n,a)&&(h[h.length]=s);return h},updateCoords:function(){var t,e,i,r,o=this.objectsList.length;for(e=0;e<o;e++)if(t=this.objectsList[e],s.Z.exists(t.coords)){if("inherit"===(i=t.evalVisProp("frozen")))for(r in t.descendants)if(t.descendants.hasOwnProperty(r)&&!0===t.descendants[r].evalVisProp("frozen")){i=!0;break}!0===i?t.is3D?t.element2D.coords.screen2usr():t.coords.screen2usr():t.is3D?t.element2D.coords.usr2screen():(t.coords.usr2screen(),s.Z.exists(t.actualCoords)&&t.actualCoords.usr2screen())}return this},moveOrigin:function(t,e,r){var n,a,h,l;return s.Z.exists(t)&&s.Z.exists(e)&&(n=this.origin.scrCoords[1],a=this.origin.scrCoords[2],this.origin.scrCoords[1]=t,this.origin.scrCoords[2]=e,r&&(this.origin.scrCoords[1]-=this.drag_dx,this.origin.scrCoords[2]-=this.drag_dy),h=new u.Z(i.Z.COORDS_BY_SCREEN,[0,0],this).usrCoords,l=new u.Z(i.Z.COORDS_BY_SCREEN,[this.canvasWidth,this.canvasHeight],this).usrCoords,(h[1]<this.maxboundingbox[0]-o.Z.eps||h[2]>this.maxboundingbox[1]+o.Z.eps||l[1]>this.maxboundingbox[2]+o.Z.eps||l[2]<this.maxboundingbox[3]-o.Z.eps)&&(this.origin.scrCoords[1]=n,this.origin.scrCoords[2]=a)),this.updateCoords().clearTraces().fullUpdate(),this.triggerEventHandlers(["boundingbox"]),this},addConditions:function(e){var r,o,n,a,h,l,c,d=[],u=e.indexOf("<data>"),p=e.indexOf("</data>"),f=function(t,e,s,r){return function(){var o,n;n=(o=t.select(e.id)).coords.usrCoords[r],2===r?o.setPositionDirectly(i.Z.COORDS_BY_USER,[s(),n]):o.setPositionDirectly(i.Z.COORDS_BY_USER,[n,s()]),o.prepareUpdate().update()}},_=function(t,e,i){return function(){var s,r;s=t.select(e.id),r=i(),s.setAttribute({visible:r})}},m=function(t,e,i,s){return function(){var r,o;r=t.select(e.id),o=i(),"strokewidth"===s?r.visProp.strokewidth=o:(o=S.Z.rgba2rgbo(o),r.visProp[s+"color"]=o[0],r.visProp[s+"opacity"]=o[1])}},g=function(t,e,i){return function(){t.select(e.id).position=i()}},b=function(t,e,i){return function(){t.select(e.id).setStyle(i())}};if(!(u<0)){for(;u>=0;){if(o=(r=e.slice(u+6,p)).indexOf("="),n=r.slice(0,o),a=r.slice(o+1),o=n.indexOf("."),h=n.slice(0,o),l=this.elementsByName[s.Z.unescapeHTML(h)],c=n.slice(o+1).replace(/\s+/g,"").toLowerCase(),a=s.Z.createFunction(a,this,"",!0),s.Z.exists(this.elementsByName[h]))switch(c){case"x":d.push(f(this,l,a,2));break;case"y":d.push(f(this,l,a,1));break;case"visible":d.push(_(this,l,a));break;case"position":d.push(g(this,l,a));break;case"stroke":d.push(m(this,l,a,"stroke"));break;case"style":d.push(b(this,l,a));break;case"strokewidth":d.push(m(this,l,a,"strokewidth"));break;case"fill":d.push(m(this,l,a,"fill"));break;case"label":break;default:t.Z.debug('property "'+c+'" in conditions not yet implemented:'+a)}else t.Z.debug("debug conditions: |"+h+"| undefined");u=(e=e.slice(p+7)).indexOf("<data>"),p=e.indexOf("</data>")}this.updateConditions=function(){var t;for(t=0;t<d.length;t++)d[t]();return this.prepareUpdate().updateElements(),!0},this.updateConditions()}},updateConditions:function(){return!1},calculateSnapSizes:function(){var t,e,r,o,n=this.getBoundingBox(),a=s.Z.evaluate(this.options.grid.majorStep),h=s.Z.evaluate(this.options.grid.gridX),l=s.Z.evaluate(this.options.grid.gridY);for(s.Z.isArray(a)||(a=[a,a]),a.length<2&&(a=[a[0],a[0]]),s.Z.exists(h)&&(a[0]=h),s.Z.exists(l)&&(a[1]=l),"auto"===a[0]?a[0]=1:a[0]=s.Z.parseNumber(a[0],Math.abs(n[1]-n[3]),1/this.unitX),"auto"===a[1]?a[1]=1:a[1]=s.Z.parseNumber(a[1],Math.abs(n[0]-n[2]),1/this.unitY),t=new u.Z(i.Z.COORDS_BY_USER,[0,0],this),e=new u.Z(i.Z.COORDS_BY_USER,[a[0],a[1]],this),r=t.scrCoords[1]-e.scrCoords[1],o=t.scrCoords[2]-e.scrCoords[2],this.options.grid.snapSizeX=a[0];Math.abs(r)>25;)this.options.grid.snapSizeX*=2,r/=2;for(this.options.grid.snapSizeY=a[1];Math.abs(o)>25;)this.options.grid.snapSizeY*=2,o/=2;return this},applyZoom:function(){return this.updateCoords().calculateSnapSizes().clearTraces().fullUpdate(),this},zoomIn:function(t,e){var i=this.getBoundingBox(),r=s.Z.evaluate(this.attr.zoom.factorx),o=s.Z.evaluate(this.attr.zoom.factory),n=(i[2]-i[0])*(1-1/r),a=(i[1]-i[3])*(1-1/o),h=.5,l=.5,c=s.Z.evaluate(this.attr.zoom.max),d=s.Z.evaluate(this.attr.zoom.eps)||s.Z.evaluate(this.attr.zoom.min)||.001;return this.zoomX>c&&r>1||this.zoomY>c&&o>1||this.zoomX<d&&r<1||this.zoomY<d&&o<1?this:(s.Z.isNumber(t)&&s.Z.isNumber(e)&&(h=(t-i[0])/(i[2]-i[0]),l=(i[1]-e)/(i[1]-i[3])),this.setBoundingBox([i[0]+n*h,i[1]-a*l,i[2]-n*(1-h),i[3]+a*(1-l)],this.keepaspectratio,"update"),this.applyZoom())},zoomOut:function(t,e){var i=this.getBoundingBox(),r=s.Z.evaluate(this.attr.zoom.factorx),o=s.Z.evaluate(this.attr.zoom.factory),n=(i[2]-i[0])*(1-r),a=(i[1]-i[3])*(1-o),h=.5,l=.5,c=s.Z.evaluate(this.attr.zoom.eps)||s.Z.evaluate(this.attr.zoom.min)||.001;return this.zoomX<c||this.zoomY<c?this:(s.Z.isNumber(t)&&s.Z.isNumber(e)&&(h=(t-i[0])/(i[2]-i[0]),l=(i[1]-e)/(i[1]-i[3])),this.setBoundingBox([i[0]+n*h,i[1]-a*l,i[2]-n*(1-h),i[3]+a*(1-l)],this.keepaspectratio,"update"),this.applyZoom())},zoom100:function(){var t,e,i;return s.Z.exists(this.attr.boundingbox)?this.setBoundingBox(this.attr.boundingbox,this.keepaspectratio,"reset"):(e=((t=this.getBoundingBox())[2]-t[0])*(1-this.zoomX)*.5,i=(t[1]-t[3])*(1-this.zoomY)*.5,this.setBoundingBox([t[0]+e,t[1]-i,t[2]-e,t[3]+i],this.keepaspectratio,"reset")),this.applyZoom()},zoomAllPoints:function(){var t,e,i,r,o=0,n=0,a=0,h=0,l=this.objectsList.length;for(t=0;t<l;t++)r=this.objectsList[t],s.Z.isPoint(r)&&r.visPropCalc.visible&&(r.coords.usrCoords[1]<o?o=r.coords.usrCoords[1]:r.coords.usrCoords[1]>n&&(n=r.coords.usrCoords[1]),r.coords.usrCoords[2]>h?h=r.coords.usrCoords[2]:r.coords.usrCoords[2]<a&&(a=r.coords.usrCoords[2]));return 50,e=50/this.unitX,i=50/this.unitY,this.setBoundingBox([o-e,h+i,n+e,a-i],this.keepaspectratio,"update"),this.applyZoom()},zoomElements:function(t){var e,i,r,o,n,a,h,l=[1/0,-1/0,-1/0,1/0];if(!s.Z.isArray(t)||0===t.length)return this;for(e=0;e<t.length;e++)i=this.select(t[e]).bounds(),s.Z.isArray(i)&&(i[0]<l[0]&&(l[0]=i[0]),i[1]>l[1]&&(l[1]=i[1]),i[2]>l[2]&&(l[2]=i[2]),i[3]<l[3]&&(l[3]=i[3]));return s.Z.isArray(l)&&(r=.5*(l[0]+l[2]),o=.5*(l[1]+l[3]),n=1.5*(l[2]-l[0])*.5,a=1.5*(l[1]-l[3])*.5,h=Math.max(n,a),this.setBoundingBox([r-h,o+h,r+h,o-h],this.keepaspectratio,"update")),this},setZoom:function(t,e){var i=this.attr.zoom.factorx,s=this.attr.zoom.factory;return this.attr.zoom.factorx=t/this.zoomX,this.attr.zoom.factory=e/this.zoomY,this.zoomIn(),this.attr.zoom.factorx=i,this.attr.zoom.factory=s,this},_removeObj:function(e,r){var o,n;if(s.Z.isArray(e)){for(n=0;n<e.length;n++)this._removeObj(e[n],r);return this}if(e=this.select(e),!s.Z.exists(e)||s.Z.isString(e))return this;try{for(o in e.childElements)e.childElements.hasOwnProperty(o)&&e.childElements[o].board._removeObj(e.childElements[o]);for(o in e.objects)e.objects.hasOwnProperty(o)&&e.objects[o].board._removeObj(e.objects[o]);if(r)for(o in this.objects)this.objects.hasOwnProperty(o)&&s.Z.exists(this.objects[o].childElements)&&s.Z.exists(this.objects[o].childElements.hasOwnProperty(e.id))&&(delete this.objects[o].childElements[e.id],delete this.objects[o].descendants[e.id]);else if(s.Z.exists(e.ancestors))for(o in e.ancestors)e.ancestors.hasOwnProperty(o)&&s.Z.exists(e.ancestors[o].childElements)&&s.Z.exists(e.ancestors[o].childElements.hasOwnProperty(e.id))&&(delete e.ancestors[o].childElements[e.id],delete e.ancestors[o].descendants[e.id]);if(e._pos>-1)for(this.objectsList.splice(e._pos,1),n=e._pos;n<this.objectsList.length;n++)this.objectsList[n]._pos--;else e.type!==i.Z.OBJECT_TYPE_TURTLE&&t.Z.debug("Board.removeObject: object "+e.id+" not found in list.");delete this.objects[e.id],delete this.elementsByName[e.name],e.visProp&&e.evalVisProp("trace")&&e.clearTrace(),s.Z.exists(e.remove)&&e.remove()}catch(i){t.Z.debug(e.id+": Could not be removed: "+i)}return this},removeObject:function(t,e){var i;if(this.renderer.suspendRedraw(this),s.Z.isArray(t))for(i=0;i<t.length;i++)this._removeObj(t[i],e);else this._removeObj(t,e);return this.renderer.unsuspendRedraw(),this.update(),this},removeAncestors:function(t){var e;for(e in t.ancestors)t.ancestors.hasOwnProperty(e)&&this.removeAncestors(t.ancestors[e]);return this.removeObject(t),this},initGeonextBoard:function(){var t,e,i;return t=this.create("point",[0,0],{id:this.id+"g00e0",name:"Ursprung",withLabel:!1,visible:!1,fixed:!0}),e=this.create("point",[1,0],{id:this.id+"gX0e0",name:"Punkt_1_0",withLabel:!1,visible:!1,fixed:!0}),i=this.create("point",[0,1],{id:this.id+"gY0e0",name:"Punkt_0_1",withLabel:!1,visible:!1,fixed:!0}),this.create("line",[t,e],{id:this.id+"gXLe0",name:"X-Achse",withLabel:!1,visible:!1}),this.create("line",[t,i],{id:this.id+"gYLe0",name:"Y-Achse",withLabel:!1,visible:!1}),this},resizeContainer:function(t,e,i,s){var r,o,n,a,h;return o=this.canvasWidth,n=this.canvasHeight,s||(r=this.getBoundingBox()),this.canvasWidth=parseFloat(t),this.canvasHeight=parseFloat(e),i||(this.containerObj.style.width=this.canvasWidth+"px",this.containerObj.style.height=this.canvasHeight+"px"),this.renderer.resize(this.canvasWidth,this.canvasHeight),s?(a=.5*(this.canvasWidth-o),h=.5*(this.canvasHeight-n),this.moveOrigin(this.origin.scrCoords[1]+a,this.origin.scrCoords[2]+h)):this.setBoundingBox(r,this.keepaspectratio,"keep"),this},showDependencies:function(){var t,e,i,s,r;for(t in e="<p>\n",this.objects)if(this.objects.hasOwnProperty(t)){for(i in r=0,this.objects[t].childElements)this.objects[t].childElements.hasOwnProperty(i)&&(r+=1);for(i in r>=0&&(e+="<strong>"+this.objects[t].id+":</strong> "),this.objects[t].childElements)this.objects[t].childElements.hasOwnProperty(i)&&(e+=this.objects[t].childElements[i].id+"("+this.objects[t].childElements[i].name+"), ");e+="<p>\n"}return e+="</p>\n",(s=window.open()).document.open(),s.document.write(e),s.document.close(),this},showXML:function(){var t=window.open("");return t.document.open(),t.document.write("<pre>"+s.Z.escapeHTML(this.xmlString)+"</pre>"),t.document.close(),this},prepareUpdate:function(t){var e,r,o,n=this.objectsList.length;for(e=0;e<n;e++)if(o=this.objectsList[e],this._change3DView||s.Z.exists(t)&&"view3d_slider"===t.elType){if(o.needsUpdate=o.visProp.element3d||"view3d"===o.elType||"view3d_slider"===o.elType||this.needsFullUpdate,o.visProp.element3d&&o.visProp.element3d.type===i.Z.OBJECT_TYPE_SPHERE3D)for(r=0;r<o.parents.length;r++)this.objects[o.parents[r]].needsUpdate=!0}else o.needsUpdate=o.needsRegularUpdate||this.needsFullUpdate;for(e in this.groups)this.groups.hasOwnProperty(e)&&((o=this.groups[e]).needsUpdate=o.needsRegularUpdate||this.needsFullUpdate);return this},updateElements:function(t){var e,r;for(t=this.select(t),e=0;e<this.objectsList.length;e++)r=this.objectsList[e],this.needsFullUpdate&&r.elementClass===i.Z.OBJECT_CLASS_TEXT&&r.updateSize(),r.update(!s.Z.exists(t)||r.id!==t.id).updateVisibility();for(e in this.groups)this.groups.hasOwnProperty(e)&&this.groups[e].update(t);return this},updateRenderer:function(){var t,e,i,s=this.objectsList.length,r=[];if(this.renderer){if("canvas"===this.renderer.type)this.updateRendererCanvas();else{for(t=0;t<s;t++)this.objectsList[t].visProp.islabel&&this.objectsList[t].visProp.autoposition?r.push(t):this.objectsList[t].updateRenderer();for(e=r.length;0!==e;)i=Math.floor(Math.random()*e),e--,[r[e],r[i]]=[r[i],r[e]];for(t=0;t<r.length;t++)this.objectsList[r[t]].updateRenderer()}return this}},updateRendererCanvas:function(){var t,e,i,s=this.objectsList.length,r=[];for(t=0;t<s;t++)if("view3d"===(e=this.objectsList[t]).elType&&e.evalVisProp("depthorder.enabled")){r=e.evalVisProp("depthorder.layers"),e.updateRenderer();break}for(s=(i=this.objectsList.toSorted((function(t,e){return t.visProp.layer!==e.visProp.layer?t.visProp.layer-e.visProp.layer:-1===r.indexOf(t.visProp.layer)?t._pos-e._pos:t.visProp.element3d||e.visProp.element3d?t.visProp.element3d&&!e.visProp.element3d?-1:e.visProp.element3d&&!t.visProp.element3d?1:t.visProp.element3d.zIndex-e.visProp.element3d.zIndex:t._pos-e._pos}))).length,t=0;t<s;t++)i[t].prepareUpdate().updateRenderer();return this},addHook:function(e,i,r){return t.Z.deprecated("Board.addHook()","Board.on()"),i=s.Z.def(i,"update"),r=s.Z.def(r,this),this.hooks.push([i,e]),this.on(i,e,r),this.hooks.length-1},addEvent:t.Z.shortcut(t.Z.Board.prototype,"on"),removeHook:function(e){return t.Z.deprecated("Board.removeHook()","Board.off()"),this.hooks[e]&&(this.off(this.hooks[e][0],this.hooks[e][1]),this.hooks[e]=null),this},removeEvent:t.Z.shortcut(t.Z.Board.prototype,"off"),updateHooks:function(e){var i=Array.prototype.slice.call(arguments,0);return t.Z.deprecated("Board.updateHooks()","Board.triggerEventHandlers()"),i[0]=s.Z.def(i[0],"update"),this.triggerEventHandlers([i[0]],arguments),this},addChild:function(t){return s.Z.exists(t)&&s.Z.exists(t.containerObj)&&(this.dependentBoards.push(t),this.update()),this},removeChild:function(t){var e;for(e=this.dependentBoards.length-1;e>=0;e--)this.dependentBoards[e]===t&&this.dependentBoards.splice(e,1);return this},update:function(t){var e,i,r,o,n;if(this.inUpdate||this.isSuspendedUpdate)return this;for(this.inUpdate=!0,"all"===this.attr.minimizereflow&&this.containerObj&&"vml"!==this.renderer.type&&(n=this.document.activeElement,o=this.renderer.removeToInsertLater(this.containerObj)),"svg"===this.attr.minimizereflow&&"svg"===this.renderer.type&&(n=this.document.activeElement,o=this.renderer.removeToInsertLater(this.renderer.svgRoot)),this.prepareUpdate(t).updateElements(t).updateConditions(),this.renderer.suspendRedraw(this),this.updateRenderer(),this.renderer.unsuspendRedraw(),this.triggerEventHandlers(["update"],[]),o&&(o(),n.focus()),i=this.dependentBoards.length,e=0;e<i;e++)r=this.dependentBoards[e],s.Z.exists(r)&&r!==this&&(r.updateQuality=this.updateQuality,r.prepareUpdate().updateElements().updateConditions(),r.renderer.suspendRedraw(this),r.updateRenderer(),r.renderer.unsuspendRedraw(),r.triggerEventHandlers(["update"],[]));return this.inUpdate=!1,this},fullUpdate:function(){return this.needsFullUpdate=!0,this.update(),this.needsFullUpdate=!1,this},addGrid:function(){return this.create("grid",[]),this},removeGrids:function(){var t;for(t=0;t<this.grids.length;t++)this.removeObject(this.grids[t]);return this.grids.length=0,this.update(),this},create:function(e,i,r){var o,n;for(e=e.toLowerCase(),s.Z.exists(i)||(i=[]),s.Z.exists(r)||(r={}),n=0;n<i.length;n++)!s.Z.isString(i[n])||"text"===e&&2===n||"solidofrevolution3d"===e&&2===n||"text3d"===e&&(2===n||4===n)||!("input"!==e&&"checkbox"!==e&&"button"!==e||2!==n&&3!==n)||"curve"===e||"functiongraph"===e||"implicitcurve"===e||(n>0&&"view3d"===i[0].elType?i[n]=i[0].select(i[n]):i[n]=this.select(i[n]));if(!s.Z.isFunction(t.Z.elements[e]))throw new Error("JSXGraph: create: Unknown element type given: "+e);return o=t.Z.elements[e](this,i,r),s.Z.exists(o)?(o.prepareUpdate&&o.update&&o.updateRenderer&&o.fullUpdate(),o):(t.Z.debug("JSXGraph: create: failure creating "+e),o)},createElement:function(){return t.Z.deprecated("Board.createElement()","Board.create()"),this.create.apply(this,arguments)},clearTraces:function(){var t;for(t=0;t<this.objectsList.length;t++)this.objectsList[t].clearTrace();return this.numTraces=0,this},suspendUpdate:function(){return this.inUpdate||(this.isSuspendedUpdate=!0),this},unsuspendUpdate:function(){return this.isSuspendedUpdate&&(this.isSuspendedUpdate=!1,this.fullUpdate()),this},setBoundingBox:function(t,i,r){var n,a,h,l,c,d,u,p,f,_=0,m=0,g=1,b=e.Z.getDimensions(this.containerObj,this.document);return s.Z.isArray(t)?(t[0]<this.maxboundingbox[0]-o.Z.eps||t[1]>this.maxboundingbox[1]+o.Z.eps||t[2]>this.maxboundingbox[2]+o.Z.eps||t[3]<this.maxboundingbox[3]-o.Z.eps||(s.Z.exists(r)||(r="reset"),h=this.unitX,l=this.unitY,this.canvasWidth=parseFloat(b.width),this.canvasHeight=parseFloat(b.height),a=this.canvasWidth,n=this.canvasHeight,i?(this.keepaspectratio?(c=h/l,isNaN(c)&&(c=1)):c=1,"keep"===r&&(g=this.zoomX/this.zoomY),p=h*(d=t[2]-t[0]),f=l*(u=t[1]-t[3]),a>=n?p>=f?(this.unitY=n/u,this.unitX=this.unitY*c):(this.unitY=n/Math.abs(d)*o.Z.sign(u)/g,this.unitX=this.unitY*c):f>p?(this.unitX=a/d,this.unitY=this.unitX/c):(this.unitX=a/Math.abs(u)*o.Z.sign(d)*g,this.unitY=this.unitX/c),_=.5*(a/this.unitX-d),m=.5*(n/this.unitY-u),this.keepaspectratio=!0):(this.unitX=a/(t[2]-t[0]),this.unitY=n/(t[1]-t[3]),this.keepaspectratio=!1),this.moveOrigin(-this.unitX*(t[0]-_),this.unitY*(t[1]+m)),"update"===r?(this.zoomX*=this.unitX/h,this.zoomY*=this.unitY/l):"reset"===r&&(this.zoomX=s.Z.exists(this.attr.zoomx)?this.attr.zoomx:1,this.zoomY=s.Z.exists(this.attr.zoomy)?this.attr.zoomy:1)),this):this},getBoundingBox:function(){var t=new u.Z(i.Z.COORDS_BY_SCREEN,[0,0],this).usrCoords,e=new u.Z(i.Z.COORDS_BY_SCREEN,[this.canvasWidth,this.canvasHeight],this).usrCoords;return[t[1],t[2],e[1],e[2]]},_set:function(t,e){t=t.toLocaleLowerCase(),null===e||!s.Z.isObject(e)||s.Z.exists(e.id)||s.Z.exists(e.name)?this.attr[t]=e:s.Z.mergeAttr(this.attr[t],e)},setAttribute:function(e){var r,o,n,a,h,l,c={};for(r=0;r<arguments.length;r++)o=arguments[r],s.Z.isString(o)?(n=o.split(":"),c[s.Z.trim(n[0])]=s.Z.trim(n[1])):s.Z.isArray(o)?c[o[0]]=o[1]:t.Z.extend(c,o);for(r in c)if(c.hasOwnProperty(r)&&(a=r.replace(/\s+/g,"").toLowerCase(),h=c[r]),h=(!h.toLowerCase||"false"!==h.toLowerCase())&&h,this.attr[a]!==h)switch(a){case"axis":!1===h&&s.Z.exists(this.defaultAxes)&&(this.defaultAxes.x.setAttribute({visible:!1}),this.defaultAxes.y.setAttribute({visible:!1}));break;case"boundingbox":this.setBoundingBox(h,this.keepaspectratio),this._set(a,h);break;case"defaultaxes":s.Z.exists(this.defaultAxes.x)&&s.Z.exists(h.x)&&this.defaultAxes.x.setAttribute(h.x),s.Z.exists(this.defaultAxes.y)&&s.Z.exists(h.y)&&this.defaultAxes.y.setAttribute(h.y);break;case"title":this.document.getElementById(this.container+"_ARIAlabel").innerHTML=h,this._set(a,h);break;case"keepaspectratio":this._set(a,h),this.setBoundingBox(this.getBoundingBox(),h,"keep");break;case"document":case"maxboundingbox":this[a]=h,this._set(a,h);break;case"zoomx":case"zoomy":this[a]=h,this._set(a,h),this.setZoom(this.attr.zoomx,this.attr.zoomy);break;case"registerevents":case"renderer":break;case"fullscreen":case"screenshot":(l=this.containerObj.ownerDocument.getElementById(this.container+"_navigation_"+a))&&s.Z.exists(h.symbol)&&(l.innerHTML=s.Z.evaluate(h.symbol)),this._set(a,h);break;case"selection":h.visible=!1,h.withLines=!1,h.vertices={visible:!1},this._set(a,h);break;case"showcopyright":"svg"===this.renderer.type&&((l=this.containerObj.ownerDocument.getElementById(this.renderer.uniqName("licenseText")))?l.style.display=s.Z.evaluate(h)?"inline":"none":s.Z.evaluate(h)&&this.renderer.displayCopyright(i.Z.licenseText,parseInt(this.options.text.fontSize,10))),this._set(a,h);break;case"showlogo":"svg"===this.renderer.type&&((l=this.containerObj.ownerDocument.getElementById(this.renderer.uniqName("licenseLogo")))?l.style.display=s.Z.evaluate(h)?"inline":"none":s.Z.evaluate(h)&&this.renderer.displayLogo(i.Z.licenseLogo,parseInt(this.options.text.fontSize,10))),this._set(a,h);break;default:s.Z.exists(this.attr[a])&&this._set(a,h)}return this.containerObj.ownerDocument.getElementById(this.container+"_navigationbar").remove(),this.renderer.drawNavigationBar(this,this.attr.navbar),this.triggerEventHandlers(["attribute"],[c,this]),this.fullUpdate(),this},addAnimation:function(t){var e=this;return this.animationObjects[t.id]=t,this.animationIntervalCode||(this.animationIntervalCode=window.setInterval((function(){e.animate()}),t.board.attr.animationdelay)),this},stopAllAnimation:function(){var t;for(t in this.animationObjects)this.animationObjects.hasOwnProperty(t)&&s.Z.exists(this.animationObjects[t])&&(this.animationObjects[t]=null,delete this.animationObjects[t]);return window.clearInterval(this.animationIntervalCode),delete this.animationIntervalCode,this},animate:function(){var t,e,r,o,n,a,h,l,c=0,d=null;for(e in this.animationObjects)if(this.animationObjects.hasOwnProperty(e)&&s.Z.exists(this.animationObjects[e])){if(c+=1,(r=this.animationObjects[e]).animationPath&&(o=s.Z.isFunction(r.animationPath)?r.animationPath((new Date).getTime()-r.animationStart):r.animationPath.pop(),!s.Z.exists(o)||!s.Z.isArray(o)&&isNaN(o)?delete r.animationPath:(r.setPositionDirectly(i.Z.COORDS_BY_USER,o),r.fullUpdate(),d=r)),r.animationData){for(n in h=0,r.animationData)r.animationData.hasOwnProperty(n)&&(a=r.animationData[n].pop(),s.Z.exists(a)?(h+=1,(t={})[n]=a,r.setAttribute(t)):delete r.animationData[a]);0===h&&delete r.animationData}s.Z.exists(r.animationData)||s.Z.exists(r.animationPath)||(this.animationObjects[e]=null,delete this.animationObjects[e],s.Z.exists(r.animationCallback)&&(l=r.animationCallback,r.animationCallback=null,l()))}return 0===c?(window.clearInterval(this.animationIntervalCode),delete this.animationIntervalCode):this.update(d),this},migratePoint:function(t,e,i){var r,o,n,a,h,l,c=!1;for(o in t=this.select(t),e=this.select(e),s.Z.exists(t.label)&&(l=t.label.id,c=!0,this.removeObject(t.label)),t.childElements)if(t.childElements.hasOwnProperty(o)){for(n in a=!1,r=t.childElements[o])r.hasOwnProperty(n)&&r[n]===t&&(r[n]=e,a=!0);for(a&&delete t.childElements[o],h=0;h<r.parents.length;h++)r.parents[h]===t.id&&(r.parents[h]=e.id);e.addChild(r)}return i&&(c&&(delete e.childElements[l],delete e.descendants[l]),e.label&&this.removeObject(e.label),delete this.elementsByName[e.name],e.name=t.name,c&&e.createLabel()),this.removeObject(t),s.Z.exists(e.name)&&""!==e.name&&(this.elementsByName[e.name]=e),this.fullUpdate(),this},emulateColorblindness:function(e){var i,r;if(s.Z.exists(e)||(e="none"),this.currentCBDef===e)return this;for(i in this.objects)this.objects.hasOwnProperty(i)&&(r=this.objects[i],"none"!==e?("none"===this.currentCBDef&&(r.visPropOriginal={strokecolor:r.visProp.strokecolor,fillcolor:r.visProp.fillcolor,highlightstrokecolor:r.visProp.highlightstrokecolor,highlightfillcolor:r.visProp.highlightfillcolor}),r.setAttribute({strokecolor:S.Z.rgb2cb(r.eval(r.visPropOriginal.strokecolor),e),fillcolor:S.Z.rgb2cb(r.eval(r.visPropOriginal.fillcolor),e),highlightstrokecolor:S.Z.rgb2cb(r.eval(r.visPropOriginal.highlightstrokecolor),e),highlightfillcolor:S.Z.rgb2cb(r.eval(r.visPropOriginal.highlightfillcolor),e)})):s.Z.exists(r.visPropOriginal)&&t.Z.extend(r.visProp,r.visPropOriginal));return this.currentCBDef=e,this.update(),this},select:function(t,e){var i,r,o,n,a=t;if(null===a)return a;if(s.Z.isString(a)&&""!==a)s.Z.exists(this.objects[a])?a=this.objects[a]:s.Z.exists(this.elementsByName[a])?a=this.elementsByName[a]:s.Z.exists(this.groups[a])&&(a=this.groups[a]);else if(!e&&(s.Z.isFunction(a)||s.Z.isObject(a)&&!s.Z.isFunction(a.setAttribute))){for(r={},n=(i=s.Z.filterElements(this.objectsList,a)).length,o=0;o<n;o++)r[i[o].id]=i[o];a=new T(r)}else s.Z.isObject(a)&&s.Z.exists(a.id)&&!s.Z.exists(this.objects[a.id])&&(a=null);return a},hasPoint:function(t,e){var i=t,r=e,o=this.getBoundingBox();return s.Z.exists(t)&&s.Z.isArray(t.usrCoords)&&(i=t.usrCoords[1],r=t.usrCoords[2]),!!(s.Z.isNumber(i)&&s.Z.isNumber(r)&&o[0]<i&&i<o[2]&&o[1]>r&&r>o[3])},updateCSSTransforms:function(){var t=this.containerObj,i=t,r=t;if(this.cssTransMat=e.Z.getCSSTransformMatrix(i),s.Z.exists(i.getRootNode)){for(i=i.parentNode===i.getRootNode()?i.parentNode.host:i.parentNode;i;)this.cssTransMat=o.Z.matMatMult(e.Z.getCSSTransformMatrix(i),this.cssTransMat),i=i.parentNode===i.getRootNode()?i.parentNode.host:i.parentNode;this.cssTransMat=o.Z.inverse(this.cssTransMat)}else{for(i=i.offsetParent;i;){for(this.cssTransMat=o.Z.matMatMult(e.Z.getCSSTransformMatrix(i),this.cssTransMat),r=r.parentNode;r!==i;)this.cssTransMat=o.Z.matMatMult(e.Z.getCSSTransformMatrix(i),this.cssTransMat),r=r.parentNode;i=i.offsetParent}this.cssTransMat=o.Z.inverse(this.cssTransMat)}return this},startSelectionMode:function(){this.selectingMode=!0,this.selectionPolygon.setAttribute({visible:!0}),this.selectingBox=[[0,0],[0,0]],this._setSelectionPolygonFromBox(),this.selectionPolygon.fullUpdate()},stopSelectionMode:function(){return this.selectingMode=!1,this.selectionPolygon.setAttribute({visible:!1}),[this.selectionPolygon.vertices[0].coords,this.selectionPolygon.vertices[2].coords]},_startSelecting:function(t){this.isSelecting=!0,this.selectingBox=[[t[0],t[1]],[t[0],t[1]]],this._setSelectionPolygonFromBox()},_moveSelecting:function(t){this.isSelecting&&(this.selectingBox[1]=[t[0],t[1]],this._setSelectionPolygonFromBox(),this.selectionPolygon.fullUpdate())},_stopSelecting:function(t){var e=this.getMousePosition(t);this.isSelecting=!1,this.selectingBox[1]=[e[0],e[1]],this._setSelectionPolygonFromBox()},_setSelectionPolygonFromBox:function(){var e=this.selectingBox[0],i=this.selectingBox[1];this.selectionPolygon.vertices[0].setPositionDirectly(t.Z.COORDS_BY_SCREEN,[e[0],e[1]]),this.selectionPolygon.vertices[1].setPositionDirectly(t.Z.COORDS_BY_SCREEN,[e[0],i[1]]),this.selectionPolygon.vertices[2].setPositionDirectly(t.Z.COORDS_BY_SCREEN,[i[0],i[1]]),this.selectionPolygon.vertices[3].setPositionDirectly(t.Z.COORDS_BY_SCREEN,[i[0],e[1]])},_testForSelection:function(t){this._isRequiredKeyPressed(t,"selection")&&(s.Z.exists(this.selectionPolygon)||this._createSelectionPolygon(this.attr),this.startSelectionMode())},_createSelectionPolygon:function(t){var e;return s.Z.exists(this.selectionPolygon)||!0===(e=s.Z.copyAttributes(t,v.Z,"board","selection")).enabled&&(this.selectionPolygon=this.create("polygon",[[0,0],[0,0],[0,0],[0,0]],e)),this},__evt__attribute:function(t){},__evt__down:function(t){},__evt__mousedown:function(t){},__evt__pendown:function(t){},__evt__pointerdown:function(t){},__evt__touchstart:function(t){},__evt__up:function(t){},__evt__mouseup:function(t){},__evt__pointerup:function(t){},__evt__touchend:function(t){},__evt__click:function(t){},__evt__dblclick:function(t){},__evt__mouseclick:function(t){},__evt__mousedblclick:function(t){},__evt__pointerclick:function(t){},__evt__pointerdblclick:function(t){},__evt__move:function(t,e){},__evt__mousemove:function(t,e){},__evt__penmove:function(t,e){},__evt__pointermove:function(t,e){},__evt__touchmove:function(t,e){},__evt__keymove:function(t,e){},__evt__hit:function(t,e,i){},__evt__mousehit:function(t,e,i){},__evt__update:function(){},__evt__boundingbox:function(){},__evt__startselecting:function(){},__evt__mousestartselecting:function(){},__evt__pointerstartselecting:function(){},__evt__touchstartselecting:function(){},__evt__stopselecting:function(){},__evt__mousestopselecting:function(){},__evt__pointerstopselecting:function(){},__evt__touchstopselecting:function(){},__evt__moveselecting:function(){},__evt__mousemoveselecting:function(){},__evt__pointermoveselecting:function(){},__evt__touchmoveselecting:function(){},__evt:function(){},toFullscreen:function(t){var e,i,r,o,n=this.document;return t=t||this.container,this._fullscreen_inner_id=t,r=n.getElementById(t),e="fullscreenwrap_"+t,s.Z.exists(r._cssFullscreenStore)||(o=this.containerObj.getBoundingClientRect(),r._cssFullscreenStore={w:o.width,h:o.height}),n.getElementById(e)?i=n.getElementById(e):((i=document.createElement("div")).classList.add("JXG_wrap_private"),i.setAttribute("id",e),r.parentNode.insertBefore(i,r),i.appendChild(r)),i.requestFullscreen=i.requestFullscreen||i.webkitRequestFullscreen||i.mozRequestFullScreen||i.msRequestFullscreen,null===(void 0!==n.fullscreenElement?n.fullscreenElement:void 0!==n.webkitFullscreenElement?n.webkitFullscreenElement:n.msFullscreenElement)?i.requestFullscreen&&(i.requestFullscreen(),this.startFullscreenResizeObserver(i)):(this.stopFullscreenResizeObserver(i),s.Z.exists(document.exitFullscreen)?document.exitFullscreen():s.Z.exists(document.webkitExitFullscreen)&&document.webkitExitFullscreen()),this},fullscreenListener:function(t){var i,r,o,n=this.document;i=this._fullscreen_inner_id,s.Z.exists(i)&&(o=void 0!==n.fullscreenElement?n.fullscreenElement:void 0!==n.webkitFullscreenElement?n.webkitFullscreenElement:n.msFullscreenElement,r=n.getElementById(i),o?(r._cssFullscreenStore.id=o.id,r._cssFullscreenStore.isFullscreen=!0,r._cssFullscreenStore.margin=r.style.margin,r._cssFullscreenStore.width=r.style.width,r._cssFullscreenStore.height=r.style.height,r._cssFullscreenStore.transform=r.style.transform,r.style.width=r._cssFullscreenStore.w+"px",r.style.height=r._cssFullscreenStore.h+"px",r.style.margin="",e.Z.scaleJSXGraphDiv(o.id,i,n,s.Z.evaluate(this.attr.fullscreen.scale)),o=null):s.Z.exists(r._cssFullscreenStore)&&(r._cssFullscreenStore.isFullscreen=!1,r.style.margin=r._cssFullscreenStore.margin,r.style.width=r._cssFullscreenStore.width,r.style.height=r._cssFullscreenStore.height,r.style.transform=r._cssFullscreenStore.transform,r._cssFullscreenStore=null,r.parentElement.replaceWith(r)),this.updateCSSTransforms())},startFullscreenResizeObserver:function(t){var i=this;return e.Z.isBrowser&&this.attr.resize&&this.attr.resize.enabled?(this.resizeObserver=new ResizeObserver((function(r){var o,n,a=i.document;i._isResizing||(i._isResizing=!0,window.setTimeout((function(){try{o=i._fullscreen_inner_id,null!==(n=void 0!==a.fullscreenElement?a.fullscreenElement:void 0!==a.webkitFullscreenElement?a.webkitFullscreenElement:a.msFullscreenElement)&&e.Z.scaleJSXGraphDiv(n.id,o,a,s.Z.evaluate(i.attr.fullscreen.scale))}catch(e){i.stopFullscreenResizeObserver(t)}finally{i._isResizing=!1}}),i.attr.resize.throttle))})),this.resizeObserver.observe(t),this):this},stopFullscreenResizeObserver:function(t){return e.Z.isBrowser&&this.attr.resize&&this.attr.resize.enabled?(s.Z.exists(this.resizeObserver)&&this.resizeObserver.unobserve(t),this):this},addLogEntry:function(t,e,i){var r,o,n=this.userLog.length-1;return s.Z.exists(e.elementClass)&&(o=e.id),s.Z.evaluate(this.attr.logging.enabled)&&(r=(new Date).getTime(),n>=0&&this.userLog[n].type===t&&this.userLog[n].id===o&&r-this.userLog[n].end<500?(this.userLog[n].end=r,this.userLog[n].endpos=i):this.userLog.push({type:t,id:o,start:r,startpos:i,end:r,endpos:i,bbox:this.getBoundingBox(),canvas:[this.canvasWidth,this.canvasHeight],zoom:[this.zoomX,this.zoomY]})),this},createRoulette:function(t,e,i,s,r,n,a){var h=this;return new function(){var l,c=0,u=0,p=0,f=i,_=d.Z.root((function(i){var s=t.X(f),r=t.Y(f),o=e.X(i),n=e.Y(i);return(s-o)*(s-o)+(r-n)*(r-n)}),[0,2*Math.PI]),m=0,g=0,v=h.create("transform",[function(){return c}],{type:"rotate"}),Z=h.create("transform",[function(){return c},function(){return t.X(f)},function(){return t.Y(f)}],{type:"rotate"}),C=h.create("transform",[function(){return u},function(){return p}],{type:"translate"}),y=function(t,e,i){var s=d.Z.D(t.X)(e),r=d.Z.D(t.Y)(e),n=d.Z.D(t.X)(i),a=d.Z.D(t.Y)(i),h=d.Z.D(t.X)(.5*(e+i)),l=d.Z.D(t.Y)(.5*(e+i)),c=o.Z.hypot(s,r),u=o.Z.hypot(n,a);return(c+4*o.Z.hypot(h,l)+u)*(i-e)/6},P=function(t){return l-y(e,_,t)},E=Math.PI/18,x=9*E,O=null;return this.rolling=function(){var i,o,n,O,M;l=y(t,f,m=f+r*s),g=d.Z.root(P,_),i=new b(t.X(m),t.Y(m)),o=new b(e.X(g),e.Y(g)),n=new b(d.Z.D(t.X)(m),d.Z.D(t.Y)(m)),O=new b(d.Z.D(e.X)(g),d.Z.D(e.Y)(g)),M=b.C.div(n,O),c=Math.atan2(M.imaginary,M.real),M.div(b.C.abs(M)),M.mult(o),u=i.real-M.real,p=i.imaginary-M.imaginary,c<-E&&c>-x?(c=-E,Z.applyOnce(a)):c>E&&c<x?(c=E,Z.applyOnce(a)):(v.applyOnce(a),C.applyOnce(a),f=m,_=g),h.update()},this.start=function(){return n>0&&(O=window.setInterval(this.rolling,n)),this},this.stop=function(){return window.clearInterval(O),this},this}}});var N=t.Z.Board;t.Z.SVGRenderer=function(t,e){var i;for(this.type="svg",this.isIE=-1!==navigator.appVersion.indexOf("MSIE")||navigator.userAgent.match(/Trident\//),this.svgRoot=null,this.svgNamespace="http://www.w3.org/2000/svg",this.xlinkNamespace="http://www.w3.org/1999/xlink",this.container=t,this.container.style.MozUserSelect="none",this.container.style.userSelect="none",this.container.style.overflow="hidden",""===this.container.style.position&&(this.container.style.position="relative"),this.svgRoot=this.container.ownerDocument.createElementNS(this.svgNamespace,"svg"),this.svgRoot.style.overflow="hidden",this.svgRoot.style.display="block",this.resize(e.width,e.height),this.container.appendChild(this.svgRoot),this.defs=this.container.ownerDocument.createElementNS(this.svgNamespace,"defs"),this.svgRoot.appendChild(this.defs),this.createShadowFilter=function(t,e,i,r,o,n){var a,h,l,c,d,u=this.container.ownerDocument.createElementNS(this.svgNamespace,"filter");return u.setAttributeNS(null,"id",t),u.setAttributeNS(null,"width","300%"),u.setAttributeNS(null,"height","300%"),u.setAttributeNS(null,"filterUnits","userSpaceOnUse"),(a=this.container.ownerDocument.createElementNS(this.svgNamespace,"feOffset")).setAttributeNS(null,"in","SourceGraphic"),a.setAttributeNS(null,"result","offOut"),a.setAttributeNS(null,"dx",n[0]),a.setAttributeNS(null,"dy",n[1]),u.appendChild(a),(h=this.container.ownerDocument.createElementNS(this.svgNamespace,"feColorMatrix")).setAttributeNS(null,"in","offOut"),h.setAttributeNS(null,"result","colorOut"),h.setAttributeNS(null,"type","matrix"),"none"===e||!s.Z.isArray(e)||e.length<3?h.setAttributeNS(null,"values","0.1 0 0 0 0  0 0.1 0 0 0  0 0 0.1 0 0  0 0 0 "+i+" 0"):(e[0]/=255,e[1]/=255,e[2]/=255,d=r+" 0 0 0 "+e[0]+"  0 "+r+" 0 0 "+e[1]+"  0 0 "+r+" 0 "+e[2]+"  0 0 0 "+i+" 0",h.setAttributeNS(null,"values",d)),u.appendChild(h),(l=this.container.ownerDocument.createElementNS(this.svgNamespace,"feGaussianBlur")).setAttributeNS(null,"in","colorOut"),l.setAttributeNS(null,"result","blurOut"),l.setAttributeNS(null,"stdDeviation",o),u.appendChild(l),(c=this.container.ownerDocument.createElementNS(this.svgNamespace,"feBlend")).setAttributeNS(null,"in","SourceGraphic"),c.setAttributeNS(null,"in2","blurOut"),c.setAttributeNS(null,"mode","normal"),u.appendChild(c),u},this.uniqName=function(){return this.container.id+"_"+Array.prototype.slice.call(arguments).join("_")},this.toStr=function(){var t=Array.prototype.slice.call(arguments).join("");return s.Z.exists(encodeURIComponent)&&(t=encodeURIComponent(t)),t},this.toURL=function(){return"url(#"+this.toStr.apply(this,arguments)+")"},this.defs.appendChild(this.createShadowFilter(this.uniqName("f1"),"none",1,.1,3,[5,5])),this.layer=[],i=0;i<v.Z.layer.numlayers;i++)this.layer[i]=this.container.ownerDocument.createElementNS(this.svgNamespace,"g"),this.svgRoot.appendChild(this.layer[i]);try{this.foreignObjLayer=this.container.ownerDocument.createElementNS(this.svgNamespace,"foreignObject"),this.foreignObjLayer.setAttribute("display","none"),this.foreignObjLayer.setAttribute("x",0),this.foreignObjLayer.setAttribute("y",0),this.foreignObjLayer.setAttribute("width","100%"),this.foreignObjLayer.setAttribute("height","100%"),this.foreignObjLayer.setAttribute("id",this.uniqName("foreignObj")),this.svgRoot.appendChild(this.foreignObjLayer),this.supportsForeignObject=!0}catch(t){this.supportsForeignObject=!1}},t.Z.SVGRenderer.prototype=new Z,t.Z.extend(t.Z.SVGRenderer.prototype,{_createArrowHead:function(t,e,r){var o,n,a,h,l=t.id+"Triangle";return s.Z.exists(e)&&(l+=e),s.Z.exists(r)&&(l+=r),(o=this.createPrim("marker",l)).setAttributeNS(null,"stroke",t.evalVisProp("strokecolor")),o.setAttributeNS(null,"stroke-opacity",t.evalVisProp("strokeopacity")),o.setAttributeNS(null,"fill",t.evalVisProp("strokecolor")),o.setAttributeNS(null,"fill-opacity",t.evalVisProp("strokeopacity")),o.setAttributeNS(null,"stroke-width",0),o.setAttributeNS(null,"orient","auto"),o.setAttributeNS(null,"markerUnits","strokeWidth"),n=this.container.ownerDocument.createElementNS(this.svgNamespace,"path"),h=5,"Start"===e?(a=0,2===r?n.setAttributeNS(null,"d","M 10,0 L 0,5 L 10,10 L 5,5 z"):3===r?n.setAttributeNS(null,"d","M 0,0 L 3.33,0 L 3.33,10 L 0,10 z"):4===r?(h=3.31,n.setAttributeNS(null,"d","M 0.00,3.31 C 3.53,3.84 7.13,4.50 10.00,6.63 C 9.33,5.52 8.67,4.42 8.00,3.31 C 8.67,2.21 9.33,1.10 10.00,0.00 C 7.13,2.13 3.53,2.79 0.00,3.31")):5===r?(h=3.28,n.setAttributeNS(null,"d","M 0.00,3.28 C 3.39,4.19 6.81,5.07 10.00,6.55 C 9.38,5.56 9.00,4.44 9.00,3.28 C 9.00,2.11 9.38,0.99 10.00,0.00 C 6.81,1.49 3.39,2.37 0.00,3.28")):6===r?(h=2.84,n.setAttributeNS(null,"d","M 0.00,2.84 C 3.39,3.59 6.79,4.35 10.00,5.68 C 9.67,4.73 9.33,3.78 9.00,2.84 C 9.33,1.89 9.67,0.95 10.00,0.00 C 6.79,1.33 3.39,2.09 0.00,2.84")):7===r?(h=5.2,n.setAttributeNS(null,"d","M 0.00,5.20 C 4.04,5.20 7.99,6.92 10.00,10.39 M 10.00,0.00 C 7.99,3.47 4.04,5.20 0.00,5.20")):n.setAttributeNS(null,"d","M 10,0 L 0,5 L 10,10 z"),t.elementClass===i.Z.OBJECT_CLASS_LINE&&(a=2===r?4.9:3===r?3.3:4===r||5===r||6===r?6.66:7===r?0:10)):(a=10,2===r?n.setAttributeNS(null,"d","M 0,0 L 10,5 L 0,10 L 5,5 z"):3===r?(a=3.3,n.setAttributeNS(null,"d","M 0,0 L 3.33,0 L 3.33,10 L 0,10 z")):4===r?(h=3.31,n.setAttributeNS(null,"d","M 10.00,3.31 C 6.47,3.84 2.87,4.50 0.00,6.63 C 0.67,5.52 1.33,4.42 2.00,3.31 C 1.33,2.21 0.67,1.10 0.00,0.00 C 2.87,2.13 6.47,2.79 10.00,3.31")):5===r?(h=3.28,n.setAttributeNS(null,"d","M 10.00,3.28 C 6.61,4.19 3.19,5.07 0.00,6.55 C 0.62,5.56 1.00,4.44 1.00,3.28 C 1.00,2.11 0.62,0.99 0.00,0.00 C 3.19,1.49 6.61,2.37 10.00,3.28")):6===r?(h=2.84,n.setAttributeNS(null,"d","M 10.00,2.84 C 6.61,3.59 3.21,4.35 0.00,5.68 C 0.33,4.73 0.67,3.78 1.00,2.84 C 0.67,1.89 0.33,0.95 0.00,0.00 C 3.21,1.33 6.61,2.09 10.00,2.84")):7===r?(h=5.2,n.setAttributeNS(null,"d","M 10.00,5.20 C 5.96,5.20 2.01,6.92 0.00,10.39 M 0.00,0.00 C 2.01,3.47 5.96,5.20 10.00,5.20")):n.setAttributeNS(null,"d","M 0,0 L 10,5 L 0,10 z"),t.elementClass===i.Z.OBJECT_CLASS_LINE&&(a=2===r?5.1:3===r?.02:4===r||5===r||6===r?3.33:7===r?10:.05)),7===r&&(o.setAttributeNS(null,"fill","none"),o.setAttributeNS(null,"stroke-width",1)),o.setAttributeNS(null,"refY",h),o.setAttributeNS(null,"refX",a),o.appendChild(n),o},_setArrowColor:function(t,e,i,r,o){t&&(s.Z.isString(e)&&(7!==o?this._setAttribute((function(){t.setAttributeNS(null,"stroke",e),t.setAttributeNS(null,"fill",e),t.setAttributeNS(null,"stroke-opacity",i),t.setAttributeNS(null,"fill-opacity",i)}),r.visPropOld.fillcolor):this._setAttribute((function(){t.setAttributeNS(null,"fill","none"),t.setAttributeNS(null,"stroke",e),t.setAttributeNS(null,"stroke-opacity",i)}),r.visPropOld.fillcolor)),r.rendNode.parentNode.insertBefore(r.rendNode,r.rendNode))},_setArrowWidth:function(t,e,i,s){var r,o;t&&(o=(r=e)*s,t.setAttributeNS(null,"viewBox","0 0 "+10*r+" "+10*r),t.setAttributeNS(null,"markerHeight",o),t.setAttributeNS(null,"markerWidth",o),t.setAttributeNS(null,"display","inherit"),i.parentNode.insertBefore(i,i))},updateTicks:function(t){var e,i,r,o,n,a,h,l,c="",d=t.ticks.length,u=!0;for(e=0;e<d;e++){for(n=(r=t.ticks[e])[0],a=r[1],h=n.length,l=" M "+n[0]+" "+a[0],s.Z.isNumber(n[0])||(u=!1),i=1;u&&i<h;++i)s.Z.isNumber(n[i])?l+=" L "+n[i]+" "+a[i]:u=!1;u&&(c+=l)}o=t.rendNode,s.Z.exists(o)||(o=this.createPrim("path",t.id),this.appendChildPrim(o,t.evalVisProp("layer")),t.rendNode=o),o.setAttributeNS(null,"stroke",t.evalVisProp("strokecolor")),o.setAttributeNS(null,"fill","none"),o.setAttributeNS(null,"stroke-opacity",t.evalVisProp("strokeopacity")),o.setAttributeNS(null,"stroke-width",t.evalVisProp("strokewidth")),this.updatePathPrim(o,c,t.board)},displayCopyright:function(t,e){var i,s,r=4+1.8*e,o=6+e;(i=this.createPrim("text","licenseText")).setAttributeNS(null,"x",r+"px"),i.setAttributeNS(null,"y",o+"px"),i.setAttributeNS(null,"style","font-family:Arial,Helvetica,sans-serif; font-size:"+e+"px; opacity:0.2;"),i.setAttributeNS(null,"aria-hidden","true"),s=this.container.ownerDocument.createTextNode(t),i.appendChild(s),this.appendChildPrim(i,0)},displayLogo:function(t,e){var i,s=1.5*e;(i=this.createPrim("image","licenseLogo")).setAttributeNS(null,"x","5px"),i.setAttributeNS(null,"y","5px"),i.setAttributeNS(null,"width",s+"px"),i.setAttributeNS(null,"height",s+"px"),i.setAttributeNS(null,"preserveAspectRatio","none"),i.setAttributeNS(null,"style","opacity:0.2;"),i.setAttributeNS(null,"aria-hidden","true"),i.setAttributeNS(this.xlinkNamespace,"xlink:href",t),this.appendChildPrim(i,0)},drawInternalText:function(t){var e=this.createPrim("text",t.id);return e.style.whiteSpace="nowrap",t.rendNodeText=this.container.ownerDocument.createTextNode(""),e.appendChild(t.rendNodeText),this.appendChildPrim(e,t.evalVisProp("layer")),e},updateInternalText:function(t){var e,i,s=t.plaintext,r=t.getAnchorX(),o=t.getAnchorY();i=t.evalVisProp("cssclass"),t.rendNode.getAttributeNS(null,"class")!==i&&(t.rendNode.setAttributeNS(null,"class",i),t.needsSizeUpdate=!0),isNaN(t.coords.scrCoords[1]+t.coords.scrCoords[2])||(e=t.coords.scrCoords[1],t.visPropOld.left!==r+e&&(t.rendNode.setAttributeNS(null,"x",e+"px"),"left"===r?t.rendNode.setAttributeNS(null,"text-anchor","start"):"right"===r?t.rendNode.setAttributeNS(null,"text-anchor","end"):"middle"===r&&t.rendNode.setAttributeNS(null,"text-anchor","middle"),t.visPropOld.left=r+e),e=t.coords.scrCoords[2],t.visPropOld.top!==o+e&&(t.rendNode.setAttributeNS(null,"y",e+.5*this.vOffsetText+"px"),"bottom"===o?(t.rendNode.setAttributeNS(null,"dy","0"),t.rendNode.setAttributeNS(null,"dominant-baseline","auto")):"top"===o?(t.rendNode.setAttributeNS(null,"dy","1.6ex"),t.rendNode.setAttributeNS(null,"dominant-baseline","auto")):"middle"===o&&(t.rendNode.setAttributeNS(null,"dy","0.6ex"),t.rendNode.setAttributeNS(null,"dominant-baseline","auto")),t.visPropOld.top=o+e)),t.htmlStr!==s&&(t.rendNodeText.data=s,t.htmlStr=s),this.transformRect(t,t.transformations)},updateInternalTextStyle:function(t,e,i,s){this.setObjectFillColor(t,e,i)},drawImage:function(t){var e=this.createPrim("image",t.id);e.setAttributeNS(null,"preserveAspectRatio","none"),this.appendChildPrim(e,t.evalVisProp("layer")),t.rendNode=e,this.updateImage(t)},transformRect:function(t,e){var s,r,o,n,a,h="";if(e.length>0&&(o=t.rendNode,-1===(s=[(r=this.joinTransforms(t,e))[1][1],r[2][1],r[1][2],r[2][2],r[1][0],r[2][0]].join(",")).indexOf("NaN")))if(h+=" matrix("+s+") ",t.elementClass===i.Z.OBJECT_CLASS_TEXT&&"html"===t.visProp.display){switch(o.style.transform=h,n=-t.coords.scrCoords[1],a=-t.coords.scrCoords[2],t.evalVisProp("anchorx")){case"right":n+=t.size[0];break;case"middle":n+=.5*t.size[0]}switch(t.evalVisProp("anchory")){case"bottom":a+=t.size[1];break;case"middle":a+=.5*t.size[1]}o.style["transform-origin"]=n+"px "+a+"px"}else o.setAttributeNS(null,"transform",h)},updateImageURL:function(t){var e=t.eval(t.url);return t._src!==e&&(t.imgIsLoaded=!1,t.rendNode.setAttributeNS(this.xlinkNamespace,"xlink:href",e),t._src=e,!0)},updateImageStyle:function(t,e){var i=t.evalVisProp(e?"highlightcssclass":"cssclass");t.rendNode.setAttributeNS(null,"class",i)},drawForeignObject:function(t){t.rendNode=this.appendChildPrim(this.createPrim("foreignObject",t.id),t.evalVisProp("layer")),this.appendNodesToElement(t,"foreignObject"),this.updateForeignObject(t)},updateForeignObject:function(t){t._useUserSize?t.rendNode.style.overflow="hidden":t.rendNode.style.overflow="visible",this.updateRectPrim(t.rendNode,t.coords.scrCoords[1],t.coords.scrCoords[2]-t.size[1],t.size[0],t.size[1]),!0===t.evalVisProp("evaluateOnlyOnce")&&t.renderedOnce||(t.rendNode.innerHTML=t.content,t.renderedOnce=!0),this._updateVisual(t,{stroke:!0,dash:!0},!0)},appendChildPrim:function(t,e){return s.Z.exists(e)?e>=v.Z.layer.numlayers&&(e=v.Z.layer.numlayers-1):e=0,this.layer[e].appendChild(t),t},createPrim:function(t,e){var i=this.container.ownerDocument.createElementNS(this.svgNamespace,t);return i.setAttributeNS(null,"id",this.uniqName(e)),i.style.position="absolute","path"===t&&(i.setAttributeNS(null,"stroke-linecap","round"),i.setAttributeNS(null,"stroke-linejoin","round"),i.setAttributeNS(null,"fill-rule","evenodd")),i},remove:function(t){s.Z.exists(t)&&s.Z.exists(t.parentNode)&&t.parentNode.removeChild(t)},setLayer:function(t,e){s.Z.exists(e)?e>=v.Z.layer.numlayers&&(e=v.Z.layer.numlayers-1):e=0,this.layer[e].appendChild(t.rendNode)},makeArrows:function(t,e){var i,r,o=e.evFirst,n=e.evLast;this.isIE&&t.visPropCalc.visible&&(o||n)?t.rendNode.parentNode.insertBefore(t.rendNode,t.rendNode):(i=t.rendNodeTriangleStart,o?(r=this.toStr(this.container.id,"_",t.id,"TriangleStart",e.typeFirst),s.Z.exists(i)&&i.id===r||(null===(i=this.container.ownerDocument.getElementById(r))&&(i=this._createArrowHead(t,"Start",e.typeFirst),this.defs.appendChild(i)),t.rendNodeTriangleStart=i,t.rendNode.setAttributeNS(null,"marker-start",this.toURL(r)))):(s.Z.exists(i)&&(this.remove(i),t.rendNodeTriangleStart=null),t.rendNode.setAttributeNS(null,"marker-start",null)),i=t.rendNodeTriangleEnd,n?(r=this.toStr(this.container.id,"_",t.id,"TriangleEnd",e.typeLast),s.Z.exists(i)&&i.id===r||(null===(i=this.container.ownerDocument.getElementById(r))&&(i=this._createArrowHead(t,"End",e.typeLast),this.defs.appendChild(i)),t.rendNodeTriangleEnd=i,t.rendNode.setAttributeNS(null,"marker-end",this.toURL(r)))):(s.Z.exists(i)&&(this.remove(i),t.rendNodeTriangleEnd=null),t.rendNode.setAttributeNS(null,"marker-end",null)))},updateEllipsePrim:function(t,e,i,s,r){var o;o=2e5,e=Math.abs(e)<o?e:o*e/Math.abs(e),i=Math.abs(i)<o?i:o*i/Math.abs(i),s=Math.abs(s)<o?s:o*s/Math.abs(s),r=Math.abs(r)<o?r:o*r/Math.abs(r),t.setAttributeNS(null,"cx",e),t.setAttributeNS(null,"cy",i),t.setAttributeNS(null,"rx",Math.abs(s)),t.setAttributeNS(null,"ry",Math.abs(r))},updateLinePrim:function(t,e,i,s,r){var o;o=2e5,isNaN(e+i+s+r)||(e=Math.abs(e)<o?e:o*e/Math.abs(e),i=Math.abs(i)<o?i:o*i/Math.abs(i),s=Math.abs(s)<o?s:o*s/Math.abs(s),r=Math.abs(r)<o?r:o*r/Math.abs(r),t.setAttributeNS(null,"x1",e),t.setAttributeNS(null,"y1",i),t.setAttributeNS(null,"x2",s),t.setAttributeNS(null,"y2",r))},updatePathPrim:function(t,e){""===e&&(e="M 0 0"),t.setAttributeNS(null,"d",e)},updatePathStringPoint:function(t,e,i){var s="",r=t.coords.scrCoords,o=e*Math.sqrt(3)*.5,n=.5*e;return"x"===i?s=" M "+(r[1]-e)+" "+(r[2]-e)+" L "+(r[1]+e)+" "+(r[2]+e)+" M "+(r[1]+e)+" "+(r[2]-e)+" L "+(r[1]-e)+" "+(r[2]+e):"+"===i?s=" M "+(r[1]-e)+" "+r[2]+" L "+(r[1]+e)+" "+r[2]+" M "+r[1]+" "+(r[2]-e)+" L "+r[1]+" "+(r[2]+e):"|"===i?s=" M "+r[1]+" "+(r[2]-e)+" L "+r[1]+" "+(r[2]+e):"-"===i?s=" M "+(r[1]-e)+" "+r[2]+" L "+(r[1]+e)+" "+r[2]:"<>"===i||"<<>>"===i?("<<>>"===i&&(e*=1.41),s=" M "+(r[1]-e)+" "+r[2]+" L "+r[1]+" "+(r[2]+e)+" L "+(r[1]+e)+" "+r[2]+" L "+r[1]+" "+(r[2]-e)+" Z "):"^"===i?s=" M "+r[1]+" "+(r[2]-e)+" L "+(r[1]-o)+" "+(r[2]+n)+" L "+(r[1]+o)+" "+(r[2]+n)+" Z ":"v"===i?s=" M "+r[1]+" "+(r[2]+e)+" L "+(r[1]-o)+" "+(r[2]-n)+" L "+(r[1]+o)+" "+(r[2]-n)+" Z ":">"===i?s=" M "+(r[1]+e)+" "+r[2]+" L "+(r[1]-n)+" "+(r[2]-o)+" L "+(r[1]-n)+" "+(r[2]+o)+" Z ":"<"===i&&(s=" M "+(r[1]-e)+" "+r[2]+" L "+(r[1]+n)+" "+(r[2]-o)+" L "+(r[1]+n)+" "+(r[2]+o)+" Z "),s},updatePathStringPrim:function(t){var e,i,s,r=" M ",o=r,n=5e3,a="";if(t.numberPoints<=0)return"";if(s=Math.min(t.points.length,t.numberPoints),1===t.bezierDegree)for(e=0;e<s;e++)i=t.points[e].scrCoords,isNaN(i[1])||isNaN(i[2])?o=r:(i[1]=Math.max(Math.min(i[1],n),-5e3),i[2]=Math.max(Math.min(i[2],n),-5e3),a+=o+i[1]+" "+i[2],o=" L ");else if(3===t.bezierDegree)for(e=0;e<s;)i=t.points[e].scrCoords,isNaN(i[1])||isNaN(i[2])?o=r:(a+=o+i[1]+" "+i[2]," C "===o&&(e+=1,a+=" "+(i=t.points[e].scrCoords)[1]+" "+i[2],e+=1,a+=" "+(i=t.points[e].scrCoords)[1]+" "+i[2]),o=" C "),e+=1;return a},updatePathStringBezierPrim:function(t){var e,i,s,r,o,n,a,h=" M ",l=h,c=5e3,u="",p=t.evalVisProp("strokewidth"),f="plot"!==t.evalVisProp("curvetype");if(t.numberPoints<=0)return"";for(f&&t.board.options.curve.RDPsmoothing&&(t.points=d.Z.RamerDouglasPeucker(t.points,.5)),a=Math.min(t.points.length,t.numberPoints),i=1;i<3;i++)for(l=h,e=0;e<a;e++)r=t.points[e].scrCoords,isNaN(r[1])||isNaN(r[2])?l=h:(r[1]=Math.max(Math.min(r[1],c),-5e3),r[2]=Math.max(Math.min(r[2],c),-5e3),l===h?u+=l+r[1]+" "+r[2]:(s=2*i,u+=[l,o+.333*(r[1]-o)+p*(s*Math.random()-i)," ",n+.333*(r[2]-n)+p*(s*Math.random()-i)," ",o+.666*(r[1]-o)+p*(s*Math.random()-i)," ",n+.666*(r[2]-n)+p*(s*Math.random()-i)," ",r[1]," ",r[2]].join("")),l=" C ",o=r[1],n=r[2]);return u},updatePolygonPrim:function(t,e){var i,s,r="",o=e.vertices.length;for(t.setAttributeNS(null,"stroke","none"),t.setAttributeNS(null,"fill-rule","evenodd"),"polygonalchain"===e.elType&&o++,i=0;i<o-1;i++){if(!e.vertices[i].isReal)return void t.setAttributeNS(null,"points","");r=r+(s=e.vertices[i].coords.scrCoords)[1]+","+s[2],i<o-2&&(r+=" ")}-1===r.indexOf("NaN")&&t.setAttributeNS(null,"points",r)},updateRectPrim:function(t,e,i,s,r){t.setAttributeNS(null,"x",e),t.setAttributeNS(null,"y",i),t.setAttributeNS(null,"width",s),t.setAttributeNS(null,"height",r)},_setAttribute:function(t,e){""===e?t():window.setTimeout(t,1)},display:function(t,e){var i;t&&t.rendNode&&(t.visPropOld.visible=e,i=t.rendNode,e?(i.setAttributeNS(null,"display","inline"),i.style.visibility="inherit"):(i.setAttributeNS(null,"display","none"),i.style.visibility="hidden"))},hide:function(e){t.Z.deprecated("Board.renderer.hide()","Board.renderer.display()"),this.display(e,!1)},setARIA:function(t){var e,i,s;for(e in t.visProp.aria)t.visProp.aria.hasOwnProperty(e)&&"enabled"!==e&&(i="aria."+e,s=t.evalVisProp("aria."+e),t.visPropOld[i]!==s&&(this.setPropertyPrim(t.rendNode,"aria-"+e,s),t.visPropOld[i]=s))},setBuffering:function(t,e){t.rendNode.setAttribute("buffered-rendering",e)},setCssClass(t,e){t.visPropOld.cssclass!==e&&(this.setPropertyPrim(t.rendNode,"class",e),t.visPropOld.cssclass=e)},setDashStyle:function(t){var e=t.evalVisProp("dash"),i=t.evalVisProp("dashscale")?.5*t.evalVisProp("strokewidth"):1,s=t.rendNode;e>0?s.setAttributeNS(null,"stroke-dasharray",this.dashArray[e-1].map((function(t){return t*i})).join(",")):s.hasAttributeNS(null,"stroke-dasharray")&&s.removeAttributeNS(null,"stroke-dasharray")},setGradient:function(t){var e,i,s,r=t.rendNode,o=t.evalVisProp("gradient");"linear"===o||"radial"===o?(e=this.createPrim(o+"Gradient",t.id+"_gradient"),i=this.createPrim("stop",t.id+"_gradient1"),s=this.createPrim("stop",t.id+"_gradient2"),e.appendChild(i),e.appendChild(s),this.defs.appendChild(e),r.setAttributeNS(null,"style","fill:"+this.toURL(this.container.id+"_"+t.id+"_gradient")),t.gradNode1=i,t.gradNode2=s,t.gradNode=e):r.removeAttributeNS(null,"style")},setLineCap:function(t){var e=t.evalVisProp("linecap");void 0!==e&&""!==e&&t.visPropOld.linecap!==e&&s.Z.exists(t.rendNode)&&(this.setPropertyPrim(t.rendNode,"stroke-linecap",e),t.visPropOld.linecap=e)},setObjectFillColor:function(e,i,r,o){var n,a,h,l,c=i,d=r,u=e.evalVisProp("gradient");d=d>0?d:0,e.visPropOld.fillcolor===c&&e.visPropOld.fillopacity===d&&null===u||(s.Z.exists(c)&&!1!==c&&(9!==c.length?(a=c,l=d):(h=S.Z.rgba2rgbo(c),a=h[0],l=d*h[1]),n=void 0===o?e.rendNode:o,"none"!==a&&""!==a&&!1!==a&&this._setAttribute((function(){n.setAttributeNS(null,"fill",a)}),e.visPropOld.fillcolor),e.type===t.Z.OBJECT_TYPE_IMAGE?this._setAttribute((function(){n.setAttributeNS(null,"opacity",l)}),e.visPropOld.fillopacity):("none"===a?(l=0,n.setAttributeNS(null,"pointer-events","visibleStroke")):n.setAttributeNS(null,"pointer-events","visiblePainted"),this._setAttribute((function(){n.setAttributeNS(null,"fill-opacity",l)}),e.visPropOld.fillopacity)),"linear"!==u&&"radial"!==u||this.updateGradient(e)),e.visPropOld.fillcolor=c,e.visPropOld.fillopacity=d)},setObjectStrokeColor:function(t,e,r){var o,n,a,h,l=e,c=r;c=c>0?c:0,t.visPropOld.strokecolor===l&&t.visPropOld.strokeopacity===c||(s.Z.exists(l)&&!1!==l&&(9!==l.length?(o=l,a=c):(n=S.Z.rgba2rgbo(l),o=n[0],a=c*n[1]),h=t.rendNode,t.elementClass===i.Z.OBJECT_CLASS_TEXT?"html"===t.evalVisProp("display")?this._setAttribute((function(){h.style.color=o,h.style.opacity=a}),t.visPropOld.strokecolor):this._setAttribute((function(){h.setAttributeNS(null,"style","fill:"+o),h.setAttributeNS(null,"style","fill-opacity:"+a)}),t.visPropOld.strokecolor):this._setAttribute((function(){h.setAttributeNS(null,"stroke",o),h.setAttributeNS(null,"stroke-opacity",a)}),t.visPropOld.strokecolor),t.elementClass!==i.Z.OBJECT_CLASS_CURVE&&t.elementClass!==i.Z.OBJECT_CLASS_LINE||(t.evalVisProp("firstarrow")&&this._setArrowColor(t.rendNodeTriangleStart,o,a,t,t.visPropCalc.typeFirst),t.evalVisProp("lastarrow")&&this._setArrowColor(t.rendNodeTriangleEnd,o,a,t,t.visPropCalc.typeLast))),t.visPropOld.strokecolor=l,t.visPropOld.strokeopacity=c)},setObjectStrokeWidth:function(t,e){var i,r=e;isNaN(r)||t.visPropOld.strokewidth===r||(i=t.rendNode,this.setPropertyPrim(i,"stroked","true"),s.Z.exists(r)&&this.setPropertyPrim(i,"stroke-width",r+"px"),t.visPropOld.strokewidth=r)},setObjectTransition:function(t,e){var i,r,o,n=[],a=0,h=["rendNode","rendNodeTriangleStart","rendNodeTriangleEnd"];if(void 0===e&&(e=t.evalVisProp("transitionduration")),i=t.evalVisProp("transitionproperties"),e!==t.visPropOld.transitionduration||i!==t.visPropOld.transitionproperties){for(s.Z.exists(i)&&(a=i.length),o=0;o<a;o++)n.push(i[o]+" "+e+"ms");for(r=n.join(", "),a=h.length,o=0;o<a;++o)t[h[o]]&&(t[h[o]].style.transition=r);t.visPropOld.transitionduration=e,t.visPropOld.transitionproperties=i}},setShadow:function(e){var i,r,o,n,a,h,l,c,d=e.evalVisProp("shadow"),u=!0,p=!1;(i=JSON.stringify(d))!==e.visPropOld.shadow&&("boolean"==typeof d?(u=!0,p=d,r="none",o=3,n=.1,a=[5,5],h=1):e.evalVisProp("shadow.enabled")?(u=!1,p=!0,r=t.Z.rgbParser(e.evalVisProp("shadow.color")),o=e.evalVisProp("shadow.blur"),n=e.evalVisProp("shadow.blend"),a=e.evalVisProp("shadow.offset"),h=e.evalVisProp("shadow.opacity")):p=!1,s.Z.exists(e.rendNode)&&(p?u?e.rendNode.setAttributeNS(null,"filter",this.toURL(this.container.id+"_f1")):((c=this.container.ownerDocument.getElementById(l))&&this.defs.removeChild(c),l=e.rendNode.id+"_f1",this.defs.appendChild(this.createShadowFilter(l,r,h,n,o,a)),e.rendNode.setAttributeNS(null,"filter",this.toURL(l))):e.rendNode.removeAttributeNS(null,"filter")),e.visPropOld.shadow=i)},setTabindex:function(t){var e;t.board.attr.keyboard.enabled&&s.Z.exists(t.rendNode)&&(e=t.evalVisProp("tabindex"),t.visPropCalc.visible||(e=null),e!==t.visPropOld.tabindex&&(t.rendNode.setAttribute("tabindex",e),t.visPropOld.tabindex=e))},setPropertyPrim:function(t,e,i){"stroked"!==e&&t.setAttributeNS(null,e,i)},show:function(e){t.Z.deprecated("Board.renderer.show()","Board.renderer.display()"),this.display(e,!0)},updateGradient:function(t){var e,i,r=t.gradNode1,o=t.gradNode2,n=t.evalVisProp("gradient");s.Z.exists(r)&&s.Z.exists(o)&&(i=(i=t.evalVisProp("fillopacity"))>0?i:0,e=t.evalVisProp("fillcolor"),r.setAttributeNS(null,"style","stop-color:"+e+";stop-opacity:"+i),o.setAttributeNS(null,"style","stop-color:"+t.evalVisProp("gradientsecondcolor")+";stop-opacity:"+t.evalVisProp("gradientsecondopacity")),r.setAttributeNS(null,"offset",100*t.evalVisProp("gradientstartoffset")+"%"),o.setAttributeNS(null,"offset",100*t.evalVisProp("gradientendoffset")+"%"),"linear"===n?this.updateGradientAngle(t.gradNode,t.evalVisProp("gradientangle")):"radial"===n&&this.updateGradientCircle(t.gradNode,t.evalVisProp("gradientcx"),t.evalVisProp("gradientcy"),t.evalVisProp("gradientr"),t.evalVisProp("gradientfx"),t.evalVisProp("gradientfy"),t.evalVisProp("gradientfr")))},updateGradientAngle:function(t,e){var i=1,s=Math.cos(e),r=Math.sin(e);Math.abs(s)>Math.abs(r)?i/=Math.abs(s):i/=Math.abs(r),s>=0?(t.setAttributeNS(null,"x1",0),t.setAttributeNS(null,"x2",s*i)):(t.setAttributeNS(null,"x1",-s*i),t.setAttributeNS(null,"x2",0)),r>=0?(t.setAttributeNS(null,"y1",0),t.setAttributeNS(null,"y2",r*i)):(t.setAttributeNS(null,"y1",-r*i),t.setAttributeNS(null,"y2",0))},updateGradientCircle:function(t,e,i,s,r,o,n){t.setAttributeNS(null,"cx",100*e+"%"),t.setAttributeNS(null,"cy",100*i+"%"),t.setAttributeNS(null,"r",100*s+"%"),t.setAttributeNS(null,"fx",100*r+"%"),t.setAttributeNS(null,"fy",100*o+"%"),t.setAttributeNS(null,"fr",100*n+"%")},suspendRedraw:function(){this.suspendHandle=this.svgRoot.suspendRedraw(1e4)},unsuspendRedraw:function(){this.svgRoot.unsuspendRedraw(this.suspendHandle)},resize:function(t,e){this.svgRoot.setAttribute("width",parseFloat(t)),this.svgRoot.setAttribute("height",parseFloat(e))},createTouchpoints:function(t){var e,i,s,r;for(this.touchpoints=[],e=0;e<t;e++)i="touchpoint1_"+e,r=this.createPrim("path",i),this.appendChildPrim(r,19),r.setAttributeNS(null,"d","M 0 0"),this.touchpoints.push(r),this.setPropertyPrim(r,"stroked","true"),this.setPropertyPrim(r,"stroke-width","1px"),r.setAttributeNS(null,"stroke","#000000"),r.setAttributeNS(null,"stroke-opacity",1),r.setAttributeNS(null,"display","none"),s="touchpoint2_"+e,r=this.createPrim("ellipse",s),this.appendChildPrim(r,19),this.updateEllipsePrim(r,0,0,0,0),this.touchpoints.push(r),this.setPropertyPrim(r,"stroked","true"),this.setPropertyPrim(r,"stroke-width","1px"),r.setAttributeNS(null,"stroke","#000000"),r.setAttributeNS(null,"stroke-opacity",1),r.setAttributeNS(null,"fill","#ffffff"),r.setAttributeNS(null,"fill-opacity",0),r.setAttributeNS(null,"display","none")},showTouchpoint:function(t){this.touchpoints&&t>=0&&2*t<this.touchpoints.length&&(this.touchpoints[2*t].setAttributeNS(null,"display","inline"),this.touchpoints[2*t+1].setAttributeNS(null,"display","inline"))},hideTouchpoint:function(t){this.touchpoints&&t>=0&&2*t<this.touchpoints.length&&(this.touchpoints[2*t].setAttributeNS(null,"display","none"),this.touchpoints[2*t+1].setAttributeNS(null,"display","none"))},updateTouchpoint:function(t,e){var i,s;this.touchpoints&&t>=0&&2*t<this.touchpoints.length&&(i=e[0],s=e[1],this.touchpoints[2*t].setAttributeNS(null,"d","M "+(i-37)+" "+s+" L "+(i+37)+" "+s+" M "+i+" "+(s-37)+" L "+i+" "+(s+37)),this.updateEllipsePrim(this.touchpoints[2*t+1],e[0],e[1],25,25))},_getValuesOfDOMElements:function(t){var e=[];if(1===t.nodeType)for(t=t.firstChild;t;)void 0!==t.id&&void 0!==t.value&&e.push([t.id,t.value]),s.Z.concat(e,this._getValuesOfDOMElements(t)),t=t.nextSibling;return e},_getImgDataURL:function(t){var e,i,s,r,o,n;if((i=(e=t.getElementsByTagName("image")).length)>0)for(s=document.createElement("canvas"),n=0;n<i;n++){e[n].setAttribute("crossorigin","anonymous"),r=s.getContext("2d"),s.width=e[n].getAttribute("width"),s.height=e[n].getAttribute("height");try{r.drawImage(e[n],0,0,s.width,s.height),o=s.toDataURL(),e[n].setAttribute("xlink:href",o)}catch(t){console.log("CORS problem! Image can not be used",t)}}return!0},dumpToDataURI:function(t){var e,i,r,o=this.svgRoot,n=window.btoa||x.encode,a=[];if(this.container.hasChildNodes()&&s.Z.exists(this.foreignObjLayer))for(t||this.foreignObjLayer.setAttribute("display","inline");o.nextSibling;)s.Z.concat(a,this._getValuesOfDOMElements(o.nextSibling)),this.foreignObjLayer.appendChild(o.nextSibling);if(this._getImgDataURL(o),o.setAttribute("xmlns","http://www.w3.org/2000/svg"),e=(new XMLSerializer).serializeToString(o),!0!==t)for(r=a.length,i=0;i<r;i++)e=e.replace('id="'+a[i][0]+'"','id="'+a[i][0]+'" value="'+a[i][1]+'"');if(this.isIE&&(e.match(/xmlns="http:\/\/www.w3.org\/2000\/svg"\s+xmlns="http:\/\/www.w3.org\/2000\/svg"/g)||[]).length>1&&(e=e.replace(/xmlns="http:\/\/www.w3.org\/2000\/svg"\s+xmlns="http:\/\/www.w3.org\/2000\/svg"/g,"")),e=e.replace(/&nbsp;/g," "),s.Z.exists(this.foreignObjLayer)&&this.foreignObjLayer.hasChildNodes()){for(;this.foreignObjLayer.firstChild;)this.container.appendChild(this.foreignObjLayer.firstChild);this.foreignObjLayer.setAttribute("display","none")}return"data:image/svg+xml;base64,"+n(unescape(encodeURIComponent(e)))},dumpToCanvas:function(t,e,i,s){var r,o,n,a;return(n=this.container.ownerDocument.getElementById(t)).width=n.width,a=n.getContext("2d"),void 0!==e&&void 0!==i&&(n.style.width=parseFloat(e)+"px",n.style.height=parseFloat(i)+"px",n.setAttribute("width",parseFloat(e)),n.setAttribute("height",parseFloat(i))),o=new Image,r=this.dumpToDataURI(s),o.src=r,"Promise"in window?new Promise((function(t,s){try{o.onload=function(){a.drawImage(o,0,0,e,i),t()}}catch(t){s(t)}})):(o.onload=function(){window.setTimeout((function(){try{a.drawImage(o,0,0,e,i)}catch(t){console.log("screenshots not longer supported on IE")}}),200)},this)},screenshot:function(t,e,i){var r,o,n,a,h,l,c,d,u,p,f,_,m=this.container.ownerDocument,g=this.container.parentNode,b=t.attr.screenshot,v=!1;return"no"===this.type||(c=b.scale*this.container.getBoundingClientRect().width,d=b.scale*this.container.getBoundingClientRect().height,void 0===e||""===e?(v=!0,(a=new Image).style.width=c+"px",a.style.height=d+"px"):(v=!1,a=m.getElementById(e)),v&&((r=m.createElement("div")).style.cssText=b.css,r.style.width=c+"px",r.style.height=d+"px",r.style.zIndex=this.container.style.zIndex+120,r.style.position="absolute",r.style.top=this.container.offsetTop+"px",r.style.left=this.container.offsetLeft+"px"),o=m.createElement("canvas"),n=Math.random().toString(36).slice(2,7),o.setAttribute("id",n),o.setAttribute("width",c),o.setAttribute("height",d),o.style.width=c+"px",o.style.height=c+"px",o.style.display="none",g.appendChild(o),v&&(h=m.createElement("span"),l=m.createTextNode("✖"),h.style.cssText=b.cssButton,h.appendChild(l),h.onclick=function(){r.parentNode.removeChild(r)},r.appendChild(a),r.appendChild(h),g.insertBefore(r,this.container.nextSibling)),u=m.getElementById(this.uniqName("navigationbar")),s.Z.exists(u)&&(p=u.style.display,u.style.display="none",f=this.removeToInsertLater(u)),_=function(){a.src=o.toDataURL("image/png"),g.removeChild(o)},"Promise"in window?this.dumpToCanvas(n,c,d,i).then(_):(this.dumpToCanvas(n,c,d,i),window.setTimeout(_,200)),s.Z.exists(u)&&(u.style.display=p,f())),this}});var A=t.Z.SVGRenderer;t.Z.VMLRenderer=function(e){this.type="vml",this.container=e,this.container.style.overflow="hidden",""===this.container.style.position&&(this.container.style.position="relative"),this.container.onselectstart=function(){return!1},this.resolution=10,s.Z.exists(t.Z.vmlStylesheet)||(e.ownerDocument.namespaces.add("jxgvml","urn:schemas-microsoft-com:vml"),t.Z.vmlStylesheet=this.container.ownerDocument.createStyleSheet(),t.Z.vmlStylesheet.addRule(".jxgvml","behavior:url(#default#VML)"));try{e.ownerDocument.namespaces.jxgvml||e.ownerDocument.namespaces.add("jxgvml","urn:schemas-microsoft-com:vml"),this.createNode=function(t){return e.ownerDocument.createElement("<jxgvml:"+t+' class="jxgvml">')}}catch(t){this.createNode=function(t){return e.ownerDocument.createElement("<"+t+' xmlns="urn:schemas-microsoft.com:vml" class="jxgvml">')}}this.dashArray=["Solid","1 1","ShortDash","Dash","LongDash","ShortDashDot","LongDashDot"]},t.Z.VMLRenderer.prototype=new Z,t.Z.extend(t.Z.VMLRenderer.prototype,{_setAttr:function(e,i,s,r){try{8===this.container.ownerDocument.documentMode?e[i]=s:e.setAttribute(i,s,r)}catch(e){t.Z.debug("_setAttr: "+i+" "+s+"<br>\n")}},updateTicks:function(t){var e,i,r,o,n,a=this.resolution,h=[];for(i=t.ticks.length,e=0;e<i;e++)o=(r=t.ticks[e])[0],n=r[1],s.Z.isNumber(o[0])&&s.Z.isNumber(o[1])&&h.push(" m "+Math.round(a*o[0])+", "+Math.round(a*n[0])+" l "+Math.round(a*o[1])+", "+Math.round(a*n[1])+" ");s.Z.exists(t.rendNode)||(t.rendNode=this.createPrim("path",t.id),this.appendChildPrim(t.rendNode,t.evalVisProp("layer"))),this._setAttr(t.rendNode,"stroked","true"),this._setAttr(t.rendNode,"strokecolor",t.evalVisProp("strokecolor"),1),this._setAttr(t.rendNode,"strokeweight",t.evalVisProp("strokewidth")),this._setAttr(t.rendNodeStroke,"opacity",100*t.evalVisProp("strokeopacity")+"%"),this.updatePathPrim(t.rendNode,h,t.board)},displayCopyright:function(t,e){var i,s;(i=this.createNode("textbox")).style.position="absolute",this._setAttr(i,"id",this.container.id+"_licenseText"),i.style.left=20,i.style.top=2,i.style.fontSize=e,i.style.color="#356AA0",i.style.fontFamily="Arial,Helvetica,sans-serif",this._setAttr(i,"opacity","30%"),i.style.filter="progid:DXImageTransform.Microsoft.Matrix(M11='1.0', sizingMethod='auto expand', enabled = false) progid:DXImageTransform.Microsoft.Alpha(opacity = 30, enabled = true)",s=this.container.ownerDocument.createTextNode(t),i.appendChild(s),this.appendChildPrim(i,0)},drawInternalText:function(t){var e;return(e=this.createNode("textbox")).style.position="absolute",t.rendNodeText=this.container.ownerDocument.createTextNode(""),e.appendChild(t.rendNodeText),this.appendChildPrim(e,9),e.style.filter="progid:DXImageTransform.Microsoft.Matrix(M11='1.0', sizingMethod='auto expand', enabled = false) progid:DXImageTransform.Microsoft.Alpha(opacity = 100, enabled = false)",e},updateInternalText:function(t){var e,i,s,r,n,a,h=t.plaintext,l=this.joinTransforms(t,t.transformations),c=[0,0],d=t.rendNode,u=[],p=t.getAnchorX(),f=t.getAnchorY();if(!isNaN(t.coords.scrCoords[1]+t.coords.scrCoords[2])){for("right"===p?c[0]=1:"middle"===p&&(c[0]=.5),"bottom"===f?c[1]=1:"middle"===f&&(c[1]=.5),u[0]=o.Z.matVecMult(l,[1,t.coords.scrCoords[1]-c[0]*t.size[0],t.coords.scrCoords[2]+(1-c[1])*t.size[1]+this.vOffsetText]),u[0][1]/=u[0][0],u[0][2]/=u[0][0],u[1]=o.Z.matVecMult(l,[1,t.coords.scrCoords[1]+(1-c[0])*t.size[0],t.coords.scrCoords[2]+(1-c[1])*t.size[1]+this.vOffsetText]),u[1][1]/=u[1][0],u[1][2]/=u[1][0],u[2]=o.Z.matVecMult(l,[1,t.coords.scrCoords[1]+(1-c[0])*t.size[0],t.coords.scrCoords[2]-c[1]*t.size[1]+this.vOffsetText]),u[2][1]/=u[2][0],u[2][2]/=u[2][0],u[3]=o.Z.matVecMult(l,[1,t.coords.scrCoords[1]-c[0]*t.size[0],t.coords.scrCoords[2]-c[1]*t.size[1]+this.vOffsetText]),u[3][1]/=u[3][0],u[3][2]/=u[3][0],i=u[0][1],r=u[0][1],s=u[0][2],n=u[0][2],a=1;a<4;a++)i=Math.max(i,u[a][1]),r=Math.min(r,u[a][1]),s=Math.max(s,u[a][2]),n=Math.min(n,u[a][2]);e=1===c[0]?Math.floor(t.board.canvasWidth-i):Math.floor(r),t.visPropOld.left!==p+e&&(1===c[0]?(t.rendNode.style.right=e+"px",t.rendNode.style.left="auto"):(t.rendNode.style.left=e+"px",t.rendNode.style.right="auto"),t.visPropOld.left=p+e),e=1===c[1]?Math.floor(t.board.canvasHeight-s):Math.floor(n),t.visPropOld.top!==f+e&&(1===c[1]?(t.rendNode.style.bottom=e+"px",t.rendNode.style.top="auto"):(t.rendNode.style.top=e+"px",t.rendNode.style.bottom="auto"),t.visPropOld.top=f+e)}t.htmlStr!==h&&(t.rendNodeText.data=h,t.htmlStr=h),d.filters.item(0).M11=l[1][1],d.filters.item(0).M12=l[1][2],d.filters.item(0).M21=l[2][1],d.filters.item(0).M22=l[2][2],d.filters.item(0).enabled=!0},drawImage:function(t){var e;(e=this.container.ownerDocument.createElement("img")).style.position="absolute",this._setAttr(e,"id",this.container.id+"_"+t.id),this.container.appendChild(e),this.appendChildPrim(e,t.evalVisProp("layer")),e.style.filter="progid:DXImageTransform.Microsoft.Matrix(M11='1.0', sizingMethod='auto expand') progid:DXImageTransform.Microsoft.Alpha(opacity = 100, enabled = false)",t.rendNode=e,this.updateImage(t)},transformRect:function(t,e){var i,s,r,n,a,h,l=t.rendNode,c=[];if(e.length>0){for(i=this.joinTransforms(t,e),c[0]=o.Z.matVecMult(i,t.coords.scrCoords),c[0][1]/=c[0][0],c[0][2]/=c[0][0],c[1]=o.Z.matVecMult(i,[1,t.coords.scrCoords[1]+t.size[0],t.coords.scrCoords[2]]),c[1][1]/=c[1][0],c[1][2]/=c[1][0],c[2]=o.Z.matVecMult(i,[1,t.coords.scrCoords[1]+t.size[0],t.coords.scrCoords[2]-t.size[1]]),c[2][1]/=c[2][0],c[2][2]/=c[2][0],c[3]=o.Z.matVecMult(i,[1,t.coords.scrCoords[1],t.coords.scrCoords[2]-t.size[1]]),c[3][1]/=c[3][0],c[3][2]/=c[3][0],s=c[0][1],n=c[0][1],r=c[0][2],a=c[0][2],h=1;h<4;h++)s=Math.max(s,c[h][1]),n=Math.min(n,c[h][1]),r=Math.max(r,c[h][2]),a=Math.min(a,c[h][2]);l.style.left=Math.floor(n)+"px",l.style.top=Math.floor(a)+"px",l.filters.item(0).M11=i[1][1],l.filters.item(0).M12=i[1][2],l.filters.item(0).M21=i[2][1],l.filters.item(0).M22=i[2][2],l.filters.item(0).enabled=!0}},updateImageURL:function(t){var e=t.eval(t.url);this._setAttr(t.rendNode,"src",e)},appendChildPrim:function(t,e){return s.Z.exists(e)||(e=0),t.style.zIndex=e,this.container.appendChild(t),t},appendNodesToElement:function(t,e){"shape"!==e&&"path"!==e&&"polygon"!==e||(t.rendNodePath=this.getElementById(t.id+"_path")),t.rendNodeFill=this.getElementById(t.id+"_fill"),t.rendNodeStroke=this.getElementById(t.id+"_stroke"),t.rendNodeShadow=this.getElementById(t.id+"_shadow"),t.rendNode=this.getElementById(t.id)},createPrim:function(t,e){var i,s,r=this.createNode("fill"),o=this.createNode("stroke"),n=this.createNode("shadow");return this._setAttr(r,"id",this.container.id+"_"+e+"_fill"),this._setAttr(o,"id",this.container.id+"_"+e+"_stroke"),this._setAttr(n,"id",this.container.id+"_"+e+"_shadow"),"circle"===t||"ellipse"===t?((i=this.createNode("oval")).appendChild(r),i.appendChild(o),i.appendChild(n)):"polygon"===t||"path"===t||"shape"===t||"line"===t?((i=this.createNode("shape")).appendChild(r),i.appendChild(o),i.appendChild(n),s=this.createNode("path"),this._setAttr(s,"id",this.container.id+"_"+e+"_path"),i.appendChild(s)):((i=this.createNode(t)).appendChild(r),i.appendChild(o),i.appendChild(n)),i.style.position="absolute",i.style.left="0px",i.style.top="0px",this._setAttr(i,"id",this.container.id+"_"+e),i},remove:function(t){s.Z.exists(t)&&t.removeNode(!0)},makeArrows:function(t){var e,i=t.evalVisProp("firstarrow"),r=t.evalVisProp("lastarrow");t.visPropOld.firstarrow===i&&t.visPropOld.lastarrow===r||(i?(e=t.rendNodeStroke,this._setAttr(e,"startarrow","block"),this._setAttr(e,"startarrowlength","long")):(e=t.rendNodeStroke,s.Z.exists(e)&&this._setAttr(e,"startarrow","none")),r?(e=t.rendNodeStroke,this._setAttr(e,"id",this.container.id+"_"+t.id+"stroke"),this._setAttr(e,"endarrow","block"),this._setAttr(e,"endarrowlength","long")):(e=t.rendNodeStroke,s.Z.exists(e)&&this._setAttr(e,"endarrow","none")),t.visPropOld.firstarrow=i,t.visPropOld.lastarrow=r)},updateEllipsePrim:function(t,e,i,s,r){t.style.left=Math.floor(e-s)+"px",t.style.top=Math.floor(i-r)+"px",t.style.width=Math.floor(2*Math.abs(s))+"px",t.style.height=Math.floor(2*Math.abs(r))+"px"},updateLinePrim:function(t,e,i,s,r,o){var n,a=this.resolution;isNaN(e+i+s+r)||(n=["m ",Math.floor(a*e),", ",Math.floor(a*i)," l ",Math.floor(a*s),", ",Math.floor(a*r)],this.updatePathPrim(t,n,o))},updatePathPrim:function(t,e,i){var s=i.canvasWidth,r=i.canvasHeight;e.length<=0&&(e=["m 0,0"]),t.style.width=s,t.style.height=r,this._setAttr(t,"coordsize",[Math.floor(this.resolution*s),Math.floor(this.resolution*r)].join(",")),this._setAttr(t,"path",e.join(""))},updatePathStringPoint:function(t,e,i){var s=[],r=Math.round,o=t.coords.scrCoords,n=e*Math.sqrt(3)*.5,a=.5*e,h=this.resolution;return"x"===i?s.push([" m ",r(h*(o[1]-e)),", ",r(h*(o[2]-e))," l ",r(h*(o[1]+e)),", ",r(h*(o[2]+e))," m ",r(h*(o[1]+e)),", ",r(h*(o[2]-e))," l ",r(h*(o[1]-e)),", ",r(h*(o[2]+e))].join("")):"+"===i?s.push([" m ",r(h*(o[1]-e)),", ",r(h*o[2])," l ",r(h*(o[1]+e)),", ",r(h*o[2])," m ",r(h*o[1]),", ",r(h*(o[2]-e))," l ",r(h*o[1]),", ",r(h*(o[2]+e))].join("")):"<>"===i||"<<>>"===i?("<<>>"===i&&(e*=1.41),s.push([" m ",r(h*(o[1]-e)),", ",r(h*o[2])," l ",r(h*o[1]),", ",r(h*(o[2]+e))," l ",r(h*(o[1]+e)),", ",r(h*o[2])," l ",r(h*o[1]),", ",r(h*(o[2]-e))," x e "].join(""))):"^"===i?s.push([" m ",r(h*o[1]),", ",r(h*(o[2]-e))," l ",r(h*(o[1]-n)),", ",r(h*(o[2]+a))," l ",r(h*(o[1]+n)),", ",r(h*(o[2]+a))," x e "].join("")):"v"===i?s.push([" m ",r(h*o[1]),", ",r(h*(o[2]+e))," l ",r(h*(o[1]-n)),", ",r(h*(o[2]-a))," l ",r(h*(o[1]+n)),", ",r(h*(o[2]-a))," x e "].join("")):">"===i?s.push([" m ",r(h*(o[1]+e)),", ",r(h*o[2])," l ",r(h*(o[1]-a)),", ",r(h*(o[2]-n))," l ",r(h*(o[1]-a)),", ",r(h*(o[2]+n))," l ",r(h*(o[1]+e)),", ",r(h*o[2])].join("")):"<"===i&&s.push([" m ",r(h*(o[1]-e)),", ",r(h*o[2])," l ",r(h*(o[1]+a)),", ",r(h*(o[2]-n))," l ",r(h*(o[1]+a)),", ",r(h*(o[2]+n))," x e "].join("")),s},updatePathStringPrim:function(t){var e,i,s=[],r=this.resolution,o=Math.round,n=" m ",a=n,h=Math.min(t.numberPoints,8192);if(t.numberPoints<=0)return"";if(h=Math.min(h,t.points.length),1===t.bezierDegree)for(e=0;e<h;e++)i=t.points[e].scrCoords,isNaN(i[1])||isNaN(i[2])?a=n:(i[1]>2e4?i[1]=2e4:i[1]<-2e4&&(i[1]=-2e4),i[2]>2e4?i[2]=2e4:i[2]<-2e4&&(i[2]=-2e4),s.push([a,o(r*i[1]),", ",o(r*i[2])].join("")),a=" l ");else if(3===t.bezierDegree)for(e=0;e<h;)i=t.points[e].scrCoords,isNaN(i[1])||isNaN(i[2])?a=n:(s.push([a,o(r*i[1]),", ",o(r*i[2])].join(""))," c "===a&&(e+=1,i=t.points[e].scrCoords,s.push([" ",o(r*i[1]),", ",o(r*i[2])].join("")),e+=1,i=t.points[e].scrCoords,s.push([" ",o(r*i[1]),", ",o(r*i[2])].join(""))),a=" c "),e+=1;return s.push(" e"),s},updatePathStringBezierPrim:function(t){var e,i,s,r,o,n,a=[],h=t.evalVisProp("strokewidth"),l=this.resolution,c=Math.round,u=" m ",p=u,f="plot"!==t.evalVisProp("curvetype"),_=Math.min(t.numberPoints,8192);if(t.numberPoints<=0)return"";for(f&&t.board.options.curve.RDPsmoothing&&(t.points=d.Z.RamerDouglasPeucker(t.points,1)),_=Math.min(_,t.points.length),i=1;i<3;i++)for(p=u,e=0;e<_;e++)r=t.points[e].scrCoords,isNaN(r[1])||isNaN(r[2])?p=u:(r[1]>2e4?r[1]=2e4:r[1]<-2e4&&(r[1]=-2e4),r[2]>2e4?r[2]=2e4:r[2]<-2e4&&(r[2]=-2e4),p===u?a.push([p,c(l*r[1])," ",c(l*r[2])].join("")):(s=2*i,a.push([p,c(l*(o+.333*(r[1]-o)+h*(s*Math.random()-i)))," ",c(l*(n+.333*(r[2]-n)+h*(s*Math.random()-i)))," ",c(l*(o+.666*(r[1]-o)+h*(s*Math.random()-i)))," ",c(l*(n+.666*(r[2]-n)+h*(s*Math.random()-i)))," ",c(l*r[1])," ",c(l*r[2])].join(""))),p=" c ",o=r[1],n=r[2]);return a.push(" e"),a},updatePolygonPrim:function(t,e){var i,s,r=e.vertices.length,o=this.resolution,n=[];if(this._setAttr(t,"stroked","false"),s=e.vertices[0].coords.scrCoords,!isNaN(s[1]+s[2])){for(n.push(["m ",Math.floor(o*s[1]),",",Math.floor(o*s[2])," l "].join("")),i=1;i<r-1;i++){if(!e.vertices[i].isReal)return void this.updatePathPrim(t,"",e.board);if(s=e.vertices[i].coords.scrCoords,isNaN(s[1]+s[2]))return;n.push(Math.floor(o*s[1])+","+Math.floor(o*s[2])),i<r-2&&n.push(", ")}n.push(" x e"),this.updatePathPrim(t,n,e.board)}},updateRectPrim:function(t,e,i,s,r){t.style.left=Math.floor(e)+"px",t.style.top=Math.floor(i)+"px",s>=0&&(t.style.width=s+"px"),r>=0&&(t.style.height=r+"px")},setPropertyPrim:function(t,e,i){var s,r="";switch(e){case"stroke":r="strokecolor";break;case"stroke-width":r="strokeweight";break;case"stroke-dasharray":r="dashstyle"}""!==r&&(s=i,this._setAttr(t,r,s))},display:function(t,e){t&&t.rendNode&&(t.visPropOld.visible=e,t.rendNode.style.visibility=e?"inherit":"hidden")},show:function(e){t.Z.deprecated("Board.renderer.show()","Board.renderer.display()"),e&&e.rendNode&&(e.rendNode.style.visibility="inherit")},hide:function(e){t.Z.deprecated("Board.renderer.hide()","Board.renderer.display()"),e&&e.rendNode&&(e.rendNode.style.visibility="hidden")},setDashStyle:function(t,e){var i;e.dash>=0&&(i=t.rendNodeStroke,this._setAttr(i,"dashstyle",this.dashArray[e.dash]))},setGradient:function(t){var e=t.rendNodeFill,i=t.evalVisProp("gradient");"linear"===i?(this._setAttr(e,"type","gradient"),this._setAttr(e,"color2",t.evalVisProp("gradientsecondcolor")),this._setAttr(e,"opacity2",t.evalVisProp("gradientsecondopacity")),this._setAttr(e,"angle",t.evalVisProp("gradientangle"))):"radial"===i?(this._setAttr(e,"type","gradientradial"),this._setAttr(e,"color2",t.evalVisProp("gradientsecondcolor")),this._setAttr(e,"opacity2",t.evalVisProp("gradientsecondopacity")),this._setAttr(e,"focusposition",100*t.evalVisProp("gradientpositionx")+"%,"+100*t.evalVisProp("gradientpositiony")+"%"),this._setAttr(e,"focussize","0,0")):this._setAttr(e,"type","solid")},setObjectFillColor:function(t,e,r){var o,n,a,h=e,l=r,c=t.rendNode;l=l>0?l:0,t.visPropOld.fillcolor===h&&t.visPropOld.fillopacity===l||(s.Z.exists(h)&&!1!==h&&(9!==h.length?(o=h,a=l):(o=(n=S.Z.rgba2rgbo(h))[0],a=l*n[1]),"none"===o||!1===o?this._setAttr(t.rendNode,"filled","false"):(this._setAttr(t.rendNode,"filled","true"),this._setAttr(t.rendNode,"fillcolor",o),s.Z.exists(a)&&t.rendNodeFill&&this._setAttr(t.rendNodeFill,"opacity",100*a+"%")),t.type===i.Z.OBJECT_TYPE_IMAGE&&c.filters.length>1&&(c.filters.item(1).opacity=Math.round(100*a),c.filters.item(1).enabled=!0)),t.visPropOld.fillcolor=h,t.visPropOld.fillopacity=l)},setObjectStrokeColor:function(t,e,r){var o,n,a,h,l=e,c=r,d=t.rendNode;c=c>0?c:0,t.visPropOld.strokecolor===l&&t.visPropOld.strokeopacity===c||(s.Z.exists(l)&&!1!==l&&(9!==l.length?(o=l,a=c):(o=(n=e.rgba2rgbo(l))[0],a=c*n[1]),t.elementClass===i.Z.OBJECT_CLASS_TEXT?(d.filters.length>1&&(d.filters.item(1).opacity=Math.round(100*a),d.filters.item(1).enabled=!0),d.style.color=o):(!1!==o&&(this._setAttr(d,"stroked","true"),this._setAttr(d,"strokecolor",o)),h=t.rendNodeStroke,s.Z.exists(a)&&t.type!==i.Z.OBJECT_TYPE_IMAGE&&this._setAttr(h,"opacity",100*a+"%"))),t.visPropOld.strokecolor=l,t.visPropOld.strokeopacity=c)},setObjectStrokeWidth:function(t,e){var i,r=s.Z.evaluate(e);isNaN(r)||t.visPropOld.strokewidth===r||(i=t.rendNode,this.setPropertyPrim(i,"stroked","true"),s.Z.exists(r)&&(this.setPropertyPrim(i,"stroke-width",r),0===r&&s.Z.exists(t.rendNodeStroke)&&this._setAttr(i,"stroked","false")),t.visPropOld.strokewidth=r)},setShadow:function(t){var e=t.rendNodeShadow,i=t.evalVisProp("shadow");e&&t.visPropOld.shadow!==i&&(i?(this._setAttr(e,"On","True"),this._setAttr(e,"Offset","3pt,3pt"),this._setAttr(e,"Opacity","60%"),this._setAttr(e,"Color","#aaaaaa")):this._setAttr(e,"On","False"),t.visPropOld.shadow=i)},suspendRedraw:function(){this.container.style.display="none"},unsuspendRedraw:function(){this.container.style.display=""}});var D=t.Z.VMLRenderer,k="0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz".split("");t.Z.Util=t.Z.Util||{},t.Z.Util.genUUID=function(t){var e,i,s=[],r=0;for(""!==(t=t||"")&&"-"!==t.slice(t.length-1)&&(t+="-"),i=0;i<36;i++)8===i||13===i||18===i||23===i?s[i]="-":14===i?s[i]="4":(r<=2&&(r=33554432+16777216*Math.random()|0),e=15&r,r>>=4,s[i]=k[19===i?3&e|8:e]);return t+s.join("")};var R=t.Z.Util;t.Z.CanvasRenderer=function(i,s){if(this.type="canvas",this.canvasRoot=null,this.suspendHandle=null,this.canvasId=R.genUUID(),this.canvasNamespace=null,e.Z.isBrowser)this.container=i,this.container.style.MozUserSelect="none",this.container.style.userSelect="none",this.container.style.overflow="hidden",""===this.container.style.position&&(this.container.style.position="relative"),this.container.innerHTML=['<canvas id="',this.canvasId,'" width="',s.width,'px" height="',s.height,'px"></canvas>'].join(""),this.canvasRoot=this.container.ownerDocument.getElementById(this.canvasId),this.canvasRoot.style.display="block",this.context=this.canvasRoot.getContext("2d");else if(e.Z.isNode())try{this.canvasRoot=t.Z.createCanvas(500,500),this.context=this.canvasRoot.getContext("2d")}catch(t){throw new Error('JXG.createCanvas not available.\nInstall the npm package `canvas`\nand call:\n    import { createCanvas } from "canvas.js";\n    JXG.createCanvas = createCanvas;\n')}},t.Z.CanvasRenderer.prototype=new Z,t.Z.extend(t.Z.CanvasRenderer.prototype,{_drawPolygon:function(t,e,i){var s,r=t.length,o=this.context;if(r>0){if(i&&(o.lineWidth=0),o.beginPath(),o.moveTo(t[0][0],t[0][1]),1===e)for(s=1;s<r;s++)o.lineTo(t[s][0],t[s][1]);else for(s=1;s<r;s+=3)o.bezierCurveTo(t[s][0],t[s][1],t[s+1][0],t[s+1][1],t[s+2][0],t[s+2][1]);i?(o.lineTo(t[0][0],t[0][1]),o.closePath(),o.fill("evenodd")):o.stroke()}},_fill:function(t){var e=this.context;e.save(),this._setColor(t,"fill")&&e.fill("evenodd"),e.restore()},_rotatePoint:function(t,e,i){return[e*Math.cos(t)-i*Math.sin(t),e*Math.sin(t)+i*Math.cos(t)]},_rotateShape:function(t,e){var i,s=[],r=t.length;if(r<=0)return t;for(i=0;i<r;i++)s.push(this._rotatePoint(e,t[i][0],t[i][1]));return s},updateGradientAngle:function(t,e){var s,r,o,n,a,h,l,c,d,p,f,_,m=1,g=Math.cos(-e),b=Math.sin(-e),v=t.getBoundingBox();return Math.abs(g)>Math.abs(b)?m/=Math.abs(g):m/=Math.abs(b),g>=0?(o=0,n=g*m):(o=-g*m,n=0),b>=0?(a=0,h=b*m):(a=-b*m,h=0),s=new u.Z(i.Z.COORDS_BY_USER,[v[0],v[1]],t.board),f=(r=new u.Z(i.Z.COORDS_BY_USER,[v[2],v[3]],t.board)).scrCoords[1]-s.scrCoords[1],_=r.scrCoords[2]-s.scrCoords[2],l=s.scrCoords[1]+f*o,d=s.scrCoords[2]+_*a,c=s.scrCoords[1]+f*n,p=s.scrCoords[2]+_*h,this.context.createLinearGradient(l,d,c,p)},updateGradientCircle:function(t,e,s,r,o,n,a){var h,l,c,d,p,f,_,m,g,b,v=t.getBoundingBox();return h=new u.Z(i.Z.COORDS_BY_USER,[v[0],v[1]],t.board),g=(l=new u.Z(i.Z.COORDS_BY_USER,[v[2],v[3]],t.board)).scrCoords[1]-h.scrCoords[1],b=h.scrCoords[2]-l.scrCoords[2],c=h.scrCoords[1]+g*e,d=l.scrCoords[2]+b*s,f=h.scrCoords[1]+g*o,_=l.scrCoords[2]+b*n,p=r*(g+b)*.5,m=a*(g+b)*.5,this.context.createRadialGradient(f,_,m,c,d,p)},updateGradient:function(t){var e,i,s=t.evalVisProp("gradient");return e=t.evalVisProp("fillcolor"),"linear"===s?i=this.updateGradientAngle(t,t.evalVisProp("gradientangle")):"radial"===s&&(i=this.updateGradientCircle(t,t.evalVisProp("gradientcx"),t.evalVisProp("gradientcy"),t.evalVisProp("gradientr"),t.evalVisProp("gradientfx"),t.evalVisProp("gradientfy"),t.evalVisProp("gradientfr"))),"none"!==e&&""!==e&&!1!==e&&(i.addColorStop(t.evalVisProp("gradientstartoffset"),e),i.addColorStop(t.evalVisProp("gradientendoffset"),t.evalVisProp("gradientsecondcolor"))),i},_setColor:function(t,e,i){var s,r,o,n,a,h,l,c,d,u=!0;return e=e||"stroke",i=i||e,r=this._getHighlighted(t),"none"!==(n=t.evalVisProp(r+e+"color"))&&""!==n&&!1!==n?(l=(l=t.evalVisProp(r+e+"opacity"))>0?l:0,9!==n.length?(h=n,c=l):(h=(a=S.Z.rgba2rgbo(n))[0],c=l*a[1]),this.context.globalAlpha=c,this.context[i+"Style"]=h):u=!1,"stroke"===e||"linear"!==(d=t.evalVisProp("gradient"))&&"radial"!==d?(o=parseFloat(t.evalVisProp(r+"strokewidth")),"stroke"!==e||isNaN(o)||(0===o?this.context.globalAlpha=0:this.context.lineWidth=o),s=t.evalVisProp("linecap"),"stroke"===e&&void 0!==s&&""!==s&&(this.context.lineCap=s),u):(this.context.globalAlpha=c,this.context[i+"Style"]=this.updateGradient(t),u)},_stroke:function(t){var e=this.context,i=t.evalVisProp("dash"),s=t.evalVisProp("dashscale")?.5*t.evalVisProp("strokewidth"):1;e.save(),i>0?e.setLineDash&&e.setLineDash(this.dashArray[i-1].map((function(t){return t*s}))):this.context.lineDashArray=[],this._setColor(t,"stroke")&&e.stroke(),e.restore()},_translateShape:function(t,e,i){var s,r=[],o=t.length;if(o<=0)return t;for(s=0;s<o;s++)r.push([t[s][0]+e,t[s][1]+i]);return r},drawPoint:function(t){var e=t.evalVisProp("face"),i=t.evalVisProp("size"),s=t.coords.scrCoords,r=i*Math.sqrt(3)*.5,o=.5*i,n=parseFloat(t.evalVisProp("strokewidth"))/2,a=this.context;if(t.visPropCalc.visible)switch(e){case"cross":case"x":a.beginPath(),a.moveTo(s[1]-i,s[2]-i),a.lineTo(s[1]+i,s[2]+i),a.moveTo(s[1]+i,s[2]-i),a.lineTo(s[1]-i,s[2]+i),a.lineCap="round",a.lineJoin="round",a.closePath(),this._stroke(t);break;case"circle":case"o":a.beginPath(),a.arc(s[1],s[2],i+1+n,0,2*Math.PI,!1),a.closePath(),this._fill(t),this._stroke(t);break;case"square":case"[]":if(i<=0)break;a.save(),this._setColor(t,"stroke","fill")&&a.fillRect(s[1]-i-n,s[2]-i-n,2*i+3*n,2*i+3*n),a.restore(),a.save(),this._setColor(t,"fill"),a.fillRect(s[1]-i+n,s[2]-i+n,2*i-n,2*i-n),a.restore();break;case"plus":case"+":a.beginPath(),a.moveTo(s[1]-i,s[2]),a.lineTo(s[1]+i,s[2]),a.moveTo(s[1],s[2]-i),a.lineTo(s[1],s[2]+i),a.lineCap="round",a.lineJoin="round",a.closePath(),this._stroke(t);break;case"divide":case"|":a.beginPath(),a.moveTo(s[1],s[2]-i),a.lineTo(s[1],s[2]+i),a.lineCap="round",a.lineJoin="round",a.closePath(),this._stroke(t);break;case"minus":case"-":a.beginPath(),a.moveTo(s[1]-i,s[2]),a.lineTo(s[1]+i,s[2]),a.lineCap="round",a.lineJoin="round",a.closePath(),this._stroke(t);break;case"diamond2":case"<<>>":i*=1.41;case"diamond":case"<>":a.beginPath(),a.moveTo(s[1]-i,s[2]),a.lineTo(s[1],s[2]+i),a.lineTo(s[1]+i,s[2]),a.lineTo(s[1],s[2]-i),a.closePath(),this._fill(t),this._stroke(t);break;case"triangleup":case"A":case"a":case"^":a.beginPath(),a.moveTo(s[1],s[2]-i),a.lineTo(s[1]-r,s[2]+o),a.lineTo(s[1]+r,s[2]+o),a.closePath(),this._fill(t),this._stroke(t);break;case"triangledown":case"v":a.beginPath(),a.moveTo(s[1],s[2]+i),a.lineTo(s[1]-r,s[2]-o),a.lineTo(s[1]+r,s[2]-o),a.closePath(),this._fill(t),this._stroke(t);break;case"triangleleft":case"<":a.beginPath(),a.moveTo(s[1]-i,s[2]),a.lineTo(s[1]+o,s[2]-r),a.lineTo(s[1]+o,s[2]+r),a.closePath(),this._fill(t),this._stroke(t);break;case"triangleright":case">":a.beginPath(),a.moveTo(s[1]+i,s[2]),a.lineTo(s[1]-o,s[2]-r),a.lineTo(s[1]-o,s[2]+r),a.closePath(),this._fill(t),this._stroke(t)}},updatePoint:function(t){this.drawPoint(t)},drawArrows:function(t,e,s,r,o){var n,a,h,l,c,d,u,p,f,_,m,g,b,v,Z,C,y,P,E,x,O=this.context,M=1,w=1,S=1,T=o.evFirst,N=o.evLast;if("none"!==t.evalVisProp("strokecolor")&&(T||N)){if(t.elementClass===i.Z.OBJECT_CLASS_LINE)n=e.scrCoords[1],a=e.scrCoords[2],h=s.scrCoords[1],l=s.scrCoords[2],E=x=Math.atan2(l-a,h-n);else{if(n=t.points[0].scrCoords[1],a=t.points[0].scrCoords[2],(P=t.points.length-1)<1)return;h=t.points[t.points.length-1].scrCoords[1],l=t.points[t.points.length-1].scrCoords[2],v=t.points[1].scrCoords[1]-t.points[0].scrCoords[1],Z=t.points[1].scrCoords[2]-t.points[0].scrCoords[2],C=t.points[P].scrCoords[1]-t.points[P-1].scrCoords[1],y=t.points[P].scrCoords[2]-t.points[P-1].scrCoords[2],T&&(E=Math.atan2(Z,v)),N&&(x=Math.atan2(y,C))}if(d=t.evalVisProp(r+"strokewidth"),T)if(c=d*o.sizeFirst,f=M=o.typeFirst,2===M)p=[[c,.5*-c],[0,0],[c,.5*c],[.5*c,0]];else if(3===M)p=[[c/3,.5*-c],[0,.5*-c],[0,.5*c],[c/3,.5*c]];else if(4===M)for(c/=10,w=3,b=(p=[[10,3.31],[6.47,3.84],[2.87,4.5],[0,6.63],[.67,5.52],[1.33,4.42],[2,3.31],[1.33,2.21],[.67,1.1],[0,0],[2.87,2.13],[6.47,2.79],[10,3.31]]).length,g=0;g<b;g++)p[g][0]*=-c,p[g][1]*=c,p[g][0]+=10*c,p[g][1]-=3.31*c;else if(5===M)for(c/=10,w=3,b=(p=[[10,3.28],[6.61,4.19],[3.19,5.07],[0,6.55],[.62,5.56],[1,4.44],[1,3.28],[1,2.11],[.62,.99],[0,0],[3.19,1.49],[6.61,2.37],[10,3.28]]).length,g=0;g<b;g++)p[g][0]*=-c,p[g][1]*=c,p[g][0]+=10*c,p[g][1]-=3.28*c;else if(6===M)for(c/=10,w=3,b=(p=[[10,2.84],[6.61,3.59],[3.21,4.35],[0,5.68],[.33,4.73],[.67,3.78],[1,2.84],[.67,1.89],[.33,.95],[0,0],[3.21,1.33],[6.61,2.09],[10,2.84]]).length,g=0;g<b;g++)p[g][0]*=-c,p[g][1]*=c,p[g][0]+=10*c,p[g][1]-=2.84*c;else if(7===M)for(c=d,w=3,b=(p=[[0,10.39],[2.01,6.92],[5.96,5.2],[10,5.2],[5.96,5.2],[2.01,3.47],[0,0]]).length,g=0;g<b;g++)p[g][0]*=-c,p[g][1]*=c,p[g][0]+=10*c,p[g][1]-=5.2*c;else p=[[c,.5*-c],[0,0],[c,.5*c]];if(N)if(c=d*o.sizeLast,_=M=o.typeLast,2===M)u=[[-c,.5*-c],[0,0],[-c,.5*c],[.5*-c,0]];else if(3===M)u=[[-c/3,.5*-c],[0,.5*-c],[0,.5*c],[-c/3,.5*c]];else if(4===M)for(c/=10,S=3,b=(u=[[10,3.31],[6.47,3.84],[2.87,4.5],[0,6.63],[.67,5.52],[1.33,4.42],[2,3.31],[1.33,2.21],[.67,1.1],[0,0],[2.87,2.13],[6.47,2.79],[10,3.31]]).length,g=0;g<b;g++)u[g][0]*=c,u[g][1]*=c,u[g][0]-=10*c,u[g][1]-=3.31*c;else if(5===M)for(c/=10,S=3,b=(u=[[10,3.28],[6.61,4.19],[3.19,5.07],[0,6.55],[.62,5.56],[1,4.44],[1,3.28],[1,2.11],[.62,.99],[0,0],[3.19,1.49],[6.61,2.37],[10,3.28]]).length,g=0;g<b;g++)u[g][0]*=c,u[g][1]*=c,u[g][0]-=10*c,u[g][1]-=3.28*c;else if(6===M)for(c/=10,S=3,b=(u=[[10,2.84],[6.61,3.59],[3.21,4.35],[0,5.68],[.33,4.73],[.67,3.78],[1,2.84],[.67,1.89],[.33,.95],[0,0],[3.21,1.33],[6.61,2.09],[10,2.84]]).length,g=0;g<b;g++)u[g][0]*=c,u[g][1]*=c,u[g][0]-=10*c,u[g][1]-=2.84*c;else if(7===M)for(c=d,S=3,b=(u=[[0,10.39],[2.01,6.92],[5.96,5.2],[10,5.2],[5.96,5.2],[2.01,3.47],[0,0]]).length,g=0;g<b;g++)u[g][0]*=c,u[g][1]*=c,u[g][0]-=10*c,u[g][1]-=5.2*c;else u=[[-c,.5*-c],[0,0],[-c,.5*c]];O.save(),this._setColor(t,"stroke","fill")&&(this._setColor(t,"stroke"),T&&(m=7!==f,this._drawPolygon(this._translateShape(this._rotateShape(p,E),n,a),w,m)),N&&(m=7!==_,this._drawPolygon(this._translateShape(this._rotateShape(u,x),h,l),S,m))),O.restore()}},drawLine:function(t){var e,s,r,o,n,a=new u.Z(i.Z.COORDS_BY_USER,t.point1.coords.usrCoords,t.board),l=new u.Z(i.Z.COORDS_BY_USER,t.point2.coords.usrCoords,t.board),c=null;t.visPropCalc.visible&&(r=this._getHighlighted(t),o=t.evalVisProp(r+"strokewidth"),((n=this.getArrowHeadData(t,o,r)).evFirst||n.evLast)&&(c=-4),h.Z.calcStraight(t,a,l,c),this.handleTouchpoints(t,a,l,n),e=new u.Z(i.Z.COORDS_BY_USER,a.usrCoords,t.board),s=new u.Z(i.Z.COORDS_BY_USER,l.usrCoords,t.board),this.getPositionArrowHead(t,a,l,n),this.context.beginPath(),this.context.moveTo(a.scrCoords[1],a.scrCoords[2]),this.context.lineTo(l.scrCoords[1],l.scrCoords[2]),this._stroke(t),(n.evFirst||n.evLast)&&this.drawArrows(t,e,s,r,n))},updateLine:function(t){this.drawLine(t)},drawTicks:function(){},updateTicks:function(t){var e,i,s,r,o,n,a=t.ticks.length,h=this.context;for(h.beginPath(),e=0;e<a;e++)for(s=(i=t.ticks[e])[0],r=i[1],o=s.length,h.moveTo(s[0],r[0]),n=1;n<o;++n)h.lineTo(s[n],r[n]);h.lineCap="round",this._stroke(t)},drawCurve:function(t){var e,i,s;t.evalVisProp("handdrawing")?this.updatePathStringBezierPrim(t):this.updatePathStringPrim(t),t.numberPoints>1&&(e=this._getHighlighted(t),i=t.evalVisProp(e+"strokewidth"),((s=this.getArrowHeadData(t,i,e)).evFirst||s.evLast)&&this.drawArrows(t,null,null,e,s))},updateCurve:function(t){this.drawCurve(t)},drawEllipse:function(t){var e=t.center.coords.scrCoords[1],i=t.center.coords.scrCoords[2],s=t.board.unitX,r=t.board.unitY,o=2*t.Radius(),n=2*t.Radius(),a=o*s,h=n*r,l=e-a/2,c=i-h/2,d=a/2*.5522848,u=h/2*.5522848,p=l+a,f=c+h,_=l+a/2,m=c+h/2,g=this.context;o>0&&n>0&&!isNaN(e+i)&&(g.beginPath(),g.moveTo(l,m),g.bezierCurveTo(l,m-u,_-d,c,_,c),g.bezierCurveTo(_+d,c,p,m-u,p,m),g.bezierCurveTo(p,m+u,_+d,f,_,f),g.bezierCurveTo(_-d,f,l,m+u,l,m),g.closePath(),this._fill(t),this._stroke(t))},updateEllipse:function(t){return this.drawEllipse(t)},displayCopyright:function(t,e){var i=this.context,s=4+1.8*e,r=6+e;i.save(),i.font=e+"px Arial",i.globalAlpha=.2,i.lineWidth=.5,i.fillText(t+".",s,r),i.restore()},displayLogo:function(t,e,i){var r=this.context,o=1.5*e;s.Z.exists(i._logo_image)||(i._logo_image=new Image,i._logo_image.src=t),i._logo_image.onload=function(){r.save(),r.globalAlpha=.2,r.drawImage(i._logo_image,5,5,o,o),r.restore()},r.save(),r.globalAlpha=.2,r.drawImage(i._logo_image,5,5,o,o),r.restore()},drawInternalText:function(t){var e=t.evalVisProp("fontsize"),i=t.evalVisProp("fontunit"),s=t.getAnchorX(),r=t.getAnchorY(),o=this.context;return o.save(),this._setColor(t,"stroke","fill")&&!isNaN(t.coords.scrCoords[1]+t.coords.scrCoords[2])&&(o.font=(e>0?e:0)+i+" Arial",this.transformRect(t,t.transformations),"left"===s?o.textAlign="left":"right"===s?o.textAlign="right":"middle"===s&&(o.textAlign="center"),"bottom"===r?o.textBaseline="bottom":"top"===r?o.textBaseline="top":"middle"===r&&(o.textBaseline="middle"),o.fillText(t.plaintext,t.coords.scrCoords[1],t.coords.scrCoords[2])),o.restore(),null},updateInternalText:function(t){this.drawInternalText(t)},setObjectStrokeColor:function(t,e,r){var o,n,a,h,l=e,c=r;c=c>0?c:0,t.visPropOld.strokecolor===l&&t.visPropOld.strokeopacity===c||(s.Z.exists(l)&&!1!==l&&(9!==l.length?(o=l,a=c):(o=(n=S.Z.rgba2rgbo(l))[0],a=c*n[1]),h=t.rendNode,t.elementClass===i.Z.OBJECT_CLASS_TEXT&&"html"===t.evalVisProp("display")&&(h.style.color=o,h.style.opacity=a)),t.visPropOld.strokecolor=l,t.visPropOld.strokeopacity=c)},drawImage:function(t){t.rendNode=new Image,t._src="",this.updateImage(t)},updateImage:function(t){var e=this.context,i=t.evalVisProp("fillopacity"),r=s.Z.bind((function(){t.imgIsLoaded=!0,t.size[0]<=0||t.size[1]<=0||(e.save(),e.globalAlpha=i,this.transformRect(t,t.transformations),e.drawImage(t.rendNode,t.coords.scrCoords[1],t.coords.scrCoords[2]-t.size[1],t.size[0],t.size[1]),e.restore())}),this);this.updateImageURL(t)?t.rendNode.onload=r:t.imgIsLoaded&&r()},transformRect:function(t,e){var s,r,n,a,h,l=e.length,c=this.context;if(l>0)if(s=this.joinTransforms(t,e),t.elementClass===i.Z.OBJECT_CLASS_TEXT&&"html"===t.visProp.display){if(-1===(r=" matrix("+[s[1][1],s[2][1],s[1][2],s[2][2],s[1][0],s[2][0]].join(",")+") ").indexOf("NaN")){switch((h=t.rendNode).style.transform=r,n=-t.coords.scrCoords[1],a=-t.coords.scrCoords[2],t.evalVisProp("anchorx")){case"right":n+=t.size[0];break;case"middle":n+=.5*t.size[0]}switch(t.evalVisProp("anchory")){case"bottom":a+=t.size[1];break;case"middle":a+=.5*t.size[1]}h.style["transform-origin"]=n+"px "+a+"px"}}else Math.abs(d.Z.det(s))>=o.Z.eps&&c.transform(s[1][1],s[2][1],s[1][2],s[2][2],s[1][0],s[2][0])},updateImageURL:function(t){var e;return e=t.eval(t.url),t._src!==e&&(t.imgIsLoaded=!1,t.rendNode.src=e,t._src=e,!0)},remove:function(t){s.Z.exists(t)&&s.Z.exists(t.parentNode)&&t.parentNode.removeChild(t)},updatePathStringPrim:function(t){var e,i,s,r,o,n="M",a=n,h=5e3,l=this.context;if(!(t.numberPoints<=0)){if(o=Math.min(t.points.length,t.numberPoints),l.beginPath(),1===t.bezierDegree)for(e=0;e<o;e++)i=t.points[e].scrCoords,isNaN(i[1])||isNaN(i[2])?a=n:(i[1]>h?i[1]=h:i[1]<-5e3&&(i[1]=-5e3),i[2]>h?i[2]=h:i[2]<-5e3&&(i[2]=-5e3),a===n?l.moveTo(i[1],i[2]):l.lineTo(i[1],i[2]),a="L");else if(3===t.bezierDegree)for(e=0;e<o;)i=t.points[e].scrCoords,isNaN(i[1])||isNaN(i[2])?a=n:(a===n?l.moveTo(i[1],i[2]):(e+=1,s=t.points[e].scrCoords,e+=1,r=t.points[e].scrCoords,l.bezierCurveTo(i[1],i[2],s[1],s[2],r[1],r[2])),a="C"),e+=1;l.lineCap="round",l.lineJoin="round",this._fill(t),this._stroke(t)}},updatePathStringBezierPrim:function(t){var e,i,s,r,o,n,a,h="M",l=h,c=5e3,u=t.evalVisProp("strokewidth"),p="plot"!==t.evalVisProp("curvetype"),f=this.context;if(!(t.numberPoints<=0)){for(p&&t.board.options.curve.RDPsmoothing&&(t.points=d.Z.RamerDouglasPeucker(t.points,.5)),a=Math.min(t.points.length,t.numberPoints),f.beginPath(),i=1;i<3;i++)for(l=h,e=0;e<a;e++)r=t.points[e].scrCoords,isNaN(r[1])||isNaN(r[2])?l=h:(r[1]>c?r[1]=c:r[1]<-5e3&&(r[1]=-5e3),r[2]>c?r[2]=c:r[2]<-5e3&&(r[2]=-5e3),l===h?f.moveTo(r[1],r[2]):(s=2*i,f.bezierCurveTo(o+.333*(r[1]-o)+u*(s*Math.random()-i),n+.333*(r[2]-n)+u*(s*Math.random()-i),o+.666*(r[1]-o)+u*(s*Math.random()-i),n+.666*(r[2]-n)+u*(s*Math.random()-i),r[1],r[2])),l="C",o=r[1],n=r[2]);f.lineCap="round",f.lineJoin="round",this._fill(t),this._stroke(t)}},updatePolygonPrim:function(t,e){var i,s,r,o=e.vertices.length,n=this.context,a=!0;if(!(o<=0)&&e.visPropCalc.visible){for("polygonalchain"===e.elType&&o++,n.beginPath(),s=0;!e.vertices[s].isReal&&s<o-1;)s++,a=!1;for(i=e.vertices[s].coords.scrCoords,n.moveTo(i[1],i[2]),r=s;r<o-1;r++)e.vertices[r].isReal||(a=!1),i=e.vertices[r].coords.scrCoords,n.lineTo(i[1],i[2]);n.closePath(),a&&this._fill(e)}},display:function(t,e){t&&t.rendNode&&(t.visPropOld.visible=e,t.rendNode.style.visibility=e?"inherit":"hidden")},show:function(e){t.Z.deprecated("Board.renderer.show()","Board.renderer.display()"),s.Z.exists(e.rendNode)&&(e.rendNode.style.visibility="inherit")},hide:function(e){t.Z.deprecated("Board.renderer.hide()","Board.renderer.display()"),s.Z.exists(e.rendNode)&&(e.rendNode.style.visibility="hidden")},setGradient:function(t){},setShadow:function(t){t.visPropOld.shadow!==t.visProp.shadow&&(t.visPropOld.shadow=t.visProp.shadow)},highlight:function(t){return t.elementClass===i.Z.OBJECT_CLASS_TEXT&&"html"===t.evalVisProp("display")?this.updateTextStyle(t,!0):(t.board.prepareUpdate(),t.board.renderer.suspendRedraw(t.board),t.board.updateRenderer(),t.board.renderer.unsuspendRedraw()),this},noHighlight:function(t){return t.elementClass===i.Z.OBJECT_CLASS_TEXT&&"html"===t.evalVisProp("display")?this.updateTextStyle(t,!1):(t.board.prepareUpdate(),t.board.renderer.suspendRedraw(t.board),t.board.updateRenderer(),t.board.renderer.unsuspendRedraw()),this},suspendRedraw:function(e){this.context.save(),this.context.clearRect(0,0,this.canvasRoot.width,this.canvasRoot.height),e&&(e.attr.showcopyright||e.attr.showlogo)&&this.displayLogo(t.Z.licenseLogo,12,e),e&&e.attr.showcopyright&&this.displayCopyright(t.Z.licenseText,12)},unsuspendRedraw:function(){this.context.restore()},resize:function(t,e){this.container?(this.canvasRoot.style.width=parseFloat(t)+"px",this.canvasRoot.style.height=parseFloat(e)+"px",this.canvasRoot.setAttribute("width",2*parseFloat(t)+"px"),this.canvasRoot.setAttribute("height",2*parseFloat(e)+"px")):(this.canvasRoot.width=2*parseFloat(t),this.canvasRoot.height=2*parseFloat(e)),this.context=this.canvasRoot.getContext("2d"),this.context.scale(2,2)},removeToInsertLater:function(){return function(){}}});var L=t.Z.CanvasRenderer;t.Z.NoRenderer=function(){this.enhancedRendering=!1,this.type="no"},t.Z.extend(t.Z.NoRenderer.prototype,{drawPoint:function(t){},updatePoint:function(t){},changePointStyle:function(t){},drawLine:function(t){},updateLine:function(t){},drawTicks:function(t){},updateTicks:function(t){},drawCurve:function(t){},updateCurve:function(t){},drawEllipse:function(t){},updateEllipse:function(t){},drawPolygon:function(t){},updatePolygon:function(t){},displayCopyright:function(t,e){},drawInternalText:function(t){},updateInternalText:function(t){},drawText:function(t){},updateText:function(t){},updateTextStyle:function(t,e){},updateInternalTextStyle:function(t,e,i){},drawImage:function(t){},updateImage:function(t){},updateImageURL:function(t){},appendChildPrim:function(t,e){},appendNodesToElement:function(t,e){},remove:function(t){},makeArrows:function(t){},updateEllipsePrim:function(t,e,i,s,r){},updateLinePrim:function(t,e,i,s,r,o){},updatePathPrim:function(t,e,i){},updatePathStringPoint:function(t,e,i){},updatePathStringPrim:function(t){},updatePathStringBezierPrim:function(t){},updatePolygonPrim:function(t,e){},updateRectPrim:function(t,e,i,s,r){},setPropertyPrim:function(t,e,i){},show:function(t){},hide:function(t){},setBuffering:function(t,e){},setDashStyle:function(t){},setDraft:function(t){},removeDraft:function(t){},setGradient:function(t){},updateGradient:function(t){},setObjectTransition:function(t,e){},setObjectFillColor:function(t,e,i){},setObjectStrokeColor:function(t,e,i){},setObjectStrokeWidth:function(t,e){},setShadow:function(t){},highlight:function(t){},noHighlight:function(t){},suspendRedraw:function(){},unsuspendRedraw:function(){},drawNavigationBar:function(t){},getElementById:function(t){return null},resize:function(t,e){},removeToInsertLater:function(){return function(){}}}),t.Z.NoRenderer.prototype=new Z;var B=t.Z.NoRenderer;t.Z.JSXGraph={rendererType:(v.Z.board.renderer="no",e.Z.supportsVML()&&(v.Z.board.renderer="vml",document.onmousemove=function(){var t;return document.body&&(t=document.body.scrollLeft,t+=document.body.scrollTop),t}),e.Z.supportsCanvas()&&(v.Z.board.renderer="canvas"),e.Z.supportsSVG()&&(v.Z.board.renderer="svg"),e.Z.isNode()&&e.Z.supportsCanvas()&&(v.Z.board.renderer="canvas"),(e.Z.isNode()||"no"===v.Z.renderer)&&(v.Z.text.display="internal",v.Z.infobox.display="internal"),v.Z.board.renderer),initRenderer:function(t,e,i,r){var o;if(s.Z.exists(i)&&!1!==i||"object"!=typeof document||(i=document),"object"==typeof i&&null!==t)for(o=s.Z.isString(t)?i.getElementById(t):t;o.firstChild;)o.removeChild(o.firstChild);else o=t;return void 0!==r&&"auto"!==r||(r=this.rendererType),"svg"===r?new A(o,e):"vml"===r?new D(o):"canvas"===r?new L(o,e):new B},_setAttributes:function(t,e){var i,r,o=s.Z.copyAttributes(t,e,"board"),n=["drag","fullscreen","intl","keyboard","logging","pan","resize","screenshot","selection","zoom"],a=n.length;for(i=0;i<a;i++)o[r=n[i]]=s.Z.copyAttributes(o,e,"board",r);return o.navbar=s.Z.copyAttributes(o.navbar,e,"navbar"),o.movetarget=t.moveTarget||t.movetarget||e.board.moveTarget,o},_fillBoard:function(e,i,s){e.initInfobox(i.infobox),e.maxboundingbox=i.maxboundingbox,e.resizeContainer(s.width,s.height,!0,!0),e._createSelectionPolygon(i),e.renderer.drawNavigationBar(e,i.navbar),t.Z.boards[e.id]=e},_setARIA:function(t,i){var r,o=i.document;if("object"!=typeof o){if(!e.Z.isBrowser)return;o=document}(r=s.Z.isString(t)?o.getElementById(t):t).setAttribute("role","region"),r.setAttribute("aria-label",i.title)},_removeARIANodes:function(t){var e,i,s;"object"==typeof(s=t.document||document)&&(i=t.containerObj.getAttribute("aria-labelledby"),(e=s.getElementById(i))&&e.parentNode&&e.parentNode.removeChild(e),i=t.containerObj.getAttribute("aria-describedby"),(e=s.getElementById(i))&&e.parentNode&&e.parentNode.removeChild(e))},initBoard:function(i,r){var o,n,a,h,l,c,d,u,p,f,_,m,g,b,Z,C=0,y=0,P={};return"default"!==(r=r||{}).theme&&s.Z.exists(t.Z.themes[r.theme])&&(P=t.Z.themes[r.theme]),b=s.Z.deepCopy(v.Z,P,!0),f=this._setAttributes(r,b),u=e.Z.getDimensions(i,f.document),f.unitx||f.unity?(o=s.Z.def(f.originx,150),n=s.Z.def(f.originy,150),a=s.Z.def(f.unitx,50),h=s.Z.def(f.unity,50)):((p=f.boundingbox)[0]<f.maxboundingbox[0]&&(p[0]=f.maxboundingbox[0]),p[1]>f.maxboundingbox[1]&&(p[1]=f.maxboundingbox[1]),p[2]>f.maxboundingbox[2]&&(p[2]=f.maxboundingbox[2]),p[3]<f.maxboundingbox[3]&&(p[3]=f.maxboundingbox[3]),l=parseInt(u.width,10),c=parseInt(u.height,10),s.Z.exists(p)&&f.keepaspectratio?(a=l/(p[2]-p[0]),h=c/(p[1]-p[3]),Math.abs(a)<Math.abs(h)?y=.5*(c/(h=Math.abs(a)*h/Math.abs(h))-(p[1]-p[3])):C=.5*(l/(a=Math.abs(h)*a/Math.abs(a))-(p[2]-p[0]))):(a=l/(p[2]-p[0]),h=c/(p[1]-p[3])),o=-a*(p[0]-C),n=h*(p[1]+y)),d=this.initRenderer(i,u,f.document,f.renderer),this._setARIA(i,f),(Z=new N(i,d,f.id,[o,n],f.zoomx,f.zoomy,a,h,u.width,u.height,f)).keepaspectratio=f.keepaspectratio,this._fillBoard(Z,f,u),Z.suspendUpdate(),(f=Z.attr).axis&&(_="object"==typeof f.axis?f.axis:{},m=s.Z.deepCopy(b.board.defaultaxes.x,_),g=s.Z.deepCopy(b.board.defaultaxes.y,_),f.defaultaxes.x&&(m=s.Z.deepCopy(m,f.defaultaxes.x)),f.defaultaxes.y&&(g=s.Z.deepCopy(g,f.defaultaxes.y)),Z.defaultAxes={},Z.defaultAxes.x=Z.create("axis",[[0,0],[1,0]],m),Z.defaultAxes.y=Z.create("axis",[[0,0],[0,1]],g)),f.grid&&Z.create("grid",[],"object"==typeof f.grid?f.grid:{}),Z.unsuspendUpdate(),Z},loadBoardFromFile:function(t,i,s,r,o){var n,a,h,l,c;return r=r||{},n=this._setAttributes(r),l=e.Z.getDimensions(t,n.document),a=this.initRenderer(t,l,n.document,n.renderer),this._setARIA(t,n),h=new N(t,a,"",[150,150],1,1,50,50,l.width,l.height,n),this._fillBoard(h,n,l),c=n.encoding||"iso-8859-1",O.parseFileContent(i,h,s,!0,c,o),h},loadBoardFromString:function(t,i,s,r,o){var n,a,h,l;return r=r||{},n=this._setAttributes(r),l=e.Z.getDimensions(t,n.document),a=this.initRenderer(t,l,n.document,n.renderer),this._setARIA(t,n),h=new N(t,a,"",[150,150],1,1,50,50,l.width,l.height,n),this._fillBoard(h,n,l),O.parseString(i,h,s,!0,o),h},freeBoard:function(e){var i;for(i in"string"==typeof e&&(e=t.Z.boards[e]),this._removeARIANodes(e),e.removeEventHandlers(),e.suspendUpdate(),e.objects)e.objects.hasOwnProperty(i)&&e.objects[i].remove();for(;e.containerObj.firstChild;)e.containerObj.removeChild(e.containerObj.firstChild);for(i in e.objects)e.objects.hasOwnProperty(i)&&delete e.objects[i];delete e.renderer,e.jc.creator.clearCache(),delete e.jc,delete t.Z.boards[e.id]},registerElement:function(e,i){t.Z.deprecated("JXG.JSXGraph.registerElement()","JXG.registerElement()"),t.Z.registerElement(e,i)}},e.Z.isBrowser&&"object"==typeof window&&"object"==typeof document&&e.Z.addEvent(window,"load",(function(){var e,i,r,o,n,a,h,l,c,d,u,p,f,_,m,g,b,v,Z=!1,C=document.getElementsByTagName("script"),y=function(e,i,s){var r=t.Z.JSXGraph.initBoard(n,{boundingbox:s,keepaspectratio:!0,grid:m,axis:_,showReload:!0});if(i.toLowerCase().indexOf("script")>-1)r.construct(e);else try{r.jc.parse(e)}catch(e){t.Z.debug(e)}return r},P=function(e,i,s,r){return function(){var o;t.Z.JSXGraph.freeBoard(e),(o=y(i,s,r)).reload=P(o,i,s,r)}};for(i=0;i<C.length;i++)if(e=C[i].getAttribute("type",!1),s.Z.exists(e)&&("text/jessiescript"===e.toLowerCase()||"jessiescript"===e.toLowerCase()||"text/jessiecode"===e.toLowerCase()||"jessiecode"===e.toLowerCase())){if(p=C[i].getAttribute("class",!1)||"",l=C[i].getAttribute("width",!1)||"",c=C[i].getAttribute("height",!1)||"",d=C[i].getAttribute("maxwidth",!1)||"100%",u=C[i].getAttribute("aspectratio",!1)||"1/1",f=C[i].getAttribute("boundingbox",!1)||"-5, 5, 5, -5",n=C[i].getAttribute("container",!1),b=C[i].getAttribute("src",!1),4!==(f=f.split(",")).length)f=[-5,5,5,-5];else for(r=0;r<f.length;r++)f[r]=parseFloat(f[r]);if(_=s.Z.str2Bool(C[i].getAttribute("axis",!1)||"false"),m=s.Z.str2Bool(C[i].getAttribute("grid",!1)||"false"),s.Z.exists(n))o=document.getElementById(n);else{n="jessiescript_autgen_jxg_"+i,(o=document.createElement("div")).setAttribute("id",n),h=""!==l?"width:"+l+";":"",h+=""!==c?"height:"+c+";":"",h+=""!==d?"max-width:"+d+";":"",h+=""!==u?"aspect-ratio:"+u+";":"",o.setAttribute("style",h),o.setAttribute("class","jxgbox "+p);try{document.body.insertBefore(o,C[i])}catch(t){"object"==typeof jQuery&&jQuery(o).insertBefore(C[i])}}g="",s.Z.exists(b)?(Z=!0,(v=new XMLHttpRequest).open("GET",b),v.overrideMimeType("text/plain; charset=x-user-defined"),v.addEventListener("load",(function(){if(!(this.status<400))throw new Error("\nJSXGraph: failed to load file",b,":",this.responseText);g=this.responseText+"\n"+g,(a=y(g,e,f)).reload=P(a,g,e,f)})),v.addEventListener("error",(function(t){throw new Error("\nJSXGraph: failed to load file",b,":",t)})),v.send()):Z=!1,document.getElementById(n)?(g=(g=C[i].innerHTML).replace(/<!\[CDATA\[/g,"").replace(/\]\]>/g,""),C[i].innerHTML=g,Z||((a=y(g,e,f)).reload=P(a,g,e,f))):t.Z.debug("JSXGraph: Apparently the div injection failed. Can't create a board, sorry.")}}),window);t.Z.JSXGraph;var I=__webpack_require__(958),j=__webpack_require__(218);t.Z.Point=function(t,e,s){this.constructor(t,s,i.Z.OBJECT_TYPE_POINT,i.Z.OBJECT_CLASS_POINT),this.element=this.board.select(s.anchor),this.coordsConstructor(e),this.elType="point",this.id=this.board.setId(this,"P"),this.board.renderer.drawPoint(this),this.board.finalizeAdding(this),this.createGradient(),this.createLabel()},t.Z.Point.prototype=new I.Z,s.Z.copyPrototypeMethods(t.Z.Point,j.Z,"coordsConstructor"),t.Z.extend(t.Z.Point.prototype,{hasPoint:function(t,e){var i,r,o,n=this.coords.scrCoords,a=this.evalVisProp("sizeunit");return s.Z.isObject(this.evalVisProp("precision"))?(o=this.board._inputDevice,r=this.evalVisProp("precision."+o)):r=this.board.options.precision.hasPoint,i=parseFloat(this.evalVisProp("size")),"user"===a&&(i*=Math.sqrt(Math.abs(this.board.unitX*this.board.unitY))),(i+=.5*parseFloat(this.evalVisProp("strokewidth")))<r&&(i=r),Math.abs(n[1]-t)<i+2&&Math.abs(n[2]-e)<i+2},update:function(t){return this.needsUpdate?(this.updateCoords(t),this.evalVisProp("trace")&&this.cloneToBackground(!0),this):this},updateTransform:function(t){var e,s;if(0===this.transformations.length||null===this.baseElement)return this;for(this.transformations[0].update(),e=this===this.baseElement?this.transformations[0].apply(this,"self"):this.transformations[0].apply(this.baseElement),s=1;s<this.transformations.length;s++)this.transformations[s].update(),e=o.Z.matVecMult(this.transformations[s].matrix,e);return this.coords.setCoordinates(i.Z.COORDS_BY_USER,e),this},updateRenderer:function(){return this.updateRendererGeneric("updatePoint"),this},bounds:function(){return this.coords.usrCoords.slice(1).concat(this.coords.usrCoords.slice(1))},makeIntersection:function(t,e,s,r){var o;t=this.board.select(t),e=this.board.select(e),o=h.Z.intersectionFunction(this.board,t,e,s,r,this.visProp.alwaysintersect),this.addConstraint([o]);try{t.addChild(this),e.addChild(this)}catch(i){throw new Error("JSXGraph: Can't create 'intersection' with parent types '"+typeof t+"' and '"+typeof e+"'.")}this.type=i.Z.OBJECT_TYPE_INTERSECTION,this.elType="intersection",this.parents=[t.id,e.id,s,r],this.generatePolynomial=function(){var i=t.generatePolynomial(this),s=e.generatePolynomial(this);return 0===i.length||0===s.length?[]:[i[0],s[0]]},this.prepareUpdate().update()},setStyle:function(t){return this.visProp.face=["cross","cross","cross","circle","circle","circle","circle","square","square","square","plus","plus","plus"][t],this.visProp.size=[2,3,4,1,2,3,4,2,3,4,2,3,4][t],this.board.renderer.changePointStyle(this),this},normalizeFace:function(e){return t.Z.deprecated("Point.normalizeFace()","JXG.normalizePointFace()"),v.Z.normalizePointFace(e)},face:function(e){t.Z.deprecated("Point.face()","Point.setAttribute()"),this.setAttribute({face:e})},size:function(e){t.Z.deprecated("Point.size()","Point.setAttribute()"),this.setAttribute({size:e})},isOn:function(e,r){var n,a;return r=r||o.Z.eps,s.Z.isPoint(e)?this.Dist(e)<r:e.elementClass===i.Z.OBJECT_CLASS_LINE?("segment"!==e.elType||this.evalVisProp("alwaysintersect")||(n=t.Z.Math.Geometry.projectCoordsToSegment(this.coords.usrCoords,e.point1.coords.usrCoords,e.point2.coords.usrCoords))[1]>=0&&n[1]<=1)&&h.Z.distPointLine(this.coords.usrCoords,e.stdform)<r:e.elementClass===i.Z.OBJECT_CLASS_CIRCLE?e.evalVisProp("hasinnerpoints")?this.Dist(e.center)<e.Radius()+r:Math.abs(this.Dist(e.center)-e.Radius())<r:e.elementClass===i.Z.OBJECT_CLASS_CURVE?(a=h.Z.projectPointToCurve(this,e,this.board)[0],h.Z.distance(this.coords.usrCoords,a.usrCoords,3)<r):e.type===i.Z.OBJECT_TYPE_POLYGON?!(!e.evalVisProp("hasinnerpoints")||!e.pnpoly(this.coords.usrCoords[1],this.coords.usrCoords[2],t.Z.COORDS_BY_USER))||(n=h.Z.projectCoordsToPolygon(this.coords.usrCoords,e),h.Z.distance(this.coords.usrCoords,n,3)<r):e.type===i.Z.OBJECT_TYPE_TURTLE&&(a=h.Z.projectPointToTurtle(this,e,this.board),h.Z.distance(this.coords.usrCoords,a.usrCoords,3)<r)},cloneToBackground:function(){var t=s.Z.getCloneObject(this);return this.board.renderer.drawPoint(t),this.traces[t.id]=t.rendNode,this}}),t.Z.createPoint=function(e,i,r){var o,n;if(n=s.Z.copyAttributes(r,e.options,"point"),!(o=j.Z.create(t.Z.Point,e,i,n)))throw new Error("JSXGraph: Can't create point with parent types '"+typeof i[0]+"' and '"+typeof i[1]+"'.\nPossible parent types: [x,y], [z,x,y], [element,transformation]");return o},t.Z.createGlider=function(t,e,i){var r,o,n=s.Z.copyAttributes(i,t.options,"glider");return o=1===e.length?[0,0]:e.slice(0,2),(r=t.create("point",o,n)).makeGlider(e[e.length-1]),r},t.Z.createIntersectionPoint=function(t,e,r){var o,n,a,l,c,d,u=s.Z.copyAttributes(r,t.options,"intersection");e.push(0,0),n=t.select(e[0]),a=t.select(e[1]),c=e[2]||0,d=e[3]||0,o=t.create("point",[0,0,0],u),l=h.Z.intersectionFunction(t,n,a,c,d,o.visProp.alwaysintersect),o.addConstraint([l]);try{n.addChild(o),a.addChild(o)}catch(t){throw new Error("JSXGraph: Can't create 'intersection' with parent types '"+typeof e[0]+"' and '"+typeof e[1]+"'.")}return o.type=i.Z.OBJECT_TYPE_INTERSECTION,o.elType="intersection",o.setParents([n.id,a.id]),o.intersectionNumbers=[c,d],o.getParents=function(){return this.parents.concat(this.intersectionNumbers)},o.generatePolynomial=function(){var t=n.generatePolynomial(o),e=a.generatePolynomial(o);return 0===t.length||0===e.length?[]:[t[0],e[0]]},o},t.Z.createOtherIntersectionPoint=function(t,e,r){var o,n,a,l,c,d,u,p=!0,f=s.Z.copyAttributes(r,t.options,"otherintersection");if(3!==e.length)p=!1;else{for(n=t.select(e[0]),a=t.select(e[1]),c=s.Z.isArray(e[2])?e[2]:[e[2]],l=0;l<c.length;l++)if(c[l]=t.select(c[l]),!s.Z.isPoint(c[l])){p=!1;break}p&&((u=[n,a]).sort((function(t,e){return e.elementClass-t.elementClass})),([i.Z.OBJECT_CLASS_CIRCLE,i.Z.OBJECT_CLASS_CURVE].indexOf(u[0].elementClass)<0||[i.Z.OBJECT_CLASS_CIRCLE,i.Z.OBJECT_CLASS_CURVE,i.Z.OBJECT_CLASS_LINE].indexOf(u[1].elementClass)<0)&&(p=!1))}if(!p)throw new Error("JSXGraph: Can't create 'other intersection point' with parent types '"+typeof e[0]+"',  '"+typeof e[1]+"'and  '"+typeof e[2]+"'.\nPossible parent types: [circle|curve|line,circle|curve|line, point], not two lines");return o=t.create("point",[0,0,0],f),d=h.Z.otherIntersectionFunction(u,c,o.visProp.alwaysintersect,o.visProp.precision),o.addConstraint([d]),o.type=i.Z.OBJECT_TYPE_INTERSECTION,o.elType="otherintersection",o.setParents([n.id,a.id]),o.addParents(c),n.addChild(o),a.addChild(o),n.elementClass===i.Z.OBJECT_CLASS_CIRCLE&&(o.generatePolynomial=function(){var t=n.generatePolynomial(o),e=a.generatePolynomial(o);return 0===t.length||0===e.length?[]:[t[0],e[0]]}),o},t.Z.createPolePoint=function(e,s,r){var o,n,a,h,l,c,d;if(s.length>1&&(h=s[0].type===i.Z.OBJECT_TYPE_CONIC||s[0].elementClass===i.Z.OBJECT_CLASS_CIRCLE,l=s[1].type===i.Z.OBJECT_TYPE_CONIC||s[1].elementClass===i.Z.OBJECT_CLASS_CIRCLE,c=s[0].elementClass===i.Z.OBJECT_CLASS_LINE,d=s[1].elementClass===i.Z.OBJECT_CLASS_LINE),2!==s.length||!(h&&d||c&&l))throw new Error("JSXGraph: Can't create 'pole point' with parent types '"+typeof s[0]+"' and '"+typeof s[1]+"'.\nPossible parent type: [conic|circle,line], [line,conic|circle]");return d?(n=e.select(s[0]),a=e.select(s[1])):(n=e.select(s[1]),a=e.select(s[0])),(o=e.create("point",[function(){var e=n.quadraticform,i=a.stdform.slice(0,3);return[t.Z.Math.Numerics.det([i,e[1],e[2]]),t.Z.Math.Numerics.det([e[0],i,e[2]]),t.Z.Math.Numerics.det([e[0],e[1],i])]}],r)).elType="polepoint",o.setParents([n.id,a.id]),n.addChild(o),a.addChild(o),o},t.Z.registerElement("point",t.Z.createPoint),t.Z.registerElement("glider",t.Z.createGlider),t.Z.registerElement("intersection",t.Z.createIntersectionPoint),t.Z.registerElement("otherintersection",t.Z.createOtherIntersectionPoint),t.Z.registerElement("polepoint",t.Z.createPolePoint);var Y=t.Z.Point;t.Z.Line=function(e,s,r,o){this.constructor(e,o,i.Z.OBJECT_TYPE_LINE,i.Z.OBJECT_CLASS_LINE),this.point1=this.board.select(s),this.point2=this.board.select(r),this.ticks=[],this.defaultTicks=null,this.parentPolygon=null,this.id=this.board.setId(this,"L"),this.board.renderer.drawLine(this),this.board.finalizeAdding(this),this.elType="line",this.point1._is_new?(this.addChild(this.point1),delete this.point1._is_new):this.point1.addChild(this),this.point2._is_new?(this.addChild(this.point2),delete this.point2._is_new):this.point2.addChild(this),this.inherits.push(this.point1,this.point2),this.updateStdform(),this.createLabel(),this.methodMap=t.Z.deepCopy(this.methodMap,{point1:"point1",point2:"point2",getSlope:"Slope",Slope:"Slope",Direction:"Direction",getRise:"getRise",Rise:"getRise",getYIntersect:"getRise",YIntersect:"getRise",getAngle:"getAngle",Angle:"getAngle",L:"L",length:"L",setFixedLength:"setFixedLength",setStraight:"setStraight"})},t.Z.Line.prototype=new I.Z,t.Z.extend(t.Z.Line.prototype,{hasPoint:function(t,e){var r,n,a,l,c,d,p,f,_,m=[],g=[1,t,e],b=this.evalVisProp("strokewidth");return s.Z.isObject(this.evalVisProp("precision"))?(_=this.board._inputDevice,f=this.evalVisProp("precision."+_)):f=this.board.options.precision.hasPoint,f+=.5*b,m[0]=this.stdform[0]-this.stdform[1]*this.board.origin.scrCoords[1]/this.board.unitX+this.stdform[2]*this.board.origin.scrCoords[2]/this.board.unitY,m[1]=this.stdform[1]/this.board.unitX,m[2]=this.stdform[2]/-this.board.unitY,r=h.Z.distPointLine(g,m),!(isNaN(r)||r>f)&&(!(!this.evalVisProp("straightfirst")||!this.evalVisProp("straightlast"))||(a=this.point1.coords,l=this.point2.coords,n=[0,m[1],m[2]],n=o.Z.crossProduct(n,g),(n=o.Z.crossProduct(n,m))[1]/=n[0],n[2]/=n[0],n[0]=1,n=new u.Z(i.Z.COORDS_BY_SCREEN,n.slice(1),this.board).usrCoords,c=a.distance(i.Z.COORDS_BY_USER,l),a=a.usrCoords.slice(0),l=l.usrCoords.slice(0),c<o.Z.eps?d=0:(c===Number.POSITIVE_INFINITY&&(c=1/o.Z.eps,Math.abs(l[0])<o.Z.eps?(c/=h.Z.distance([0,0,0],l),l=[1,a[1]+l[1]*c,a[2]+l[2]*c]):(c/=h.Z.distance([0,0,0],a),a=[1,l[1]+a[1]*c,l[2]+a[2]*c])),c=l[p=1]-a[p],Math.abs(c)<o.Z.eps&&(c=l[p=2]-a[p]),d=(n[p]-a[p])/c),!(!this.evalVisProp("straightfirst")&&d<0)&&!(!this.evalVisProp("straightlast")&&d>1)))},update:function(){var t;return this.needsUpdate?(this.constrained&&(s.Z.isFunction(this.funps)?(t=this.funps())&&t.length&&2===t.length&&(this.point1=t[0],this.point2=t[1]):(s.Z.isFunction(this.funp1)&&(t=this.funp1(),s.Z.isPoint(t)?this.point1=t:t&&t.length&&2===t.length&&this.point1.setPositionDirectly(i.Z.COORDS_BY_USER,t)),s.Z.isFunction(this.funp2)&&(t=this.funp2(),s.Z.isPoint(t)?this.point2=t:t&&t.length&&2===t.length&&this.point2.setPositionDirectly(i.Z.COORDS_BY_USER,t)))),this.updateSegmentFixedLength(),this.updateStdform(),this.evalVisProp("trace")&&this.cloneToBackground(!0),this):this},updateSegmentFixedLength:function(){var t,e,s,r,n,a,h,l;return this.hasFixedLength?(t=this.point1.Dist(this.point2),e=this.evalVisProp("nonnegativeonly")?Math.max(0,this.fixedLength()):Math.abs(this.fixedLength()),s=this.fixedLengthOldCoords[0].distance(i.Z.COORDS_BY_USER,this.point1.coords),r=this.fixedLengthOldCoords[1].distance(i.Z.COORDS_BY_USER,this.point2.coords),(s>o.Z.eps||r>o.Z.eps||t!==e)&&(n=this.point1.isDraggable&&this.point1.type!==i.Z.OBJECT_TYPE_GLIDER&&!this.point1.evalVisProp("fixed"),a=this.point2.isDraggable&&this.point2.type!==i.Z.OBJECT_TYPE_GLIDER&&!this.point2.evalVisProp("fixed"),t>o.Z.eps?s>r&&a||s<=r&&a&&!n?(this.point2.setPositionDirectly(i.Z.COORDS_BY_USER,[this.point1.X()+(this.point2.X()-this.point1.X())*e/t,this.point1.Y()+(this.point2.Y()-this.point1.Y())*e/t]),this.point2.fullUpdate()):(s<=r&&n||s>r&&n&&!a)&&(this.point1.setPositionDirectly(i.Z.COORDS_BY_USER,[this.point2.X()+(this.point1.X()-this.point2.X())*e/t,this.point2.Y()+(this.point1.Y()-this.point2.Y())*e/t]),this.point1.fullUpdate()):(h=Math.random()-.5,l=Math.random()-.5,t=o.Z.hypot(h,l),a?(this.point2.setPositionDirectly(i.Z.COORDS_BY_USER,[this.point1.X()+h*e/t,this.point1.Y()+l*e/t]),this.point2.fullUpdate()):n&&(this.point1.setPositionDirectly(i.Z.COORDS_BY_USER,[this.point2.X()+h*e/t,this.point2.Y()+l*e/t]),this.point1.fullUpdate())),this.fixedLengthOldCoords[0].setCoordinates(i.Z.COORDS_BY_USER,this.point1.coords.usrCoords),this.fixedLengthOldCoords[1].setCoordinates(i.Z.COORDS_BY_USER,this.point2.coords.usrCoords)),this):this},updateStdform:function(){var t=o.Z.crossProduct(this.point1.coords.usrCoords,this.point2.coords.usrCoords);this.stdform[0]=t[0],this.stdform[1]=t[1],this.stdform[2]=t[2],this.stdform[3]=0,this.normalize()},updateRenderer:function(){return this.needsUpdate?(this.visPropCalc.visible&&(this.isReal=!isNaN(this.point1.coords.usrCoords[1]+this.point1.coords.usrCoords[2]+this.point2.coords.usrCoords[1]+this.point2.coords.usrCoords[2])&&o.Z.innerProduct(this.stdform,this.stdform,3)>=o.Z.eps*o.Z.eps,this.isReal||this.updateVisibility(!1)),this.visPropCalc.visible&&this.board.renderer.updateLine(this),this.hasLabel&&this.visPropCalc.visible&&this.label&&this.label.visPropCalc.visible&&this.isReal&&(this.label.update(),this.board.renderer.updateText(this.label)),this.setDisplayRendNode(),this.needsUpdate=!1,this):this},generatePolynomial:function(t){var e=this.point1.symbolic.x,i=this.point1.symbolic.y,s=this.point2.symbolic.x,r=this.point2.symbolic.y,o=t.symbolic.x,n=t.symbolic.y;return[["(",i,")*(",o,")-(",i,")*(",s,")+(",n,")*(",s,")-(",e,")*(",n,")+(",e,")*(",r,")-(",o,")*(",r,")"].join("")]},getRise:function(){return Math.abs(this.stdform[2])>=o.Z.eps?-this.stdform[0]/this.stdform[2]:1/0},Slope:function(){return Math.abs(this.stdform[2])>=o.Z.eps?-this.stdform[1]/this.stdform[2]:1/0},getSlope:function(){return this.Slope()},getAngle:function(){return Math.atan2(-this.stdform[1],this.stdform[2])},Direction:function(){var t=this.point1.coords.usrCoords,e=this.point2.coords.usrCoords;return 0===e[0]&&0!==t[0]?e.slice(1):0===t[0]&&0!==e[0]?[-t[1],-t[2]]:[e[1]-t[1],e[2]-t[2]]},isVertical:function(){var t=this.Direction();return 0===t[0]&&0!==t[1]},isHorizontal:function(){var t=this.Direction();return 0===t[1]&&0!==t[0]},setStraight:function(t,e){return this.visProp.straightfirst=t,this.visProp.straightlast=e,this.board.renderer.updateLine(this),this},getTextAnchor:function(){return new u.Z(i.Z.COORDS_BY_USER,[.5*(this.point2.X()+this.point1.X()),.5*(this.point2.Y()+this.point1.Y())],this.board)},setLabelRelativeCoords:function(t){s.Z.exists(this.label)&&(this.label.relativeCoords=new u.Z(i.Z.COORDS_BY_SCREEN,[t[0],-t[1]],this.board))},getLabelAnchor:function(){var t,e,r,n,a,l,c,d,p=1.5,f=0,_=new u.Z(i.Z.COORDS_BY_USER,this.point1.coords.usrCoords,this.board),m=new u.Z(i.Z.COORDS_BY_USER,this.point2.coords.usrCoords,this.board),g=this.evalVisProp("straightfirst"),b=this.evalVisProp("straightlast");if((g||b)&&h.Z.calcStraight(this,_,m,0),_=_.scrCoords,m=m.scrCoords,!s.Z.exists(this.label))return new u.Z(i.Z.COORDS_BY_SCREEN,[NaN,NaN],this.board);if(r=this.label.evalVisProp("position"),!s.Z.isString(r))return new u.Z(i.Z.COORDS_BY_SCREEN,[NaN,NaN],this.board);if(r.indexOf("right")<0&&r.indexOf("left")<0)switch(r){case"last":t=m[1],e=m[2];break;case"first":t=_[1],e=_[2];break;case"lft":case"llft":case"ulft":_[1]<m[1]+o.Z.eps?(t=_[1],e=_[2]):(t=m[1],e=m[2]);break;case"rt":case"lrt":case"urt":_[1]>m[1]+o.Z.eps?(t=_[1],e=_[2]):(t=m[1],e=m[2]);break;default:t=.5*(_[1]+m[1]),e=.5*(_[2]+m[2])}else n=s.Z.parsePosition(r),a=s.Z.parseNumber(n.pos,1,1),l=m[1]-_[1],c=m[2]-_[2],d=o.Z.hypot(l,c),n.pos.indexOf("px")>=0||n.pos.indexOf("fr")>=0||n.pos.indexOf("%")>=0?(n.pos.indexOf("px")>=0&&(a/=d),t=_[1]+a*l,e=_[2]+a*c):(t=_[1]+a*this.board.unitX*l/d,e=_[2]+a*this.board.unitY*c/d),"left"===n.side?l*=-1:c*=-1,s.Z.exists(this.label)&&(p=.5*this.label.evalVisProp("distance")/d),t+=c*this.label.size[0]*p,e+=l*this.label.size[1]*p;return(g||b)&&(s.Z.exists(this.label)&&(f=this.label.evalVisProp("fontsize")),Math.abs(t)<o.Z.eps?t=f:this.board.canvasWidth+o.Z.eps>t&&t>this.board.canvasWidth-f-o.Z.eps&&(t=this.board.canvasWidth-f),o.Z.eps+f>e&&e>-o.Z.eps?e=f:this.board.canvasHeight+o.Z.eps>e&&e>this.board.canvasHeight-f-o.Z.eps&&(e=this.board.canvasHeight-f)),new u.Z(i.Z.COORDS_BY_SCREEN,[t,e],this.board)},cloneToBackground:function(){var t,e,i,r=s.Z.getCloneObject(this);return r.point1=this.point1,r.point2=this.point2,r.stdform=this.stdform,e=this.getSlope(),t=this.getRise(),r.getSlope=function(){return e},r.getRise=function(){return t},i=this.board.renderer.enhancedRendering,this.board.renderer.enhancedRendering=!0,this.board.renderer.drawLine(r),this.board.renderer.enhancedRendering=i,this.traces[r.id]=r.rendNode,this},addTransform:function(t){var e,i=s.Z.isArray(t)?t:[t],r=i.length;for(e=0;e<r;e++)this.point1.transformations.push(i[e]),this.point2.transformations.push(i[e]);return this},snapToGrid:function(t){var e,r,o,n,a,l,c,d;return this.evalVisProp("snaptogrid")?this.parents.length<3?(this.point1.handleSnapToGrid(!0,!0),this.point2.handleSnapToGrid(!0,!0)):s.Z.exists(t)&&(c=this.evalVisProp("snapsizex"),d=this.evalVisProp("snapsizey"),a=(e=new u.Z(i.Z.COORDS_BY_SCREEN,[t.Xprev,t.Yprev],this.board)).usrCoords[1],l=e.usrCoords[2],c<=0&&this.board.defaultAxes&&this.board.defaultAxes.x.defaultTicks&&(c=(n=this.board.defaultAxes.x.defaultTicks).ticksDelta*(n.evalVisProp("minorticks")+1)),d<=0&&this.board.defaultAxes&&this.board.defaultAxes.y.defaultTicks&&(d=(n=this.board.defaultAxes.y.defaultTicks).ticksDelta*(n.evalVisProp("minorticks")+1)),c>0&&d>0&&(r=h.Z.projectPointToLine({coords:e},this,this.board),o=p.Z.subtract([1,Math.round(a/c)*c,Math.round(l/d)*d],r.usrCoords),this.board.create("transform",o.slice(1),{type:"translate"}).applyOnce([this.point1,this.point2]))):(this.point1.handleSnapToGrid(!1,!0),this.point2.handleSnapToGrid(!1,!0)),this},snapToPoints:function(){var t=this.evalVisProp("snaptopoints");return this.parents.length<3&&(this.point1.handleSnapToPoints(t),this.point2.handleSnapToPoints(t)),this},X:function(t){var e,i=this.stdform[2];return e=Math.abs(this.point1.coords.usrCoords[0])>o.Z.eps?this.point1.coords.usrCoords[1]:this.point2.coords.usrCoords[1],t=2*(t-.5),(1-Math.abs(t))*e-t*i},Y:function(t){var e,i=this.stdform[1];return e=Math.abs(this.point1.coords.usrCoords[0])>o.Z.eps?this.point1.coords.usrCoords[2]:this.point2.coords.usrCoords[2],t=2*(t-.5),(1-Math.abs(t))*e+t*i},Z:function(t){var e=Math.abs(this.point1.coords.usrCoords[0])>o.Z.eps?this.point1.coords.usrCoords[0]:this.point2.coords.usrCoords[0];return t=2*(t-.5),(1-Math.abs(t))*e},L:function(){return this.point1.Dist(this.point2)},setFixedLength:function(t){return this.hasFixedLength?(this.fixedLength=s.Z.createFunction(t,this.board),this.hasFixedLength=!0,this.addParentsFromJCFunctions([this.fixedLength]),this.board.update(),this):this},minX:function(){return 0},maxX:function(){return 1},bounds:function(){var t=this.point1.coords.usrCoords,e=this.point2.coords.usrCoords;return[Math.min(t[1],e[1]),Math.max(t[2],e[2]),Math.max(t[1],e[1]),Math.min(t[2],e[2])]},remove:function(){this.removeAllTicks(),I.Z.prototype.remove.call(this)}}),t.Z.createLine=function(e,i,r){var o,n,a,h,l,c,d,u=[],p=!1,f=!1;if(2===i.length){if(c=s.Z.copyAttributes(r,e.options,"line","point1"),s.Z.isArray(i[0])&&i[0].length>1)a=e.create("point",i[0],c);else if(s.Z.isString(i[0])||s.Z.isPoint(i[0]))a=e.select(i[0]);else if(s.Z.isFunction(i[0])&&s.Z.isPoint(i[0]()))a=i[0](),f=!0;else if(s.Z.isFunction(i[0])&&i[0]().length&&i[0]().length>=2)a=t.Z.createPoint(e,i[0](),c),f=!0;else{if(!s.Z.isObject(i[0])||!s.Z.isTransformationOrArray(i[1]))throw new Error("JSXGraph: Can't create line with parent types '"+typeof i[0]+"' and '"+typeof i[1]+"'.\nPossible parent types: [point,point], [[x1,y1],[x2,y2]], [a,b,c]");p=!0,a=e.create("point",[i[0].point1,i[1]],c)}if(c=s.Z.copyAttributes(r,e.options,"line","point2"),p)h=e.create("point",[i[0].point2,i[1]],c);else if(s.Z.isArray(i[1])&&i[1].length>1)h=e.create("point",i[1],c);else if(s.Z.isString(i[1])||s.Z.isPoint(i[1]))h=e.select(i[1]);else if(s.Z.isFunction(i[1])&&s.Z.isPoint(i[1]()))h=i[1](),f=!0;else{if(!(s.Z.isFunction(i[1])&&i[1]().length&&i[1]().length>=2))throw new Error("JSXGraph: Can't create line with parent types '"+typeof i[0]+"' and '"+typeof i[1]+"'.\nPossible parent types: [point,point], [[x1,y1],[x2,y2]], [a,b,c]");h=t.Z.createPoint(e,i[1](),c),f=!0}c=s.Z.copyAttributes(r,e.options,"line"),n=new t.Z.Line(e,a,h,c),f?(n.constrained=!0,n.funp1=i[0],n.funp2=i[1]):p||(n.isDraggable=!0),n.setParents([a.id,h.id])}else if(3===i.length){for(d=!0,l=0;l<3;l++)if(s.Z.isNumber(i[l]))u[l]=s.Z.createFunction(i[l]);else{if(!s.Z.isFunction(i[l]))throw new Error("JSXGraph: Can't create line with parent types '"+typeof i[0]+"' and '"+typeof i[1]+"' and '"+typeof i[2]+"'.\nPossible parent types: [point,point], [[x1,y1],[x2,y2]], [a,b,c]");u[l]=i[l],d=!1}c=s.Z.copyAttributes(r,e.options,"line","point1"),a=d?e.create("point",[u[2]()*u[2]()+u[1]()*u[1](),u[2]()-u[1]()*u[0]()+u[2](),-u[1]()-u[2]()*u[0]()-u[1]()],c):e.create("point",[function(){return.5*(u[2]()*u[2]()+u[1]()*u[1]())},function(){return.5*(u[2]()-u[1]()*u[0]()+u[2]())},function(){return.5*(-u[1]()-u[2]()*u[0]()-u[1]())}],c),c=s.Z.copyAttributes(r,e.options,"line","point2"),h=d?e.create("point",[u[2]()*u[2]()+u[1]()*u[1](),-u[1]()*u[0]()+u[2](),-u[2]()*u[0]()-u[1]()],c):e.create("point",[function(){return u[2]()*u[2]()+u[1]()*u[1]()},function(){return-u[1]()*u[0]()+u[2]()},function(){return-u[2]()*u[0]()-u[1]()}],c),a.prepareUpdate().update(),h.prepareUpdate().update(),c=s.Z.copyAttributes(r,e.options,"line"),(n=new t.Z.Line(e,a,h,c)).isDraggable=d,n.setParents([a,h])}else if(1===i.length&&s.Z.isFunction(i[0])&&2===i[0]().length&&s.Z.isPoint(i[0]()[0])&&s.Z.isPoint(i[0]()[1]))o=i[0](),c=s.Z.copyAttributes(r,e.options,"line"),(n=new t.Z.Line(e,o[0],o[1],c)).constrained=!0,n.funps=i[0],n.setParents(o);else{if(!(1===i.length&&s.Z.isFunction(i[0])&&3===i[0]().length&&s.Z.isNumber(i[0]()[0])&&s.Z.isNumber(i[0]()[1])&&s.Z.isNumber(i[0]()[2])))throw new Error("JSXGraph: Can't create line with parent types '"+typeof i[0]+"' and '"+typeof i[1]+"'.\nPossible parent types: [point,point], [[x1,y1],[x2,y2]], [a,b,c]");o=i[0],c=s.Z.copyAttributes(r,e.options,"line","point1"),a=e.create("point",[function(){var t=o();return[.5*(t[2]*t[2]+t[1]*t[1]),.5*(t[2]-t[1]*t[0]+t[2]),.5*(-t[1]-t[2]*t[0]-t[1])]}],c),c=s.Z.copyAttributes(r,e.options,"line","point2"),h=e.create("point",[function(){var t=o();return[t[2]*t[2]+t[1]*t[1],-t[1]*t[0]+t[2],-t[2]*t[0]-t[1]]}],c),c=s.Z.copyAttributes(r,e.options,"line"),(n=new t.Z.Line(e,a,h,c)).constrained=!0,n.funps=i[0],n.setParents([a,h])}return n},t.Z.registerElement("line",t.Z.createLine),t.Z.createSegment=function(t,e,r){var o,n;if(r.straightFirst=!1,r.straightLast=!1,n=s.Z.copyAttributes(r,t.options,"segment"),o=t.create("line",e.slice(0,2),n),3===e.length){try{o.hasFixedLength=!0,o.fixedLengthOldCoords=[],o.fixedLengthOldCoords[0]=new u.Z(i.Z.COORDS_BY_USER,o.point1.coords.usrCoords.slice(1,3),t),o.fixedLengthOldCoords[1]=new u.Z(i.Z.COORDS_BY_USER,o.point2.coords.usrCoords.slice(1,3),t),o.setFixedLength(e[2])}catch(t){throw new Error("JSXGraph: Can't create segment with third parent type '"+typeof e[2]+"'.\nPossible third parent types: number or function")}o.getParents=function(){return this.parents.concat(this.fixedLength())}}return o.elType="segment",o},t.Z.registerElement("segment",t.Z.createSegment),t.Z.createArrow=function(t,e,r){var o,n;return r.straightFirst=!1,r.straightLast=!1,n=s.Z.copyAttributes(r,t.options,"arrow"),(o=t.create("line",e,n)).type=i.Z.OBJECT_TYPE_VECTOR,o.elType="arrow",o},t.Z.registerElement("arrow",t.Z.createArrow),t.Z.createAxis=function(e,r,o){var n,a,h,l;a=s.Z.copyAttributes(o,e.options,"axis");try{n=e.create("line",r,a)}catch(t){throw new Error("JSXGraph: Can't create axis with parent types '"+typeof r[0]+"' and '"+typeof r[1]+"'.\nPossible parent types: [point,point], [[x1,y1],[x2,y2]]")}for(h in n.type=i.Z.OBJECT_TYPE_AXIS,n.isDraggable=!1,n.point1.isDraggable=!1,n.point2.isDraggable=!1,n._point1UsrCoordsOrg=n.point1.coords.usrCoords.slice(),n._point2UsrCoordsOrg=n.point2.coords.usrCoords.slice(),n.ancestors)n.ancestors.hasOwnProperty(h)&&(n.ancestors[h].type=i.Z.OBJECT_TYPE_AXISPOINT);return l=s.Z.exists(a.ticks.ticksdistance)?a.ticks.ticksdistance:s.Z.isArray(a.ticks.ticks)?a.ticks.ticks:1,n.defaultTicks=e.create("ticks",[n,l],a.ticks),n.defaultTicks.dump=!1,n.elType="axis",n.subs={ticks:n.defaultTicks},n.inherits.push(n.defaultTicks),n.update=function(){var e,i,r,o,a,h,l,c,d,u,p,f,_,m,g,b,v,Z,C;if(!this.needsUpdate)return this;if(e=this.board.getBoundingBox(),i=this.evalVisProp("position"),o=this.Direction(),a=this.isHorizontal(),h=this.isVertical(),l=this.evalVisProp("ticksautopos"),c=this.evalVisProp("ticksautoposthreshold"),c=a?s.Z.parseNumber(c,Math.abs(e[1]-e[3]),1/this.board.unitX)*this.board.unitX:h?s.Z.parseNumber(c,Math.abs(e[1]-e[3]),1/this.board.unitY)*this.board.unitY:s.Z.parseNumber(c,1,1),p=(u=this.evalVisProp("anchor")).indexOf("left")>-1,f=u.indexOf("right")>-1,_=this.evalVisProp("anchordist"),_=a?s.Z.parseNumber(_,Math.abs(e[1]-e[3]),1/this.board.unitX):h?s.Z.parseNumber(_,Math.abs(e[0]-e[2]),1/this.board.unitY):0,b=this.board.getPointLoc(this._point1UsrCoordsOrg,_),m=this.point1.coords.usrCoords.slice(),g=this.point2.coords.usrCoords.slice(),"static"===i||!h&&!a||("fixed"===i?(a&&(o[0]>0&&f||o[0]<0&&p?(m[2]=e[3]+_,g[2]=e[3]+_):o[0]>0&&p||o[0]<0&&f?(m[2]=e[1]-_,g[2]=e[1]-_):(m=this._point1UsrCoordsOrg.slice(),g=this._point2UsrCoordsOrg.slice())),h&&(o[1]>0&&p||o[1]<0&&f?(m[1]=e[0]+_,g[1]=e[0]+_):o[1]>0&&f||o[1]<0&&p?(m[1]=e[2]-_,g[1]=e[2]-_):(m=this._point1UsrCoordsOrg.slice(),g=this._point2UsrCoordsOrg.slice()))):"sticky"===i&&(a&&(b[1]<0&&(o[0]>0&&f||o[0]<0&&p)?(m[2]=e[3]+_,g[2]=e[3]+_):b[1]>0&&(o[0]>0&&p||o[0]<0&&f)?(m[2]=e[1]-_,g[2]=e[1]-_):(m=this._point1UsrCoordsOrg.slice(),g=this._point2UsrCoordsOrg.slice())),h&&(b[0]<0&&(o[1]>0&&p||o[1]<0&&f)?(m[1]=e[0]+_,g[1]=e[0]+_):b[0]>0&&(o[1]>0&&f||o[1]<0&&p)?(m[1]=e[2]-_,g[1]=e[2]-_):(m=this._point1UsrCoordsOrg.slice(),g=this._point2UsrCoordsOrg.slice())))),this.point1.setPositionDirectly(t.Z.COORDS_BY_USER,m),this.point2.setPositionDirectly(t.Z.COORDS_BY_USER,g),s.Z.exists(this.defaultTicks)){if(v=this.defaultTicks.visProp.label,l&&(a||h)){if(s.Z.exists(v._anchorx_org)||(v._anchorx_org=s.Z.def(v.anchorx,this.board.options.text.anchorX)),s.Z.exists(v._anchory_org)||(v._anchory_org=s.Z.def(v.anchory,this.board.options.text.anchorY)),s.Z.exists(v._offset_org)||(v._offset_org=v.offset.slice()),C=v.offset,a){for(d=n.point1.coords.scrCoords[2]-.5*this.board.canvasHeight,Z=v.anchory,d<0&&Math.abs(d)>c?"bottom"===v._side&&("top"===v.anchory&&(Z="bottom"),C[1]*=-1,v._side="top"):d>0&&Math.abs(d)>c?"top"===v._side&&("bottom"===v.anchory&&(Z="top"),C[1]*=-1,v._side="bottom"):(Z=v._anchory_org,C=v._offset_org.slice(),"top"===Z?v._side="bottom":"bottom"===Z?v._side="top":C[1]<0?v._side="bottom":v._side="top"),r=0;r<n.defaultTicks.labels.length;r++)this.defaultTicks.labels[r].visProp.anchory=Z;v.anchory=Z}else if(h){for((d=n.point1.coords.scrCoords[1]-.5*this.board.canvasWidth)<0&&Math.abs(d)>c?"right"===v._side&&("left"===v.anchorx&&(Z="right"),C[0]*=-1,v._side="left"):d>0&&Math.abs(d)>c?"left"===v._side&&("right"===v.anchorx&&(Z="left"),C[0]*=-1,v._side="right"):(Z=v._anchorx_org,C=v._offset_org.slice(),"left"===Z?v._side="right":"right"===Z||C[0]<0?v._side="left":v._side="right"),r=0;r<n.defaultTicks.labels.length;r++)this.defaultTicks.labels[r].visProp.anchorx=Z;v.anchorx=Z}v.offset=C}else delete v._anchorx_org,delete v._anchory_org,delete v._offset_org;this.defaultTicks.needsUpdate=!0}return t.Z.Line.prototype.update.call(this),this},n},t.Z.registerElement("axis",t.Z.createAxis),t.Z.createTangent=function(t,e,r){var n,a,l,c,u,p,f,_,m,g=[];if(1===e.length)n=e[0],a=n.slideObject;else{if(2!==e.length)throw new Error("JSXGraph: Can't create tangent with parent types '"+typeof e[0]+"' and '"+typeof e[1]+"'.\nPossible parent types: [glider|point], [point,line|curve|circle|conic]");if(s.Z.isPoint(e[0]))n=e[0],a=e[1];else{if(!s.Z.isPoint(e[1]))throw new Error("JSXGraph: Can't create tangent with parent types '"+typeof e[0]+"' and '"+typeof e[1]+"'.\nPossible parent types: [glider|point], [point,line|curve|circle|conic]");a=e[0],n=e[1]}}if(p=s.Z.copyAttributes(r,t.options,"tangent"),a.elementClass===i.Z.OBJECT_CLASS_LINE)(u=t.create("line",[a.point1,a.point2],p)).glider=n;else if(a.elementClass===i.Z.OBJECT_CLASS_CURVE&&a.type!==i.Z.OBJECT_TYPE_CONIC){if(_=a.getTransformationSource(),m=_[0])for(g.push(a);_[0]&&s.Z.exists(_[1]._transformationSource);)g.push(_[1]),_=_[1].getTransformationSource();"plot"!==a.evalVisProp("curvetype")||m?(u=t.create("line",[function(){var e,s,r,l,c,u,p,f,_=a.X,b=a.Y;if(f=n.type===i.Z.OBJECT_TYPE_GLIDER?n.position:"functiongraph"===a.evalVisProp("curvetype")?n.X():h.Z.projectPointToCurve(n,a,t)[1],p=n.Coords(!0),m){for(_=(c=g[g.length-1]._transformationSource).X,b=c.Y,l=0;l<g.length;l++)g[l].updateTransformMatrix(),u=o.Z.inverse(g[l].transformMat),p=o.Z.matVecMult(u,p);n.type!==i.Z.OBJECT_TYPE_GLIDER&&(p[1]/=p[0],p[2]/=p[0],p[0]/=p[0],f=h.Z.projectCoordsToCurve(p[1],p[2],0,c,t)[1])}if(e=d.Z.D(b)(f),s=d.Z.D(_)(f),r=[-p[1]*e+p[2]*s,p[0]*e,-p[0]*s],m)for(l=g.length-1;l>=0;l--)u=o.Z.transpose(o.Z.inverse(g[l].transformMat)),r=o.Z.matVecMult(u,r);return r}],p),n.addChild(u),u.glider=n):(f=function(t,e,i){var s,r,a,h,l,c,d,u,p,f,_,m,g=Math.floor(t);if(1===e.bezierDegree)g===e.numberPoints-1&&g--;else{if(3!==e.bezierDegree)return 0;a=(t*((m="sector"===e.elType?(_=e.points.slice(3,e.numberPoints-3)).length:(_=e.points).length)-1)-(g=3*Math.floor(t*(m-1)/3)))/3,g>=m-1&&(g=m-4,a=1)}if(g<0)return 1;switch(1===e.bezierDegree?(s=e.points[g].usrCoords,r=e.points[g+1].usrCoords):(h=_[g].usrCoords,l=_[g+1].usrCoords,c=_[g+2].usrCoords,d=_[g+3].usrCoords,u=(1-a)*(1-a)*(l[1]-h[1])+2*(1-a)*a*(c[1]-l[1])+a*a*(d[1]-c[1]),p=(1-a)*(1-a)*(l[2]-h[2])+2*(1-a)*a*(c[2]-l[2])+a*a*(d[2]-c[2]),u/=f=o.Z.hypot(u,p),p/=f,r=[1,(s=n.coords.usrCoords)[1]+u,s[2]+p]),i){case 0:return s[2]*r[1]-s[1]*r[2];case 1:return r[2]-s[2];case 2:return s[1]-r[1];default:return[s[2]*r[1]-s[1]*r[2],r[2]-s[2],s[1]-r[1]]}},u=t.create("line",[function(){var e;return e=n.type===i.Z.OBJECT_TYPE_GLIDER?n.position:h.Z.projectPointToCurve(n,a,t)[1],f(e,a)}],p),n.addChild(u),u.glider=n)}else a.type===i.Z.OBJECT_TYPE_TURTLE?(u=t.create("line",[function(){var e,s;for(s=n.type===i.Z.OBJECT_TYPE_GLIDER?n.position:h.Z.projectPointToTurtle(n,a,t)[1],e=Math.floor(s),l=0;l<a.objects.length;l++)if((c=a.objects[l]).type===i.Z.OBJECT_TYPE_CURVE){if(e<c.numberPoints)break;e-=c.numberPoints}return e===c.numberPoints-1&&e--,e<0?[1,0,0]:[c.Y(e)*c.X(e+1)-c.X(e)*c.Y(e+1),c.Y(e+1)-c.Y(e),c.X(e)-c.X(e+1)]}],p),n.addChild(u),u.glider=n):a.elementClass!==i.Z.OBJECT_CLASS_CIRCLE&&a.type!==i.Z.OBJECT_TYPE_CONIC||(u=t.create("line",[function(){return o.Z.matVecMult(a.quadraticform,n.coords.usrCoords)}],p),n.addChild(u),u.glider=n);if(!s.Z.exists(u))throw new Error("JSXGraph: Couldn't create tangent with the given parents.");return u.elType="tangent",u.type=i.Z.OBJECT_TYPE_TANGENT,u.setParents(e),u},t.Z.createNormal=function(t,e,r){var n,a,l,c,u,p,f,_,m,g,b=[];for(c=0;c<e.length;++c)e[c]=t.select(e[c]);if(1===e.length)n=e[0],a=n.slideObject;else{if(2!==e.length)throw new Error("JSXGraph: Can't create normal with parent types '"+typeof e[0]+"' and '"+typeof e[1]+"'.\nPossible parent types: [point,line], [point,circle], [glider]");if(s.Z.isPointType(t,e[0]))n=s.Z.providePoints(t,[e[0]],r,"point")[0],a=e[1];else{if(!s.Z.isPointType(t,e[1]))throw new Error("JSXGraph: Can't create normal with parent types '"+typeof e[0]+"' and '"+typeof e[1]+"'.\nPossible parent types: [point,line], [point,circle], [glider]");a=e[0],n=s.Z.providePoints(t,[e[1]],r,"point")[0]}}if(u=s.Z.copyAttributes(r,t.options,"normal"),a.elementClass===i.Z.OBJECT_CLASS_LINE)f=s.Z.copyAttributes(r,t.options,"normal","point"),p=t.create("point",[function(){var t=o.Z.crossProduct([1,0,0],a.stdform);return[t[0],-t[2],t[1]]}],f),p.isDraggable=!0,(l=t.create("line",[n,p],u)).point=p,l.subs={point:p},l.inherits.push(p);else if(a.elementClass===i.Z.OBJECT_CLASS_CIRCLE)l=t.create("line",[a.midpoint,n],u);else if(a.elementClass===i.Z.OBJECT_CLASS_CURVE){if(m=a.getTransformationSource(),g=m[0])for(b.push(a);m[0]&&s.Z.exists(m[1]._transformationSource);)b.push(m[1]),m=m[1].getTransformationSource();"plot"!==a.evalVisProp("curvetype")||g?l=t.create("line",[function(){var e,s,r,l,c,u,p,f,_=a.X,m=a.Y;if(f=n.type===i.Z.OBJECT_TYPE_GLIDER?n.position:"functiongraph"===a.evalVisProp("curvetype")?n.X():h.Z.projectPointToCurve(n,a,t)[1],p=n.Coords(!0),g){for(_=(c=b[b.length-1]._transformationSource).X,m=c.Y,l=0;l<b.length;l++)b[l].updateTransformMatrix(),u=o.Z.inverse(b[l].transformMat),p=o.Z.matVecMult(u,p);n.type!==i.Z.OBJECT_TYPE_GLIDER&&(p[1]/=p[0],p[2]/=p[0],p[0]/=p[0],f=h.Z.projectCoordsToCurve(p[1],p[2],0,c,t)[1])}if(e=d.Z.D(m)(f),s=d.Z.D(_)(f),r=[-p[1]*s-p[2]*e,p[0]*s,p[0]*e],g)for(l=b.length-1;l>=0;l--)u=o.Z.transpose(o.Z.inverse(b[l].transformMat)),r=o.Z.matVecMult(u,r);return r}],u):(_=function(t,e,i){var s,r,a,h,l,c,d,u,p,f,_,m,g,b,v,Z,C=Math.floor(t);if(1===e.bezierDegree)C===e.numberPoints-1&&C--,h=t;else{if(3!==e.bezierDegree)return 0;h=(t*((Z="sector"===e.elType?(v=e.points.slice(3,e.numberPoints-3)).length:(v=e.points).length)-1)-(C=3*Math.floor(t*(Z-1)/3)))/3,C>=Z-1&&(C=Z-4,h=1)}if(C<0)return 1;switch(s=h-C,1===e.bezierDegree?(r=e.points[C].usrCoords,a=e.points[C+1].usrCoords,g=[r[0]+s*(a[0]-r[0]),r[1]+s*(a[1]-r[1]),r[2]+s*(a[2]-r[2])],m=o.Z.crossProduct(r,a),b=[(b=o.Z.crossProduct([1,0,0],m))[0],-b[2],b[1]],m=o.Z.crossProduct(g,b)):(l=v[C].usrCoords,c=v[C+1].usrCoords,d=v[C+2].usrCoords,u=v[C+3].usrCoords,p=(1-h)*(1-h)*(c[1]-l[1])+2*(1-h)*h*(d[1]-c[1])+h*h*(u[1]-d[1]),f=(1-h)*(1-h)*(c[2]-l[2])+2*(1-h)*h*(d[2]-c[2])+h*h*(u[2]-d[2]),p/=_=o.Z.hypot(p,f),f/=_,a=[1,(r=n.coords.usrCoords)[1]-f,r[2]+p],m=[r[2]*a[1]-r[1]*a[2],a[2]-r[2],r[1]-a[1]]),i){case 0:return m[0];case 1:return m[1];case 2:return m[2];default:return m}},l=t.create("line",[function(){var e;return e=n.type===i.Z.OBJECT_TYPE_GLIDER?n.position:h.Z.projectPointToCurve(n,a,t)[1],_(e,a)}],u),n.addChild(l),l.glider=n)}else{if(a.type!==i.Z.OBJECT_TYPE_TURTLE)throw new Error("JSXGraph: Can't create normal with parent types '"+typeof e[0]+"' and '"+typeof e[1]+"'.\nPossible parent types: [point,line], [point,circle], [glider]");l=t.create("line",[function(){var t,e,s=Math.floor(n.position),r=n.position-s;for(e=0;e<a.objects.length;e++)if((t=a.objects[e]).type===i.Z.OBJECT_TYPE_CURVE){if(s<t.numberPoints)break;s-=t.numberPoints}return s===t.numberPoints-1&&(s-=1,r=1),s<0?1:(t.Y(s)+r*(t.Y(s+1)-t.Y(s)))*(t.Y(s)-t.Y(s+1))-(t.X(s)+r*(t.X(s+1)-t.X(s)))*(t.X(s+1)-t.X(s))},function(){var t,e,s=Math.floor(n.position);for(e=0;e<a.objects.length;e++)if((t=a.objects[e]).type===i.Z.OBJECT_TYPE_CURVE){if(s<t.numberPoints)break;s-=t.numberPoints}return s===t.numberPoints-1&&(s-=1),s<0?0:t.X(s+1)-t.X(s)},function(){var t,e,s=Math.floor(n.position);for(e=0;e<a.objects.length;e++)if((t=a.objects[e]).type===i.Z.OBJECT_TYPE_CURVE){if(s<t.numberPoints)break;s-=t.numberPoints}return s===t.numberPoints-1&&(s-=1),s<0?0:t.Y(s+1)-t.Y(s)}],u)}return l.elType="normal",l.setParents(e),s.Z.exists(n._is_new)?(l.addChild(n),delete n._is_new):n.addChild(l),a.addChild(l),l},t.Z.createRadicalAxis=function(t,e,s){var r,n,a;if(2!==e.length||e[0].elementClass!==i.Z.OBJECT_CLASS_CIRCLE||e[1].elementClass!==i.Z.OBJECT_CLASS_CIRCLE)throw new Error("JSXGraph: Can't create 'radical axis' with parent types '"+typeof e[0]+"' and '"+typeof e[1]+"'.\nPossible parent type: [circle,circle]");return n=t.select(e[0]),a=t.select(e[1]),(r=t.create("line",[function(){var t=n.stdform,e=a.stdform;return o.Z.matVecMult(o.Z.transpose([t.slice(0,3),e.slice(0,3)]),[e[3],-t[3]])}],s)).elType="radicalaxis",r.setParents([n.id,a.id]),n.addChild(r),a.addChild(r),r},t.Z.createPolarLine=function(t,e,r){var o,n,a,h,l,c,d;if(e.length>1&&(h=e[0].type===i.Z.OBJECT_TYPE_CONIC||e[0].elementClass===i.Z.OBJECT_CLASS_CIRCLE,l=e[1].type===i.Z.OBJECT_TYPE_CONIC||e[1].elementClass===i.Z.OBJECT_CLASS_CIRCLE,c=s.Z.isPoint(e[0]),d=s.Z.isPoint(e[1])),2!==e.length||!(h&&d||c&&l))throw new Error("JSXGraph: Can't create 'polar line' with parent types '"+typeof e[0]+"' and '"+typeof e[1]+"'.\nPossible parent type: [conic|circle,point], [point,conic|circle]");return d?(n=t.select(e[0]),a=t.select(e[1])):(n=t.select(e[1]),a=t.select(e[0])),(o=t.create("tangent",[n,a],r)).elType="polarline",o},t.Z.createTangentTo=function(t,e,r){var o,n,a,h,l,c,d;if(a=t.select(e[0]),h=s.Z.providePoints(t,e[1],r,"point")[0],l=s.Z.def(e[2],0),a.type!==i.Z.OBJECT_TYPE_CIRCLE&&a.type!==i.Z.OBJECT_TYPE_CONIC||h.elementClass!==i.Z.OBJECT_CLASS_POINT)throw new Error("JSXGraph: Can't create tangentto with parent types '"+typeof e[0]+"' and '"+typeof e[1]+"' and '"+typeof e[2]+"'.\nPossible parent types: [circle|conic,point,number]");return n=s.Z.copyAttributes(r,t.options,"tangentto"),d=t.create("polar",[a,h],n.polar),c=t.create("intersection",[d,a,l],n.point),(o=t.create("tangent",[a,c],n)).point=c,o.polar=d,o.elType="tangentto",o},t.Z.registerElement("tangent",t.Z.createTangent),t.Z.registerElement("normal",t.Z.createNormal),t.Z.registerElement("tangentto",t.Z.createTangentTo),t.Z.registerElement("polar",t.Z.createTangent),t.Z.registerElement("radicalaxis",t.Z.createRadicalAxis),t.Z.registerElement("polarline",t.Z.createPolarLine);t.Z.Line;t.Z.Group=function(t,e,r,o,n){var a,h,l,c;for(this.board=t,this.objects={},a=this.board.numObjects,this.board.numObjects+=1,""!==e&&s.Z.exists(e)?this.id=e:this.id=this.board.id+"Group"+a,this.board.groups[this.id]=this,this.type=i.Z.OBJECT_TYPE_POINT,this.elementClass=i.Z.OBJECT_CLASS_POINT,""!==r&&s.Z.exists(r)?this.name=r:this.name="group_"+this.board.generateName(this),delete this.type,this.coords={},this.needsRegularUpdate=n.needsregularupdate,this.rotationCenter="centroid",this.scaleCenter=null,this.rotationPoints=[],this.translationPoints=[],this.scalePoints=[],this.scaleDirections={},this.parents=[],h=s.Z.isArray(o)?o:Array.prototype.slice.call(arguments,3),l=0;l<h.length;l++)!(c=this.board.select(h[l])).evalVisProp("fixed")&&s.Z.exists(c.coords)&&this.addPoint(c);this.methodMap={ungroup:"ungroup",add:"addPoint",addPoint:"addPoint",addPoints:"addPoints",addGroup:"addGroup",remove:"removePoint",removePoint:"removePoint",setAttribute:"setAttribute",setProperty:"setAttribute"}},t.Z.extend(t.Z.Group.prototype,{ungroup:function(){var t,e,i;for(t in this.objects)this.objects.hasOwnProperty(t)&&(e=this.objects[t].point,s.Z.isArray(e.groups)&&(i=s.Z.indexOf(e.groups,this.id))>=0&&delete e.groups[i]);return this.objects={},this},addParents:function(t){var e,i,r;for(i=(r=s.Z.isArray(t)?t:arguments).length,e=0;e<i;++e)s.Z.isId(this.board,r[e])?this.parents.push(r[e]):s.Z.exists(r[e].id)&&this.parents.push(r[e].id);this.parents=s.Z.uniqueArray(this.parents)},setParents:function(t){return this.parents=[],this.addParents(t),this},getParents:function(){return s.Z.isArray(this.parents)?this.parents:[]},_updateCoordsCache:function(t){var e;""!==t&&s.Z.exists(this.objects[t])&&(e=this.objects[t].point,this.coords[e.id]={usrCoords:e.coords.usrCoords.slice(0)})},update:function(){var e,r,n,a,l,c,d,u,p,f,_,m,g=null;if(!this.needsUpdate)return this;if("nothing"===(r=this._update_find_drag_type()).action)return this._updateCoordsCache(r.id),this;if(g=this.objects[r.id].point,"translation"===r.action){if(f=[g.coords.usrCoords[1]-this.coords[r.id].usrCoords[1],g.coords.usrCoords[2]-this.coords[r.id].usrCoords[2]],g.elementClass!==i.Z.OBJECT_CLASS_POINT){for(f.unshift(0),e=0;e<g.transformations.length;e++)f=o.Z.matVecMult(g.transformations[e].matrix,f);f.shift()}(_=this.board.create("transform",f,{type:"translate"})).update()}else if("rotation"===r.action||"scaling"===r.action){if(a="rotation"===r.action?"rotationCenter":"scaleCenter",s.Z.exists(this[a].coords))m=this[a].coords.usrCoords.slice(1);else if("centroid"===this[a])m=this._update_centroid_center();else if(s.Z.isArray(this[a]))m=this[a];else{if(!s.Z.isFunction(this[a]))return t.Z.debug("Group.update: No valid center for this transformation, get out of here."),this;m=this[a]()}if("rotation"===r.action)p=h.Z.rad(this.coords[r.id].usrCoords.slice(1),m,this.objects[r.id].point),(f=this.board.create("transform",[p,m[0],m[1]],{type:"rotate"})).update();else{if("scaling"!==r.action)return this;if(c=h.Z.distance(this.coords[r.id].usrCoords.slice(1),m),Math.abs(c)<o.Z.eps)return this;c=h.Z.distance(g.coords.usrCoords.slice(1),m)/c,d=this.scaleDirections[r.id].indexOf("x")>=0?c:1,u=this.scaleDirections[r.id].indexOf("y")>=0?c:1,(f=this.board.create("transform",[1,0,0,m[0]*(1-d),d,0,m[1]*(1-u),0,u],{type:"generic"})).update()}}for(n in this.objects)(g=this.objects[n].point).elementClass!==i.Z.OBJECT_CLASS_POINT&&(s.Z.exists(f.board)?f.meltTo(g):r.id!==g.id&&_.meltTo(g));for(n in this._update_apply_transformation(r,f),this.needsUpdate=!1,this.objects)if(this.objects.hasOwnProperty(n))for(l in this.objects[n].descendants)this.objects[n].descendants.hasOwnProperty(l)&&(this.objects[n].descendants.needsUpdate=this.objects[n].descendants.needsRegularUpdate||this.board.needsFullUpdate);for(n in this.board.updateElements(r),this.objects)this.objects.hasOwnProperty(n)&&this._updateCoordsCache(n);return this},_update_find_drag_type:function(){var t,e,r,n="nothing",a=[];for(t in this.objects)this.objects.hasOwnProperty(t)&&(e=this.objects[t].point).coords.distance(i.Z.COORDS_BY_USER,this.coords[t])>o.Z.eps&&a.push(e.id);return 0===a.length?{action:n,id:"",changed:a}:(r=a[0],e=this.objects[r].point,a.length>1?n="translation":s.Z.isInArray(this.rotationPoints,e)&&s.Z.exists(this.rotationCenter)?n="rotation":s.Z.isInArray(this.scalePoints,e)&&s.Z.exists(this.scaleCenter)?n="scaling":s.Z.isInArray(this.translationPoints,e)&&(n="translation"),{action:n,id:r,changed:a})},_update_centroid_center:function(){var t,e,i;for(i in t=[0,0],e=0,this.coords)this.coords.hasOwnProperty(i)&&(t[0]+=this.coords[i].usrCoords[1],t[1]+=this.coords[i].usrCoords[2],++e);return e>0&&(t[0]/=e,t[1]/=e),t},_update_apply_transformation:function(t,e){var r,n;for(r in this.objects)this.objects.hasOwnProperty(r)&&(s.Z.exists(this.board.objects[r])?(n=this.objects[r].point).id!==t.id?"translation"===t.action?s.Z.isInArray(t.changed,n.id)||n.elementClass===i.Z.OBJECT_CLASS_POINT&&n.coords.setCoordinates(i.Z.COORDS_BY_USER,[this.coords[r].usrCoords[1]+e[0],this.coords[r].usrCoords[2]+e[1]]):"rotation"!==t.action&&"scaling"!==t.action||n.elementClass===i.Z.OBJECT_CLASS_POINT&&e.applyOnce([n]):"rotation"!==t.action&&"scaling"!==t.action||n.coords.setCoordinates(i.Z.COORDS_BY_USER,o.Z.matVecMult(e.matrix,this.coords[n.id].usrCoords)):delete this.objects[r])},addPoint:function(t){return this.objects[t.id]={point:this.board.select(t)},this._updateCoordsCache(t.id),this.translationPoints.push(t),t.groups.push(this.id),t.groups=s.Z.uniqueArray(t.groups),this},addPoints:function(t){var e;for(e=0;e<t.length;e++)this.addPoint(t[e]);return this},addGroup:function(t){var e;for(e in t.objects)t.objects.hasOwnProperty(e)&&this.addPoint(t.objects[e].point);return this},removePoint:function(t){return delete this.objects[t.id],this},setRotationCenter:function(t){return this.rotationCenter=t,this},setRotationPoints:function(t){return this._setActionPoints("rotation",t)},addRotationPoint:function(t){return this._addActionPoint("rotation",t)},removeRotationPoint:function(t){return this._removeActionPoint("rotation",t)},setTranslationPoints:function(t){return this._setActionPoints("translation",t)},addTranslationPoint:function(t){return this._addActionPoint("translation",t)},removeTranslationPoint:function(t){return this._removeActionPoint("translation",t)},setScaleCenter:function(t){return this.scaleCenter=t,this},setScalePoints:function(t,e){var i,r,o;for(o=(i=s.Z.isArray(t)?t:arguments).length,r=0;r<o;++r)this.scaleDirections[this.board.select(i[r]).id]=e||"xy";return this._setActionPoints("scale",t)},addScalePoint:function(t,e){return this._addActionPoint("scale",t),this.scaleDirections[this.board.select(t).id]=e||"xy",this},removeScalePoint:function(t){return this._removeActionPoint("scale",t)},_setActionPoints:function(t,e){var i,r,o;for(o=(i=s.Z.isArray(e)?e:arguments).length,this[t+"Points"]=[],r=0;r<o;++r)this._addActionPoint(t,i[r]);return this},_addActionPoint:function(t,e){return this[t+"Points"].push(this.board.select(e)),this},_removeActionPoint:function(t,e){var i=this[t+"Points"].indexOf(this.board.select(e));return i>-1&&this[t+"Points"].splice(i,1),this},setProperty:function(){t.Z.deprecated("Group.setProperty","Group.setAttribute()"),this.setAttribute.apply(this,arguments)},setAttribute:function(){var t;for(t in this.objects)this.objects.hasOwnProperty(t)&&this.objects[t].point.setAttribute.apply(this.objects[t].point,arguments);return this}}),t.Z.createGroup=function(e,i,r){var o=s.Z.copyAttributes(r,e.options,"group"),n=new t.Z.Group(e,o.id,o.name,i,o);return n.elType="group",n.setParents(i),n},t.Z.registerElement("group",t.Z.createGroup);t.Z.Group;t.Z.Circle=function(t,e,r,o,n){this.constructor(t,n,i.Z.OBJECT_TYPE_CIRCLE,i.Z.OBJECT_CLASS_CIRCLE),this.method=e,this.midpoint=this.board.select(r),this.center=this.board.select(r),this.point2=null,this.radius=0,this.line=null,this.circle=null,this.points=[],"twoPoints"===e?(this.point2=t.select(o),this.radius=this.Radius()):"pointRadius"===e?(this.gxtterm=o,this.updateRadius=s.Z.createFunction(o,this.board),this.updateRadius(),this.addParentsFromJCFunctions([this.updateRadius])):"pointLine"===e?(this.line=t.select(o),this.radius=this.line.point1.coords.distance(i.Z.COORDS_BY_USER,this.line.point2.coords)):"pointCircle"===e&&(this.circle=t.select(o),this.radius=this.circle.Radius()),this.id=this.board.setId(this,"C"),this.board.renderer.drawEllipse(this),this.board.finalizeAdding(this),this.createGradient(),this.elType="circle",this.createLabel(),s.Z.exists(this.center._is_new)?(this.addChild(this.center),delete this.center._is_new):this.center.addChild(this),"pointRadius"===e?this.notifyParents(o):"pointLine"===e?this.line.addChild(this):"pointCircle"===e?this.circle.addChild(this):"twoPoints"===e&&(s.Z.exists(this.point2._is_new)?(this.addChild(this.point2),delete this.point2._is_new):this.point2.addChild(this)),this.methodMap=s.Z.deepCopy(this.methodMap,{setRadius:"setRadius",getRadius:"getRadius",Area:"Area",area:"Area",Perimeter:"Perimeter",Circumference:"Perimeter",radius:"Radius",Radius:"Radius",Diameter:"Diameter",center:"center",line:"line",point2:"point2"})},t.Z.Circle.prototype=new I.Z,t.Z.extend(t.Z.Circle.prototype,{hasPoint:function(t,e){var r,n,a,h,l,c=this.center.coords.usrCoords,d=new u.Z(i.Z.COORDS_BY_SCREEN,[t,e],this.board),p=this.Radius();return s.Z.isObject(this.evalVisProp("precision"))?(n=this.board._inputDevice,r=this.evalVisProp("precision."+n)):r=this.board.options.precision.hasPoint,a=c[1]-d.usrCoords[1],h=c[2]-d.usrCoords[2],l=o.Z.hypot(a,h),r+=.5*this.evalVisProp("strokewidth"),r/=Math.sqrt(Math.abs(this.board.unitX*this.board.unitY)),this.evalVisProp("hasinnerpoints")?l<p+r:Math.abs(l-p)<r},generatePolynomial:function(t){var e=this.center.symbolic.x,i=this.center.symbolic.y,s=t.symbolic.x,r=t.symbolic.y,o=this.generateRadiusSquared();return""===o?[]:["(("+s+")-("+e+"))^2 + (("+r+")-("+i+"))^2 - ("+o+")"]},generateRadiusSquared:function(){var t,e,i,r,o="";return"twoPoints"===this.method?(t=this.center.symbolic.x,e=this.center.symbolic.y,o="(("+(i=this.point2.symbolic.x)+")-("+t+"))^2 + (("+(r=this.point2.symbolic.y)+")-("+e+"))^2"):"pointRadius"===this.method?s.Z.isNumber(this.radius)&&(o=(this.radius*this.radius).toString()):"pointLine"===this.method?(i=this.line.point1.symbolic.x,r=this.line.point1.symbolic.y,o="(("+i+")-("+this.line.point2.symbolic.x+"))^2 + (("+r+")-("+this.line.point2.symbolic.y+"))^2"):"pointCircle"===this.method&&(o=this.circle.Radius()),o},update:function(){var t,e,s,r,o,n;if(this.needsUpdate)for(this.evalVisProp("trace")&&this.cloneToBackground(!0),"pointLine"===this.method?this.radius=this.line.point1.coords.distance(i.Z.COORDS_BY_USER,this.line.point2.coords):"pointCircle"===this.method?this.radius=this.circle.Radius():"pointRadius"===this.method&&(this.radius=this.updateRadius()),this.radius=Math.abs(this.radius),this.updateStdform(),this.updateQuadraticform(),s=this.center.coords.usrCoords[0],t=this.center.coords.usrCoords[1]/s,e=this.center.coords.usrCoords[2]/s,s/=s,r=this.Radius(),o=.551915024494,this.numberPoints=13,this.dataX=[t+r,t+r,t+r*o,t,t-r*o,t-r,t-r,t-r,t-r*o,t,t+r*o,t+r,t+r],this.dataY=[e,e+r*o,e+r,e+r,e+r,e+r*o,e,e-r*o,e-r,e-r,e-r,e-r*o,e],this.bezierDegree=3,n=0;n<this.numberPoints;n++)this.points[n]=new u.Z(i.Z.COORDS_BY_USER,[this.dataX[n],this.dataY[n]],this.board);return this},updateQuadraticform:function(){var t=this.center,e=t.X(),i=t.Y(),s=this.Radius();this.quadraticform=[[e*e+i*i-s*s,-e,-i],[-e,1,0],[-i,0,1]]},updateStdform:function(){this.stdform[3]=.5,this.stdform[4]=this.Radius(),this.stdform[1]=-this.center.coords.usrCoords[1],this.stdform[2]=-this.center.coords.usrCoords[2],isFinite(this.stdform[4])||(this.stdform[0]=s.Z.exists(this.point2)?-(this.stdform[1]*this.point2.coords.usrCoords[1]+this.stdform[2]*this.point2.coords.usrCoords[2]):0),this.normalize()},updateRenderer:function(){return this.needsUpdate?(this.visPropCalc.visible&&(this.isReal=!isNaN(this.center.coords.usrCoords[1]+this.center.coords.usrCoords[2]+this.Radius())&&this.center.isReal,this.isReal||this.updateVisibility(!1)),this.visPropCalc.visible&&this.board.renderer.updateEllipse(this),this.hasLabel&&this.visPropCalc.visible&&this.label&&this.label.visPropCalc.visible&&this.isReal&&(this.label.update(),this.board.renderer.updateText(this.label)),this.setDisplayRendNode(),this.needsUpdate=!1,this):this},notifyParents:function(t){s.Z.isString(t)&&M.Z.findDependencies(this,t,this.board)},setRadius:function(t){return this.updateRadius=s.Z.createFunction(t,this.board),this.addParentsFromJCFunctions([this.updateRadius]),this.board.update(),this},Radius:function(t){return s.Z.exists(t)?(this.setRadius(t),this.Radius()):"twoPoints"===this.method?s.Z.cmpArrays(this.point2.coords.usrCoords,[0,0,0])||s.Z.cmpArrays(this.center.coords.usrCoords,[0,0,0])?NaN:this.center.Dist(this.point2):"pointLine"===this.method||"pointCircle"===this.method?this.radius:"pointRadius"===this.method?this.evalVisProp("nonnegativeonly")?Math.max(0,this.updateRadius()):Math.abs(this.updateRadius()):NaN},Diameter:function(){return 2*this.Radius()},getRadius:function(){return t.Z.deprecated("Circle.getRadius()","Circle.Radius()"),this.Radius()},getTextAnchor:function(){return this.center.coords},getLabelAnchor:function(){var t,e,r,o,n,a,h=1.5,l=this.Radius(),c=this.center.coords.usrCoords,d=.7071067811865;if(!s.Z.exists(this.label))return new u.Z(i.Z.COORDS_BY_SCREEN,[NaN,NaN],this.board);if(r=this.label.evalVisProp("position"),!s.Z.isString(r))return new u.Z(i.Z.COORDS_BY_SCREEN,[NaN,NaN],this.board);if(!(r.indexOf("right")<0&&r.indexOf("left")<0))return c=this.center.coords.scrCoords,o=s.Z.parsePosition(r),n=s.Z.parseNumber(o.pos,2*Math.PI,1),o.pos.indexOf("fr")<0&&o.pos.indexOf("%")<0&&(o.pos.indexOf("px")>=0?n=0:n*=Math.PI/180),a=1,"left"===o.side&&(a=-1),s.Z.exists(this.label)&&(h=.5*a*this.label.evalVisProp("distance")),t=c[1]+(l*this.board.unitX+this.label.size[0]*h)*Math.cos(n),e=c[2]-(l*this.board.unitY+this.label.size[1]*h)*Math.sin(n),new u.Z(i.Z.COORDS_BY_SCREEN,[t,e],this.board);switch(this.evalVisProp("label.position")){case"lft":t=c[1]-l,e=c[2];break;case"llft":t=c[1]-d*l,e=c[2]-d*l;break;case"rt":t=c[1]+l,e=c[2];break;case"lrt":t=c[1]+d*l,e=c[2]-d*l;break;case"urt":t=c[1]+d*l,e=c[2]+d*l;break;case"top":t=c[1],e=c[2]+l;break;case"bot":t=c[1],e=c[2]-l;break;default:t=c[1]-d*l,e=c[2]+d*l}return new u.Z(i.Z.COORDS_BY_USER,[t,e],this.board)},cloneToBackground:function(){var t,e=this.Radius(),i=s.Z.getCloneObject(this);return i.center={coords:this.center.coords},i.Radius=function(){return e},i.getRadius=function(){return e},t=this.board.renderer.enhancedRendering,this.board.renderer.enhancedRendering=!0,this.board.renderer.drawEllipse(i),this.board.renderer.enhancedRendering=t,this.traces[i.id]=i.rendNode,this},addTransform:function(t){var e,i=s.Z.isArray(t)?t:[t],r=i.length;for(e=0;e<r;e++)this.center.transformations.push(i[e]),"twoPoints"===this.method&&this.point2.transformations.push(i[e]);return this},snapToGrid:function(){var t=this.evalVisProp("snaptogrid");return this.center.handleSnapToGrid(t,!0),"twoPoints"===this.method&&this.point2.handleSnapToGrid(t,!0),this},snapToPoints:function(){var t=this.evalVisProp("snaptopoints");return this.center.handleSnapToPoints(t),"twoPoints"===this.method&&this.point2.handleSnapToPoints(t),this},X:function(t){return this.Radius()*Math.cos(2*t*Math.PI)+this.center.coords.usrCoords[1]},Y:function(t){return this.Radius()*Math.sin(2*t*Math.PI)+this.center.coords.usrCoords[2]},Z:function(t){return 1},minX:function(){return 0},maxX:function(){return 1},Area:function(){var t=this.Radius();return t*t*Math.PI},Perimeter:function(){return 2*this.Radius()*Math.PI},bounds:function(){var t=this.center.coords.usrCoords,e=this.Radius();return[t[1]-e,t[2]+e,t[1]+e,t[2]-e]},getParents:function(){return 1===this.parents.length?this.parents.concat(this.radius):this.parents}}),t.Z.createCircle=function(e,r,o){var n,a,h,l,c,d=["center","point2"];if(a=[],c=e.select(r[0]),s.Z.isObject(c)&&c.elementClass===i.Z.OBJECT_CLASS_CIRCLE&&s.Z.isTransformationOrArray(r[1]))return l=s.Z.copyAttributes(o,e.options,"circle"),(n=t.Z.createEllipse(e,[c.center,c.center,function(){return 2*c.Radius()}],l)).addTransform(r[1]),n;for(h=0;h<r.length;h++)if(s.Z.isPointType(e,r[h])){if(r.length<3?a.push(s.Z.providePoints(e,[r[h]],o,"circle",[d[h]])[0]):a.push(s.Z.providePoints(e,[r[h]],o,"point")[0]),!1===a[a.length-1])throw new Error("JSXGraph: Can't create circle from this type. Please provide a point type.")}else a.push(r[h]);if(l=s.Z.copyAttributes(o,e.options,"circle"),2===a.length&&s.Z.isPoint(a[0])&&s.Z.isPoint(a[1]))n=new t.Z.Circle(e,"twoPoints",a[0],a[1],l);else if((s.Z.isNumber(a[0])||s.Z.isFunction(a[0])||s.Z.isString(a[0]))&&s.Z.isPoint(a[1]))n=new t.Z.Circle(e,"pointRadius",a[1],a[0],l);else if((s.Z.isNumber(a[1])||s.Z.isFunction(a[1])||s.Z.isString(a[1]))&&s.Z.isPoint(a[0]))n=new t.Z.Circle(e,"pointRadius",a[0],a[1],l);else if(a[0].elementClass===i.Z.OBJECT_CLASS_CIRCLE&&s.Z.isPoint(a[1]))n=new t.Z.Circle(e,"pointCircle",a[1],a[0],l);else if(a[1].elementClass===i.Z.OBJECT_CLASS_CIRCLE&&s.Z.isPoint(a[0]))n=new t.Z.Circle(e,"pointCircle",a[0],a[1],l);else if(a[0].elementClass===i.Z.OBJECT_CLASS_LINE&&s.Z.isPoint(a[1]))n=new t.Z.Circle(e,"pointLine",a[1],a[0],l);else if(a[1].elementClass===i.Z.OBJECT_CLASS_LINE&&s.Z.isPoint(a[0]))n=new t.Z.Circle(e,"pointLine",a[0],a[1],l);else{if(!(3===r.length&&s.Z.isPoint(a[0])&&s.Z.isPoint(a[1])&&s.Z.isPoint(a[2])))throw new Error("JSXGraph: Can't create circle with parent types '"+typeof r[0]+"' and '"+typeof r[1]+"'.\nPossible parent types: [point,point], [point,number], [point,function], [point,circle], [point,point,point], [circle,transformation]");if(!t.Z.elements.circumcircle)throw new Error("JSXGraph: Can't create circle with three points. Please include the circumcircle element (element/composition).");n=t.Z.elements.circumcircle(e,a,l)}for(n.isDraggable=!0,n.setParents(a),n.elType="circle",h=0;h<a.length;h++)s.Z.isPoint(a[h])&&n.inherits.push(a[h]);return n},t.Z.registerElement("circle",t.Z.createCircle);var V=t.Z.Circle;t.Z.createEllipse=function(t,e,r){var o,n,a,h,l,c,d,p=[],f=s.Z.copyAttributes(r,t.options,"conic","foci"),_=s.Z.copyAttributes(r,t.options,"conic","center"),m=s.Z.copyAttributes(r,t.options,"conic");for(c=0;c<2;c++)if(e[c].length>1)p[c]=t.create("point",e[c],f);else if(s.Z.isPoint(e[c]))p[c]=t.select(e[c]);else if(s.Z.isFunction(e[c])&&s.Z.isPoint(e[c]()))p[c]=e[c]();else{if(!s.Z.isString(e[c]))throw new Error("JSXGraph: Can't create Ellipse with parent types '"+typeof e[0]+"' and '"+typeof e[1]+"'.\nPossible parent types: [point,point,point], [point,point,number|function]");p[c]=t.select(e[c])}if(s.Z.isNumber(e[2]))l=s.Z.createFunction(e[2],t);else if(s.Z.isFunction(e[2])&&s.Z.isNumber(e[2]()))l=e[2];else{if(s.Z.isPoint(e[2]))h=t.select(e[2]);else if(e[2].length>1)h=t.create("point",e[2],f);else if(s.Z.isFunction(e[2])&&s.Z.isPoint(e[2]()))h=e[2]();else{if(!s.Z.isString(e[2]))throw new Error("JSXGraph: Can't create Ellipse with parent types '"+typeof e[0]+"' and '"+typeof e[1]+"' and '"+typeof e[2]+"'.\nPossible parent types: [point,point,point], [point,point,number|function]");h=t.select(e[2])}l=function(){return h.Dist(p[0])+h.Dist(p[1])}}for(s.Z.exists(e[4])||(e[4]=2*Math.PI),s.Z.exists(e[3])||(e[3]=0),a=t.create("point",[function(){return.5*(p[0].X()+p[1].X())},function(){return.5*(p[0].Y()+p[1].Y())}],_),(n=t.create("curve",[function(t){return 0},function(t){return 0},e[3],e[4]],m)).majorAxis=l,d=n.hasPoint,o=function(t,e){var i,s,r,o,a,h,c,d,u;e||(s=(i=l())*i,r=p[0].X(),o=p[0].Y(),c=r-(a=p[1].X()),d=o-(h=p[1].Y()),u=(s-r*r-o*o+a*a+h*h)/(2*i),n.quadraticform=[[u*u-a*a-h*h,u*c/i+a,u*d/i+h],[u*c/i+a,c*c/s-1,c*d/s],[u*d/i+h,c*d/s,d*d/s-1]])},n.X=function(t,e){var i=l(),s=p[1].Dist(p[0]),r=.5*(s*s-i*i)/(s*Math.cos(t)-i),n=Math.atan2(p[1].Y()-p[0].Y(),p[1].X()-p[0].X());return e||o(0,e),p[0].X()+Math.cos(n+t)*r},n.Y=function(t,e){var i=l(),s=p[1].Dist(p[0]),r=.5*(s*s-i*i)/(s*Math.cos(t)-i),o=Math.atan2(p[1].Y()-p[0].Y(),p[1].X()-p[0].X());return p[0].Y()+Math.sin(o+t)*r},n.midpoint=n.center=a,n.type=i.Z.OBJECT_TYPE_CONIC,n.subs={center:n.center},n.inherits.push(n.center,p[0],p[1]),s.Z.isPoint(h)&&n.inherits.push(h),n.hasPoint=function(t,e){var s,r,o,n;return this.evalVisProp("hasinnerpoints")?(s=p[0].coords,r=p[1].coords,o=this.majorAxis(),(n=new u.Z(i.Z.COORDS_BY_SCREEN,[t,e],this.board)).distance(i.Z.COORDS_BY_USER,s)+n.distance(i.Z.COORDS_BY_USER,r)<=o):d.apply(this,arguments)},a.addChild(n),c=0;c<2;c++)s.Z.isPoint(p[c])&&p[c].addChild(n);return s.Z.isPoint(h)&&h.addChild(n),n.setParents(e),n},t.Z.createHyperbola=function(t,e,r){var o,n,a,h,l,c,d=[],u=s.Z.copyAttributes(r,t.options,"conic","foci"),p=s.Z.copyAttributes(r,t.options,"conic","center"),f=s.Z.copyAttributes(r,t.options,"conic");for(c=0;c<2;c++)if(e[c].length>1)d[c]=t.create("point",e[c],u);else if(s.Z.isPoint(e[c]))d[c]=t.select(e[c]);else if(s.Z.isFunction(e[c])&&s.Z.isPoint(e[c]()))d[c]=e[c]();else{if(!s.Z.isString(e[c]))throw new Error("JSXGraph: Can't create Hyperbola with parent types '"+typeof e[0]+"' and '"+typeof e[1]+"'.\nPossible parent types: [point,point,point], [point,point,number|function]");d[c]=t.select(e[c])}if(s.Z.isNumber(e[2]))l=s.Z.createFunction(e[2],t);else if(s.Z.isFunction(e[2])&&s.Z.isNumber(e[2]()))l=e[2];else{if(s.Z.isPoint(e[2]))h=t.select(e[2]);else if(e[2].length>1)h=t.create("point",e[2],u);else if(s.Z.isFunction(e[2])&&s.Z.isPoint(e[2]()))h=e[2]();else{if(!s.Z.isString(e[2]))throw new Error("JSXGraph: Can't create Hyperbola with parent types '"+typeof e[0]+"' and '"+typeof e[1]+"' and '"+typeof e[2]+"'.\nPossible parent types: [point,point,point], [point,point,number|function]");h=t.select(e[2])}l=function(){return h.Dist(d[0])-h.Dist(d[1])}}for(s.Z.exists(e[4])||(e[4]=1.0001*Math.PI),s.Z.exists(e[3])||(e[3]=-1.0001*Math.PI),a=t.create("point",[function(){return.5*(d[0].X()+d[1].X())},function(){return.5*(d[0].Y()+d[1].Y())}],p),(n=t.create("curve",[function(t){return 0},function(t){return 0},e[3],e[4]],f)).majorAxis=l,o=function(t,e){var i,s,r,o,a,h,c,u,p;e||(s=(i=l())*i,r=d[0].X(),o=d[0].Y(),c=r-(a=d[1].X()),u=o-(h=d[1].Y()),p=(s-r*r-o*o+a*a+h*h)/(2*i),n.quadraticform=[[p*p-a*a-h*h,p*c/i+a,p*u/i+h],[p*c/i+a,c*c/s-1,c*u/s],[p*u/i+h,c*u/s,u*u/s-1]])},n.X=function(t,e){var i=l(),s=d[1].Dist(d[0]),r=.5*(s*s-i*i)/(s*Math.cos(t)+i),n=Math.atan2(d[1].Y()-d[0].Y(),d[1].X()-d[0].X());return e||o(0,e),d[0].X()+Math.cos(n+t)*r},n.Y=function(t,e){var i=l(),s=d[1].Dist(d[0]),r=.5*(s*s-i*i)/(s*Math.cos(t)+i),o=Math.atan2(d[1].Y()-d[0].Y(),d[1].X()-d[0].X());return d[0].Y()+Math.sin(o+t)*r},n.midpoint=n.center=a,n.subs={center:n.center},n.inherits.push(n.center,d[0],d[1]),s.Z.isPoint(h)&&n.inherits.push(h),n.type=i.Z.OBJECT_TYPE_CONIC,a.addChild(n),c=0;c<2;c++)s.Z.isPoint(d[c])&&d[c].addChild(n);return s.Z.isPoint(h)&&h.addChild(n),n.setParents(e),n},t.Z.createParabola=function(t,e,r){var o,n,a,l,c=e[0],d=e[1],u=s.Z.copyAttributes(r,t.options,"conic","foci"),p=s.Z.copyAttributes(r,t.options,"conic","center"),f=s.Z.copyAttributes(r,t.options,"conic");if(e[0].length>1)c=t.create("point",e[0],u);else if(s.Z.isPoint(e[0]))c=t.select(e[0]);else if(s.Z.isFunction(e[0])&&s.Z.isPoint(e[0]()))c=e[0]();else{if(!s.Z.isString(e[0]))throw new Error("JSXGraph: Can't create Parabola with parent types '"+typeof e[0]+"' and '"+typeof e[1]+"'.\nPossible parent types: [point,line]");c=t.select(e[0])}return s.Z.isArray(d)&&2===d.length&&(l=s.Z.copyAttributes(r,t.options,"conic","line"),d=t.create("line",d,l)),s.Z.exists(e[3])||(e[3]=2*Math.PI),s.Z.exists(e[2])||(e[2]=0),a=t.create("point",[function(){return h.Z.projectPointToLine(c,d,t).usrCoords}],p),(n=t.create("curve",[function(t){return 0},function(t){return 0},e[2],e[3]],f)).midpoint=n.center=a,n.subs={center:n.center},n.inherits.push(n.center),o=function(t,e){var i,s,r,o,a,h;e||(i=d.stdform[1],s=d.stdform[2],r=d.stdform[0],o=i*i+s*s,a=c.X(),h=c.Y(),n.quadraticform=[[r*r-o*(a*a+h*h),r*i+o*a,r*s+o*h],[r*i+o*a,-s*s,i*s],[r*s+o*h,i*s,-i*i]])},n.X=function(t,e){var i,s=d.getAngle(),r=h.Z.distPointLine(c.coords.usrCoords,d.stdform),n=d.point1.coords.usrCoords,a=d.point2.coords.usrCoords,l=c.coords.usrCoords;return 0===n[0]?n=[1,a[1]+d.stdform[2],a[2]-d.stdform[1]]:0===a[0]&&(a=[1,n[1]+d.stdform[2],n[2]-d.stdform[1]]),i=((a[1]-n[1])*(l[2]-n[2])-(a[2]-n[2])*(l[1]-n[1])>=0?1:-1)*r/(1-Math.sin(t)),e||o(0,e),c.X()+Math.cos(t+s)*i},n.Y=function(t,e){var i,s=d.getAngle(),r=h.Z.distPointLine(c.coords.usrCoords,d.stdform),o=d.point1.coords.usrCoords,n=d.point2.coords.usrCoords,a=c.coords.usrCoords;return 0===o[0]?o=[1,n[1]+d.stdform[2],n[2]-d.stdform[1]]:0===n[0]&&(n=[1,o[1]+d.stdform[2],o[2]-d.stdform[1]]),i=((n[1]-o[1])*(a[2]-o[2])-(n[2]-o[2])*(a[1]-o[1])>=0?1:-1)*r/(1-Math.sin(t)),c.Y()+Math.sin(t+s)*i},n.type=i.Z.OBJECT_TYPE_CONIC,a.addChild(n),s.Z.isPoint(c)&&(c.addChild(n),n.inherits.push(c)),d.addChild(n),n.setParents(e),n},t.Z.createConic=function(t,e,r){var n,a,h,l,c,u,p,f,_,m,g,b,v,Z,C=[[1,0,0],[0,1,0],[0,0,1]],y=[[1,0,0],[0,1,0],[0,0,1]],P=[],E=[],x=s.Z.copyAttributes(r,t.options,"conic","point"),O=s.Z.copyAttributes(r,t.options,"conic","center"),M=s.Z.copyAttributes(r,t.options,"conic");if(5===e.length)Z=!0;else{if(6!==e.length)throw new Error("JSXGraph: Can't create generic Conic with "+e.length+" parameters.");Z=!1}if(Z)for(b=0;b<5;b++)if(e[b].length>1)P[b]=t.create("point",e[b],x);else if(s.Z.isPoint(e[b]))P[b]=t.select(e[b]);else if(s.Z.isFunction(e[b])&&s.Z.isPoint(e[b]()))P[b]=e[b]();else{if(!s.Z.isString(e[b]))throw new Error("JSXGraph: Can't create Conic section with parent types '"+typeof e[b]+"'.\nPossible parent types: [point,point,point,point,point], [a00,a11,a22,a01,a02,a12]");P[b]=t.select(e[b])}else(v=[[0,0,0],[0,0,0],[0,0,0]])[0][0]=s.Z.isFunction(e[2])?function(){return e[2]()}:function(){return e[2]},v[0][1]=s.Z.isFunction(e[4])?function(){return e[4]()}:function(){return e[4]},v[0][2]=s.Z.isFunction(e[5])?function(){return e[5]()}:function(){return e[5]},v[1][1]=s.Z.isFunction(e[0])?function(){return e[0]()}:function(){return e[0]},v[1][2]=s.Z.isFunction(e[3])?function(){return e[3]()}:function(){return e[3]},v[2][2]=s.Z.isFunction(e[1])?function(){return e[1]()}:function(){return e[1]};if(c=function(t){var e,i;for(e=0;e<3;e++)for(i=e;i<3;i++)t[e][i]+=t[i][e];for(e=0;e<3;e++)for(i=0;i<e;i++)t[e][i]=t[i][e];return t},l=function(t,e){var i,s,r=[[0,0,0],[0,0,0],[0,0,0]];for(i=0;i<3;i++)for(s=0;s<3;s++)r[i][s]=t[i]*e[s];return c(r)},h=function(t,e,i){var s,r,n,a,h,l=[[0,0,0],[0,0,0],[0,0,0]];for(h=o.Z.matVecMult(e,i),n=o.Z.innerProduct(i,h),h=o.Z.matVecMult(t,i),a=o.Z.innerProduct(i,h),s=0;s<3;s++)for(r=0;r<3;r++)l[s][r]=n*t[s][r]-a*e[s][r];return l},a=t.create("curve",[function(t){return 0},function(t){return 0},0,2*Math.PI],M),n=function(t,e){var i,r,n;if(!e){if(Z){for(i=0;i<5;i++)E[i]=P[i].coords.usrCoords;m=l(o.Z.crossProduct(E[0],E[1]),o.Z.crossProduct(E[2],E[3])),g=l(o.Z.crossProduct(E[0],E[2]),o.Z.crossProduct(E[1],E[3])),y=h(m,g,E[4])}else for(i=0;i<3;i++)for(r=i;r<3;r++)y[i][r]=v[i][r](),r>i&&(y[r][i]=y[i][r]);a.quadraticform=y,(u=d.Z.Jacobi(y))[0][0][0]<0&&(u[0][0][0]*=-1,u[0][1][1]*=-1,u[0][2][2]*=-1),C=u[1],_=Math.sqrt(Math.abs(u[0][0][0])),p=Math.sqrt(Math.abs(u[0][1][1])),f=Math.sqrt(Math.abs(u[0][2][2]))}return u[0][1][1]<=0&&u[0][2][2]<=0?n=o.Z.matVecMult(C,[1/_,Math.cos(t)/p,Math.sin(t)/f]):u[0][1][1]<=0&&u[0][2][2]>0?n=o.Z.matVecMult(C,[Math.cos(t)/_,1/p,Math.sin(t)/f]):u[0][2][2]<0&&(n=o.Z.matVecMult(C,[Math.sin(t)/_,Math.cos(t)/p,1/f])),s.Z.exists(n)?(n[1]/=n[0],n[2]/=n[0],n[0]=1):n=[1,NaN,NaN],n},a.X=function(t,e){return n(t,e)[1]},a.Y=function(t,e){return n(t,e)[2]},a.midpoint=t.create("point",[function(){var t=a.quadraticform;return[t[1][1]*t[2][2]-t[1][2]*t[1][2],t[1][2]*t[0][2]-t[2][2]*t[0][1],t[0][1]*t[1][2]-t[1][1]*t[0][2]]}],O),a.type=i.Z.OBJECT_TYPE_CONIC,a.center=a.midpoint,a.subs={center:a.center},a.inherits.push(a.center),a.inherits=a.inherits.concat(P),Z){for(b=0;b<5;b++)s.Z.isPoint(P[b])&&P[b].addChild(a);a.setParents(e)}return a.addChild(a.center),a},t.Z.registerElement("ellipse",t.Z.createEllipse),t.Z.registerElement("hyperbola",t.Z.createHyperbola),t.Z.registerElement("parabola",t.Z.createParabola),t.Z.registerElement("conic",t.Z.createConic),t.Z.Polygon=function(e,r,o){this.constructor(e,o,i.Z.OBJECT_TYPE_POLYGON,i.Z.OBJECT_CLASS_AREA);var n,a,h,l,c,d=s.Z.copyAttributes(o,e.options,"polygon","borders");for(this.withLines=o.withlines,this.attr_line=d,this.vertices=[],n=0;n<r.length;n++)this.vertices[n]=this.board.select(r[n]),this.vertices[n]._is_new&&(delete this.vertices[n]._is_new,this.vertices[n]._is_new_pol=!0);if(this.vertices.length>0&&this.vertices[this.vertices.length-1].id!==this.vertices[0].id&&this.vertices.push(this.vertices[0]),this.borders=[],this.withLines)for(h=this.vertices.length-1,l=0;l<h;l++)n=(l+1)%h,d.id=d.ids&&d.ids[n],d.name=d.names&&d.names[n],d.strokecolor=s.Z.isArray(d.colors)&&d.colors[n%d.colors.length]||d.strokecolor,d.visible=s.Z.exists(o.borders.visible)?o.borders.visible:o.visible,!1===d.strokecolor&&(d.strokecolor="none"),(a=e.create("segment",[this.vertices[n],this.vertices[n+1]],d)).dump=!1,this.borders[n]=a,a.parentPolygon=this,this.addChild(a);for(this.inherits.push(this.vertices,this.borders),this.id=this.board.setId(this,"Py"),n=0;n<this.vertices.length-1;n++)c=this.board.select(this.vertices[n]),s.Z.exists(c._is_new_pol)?(this.addChild(c),delete c._is_new_pol):c.addChild(this);this.board.renderer.drawPolygon(this),this.board.finalizeAdding(this),this.createGradient(),this.elType="polygon",this.createLabel(),this.methodMap=t.Z.deepCopy(this.methodMap,{borders:"borders",vertices:"vertices",A:"Area",Area:"Area",Perimeter:"Perimeter",L:"Perimeter",boundingBox:"bounds",BoundingBox:"bounds",addPoints:"addPoints",insertPoints:"insertPoints",removePoints:"removePoints",Intersect:"intersect"})},t.Z.Polygon.prototype=new I.Z,t.Z.extend(t.Z.Polygon.prototype,{pnpoly:function(t,e,i){return h.Z.pnpoly(t,e,this.vertices,i,this.board)},hasPoint:function(t,e){var i,s;if(this.evalVisProp("hasinnerpoints")&&this.pnpoly(t,e))return!0;for(s=this.borders.length,i=0;i<s;i++)if(this.borders[i].hasPoint(t,e))return!0;return!1},updateRenderer:function(){var t,e;if(!this.needsUpdate)return this;if(this.visPropCalc.visible){for(e=this.vertices.length-("polygonalchain"===this.elType?0:1),this.isReal=!0,t=0;t<e;++t)if(!this.vertices[t].isReal){this.isReal=!1;break}if(!this.isReal)for(t in this.updateVisibility(!1),this.childElements)this.childElements.hasOwnProperty(t)&&this.childElements[t].setDisplayRendNode(!1)}return this.visPropCalc.visible&&this.board.renderer.updatePolygon(this),this.hasLabel&&this.visPropCalc.visible&&this.label&&this.label.visPropCalc.visible&&this.isReal&&(this.label.update(),this.board.renderer.updateText(this.label)),this.setDisplayRendNode(),this.needsUpdate=!1,this},getTextAnchor:function(){var t,e,s,r,o;if(0===this.vertices.length)return new u.Z(i.Z.COORDS_BY_USER,[1,0,0],this.board);for(s=t=this.vertices[0].X(),r=e=this.vertices[0].Y(),o=0;o<this.vertices.length;o++)this.vertices[o].X()<t&&(t=this.vertices[o].X()),this.vertices[o].X()>s&&(s=this.vertices[o].X()),this.vertices[o].Y()>e&&(e=this.vertices[o].Y()),this.vertices[o].Y()<r&&(r=this.vertices[o].Y());return new u.Z(i.Z.COORDS_BY_USER,[.5*(t+s),.5*(e+r)],this.board)},getLabelAnchor:t.Z.shortcut(t.Z.Polygon.prototype,"getTextAnchor"),cloneToBackground:function(){var t,e=s.Z.getCloneObject(this);return e.vertices=this.vertices,t=this.board.renderer.enhancedRendering,this.board.renderer.enhancedRendering=!0,this.board.renderer.drawPolygon(e),this.board.renderer.enhancedRendering=t,this.traces[e.id]=e.rendNode,this},hideElement:function(e){var i;if(t.Z.deprecated("Element.hideElement()","Element.setDisplayRendNode()"),this.visPropCalc.visible=!1,this.board.renderer.display(this,!1),!e)for(i=0;i<this.borders.length;i++)this.borders[i].hideElement();this.hasLabel&&s.Z.exists(this.label)&&(this.label.hiddenByParent=!0,this.label.visPropCalc.visible&&this.label.hideElement())},showElement:function(e){var i;if(t.Z.deprecated("Element.showElement()","Element.setDisplayRendNode()"),this.visPropCalc.visible=!0,this.board.renderer.display(this,!0),!e)for(i=0;i<this.borders.length;i++)this.borders[i].showElement().updateRenderer();return s.Z.exists(this.label)&&this.hasLabel&&this.label.hiddenByParent&&(this.label.hiddenByParent=!1,this.label.visPropCalc.visible||this.label.showElement().updateRenderer()),this},Area:function(){return Math.abs(h.Z.signedPolygon(this.vertices,!0))},Perimeter:function(){var t,e=this.vertices.length,i=0;for(t=1;t<e;++t)i+=this.vertices[t].Dist(this.vertices[t-1]);return i},L:function(){return this.Perimeter()},boundingBox:function(){var t,e,i=[0,0,0,0],s=this.vertices.length-1;if(0===s)return i;for(i[0]=this.vertices[0].X(),i[2]=i[0],i[1]=this.vertices[0].Y(),i[3]=i[1],t=1;t<s;++t)(e=this.vertices[t].X())<i[0]?i[0]=e:e>i[2]&&(i[2]=e),(e=this.vertices[t].Y())>i[1]?i[1]=e:e<i[3]&&(i[3]=e);return i},bounds:function(){return this.boundingBox()},remove:function(){var t;for(t=0;t<this.borders.length;t++)this.board.removeObject(this.borders[t]);I.Z.prototype.remove.call(this)},findPoint:function(t){var e;if(!s.Z.isPoint(t))return-1;for(e=0;e<this.vertices.length;e++)if(this.vertices[e].id===t.id)return e;return-1},addPoints:function(t){var e,i=Array.prototype.slice.call(arguments);return e="polygonalchain"===this.elType?this.vertices.length-1:this.vertices.length-2,this.insertPoints.apply(this,[e].concat(i))},insertPoints:function(t,e){var i,r,o,n,a;if(0===arguments.length)return this;if(o=this.vertices.length-1,"polygon"===this.elType&&o--,t<-1||t>o)return this;for(r=arguments.length-1,i=1;i<r+1;i++)(a=s.Z.providePoints(this.board,[arguments[i]],{},"polygon",["vertices"])[0])._is_new&&(this.addChild(a),delete a._is_new),this.vertices.splice(t+i,0,a);if(this.withLines)for(n=t+1,"polygon"===this.elType?t<0?(this.vertices[this.vertices.length-1]=this.vertices[0],this.borders[this.borders.length-1].point2=this.vertices[this.vertices.length-1]):this.borders[t].point2=this.vertices[n]:t>=0&&(t<this.borders.length?this.borders[t].point2=this.vertices[n]:n=t),i=n;i<n+r;i++)this.borders.splice(i,0,this.board.create("segment",[this.vertices[i],this.vertices[i+1]],this.attr_line));return this.inherits=[],this.inherits.push(this.vertices,this.borders),this.board.update(),this},removePoints:function(t){var e,i,r,o,n=[],a=[],h=[],l=[];for("polygon"===this.elType&&(o=this.vertices.pop()),e=0;e<arguments.length;e++)r=arguments[e],s.Z.isPoint(r)&&(r=this.findPoint(r)),s.Z.isNumber(r)&&r>-1&&r<this.vertices.length&&-1===s.Z.indexOf(h,r)&&h.push(r);if(0===h.length)return"polygon"===this.elType&&this.vertices.push(o),this;for(e=0;e<h.length;e++)this.vertices[h[e]].removeChild(this);for(h=h.sort(),n=this.vertices.slice(),a=this.borders.slice(),this.withLines&&l.push([h[h.length-1]]),e=h.length-1;e>-1;e--)n[h[e]]=-1,this.withLines&&h.length>1&&h[e]-1>h[e-1]&&(l[l.length-1][1]=h[e],l.push([h[e-1]]));for(this.withLines&&(l[l.length-1][1]=h[0]),this.vertices=[],e=0;e<n.length;e++)s.Z.isPoint(n[e])&&this.vertices.push(n[e]);if("polygon"===this.elType&&this.vertices.length>1&&this.vertices[this.vertices.length-1].id!==this.vertices[0].id&&this.vertices.push(this.vertices[0]),this.withLines){for(e=0;e<l.length;e++){for(i=l[e][1]-1;i<l[e][0]+1;i++)i<0?"polygon"===this.elType&&(this.board.removeObject(this.borders[a.length-1]),a[a.length-1]=-1):i<a.length&&(this.board.removeObject(this.borders[i]),a[i]=-1);0!==l[e][1]&&l[e][0]!==n.length-1&&(a[l[e][0]-1]=this.board.create("segment",[n[l[e][1]-1],n[l[e][0]+1]],this.attr_line))}for(this.borders=[],e=0;e<a.length;e++)-1!==a[e]&&this.borders.push(a[e]);"polygon"===this.elType&&this.vertices.length>2&&(l[0][1]===this.vertices.length-1||0===l[l.length-1][1])&&this.borders.push(this.board.create("segment",[this.vertices[this.vertices.length-2],this.vertices[0]],this.attr_line))}return this.inherits=[],this.inherits.push(this.vertices,this.borders),this.board.update(),this},getParents:function(){return this.setParents(this.vertices),this.parents},getAttributes:function(){var t,e=I.Z.prototype.getAttributes.call(this);if(this.withLines)for(e.lines=e.lines||{},e.lines.ids=[],e.lines.colors=[],t=0;t<this.borders.length;t++)e.lines.ids.push(this.borders[t].id),e.lines.colors.push(this.borders[t].visProp.strokecolor);return e},snapToGrid:function(){var t,e;for(e=!!this.evalVisProp("snaptogrid"),t=0;t<this.vertices.length;t++)this.vertices[t].handleSnapToGrid(e,!0)},setPositionDirectly:function(t,e,i){var s,r,o,n=new u.Z(t,e,this.board),a=new u.Z(t,i,this.board);for(o=this.vertices.length-1,r=0;r<o;r++)if(!this.vertices[r].draggable())return this;return s=p.Z.subtract(n.usrCoords,a.usrCoords),this.board.create("transform",s.slice(1),{type:"translate"}).applyOnce(this.vertices.slice(0,-1)),this},sutherlandHodgman:function(e){var i,s,r,o,n,a,h,l=t.Z.Math.Geometry.sortVertices(this.vertices),c=t.Z.Math.Geometry.sortVertices(e.vertices),d=l.length-1,u=c.length-1,p=[],f=function(t,e,i){return(e[1]-t[1])*(i[2]-t[2])-(e[2]-t[2])*(i[1]-t[1])>=0};for(r=0;r<u;r++)p.push(c[r]);for(r=0;r<d;r++)for(s=p.slice(0),p=[],n=s[(i=s.length)-1],o=0;o<i;o++)a=s[o],f(l[r],l[r+1],a)?(f(l[r],l[r+1],n)||((h=t.Z.Math.Geometry.meetSegmentSegment(n,a,l[r],l[r+1]))[0][1]/=h[0][0],h[0][2]/=h[0][0],h[0][0]=1,p.push(h[0])),p.push(a)):f(l[r],l[r+1],n)&&((h=t.Z.Math.Geometry.meetSegmentSegment(n,a,l[r],l[r+1]))[0][1]/=h[0][0],h[0][2]/=h[0][0],h[0][0]=1,p.push(h[0])),n=a;return p},intersect:function(t){return this.sutherlandHodgman(t)}}),t.Z.createPolygon=function(e,r,o){var n,a,h,l,c,d,u=[],p=!1;if(s.Z.copyAttributes(o,e.options,"polygon"),null===(l=e.select(r[0]))&&(l=r[0]),s.Z.isObject(l)&&l.type===i.Z.OBJECT_TYPE_POLYGON&&s.Z.isTransformationOrArray(r[1]))for(p=!0,h=l.vertices.length-1,d=s.Z.copyAttributes(o,e.options,"polygon","vertices"),a=0;a<h;a++)d.withlabel&&(d.name=""===l.vertices[a].name?"":l.vertices[a].name+"'"),u.push(e.create("point",[l.vertices[a],r[1]],d));else if(!1===(u=s.Z.providePoints(e,r,o,"polygon",["vertices"])))throw new Error("JSXGraph: Can't create polygon / polygonalchain with parent types other than 'point' and 'coordinate arrays' or a function returning an array of coordinates. Alternatively, a polygon and a transformation can be supplied");if(c=s.Z.copyAttributes(o,e.options,"polygon"),(n=new t.Z.Polygon(e,u,c)).isDraggable=!0,p)for(n.prepareUpdate().update().updateVisibility().updateRenderer(),h=l.vertices.length-1,a=0;a<h;a++)u[a].prepareUpdate().update().updateVisibility().updateRenderer();return n},t.Z.createRegularPolygon=function(t,e,r){var o,n,a,h,l,c,d,u=[];if(a=e[(l=e.length)-1],s.Z.isNumber(a)&&(3!==e.length||a<3))throw new Error("JSXGraph: A regular polygon needs two point types and a number > 2 as input.");if(s.Z.isNumber(t.select(a))?(l--,c=!1):(a=l,c=!0),!1===(u=s.Z.providePoints(t,e.slice(0,l),r,"regularpolygon",["vertices"])))throw new Error("JSXGraph: Can't create regular polygon with parent types other than 'point' and 'coordinate arrays' or a function returning an array of coordinates");for(d=s.Z.copyAttributes(r,t.options,"regularpolygon","vertices"),n=2;n<a;n++)h=t.create("transform",[Math.PI*(2-(a-2)/a),u[n-1]],{type:"rotate"}),c?(u[n].addTransform(u[n-2],h),u[n].fullUpdate()):(s.Z.isArray(d.ids)&&d.ids.length>=a-2&&(d.id=d.ids[n-2]),u[n]=t.create("point",[u[n-2],h],d),u[n].type=i.Z.OBJECT_TYPE_CAS,u[n].isDraggable=!0,u[n].visProp.fixed=!1);return d=s.Z.copyAttributes(r,t.options,"regularpolygon"),(o=t.create("polygon",u,d)).elType="regularpolygon",o},t.Z.createPolygonalChain=function(t,e,i){var r,o;return r=s.Z.copyAttributes(i,t.options,"polygonalchain"),(o=t.create("polygon",e,r)).elType="polygonalchain",o.vertices.pop(),t.removeObject(o.borders[o.borders.length-1]),o.borders.pop(),o},t.Z.createParallelogram=function(t,e,i){var r,o,n,a,h;if(!1===(n=s.Z.providePoints(t,e,i,"polygon",["vertices"]))||n.length<3)throw new Error("JSXGraph: Can't create parallelogram with parent types other than 'point' and 'coordinate arrays' or a function returning an array of coordinates.");return h=s.Z.copyAttributes(i,t.options,"parallelogram","parallelpoint"),o=t.create("parallelpoint",n,h),a=s.Z.copyAttributes(i,t.options,"parallelogram"),(r=t.create("polygon",[n[0],n[1],o,n[2]],a)).elType="parallelogram",r.parallelPoint=o,r.isDraggable=!0,o.isDraggable=!0,o.visProp.fixed=!1,r},t.Z.registerElement("polygon",t.Z.createPolygon),t.Z.registerElement("regularpolygon",t.Z.createRegularPolygon),t.Z.registerElement("polygonalchain",t.Z.createPolygonalChain),t.Z.registerElement("parallelogram",t.Z.createParallelogram);var X=t.Z.Polygon;t.Z.Curve=function(t,e,r){this.constructor(t,r,i.Z.OBJECT_TYPE_CURVE,i.Z.OBJECT_CLASS_CURVE),this.points=[],this.numberPoints=this.evalVisProp("numberpointshigh"),this.bezierDegree=1,this.dataX=null,this.dataY=null,this.ticks=[],this.qdt=null,s.Z.exists(e[0])?this.varname=e[0]:this.varname="x",this.xterm=e[1],this.yterm=e[2],this.generateTerm(this.varname,this.xterm,this.yterm,e[3],e[4]),this.updateCurve(),this.id=this.board.setId(this,"G"),this.board.renderer.drawCurve(this),this.board.finalizeAdding(this),this.createGradient(),this.elType="curve",this.createLabel(),s.Z.isString(this.xterm)&&this.notifyParents(this.xterm),s.Z.isString(this.yterm)&&this.notifyParents(this.yterm),this.methodMap=s.Z.deepCopy(this.methodMap,{generateTerm:"generateTerm",setTerm:"generateTerm",move:"moveTo",moveTo:"moveTo",MinX:"minX",MaxX:"maxX"})},t.Z.Curve.prototype=new I.Z,t.Z.extend(t.Z.Curve.prototype,{minX:function(){return"polar"===this.evalVisProp("curvetype")?0:new u.Z(i.Z.COORDS_BY_SCREEN,[.1*-this.board.canvasWidth,0],this.board,!1).usrCoords[1]},maxX:function(){return"polar"===this.evalVisProp("curvetype")?2*Math.PI:new u.Z(i.Z.COORDS_BY_SCREEN,[1.1*this.board.canvasWidth,0],this.board,!1).usrCoords[1]},X:function(t){return NaN},Y:function(t){return NaN},Z:function(t){return 1},hasPoint:function(t,e,r){var n,a,l,c,d,p,f,_,m,g,b,v,Z,C,y,P,E=[],x=this.evalVisProp("numberpointslow"),O=(this.maxX()-this.minX())/x,M=1/0;if(s.Z.isObject(this.evalVisProp("precision"))?(v=this.board._inputDevice,b=this.evalVisProp("precision."+v)):b=this.board.options.precision.hasPoint,t=(p=new u.Z(i.Z.COORDS_BY_SCREEN,[t,e],this.board,!1)).usrCoords[1],e=p.usrCoords[2],1===this.bezierDegree&&this.evalVisProp("hasinnerpoints")&&0!==h.Z.windingNumber([1,t,e],this.points,!0))return!0;if(b+=.5*this.evalVisProp("strokewidth"),b*=b,Z=this.board.unitX*this.board.unitX,C=this.board.unitY*this.board.unitY,P=this.minX(),this.maxX(),s.Z.exists(this._visibleArea)&&(P=this._visibleArea[0],O=(this._visibleArea[1]-P)/x),"parameter"===(y=this.evalVisProp("curvetype"))||"polar"===y)for(this.transformations.length>0&&(this.updateTransformMatrix(),_=o.Z.inverse(this.transformMat),t=(a=o.Z.matVecMult(_,[1,t,e]))[1],e=a[2]),l=0,n=P;l<x;l++){if((M=(t-(c=this.X(n,true)))*(t-c)*Z+(e-(d=this.Y(n,true)))*(e-d)*C)<=b)return!0;n+=O}else if("plot"===y||"functiongraph"===y){for((!s.Z.exists(r)||r<0)&&(r=0),s.Z.exists(this.qdt)&&this.evalVisProp("useqdt")&&3!==this.bezierDegree?f=(m=(g=this.qdt.query(new u.Z(i.Z.COORDS_BY_USER,[t,e],this.board))).points).length:(m=this.points,f=this.numberPoints-1),l=r;l<f;l++)if(3===this.bezierDegree?E=h.Z.projectCoordsToBeziersegment([1,t,e],this,l):g?(m[l].prev&&(E=h.Z.projectCoordsToSegment([1,t,e],m[l].prev.usrCoords,m[l].usrCoords)),m[l].next&&m[l+1]!==m[l].next&&(E=h.Z.projectCoordsToSegment([1,t,e],m[l].usrCoords,m[l].next.usrCoords))):E=h.Z.projectCoordsToSegment([1,t,e],m[l].usrCoords,m[l+1].usrCoords),E[1]>=0&&E[1]<=1&&(t-E[0][1])*(t-E[0][1])*Z+(e-E[0][2])*(e-E[0][2])*C<=b)return!0;return!1}return M<b},allocatePoints:function(){var t,e;if(e=this.numberPoints,this.points.length<this.numberPoints)for(t=this.points.length;t<e;t++)this.points[t]=new u.Z(i.Z.COORDS_BY_USER,[0,0],this.board,!1)},update:function(){return this.needsUpdate&&(this.evalVisProp("trace")&&this.cloneToBackground(!0),this.updateCurve()),this},updateRenderer:function(){return this.needsUpdate?(this.visPropCalc.visible&&(this.isReal=f.checkReal(this.points),this.isReal||this.updateVisibility(!1)),this.visPropCalc.visible&&this.board.renderer.updateCurve(this),this.hasLabel&&this.visPropCalc.visible&&this.label&&this.label.visPropCalc.visible&&this.isReal&&(this.label.update(),this.board.renderer.updateText(this.label)),this.setDisplayRendNode(),this.needsUpdate=!1,this):this},updateDataArray:function(){},updateCurve:function(){var t,e,r,o,n,a,h=this.visProp.plotversion,c=!1;if(this.updateTransformMatrix(),this.updateDataArray(),r=this.minX(),o=this.maxX(),s.Z.exists(this.dataX))for(this.numberPoints=this.dataX.length,e=this.numberPoints,this.allocatePoints(),t=0;t<e;t++)n=t,s.Z.exists(this.dataY)?(a=t,this.points[t].setCoordinates(i.Z.COORDS_BY_USER,[this.dataX[t],this.dataY[t]],!1)):(a=this.X(n),this.points[t].setCoordinates(i.Z.COORDS_BY_USER,[this.dataX[t],this.Y(a,c)],!1)),this.points[t]._t=t,c=!0;else if(this.evalVisProp("doadvancedplot")?1===h||this.evalVisProp("doadvancedplotold")?f.updateParametricCurveOld(this,r,o):2===h?f.updateParametricCurve_v2(this,r,o):3===h?f.updateParametricCurve_v3(this,r,o):4===h?f.updateParametricCurve_v4(this,r,o):f.updateParametricCurve_v2(this,r,o):(this.board.updateQuality===this.board.BOARD_QUALITY_HIGH?this.numberPoints=this.evalVisProp("numberpointshigh"):this.numberPoints=this.evalVisProp("numberpointslow"),this.allocatePoints(),f.updateParametricCurveNaive(this,r,o,this.numberPoints)),e=this.numberPoints,this.evalVisProp("useqdt")&&this.board.updateQuality===this.board.BOARD_QUALITY_HIGH)for(this.qdt=new l(this.board.getBoundingBox()),t=0;t<this.points.length;t++)this.qdt.insert(this.points[t]),t>0&&(this.points[t].prev=this.points[t-1]),t<e-1&&(this.points[t].next=this.points[t+1]);for("plot"!==this.evalVisProp("curvetype")&&this.evalVisProp("rdpsmoothing")&&(this.points=d.Z.RamerDouglasPeucker(this.points,.2),this.numberPoints=this.points.length),e=this.numberPoints,t=0;t<e;t++)this.updateTransform(this.points[t]);return this},updateTransformMatrix:function(){var t,e,i=this.transformations.length;for(this.transformMat=[[1,0,0],[0,1,0],[0,0,1]],e=0;e<i;e++)(t=this.transformations[e]).update(),this.transformMat=o.Z.matMatMult(t.matrix,this.transformMat);return this},updateTransform:function(t){var e;return this.transformations.length>0&&(e=o.Z.matVecMult(this.transformMat,t.usrCoords),t.setCoordinates(i.Z.COORDS_BY_USER,e,!1,!0)),t},addTransform:function(t){var e,i=s.Z.isArray(t)?t:[t],r=i.length;for(e=0;e<r;e++)this.transformations.push(i[e]);return this},interpolationFunctionFromArray:function(t){var e="data"+t,i=this;return function(t,r){var o,n,a,h,l=i[e],c=l.length,d=[];if(isNaN(t))return NaN;if(t<0)return s.Z.isFunction(l[0])?l[0]():l[0];if(3===i.bezierDegree){if(t>=(c-1)/3)return s.Z.isFunction(l[l.length-1])?l[l.length-1]():l[l.length-1];for(o=3*Math.floor(t),h=1-(a=t%1),n=0;n<4;n++)s.Z.isFunction(l[o+n])?d[n]=l[o+n]():d[n]=l[o+n];return h*h*(h*d[0]+3*a*d[1])+(3*h*d[2]+a*d[3])*a*a}if((o=t>c-2?c-2:parseInt(Math.floor(t),10))===t)return s.Z.isFunction(l[o])?l[o]():l[o];for(n=0;n<2;n++)s.Z.isFunction(l[o+n])?d[n]=l[o+n]():d[n]=l[o+n];return d[0]+(d[1]-d[0])*(t-o)}},generateTerm:function(t,e,i,r,n){var a,h,l;s.Z.isArray(e)?(this.dataX=e,this.numberPoints=this.dataX.length,this.X=this.interpolationFunctionFromArray.apply(this,["X"]),this.visProp.curvetype="plot",this.isDraggable=!0):(this.X=s.Z.createFunction(e,this.board,t),s.Z.isString(e)?this.visProp.curvetype="functiongraph":(s.Z.isFunction(e)||s.Z.isNumber(e))&&(this.visProp.curvetype="parameter"),this.isDraggable=!0),s.Z.isArray(i)?(this.dataY=i,this.Y=this.interpolationFunctionFromArray.apply(this,["Y"])):s.Z.exists(i)?this.Y=s.Z.createFunction(i,this.board,t):(l=o.Z.transpose(e),this.dataX=l[0],this.dataY=l[1],this.numberPoints=this.dataX.length,this.Y=this.interpolationFunctionFromArray.apply(this,["Y"])),s.Z.isFunction(e)&&s.Z.isArray(i)&&(a=s.Z.createFunction(i[0],this.board,""),h=s.Z.createFunction(i[1],this.board,""),this.X=function(t){return e(t)*Math.cos(t)+a()},this.X.deps=a.deps,this.Y=function(t){return e(t)*Math.sin(t)+h()},this.Y.deps=h.deps,this.visProp.curvetype="polar"),s.Z.exists(r)?this.minX=s.Z.createFunction(r,this.board,""):delete this.minX,s.Z.exists(n)?this.maxX=s.Z.createFunction(n,this.board,""):delete this.maxX,this.addParentsFromJCFunctions([this.X,this.Y,this.minX,this.maxX])},notifyParents:function(t){var e,i,s,r=!1;for(e in s={xterm:1,yterm:1})if(s.hasOwnProperty(e)&&this.hasOwnProperty(e)&&this[e].origin)for(i in r=!0,this[e].origin.deps)this[e].origin.deps.hasOwnProperty(i)&&this[e].origin.deps[i].addChild(this);r||M.Z.findDependencies(this,t,this.board)},getLabelPosition:function(t,e){var r,n,a,h,l,c,p,f,_,m,g,b,v,Z=1.5;return f=this.minX(),_=this.maxX(),m=d.Z.findDomain(this.X,[f,_],null,!1),m=d.Z.findDomain(this.Y,m,null,!1),f=Math.max(m[0],m[0]),_=Math.min(m[1],m[1]),a=s.Z.parsePosition(t),p=s.Z.parseNumber(a.pos,_-f,1),a.pos.indexOf("fr")<0&&a.pos.indexOf("%")<0&&(p=0),g=f+p,r=this.X(g),n=this.Y(g),isNaN(r+n)&&(g=f+(p/=_-f),r=this.X(f)+p*(this.X(_)-this.X(f)),n=this.Y(f)+p*(this.Y(_)-this.Y(f))),h=new u.Z(i.Z.COORDS_BY_USER,[r,n],this.board).scrCoords,g<f+(c=o.Z.eps)?(b=(this.X(g+c)-this.X(g))/c,v=(this.Y(g+c)-this.Y(g))/c):g>_-c?(b=(this.X(g)-this.X(g-c))/c,v=(this.Y(g)-this.Y(g-c))/c):(b=.5*(this.X(g+c)-this.X(g-c))/c,v=.5*(this.Y(g+c)-this.Y(g-c))/c),b=isNaN(b)?1:b,v=isNaN(v)?1:v,l=o.Z.hypot(b,v),"left"===a.side?v*=-1:b*=-1,s.Z.exists(this.label)&&(Z=.5*e/l),r=h[1]+v*this.label.size[0]*Z,n=h[2]-b*this.label.size[1]*Z,new u.Z(i.Z.COORDS_BY_SCREEN,[r,n],this.board)},getLabelAnchor:function(){var t,e,r,o,n=.05*this.board.canvasWidth,a=.05*this.board.canvasHeight,l=.95*this.board.canvasWidth,c=.95*this.board.canvasHeight;if(!s.Z.exists(this.label))return new u.Z(i.Z.COORDS_BY_SCREEN,[NaN,NaN],this.board);if(r=this.label.evalVisProp("position"),!s.Z.isString(r))return new u.Z(i.Z.COORDS_BY_SCREEN,[NaN,NaN],this.board);if(!(r.indexOf("right")<0&&r.indexOf("left")<0))return this.getLabelPosition(r,this.label.evalVisProp("distance"));switch(this.evalVisProp("label.position")){case"ulft":t=n,e=a;break;case"llft":t=n,e=c;break;case"rt":t=l,e=.5*c;break;case"lrt":t=l,e=c;break;case"urt":t=l,e=a;break;case"top":t=.5*l,e=a;break;case"bot":t=.5*l,e=c;break;default:t=n,e=.5*c}return o=new u.Z(i.Z.COORDS_BY_SCREEN,[t,e],this.board,!1),h.Z.projectCoordsToCurve(o.usrCoords[1],o.usrCoords[2],0,this,this.board)[0]},cloneToBackground:function(){var t,e=s.Z.getCloneObject(this);return e.points=this.points.slice(0),e.bezierDegree=this.bezierDegree,e.numberPoints=this.numberPoints,t=this.board.renderer.enhancedRendering,this.board.renderer.enhancedRendering=!0,this.board.renderer.drawCurve(e),this.board.renderer.enhancedRendering=t,this.traces[e.id]=e.rendNode,this},bounds:function(){var t,e,i,r=1/0,o=-1/0,n=1/0,a=-1/0,h=this.points.length;if(3===this.bezierDegree){for(t=0;t<h;t++)this.points[t].X=s.Z.bind((function(){return this.usrCoords[1]}),this.points[t]),this.points[t].Y=s.Z.bind((function(){return this.usrCoords[2]}),this.points[t]);return i=(e=d.Z.bezier(this.points))[3](),r=d.Z.fminbr((function(t){return e[0](t)}),[0,i]),o=d.Z.fminbr((function(t){return-e[0](t)}),[0,i]),n=d.Z.fminbr((function(t){return e[1](t)}),[0,i]),a=d.Z.fminbr((function(t){return-e[1](t)}),[0,i]),r=e[0](r),o=e[0](o),n=e[1](n),[r,a=e[1](a),o,n]}for(t=0;t<h;t++)r>this.points[t].usrCoords[1]&&(r=this.points[t].usrCoords[1]),o<this.points[t].usrCoords[1]&&(o=this.points[t].usrCoords[1]),n>this.points[t].usrCoords[2]&&(n=this.points[t].usrCoords[2]),a<this.points[t].usrCoords[2]&&(a=this.points[t].usrCoords[2]);return[r,a,o,n]},getParents:function(){var t=[this.xterm,this.yterm,this.minX(),this.maxX()];return 0!==this.parents.length&&(t=this.parents),t},moveTo:function(t){var e,s=[];return this.points.length>0&&!this.evalVisProp("fixed")?(e=this.points[0],s=3===t.length?[t[0]-e.usrCoords[0],t[1]-e.usrCoords[1],t[2]-e.usrCoords[2]]:[t[0]-e.usrCoords[1],t[1]-e.usrCoords[2]],this.setPosition(i.Z.COORDS_BY_USER,s),this.board.update(this)):this},getTransformationSource:function(){var t,e;return s.Z.exists(this._transformationSource)&&(e=this._transformationSource).elementClass===i.Z.OBJECT_CLASS_CURVE&&(t=!0),[t,e]}}),t.Z.createCurve=function(e,r,o){var n,a,h=s.Z.copyAttributes(o,e.options,"curve");return n=e.select(r[0],!0),s.Z.isTransformationOrArray(r[1])&&s.Z.isObject(n)&&(n.type===i.Z.OBJECT_TYPE_CURVE||n.type===i.Z.OBJECT_TYPE_ANGLE||n.type===i.Z.OBJECT_TYPE_ARC||n.type===i.Z.OBJECT_TYPE_CONIC||n.type===i.Z.OBJECT_TYPE_SECTOR)?(n.type===i.Z.OBJECT_TYPE_SECTOR?h=s.Z.copyAttributes(o,e.options,"sector"):n.type===i.Z.OBJECT_TYPE_ARC?h=s.Z.copyAttributes(o,e.options,"arc"):n.type===i.Z.OBJECT_TYPE_ANGLE?(s.Z.exists(o.withLabel)||(o.withLabel=!1),h=s.Z.copyAttributes(o,e.options,"angle")):h=s.Z.copyAttributes(o,e.options,"curve"),h=s.Z.copyAttributes(h,e.options,"curve"),(a=new t.Z.Curve(e,["x",[],[]],h)).updateDataArray=function(){var t,e=n.numberPoints;for(this.bezierDegree=n.bezierDegree,this.dataX=[],this.dataY=[],t=0;t<e;t++)this.dataX.push(n.points[t].usrCoords[1]),this.dataY.push(n.points[t].usrCoords[2]);return this},a.addTransform(r[1]),n.addChild(a),a.setParents([n]),a._transformationSource=n,a):(h=s.Z.copyAttributes(o,e.options,"curve"),new t.Z.Curve(e,["x"].concat(r),h))},t.Z.registerElement("curve",t.Z.createCurve),t.Z.createFunctiongraph=function(e,i,r){var o,n=["x","x"].concat(i);return o=s.Z.copyAttributes(r,e.options,"functiongraph"),(o=s.Z.copyAttributes(o,e.options,"curve")).curvetype="functiongraph",new t.Z.Curve(e,n,o)},t.Z.registerElement("functiongraph",t.Z.createFunctiongraph),t.Z.registerElement("plot",t.Z.createFunctiongraph),t.Z.createSpline=function(e,i,r){var o,n,a;return n=function(){var t,e=[],r=[];return[function(o,n){var a,h,l;if(!n){if(e=[],r=[],2===i.length&&s.Z.isArray(i[0])&&s.Z.isArray(i[1])&&i[0].length===i[1].length)for(a=0;a<i[0].length;a++)s.Z.isFunction(i[0][a])?e.push(i[0][a]()):e.push(i[0][a]),s.Z.isFunction(i[1][a])?r.push(i[1][a]()):r.push(i[1][a]);else for(a=0;a<i.length;a++)if(s.Z.isPoint(i[a]))e.push(i[a].X()),r.push(i[a].Y());else if(s.Z.isArray(i[a])&&2===i[a].length)for(h=0;h<i.length;h++)s.Z.isFunction(i[h][0])?e.push(i[h][0]()):e.push(i[h][0]),s.Z.isFunction(i[h][1])?r.push(i[h][1]()):r.push(i[h][1]);else s.Z.isFunction(i[a])&&2===i[a]().length&&(l=i[a](),e.push(l[0]),r.push(l[1]));t=d.Z.splineDef(e,r)}return d.Z.splineEval(o,e,r,t)},function(){return e[0]},function(){return e[e.length-1]}]},(r=s.Z.copyAttributes(r,e.options,"curve")).curvetype="functiongraph",a=n(),(o=new t.Z.Curve(e,["x","x",a[0],a[1],a[2]],r)).setParents(i),o.elType="spline",o},t.Z.registerElement("spline",t.Z.createSpline),t.Z.createCardinalSpline=function(e,i,r){var n,a,h,l,c,u,p,f,_,m,g="\nPossible parent types: [points:array, tau:number|function, type:string]";if(!s.Z.exists(i[0])||!s.Z.isArray(i[0]))throw new Error("JSXGraph: JXG.createCardinalSpline: argument 1 'points' has to be array of points or coordinate pairs"+g);if(!s.Z.exists(i[1])||!s.Z.isNumber(i[1])&&!s.Z.isFunction(i[1]))throw new Error("JSXGraph: JXG.createCardinalSpline: argument 2 'tau' has to be number between [0,1] or function'"+g);if(c=s.Z.exists(i[2])&&s.Z.isString(i[2])?i[2]:"uniform",r=s.Z.copyAttributes(r,e.options,"curve"),(r=s.Z.copyAttributes(r,e.options,"cardinalspline")).curvetype="parameter",u=i[0],p=[],!r.isarrayofcoordinates&&2===u.length&&s.Z.isArray(u[0])&&s.Z.isArray(u[1])&&u[0].length===u[1].length)for(f=0;f<u[0].length;f++)p[f]=[],s.Z.isFunction(u[0][f])?p[f].push(u[0][f]()):p[f].push(u[0][f]),s.Z.isFunction(u[1][f])?p[f].push(u[1][f]()):p[f].push(u[1][f]);else for(f=0;f<u.length;f++)s.Z.isString(u[f])?p.push(e.select(u[f])):s.Z.isPoint(u[f])?p.push(u[f]):s.Z.isArray(u[f])&&2===u[f].length?(p[f]=[],s.Z.isFunction(u[f][0])?p[f].push(u[f][0]()):p[f].push(u[f][0]),s.Z.isFunction(u[f][1])?p[f].push(u[f][1]()):p[f].push(u[f][1])):s.Z.isFunction(u[f])&&2===u[f]().length&&p.push(i[f]());if(!0===r.createpoints)h=s.Z.providePoints(e,p,r,"cardinalspline",["points"]);else for(h=[],a=function(t){return{X:function(){return p[t][0]},Y:function(){return p[t][1]},Dist:function(t){var e=this.X()-t.X(),i=this.Y()-t.Y();return o.Z.hypot(e,i)}}},f=0;f<p.length;f++)s.Z.isPoint(p[f])?h.push(p[f]):h.push(a(f));for(l=i[1],m=["x"].concat(d.Z.CardinalSpline(h,l,c)),n=new t.Z.Curve(e,m,r),_=h.length,n.setParents(h),f=0;f<_;f++)u=h[f],s.Z.isPoint(u)&&(s.Z.exists(u._is_new)?(n.addChild(u),delete u._is_new):u.addChild(n));return n.elType="cardinalspline",n},t.Z.registerElement("cardinalspline",t.Z.createCardinalSpline),t.Z.createMetapostSpline=function(e,i,r){var o,n,a,h,l,c,d,u,p="\nPossible parent types: [points:array, controls:object";if(!s.Z.exists(i[0])||!s.Z.isArray(i[0]))throw new Error("JSXGraph: JXG.createMetapostSpline: argument 1 'points' has to be array of points or coordinate pairs"+p);if(!s.Z.exists(i[1])||!s.Z.isObject(i[1]))throw new Error("JSXGraph: JXG.createMetapostSpline: argument 2 'controls' has to be a JavaScript object'"+p);if(r=s.Z.copyAttributes(r,e.options,"curve"),(r=s.Z.copyAttributes(r,e.options,"metapostspline")).curvetype="parameter",l=i[0],c=[],!r.isarrayofcoordinates&&2===l.length&&s.Z.isArray(l[0])&&s.Z.isArray(l[1])&&l[0].length===l[1].length)for(d=0;d<l[0].length;d++)c[d]=[],s.Z.isFunction(l[0][d])?c[d].push(l[0][d]()):c[d].push(l[0][d]),s.Z.isFunction(l[1][d])?c[d].push(l[1][d]()):c[d].push(l[1][d]);else for(d=0;d<l.length;d++)s.Z.isString(l[d])?c.push(e.select(l[d])):s.Z.isPoint(l[d])?c.push(l[d]):s.Z.isArray(l[d])&&2===l[d].length?(c[d]=[],s.Z.isFunction(l[d][0])?c[d].push(l[d][0]()):c[d].push(l[d][0]),s.Z.isFunction(l[d][1])?c[d].push(l[d][1]()):c[d].push(l[d][1])):s.Z.isFunction(l[d])&&2===l[d]().length&&c.push(i[d]());if(!0===r.createpoints)a=s.Z.providePoints(e,c,r,"metapostspline",["points"]);else for(a=[],n=function(t){return{X:function(){return c[t][0]},Y:function(){return c[t][1]}}},d=0;d<c.length;d++)s.Z.isPoint(c[d])?a.push(c[d]):a.push(n);for(h=i[1],(o=new t.Z.Curve(e,["t",[],[],0,l.length-1],r)).updateDataArray=function(){var t,e,i=a.length,s=[];for(e=0;e<i;e++)s.push([a[e].X(),a[e].Y()]);t=m.curve(s,h),this.dataX=t[0],this.dataY=t[1]},o.bezierDegree=3,u=a.length,o.setParents(a),d=0;d<u;d++)s.Z.isPoint(a[d])&&a[d].addChild(o);return o.elType="metapostspline",o},t.Z.registerElement("metapostspline",t.Z.createMetapostSpline),t.Z.createRiemannsum=function(t,e,i){var r,o,n,a,h,l;if((l=s.Z.copyAttributes(i,t.options,"riemannsum")).curvetype="plot",n=e[0],r=s.Z.createFunction(e[1],t,""),!s.Z.exists(r))throw new Error("JSXGraph: JXG.createRiemannsum: argument '2' n has to be number or function.\nPossible parent types: [function,n:number|function,type,start:number|function,end:number|function]");if("string"==typeof e[2]&&(e[2]="'"+e[2]+"'"),o=s.Z.createFunction(e[2],t,""),!s.Z.exists(o))throw new Error("JSXGraph: JXG.createRiemannsum: argument 3 'type' has to be string or function.\nPossible parent types: [function,n:number|function,type,start:number|function,end:number|function]");return a=[[0],[0]].concat(e.slice(3)),(h=t.create("curve",a,l)).sum=0,h.Value=function(){return this.sum},h.updateDataArray=function(){var t=d.Z.riemann(n,r(),o(),this.minX(),this.maxX());this.dataX=t[0],this.dataY=t[1],this.sum=t[2]},h.addParentsFromJCFunctions([r,o]),h},t.Z.registerElement("riemannsum",t.Z.createRiemannsum),t.Z.createTracecurve=function(t,e,r){var o,n,a,h;if(2!==e.length)throw new Error("JSXGraph: Can't create trace curve with given parent'\nPossible parent types: [glider, point]");if(n=t.select(e[0]),a=t.select(e[1]),n.type!==i.Z.OBJECT_TYPE_GLIDER||!s.Z.isPoint(a))throw new Error("JSXGraph: Can't create trace curve with parent types '"+typeof e[0]+"' and '"+typeof e[1]+"'.\nPossible parent types: [glider, point]");return(h=s.Z.copyAttributes(r,t.options,"tracecurve")).curvetype="plot",(o=t.create("curve",[[0],[0]],h)).updateDataArray=function(){var t,e,s,r,o,h,l,c,d,u=this.visProp.numberpoints,p=n.position,f=n.slideObject,_=f.minX();for(e=(f.maxX()-_)/u,this.dataX=[],this.dataY=[],f.elementClass!==i.Z.OBJECT_CLASS_CURVE&&u++,t=0;t<u;t++){for(r in s=_+t*e,h=f.X(s)/f.Z(s),l=f.Y(s)/f.Z(s),n.setPositionDirectly(i.Z.COORDS_BY_USER,[h,l]),c=!1,this.board.objects)if(this.board.objects.hasOwnProperty(r)&&((o=this.board.objects[r])===n&&(c=!0),c&&o.needsRegularUpdate&&(d=o.visProp.trace,o.visProp.trace=!1,o.needsUpdate=!0,o.update(!0),o.visProp.trace=d,o===a)))break;this.dataX[t]=a.X(),this.dataY[t]=a.Y()}for(r in n.position=p,c=!1,this.board.objects)if(this.board.objects.hasOwnProperty(r)&&((o=this.board.objects[r])===n&&(c=!0),c&&o.needsRegularUpdate&&(d=o.visProp.trace,o.visProp.trace=!1,o.needsUpdate=!0,o.update(!0),o.visProp.trace=d,o===a)))break},o},t.Z.registerElement("tracecurve",t.Z.createTracecurve),t.Z.createStepfunction=function(t,e,i){var r,o;if(2!==e.length)throw new Error("JSXGraph: Can't create step function with given parent'\nPossible parent types: [array, array|function]");return o=s.Z.copyAttributes(i,t.options,"stepfunction"),(r=t.create("curve",e,o)).updateDataArray=function(){var t,e=0,i=this.xterm.length;if(this.dataX=[],this.dataY=[],0!==i)for(this.dataX[e]=this.xterm[0],this.dataY[e]=this.yterm[0],++e,t=1;t<i;++t)this.dataX[e]=this.xterm[t],this.dataY[e]=this.dataY[e-1],++e,this.dataX[e]=this.xterm[t],this.dataY[e]=this.yterm[t],++e},r},t.Z.registerElement("stepfunction",t.Z.createStepfunction),t.Z.createDerivative=function(t,e,r){var o,n,a,h,l;if(1!==e.length&&e[0].class!==i.Z.OBJECT_CLASS_CURVE)throw new Error("JSXGraph: Can't create derivative curve with given parent'\nPossible parent types: [curve]");return l=s.Z.copyAttributes(r,t.options,"curve"),n=e[0],a=d.Z.D(n.X),h=d.Z.D(n.Y),(o=t.create("curve",[function(t){return n.X(t)},function(t){return h(t)/a(t)},n.minX(),n.maxX()],l)).setParents(n),o},t.Z.registerElement("derivative",t.Z.createDerivative),t.Z.createCurveIntersection=function(t,e,i){var s;if(2!==e.length)throw new Error("JSXGraph: Can't create curve intersection with given parent'\nPossible parent types: [array, array|function]");return(s=t.create("curve",[[],[]],i)).updateDataArray=function(){var t=g.intersection(e[0],e[1],this.board);this.dataX=t[0],this.dataY=t[1]},s},t.Z.createCurveUnion=function(t,e,i){var s;if(2!==e.length)throw new Error("JSXGraph: Can't create curve union with given parent'\nPossible parent types: [array, array|function]");return(s=t.create("curve",[[],[]],i)).updateDataArray=function(){var t=g.union(e[0],e[1],this.board);this.dataX=t[0],this.dataY=t[1]},s},t.Z.createCurveDifference=function(t,e,i){var s;if(2!==e.length)throw new Error("JSXGraph: Can't create curve difference with given parent'\nPossible parent types: [array, array|function]");return(s=t.create("curve",[[],[]],i)).updateDataArray=function(){var t=g.difference(e[0],e[1],this.board);this.dataX=t[0],this.dataY=t[1]},s},t.Z.registerElement("curvedifference",t.Z.createCurveDifference),t.Z.registerElement("curveintersection",t.Z.createCurveIntersection),t.Z.registerElement("curveunion",t.Z.createCurveUnion),t.Z.createBoxPlot=function(t,e,i){var r,o,n,a=s.Z.copyAttributes(i,t.options,"boxplot");if(3!==e.length)throw new Error("JSXGraph: Can't create box plot with given parent'\nPossible parent types: [array, number|function, number|function] containing quantiles, axis, width");if(e[0].length<5)throw new Error("JSXGraph: Can't create box plot with given parent[0]'\nparent[0] has to contain at least 5 quantiles.");for(r=t.create("curve",[[],[]],a),n=e[0].length,r.Q=[],o=0;o<n;o++)r.Q[o]=s.Z.createFunction(e[0][o],t);return r.x=s.Z.createFunction(e[1],t),r.w=s.Z.createFunction(e[2],t),r.updateDataArray=function(){var t,e,i,s,r,o,n,a,h;n=this.evalVisProp("smallwidth"),a=this.evalVisProp("dir"),i=(h=this.x())-.5*this.w(),s=h-.5*this.w()*n,r=h+.5*this.w(),t=[h,s,o=h+.5*this.w()*n,h,h,i,i,r,r,h,NaN,i,r,NaN,h,h,s,o,h],e=[this.Q[0](),this.Q[0](),this.Q[0](),this.Q[0](),this.Q[1](),this.Q[1](),this.Q[3](),this.Q[3](),this.Q[1](),this.Q[1](),NaN,this.Q[2](),this.Q[2](),NaN,this.Q[3](),this.Q[4](),this.Q[4](),this.Q[4](),this.Q[4]()],"vertical"===a?(this.dataX=t,this.dataY=e):(this.dataX=e,this.dataY=t)},r.addParentsFromJCFunctions([r.Q,r.x,r.w]),r},t.Z.registerElement("boxplot",t.Z.createBoxPlot),t.Z.createImplicitCurve=function(t,e,i){var r,o;if([1,3,5].indexOf(e.length)<0)throw new Error("JSXGraph: Can't create curve implicitCurve with given parent'\nPossible parent types: [f], [f, rangex, rangey], [f, dfx, dfy] or [f, dfx, dfy, rangex, rangey]\nwith functions f, dfx, dfy and arrays of length 2 rangex, rangey.");return o=s.Z.copyAttributes(i,t.options,"implicitcurve"),(r=t.create("curve",[[],[]],o)).f=s.Z.createFunction(e[0],t,"x, y"),5===e.length||s.Z.isString(e[1])||s.Z.isFunction(e[1])?r.dfx=s.Z.createFunction(e[1],t,"x, y"):r.dfx=null,5===e.length||s.Z.isString(e[2])||s.Z.isFunction(e[2])?r.dfy=s.Z.createFunction(e[2],t,"x, y"):r.dfy=null,r.domain=null,5===e.length?r.domain=[e[3],e[4]]:3===e.length&&(r.domain=[e[1],e[2]]),r.updateDataArray=function(){var t,e,i,r,o,n,a;null===this.domain?(a=this.evalVisProp("margin"),(t=this.board.getBoundingBox())[0]-=a,t[1]+=a,t[2]+=a,t[3]-=a):(e=s.Z.evaluate(this.domain[0]),i=s.Z.evaluate(this.domain[1]),t=[Math.min(e[0],e[1]),Math.max(i[0],i[1]),Math.max(e[0],e[1]),Math.min(i[0],i[1])]),o={resolution_out:Math.max(.01,this.evalVisProp("resolution_outer")),resolution_in:Math.max(.01,this.evalVisProp("resolution_inner")),max_steps:this.evalVisProp("max_steps"),alpha_0:this.evalVisProp("alpha_0"),tol_u0:this.evalVisProp("tol_u0"),tol_newton:this.evalVisProp("tol_newton"),tol_cusp:this.evalVisProp("tol_cusp"),tol_progress:this.evalVisProp("tol_progress"),qdt_box:this.evalVisProp("qdt_box"),kappa_0:this.evalVisProp("kappa_0"),delta_0:this.evalVisProp("delta_0"),h_initial:this.evalVisProp("h_initial"),h_critical:this.evalVisProp("h_critical"),h_max:this.evalVisProp("h_max"),loop_dist:this.evalVisProp("loop_dist"),loop_dir:this.evalVisProp("loop_dir"),loop_detection:this.evalVisProp("loop_detection"),unitX:this.board.unitX,unitY:this.board.unitY},this.dataX=[],this.dataY=[],r=new _(t,o,this.f,this.dfx,this.dfy),this.qdt=r.qdt,n=r.plot(),this.dataX=n[0],this.dataY=n[1]},r.elType="implicitcurve",r},t.Z.registerElement("implicitcurve",t.Z.createImplicitCurve);t.Z.Curve;t.Z.createArc=function(e,r,n){var a,l,c;if(!1===(c=s.Z.providePoints(e,r,n,"arc",["center","radiuspoint","anglepoint"]))||c.length<3)throw new Error("JSXGraph: Can't create Arc with parent types '"+typeof r[0]+"' and '"+typeof r[1]+"' and '"+typeof r[2]+"'.\nPossible parent types: [point,point,point], [arc, transformation]");return l=s.Z.copyAttributes(n,e.options,"arc"),(a=e.create("curve",[[0],[0],0,4],l)).elType="arc",a.setParents(c),a.type=i.Z.OBJECT_TYPE_ARC,a.center=c[0],a.radiuspoint=c[1],a.point2=a.radiuspoint,a.anglepoint=c[2],a.point3=a.anglepoint,s.Z.exists(a.center._is_new)?(a.addChild(a.center),delete a.center._is_new):a.center.addChild(a),s.Z.exists(a.radiuspoint._is_new)?(a.addChild(a.radiuspoint),delete a.radiuspoint._is_new):a.radiuspoint.addChild(a),s.Z.exists(a.anglepoint._is_new)?(a.addChild(a.anglepoint),delete a.anglepoint._is_new):a.anglepoint.addChild(a),a.useDirection=l.usedirection,a.updateDataArray=function(){var t,e,i,s,r,o=1,n=this.radiuspoint,a=this.center,l=this.anglepoint,d=this.evalVisProp("selection");e=h.Z.rad(n,a,l),("minor"===d&&e>Math.PI||"major"===d&&e<Math.PI)&&(o=-1),this.useDirection&&(i=c[1].coords.usrCoords,s=c[3].coords.usrCoords,r=c[2].coords.usrCoords,(i[1]-r[1])*(i[2]-s[2])-(i[2]-r[2])*(i[1]-s[1])<0?(this.radiuspoint=c[1],this.anglepoint=c[2]):(this.radiuspoint=c[2],this.anglepoint=c[1])),n=n.coords.usrCoords,a=a.coords.usrCoords,l=l.coords.usrCoords,t=h.Z.bezierArc(n,a,l,!1,o),this.dataX=t[0],this.dataY=t[1],this.bezierDegree=3,this.updateStdform(),this.updateQuadraticform()},a.Radius=function(){return this.radiuspoint.Dist(this.center)},a.getRadius=function(){return t.Z.deprecated("Arc.getRadius()","Arc.Radius()"),this.Radius()},a.Value=function(t,e){var i;return e=e||h.Z.rad(this.radiuspoint,this.center,this.anglepoint),""===(t=(t=t||"length").toLocaleLowerCase())||0===t.indexOf("len")?i=e*this.Radius():0===t.indexOf("rad")?i=e:0===t.indexOf("deg")?i=180*e/Math.PI:0===t.indexOf("sem")?i=e/Math.PI:0===t.indexOf("cir")&&(i=.5*e/Math.PI),i},a.L=function(){return this.Value("length")},a.hasPoint=function(t,e){var r,n,a,l,c,d,p,f=this.Radius();return this.evalVisProp("hasinnerpoints")?this.hasPointSector(t,e):(s.Z.isObject(this.evalVisProp("precision"))?(p=this.board._inputDevice,d=this.evalVisProp("precision."+p)):d=this.board.options.precision.hasPoint,d/=Math.min(Math.abs(this.board.unitX),Math.abs(this.board.unitY)),n=new u.Z(i.Z.COORDS_BY_SCREEN,[t,e],this.board),this.transformations.length>0&&(this.updateTransformMatrix(),l=o.Z.inverse(this.transformMat),c=o.Z.matVecMult(l,n.usrCoords),n=new u.Z(i.Z.COORDS_BY_USER,c,this.board)),r=this.center.coords.distance(i.Z.COORDS_BY_USER,n),(a=Math.abs(r-f)<d)&&(a=h.Z.coordsOnArc(this,n)),a)},a.hasPointSector=function(t,e){var s=new u.Z(i.Z.COORDS_BY_SCREEN,[t,e],this.board),r=this.Radius(),o=this.center.coords.distance(i.Z.COORDS_BY_USER,s)<r;return o&&(o=h.Z.coordsOnArc(this,s)),o},a.getTextAnchor=function(){return this.center.coords},a.getLabelAnchor=function(){var t,e,r,n,a,l=this.label.evalVisProp("position"),c=h.Z.rad(this.radiuspoint,this.center,this.anglepoint),d=10/this.board.unitX,p=10/this.board.unitY,f=this.point2.coords.usrCoords,_=this.center.coords.usrCoords,m=f[1]-_[1],g=f[2]-_[2],b=this.evalVisProp("selection"),v=this.label?this.label.visProp:this.visProp.label;return!s.Z.isString(l)||l.indexOf("right")<0&&l.indexOf("left")<0?(("minor"===b&&c>Math.PI||"major"===b&&c<Math.PI)&&(c=-(2*Math.PI-c)),r=(t=new u.Z(i.Z.COORDS_BY_USER,[_[1]+Math.cos(.5*c)*m-Math.sin(.5*c)*g,_[2]+Math.sin(.5*c)*m+Math.cos(.5*c)*g],this.board)).usrCoords[1]-_[1],n=t.usrCoords[2]-_[2],r=r*((a=o.Z.hypot(r,n))+d)/a,n=n*(a+p)/a,e=[_[1]+r,_[2]+n],v.position=h.Z.calcLabelQuadrant(h.Z.rad([1,0],[0,0],e)),new u.Z(i.Z.COORDS_BY_USER,e,this.board)):this.getLabelPosition(l,this.label.evalVisProp("distance"))},a.updateQuadraticform=V.prototype.updateQuadraticform,a.updateStdform=V.prototype.updateStdform,a.methodMap=t.Z.deepCopy(a.methodMap,{getRadius:"getRadius",radius:"Radius",Radius:"Radius",center:"center",radiuspoint:"radiuspoint",anglepoint:"anglepoint",Value:"Value",L:"L"}),a.prepareUpdate().update(),a},t.Z.registerElement("arc",t.Z.createArc),t.Z.createSemicircle=function(t,e,i){var r,o,n,a;if(!1===(a=s.Z.providePoints(t,e,i,"point"))||2!==a.length)throw new Error("JSXGraph: Can't create Semicircle with parent types '"+typeof e[0]+"' and '"+typeof e[1]+"'.\nPossible parent types: [point,point]");return n=s.Z.copyAttributes(i,t.options,"semicircle","center"),(o=t.create("midpoint",a,n)).dump=!1,n=s.Z.copyAttributes(i,t.options,"semicircle"),(r=t.create("arc",[o,a[1],a[0]],n)).elType="semicircle",r.setParents([a[0].id,a[1].id]),r.subs={midpoint:o},r.inherits.push(o),r.midpoint=r.center=o,r},t.Z.registerElement("semicircle",t.Z.createSemicircle),t.Z.createCircumcircleArc=function(t,e,i){var r,o,n,a;if(!1===(a=s.Z.providePoints(t,e,i,"point"))||3!==a.length)throw new Error("JSXGraph: create Circumcircle Arc with parent types '"+typeof e[0]+"' and '"+typeof e[1]+"' and '"+typeof e[2]+"'.\nPossible parent types: [point,point,point]");return n=s.Z.copyAttributes(i,t.options,"circumcirclearc","center"),(o=t.create("circumcenter",a,n)).dump=!1,(n=s.Z.copyAttributes(i,t.options,"circumcirclearc")).usedirection=!0,(r=t.create("arc",[o,a[0],a[2],a[1]],n)).elType="circumcirclearc",r.setParents([a[0].id,a[1].id,a[2].id]),r.subs={center:o},r.inherits.push(o),r.center=o,r},t.Z.registerElement("circumcirclearc",t.Z.createCircumcircleArc),t.Z.createMinorArc=function(e,i,s){return s.selection="minor",t.Z.createArc(e,i,s)},t.Z.registerElement("minorarc",t.Z.createMinorArc),t.Z.createMajorArc=function(e,i,s){return s.selection="major",t.Z.createArc(e,i,s)},t.Z.registerElement("majorarc",t.Z.createMajorArc),t.Z.createSector=function(e,r,n){var a,l,c,d,f,_,m=1e-14,g="invalid";if(r[0].elementClass===i.Z.OBJECT_CLASS_LINE&&r[1].elementClass===i.Z.OBJECT_CLASS_LINE&&(s.Z.isArray(r[2])||s.Z.isNumber(r[2]))&&(s.Z.isArray(r[3])||s.Z.isNumber(r[3]))&&(s.Z.isNumber(r[4])||s.Z.isFunction(r[4])||s.Z.isString(r[4])))g="2lines";else{if(!1===(_=s.Z.providePoints(e,r,n,"sector",["center","radiusPoint","anglePoint"])))throw new Error("JSXGraph: Can't create Sector with parent types '"+typeof r[0]+"' and '"+typeof r[1]+"' and '"+typeof r[2]+"'.");g="3points"}if(l=s.Z.copyAttributes(n,e.options,"sector"),(a=e.create("curve",[[0],[0],0,6],l)).type=i.Z.OBJECT_TYPE_SECTOR,a.elType="sector",a.autoRadius=function(){var t=20/a.board.unitX,e=1/0,i=50/a.board.unitX;return s.Z.isPoint(a.center)&&(e=.3333*a.center.Dist(a.point2)),Math.max(t,Math.min(e,i))},"2lines"===g)a.Radius=function(){var t=s.Z.evaluate(r[4]);return"auto"===t?this.autoRadius():t},a.line1=e.select(r[0]),a.line2=e.select(r[1]),a.line1.addChild(a),a.line2.addChild(a),a.setParents(r),a.point1={visProp:{}},a.point2={visProp:{}},a.point3={visProp:{}},d=h.Z.meetLineLine(a.line1.stdform,a.line2.stdform,0,e),h.Z.distance(d.usrCoords,[0,0,0],3)<m&&(a.line1.point1.Dist(a.line2.point1)<m||a.line1.point1.Dist(a.line2.point2)<m?d=a.line1.point1.coords:a.line1.point2.Dist(a.line2.point1)<m||a.line1.point2.Dist(a.line2.point1)<m?d=a.line1.point2.coords:console.log("JSXGraph warning: Can't create Sector from parallel lines with no common defining point.")),s.Z.isArray(r[2])?(2===r[2].length&&(r[2]=[1].concat(r[2])),f=h.Z.projectPointToLine({coords:{usrCoords:r[2]}},a.line1,e),f=p.Z.subtract(f.usrCoords,d.usrCoords),a.direction1=o.Z.innerProduct(f,[0,a.line1.stdform[2],-a.line1.stdform[1]],3)>=0?1:-1):a.direction1=r[2]>=0?1:-1,s.Z.isArray(r[3])?(2===r[3].length&&(r[3]=[1].concat(r[3])),f=h.Z.projectPointToLine({coords:{usrCoords:r[3]}},a.line2,e),f=p.Z.subtract(f.usrCoords,d.usrCoords),a.direction2=o.Z.innerProduct(f,[0,a.line2.stdform[2],-a.line2.stdform[1]],3)>=0?1:-1):a.direction2=r[3]>=0?1:-1,a.methodMap=t.Z.deepCopy(a.methodMap,{arc:"arc",center:"center",line1:"line1",line2:"line2"}),a.updateDataArray=function(){var t,e,s,r,n,l,c=1e-14,d=[0,0,0];if(e=this.line1,s=this.line2,d=o.Z.crossProduct(e.stdform,s.stdform),h.Z.distance(d,[0,0,0],3)<c&&(e.point1.Dist(s.point1)<c||e.point1.Dist(s.point2)<c?d=e.point1.coords.usrCoords:(e.point2.Dist(s.point1)<c||e.point2.Dist(s.point1)<c)&&(d=e.point2.coords.usrCoords)),Math.abs(d[0])>c&&(d[1]/=d[0],d[2]/=d[0],d[0]/=d[0]),t=this.direction1*this.Radius(),r=p.Z.add(d,[0,t*e.stdform[2],-t*e.stdform[1]]),t=this.direction2*this.Radius(),n=p.Z.add(d,[0,t*s.stdform[2],-t*s.stdform[1]]),this.point2.coords=new u.Z(i.Z.COORDS_BY_USER,r,a.board),this.point1.coords=new u.Z(i.Z.COORDS_BY_USER,d,a.board),this.point3.coords=new u.Z(i.Z.COORDS_BY_USER,n,a.board),Math.abs(r[0])<o.Z.eps||Math.abs(d[0])<o.Z.eps||Math.abs(n[0])<o.Z.eps)return this.dataX=[NaN],void(this.dataY=[NaN]);l=h.Z.bezierArc(r,d,n,!0,1),this.dataX=l[0],this.dataY=l[1],this.bezierDegree=3};else if("3points"===g){for(a.point1=_[0],a.point2=_[1],a.point3=_[2],c=0;c<3;c++)s.Z.exists(_[c]._is_new)?(a.addChild(_[c]),delete _[c]._is_new):_[c].addChild(a);a.useDirection=n.usedirection,a.setParents(_),s.Z.exists(_[3])&&(a.point4=_[3],a.point4.addChild(a)),a.methodMap=t.Z.deepCopy(a.methodMap,{arc:"arc",center:"center",radiuspoint:"radiuspoint",anglepoint:"anglepoint"}),a.updateDataArray=function(){var t,e,i,r,o,n=this.point2,a=this.point1,l=this.point3,c=1,d=this.evalVisProp("selection");if(!n.isReal||!a.isReal||!l.isReal)return this.dataX=[NaN],void(this.dataY=[NaN]);o=h.Z.rad(n,a,l),("minor"===d&&o>Math.PI||"major"===d&&o<Math.PI)&&(c=-1),this.useDirection&&s.Z.exists(this.point4)&&(e=this.point2.coords.usrCoords,i=this.point4.coords.usrCoords,r=this.point3.coords.usrCoords,(e[1]-r[1])*(e[2]-i[2])-(e[2]-r[2])*(e[1]-i[1])>=0&&(l=this.point2,n=this.point3)),n=n.coords.usrCoords,a=a.coords.usrCoords,l=l.coords.usrCoords,t=h.Z.bezierArc(n,a,l,!0,c),this.dataX=t[0],this.dataY=t[1],this.bezierDegree=3},a.Radius=function(){return this.point2.Dist(this.point1)}}return a.center=a.point1,a.radiuspoint=a.point2,a.anglepoint=a.point3,l=s.Z.copyAttributes(n,e.options,"arc"),(l=s.Z.copyAttributes(l,e.options,"sector","arc")).withlabel=!1,l.selection=a.visProp.selection,l.name+="_arc","2lines"===g?(a.updateDataArray(),a.arc=e.create("arc",[function(){return a.point1.coords.usrCoords},function(){var t=a.point2.coords.distance(i.Z.COORDS_BY_USER,a.point1.coords);return 0===t?[a.point1.coords.usrCoords[1],a.point1.coords.usrCoords[2]]:[a.point1.coords.usrCoords[1]+a.Radius()*(a.point2.coords.usrCoords[1]-a.point1.coords.usrCoords[1])/t,a.point1.coords.usrCoords[2]+a.Radius()*(a.point2.coords.usrCoords[2]-a.point1.coords.usrCoords[2])/t]},function(){return a.point3.coords.usrCoords}],l)):a.arc=e.create("arc",[a.point1,function(){var t=a.point2.Dist(a.point1);return 0===t?[a.point1.X(),a.point1.Y()]:[a.point1.X()+a.Radius()*(a.point2.X()-a.point1.X())/t,a.point1.Y()+a.Radius()*(a.point2.Y()-a.point1.Y())/t]},a.point3],l),a.addChild(a.arc),a.hasPointCurve=function(t,e){var r,o,n,a,l,c,d=new u.Z(i.Z.COORDS_BY_SCREEN,[t,e],this.board),p=this.Radius(),f=this.center.coords.distance(i.Z.COORDS_BY_USER,d),_=this.evalVisProp("selection");return s.Z.isObject(this.evalVisProp("precision"))?(l=this.board._inputDevice,a=this.evalVisProp("precision."+l)):a=this.board.options.precision.hasPoint,a/=Math.min(Math.abs(this.board.unitX),Math.abs(this.board.unitY)),(c=Math.abs(f-p)<a)&&(r=h.Z.rad(this.point2,this.center,d.usrCoords.slice(1)),o=0,n=h.Z.rad(this.point2,this.center,this.point3),("minor"===_&&n>Math.PI||"major"===_&&n<Math.PI)&&(o=n,n=2*Math.PI),(r<o||r>n)&&(c=!1)),c},a.hasPointSector=function(t,e){var s,r,o,n=new u.Z(i.Z.COORDS_BY_SCREEN,[t,e],this.board),a=this.Radius(),l=this.point1.coords.distance(i.Z.COORDS_BY_USER,n)<a,c=this.evalVisProp("selection");return l&&(s=h.Z.rad(this.radiuspoint,this.center,n.usrCoords.slice(1)),r=0,o=h.Z.rad(this.radiuspoint,this.center,this.anglepoint),("minor"===c&&o>Math.PI||"major"===c&&o<Math.PI)&&(r=o,o=2*Math.PI),(s<r||s>o)&&(l=!1)),l},a.hasPoint=function(t,e){return this.evalVisProp("highlightonsector")||this.evalVisProp("hasinnerpoints")?this.hasPointSector(t,e):this.hasPointCurve(t,e)},a.getTextAnchor=function(){return this.point1.coords},a.getLabelAnchor=function(){var t,e,r,n,a,l=this.label.evalVisProp("position"),c=h.Z.rad(this.point2,this.point1,this.point3),d=13/this.board.unitX,p=13/this.board.unitY,f=this.point2.coords.usrCoords,_=this.point1.coords.usrCoords,m=f[1]-_[1],g=f[2]-_[2],b=this.evalVisProp("selection"),v=this.label?this.label.visProp:this.visProp.label;return!s.Z.isString(l)||l.indexOf("right")<0&&l.indexOf("left")<0?(("minor"===b&&c>Math.PI||"major"===b&&c<Math.PI)&&(c=-(2*Math.PI-c)),r=(t=new u.Z(i.Z.COORDS_BY_USER,[_[1]+Math.cos(.5*c)*m-Math.sin(.5*c)*g,_[2]+Math.sin(.5*c)*m+Math.cos(.5*c)*g],this.board)).usrCoords[1]-_[1],n=t.usrCoords[2]-_[2],r=r*((a=o.Z.hypot(r,n))+d)/a,n=n*(a+p)/a,e=[_[1]+r,_[2]+n],v.position=h.Z.calcLabelQuadrant(h.Z.rad([1,0],[0,0],e)),new u.Z(i.Z.COORDS_BY_USER,e,this.board)):this.getLabelPosition(l,this.label.evalVisProp("distance"))},a.setRadius=function(t){var e=s.Z.evaluate(t);"undefined"!==("auto"===t||"auto"===e?"auto":s.Z.isNumber(t)?"number":s.Z.isFunction(t)&&!s.Z.isString(e)?"function":"undefined")&&(this.visProp.radius=t),a.Radius=function(){var e=s.Z.evaluate(t);return"auto"===e?this.autoRadius():e}},a.getRadius=function(){return t.Z.deprecated("Sector.getRadius()","Sector.Radius()"),this.Radius()},a.Value=function(t){return this.arc.Value(t)},a.L=function(){return this.arc.L()},a.Area=function(){var t=this.Radius();return.5*t*t*this.Value("radians")},a.Perimeter=function(){return this.L()+2*this.Radius()},"3points"===g&&(a.setPositionDirectly=function(t,e,i){var s,r=new u.Z(t,e,this.board),o=new u.Z(t,i,this.board);return a.point1.draggable()&&a.point2.draggable()&&a.point3.draggable()?(s=p.Z.subtract(r.usrCoords,o.usrCoords),this.board.create("transform",s.slice(1),{type:"translate"}).applyOnce([a.point1,a.point2,a.point3]),this):this}),a.methodMap=t.Z.deepCopy(a.methodMap,{radius:"Radius",Radius:"Radius",getRadius:"Radius",setRadius:"setRadius",Value:"Value",L:"L",Area:"Area",Perimeter:"Perimeter"}),a},t.Z.registerElement("sector",t.Z.createSector),t.Z.createCircumcircleSector=function(t,e,i){var r,o,n,a;if(!1===(a=s.Z.providePoints(t,e,i,"point")))throw new Error("JSXGraph: Can't create circumcircle sector with parent types '"+typeof e[0]+"' and '"+typeof e[1]+"' and '"+typeof e[2]+"'.");return(o=t.create("circumcenter",a.slice(0,3),n)).dump=!1,n=s.Z.copyAttributes(i,t.options,"circumcirclesector"),(r=t.create("sector",[o,a[0],a[2],a[1]],n)).elType="circumcirclesector",r.setParents(a),r.center=o,r.subs={center:o},r},t.Z.registerElement("circumcirclesector",t.Z.createCircumcircleSector),t.Z.createMinorSector=function(e,i,s){return s.selection="minor",t.Z.createSector(e,i,s)},t.Z.registerElement("minorsector",t.Z.createMinorSector),t.Z.createMajorSector=function(e,i,s){return s.selection="major",t.Z.createSector(e,i,s)},t.Z.registerElement("majorsector",t.Z.createMajorSector),t.Z.createAngle=function(t,e,r){var n,a,l,c,d,p,f="invalid";if(e[0].elementClass===i.Z.OBJECT_CLASS_LINE&&e[1].elementClass===i.Z.OBJECT_CLASS_LINE&&(s.Z.isArray(e[2])||s.Z.isNumber(e[2]))&&(s.Z.isArray(e[3])||s.Z.isNumber(e[3])))f="2lines";else{if(l={name:""},!1===(p=s.Z.providePoints(t,e,l,"point")))throw new Error("JSXGraph: Can't create angle with parent types '"+typeof e[0]+"' and '"+typeof e[1]+"' and '"+typeof e[2]+"'.");f="3points"}if(l=s.Z.copyAttributes(r,t.options,"angle"),s.Z.exists(l.name)||(l.name=t.generateName({type:i.Z.OBJECT_TYPE_ANGLE})),a=s.Z.exists(l.radius)?l.radius:0,t.suspendUpdate(),"2lines"===f?(e.push(a),(n=t.create("sector",e,l)).updateDataArraySector=n.updateDataArray,n.setAngle=function(t){},n.free=function(t){}):((n=t.create("sector",[p[1],p[0],p[2]],l)).arc.visProp.priv=!0,n.point=n.point2=n.radiuspoint=p[0],n.pointsquare=n.point3=n.anglepoint=p[2],n.Radius=function(){var t=s.Z.evaluate(a);return"auto"===t?n.autoRadius():t},n.updateDataArraySector=function(){var t,e,i=this.point2,s=this.point1,r=this.point3,o=this.Radius(),n=s.Dist(i),a=1,l=this.evalVisProp("selection");e=h.Z.rad(i,s,r),("minor"===l&&e>Math.PI||"major"===l&&e<Math.PI)&&(a=-1),i=i.coords.usrCoords,s=s.coords.usrCoords,r=r.coords.usrCoords,i=[1,s[1]+(i[1]-s[1])*o/n,s[2]+(i[2]-s[2])*o/n],r=[1,s[1]+(r[1]-s[1])*o/n,s[2]+(r[2]-s[2])*o/n],t=h.Z.bezierArc(i,s,r,!0,a),this.dataX=t[0],this.dataY=t[1],this.bezierDegree=3},n.setAngle=function(t){var e,r,n,a=this.anglepoint,h=this.radiuspoint;return a.draggable()&&(e=this.board.create("transform",[t,this.center],{type:"rotate"}),a.addTransform(h,e),e.update(),a.moveTo(o.Z.matVecMult(e.matrix,h.coords.usrCoords)),n=s.Z.isFunction(t)?function(){return 2*Math.PI-t()}:function(){return 2*Math.PI-t},r=this.board.create("transform",[n,this.center],{type:"rotate"}),a.coords.on("update",(function(){r.update(),h.setPositionDirectly(i.Z.COORDS_BY_USER,o.Z.matVecMult(r.matrix,a.coords.usrCoords))})),a.setParents(h),this.hasFixedAngle=!0),this},n.free=function(){var t=this.anglepoint;return t.transformations.length>0&&(t.transformations.pop(),t.isDraggable=!0,t.parents=[],t.coords.off("update")),this.hasFixedAngle=!1,this},n.setParents(p)),s.Z.exists(n.visProp.text)&&n.label.setText(n.evalVisProp("text")),n.elType="angle",n.type=i.Z.OBJECT_TYPE_ANGLE,n.subs={},n.updateDataArraySquare=function(){var t,e,i,s,r,n,a,l,c=this.Radius();"2lines"===f&&this.updateDataArraySector(),t=this.point2,e=this.point1,i=this.point3,t=t.coords.usrCoords,e=e.coords.usrCoords,i=i.coords.usrCoords,s=h.Z.distance(t,e,3),r=h.Z.distance(i,e,3),t=[1,e[1]+(t[1]-e[1])*c/s,e[2]+(t[2]-e[2])*c/s],i=[1,e[1]+(i[1]-e[1])*c/r,e[2]+(i[2]-e[2])*c/r],n=o.Z.crossProduct(i,e),a=[-t[1]*n[1]-t[2]*n[2],t[0]*n[1],t[0]*n[2]],n=o.Z.crossProduct(t,e),l=[-i[1]*n[1]-i[2]*n[2],i[0]*n[1],i[0]*n[2]],(n=o.Z.crossProduct(a,l))[1]/=n[0],n[2]/=n[0],this.dataX=[e[1],t[1],n[1],i[1],e[1]],this.dataY=[e[2],t[2],n[2],i[2],e[2]],this.bezierDegree=1},n.updateDataArrayNone=function(){this.dataX=[NaN],this.dataY=[NaN],this.bezierDegree=1},n.updateDataArray=function(){var t=this.evalVisProp("type"),e=h.Z.trueAngle(this.point2,this.point1,this.point3),i=this.evalVisProp("selection");("minor"===i&&e>180||"major"===i&&e<180)&&(e=360-e),Math.abs(e-90)<this.evalVisProp("orthosensitivity")+o.Z.eps&&(t=this.evalVisProp("orthotype")),"none"===t?(this.updateDataArrayNone(),this.maxX=function(){return 0}):"square"===t?(this.updateDataArraySquare(),this.maxX=function(){return 4}):"sector"===t?(this.updateDataArraySector(),this.maxX=function(){return 6}):"sectordot"===t&&(this.updateDataArraySector(),this.maxX=function(){return 6},this.dot.visProp.visible||this.dot.setAttribute({visible:!0})),(!this.visProp.visible||"sectordot"!==t&&this.dot.visProp.visible)&&this.dot.setAttribute({visible:!1})},c=s.Z.copyAttributes(r,t.options,"angle","dot"),n.dot=t.create("point",[function(){var t,e,i,r,a,l,c,d,u;return s.Z.exists(n.dot)&&!n.dot.visProp.visible?[0,0]:(t=n.point2.coords.usrCoords,e=n.point1.coords.usrCoords,i=n.Radius(),r=h.Z.distance(t,e,3),a=h.Z.rad(n.point2,n.point1,n.point3),("minor"===(u=n.evalVisProp("selection"))&&a>Math.PI||"major"===u&&a<Math.PI)&&(a=-(2*Math.PI-a)),a*=.5,l=Math.cos(a),c=Math.sin(a),t=[1,e[1]+(t[1]-e[1])*i/r,e[2]+(t[2]-e[2])*i/r],d=[[1,0,0],[e[1]-.5*e[1]*l+.5*e[2]*c,.5*l,.5*-c],[e[2]-.5*e[1]*c-.5*e[2]*l,.5*c,.5*l]],o.Z.matVecMult(d,t))}],c),n.dot.dump=!1,n.subs.dot=n.dot,"2lines"===f)for(d=0;d<2;d++)t.select(e[d]).addChild(n.dot);else for(d=0;d<3;d++)t.select(p[d]).addChild(n.dot);return t.unsuspendUpdate(),n.Value=function(t){return""===(t=t||"radians")&&(t="radians"),n.arc.Value(t)},n.getLabelAnchor=function(){var t,e,r,a,l,c,d,p,f,_=12,m=n.evalVisProp("selection"),g=this.label.evalVisProp("position"),b=this.label?this.label.visProp:this.visProp.label;return!s.Z.isString(g)||g.indexOf("right")<0&&g.indexOf("left")<0?(s.Z.exists(this.label)&&s.Z.exists(this.label.visProp.fontsize)&&(_=this.label.evalVisProp("fontsize")),_/=this.board.unitX,e=n.point2.coords.usrCoords,r=n.point1.coords.usrCoords,a=n.Radius(),l=h.Z.distance(e,r,3),c=h.Z.rad(n.point2,n.point1,n.point3),("minor"===m&&c>Math.PI||"major"===m&&c<Math.PI)&&(c=-(2*Math.PI-c)),c*=.5,d=Math.cos(c),p=Math.sin(c),e=[1,r[1]+(e[1]-r[1])*a/l,r[2]+(e[2]-r[2])*a/l],f=[[1,0,0],[r[1]-.5*r[1]*d+.5*r[2]*p,.5*d,.5*-p],[r[2]-.5*r[1]*p-.5*r[2]*d,.5*p,.5*d]],(t=o.Z.matVecMult(f,e))[1]/=t[0],t[2]/=t[0],t[0]/=t[0],l=h.Z.distance(t,r,3),t=[t[0],r[1]+(t[1]-r[1])*(a+_)/l,r[2]+(t[2]-r[2])*(a+_)/l],b.position=h.Z.calcLabelQuadrant(h.Z.rad([1,0],[0,0],t)),new u.Z(i.Z.COORDS_BY_USER,t,this.board)):this.getLabelPosition(g,this.label.evalVisProp("distance"))},n.methodMap=s.Z.deepCopy(n.methodMap,{setAngle:"setAngle",Value:"Value",free:"free"}),n},t.Z.registerElement("angle",t.Z.createAngle),t.Z.createNonreflexAngle=function(e,i,r){var o;return r.selection="minor",r=s.Z.copyAttributes(r,e.options,"nonreflexangle"),(o=t.Z.createAngle(e,i,r)).Value=function(t){var e=h.Z.rad(this.point2,this.point1,this.point3);return""===(t=t||"radians")&&(t="radians"),e=e<Math.PI?e:2*Math.PI-e,this.arc.Value(t,e)},o},t.Z.registerElement("nonreflexangle",t.Z.createNonreflexAngle),t.Z.createReflexAngle=function(e,i,r){var o;return r.selection="major",r=s.Z.copyAttributes(r,e.options,"reflexangle"),(o=t.Z.createAngle(e,i,r)).Value=function(t){var e=h.Z.rad(this.point2,this.point1,this.point3);return""===(t=t||"radians")&&(t="radians"),e=e>=Math.PI?e:2*Math.PI-e,this.arc.Value(t,e)},o},t.Z.registerElement("reflexangle",t.Z.createReflexAngle),t.Z.createOrthogonalProjection=function(t,e,r){var o,n,a,l;if(e[0]=t.select(e[0]),e[1]=t.select(e[1]),s.Z.isPointType(t,e[0])&&e[1].elementClass===i.Z.OBJECT_CLASS_LINE)n=s.Z.providePoints(t,[e[0]],r,"point")[0],o=e[1];else{if(!s.Z.isPointType(t,e[1])||e[0].elementClass!==i.Z.OBJECT_CLASS_LINE)throw new Error("JSXGraph: Can't create perpendicular point with parent types '"+typeof e[0]+"' and '"+typeof e[1]+"'.\nPossible parent types: [point,line]");n=s.Z.providePoints(t,[e[1]],r,"point")[0],o=e[0]}return l=s.Z.copyAttributes(r,t.options,"orthogonalprojection"),a=t.create("point",[function(){return h.Z.projectPointToLine(n,o,t)}],l),s.Z.exists(n._is_new)?(a.addChild(n),delete n._is_new):n.addChild(a),o.addChild(a),a.elType="orthogonalprojection",a.setParents([n.id,a.id]),a.update(),a.generatePolynomial=function(){var t=o.point1.symbolic.x,e=o.point1.symbolic.y,i=o.point2.symbolic.x,s=o.point2.symbolic.y,r=n.symbolic.x,h=n.symbolic.y,l=a.symbolic.x,c=a.symbolic.y;return["("+e+")*("+l+")-("+e+")*("+i+")+("+c+")*("+i+")-("+t+")*("+c+")+("+t+")*("+s+")-("+l+")*("+s+")","("+h+")*("+e+")-("+h+")*("+s+")-("+c+")*("+e+")+("+c+")*("+s+")+("+r+")*("+t+")-("+r+")*("+i+")-("+l+")*("+t+")+("+l+")*("+i+")"]},a},t.Z.createPerpendicular=function(e,r,o){var n,a,h,l;if(r[0]=e.select(r[0]),r[1]=e.select(r[1]),s.Z.isPointType(e,r[0])&&r[1].elementClass===i.Z.OBJECT_CLASS_LINE)a=r[1],n=s.Z.providePoints(e,[r[0]],o,"point")[0];else{if(!s.Z.isPointType(e,r[1])||r[0].elementClass!==i.Z.OBJECT_CLASS_LINE)throw new Error("JSXGraph: Can't create perpendicular with parent types '"+typeof r[0]+"' and '"+typeof r[1]+"'.\nPossible parent types: [line,point]");a=r[0],n=s.Z.providePoints(e,[r[1]],o,"point")[0]}return l=s.Z.copyAttributes(o,e.options,"perpendicular"),(h=t.Z.createLine(e,[function(){return a.stdform[2]*n.X()-a.stdform[1]*n.Y()},function(){return-a.stdform[2]*n.Z()},function(){return a.stdform[1]*n.Z()}],l)).elType="perpendicular",h.setParents([a.id,n.id]),s.Z.exists(n._is_new)?(h.addChild(n),delete n._is_new):n.addChild(h),a.addChild(h),h},t.Z.createPerpendicularPoint=function(t,e,r){var o,n,a;if(e[0]=t.select(e[0]),e[1]=t.select(e[1]),s.Z.isPointType(t,e[0])&&e[1].elementClass===i.Z.OBJECT_CLASS_LINE)n=s.Z.providePoints(t,[e[0]],r,"point")[0],o=e[1];else{if(!s.Z.isPointType(t,e[1])||e[0].elementClass!==i.Z.OBJECT_CLASS_LINE)throw new Error("JSXGraph: Can't create perpendicular point with parent types '"+typeof e[0]+"' and '"+typeof e[1]+"'.\nPossible parent types: [point,line]");n=s.Z.providePoints(t,[e[1]],r,"point")[0],o=e[0]}return a=t.create("point",[function(){return h.Z.perpendicular(o,n,t)[0]}],r),s.Z.exists(n._is_new)?(a.addChild(n),delete n._is_new):n.addChild(a),o.addChild(a),a.elType="perpendicularpoint",a.setParents([n.id,o.id]),a.update(),a.generatePolynomial=function(){var t=o.point1.symbolic.x,e=o.point1.symbolic.y,i=o.point2.symbolic.x,s=o.point2.symbolic.y,r=n.symbolic.x,h=n.symbolic.y,l=a.symbolic.x,c=a.symbolic.y;return["("+e+")*("+l+")-("+e+")*("+i+")+("+c+")*("+i+")-("+t+")*("+c+")+("+t+")*("+s+")-("+l+")*("+s+")","("+h+")*("+e+")-("+h+")*("+s+")-("+c+")*("+e+")+("+c+")*("+s+")+("+r+")*("+t+")-("+r+")*("+i+")-("+l+")*("+t+")+("+l+")*("+i+")"]},a},t.Z.createPerpendicularSegment=function(e,r,o){var n,a,l,c,d;if(r[0]=e.select(r[0]),r[1]=e.select(r[1]),s.Z.isPointType(e,r[0])&&r[1].elementClass===i.Z.OBJECT_CLASS_LINE)a=r[1],n=s.Z.providePoints(e,[r[0]],o,"point")[0];else{if(!s.Z.isPointType(e,r[1])||r[0].elementClass!==i.Z.OBJECT_CLASS_LINE)throw new Error("JSXGraph: Can't create perpendicular with parent types '"+typeof r[0]+"' and '"+typeof r[1]+"'.\nPossible parent types: [line,point]");a=r[0],n=s.Z.providePoints(e,[r[1]],o,"point")[0]}return d=s.Z.copyAttributes(o,e.options,"perpendicularsegment","point"),(c=t.Z.createPerpendicularPoint(e,[a,n],d)).dump=!1,s.Z.exists(o.layer)||(o.layer=e.options.layer.line),d=s.Z.copyAttributes(o,e.options,"perpendicularsegment"),(l=t.Z.createLine(e,[function(){return h.Z.perpendicular(a,n,e)[1]?[c,n]:[n,c]}],d)).point=c,s.Z.exists(n._is_new)?(l.addChild(n),delete n._is_new):n.addChild(l),a.addChild(l),l.elType="perpendicularsegment",l.setParents([n.id,a.id]),l.subs={point:c},l.inherits.push(c),l},t.Z.createMidpoint=function(t,e,r){var n,a,h,l,c;for(l=0;l<e.length;++l)e[l]=t.select(e[l]);if(2===e.length&&s.Z.isPointType(t,e[0])&&s.Z.isPointType(t,e[1]))e=s.Z.providePoints(t,e,r,"point"),n=e[0],a=e[1];else{if(1!==e.length||e[0].elementClass!==i.Z.OBJECT_CLASS_LINE)throw new Error("JSXGraph: Can't create midpoint.\nPossible parent types: [point,point], [line]");n=e[0].point1,a=e[0].point2}return c=s.Z.copyAttributes(r,t.options,"midpoint"),h=t.create("point",[function(){var t=n.coords.usrCoords[1]+a.coords.usrCoords[1];return isNaN(t)||Math.abs(n.coords.usrCoords[0])<o.Z.eps||Math.abs(a.coords.usrCoords[0])<o.Z.eps?NaN:.5*t},function(){var t=n.coords.usrCoords[2]+a.coords.usrCoords[2];return isNaN(t)||Math.abs(n.coords.usrCoords[0])<o.Z.eps||Math.abs(a.coords.usrCoords[0])<o.Z.eps?NaN:.5*t}],c),s.Z.exists(n._is_new)?(h.addChild(n),delete n._is_new):n.addChild(h),s.Z.exists(a._is_new)?(h.addChild(a),delete a._is_new):a.addChild(h),h.elType="midpoint",h.setParents([n.id,a.id]),h.prepareUpdate().update(),h.generatePolynomial=function(){var t=n.symbolic.x,e=n.symbolic.y,i=a.symbolic.x,s=a.symbolic.y,r=h.symbolic.x,o=h.symbolic.y;return["("+e+")*("+r+")-("+e+")*("+i+")+("+o+")*("+i+")-("+t+")*("+o+")+("+t+")*("+s+")-("+r+")*("+s+")","("+t+")^2 - 2*("+t+")*("+r+")+("+e+")^2-2*("+e+")*("+o+")-("+i+")^2+2*("+i+")*("+r+")-("+s+")^2+2*("+s+")*("+o+")"]},h},t.Z.createParallelPoint=function(t,e,r){var o,n,a,h,l,c;for(l=0;l<e.length;++l)e[l]=t.select(e[l]);if(3===e.length&&s.Z.isPointType(t,e[0])&&s.Z.isPointType(t,e[1])&&s.Z.isPointType(t,e[2]))e=s.Z.providePoints(t,e,r,"point"),o=e[0],n=e[1],a=e[2];else if(s.Z.isPointType(t,e[0])&&e[1].elementClass===i.Z.OBJECT_CLASS_LINE)a=s.Z.providePoints(t,[e[0]],r,"point")[0],o=e[1].point1,n=e[1].point2;else{if(!s.Z.isPointType(t,e[1])||e[0].elementClass!==i.Z.OBJECT_CLASS_LINE)throw new Error("JSXGraph: Can't create parallel point with parent types '"+typeof e[0]+"', '"+typeof e[1]+"' and '"+typeof e[2]+"'.\nPossible parent types: [line,point], [point,point,point]");a=s.Z.providePoints(t,[e[1]],r,"point")[0],o=e[0].point1,n=e[0].point2}return c=s.Z.copyAttributes(r,t.options,"parallelpoint"),h=t.create("point",[function(){return a.coords.usrCoords[1]+n.coords.usrCoords[1]-o.coords.usrCoords[1]},function(){return a.coords.usrCoords[2]+n.coords.usrCoords[2]-o.coords.usrCoords[2]}],c),s.Z.exists(o._is_new)?(h.addChild(o),delete o._is_new):o.addChild(h),s.Z.exists(n._is_new)?(h.addChild(n),delete n._is_new):n.addChild(h),s.Z.exists(a._is_new)?(h.addChild(a),delete a._is_new):a.addChild(h),h.elType="parallelpoint",h.setParents([o.id,n.id,a.id]),h.prepareUpdate().update(),h.generatePolynomial=function(){var t=o.symbolic.x,e=o.symbolic.y,i=n.symbolic.x,s=n.symbolic.y,r=a.symbolic.x,l=a.symbolic.y,c=h.symbolic.x,d=h.symbolic.y;return["("+s+")*("+c+")-("+s+")*("+r+")-("+e+")*("+c+")+("+e+")*("+r+")-("+d+")*("+i+")+("+d+")*("+t+")+("+l+")*("+i+")-("+l+")*("+t+")","("+d+")*("+t+")-("+d+")*("+r+")-("+s+")*("+t+")+("+s+")*("+r+")-("+c+")*("+e+")+("+c+")*("+l+")+("+i+")*("+e+")-("+i+")*("+l+")"]},h},t.Z.createParallel=function(t,e,i){var r,n,a,h,l,c,d=1;for(l=0;l<e.length;++l)e[l]=t.select(e[l]);return r=null,3===e.length?(r=(e=s.Z.providePoints(t,e,i,"point"))[2],d=0):s.Z.isPointType(t,e[0])?(r=s.Z.providePoints(t,[e[0]],i,"point")[0],h=function(){return e[1].stdform}):s.Z.isPointType(t,e[1])&&(r=s.Z.providePoints(t,[e[1]],i,"point")[0],h=function(){return e[0].stdform}),s.Z.exists(i.layer)||(i.layer=t.options.layer.line),c=s.Z.copyAttributes(i,t.options,"parallel","point"),(n=1===d?t.create("point",[function(){return o.Z.crossProduct([1,0,0],h())}],c):t.create("parallelpoint",e,c)).isDraggable=!0,c=s.Z.copyAttributes(i,t.options,"parallel"),(a=t.create("line",[r,n],c)).elType="parallel",a.subs={point:n},a.inherits.push(n),a.setParents([e[0].id,e[1].id]),3===e.length&&a.addParents(e[2].id),a.point=n,a},t.Z.createArrowParallel=function(e,r,o){var n,a;try{return!1===(a=s.Z.copyAttributes(o,e.options,"arrowparallel")).lastArrow&&(a.lastArrow=!0),(n=t.Z.createParallel(e,r,a).setAttribute({straightFirst:!1,straightLast:!1})).type=i.Z.OBJECT_TYPE_VECTOR,n.elType="arrowparallel",n}catch(t){throw new Error("JSXGraph: Can't create arrowparallel with parent types '"+typeof r[0]+"' and '"+typeof r[1]+"'.\nPossible parent types: [line,point], [point,point,point]")}},t.Z.createBisector=function(e,i,r){var o,n,a,l;if(i=s.Z.providePoints(e,i,r,"point"),s.Z.isPoint(i[0])&&s.Z.isPoint(i[1])&&s.Z.isPoint(i[2])){for((l=s.Z.copyAttributes(r,e.options,"bisector","point")).snapToGrid=!1,(o=e.create("point",[function(){return h.Z.angleBisector(i[0],i[1],i[2],e)}],l)).dump=!1,a=0;a<3;a++)s.Z.exists(i[a]._is_new)?(o.addChild(i[a]),delete i[a]._is_new):i[a].addChild(o);return s.Z.exists(r.layer)||(r.layer=e.options.layer.line),l=s.Z.copyAttributes(r,e.options,"bisector"),(n=t.Z.createLine(e,[i[1],o],l)).point=o,n.elType="bisector",n.setParents(i),n.subs={point:o},n.inherits.push(o),n}throw new Error("JSXGraph: Can't create angle bisector with parent types '"+typeof i[0]+"' and '"+typeof i[1]+"'.\nPossible parent types: [point,point,point]")},t.Z.createAngularBisectorsOfTwoLines=function(t,e,r){var n,a,h,l,c=t.select(e[0]),d=t.select(e[1]);if(c.elementClass!==i.Z.OBJECT_CLASS_LINE||d.elementClass!==i.Z.OBJECT_CLASS_LINE)throw new Error("JSXGraph: Can't create angle bisectors of two lines with parent types '"+typeof e[0]+"' and '"+typeof e[1]+"'.\nPossible parent types: [line,line]");return s.Z.exists(r.layer)||(r.layer=t.options.layer.line),h=s.Z.copyAttributes(r,t.options,"bisectorlines","line1"),n=t.create("line",[function(){var t=o.Z.hypot(c.stdform[1],c.stdform[2]),e=o.Z.hypot(d.stdform[1],d.stdform[2]);return c.stdform[0]/t-d.stdform[0]/e},function(){var t=o.Z.hypot(c.stdform[1],c.stdform[2]),e=o.Z.hypot(d.stdform[1],d.stdform[2]);return c.stdform[1]/t-d.stdform[1]/e},function(){var t=o.Z.hypot(c.stdform[1],c.stdform[2]),e=o.Z.hypot(d.stdform[1],d.stdform[2]);return c.stdform[2]/t-d.stdform[2]/e}],h),s.Z.exists(r.layer)||(r.layer=t.options.layer.line),h=s.Z.copyAttributes(r,t.options,"bisectorlines","line2"),a=t.create("line",[function(){var t=o.Z.hypot(c.stdform[1],c.stdform[2]),e=o.Z.hypot(d.stdform[1],d.stdform[2]);return c.stdform[0]/t+d.stdform[0]/e},function(){var t=o.Z.hypot(c.stdform[1],c.stdform[2]),e=o.Z.hypot(d.stdform[1],d.stdform[2]);return c.stdform[1]/t+d.stdform[1]/e},function(){var t=o.Z.hypot(c.stdform[1],c.stdform[2]),e=o.Z.hypot(d.stdform[1],d.stdform[2]);return c.stdform[2]/t+d.stdform[2]/e}],h),l=new T({line1:n,line2:a}),n.dump=!1,a.dump=!1,l.elType="bisectorlines",l.setParents([c.id,d.id]),l.subs={line1:n,line2:a},l},t.Z.createCircumcenter=function(e,i,r){var o,n,a,l,c;if(i=s.Z.providePoints(e,i,r,"point"),s.Z.isPoint(i[0])&&s.Z.isPoint(i[1])&&s.Z.isPoint(i[2])){for(a=i[0],l=i[1],c=i[2],o=t.Z.createPoint(e,[function(){return h.Z.circumcenter(a,l,c,e)}],r),n=0;n<3;n++)s.Z.exists(i[n]._is_new)?(o.addChild(i[n]),delete i[n]._is_new):i[n].addChild(o);return o.elType="circumcenter",o.setParents(i),o.generatePolynomial=function(){var t=a.symbolic.x,e=a.symbolic.y,i=l.symbolic.x,s=l.symbolic.y,r=c.symbolic.x,n=c.symbolic.y,h=o.symbolic.x,d=o.symbolic.y;return[["((",h,")-(",t,"))^2+((",d,")-(",e,"))^2-((",h,")-(",i,"))^2-((",d,")-(",s,"))^2"].join(""),["((",h,")-(",t,"))^2+((",d,")-(",e,"))^2-((",h,")-(",r,"))^2-((",d,")-(",n,"))^2"].join("")]},o}throw new Error("JSXGraph: Can't create circumcircle midpoint with parent types '"+typeof i[0]+"', '"+typeof i[1]+"' and '"+typeof i[2]+"'.\nPossible parent types: [point,point,point]")},t.Z.createIncenter=function(t,e,r){var n,a,h,l,c;if(!((e=s.Z.providePoints(t,e,r,"point")).length>=3&&s.Z.isPoint(e[0])&&s.Z.isPoint(e[1])&&s.Z.isPoint(e[2])))throw new Error("JSXGraph: Can't create incenter with parent types '"+typeof e[0]+"', '"+typeof e[1]+"' and '"+typeof e[2]+"'.\nPossible parent types: [point,point,point]");for(a=e[0],h=e[1],l=e[2],n=t.create("point",[function(){var e,s,r;return e=o.Z.hypot(h.X()-l.X(),h.Y()-l.Y()),s=o.Z.hypot(a.X()-l.X(),a.Y()-l.Y()),r=o.Z.hypot(h.X()-a.X(),h.Y()-a.Y()),new u.Z(i.Z.COORDS_BY_USER,[(e*a.X()+s*h.X()+r*l.X())/(e+s+r),(e*a.Y()+s*h.Y()+r*l.Y())/(e+s+r)],t)}],r),c=0;c<3;c++)s.Z.exists(e[c]._is_new)?(n.addChild(e[c]),delete e[c]._is_new):e[c].addChild(n);return n.elType="incenter",n.setParents(e),n},t.Z.createCircumcircle=function(e,i,r){var o,n,a,h;if(!1===(i=s.Z.providePoints(e,i,r,"point")))throw new Error("JSXGraph: Can't create circumcircle with parent types '"+typeof i[0]+"', '"+typeof i[1]+"' and '"+typeof i[2]+"'.\nPossible parent types: [point,point,point]");try{for(a=s.Z.copyAttributes(r,e.options,"circumcircle","center"),(o=t.Z.createCircumcenter(e,i,a)).dump=!1,s.Z.exists(r.layer)||(r.layer=e.options.layer.circle),a=s.Z.copyAttributes(r,e.options,"circumcircle"),(n=t.Z.createCircle(e,[o,i[0]],a)).elType="circumcircle",n.setParents(i),n.subs={center:o},n.inherits.push(n),h=0;h<3;h++)s.Z.exists(i[h]._is_new)?(n.addChild(i[h]),delete i[h]._is_new):i[h].addChild(n)}catch(t){throw new Error("JSXGraph: Can't create circumcircle with parent types '"+typeof i[0]+"', '"+typeof i[1]+"' and '"+typeof i[2]+"'.\nPossible parent types: [point,point,point]")}return n},t.Z.createIncircle=function(e,i,r){var n,a,h,l;if(!1===(i=s.Z.providePoints(e,i,r,"point")))throw new Error("JSXGraph: Can't create circumcircle with parent types '"+typeof i[0]+"', '"+typeof i[1]+"' and '"+typeof i[2]+"'.\nPossible parent types: [point,point,point]");try{for(l=s.Z.copyAttributes(r,e.options,"incircle","center"),(a=t.Z.createIncenter(e,i,l)).dump=!1,s.Z.exists(r.layer)||(r.layer=e.options.layer.circle),l=s.Z.copyAttributes(r,e.options,"incircle"),(h=t.Z.createCircle(e,[a,function(){var t=o.Z.hypot(i[1].X()-i[2].X(),i[1].Y()-i[2].Y()),e=o.Z.hypot(i[0].X()-i[2].X(),i[0].Y()-i[2].Y()),s=o.Z.hypot(i[1].X()-i[0].X(),i[1].Y()-i[0].Y()),r=(t+e+s)/2;return Math.sqrt((r-t)*(r-e)*(r-s)/r)}],l)).elType="incircle",h.setParents(i),n=0;n<3;n++)s.Z.exists(i[n]._is_new)?(h.addChild(i[n]),delete i[n]._is_new):i[n].addChild(h);h.center=a,h.subs={center:h.center},h.inherits.push(a)}catch(t){throw new Error("JSXGraph: Can't create circumcircle with parent types '"+typeof i[0]+"', '"+typeof i[1]+"' and '"+typeof i[2]+"'.\nPossible parent types: [point,point,point]")}return h},t.Z.createReflection=function(e,r,o){var n,a,h,l,c,d,u,p,f="\nPossible parent types: [point|line|curve|polygon|circle|arc|sector, line]";for(d=0;d<r.length;++d)r[d]=e.select(r[d]);if(u=s.Z.copyAttributes(o,e.options,"reflection"),s.Z.isPoint(r[0]))a=s.Z.providePoints(e,[r[0]],p)[0];else{if(r[0].elementClass!==i.Z.OBJECT_CLASS_CURVE&&r[0].elementClass!==i.Z.OBJECT_CLASS_LINE&&r[0].type!==i.Z.OBJECT_TYPE_POLYGON&&r[0].elementClass!==i.Z.OBJECT_CLASS_CIRCLE)throw new Error("JSXGraph: Can't create reflection element with parent types '"+typeof r[0]+"' and '"+typeof r[1]+"'."+f);a=r[0]}if(r[1].elementClass!==i.Z.OBJECT_CLASS_LINE)throw new Error("JSXGraph: Can't create reflected element with parent types '"+typeof r[0]+"' and '"+typeof r[1]+"'."+f);if(n=r[1],c=t.Z.createTransform(e,[n],{type:"reflect"}),s.Z.isPoint(a))h=t.Z.createPoint(e,[a,c],u);else if(a.elementClass===i.Z.OBJECT_CLASS_CURVE)h=t.Z.createCurve(e,[a,c],u);else if(a.elementClass===i.Z.OBJECT_CLASS_LINE)h=t.Z.createLine(e,[a,c],u);else if(a.type===i.Z.OBJECT_TYPE_POLYGON)h=t.Z.createPolygon(e,[a,c],u);else{if(a.elementClass!==i.Z.OBJECT_CLASS_CIRCLE)throw new Error("JSXGraph: Can't create reflected element with parent types '"+typeof r[0]+"' and '"+typeof r[1]+"'."+f);"euclidean"===u.type.toLowerCase()?(p=s.Z.copyAttributes(o,e.options,"reflection","center"),(l=t.Z.createPoint(e,[a.center,c],p)).prepareUpdate().update().updateVisibility(l.evalVisProp("visible")).updateRenderer(),h=t.Z.createCircle(e,[l,function(){return a.Radius()}],u)):h=t.Z.createCircle(e,[a,c],u)}return s.Z.exists(a._is_new)&&(h.addChild(a),delete a._is_new),n.addChild(h),h.elType="reflection",h.addParents(n),h.prepareUpdate().update(),s.Z.isPoint(h)&&(h.generatePolynomial=function(){var t=n.point1.symbolic.x,e=n.point1.symbolic.y,i=n.point2.symbolic.x,s=n.point2.symbolic.y,r=a.symbolic.x,o=a.symbolic.y,l=h.symbolic.x,c=h.symbolic.y;return[["((",c,")-(",o,"))*((",e,")-(",s,"))+((",t,")-(",i,"))*((",l,")-(",r,"))"].join(""),["((",l,")-(",t,"))^2+((",c,")-(",e,"))^2-((",r,")-(",t,"))^2-((",o,")-(",e,"))^2"].join("")]}),h},t.Z.createMirrorElement=function(e,r,o){var n,a,h,l,c,d,u,p,f="\nPossible parent types: [point|line|curve|polygon|circle|arc|sector, point]";for(a=0;a<r.length;++a)r[a]=e.select(r[a]);if(u=s.Z.copyAttributes(o,e.options,"mirrorelement"),s.Z.isPoint(r[0]))n=s.Z.providePoints(e,[r[0]],u)[0];else{if(r[0].elementClass!==i.Z.OBJECT_CLASS_CURVE&&r[0].elementClass!==i.Z.OBJECT_CLASS_LINE&&r[0].type!==i.Z.OBJECT_TYPE_POLYGON&&r[0].elementClass!==i.Z.OBJECT_CLASS_CIRCLE)throw new Error("JSXGraph: Can't create mirror element with parent types '"+typeof r[0]+"' and '"+typeof r[1]+"'."+f);n=r[0]}if(!s.Z.isPoint(r[1]))throw new Error("JSXGraph: Can't create mirror element with parent types '"+typeof r[0]+"' and '"+typeof r[1]+"'."+f);if(p=s.Z.copyAttributes(o,e.options,"mirrorelement","point"),h=s.Z.providePoints(e,[r[1]],p)[0],d=t.Z.createTransform(e,[Math.PI,h],{type:"rotate"}),s.Z.isPoint(n))l=t.Z.createPoint(e,[n,d],u);else if(n.elementClass===i.Z.OBJECT_CLASS_CURVE)l=t.Z.createCurve(e,[n,d],u);else if(n.elementClass===i.Z.OBJECT_CLASS_LINE)l=t.Z.createLine(e,[n,d],u);else if(n.type===i.Z.OBJECT_TYPE_POLYGON)l=t.Z.createPolygon(e,[n,d],u);else{if(n.elementClass!==i.Z.OBJECT_CLASS_CIRCLE)throw new Error("JSXGraph: Can't create mirror element with parent types '"+typeof r[0]+"' and '"+typeof r[1]+"'."+f);"euclidean"===u.type.toLowerCase()?(p=s.Z.copyAttributes(o,e.options,"mirrorelement","center"),(c=t.Z.createPoint(e,[n.center,d],p)).prepareUpdate().update().updateVisibility(c.evalVisProp("visible")).updateRenderer(),l=t.Z.createCircle(e,[c,function(){return n.Radius()}],u)):l=t.Z.createCircle(e,[n,d],u)}return s.Z.exists(n._is_new)&&(l.addChild(n),delete n._is_new),h.addChild(l),l.elType="mirrorelement",l.addParents(h),l.prepareUpdate().update(),l},t.Z.createMirrorPoint=function(e,i,s){var r=t.Z.createMirrorElement(e,i,s);return r.elType="mirrorpoint",r},t.Z.createIntegral=function(e,r,o){var n,a,h,l,c,p,f,_,m,g,b,v,Z,C,y,P=null;if(s.Z.isArray(r[0])&&r[1].elementClass===i.Z.OBJECT_CLASS_CURVE)n=r[0],a=r[1];else{if(!s.Z.isArray(r[1])||r[0].elementClass!==i.Z.OBJECT_CLASS_CURVE)throw new Error("JSXGraph: Can't create integral with parent types '"+typeof r[0]+"' and '"+typeof r[1]+"'.\nPossible parent types: [[number|function,number|function],curve]");n=r[1],a=r[0]}return(h=s.Z.copyAttributes(o,e.options,"integral")).withlabel=!1,y=e.create("curve",[[0],[0]],h),e.objectsList.pop(),l=n[0],c=n[1],s.Z.isFunction(l)?(f=function(){return a.Y(p())},l=(p=l)()):(p=l,f=a.Y(l)),s.Z.isFunction(c)?(m=function(){return a.Y(_())},c=(_=c)()):(_=c,m=a.Y(c)),h=s.Z.copyAttributes(o,e.options,"integral","curveleft"),g=e.create("glider",[p,f,a],h),s.Z.isFunction(p)&&g.hideElement(),h=s.Z.copyAttributes(o,e.options,"integral","baseleft"),b=e.create("point",[function(){return"y"===y.evalVisProp("axis")?0:g.X()},function(){return"y"===y.evalVisProp("axis")?g.Y():0}],h),h=s.Z.copyAttributes(o,e.options,"integral","curveright"),v=e.create("glider",[_,m,a],h),s.Z.isFunction(_)&&v.hideElement(),h=s.Z.copyAttributes(o,e.options,"integral","baseright"),Z=e.create("point",[function(){return"y"===y.evalVisProp("axis")?0:v.X()},function(){return"y"===y.evalVisProp("axis")?v.Y():0}],h),y._pos=e.objectsList.length,e.objectsList.push(y),!1!==(h=s.Z.copyAttributes(o,e.options,"integral")).withlabel&&"y"!==h.axis&&(h=s.Z.copyAttributes(o,e.options,"integral","label"),h=s.Z.copyAttributes(h,e.options,"label"),C=function(){var t,e="&int;",i=d.Z.NewtonCotes([b.X(),Z.X()],a.Y),r=P.evalVisProp("digits");return t=P.useLocale()?P.formatNumberLocale(i,r):s.Z.toFixed(i,r),(P.evalVisProp("usemathjax")||P.evalVisProp("usekatex"))&&(e="\\int"),e+" = "+t},(P=e.create("text",[function(){var t=new u.Z(i.Z.COORDS_BY_SCREEN,[this.evalVisProp("offset.0")+this.board.origin.scrCoords[1],0],this.board,!1),e=this.board.getBoundingBox(),s=.1*(e[2]-e[0]),r=v.X();return r<e[0]?r=e[0]+s:r>e[2]&&(r=e[2]-s),r+t.usrCoords[1]},function(){var t=new u.Z(i.Z.COORDS_BY_SCREEN,[0,this.evalVisProp("offset.1")+this.board.origin.scrCoords[2]],this.board,!1),e=this.board.getBoundingBox(),s=.1*(e[1]-e[3]),r=v.Y();return r>e[1]?r=e[1]-s:r<e[3]&&(r=e[3]+s),r+t.usrCoords[2]},""],h)).setText(C),P.dump=!1,g.addChild(P),v.addChild(P)),g.dump=!1,b.dump=!1,v.dump=!1,Z.dump=!1,y.elType="integral",y.setParents([a.id,n]),y.subs={curveLeft:g,baseLeft:b,curveRight:v,baseRight:Z},y.inherits.push(g,b,v,Z),h.withlabel&&(y.subs.label=P,y.inherits.push(P)),y.Value=function(){return d.Z.I([b.X(),Z.X()],a.Y)},y.updateDataArray=function(){var t,e,i,s,r,o,n,h,l;if("y"===this.evalVisProp("axis")){for(g.Y()<v.Y()?(o=g.X(),h=g.Y(),n=v.X(),l=v.Y()):(o=v.X(),h=v.Y(),n=g.X(),l=g.Y()),s=Math.min(o,n),r=Math.max(o,n),t=[0,o],e=[h,h],i=0;i<a.numberPoints;i++)h<=a.points[i].usrCoords[2]&&s<=a.points[i].usrCoords[1]&&a.points[i].usrCoords[2]<=l&&a.points[i].usrCoords[1]<=r&&(t.push(a.points[i].usrCoords[1]),e.push(a.points[i].usrCoords[2]));t.push(n),e.push(l),t.push(0),e.push(l),t.push(0),e.push(h)}else{for(b.X()<Z.X()?(s=b.X(),r=Z.X()):(s=Z.X(),r=b.X()),t=[s,s],e=[0,a.Y(s)],i=0;i<a.numberPoints;i++)s<=a.points[i].usrCoords[1]&&a.points[i].usrCoords[1]<=r&&(t.push(a.points[i].usrCoords[1]),e.push(a.points[i].usrCoords[2]));t.push(r),e.push(a.Y(r)),t.push(r),e.push(0),t.push(s),e.push(0)}this.dataX=t,this.dataY=e},g.addChild(y),v.addChild(y),b.addChild(y),Z.addChild(y),y.baseLeft=b,y.baseRight=Z,y.curveLeft=g,y.curveRight=v,y.methodMap=t.Z.deepCopy(y.methodMap,{curveLeft:"curveLeft",baseLeft:"baseLeft",curveRight:"curveRight",baseRight:"baseRight",Value:"Value"}),y.label=P,y},t.Z.createInequality=function(t,e,r){var n,a,l;if(l=s.Z.copyAttributes(r,t.options,"inequality"),e[0].elementClass===i.Z.OBJECT_CLASS_LINE)(a=t.create("curve",[[],[]],l)).hasPoint=function(){return!1},a.updateDataArray=function(){var s,r,n,a=t.getBoundingBox(),l=this.evalVisProp("inverse"),c=l?-1:1,d=1.5*Math.max(a[2]-a[0],a[1]-a[3]),u={coords:{usrCoords:[1,.5*(a[0]+a[2]),l?a[1]:a[3]]}},p=e[0].stdform.slice(1),f=p;n=1.5*Math.max(h.Z.perpendicular(e[0],u,t)[0].distance(i.Z.COORDS_BY_USER,u.coords),d),n*=c,u={coords:{usrCoords:[1,.5*(a[0]+a[2]),.5*(a[1]+a[3])]}},s=[1,(u=Math.abs(o.Z.innerProduct(u.coords.usrCoords,e[0].stdform,3))>=o.Z.eps?h.Z.perpendicular(e[0],u,t)[0].usrCoords:u.coords.usrCoords)[1]+p[1]*d,u[2]-p[0]*d],r=[1,u[1]-f[1]*d,u[2]+f[0]*d],this.dataX=[s[1],s[1]+p[0]*n,r[1]+f[0]*n,r[1],s[1]],this.dataY=[s[2],s[2]+p[1]*n,r[2]+f[1]*n,r[2],s[2]]};else if(e[0].elementClass===i.Z.OBJECT_CLASS_CURVE&&"functiongraph"===e[0].visProp.curvetype)(a=t.create("curve",[[],[]],l)).updateDataArray=function(){var t,i,s,r,o,n,a,h,l,c=this.board.getBoundingBox(),d=[],u=e[0].minX(),p=e[0].maxX(),f=.3*(c[1]-c[3]);if(t=this.evalVisProp("inverse")?1:3,this.dataX=[],this.dataY=[],0!==(r=e[0].points.length))for(c[1]+=f,c[3]-=f,s=-1;s<r-1;){for(o=s+1,i=r;o<r;o++)if(e[0].points[o].isReal()){i=o;break}if(i>=r)break;for(o=i,s=r-1;o<r-1;o++)if(!e[0].points[o+1].isReal()){s=o;break}for(h=e[0].points[i].usrCoords[1],l=e[0].points[s].usrCoords[1],n=c[0]<u?u:c[0],a=c[2]>p?p:c[2],n=0===i?n:Math.max(n,h),a=s===r-1?a:Math.min(a,l),a=s===r-1?p:l,(d=[]).push([1,n=0===i?u:h,c[t]]),d.push([1,n,e[0].points[i].usrCoords[2]]),o=i;o<=s;o++)d.push(e[0].points[o].usrCoords);for(d.push([1,a,e[0].points[s].usrCoords[2]]),d.push([1,a,c[t]]),d.push(d[0]),o=0;o<d.length;o++)this.dataX.push(d[o][1]),this.dataY.push(d[o][2]);s<r-1&&(this.dataX.push(NaN),this.dataY.push(NaN))}},a.hasPoint=function(){return!1};else if(n=s.Z.createFunction(e[0]),a.addParentsFromJCFunctions([n]),!s.Z.exists(n))throw new Error("JSXGraph: Can't create area with the given parents.\nPossible parent types: [line], [function]");return a.addParents(e[0]),a},t.Z.registerElement("arrowparallel",t.Z.createArrowParallel),t.Z.registerElement("bisector",t.Z.createBisector),t.Z.registerElement("bisectorlines",t.Z.createAngularBisectorsOfTwoLines),t.Z.registerElement("msector",t.Z.createMsector),t.Z.registerElement("circumcircle",t.Z.createCircumcircle),t.Z.registerElement("circumcirclemidpoint",t.Z.createCircumcenter),t.Z.registerElement("circumcenter",t.Z.createCircumcenter),t.Z.registerElement("incenter",t.Z.createIncenter),t.Z.registerElement("incircle",t.Z.createIncircle),t.Z.registerElement("integral",t.Z.createIntegral),t.Z.registerElement("midpoint",t.Z.createMidpoint),t.Z.registerElement("mirrorelement",t.Z.createMirrorElement),t.Z.registerElement("mirrorpoint",t.Z.createMirrorPoint),t.Z.registerElement("orthogonalprojection",t.Z.createOrthogonalProjection),t.Z.registerElement("parallel",t.Z.createParallel),t.Z.registerElement("parallelpoint",t.Z.createParallelPoint),t.Z.registerElement("perpendicular",t.Z.createPerpendicular),t.Z.registerElement("perpendicularpoint",t.Z.createPerpendicularPoint),t.Z.registerElement("perpendicularsegment",t.Z.createPerpendicularSegment),t.Z.registerElement("reflection",t.Z.createReflection),t.Z.registerElement("inequality",t.Z.createInequality),t.Z.createGrid=function(e,r,n){var a,h,l,c,d,u,p,f,_=o.Z.eps,m=5e3,g=[],b=[];if((l=r).length>2||l.length>=1&&"axis"!==l[0].elType||l.length>=2&&"axis"!==l[1].elType)throw new Error("JSXGraph: Can't create 'grid' with parent type '"+r[0].elType+"'. Possible parent types: [axis,axis]");return!s.Z.exists(l[0])&&s.Z.exists(e.defaultAxes)&&(l[0]=e.defaultAxes.x),!s.Z.exists(l[1])&&s.Z.exists(e.defaultAxes)&&(l[1]=e.defaultAxes.y),f=function(t,e,i,s,r,o,n){var a,h,l,c,d,u,p;switch(t.toLowerCase()){case".":case"point":return e.visProp.linecap="round",e.visProp.strokewidth=r*e.board.unitX+o*e.board.unitY,[[i,i,NaN],[s,s,NaN]];case"o":case"circle":return e.visProp.linecap="square",e.bezierDegree=3,[[i+r,i+r,i+(h=4*Math.tan(Math.PI/8)/3)*r,i,i-h*r,i-r,i-r,i-r,i-h*r,i,i+h*r,i+r,i+r,NaN],[s,s+h*o,s+o,s+o,s+o,s+h*o,s,s-h*o,s-o,s-o,s-o,s-h*o,s,NaN]];case"regpol":case"regularpolygon":for(e.visProp.linecap="round",c=e.evalVisProp("polygonvertices"),d=[[],[]],a=0;a<=2*Math.PI;a+=2*Math.PI/c)d[0].push(i-r*Math.sin(a)),d[1].push(s-o*Math.cos(a));return d[0].push(NaN),d[1].push(NaN),d;case"[]":case"square":return e.visProp.linecap="square",[[i-r,i+r,i+r,i-r,i-r,NaN],[s+o,s+o,s-o,s-o,s+o,NaN]];case"<>":case"diamond":return e.visProp.linecap="square",[[i,i+r,i,i-r,i,NaN],[s+o,s,s-o,s,s+o,NaN]];case"<<>>":case"diamond2":return e.visProp.linecap="square",[[i,i+(u=r*Math.sqrt(2)),i,i-u,i,NaN],[s+(p=o*Math.sqrt(2)),s,s-p,s,s+p,NaN]];case"x":case"cross":return[[i-r,i+r,NaN,i-r,i+r,NaN],[s+o,s-o,NaN,s-o,s+o,NaN]];case"+":case"plus":return[[i-r,i+r,NaN,i,i,NaN],[s,s,NaN,s-o,s+o,NaN]];case"-":case"minus":return[[i-r,i+r,NaN],[s,s,NaN]];case"|":case"divide":return[[i,i,NaN],[s-o,s+o,NaN]];case"^":case"a":case"A":case"triangleup":return[[i-r,i,i+r,NaN],[s-o,s,s-o,NaN]];case"v":case"triangledown":return[[i-r,i,i+r,NaN],[s+o,s,s+o,NaN]];case"<":case"triangleleft":return[[i+r,i,i+r,NaN],[s+o,s,s-o,NaN]];case">":case"triangleright":return[[i-r,i,i-r,NaN],[s+o,s,s-o,NaN]];case"line":return l=e.evalVisProp("margin"),[[i,i,NaN,n[0]-l/e.board.unitX,n[2]+l/e.board.unitX,NaN],[n[1]+l/e.board.unitY,n[3]-l/e.board.unitY,NaN,s,s,NaN]];default:return[[],[]]}},c=s.Z.copyAttributes(n,e.options,"grid"),s.Z.mergeAttr(c,c.themes[c.theme],!1),d={},s.Z.mergeAttr(d,c,!0,!0),s.Z.mergeAttr(d,c.major,!0,!0),(a=e.create("curve",[[null],[null]],d)).elType="grid",a.type=i.Z.OBJECT_TYPE_GRID,u={},s.Z.mergeAttr(u,c,!0,!0),s.Z.mergeAttr(u,c.minor,!0,!0),u.id===d.id&&(u.id=a.id+"_minor"),u.name===d.name&&(u.name=a.name+"_minor"),(h=e.create("curve",[[null],[null]],u)).elType="grid",h.type=i.Z.OBJECT_TYPE_GRID,a.minorGrid=h,h.majorGrid=a,a.hasPoint=function(){return!1},h.hasPoint=function(){return!1},a.inherits.push(h),a.updateDataArray=function(){var e,i,r,n,h,c,d,u,v=this.board.getBoundingBox(),Z=this.evalVisProp("gridx"),C=this.evalVisProp("gridy"),y=this.evalVisProp("face"),P=this.evalVisProp("drawzero"),E=!0===P||s.Z.isObject(P)&&!0===this.eval(P.origin),x=!0===P||s.Z.isObject(P)&&!0===this.eval(P.x),O=!0===P||s.Z.isObject(P)&&!0===this.eval(P.y),M=this.evalVisProp("includeboundaries"),w=this.evalVisProp("forcesquare");if(this.dataX=[],this.dataY=[],p=this.evalVisProp("majorstep"),s.Z.isArray(p)||(p=[p,p]),p.length<2&&(p=[p[0],p[0]]),s.Z.exists(Z)&&(t.Z.deprecated("gridX","majorStep"),p[0]=Z),s.Z.exists(C)&&(t.Z.deprecated("gridY","majorStep"),p[1]=C),"auto"===p[0]?(u=Math.pow(10,Math.floor(Math.log(50/this.board.unitX)/Math.LN10)),p[0]=u,s.Z.exists(l[0])&&(p[0]=l[0].ticks[0].getDistanceMajorTicks())):p[0]=s.Z.parseNumber(p[0],Math.abs(v[1]-v[3]),1/this.board.unitX),"auto"===p[1]?(u=Math.pow(10,Math.floor(Math.log(50/this.board.unitY)/Math.LN10)),p[1]=u,s.Z.exists(l[1])&&(p[1]=l[1].ticks[0].getDistanceMajorTicks())):p[1]=s.Z.parseNumber(p[1],Math.abs(v[0]-v[2]),1/this.board.unitY),"min"===w||!0===w?p[0]*this.board.unitX<=p[1]*this.board.unitY?p[1]=p[0]/this.board.unitY*this.board.unitX:p[0]=p[1]/this.board.unitX*this.board.unitY:"max"===w&&(p[0]*this.board.unitX<=p[1]*this.board.unitY?p[0]=p[1]/this.board.unitX*this.board.unitY:p[1]=p[0]/this.board.unitY*this.board.unitX),g=this.evalVisProp("size"),s.Z.isArray(g)||(g=[g,g]),g.length<2&&(g=[g[0],g[0]]),s.Z.isNumber(g[0],!0)&&(g[0]=g[0]+"px"),s.Z.isNumber(g[1],!0)&&(g[1]=g[1]+"px"),g[0]=s.Z.parseNumber(g[0],p[0],1/this.board.unitX),g[1]=s.Z.parseNumber(g[1],p[1],1/this.board.unitY),b[0]=g[0]/2,b[1]=g[1]/2,e=o.Z.roundToStep(v[0],p[0]),i=o.Z.roundToStep(v[1],p[1]),d=isFinite(e)&&isFinite(i)&&isFinite(v[2])&&isFinite(v[3])&&Math.abs(v[2])<Math.abs(p[0]*m)&&Math.abs(v[3])<Math.abs(p[1]*m),"line"===y.toLowerCase()){for(h=a.evalVisProp("margin"),n=i;d&&n>=v[3];n-=p[1])!E&&Math.abs(n)<_||!O&&Math.abs(n)<_||!M&&(n<=v[3]+b[1]||n>=v[1]-b[1])||(c=[[v[0]-h/a.board.unitX,v[2]+h/a.board.unitX,NaN],[n,n,NaN]],s.Z.concat(this.dataX,c[0]),s.Z.concat(this.dataY,c[1]));for(r=e;d&&r<=v[2];r+=p[0])!E&&Math.abs(r)<_||!x&&Math.abs(r)<_||!M&&(r<=v[0]+b[0]||r>=v[2]-b[0])||(c=[[r,r,NaN],[v[1]+h/a.board.unitY,v[3]-h/a.board.unitY,NaN]],s.Z.concat(this.dataX,c[0]),s.Z.concat(this.dataY,c[1]))}else for(n=i;d&&n>=v[3];n-=p[1])for(r=e;d&&r<=v[2];r+=p[0])!E&&Math.abs(n)<_&&Math.abs(r)<_||!x&&Math.abs(n)<_&&Math.abs(r)>=_||!O&&Math.abs(r)<_&&Math.abs(n)>=_||!M&&(r<=v[0]+b[0]||r>=v[2]-b[0]||n<=v[3]+b[1]||n>=v[1]-b[1])||(c=f(y,a,r,n,b[0],b[1],v),s.Z.concat(this.dataX,c[0]),s.Z.concat(this.dataY,c[1]))},h.updateDataArray=function(){var t,e,i,r,n,a,c,d,u,g,v,Z,C,y,P,E,x,O,M,w=this.board.getBoundingBox(),S=[],T=[],N=this.evalVisProp("minorelements"),A=this.evalVisProp("size"),D=this.evalVisProp("face"),k=this.evalVisProp("drawzero"),R=!0===k||s.Z.isObject(k)&&!0===this.eval(k.x),L=!0===k||s.Z.isObject(k)&&!0===this.eval(k.y),B=this.majorGrid.evalVisProp("face"),I=this.majorGrid.evalVisProp("drawzero"),j=!0===I||s.Z.isObject(I)&&!0===this.eval(I.origin),Y=!0===I||s.Z.isObject(I)&&!0===this.eval(I.x),V=!0===I||s.Z.isObject(I)&&!0===this.eval(I.y),X=this.evalVisProp("includeboundaries");if(this.dataX=[],this.dataY=[],s.Z.isArray(N)||(N=[N,N]),N.length<2&&(N=[N[0],N[0]]),s.Z.isNumber(N[0],!0)?N[0]=parseFloat(N[0]):(N[0]=3,s.Z.exists(l[0])&&(N[0]=l[0].eval(l[0].getAttribute("ticks").minorticks))),S[0]=p[0]/(N[0]+1),s.Z.isNumber(N[1],!0)?N[1]=parseFloat(N[1]):(N[1]=3,s.Z.exists(l[1])&&(N[1]=l[1].eval(l[1].getAttribute("ticks").minorticks))),S[1]=p[1]/(N[1]+1),s.Z.isArray(A)||(A=[A,A]),A.length<2&&(A=[A[0],A[0]]),s.Z.isNumber(A[0],!0)&&(A[0]=A[0]+"px"),s.Z.isNumber(A[1],!0)&&(A[1]=A[1]+"px"),A[0]=s.Z.parseNumber(A[0],S[0],1/this.board.unitX),A[1]=s.Z.parseNumber(A[1],S[1],1/this.board.unitY),T[0]=.5*A[0],T[1]=.5*A[1],t=o.Z.roundToStep(w[0],S[0]),e=o.Z.roundToStep(w[1],S[1]),c=isFinite(t)&&isFinite(e)&&isFinite(w[2])&&isFinite(w[3])&&Math.abs(w[2])<=Math.abs(S[0]*m)&&Math.abs(w[3])<Math.abs(S[1]*m),"line"!==D.toLowerCase())for(r=e;c&&r>=w[3];r-=S[1])for(i=t;c&&i<=w[2];i+=S[0]){if(d=o.Z.roundToStep(Math.abs(i),p[0]),d=Math.abs(d-Math.abs(i)),u=p[0]-d,g=o.Z.roundToStep(Math.abs(r),p[1]),g=Math.abs(g-Math.abs(r)),v=p[1]-g,"line"===B){if(d-T[0]-b[0]<_||u-T[0]-b[0]<_||g-T[1]-b[1]<_||v-T[1]-b[1]<_)continue}else if((d-T[0]-b[0]<_||u-T[0]-b[0]<_)&&(g-T[1]-b[1]<_||v-T[1]-b[1]<_)&&(j||b[1]-Math.abs(r)+T[1]<_||b[0]-Math.abs(i)+T[0]<_)&&(Y||b[1]-Math.abs(r)+T[1]<_||b[0]+Math.abs(i)-T[0]<_)&&(V||b[0]-Math.abs(i)+T[0]<_||b[1]+Math.abs(r)-T[1]<_))continue;!L&&Math.abs(i)<_||!R&&Math.abs(r)<_||(Z=Math.abs(w[0]%p[0]),C=Math.abs(w[1]%p[1]),y=Math.abs(w[2]%p[0]),P=Math.abs(w[3]%p[1]),E=p[0]-Z,x=p[1]-C,O=p[0]-y,M=p[1]-P,!X&&(i-T[0]-w[0]-b[0]+E<_&&E-b[0]<_||i-T[0]-w[0]-b[0]-Z<_&&Z-b[0]<_||-i-T[0]+w[2]-b[0]+O<_&&O-b[0]<_||-i-T[0]+w[2]-b[0]-y<_&&y-b[0]<_||-r-T[1]+w[1]-b[1]+x<_&&x-b[1]<_||-r-T[1]+w[1]-b[1]-C<_&&C-b[1]<_||r-T[1]-w[3]-b[1]+M<_&&M-b[1]<_||r-T[1]-w[3]-b[1]-P<_&&P-b[1]<_||-r-T[1]+w[1]<_||i-T[0]-w[0]<_||r-T[1]-w[3]<_||-i-T[0]+w[2]<_)||(a=f(D,h,i,r,T[0],T[1],w),s.Z.concat(this.dataX,a[0]),s.Z.concat(this.dataY,a[1])))}else{for(n=h.evalVisProp("margin"),r=e;c&&r>=w[3];r-=S[1]){if(g=o.Z.roundToStep(Math.abs(r),p[1]),g=Math.abs(g-Math.abs(r)),v=p[1]-g,"line"===B){if(g-T[1]-b[1]<_||v-T[1]-b[1]<_)continue}else if((g-T[1]-b[1]<_||v-T[1]-b[1]<_)&&(j||b[1]-Math.abs(r)+T[1]<_)&&(Y||b[1]-Math.abs(r)+T[1]<_)&&(V||b[1]+Math.abs(r)-T[1]<_))continue;!R&&Math.abs(r)<_||(Z=Math.abs(w[0]%p[0]),C=Math.abs(w[1]%p[1]),y=Math.abs(w[2]%p[0]),P=Math.abs(w[3]%p[1]),E=p[0]-Z,x=p[1]-C,O=p[0]-y,M=p[1]-P,!X&&(-r-T[1]+w[1]-b[1]+x<_&&x-b[1]<_||-r-T[1]+w[1]-b[1]-C<_&&C-b[1]<_||r-T[1]-w[3]-b[1]+M<_&&M-b[1]<_||r-T[1]-w[3]-b[1]-P<_&&P-b[1]<_||-r-T[1]+w[1]<_||r-T[1]-w[3]<_)||(a=[[w[0]-n/h.board.unitX,w[2]+n/h.board.unitX,NaN],[r,r,NaN]],s.Z.concat(this.dataX,a[0]),s.Z.concat(this.dataY,a[1])))}for(i=t;c&&i<=w[2];i+=S[0]){if(d=o.Z.roundToStep(Math.abs(i),p[0]),d=Math.abs(d-Math.abs(i)),u=p[0]-d,"line"===B){if(d-T[0]-b[0]<_||u-T[0]-b[0]<_)continue}else if((d-T[0]-b[0]<_||u-T[0]-b[0]<_)&&(j||b[0]-Math.abs(i)+T[0]<_)&&(Y||b[0]+Math.abs(i)-T[0]<_)&&(V||b[0]-Math.abs(i)+T[0]<_))continue;!L&&Math.abs(i)<_||(Z=Math.abs(w[0]%p[0]),C=Math.abs(w[1]%p[1]),y=Math.abs(w[2]%p[0]),P=Math.abs(w[3]%p[1]),E=p[0]-Z,x=p[1]-C,O=p[0]-y,M=p[1]-P,!X&&(i-T[0]-w[0]-b[0]+E<_&&E-b[0]<_||i-T[0]-w[0]-b[0]-Z<_&&Z-b[0]<_||-i-T[0]+w[2]-b[0]+O<_&&O-b[0]<_||-i-T[0]+w[2]-b[0]-y<_&&y-b[0]<_||i-T[0]-w[0]<_||-i-T[0]+w[2]<_)||(a=[[i,i,NaN],[w[1]+n/h.board.unitY,w[3]-n/h.board.unitY,NaN]],s.Z.concat(this.dataX,a[0]),s.Z.concat(this.dataY,a[1])))}}},e.grids.push(a),e.grids.push(h),h.dump=!1,a.getParents=h.getParents=function(){return l.slice()},a},t.Z.registerElement("grid",t.Z.createGrid);__webpack_require__(573);t.Z.Image=function(e,r,o,n,a){this.constructor(e,o,i.Z.OBJECT_TYPE_IMAGE,i.Z.OBJECT_CLASS_OTHER),this.element=this.board.select(o.anchor),this.coordsConstructor(r),this.W=s.Z.createFunction(a[0],this.board,""),this.H=s.Z.createFunction(a[1],this.board,""),this.addParentsFromJCFunctions([this.W,this.H]),this.usrSize=[this.W(),this.H()],this.size=[Math.abs(this.usrSize[0]*e.unitX),Math.abs(this.usrSize[1]*e.unitY)],this.url=n,this.elType="image",this.span=[this.coords.usrCoords.slice(0),[this.coords.usrCoords[0],this.W(),0],[this.coords.usrCoords[0],0,this.H()]],this.id=this.board.setId(this,"Im"),this.board.renderer.drawImage(this),this.board.finalizeAdding(this),this.methodMap=t.Z.deepCopy(this.methodMap,{addTransformation:"addTransform",trans:"addTransform",W:"W",Width:"W",H:"H",Height:"H",setSize:"setSize"})},t.Z.Image.prototype=new I.Z,s.Z.copyPrototypeMethods(t.Z.Image,j.Z,"coordsConstructor"),t.Z.extend(t.Z.Image.prototype,{hasPoint:function(t,e){var r,n,a,h,l,c,d,p,f,_=this.transformations.length;return s.Z.isObject(this.evalVisProp("precision"))?(h=this.board._inputDevice,l=this.evalVisProp("precision."+h)):l=this.board.options.precision.hasPoint,0===_?(r=t-this.coords.scrCoords[1],n=this.coords.scrCoords[2]-e,r>=-(a=l)&&r-this.size[0]<=a&&n>=-a&&n-this.size[1]<=a):(d=[(c=(c=new u.Z(i.Z.COORDS_BY_SCREEN,[t,e],this.board)).usrCoords)[0]-this.span[0][0],c[1]-this.span[0][1],c[2]-this.span[0][2]],0<=(p=(f=o.Z.innerProduct)(d,this.span[1]))&&p<=f(this.span[1],this.span[1])&&0<=(p=f(d,this.span[2]))&&p<=f(this.span[2],this.span[2]))},update:function(t){return this.needsUpdate?(this.updateCoords(t),this.updateSize(),this.updateSpan(),this):this},updateRenderer:function(){return this.updateRendererGeneric("updateImage")},updateSize:function(){return this.usrSize=[this.W(),this.H()],this.size=[Math.abs(this.usrSize[0]*this.board.unitX),Math.abs(this.usrSize[1]*this.board.unitY)],this},updateSpan:function(){var t,e,i=this.transformations.length,s=[];if(0===i)this.span=[[this.Z(),this.X(),this.Y()],[this.Z(),this.W(),0],[this.Z(),0,this.H()]];else{for(s[0]=[this.Z(),this.X(),this.Y()],s[1]=[this.Z(),this.X()+this.W(),this.Y()],s[2]=[this.Z(),this.X(),this.Y()+this.H()],t=0;t<i;t++)for(e=0;e<3;e++)s[e]=o.Z.matVecMult(this.transformations[t].matrix,s[e]);for(e=0;e<3;e++)s[e][1]/=s[e][0],s[e][2]/=s[e][0],s[e][0]/=s[e][0];for(e=1;e<3;e++)s[e][0]-=s[0][0],s[e][1]-=s[0][1],s[e][2]-=s[0][2];this.span=s}return this},addTransform:function(t){var e;if(s.Z.isArray(t))for(e=0;e<t.length;e++)this.transformations.push(t[e]);else this.transformations.push(t);return this},getParents:function(){var t=[this.url,[this.Z(),this.X(),this.Y()],this.usrSize];return 0!==this.parents.length&&(t=this.parents),t},setSize:function(t,e){return this.W=s.Z.createFunction(t,this.board,""),this.H=s.Z.createFunction(e,this.board,""),this.addParentsFromJCFunctions([this.W,this.H]),this},W:function(){},H:function(){}}),t.Z.createImage=function(e,i,r){var o,n,a=i[0],h=i[1],l=i[2];if(o=s.Z.copyAttributes(r,e.options,"image"),!(n=j.Z.create(t.Z.Image,e,h,o,a,l)))throw new Error("JSXGraph: Can't create image with parent types '"+typeof i[0]+"' and '"+typeof i[1]+"'.\nPossible parent types: [x,y], [z,x,y], [element,transformation]");return 0!==o.rotate&&n.addRotation(o.rotate),n},t.Z.registerElement("image",t.Z.createImage);t.Z.Image;t.Z.createSlider=function(t,e,r){var n,a,h,l,c,d,p,f,_,m,g,b,v,Z,C,y,P,E,x,O,M,w,S;return P=(S=s.Z.copyAttributes(r,t.options,"slider")).withticks,y=S.withlabel,O=S.snapwidth,E=S.snapvalues,x=S.snapvaluedistance,p=t.create("point",e[0],S.point1),f=t.create("point",e[1],S.point2),(m=t.create("segment",[p,f],S.baseline)).updateStdform(),n=p.coords.usrCoords.slice(1),a=f.coords.usrCoords.slice(1),h=e[2][0],l=e[2][1],c=e[2][2],d=c-h,w=-1===(M=s.Z.evaluate(O))?l:Math.round((l-h)/M)*M+h,Z=n[0]+(a[0]-n[0])*(w-h)/(c-h),C=n[1]+(a[1]-n[1])*(w-h)/(c-h),S.withlabel=!1,(_=t.create("glider",[Z,C,m],S)).setAttribute({snapwidth:O,snapvalues:E,snapvaluedistance:x}),g=t.create("segment",[p,_],S.highline),_.Value=function(){var t=this._smax-this._smin,e=this.evalVisProp("snapwidth");return-1===e?this.position*t+this._smin:Math.round(this.position*t/e)*e+this._smin},_.methodMap=s.Z.deepCopy(_.methodMap,{Value:"Value",setValue:"setValue",smax:"_smax",smin:"_smin",setMax:"setMax",setMin:"setMin",point1:"point1",point2:"point2",baseline:"baseline",highline:"highline",ticks:"ticks",label:"label"}),_._smax=c,_._smin=h,_.setMax=function(t){return this._smax=t,this},_.setValue=function(t){var e=this._smax-this._smin;return Math.abs(e)>o.Z.eps?this.position=(t-this._smin)/e:this.position=0,this.position=Math.max(0,Math.min(1,this.position)),this},_.setMin=function(t){return this._smin=t,this},y&&(v=t.create("text",[function(){return.05*(f.X()-p.X())+f.X()},function(){return.05*(f.Y()-p.Y())+f.Y()},function(){var t,e=_.evalVisProp("digits"),i=_.evalVisProp("suffixlabel"),r=_.evalVisProp("unitlabel"),o=_.evalVisProp("postlabel");return 2===e&&2!==_.evalVisProp("precision")&&(e=_.evalVisProp("precision")),t=null!==i?i:_.name&&""!==_.name?_.name+" = ":"",_.useLocale()?t+=_.formatNumberLocale(_.Value(),e):t+=s.Z.toFixed(_.Value(),e),null!==r&&(t+=r),null!==o&&(t+=o),t}],S.label),_.label=v,_.visProp.withlabel=!0,_.hasLabel=!0),_.point1=p,_.point2=f,_.baseline=m,_.highline=g,P&&(s.Z.exists(S.generatelabeltext)||(S.ticks.generateLabelText=function(t,e,i){var s=_.point1.Dist(_.point2),r=_._smin,n=_._smax,a=this.getDistanceFromZero(e,t)*(n-r)/s+r;return s<o.Z.eps||Math.abs(a)<o.Z.eps?"0":this.formatLabelText(a)}),2,b=t.create("ticks",[_.baseline,_.point1.Dist(p)/2,function(t){var e=_.point1.Dist(_.point2),s=_.point1.coords.distance(i.Z.COORDS_BY_USER,t);return e<o.Z.eps?0:s/e*d+h}],S.ticks),_.ticks=b),_.remove=function(){y&&t.removeObject(v),t.removeObject(g),t.removeObject(m),t.removeObject(f),t.removeObject(p),Y.prototype.remove.call(_)},p.dump=!1,f.dump=!1,m.dump=!1,g.dump=!1,y&&(v.dump=!1),_.elType="slider",_.parents=e,_.subs={point1:p,point2:f,baseLine:m,highLine:g},_.inherits.push(p,f,m,g),m.inherits=[],g.inherits=[],P&&(b.dump=!1,_.subs.ticks=b,_.inherits.push(b)),_.getParents=function(){return[this.point1.coords.usrCoords.slice(1),this.point2.coords.usrCoords.slice(1),[this._smin,this.position*(this._smax-this._smin)+this._smin,this._smax]]},_.baseline.on("up",(function(t){var e,s;_.evalVisProp("moveonup")&&!_.evalVisProp("fixed")&&(e=m.board.getMousePosition(t,0),s=new u.Z(i.Z.COORDS_BY_SCREEN,e,this.board),_.moveTo([s.usrCoords[1],s.usrCoords[2]]),_.triggerEventHandlers(["drag"],[t]))})),_.prepareUpdate().update(),t.isSuspendedUpdate||(_.updateVisibility().updateRenderer(),_.baseline.updateVisibility().updateRenderer(),_.highline.updateVisibility().updateRenderer(),P&&_.ticks.updateVisibility().updateRenderer()),_},t.Z.registerElement("slider",t.Z.createSlider),t.Z.PrefixParser={parse:function(t,e){var i,r,n,a,h,l;if(s.Z.isNumber(t)||s.Z.isString(t))return t;if(!s.Z.isArray(t)||t.length<2)throw new Error("prefixParser.parse: term is not an array, number or string");if(i=t[0],n=t.length,"execute"===e)if(s.Z.isInArray(["+","-","*","/"],i))for(a=this.parse(t[1],e),r=2;r<n;r++)switch(l=this.parse(t[r],e),i){case"+":a+=l;break;case"-":a-=l;break;case"*":a*=l;break;case"/":a/=l}else if("exec"===i){for(h=t[1],l=[],r=2;r<n;r++)l.push(this.parse(t[r],e));if(s.Z.exists(Math[h]))a=Math[h].apply(this,l);else{if(!s.Z.exists(o.Z[h]))throw new Error("PrefixParser.parse: "+h+" is not allowed");a=o.Z[h].apply(this,l)}}else{if("V"===(h=t[0])&&(h="Value"),"Coords"===h&&(t[2]="true"),!s.Z.exists(t[1][h]))throw new Error("PrefixParser.parse: "+h+" is not a method of "+t[1]);for(l=[],r=2;r<n;r++)l.push(this.parse(t[r],e));a=t[1][h].apply(t[1],l)}return a},dimension:function(t){var e,r,o,n,a,h,l,c;if(s.Z.isNumber(t))return 0;if(!s.Z.isArray(t)||t.length<2)throw new Error("PrefixParser.dimension: term is not an array");if(e=t[0],o=t.length,s.Z.isInArray(["+","-","*","/"],e))for(n=this.dimension(t[1]),r=2;r<o;r++)switch(l=this.dimension(t[r]),e){case"+":case"-":l!==n&&(n=NaN);break;case"*":n+=l;break;case"/":n-=l}else if("exec"===e)t[2].type===s.Z.OBJECT_TYPE_MEASUREMENT?(n=t[2].Dimension(),s.Z.exists(t[2].visProp.dim)&&null!==(h=t[2].evalVisProp("dim"))&&(n=h)):n=0;else switch(a=t[0]){case"L":case"Length":case"Perimeter":case"Radius":case"R":n=1;break;case"Area":case"A":n=2;break;default:t[1].type===s.Z.OBJECT_TYPE_MEASUREMENT?(n=t[1].Dimension(),s.Z.exists(t[1].visProp.dim)&&null!==(h=t[1].evalVisProp("dim"))&&(n=h)):(n=0,"Value"!==a&&"V"!==a||[i.Z.OBJECT_TYPE_ARC,i.Z.OBJECT_TYPE_SECTOR,i.Z.OBJECT_TYPE_ANGLE].indexOf(t[1].type)>=0&&(c="",3===t.length&&s.Z.isString(t[2])&&(c=t[2].toLowerCase()),n=""===c?t[1].type===i.Z.OBJECT_TYPE_ANGLE?0:1:0===c.indexOf("len")?1:0))}return n},toPrefix:function(t){var e,i,r,o;if(s.Z.isNumber(t))return t;if(!s.Z.isArray(t)||t.length<2)throw new Error("PrefixParser.toPrefix: term is not an array");for(e=t[0],r=t.length,o=[e],i=1;i<r;i++)s.Z.isInArray(["+","-","*","/"],e)?o.push(this.toPrefix(t[i])):"V"===e&&t[i].type===s.Z.OBJECT_TYPE_MEASUREMENT?o=t[i].toPrefix():"exec"===e?1===i?o.push(t[i]):o.push(this.toPrefix(t[i])):o=[e,t[i].id];return o},getParents:function(t){var e,i,r,o;if(s.Z.isNumber(t))return[];if(!s.Z.isArray(t)||t.length<2)throw new Error("PrefixParser.getParents: term is not an array");for(e=t[0],r=t.length,o=[],i=1;i<r;i++)s.Z.isInArray(["+","-","*","/"],e)?s.Z.concat(o,this.getParents(t[i])):"V"===e&&t[i].type===s.Z.OBJECT_TYPE_MEASUREMENT?s.Z.concat(o,t[i].getParents()):"exec"===e?i>1&&s.Z.concat(o,this.getParents(t[i])):o.push(t[i]);return o}};var U=t.Z.PrefixParser;t.Z.createTapemeasure=function(e,i,r){var o,n,a,h,l,c,d,u,p,f;return o=i[0],n=i[1],a=s.Z.copyAttributes(r,e.options,"tapemeasure","point1"),d=e.create("point",o,a),a=s.Z.copyAttributes(r,e.options,"tapemeasure","point2"),u=e.create("point",n,a),d.setAttribute({ignoredSnapToPoints:[u.id]}),u.setAttribute({ignoredSnapToPoints:[d.id]}),a=s.Z.copyAttributes(r,e.options,"tapemeasure"),h=a.withticks,l=a.withlabel,2===a.digits&&2!==a.precision&&a.precision,l&&(a.withlabel=!0),c=e.create("segment",[d,u],a),l&&(p=r.name&&""!==r.name?r.name+" = ":"",c.label.setText((function(){var t=c.label.evalVisProp("digits");return c.label.useLocale()?p+c.label.formatNumberLocale(d.Dist(u),t):p+s.Z.toFixed(d.Dist(u),t)}))),h&&(a=s.Z.copyAttributes(r,e.options,"tapemeasure","ticks"),f=e.create("ticks",[c,.1],a),c.inherits.push(f)),c.remove=function(){h&&c.removeTicks(f),e.removeObject(u),e.removeObject(d),I.Z.prototype.remove.call(this)},c.Value=function(){return d.Dist(u)},d.dump=!1,u.dump=!1,c.elType="tapemeasure",c.getParents=function(){return[[d.X(),d.Y()],[u.X(),u.Y()]]},c.subs={point1:d,point2:u},h&&(f.dump=!1),c.methodMap=t.Z.deepCopy(c.methodMap,{Value:"Value"}),c.prepareUpdate().update(),e.isSuspendedUpdate||(c.updateVisibility().updateRenderer(),c.point1.updateVisibility().updateRenderer(),c.point2.updateVisibility().updateRenderer()),c},t.Z.registerElement("tapemeasure",t.Z.createTapemeasure),t.Z.createMeasurement=function(t,e,i){var r,o,n,a,h,l;for(o=s.Z.copyAttributes(i,t.options,"measurement"),n=e[0],a=e[1],h=e[2],(r=t.create("text",[n,a,""],o)).type=s.Z.OBJECT_TYPE_MEASUREMENT,r.elType="measurement",r.Value=function(){return U.parse(h,"execute")},r.Dimension=function(){var t=r.evalVisProp("dim");return null!==t?t:U.dimension(h)},r.Unit=function(t){var e,i="",o=r.evalVisProp("units"),n=t,a={};if(s.Z.exists(n)||(n=r.Dimension()),s.Z.isArray(t)){for(e=0;e<t.length;e++)a["dim"+t[e]]=r.Unit(t[e]);return a}return s.Z.isObject(o)&&s.Z.exists(o[n])&&!1!==o[n]?i=r.eval(o[n]):s.Z.isObject(o)&&s.Z.exists(o["dim"+n])&&!1!==o["dim"+n]?i=r.eval(o["dim"+n]):(i=r.evalVisProp("baseunit"),0===n?i="":n>1&&""!==i&&(i=i+"^{"+n+"}")),i},r.getTerm=function(){return h},r.getMethod=function(){var t=h[0];return"V"===t&&(t="Value"),t},r.toPrefix=function(){return U.toPrefix(h)},r.getParents=function(){return U.getParents(h)},r.addParents(r.getParents()),l=0;l<r.parents.length;l++)t.select(r.parents[l]).addChild(r);return r.setText((function(){var t,e="",i="",o=r.Dimension(),n=r.evalVisProp("digits"),a=r.Unit(),h=r.Value();if(r.evalVisProp("showprefix")&&(e=r.evalVisProp("prefix")),r.evalVisProp("showsuffix")&&(i=r.evalVisProp("suffix")),s.Z.isNumber(h))"none"===n||(h="auto"===n?r.useLocale()?r.formatNumberLocale(h):s.Z.autoDigits(h):r.useLocale()?r.formatNumberLocale(h,n):s.Z.toFixed(h,n));else if(s.Z.isArray(h))for(t=0;t<h.length;t++)s.Z.isNumber(h[t])&&("none"===n||("auto"===n?r.useLocale()?h[t]=r.formatNumberLocale(h[t]):h[t]=s.Z.autoDigits(h[t]):r.useLocale()?h[t]=r.formatNumberLocale(h[t],n):h[t]=s.Z.toFixed(h[t],n)));return"coords"===o&&s.Z.isArray(h)&&(2===h.length&&h.unshift(void 0),h=r.visProp.formatcoords(r,h[1],h[2],h[0])),"direction"===o&&s.Z.isArray(h)&&(h=r.visProp.formatdirection(r,h[0],h[1])),s.Z.isString(o)?e+h+i:isNaN(o)?e+"NaN"+i:e+h+a+i})),r.methodMap=s.Z.deepCopy(r.methodMap,{Value:"Value",Dimension:"Dimension",Unit:"Unit",getTerm:"getTerm",Term:"getTerm",getMethod:"getMethod",Method:"getMethod",getParents:"getParents",Parents:"getParents"}),r},t.Z.registerElement("measurement",t.Z.createMeasurement),t.Z.DataSource=function(){return this.data=[],this.columnHeaders=[],this.rowHeaders=[],this},t.Z.extend(t.Z.DataSource.prototype,{loadFromArray:function(t,e,i){var r,o,n;if(s.Z.isArray(e)&&(this.columnHeaders=e,e=!1),s.Z.isArray(i)&&(this.rowHeaders=i,i=!1),this.data=[],e&&(this.columnHeaders=[]),i&&(this.rowHeaders=[]),s.Z.exists(t)){for(this.data=[],r=0;r<t.length;r++)for(this.data[r]=[],o=0;o<t[r].length;o++)n=t[r][o],parseFloat(n).toString()===n?this.data[r][o]=parseFloat(n):this.data[r][o]="-"!==n?n:NaN;if(e&&(this.columnHeaders=this.data[0].slice(1),this.data=this.data.slice(1)),i)for(this.rowHeaders=[],r=0;r<this.data.length;r++)this.rowHeaders.push(this.data[r][0]),this.data[r]=this.data[r].slice(1)}return this},loadFromTable:function(t,e,i){var r,o,n,a,h;if(s.Z.isArray(e)&&(this.columnHeaders=e,e=!1),s.Z.isArray(i)&&(this.rowHeaders=i,i=!1),this.data=[],e&&(this.columnHeaders=[]),i&&(this.rowHeaders=[]),t=document.getElementById(t),s.Z.exists(t)){for(r=t.getElementsByTagName("tr"),this.data=[],o=0;o<r.length;o++)for(a=r[o].getElementsByTagName("td"),this.data[o]=[],n=0;n<a.length;n++)h=a[n].innerHTML,parseFloat(h).toString()===h?this.data[o][n]=parseFloat(h):this.data[o][n]="-"!==h?h:NaN;if(e&&(this.columnHeaders=this.data[0].slice(1),this.data=this.data.slice(1)),i)for(this.rowHeaders=[],o=0;o<this.data.length;o++)this.rowHeaders.push(this.data[o][0]),this.data[o]=this.data[o].slice(1)}return this},addColumn:function(t,e,i){throw new Error("not implemented")},addRow:function(t,e,i){throw new Error("not implemented")},getColumn:function(t){var e,i=[];if(s.Z.isString(t))for(e=0;e<this.columnHeaders.length;e++)if(t===this.columnHeaders[e]){t=e;break}for(e=0;e<this.data.length;e++)this.data[e].length>t&&(i[e]=parseFloat(this.data[e][t]));return i},getRow:function(t){var e,i;if(s.Z.isString(t))for(i=0;i<this.rowHeaders.length;i++)if(t===this.rowHeaders[i]){t=i;break}for(e=[],i=0;i<this.data[t].length;i++)e[i]=this.data[t][i];return e}});var F=t.Z.DataSource;t.Z.Chart=function(t,e,i){var r,o,n,a,h,l;if(this.constructor(t,i),!s.Z.isArray(e)||0===e.length)throw new Error("JSXGraph: Can't create a chart without data");if(this.elements=[],s.Z.isNumber(e[0]))for(o=e,r=[],n=0;n<o.length;n++)r[n]=n+1;else if(1===e.length&&s.Z.isArray(e[0]))for(o=e[0],r=[],l=s.Z.evaluate(o).length,n=0;n<l;n++)r[n]=n+1;else 2===e.length&&(l=Math.min(e[0].length,e[1].length),r=e[0].slice(0,l),o=e[1].slice(0,l));if(s.Z.isArray(o)&&0===o.length)throw new Error("JSXGraph: Can't create charts without data.");for(h=i.chartstyle.replace(/ /g,"").split(","),n=0;n<h.length;n++){switch(h[n]){case"bar":a=this.drawBar(t,r,o,i);break;case"line":a=this.drawLine(t,r,o,i);break;case"fit":a=this.drawFit(t,r,o,i);break;case"spline":a=this.drawSpline(t,r,o,i);break;case"pie":a=this.drawPie(t,o,i);break;case"point":a=this.drawPoints(t,r,o,i);break;case"radar":a=this.drawRadar(t,e,i)}this.elements.push(a)}return this.id=this.board.setId(this,"Chart"),this.elements},t.Z.Chart.prototype=new I.Z,t.Z.extend(t.Z.Chart.prototype,{drawLine:function(t,e,i,s){return s.fillcolor="none",s.highlightfillcolor="none",t.create("curve",[e,i],s)},drawSpline:function(t,e,i,s){return s.fillColor="none",s.highlightfillcolor="none",t.create("spline",[e,i],s)},drawFit:function(t,e,i,s){var r=s.degree;return r=Math.max(parseInt(r,10),1)||1,s.fillcolor="none",s.highlightfillcolor="none",t.create("functiongraph",[d.Z.regressionPolynomial(r,e,i)],s)},drawBar:function(t,e,i,r){var o,n,a,h,l,c,d,u,p,f,_=[],m=[],g=function(t,i){return function(){return e[t]()-i*a}},b={fixed:!0,withLabel:!1,visible:!1,name:""};if((p=s.Z.copyAttributes(r,t.options,"chart"))&&p.width)a=p.width;else{if(e.length<=1)a=1;else for(a=e[1]-e[0],o=1;o<e.length-1;o++)a=e[o+1]-e[o]<a?e[o+1]-e[o]:a;a*=.8}for(f=s.Z.copyAttributes(r,t.options,"chart","label"),o=0;o<e.length;o++)s.Z.isFunction(e[o])?(h=g(o,-.5),l=g(o,0),c=g(o,.5)):(h=e[o]-.5*a,l=e[o],c=e[o]+.5*a),d=s.Z.isFunction(i[o])?i[o]():i[o],d=i[o],"horizontal"===p.dir?(m[0]=t.create("point",[0,h],b),m[1]=t.create("point",[d,h],b),m[2]=t.create("point",[d,c],b),m[3]=t.create("point",[0,c],b),s.Z.exists(p.labels)&&s.Z.exists(p.labels[o])&&(f.anchorX=function(t){return t.X()>=0?"left":"right"},f.anchorY="middle",n=t.create("text",[d,l,p.labels[o]],f))):(m[0]=t.create("point",[h,0],b),m[1]=t.create("point",[h,d],b),m[2]=t.create("point",[c,d],b),m[3]=t.create("point",[c,0],b),s.Z.exists(p.labels)&&s.Z.exists(p.labels[o])&&(f.anchorX="middle",f.anchorY=function(t){return t.Y()>=0?"bottom":"top"},n=t.create("text",[l,d,p.labels[o]],f))),s.Z.isArray(p.colors)&&(u=p.colors,p.fillcolor=u[o%u.length]),_[o]=t.create("polygon",m,p),s.Z.exists(p.labels)&&s.Z.exists(p.labels[o])&&(_[o].text=n,_[o].addChild(n));return _},drawPoints:function(t,e,i,s){var r,o=[],n=s.infoboxarray;for(s.fixed=!0,s.name="",r=0;r<e.length;r++)s.infoboxtext=!!n&&n[r%n.length],o[r]=t.create("point",[e[r],i[r]],s);return o},drawPie:function(t,e,r){var o,n,a=[],h=[],l=r.colors,c=r.highlightcolors,d=r.labels,p=r.radius||4,f=p,_=r.center||[0,0],m=_[0],g=_[1],b=function(t,i,r){return function(){var o,n,a,h=0;for(n=0;n<=t;n++)h+=parseFloat(s.Z.evaluate(e[n]));for(o=h,n=t+1;n<e.length;n++)o+=parseFloat(s.Z.evaluate(e[n]));return a=0!==o?2*Math.PI*h/o:0,f()*Math[i](a)+r}},v=function(t,e){var r=-this.point1.coords.usrCoords[1]+this.point2.coords.usrCoords[1],o=-this.point1.coords.usrCoords[2]+this.point2.coords.usrCoords[2];s.Z.exists(this.label)&&(this.label.rendNode.style.fontSize=e*this.label.evalVisProp("fontsize")+"px",this.label.fullUpdate()),this.point2.coords=new u.Z(i.Z.COORDS_BY_USER,[this.point1.coords.usrCoords[1]+r*t,this.point1.coords.usrCoords[2]+o*t],this.board),this.fullUpdate()},Z=function(){this.highlighted||(this.highlighted=!0,this.board.highlightedObjects[this.id]=this,this.board.renderer.highlight(this),v.call(this,1.1,2))},C=function(){this.highlighted&&(this.highlighted=!1,this.board.renderer.noHighlight(this),v.call(this,.9090909,1))},y={fixed:!0,withLabel:!1,visible:!1,name:""};if(!s.Z.isArray(d))for(d=[],o=0;o<e.length;o++)d[o]="";for(s.Z.isFunction(p)||(f=function(){return p}),r.highlightonsector=r.highlightonsector||!1,r.straightfirst=!1,r.straightlast=!1,n=t.create("point",[m,g],y),a[0]=t.create("point",[function(){return f()+m},function(){return g}],y),o=0;o<e.length;o++)a[o+1]=t.create("point",[b(o,"cos",m),b(o,"sin",g)],y),r.name=d[o],r.withlabel=""!==r.name,r.fillcolor=l&&l[o%l.length],r.labelcolor=l&&l[o%l.length],r.highlightfillcolor=c&&c[o%c.length],h[o]=t.create("sector",[n,a[o],a[o+1]],r),r.highlightonsector&&(h[o].hasPoint=h[o].hasPointSector),r.highlightbysize&&(h[o].highlight=Z,h[o].noHighlight=C);return{sectors:h,points:a,midpoint:n}},drawRadar:function(e,r,o){var n,a,h,l,c,d,p,f,_,m,g,b,v,Z,C,y,P,E,x,O,M,w,S,T,N,A,D,k,R,L,B,I,j,Y,V,X,U,F,z,J,G,W,H=r.length,K=function(){var t,e,s,r,o=this.evalVisProp("label.offset).slice(0");return t=this.point1.X(),e=this.point2.X(),s=this.point1.Y(),r=this.point2.Y(),e<t&&(o[0]=-o[0]),r<s&&(o[1]=-o[1]),this.setLabelRelativeCoords(o),new u.Z(i.Z.COORDS_BY_USER,[this.point2.X(),this.point2.Y()],this.board)},$=function(t,i){var s,r,o;return s=e.create("transform",[-(C[i]-v[i]),0],{type:"translate"}),r=e.create("transform",[x/(y[i]+Z[i]-(C[i]-v[i])),1],{type:"scale"}),s.melt(r),o=e.create("transform",[t],{type:"rotate"}),s.melt(o),s};if(H<=0)throw new Error("JSXGraph radar chart: no data");if(h=o.paramarray,!s.Z.exists(h))throw new Error("JSXGraph radar chart: need paramArray attribute");if((l=h.length)<=1)throw new Error("JSXGraph radar chart: need more than one param in paramArray");for(n=0;n<H;n++)if(l!==r[n].length)throw new Error("JSXGraph radar chart: use data length equal to number of params ("+r[n].length+" != "+l+")");for(c=[],d=[],a=0;a<l;a++)c[a]=r[0][a],d[a]=c[a];for(n=1;n<H;n++)for(a=0;a<l;a++)r[n][a]>c[a]&&(c[a]=r[n][a]),r[n][a]<d[a]&&(d[a]=r[n][a]);for(p=[],f=[],n=0;n<H;n++)p[n]="",f[n]=[];for(_=[],m=[],g=o.startshiftratio||0,b=o.endshiftratio||0,n=0;n<l;n++)_[n]=(c[n]-d[n])*g,m[n]=(c[n]-d[n])*b;if(v=o.startshiftarray||_,Z=o.endshiftarray||m,C=o.startarray||d,s.Z.exists(o.start))for(n=0;n<l;n++)C[n]=o.start;if(y=o.endarray||c,s.Z.exists(o.end))for(n=0;n<l;n++)y[n]=o.end;if(v.length!==l)throw new Error("JSXGraph radar chart: start shifts length is not equal to number of parameters");if(Z.length!==l)throw new Error("JSXGraph radar chart: end shifts length is not equal to number of parameters");if(C.length!==l)throw new Error("JSXGraph radar chart: starts length is not equal to number of parameters");if(y.length!==l)throw new Error("JSXGraph radar chart: snds length is not equal to number of parameters");for(P=o.labelarray||p,E=o.colors,x=o.radius||10,G=o.strokewidth||1,s.Z.exists(o.highlightonsector)||(o.highlightonsector=!1),O={name:o.name,id:o.id,strokewidth:G,polystrokewidth:o.polystrokewidth||G,strokecolor:o.strokecolor||"black",straightfirst:!1,straightlast:!1,fillcolor:o.fillColor||"#FFFF88",fillopacity:o.fillOpacity||.4,highlightfillcolor:o.highlightFillColor||"#FF7400",highlightstrokecolor:o.highlightStrokeColor||"black",gradient:o.gradient||"none"},w=(M=o.center||[0,0])[0],S=M[1],T=e.create("point",[w,S],{name:"",fixed:!0,withlabel:!1,visible:!1}),Math.PI/2-Math.PI/l,A=N=o.startangle||0,D=[],k=[],n=0;n<l;n++)for(A+=2*Math.PI/l,L=x*Math.cos(A)+w,B=x*Math.sin(A)+S,D[n]=e.create("point",[L,B],{name:"",fixed:!0,withlabel:!1,visible:!1}),k[n]=e.create("line",[T,D[n]],{name:h[n],strokeColor:O.strokecolor,strokeWidth:O.strokewidth,strokeOpacity:1,straightFirst:!1,straightLast:!1,withLabel:!0,highlightStrokeColor:O.highlightstrokecolor}),k[n].getLabelAnchor=K,R=$(A,n),a=0;a<r.length;a++)W=r[a][n],f[a][n]=e.create("point",[W,0],{name:"",fixed:!0,withlabel:!1,visible:!1}),f[a][n].addTransform(f[a][n],R);for(I=[],n=0;n<H;n++)for(O.labelcolor=E&&E[n%E.length],O.strokecolor=E&&E[n%E.length],O.fillcolor=E&&E[n%E.length],I[n]=e.create("polygon",f[n],{withLines:!0,withLabel:!1,fillColor:O.fillcolor,fillOpacity:O.fillopacity,highlightFillColor:O.highlightfillcolor}),a=0;a<l;a++)I[n].borders[a].setAttribute("strokecolor:"+E[n%E.length]),I[n].borders[a].setAttribute("strokewidth:"+O.polystrokewidth);switch(o.legendposition||"none"){case"right":Y=o.legendleftoffset||2,V=o.legendtopoffset||1,this.legend=e.create("legend",[w+x+Y,S+x-V],{labels:P,colors:E});break;case"none":break;default:t.Z.debug("Unknown legend position")}if(j=[],o.showcircles){for(X=[],n=0;n<6;n++)X[n]=20*n;if(X[0]="0",(F=(U=o.circlelabelarray||X).length)<2)throw new Error("JSXGraph radar chart: too less circles in circleLabelArray");for(z=[],R=$(N+Math.PI/l,0),O.fillcolor="none",O.highlightfillcolor="none",O.strokecolor=o.strokecolor||"black",O.strokewidth=o.circlestrokewidth||.5,O.layer=0,J=(y[0]-C[0])/(F-1),n=0;n<F;n++)z[n]=e.create("point",[C[0]+n*J,0],{name:U[n],size:0,fixed:!0,withLabel:!0,visible:!0}),z[n].addTransform(z[n],R),j[n]=e.create("circle",[T,z[n]],O)}return this.rendNode=I[0].rendNode,{circles:j,lines:k,points:f,midpoint:T,polygons:I}},updateRenderer:function(){return this},update:function(){return this.needsUpdate&&this.updateDataArray(),this},updateDataArray:function(){return this}}),t.Z.createChart=function(i,r,o){var n,a,h,l,c,d,u,p,f,_,m,g,b,v,Z,C,y=[],P=e.Z.isBrowser?i.document.getElementById(r[0]):null;if(1===r.length&&s.Z.isString(r[0])){if(s.Z.exists(P)){if(f=s.Z.copyAttributes(o,i.options,"chart"),n=(P=(new F).loadFromTable(r[0],f.withheaders,f.withheaders)).data,c=P.columnHeaders,a=P.rowHeaders,_=f.width,m=f.name,g=f.strokecolor,b=f.fillcolor,v=f.highlightstrokecolor,Z=f.highlightfillcolor,i.suspendUpdate(),C=n.length,p=[],f.rows&&s.Z.isArray(f.rows)){for(h=0;h<C;h++)for(l=0;l<f.rows.length;l++)if(f.rows[l]===h||f.withheaders&&f.rows[l]===a[h]){p.push(n[h]);break}}else p=n;for(C=p.length,h=0;h<C;h++){if(u=[],f.chartstyle&&-1!==f.chartstyle.indexOf("bar")){for(d=_||.8,u.push(1-d/2+(h+.5)*d/C),l=1;l<p[h].length;l++)u.push(u[l-1]+1);f.width=d/C}m&&m.length===C?f.name=m[h]:f.withheaders&&(f.name=c[h]),g&&g.length===C?f.strokecolor=g[h]:f.strokecolor=S.Z.hsv2rgb((h+1)/C*360,.9,.6),b&&b.length===C?f.fillcolor=b[h]:f.fillcolor=S.Z.hsv2rgb((h+1)/C*360,.9,1),v&&v.length===C?f.highlightstrokecolor=v[h]:f.highlightstrokecolor=S.Z.hsv2rgb((h+1)/C*360,.9,1),Z&&Z.length===C?f.highlightfillcolor=Z[h]:f.highlightfillcolor=S.Z.hsv2rgb((h+1)/C*360,.9,.6),f.chartstyle&&-1!==f.chartstyle.indexOf("bar")?y.push(new t.Z.Chart(i,[u,p[h]],f)):y.push(new t.Z.Chart(i,[p[h]],f))}i.unsuspendUpdate()}return y}return f=s.Z.copyAttributes(o,i.options,"chart"),new t.Z.Chart(i,r,f)},t.Z.registerElement("chart",t.Z.createChart),t.Z.Legend=function(t,e,r){var o;if(this.constructor(),o=s.Z.copyAttributes(r,t.options,"legend"),this.board=t,this.coords=new u.Z(i.Z.COORDS_BY_USER,e,this.board),this.myAtts={},this.label_array=o.labelarray||o.labels,this.color_array=o.colorarray||o.colors,this.opacy_array=o.strokeopacity||[1],this.lines=[],this.myAtts.strokewidth=o.strokewidth||5,this.myAtts.straightfirst=!1,this.myAtts.straightlast=!1,this.myAtts.withlabel=!0,this.myAtts.fixed=!0,this.myAtts.frozen=o.frozen||!1,this.style=o.legendstyle||o.style,"vertical"!==this.style)throw new Error("JSXGraph: Unknown legend style: "+this.style);this.drawVerticalLegend(t,o),this.id=this.board.setId(this,"Leg")},t.Z.Legend.prototype=new I.Z,t.Z.Legend.prototype.drawVerticalLegend=function(t,e){var s,r=e.linelength||1,o=(e.rowheight||20)/this.board.unitY,n=function(){return this.setLabelRelativeCoords(this.visProp.label.offset),new u.Z(i.Z.COORDS_BY_USER,[this.point2.X(),this.point2.Y()],this.board)};for(s=0;s<this.label_array.length;s++)this.myAtts.name=this.label_array[s],this.myAtts.strokecolor=this.color_array[s%this.color_array.length],this.myAtts.highlightstrokecolor=this.color_array[s%this.color_array.length],this.myAtts.strokeopacity=this.opacy_array[s%this.opacy_array.length],this.myAtts.highlightstrokeopacity=this.opacy_array[s%this.opacy_array.length],this.myAtts.label={offset:[10,0],strokeColor:this.color_array[s%this.color_array.length],strokeWidth:this.myAtts.strokewidth},this.lines[s]=t.create("line",[[this.coords.usrCoords[1],this.coords.usrCoords[2]-s*o],[this.coords.usrCoords[1]+r,this.coords.usrCoords[2]-s*o]],this.myAtts),this.myAtts.frozen&&this.lines[s].setAttribute({point1:{frozen:!0},point2:{frozen:!0}}),this.lines[s].getLabelAnchor=n,this.lines[s].prepareUpdate().update().updateVisibility(this.lines[s].evalVisProp("visible")).updateRenderer(),this.addChild(this.lines[s])},t.Z.createLegend=function(e,i,r){var o=[0,0];if(!s.Z.exists(i)||2!==i.length)throw new Error("JSXGraph: Legend element needs two numbers as parameters");return o=i,new t.Z.Legend(e,o,r)},t.Z.registerElement("legend",t.Z.createLegend);t.Z.Chart,t.Z.Legend;t.Z.Transformation=function(t,e,s,r){this.elementClass=i.Z.OBJECT_CLASS_OTHER,this.type=i.Z.OBJECT_TYPE_TRANSFORMATION,r?(this.is3D=!0,this.matrix=[[1,0,0,0],[0,1,0,0],[0,0,1,0],[0,0,0,1]]):(this.is3D=!1,this.matrix=[[1,0,0],[0,1,0],[0,0,1]]),this.board=t,this.isNumericMatrix=!1,this.is3D?this.setMatrix3D(s[0],e,s.slice(1)):this.setMatrix(t,e,s),this.methodMap={apply:"apply",applyOnce:"applyOnce",bindTo:"bindTo",bind:"bindTo",melt:"melt",meltTo:"meltTo"}},t.Z.Transformation.prototype={},t.Z.extend(t.Z.Transformation.prototype,{update:function(){return this},setMatrix:function(t,e,i){var r;for(this.isNumericMatrix=!0,r=0;r<i.length;r++)if("number"!=typeof i[r]){this.isNumericMatrix=!1;break}if("translate"===e){if(2!==i.length)throw new Error("JSXGraph: translate transformation needs 2 parameters.");this.evalParam=s.Z.createEvalFunction(t,i,2),this.update=function(){this.matrix[1][0]=this.evalParam(0),this.matrix[2][0]=this.evalParam(1)}}else if("scale"===e){if(2!==i.length)throw new Error("JSXGraph: scale transformation needs 2 parameters.");this.evalParam=s.Z.createEvalFunction(t,i,2),this.update=function(){this.matrix[1][1]=this.evalParam(0),this.matrix[2][2]=this.evalParam(1)}}else if("reflect"===e)i.length<4&&(i[0]=t.select(i[0])),2===i.length&&(i[1]=t.select(i[1])),4===i.length&&(this.evalParam=s.Z.createEvalFunction(t,i,4)),this.update=function(){var t,e,s,r,n,a,h,l;1===i.length?h=i[0].stdform:2===i.length?h=o.Z.crossProduct(i[1].coords.usrCoords,i[0].coords.usrCoords):4===i.length&&(h=o.Z.crossProduct([1,this.evalParam(2),this.evalParam(3)],[1,this.evalParam(0),this.evalParam(1)])),t=h[1],e=h[2],a=(l=[-(s=h[0])*t,-s*e,t*t+e*e])[2],r=l[0]/l[2],n=l[1]/l[2],t=-h[2],e=h[1],this.matrix[1][1]=(t*t-e*e)/a,this.matrix[1][2]=2*t*e/a,this.matrix[2][1]=this.matrix[1][2],this.matrix[2][2]=-this.matrix[1][1],this.matrix[1][0]=r*(1-this.matrix[1][1])-n*this.matrix[1][2],this.matrix[2][0]=n*(1-this.matrix[2][2])-r*this.matrix[2][1]};else if("rotate"===e)3===i.length?this.evalParam=s.Z.createEvalFunction(t,i,3):i.length>0&&i.length<=2&&(this.evalParam=s.Z.createEvalFunction(t,i,1),2!==i.length||s.Z.isArray(i[1])||(i[1]=t.select(i[1]))),this.update=function(){var t,e,r=this.evalParam(0),o=Math.cos(r),n=Math.sin(r);this.matrix[1][1]=o,this.matrix[1][2]=-n,this.matrix[2][1]=n,this.matrix[2][2]=o,i.length>1&&(3===i.length?(t=this.evalParam(1),e=this.evalParam(2)):s.Z.isArray(i[1])?(t=i[1][0],e=i[1][1]):(t=i[1].X(),e=i[1].Y()),this.matrix[1][0]=t*(1-o)+e*n,this.matrix[2][0]=e*(1-o)-t*n)};else if("shear"===e){if(2!==i.length)throw new Error("JSXGraph: shear transformation needs 2 parameters.");this.evalParam=s.Z.createEvalFunction(t,i,2),this.update=function(){this.matrix[1][2]=this.evalParam(0),this.matrix[2][1]=this.evalParam(1)}}else if("generic"===e){if(9!==i.length)throw new Error("JSXGraph: generic transformation needs 9 parameters.");this.evalParam=s.Z.createEvalFunction(t,i,9),this.update=function(){this.matrix[0][0]=this.evalParam(0),this.matrix[0][1]=this.evalParam(1),this.matrix[0][2]=this.evalParam(2),this.matrix[1][0]=this.evalParam(3),this.matrix[1][1]=this.evalParam(4),this.matrix[1][2]=this.evalParam(5),this.matrix[2][0]=this.evalParam(6),this.matrix[2][1]=this.evalParam(7),this.matrix[2][2]=this.evalParam(8)}}else if("matrix"===e){if(1!==i.length)throw new Error("JSXGraph: transformation of type 'matrix' needs 1 parameter.");this.evalParam=i[0].slice(),this.update=function(){var t,e;for(t=0;t<3;t++)for(e=0;e<3;e++)this.matrix[t][e]=s.Z.evaluate(this.evalParam[t][e])}}},setMatrix3D:function(t,e,i){var r,n=t.board;for(this.isNumericMatrix=!0,r=0;r<i.length;r++)if("number"!=typeof i[r]){this.isNumericMatrix=!1;break}if("translate"===e){if(3!==i.length)throw new Error("JSXGraph: 3D translate transformation needs 3 parameters.");this.evalParam=s.Z.createEvalFunction(n,i,3),this.update=function(){this.matrix[1][0]=this.evalParam(0),this.matrix[2][0]=this.evalParam(1),this.matrix[3][0]=this.evalParam(2)}}else if("scale"===e){if(3!==i.length&&4!==i.length)throw new Error("JSXGraph: 3D scale transformation needs either 3 or 4 parameters.");this.evalParam=s.Z.createEvalFunction(n,i,3),this.update=function(){var t=this.evalParam(0),e=this.evalParam(1),i=this.evalParam(2);this.matrix[1][1]=t,this.matrix[2][2]=e,this.matrix[3][3]=i}}else if("rotateX"===e)i.splice(1,0,[1,0,0]),this.setMatrix3D(t,"rotate",i);else if("rotateY"===e)i.splice(1,0,[0,1,0]),this.setMatrix3D(t,"rotate",i);else if("rotateZ"===e)i.splice(1,0,[0,0,1]),this.setMatrix3D(t,"rotate",i);else if("rotate"===e){if(i.length<2)throw new Error("JSXGraph: 3D rotate transformation needs 2 or 3 parameters.");3!==i.length||s.Z.isFunction(i[2])||s.Z.isArray(i[2])?this.evalParam=s.Z.createEvalFunction(n,i,i.length):(this.evalParam=s.Z.createEvalFunction(n,i,2),i[2]=t.select(i[2])),this.update=function(){var t,e,r,n=this.evalParam(0),a=this.evalParam(1),h=[1,0,0,0],l=Math.cos(n),c=Math.sin(n),d=[[1,0,0,0],[0,1,0,0],[0,0,1,0],[0,0,0,1]],u=[[1,0,0,0],[0,1,0,0],[0,0,1,0],[0,0,0,1]],p=o.Z.norm(a);3===a.length?(t=a[0]/p,e=a[1]/p,r=a[2]/p):(t=a[1]/p,e=a[2]/p,r=a[3]/p),3===i.length&&(3===(h=3===i.length&&s.Z.exists(i[2].is3D)?i[2].coords.slice():this.evalParam(2)).length&&h.unshift(1),d[1][0]=-h[1],d[2][0]=-h[2],d[3][0]=-h[3],u[1][0]=h[1],u[2][0]=h[2],u[3][0]=h[3]),this.matrix=[[1,0,0,0],[0,t*t*(1-l)+l,t*e*(1-l)-r*c,t*r*(1-l)+e*c],[0,e*t*(1-l)+r*c,e*e*(1-l)+l,e*r*(1-l)-t*c],[0,r*t*(1-l)-e*c,r*e*(1-l)+t*c,r*r*(1-l)+l]],this.matrix=o.Z.matMatMult(this.matrix,d),this.matrix=o.Z.matMatMult(u,this.matrix)}}},apply:function(t,e){var i;return this.update(),i=this.is3D?t.coords:s.Z.exists(e)?t.initialCoords.usrCoords:t.coords.usrCoords,o.Z.matVecMult(this.matrix,i)},applyOnce:function(t){var e,r,n;for(s.Z.isArray(t)||(t=[t]),r=t.length,n=0;n<r;n++)this.update(),this.is3D?t[n].coords=o.Z.matVecMult(this.matrix,t[n].coords):(e=o.Z.matVecMult(this.matrix,t[n].coords.usrCoords),t[n].coords.setCoordinates(i.Z.COORDS_BY_USER,e))},bindTo:function(t){var e,i;if(s.Z.isArray(t))for(i=t.length,e=0;e<i;e++)t[e].transformations.push(this);else t.transformations.push(this)},meltTo:function(t){var e,i,r;if(s.Z.isArray(t))for(e=0;e<t.length;e++)this.meltTo(t[e]);else(i=t.transformations).length>0&&i[i.length-1].isNumericMatrix&&this.isNumericMatrix?i[i.length-1].melt(this):(r=this.clone(),i.push(r))},clone:function(){var e=null;return this.isNumericMatrix&&((e=new t.Z.Transformation(this.board,"none",[])).matrix=this.matrix.slice()),e},setProperty:function(e){t.Z.deprecated("Transformation.setProperty()","Transformation.setAttribute()")},setAttribute:function(t){},melt:function(t){var e;return this.update(),t.update(),e=o.Z.matMatMult(t.matrix,this.matrix),this.update=function(){this.matrix=e},this},getParents:function(){var t=[[].concat.apply([],this.matrix)];return 0!==this.parents.length&&(t=this.parents),t}}),t.Z.createTransform=function(e,i,s){return new t.Z.Transformation(e,s.type,i)},t.Z.registerElement("transform",t.Z.createTransform),t.Z.createTransform3D=function(e,i,s){return new t.Z.Transformation(e,s.type,i,!0)},t.Z.registerElement("transform3d",t.Z.createTransform3D);t.Z.Transformation;t.Z.Turtle=function(t,e,r){var o,n,a;return this.constructor(t,r,i.Z.OBJECT_TYPE_TURTLE,i.Z.OBJECT_CLASS_OTHER),this.turtleIsHidden=!1,this.board=t,this.visProp.curveType="plot",this._attributes=s.Z.copyAttributes(this.visProp,t.options,"turtle"),delete this._attributes.id,o=0,n=0,a=90,0!==e.length&&(3===e.length?(o=e[0],n=e[1],a=e[2]):2===e.length?s.Z.isArray(e[0])?(o=e[0][0],n=e[0][1],a=e[1]):(o=e[0],n=e[1]):(o=e[0][0],n=e[0][1])),this.init(o,n,a),this.methodMap=s.Z.deepCopy(this.methodMap,{forward:"forward",fd:"forward",back:"back",bk:"back",right:"right",rt:"right",left:"left",lt:"left",penUp:"penUp",pu:"penUp",up:"penUp",penDown:"penDown",pd:"penDown",down:"penDown",clearScreen:"clearScreen",cs:"clearScreen",clean:"clean",setPos:"setPos",home:"home",hideTurtle:"hideTurtle",ht:"hideTurtle",hide:"hideTurtle",showTurtle:"showTurtle",st:"showTurtle",show:"showTurtle",penSize:"setPenSize",setPenSize:"setPenSize",penColor:"setPenColor",setPenColor:"setPenColor",highlightPenColor:"setHighlightPenColor",setHighlightPenColor:"setHighlightPenColor",getPenColor:"getPenColor",Color:"getPenColor",getHighlightPenColor:"getHighlightPenColor",HighlightColor:"getHighlightPenColor",getPenSize:"getPenSize",Size:"getPenSize",pushTurtle:"pushTurtle",push:"pushTurtle",popTurtle:"popTurtle",pop:"popTurtle",lookTo:"lookTo",pos:"pos",Pos:"pos",moveTo:"moveTo",X:"X",Y:"Y"}),this},t.Z.Turtle.prototype=new I.Z,t.Z.extend(t.Z.Turtle.prototype,{init:function(t,e,i){var s={fixed:!0,name:"",visible:!1,withLabel:!1};this.arrowLen=20/o.Z.hypot(this.board.unitX,this.board.unitY),this.pos=[t,e],this.isPenDown=!0,this.dir=90,this.stack=[],this.objects=[],this.curve=this.board.create("curve",[[this.pos[0]],[this.pos[1]]],this._attributes),this.objects.push(this.curve),this.turtle=this.board.create("point",this.pos,s),this.objects.push(this.turtle),this.turtle2=this.board.create("point",[this.pos[0],this.pos[1]+this.arrowLen],s),this.objects.push(this.turtle2),this.visProp.arrow.lastArrow=!0,this.visProp.arrow.straightFirst=!1,this.visProp.arrow.straightLast=!1,this.arrow=this.board.create("line",[this.turtle,this.turtle2],this.visProp.arrow),this.objects.push(this.arrow),this.subs={arrow:this.arrow},this.inherits.push(this.arrow),this.right(90-i),this.board.update()},forward:function(t){if(0===t)return this;var e,i=t*Math.cos(this.dir*Math.PI/180),s=t*Math.sin(this.dir*Math.PI/180);return this.turtleIsHidden||((e=this.board.create("transform",[i,s],{type:"translate"})).applyOnce(this.turtle),e.applyOnce(this.turtle2)),this.isPenDown&&this.curve.dataX.length>=8192&&(this.curve=this.board.create("curve",[[this.pos[0]],[this.pos[1]]],this._attributes),this.objects.push(this.curve)),this.pos[0]+=i,this.pos[1]+=s,this.isPenDown&&(this.curve.dataX.push(this.pos[0]),this.curve.dataY.push(this.pos[1])),this.board.update(),this},back:function(t){return this.forward(-t)},right:function(t){(this.dir-=t,this.dir%=360,this.turtleIsHidden)||this.board.create("transform",[-t*Math.PI/180,this.turtle],{type:"rotate"}).applyOnce(this.turtle2);return this.board.update(),this},left:function(t){return this.right(-t)},penUp:function(){return this.isPenDown=!1,this},penDown:function(){return this.isPenDown=!0,this.curve=this.board.create("curve",[[this.pos[0]],[this.pos[1]]],this._attributes),this.objects.push(this.curve),this},clean:function(){var t,e;for(t=0;t<this.objects.length;t++)(e=this.objects[t]).type===i.Z.OBJECT_TYPE_CURVE&&(this.board.removeObject(e),this.objects.splice(t,1));return this.curve=this.board.create("curve",[[this.pos[0]],[this.pos[1]]],this._attributes),this.objects.push(this.curve),this.board.update(),this},clearScreen:function(){var t,e,i=this.objects.length;for(t=0;t<i;t++)e=this.objects[t],this.board.removeObject(e);return this.init(0,0,90),this},setPos:function(t,e){return s.Z.isArray(t)?this.pos=t:this.pos=[t,e],this.turtleIsHidden||(this.turtle.setPositionDirectly(i.Z.COORDS_BY_USER,[t,e]),this.turtle2.setPositionDirectly(i.Z.COORDS_BY_USER,[t,e+this.arrowLen]),this.board.create("transform",[-(this.dir-90)*Math.PI/180,this.turtle],{type:"rotate"}).applyOnce(this.turtle2)),this.curve=this.board.create("curve",[[this.pos[0]],[this.pos[1]]],this._attributes),this.objects.push(this.curve),this.board.update(),this},setPenSize:function(t){return this.curve=this.board.create("curve",[[this.pos[0]],[this.pos[1]]],this.copyAttr("strokeWidth",t)),this.objects.push(this.curve),this},setPenColor:function(t){return this.curve=this.board.create("curve",[[this.pos[0]],[this.pos[1]]],this.copyAttr("strokeColor",t)),this.objects.push(this.curve),this},getPenAttribute:function(t){var e,i=this.objects.length;return e=4===i?0:i-1,this.objects[e].evalVisProp(t)},getPenSize:function(){return this.getPenAttribute("strokewidth")},getPenColor:function(){return this.getPenAttribute("strokecolor")},getHighlightPenColor:function(){return this.getPenAttribute("highlightstrokecolor")},setHighlightPenColor:function(t){return this.curve=this.board.create("curve",[[this.pos[0]],[this.pos[1]]],this.copyAttr("highlightStrokeColor",t)),this.objects.push(this.curve),this},setAttribute:function(t){var e,r,o,n=this.objects.length;for(e=0;e<n;e++)(r=this.objects[e]).type===i.Z.OBJECT_TYPE_CURVE&&r.setAttribute(t);return o=this.visProp.id,this.visProp=s.Z.deepCopy(this.curve.visProp),this.visProp.id=o,this._attributes=s.Z.deepCopy(this.visProp),delete this._attributes.id,this},copyAttr:function(t,e){return this._attributes[t.toLowerCase()]=e,this._attributes},showTurtle:function(){return this.turtleIsHidden=!1,this.arrow.setAttribute({visible:!0}),this.visProp.arrow.visible=!1,this.setPos(this.pos[0],this.pos[1]),this.board.update(),this},hideTurtle:function(){return this.turtleIsHidden=!0,this.arrow.setAttribute({visible:!1}),this.visProp.arrow.visible=!1,this.board.update(),this},home:function(){return this.pos=[0,0],this.setPos(this.pos[0],this.pos[1]),this},pushTurtle:function(){return this.stack.push([this.pos[0],this.pos[1],this.dir]),this},popTurtle:function(){var t=this.stack.pop();return this.pos[0]=t[0],this.pos[1]=t[1],this.dir=t[2],this.setPos(this.pos[0],this.pos[1]),this},lookTo:function(t){var e,i,r,o,n;return s.Z.isArray(t)?(e=this.pos[0],i=this.pos[1],r=t[0],o=t[1],n=Math.atan2(o-i,r-e),this.right(this.dir-180*n/Math.PI)):s.Z.isNumber(t)&&this.right(this.dir-t),this},moveTo:function(t){var e,i,r;return s.Z.isArray(t)&&(e=t[0]-this.pos[0],i=t[1]-this.pos[1],this.turtleIsHidden||((r=this.board.create("transform",[e,i],{type:"translate"})).applyOnce(this.turtle),r.applyOnce(this.turtle2)),this.isPenDown&&this.curve.dataX.length>=8192&&(this.curve=this.board.create("curve",[[this.pos[0]],[this.pos[1]]],this._attributes),this.objects.push(this.curve)),this.pos[0]=t[0],this.pos[1]=t[1],this.isPenDown&&(this.curve.dataX.push(this.pos[0]),this.curve.dataY.push(this.pos[1])),this.board.update()),this},fd:function(t){return this.forward(t)},bk:function(t){return this.back(t)},lt:function(t){return this.left(t)},rt:function(t){return this.right(t)},pu:function(){return this.penUp()},pd:function(){return this.penDown()},ht:function(){return this.hideTurtle()},st:function(){return this.showTurtle()},cs:function(){return this.clearScreen()},push:function(){return this.pushTurtle()},pop:function(){return this.popTurtle()},evalAt:function(t,e){var s,r,o,n,a=this.objects.length;for(s=0,r=0;s<a;s++)if((o=this.objects[s]).elementClass===i.Z.OBJECT_CLASS_CURVE){if(r<=t&&t<r+o.numberPoints)return n=t-r,o[e](n);r+=o.numberPoints}return this[e]()},X:function(t){return s.Z.exists(t)?this.evalAt(t,"X"):this.pos[0]},Y:function(t){return s.Z.exists(t)?this.evalAt(t,"Y"):this.pos[1]},Z:function(t){return 1},minX:function(){return 0},maxX:function(){var t,e=this.objects.length,s=0;for(t=0;t<e;t++)this.objects[t].elementClass===i.Z.OBJECT_CLASS_CURVE&&(s+=this.objects[t].numberPoints);return s},hasPoint:function(t,e){var s,r;for(s=0;s<this.objects.length;s++)if((r=this.objects[s]).type===i.Z.OBJECT_TYPE_CURVE&&r.hasPoint(t,e))return!0;return!1}}),t.Z.createTurtle=function(e,i,r){var o;return i=i||[],o=s.Z.copyAttributes(r,e.options,"turtle"),new t.Z.Turtle(e,i,o)},t.Z.registerElement("turtle",t.Z.createTurtle);t.Z.Turtle;t.Z.Ticks=function(t,e,r){if(this.constructor(t.board,r,i.Z.OBJECT_TYPE_TICKS,i.Z.OBJECT_CLASS_OTHER),this.line=t,this.board=this.line.board,this.fixedTicks=null,this.equidistant=!1,this.labelsData=[],s.Z.isFunction(e))throw this.ticksFunction=e,new Error("Function arguments are no longer supported.");s.Z.isArray(e)?this.fixedTicks=e:this.equidistant=!0,this.ticks=[],this.labels=[],this.labelCounter=0,this.id=this.line.addTicks(this),this.elType="ticks",this.inherits.push(this.labels),this.board.setId(this,"Ti")},t.Z.Ticks.prototype=new I.Z,t.Z.extend(t.Z.Ticks.prototype,{hasPoint:function(t,e){var r,n,a,h,l=this.ticks&&this.ticks.length||0;if(!this.line.evalVisProp("scalable")||this.line.elementClass===i.Z.OBJECT_CLASS_CURVE)return!1;if(s.Z.isObject(this.evalVisProp("precision"))?(h=this.board._inputDevice,a=this.evalVisProp("precision."+h)):a=this.board.options.precision.hasPoint,a+=.5*this.evalVisProp("strokewidth"),0!==this.line.stdform[1]&&0!==this.line.stdform[2]&&this.line.type!==i.Z.OBJECT_TYPE_AXIS)return!1;for(r=0;r<l;r++)if((n=this.ticks[r])[2]&&!(0===this.line.stdform[1]&&Math.abs(n[0][0]-this.line.point1.coords.scrCoords[1])<o.Z.eps||0===this.line.stdform[2]&&Math.abs(n[1][0]-this.line.point1.coords.scrCoords[2])<o.Z.eps)&&(Math.abs(n[0][0]-n[0][1])>=1||Math.abs(n[1][0]-n[1][1])>=1))if(0===this.line.stdform[1]){if(Math.abs(e-this.line.point1.coords.scrCoords[2])<2*a&&n[0][0]-a<t&&t<n[0][1]+a)return!0}else if(0===this.line.stdform[2]&&Math.abs(t-this.line.point1.coords.scrCoords[1])<2*a&&n[1][0]-a<e&&e<n[1][1]+a)return!0;return!1},setPositionDirectly:function(t,e,s){var r,n,a=new u.Z(t,e,this.board),h=new u.Z(t,s,this.board),l=this.board.getBoundingBox();return this.line.type===i.Z.OBJECT_TYPE_AXIS&&this.line.evalVisProp("scalable")?(Math.abs(this.line.stdform[1])<o.Z.eps&&Math.abs(a.usrCoords[1]*h.usrCoords[1])>o.Z.eps?(r=h.usrCoords[1]/a.usrCoords[1],l[0]*=r,l[2]*=r,this.board.setBoundingBox(l,this.board.keepaspectratio,"update")):Math.abs(this.line.stdform[2])<o.Z.eps&&Math.abs(a.usrCoords[2]*h.usrCoords[2])>o.Z.eps&&(n=h.usrCoords[2]/a.usrCoords[2],l[3]*=n,l[1]*=n,this.board.setBoundingBox(l,this.board.keepaspectratio,"update")),this):this},calculateTicksCoordinates:function(){var t,e,s,r;if(!(this.line.elementClass===i.Z.OBJECT_CLASS_LINE&&(this.setTicksSizeVariables(),Math.abs(this.dx)<o.Z.eps&&Math.abs(this.dy)<o.Z.eps)))return t=this.getZeroCoordinates(),e=this.line.elementClass===i.Z.OBJECT_CLASS_LINE?this.getLowerAndUpperBounds(t,"ticksdistance"):{lower:this.line.minX(),upper:this.line.maxX(),a1:0,a2:0,m1:0,m2:0},"polar"===this.evalVisProp("type")&&(r=this.board.getBoundingBox(),s=Math.max(o.Z.hypot(r[0],r[1]),o.Z.hypot(r[2],r[3])),e.upper=s),this.ticks=[],this.labelsData=[],this.equidistant?this.generateEquidistantTicks(t,e):this.generateFixedTicks(t,e),this},setTicksSizeVariables:function(t){var e,i,r,n,a=.5*this.evalVisProp("majorheight"),h=.5*this.evalVisProp("minorheight");s.Z.exists(t)?(i=this.line.minX(),r=this.line.maxX(),(n=this.line.points.length)<2?(this.dxMaj=0,this.dyMaj=0):o.Z.relDif(t,i)<o.Z.eps?(this.dxMaj=this.line.points[0].usrCoords[2]-this.line.points[1].usrCoords[2],this.dyMaj=this.line.points[1].usrCoords[1]-this.line.points[0].usrCoords[1]):o.Z.relDif(t,r)<o.Z.eps?(this.dxMaj=this.line.points[n-2].usrCoords[2]-this.line.points[n-1].usrCoords[2],this.dyMaj=this.line.points[n-1].usrCoords[1]-this.line.points[n-2].usrCoords[1]):(this.dxMaj=-d.Z.D(this.line.Y)(t),this.dyMaj=d.Z.D(this.line.X)(t))):(this.dxMaj=this.line.stdform[1],this.dyMaj=this.line.stdform[2]),this.dxMin=this.dxMaj,this.dyMin=this.dyMaj,this.dx=this.dxMaj,this.dy=this.dyMaj,e=o.Z.hypot(this.dxMaj*this.board.unitX,this.dyMaj*this.board.unitY),this.dxMaj*=a/e*this.board.unitX,this.dyMaj*=a/e*this.board.unitY,this.dxMin*=h/e*this.board.unitX,this.dyMin*=h/e*this.board.unitY,this.minStyle=this.evalVisProp("minorheight")<0?"infinite":"finite",this.majStyle=this.evalVisProp("majorheight")<0?"infinite":"finite"},getZeroCoordinates:function(){var t,e,r,o,n,a,l,c,d=this.evalVisProp("anchor");return this.line.elementClass===i.Z.OBJECT_CLASS_LINE?this.line.type===i.Z.OBJECT_TYPE_AXIS?h.Z.projectPointToLine({coords:{usrCoords:[1,0,0]}},this.line,this.board):(r=this.line.point1.coords.usrCoords[0],t=this.line.point1.coords.usrCoords[1],e=this.line.point1.coords.usrCoords[2],a=this.line.point2.coords.usrCoords[0],o=this.line.point2.coords.usrCoords[1],n=this.line.point2.coords.usrCoords[2],"right"===d?this.line.point2.coords:"middle"===d?new u.Z(i.Z.COORDS_BY_USER,[.5*(r+a),.5*(t+o),.5*(e+n)],this.board):s.Z.isNumber(d)?new u.Z(i.Z.COORDS_BY_USER,[r+(a-r)*d,t+(o-t)*d,e+(n-e)*d],this.board):this.line.point1.coords):(l=this.line.minX(),c=this.line.maxX(),"right"===d?c:"middle"===d?.5*(l+c):s.Z.isNumber(d)?l*(1-d)+c*d:l)},getLowerAndUpperBounds:function(t,e){var r,n,a,l,c,d,p,f,_,m,g,b,v,Z,C,y=10*o.Z.eps,P=this.line.evalVisProp("straightfirst"),E=this.line.evalVisProp("straightlast"),x=this.evalVisProp("includeboundaries");return this.line.elementClass===i.Z.OBJECT_CLASS_CURVE?{lower:this.line.minX(),upper:this.line.maxX()}:(c=new u.Z(i.Z.COORDS_BY_USER,this.line.point1.coords.usrCoords,this.board),d=new u.Z(i.Z.COORDS_BY_USER,this.line.point2.coords.usrCoords,this.board),p=Math.abs(c.usrCoords[0])>=o.Z.eps&&c.scrCoords[1]>=0&&c.scrCoords[1]<=this.board.canvasWidth&&c.scrCoords[2]>=0&&c.scrCoords[2]<=this.board.canvasHeight,f=Math.abs(d.usrCoords[0])>=o.Z.eps&&d.scrCoords[1]>=0&&d.scrCoords[1]<=this.board.canvasWidth&&d.scrCoords[2]>=0&&d.scrCoords[2]<=this.board.canvasHeight,s.Z.exists(e)&&"ticksdistance"===e?(h.Z.calcStraight(this.line,c,d,0),Z=this.getDistanceFromZero(t,c),C=this.getDistanceFromZero(t,d),h.Z.calcStraight(this.line,c,d,this.line.evalVisProp("margin")),Z=this.getDistanceFromZero(t,c)-Z,C=this.getDistanceFromZero(t,d).m2):h.Z.calcLineDelimitingPoints(this.line,c,d),a=this.line.evalVisProp("firstarrow"),l=this.line.evalVisProp("lastarrow"),b=this.getDistanceFromZero(t,c),v=this.getDistanceFromZero(t,d),(a||l)&&(g=this.board.renderer.getArrowHeadData(this.line,this.line.evalVisProp("strokewidth"),""),this.board.renderer.getPositionArrowHead(this.line,c,d,g)),(_=this.getDistanceFromZero(t,c))<(m=this.getDistanceFromZero(t,d))?(r=_,n=m,P||!p||x||(r+=y),E||!f||x||(n-=y)):m<_?(r=m,n=_,E||!f||x||(r+=y),P||!p||x||(n-=y)):(r=0,n=0),{lower:r,upper:n,a1:b=_-b,a2:v=_-v,m1:Z,m2:C})},getDistanceFromZero:function(t,e){var s,r,n,a,h;return s=this.line.point1.coords,r=this.line.point2.coords,h=t.distance(i.Z.COORDS_BY_USER,e),n=[r.usrCoords[0]-s.usrCoords[0],r.usrCoords[1]-s.usrCoords[1],r.usrCoords[2]-s.usrCoords[2]],a=[e.usrCoords[0]-t.usrCoords[0],e.usrCoords[1]-t.usrCoords[1],e.usrCoords[2]-t.usrCoords[2]],o.Z.innerProduct(n,a,3)<0&&(h*=-1),h},generateEquidistantTicks:function(e,s){var r,n,a,h=o.Z.eps,l=this.evalVisProp("minorticks");if(a=this.getDistanceMajorTicks(),this.line.elementClass===i.Z.OBJECT_CLASS_LINE&&(n=this.getXandYdeltas()),a*=this.evalVisProp("scale"),a/=l+1,this.ticksDelta=a,!(a<o.Z.eps))if(Math.abs(s.upper-s.lower)>2048*a)t.Z.warn("JSXGraph ticks: too many ticks (>2048). Please increase ticksDistance.");else{for(r=0,this.evalVisProp("drawzero")||(r=a),r<s.lower&&(r=Math.floor((s.lower-h)/a)*a);r<=s.upper+h&&(r>=s.lower-h&&this.processTickPosition(e,r,a,n),r+=a,!(s.upper-r>1e4*a)););for((r=-a)>s.upper&&(r=Math.ceil((s.upper+h)/-a)*-a);r>=s.lower-h&&(r<=s.upper+h&&this.processTickPosition(e,r,a,n),!((r-=a)-s.lower>1e4*a)););}},getDistanceMajorTicks:function(){var t,e,i,s,r,n,a,h;return this.evalVisProp("insertticks")?(n=((i=this.getLowerAndUpperBounds(this.getZeroCoordinates(),"")).upper-i.lower)/6/(r=this.evalVisProp("scale")),a=this.evalVisProp("minticksdistance")/r,h=this.evalVisProp("minorticks"),(s=this.getXandYdeltas()).x*=this.board.unitX,s.y*=this.board.unitY,a/=o.Z.hypot(s.x,s.y),a*=h+1,2*(t=Math.pow(10,Math.floor(Math.log(a)/Math.LN10)))>=a?t*=2:5*t>=a&&(t*=5),5*(e=Math.pow(10,Math.floor(Math.log(n)/Math.LN10)))<n?e*=5:2*e<n&&(e*=2),t=Math.max(t,e)):this.evalVisProp("ticksdistance")},processTickPosition:function(t,e,s,r){var o,n,a,h,l,c,d=this.evalVisProp("ticksperlabel"),p=null;this.line.elementClass===i.Z.OBJECT_CLASS_LINE?(o=t.usrCoords[1]+e*r.x,n=t.usrCoords[2]+e*r.y):(o=this.line.X(t+e),n=this.line.Y(t+e)),a=new u.Z(i.Z.COORDS_BY_USER,[o,n],this.board),this.line.elementClass===i.Z.OBJECT_CLASS_CURVE&&(p=t+e,this.setTicksSizeVariables(p)),l=this.evalVisProp("minorticks"),a.major=Math.round(e/s)%(l+1)==0,d||(d=l+1),c=Math.round(e/s)%d==0,3===(h=this.createTickPath(a,a.major)).length&&(this.ticks.push(h),c&&this.evalVisProp("drawlabels")?this.labelsData.push(this.generateLabelData(this.generateLabelText(a,t,p),a,this.ticks.length)):this.labelsData.push(null))},generateFixedTicks:function(t,e){var r,n,a,h,l,c,d,p,f=o.Z.eps,_=s.Z.isArray(this.visProp.labels),m=this.evalVisProp("drawlabels");for(this.line.elementClass===i.Z.OBJECT_CLASS_LINE&&(p=this.getXandYdeltas()),a=0;a<this.fixedTicks.length;a++)this.line.elementClass===i.Z.OBJECT_CLASS_LINE?(d=this.fixedTicks[a],l=t.usrCoords[1]+d*p.x,c=t.usrCoords[2]+d*p.y):(d=t+this.fixedTicks[a],l=this.line.X(d),c=this.line.Y(d)),r=new u.Z(i.Z.COORDS_BY_USER,[l,c],this.board),this.line.elementClass===i.Z.OBJECT_CLASS_CURVE&&this.setTicksSizeVariables(d),3===(h=this.createTickPath(r,!0)).length&&d>=e.lower-f&&d<=e.upper+f&&(this.ticks.push(h),m&&(_||s.Z.exists(this.visProp.labels[a]))?(n=_?this.evalVisProp("labels."+a):d,this.labelsData.push(this.generateLabelData(this.generateLabelText(r,t,n),r,a))):this.labelsData.push(null))},getXandYdeltas:function(){var t,e,i=this.line.point1.Dist(this.line.point2);return t=this.line.point1.coords.usrCoords,{x:((e=this.line.point2.coords.usrCoords)[1]-t[1])/i,y:(e[2]-t[2])/i}},_isInsideCanvas:function(t,e,i){var s=this.board.canvasWidth,r=this.board.canvasHeight;return void 0===i&&(i=0),t[0]>=i&&t[0]<=s-i&&e[0]>=i&&e[0]<=r-i||t[1]>=i&&t[1]<=s-i&&e[1]>=i&&e[1]<=r-i},createTickPath:function(t,e){var i,s,r,n,a,l,c,d,u,p,f,_,m,g,b,v,Z,C=[-2e6,-2e6],y=[-2e6,-2e6];if(i=t.scrCoords,e?(n=this.dxMaj,a=this.dyMaj,u=this.majStyle,b=this.evalVisProp("majortickendings.0")>0,v=this.evalVisProp("majortickendings.1")>0):(n=this.dxMin,a=this.dyMin,u=this.minStyle,b=this.evalVisProp("tickendings.0")>0,v=this.evalVisProp("tickendings.1")>0),s=[-a*i[1]-n*i[2],a,n],e&&"polar"===this.evalVisProp("type")){if(_=this.board.getBoundingBox(),g=(m=2*Math.PI)/180,i=t.usrCoords,(f=o.Z.hypot(i[1],i[2]))<Math.max(o.Z.hypot(_[0],_[1]),o.Z.hypot(_[2],_[3]))){for(C=[],y=[],p=0;p<=m;p+=g)C.push(this.board.origin.scrCoords[1]+f*Math.cos(p)*this.board.unitX),y.push(this.board.origin.scrCoords[2]+f*Math.sin(p)*this.board.unitY);return[C,y,e]}}else{if("infinite"===u){if(this.evalVisProp("ignoreinfinitetickendings")&&(b=v=!0),r=h.Z.meetLineBoard(s,this.board),b&&v)C[0]=r[0].scrCoords[1],C[1]=r[1].scrCoords[1],y[0]=r[0].scrCoords[2],y[1]=r[1].scrCoords[2];else if((Z=[o.Z.innerProduct(r[0].usrCoords.slice(1,3),this.line.stdform.slice(1,3))+this.line.stdform[0],o.Z.innerProduct(r[1].usrCoords.slice(1,3),this.line.stdform.slice(1,3))+this.line.stdform[0]])[0]<Z[1]&&(r.reverse(),Z.reverse()),b){if(Z[0]<0)return[];Z[1]<0?(C[0]=r[0].scrCoords[1],y[0]=r[0].scrCoords[2],C[1]=i[1],y[1]=i[2]):(C[0]=r[0].scrCoords[1],y[0]=r[0].scrCoords[2],C[1]=r[1].scrCoords[1],y[1]=r[1].scrCoords[2])}else if(v){if(Z[1]>0)return[];Z[0]>0?(C[0]=i[1],y[0]=i[2],C[1]=r[1].scrCoords[1],y[1]=r[1].scrCoords[2]):(C[0]=r[0].scrCoords[1],y[0]=r[0].scrCoords[2],C[1]=r[1].scrCoords[1],y[1]=r[1].scrCoords[2])}}else d=">"===this.evalVisProp("face")?Math.PI/4:"<"===this.evalVisProp("face")?-Math.PI/4:0,l=Math.cos(d)*n-Math.sin(d)*a,c=Math.sin(d)*n+Math.cos(d)*a,C[0]=i[1]+l*b,y[0]=i[2]-c*b,C[1]=i[1],y[1]=i[2],d=-d,l=Math.cos(d)*n-Math.sin(d)*a,c=Math.sin(d)*n+Math.cos(d)*a,C[2]=i[1]-l*v,y[2]=i[2]+c*v;if(this._isInsideCanvas(C,y))return[C,y,e]}return[]},formatLabelText:function(t){var e,i,r=this.evalVisProp("label.usemathjax"),o=this.evalVisProp("label.usekatex"),n=this.evalVisProp("scalesymbol");return s.Z.isNumber(t)?(this.evalVisProp("label.tofraction")?e=r?"\\("+s.Z.toFraction(t,!0)+"\\)":s.Z.toFraction(t,o):(i=this.evalVisProp("digits"),this.useLocale()?e=this.formatNumberLocale(t,i):((e=(Math.round(1e11*t)/1e11).toString()).length>this.evalVisProp("maxlabellength")||-1!==e.indexOf("e"))&&(3!==this.evalVisProp("precision")&&3===i&&(i=this.evalVisProp("precision")),e=t.toExponential(i).toString())),this.evalVisProp("beautifulscientificticklabels")&&(e=this.beautifyScientificNotationLabel(e)),e.indexOf(".")>-1&&-1===e.indexOf("e")&&(e=(e=e.replace(/0+$/,"")).replace(/\.$/,""))):e=t.toString(),n.length>0&&("1"===e?e=n:"-1"===e?e="-"+n:"0"!==e&&(e+=n)),this.evalVisProp("useunicodeminus")&&(e=e.replace(/-/g,"−")),e},beautifyScientificNotationLabel:function(t){return-1===t.indexOf("e")?t:(parseFloat(t.substring(0,t.indexOf("e")))+t.substring(t.indexOf("e"))).replace(/e(.*)$/g,(function(t,e){var i="•10";return i+=e.replace(/-/g,"⁻").replace(/\+/g,"").replace(/0/g,"⁰").replace(/1/g,"¹").replace(/2/g,"²").replace(/3/g,"³").replace(/4/g,"⁴").replace(/5/g,"⁵").replace(/6/g,"⁶").replace(/7/g,"⁷").replace(/8/g,"⁸").replace(/9/g,"⁹")}))},generateLabelText:function(t,e,i){var r;if(!s.Z.exists(i)){if(r=this.getDistanceFromZero(e,t),Math.abs(r)<o.Z.eps)return"0";i=r/this.evalVisProp("scale")}return this.formatLabelText(i)},generateLabelData:function(t,e,i){var s,r,o,n;return n=this.evalVisProp("label.fontsize"),s=[e.scrCoords[1],e.scrCoords[1]],r=[e.scrCoords[2],e.scrCoords[2]],o=void 0===n?12:n,o*=.5,this._isInsideCanvas(s,r,o)?(s=this.evalVisProp("label.offset")[0],r=this.evalVisProp("label.offset")[1],{x:e.usrCoords[1]+s/this.board.unitX,y:e.usrCoords[2]+r/this.board.unitY,t:t,i:i}):null},update:function(){return this.needsUpdate&&0!==this.board.canvasWidth&&0!==this.board.canvasHeight&&this.calculateTicksCoordinates(),this},updateRenderer:function(){return this.needsUpdate?(this.visPropCalc.visible&&this.board.renderer.updateTicks(this),this.updateRendererLabels(),this.setDisplayRendNode(),this.needsUpdate=!1,this):this},updateRendererLabels:function(){var e,i,r,o,n,a,h,l;for(r=this.labelsData.length,o=this.labels.length,e=0,i=0;e<r;e++)null!==this.labelsData[e]&&(h=this.labelsData[e],i<o?((a=this.labels[i]).setText(h.t),a.setCoords(h.x,h.y),i++):(this.labelCounter+=1,(n=s.Z.deepCopy(this.visProp.label)).isLabel=!0,n.priv=this.visProp.priv,n.id=this.id+h.i+"Label"+this.labelCounter,a=t.Z.createText(this.board,[h.x,h.y,h.t],n),this.addChild(a),a.setParents(this),a.isDraggable=!1,a.dump=!1,this.labels.push(a)),"inherit"===(l=this.evalVisProp("label.visible"))&&(l=this.visPropCalc.visible),a.prepareUpdate().updateVisibility(l).updateRenderer(),a.distanceX=this.evalVisProp("label.offset")[0],a.distanceY=this.evalVisProp("label.offset")[1]);for(i=r=i;i<o;i++)this.board.renderer.display(this.labels[i],!1),this.labels[i].visProp.visible=this.labels[i].visPropCalc.visible=!1;return this},hideElement:function(){var e;for(t.Z.deprecated("Element.hideElement()","Element.setDisplayRendNode()"),this.visPropCalc.visible=!1,this.board.renderer.display(this,!1),e=0;e<this.labels.length;e++)s.Z.exists(this.labels[e])&&this.labels[e].hideElement();return this},showElement:function(){var e;for(t.Z.deprecated("Element.showElement()","Element.setDisplayRendNode()"),this.visPropCalc.visible=!0,this.board.renderer.display(this,!1),e=0;e<this.labels.length;e++)s.Z.exists(this.labels[e])&&this.labels[e].showElement();return this}}),t.Z.createTicks=function(e,r,o){var n,a,h=s.Z.copyAttributes(o,e.options,"ticks");if(a=r.length<2?h.ticksdistance:r[1],r[0].elementClass!==i.Z.OBJECT_CLASS_LINE&&r[0].elementClass!==i.Z.OBJECT_CLASS_CURVE)throw new Error("JSXGraph: Can't create Ticks with parent types '"+typeof r[0]+"'.");return n=new t.Z.Ticks(r[0],a,h),s.Z.isFunction(h.generatelabelvalue)&&(n.generateLabelText=h.generatelabelvalue),s.Z.isFunction(h.generatelabeltext)&&(n.generateLabelText=h.generatelabeltext),n.setParents(r[0]),n.isDraggable=!0,n.fullUpdate(r[0].visPropCalc.visible),n},t.Z.createHatchmark=function(t,e,r){var o,n,a,h,l,c=[],d=s.Z.copyAttributes(r,t.options,"hatch");if(e[0].elementClass!==i.Z.OBJECT_CLASS_LINE&&e[0].elementClass!==i.Z.OBJECT_CLASS_CURVE||"number"!=typeof e[1])throw new Error("JSXGraph: Can't create Hatch mark with parent types '"+typeof e[0]+"' and '"+typeof e[1]+" and ''"+typeof e[2]+"'.");for(a=.5*-(((o=e[1])-1)*(h=d.ticksdistance)),n=0;n<o;n++)c[n]=a+n*h;return(l=t.create("ticks",[e[0],c],d)).elType="hatch",e[0].inherits.push(l),l},t.Z.registerElement("ticks",t.Z.createTicks),t.Z.registerElement("hash",t.Z.createHatchmark),t.Z.registerElement("hatch",t.Z.createHatchmark);t.Z.Ticks;var z=[0,128,64,192,32,160,96,224,16,144,80,208,48,176,112,240,8,136,72,200,40,168,104,232,24,152,88,216,56,184,120,248,4,132,68,196,36,164,100,228,20,148,84,212,52,180,116,244,12,140,76,204,44,172,108,236,28,156,92,220,60,188,124,252,2,130,66,194,34,162,98,226,18,146,82,210,50,178,114,242,10,138,74,202,42,170,106,234,26,154,90,218,58,186,122,250,6,134,70,198,38,166,102,230,22,150,86,214,54,182,118,246,14,142,78,206,46,174,110,238,30,158,94,222,62,190,126,254,1,129,65,193,33,161,97,225,17,145,81,209,49,177,113,241,9,137,73,201,41,169,105,233,25,153,89,217,57,185,121,249,5,133,69,197,37,165,101,229,21,149,85,213,53,181,117,245,13,141,77,205,45,173,109,237,29,157,93,221,61,189,125,253,3,131,67,195,35,163,99,227,19,147,83,211,51,179,115,243,11,139,75,203,43,171,107,235,27,155,91,219,59,187,123,251,7,135,71,199,39,167,103,231,23,151,87,215,55,183,119,247,15,143,79,207,47,175,111,239,31,159,95,223,63,191,127,255],J=[3,4,5,6,7,8,9,10,11,13,15,17,19,23,27,31,35,43,51,59,67,83,99,115,131,163,195,227,258,0,0],G=[0,0,0,0,0,0,0,0,1,1,1,1,2,2,2,2,3,3,3,3,4,4,4,4,5,5,5,5,0,99,99],W=[1,2,3,4,5,7,9,13,17,25,33,49,65,97,129,193,257,385,513,769,1025,1537,2049,3073,4097,6145,8193,12289,16385,24577],H=[0,0,0,0,1,1,2,2,3,3,4,4,5,5,6,6,7,7,8,8,9,9,10,10,11,11,12,12,13,13],K=[16,17,18,0,8,7,9,6,10,5,11,4,12,3,13,2,14,1,15];t.Z.Util=t.Z.Util||{},t.Z.Util.Unzip=function(e){var i,s,r,o,n=[],a=0,h=[],l=new Array(32768),c=0,d=!1,u=e.length,p=0,f=1,_=new Array(288),m=new Array(32),g=0,b=null,v=0,Z=new Array(17),C=[];function y(){return p<u?e[p++]:-1}function P(){f=1}function E(){var t;try{t=1&f,0===(f>>=1)&&(t=1&(f=y()),f=f>>1|128)}catch(t){throw console.log("Probably problems on iOS7 with >>"),t}return t}function x(t){var e=0,i=t;try{for(;i--;)e=e<<1|E();t&&(e=z[e]>>8-t)}catch(t){throw console.log("Probably problems on iOS7 with >>"),t}return e}function O(){c=0}function M(t){l[c++]=t,n.push(String.fromCharCode(t)),32768===c&&(c=0)}function w(){this.b0=0,this.b1=0,this.jump=null,this.jumppos=-1}function S(){for(;;){if(Z[v]>=o)return-1;if(r[Z[v]]===v)return Z[v]++;Z[v]++}}function T(){var t,e=b[g];if(17===v)return-1;if(g++,v++,(t=S())>=0)e.b0=t;else if(e.b0=32768,T())return-1;if((t=S())>=0)e.b1=t,e.jump=null;else if(e.b1=32768,e.jump=b[g],e.jumppos=g,T())return-1;return v--,0}function N(t,e,i,s){var n;for(b=t,g=0,r=i,o=e,n=0;n<17;n++)Z[n]=0;return v=0,T()?-1:0}function A(t){for(var e,i,s=0,r=t[s];;)if(E()){if(!(32768&r.b1))return r.b1;for(r=r.jump,e=t.length,i=0;i<e;i++)if(t[i]===r){s=i;break}}else{if(!(32768&r.b0))return r.b0;r=t[++s]}}function D(){var e,i,s,r,o,n,a,h,d,u,p,f,g,b,v;do{if(e=E(),0===(i=x(2)))for(P(),d=y(),d|=y()<<8,p=y(),65535&(d^~(p|=y()<<8))&&t.Z.debug("BlockLen checksum mismatch\n");d--;)M(y());else if(1===i)for(;;)if((r=z[x(7)]>>1)>23?(r=r<<1|E())>199?r=(r-=128)<<1|E():(r-=48)>143&&(r+=136):r+=256,r<256)M(r);else{if(256===r)break;for(h=x(G[r-=257])+J[r],r=z[x(5)]>>3,H[r]>8?(u=x(8),u|=x(H[r]-8)<<8):u=x(H[r]),u+=W[r],r=0;r<h;r++)M(l[c-u&32767])}else if(2===i){for(n=new Array(320),g=257+x(5),b=1+x(5),v=4+x(4),r=0;r<19;r++)n[r]=0;for(r=0;r<v;r++)n[K[r]]=x(3);for(h=m.length,s=0;s<h;s++)m[s]=new w;if(N(m,19,n))return O(),1;for(f=g+b,s=0;s<f;)if((r=A(m))<16)n[s++]=r;else if(16===r){if(s+(r=3+x(2))>f)return O(),1;for(o=s?n[s-1]:0;r--;)n[s++]=o}else{if(s+(r=17===r?3+x(3):11+x(7))>f)return O(),1;for(;r--;)n[s++]=0}for(h=_.length,s=0;s<h;s++)_[s]=new w;if(N(_,g,n))return O(),1;for(h=_.length,s=0;s<h;s++)m[s]=new w;for(a=[],s=g;s<n.length;s++)a[s-g]=n[s];if(N(m,b,a))return O(),1;for(;;)if((r=A(_))>=256){if(0===(r-=256))break;for(h=x(G[r-=1])+J[r],r=A(m),H[r]>8?(u=x(8),u|=x(H[r]-8)<<8):u=x(H[r]),u+=W[r];h--;)M(l[c-u&32767])}else M(r)}}while(!e);return O(),P(),0}function k(){var t,e,r,o,l,c=[];try{if(n=[],d=!1,c[0]=y(),c[1]=y(),120===c[0]&&218===c[1]&&(D(),h[a]=[n.join(""),"geonext.gxt"],a++),31===c[0]&&139===c[1]&&(R(),h[a]=[n.join(""),"file"],a++),80===c[0]&&75===c[1]){if(d=!0,c[2]=y(),c[3]=y(),3===c[2]&&4===c[3]){for(c[0]=y(),c[1]=y(),i=y(),i|=y()<<8,l=y(),l|=y()<<8,y(),y(),y(),y(),y(),y()<<8,y()<<16,y()<<24,y(),y()<<8,y()<<16,y()<<24,y(),y()<<8,y()<<16,y()<<24,o=y(),o|=y()<<8,r=y(),r|=y()<<8,t=0,C=[];o--;)"/"===(e=y())|":"===e?t=0:t<255&&(C[t++]=String.fromCharCode(e));for(s||(s=C),t=0;t<r;)e=y(),t++;if(8===l&&(D(),h[a]=new Array(2),h[a][0]=n.join(""),h[a][1]=C.join(""),a++),R())return!1}return!0}}catch(t){throw console.log("Probably problems on iOS7 with >>"),t}return!1}function R(){var t,e,s=[];if(8&i&&(s[0]=y(),s[1]=y(),s[2]=y(),s[3]=y(),80===s[0]&&75===s[1]&&7===s[2]&&8===s[3]?(y(),y()<<8,y()<<16,y()<<24):s[0]|s[1]<<8|s[2]<<16|s[3]<<24,y(),y()<<8,y()<<16,y()<<24,y(),y()<<8,y()<<16,y()<<24),d&&k())return!1;if(s[0]=y(),8!==s[0])return!0;if(i=y(),y(),y(),y(),y(),y(),y(),4&i)for(s[0]=y(),s[2]=y(),v=s[0]+256*s[1],t=0;t<v;t++)y();if(8&i)for(t=0,C=[],e=y();e;)"7"!==e&&":"!==e||(t=0),t<255&&(C[t++]=e),e=y();if(16&i)for(e=y();e;)e=y();return 2&i&&(y(),y()),D(),y(),y()<<8,y()<<16,y()<<24,y(),y()<<8,y()<<16,y()<<24,d&&k(),!1}Z[0]=0,t.Z.Util.Unzip.prototype.unzipFile=function(t){var e;for(this.unzip(),e=0;e<h.length;e++)if(h[e][1]===t)return h[e][0];return""},t.Z.Util.Unzip.prototype.unzip=function(){return k(),h}};t.Z.Util;t.Z.CA=function(t,e,i){this.node=t,this.createNode=e,this.parser=i},t.Z.extend(t.Z.CA.prototype,{findMapNode:function(t,e){var i,s,r;if("op_assign"===e.value&&e.children[0].value===t)return e.children[1];if(e.children)for(s=e.children.length,i=0;i<s;++i)if(null!==(r=this.findMapNode(t,e.children[i])))return r;return null},setMath:function(t){var e,i;if(("node_op"!=t.type||"op_add"!=t.value&&"op_sub"!=t.value&&"op_mul"!=t.value&&"op_div"!=t.value&&"op_neg"!=t.value&&"op_execfun"!=t.value&&"op_exp"!=t.value)&&"node_var"!=t.type&&"node_const"!=t.type||(t.isMath=!0),t.children)for(i=t.children.length,e=0;e<i;++e)this.setMath(t.children[e])},deriveElementary:function(t,e){var i,r=t.children[0].value,o=t.children[1];switch(r){case"abs":i=this.createNode("node_op","op_div",o[0],this.createNode("node_op","op_execfun",this.createNode("node_var","sqrt"),[this.createNode("node_op","op_mul",s.Z.deepCopy(o[0]),s.Z.deepCopy(o[0]))]));break;case"sqrt":i=this.createNode("node_op","op_div",this.createNode("node_const",1),this.createNode("node_op","op_mul",this.createNode("node_const",2),this.createNode(t.type,t.value,s.Z.deepCopy(t.children[0]),s.Z.deepCopy(t.children[1]))));break;case"sin":i=this.createNode("node_op","op_execfun",this.createNode("node_var","cos"),s.Z.deepCopy(o));break;case"cos":i=this.createNode("node_op","op_neg",this.createNode("node_op","op_execfun",this.createNode("node_var","sin"),s.Z.deepCopy(o)));break;case"tan":i=this.createNode("node_op","op_div",this.createNode("node_const",1),this.createNode("node_op","op_exp",this.createNode("node_op","op_execfun",this.createNode("node_var","cos"),s.Z.deepCopy(o)),this.createNode("node_const",2)));break;case"cot":i=this.createNode("node_op","op_neg",this.createNode("node_op","op_div",this.createNode("node_const",1),this.createNode("node_op","op_exp",this.createNode("node_op","op_execfun",this.createNode("node_var","sin"),s.Z.deepCopy(o)),this.createNode("node_const",2))));break;case"exp":i=this.createNode(t.type,t.value,s.Z.deepCopy(t.children[0]),s.Z.deepCopy(t.children[1]));break;case"pow":i=this.createNode("node_op","op_mul",this.createNode("node_op","op_execfun",s.Z.deepCopy(t.children[0]),s.Z.deepCopy(t.children[1])),this.createNode("node_op","op_add",this.createNode("node_op","op_mul",this.derivative(t.children[1][0],e),this.createNode("node_op","op_div",s.Z.deepCopy(t.children[1][1]),s.Z.deepCopy(t.children[1][0]))),this.createNode("node_op","op_mul",this.derivative(t.children[1][1],e),this.createNode("node_op","op_execfun",this.createNode("node_var","log"),[s.Z.deepCopy(t.children[1][0])]))));break;case"log":case"ln":i=this.createNode("node_op","op_div",this.createNode("node_const",1),s.Z.deepCopy(o[0]));break;case"log2":case"lb":case"ld":i=this.createNode("node_op","op_mul",this.createNode("node_op","op_div",this.createNode("node_const",1),s.Z.deepCopy(o[0])),this.createNode("node_const",1.4426950408889634));break;case"log10":case"lg":i=this.createNode("node_op","op_mul",this.createNode("node_op","op_div",this.createNode("node_const",1),s.Z.deepCopy(o[0])),this.createNode("node_const",.43429448190325176));break;case"asin":i=this.createNode("node_op","op_div",this.createNode("node_const",1),this.createNode("node_op","op_execfun",this.createNode("node_var","sqrt"),[this.createNode("node_op","op_sub",this.createNode("node_const",1),this.createNode("node_op","op_mul",s.Z.deepCopy(o[0]),s.Z.deepCopy(o[0])))]));break;case"acos":i=this.createNode("node_op","op_neg",this.createNode("node_op","op_div",this.createNode("node_const",1),this.createNode("node_op","op_execfun",this.createNode("node_var","sqrt"),[this.createNode("node_op","op_sub",this.createNode("node_const",1),this.createNode("node_op","op_mul",s.Z.deepCopy(o[0]),s.Z.deepCopy(o[0])))])));break;case"atan":i=this.createNode("node_op","op_div",this.createNode("node_const",1),this.createNode("node_op","op_add",this.createNode("node_const",1),this.createNode("node_op","op_mul",s.Z.deepCopy(o[0]),s.Z.deepCopy(o[0]))));break;case"acot":i=this.createNode("node_op","op_neg",this.createNode("node_op","op_div",this.createNode("node_const",1),this.createNode("node_op","op_add",this.createNode("node_const",1),this.createNode("node_op","op_mul",s.Z.deepCopy(o[0]),s.Z.deepCopy(o[0])))));break;case"sinh":i=this.createNode("node_op","op_execfun",this.createNode("node_var","cosh"),[s.Z.deepCopy(o[0])]);break;case"cosh":i=this.createNode("node_op","op_execfun",this.createNode("node_var","sinh"),[s.Z.deepCopy(o[0])]);break;case"tanh":i=this.createNode("node_op","op_sub",this.createNode("node_const",1),this.createNode("node_op","op_exp",this.createNode("node_op","op_execfun",this.createNode("node_var","tanh"),[s.Z.deepCopy(o[0])]),this.createNode("node_const",2)));break;case"asinh":i=this.createNode("node_op","op_div",this.createNode("node_const",1),this.createNode("node_op","op_execfun",this.createNode("node_var","sqrt"),[this.createNode("node_op","op_add",this.createNode("node_op","op_mul",s.Z.deepCopy(o[0]),s.Z.deepCopy(o[0])),this.createNode("node_const",1))]));break;case"acosh":i=this.createNode("node_op","op_div",this.createNode("node_const",1),this.createNode("node_op","op_execfun",this.createNode("node_var","sqrt"),[this.createNode("node_op","op_sub",this.createNode("node_op","op_mul",s.Z.deepCopy(o[0]),s.Z.deepCopy(o[0])),this.createNode("node_const",1))]));break;case"atanh":i=this.createNode("node_op","op_div",this.createNode("node_const",1),this.createNode("node_op","op_sub",this.createNode("node_const",1),this.createNode("node_op","op_mul",s.Z.deepCopy(o[0]),s.Z.deepCopy(o[0]))));break;default:throw i=this.createNode("node_const",0),console.log('Derivative of "'+r+'" not yet implemented'),new Error("Error("+this.line+"): ")}return i},derivative:function(t,e){var i;switch(t.type){case"node_op":switch(t.value){case"op_execfun":i="pow"==t.children[0].value?this.deriveElementary(t,e):0===t.children[1].length?this.createNode("node_const",0):this.createNode("node_op","op_mul",this.deriveElementary(t,e),this.derivative(t.children[1][0],e));break;case"op_div":i=this.createNode("node_op","op_div",this.createNode("node_op","op_sub",this.createNode("node_op","op_mul",this.derivative(t.children[0],e),s.Z.deepCopy(t.children[1])),this.createNode("node_op","op_mul",s.Z.deepCopy(t.children[0]),this.derivative(t.children[1],e))),this.createNode("node_op","op_mul",s.Z.deepCopy(t.children[1]),s.Z.deepCopy(t.children[1])));break;case"op_mul":i=this.createNode("node_op","op_add",this.createNode("node_op","op_mul",s.Z.deepCopy(t.children[0]),this.derivative(t.children[1],e)),this.createNode("node_op","op_mul",this.derivative(t.children[0],e),s.Z.deepCopy(t.children[1])));break;case"op_neg":i=this.createNode("node_op","op_neg",this.derivative(t.children[0],e));break;case"op_add":case"op_sub":i=this.createNode("node_op",t.value,this.derivative(t.children[0],e),this.derivative(t.children[1],e));break;case"op_exp":i=this.createNode("node_op","op_mul",s.Z.deepCopy(t),this.createNode("node_op","op_add",this.createNode("node_op","op_mul",this.derivative(t.children[0],e),this.createNode("node_op","op_div",s.Z.deepCopy(t.children[1]),s.Z.deepCopy(t.children[0]))),this.createNode("node_op","op_mul",this.derivative(t.children[1],e),this.createNode("node_op","op_execfun",this.createNode("node_var","log"),[s.Z.deepCopy(t.children[0])]))))}break;case"node_var":i=t.value===e?this.createNode("node_const",1):this.createNode("node_const",0);break;case"node_const":i=this.createNode("node_const",0)}return i},expandDerivatives:function(t,e,i){var r,o,n,a,h,l,c,d,u,p,f,_;if(0,!t)return 0;for(this.line=t.line,this.col=t.col,r=t.children.length,o=0;o<r;++o)if(t.children[o]&&t.children[o].type)t.children[o]=this.expandDerivatives(t.children[o],t,i);else if(s.Z.isArray(t.children[o]))for(n=0;n<t.children[o].length;++n)t.children[o][n]&&t.children[o][n].type&&(t.children[o][n]=this.expandDerivatives(t.children[o][n],t,i));if("node_op"===t.type)if("op_execfun"===t.value)if(t.children[0]&&"D"===t.children[0].value){if(_=!1,"node_var"==t.children[1][0].type&&(d=t.children[1][0].value,null!==(a=this.findMapNode(d,i))&&(_=!0)),_?(p=a.children[0],u=t.children[1].length>=2?t.children[1][1].value:a.children[0][0],h=a.children[1]):(h=t.children[1][0],p=["x"],u=t.children[1].length>=2?t.children[1][1].value:"x"),c=h,(f=t.children[1].length>=3?t.children[1][2].value:1)>=1)for(;f>=1;)c=this.derivative(c,u),c=this.removeTrivialNodes(c),f--;l="node_op"==e.type&&"op_assign"==e.value?this.createNode("node_op","op_map",p,c):c,this.setMath(l),t.type=l.type,t.value=l.value,l.children.length>0&&(t.children[0]=l.children[0]),l.children.length>1&&(t.children[1]=l.children[1])}return t},removeTrivialNodes:function(t){var e,i,r,o,n;if(s.Z.isArray(t))for(i=t.length,e=0;e<i;++e)t[e]=this.removeTrivialNodes(t[e]);if("node_op"!=t.type||!t.children)return t;for(i=t.children.length,e=0;e<i;++e){this.mayNotBeSimplified=!1;do{t.children[e]=this.removeTrivialNodes(t.children[e])}while(this.mayNotBeSimplified)}switch(t.value){case"op_map":if(r=t.children[0],"node_var"==(o=t.children[1]).type)for(e=0;e<r.length;++e)if(r[e]==o.value){o.isMath=!0;break}break;case"op_add":if(r=t.children[0],o=t.children[1],"node_const"==r.type&&0===r.value)return o;if("node_const"==o.type&&0===o.value)return r;if("node_const"==r.type&&"node_const"==o.type)return r.value+=o.value,r;break;case"op_mul":if(r=t.children[0],o=t.children[1],"node_const"==r.type&&1==r.value)return o;if("node_const"==o.type&&1==o.value)return r;if("node_const"==r.type&&0===r.value)return r;if("node_const"==o.type&&0===o.value)return o;if("node_const"==o.type&&0===o.value)return o;if("node_op"==r.type&&"op_neg"==r.value&&"node_op"==o.type&&"op_neg"==o.value)return t.children=[r.children[0],o.children[0]],this.mayNotBeSimplified=!0,t;if("op_neg"==r.value&&"op_neg"!=o.value)return t.type="node_op",t.value="op_neg",t.children=[this.createNode("node_op","op_mul",r.children[0],o)],this.mayNotBeSimplified=!0,t;if("op_neg"!=r.value&&"op_neg"==o.value)return t.type="node_op",t.value="op_neg",t.children=[this.createNode("node_op","op_mul",r,o.children[0])],this.mayNotBeSimplified=!0,t;if("op_div"==r.value&&"node_const"==r.children[0].type&&1==r.children[0].value)return t.type="node_op",t.value="op_div",t.children=[o,r.children[1]],this.mayNotBeSimplified=!0,t;if("op_div"==o.value&&"node_const"==o.children[0].type&&1==o.children[0].value)return t.type="node_op",t.value="op_div",t.children=[r,o.children[1]],this.mayNotBeSimplified=!0,t;if("node_const"!=r.type&&"node_const"==o.type)return t.children=[o,r],this.mayNotBeSimplified=!0,t;if("node_const"!=r.type&&"node_op"==o.type&&"op_neg"==o.value&&"node_const"==o.children[0].type)return t.children=[o,r],this.mayNotBeSimplified=!0,t;if("node_op"==r.type&&"op_execfun"!=r.value&&("node_var"==o.type||"node_op"==o.type&&"op_execfun"==o.value))return t.children=[o,r],this.mayNotBeSimplified=!0,t;if("node_op"!=r.type&&"node_op"==o.type&&"op_neg"==o.value&&"node_var"==o.children[0].type)return t.children=[o,r],this.mayNotBeSimplified=!0,t;if("node_const"!=r.type&&"node_op"==o.type&&("op_mul"==o.value||"op_div"==o.value)&&"node_const"==o.children[0].type)return n=o.children[0],o.children[0]=r,t.children=[n,o],this.mayNotBeSimplified=!0,t;if("node_const"!=o.type&&"node_op"==r.type&&"op_mul"==r.value&&"node_const"==r.children[0].type)return t.children=[r.children[0],this.createNode("node_op","op_mul",r.children[1],o)],this.mayNotBeSimplified=!0,t;if("node_const"==r.type&&"node_const"==o.type)return r.value*=o.value,r;if("node_const"==r.type&&"node_op"==o.type&&("op_mul"==o.value||"op_div"==o.value)&&"node_const"==o.children[0].type)return o.children[0].value*=r.value,o;if(r.hash=this.parser.compile(r),o.hash=this.parser.compile(o),r.hash===o.hash)return t.value="op_exp",t.children[1]=this.createNode("node_const",2),t;if("node_const"==r.type&&"node_op"==o.type&&("op_mul"==o.value||"op_div"==o.value)&&"node_const"==o.children[0].type)return o.children[0].value*=r.value,o;if("node_op"==o.type&&"op_exp"==o.value&&(r.hash||(r.hash=this.parser.compile(r)),o.children[0].hash||(o.children[0].hash=this.parser.compile(o.children[0])),r.hash===o.children[0].hash))return o.children[1]=this.createNode("node_op","op_add",o.children[1],this.createNode("node_const",1)),this.mayNotBeSimplified=!0,o;if("node_op"==r.type&&"op_exp"==r.value&&"node_op"==o.type&&"op_exp"==o.value&&(r.children[0].hash=this.parser.compile(r.children[0]),o.children[0].hash=this.parser.compile(o.children[0]),r.children[0].hash===o.children[0].hash))return r.children[1]=this.createNode("node_op","op_add",r.children[1],o.children[1]),this.mayNotBeSimplified=!0,r;break;case"op_sub":if(r=t.children[0],o=t.children[1],"node_const"==r.type&&0===r.value)return t.value="op_neg",t.children[0]=o,t;if("node_const"==o.type&&0===o.value)return r;if("node_const"==r.type&&"node_const"==o.type&&r.value==o.value)return this.createNode("node_const",0);if("node_var"==r.type&&"node_var"==o.type&&r.value==o.value)return this.createNode("node_const",0);if("node_const"==r.type&&"node_const"==o.type)return r.value-=o.value,r;if("node_op"==r.type&&"op_mul"==r.value&&"node_op"==o.type&&"op_mul"==o.value&&(r.children[1].hash=this.parser.compile(r.children[1]),o.children[1].hash=this.parser.compile(o.children[1]),r.children[1].hash===o.children[1].hash))return t.value="op_mul",t.children=[this.createNode("node_op","op_sub",r.children[0],o.children[0]),r.children[1]],this.mayNotBeSimplified=!0,t;if("node_op"==r.type&&"op_mul"==r.value&&(r.children[1].hash=this.parser.compile(r.children[1]),o.hash=this.parser.compile(o),r.children[1].hash===o.hash))return t.value="op_mul",t.children=[this.createNode("node_op","op_sub",r.children[0],this.createNode("node_const",1)),o],this.mayNotBeSimplified=!0,t;if("node_op"==o.type&&"op_mul"==o.value&&(o.children[1].hash=this.parser.compile(o.children[1]),r.hash=this.parser.compile(r),o.children[1].hash===r.hash))return t.value="op_mul",t.children=[this.createNode("node_op","op_sub",this.createNode("node_const",1),o.children[0]),r],this.mayNotBeSimplified=!0,t;break;case"op_neg":if("node_const"==(r=t.children[0]).type&&0===r.value)return r;if("node_op"==r.type&&"op_neg"==r.value)return r.children[0];break;case"op_div":if(r=t.children[0],o=t.children[1],"node_const"==r.type&&"node_const"==o.type&&r.value==o.value&&0!==r.value)return r.value=1,r;if("node_const"==r.type&&0===r.value&&"node_const"==o.type&&0!==o.value)return r.value=0,r;if("node_const"==r.type&&0===r.value&&("node_op"==o.type||"node_var"==o.type))return t.type="node_const",t.value=0,t;if("node_var"==r.type&&"node_var"==o.type&&r.value==o.value)return this.createNode("node_const",1);if("node_const"==r.type&&0!==r.value&&"node_const"==o.type&&0===o.value)return r.value>0?r.value=1/0:r.value=-1/0,r;if("node_op"==r.type&&"op_neg"==r.value&&"node_op"==o.type&&"op_neg"==o.value)return t.children=[r.children[0],o.children[0]],this.mayNotBeSimplified=!0,t;if("op_neg"==r.value&&"op_neg"!=o.value)return t.type="node_op",t.value="op_neg",t.children=[this.createNode("node_op","op_div",r.children[0],o)],this.mayNotBeSimplified=!0,t;if("op_neg"!=r.value&&"op_neg"==o.value)return t.type="node_op",t.value="op_neg",t.children=[this.createNode("node_op","op_div",r,o.children[0])],this.mayNotBeSimplified=!0,t;if("node_op"==r.type&&"op_exp"==r.value&&(o.hash||(o.hash=this.parser.compile(o)),r.children[0].hash||(r.children[0].hash=this.parser.compile(r.children[0])),o.hash===r.children[0].hash))return r.children[1]=this.createNode("node_op","op_sub",r.children[1],this.createNode("node_const",1)),this.mayNotBeSimplified=!0,r;if("node_const"!=o.type&&"node_op"==r.type&&"op_mul"==r.value&&"node_const"==r.children[0].type)return t.value="op_mul",t.children=[r.children[0],this.createNode("node_op","op_div",r.children[1],o)],this.mayNotBeSimplified=!0,t;if("node_op"==r.type&&"op_exp"==r.value&&"node_op"==o.type&&"op_exp"==o.value&&(r.children[0].hash=this.parser.compile(r.children[0]),o.children[0].hash=this.parser.compile(o.children[0]),r.children[0].hash===o.children[0].hash))return r.children[1]=this.createNode("node_op","op_sub",r.children[1],o.children[1]),this.mayNotBeSimplified=!0,r;break;case"op_exp":if(r=t.children[0],"node_const"==(o=t.children[1]).type&&0===o.value)return o.value=1,o;if("node_const"==o.type&&1==o.value)return r;if("node_const"==r.type&&1==r.value)return r;if("node_const"==r.type&&0===r.value&&"node_const"==o.type&&0!==o.value)return r;if("node_op"==r.type&&"op_exp"==r.value)return t.children=[r.children[0],this.createNode("node_op","op_mul",r.children[1],o)],t}switch(t.value){case"op_add":if(r=t.children[0],o=t.children[1],"node_const"==r.type&&"node_const"==o.type&&r.value==o.value)return r.value+=o.value,r;if("node_var"==r.type&&"node_var"==o.type&&r.value==o.value)return t.children[0]=this.createNode("node_const",2),t.value="op_mul",t;if("node_op"==r.type&&"op_neg"==r.value)return t.value="op_sub",t.children[0]=o,t.children[1]=r.children[0],this.mayNotBeSimplified=!0,t;if("node_op"==o.type&&"op_neg"==o.value)return t.value="op_sub",t.children[1]=o.children[0],this.mayNotBeSimplified=!0,t;if("node_op"==r.type&&"op_mul"==r.value&&"node_op"==o.type&&"op_mul"==o.value&&(r.children[1].hash=this.parser.compile(r.children[1]),o.children[1].hash=this.parser.compile(o.children[1]),r.children[1].hash===o.children[1].hash))return t.value="op_mul",t.children=[this.createNode("node_op","op_add",r.children[0],o.children[0]),r.children[1]],this.mayNotBeSimplified=!0,t;if("node_op"==r.type&&"op_mul"==r.value&&(r.children[1].hash=this.parser.compile(r.children[1]),o.hash=this.parser.compile(o),r.children[1].hash===o.hash))return t.value="op_mul",t.children=[this.createNode("node_op","op_add",r.children[0],this.createNode("node_const",1)),o],this.mayNotBeSimplified=!0,t;if("node_op"==o.type&&"op_mul"==o.value&&(o.children[1].hash=this.parser.compile(o.children[1]),r.hash=this.parser.compile(r),o.children[1].hash===r.hash))return t.value="op_mul",t.children=[this.createNode("node_op","op_add",this.createNode("node_const",1),o.children[0]),r],this.mayNotBeSimplified=!0,t;break;case"op_sub":if(r=t.children[0],"node_op"==(o=t.children[1]).type&&"op_neg"==o.value)return t.value="op_add",t.children[1]=o.children[0],this.mayNotBeSimplified=!0,t;break;case"op_execfun":return this.simplifyElementary(t)}return t},simplifyElementary:function(t){var e=t.children[0].value,i=t.children[1];if(0==i.length)return t;switch(e){case"sin":case"tan":if("node_const"==i[0].type&&0===i[0].value)return t.type="node_const",t.value=0,t;if("node_var"==i[0].type&&"PI"==i[0].value)return t.type="node_const",t.value=0,t;if("node_op"==i[0].type&&"op_mul"==i[0].value&&"node_const"==i[0].children[0].type&&i[0].children[0].value%1==0&&"node_var"==i[0].children[1].type&&"PI"==i[0].children[1].value)return t.type="node_const",t.value=0,t;break;case"cos":if("node_const"==i[0].type&&0===i[0].value)return t.type="node_const",t.value=1,t;if("node_var"==i[0].type&&"PI"==i[0].value)return t.type="node_op",t.value="op_neg",t.children=[this.createNode("node_const",1)],t;break;case"exp":if("node_const"==i[0].type&&0===i[0].value)return t.type="node_const",t.value=1,t;break;case"pow":if("node_const"==i[1].type&&0===i[1].value)return t.type="node_const",t.value=1,t}return t}});t.Z.CA;t.Z.Dump={addMarkers:function(t,e,i){var r,o,n;for(r in s.Z.isArray(e)||(e=[e]),s.Z.isArray(i)||(i=[i]),o=Math.min(e.length,i.length),e.length=o,i.length=o,t.objects)if(t.objects.hasOwnProperty(r))for(n=0;n<o;n++)t.objects[r][e[n]]=i[n]},deleteMarkers:function(t,e){var i,r,o;for(i in s.Z.isArray(e)||(e=[e]),r=e.length,e.length=r,t.objects)if(t.objects.hasOwnProperty(i))for(o=0;o<r;o++)delete t.objects[i][e[o]]},str:function(t){return"string"==typeof t&&"function"!==t.slice(0,7)&&(t='"'+t+'"'),t},minimizeObject:function(e,i){var r,o,n,a={},h=s.Z.deepCopy(e),l=[];for(n=1;n<arguments.length;n++)l.push(arguments[n]);for(a=s.Z.deepCopy(a,t.Z.Options.elements,!0),n=l.length;n>0;n--)a=s.Z.deepCopy(a,l[n-1],!0);for(r in a)a.hasOwnProperty(r)&&(o=r.toLowerCase(),"object"!=typeof a[r]&&a[r]===h[o]&&delete h[o]);return h},prepareAttributes:function(e,i){var s,r;for(r in s=this.minimizeObject(i.getAttributes(),t.Z.Options[i.elType]),i.subs)i.subs.hasOwnProperty(r)&&(s[r]=this.minimizeObject(i.subs[r].getAttributes(),t.Z.Options[i.elType][r],t.Z.Options[i.subs[r].elType]),s[r].id=i.subs[r].id,s[r].name=i.subs[r].name);return s.id=i.id,s.name=i.name,s},setBoundingBox:function(t,e,i){return t.push({obj:i,method:"setBoundingBox",params:[e.getBoundingBox(),e.keepaspectratio]}),t},dump:function(t){var e,i,r,o,n=[],a=[],h=t.objectsList.length;for(this.addMarkers(t,"dumped",!1),e=0;e<h;e++)if(r={},!(i=t.objectsList[e]).dumped&&i.dump){for(r.type=i.getType(),r.parents=i.getParents().slice(),"point"===r.type&&1===r.parents[0]&&(r.parents=r.parents.slice(1)),o=0;o<r.parents.length;o++)s.Z.isString(r.parents[o])&&"'"!==r.parents[o][0]&&'"'!==r.parents[o][0]?r.parents[o]='"'+r.parents[o]+'"':s.Z.isArray(r.parents[o])&&(r.parents[o]="["+r.parents[o].toString()+"]");r.attributes=this.prepareAttributes(t,i),"glider"===r.type&&i.onPolygon&&n.push({obj:i.id,prop:"onPolygon",val:!0}),a.push(r)}return this.deleteMarkers(t,"dumped"),{elements:a,props:n,methods:[]}},arrayToParamStr:function(t,e){var i,s=[];for(i=0;i<t.length;i++)s.push(e.call(this,t[i]));return s.join(", ")},toJCAN:function(t){var e,i,r;switch(typeof t){case"object":if(t){if(i=[],s.Z.isArray(t)){for(e=0;e<t.length;e++)i.push(this.toJCAN(t[e]));return"["+i.join(",")+"]"}for(r in t)t.hasOwnProperty(r)&&i.push(r+": "+this.toJCAN(t[r]));return"<<"+i.join(", ")+">> "}return"null";case"string":return"'"+t.replace(/\\/g,"\\\\").replace(/(["'])/g,"\\$1")+"'";case"number":case"boolean":return t.toString();case"null":return"null"}},toJessie:function(t){var e,i,s,r,o,n=this.dump(t),a=[];for(n.methods=this.setBoundingBox(n.methods,t,"$board"),r=n.elements,e=0;e<r.length;e++){for(r[e].attributes.name.length>0&&a.push("// "+r[e].attributes.name),s=[],i=0;i<r[e].parents.length;i++)s.push("number"==typeof r[e].parents[i]?r[e].parents[i].toFixed(100).replace(/(\.\d*?)0*$/,"$1"):r[e].parents[i]);a.push("s"+e+" = "+r[e].type+"("+s.join(", ")+") "+this.toJCAN(r[e].attributes).replace(/\n/,"\\n")+";"),"axis"===r[e].type&&(o=r[e].attributes.id,null===t.objects[o].defaultTicks&&a.push("s"+e+".removeAllTicks();")),a.push("")}for(e=0;e<n.methods.length;e++)a.push(n.methods[e].obj+"."+n.methods[e].method+"("+this.arrayToParamStr(n.methods[e].params,this.toJCAN)+");"),a.push("");for(e=0;e<n.props.length;e++)a.push(n.props[e].obj+"."+n.props[e].prop+" = "+this.toJCAN(n.props[e].val)+";"),a.push("");return a.join("\n")},toJavaScript:function(t){var e,i,r,o=this.dump(t),n=[];for(o.methods=this.setBoundingBox(o.methods,t,"board"),i=o.elements,e=0;e<i.length;e++)n.push('board.create("'+i[e].type+'", ['+i[e].parents.join(", ")+"], "+s.Z.toJSON(i[e].attributes)+");"),"axis"===i[e].type&&(r=i[e].attributes.id,null===t.objects[r].defaultTicks&&n.push('board.objects["'+r+'"].removeTicks(board.objects["'+r+'"].defaultTicks);'));for(e=0;e<o.methods.length;e++)n.push(o.methods[e].obj+"."+o.methods[e].method+"("+this.arrayToParamStr(o.methods[e].params,s.Z.toJSON)+");"),n.push("");for(e=0;e<o.props.length;e++)n.push(o.props[e].obj+"."+o.props[e].prop+" = "+s.Z.toJSON(o.props[e].val)+";"),n.push("");return n.join("\n")}};t.Z.Dump;t.Z.createComb=function(e,i,r){var o,n,a,h,l;if(2!==i.length)throw l=i.map((function(t){return"'"+typeof t+"'"})),new Error("JSXGraph: Can't create comb with parent types "+l.join(", ")+".\nPossible parent types: [point,point], [[x1,y1],[x2,y2]]");if(s.Z.isArray(i[0])&&i[0].length>1)h=s.Z.copyAttributes(r,e.options,"comb","point1"),o=e.create("point",i[0],h);else if(s.Z.isString(i[0])||s.Z.isPoint(i[0]))o=e.select(i[0]);else if(s.Z.isFunction(i[0])&&s.Z.isPoint(i[0]()))o=i[0]();else{if(!(s.Z.isFunction(i[0])&&i[0]().length&&i[0]().length>=2))throw new Error("JSXGraph: Can't create comb with parent types '"+typeof i[0]+"' and '"+typeof i[1]+"'.\nPossible parent types: [point,point], [[x1,y1],[x2,y2]]");h=s.Z.copyAttributes(r,e.options,"comb","point1"),o=t.Z.createPoint(e,i[0](),h)}if(s.Z.isArray(i[1])&&i[1].length>1)h=s.Z.copyAttributes(r,e.options,"comb","point2"),n=e.create("point",i[1],h);else if(s.Z.isString(i[1])||s.Z.isPoint(i[1]))n=e.select(i[1]);else if(s.Z.isFunction(i[1])&&s.Z.isPoint(i[1]()))n=i[1]();else{if(!(s.Z.isFunction(i[1])&&i[1]().length&&i[1]().length>=2))throw new Error("JSXGraph: Can't create comb with parent types '"+typeof i[0]+"' and '"+typeof i[1]+"'.\nPossible parent types: [point,point], [[x1,y1],[x2,y2]]");h=s.Z.copyAttributes(r,e.options,"comb","point2"),n=t.Z.createPoint(e,i[1](),h)}return h=s.Z.copyAttributes(r,e.options,"comb"),(a=e.create("curve",[[0],[0]],h)).updateDataArray=function(){var t,e,i,s,r,h,l,c,d,u,p=0,f=o.Dist(n),_=o,m=n;for(c=a.evalVisProp("frequency"),d=-a.evalVisProp("angle"),u=a.evalVisProp("width"),a.evalVisProp("reverse")&&(_=n,m=o,d=-d),t=Math.cos(d),e=Math.sin(d),i=(m.X()-_.X())/f,s=(m.Y()-_.Y())/f,t*=u/Math.abs(e),e*=u/Math.abs(e),this.dataX=[],this.dataY=[];p<f;)r=_.X()+i*p,h=_.Y()+s*p,e*=l=Math.min(t,f-p)/Math.abs(t),t*=l,this.dataX.push(r),this.dataY.push(h),this.dataX.push(r+i*t+s*e),this.dataY.push(h-i*e+s*t),this.dataX.push(NaN),this.dataY.push(NaN),p+=c},a},t.Z.registerElement("comb",t.Z.createComb);var $=function(){X.prototype.remove.call(this),this.board.removeObject(this.toppoint),this.board.removeObject(this.glider),this.board.removeObject(this.baseline),this.board.removeObject(this.basepoint),this.board.removeObject(this.label),this._isPrivateTangent&&this.board.removeObject(this.tangent)},q=function(){return this.tangent.getSlope()},Q=function(){return this.tangent.Direction()};t.Z.createSlopeTriangle=function(e,r,o){var n,a,h,l,c,d,u,p,f,_=!1;if(1===r.length&&r[0].type===i.Z.OBJECT_TYPE_TANGENT)a=r[0],h=a.glider;else if(1===r.length&&r[0].type===i.Z.OBJECT_TYPE_GLIDER)h=r[0],f=s.Z.copyAttributes(o,e.options,"slopetriangle","tangent"),a=e.create("tangent",[h],f),_=!0;else{if(2!==r.length||r[0].elementClass!==i.Z.OBJECT_CLASS_LINE||!s.Z.isPoint(r[1]))throw new Error("JSXGraph: Can't create slope triangle with parent types '"+typeof r[0]+"'.");a=r[0],h=r[1]}return f=s.Z.copyAttributes(o,e.options,"slopetriangle","basepoint"),u=e.create("point",[function(){return[h.X()+1,h.Y()]}],f),f=s.Z.copyAttributes(o,e.options,"slopetriangle","baseline"),d=e.create("line",[h,u],f),f=s.Z.copyAttributes(o,e.options,"slopetriangle","glider"),l=e.create("glider",[h.X()+1,h.Y(),d],f),f=s.Z.copyAttributes(o,e.options,"slopetriangle","toppoint"),c=e.create("point",[function(){return[l.X(),l.Y()+(l.X()-h.X())*a.getSlope()]}],f),f=s.Z.copyAttributes(o,e.options,"slopetriangle"),(n=e.create("polygon",[h,l,c],f)).Value=q,n.Direction=Q,n.tangent=a,n._isPrivateTangent=_,n.borders[2].setArrow(!1,!1),f=s.Z.copyAttributes(o,e.options,"slopetriangle","label"),(f=s.Z.copyAttributes(f,e.options,"label")).isLabel=!0,f.anchor=n.borders[1],f.priv=n.borders[1].visProp.priv,f.id=n.borders[1].id+"Label",(p=e.create("text",[0,0,function(){return""}],f)).needsUpdate=!0,p.dump=!1,n.borders[1].label=p,n.borders[1].hasLabel=!0,n.borders[1].visProp.withlabel=!0,p._setText((function(){var t=p.evalVisProp("digits");return p.useLocale()?p.formatNumberLocale(n.Value(),t):s.Z.toFixed(n.Value(),t)})),p.fullUpdate(),n.glider=l,n.basepoint=u,n.baseline=d,n.toppoint=c,n.label=p,n.subs={glider:l,basePoint:u,baseLine:d,topPoint:c,label:p},n.inherits.push(l,u,d,c,p),n.methodMap=t.Z.deepCopy(n.methodMap,{tangent:"tangent",glider:"glider",basepoint:"basepoint",baseline:"baseline",toppoint:"toppoint",label:"label",Value:"Value",V:"Value",Direction:"Direction"}),n.remove=$,n},t.Z.registerElement("slopetriangle",t.Z.createSlopeTriangle);var tt=function(){this._value=this.rendNodeCheckbox.checked,this.board.update()};t.Z.createCheckbox=function(i,r,o){var n,a,h=s.Z.copyAttributes(o,i.options,"checkbox");return a=[r[0],r[1],'<span style="display:inline"><input type="checkbox" /><label for=""></label></span>'],(n=i.create("text",a,h)).type=s.Z.OBJECT_TYPE_CHECKBOX,n.rendNodeCheckbox=n.rendNode.childNodes[0].childNodes[0],n.rendNodeLabel=n.rendNode.childNodes[0].childNodes[1],n.rendNodeTag=n.rendNodeCheckbox,n.rendNodeTag.disabled=!!h.disabled,n.rendNodeCheckbox.id=n.rendNode.id+"_checkbox",n.rendNodeLabel.id=n.rendNode.id+"_label",n.rendNodeLabel.setAttribute("for",n.rendNodeCheckbox.id),n.setText(r[2]),n.visPropOld.fontsize="0px",i.renderer.updateTextStyle(n,!1),n.rendNodeCheckbox.checked=h.checked,n._value=h.checked,n.Value=function(){return this._value},n.update=function(){return this.needsUpdate&&(t.Z.Text.prototype.update.call(this),this._value=this.rendNodeCheckbox.checked),this},e.Z.addEvent(n.rendNodeCheckbox,"change",tt,n),n},t.Z.registerElement("checkbox",t.Z.createCheckbox);var et=function(t){this._value=this.rendNodeInput.value,this.board.update()};t.Z.createInput=function(i,r,o){var n,a,h=s.Z.copyAttributes(o,i.options,"input");return a=[r[0],r[1],'<span style="display:inline; white-space:nowrap; padding:0px;"><label></label><input type="text" maxlength="'+h.maxlength+'" style="width:100%"/></span>'],(n=i.create("text",a,h)).type=s.Z.OBJECT_TYPE_INPUT,n.rendNodeLabel=n.rendNode.childNodes[0].childNodes[0],n.rendNodeInput=n.rendNode.childNodes[0].childNodes[1],n.rendNodeInput.value=r[2],n.rendNodeTag=n.rendNodeInput,n.rendNodeTag.disabled=!!h.disabled,n.rendNodeLabel.id=n.rendNode.id+"_label",n.rendNodeInput.id=n.rendNode.id+"_input",n.rendNodeInput.setAttribute("aria-labelledby",n.rendNodeLabel.id),n.setText(r[3]),n._value=r[2],n.rendNodeInput.addEventListener("keydown",(t=>{"ArrowLeft"!==t.key&&"ArrowRight"!==t.key||t.stopPropagation()})),n.update=function(){return this.needsUpdate&&(t.Z.Text.prototype.update.call(this),this._value=this.rendNodeInput.value),this},n.Value=function(){return this._value},n.set=function(t){return this._value=t,this.rendNodeInput.value=t,this},e.Z.addEvent(n.rendNodeInput,"input",et,n),e.Z.addEvent(n.rendNodeInput,"mousedown",(function(t){s.Z.exists(t.stopPropagation)&&t.stopPropagation()}),n),e.Z.addEvent(n.rendNodeInput,"touchstart",(function(t){s.Z.exists(t.stopPropagation)&&t.stopPropagation()}),n),e.Z.addEvent(n.rendNodeInput,"pointerdown",(function(t){s.Z.exists(t.stopPropagation)&&t.stopPropagation()}),n),n.visPropOld.fontsize="0px",i.renderer.updateTextStyle(n,!1),n},t.Z.registerElement("input",t.Z.createInput);var it=function(){this._handler&&this._handler(),this.board.update()};t.Z.createButton=function(i,r,o){var n,a,h=s.Z.copyAttributes(o,i.options,"button");return a=[r[0],r[1],'<button type="button" style="width:100%; height:100%;" tabindex="0"></button>'],(n=i.create("text",a,h)).type=s.Z.OBJECT_TYPE_BUTTON,n.rendNodeButton=n.rendNode.childNodes[0],n.rendNodeButton.id=n.rendNode.id+"_button",n.rendNodeTag=n.rendNodeButton,n.rendNodeTag.disabled=!!h.disabled,n.setText(r[2]),n.visPropOld.fontsize="0px",i.renderer.updateTextStyle(n,!1),r[3]&&(s.Z.isString(r[3])?(n._jc=new t.Z.JessieCode,n._jc.use(i),n._handler=function(){n._jc.parse(r[3])}):n._handler=r[3]),e.Z.addEvent(n.rendNodeButton,"click",it,n),e.Z.addEvent(n.rendNodeButton,"mousedown",(function(t){s.Z.exists(t.stopPropagation)&&t.stopPropagation()}),n),e.Z.addEvent(n.rendNodeButton,"touchstart",(function(t){s.Z.exists(t.stopPropagation)&&t.stopPropagation()}),n),e.Z.addEvent(n.rendNodeButton,"pointerdown",(function(t){s.Z.exists(t.stopPropagation)&&t.stopPropagation()}),n),n},t.Z.registerElement("button",t.Z.createButton),t.Z.createVectorField=function(t,e,i){var r,o;if(!(e.length>=3&&(s.Z.isArray(e[0])||s.Z.isFunction(e[0])||s.Z.isString(e[0]))&&s.Z.isArray(e[1])&&3===e[1].length&&s.Z.isArray(e[2])&&3===e[2].length))throw new Error("JSXGraph: Can't create vector field with parent types '"+typeof e[0]+"', '"+typeof e[1]+"', '"+typeof e[2]+"'.");return o=s.Z.copyAttributes(i,t.options,"vectorfield"),(r=t.create("curve",[[],[]],o)).elType="vectorfield",r.setF=function(t,e){var i,o;return s.Z.isArray(t)?(i=s.Z.createFunction(t[0],this.board,e),o=s.Z.createFunction(t[1],this.board,e),this.F=function(t,e){return[i(t,e),o(t,e)]}):this.F=s.Z.createFunction(t,r.board,e),this},r.setF(e[0],"x, y"),r.xData=e[1],r.yData=e[2],r.updateDataArray=function(){var e,i,r,o,n,a,h,l,c,d,u,p,f=this.evalVisProp("scale"),_=s.Z.evaluate(this.xData[0]),m=s.Z.evaluate(this.xData[1]),g=(s.Z.evaluate(this.xData[2])-_)/m,b=s.Z.evaluate(this.yData[0]),v=s.Z.evaluate(this.yData[1]),Z=(s.Z.evaluate(this.yData[2])-b)/v,C=this.evalVisProp("arrowhead.enabled");for(C&&(d=(c=this.evalVisProp("arrowhead.size"))/t.unitX,u=c/t.unitY,p=this.evalVisProp("arrowhead.angle")),this.dataX=[],this.dataY=[],r=0,e=_;r<=m;e+=g,r++)for(o=0,i=b;o<=v;i+=Z,o++)(n=this.F(e,i))[0]*=f,n[1]*=f,s.Z.concat(this.dataX,[e,e+n[0],NaN]),s.Z.concat(this.dataY,[i,i+n[1],NaN]),C&&Math.abs(n[0])+Math.abs(n[1])>0&&(h=(a=Math.atan2(n[1],n[0]))+p,l=a-p,s.Z.concat(this.dataX,[e+n[0]-Math.cos(h)*d,e+n[0],e+n[0]-Math.cos(l)*d,NaN]),s.Z.concat(this.dataY,[i+n[1]-Math.sin(h)*u,i+n[1],i+n[1]-Math.sin(l)*u,NaN]))},r.methodMap=s.Z.deepCopy(r.methodMap,{setF:"setF"}),r},t.Z.registerElement("vectorfield",t.Z.createVectorField),t.Z.createSlopeField=function(t,e,i){var r,o,n;if(!(e.length>=3&&(s.Z.isFunction(e[0])||s.Z.isString(e[0]))&&s.Z.isArray(e[1])&&3===e[1].length&&s.Z.isArray(e[2])&&3===e[2].length))throw new Error("JSXGraph: Can't create slope field with parent types '"+typeof e[0]+"', '"+typeof e[1]+"', '"+typeof e[2]+"'.");return o=s.Z.createFunction(e[0],t,"x, y"),e[0]=function(t,e){var i=o(t,e),s=Math.sqrt(1+i*i);return[1/s,i/s]},n=s.Z.copyAttributes(i,t.options,"slopefield"),(r=t.create("vectorfield",e,n)).elType="slopefield",r.setF=function(t,e){var i=s.Z.createFunction(t,r.board,e);this.F=function(t,e){var s=i(t,e),r=Math.sqrt(1+s*s);return[1/r,s/r]}},r.methodMap=s.Z.deepCopy(r.methodMap,{setF:"setF"}),r},t.Z.registerElement("slopefield",t.Z.createSlopeField),t.Z.createSmartLabel=function(t,e,r){var o,n,a,h,l,c;if(0===e.length||[i.Z.OBJECT_CLASS_POINT,i.Z.OBJECT_CLASS_LINE,i.Z.OBJECT_CLASS_CIRCLE].indexOf(e[0].elementClass)<0&&[i.Z.OBJECT_TYPE_POLYGON,i.Z.OBJECT_TYPE_ANGLE].indexOf(e[0].type)<0)throw new Error("JSXGraph: Can't create smartlabel with parent types '"+typeof e[0]+"', '"+typeof e[1]+"'.");return a=e[0],h=e[1]||"",a.elementClass===i.Z.OBJECT_CLASS_POINT?n=s.Z.copyAttributes(r,t.options,"smartlabelpoint"):a.elementClass===i.Z.OBJECT_CLASS_LINE?((n=s.Z.copyAttributes(r,t.options,"smartlabelline")).rotate=function(){return 180*Math.atan(a.getSlope())/Math.PI},n.visible=function(){return!(a.L()<1.5)}):a.elementClass===i.Z.OBJECT_CLASS_CIRCLE?(n=s.Z.copyAttributes(r,t.options,"smartlabelcircle")).visible=function(){return!(a.Radius()<1.5)}:a.type===i.Z.OBJECT_TYPE_POLYGON?n=s.Z.copyAttributes(r,t.options,"smartlabelpolygon"):a.type===i.Z.OBJECT_TYPE_ANGLE&&((n=s.Z.copyAttributes(r,t.options,"smartlabelangle")).rotate=function(){var t=a.center.coords.usrCoords,e=a.getLabelAnchor().usrCoords,i=180*Math.atan2(e[2]-t[2],e[1]-t[1])/Math.PI;return i>90&&i<270?i+180:i},n.anchorX=function(){var t=a.center.coords.usrCoords,e=a.getLabelAnchor().usrCoords,i=180*Math.atan2(e[2]-t[2],e[1]-t[1])/Math.PI;return i>90&&i<270?"right":"left"}),l=function(t,e,i,r){var o;switch(r){case"length":o=function(){return e.L()};break;case"slope":o=function(){return e.Slope()};break;case"area":o=function(){return e.Area()};break;case"radius":o=function(){return e.Radius()};break;case"perimeter":o=function(){return e.Perimeter()};break;case"rad":o=function(){return e.Value()};break;case"deg":o=function(){return 180*e.Value()/Math.PI};break;default:o=function(){return 0}}return function(){var e,i="",r=s.Z.evaluate(h),n=t.evalVisProp("digits"),a=t.evalVisProp("unit"),l=t.evalVisProp("prefix"),c=t.evalVisProp("suffix"),d=t.evalVisProp("usemathjax")||t.evalVisProp("usekatex");return""===r?(e=t.useLocale()?t.formatNumberLocale(o(),n):s.Z.toFixed(o(),n),i=d?["\\(",l,e,"\\,",a,c,"\\)"].join(""):[l,e,a,c].join("")):i=r,i}},a.elementClass===i.Z.OBJECT_CLASS_POINT?(o=t.create("text",[function(){return a.X()},function(){return a.Y()},""],n),c=function(){var t,e,i="",r=s.Z.evaluate(h),n=o.evalVisProp("digits"),l=o.evalVisProp("unit"),c=o.evalVisProp("prefix"),d=o.evalVisProp("suffix"),u=o.evalVisProp("dir"),p=o.evalVisProp("usemathjax")||o.evalVisProp("usekatex");return o.useLocale()?(t=o.formatNumberLocale(a.X(),n),e=o.formatNumberLocale(a.Y(),n)):(t=s.Z.toFixed(a.X(),n),e=s.Z.toFixed(a.Y(),n)),""===r?"row"===u?i=p?["\\(",c,t,"\\,",l," / ",e,"\\,",l,d,"\\)"].join(""):[c,t," ",l," / ",e," ",l,d].join(""):0===u.indexOf("col")&&(i=p?["\\(",c,"\\left(\\array{",t,"\\,",l,"\\\\ ",e,"\\,",l,"}\\right)",d,"\\)"].join(""):[c,t," ",l,"<br/>",e," ",l,d].join("")):i=r,i}):a.elementClass===i.Z.OBJECT_CLASS_LINE?"length"===n.measure?c=l(o=t.create("text",[function(){return.5*(a.point1.X()+a.point2.X())},function(){return.5*(a.point1.Y()+a.point2.Y())},""],n),a,0,"length"):"slope"===n.measure&&(c=l(o=t.create("text",[function(){return.25*a.point1.X()+.75*a.point2.X()},function(){return.25*a.point1.Y()+.75*a.point2.Y()},""],n),a,0,"slope")):a.elementClass===i.Z.OBJECT_CLASS_CIRCLE?"radius"===n.measure?c=l(o=t.create("text",[function(){return a.center.X()+.5*a.Radius()},function(){return a.center.Y()},""],n),a,0,"radius"):"area"===n.measure?c=l(o=t.create("text",[function(){return a.center.X()},function(){return a.center.Y()+.5*a.Radius()},""],n),a,0,"area"):"circumference"!==n.measure&&"perimeter"!==n.measure||(c=l(o=t.create("text",[function(){return a.getLabelAnchor()},""],n),a,0,"perimeter")):a.type===i.Z.OBJECT_TYPE_POLYGON?"area"===n.measure?c=l(o=t.create("text",[function(){return a.getTextAnchor()},""],n),a,0,"area"):"perimeter"===n.measure&&(c=l(o=t.create("text",[function(){var t=a.borders.length-1;return t>=0?[.5*(a.borders[t].point1.X()+a.borders[t].point2.X()),.5*(a.borders[t].point1.Y()+a.borders[t].point2.Y())]:a.getTextAnchor()},""],n),a,0,"perimeter")):a.type===i.Z.OBJECT_TYPE_ANGLE&&(c=l(o=t.create("text",[function(){return a.getLabelAnchor()},""],n),a,0,n.measure)),s.Z.exists(o)&&(o.setText(c),a.addChild(o),o.setParents([a])),o},t.Z.registerElement("smartlabel",t.Z.createSmartLabel),t.Z.ForeignObject=function(e,r,o,n,a){this.constructor(e,o,i.Z.OBJECT_TYPE_FOREIGNOBJECT,i.Z.OBJECT_CLASS_OTHER),this.element=this.board.select(o.anchor),this.coordsConstructor(r),this._useUserSize=!1,this.size=[1,1],s.Z.exists(a)&&a.length>0&&(this._useUserSize=!0,this.W=s.Z.createFunction(a[0],this.board,""),this.H=s.Z.createFunction(a[1],this.board,""),this.addParentsFromJCFunctions([this.W,this.H]),this.usrSize=[this.W(),this.H()]),this.content=n,this.elType="foreignobject",this.id=this.board.setId(this,"Im"),this.board.renderer.drawForeignObject(this),this.board.finalizeAdding(this),this.methodMap=t.Z.deepCopy(this.methodMap,{addTransformation:"addTransform",trans:"addTransform",W:"W",Width:"W",H:"H",Height:"H"})},t.Z.ForeignObject.prototype=new I.Z,s.Z.copyPrototypeMethods(t.Z.ForeignObject,j.Z,"coordsConstructor"),t.Z.extend(t.Z.ForeignObject.prototype,{hasPoint:function(t,e){var r,n,a,h,l,c,d,p,f,_=this.transformations.length;return s.Z.isObject(this.evalVisProp("precision"))?(h=this.board._inputDevice,l=this.evalVisProp("precision."+h)):l=this.board.options.precision.hasPoint,0===_?(r=t-this.coords.scrCoords[1],n=this.coords.scrCoords[2]-e,r>=-(a=l)&&r-this.size[0]<=a&&n>=-a&&n-this.size[1]<=a):(d=[(c=(c=new u.Z(i.Z.COORDS_BY_SCREEN,[t,e],this.board)).usrCoords)[0]-this.span[0][0],c[1]-this.span[0][1],c[2]-this.span[0][2]],0<=(p=(f=o.Z.innerProduct)(d,this.span[1]))&&p<=f(this.span[1],this.span[1])&&0<=(p=f(d,this.span[2]))&&p<=f(this.span[2],this.span[2]))},update:function(t){return this.needsUpdate?(this.updateCoords(t),this.updateSize(),this):this},updateRenderer:function(){return this.updateRendererGeneric("updateForeignObject")},updateSize:function(){var t=[0,0];return this._useUserSize?(this.usrSize=[this.W(),this.H()],this.size=[Math.abs(this.usrSize[0]*this.board.unitX),Math.abs(this.usrSize[1]*this.board.unitY)]):this.rendNode.hasChildNodes()&&(t=this.rendNode.childNodes[0].getBoundingClientRect(),this.size=[t.width,t.height]),this},updateSpan:function(){var t,e,i=this.transformations.length,s=[];if(0===i)this.span=[[this.Z(),this.X(),this.Y()],[this.Z(),this.W(),0],[this.Z(),0,this.H()]];else{for(s[0]=[this.Z(),this.X(),this.Y()],s[1]=[this.Z(),this.X()+this.W(),this.Y()],s[2]=[this.Z(),this.X(),this.Y()+this.H()],t=0;t<i;t++)for(e=0;e<3;e++)s[e]=o.Z.matVecMult(this.transformations[t].matrix,s[e]);for(e=0;e<3;e++)s[e][1]/=s[e][0],s[e][2]/=s[e][0],s[e][0]/=s[e][0];for(e=1;e<3;e++)s[e][0]-=s[0][0],s[e][1]-=s[0][1],s[e][2]-=s[0][2];this.span=s}return this},addTransform:function(t){var e;if(s.Z.isArray(t))for(e=0;e<t.length;e++)this.transformations.push(t[e]);else this.transformations.push(t);return this},getParents:function(){var t=[this.url,[this.Z(),this.X(),this.Y()],this.usrSize];return 0!==this.parents.length&&(t=this.parents),t},setSize:function(t,e){return this.W=s.Z.createFunction(t,this.board,""),this.H=s.Z.createFunction(e,this.board,""),this._useUserSize=!0,this.addParentsFromJCFunctions([this.W,this.H]),this},W:function(){},H:function(){}}),t.Z.createForeignObject=function(e,i,r){var o,n,a=i[0],h=i[1],l=[];if(i.length>=2&&(l=i[2]),o=s.Z.copyAttributes(r,e.options,"foreignobject"),!(n=j.Z.create(t.Z.ForeignObject,e,h,o,a,l)))throw new Error("JSXGraph: Can't create foreignObject with parent types '"+typeof i[0]+"' and '"+typeof i[1]+"'.\nPossible parent types: [string, [x, y], [w, h]], [string, [x, y]], [element,transformation]");return n},t.Z.registerElement("foreignobject",t.Z.createForeignObject),t.Z.registerElement("fo",t.Z.createForeignObject);t.Z.ForeignObject;t.Z.extend(v.Z,{axes3d:{axesPosition:"center",xAxis:{visible:!0,point2:{name:"x"},strokeColor:t.Z.palette.red},yAxis:{visible:!0,point2:{name:"y"},strokeColor:t.Z.palette.green},zAxis:{visible:!0,point2:{name:"z"},strokeColor:t.Z.palette.blue},xAxisBorder:{name:"x",visible:"ìnherit",withLabel:!1,label:{position:"50% left",offset:[30,0],fontsize:15},strokeWidth:1,lastArrow:!1,ticks3d:{visible:"ìnherit",label:{anchorX:"middle",anchorY:"middle"}}},yAxisBorder:{name:"y",visible:"ìnherit",withLabel:!1,label:{position:"50% right",offset:[0,-30],fontsize:15},strokeWidth:1,lastArrow:!1,ticks3d:{visible:"ìnherit",label:{anchorX:"middle"}}},zAxisBorder:{name:"z",visible:"ìnherit",withLabel:!1,label:{position:"50% right",offset:[30,0],fontsize:15},strokeWidth:1,lastArrow:!1,ticks3d:{visible:"ìnherit",label:{anchorX:"middle",anchorY:"middle"}}},xPlaneRear:{visible:!0,layer:0,strokeWidth:1,strokeColor:"#dddddd",fillColor:"#dddddd",mesh3d:{layer:1}},yPlaneRear:{visible:!0,strokeWidth:1,strokeColor:"#dddddd",fillColor:"#dddddd",layer:0,mesh3d:{layer:1}},zPlaneRear:{visible:!0,strokeWidth:1,strokeColor:"#dddddd",fillColor:"#dddddd",layer:0,mesh3d:{layer:1}},xPlaneFront:{visible:!1,strokeWidth:1,strokeColor:"#dddddd",fillColor:"#dddddd",layer:0,mesh3d:{layer:1}},yPlaneFront:{visible:!1,strokeWidth:1,strokeColor:"#dddddd",fillColor:"#dddddd",layer:0,mesh3d:{layer:1}},zPlaneFront:{visible:!1,strokeWidth:1,strokeColor:"#dddddd",fillColor:"#dddddd",layer:0,mesh3d:{layer:1}},xPlaneRearYAxis:{visible:"inherit",strokeColor:"#888888",strokeWidth:1},xPlaneRearZAxis:{visible:"inherit",strokeColor:"#888888",strokeWidth:1},xPlaneFrontYAxis:{visible:"inherit",strokeColor:"#888888",strokeWidth:1},xPlaneFrontZAxis:{visible:"inherit",strokeColor:"#888888",strokeWidth:1},yPlaneRearXAxis:{visible:"inherit",strokeColor:"#888888",strokeWidth:1},yPlaneRearZAxis:{visible:"inherit",strokeColor:"#888888",strokeWidth:1},yPlaneFrontXAxis:{visible:"inherit",strokeColor:"#888888",strokeWidth:1},yPlaneFrontZAxis:{visible:"inherit",strokeColor:"#888888",strokeWidth:1},zPlaneRearXAxis:{visible:"inherit",strokeColor:"#888888",strokeWidth:1},zPlaneRearYAxis:{visible:"inherit",strokeColor:"#888888",strokeWidth:1},zPlaneFrontXAxis:{visible:"inherit",strokeColor:"#888888",strokeWidth:1},zPlaneFrontYAxis:{visible:"inherit",strokeColor:"#888888",strokeWidth:1}},axis3d:{highlight:!1,fixed:!0,strokeColor:"black",strokeWidth:1,tabindex:null,point1:{visible:!1,name:"",withLabel:!1},point2:{visible:!1,name:"",withLabel:!1,label:{visible:!0}}},circle3d:{layer:12,point:{visible:!1,name:""},needsRegularUpdate:!0},curve3d:{layer:12,highlight:!1,tabindex:-1,strokeWidth:1,numberPointsHigh:200,needsRegularUpdate:!0},face3d:{transitionProperties:[],layer:12,highlight:!1,tabindex:null,strokeWidth:1,fillColor:t.Z.palette.yellow,fillOpacity:.4,needsRegularUpdate:!0,shader:{enabled:!1,type:"angle",hue:60,saturation:90,minLightness:30,maxLightness:90}},intersectionline3d:{point1:{visible:!1,name:""},point2:{visible:!1,name:""}},line3d:{layer:12,strokeWidth:1,strokeColor:"black",fixed:!0,tabindex:null,needsRegularUpdate:!0,point:{visible:!1,name:""},point1:{visible:!1,name:""},point2:{visible:!1,name:""},straightFirst:!1,straightLast:!1},mesh3d:{layer:12,strokeWidth:1,strokeColor:"#9a9a9a",strokeOpacity:.6,highlight:!1,tabindex:null,needsRegularUpdate:!0,stepWidthU:1,stepWidthV:1},plane3d:{layer:12,strokeWidth:0,strokeColor:"black",strokeOpacity:1,highlight:!1,tabindex:null,needsRegularUpdate:!0,visible:!0,gradient:"linear",gradientSecondColor:"#ffffff",gradientAngle:Math.PI,fillColor:"#a7a7a7",fillOpacity:.6,mesh3d:{visible:"inherit"},threePoints:!1,point:{visible:!1,name:"",fixed:!0},point1:{visible:!1,name:""},point2:{visible:!1,name:""},point3:{visible:!1,name:""}},point3d:{layer:13,infoboxDigits:"auto",strokeWidth:0,gradient:"radial",gradientSecondColor:"#555555",fillColor:"yellow",highlightStrokeColor:"#555555",gradientFX:.7,gradientFY:.3,needsRegularUpdate:!0},polygon3d:{layer:12,highlight:!1,tabindex:-1,strokeWidth:1,fillColor:"none",needsRegularUpdate:!0},polyhedron3d:{fillColorArray:["white","black"],needsRegularUpdate:!0},sphere3d:{layer:12,highlight:!1,strokeWidth:1,strokeColor:"#00ff80",fillColor:"white",gradient:"radial",gradientSecondColor:"#00ff80",gradientFX:.7,gradientFY:.3,fillOpacity:.4,needsRegularUpdate:!0},surface3d:{layer:12,highlight:!1,tabindex:-1,strokeWidth:1,stepsU:30,stepsV:30,needsRegularUpdate:!0},text3d:{withLabel:!1,needsRegularUpdate:!0},ticks3d:{visible:!0,ticksDistance:1,majorHeight:10,minorTicks:0,tickEndings:[0,1],drawLabels:!0,needsRegularUpdate:!0,label:{visible:!0,withLabel:!1}},vectorfield3d:{scale:1,arrowhead:{enabled:!0,size:5,angle:.125*Math.PI},needsRegularUpdate:!0},view3d:{needsRegularUpdate:!0,depthOrder:{enabled:!1,layers:[12,13]},projection:"parallel",verticalDrag:{enabled:!0,key:"shift"},az:{pointer:{enabled:!0,speed:1,outside:!0,button:-1,key:"none"},keyboard:{enabled:!0,step:10,key:"ctrl"},continuous:!0,slider:{visible:"inherit",style:6,point1:{pos:"auto",frozen:!1},point2:{pos:"auto",frozen:!1},min:0,max:2*Math.PI,start:1}},el:{pointer:{enabled:!0,speed:1,outside:!0,button:-1,key:"none"},keyboard:{enabled:!0,step:10,key:"ctrl"},continuous:!0,slider:{visible:"inherit",style:6,point1:{frozen:!1,pos:"auto"},point2:{frozen:!1,pos:"auto"},min:0,max:2*Math.PI,start:.3}},bank:{pointer:{enabled:!0,speed:.08,outside:!0,button:-1,key:"none"},keyboard:{enabled:!0,step:10,key:"ctrl"},continuous:!0,slider:{visible:"inherit",style:6,point1:{frozen:!1,pos:"auto"},point2:{frozen:!1,pos:"auto"},min:-Math.PI,max:Math.PI,start:0}},trackball:{enabled:!1,outside:!0,button:-1,key:"none"},r:"auto",fov:.4*Math.PI,values:[[0,1.57],[.78,.62],[0,0],[5.49,.62],[4.71,0],[3.93,.62],[3.14,0],[2.36,.62],[1.57,1.57]],infobox:{strokeColor:"#000000",fontSize:16,useKatex:!1,useMathjax:!1,intl:{enabled:!0,options:{minimumFractionDigits:2,maximumFractionDigits:3}}},_currentView:-1}});t.Z.Options;t.Z.View3D=function(t,e,r){this.constructor(t,r,i.Z.OBJECT_TYPE_VIEW3D,i.Z.OBJECT_CLASS_3D),this.objects={},this.depthOrdered={},this.elementsByName={},this.defaultAxes=null,this.angles={az:null,el:null,bank:null},this.matrix3DRot=[[1,0,0,0],[0,1,0,0],[0,0,1,0],[0,0,0,1]],this.matrix3DRotShift=[[1,0,0,0],[0,1,0,0],[0,0,1,0],[0,0,0,1]],this.matrix3D=[[1,0,0,0],[0,1,0,0],[0,0,1,0]],this.boxToCam=[],this.llftCorner=e[0],this.size=e[1],this.bbox3D=e[2],this.r=-1,this.focalDist=-1,this.projectionType="parallel",this.trackballEnabled=!1,this.timeoutAzimuth=null,this.zIndexMin=1/0,this.zIndexMax=-1/0,this.id=this.board.setId(this,"V"),this.board.finalizeAdding(this),this.elType="view3d",this.methodMap=s.Z.deepCopy(this.methodMap,{})},t.Z.View3D.prototype=new I.Z,t.Z.extend(t.Z.View3D.prototype,{create:function(t,e,i){var s=[];return t.indexOf("3d")>0&&s.push(this),this.board.create(t,s.concat(e),i)},select:function(t,e){var i,r,o,n,a=t;if(null===a)return a;if(s.Z.isString(a)&&""!==a)s.Z.exists(this.objects[a])?a=this.objects[a]:s.Z.exists(this.elementsByName[a])&&(a=this.elementsByName[a]);else if(!e&&(s.Z.isFunction(a)||s.Z.isObject(a)&&!s.Z.isFunction(a.setAttribute))){for(r={},n=(i=s.Z.filterElements(this.objectsList,a)).length,o=0;o<n;o++)r[i[o].id]=i[o];a=new T(r)}else s.Z.isObject(a)&&s.Z.exists(a.id)&&!s.Z.exists(this.objects[a.id])&&(a=null);return a},setAnglesFromRotation:function(){var t,e,i,s,r,n,a,h=this.matrix3DRot;(t=Math.sqrt(h[1][3]*h[1][3]+h[2][3]*h[2][3]))>o.Z.eps?(e=h[2][3]/t,i=h[1][3]/t):(e=Math.cos(this.angles.bank),i=Math.sin(this.angles.bank)),h=o.Z.matMatMult([[1,0,0,0],[0,e,-i,0],[0,i,e,0],[0,0,0,1]],h),this.angles.bank=Math.atan2(i,e),s=h[2][3],r=h[3][3],h=o.Z.matMatMult([[1,0,0,0],[0,1,0,0],[0,0,s,r],[0,0,-r,s]],h),this.angles.el=Math.atan2(r,s),n=-h[1][1],a=h[3][1],this.angles.az=Math.atan2(a,n),this.angles.az<0&&(this.angles.az+=2*Math.PI),this.setSlidersFromAngles()},anglesHaveMoved:function(){return this._hasMoveAz||this._hasMoveEl||Math.abs(this.angles.az-this.az_slide.Value())>o.Z.eps||Math.abs(this.angles.el-this.el_slide.Value())>o.Z.eps||Math.abs(this.angles.bank-this.bank_slide.Value())>o.Z.eps},getAnglesFromSliders:function(){this.angles.az=this.az_slide.Value(),this.angles.el=this.el_slide.Value(),this.angles.bank=this.bank_slide.Value()},setSlidersFromAngles:function(){this.az_slide.setValue(this.angles.az),this.el_slide.setValue(this.angles.el),this.bank_slide.setValue(this.angles.bank)},getRotationFromAngles:function(){var t,e,i,s,r,n,a=[[1,0,0,0],[0,1,0,0],[0,0,1,0],[0,0,0,1]];return t=this.angles.az,e=this.angles.el,i=this.angles.bank,s=-Math.sin(e),a[1][1]=-Math.cos(t),a[1][2]=Math.sin(t),a[1][3]=0,a[2][1]=s*Math.sin(t),a[2][2]=s*Math.cos(t),a[2][3]=Math.cos(e),a[3][1]=Math.cos(e)*Math.sin(t),a[3][2]=Math.cos(e)*Math.cos(t),a[3][3]=Math.sin(e),r=Math.cos(i),n=Math.sin(i),a=o.Z.matMatMult([[1,0,0,0],[0,r,n,0],[0,-n,r,0],[0,0,0,1]],a)},_projectToSphere:function(t,e,i){var s,r=o.Z.hypot(e,i);return r<.7071067811865475*t?Math.sqrt(t*t-r*r):(s=t/1.414213562373095)*s/r},updateProjectionTrackball:function(t){var e,i,r,n,a,l,c,d,u,p,f,_,m=100,g=[[1,0,0,0],[0,1,0,0],[0,0,1,0],[0,0,0,1]];return s.Z.exists(this._trackball)?((e=this._trackball.dx)*e+(i=this._trackball.dy)*i>o.Z.eps&&(m=.25*(this.size[0]*this.board.unitX+this.size[1]*this.board.unitY),n=[a=this._trackball.x,l=this._trackball.y,this._projectToSphere(m,a,l)],r=[a-=e,l-=i,this._projectToSphere(m,a,l)],_=o.Z.crossProduct(r,n),u=o.Z.hypot(_[0],_[1],_[2]),_[0]/=u,_[1]/=u,_[2]/=u,d=(d=(d=h.Z.distance(n,r,3)/(2*m))>1?1:d)<-1?-1:d,c=2*Math.asin(d),d=1-(p=Math.cos(c)),f=Math.sin(c),g[1][1]=p+_[0]*_[0]*d,g[2][1]=_[1]*_[0]*d+_[2]*f,g[3][1]=_[2]*_[0]*d-_[1]*f,g[1][2]=_[0]*_[1]*d-_[2]*f,g[2][2]=p+_[1]*_[1]*d,g[3][2]=_[2]*_[1]*d+_[0]*f,g[1][3]=_[0]*_[2]*d+_[1]*f,g[2][3]=_[1]*_[2]*d-_[0]*f,g[3][3]=p+_[2]*_[2]*d),g=o.Z.matMatMult(g,this.matrix3DRot)):this.matrix3DRot},updateAngleSliderBounds:function(){var t,e,i,s,r,n,a,h,l,c,d;this.trackballEnabled=this.evalVisProp("trackball.enabled"),this.trackballEnabled?(this.az_slide.setMin(0),this.az_slide.setMax(2*Math.PI),this.el_slide.setMin(-.5*Math.PI),this.el_slide.setMax(.5*Math.PI),this.bank_slide.setMin(-Math.PI),this.bank_slide.setMax(Math.PI)):(this.az_slide.setMin(this.visProp.az.slider.min),this.az_slide.setMax(this.visProp.az.slider.max),this.el_slide.setMin(this.visProp.el.slider.min),this.el_slide.setMax(this.visProp.el.slider.max),this.bank_slide.setMin(this.visProp.bank.slider.min),this.bank_slide.setMax(this.visProp.bank.slider.max)),t=this.az_slide._smax,e=this.az_slide._smin,i=this.el_slide._smax,s=this.el_slide._smin,c=this.bank_slide._smax,d=this.bank_slide._smin,this.trackballEnabled?(r=o.Z.mod(this.angles.el,2*Math.PI),.5*Math.PI<r&&r<1.5*Math.PI&&(this.angles.el=Math.PI-r,this.angles.az=o.Z.wrap(this.angles.az+Math.PI,e,t),this.angles.bank=o.Z.wrap(this.angles.bank+Math.PI,d,c)),this.angles.az=o.Z.wrap(this.angles.az,e,t),this.angles.el=o.Z.wrap(this.angles.el,s,i),this.angles.bank=o.Z.wrap(this.angles.bank,d,c)):(l=function(t){return t<s?s-t:i<t?t-i:0},n=.5*(s+i),a=o.Z.wrap(this.angles.el,n-Math.PI,n+Math.PI),h=o.Z.wrap(Math.PI-this.angles.el,n-Math.PI,n+Math.PI),l(a)<=l(h)?this.angles.el=o.Z.clamp(a,s,i):(this.angles.el=o.Z.clamp(h,s,i),this.angles.az=o.Z.wrap(this.angles.az+Math.PI,e,t),this.angles.bank=o.Z.wrap(this.angles.bank+Math.PI,d,c)),this.angles.az=o.Z.wrapAndClamp(this.angles.az,e,t,2*Math.PI),this.angles.bank=o.Z.wrapAndClamp(this.angles.bank,d,c,2*Math.PI),this.matrix3DRot=this.getRotationFromAngles()),this.setSlidersFromAngles()},_updateCentralProjection:function(){var t,e;return"auto"===(t=this.evalVisProp("r"))&&(t=1.01*o.Z.hypot(this.bbox3D[0][0]-this.bbox3D[0][1],this.bbox3D[1][0]-this.bbox3D[1][1],this.bbox3D[2][0]-this.bbox3D[2][1])),this.boxToCam=this.matrix3DRot.map((function(t){return t.slice()})),this.boxToCam[3][0]=-t,this.focalDist=1/Math.tan(.5*this.evalVisProp("fov")),e=[[0,0,0,-1],[0,this.focalDist,0,0],[0,0,this.focalDist,0],[320/-12,0,0,28/-12]],o.Z.matMatMult(e,this.boxToCam)},update:function(){var t,e,i,r,n,a=this.r,h=[[1,0,0,0],[0,-a,0,0],[0,0,-a,0],[0,0,0,1]];if(!(s.Z.exists(this.el_slide)&&s.Z.exists(this.az_slide)&&s.Z.exists(this.bank_slide)&&this.needsUpdate))return this.needsUpdate=!1,this;if(t=[[1,0,0],[0,1,0],[0,0,1]],this.projectionType=this.evalVisProp("projection").toLowerCase(),this.trackballEnabled!==this.evalVisProp("trackball.enabled")&&this.updateAngleSliderBounds(),this._hasMoveTrackball?(this.matrix3DRot=this.updateProjectionTrackball(),this.setAnglesFromRotation()):this.anglesHaveMoved()&&(this.getAnglesFromSliders(),this.matrix3DRot=this.getRotationFromAngles()),this.shift=[[1,0,0,0],[-.5*(this.bbox3D[0][0]+this.bbox3D[0][1]),1,0,0],[-.5*(this.bbox3D[1][0]+this.bbox3D[1][1]),0,1,0],[-.5*(this.bbox3D[2][0]+this.bbox3D[2][1]),0,0,1]],"central"===this.projectionType)i=.8,t[1][1]=this.size[0]/i,t[2][2]=this.size[1]/i,t[1][0]=this.llftCorner[0]+.5*t[1][1]*i,t[2][0]=this.llftCorner[1]+.5*t[2][2]*i,this.viewPortTransform=t,e=this._updateCentralProjection(),this.matrix3D=o.Z.matMatMult(e,this.shift);else r=this.bbox3D[0][1]-this.bbox3D[0][0],n=this.bbox3D[1][1]-this.bbox3D[1][0],t[1][1]=this.size[0]/r,t[2][2]=this.size[1]/n,t[1][0]=this.llftCorner[0]+.5*t[1][1]*r,t[2][0]=this.llftCorner[1]+.5*t[2][2]*n,this.matrix3D=o.Z.matMatMult(t,o.Z.matMatMult(o.Z.matMatMult(this.matrix3DRot,h),this.shift).slice(0,3));return this.matrix3DRotShift=o.Z.matMatMult(this.matrix3DRot,this.shift),this},compareDepth:function(t,e){return t.zIndex-e.zIndex},updateZIndices:function(){var t,e;for(t in this.objects)this.objects.hasOwnProperty(t)&&((e=this.objects[t]).type===i.Z.OBJECT_TYPE_LINE3D||e.type===i.Z.OBJECT_TYPE_POLYGON3D)&&s.Z.exists(e.element2D)&&e.element2D.evalVisProp("visible")&&e.updateZIndex()},updateShaders:function(){var t,e,i;for(t in this.objects)this.objects.hasOwnProperty(t)&&(e=this.objects[t],s.Z.exists(e.shader)&&((i=e.shader())<this.zIndexMin?this.zIndexMin=i:i>this.zIndexMax&&(this.zIndexMax=i)))},updateDepthOrdering:function(){var t,e,r,o,n,a,h;for(a=this.evalVisProp("depthorder.layers"),r=0;r<a.length;r++)this.depthOrdered[a[r]]=[];for(t in this.objects)this.objects.hasOwnProperty(t)&&((e=this.objects[t]).type===i.Z.OBJECT_TYPE_FACE3D||e.type===i.Z.OBJECT_TYPE_LINE3D||e.type===i.Z.OBJECT_TYPE_POINT3D||e.type===i.Z.OBJECT_TYPE_POLYGON3D)&&s.Z.exists(e.element2D)&&e.element2D.evalVisProp("visible")&&(h=e.element2D.evalVisProp("layer"),a.indexOf(h)>=0&&this.depthOrdered[h].push(e));if(this.board.renderer&&"svg"===this.board.renderer.type)for(r=0;r<a.length;r++)for(h=a[r],this.depthOrdered[h].sort(this.compareDepth.bind(this)),n=this.depthOrdered[h],o=0;o<n.length;o++)this.board.renderer.setLayer(n[o].element2D,h);return this},updateRenderer:function(){return this.needsUpdate?(this.depthOrdered={},void 0!==this.shift&&this.evalVisProp("depthorder.enabled")&&(this.updateZIndices(),this.updateShaders(),this.board.renderer&&"svg"===this.board.renderer.type&&this.updateDepthOrdering()),this.needsUpdate=!1,this):this},removeObject:function(e,i){var r,o;if(s.Z.isArray(e)){for(r=0;r<e.length;r++)this.removeObject(e[r]);return this}if(e=this.select(e),!s.Z.exists(e)||s.Z.isString(e))return this;try{for(o in e.childElements)e.childElements.hasOwnProperty(o)&&this.removeObject(e.childElements[o]);delete this.objects[e.id]}catch(i){t.Z.debug("View3D "+e.id+": Could not be removed: "+i)}return this.board.removeObject(e,i),this},worldToFocal:function(t,e=!0){var i,s=o.Z.matVecMult(this.boxToCam,o.Z.matVecMult(this.shift,t));if(s[3]-=s[0]*this.focalDist,e)return s;for(i=1;i<4;i++)s[i]/=s[0];return s.slice(1,4)},project3DTo2D:function(t,e,i){var s,r;return 3===arguments.length?s=[1,t,e,i]:3===t.length?(s=t.slice()).unshift(1):s=t,r=o.Z.matVecMult(this.matrix3D,s),"central"===this.projectionType?(r[1]/=r[0],r[2]/=r[0],r[3]/=r[0],r[0]/=r[0],o.Z.matVecMult(this.viewPortTransform,r.slice(0,3))):r},_getW0:function(t,e,i){var s,r=o.Z.inverse(t),n=r[0][0]+e[1]*r[0][1]+e[2]*r[0][2],a=r[3][0]+e[1]*r[3][1]+e[2]*r[3][2];return s=i*r[0][3]-r[3][3],[1/((a*r[0][3]-n*r[3][3])/s),(a-n*i)/s]},project2DTo3DPlane:function(t,e,i){var s,r,n,a,h,l,c,d,u=i.slice(1)||[0,0,0],p=e.slice(1);if(a=o.Z.norm(p,3),n=o.Z.innerProduct(u,p,3)/a,"parallel"===this.projectionType){(s=this.matrix3D.slice(0,3)).push([0,p[0],p[1],p[2]]),(r=t.coords.usrCoords.slice()).push(n);try{1===s[2][3]&&(s[2][1]=s[2][2]=.001*o.Z.eps),h=o.Z.Numerics.Gauss(s,r)}catch(t){h=[0,NaN,NaN,NaN]}}else{s=this.matrix3D,r=t.coords.usrCoords.slice(),l=o.Z.Numerics.Gauss(this.viewPortTransform,r),c=(d=this._getW0(s,l,n))[0],r=[l[0]*c,l[1]*c,l[2]*c,d[1]*c];try{1===s[2][3]&&(s[2][1]=s[2][2]=.001*o.Z.eps),(h=o.Z.Numerics.Gauss(s,r))[1]/=h[0],h[2]/=h[0],h[3]/=h[0],h[0]/=h[0]}catch(t){h=[0,NaN,NaN,NaN]}}return h},projectScreenToSegment:function(t,e,i){var s,r,n,a,h,l,c,d=this.project3DTo2D(e).slice(1,3),u=this.project3DTo2D(i).slice(1,3),p=[u[0]-d[0],u[1]-d[1]],f=o.Z.innerProduct(p,p),_=[t[0]-d[0],t[1]-d[1]],m=o.Z.innerProduct(_,p)/f;return"central"===this.projectionType?(s=[.5*(e[0]+i[0]),.5*(e[1]+i[1]),.5*(e[2]+i[2])],n=[(r=this.project3DTo2D(s).slice(1,3))[0]-d[0],r[1]-d[1]],h=(1-(a=o.Z.innerProduct(n,p)/f))*m/((1-2*a)*m+a)):h=m,[1,(c=1-(l=Math.min(Math.max(h,0),1)))*e[0]+l*i[0],c*e[1]+l*i[1],c*e[2]+l*i[2]]},project2DTo3DVertical:function(t,e){var i=t.coords.usrCoords.slice(1,3),s=[e[1],e[2],this.bbox3D[2][0]],r=[e[1],e[2],this.bbox3D[2][1]];return this.projectScreenToSegment(i,s,r)},project3DToCube:function(t){var e=this.bbox3D,i=!1;return t[1]<e[0][0]&&(t[1]=e[0][0],i=!0),t[1]>e[0][1]&&(t[1]=e[0][1],i=!0),t[2]<e[1][0]&&(t[2]=e[1][0],i=!0),t[2]>e[1][1]&&(t[2]=e[1][1],i=!0),t[3]<=e[2][0]&&(t[3]=e[2][0],i=!0),t[3]>=e[2][1]&&(t[3]=e[2][1],i=!0),[t,i]},intersectionLineCube:function(t,e,i){var s,r,o,n,a;for(a=3===e.length?e:e.slice(1),s=i,r=0;r<3;r++)0!==a[r]&&(o=(this.bbox3D[r][0]-t[r+1])/a[r],n=(this.bbox3D[r][1]-t[r+1])/a[r],s=i<0?Math.max(s,Math.min(o,n)):Math.min(s,Math.max(o,n)));return s},isInCube:function(t,e){var i;if(4===t.length){if(0===t[0])return!1;i=t.slice(1)}return i[0]>this.bbox3D[0][0]-o.Z.eps&&i[0]<this.bbox3D[0][1]+o.Z.eps&&i[1]>this.bbox3D[1][0]-o.Z.eps&&i[1]<this.bbox3D[1][1]+o.Z.eps&&i[2]>this.bbox3D[2][0]-o.Z.eps&&i[2]<this.bbox3D[2][1]+o.Z.eps},intersectionPlanePlane:function(t,e,i){var s,r,n,a,h,l=[!1,!1];return i=i||e.d,(a=o.Z.crossProduct(t.normal.slice(1),e.normal.slice(1))).unshift(0),s=o.Z.Geometry.meet3Planes(t.normal,t.d,e.normal,i,a,0),h=o.Z.Geometry.meetPlanePlane(t.vec1,t.vec2,e.vec1,e.vec2),n=this.intersectionLineCube(s,h,1/0),r=o.Z.axpy(n,h,s),this.isInCube(r)&&(l[0]=r),n=this.intersectionLineCube(s,h,-1/0),r=o.Z.axpy(n,h,s),this.isInCube(r)&&(l[1]=r),l},intersectionPlaneFace:function(t,e){var i,s,r,n,a,l,c,u,p,f,_,m,g,b,v,Z,C=[],y=[],P=[];for((Z=o.Z.crossProduct(t.normal.slice(1),e.normal.slice(1))).unshift(0),r=h.Z.meet3Planes(t.normal,t.d,e.normal,e.d,Z,0),b=h.Z.meetPlanePlane(t.vec1,t.vec2,e.vec1,e.vec2),u=e.polyhedron.faces[e.faceNumber],n=e.polyhedron.coords,p=u.length,i=1;i<=p;i++)a=n[u[i-1]],v=[0,(l=n[u[i%p]])[1]-a[1],l[2]-a[2],l[3]-a[3]],f=Math.random(),_=Math.random(),m=Math.random(),g=Math.random(),y=[[f*b[1]+_*b[3],f*-v[1]+_*-v[3]],[m*b[2]+g*b[3],m*-v[2]+g*-v[3]]],P=[f*(a[1]-r[1])+_*(a[3]-r[3]),m*(a[2]-r[2])+g*(a[3]-r[3])],(s=d.Z.Gauss(y,P)[1])>-o.Z.eps&&s<1+o.Z.eps&&(c=[1,a[1]+s*v[1],a[2]+s*v[2],a[3]+s*v[3]],C.push(c));return C},intersectionPlanePolyhedron:function(t,e){var i,s,r,o,n,a,l=1e-12,c=[],d=[],u=[],p=[];for(i=0;i<e.numberFaces;i++)if(!(e.def.faces[i].length<3||(r=this.intersectionPlaneFace(t,e.faces[i])).length<2||4!==r[0].length||4!==r[1].length)){for(s=0;s<c.length&&!(h.Z.distance(r[0],c[s][0],4)<l&&h.Z.distance(r[1],c[s][1],4)<l||h.Z.distance(r[0],c[s][1],4)<l&&h.Z.distance(r[1],c[s][0],4)<l);s++);s===c.length&&c.push(r.slice())}if(0===c.length)return{X:d,Y:u,Z:p};n=0,i=0;do{for(4===(o=c[n][i]).length&&(d.push(o[1]),u.push(o[2]),p.push(o[3])),i=(i+1)%2,o=c[n][i],a=n,s=0;s<c.length;s++){if(s!==n&&h.Z.distance(o,c[s][0])<l){n=s,i=0;break}if(s!==n&&h.Z.distance(o,c[s][1])<l){n=s,i=1;break}}if(n===a){console.log("Error face3d intersection update: did not find next",n,i);break}}while(0!==n);return d.push(d[0]),u.push(u[0]),p.push(p[0]),{X:d,Y:u,Z:p}},getMesh:function(t,e,i){var r,o,n,a,h,l,c,d=[0,0,0],u=s.Z.evaluate(e[2]),p=s.Z.evaluate(i[2]),f=[],_=[];for(l=(s.Z.evaluate(e[1])-s.Z.evaluate(e[0]))/u,c=(s.Z.evaluate(i[1])-s.Z.evaluate(i[0]))/p,r=0;r<=u;r++){for(n=e[0]+l*r,o=0;o<=p;o++)a=i[0]+c*o,d=s.Z.isFunction(t)?t(n,a):[t[0](n,a),t[1](n,a),t[2](n,a)],h=this.project3DTo2D(d),f.push(h[1]),_.push(h[2]);f.push(NaN),_.push(NaN)}for(o=0;o<=p;o++){for(a=i[0]+c*o,r=0;r<=u;r++)n=e[0]+l*r,d=s.Z.isFunction(t)?t(n,a):[t[0](n,a),t[1](n,a),t[2](n,a)],h=this.project3DTo2D(d),f.push(h[1]),_.push(h[2]);f.push(NaN),_.push(NaN)}return[f,_]},animateAzimuth:function(){var t=this.az_slide._smin,e=this.az_slide._smax-t,i=this.az_slide.Value()+.1;this.az_slide.position=(i-t)/e,this.az_slide.position>1&&(this.az_slide.position=0),this.board._change3DView=!0,this.board.update(),this.board._change3DView=!1,this.timeoutAzimuth=setTimeout(function(){this.animateAzimuth()}.bind(this),200)},stopAzimuth:function(){clearTimeout(this.timeoutAzimuth),this.timeoutAzimuth=null},isVerticalDrag:function(){var t=this.board;return!!this.evalVisProp("verticaldrag.enabled")&&t["_"+this.evalVisProp("verticaldrag.key")+"Key"]},setView:function(t,e,i){return i=i||this.r,this.az_slide.setValue(t),this.el_slide.setValue(e),this.r=i,this.board.update(),this},nextView:function(){var t=this.evalVisProp("values"),e=this.visProp._currentview;return e=(e+1)%t.length,this.setCurrentView(e),this},previousView:function(){var t=this.evalVisProp("values"),e=this.visProp._currentview;return e=(e+t.length-1)%t.length,this.setCurrentView(e),this},setCurrentView:function(t){var e=this.evalVisProp("values");return(t<0||t>=e.length)&&(t=(t%e.length+e.length)%e.length),this.setView(e[t][0],e[t][1],e[t][2]),this.visProp._currentview=t,this},_azEventHandler:function(t){var e=this.az_slide._smax,i=this.az_slide._smin,s=(e-i)/this.board.canvasWidth*this.evalVisProp("az.pointer.speed"),r=t.movementX,n=this.az_slide.Value(),a=this.el_slide.Value();return this.board.mode===this.board.BOARD_MODE_DRAG||(this.evalVisProp("az.keyboard.enabled")&&("ArrowRight"===t.key?n+=this.evalVisProp("az.keyboard.step")*Math.PI/180:"ArrowLeft"===t.key&&(n-=this.evalVisProp("az.keyboard.step")*Math.PI/180)),this.evalVisProp("az.pointer.enabled")&&0!==r&&null==t.key&&(n+=r*s),this.evalVisProp("az.continuous")?n=o.Z.wrap(n,i,e):n>0?n=Math.min(e,n):n<0&&(n=Math.max(i,n)),this.setView(n,a)),this},_elEventHandler:function(t){var e=this.el_slide._smax,i=this.el_slide._smin,s=(e-i)/this.board.canvasHeight*this.evalVisProp("el.pointer.speed"),r=t.movementY,n=this.az_slide.Value(),a=this.el_slide.Value();return this.board.mode===this.board.BOARD_MODE_DRAG||(this.evalVisProp("el.keyboard.enabled")&&("ArrowUp"===t.key?a-=this.evalVisProp("el.keyboard.step")*Math.PI/180:"ArrowDown"===t.key&&(a+=this.evalVisProp("el.keyboard.step")*Math.PI/180)),this.evalVisProp("el.pointer.enabled")&&0!==r&&null==t.key&&(a+=r*s),this.evalVisProp("el.continuous")&&!this.trackballEnabled?a=o.Z.wrap(a,i,e):a>0?a=Math.min(e,a):a<0&&(a=Math.max(i,a)),this.setView(n,a)),this},_bankEventHandler:function(t){var e,i=this.bank_slide._smax,s=this.bank_slide._smin,r=t.deltaY,n=this.bank_slide.Value();return this.board.mode===this.board.BOARD_MODE_DRAG||(this.evalVisProp("bank.keyboard.enabled")&&(e=this.evalVisProp("bank.keyboard.step")*Math.PI/180,"."===t.key||"<"===t.key?n-=e:","!==t.key&&">"!==t.key||(n+=e)),this.evalVisProp("bank.pointer.enabled")&&0!==r&&null==t.key&&(n+=r*((i-s)/this.board.canvasHeight*this.evalVisProp("bank.pointer.speed")),t.preventDefault()),n=this.evalVisProp("bank.continuous")?o.Z.wrap(n,s,i):o.Z.clamp(n,s,i),this.bank_slide.setValue(n),this.board.update()),this},_trackballHandler:function(t){var e,s,r,o=this.board.getMousePosition(t);return r=new u.Z(i.Z.COORDS_BY_USER,[this.llftCorner[0]+.5*this.size[0],this.llftCorner[1]+.5*this.size[1]],this.board),e=o[0]-r.scrCoords[1],s=o[1]-r.scrCoords[2],this._trackball={dx:t.movementX,dy:-t.movementY,x:e,y:-s},this.board.update(),this},pointerDownHandler:function(t){var i,s,r;this._hasMoveAz=!1,this._hasMoveEl=!1,this._hasMoveBank=!1,this._hasMoveTrackball=!1,this.board.mode===this.board.BOARD_MODE_NONE&&(this.board._change3DView=!0,this.evalVisProp("trackball.enabled")?(i=this.evalVisProp("trackball.button"),s=this.evalVisProp("trackball.key"),-1!==i&&i!==t.button||!("none"===s||s.indexOf("shift")>-1&&t.shiftKey||s.indexOf("ctrl")>-1&&t.ctrlKey)||(r=this.evalVisProp("trackball.outside")?document:this.board.containerObj,e.Z.addEvent(r,"pointermove",this._trackballHandler,this),this._hasMoveTrackball=!0)):(this.evalVisProp("az.pointer.enabled")&&(i=this.evalVisProp("az.pointer.button"),s=this.evalVisProp("az.pointer.key"),-1!==i&&i!==t.button||!("none"===s||s.indexOf("shift")>-1&&t.shiftKey||s.indexOf("ctrl")>-1&&t.ctrlKey)||(r=this.evalVisProp("az.pointer.outside")?document:this.board.containerObj,e.Z.addEvent(r,"pointermove",this._azEventHandler,this),this._hasMoveAz=!0)),this.evalVisProp("el.pointer.enabled")&&(i=this.evalVisProp("el.pointer.button"),s=this.evalVisProp("el.pointer.key"),-1!==i&&i!==t.button||!("none"===s||s.indexOf("shift")>-1&&t.shiftKey||s.indexOf("ctrl")>-1&&t.ctrlKey)||(r=this.evalVisProp("el.pointer.outside")?document:this.board.containerObj,e.Z.addEvent(r,"pointermove",this._elEventHandler,this),this._hasMoveEl=!0)),this.evalVisProp("bank.pointer.enabled")&&(i=this.evalVisProp("bank.pointer.button"),s=this.evalVisProp("bank.pointer.key"),-1!==i&&i!==t.button||!("none"===s||s.indexOf("shift")>-1&&t.shiftKey||s.indexOf("ctrl")>-1&&t.ctrlKey)||(r=this.evalVisProp("bank.pointer.outside")?document:this.board.containerObj,e.Z.addEvent(r,"wheel",this._bankEventHandler,this,{passive:!1}),this._hasMoveBank=!0))),e.Z.addEvent(document,"pointerup",this.pointerUpHandler,this))},pointerUpHandler:function(t){var i;this._hasMoveAz&&(i=this.evalVisProp("az.pointer.outside")?document:this.board.containerObj,e.Z.removeEvent(i,"pointermove",this._azEventHandler,this),this._hasMoveAz=!1),this._hasMoveEl&&(i=this.evalVisProp("el.pointer.outside")?document:this.board.containerObj,e.Z.removeEvent(i,"pointermove",this._elEventHandler,this),this._hasMoveEl=!1),this._hasMoveBank&&(i=this.evalVisProp("bank.pointer.outside")?document:this.board.containerObj,e.Z.removeEvent(i,"wheel",this._bankEventHandler,this),this._hasMoveBank=!1),this._hasMoveTrackball&&(i=this.evalVisProp("trackball.outside")?document:this.board.containerObj,e.Z.removeEvent(i,"pointermove",this._trackballHandler,this),this._hasMoveTrackball=!1),e.Z.removeEvent(document,"pointerup",this.pointerUpHandler,this),this.board._change3DView=!1}}),t.Z.createView3D=function(i,r,n){var a,h,l,c,d,u,p,f,_,m,g,b,v=r[0],Z=r[1];return h=s.Z.copyAttributes(n,i.options,"view3d"),(a=new t.Z.View3D(i,r,h)).defaultAxes=a.create("axes3d",[],h),u=v[0],p=v[1],f=Z[0],_=Z[1],(l=s.Z.copyAttributes(h,i.options,"view3d","az","slider")).name="az",(c=s.Z.copyAttributes(h,i.options,"view3d","el","slider")).name="el",(d=s.Z.copyAttributes(h,i.options,"view3d","bank","slider")).name="bank",b=s.Z.evaluate(l.point1.pos),m=s.Z.isArray(b)?b:[u-1,p-2],b=s.Z.evaluate(l.point2.pos),g=s.Z.isArray(b)?b:[u+f+1,p-2],a.az_slide=i.create("slider",[m,g,[s.Z.evaluate(l.min),s.Z.evaluate(l.start),s.Z.evaluate(l.max)]],l),a.inherits.push(a.az_slide),a.az_slide.elType="view3d_slider",b=s.Z.evaluate(c.point1.pos),m=s.Z.isArray(b)?b:[u-1,p],b=s.Z.evaluate(c.point2.pos),g=s.Z.isArray(b)?b:[u-1,p+_],a.el_slide=i.create("slider",[m,g,[s.Z.evaluate(c.min),s.Z.evaluate(c.start),s.Z.evaluate(c.max)]],c),a.inherits.push(a.el_slide),a.el_slide.elType="view3d_slider",b=s.Z.evaluate(d.point1.pos),m=s.Z.isArray(b)?b:[u-1,p+_+2],b=s.Z.evaluate(d.point2.pos),g=s.Z.isArray(b)?b:[u+f+1,p+_+2],a.bank_slide=i.create("slider",[m,g,[s.Z.evaluate(d.min),s.Z.evaluate(d.start),s.Z.evaluate(d.max)]],d),a.inherits.push(a.bank_slide),a.bank_slide.elType="view3d_slider",a.board.infobox.visProp=s.Z.merge(a.board.infobox.visProp,h.infobox),a.board.highlightInfobox=function(t,e,i){var r,n,h,l,c,d,u="",p=i.board,f=null;for(this.mode===this.BOARD_MODE_DRAG&&(u=a.isVerticalDrag()?'<span style="color:black; font-size:200%">⇅ &nbsp;</span>':'<span style="color:black; font-size:200%">⇄ &nbsp;</span>'),n=0;n<i.parents.length&&!(f=p.objects[i.parents[n]]).is3D;n++);if(f&&s.Z.exists(f.element2D)){if(l=[1,0,0,f.coords[3]],a._w0=o.Z.innerProduct(a.matrix3D[0],l,4),h=a.project2DTo3DPlane(f.element2D,[1,0,0,1],l),!a.isInCube(h))return void a.board.highlightCustomInfobox("",f);r=f.evalVisProp("infoboxdigits"),d=a.board.infobox,c="auto"===r?d.useLocale()?[u,"(",d.formatNumberLocale(f.X())," | ",d.formatNumberLocale(f.Y())," | ",d.formatNumberLocale(f.Z()),")"]:[u,"(",s.Z.autoDigits(f.X())," | ",s.Z.autoDigits(f.Y())," | ",s.Z.autoDigits(f.Z()),")"]:d.useLocale()?[u,"(",d.formatNumberLocale(f.X(),r)," | ",d.formatNumberLocale(f.Y(),r)," | ",d.formatNumberLocale(f.Z(),r),")"]:[u,"(",s.Z.toFixed(f.X(),r)," | ",s.Z.toFixed(f.Y(),r)," | ",s.Z.toFixed(f.Z(),r),")"],a.board.highlightCustomInfobox(c.join(""),f)}else a.board.highlightCustomInfobox("("+t+", "+e+")",i)},a.BOARD_MODE_NONE=0,e.Z.addEvent(i.containerObj,"keydown",(function(t){var e,i=!1;!a.evalVisProp("el.keyboard.enabled")||"ArrowUp"!==t.key&&"ArrowDown"!==t.key||("none"===(e=a.evalVisProp("el.keyboard.key"))||e.indexOf("shift")>-1&&t.shiftKey||e.indexOf("ctrl")>-1&&t.ctrlKey)&&(a._elEventHandler(t),i=!0),!a.evalVisProp("az.keyboard.enabled")||"ArrowLeft"!==t.key&&"ArrowRight"!==t.key||("none"===(e=a.evalVisProp("az.keyboard.key"))||e.indexOf("shift")>-1&&t.shiftKey||e.indexOf("ctrl")>-1&&t.ctrlKey)&&(a._azEventHandler(t),i=!0),!a.evalVisProp("bank.keyboard.enabled")||","!==t.key&&"<"!==t.key&&"."!==t.key&&">"!==t.key||("none"===(e=a.evalVisProp("bank.keyboard.key"))||e.indexOf("shift")>-1&&t.shiftKey||e.indexOf("ctrl")>-1&&t.ctrlKey)&&(a._bankEventHandler(t),i=!0),"PageUp"===t.key?(a.nextView(),i=!0):"PageDown"===t.key&&(a.previousView(),i=!0),i&&t.preventDefault(),this.board._change3DView=!1}),a),e.Z.addEvent(i.containerObj,"pointerdown",a.pointerDownHandler,a),a.getAnglesFromSliders(),a.matrix3DRot=a.getRotationFromAngles(),a.updateAngleSliderBounds(),a.board.update(),a},t.Z.registerElement("view3d",t.Z.createView3D);t.Z.View3D;t.Z.GeometryElement3D=function(t,e){this.elType=e,this.view=t,this.id=this.view.board.setId(this,e),this.element2D=null,this.is3D=!0,this.zIndex=0,this.view.objects[this.id]=this,""!==this.name&&(this.view.elementsByName[this.name]=this)},t.Z.extend(t.Z.GeometryElement3D.prototype,{setAttr2D:function(t){var e=t;return e.name=this.name,e.element3d=this,e.id=null,e},setAttribute:function(e){var i,r,o,n,a,h={};for(i=0;i<arguments.length;i++)n=arguments[i],s.Z.isString(n)?(a=n.split(":"),h[s.Z.trim(a[0])]=s.Z.trim(a[1])):s.Z.isArray(n)?h[n[0]]=n[1]:t.Z.extend(h,n);for(i in h)if(h.hasOwnProperty(i))switch(r=i.replace(/\s+/g,"").toLowerCase(),o=h[i],r){case"numberpointshigh":case"stepsu":case"stepsv":s.Z.exists(this.visProp[r])&&(!t.Z.Validator[r]||t.Z.Validator[r]&&t.Z.Validator[r](o)||t.Z.Validator[r]&&s.Z.isFunction(o)&&t.Z.Validator[r](o()))&&(o=(!o.toLowerCase||"false"!==o.toLowerCase())&&o,this._set(r,o));break;default:s.Z.exists(this.element2D)&&this.element2D.setAttribute(h)}},getAttribute:function(t){var e;switch(t=t.toLowerCase()){case"numberpointshigh":case"stepsu":case"stepsv":e=this.visProp[t];break;default:s.Z.exists(this.element2D)&&(e=this.element2D.getAttribute(t))}return e},getAttributes:function(){var t,e,i={},r=["numberpointshigh","stepsu","stepsv"],o=r.length;for(s.Z.exists(this.element2D)&&(i=s.Z.merge(this.element2D.getAttributes())),t=0;t<o;t++)e=r[t],s.Z.exists(this.visProp[e])&&(i[e]=this.visProp[e]);return i},addTransformGeneric:function(t,e){var i,r=s.Z.isArray(e)?e:[e],o=r.length;for(0===this.transformations.length&&(this.baseElement=t),i=0;i<o;i++)this.transformations.push(r[i]);return this},setPosition2D:function(t){},projectCoords:function(t,e){},remove:function(){}});t.Z.GeometryElement3D;t.Z.createAxes3D=function(t,e,i){var r,o,n,a,h,l,c,d,u,p,f,_,m,g,b,v,Z,C,y=e[0],P=["x","y","z"],E="Axis",x=["Rear","Front"],O=[0,0,0],M=[0,0,0],w={};if(s.Z.exists(y.bbox3D))for(r=0;r<P.length;r++)O[r]=y.bbox3D[r][0],M[r]=y.bbox3D[r][1];else for(r=0;r<P.length;r++)O[r]=e[1][r],M[r]=e[2][1];for(c=(l=s.Z.copyAttributes(i,t.options,"axes3d")).axesposition,r=0;r<P.length;r++)v=(d=P[r])+E,"center"===c?(p=[0,0,0],(f=[0,0,0])[r]=M[r],w[v]=y.create("axis3d",[p,f],l[v.toLowerCase()]),w[v].view=y):"border"===c&&(v+="Border",p=O.slice(),f=M.slice(),"z"===d?(p[1]=M[1],f[0]=O[0]):"x"===d?(p=[O[0],M[1],O[2]],f=[M[0],M[1],O[2]]):(p=[M[0],O[1],O[2]],f=[M[0],M[1],O[2]]),f[r]=M[r],w[v]=y.create("axis3d",[p,f],l[v.toLowerCase()]),w[v].view=y,(C=l[v.toLowerCase()].ticks3d).element3d=!0,u=M[r]-O[r],w[v+"Ticks"]="x"===d?y.create("ticks3d",[p,[1,0,0],u,[0,1,0]],C):"y"===d?y.create("ticks3d",[p,[0,1,0],u,[1,0,0]],C):y.create("ticks3d",[p,[0,0,1],u,[0,1,0]],C),w[v+"Ticks"].view=y);for("center"===c?(w.O=y.create("intersection",[w[P[0]+E],w[P[1]+E]],{name:"",visible:!1,withLabel:!1}),w.O.view=y):w.O=null,r=0;r<P.length;r++)for(a=(r+1)%3,h=(r+2)%3,d=P[r],o=0;o<x.length;o++)v=d+"Plane"+x[o],(p=[0,0,0])[r]=0===o?O[r]:M[r],m=[0,0,0],(_=[0,0,0])[a]=1,m[h]=1,g=[O[a],M[a]],b=[O[h],M[h]],l=s.Z.copyAttributes(i,t.options,"axes3d",v),w[v]=y.create("plane3d",[p,_,m,g,b],l),w[v].elType="axisplane3d";for(r=0;r<P.length;r++)for(d=P[r],o=0;o<x.length;o++)for(n=1;n<=2;n++)v=d+"Plane"+x[o]+P[a=(r+n)%3].toUpperCase()+"Axis",Z=d+"Plane"+x[o],(p=[0,0,0])[r]=(f=[0,0,0])[r]=0===o?O[r]:M[r],p[a]=O[a],f[a]=M[a],l=s.Z.copyAttributes(i,t.options,"axes3d",v),w[v]=y.create("axis3d",[p,f],l),w[v].view=y,w[Z].addChild(w[v]),w[Z].element2D.inherits.push(w[v]);return w},t.Z.registerElement("axes3d",t.Z.createAxes3D),t.Z.createAxis3D=function(t,e,i){var r,o,n,a,h,l,c=e[0],d=e[1],u=e[2];return(r=s.Z.copyAttributes(i.point1,t.options,"axis3d","point1")).element3d=!0,o=c.create("point",[(a=d[0],h=d[1],l=d[2],function(){return c.project3DTo2D(a,h,l)[1]}),function(t,e,i){return function(){return c.project3DTo2D(t,e,i)[2]}}(d[0],d[1],d[2])],r),(r=s.Z.copyAttributes(i.point2,t.options,"axis3d","point2")).element3d=!0,n=c.create("point",[function(t,e,i){return function(){return c.project3DTo2D(t,e,i)[1]}}(u[0],u[1],u[2]),function(t,e,i){return function(){return c.project3DTo2D(t,e,i)[2]}}(u[0],u[1],u[2])],r),(r=s.Z.copyAttributes(i,t.options,"axis3d")).element3d=!0,c.create("arrow",[o,n],r)},t.Z.registerElement("axis3d",t.Z.createAxis3D),t.Z.createMesh3D=function(e,i,r){var o,n,a=i[0];return(o=s.Z.copyAttributes(r,e.options,"mesh3d")).element3d=!0,(n=a.create("curve",[[],[]],o)).point=i[1],n.direction1=i[2],n.direction2=i[3],n.range1=i[4],n.range2=i[5],n.updateDataArray=function(){var e,i,r,o,n=s.Z.evaluate(this.range1),h=s.Z.evaluate(this.range2),l=n[0],c=n[1],d=h[0],u=h[1],p=[0,0,0],f=[0,0,0],_=this.evalVisProp("stepwidthu"),m=this.evalVisProp("stepwidthv"),g=[0,0,0];if(this.dataX=[],this.dataY=[],s.Z.isFunction(this.point))g=this.point();else if(s.Z.isPoint3D(this.point))g=this.point.coords;else for(o=0;o<this.point.length;o++)g[o]=s.Z.evaluate(this.point[o]);if(s.Z.isFunction(this.direction1))p=s.Z.evaluate(this.direction1);else for(o=0;o<this.direction1.length;o++)p[o]=s.Z.evaluate(this.direction1[o]);if(s.Z.isFunction(this.direction2))f=s.Z.evaluate(this.direction2);else for(o=0;o<this.direction2.length;o++)f[o]=s.Z.evaluate(this.direction2[o]);for(4===g.length&&(g=g.slice(1)),4===p.length&&(p=p.slice(1)),4===f.length&&(f=f.slice(1)),e=t.Z.Math.norm(p,3),i=t.Z.Math.norm(f,3),o=0;o<3;o++)p[o]/=e,f[o]/=i;r=a.getMesh([function(t,e){return g[0]+t*p[0]+e*f[0]},function(t,e){return g[1]+t*p[1]+e*f[1]},function(t,e){return g[2]+t*p[2]+e*f[2]}],[Math.ceil(l),Math.floor(c),(Math.ceil(c)-Math.floor(l))/_],[Math.ceil(d),Math.floor(u),(Math.ceil(u)-Math.floor(d))/m]),this.dataX=r[0],this.dataY=r[1]},n},t.Z.registerElement("mesh3d",t.Z.createMesh3D),t.Z.Circle3D=function(t,e,r,n,a){var h,l;this.constructor(t.board,a,i.Z.OBJECT_TYPE_CIRCLE3D,i.Z.OBJECT_CLASS_3D),this.constructor3D(t,"circle3d"),this.center=this.board.select(e),this.normalFunc=r,this.normal=[0,0,0,0],this.curve,this.frame1,this.frame2,s.Z.exists(this.center._is_new)?(this.addChild(this.center),delete this.center._is_new):this.center.addChild(this),this.updateRadius=s.Z.createFunction(n,this.board),this.addParentsFromJCFunctions([this.updateRadius]),this.updateNormal(),this.frame1=o.Z.crossProduct(this.normal.slice(1),[1,0,0]),this.frame1.unshift(0),(h=o.Z.crossProduct(this.normal.slice(1),[-.5,.8660254037844386,0])).unshift(0),o.Z.norm(h)>o.Z.norm(this.frame1)&&(this.frame1=h),this.frame2=o.Z.crossProduct(this.normal.slice(1),this.frame1.slice(1)),this.frame2.unshift(0),this.normalizeFrame(),l=this,this.curve=t.create("curve3d",[function(t){var e=l.Radius(),i=Math.sin(t),s=Math.cos(t);return[l.center.coords[1]+e*(s*l.frame1[1]+i*l.frame2[1]),l.center.coords[2]+e*(s*l.frame1[2]+i*l.frame2[2]),l.center.coords[3]+e*(s*l.frame1[3]+i*l.frame2[3])]},[0,2*Math.PI]],a)},t.Z.Circle3D.prototype=new t.Z.GeometryElement,s.Z.copyPrototypeMethods(t.Z.Circle3D,t.Z.GeometryElement3D,"constructor3D"),t.Z.extend(t.Z.Circle3D.prototype,{update:function(){return this.needsUpdate&&(this.updateNormal().updateFrame(),this.curve.visProp.visible=!isNaN(this.Radius())),this},updateRenderer:function(){return this.needsUpdate=!1,this},setRadius:function(t){return this.updateRadius=s.Z.createFunction(t,this.board),this.addParentsFromJCFunctions([this.updateRadius]),this.board.update(),this},Radius:function(t){return s.Z.exists(t)?(this.setRadius(t),this.Radius()):Math.abs(this.updateRadius())},normalizeFrame:function(){var t,e=o.Z.norm(this.frame1),i=o.Z.norm(this.frame2);for(t=0;t<4;t++)this.frame1[t]/=e,this.frame2[t]/=i;return this},updateNormal:function(){var t,e;if(this.normal=s.Z.evaluate(this.normalFunc),e=o.Z.norm(this.normal),Math.abs(e)>1e-12)for(t=0;t<4;t++)this.normal[t]/=e;return this},updateFrame:function(){return this.frame1=o.Z.crossProduct(this.frame2.slice(1),this.normal.slice(1)),this.frame1.unshift(0),this.frame2=o.Z.crossProduct(this.normal.slice(1),this.frame1.slice(1)),this.frame2.unshift(0),this.normalizeFrame(),this},projectCoords:function(t,e){return this.curve.projectCoords(t,e)}}),t.Z.createCircle3D=function(e,i,r){var o,n=i[0],a=s.Z.copyAttributes(r,e.options,"circle3d"),h=s.Z.providePoints3D(n,[i[1]],r,"circle3d",["point"])[0],l=i[2],c=i[3];return(o=new t.Z.Circle3D(n,h,l,c,a)).curve.addParents([o]),o.addChild(o.curve),o.update(),o},t.Z.registerElement("circle3d",t.Z.createCircle3D),t.Z.createIntersectionCircle3D=function(t,e,r){var o,n,a,l=e[0],c=e[1],d=e[2],u=s.Z.copyAttributes(r,t.options,"intersectioncircle3d");a=h.Z.intersectionFunction3D(l,c,d),n=l.create("point3d",a[0],{visible:!1}),o=l.create("circle3d",[n,a[1],a[2]],u);try{c.addChild(o),d.addChild(o)}catch(t){throw new Error("JSXGraph: Can't create 'intersection' with parent types '"+typeof e[1]+"' and '"+typeof e[2]+"'.")}return o.type=i.Z.OBJECT_TYPE_INTERSECTION_CIRCLE3D,o.elType="intersectioncircle3d",o.setParents([c.id,d.id]),o},t.Z.registerElement("intersectioncircle3d",t.Z.createIntersectionCircle3D),t.Z.Point3D=function(t,e,r,o){this.constructor(t.board,o,i.Z.OBJECT_TYPE_POINT3D,i.Z.OBJECT_CLASS_3D),this.constructor3D(t,"point3d"),this.board.finalizeAdding(this),this.coords=[0,0,0,0],this.initialCoords=[0,0,0,0],this.F=e,this.slide=r,this.position=[],this.movePath=[],this.moveCallback=null,this.moveInterval=null,this._c2d=null,this.methodMap=s.Z.deepCopy(this.methodMap,{})},t.Z.Point3D.prototype=new t.Z.GeometryElement,s.Z.copyPrototypeMethods(t.Z.Point3D,t.Z.GeometryElement3D,"constructor3D"),t.Z.extend(t.Z.Point3D.prototype,{X:function(){return this.coords[1]},Y:function(){return this.coords[2]},Z:function(){return this.coords[3]},W:function(){return this.coords[0]},updateCoords:function(){var t,e=0;if(s.Z.isFunction(this.F))this.coords=s.Z.evaluate(this.F),3===this.coords.length&&this.coords.unshift(1);else for(3===this.F.length&&(this.coords[0]=1,e=1),t=0;t<this.F.length;t++)s.Z.isFunction(this.F[t])&&(this.coords[e+t]=s.Z.evaluate(this.F[t]));return this},initCoords:function(){var t,e=0;if(s.Z.isFunction(this.F))this.coords=s.Z.evaluate(this.F),3===this.coords.length&&this.coords.unshift(1);else for(3===this.F.length&&(this.coords[0]=1,e=1),t=0;t<this.F.length;t++)this.coords[e+t]=s.Z.evaluate(this.F[t]);return this.initialCoords=this.coords.slice(),this},normalizeCoords:function(){return Math.abs(this.coords[0])>1e-14&&(this.coords[1]/=this.coords[0],this.coords[2]/=this.coords[0],this.coords[3]/=this.coords[0],this.coords[0]=1),this},setPosition:function(t,e){var i=this.coords;return 3===t.length?(i[0]=1,i[1]=t[0],i[2]=t[1],i[3]=t[2]):(i[0]=t[0],i[1]=t[1],i[2]=t[2],i[3]=t[2],this.normalizeCoords()),this},addTransform:function(t,e){return this.addTransformGeneric(t,e),this},updateTransform:function(){var t,e;if(0===this.transformations.length||null===this.baseElement)return this;for(t=this===this.baseElement?this.initialCoords:this.baseElement.coords,e=0;e<this.transformations.length;e++)this.transformations[e].update(),t=o.Z.matVecMult(this.transformations[e].matrix,t);return this.coords=t,this},update:function(t){var e,s,r;return this.element2D.draggable()&&0!==h.Z.distance(this._c2d,this.element2D.coords.usrCoords)?(this.view.isVerticalDrag()?e=this.view.project2DTo3DVertical(this.element2D,this.coords):(s=[1,0,0,this.coords[3]],e=this.view.project2DTo3DPlane(this.element2D,[1,0,0,1],s)),0!==e[0]&&(r=this.view.project3DToCube(e),this.coords=r[0],r[1]&&this.element2D.coords.setCoordinates(i.Z.COORDS_BY_USER,this.view.project3DTo2D(this.coords)),this.slide&&(this.coords=this.slide.projectCoords([1,this.X(),this.Y(),this.Z()],this.position),this.element2D.coords.setCoordinates(i.Z.COORDS_BY_USER,this.view.project3DTo2D(this.coords))))):(this.updateCoords().updateTransform(),this.slide&&(this.coords=this.slide.projectCoords([1,this.X(),this.Y(),this.Z()],this.position)),e=this.coords,this.element2D.coords.setCoordinates(i.Z.COORDS_BY_USER,this.view.project3DTo2D(e)),this.zIndex=o.Z.innerProduct(this.view.matrix3DRotShift[3],e)),this._c2d=this.element2D.coords.usrCoords.slice(),this},updateRenderer:function(){return this.needsUpdate=!1,this},testIfFinite:function(){return Math.abs(this.coords[0])>1e-12},distance:function(t){var e=this.coords,i=t.coords;return e[0]*e[0]>1e-12&&i[0]*i[0]>1e-12?o.Z.hypot(i[1]-e[1],i[2]-e[2],i[3]-e[3]):Number.POSITIVE_INFINITY},moveTo:function(t,e,i){i=i||{};var r,n,a=Math.ceil(e/this.board.attr.animationdelay),h=t[0],l=t[1],c=t[2],d=this.coords[1]-h,u=this.coords[2]-l,p=this.coords[3]-c,f=()=>{};if(i.callback&&(f=i.callback),n=function(t){var e=t/a;if(i.effect){if("<>"===i.effect)return Math.pow(Math.sin(e*Math.PI/2),2);if("<"===i.effect)return e*e*e;if(">"===i.effect)return 1-Math.pow(1-e,3);if("=="===i.effect)return t/a;throw new Error("valid effects are '==', '<>', '>', and '<'.")}return t/a},!s.Z.exists(e)||0===e)return this.setPosition([h,l,c],!0),this.board.update(this);if(!s.Z.exists(i.callback)&&Math.abs(d)<o.Z.eps&&Math.abs(u)<o.Z.eps&&Math.abs(p)<o.Z.eps)return this;for(this.animationPath=[],r=a;r>=0;r--)this.animationPath[a-r]=[h+d*n(r),l+u*n(r),c+p*n(r)];return this.moveAlong(this.animationPath,e,{callback:f})},moveAlong:function(t,e,i){let r=e/t.length;if(s.Z.isObject(i)&&"callback"in i&&(this.moveCallback=i.callback),!(this.movePath.length>0))return this.movePath=t,new Promise(((t,e)=>{this.moveInterval=setInterval((()=>{if(this.movePath.length>0){let t=this.movePath.shift();this.setPosition(t,!0),this.board.update(this)}0===this.movePath.length&&(clearInterval(this.moveInterval),t(),s.Z.isFunction(this.moveCallback)&&this.moveCallback())}),r)}));this.movePath=t},__evt__update3D:function(t){}}),t.Z.createPoint3D=function(e,i,r){var o,n,a,h,l,c=i[0],d=null,u=null;if(a=i.length>2&&s.Z.exists(i[i.length-1].is3D)&&!s.Z.isTransformationOrArray(i[i.length-1])?i.pop():null,2===i.length)n=i[1];else if(3===i.length&&s.Z.isPoint3D(i[1])&&s.Z.isTransformationOrArray(i[2]))n=[0,0,0],d=i[1],u=i[2];else if(4===i.length)n=i.slice(1);else{if(5!==i.length)throw new Error("JSXGraph: Can't create point3d with parent types '"+typeof i[1]+"' and '"+typeof i[2]+"'.\nPossible parent types: [[x,y,z]], [x,y,z], or [[x,y,z], slide], () => [x, y, z], or [point, transformation(s)]");n=i.slice(1)}return o=s.Z.copyAttributes(r,e.options,"point3d"),(l=new t.Z.Point3D(c,n,a,o)).initCoords(),null!==d&&null!==u&&l.addTransform(d,u),h=c.project3DTo2D(l.coords),o=l.setAttr2D(o),l.element2D=c.create("point",h,o),l.element2D.view=c,l.addChild(l.element2D),l.inherits.push(l.element2D),l.element2D.setParents(l),l.slide&&(l.slide.addChild(l),l.setParents(l.slide)),d&&l.setParents(d),l._c2d=l.element2D.coords.usrCoords.slice(),l},t.Z.registerElement("point3d",t.Z.createPoint3D),t.Z.Curve3D=function(t,e,r,o,n,a,h){this.constructor(t.board,h,i.Z.OBJECT_TYPE_CURVE3D,i.Z.OBJECT_CLASS_3D),this.constructor3D(t,"curve3d"),this.board.finalizeAdding(this),this._F=e,this._X=r,this._Y=o,this._Z=n,this.points=[],this.numberPoints=0,this.dataX=null,this.dataY=null,this.dataZ=null,null!==this._F?(this._X=function(t){return this._F(t)[0]},this._Y=function(t){return this._F(t)[1]},this._Z=function(t){return this._F(t)[2]}):s.Z.isFunction(this._X)?this._F=function(t){return[this._X(t),this._Y(t),this._Z(t)]}:this._F=null,this.range=a,this.methodMap=s.Z.deepCopy(this.methodMap,{})},t.Z.Curve3D.prototype=new t.Z.GeometryElement,s.Z.copyPrototypeMethods(t.Z.Curve3D,t.Z.GeometryElement3D,"constructor3D"),t.Z.extend(t.Z.Curve3D.prototype,{updateCoords:function(){var t,e,i,r,o,n,a=this.evalVisProp("numberpointshigh"),h=[1,0,0,0];if(this.points=[],s.Z.exists(this.dataX))for(a=this.dataX.length,o=0;o<a;o++)this.points.push([1,this.dataX[o],this.dataY[o],this.dataZ[o]]);else if(s.Z.isArray(this._X))for(a=this._X.length,o=0;o<a;o++)this.points.push([1,this._X[o],this._Y[o],this._Z[o]]);else for(t=s.Z.evaluate(this.range),e=s.Z.evaluate(t[0]),r=((i=s.Z.evaluate(t[1]))-e)/(a-1),n=0,o=e;n<a&&o<=i;n++,o+=r)(h=this.F(o)).unshift(1),this.points.push(h);return this.numberPoints=this.points.length,this},evalF:function(t){var e,i,r=[0,0,0,0];if(0===this.transformations.length||!s.Z.exists(this.baseElement))return r=s.Z.exists(this._F)?this._F(t):[this._X[t],this._Y[t],this._Z[t]];for(e=this.transformations,i=0;i<e.length;i++)e[i].update();for(3===r.length&&r.unshift(1),(r=this===this.baseElement?s.Z.exists(this._F)?this._F(t):[this._X[t],this._Y[t],this._Z[t]]:this.baseElement.evalF(t)).unshift(1),r=o.Z.matVecMult(e[0].matrix,r),i=1;i<e.length;i++)r=o.Z.matVecMult(e[i].matrix,r);return r.slice(1)},F:function(t){return this.evalF(t)},X:function(t){return this.evalF(t)[0]},Y:function(t){return this.evalF(t)[1]},Z:function(t){return this.evalF(t)[2]},updateDataArray2D:function(){var t,e,i=[],s=[],r=this.points.length;for(t=0;t<r;t++)e=this.view.project3DTo2D(this.points[t]),i.push(e[1]),s.push(e[2]);return{X:i,Y:s}},addTransform:function(t,e){return this.addTransformGeneric(t,e),this},updateTransform:function(){var t,e,i,r,n;if(0===this.transformations.length||null===this.baseElement||s.Z.exists(this._F))return this;for(t=this.transformations,i=0;i<t.length;i++)t[i].update();for(n=this.baseElement.numberPoints,i=0;i<n;i++){for(e=this===this.baseElement?this.points[i]:this.baseElement.points[i],r=0;r<t.length;r++)e=o.Z.matVecMult(t[r].matrix,e);this.points[i]=e}return this.numberPoints=n,this},updateDataArray:function(){},update:function(){return this.needsUpdate&&(this.updateDataArray(),this.updateCoords().updateTransform()),this},updateRenderer:function(){return this.needsUpdate=!1,this},projectCoords:function(t,e){return h.Z.projectCoordsToParametric(t,this,1,e)}}),t.Z.createCurve3D=function(e,r,n){var a,h,l,c,d,u,p,f,_=r[0],m=null,g=null;return 3===r.length?s.Z.isTransformationOrArray(r[2])&&r[1].type===i.Z.OBJECT_TYPE_CURVE3D?(m=r[1],g=r[2],a=null,h=[],l=[],c=[]):(a=r[1],d=r[2],h=null,l=null,c=null):2===r.length&&s.Z.isArray(r[1])?(h=(f=o.Z.transpose(r[1]))[0],l=f[1],c=f[2],a=null):(h=r[1],l=r[2],c=r[3],d=r[4],a=null),u=s.Z.copyAttributes(n,e.options,"curve3d"),u=(p=new t.Z.Curve3D(_,a,h,l,c,d,u)).setAttr2D(u),p.element2D=_.create("curve",[[],[]],u),p.element2D.view=_,null!==m&&(p.addTransform(m,g),p.addParents(m)),p.element2D.updateDataArray=function(){var t=p.updateDataArray2D();this.dataX=t.X,this.dataY=t.Y},p.addChild(p.element2D),p.inherits.push(p.element2D),p.element2D.setParents(p),p.element2D.prepareUpdate().update(),e.isSuspendedUpdate||p.element2D.updateVisibility().updateRenderer(),p},t.Z.registerElement("curve3d",t.Z.createCurve3D),t.Z.createVectorfield3D=function(t,e,i){var r,n,a=e[0];if(!(e.length>=5&&(s.Z.isArray(e[1])||s.Z.isFunction(e[1])||s.Z.isString(e[1]))&&s.Z.isArray(e[2])&&3===e[1].length&&s.Z.isArray(e[3])&&3===e[2].length&&s.Z.isArray(e[4])&&3===e[3].length))throw new Error("JSXGraph: Can't create vector field 3D with parent types '"+typeof e[1]+"', '"+typeof e[2]+"', '"+typeof e[3]+"'.'"+typeof e[4]+"', ");return n=s.Z.copyAttributes(i,t.options,"vectorfield3d"),(r=a.create("curve3d",[[],[],[]],n)).setF=function(t,e){var i,o,n;return s.Z.isArray(t)?(i=s.Z.createFunction(t[0],this.board,e),o=s.Z.createFunction(t[1],this.board,e),n=s.Z.createFunction(t[2],this.board,e),this.F=function(t,e,s){return[i(t,e,s),o(t,e,s),n(t,e,s)]}):this.F=s.Z.createFunction(t,r.board,e),this},r.setF(e[1],"x, y, z"),r.xData=e[2],r.yData=e[3],r.zData=e[4],r.updateDataArray=function(){var e,i,r,n,a,h,l,c,d,u,p,f,_,m,g,b,v,Z=this.evalVisProp("scale"),C=[s.Z.evaluate(this.xData[0]),s.Z.evaluate(this.yData[0]),s.Z.evaluate(this.zData[0])],y=[s.Z.evaluate(this.xData[1]),s.Z.evaluate(this.yData[1]),s.Z.evaluate(this.zData[1])],P=[s.Z.evaluate(this.xData[2]),s.Z.evaluate(this.yData[2]),s.Z.evaluate(this.zData[2])],E=[(P[0]-C[0])/y[0],(P[1]-C[1])/y[1],(P[2]-C[2])/y[2]],x=this.evalVisProp("arrowhead.enabled");for(x&&(_=this.evalVisProp("arrowhead.size"),v=this.evalVisProp("arrowhead.angle"),m=_/t.unitX,g=_/t.unitY,b=_/Math.sqrt(t.unitX*t.unitY)),this.dataX=[],this.dataY=[],this.dataZ=[],i=0,h=C[0];i<=y[0];h+=E[0],i++)for(r=0,l=C[1];r<=y[1];l+=E[1],r++)for(e=0,c=C[2];e<=y[2];c+=E[2],e++)n=this.F(h,l,c),(a=o.Z.norm(n))<Number.EPSILON||(n[0]*=Z,n[1]*=Z,n[2]*=Z,s.Z.concat(this.dataX,[h,h+n[0],NaN]),s.Z.concat(this.dataY,[l,l+n[1],NaN]),s.Z.concat(this.dataZ,[c,c+n[2],NaN]),x&&(a*=Z,d=Math.atan2(n[1],n[0]),u=(f=Math.asin(n[2]/a))-v,p=f+v,s.Z.concat(this.dataX,[h+n[0]-m*Math.cos(d)*Math.cos(u),h+n[0],h+n[0]-m*Math.cos(d)*Math.cos(p),NaN]),s.Z.concat(this.dataY,[l+n[1]-g*Math.sin(d)*Math.cos(u),l+n[1],l+n[1]-g*Math.sin(d)*Math.cos(p),NaN]),s.Z.concat(this.dataZ,[c+n[2]-b*Math.sin(p),c+n[2],c+n[2]-b*Math.sin(u),NaN])))},r.methodMap=s.Z.deepCopy(r.methodMap,{setF:"setF"}),r},t.Z.registerElement("vectorfield3D",t.Z.createVectorfield3D),t.Z.Line3D=function(t,e,r,o,n){this.constructor(t.board,n,i.Z.OBJECT_TYPE_LINE3D,i.Z.OBJECT_CLASS_3D),this.constructor3D(t,"line3d"),this.point=e,this.direction=r,this.vec=[0,0,0,0],this.range=o||[-1/0,1/0],this.point1=null,this.point2=null,this.board.finalizeAdding(this),this.methodMap=s.Z.deepCopy(this.methodMap,{})},t.Z.Line3D.prototype=new t.Z.GeometryElement,s.Z.copyPrototypeMethods(t.Z.Line3D,t.Z.GeometryElement3D,"constructor3D"),t.Z.extend(t.Z.Line3D.prototype,{updateCoords:function(){var t,e=0;if(s.Z.exists(this.direction.view)&&this.direction.type===i.Z.OBJECT_TYPE_LINE3D)this.vec=this.direction.vec.slice();else if(s.Z.isFunction(this.direction))this.vec=s.Z.evaluate(this.direction),3===this.vec.length&&this.vec.unshift(0);else for(3===this.direction.length&&(this.vec[0]=0,e=1),t=0;t<this.direction.length;t++)this.vec[e+t]=s.Z.evaluate(this.direction[t]);return this},getPointCoords:function(t){var e,i,r;return e=this.point.coords,i=this.vec,r=s.Z.evaluate(t),t=this.view.intersectionLineCube(e,i,r),Math.abs(t)===1/0&&(t=0),[e[0]+i[0]*t,e[1]+i[1]*t,e[2]+i[2]*t,e[3]+i[3]*t]},addTransform:function(t,e){return this.point.addTransform(t.point,e),this.addTransformGeneric(t,e),this},updateTransform:function(){var t,e;if(0===this.transformations.length||null===this.baseElement)return this;for(t=this===this.baseElement?this.vec:this.baseElement.vec,e=0;e<this.transformations.length;e++)this.transformations[e].update(),t=o.Z.matVecMult(this.transformations[e].matrix,t);return this.vec=t,this},update:function(){return this.needsUpdate&&this.updateCoords().updateTransform(),this},setPosition2D:function(t){var e,i;for(e=0;e<this.parents.length;e++)"point3d"===(i=this.view.select(this.parents[e])).elType&&i.element2D.draggable()&&t.applyOnce(i.element2D);this.endpoints[0].update(),this.endpoints[1].update()},updateRenderer:function(){return this.needsUpdate=!1,this},projectCoords:function(t,e){var i,r=this.getPointCoords(0),n=this.getPointCoords(1),a=[n[0]-r[0],n[1]-r[1],n[2]-r[2]],h=[t[0]-r[0],t[1]-r[1],t[2]-r[2]],l=o.Z.innerProduct(h,a)/o.Z.innerProduct(a,a),c=Math.min(Math.max(l,s.Z.evaluate(this.range[0])),s.Z.evaluate(this.range[1]));return i=this.getPointCoords(c).slice(),e[0]=c,i},updateZIndex:function(){var t=this.endpoints[0],e=this.endpoints[1],i=[1,t.X()+e.X(),t.Y()+e.Y(),t.Z()+e.Z()];return i[1]*=.5,i[2]*=.5,i[3]*=.5,this.zIndex=o.Z.innerProduct(this.view.matrix3DRotShift[3],i),this}}),t.Z.createLine3D=function(e,r,o){var n,a,h,l,c,d,u,p,f,_=r[0],m=null,g=null;if(n=s.Z.copyAttributes(o,e.options,"line3d"),r[1].type===i.Z.OBJECT_TYPE_LINE3D&&s.Z.isTransformationOrArray(r[2])&&(m=r[1],g=r[2],a=s.Z.providePoints3D(_,[[0,0,0,0],[0,0,0,0]],o,"line3d",["point1","point2"])),null===m&&(s.Z.isPoint3D(r[2])||3===r.length&&(s.Z.isArray(r[2])||s.Z.isFunction(r[2]))))d=s.Z.providePoints3D(_,[r[1]],o,"line3d",["point1"])[0],u=s.Z.providePoints3D(_,[r[2]],o,"line3d",["point2"])[0],l=function(){return[0,u.X()-d.X(),u.Y()-d.Y(),u.Z()-d.Z()]},c=[0,1],(f=new t.Z.Line3D(_,d,l,c,n)).prepareUpdate().update(),(p=s.Z.providePoints3D(_,[[1,0,0,0],[1,0,0,0]],{visible:!0},"line3d",["point1","point2"]))[0].F=function(){var t=0;return f.evalVisProp("straightfirst")&&(t=-1/0),f.getPointCoords(t)},p[1].F=function(){var t=1;return f.evalVisProp("straightlast")&&(t=1/0),f.getPointCoords(t)},p[0].prepareUpdate().update(),p[1].prepareUpdate().update(),(n=f.setAttr2D(n)).straightfirst=!1,n.straightlast=!1,f.element2D=_.create("segment",[p[0].element2D,p[1].element2D],n),f.element2D.view=_,f.endpoints=p,f.addChild(p[0]),f.addChild(p[1]),f.addParents([d,u]);else{if(null!==m)h=s.Z.providePoints3D(_,[[0,0,0]],o,"line3d",["point"])[0],l=[0,0,0,1e-4],c=r[3]||[-1/0,1/0];else{if(!(s.Z.exists(r[2].view)&&r[2].type===i.Z.OBJECT_TYPE_LINE3D||s.Z.isFunction(r[2])||3===r[2].length||4===r[2].length))throw new Error("JSXGraph: Can't create line3d with parents of type '"+typeof r[1]+", "+typeof r[2]+", "+typeof r[3]+"'.");h=s.Z.providePoints3D(_,[r[1]],o,"line3d",["point"])[0],l=r[2],c=r[3]}a=s.Z.providePoints3D(_,[[1,0,0,0],[1,0,0,0]],o,"line3d",["point1","point2"]),(f=new t.Z.Line3D(_,h,l,c,n)).prepareUpdate().update(),a[0].F=function(){return f.getPointCoords(s.Z.evaluate(f.range[0]))},a[0].prepareUpdate().update(),d=a[0],a[1].F=function(){return f.getPointCoords(s.Z.evaluate(f.range[1]))},a[1].prepareUpdate().update(),u=a[1],(n=f.setAttr2D(n)).straightfirst=!1,n.straightlast=!1,f.element2D=_.create("segment",[d.element2D,u.element2D],n),f.element2D.view=_,f.endpoints=a,f.addParents(h),null!==m&&null!==g&&(f.addTransform(m,g),f.addParents(m))}return f.addChild(f.element2D),f.inherits.push(f.element2D),f.element2D.addParents(f),f.point1=d,f.point2=u,f.point1._is_new?(f.addChild(f.point1),delete f.point1._is_new):f.point1.addChild(f),f.point2._is_new?(f.addChild(f.point2),delete f.point2._is_new):f.point2.addChild(f),s.Z.exists(h)&&(h._is_new?(f.addChild(h),delete h._is_new):h.addChild(f)),f.update(),f.element2D.prepareUpdate().update().updateRenderer(),f},t.Z.registerElement("line3d",t.Z.createLine3D),t.Z.Plane3D=function(t,e,r,o,n,a,h){this.constructor(t.board,h,i.Z.OBJECT_TYPE_PLANE3D,i.Z.OBJECT_CLASS_3D),this.constructor3D(t,"plane3d"),this.board.finalizeAdding(this),this.point=e,this.direction1=r,this.direction2=n,this.range_u=o||[-1/0,1/0],this.range_v=a||[-1/0,1/0],this.vec1=[0,0,0,0],this.vec2=[0,0,0,0],this.mesh3d=null,this.normal=[0,0,0,0],this.d=0,this.updateCoords(),this.updateNormal(),this.methodMap=s.Z.deepCopy(this.methodMap,{})},t.Z.Plane3D.prototype=new t.Z.GeometryElement,s.Z.copyPrototypeMethods(t.Z.Plane3D,t.Z.GeometryElement3D,"constructor3D"),t.Z.extend(t.Z.Plane3D.prototype,{F:function(t,e){var i,s,r,n,a;for(s=this.vec1.slice(),r=this.vec2.slice(),n=o.Z.norm(s,3),a=o.Z.norm(r,3),i=0;i<3;i++)s[i]/=n,r[i]/=a;return[this.point.X()+t*s[0]+e*r[0],this.point.Y()+t*s[1]+e*r[1],this.point.Z()+t*s[2]+e*r[2]]},X:function(t,e){return this.F(t,e)[0]},Y:function(t,e){return this.F(t,e)[1]},Z:function(t,e){return this.F(t,e)[2]},updateCoords:function(){var t,e;if(s.Z.exists(this.direction1.view)&&this.direction1.type===i.Z.OBJECT_TYPE_LINE3D)this.vec1=this.direction1.vec.slice();else if(s.Z.isFunction(this.direction1))this.vec1=s.Z.evaluate(this.direction1),3===this.vec1.length&&this.vec1.unshift(0);else for(e=0,3===this.direction1.length&&(this.vec1[0]=0,e=1),t=0;t<this.direction1.length;t++)this.vec1[e+t]=s.Z.evaluate(this.direction1[t]);if(s.Z.exists(this.direction2.view)&&this.direction2.type===i.Z.OBJECT_TYPE_LINE3D)this.vec2=this.direction2.vec.slice();else if(s.Z.isFunction(this.direction2))this.vec2=s.Z.evaluate(this.direction2),3===this.vec2.length&&this.vec2.unshift(0);else for(e=0,3===this.direction2.length&&(this.vec2[0]=0,e=1),t=0;t<this.direction2.length;t++)this.vec2[e+t]=s.Z.evaluate(this.direction2[t]);return this},updateNormal:function(){var t,e;if(this.needsUpdate,this.normal=o.Z.crossProduct(this.vec1.slice(1),this.vec2.slice(1)),e=o.Z.norm(this.normal),Math.abs(e)>o.Z.eps*o.Z.eps)for(t=0;t<3;t++)this.normal[t]/=e;return this.normal.unshift(0),this.d=o.Z.innerProduct(this.point.coords,this.normal,4),this},updateDataArray:function(){var t,e,i,r,n,a,l,c,d,u,p,f,_,m,g=["xPlaneRear","yPlaneRear","zPlaneRear"],b=[],v=[0,0,0],Z=[0,0,0],C=[0,0,0],y=[0,0,0],P=1e-12,E=this.view;if(this.dataX=[],this.dataY=[],this.updateNormal(),"axisplane3d"!==this.elType&&E.defaultAxes&&s.Z.evaluate(this.range_u[0])===-1/0&&s.Z.evaluate(this.range_u[1])===1/0&&s.Z.evaluate(this.range_v[0])===-1/0&&s.Z.evaluate(this.range_v[1])===1/0){for(u=0;u<g.length;u++){if(!1!==(y=E.intersectionPlanePlane(this,E.defaultAxes[g[u]]))[0]&&!1!==y[1]){for(d=0;d<b.length&&!(h.Z.distance(y[0],b[d][0],4)<P&&h.Z.distance(y[1],b[d][1],4)<P||h.Z.distance(y[0],b[d][1],4)<P&&h.Z.distance(y[1],b[d][0],4)<P);d++);d===b.length&&b.push(y.slice())}if((y=[1,0,0,0])[u+1]=E.bbox3D[u][1],c=o.Z.innerProduct(y,E.defaultAxes[g[u]].normal,4),!1!==(y=E.intersectionPlanePlane(this,E.defaultAxes[g[u]],c))[0]&&!1!==y[1]){for(d=0;d<b.length&&!(h.Z.distance(y[0],b[d][0],4)<P&&h.Z.distance(y[1],b[d][1],4)<P||h.Z.distance(y[0],b[d][1],4)<P&&h.Z.distance(y[1],b[d][0],4)<P);d++);d===b.length&&b.push(y.slice())}}if(0===b.length)return{X:this.dataX,Y:this.dataY};0,_=0,d=0;do{for(4===(y=b[_][d]).length&&(n=E.project3DTo2D(y),this.dataX.push(n[1]),this.dataY.push(n[2])),d=(d+1)%2,y=b[_][d],m=_,u=0;u<b.length;u++){if(u!==_&&h.Z.distance(y,b[u][0])<P){_=u,d=0;break}if(u!==_&&h.Z.distance(y,b[u][1])<P){_=u,d=1;break}}if(_===m){console.log("Error plane3d update: did not find next",_);break}}while(0!==_);n=E.project3DTo2D(b[0][0]),this.dataX.push(n[1]),this.dataY.push(n[2])}else{for(t=s.Z.evaluate(this.range_u[0]),e=s.Z.evaluate(this.range_u[1]),i=s.Z.evaluate(this.range_v[0]),r=s.Z.evaluate(this.range_v[1]),C=this.point.coords,v=this.vec1.slice(),Z=this.vec2.slice(),a=o.Z.norm(v,4),l=o.Z.norm(Z,4),d=1;d<4;d++)v[d]/=a,Z[d]/=l;for(u=0;u<4;u++){switch(u){case 0:p=t,f=i;break;case 1:p=e,f=i;break;case 2:p=e,f=r;break;case 3:p=t,f=r}for(d=0;d<4;d++)y[d]=C[d]+p*v[d]+f*Z[d];n=E.project3DTo2D(y),this.dataX.push(n[1]),this.dataY.push(n[2])}this.dataX.push(this.dataX[0]),this.dataY.push(this.dataY[0])}return{X:this.dataX,Y:this.dataY}},addTransform:function(t,e){return this.addTransformGeneric(t,e),this.point.addTransform(t.point,e),this},updateTransform:function(){var t,e,i;if(0===this.transformations.length||null===this.baseElement)return this;for(this===this.baseElement?(t=this.vec1,e=this.vec2):(t=this.baseElement.vec1,e=this.baseElement.vec2),i=0;i<this.transformations.length;i++)this.transformations[i].update(),t=o.Z.matVecMult(this.transformations[i].matrix,t),e=o.Z.matVecMult(this.transformations[i].matrix,e);return this.vec1=t,this.vec2=e,this},update:function(){return this.needsUpdate&&this.updateCoords().updateTransform(),this},updateRenderer:function(){return this.needsUpdate=!1,this},projectCoords:function(t,e){return h.Z.projectCoordsToParametric(t,this,2,e)}}),t.Z.createPlane3D=function(e,r,o){var n,a,h,l,c,d,u,p,f,_,m=r[0],g=null,b=null;if((n=s.Z.copyAttributes(o,e.options,"plane3d")).threepoints||s.Z.isPoint3D(r[2])||s.Z.isPoint3D(r[3]))a=s.Z.providePoints3D(m,[r[1]],o,"plane3d",["point1"])[0],h=s.Z.providePoints3D(m,[r[2]],o,"plane3d",["point2"])[0],l=s.Z.providePoints3D(m,[r[3]],o,"plane3d",["point3"])[0],c=function(){return[h.X()-a.X(),h.Y()-a.Y(),h.Z()-a.Z()]},d=function(){return[l.X()-a.X(),l.Y()-a.Y(),l.Z()-a.Z()]},u=r[4]||[-1/0,1/0],p=r[5]||[-1/0,1/0];else{if(r[1].type===i.Z.OBJECT_TYPE_PLANE3D&&s.Z.isTransformationOrArray(r[2])?(g=r[1],b=r[2],a=s.Z.providePoints3D(m,[[0,0,0,0]],o,"plane3d",["point"])[0],c=[0,1e-4,0,0],d=[0,0,1e-4,0],u=r[3]||[-1/0,1/0],p=r[4]||[-1/0,1/0]):(a=s.Z.providePoints3D(m,[r[1]],o,"plane3d",["point"])[0],c=r[2],d=r[3],u=r[4]||[-1/0,1/0],p=r[5]||[-1/0,1/0]),!1===a)throw new Error("JSXGraph: Can't create plane3d with first parent of type '"+typeof r[1]+"'.\nPossible first parent types are: point3d, array of length 3, function returning an array of length 3.");if(null!==g&&r<3||null===g&&r.length<4)throw new Error("JSXGraph: Can't create plane3d with parents of type '"+typeof r[1]+", "+typeof r[2]+", "+typeof r[3]+", "+typeof r[4]+", "+typeof r[5]+"'.")}return f=new t.Z.Plane3D(m,a,c,u,d,p,n),a.addChild(f),n=f.setAttr2D(n),f.element2D=m.create("curve",[[],[]],n),f.element2D.view=m,null!==g&&null!==b&&(f.addTransform(g,b),f.addParents(g)),f.element2D.updateDataArray=function(){var t=f.updateDataArray();this.dataX=t.X,this.dataY=t.Y},f.addChild(f.element2D),f.inherits.push(f.element2D),f.element2D.setParents(f),Math.abs(f.range_u[0])!==1/0&&Math.abs(f.range_u[1])!==1/0&&Math.abs(f.range_v[0])!==1/0&&Math.abs(f.range_v[1])!==1/0&&(n=s.Z.copyAttributes(n.mesh3d,e.options,"mesh3d"),_=m.create("mesh3d",[function(){return a.coords},function(){return f.vec1},function(){return f.vec2},f.range_u,f.range_v],n),f.mesh3d=_,f.addChild(_),f.inherits.push(_),f.element2D.inherits.push(_),_.setParents(f),f.mesh3d.view=m),f.element2D.prepareUpdate().update(),e.isSuspendedUpdate||f.element2D.updateVisibility().updateRenderer(),f},t.Z.registerElement("plane3d",t.Z.createPlane3D),t.Z.createIntersectionLine3D=function(t,e,r){var o,n,a,l=e[0],c=e[1],d=e[2],u=s.Z.copyAttributes(r,t.options,"intersectionline3d"),p=[];for(a=h.Z.intersectionFunction3D(l,c,d),n=0;n<2;n++)p[n]=l.create("point3d",a[n],u["point"+(n+1)]);o=l.create("line3d",p,u);try{c.addChild(o),d.addChild(o)}catch(t){throw new Error("JSXGraph: Can't create 'intersection' with parent types '"+typeof e[1]+"' and '"+typeof e[2]+"'.")}return o.type=i.Z.OBJECT_TYPE_INTERSECTION_LINE3D,o.elType="intersectionline3d",o.setParents([c.id,d.id]),o},t.Z.registerElement("intersectionline3d",t.Z.createIntersectionLine3D),t.Z.Text3D=function(t,e,r,o,n){this.constructor(t.board,n,i.Z.OBJECT_TYPE_TEXT3D,i.Z.OBJECT_CLASS_3D),this.constructor3D(t,"text3d"),this.board.finalizeAdding(this),this.coords=[0,0,0,0],this.F=e,this.slide=o,this.X=function(){return this.coords[1]},this.Y=function(){return this.coords[2]},this.Z=function(){return this.coords[3]},this.position=[],this._c2d=null,this.methodMap=s.Z.deepCopy(this.methodMap,{})},t.Z.Text3D.prototype=new t.Z.GeometryElement,s.Z.copyPrototypeMethods(t.Z.Text3D,t.Z.GeometryElement3D,"constructor3D"),t.Z.extend(t.Z.Text3D.prototype,{updateCoords:function(){var t;if(s.Z.isFunction(this.F))this.coords=s.Z.evaluate(this.F),this.coords.unshift(1);else for(this.coords[0]=1,t=0;t<3;t++)s.Z.isFunction(this.F[t])&&(this.coords[t+1]=s.Z.evaluate(this.F[t]));return this},initCoords:function(){var t;if(s.Z.isFunction(this.F))this.coords=s.Z.evaluate(this.F),this.coords.unshift(1);else for(this.coords[0]=1,t=0;t<3;t++)this.coords[t+1]=s.Z.evaluate(this.F[t]);return this},normalizeCoords:function(){return Math.abs(this.coords[0])>o.Z.eps&&(this.coords[1]/=this.coords[0],this.coords[2]/=this.coords[0],this.coords[3]/=this.coords[0],this.coords[0]=1),this},setPosition:function(t,e){var i=this.coords;return 3===t.length?(i[0]=1,i[1]=t[0],i[2]=t[1],i[3]=t[2]):(i[0]=t[0],i[1]=t[1],i[2]=t[2],i[3]=t[2],this.normalizeCoords()),this},update:function(t){var e,s,r;return this.element2D.draggable()&&0!==h.Z.distance(this._c2d,this.element2D.coords.usrCoords)?(this.view.isVerticalDrag()?e=this.view.project2DTo3DVertical(this.element2D,this.coords):(s=[1,0,0,this.coords[3]],e=this.view.project2DTo3DPlane(this.element2D,[1,0,0,1],s)),0!==e[0]&&(r=this.view.project3DToCube(e),this.coords=r[0],r[1]&&this.element2D.coords.setCoordinates(i.Z.COORDS_BY_USER,this.view.project3DTo2D(this.coords)),this.slide&&(this.coords=this.slide.projectCoords([this.X(),this.Y(),this.Z()],this.position),this.element2D.coords.setCoordinates(i.Z.COORDS_BY_USER,this.view.project3DTo2D(this.coords))))):(this.updateCoords(),this.slide&&(this.coords=this.slide.projectCoords([this.X(),this.Y(),this.Z()],this.position)),e=this.coords,this.element2D.coords.setCoordinates(i.Z.COORDS_BY_USER,this.view.project3DTo2D(e)),this.zIndex=o.Z.innerProduct(this.view.matrix3DRotShift[3],e),this.element2D.prepareUpdate().update()),this._c2d=this.element2D.coords.usrCoords.slice(),this},updateRenderer:function(){return this.needsUpdate=!1,this},testIfFinite:function(){return Math.abs(this.coords[0])>o.Z.eps},__evt__update3D:function(t){}}),t.Z.createText3D=function(e,i,r){var o,n,a,h,l,c,d=i[0];if(a=i.length>2&&s.Z.exists(i[i.length-1].is3D)?i.pop():null,3===i.length)n=i[1],h=i[2];else{if(5!==i.length)throw new Error("JSXGraph: Can't create text3d with parent types '"+typeof i[1]+"' and '"+typeof i[2]+"'.\nPossible parent types: [[x,y,z], text], [x,y,z, text]");n=i.slice(1,4),h=i[4]}return o=s.Z.copyAttributes(r,e.options,"text3d"),(c=new t.Z.Text3D(d,n,h,a,o)).initCoords(),l=d.project3DTo2D(c.coords),o=c.setAttr2D(o),c.element2D=d.create("text",[l[1],l[2],h],o),c.element2D.view=d,c.addChild(c.element2D),c.inherits.push(c.element2D),c.element2D.setParents(c),c.slide&&(c.slide.addChild(c),c.setParents(c.slide)),c._c2d=c.element2D.coords.usrCoords.slice(),c},t.Z.registerElement("text3d",t.Z.createText3D),t.Z.createTicks3D=function(e,i,r){var n,a,h=i[0],l=i[1],c=i[2],d=i[3],u=i[4];return a=s.Z.copyAttributes(r,e.options,"ticks3d"),(n=h.create("curve",[[],[]],a)).point=l,n.direction1=c,n.len=d,n.direction2=u,n.drawLabels=function(i){var r,n,a,l,c,d,u,p=this.len,f=this.evalVisProp("ticksdistance"),_=this.evalVisProp("tickendings"),m=this.evalVisProp("majorheight"),g=[0,0,0],b=[0,0,0],v=[0,0,0],Z=[];if(r=(m/=Math.sqrt(e.unitX*e.unitY))*_[1]*2,this.dataX=[],this.dataY=[],s.Z.isFunction(this.point))v=this.point().slice(1);else for(l=0;l<3;l++)v[l]=s.Z.evaluate(this.point[l]);for(l=0;l<3;l++)g[l]=s.Z.evaluate(this.direction1[l]),b[l]=s.Z.evaluate(this.direction2[l]);for(n=t.Z.Math.norm(g,3),a=t.Z.Math.norm(b,3),l=0;l<3;l++)g[l]/=n,b[l]/=a;if(!(Math.abs(f)<o.Z.eps)){for(c=0;c<=p;c+=f){for(u=[v[0]+c*g[0]+r*b[0],v[1]+c*g[1]+r*b[1],v[2]+c*g[2]+r*b[2]],l=0;l<3;l++)0!==g[l]&&(d=v[l]+c*g[l]);Z.push(h.create("text3d",[u,d],i))}return Z}},n.evalVisProp("drawlabels")&&(n.labels=n.drawLabels(a.label)),n.updateDataArray=function(){var i,r,n,a,l,c,d,u,p=this.len,f=this.evalVisProp("ticksdistance"),_=this.evalVisProp("tickendings"),m=this.evalVisProp("majorheight"),g=[0,0,0],b=[0,0,0],v=[0,0,0];if(i=(m/=Math.sqrt(e.unitX*e.unitY))*-_[0],r=m*_[1],this.dataX=[],this.dataY=[],s.Z.isFunction(this.point))v=this.point().slice(1);else for(l=0;l<3;l++)v[l]=s.Z.evaluate(this.point[l]);for(l=0;l<3;l++)g[l]=s.Z.evaluate(this.direction1[l]),b[l]=s.Z.evaluate(this.direction2[l]);for(n=t.Z.Math.norm(g,3),a=t.Z.Math.norm(b,3),l=0;l<3;l++)g[l]/=n,b[l]/=a;if(!(Math.abs(f)<o.Z.eps))for(c=0;c<=p;c+=f)u=[v[0]+c*g[0]+i*b[0],v[1]+c*g[1]+i*b[1],v[2]+c*g[2]+i*b[2]],d=h.project3DTo2D(u),this.dataX.push(d[1]),this.dataY.push(d[2]),u=[v[0]+c*g[0]+r*b[0],v[1]+c*g[1]+r*b[1],v[2]+c*g[2]+r*b[2]],d=h.project3DTo2D(u),this.dataX.push(d[1]),this.dataY.push(d[2]),this.dataX.push(NaN),this.dataY.push(NaN)},n},t.Z.registerElement("ticks3d",t.Z.createTicks3D),t.Z.Polygon3D=function(t,e,s){var r;for(this.constructor(t.board,s,i.Z.OBJECT_TYPE_POLYGON3D,i.Z.OBJECT_CLASS_3D),this.constructor3D(t,"polygon3d"),this.board.finalizeAdding(this),this.vertices=[],r=0;r<e.length;r++)this.vertices[r]=this.board.select(e[r]),this.vertices[r]._is_new&&(delete this.vertices[r]._is_new,this.vertices[r]._is_new_pol=!0)},t.Z.Polygon3D.prototype=new t.Z.GeometryElement,s.Z.copyPrototypeMethods(t.Z.Polygon3D,t.Z.GeometryElement3D,"constructor3D"),t.Z.extend(t.Z.Polygon3D.prototype,{update:function(){return this},updateRenderer:function(){return this.needsUpdate=!1,this},updateZIndex:function(){var t,e=this.vertices.length,i=[1,0,0,0];if(e<=0)return[NaN,NaN,NaN,NaN];for(t=0;t<e;t++)i[1]+=this.vertices[t].coords[1],i[2]+=this.vertices[t].coords[2],i[3]+=this.vertices[t].coords[3];return i[1]/=e,i[2]/=e,i[3]/=e,this.zIndex=o.Z.innerProduct(this.view.matrix3DRotShift[3],i),this}}),t.Z.createPolygon3D=function(e,r,o){var n,a,h,l,c,d,u=r[0],p=[],f=[],_=!1;if(c=s.Z.copyAttributes(o,e.options,"polygon3d"),null===(l=e.select(r[1]))&&(l=r[1]),s.Z.isObject(l)&&l.type===i.Z.OBJECT_TYPE_POLYGON3D&&s.Z.isTransformationOrArray(r[2]))for(_=!0,h=l.vertices.length-1,d=s.Z.copyAttributes(o,e.options,"polygon3d","vertices"),a=0;a<h;a++)d.withlabel&&(d.name=""===l.vertices[a].name?"":l.vertices[a].name+"'"),p.push(e.create("point3d",[l.vertices[a],r[2]],d));else if(s.Z.isArray(r[1])&&r[1].every((t=>s.Z.isPoint3D(t))))p=r[1];else if(!1===(p=s.Z.providePoints3D(u,r.slice(1),o,"polygon3d",["vertices"])))throw new Error("JSXGraph: Can't create polygon3d with parent types other than 'point' and 'coordinate arrays' or a function returning an array of coordinates. Alternatively, a polygon3d and a transformation can be supplied");for((n=new t.Z.Polygon3D(u,p,c)).isDraggable=!0,c=n.setAttr2D(c),a=0;a<p.length;a++)f.push(p[a].element2D);if(n.element2D=e.create("polygon",f,c),n.element2D.view=u,n.addChild(n.element2D),n.inherits.push(n.element2D),n.element2D.setParents(n),_)for(n.prepareUpdate().update().updateVisibility().updateRenderer(),h=l.vertices.length-1,a=0;a<h;a++)p[a].prepareUpdate().update().updateVisibility().updateRenderer();return n},t.Z.registerElement("polygon3d",t.Z.createPolygon3D),t.Z.Face3D=function(t,e,r,o){this.constructor(t.board,o,i.Z.OBJECT_TYPE_FACE3D,i.Z.OBJECT_CLASS_3D),this.constructor3D(t,"face3d"),this.board.finalizeAdding(this),this.polyhedron=e,this.faceNumber=r,this.normal=[0,0,0,0],this.d=0,this.vec1=[0,0,0,0],this.vec2=[0,0,0,0],0===this.faceNumber&&this.updateCoords(),this.methodMap=s.Z.deepCopy(this.methodMap,{})},t.Z.Face3D.prototype=new t.Z.GeometryElement,s.Z.copyPrototypeMethods(t.Z.Face3D,t.Z.GeometryElement3D,"constructor3D"),t.Z.extend(t.Z.Face3D.prototype,{updateCoords:function(){var t,e,i,r,o=this.polyhedron;for(t in o.vertices){if(r=o.vertices[t],s.Z.isFunction(r))o.coords[t]=s.Z.evaluate(r);else if(s.Z.isArray(r))for(o.coords[t]=[],i=r.length,e=0;e<i;e++)o.coords[t][e]=s.Z.evaluate(r[e]);else{if(r=o.view.select(r),!s.Z.isPoint3D(r))throw new Error("Polyhedron3D.updateCoords: unknown vertices type!");o.coords[t]=r.coords}3===o.coords[t].length&&o.coords[t].unshift(1)}return this},updateDataArray2D:function(){var t,e,i,s,r=[],n=[],a=this.polyhedron,h=a.faces[this.faceNumber];if(0===this.faceNumber)for(t in a.vertices)a.coords2D[t]=[];for(e=h.length,this.zIndex=0,t=0;t<e;t++)0===(s=a.coords2D[h[t]]).length&&(i=a.coords[h[t]],s=this.view.project3DTo2D(i),a.coords2D[h[t]]=s,a.zIndex[h[t]]=o.Z.innerProduct(this.view.matrix3DRotShift[3],i)),r.push(s[1]),n.push(s[2]),this.zIndex+=a.zIndex[h[t]];return e>0&&(this.zIndex/=e),2!==e&&(r.push(r[0]),n.push(n[0])),{X:r,Y:n}},addTransform:function(t,e){return 0===this.faceNumber&&this.addTransformGeneric(t,e),this},updateTransform:function(){var t,e,i,s,r;if(0!==this.faceNumber)return this;if(0===this.transformations.length||null===this.baseElement)return this;for(t=this.transformations,i=0;i<t.length;i++)t[i].update();for(i in(r=this===this.baseElement?this.polyhedron:this.baseElement.polyhedron).coords)if(r.coords.hasOwnProperty(i)){for(e=r.coords[i],s=0;s<t.length;s++)e=o.Z.matVecMult(t[s].matrix,e);this.polyhedron.coords[i]=e}return this},update:function(){var t,e,i,s,r,n;if(this.needsUpdate&&!this.view.board._change3DView){if(e=this.polyhedron,0===this.faceNumber&&this.updateCoords().updateTransform(),(n=e.faces[this.faceNumber]).length<3)return this;if(s=e.coords[n[0]],r=e.coords[n[1]],this.vec1=[r[0]-s[0],r[1]-s[1],r[2]-s[2],r[3]-s[3]],r=e.coords[n[2]],this.vec2=[r[0]-s[0],r[1]-s[1],r[2]-s[2],r[3]-s[3]],this.normal=o.Z.crossProduct(this.vec1.slice(1),this.vec2.slice(1)),i=o.Z.norm(this.normal),this.normal.unshift(0),Math.abs(i)>1e-12)for(t=1;t<4;t++)this.normal[t]/=i;this.d=o.Z.innerProduct(s,this.normal,4)}return this},updateRenderer:function(){return this.needsUpdate&&(this.needsUpdate=!1,this.shader()),this},shader:function(){var t,e,i,s,r,n,a,h,l;if(this.evalVisProp("shader.enabled"))return t=this.evalVisProp("shader.hue"),e=this.evalVisProp("shader.saturation"),h=this.evalVisProp("shader.minlightness"),l=this.evalVisProp("shader.maxlightness"),"angle"===this.evalVisProp("shader.type").toLowerCase()?(s=o.Z.innerProduct(this.view.matrix3DRotShift[3],this.normal),i=h+(l-h)*(s=Math.abs(s))):(a=this.view.zIndexMax,n=this.view.zIndexMin,i=h+(l-h)*((this.zIndex-n)/(a-n))),r="hsl("+t+","+e+"%,"+i+"%)",this.element2D.visProp.fillcolor=r,this.zIndex}}),t.Z.createFace3D=function(e,i,r){var o,n,a=i[0],h=i[1],l=i[2];return o=s.Z.copyAttributes(r,e.options,"face3d"),o=(n=new t.Z.Face3D(a,h,l,o)).setAttr2D(o),n.element2D=a.create("curve",[[],[]],o),n.element2D.view=a,n.element2D.updateDataArray=function(){var t=n.updateDataArray2D();this.dataX=t.X,this.dataY=t.Y},n.addChild(n.element2D),n.inherits.push(n.element2D),n.element2D.setParents(n),n.element2D.prepareUpdate().update(),e.isSuspendedUpdate||n.element2D.updateVisibility().updateRenderer(),n},t.Z.registerElement("face3d",t.Z.createFace3D),t.Z.Polyhedron3D=function(t,e,r,o){var n,a,h,l=this;for(this.constructor(t.board,o,i.Z.OBJECT_TYPE_POLYHEDRON3D,i.Z.OBJECT_CLASS_3D),this.constructor3D(t,"polyhedron3d"),this.board.finalizeAdding(this),this.elType="polyhedron3d",this.faces=r,this.numberFaces=r.length,this.def=e,a=["setAttribute","setParents","prepareUpdate","updateRenderer","update","fullUpdate","highlight","noHighlight"],h=function(t){return function(){var e;for(e in l.faces)l.faces.hasOwnProperty(e)&&s.Z.exists(l.faces[e][t])&&l.faces[e][t].apply(l.faces[e],arguments);return l}},n=0;n<a.length;n++)this[a[n]]=h(a[n]);this.methodMap=s.Z.deepCopy(this.methodMap,{setAttribute:"setAttribute",setParents:"setParents",addTransform:"addTransform"})},t.Z.Polyhedron3D.prototype=new t.Z.GeometryElement,s.Z.copyPrototypeMethods(t.Z.Polyhedron3D,t.Z.GeometryElement3D,"constructor3D"),t.Z.extend(t.Z.Polyhedron3D.prototype,{addTransform:function(t,e){if(!(this.faces.length>0&&t.faces.length>0))throw new Error("Adding transformation failed. At least one of the two polyhedra has no faces.");return this.faces[0].addTransform(t.faces[0],e),this},toSTL:function(t){var e,i,s,r,o,n,a="model ";for(void 0===t&&(t=this.name),a+=t+"\n",e=0;e<this.def.faces.length;e++){for(a+=" facet normal 0 0 0\n  outer loop\n",n=(r=this.def.faces[e]).length,s=this.def.coords,i=0;i<n;i++)a+="   vertex "+(o=s[r[i]])[1]+" "+o[2]+" "+o[3]+"\n";a+="  endloop\n endfacet\n"}return a+="endsolid "+t+"\n"}}),t.Z.createPolyhedron3D=function(e,r,o){var n,a,h,l,c,d,u,p=r[0],f=[],_=null,m=null,g={view:p,vertices:{},coords:{},coords2D:{},zIndex:{},faces:[]};if(s.Z.exists(r[1].type)&&r[1].type===i.Z.OBJECT_TYPE_POLYHEDRON3D)_=r[1],m=r[2],g.vertices=_.def.vertices,g.faces=_.def.faces;else{if(s.Z.isArray(r[1]))for(a=r[1].length,n=0;n<a;n++)g.vertices[n]=r[1][n];else if(s.Z.isObject(r[1]))for(n in r[1])r[1].hasOwnProperty(n)&&(g.vertices[n]=r[1][n]);g.faces=r[2]}for(u=s.Z.copyAttributes(o,e.options,"polyhedron3d"),console.time("polyhedron"),p.board.suspendUpdate(),a=g.faces.length,n=0;n<a;n++)d=s.Z.copyAttributes(o,e.options,"face3d"),u.fillcolorarray.length>0&&(d.fillcolor=u.fillcolorarray[n%u.fillcolorarray.length]),l=g.faces[n],s.Z.isArray(l)&&2===l.length&&s.Z.isObject(l[1])&&s.Z.isArray(l[0])&&(s.Z.mergeAttr(d,l[1]),g.faces[n]=l[0]),h=p.create("face3d",[g,n],d),f.push(h);for((c=new t.Z.Polyhedron3D(p,g,f,u)).setParents(c),n=0;n<a;n++)c.inherits.push(c.faces[n]),c.addChild(c.faces[n]);return null!==_&&(c.addTransform(_,m),c.addParents(_)),p.board.unsuspendUpdate(),console.timeEnd("polyhedron"),c},t.Z.registerElement("polyhedron3d",t.Z.createPolyhedron3D),t.Z.Sphere3D=function(t,e,r,o,n){this.constructor(t.board,n,i.Z.OBJECT_TYPE_SPHERE3D,i.Z.OBJECT_CLASS_3D),this.constructor3D(t,"sphere3d"),this.board.finalizeAdding(this),this.method=e,this.center=this.board.select(r),this.point2=null,this.points=[],this.element2D=null,this.aux2D=[],this.projectionType=t.projectionType,"twoPoints"===e?(this.point2=this.board.select(o),this.radius=this.Radius()):"pointRadius"===e&&(this.updateRadius=s.Z.createFunction(o,this.board),this.updateRadius(),this.addParentsFromJCFunctions([this.updateRadius])),s.Z.exists(this.center._is_new)?(this.addChild(this.center),delete this.center._is_new):this.center.addChild(this),"twoPoints"===e&&(s.Z.exists(this.point2._is_new)?(this.addChild(this.point2),delete this.point2._is_new):this.point2.addChild(this)),this.methodMap=s.Z.deepCopy(this.methodMap,{center:"center",point2:"point2",Radius:"Radius"})},t.Z.Sphere3D.prototype=new t.Z.GeometryElement,s.Z.copyPrototypeMethods(t.Z.Sphere3D,t.Z.GeometryElement3D,"constructor3D"),t.Z.extend(t.Z.Sphere3D.prototype,{X:function(t,e){return this.Radius()*Math.sin(t)*Math.cos(e)},Y:function(t,e){return this.Radius()*Math.sin(t)*Math.sin(e)},Z:function(t,e){return this.Radius()*Math.cos(t)},range_u:[0,2*Math.PI],range_v:[0,Math.PI],update:function(){return this.projectionType!==this.view.projectionType&&this.rebuildProjection(),this},updateRenderer:function(){return this.needsUpdate=!1,this},setRadius:function(t){return this.updateRadius=s.Z.createFunction(t,this.board),this.addParentsFromJCFunctions([this.updateRadius]),this.board.update(),this},Radius:function(t){return s.Z.exists(t)?(this.setRadius(t),this.Radius()):"twoPoints"===this.method?this.center.testIfFinite()&&this.point2.testIfFinite()?this.center.distance(this.point2):NaN:"pointRadius"===this.method?Math.abs(this.updateRadius()):NaN},focusFn:function(t){var e=this;return function(){var i=e.view.boxToCam[3],s=e.Radius();return e.view.project3DTo2D([e.center.X()+t*s*i[1],e.center.Y()+t*s*i[2],e.center.Z()+t*s*i[3]]).slice(1,3)}},innerVertexFn:function(){var t=this;return function(){var e,i=t.view,s=i.worldToFocal(t.center.coords,!1),r=o.Z.hypot(s[0],s[1]),n=i.boxToCam,a=t.Radius(),h=Math.atan(-r/s[2])+Math.acos(a/o.Z.norm(s)),l=Math.cos(h),c=Math.sin(h);return e=r>1e-8?[-(s[0]*n[1][1]+s[1]*n[2][1])/r,-(s[0]*n[1][2]+s[1]*n[2][2])/r,-(s[0]*n[1][3]+s[1]*n[2][3])/r]:[n[1][1],n[1][2],n[1][3]],i.project3DTo2D([t.center.X()+a*(c*e[0]+l*n[3][1]),t.center.Y()+a*(c*e[1]+l*n[3][2]),t.center.Z()+a*(c*e[2]+l*n[3][3])])}},buildCentralProjection:function(t){var e=this.view,i={visible:!1,withLabel:!1},s=e.create("point",this.focusFn(-1),i),r=e.create("point",this.focusFn(1),i),o=e.create("point",this.innerVertexFn(e),i);this.aux2D=[s,r,o],this.element2D=e.create("ellipse",this.aux2D,void 0===t?this.visProp:t)},buildParallelProjection:function(t){var e=this;this.aux2D=[],this.element2D=this.view.create("circle",[e.center.element2D,function(){var t=e.view.bbox3D[0][1]-e.view.bbox3D[0][0];return e.Radius()*e.view.size[0]/t}],void 0===t?this.visProp:t)},rebuildProjection:function(t){var e;if(this.element2D)for(e in this.view.board.removeObject(this.element2D),this.aux2D)this.aux2D.hasOwnProperty(e)&&this.view.board.removeObject(this.aux2D[e]);this.projectionType=this.view.projectionType,"central"===this.projectionType?this.buildCentralProjection(t):this.buildParallelProjection(t),this.addChild(this.element2D),this.inherits.push(this.element2D),this.element2D.view=this.view},projectCoords:function(t,e){var i=this.Radius(),s=[1].concat(t),r=this.center.coords,o=h.Z.distance(r,s,4),n=p.Z.subtract(s,r);return 0===o?(e[0]=0,e[1]=0,[1,i,0,0]):0===i?(e[0]=0,e[1]=0,this.center.coords):(o=i/o,n[0]=1,n[1]*=o,n[2]*=o,n[3]*=o,e[1]=Math.atan2(n[2],n[1]),e[1]+=e[1]<0?Math.PI:0,0!==e[1]?e[0]=Math.atan2(n[2],n[3]*Math.sin(e[1])):e[0]=Math.atan2(n[1],n[3]*Math.cos(e[1])),e[0]+=e[0]<0?2*Math.PI:0,n)}}),t.Z.createSphere3D=function(e,i,r){var o,n,a,h,l,c,d=i[0];for(o=s.Z.copyAttributes(r,e.options,"sphere3d"),n=[],c=1;c<i.length;c++)if(s.Z.isPointType3D(e,i[c])){if(a=0===n.length?"center":"point",!1===(h=s.Z.providePoints3D(d,[i[c]],r,"sphere3d",[a])[0]))throw new Error("JSXGraph: Can't create sphere3d from this type. Please provide a point type.");n.push(h)}else n.push(i[c]);if(s.Z.isPoint3D(n[0])&&s.Z.isPoint3D(n[1]))l=new t.Z.Sphere3D(d,"twoPoints",n[0],n[1],o);else{if(!s.Z.isPoint3D(n[0])||!(s.Z.isNumber(n[1])||s.Z.isFunction(n[1])||s.Z.isString(n[1])))throw new Error("JSXGraph: Can't create sphere3d with parent types '"+typeof i[1]+"' and '"+typeof i[2]+"'.\nPossible parent types: [point,point], [point,number], [point,function]");l=new t.Z.Sphere3D(d,"pointRadius",n[0],n[1],o)}return o=l.setAttr2D(o),l.rebuildProjection(o),l.element2D.prepareUpdate().update(),e.isSuspendedUpdate||l.element2D.updateVisibility().updateRenderer(),l},t.Z.registerElement("sphere3d",t.Z.createSphere3D),t.Z.Surface3D=function(t,e,r,o,n,a,h,l){this.constructor(t.board,l,i.Z.OBJECT_TYPE_SURFACE3D,i.Z.OBJECT_CLASS_3D),this.constructor3D(t,"surface3d"),this.board.finalizeAdding(this),this._F=e,this._X=r,this._Y=o,this._Z=n,null!==this._F?(this._X=function(t,e){return this._F(t,e)[0]},this._Y=function(t,e){return this._F(t,e)[1]},this._Z=function(t,e){return this._F(t,e)[2]}):null!==this._X&&(this._F=function(t,e){return[this._X(t,e),this._Y(t,e),this._Z(t,e)]}),this.range_u=a,this.range_v=h,this.dataX=null,this.dataY=null,this.dataZ=null,this.points=[],this.methodMap=s.Z.deepCopy(this.methodMap,{})},t.Z.Surface3D.prototype=new t.Z.GeometryElement,s.Z.copyPrototypeMethods(t.Z.Surface3D,t.Z.GeometryElement3D,"constructor3D"),t.Z.extend(t.Z.Surface3D.prototype,{updateWireframe:function(){var t,e,i,r,o,n,a,h,l,c,d,u,p=[1,0,0,0];for(this.points=[],t=this.evalVisProp("stepsu"),e=this.evalVisProp("stepsv"),o=s.Z.evaluate(this.range_u),n=s.Z.evaluate(this.range_v),a=s.Z.evaluate(o[0]),h=s.Z.evaluate(n[0]),l=(s.Z.evaluate(o[1])-a)/t,c=(s.Z.evaluate(n[1])-h)/e,i=0,d=a;i<=t;i++,d+=l)for(this.points.push([]),r=0,u=h;r<=e;r++,u+=c)(p=this.F(d,u)).unshift(1),this.points[i].push(p);return this},updateCoords:function(){return null!==this._F?this.updateWireframe():this.updateTransform(),this},evalF:function(t,e){var i,r,n=[0,0,0,0];if(0===this.transformations.length||!s.Z.exists(this.baseElement))return n=this._F(t,e);for(i=this.transformations,r=0;r<i.length;r++)i[r].update();for((n=this===this.baseElement?this._F(t,e):this.baseElement.evalF(t,e)).unshift(1),n=o.Z.matVecMult(i[0].matrix,n),r=1;r<i.length;r++)n=o.Z.matVecMult(i[r].matrix,n);return n.slice(1)},F:function(t,e){return this.evalF(t,e)},X:function(t,e){return this.evalF(t,e)[0]},Y:function(t,e){return this.evalF(t,e)[1]},Z:function(t,e){return this.evalF(t,e)[2]},updateDataArray2D:function(){var t,e,i,s,r,o=[],n=[];if(0!==(i=this.points.length)){for(s=this.points[0].length,t=0;t<i;t++){for(e=0;e<s;e++)r=this.view.project3DTo2D(this.points[t][e]),o.push(r[1]),n.push(r[2]);o.push(NaN),n.push(NaN)}for(e=0;e<s;e++){for(t=0;t<i;t++)r=this.view.project3DTo2D(this.points[t][e]),o.push(r[1]),n.push(r[2]);o.push(NaN),n.push(NaN)}}return{X:o,Y:n}},addTransform:function(t,e){return this.addTransformGeneric(t,e),this},updateTransform:function(){var t,e,i,r,n,a,h;if(0===this.transformations.length||null===this.baseElement||s.Z.exists(this._F))return this;for(t=this.transformations,i=0;i<t.length;i++)t[i].update();if(this!==this.baseElement&&(this.points=[]),(a=this.baseElement.points.length)>0)for(h=this.baseElement.points[0].length,i=0;i<a;i++)for(this!==this.baseElement&&this.points.push([]),r=0;r<h;r++){for(e=this===this.baseElement?this.points[i][r]:this.baseElement.points[i][r],n=0;n<t.length;n++)e=o.Z.matVecMult(t[n].matrix,e);this===this.baseElement?this.points[i][r]=e:this.points[i].push(e)}return this},updateDataArray:function(){},update:function(){return this.needsUpdate&&(this.updateDataArray(),this.updateCoords()),this},updateRenderer:function(){return this.needsUpdate=!1,this},projectCoords:function(t,e){return h.Z.projectCoordsToParametric(t,this,2,e)}}),t.Z.createParametricSurface3D=function(e,i,r){var o,n,a,h,l,c,d,u,p=i[0],f=null,_=null;return 3===i.length?(f=i[1],_=i[2],o=null,n=null,a=null,h=null):4===i.length?(o=i[1],l=i[2],c=i[3],n=null,a=null,h=null):(n=i[1],a=i[2],h=i[3],l=i[4],c=i[5],o=null),d=s.Z.copyAttributes(r,e.options,"surface3d"),d=(u=new t.Z.Surface3D(p,o,n,a,h,l,c,d)).setAttr2D(d),u.element2D=p.create("curve",[[],[]],d),u.element2D.view=p,null!==f&&(u.addTransform(f,_),u.addParents(f)),u.element2D.updateDataArray=function(){var t=u.updateDataArray2D();this.dataX=t.X,this.dataY=t.Y},u.addChild(u.element2D),u.inherits.push(u.element2D),u.element2D.setParents(u),u.element2D.prepareUpdate().update(),e.isSuspendedUpdate||u.element2D.updateVisibility().updateRenderer(),u},t.Z.registerElement("parametricsurface3d",t.Z.createParametricSurface3D),t.Z.createFunctiongraph3D=function(t,e,i){var r,o=e[0],n=s.Z.createFunction(e[1],t,"x, y"),a=e[2],h=e[3];return(r=o.create("parametricsurface3d",[function(t,e){return t},function(t,e){return e},n,a,h],i)).elType="functiongraph3d",r},t.Z.registerElement("functiongraph3d",t.Z.createFunctiongraph3D),t.Z.Parse3D={STL:function(e){var i,s,r,o,n,a,h,l,c,d=[],u=[],p=[];for(o=(a=e.split("\n")).length,i=0;i<o;i++)if(0===(n=a[i].trim()).indexOf("solid"))l=-1,u=[],p=[];else if(0===n.indexOf("endsolid"))d.push([u.slice(),p.slice()]);else if(0===n.indexOf("facet"))l++,p.push([]);else{if(0===n.indexOf("outer loop")||0===n.indexOf("endloop"))continue;if(0===n.indexOf("vertex")){for(h=n.split(" ").slice(1).map((t=>parseFloat(t))),c=!1,s=0;s<u.length;s++)if(t.Z.Math.Geometry.distance(u[s],h,3)<t.Z.Math.eps){c=!0,r=s;break}!1===c&&(r=u.length,u.push(h)),p[l].push(r)}}return d}};t.Z.Parse3D;let st=.75,rt="#000";t.Z.themes.mono_thin={board:{showInfobox:!1,showCopyright:!0,defaultAxes:{x:{ticks:{minorTicks:0,majorHeight:10,majorTickEndings:[1,0]}},y:{ticks:{minorTicks:0,majorHeight:10,majorTickEndings:[0,1]}}}},navbar:{strokeColor:"#bbb",fillColor:"none"},elements:{strokeColor:rt,highlightStrokeColor:rt,fillColor:"none",highlightFillColor:"none",strokeOpacity:.6,highlightStrokeOpacity:1},angle:{strokeColor:rt,fillColor:"#aaaaaa55",fillOpacity:.3,highlightFillColor:"#aaaaaa33",highlightFillOpacity:.3,label:{strokeColor:rt}},arc:{strokeColor:rt,strokeWidth:st,highlightStrokeColor:rt,highlightStrokeWidth:st},axis:{},boxplot:{strokeWidth:st,strokeColor:rt,fillColor:rt,fillOpacity:.2,highlightStrokeWidth:st,highlightStrokeColor:rt,highlightFillColor:rt,highlightFillOpacity:.1},circle:{strokeWidth:st,highlightStrokeWidth:1.125,strokeColor:rt,highlightFillColor:"none",highlightStrokeColor:rt,center:{size:st,fillColor:rt,strokeColor:rt,highlightStrokeWidth:3,highlightFillColor:rt,highlightStrokeColor:rt},point2:{size:st,fillColor:rt,strokeColor:rt,highlightStrokeWidth:3,highlightFillColor:rt,highlightStrokeColor:rt}},circumcircle:{strokeWidth:st,highlightStrokeWidth:1.125,strokeColor:rt,highlightFillColor:"none",highlightStrokeColor:rt,center:{size:st,fillColor:rt,strokeColor:rt,highlightStrokeWidth:3,highlightFillColor:rt,highlightStrokeColor:rt}},circumcirclearc:{strokeColor:rt,strokeWidth:st,highlightStrokeColor:rt,highlightStrokeWidth:st},circumcirclesector:{strokeColor:rt,fillColor:"#aaaaaa55",fillOpacity:.3,highlightFillColor:"#aaaaaa33",highlightFillOpacity:.3},comb:{strokeColor:rt,strokeWidth:st},conic:{strokeWidth:st,highlightStrokeWidth:1.125,strokeColor:rt,highlightStrokeColor:rt,fillColor:"none",highlightFillColor:"none"},curve:{strokeColor:rt,strokeWidth:st,highlightStrokeColor:rt,highlightStrokeWidth:st},grid:{strokeWidth:st},hatch:{strokeColor:rt,strokeWidth:st},incircle:{strokeWidth:st,highlightStrokeWidth:1.125,strokeColor:rt,highlightFillColor:"none",highlightStrokeColor:rt,center:{size:st,fillColor:rt,strokeColor:rt,highlightStrokeWidth:3,highlightFillColor:rt,highlightStrokeColor:rt}},inequality:{fillColor:"#aaaaaa55",fillOpacity:.2},integral:{fillColor:"#aaaaaa55",highlightFillColor:"#aaaaa33",fillOpacity:.3,highlightFillOpacity:.3,curveLeft:{color:rt},baseLeft:{color:rt},curveRight:{color:rt},baseRight:{color:rt}},label:{strokeColor:rt},line:{strokeColor:rt,strokeWidth:st,highlightStrokeColor:rt,highlightStrokeWidth:st,point1:{size:st,fillColor:rt,strokeColor:rt,highlightStrokeWidth:3,highlightFillColor:rt,highlightStrokeColor:rt},point2:{size:st,fillColor:rt,strokeColor:rt,highlightStrokeWidth:3,highlightFillColor:rt,highlightStrokeColor:rt}},normal:{strokeColor:rt},parallel:{strokeColor:rt},perpendicular:{strokeColor:rt},perpendicularsegment:{strokeColor:rt},point:{size:st,fillColor:rt,strokeColor:rt,highlightStrokeWidth:3,highlightFillColor:rt,highlightStrokeColor:rt},polygon:{fillColor:"#aaaaaa55",highlightFillColor:"#aaaaaa33",fillOpacity:.3,highlightFillOpacity:.3,vertices:{size:st,fillColor:rt,strokeColor:rt,highlightStrokeWidth:3,highlightFillColor:rt,highlightStrokeColor:rt},borders:{strokeColor:rt,strokeWidth:st,highlightStrokeColor:rt,highlightStrokeWidth:st}},sector:{strokeColor:rt,fillColor:"#aaaaaa55",fillOpacity:.3,highlightFillColor:"#aaaaaa33",highlightFillOpacity:.3},semicircle:{center:{size:st,fillColor:rt,strokeColor:rt,highlightStrokeWidth:3,highlightFillColor:rt,highlightStrokeColor:rt}},slider:{size:st,fillColor:rt,strokeColor:rt,highlightStrokeWidth:3,highlightFillColor:rt,highlightStrokeColor:rt,baseline:{strokeWidth:st,strokeColor:rt,highlightStrokeColor:rt},label:{strokeColor:rt},highline:{strokeWidth:2.25,name:"",strokeColor:rt,highlightStrokeColor:rt},ticks:{strokeColor:rt}},slopefield:{strokeWidth:.5625,highlightStrokeWidth:st,highlightStrokeColor:rt,highlightStrokeOpacity:.8},tapemeasure:{strokeColor:rt,strokeWidth:st,highlightStrokeColor:rt,highlightStrokeWidth:st,point1:{size:st,fillColor:rt,strokeColor:rt,highlightStrokeWidth:3,highlightFillColor:rt,highlightStrokeColor:rt},point2:{size:st,fillColor:rt,strokeColor:rt,highlightStrokeWidth:3,highlightFillColor:rt,highlightStrokeColor:rt},ticks:{strokeWidth:st}},text:{strokeColor:rt},tracecurve:{strokeColor:rt},turtle:{strokeWidth:st,strokeColor:rt,arrow:{strokeWidth:1.5,strokeColor:"#aaaaaa55"}},vectorfield:{strokeWidth:.5625,highlightStrokeWidth:st,highlightStrokeColor:rt,highlightStrokeOpacity:.8}};t.Z.COORDS_BY_SCREEN,t.Z.COORDS_BY_USER,t.Z.Dump,t.Z.Expect,t.Z.JSXGraph,t.Z.Math,t.Z.Options,t.Z.boards,t.Z.elements,t.Z.palette,t.Z.paletteWong,t.Z.Board,t.Z.Chart,t.Z.Circle,t.Z.Complex,t.Z.Composition,t.Z.Coords,t.Z.CoordsElement,t.Z.Curve,t.Z.GeometryElement,t.Z.Group,t.Z.Image,t.Z.JessieCode,t.Z.PrefixParser,t.Z.Line,t.Z.Point,t.Z.Polygon,t.Z.Text,t.Z.Ticks,t.Z.Transformation,t.Z.Turtle,t.Z.View3D,t.Z.LMS2rgb,t.Z.addEvent,t.Z.autoDigits,t.Z.autoHighlight,t.Z.bind,t.Z.capitalize,t.Z.clearVisPropOld,t.Z.clone,t.Z.cloneAndCopy,t.Z.cmpArrays,t.Z.coordsArrayToMatrix,t.Z.copyAttributes,t.Z.createEvalFunction,t.Z.createFunction,t.Z.createHTMLSlider,t.Z.darkenColor,t.Z.debug,t.Z.debugInt,t.Z.debugLine,t.Z.debugWST,t.Z.deepCopy,t.Z.def,t.Z.deprecated,t.Z.eliminateDuplicates,t.Z.escapeHTML,t.Z.evalSlider,t.Z.evaluate,t.Z.filterElements,t.Z.getBoardByContainerId,t.Z.getCSSTransformMatrix,t.Z.getCSSTransform,t.Z.getDimensions,t.Z.getOffset,t.Z.getPosition,t.Z.getProp,t.Z.hex2rgb,t.Z.hsv2rgb,t.Z.isAndroid,t.Z.isApple,t.Z.isArray,t.Z.isDesktop,t.Z.isInArray,t.Z.isInObject,t.Z.isMetroApp,t.Z.isMobile,t.Z.isMozilla,t.Z.isBoard,t.Z.isName,t.Z.isNode,t.Z.isNumber,t.Z.isObject,t.Z.isPoint,t.Z.isPoint3D,t.Z.isPointType,t.Z.isPointType3D,t.Z.isString,t.Z.isTouchDevice,t.Z.isTransformationOrArray,t.Z.isWebWorker,t.Z.isWebkitAndroid,t.Z.isWebkitApple,t.Z.keys,t.Z.lightenColor,t.Z.merge,t.Z.normalizePointFace,t.Z.providePoints,t.Z.registerElement,t.Z.registerReader,t.Z.removeAllEvents,t.Z.removeElementFromArray,t.Z.removeEvent,t.Z.rgb2LMS,t.Z.rgb2bw,t.Z.rgb2cb,t.Z.rgb2css,t.Z.rgb2hex,t.Z.rgb2hsv,t.Z.rgbParser,t.Z.rgba2rgbo,t.Z.rgbo2rgba,t.Z.sanitizeHTML,t.Z.shortcut,t.Z.str2Bool,t.Z.supportsCanvas,t.Z.supportsPointerEvents,t.Z.supportsSVG,t.Z.supportsVML,t.Z.swap,t.Z.timedChunk,t.Z.toFixed,t.Z.toFullscreen,t.Z.toJSON,t.Z.trim,t.Z.trimNumber,t.Z.truncate,t.Z.unescapeHTML,t.Z.uniqueArray,t.Z.useBlackWhiteOptions,t.Z.useStandardOptions,t.Z.warn;e.Z.isBrowser?window.JXG=t.Z:e.Z.isWebWorker()&&(self.JXG=t.Z);var ot=t.Z}(),__webpack_exports__=__webpack_exports__.default,__webpack_exports__}()}));

try {
    JXG = window.JXG = module.exports;
} catch(e) {
}

JXG.Options.text.useMathJax = true;
