// Original: https://github.com/numbas/numbas-extension-linearalgebra
// Changes / extensions by Ulrich Görtz


Numbas.addExtension('linearalgebra2',['jme','jme-display'],function(extension) {

var math = Numbas.math;
var vectormath = Numbas.vectormath;
var matrixmath = Numbas.matrixmath;

var Fraction = extension.Fraction = function(n) {
    if(typeof(n)=='number') {
        var c = math.rationalApproximation(n);
        this.n = c[0];
        this.d = c[1];
    } else {
        this.n = n.n;
        this.d = n.d;
    }
    this.tidy();
}
Fraction.prototype = {
    toString: function() {
        return this.d==1 ? this.n+'' : this.n+'/'+this.d;
    },

    toLaTeX: function() {
        return this.d==1 ? this.n+'' : (this.n<0 ? '-': '')+'\\frac{'+(Math.abs(this.n))+'}{'+this.d+'}';
    },

    /** Ensure fraction is reduced, and denominator is positive
     */
    tidy: function() {
        if(this.n==0) {
            this.n = 0;
            this.d = 1;
        }
        var g = math.gcd(this.n,this.d) * (this.d<0 ? -1 : 1);
        this.n /= g;
        this.d /= g;
    },

    add: function(f2) {
        return new Fraction({n: this.n*f2.d + f2.n*this.d, d: this.d*f2.d});
    },
    sub: function(f2) {
        return new Fraction({n: this.n*f2.d - f2.n*this.d, d: this.d*f2.d});
    },
    mul: function(f2) {
        return new Fraction({n: this.n*f2.n, d: this.d*f2.d});
    },
    div: function(f2) {
        return new Fraction({n: this.n*f2.d, d: this.d*f2.n});
    },
    is_zero: function() {
        return this.n==0;
    },
    is_one: function() {
        return this.n==this.d;
    },
    reciprocal: function() {
        return new Fraction({n:this.d,d:this.n});
    }
}

var fraction_matrix = extension.fraction_matrix = function(matrix) {
    var o = matrix.map(function(r){return r.map(function(c){ return c instanceof Fraction ? c : new Fraction(c)})});
    o.rows = matrix.rows;
    o.columns = matrix.columns;
    return o;
}
var unfraction_matrix = extension.unfraction_matrix = function(matrix) {
    var o = matrix.map(function(r){return r.map(function(c){return c.n/c.d})});
    o.rows = matrix.rows;
    o.columns = matrix.columns;
    return o;
}

function wrap_fraction_matrix(fn,unwrap_res) {
    return function(matrix) {
        matrix = fraction_matrix(matrix);
        var res = fn(matrix);
        if(unwrap_res) {
            res = res.matrix;
        }
        return unfraction_matrix(res);
    }
}

function logger(operations,matrix) {
    return function log(message,include_matrix,options) {
        include_matrix = include_matrix===undefined ? true : include_matrix;
        var lmatrix;
        if(include_matrix) {
            lmatrix = matrix.map(function(r){return r.slice()});
            lmatrix.rows = matrix.rows;
            lmatrix.columns = matrix.columns;
        }
        var l = {message:message, matrix: lmatrix};
        for(var key in options) {
            l[key] = options[key];
        }
        operations.push(l);
    }
}

var row_echelon_form = function(matrix) {
    /** Put a matrix representing a system of equations in row-echelon form.
    * Can:
    * * Swap two rows
    * * Multiply a row by a scalar
    * * Subtract a multiple of one row from another
    * For each row of the output, the first non-zero entry is 1, and strictly to the right of the first non-zero entry in the row above.
    * Works over the rationals: input is a matrix of objects {n: numerator,d: denominator}.
    * Output is an object {matrix, operations}, where operations is a list of descriptions of each step of the process, of the form {message: string, matrix: the state of the matrix after the operation}.
    */
    matrix = matrix.map(function(r){return r.slice()});
    var rows = matrix.length;
    var columns = matrix[0].length;
    matrix.rows = rows;
    matrix.columns = columns;
    
    var operations = [];
    var log = logger(operations,matrix);

    var current_row = 0;
    // for each column, there should be at most one row with a 1 in that column, and every other row should have 0 in that column
    for(var leader_column=0;leader_column<columns;leader_column++) {
        // find the first row with a non-zero in that column
        for(var row=current_row;row<rows;row++) {
            if(!matrix[row][leader_column].is_zero()) {
                break;
            }
        }
        // if we found a row with a non-zero in the leader column 
        if(row<rows) {
            // swap that row with the <current_row>th one
            if(row!=current_row) {
                var tmp = matrix[row];
                matrix[row] = matrix[current_row];
                matrix[current_row] = tmp;
                log("Row "+(row+1)+" has a non-zero entry in column "+(leader_column+1)+", so it should go before row "+(current_row+1)+". Swap row "+(row+1)+" with row "+(current_row+1)+".",true,{determinant_scale:new Fraction(-1)});
            }

            // multiply this row so the leader column has a 1 in it
            var leader = matrix[current_row][leader_column];
            if(!leader.is_one()) {
                matrix[current_row] = matrix[current_row].map(function(c){ return c.div(leader)});
                log("Divide row "+(current_row+1)+" by \\("+leader+"\\), so that the first non-zero entry is \\(1\\).",true, {determinant_scale:leader.reciprocal()});
            }

            // subtract multiples of this row from every other row so they all have a zero in this column
            var sub = function(a,b){ return a.sub(b); };
            var add = function(a,b){ return a.add(b); };
            for(var row=current_row+1;row<rows;row++) {
                if(row!=current_row && !matrix[row][leader_column].is_zero()) {
                    var scale = matrix[row][leader_column];
                    var op = sub;
                    if(scale.n<0) {
                        scale = new Fraction({n:-scale.n,d:scale.d});
                        op = add;
                    }
                    matrix[row] = matrix[row].map(function(c,i) { 
                        var res = op(c,matrix[current_row][i].mul(scale));
                        return res;
                    });
                    var mop = op==sub ? "Subtract" : "Add";
                    var mverb = op==sub ? "from" : "to";
                    log(mop+" "+(scale.is_one() ? "" : "\\("+scale+"\\) times ")+"row "+(current_row+1)+" "+mverb+" row "+(row+1)+".");
                }
            }
            current_row += 1;
        }
    }
    if(operations.length>0) {
        log("The matrix is now in row echelon form.",false);
    }
    return {
        matrix: matrix,
        operations: operations
    };
}
extension.row_echelon_form = wrap_fraction_matrix(row_echelon_form,true);

var reduced_row_echelon_form = function(matrix) {
    /** Put a matrix representing a system of equations in reduced row-echelon form.
     * Can:
     * * Swap two rows
     * * Multiply a row by a scalar
     * * Subtract a multiple of one row from another
     * As well as being in row-echelon form, the matrix has the property that the first non-zero entry in each row is also the only non-zero entry in its column.
     * Works over the rationals: input is a matrix of objects {n: numerator,d: denominator}.
     * Output is an object {matrix, operations}, where operations is a list of descriptions of each step of the process, of the form {message: string, matrix: the state of the matrix after the operation}.
     */
    matrix = matrix.map(function(r){return r.slice()});
    var res = row_echelon_form(matrix);
    matrix = res.matrix;
    var operations = res.operations.slice();

    var rows = matrix.length;
    var columns = matrix[0].length;
    matrix.rows = rows;
    matrix.columns = columns;

    var log = logger(operations,matrix);

    var sub = function(a,b){ return a.sub(b); };
    var add = function(a,b){ return a.add(b); };

    for(var row=0;row<rows;row++) {
        for(var column=0;column<columns && matrix[row][column].is_zero();column++) {}
        
        if(column==columns) {
            continue;
        }
        for(var vrow = 0;vrow<rows;vrow++) {
            if(vrow!=row && !matrix[vrow][column].is_zero()) {
                
                var scale = matrix[vrow][column];
                if(!scale.is_zero()) {
                    var op = sub;
                    if(scale.n<0) {
                        op = add;
                        scale = new Fraction({n:-scale.n, d:scale.d});
                    }
                    matrix[vrow] = matrix[vrow].map(function(c,i) { 
                        return op(c,matrix[row][i].mul(scale));
                    });

                    var mop = op==sub ? "subtract" : "add";
                    var mverb = op==sub ? "from" : "to";
                    log("We want a zero in column "+(column+1)+" of row "+(vrow+1)+": "+mop+" "+(scale.is_one() ? "" : "\\("+scale+"\\) times ")+"row "+(row+1)+" "+mverb+" row "+(vrow+1)+".");
                }
            }
        }
    }
    if(operations.length>0) {
        log("The matrix is now in reduced row echelon form.",false);
    }
    return {
        matrix: matrix,
        operations: operations
    };
}
extension.reduced_row_echelon_form = wrap_fraction_matrix(reduced_row_echelon_form,true);

function rref_without_zero(matrix) {
    var rref = reduced_row_echelon_form(matrix).matrix;
    return rref.filter(function(row) { return !vectormath.is_zero(row); });
}

function rank(matrix) {
    return rref_without_zero(matrix).length;
}
extension.rank = function(matrix) {
    matrix = fraction_matrix(matrix);
    return rank(matrix);
}

function is_linearly_independent(vectors) {
    return rref_without_zero(vectors).length==vectors.length;
}
extension.is_linearly_independent = function(vectors) {
    vectors = fraction_matrix(vectors);
    return is_linearly_independent(vectors);
}

var adjoin = extension.adjoin = function(matrix,vector) {
    var o = [];
    for(var i=0;i<matrix.length;i++) {
        var row = matrix[i].slice();
        row.push(vector[i] || 0);
        o.push(row);
    }
    o.rows = matrix.rows;
    o.columns = matrix.columns+1;
}

/** Subset of the given vectors, with the given dimension.
 * Not always possible - if the vectors have length k, you can't have d>k. 
 * If the input list has dimension less than d, it can't be done.
 * Likewise with extra dependent vectors - if there aren't enough, it'll fail.
 * The vectors are processed in order, so if you want a random subset you should shuffle the list first.
 *
 * @param {Array.<vector>} vectors 
 * @param {Number} n - number of vectors to return
 * @param {Number} d - dimension of the set (the first d vectors will be linearly independent, and any others will be multiples of those.
 * @returns {Array.<vector>}
 */
function subset_with_dimension(vectors,n,d) {
    vectors = vectors.filter(function(v){ return !vectormath.is_zero(v); });
    var independent = [];
    var combos = [];
    for(var i=0; i<vectors.length && (independent.length<d || combos.length<n-d); i++) {
        var v = vectors[i];
        if(is_linearly_independent(independent.concat([v]))) {
            independent.push(v);
        } else {
            combos.push(v);
        }
    }
    if(independent.length<d || combos.length<n-d) {
        throw(new Error("Couldn't generate a subset of the required size and dimension"));
    }
    return independent.slice(0,d).concat(combos.slice(0,n-d));
}

extension.subset_with_dimension = function(vectors,n,d) {
    vectors = fraction_matrix(vectors);
    var subset = subset_with_dimension(vectors,n,d);
    return unfraction_matrix(subset);
}

/** Span of vectors in Z^n, with no element bigger than max
 */
function span(vectors,max) {
    var dim = vectors[0].length;
    var zero = [];
    for(var i=0;i<dim;i++) {
        zero.push(0);
    }
    var out = [zero];
    vectors.forEach(function(v) {
        var biggest = v.reduce(function(best,x){ return Math.max(best,Math.abs(x)); },0);
        var lim = vectors.length*max/biggest;
        var mults = [];
        for(var i=0;i<=lim;i++) {
            mults.push(vectormath.mul(i,v));
            mults.push(vectormath.mul(-i,v));
        }
        
        var nout = [];
        out.forEach(function(v2) {
            mults.forEach(function(m) {
                var s = vectormath.add(m,v2);
                if(!nout.find(function(v3){return vectormath.eq(v3,s)})) {
                    nout.push(s);
                }
            });
        });
        out = nout;
    })
    out = out.filter(function(v){ return v.every(function(x){return Math.abs(x)<=max}) });
    return out;
}
extension.span = span;

function as_sum_of_basis(basis,v) {
    basis.rows = basis.length;
    basis.columns = basis.rows>0 ? basis[0].length : 0;
    var matrix = matrixmath.transpose(basis);
    var augmented_matrix = matrix.map(function(row,i) {
        row = row.slice();
        row.push(v[i] || 0);
        return row;
    });
    augmented_matrix = fraction_matrix(augmented_matrix);
    augmented_matrix.rows = matrix.rows;
    augmented_matrix.columns = matrix.columns;
    var rref = reduced_row_echelon_form(augmented_matrix).matrix;
    rref = unfraction_matrix(rref);
    return rref.map(function(row){return row[matrix.columns]});
}
extension.as_sum_of_basis = as_sum_of_basis;

/** Is the given matrix in row echelon form?
 * If not, throws an error with an explanation why it isn't.
 */
var is_row_echelon_form = extension.is_row_echelon_form = function(matrix) {
    var leader = -1;
    var rows = matrix.length;
    var columns = matrix[0].length;
    for(var row=0;row<rows;row++) {
        for(var column=0;column<columns;column++) {
            var cell = matrix[row][column];
            if(column<=leader) {
                if(!cell.is_zero()) {
                    throw(new Error("The first non-zero entry in row "+(row+1)+" is not strictly to the right of the first non-zero entries in the rows above."));
                } 
            } else {
                leader = column;
                break;
            }
        }
    }
    return true;
}
extension.is_row_echelon_form = function(matrix) {
    matrix = fraction_matrix(matrix);
    return is_row_echelon_form(matrix);
}

/** Is the given matrix in row echelon form?
 * If not, throws an error with an explanation why it isn't.
 */
var is_reduced_row_echelon_form = extension.is_reduced_row_echelon_form = function(matrix) {
    is_row_echelon_form(matrix); // this will throw an error if the matrix is not in row echelon form

    for(var row=0;row<matrix.rows;row++) {
        for(var column=0;column<matrix.columns;column++) {
            var cell = matrix[row][column];
            if(!cell.is_zero()) {
                if(!cell.is_one()) {
                    throw(new Error("The first non-zero entry in row "+(row+1)+" is not 1."))
                }
                for(var vrow=0;vrow<matrix.rows;vrow++) {
                    if(vrow!=row && !matrix[vrow][column].is_zero()) {
                        throw(new Error("There is more than one non-zero value in column "+(column+1)+"."));
                    }
                }
                break;
            }
        }
    }
    return true;
}
extension.is_reduced_row_echelon_form = function(matrix) {
    matrix = fraction_matrix(matrix);
    return is_reduced_row_echelon_form(matrix);
}

var scope = extension.scope;
var jme = Numbas.jme;
var funcObj = jme.funcObj;
var TNum = jme.types.TNum;
var TList = jme.types.TList;
var TVector = jme.types.TVector;
var TMatrix = jme.types.TMatrix;
var TString = jme.types.TString;
var THTML = jme.types.THTML;
var TBool = jme.types.TBool;

function element(name,attrs,content) {
    var e = document.createElement(name);
    for(var k in attrs) {
        e.setAttribute(k,attrs[k]);
    }
    if(content!==undefined) {
        e.innerHTML = content;
    }
    return e;
}

scope.addFunction(new funcObj('row_echelon_form',[TMatrix],TMatrix,function(matrix) {
    matrix = fraction_matrix(matrix);
    var res = row_echelon_form(matrix);
    var omatrix = unfraction_matrix(res.matrix);
    return omatrix;
}));

function show_steps(steps,describe_determinant) {
    var ops = element('ul');
    if(describe_determinant) {
        var li = element('li',{},'Let the determinant of the matrix be \\(d\\)');
    }
    var d = new Fraction(1);
    steps.map(function(o) {
        var li = element('li');
        li.appendChild(element('span',{},o.message));
        if(o.matrix) {
            var m = new TMatrix(unfraction_matrix(o.matrix));
            li.appendChild(element('span',{},'\\['+jme.display.texify({tok:m},{fractionnumbers:true})+'\\]'));
            if(describe_determinant && o.determinant_scale) {
                d = d.mul(o.determinant_scale);
                li.appendChild(element('p',{},'The determinant of this matrix is \\('+(Math.abs(d.n)==d.d ? d.n<0 ? '-' : '' : d.toLaTeX())+' d\\).'));
            }
        }
        ops.appendChild(li);
    });
    return new THTML(ops);
}

scope.addFunction(new funcObj('row_echelon_form_display',[TMatrix],THTML,function(matrix) {
    matrix = fraction_matrix(matrix);
    var res = row_echelon_form(matrix);
    return show_steps(res.operations);
},{unwrapValues:true}));

scope.addFunction(new funcObj('row_echelon_form_display_determinant',[TMatrix],THTML,function(matrix) {
    matrix = fraction_matrix(matrix);
    var res = row_echelon_form(matrix);
    return show_steps(res.operations,true);
},{unwrapValues:true}));

scope.addFunction(new funcObj('is_row_echelon_form',[TMatrix],TBool,function(matrix) {
    matrix = fraction_matrix(matrix);
    try {
        return is_row_echelon_form(matrix);
    } catch(e) {
        return false;
    }
}));

scope.addFunction(new funcObj('describe_why_row_echelon_form',[TMatrix],TString,function(matrix) {
    matrix = fraction_matrix(matrix);
    try {
        is_row_echelon_form(matrix);
        return "The matrix is in row echelon form.";
    } catch(e) {
        return e.message;
    }
}));

scope.addFunction(new funcObj('reduced_row_echelon_form',[TMatrix],TMatrix,function(matrix) {
    matrix = fraction_matrix(matrix);
    var res = reduced_row_echelon_form(matrix);
    var omatrix = unfraction_matrix(res.matrix);
    return omatrix;
}));

scope.addFunction(new funcObj('reduced_row_echelon_form_display',[TMatrix],THTML,function(matrix) {
    matrix = fraction_matrix(matrix);
    var res = reduced_row_echelon_form(matrix);
    return show_steps(res.operations);
},{unwrapValues:true}));

scope.addFunction(new funcObj('is_reduced_row_echelon_form',[TMatrix],TBool,function(matrix) {
    matrix = fraction_matrix(matrix);
    try {
        return is_reduced_row_echelon_form(matrix);
    } catch(e) {
        return false;
    }
}));

scope.addFunction(new funcObj('describe_why_reduced_row_echelon_form',[TMatrix],TString,function(matrix) {
    matrix = fraction_matrix(matrix);
    try {
        is_reduced_row_echelon_form(matrix);
        return "The matrix is in reduced row echelon form.";
    } catch(e) {
        return e.message;
    }
}));

scope.addFunction(new funcObj('rank',[TMatrix],TNum,extension.rank));

scope.addFunction(new funcObj('is_linearly_independent',[TList],TBool,extension.is_linearly_independent,{unwrapValues:true}));

scope.addFunction(new funcObj('adjoin',[TMatrix,TVector],TMatrix,adjoin,{unwrapValues:true}));

scope.addFunction(new funcObj('subset_with_dimension',[TList,TNum,TNum],TList,function(vectors,n,d) {
    var out = extension.subset_with_dimension(vectors,n,d);
    return out.map(function(v){return new TVector(v); });
},{unwrapValues:true}));

scope.addFunction(new funcObj('as_sum_of_basis',[TList,TVector],TList,extension.as_sum_of_basis,{unwrapValues:true}));




scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "random_elt",
    parameters: [],
    outtype: "number",
    language: "jme",
    definition: "random(-3..3 except 0)"
}, scope));

scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "random_nonzero_elt",
    parameters: [],
    outtype: "anything",
    language: "jme",
    definition: "random(-2..2 except 0)"
}, scope));
  

scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "get_row",
    parameters: [
        { name: "a", type: "number" },
        { name: "num_columns", type: "number" },
        { name: "pivots", type: "list" }
    ],
    outtype: "list",
    language: "jme",
    definition: "if(\n  a>len(pivots),\n  repeat(0, num_columns),\n  repeat(0, pivots[a-1]-1) + [1] + map(if(x in pivots, 0, random_elt()), x, pivots[a-1]+1..num_columns))"
}, scope));

scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "random_ref",
    parameters: [
        { name: "num_rows", type: "number" },
        { name: "num_columns", type: "number" },
        { name: "pivots", type: "list" }
    ],
    outtype: "matrix",
    language: "jme",
    definition: "matrix(map(get_row(j, num_columns, pivots), j, 1..num_rows))"
}, scope));

scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "get_row",
    parameters: [
        { name: "a", type: "number" },
        { name: "num_columns", type: "number" },
        { name: "pivots", type: "list" },
        { name: "rng", type: "list of number" }
    ],
    outtype: "list",
    language: "jme",
    definition: "if(\n  a>len(pivots),\n  repeat(0, num_columns),\n  repeat(0, pivots[a-1]-1) + [1] + map(if(x in pivots, 0, random(rng)), x, pivots[a-1]+1..num_columns))"
}, scope));

scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "random_ref",
    parameters: [
        { name: "num_rows", type: "number" },
        { name: "num_columns", type: "number" },
        { name: "pivots", type: "list" },
        { name: "rng", type: "list of number" }
    ],
    outtype: "matrix",
    language: "jme",
    definition: "matrix(map(get_row(j, num_columns, pivots, rng), j, 1..num_rows))"
}, scope));
  
  
scope.addFunction(Numbas.jme.variables.makeFunction({
    // a random row transformation (i.e., returns a matrix S such that
    // multiplying by S in the left amounts to a row transformation of type t,
    // where t=1 means add a random multiple of a random row to another random
    // row, t=2 means switch two random rows, t=3 means multiply a random row by
    // a random nonzero element.
    // The random values in the ground field are generated by invoking
    // random_nonzero_elt
    name: "random_row_transformation",
    parameters: [
        { name: "t", type: "number" },
        { name: "size", type: "number" }
    ],
    outtype: "anything",
    language: "jme",
    definition: "switch(t=1, elementary_matrix(let(a, random(1..size), b, random(1..size except a), [a,b]), random_nonzero_elt(), size), t=2, transposition_matrix(let(a, random(1..size), b, random(1..size except a), [a,b]), size), t=3, diag(let(pos, random(1..size), val, random_nonzero_elt(), repeat(1, pos-1) + [val] + repeat(1, size-pos))), unit_matrix(size))"
}, scope));

scope.addFunction(Numbas.jme.variables.makeFunction({
    // return product of elementary row transformation matrices of the given
    // types as produced by random_row_transformation (using random_nonzero_elt
    // for random nonzero elements)
    name: "random_trafo",
    parameters: [
        { name: "size", type: "number" },
        { name: "types", type: "list of number" }
    ],
    outtype: "matrix",
    language: "jme",
    definition: "if(len(types)=0, unit_matrix(size), random_trafo(size, types[0..len(types)-1]) * random_row_transformation(types[len(types)-1], size))"
}, scope));
  
scope.addFunction(Numbas.jme.variables.makeFunction({
    // a random row transformation (i.e., returns a matrix S such that
    // multiplying by S in the left amounts to a row transformation of type t,
    // where t=1 means add a random multiple of a random row to another random
    // row, t=2 means switch two random rows, t=3 means multiply a random row by
    // a random nonzero element.
    // The random values in the ground field are generated by choosing a random
    // element in the list rng whis is passed as a parameter.
    name: "random_row_transformation_rng",
    parameters: [
        { name: "t", type: "number" },
        { name: "size", type: "number" },
        { name: "rng", type: "list of number" }  // list of random_nonzero_elts
    ],
    outtype: "anything",
    language: "jme",
    definition: "switch(t=1, elementary_matrix(let(a, random(1..size), b, random(1..size except a), [a,b]), random(rng except 0), size), t=2, transposition_matrix(let(a, random(1..size), b, random(1..size except a), [a,b]), size), t=3, diag(let(pos, random(1..size), val, random(rng except [0,1]), repeat(1, pos-1) + [val] + repeat(1, size-pos))), unit_matrix(size))"
}, scope));

scope.addFunction(Numbas.jme.variables.makeFunction({
    // return product of elementary row transformation matrices of the given
    // types as produced by random_row_transformation (using rng for random
    // nonzero elements)
    name: "random_trafo_rng",
    parameters: [
        { name: "size", type: "number" },
        { name: "types", type: "list of number" },
        { name: "rng", type: "list of number" }  // list of random_nonzero_elts
    ],
    outtype: "matrix",
    language: "jme",
    definition: "if(len(types)=0, unit_matrix(size), random_trafo_rng(size, types[0..len(types)-1], rng) * random_row_transformation_rng(types[len(types)-1], size, rng))"
}, scope));


// --------------- Special matrices

scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "unit_matrix",
    parameters: [
        { name: "size", type: "number" }
    ],
    outtype: "anything",
    language: "jme",
    definition: "diag(repeat(1, size))"
}, scope));


scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "zero_matrix",
    parameters: [
        { name: "size", type: "number" }
    ],
    outtype: "anything",
    language: "jme",
    definition: "diag(repeat(0, size))"
}, scope));
  
scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "single_entry",
    parameters: [
        { name: "size", type: "number" },
        { name: "pos", type: "number" },
        { name: "val", type: "number" }
    ],
    outtype: "anything",
    language: "jme",
    definition: "repeat(0, pos-1) + [val] + repeat(0, size-pos)"
}, scope));

scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "diag",
    parameters: [
        { name: "values", type: "list of number" }
    ],
    outtype: "anything",
    language: "jme",
    definition: "matrix(map(single_entry(len(values), x, values[x-1]), x, 1..len(values)))"
}, scope));


scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "transposition_matrix",
    parameters: [
        { name: "t", type: "list of number" },
        { name: "size", type: "number" }
    ],
    outtype: "anything",
    language: "jme",
    definition: "permutation_matrix(map(switch(r = t[0], t[1], r = t[1], t[0], r), r, 1..size))"
}, scope));

scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "permutation_matrix",
    parameters: [
        { name: "p", type: "list of number" }
    ],
    outtype: "matrix",
    language: "jme",
    definition: "transpose(matrix(map(single_entry(len(p), x, 1), x, p)))"
}, scope));

scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "elementary_matrix",
    parameters: [
        { name: "x", type: "list of number" },
        { name: "a", type: "number" },
        { name: "size", type: "number" }
    ],
    outtype: "anything",
    language: "jme",
    definition: "unit_matrix(size) + matrix(map(switch(r = x[0], single_entry(size, x[1], a), repeat(0, size)), r, 1..size))"
}, scope));

scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "jordan_block",
    parameters: [
        { name: "lambda", type: "number" },
        { name: "size", type: "number" }
    ],
    outtype: "matrix",
    language: "jme",
    definition: "matrix(map(map(if(x=y, lambda, if(x-y=1, 1, 0)), x, 1..size), y, 1..size))"
}, scope));

  
scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "companion_matrix",
    parameters: [
        { name: "pol", type: "list" },
        { name: "size", type: "number" }
    ],
    outtype: "matrix",
    language: "jme",
    definition: "matrix(map(map(if(x=size, -pol[y-1], if(y-x=1, 1, 0)), x, 1..size), y, 1..size))"
}, scope));
  
  
// ---------------------------------------------------

scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "columns",
    parameters: [
        { name: "A", type: "matrix" }
    ],
    outtype: "list",
    language: "jme",
    definition: "map(vector(x), x, list(transpose(A)))"
}, scope));


scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "modp",
    parameters: [
        { name: "A", type: "matrix" },
        { name: "p", type: "number" }
    ],
    outtype: "anything",
    language: "jme",
    definition: "matrix(map(map(mod(x, p), x, row), row, list(A)))"
}, scope));


scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "inverse_matrix",
    parameters: [
        { name: "A", type: "matrix" }
    ],
    outtype: "anything",
    language: "jme",
    definition: "let(En, unit_matrix(numcolumns(A)), matrix(map(x[len(list(A))..2*len(list(A))], x, list(reduced_row_echelon_form(matrix(map(list(A)[x] + list(En)[x], x, 0..len(list(A))-1))))))\n)"
}, scope));

scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "basis_solution_set",
    parameters: [
        { name: "A", type: "matrix" }
    ],
    outtype: "matrix",
    language: "jme",
    // return basis of solution set of homogeneous system of linear equations defined by matrix $A$
    // return value is a matrix whose columns are the basis
    definition: "let(num_columns, numcolumns(A), rankA, rank(A), switch(rankA = 0, matrix(unit_matrix(num_columns)), rankA = num_columns, matrix([]), let(B, matrix(list(reduced_row_echelon_form(A))[0..rankA]), pivots, map(indices(x, 1)[0]+1, x, list(B)), perm, permutation_matrix(pivots + sort(list(set(1..num_columns)-set(pivots)))), C, transpose(matrix(columns(-B * transpose(perm))[len(pivots)..num_columns])), perm * matrix(list(C) + list(unit_matrix(num_columns-rankA))))))"
}, scope));

scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "all_entries_zero",
    parameters: [
        { name: "A", type: "matrix" }
    ],
    outtype: "boolean",
    language: "jme",
    definition: "all(map(all(map(c=0, c, row)), row, list(A)))"
}, scope));


scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "trace",
    parameters: [
        { name: "A", type: "matrix" }
    ],
    outtype: "number",
    language: "jme",
    definition: "sum(map(A[x][x], x, 0..numrows(A)-1))"
}, scope));
  
scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "max_abs",
    parameters: [
        { name: "M", type: "matrix" }
    ],
    outtype: "number",
    language: "jme",
    definition: "max(map(max(map(abs(x), x, row)), row, list(M)))"
}, scope));
  
  


// ------------ LaTeX ------------------------

scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "latex_columns_as_vectors",
    parameters: [
        { name: "A", type: "matrix" }
    ],
    outtype: "string",
    language: "jme",
    definition: "join(map(\"\\\\begin\\{pmatrix\\} \" + join(map(string(rational(cc)), cc, list(x)), \"\\\\\\\\\")  + \"\\\\end\\{pmatrix\\}\", x, columns(A)), \", \")"
}, scope));

scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "latex_solution_as_linear_combination",
    parameters: [
        { name: "solution", type: "list of number" },
        { name: "v", type: "string" }
    ],
    outtype: "string",
    language: "jme",
    definition: "join(map(\n    latex_coeff(c[1]) + \" \" + v + \"_\" + string(c[0]+1),\n    c, enumerate(solution)), \" + \")"
}, scope));

scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "latex_coeff",
    parameters: [
        { name: "c", type: "number" }
    ],
    outtype: "string",
    language: "jme",
    definition: "if(c=1, \"\",\n   if(c >= 0, string(rational(c)), \"(\" + string(rational(c)) + \")\"))"
}, scope));

scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "latex_columns_as_vectors_v",
    parameters: [
        { name: "A", type: "matrix" },
        { name: "v", type: "string" }
    ],
    outtype: "anything",
    language: "jme",
    definition: "join(map(v + \"_\" + string(x[0]+1) + \" = \\\\begin\\{pmatrix\\} \" + join(map(string(rational(cc)), cc, list(x[1])), \"\\\\\\\\\")  + \"\\\\end\\{pmatrix\\}\", x, enumerate(columns(A))), \",\\\\quad \")"
}, scope));


scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "latex_mapsto",
    parameters: [
        { name: "A", type: "matrix" },
        { name: "B", type: "matrix" }
    ],
    outtype: "anything",
    language: "jme",
    definition: "let(\n  colA, columns(A),\n  colB, columns(B),\n  num_columns, len(columns(A)),\n  join(map(\"\\\\begin\\{pmatrix\\} \" + join(map(string(rational(cc)), cc, list(colA[c])), \"\\\\\\\\\")  + \"\\\\end\\{pmatrix\\}\" + \"\\\\mapsto \\\\begin\\{pmatrix\\} \" + join(map(string(rational(cc)), cc, list(colB[c])), \"\\\\\\\\\")  + \"\\\\end\\{pmatrix\\}\", c, 0..num_columns-1), \",\\\\qquad\")\n)"
}, scope));
  
// ------------ stuff related to characteristic polynomial, minimal polynomial, jordan normal form
scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "poly_prod",
    parameters: [
        { name: "ps", type: "list" }
    ],
    outtype: "anything",
    language: "jme",
    definition: "if(len(ps)=1, ps[0], ps[0] * poly_prod(ps[1..len(ps)]))"
}, scope));
  
scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "block_diag_two_matrices",
    parameters: [
        { name: "A", type: "matrix" },
        { name: "B", type: "matrix" }
    ],
    outtype: "matrix",
    language: "jme",
    definition: "let(sizeA, len(columns(A)), sizeB, len(columns(B)), matrix(map(x+repeat(0, sizeB), x, list(A)) + map(repeat(0, sizeA)+x, x, list(B))))"
}, scope)); 
 
scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "block_diag",
    parameters: [
        { name: "ms", type: "list" }
    ],
    outtype: "matrix",
    language: "jme",
    definition: "if(len(ms)=1, ms[0], block_diag_two_matrices(ms[0], block_diag(ms[1..len(ms)])))"
}, scope));  

scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "matrix_power",
    parameters: [
        { name: "A", type: "matrix" },
        { name: "x", type: "number" }
    ],
    outtype: "matrix",
    language: "jme",
    definition: "if(x=0, unit_matrix(len(list(A))), A*matrix_power(A, x-1))"
}, scope));  

scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "matrix_sum",
    parameters: [
        { name: "l", type: "list" }
    ],
    outtype: "matrix",
    language: "jme",
    definition: "if(len(l)=1, l[0], l[len(l)-1] + matrix_sum(l[0..len(l)-1]))"
}, scope));  

scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "matrix_poly",
    parameters: [
        { name: "A", type: "matrix" },
        { name: "pl", type: "polynomial" }
    ],
    outtype: "matrix",
    language: "jme",
    definition: "matrix_sum(map(pl[x]*matrix_power(A,x), x, 0..degree(pl)))"
}, scope));    
  
  
scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "latex_matrix",
    parameters: [
        { name: "A", type: "matrix" }
    ],
    outtype: "string",
    language: "jme",
    definition: '"\\\\begin\\{pmatrix\\}" + join(map(join(map(string(s), s, r), " & "), r, list(A)   ), "\\\\\\\\") + "\\\\end\\{pmatrix\\}"'
}, scope));  


scope.addFunction(Numbas.jme.variables.makeFunction({
    name: "latex_extd_coeff_matrix",
    parameters: [
        { name: "A", type: "matrix" }
    ],
    outtype: "string",
    language: "jme",
    definition: '"\\\\left( \\\\begin\\{array\\}\\{" + join(repeat("r", numcolumns(A)-1), "") + "|r\\}" + join(map(join(map(string(rational(s)), s, r), " & "), r, list(A)   ), "\\\\\\\\") + "\\\\end\\{array\\} \\\\right)"'
}, scope));  
});