/*
 * This file contains code to implement a boolean extension for Numbas.
 * 
 */

/**
 * @external Numbas
 */

//import "Numbas"
var Numbas


Numbas.addExtension('propositionallogic',['jme','base'],function(propositionallogic) {
  // Import the Numbas types that we are going to need.
  var TExpression = Numbas.jme.types.TExpression
  var TNum = Numbas.jme.types.TNum
  var TList = Numbas.jme.types.TList
  var TBool = Numbas.jme.types.TBool
  var TOp = Numbas.jme.types.TOp
  var TString = Numbas.jme.types.TString
  var TName = Numbas.jme.types.TName


  // Numbas exposes a number of functions to work
  // with expressions.
  var isType = Numbas.jme.isType
  var funcObj = Numbas.jme.funcObj
  var isOp = Numbas.jme.isOp
  var treesSame = Numbas.jme.treesSame      // Compare expressions
  var arraysEqual = Numbas.util.arraysEqual // Compare arrays
  var compile = Numbas.jme.compile          // Create Numbas Expression from tree
  var findvars = Numbas.jme.findvars        // Identify the variables in Numbas expression


  propositionallogic.generateExpression = function(nops,lnames,operators) {
    if (!operators) {operators = ["or","and","not"]}
  var p = randomExpression(nops,lnames,operators)
  return(new TExpression(p))
  }

  propositionallogic.equivExpr = function(eq1,eq2) {
  var v1 = findvars(eq1).concat(findvars(eq2))
  var varnames = v1.filter(onlyUnique).sort()
  for (var t of ttInput(varnames)) {
    if (Numbas.jme.builtinScope.evaluate(eq1,t).value !== Numbas.jme.builtinScope.evaluate(eq2,t).value) { return false;}
  }
  return true
  }
  
  propositionallogic.getFeedbackLaw = function(sA,sB,option) {
    return getFeedback(sA,sB,propositionallogic.scope.getVariable('laws'),option)
  }
    
   propositionallogic.checkStep = function(sA,sB,option) {
    return checkStep(sA,sB,propositionallogic.scope.getVariable('laws'),option)
  }
  
  propositionallogic.scope.setVariable('laws',
                                       new TList(["Boolean Operation",
                                                  "Definition of ⟹",
                                                  "Absorption Law",
                                                  "Associative Law",
                                                  "Commutative Law",
                                                  "De Morgan's Law",
                                                  "Distributive Law",
                                                  "Domination Law",
                                                  "Double Negation Law",
                                                  "Identity Law",
                                                  "Idempotent Law",
                                                  "Negation Law"]))
  // JME functions:
  // generateExpression
  propositionallogic.scope.addFunction(new funcObj('generateExpression',[TNum,TList],TExpression,
                                                   propositionallogic.generateExpression,
                                                   {unwrapValues:true}))
    // equivExpression
  propositionallogic.scope.addFunction(new funcObj('equivExpressions',[TExpression,TExpression],TBool,
                                                   propositionallogic.equivExpr,
                                                   {unwrapValues:true}))
  propositionallogic.scope.addFunction(new funcObj('ttInput',[TExpression],TList,
                                                   ttInput
                                                   ,{unwrapValues:true}))
  propositionallogic.scope.addFunction(new funcObj('truthColumn',[TExpression],TList,
                                                   truthColumn
                                                   ,{unwrapValues:true}))
  propositionallogic.scope.addFunction(new funcObj('truthTable',[TExpression],TList,
                                                   truthTable
                                                   ,{unwrapValues:true}))
  propositionallogic.scope.addFunction(new funcObj('htmlTruthTable',[TExpression],TString,
                                                   htmlTruthTable
                                                   ,{unwrapValues:true}))
  
  propositionallogic.scope.addFunction(new funcObj('latexify',[TExpression],TString,
                                                   ex=>toLatex(ex,false),{unwrapValues:true}))
  propositionallogic.scope.addFunction(new funcObj('stringify',[TExpression],TString,
                                                   ex=>toString(ex,false),{unwrapValues:true}))
  propositionallogic.scope.addFunction(new funcObj('simplifyProposition',[TExpression],TExpression,
                                                   simplifyProposition
                                                   ,{unwrapValues:true}))
  propositionallogic.scope.addFunction(new funcObj('findLaw',[TExpression,TExpression],TString,
                                                   propositionallogic.getFeedbackLaw
                                                   ,{unwrapValues:true}))
  propositionallogic.scope.addFunction(new funcObj('getMarkingMatrix',[TExpression,TExpression,TList],TList,
                                                   getMarkingMatrix
                                                   ,{unwrapValues:true}))
  
  propositionallogic.scope.addFunction(new funcObj('compareExprLists',[TList,TList],TBool,
                                                     compareExprLists,
                                                     {unwrapValues:true}))
  
  propositionallogic.scope.addFunction(new funcObj('checkSubExprList',[TList,TExpression],TString,
                                                     checkSubExprList,
                                                     {unwrapValues:true}))

  propositionallogic.scope.addFunction(new funcObj('normalize',[TString],TString,
                                                     normalize,
                                                     {unwrapValues:true}))  

  propositionallogic.scope.addFunction(new funcObj('checkStep',[TExpression,TExpression,TList],TBool,
                                                     function(sA,sB,option) {
                                        return getFeedback(sA,sB,propositionallogic.scope.getVariable('laws'),option)
                                                      },
                                                     {unwrapValues:true}))    
  
  //Numbas.jme.display.texify = toLatex


  /**
   * Construct the input columns of a truth table.
   * 
   * @param {[string]} - a list of atoms that are input to the table.
   * @return {[keypair]} - a list of objects corresponding to the possible combinations of values.
   */
  function ttInput(v) {
    if(v.length == 0)
      return [{}]
    var l = []
    var t1 = ttInput(v.slice(1))
    for(var t of t1)
    {
      l.push(Object.assign({},{[v[0]]:true},t))
      l.push(Object.assign({},{[v[0]]:false},t))
    }
    return l
  }

  /** 
   * Construct the truth table of an expression.
   * 
   * By default it constructs the table based on the atoms present in the expression,
   * sorted in alphabetical order. It is possible to specifiy another list, in
   * any order.
   * 
   * @param {expr} - the expression whose truth column to construct.
   * @param {varnames} [vars(expr)} - the atoms to construct against.
   * @return {[TBool]} a list of values corresponding to the truth table of the input.
   */
  function truthColumn(expr,varnames) {
    if (typeof(varnames)==='undefined') varnames = findvars(expr).sort()
    if (expr.tok.valueOf() === true || expr.tok.valueOf() === false) 
    { return Array(2**(varnames.length)).fill(expr.tok.valueOf());}
    var l = []
    for (var t of ttInput(varnames)) {
      l.push(Numbas.jme.builtinScope.evaluate(expr,t).value)
      }
    return l
  }

  /**
   *  Check whether two expressions are logically equivalent by comparing their truth tables.
   * 
   * This function is more robust than the Numbas one, which tests random input.
   * It can be slower if there are many atoms in the expression.
   * 
   * @param {Tree} expr1 - first expression to compare.
   * @param {Tree} expr2 - second expression to compare. 
   * @return {bool} - true if the two expressions are logically equivalent, else false.
   **/
  function equivExpr(expr1,expr2) {
    var v1 = findvars(expr1).concat(findvars(expr2))
    var varnames = v1.filter(onlyUnique).sort()
    for (var t of ttInput(varnames)) {
      if (Numbas.jme.builtinScope.evaluate(expr1,t).value !== Numbas.jme.builtinScope.evaluate(expr2,t).value) { return false;}
      }
    return true
    }


  /* See https://stackoverflow.com/questions/1960473/get-all-unique-values-in-a-javascript-array-remove-duplicates */
  function onlyUnique(value, index, self) { 
      return self.indexOf(value) === index
  }

  /**
   * Generate a random Numbas expression.
   * 
   * @param{unsigned} n - number of operators in the expressions.
   * @param{[string]} atoms - list of atoms to use.
   * @param{[string]} ops - list of operators to use.
   * @return {Tree} - a Numbas expression tree containing a random expression.
   */
  function randomExpression(n,atoms,ops) {
    var lhs, rhs;
    if(n==0){
      var whichAtom = Numbas.math.randomint(atoms.length)
      return {tok: new TName(atoms[whichAtom]), type: "name"}
    }
    var whichop = Numbas.math.randomint(ops.length)
    var op = new TOp(ops[whichop])
    if (op.name == "not") {
      lhs = randomExpression(n-1,atoms,ops)
      while (isOp(lhs.tok,"not") && Math.random() < 0.9) {
        lhs = randomExpression(n-1,atoms,ops)
      }
      return {tok:op,args:[lhs],type:"op"}
    }
    var k = Numbas.math.randomint(n-1)
    lhs = randomExpression(k,atoms,ops)
    rhs = randomExpression(n-1-k,atoms,ops)
    // Accept expressions like (p or p) , or (p or (q or r)) with probability 10%.
    while( (treesSame(rhs,lhs) || isOp(rhs.tok,op.name) || isOp(lhs.tok,op.name)) && Math.random() < 0.99)
    {
      k = Numbas.math.randomint(n-1)
      lhs = randomExpression(k,atoms,ops)
      rhs = randomExpression(n-1-k,atoms,ops)
    }
    
    return {tok:op,args:[lhs,rhs],type:"op"}
  }

  /**
   * Get all the subexpressions of a boolean expression.
   * 
   * @paramn {Tree} expr - the expression to parse.
   * @return {[Texpression]} - list of expressions that are subexpressions of the input.
   */
  function getSubtree(expr) {
  if(isType(expr.tok,"name")) {return [expr]}
   // return([])
   
    var X = expr.args.map(getSubtree)
    var Y = X.reduce((x,y) => x.concat(y))
    Y.push(expr)
    return Y
  }

  /**
   * Remove duplicate elements from a list.
   * 
   * The order of the elements in the returned list is the order
   * of first appearance in the input list.
   * 
   * @param {list} L - list of elements to check;
   * @return {list} list of elements, each appearing only once.
   */
  function removeDuplicates(L) {
    return L.filter((x,i,A) => A.findIndex(y => (treesSame(x,y))) == i)
  }


  /**
   * Find the elements in L1 that are not in L2.
   *
   * @param {list} L1 - the list of elements to look up
   * @param {list} L2 - the list of elements to compare with
   * @param {function} comp - a comparison function returning true
   *                   if two elements are equal and false otherwise
   * @returns {list} the list of elements in L1 that are not in L2.
   */
  function findMissing(L1,L2,comp) {
    var missing = []
    for (var i=0;i < L1.length;i++){
      var e1 = L1[i]
      if(!L2.some(e => comp(e,e1))) {
        missing.push(toLatex(e1))
      }
    }
    return missing
  }

  /**
   * Compare two lists of expressions to make sure they contain the same
   * expressions, not necessarily in the same order.
   * 
   * @param {list} L1 - list of expressions to compare.
   * @param {list} L2 - list of expressions to compare against.
   * @return {[list,list]} - list of expressions in L2\L1 and in L1\L2.
   */
  function compareExprLists(L1,L2) {
    var missing = findMissing(L2, L1, treesSame) /* The expressions in L1 that are not in L2. */
    var excess = findMissing(L1, L2, treesSame)  /* The expressions in L2 that are not in L1. */
    return [missing,excess]
  }

  /**
   * Generate an informative feedback message on
   * the elements of two lists.
   * 
   * @param {List[Texpression]} - missing  list of expressions missing 
   *                            from what is expected.
   * @param {List[Texpression]} - excess  list of expressions that should
   *                            not be there.
   * @returns {string} a feedback string and boolean that is true only
   *                         when the two lists are empty.
   */
  function generateHeaderFeedback(missing,excess){
    var msg = ""
    if(missing.length > 0){
      msg += "<p>The following subexpressions of the original are missing from the headers (and columns) in your table: "
      msg += missing.map(x => "$"+x+"$").join(", ")
      msg += "</p>"
    }
    if(excess.length > 0){
      msg += "<p>The following expressions are in the headers in your table but are not subexpressions of the original expression: "
      msg += excess.map(x => "$"+x+"$").join(", ")
      msg += "</p>"
    }
    return msg
  }

  /**
   * Check whether a list of expressions is the list of all
   * subsexpressions of a given expression.
   * 
   * @param {List[Texpression]} - L1 a list of expressions to compare
   * @param {Texpression} - expr the expression whose subexpressions we will compare
   *                        against.
   * 
   * @returns {string} - feedback with the difference between the list of expression
   *                     and the subsexpressions of `expr`.
   */ 
  function checkSubExprList(L1,expr) {
    var [missing,excess] =  compareExprLists(removeDuplicates(L1.map(compile)),removeDuplicates(getSubtree(expr)))
    var msg = generateHeaderFeedback(missing,excess)
    return msg
  }


  /**
   * Generate the truth table of an expression.
   * 
   * @param {TExpression} - expr the expression for which to build the truth table.
   * 
   * @returns {Keymap} - a key-value pair mapping each subexpression to its truth column.
   */
  function truthTable(expr) {
    var atoms = findvars(expr)
    var X = removeDuplicates(getSubtree(expr))
    return [X.map(x => new TExpression(x)),X.map( x => truthColumn(x,atoms).map(y => +y) )]
  }

  /**
   * Create a html table for a given truth table.
   * 
   * @param {Keymap} - a key-value pair mapping every subsexpression to its truth column.
   * @returns {string} - a string containing the html code for the table.
   */
  function htmlTruthTable(expr) {
    var table = truthTable(expr).map(e => toLatex(e))
    var html = '<table class="truthtable"><thead>'
    html += '<tr><th>'+table[0].join("</th><th>")+'</th></tr>'
    html += "</thead><tbody>"
    var nrows = table[1][1].length
    for(var i=0;i<nrows;i++){
      html += '<tr>'
      for(let r of table[1]){
        html += '<td>' + (r[i] ? 'T':'F') + '</td>'
      }
      html += '</tr>'
    }
    html += '</tbody></table>'
    return html
  }

  /**
   * The boolean operation is an algebraic operation that replaces ¬F with T, or
   * ¬T with F.
   */
  const booleanOp = {
    name: "Boolean Operation",
    expression: "$$¬T ≡ F$$ or $$¬F ≡ T$$",
    check(t1,t2,checkrtl=true){
      if(!isOp(t1.tok,"not")) {return checkrtl && this.check(t2,t1,false);}
      if(!isType(t1.args[0].tok,"boolean") || !isType(t2.tok,"boolean")) {return false;}
      return t1.args[0].tok.value != t2.tok.value
    }
  }

  /**
   * In boolean logic the implication is defined as a ⟹ b ≡ ¬a ∨ b.
   */
  const implicationDef = {
    name: "Definition of ⟹",
    expression: "$$a ⟹ b ≡ ¬a ∨ b$$",
    check(t1,t2,checkrtl=true){
    /* Check that t1 is an implication */
    if (!isOp(t1.tok,"implies")) {return checkrtl && this.check(t2,t1,false)}
    /* Check that t2 is an or operation */
    if (!isOp(t2.tok,"or")) {return checkrtl && this.check(t2,t1,false)}
    /* Check that one of the arguments of t2 is the second argument of t1. */
    var i=0;
    while (i<2 && !treesSame(t1.args[1],t2.args[i])) {i++}
    if (i==2) {return false}
    /* Finally check that the other argument of t2 is a negation. */
    return isOp(t2.args[1-i].tok,"not") && treesSame(t2.args[1-i].args[0],t1.args[0])
    }
  }

  /**
   * The idempotent law
   */
  const idempotentLaw = {
    name: "Idempotent Law",
    expression: "$$a ∨ a ≡ a$$ or $$a ∧ a ≡ a$$",
    check(t1,t2,checkrtl=true) {
      /* Step 1: check that t1 is an operation, or or and */
      if (!isOp(t1.tok,"or") && !isOp(t1.tok,"and"))
      {return checkrtl && this.check(t2,t1,false);}
      /* Step 2: check that both arguments of t1 are the same as t2 */
      if(!treesSame(t2,t1.args[0]) || !treesSame(t2,t1.args[1]))
      {return checkrtl && this.check(t2,t1,false);}
      return true
    }
  }

  const negationLaw = {
    name: "Negation Law",
    expression: "$$a ∨ ¬a ≡ T$$ or $$a ∧ ¬a ≡ F$$",
     check(t1,t2,checkrtl=true) {
      /* Step 1: t2 should be a boolean */
    
      if (!isType(t2.tok,"boolean")) {return checkrtl && this.check(t2,t1,false);}
      /* Step 2: t1 should be "and" or "or" */
      if (!isOp(t1.tok,"or") && !isOp(t1.tok,"and"))
      {return checkrtl && this.check(t2,t1,false);}
      /* Step 3: the value of t2 must agree with the operation of t1 */
      if(t2.tok.value == true && isOp(t1.tok,"and")) { return false;}
      if(t2.tok.value == false && isOp(t1.tok,"or")) { return false;}
      /* Step 4: one of the arguments of t1 is a negation. */
      var i=0
      while(i<2 && !isOp(t1.args[i].tok,"not")) i++
      if(i==2) {return checkrtl && this.check(t2,t1,false);}
      /* Step 5: the arguments must coincide */
      if(!treesSame(t1.args[i].args[0],t1.args[1-i]))
      {return checkrtl && this.check(t2,t1,false);} 
      return true
    }
  }

  const dominationLaw = {
    name: "Domination Law",
    expression: "$$a ∨ T ≡ T$$ or $$a ∧ F ≡ F$$",
    check(t1,t2,checkrtl=true) {
      /* Step 1: t2 should be true or false */
      if (!isType(t2.tok,"boolean")) { return checkrtl && this.check(t2,t1,false);}
      /* Step 2: t1 should be the correct operation */
      if (t2.tok.value == true && !isOp(t1.tok,"or"))
        {return checkrtl && this.check(t2,t1,false);}
      if (t2.tok.value == false && !isOp(t1.tok,"and"))
        {return checkrtl && this.check(t2,t1,false);}
      /* Step 3: t1 should have either RHS or LHS the value of t2. */
      if (t1.args[0].tok.value != t2.tok.value && t1.args[1].tok.value != t2.tok.value) { return checkrtl && this.check(t2,t1,false);}
      return true
    }
  }

  const identityLaw = {
    name: "Identity Law",
    expression: "$$a ∨ F ≡ a$$ or $$a ∧ T ≡ a$$",
    check(t1,t2,checkrtl=true) {
      /* Step 1: t1 should be or or and */
      if (t1.tok.type != "op" || isOp(t1.tok,"not"))
        {return checkrtl && this.check(t2,t1,false);}
      /* Step 2: t1 should have a boolean as an argument. */
      if (t1.args[0].tok.type != "boolean" && t1.args[1].tok.type != "boolean")
        {return checkrtl && this.check(t2,t1,false);}
      /* Step 3: the other argument of t1 should be t2. */
      if (!treesSame(t2,t1.args[0]) && !treesSame(t2,t1.args[1])) { return checkrtl && checkDomination(t2,t1,false);}
      /* Step 4: the operation should match the boolean value. */
      if(isOp(t1.tok,"or")) {
        if (t1.args[0].tok.value !== false && t1.args[1].tok.value !== false)
          { return checkrtl && this.check(t2,t1,false);}
      }
      else {
        if (t1.args[0].tok.value !== true && t1.args[1].tok.value !== true)
          { return checkrtl && this.check(t2,t1,false);}
      }
      return true
    }
  }

  const commutativeLaw = {
    name: "Commutative Law",
    expression: "$$a ∨ b ≡ b ∨ a$$ or $$a ∧ b ≡ b ∧ a$$",
    check(t1,t2) { 
      /* Step 1: both  must be operators, and not negations */
      if (!isType(t1.tok,"op") || !isType(t2.tok,"op") || isOp(t1.tok,"not")) {return false;}
      /* Step 2: is it the same operation? */
      if (t1.tok.name != t2.tok.name) {return false;}
      /* Step 3: are the arguments swapped? */
      return (treesSame(t1.args[0], t2.args[1]) && treesSame(t1.args[1], t2.args[0]))
    }
  }

  const associativeLaw = {
    name:  "Associative Law",
    expression: "$$a ∨ (b∨c) ≡ (a∨b) ∨ c$$ or $$a ∧ (b∧c) ≡ (a∧b) ∧ c$$",
    check(t1,t2,checkrtl=true) {
      /* Step 1: both  must be operators, and not negations */
      if (!isType(t1.tok,"op") || !isType(t2.tok,"op") || isOp(t1.tok,"not")) {return false;}
      /* Step 2: is it the same operation? */
      if (t1.tok.name != t2.tok.name) {return false;}
      /* Step 3: in t1, the 2nd argument should have the same operation */
      if (!isType(t1.args[1].tok,"op") || t1.args[1].tok.name != t1.tok.name) {return checkrtl && this.check(t2,t1,false);}
      if (!isType(t2.args[0].tok,"op") || t2.args[0].tok.name != t2.tok.name) {return checkrtl && this.check(t2,t1,false);}
      return treesSame(t1.args[0],t2.args[0].args[0]) && treesSame(t2.args[1],t1.args[1].args[1]) && treesSame(t1.args[1].args[0],t2.args[0].args[1])
    }
  }

  const doublenegationLaw = {
    name: "Double Negation Law",
    expression: "$$¬¬a ≡ a$$",
    check(t1,t2,checkrtl=true) {
      /* Step 1: at least one of the trees must be an operator */
      if (!isType(t1.tok,"op") && !isType(t2.tok,"op")) {return false;}
      /* Step 2: that operator must be a negation.*/
      if ( !isOp(t1.tok,"not")){ return checkrtl && this.check(t2,t1,false);}
      /* Step 3: that operator must be a double negation. */
      if (!isType(t1.args[0].tok,"op") || !isOp(t1.tok,"not")) { return checkrtl && this.check(t2,t1,false);}
      /* Step 4: the argument of that double negation is the other expression. */
      if (!treesSame(t1.args[0].args[0],t2)) { return checkrtl && this.check(t2,t1,false);}
      return true
    }
  }

  const distributiveLaw = {
    name: "Distributive Law",
    expression: "$$a ∨ (b∧c) ≡ (a∨b) ∧ (a∨c)$$ or $$a ∧ (b∨c) ≡ (a∧b) ∨ (a∧c)$$",
    direction(t1,t2,r) {
        /* Step 5: the right argument of t1 is the same operator as t2 */ 
      if (t1.args[1-r].tok.name != t2.tok.name)  { return false;}
      /* Step 6: the left argument of t1 is the same as the left argument for both of t2's sides */
      if ( !treesSame(t1.args[r],t2.args[0].args[r]) || !treesSame(t1.args[r],t2.args[1].args[r]))
      {    return false;  }
      /* Step 7: the arguments of the rhs of t1 are the rhs of each arguments of t2 */
      if ( !treesSame(t1.args[1-r].args[0],t2.args[0].args[1-r]) || !treesSame(t1.args[1-r].args[1],t2.args[1].args[1-r]))
      {  return false; }
      return true
    },
    check(t1,t2,checkrtl=true) {
      /* Step 1: both  must be operators, and not negations */
      if (!isType(t1.tok,"op") || !isType(t2.tok,"op") || isOp(t1.tok,"not") || isOp(t2.tok,"not")) {return false;}
      /* Step 2: the operators should be different */
      if (t1.tok.name == t2.tok.name) {return false;}
      /* Step 3: both sides of t2 should be the same operators */
      if (!isType(t2.args[0].tok,"op") || !isType(t2.args[1].tok,"op") || t2.args[0].tok.name != t2.args[1].tok.name)
      {    return checkrtl && this.check(t2,t1,false);  }
      /* Step 4: that operator should not be the same as t1's operator */
      if (t2.args[0].tok.name != t1.tok.name) 
      {    return checkrtl && this.check(t2,t1,false);  }
      /* Step 6: Right distribution or Left distribution */
      if (!this.direction(t1,t2,0) && !this.direction(t1,t2,1))
      {    return checkrtl && this.check(t2,t1,false);  }
      return true
    }
  }

  const deMorganLaw = {
    name: "De Morgan's Law",
    expression: "$$¬(a∨b) ≡ ¬a ∧ ¬b$$ or $$¬(a∧b) ≡ ¬a ∨ ¬b$$",
    check(t1,t2) {
      /* Step 1: both  must be operators */
      if (!isType(t1.tok,"op") || !isType(t2.tok,"op")) {return false;}
      /* Step 2: t1 or t2 is a negation of an operation */
      if (!isOp(t1.tok,"not")) {return isOp(t2.tok,"not") && this.check(t2,t1);}
      /* Step 3: t1.args is the opposite operation to t2 */
      if (!isType(t1.args[0].tok,"op") || t2.tok.name == t1.args[0].tok.name) {return false;}
      if (isOp(t2.tok,"not") || isOp(t1.args[0].tok,"not")) {return false;}
      /* Step 4: both arguments of t2 must be negations */
      if (!isType(t2.args[0].tok,"op") || !isType(t2.args[1].tok,"op")) {return false;}
      if (!isOp(t2.args[0].tok,"not") || !isOp(t2.args[1].tok,"not")) {return false;}
      /* Step 5: the arguments must be equal */
      if (!treesSame(t1.args[0].args[0],t2.args[0].args[0]) || !treesSame(t1.args[0].args[1],t2.args[1].args[0])) {return false;}
      return true
    }
  }

  const absorptionLaw = {
    name: "Absorption Law",
    expression: "$$a ∨ (a∧b) ≡ a$$ or $$a ∧ (a∨b) ≡ a$$",
    check(t1,t2,checkrtl=true) {
      /* Step 1: t1 must be an operator: `and` or `or` */
      if (!isOp(t1.tok,"or") && !isOp(t1.tok,"and")) { return checkrtl && this.check(t2,t1,false);}
      /* Step 2: t1 must have t2 as an argument */
      var i=0
      while (i<2 && !treesSame(t2,t1.args[i])) {i++}
      if (i==2) { return checkrtl && this.check(t2,t1,false)}
      let t3 = t1.args[1-i]
      /* Step 3: the *other* argument of p must  be an operation */
      if (!isOp(t3.tok,"or") && !isOp(t3.tok,"and")) { return checkrtl && this.check(t2,t1,false);}
      /* Step 4: that argument must be different from t1 */
      if (t1.tok.name == t3.tok.name) { return checkrtl && this.check(t2,t1,false);}
      /* Step 5: that argument must have t2 as an argument */
      if (!treesSame(t2,t3.args[0]) && !treesSame(t2,t3.args[1])) { return checkrtl && this.check(t2,t1,false);}
      return true
    }
  }

  const laws = [ absorptionLaw, commutativeLaw, associativeLaw, doublenegationLaw, distributiveLaw, deMorganLaw,
  dominationLaw, idempotentLaw, negationLaw, identityLaw, booleanOp, implicationDef ]

  /**
   * Find the boolean logic law that was applied betwen two expressions, if any.
   * 
   * @param {TExpression} t1 - first expression to compare
   * @param {TExpression} t2 - second expression to compare
   * @param {string} name of the law that was applied, or nothing.
   */
  function checkLaw(t1,t2) {
    for(let law of laws) { if(law.check(t1,t2)) {return law.name} }

    if (isType(t1.tok,"op") && isType(t2.tok,"op") && t1.tok.name == t2.tok.name) {
    /* The operation might be in one of the subtrees.*/
    if(isOp(t1.tok,"not")) return checkLaw(t1.args[0],t2.args[0])
    if(treesSame(t1.args[0],t2.args[0])) return checkLaw(t1.args[1],t2.args[1])
    if(treesSame(t1.args[1],t2.args[1])) return checkLaw(t1.args[0],t2.args[0])
    }
    return "Nothing"
  }

  function checkStep(sA,sB) {
    if(sA === "" || sB === "") {return "";}
    var A = compile(sA)
    var B = compile(sB)
    var m = ""
    if(A && B)
      m = checkLaw(A,B)
    return m
  }

  function getFeedback(sA,sB,laws,option) {
    if (!laws) laws = propositionallogic.scope.getVariable('laws')
    var listOfLaws = laws.value
    var optionItem = option.findIndex(x => x[0])
    var appliedLaw = "At this step you went from:"
    //if(optionItem > -1) {appliedLaw = "So, you applied the "+listOfLaws[optionItem]+" to "}
    var expressions = '<div class="information"> $$'+toLatexdiff(sA,sB,false)+"$$ to: $$"+toLatexdiff(sB,sA,false)+".$$ The parts that you changed are highlighted in blue (the operators on which the law was applied are in red).</div>"

    var subexpressions = diffTree(sA,sB)
    var ssA = sA
    var ssB = sB
    if(subexpressions.length > 1) {
      ssA = subexpressions[0]
      ssB = subexpressions[1]
    }

    var specific = '<div class="warning">'+appliedLaw+"$$"+toLatex(ssA,false)+"$$ to obtain $$"+toLatex(ssB,false)+".$$</div>"
    if (!equivExpr(ssA,ssB)) {
      var errval = findErrors(ssA,ssB);
      var names = Object.keys(errval)
      var sAVals = toLatex(ssA)
      var sBVals = toLatex(ssB)
      for (let n of names){
        var r = new RegExp("\\b" + n +"\\b","g")
        sAVals = sAVals.replace(r,errval[n]?'T':'F')
        sBVals = sBVals.replace(r,errval[n]?'T':'F')
      }
      var exampleValues = names.map(k => k + "=" + (errval[k]?'T':'F')).join(', ')
      var examplelhs = Numbas.jme.builtinScope.evaluate(ssA,errval).value
      var examplerhs = Numbas.jme.builtinScope.evaluate(ssB,errval).value
      var example = "if you set " + exampleValues + ", then <ul><li>$" + 
                    toLatex(ssA,false) +
                    '$ becomes $' + sAVals +'$'+
                    ', which is ' + examplelhs+
                    ' and </li><li>$'+
                    toLatex(ssB,false)+
                    '$ becomes $' + sBVals + '$'+
                    ', which is ' + examplerhs + '.</li></ul>'
      return [0,expressions+specific + '<div class="error">These two expressions are not logically equivalent. For example, ' + example + '</div>',""]
    }
    var m = checkLaw(ssA,ssB)
    if (m == "") {return [1,"There is an issue with the marking algorithm. Please contact your teacher.",""];} /* That should never happen */
    /* TODO: Ask to enter a law if no law has been selected */
    var l = listOfLaws.indexOf(m)
    if (m == "Nothing" || l == -1) {return [0,expressions+specific + '<p class="error"> Although the expressions are logically equivalent, they cannot be obtained from one another by applying a single law.</p>',m];}
    if (!option[l][0]) {return [0.5,'<p class="error">You identified the wrong law: the law you applied was in fact ' + m +'.</p>',m];}
    else {return [1,"",m];}
  }

  /**
   * Find for what values of the variables two boolean expressions are different.
   * 
   * @param {Tree} expr1 - first expression to compare.
   * @param {Tree} expr2 - second expression to compare.
   * @return {[keypair]} list of value combinations differentiating the two expressions.
   * 
   **/
  function findErrors(expr1,expr2) {
    var v1 = findvars(expr1).concat(findvars(expr2))
    var varnames = v1.filter(onlyUnique).sort()
    for (var t of ttInput(varnames)) {
      if (Numbas.jme.builtinScope.evaluate(expr1,t).value !== Numbas.jme.builtinScope.evaluate(expr2,t).value) { return t;}
      }
    return []
    }

  /**
   * For two different expression tree, find the part of the trees that are different
   * 
   * @param {Tree} t1 - first tree to compare.
   * @param {Tree} t2 - second tree to compare.
   * @return {[Tree,Tree]} - the subexpressions for `t1` and `t2` that differ. 
   **/
  function diffTree(t1,t2) {
    /* Case 1: the expressions are the same.*/
    if(treesSame(t1,t2)) return [];
    /* Case 2: one of the expressions is an atom. In that case, we do not need to descend
       into any subtree. Return the expressions.*/
    if(!isType(t1.tok,"op") || !isType(t2.tok,"op")) return [t1,t2];
    /* Case 3: the operators are different. Again, that means that the expressions are different
       at that node and we can return them.*/
    if(t1.tok.name != t2.tok.name) return [t1,t2];
    /* Case 4: the operators are both negations.In that case, 
       descend into the argument and return the result.*/
    if(isOp(t1.tok,"not") && isOp(t2.tok,"not")) return diffTree(t1.args[0],t2.args[0]);
    /* Case 5: one of the sides of the argument is the same. In that case descend into the other
       side.*/
    for(let i=0;i<2;i++){
      if(treesSame(t1.args[i],t2.args[i])) return diffTree(t1.args[1-i],t2.args[1-i]);
    }
    /* Case 6: this is the last case: both sides of the operator are different. We stop here and return
       the expression at that node.*/
    return [t1,t2];
  }

  function getMarkingMatrix(sA,sB,laws) {
    //laws = Numbas.extensions.propositionallogic.scope.getVariable('laws')
    var listOfLaws = laws.value
    var m = ""
    var marks = listOfLaws.map( () => 0)
    if( sA && sB) {m = checkLaw(sA,sB);}
    
    if (m !== "") {
      var l = listOfLaws.indexOf(m)
      if (l > -1) {marks[l] = 1;}
    }
    return marks
  }

  function getProps(d,tCols,varnames) {
    var n = d.length+1
    var l = []
    var p, q, newtCol,newop;
    /* Start by negating all the expressions at the last level. */
    for (p of d[n-2]) {
      var ex = p[0]
      if (typeof(ex) === "string") { ex = {tok: new TName(ex), type:"name"};}
      newop = {tok:new TOp("not"),args:[ex],type:"op"}
      /* Get the truth values of the new expression: if that truth column is already in the list, it means that there
      is a simpler expression than this one, and we can discard it.
      So, we only add the new expression if that truth column is not in the list. */
      newtCol = truthColumn(newop,varnames)
      if(!(tCols.some(a => arraysEqual(a,newtCol)))) {
        l.push([newop,newtCol])
          tCols.push(newtCol)
      }
    }
    /* Now, we apply diadic operations (and, or, implies) to the list. We apply the same strategies as for the negation. */
    for (var i=1;i<n/2;i++) {
      for (p of d[i]) {
        for (q of d[n-1-i]) {
          for (var sop of ["and","or","implies"]) {
              var ex1 = p[0]
              if (typeof(ex1) == "string") {ex1 = {tok: new TName(ex1), type: "name"}}
              var ex2 = q[0]
              if (typeof(ex2) == "string") {ex2 = {tok: new TName(ex2), type: "name"}}
              newop = {tok:new TOp(sop),args:[ex1,ex2], type: "op"}
              /* Get the truth values of the new expression: if that truth column is already in the list, it means that there
              is a simpler expression than this one, and we can discard it.
              So, we only add the new expression if that truth column is not in the list. */
              newtCol = truthColumn(newop,varnames)
              if(!(tCols.some(a => arraysEqual(a,newtCol))))
              {
                  l.push([newop,newtCol])
                  tCols.push(newtCol)
              }
        }
    }
    }
    }
    d.push(l)
    return d
  }

  /**
   * Generate the list of propositions corresponding to all the possible
   * truth tables based on the atoms.
   * 
   * The propositions in this list are the simplest possible, in the sense
   * that they contain as few operations as possible.
   * 
   * @param{[string]} atoms - list of atoms to use in the expressions
   * @return{[[TExpression,keymap]]} - list of expressions and their corresponding truth column.
   */
  function simpleProps(atoms) {
    var d = []
    var tCols = []
    var natoms = atoms.length
    var nprops = 2**(2**natoms)
    var l = []
    var ex,expr,tCol
    for (ex of [new TBool(true),new TBool(false)]) {
      expr = {tok:ex, type: "boolean"}
      tCol = truthColumn(expr,atoms)
      tCols.push(tCol)
      l.push([expr,tCol])
    }
    d.push(l)
    l = []
    for (ex of atoms.map(x=>new TName(x))) {
      expr = {tok:ex, type: "boolean"}
      tCol = truthColumn(expr,atoms)
      tCols.push(tCol)
      l.push([expr,tCol])
    }
    d.push(l)
    while (tCols.length < nprops) {
      getProps(d,tCols,atoms)
    }
    var allprops = d.reduce((a,b) => a.concat(b))
    return allprops.map(v => [new TExpression(v[0]),v[1]])
  }

  /**
   * Find the simplest boolean expression logically equivalent to the input.
   * Right now it only works when there are two atoms in the expression.
   * 
   * @param {Tree} expr - the expression to simplify.
   * @return {Tree} an expression logically equivalent to the input.
   **/
  function simplifyProposition(expr) {
    if(!expr) {return expr}
    let varnames = findvars(expr).sort()
    let a = varnames.map(x => new Object({tok : new TName(x)}))
    let natoms = a.length
    if (natoms == 0) {natoms=1; varnames=["p","q"]}
    let tt = truthColumn(expr,varnames)
    let props = {
      1: [{tok:new Boolean(true), type: "boolean"},  {tok:new Boolean(false), type: "boolean"},
          a[0],{tok: new TOp("not"),args : [a[0]], type: "op"}],
      2: [{tok:new Boolean(true)},  {tok:new Boolean(false)}].concat(
          a,a.map(x => new Object({tok: new TOp("not"),args: [x], type: "op"})),[
               {tok: new TOp("and"),args : [a[0],a[1]], type: "op"}, {tok: new TOp("or"),args : [a[0],a[1]], type: "op"},
               {tok: new TOp("implies"),args : [a[0],a[1]], type: "op"},{tok: new TOp("implies"),args : [a[1],a[0]], type: "op"},
               {tok: new TOp("xor"),args : [a[0],a[1]], type: "op"},
               {tok: new TOp("not"),args : [{tok: new TOp("and"),args : [a[0],a[1]]}], type: "op"},
               {tok: new TOp("not"),args : [{tok: new TOp("or"),args : [a[0],a[1]]}], type: "op"},
               {tok: new TOp("not"),args : [{tok: new TOp("xor"),args : [a[0],a[1]]}], type: "op"},
               {tok: new TOp("and"),args : [{tok: new TOp("not"),args : [a[0]]},a[1]], type: "op"},
               {tok: new TOp("and"),args : [a[0],{tok: new TOp("not"),args : [a[1]]}], type: "op"}
              ]),
      3: [{tok:new Boolean(true), type: "boolean"},  {tok:new Boolean(false), type: "boolean"},
          a,a.map(x => new Object({tok: new TOp("not"),args: [x]})),
           {tok: new TOp("not"),args : [a[0]]}, {tok: new TOp("not"),args : [a[1]]},
           {tok: new TOp("and"),args : [a[0],a[1]]}, {tok: new TOp("or"),args : [a[0],a[1]]},
           {tok: new TOp("implies"),args : [a[0],a[1]]},{tok: new TOp("implies"),args : [a[1],a[0]]},
           {tok: new TOp("not"),args : [{tok: new TOp("and"),args : [a[0],a[1]]}]},
           {tok: new TOp("not"),args : [{tok: new TOp("or"),args : [a[0],a[1]]}]},
           {tok: new TOp("and"),args : [{tok: new TOp("not"),args : [a[0]]},a[1]]},
           {tok: new TOp("and"),args : [a[0],{tok: new TOp("not"),args : [a[1]]}]}
          ],
    }
    let simpleExpr = props[natoms].find(x=>arraysEqual(truthColumn(x,varnames),tt))
    if (tt.every(x => x)) return new TExpression("true")
    if (tt.every(x => !x)) return new TExpression("false")
    return new TExpression(simpleExpr)
  }

  /**
   * Generate a LaTeX formulation of a boolean expression.
   * 
   * @param {Tree} expr - the expression to transform into LaTeX.
   * @param {bool} [false] parent - whether to surround the expression with parentheses.
   */
  function toLatex(expr,paren) {
    if(expr.tok.type == "name") return expr.tok.name
    if(expr.tok.type == "boolean") return expr.tok.value ? "T":"F"
    if(expr.tok.type == "op") {
    if(expr.tok.name == "not") return "\\neg "+toLatex(expr.args[0],true)
    var binops = [["or"," \\lor "],["and"," \\land "],["implies"," \\implies "],["xor"," \\oplus "]]
    var op = binops.find(x => x[0] == expr.tok.name)[1]
    //if(expr.tok.name == "or") {op \ " \\lor "}
    //op = (expr.tok.name == "or") ? " \\lor " : " \\land " 
    var str = toLatex(expr.args[0],true) + op + toLatex(expr.args[1],true)
    if(paren) return "(" + str + ")"
    else return str;}
  }

  /**
   * Generate a LaTeX formulation of a boolean
   * expression, with the differences from another expression
   * highlighted.
   * 
   * @param {Tree} t1 - the tree whose expression to latexify. 
   * @param {Tree} t2 - the tree to compare `t1` to and highlight the differences.
   * @return {String} - a string containing colored LaTeX/Mathjax code.
   */
  function toLatexdiff(t1,t2,paren){
    var msg = ""
    /* Case 1: the expressions are the same.*/
    if(treesSame(t1,t2)) return toLatex(t1,paren,true);
    /* Case 2: one of the expressions is an atom. In that case, we do not need to descend
       into any subtree. Return the expressions.*/
    if(!isType(t1.tok,"op") || !isType(t2.tok,"op")) return "{\\color{blue}{" + toLatex(t1,paren,true)+"}}";
    var optxt
    if(isOp(t1.tok,"not")) {optxt = "\\neg "}
    if(isOp(t1.tok,"and")) {optxt = "\\land "}
    if(isOp(t1.tok,"or")) {optxt = "\\lor "}
    if(isOp(t1.tok,"implies")) {optxt = "\\implies "}
    /* Case 3: the operators are different. Again, that means that the expressions are different
       at that node and we can return them.*/
    if(t1.tok.name != t2.tok.name) {
      if(t1.args.length > 1){
        if(paren){msg = "("}
        msg += "{\\color{blue}{"+toLatex(t1.args[0],true)+"}}"
        msg += "{\\color{red}{"+optxt+"}}"
        msg += "{\\color{blue}{"+toLatex(t1.args[1],true)+"}}"
        if(paren){msg += ")"}
      }
      else {
        msg = "{\\color{red}{\\neg}}"
        msg += "{\\color{blue}{"+toLatex(t1.args[0],true)+"}}"
      }
      return msg
    }
    /* Case 4: the operators are both negations.In that case, 
       descend into the argument and return the result.*/
    if(isOp(t1.tok,"not") && isOp(t2.tok,"not")){
        return "\\neg"+toLatexdiff(t1.args[0],t2.args[0],true)
     }
    /* Case 5: one of the sides of the argument is the same. In that case descend into the other
       side.*/
    if(treesSame(t1.args[0],t2.args[0])){
      return toLatex(t1.args[0],true) + optxt + toLatexdiff(t1.args[1],t2.args[1],true) 
    }
    if(treesSame(t1.args[1],t2.args[1])){
      return toLatexdiff(t1.args[0],t2.args[0],true)+ optxt +  toLatex(t1.args[1],true) 
    }
    /* Case 6: this is the last case: both sides of the operator are different. We stop here and return
       the expression at that node.*/
    if(paren){msg = "("}
    msg += "{\\color{blue}{" + toLatex(t1.args[0],true)+"}}"
    msg += "{\\color{red}{" + optxt + "}}"
    msg += "{\\color{blue}{" + toLatex(t1.args[1],true)+"}}"
    if(paren){msg += ")"}
    return msg
  }

  /**
   * Write an expression as a unicode string.
   * 
   * @param {Tree} expr - the expression to express as a string.
   * @param {bool} [false] paren - whether to surround the expression with parentheses.
   * @return {string} the string representing the input expression.
   */
  function toString(expr,paren) {
    if(expr.tok.type == "name") return expr.tok.name
    if(expr.tok.type == "boolean") return expr.tok.value ? "True":"False"
    if(expr.tok.type == "op") {
    if(expr.tok.name == "not") return "¬"+toString(expr.args[0],true)
    var op;
    if (expr.tok.name == "or") op = " ∨ "
    if(expr.tok.name == "and") op = " ∧ " 
    if(expr.tok.name == "implies") op = " ⟹ " 
    var str = toString(expr.args[0],true) + op + toString(expr.args[1],true);
    if(paren) return "(" + str + ")"
    else return str;}
  }

  /**
   * Replace the input to be properly interpreted by Numbas.
   * Identify various ways of entering "True" and "False" ("T","true", etc.).
   * Identify various ways of entering "p" and "q".
   * 
   * On Apple, the expression generated by Numbas replaces letters with their
   * slanted unicode version, which causes issues with interpreting on Numbas. 
   * 
   * @param {string} expr - the input to sanitize. 
   * @return {string} - the sanitized string.
   */ 
  function normalize(expr) {
    /* Start from replacing whatever could be "true" with the boolean value. */
    const tregex = /\bT(rue)?/gi
    const fregex = /\bF(alse)?/gi

    var e = expr
    /* Replace various unicode versions of letters with their ANSI ones. */
    e = e.normalize("NFKD")
    e = e.toLowerCase()

    e = expr.replace(tregex,'True')
    e = e.replace(fregex,'False')

    /* Replace square brackets with rounded ones. */
    e = e.replace(/\[/g,'(')
    e = e.replace(/\]/g,')')

    return e;
  }
})
